# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2002,2007 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
package CLOG;

=head1 NAME

CLOG - Perl API to CSM CLOG logging system.

=head1 SYNOPSIS

use CLOG;

=head1 DESCRIPTION

Logging via CSM CLOG from Perl programs.

All functions that take parameters use named parameters with either the "-name=>value" or "name=>value"
styles, though the "-name=>value" is the standard. So, you can call these functions and pass the 
parameters in any order you choose. Rest assured that if any required parameters are not provided, 
you will get an error return code from the function.

This class caches basic required data values needed in the CLOG C API and only requires the 
data specific to each message being recorded. For example, the message catalog name will not 
generally change, so that is set once in the constructor and this value is used for the life of
the object. 

However, these values may be overridden at each logging request. Continuing with the our example,
when the actual request to log an entry is made, a new value for the message catalog may be passed 
(as the -messageCatalog parameter) and this value will be in effect for this logging request only.

=head1 EXPORTED CONTSTANTS

=item B<Message Types:>

=item B<CLOG_ERROR_MSG>              - Error messages         

=item B<CLOG_WARNING_MSG>            - Warning messages       

=item B<CLOG_INFO_MSG>               - Informational messages 

=item B<CLOG_ALL_MSGS>               - Value to indicate all message types 

=item B<Function return codes:>

=item B<MAP_FILE_ERROR>              - Error in the map file - see error message for details.

=item B<BAD_OR_MISSING_CLOGGER>      - The clogger command is bad or missing

=item B<MISSING_PARAM>               - Bad parameter passed to function

=item B<BAD_PARAM>                   - Bad parameter passed to function

=item B<CLOG C API return codes: note that every effort to keep this list 

=item B<CLOG_SUCCESS>                - Clog function call was successful.       

=item B<CLOG_ALLOC_FAILURE>          - Unable to allocate storage.              

=item B<CLOG_INVALID_ENV>            - Bad ct_clog_ptr or ct_clog_t.            

=item B<CLOG_INVALID_CATALOG>        - Bad or missing catalog name, or unable to open message catalog. 

=item B<CLOG_INVALID_MESSAGE_SET>    - Bad or missing message set.              

=item B<CLOG_INVALID_MESSAGE_ID>     - Bad or missing message id.               

=item B<CLOG_INVALID_COMP_ID>        - Bad or missing component name.           

=item B<CLOG_INVALID_LPP_ID>         - Bad or missing LPP name.                 

=item B<CLOG_FFDC_ERROR>             - Error in call to FFDC.                   

=item B<CLOG_RMC_CONN_ERROR>         - Unable to connect to RMC.                

=item B<CLOG_RMC_PERM_DENIED>        - Denied permission by RMC.                

=item B<CLOG_RMC_ERROR>              - Other RMC error.                         

=item B<CLOG_BAD_NODE_ID>            - Call to cu_get_node_id() failed          

=item B<CLOG_BAD_DETAIL_DATA>        - Invalid or missing detail data.          

=item B<CLOG_DETAIL_DATA_OVERSIZE>   - Detail data is too large (100 bytes max).

=item B<CLOG_BAD_AIX_TEMP_ID>        - Bad AIX template ID.                     

=item B<CLOG_INVALID_MSG_TYPE>       - Invalid message type.                    

=item B<CLOG_INVALID_DEF_MSG>        - Bad or missing default message.          

=item B<CLOG_ERROR_LOG_NOT_ENABLED>  - Error logging is not enabled - call clog_set_error_log_info() first. 

=item B<CLOG_MSG_TYPE_NOT_ENABLED>   - Logging is not enabled for passed message type 

=item B<CLOG_BAD_PARAMS>             - Some other error was found in the passed parameters. 

=item B<CLOG_TRUE>                   - A true result value from CLOG.           

=item B<CLOG_FALSE>                  - A false result value from CLOG.          

=item B<CLOG_INTERNAL_ERROR>         - Internal error in CLOG API

=head1 EXPORTED VARIABLES

=item B<ERROR_MESSAGE>              - last error that occured in *module*

=item B<ERROR_CODE>                 - last error that occured in *module*

=cut

BEGIN
{ 
	 $::csmpm = $ENV{'CSM_PM'} ? $ENV{'CSM_PM'} : '/opt/csm/pm';
} #for development use

use lib $::csmpm;
use NodeUtils();
require MessageUtils;
use strict;

# This tells perl to honor locale for sorting, dates, etc.
# More info about locale.pm and perl handling of locales can be found in
# /usr/lib/perl5/5.6.0/locale.pm and the man page for perllocale.
use locale;

use Carp;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS); # $AUTOLOAD);

require Exporter;
require AutoLoader;
#require DynaLoader;

@ISA = qw(Exporter);

%EXPORT_TAGS = ( );

#@EXPORT    = qw( );
#@EXPORT_OK = ( );
#@EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );


# CLOGMessageTypes
use constant CLOG_ALL_MSGS              =>-1;  # Value to indicate all message types 
use constant CLOG_ERROR_MSG             => 1;  # Error messages         
use constant CLOG_WARNING_MSG           => 2;  # Warning messages       
use constant CLOG_INFO_MSG              => 4;  # Informational messages 
use constant CLOG_LAST_MESSAGE_TYPE_VAL => 4;  # Internal use only

# CLOG Return codes 
use constant MAP_FILE_ERROR              =>-13; # Error in the map file - see error message for details.
use constant BAD_OR_MISSING_CLOGGER      =>-12; # The clogger command is bad or missing
use constant MISSING_PARAM               =>-11; # Bad parameter passed to function
use constant BAD_PARAM                   =>-10; # Bad parameter passed to function
use constant CLOG_SUCCESS                => 0; # Clog function call was successful.       
use constant CLOG_ALLOC_FAILURE          => 1; # Unable to allocate storage.              
use constant CLOG_INVALID_ENV            => 2; # Bad ct_clog_ptr or ct_clog_t.            
use constant CLOG_INVALID_CATALOG        => 3; # Bad or missing catalog name, or unable to open message catalog. 
use constant CLOG_INVALID_MESSAGE_SET    => 4; # Bad or missing message set.              
use constant CLOG_INVALID_MESSAGE_ID     => 5; # Bad or missing message id.               
use constant CLOG_INVALID_COMP_ID        => 6; # Bad or missing component name.           
use constant CLOG_INVALID_LPP_ID         => 7; # Bad or missing LPP name.                 
use constant CLOG_FFDC_ERROR             => 8; # Error in call to FFDC.                   
use constant CLOG_RMC_CONN_ERROR         => 9; # Unable to connect to RMC.                
use constant CLOG_RMC_PERM_DENIED        =>10; # Denied permission by RMC.                
use constant CLOG_RMC_ERROR              =>11; # Other RMC error.                         
use constant CLOG_BAD_NODE_ID            =>12; # Call to cu_get_node_id() failed          
use constant CLOG_BAD_DETAIL_DATA        =>13; # Invalid or missing detail data.          
use constant CLOG_DETAIL_DATA_OVERSIZE   =>14; # Detail data is too large (100 bytes max).
use constant CLOG_BAD_AIX_TEMP_ID        =>15; # Bad AIX template ID.                     
use constant CLOG_INVALID_MSG_TYPE       =>16; # Invalid message type.                    
use constant CLOG_INVALID_DEF_MSG        =>17; # Bad or missing default message.          
use constant CLOG_ERROR_LOG_NOT_ENABLED  =>18; # Error logging is not enabled - call clog_set_error_log_info() first. 
use constant CLOG_MSG_TYPE_NOT_ENABLED   =>19; # Logging is not enabled for passed message type 
use constant CLOG_BAD_PARAMS             =>20; # Some other error was found in the passed parameters. 
use constant CLOG_TRUE                   =>21; # A true result value from CLOG.           
use constant CLOG_FALSE                  =>22; # A false result value from CLOG.          
use constant CLOG_INTERNAL_ERROR         =>23; # Internal error in CLOG API - see trace file for details of error.    

$VERSION = '1.0';
my $CLOGGER = "/opt/csm/csmbin/clogger";

#               catalog name     directory where map file is    message set name
my @CATALOG = ("IBM.CLOG.cat",    "/opt/csm/msgmaps/",               "Perl");

################################################################################
# ;mash
################################################################################
sub mash
{
	 my $self = shift;
	 my $str  = shift;

	 return NodeUtils->quote($str);


#	 print "Before:$str\n";
#	 $str =~ s/\\/\\\\/sg; # replace single \ with \
#	 $str =~ s/\$/\\\$/sg; # replace $ with \$
#	 $str =~ s/\'/\\\'/sg; # replace ' with \'
#	 $str =~ s/\"/\\\"/sg; # replace " with \"
#	 print "After :$str\n";
#	 return "\"$str\"";

}
################################################################################
# ;clear_errors
################################################################################
sub clear_errors {
	 my $self = shift;
	 $self->{error_message} = "";
	 $self->{error_code}    = 0;
}

################################################################################
# ;validate_args
################################################################################
sub validate_args {

	 my $self = shift;
    my %input;
	 my $match;
	 my $required_match;
	 my %match_words_not_seen;
	 my $key;

	 # $#_ returns last index of @_, so if its 6, then we really have 7 elements...
	 if ( $#_ % 2 == 0 ) {
		  $self->{error_code}     = MISSING_PARAM;
		  $self->{error_message}  = MessageUtils->getMessageFromCat(@CATALOG,
																					'EMsgMissingParamError', "");
		  return MISSING_PARAM;
	 }

	 %input = @_;
#	 print "input{_match}=[".$input{"_match"}."]\n";
	 if ( $input{"_match"} ne "" ) {
		  $match = $input{"_match"};
		  delete $input{"_match"};
	 }
	 else {
		  $self->{error_code}   = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " _match");
		  return MISSING_PARAM;
	 }

	 if ( $input{"_required"} ne "" ) {
		  $required_match = $input{"_required"};
		  delete $input{"_required"};
	 }
	 else {
		  $required_match = $match;
	 }

	 # store a hash of all match words for use later.
	 my $word;
	 foreach $word ( split(/\|/,$required_match) ) {
		  $match_words_not_seen{$word}=1;
	 }

 
	 # this verifies that the key names are OK
    foreach $key ( keys %input ) { 
#		  print "validate_args. Checking [$key]\n";
		  $key =~ s/^-//; #remove starting dash if there is one
		  if( $key !~ /($match)$/)
		  {
				$self->{error_code}   = BAD_PARAM;
				$self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																						'EMsgInvalidParamError', $key);
				return BAD_PARAM;
		  }
	 }

	 # and this verifies that the values are OK
	 foreach $key ( keys %input ) {
		  if( !$input{$key} ) {
				$self->{error_code}   = MISSING_PARAM;
				$self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																						'EMsgMissingParamError', " $key");
				return MISSING_PARAM;
		  }
		  $key =~ s/^-//;
		  delete $match_words_not_seen{$key};
	 }

	 # finally, verify that we have seen each and every parameter in the match set
  	 foreach $key (keys %match_words_not_seen ) {
  		  if ( $match_words_not_seen{$key} == 1 ) {
  				print "$key = $match_words_not_seen{$key}\n";
  				$self->{error_code}    = MISSING_PARAM;
  				$self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																						'EMsgMissingParamError', " $key");
  				return MISSING_PARAM;
  		  }
  	 }
	 return 0;

}

=head1 FUNCTIONS

Note - each parameter is required to be present and non zero. 
If the parameter is within [brackets], then it is optional.

=cut 

################################################################################
################################################################################

=item B<new()>

Creates a CLOG object.

=item PARAMETERS

 compID         - Component ID 
 lppID          - Licenced Product ID
 messageCatalog - Message catalog file name
 messageSet     - Message set name
 messageMapDir  - Directory where Perl message map is for the message catalog

=item RETURNS 

 blessed reference to new object on success
 BAD_PARAM     - bad parameter passed
 MISSING_PARAM - missing parameter

 

=cut
################################################################################
# ;new
################################################################################
sub new {
    my $self = {
		  error_code => "",       # numeric return code of last action
		  error_message => "",    # string error of last action, if any.
		  compID => "",
		  lppID => "",
		  messageCatalog => "",
		  messageSet => "", 
		  messageMapDir => "",
		  errorTemplate => "",
		  detailedData => "",
		  detailedDataSize => "",
	 };

    my %input;
	 my $key;

    shift @_; #gets rid of the CLOG input

	 # $#_ returns last index, so if its 6, then we really have 7 elements...
	 if ( $#_ % 2 == 0 ) {
		  $CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																				'EMsgMissingParamError',"");
		  $CLOG::ERROR_CODE    = MISSING_PARAM;
		  return undef;
	 }

    while(@_){
		  my ($name, $value) = (shift, shift);
		  $name =~ s/^-//; #remove starting dash if there is one
		  if( $name !~ /(compID|lppID|messageCatalog|messageSet|messageMapDir)$/ )
		  {
				$CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																					 'EMsgInvalidParamError', $name);
				$CLOG::ERROR_CODE    = BAD_PARAM;
				return BAD_PARAM;
		  }
		  else{
				$input{$name} = $value;
		  }
    }

	 # ensure that if a parameter was passed, that its value is present
	 foreach $key ( keys %input ) {
		  if( !$input{$key} ) {
				$CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																					 'EMsgMissingParamError', " $key");
				$CLOG::ERROR_CODE    = BAD_PARAM;
				return MISSING_PARAM;
		  }
	 }
	 
	 
	 if ( $input{"compID"} ) {
		  $self->{compID} = $input{"compID"};
	 }
	 else {
		  $CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																				'EMsgMissingParamError', " compID");
		  $CLOG::ERROR_CODE    = BAD_PARAM;
		  return MISSING_PARAM;
	 }

	 if ( $input{"lppID"} ) {
		  $self->{lppID} = $input{"lppID"};
	 }
	 else {
		  $CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																				'EMsgMissingParamError', " lppID");
		  $CLOG::ERROR_CODE    = BAD_PARAM;
		  return MISSING_PARAM;
	 }

	 if ( $input{"messageCatalog"} ) {  
		  $self->{messageCatalog} = $input{"messageCatalog"};
	 }
	 else {
		  $CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																				'EMsgMissingParamError', " messageCatalog");
		  $CLOG::ERROR_CODE    = BAD_PARAM;
		  return MISSING_PARAM;
	 }

	 if ( $input{"messageSet"} ) {
		  $self->{messageSet} = $input{"messageSet"};
	 }
	 else {
		  $CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																				'EMsgMissingParamError', " messageSet");
		  $CLOG::ERROR_CODE    = BAD_PARAM;
		  return MISSING_PARAM;
	 }

	 if ( $input{"messageMapDir"} ) {
		  $self->{messageMapDir} = $input{"messageMapDir"};
	 }
	 else {
		  $CLOG::ERROR_MESSAGE = MessageUtils->getMessageFromCat(@CATALOG,
																				'EMsgMissingParamError', " messageMapDir");
		  $CLOG::ERROR_CODE    = BAD_PARAM;
		  return MISSING_PARAM;
	 }

	 
    bless $self;

	 $self->enable_msgs(-messageType => CLOG_ALL_MSGS);
	 $self->clear_errors();

    return $self;
}

################################################################################
################################################################################

=item B<get_error()>

Returns a list containing the last return error code and last return error message.
If the last operation did not return an error, then the error message will be a blank string
and the error code will be 0.

Note: the error message may not always be set even in cases where the error code is not 0.

=item PARAMETERS

none

=item RETURNS 

($ErrorCode, $ErrorMessage)

=cut

################################################################################
# ;get_error
################################################################################
sub get_error {
    my $self = shift;
    return ($self->{error_code}, $self->{error_message});
}

################################################################################
################################################################################

=item B<clog_rec()>

Record a message.

=item PARAMETERS
 
 messageType        - one of the error type constants
 fileName           - name of source file logging this entry
 lineNumber         - line number in source file logging this entry
 sccsID             - sccs ID of source file logging this entry
 messageID          - message ID nmumonic from map file
 [fields]           - reference to a list of substitutional field values
 [compID]           - component ID for this entry only
 [lppID]            - lpp ID for this entry only
 [messageCatalog]   - message catalog for this entry only
 [messageSet]       - message set for this entry only
 [messageMapDir]    - message map directory for this entry only
 [errorTemplate]    - AIX error template ID for this entry only
 [detailedData]     - detailed data for this entry only
 [detailedDataSize] - detailed data size for this entry only

=cut

# ;clog_rec
sub clog_rec {
    my $self = shift;
    my %input;
	 my $options;

	 if ( $self->validate_args(@_, 
										_match=>"fileName|lineNumber|sccsID|messageMapDir|messageType|messageID|fields|".
										"compID|lppID|messageCatalog|messageSet|errorTemplate|detailedData|detailedDataSize",
										_required=>"fileName|lineNumber|sccsID|messageType|messageID") != 0 ) {
		  return $self->{error_code};
	 }


	 %input = @_;
	 
	 # Get all required data to be passed to clogger command. We allow any values passed to this 
	 # method to override temporarily those stored in this object. Some of these values are never
	 # stored in the object (eg, lineNumber), but we do it this way since
	 #  1) its clearer what we are doing
	 #  2) it does no real harm other than minor run-time cost
	 #  3) if we decide in the future to cache some data internally, this will still have the correct behavior.

	 # basic stuff from constructor
	 my $compID        = $input{compID}         ? $input{compID}          : $self->{compID};
	 my $lppID         = $input{lppID}          ? $input{lppID}           : $self->{lppID};
	 my $messageMapDir = $input{messageMapDir}  ? $input{messageMapDir}   : $self->{messageMapDir};
	 my $messageSet    = $input{messageSet}     ? $input{messageSet}      : $self->{messageSet};
	 my $messageCat    = $input{messageCatalog} ? $input{messageCatalog}  : $self->{messageCatalog};

	 # AIX error log information
	 my $detailedData      = $input{detailedData}     ? $input{detailedData}     : $self->{detailedData};
	 my $detailedDataSize  = $input{detailedDataSize} ? $input{detailedDataSize} : $self->{detailedDataSize};
	 my $errorTemplate     = $input{errorTemplate}    ? $input{errorTemplate}    : $self->{errorTemplate};

	 # Specific to this error logging call
	 my $fileName      = $input{fileName}       ? $input{fileName}        : $self->{fileName};
	 my $lineNumber    = $input{lineNumber}     ? $input{lineNumber}      : $self->{lineNumber};
	 my $sccsID        = $input{sccsID}         ? $input{sccsID}          : $self->{sccsID};
	 my $messageID     = $input{messageID}      ? $input{messageID}       : $self->{messageID};
	 my $messageType   = $input{messageType}    ? $input{messageType}     : $self->{messageType};

	 if ( ! $self->is_enabled(-messageType=>$messageType) ) {
		  $self->clear_errors();
		  return CLOG_SUCCESS;
	 }

#	 print "messageID=$messageID\n";
#	 print "messageMapDir=$messageMapDir\n";
#	 print "messageSet=$messageSet\n";
#	 print "messageCat=$messageCat\n";


	 # be sure $CLOGGER is runnable
	 if ( ! -x $CLOGGER ) {
		  $self->{error_code}    = BAD_OR_MISSING_CLOGGER;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsg_Missing_Clogger');
		  return BAD_OR_MISSING_CLOGGER;
	 }

	 %input = @_;

	 #formulate the command string
	 my @CommandArgs;
	 

	 # messageMapDir
	 if ( $messageMapDir eq "" ) {
		  $self->{error_code}    = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " messageMapDir");
		  return MISSING_PARAM;
	 }

	 # messageSet
	 if ( $messageSet eq "" ) {
		  $self->{error_code}    = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " messageSet");
		  return MISSING_PARAM;
	 }

	 # messageCat
	 if ( $messageCat eq "" ) {
		  $self->{error_code}    = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " messageCat");
		  return MISSING_PARAM;
	 }
	 # messageID
	 if ( $messageID eq "" ) {
		  $self->{error_code}   = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " messageID");
		  return MISSING_PARAM;
	 }

	 push (@CommandArgs," -I0");
	 push (@CommandArgs," -S0");
	 push (@CommandArgs," -CIBM.CLOG.cat");

	 my $field;
	 my @fields;
	 foreach $field ( @{$input{fields}} ) {
		  push (@fields,$self->mash($field));
	 }

  	 # verify that the message can be fetched.
	 my ($msgnum, $msgset, $defaultmsg) = MessageUtils->getmsgmap($messageMapDir,
																				  $messageSet,
																				  $messageCat,
																				  $messageID);
	 
#	 print "getmsgmap() returned =  ($msgnum, $msgset, [$defaultmsg] )\n";
	 
	 if ( $msgnum == -1 ) {
		  $self->{error_code}    = MAP_FILE_ERROR;
		  $self->{error_message} = $defaultmsg;
		  return MAP_FILE_ERROR;
	 }

  	 # Now fetch the message from the users catalog with any substitution already done.
	 my $FullMessage = MessageUtils->getMessageFromCat($messageCat, $messageMapDir, $messageSet, $messageID, @fields );

	 push (@CommandArgs," -M".$self->mash($FullMessage));

	 # compID
	 if ( $compID ) {
		  push(@CommandArgs," -c".$self->mash($compID));
	 }
	 else {
		  $self->{error_code}    = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " compID");
		  return MISSING_PARAM;
	 }
	 # lppID
	 if ( $lppID ) {
		  push(@CommandArgs," -l".$self->mash($lppID));
	 }
	 else {
		  $self->{error_code}    = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " lppID");
		  return MISSING_PARAM;
	 }


	 # messageType
	 if ( $messageType ) {
		  push(@CommandArgs," -t$messageType");
	 }
	 else {
		  $self->{error_code}   = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " messageType");
		  return MISSING_PARAM;
	 }

	 if ( $messageType == CLOG_ERROR_MSG || $messageType == CLOG_WARNING_MSG ) 
	 {
		  # detailedData
		  if ( $detailedData ) {
				push(@CommandArgs," -D".$self->mash($detailedData));
		  }
		  else {
				$self->{error_code}    = MISSING_PARAM;
				$self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																						'EMsgMissingParamError', " detailedData");
				return MISSING_PARAM;
		  }

		  # detailedDataSize
#  		  if ( $detailedDataSize ) {
#  				push(@CommandArgs," -d".$self->mash($detailedDataSize);
#  		  }
#  		  else {
#  				$self->{error_code}    = MISSING_PARAM;
#  				$self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
#  																						'EMsgMissingParamError', " detailedDataSize");
#  				return MISSING_PARAM;
#  		  }

		  # errorTemplate
		  if ( $errorTemplate ) {
				push(@CommandArgs," -A".$self->mash($errorTemplate));
		  }
		  else {
				$self->{error_code}    = MISSING_PARAM;
				$self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																						'EMsgMissingParamError', " errorTemplate");
				return MISSING_PARAM;
		  }
	 }

	 # fileName
	 if ( $fileName ) {
		  push(@CommandArgs," -f".$self->mash($fileName));
	 }
	 else {
		  $self->{error_code}   = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " fileName");
		  return MISSING_PARAM;
	 }
	 # lineNumber
	 if ( $lineNumber ) {
		  push(@CommandArgs," -n$lineNumber");
	 }
	 else {
		  $self->{error_code}   = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " lineNumber");
		  return MISSING_PARAM;
	 }
	 # sccsID
	 if ( $sccsID ) {
		  push(@CommandArgs," -s".$self->mash($sccsID));
	 }
	 else {
		  $self->{error_code}   = MISSING_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgMissingParamError', " sccsID");
		  return MISSING_PARAM;
	 }

	 my $command = "LC_ALL=C $CLOGGER @CommandArgs";
#	 print "Command = [$command]\n";

	 # run the command, capture output
	 my $output = `$command`;
	 chomp($output);
	 my $ret = ($? >> 8);

#	 print "$CLOGGER return code was [$ret]\n";
#	 print "$CLOGGER output was [$output]\n";

	 $self->clear_errors();

	 $self->{error_code}  = $ret;

	 if ( $self->{error_code} != 0 ) {
		  $self->{error_message} = $output;
	 }

    return $self->{error_code};
}


=item B<enable_msgs()>

Enables logging of the passed message type.

=item PARAMETERS
	 
 -messageType : the message type to disnable.

=item RETURNS 

 CLOG::CLOG_SUCCESS on success
 An error code on error - see get_error()	 	 
	 
=cut
################################################################################
# ;enable_msgs
################################################################################
sub enable_msgs
{
	 my $self = shift;
	 my %input;

	 if ( $self->validate_args(@_, _match=>"messageType", _required=>"messageType" ) != 0 ) 
	 {
		  return $self->{error_code};
	 }
	 
	 %input = @_;
	 
	 my $message_type = $input{"messageType"} ? $input{"messageType"} : $input{"-messageType"};

	 if( $message_type != CLOG_ALL_MSGS &&
		  $message_type != CLOG_ERROR_MSG &&
		  $message_type != CLOG_WARNING_MSG &&
		  $message_type != CLOG_INFO_MSG )
	 {
		  $self->{error_code}   = BAD_PARAM;
		  $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				  'EMsgInvalidParamError', "messageType");
		  return BAD_PARAM;
	 }

	 if ( $message_type == CLOG_ALL_MSGS ) {
		  $self->{message_mask} = (CLOG_LAST_MESSAGE_TYPE_VAL*2)-1;
	 }
	 else {
		  $self->{message_mask} |= $message_type;
	 }

	$self->clear_errors();
	return CLOG_SUCCESS;
}


=item B<disable_msgs()>

Disables the logging of the passed message type.

=item PARAMETERS
	 
 -messageType : the message type to enable.

=item RETURNS 

 CLOG_SUCCESS on success
 An error code on error - see get_error()	 
	 
=cut

################################################################################
# ;disable_msgs
################################################################################
sub disable_msgs
{
 	my $self = shift;
	my %input;

	if ( $self->validate_args(@_, _match=>"messageType", _required=>"messageType" ) != 0 ) 
	{
		 return $self->{error_code};
	}
	
	%input = @_;
	
	my $message_type = $input{"messageType"} ? $input{"messageType"} : $input{"-messageType"};
	
	if( $message_type != CLOG_ALL_MSGS &&
		 $message_type != CLOG_ERROR_MSG &&
		 $message_type != CLOG_WARNING_MSG &&
		 $message_type != CLOG_INFO_MSG )
	{
		 $self->{error_code}   = BAD_PARAM;
		 $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				 'EMsgInvalidParamError', "messageType");
		 return BAD_PARAM;
	}
	
	if ( $message_type == CLOG_ALL_MSGS ) {
		 $self->{message_mask} = 0;
	}
	else {
		 $self->{message_mask} &= ~$message_type;
	}

	$self->clear_errors();
	return CLOG_SUCCESS;

}

=item B<is_enabled()>

Checks if the passed message type is enabled.

=item PARAMETERS
	 
 -messageType : the message type to check.

=item RETURNS 

 CLOG_TRUE  : message type is enabled
 CLOG_FALSE : message type is not enabled
 An error code on error - see get_error()
	 
=cut
################################################################################
# ;is_enabled
################################################################################
sub is_enabled
{
	my $self = shift;
	my %input;

	if ( $self->validate_args(@_, _match=>"messageType", _required=>"messageType" ) != 0 ) 
	{
		 return $self->{error_code};
	}
	
	%input = @_;
	
	my $message_type = $input{"messageType"} ? $input{"messageType"} : $input{"-messageType"};
	
	if( $message_type != CLOG_ERROR_MSG &&
		 $message_type != CLOG_WARNING_MSG &&
		 $message_type != CLOG_INFO_MSG )
	{
		 $self->{error_code}   = BAD_PARAM;
		 $self->{error_message} = MessageUtils->getMessageFromCat(@CATALOG,
																				 'EMsgInvalidParamError', "messageType");
		 return BAD_PARAM;
	}

	$self->clear_errors();
	if ( $self->{message_mask} & $message_type ) {
		 return CLOG_TRUE;
	}
	else {
		 return CLOG_FALSE;
	}
}


=item B<set_lpp_id()>

Sets the LPP ID.

=item PARAMETERS
	 
 -lppID : the LPP ID to set.

=item RETURNS 

 CLOG_SUCCESS  : success
 An error code on error - see get_error()
	 
=cut

################################################################################
# ;set_lpp_id
################################################################################
sub set_lpp_id
{
	 my $self = shift;
	 my %input;

	 if ( $self->validate_args(@_, _match=>"lppID", _required=>"lppID") != 0 ) 
	 {
		  return $self->{error_code};
	 }
	 
	 %input = @_;
	 
	 $self->{lppID} = $input{"lppID"} ? $input{"lppID"} : $input{"-lppID"};

	 $self->clear_errors();
	 return CLOG_SUCCESS;
}

=item B<set_message_catalog()>

Sets the message catalog.

=item PARAMETERS
	 
 -messageCatalog : the message catalog name.

=item RETURNS 

 CLOG_SUCCESS  : success
 An error code on error - see get_error()
	 
=cut

################################################################################
# ;set_message_catalog
################################################################################
sub set_message_catalog
{
	my $self = shift;
	my %input;

	if ( $self->validate_args(@_, _match=>"messageCatalog" ) != 0 ) 
	{
		 return $self->{error_code};
	}
	 
	%input = @_;
	
	$self->{messageCatalog} = $input{"messageCatalog"} ? $input{"messageCatalog"} : $input{"-messageCatalog"};

	$self->clear_errors();
	return CLOG_SUCCESS;
}

=item B<set_message_set()>

Sets the message set.

=item PARAMETERS
	 
 -messageSet : the message set numonic (requires message map file) see new()

=item RETURNS 

 CLOG_SUCCESS  : success
 An error code on error - see get_error()
	 
=cut
################################################################################
# ;set_message_set
################################################################################
sub set_message_set
{
	 my $self = shift;
	 my %input;

	 if ( $self->validate_args(@_, _match=>"messageSet",_required=>"messageSet" ) != 0 ) 
	 {
		  return $self->{error_code};
	 }
	 
	 %input = @_;
	 
	 $self->{messageSet} = $input{"messageSet"} ? $input{"messageSet"} : $input{"-messageSet"};
	 
	 $self->clear_errors();
	 return CLOG_SUCCESS;
}

=item B<set_catalog_and_set()>

Sets both the message catalog and message set.

=item PARAMETERS

 -messageCatalog : the message catalog name
 -messageSet     : the message set numonic (requires message map file) see new()

=item RETURNS 

 CLOG_SUCCESS  : success
 An error code on error - see get_error()
	 
=cut
################################################################################
# ;set_catalog_and_set
################################################################################
sub set_catalog_and_set
{
	 my $self = shift;
	 my %input;
	 
	 if ( $self->validate_args(@_, _match=>"messageSet|messageCatalog" ) != 0 ) 
	 {
		  return $self->{error_code};
	 }
	 
	 %input = @_;
	 
	 my $messageCat = $input{"messageCatalog"} ? $input{"messageCatalog"} : $input{"-messageCatalog"};
	 my $ret = $self->set_message_catalog(messageCatalog=>$messageCat);
	 if ( $ret != CLOG_SUCCESS ) {
		  return $ret;
	 }

	 my $messageSet = $input{"messageSet"} ? $input{"messageSet"} : $input{"-messageSet"};
	 $ret = $self->set_message_set(messageSet=>$messageSet);
	 if ( $ret != CLOG_SUCCESS ) {
		  return $ret;
	 }

	 $self->clear_errors();
	 return CLOG_SUCCESS;
}

=item B<set_component_id()>

Sets the component ID.

=item PARAMETERS

 -compID : the component name.

=item RETURNS 

 CLOG_SUCCESS  : success
 An error code on error - see get_error()
	 
=cut
################################################################################
# ;set_component_id
################################################################################
sub set_component_id
{
	 my $self = shift;
	 my %input;
	 
	 if ( $self->validate_args(@_,_match=>"compID",_required=>"compID" ) != 0 )
	 {
		  return $self->{error_code};
	 }
	 
	 %input = @_;
	 
	 $self->{compID} = $input{"compID"} ? $input{"compID"} : $input{"-compID"};
	 
	 $self->clear_errors();
	 return CLOG_SUCCESS;
}

=item B<set_eror_log_info()>

Sets the field required to log messages via FFDC, specifically, AIX error log 
template information.

=item PARAMETERS

 -errorTemplate    :
 -detailedData     :
 -detailedDataSize :

=item RETURNS 

 CLOG_SUCCESS  : success
 An error code on error - see get_error()

 
=cut
################################################################################
# ;set_error_log_info
################################################################################
sub set_error_log_info
{
	 my $self = shift;
	 my %input;

	 if ( $self->validate_args(@_, 
										_match=>"errorTemplate|detailedData|detailedDataSize",
										_required=>"errorTemplate|detailedData|detailedDataSize") != 0 ) 
	 {
		  return $self->{error_code};
	 }

	 %input = @_;

	 $self->{errorTemplate}    = $input{"errorTemplate"}    ? $input{"errorTemplate"}    : $input{"-errorTemplate"};
	 $self->{detailedData}     = $input{"detailedData"}     ? $input{"detailedData"}     : $input{"-detailedData"};
	 $self->{detailedDataSize} = $input{"detailedDataSize"} ? $input{"detailedDataSize"} : $input{"-detailedDataSize"};

	 $self->clear_errors();
	 return CLOG_SUCCESS;
}

################################################################################

################################################################################
sub DESTROY{
	my $self = shift;
#	print "DESTROY called\n";
#	if($self =~ "CLOG=HASH"){
#		 print "FOOEY\n";
#		 c_destroy($self->{clogptr});
#	}
}

################################################################################
# _dump 
################################################################################
sub _dump 
{
	 my $self = shift;
	 my $key;

	 print "CLOG environment dump:\n";
	 printf ("%-20s\t%s\n","Name","Value");
	 print "----------------------------------------\n";
	 foreach $key ( sort keys %{$self} ) {
		  printf ("%-20s\t%s\n",$key,$self->{$key});
	 }
	 print "----------------------------------------\n";
}


1;
__END__

=back 

=head1

AUTHOR

Anthony Pioli (cluster@us.ibm.com)

=cut
