#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2005,2007 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
# @(#)52   1.92.1.4   src/csm/core/pm/DCUtils.pm.perl, setup, csm_rfish, rfishs001b 5/8/07 04:53:42
#--------------------------------------------------------------------------------

package DCUtils;
use strict;

use locale;
use Socket;

use Fcntl qw(:flock);
use File::Glob ':glob';
use File::Basename;
use File::Find;
use File::Copy;
use File::Path;    # Provides mkpath() and rmtree()

umask(0022);

my $csmroot;
my ($MSGCAT, $MSGMAPPATH, $MSGSET);

BEGIN
{
    $csmroot = $ENV{'CSM_ROOT'} ? $ENV{'CSM_ROOT'} : '/opt/csm';
    $::csmpm = $ENV{'CSM_PM'}   ? $ENV{'CSM_PM'}   : '/opt/csm/pm';
    $MSGCAT  = 'csmInstall.cat';
    if (defined $::MSGMAPPATH)
    {
        $MSGMAPPATH = $ENV{'CSM_ROOT'} ? "$csmroot/msgmaps" : $::MSGMAPPATH;
    }
    else
    {
        $MSGMAPPATH = "$csmroot/msgmaps";
    }
    $MSGSET = 'csminstall';
}

use lib $::csmpm;
use Cwd;
require MessageUtils;
require NetworkUtils;
require NodeUtils;
require NodesetUtils;
require CSMDefs;

# The DC group mapping table.
my %group_map_table = (
    AllNodes         => "Hostname like '%'",
    EmptyGroup       => "Hostname==''",
    AIXNodes         => "InstallOSName=='AIX'",
    LinuxNodes       => "InstallOSName=='Linux'",
    KickstartNodes   => "InstallMethod=='kickstart'",
    AutoyastNodes    => "InstallMethod=='autoyast'",
    NIMNodes         => "InstallMethod=='nim'",
    xSeriesNodes     => "InstallPkgArchitecture like 'i%86'",
    pSeriesNodes     => "InstallPkgArchitecture=='ppc64'",
    SystemxNodes     => "InstallPkgArchitecture like 'i%86'",
    SystempNodes     => "InstallPkgArchitecture=='ppc64'",
    RedHatELAS3Nodes =>
      "InstallDistributionName=='RedHatEL-AS' && InstallDistributionVersion=='3'",
    RedHatELES3Nodes =>
      "InstallDistributionName=='RedHatEL-ES' && InstallDistributionVersion=='3'",
    RedHatELWS3Nodes =>
      "InstallDistributionName=='RedHatEL-WS' && InstallDistributionVersion=='3'",
    RedHatELAS4Nodes =>
      "InstallDistributionName=='RedHatEL-AS' && InstallDistributionVersion=='4'",
    RedHatELES4Nodes =>
      "InstallDistributionName=='RedHatEL-ES' && InstallDistributionVersion=='4'",
    RedHatELWS4Nodes =>
      "InstallDistributionName=='RedHatEL-WS' && InstallDistributionVersion=='4'",
    RedHatELClient5Nodes =>
      "InstallDistributionName=='RedHatEL-Client' && InstallDistributionVersion=='5'",
    RedHatELServer5Nodes =>
      "InstallDistributionName=='RedHatEL-Server' && InstallDistributionVersion=='5'",
    SLES81Nodes =>
      "InstallDistributionName=='SLES' && InstallDistributionVersion=='8.1'",
    SLES9Nodes =>
      "InstallDistributionName=='SLES' && InstallDistributionVersion=='9'",
    ppcSLES81Nodes =>
      "InstallDistributionName=='SLES' && InstallDistributionVersion=='8.1' && InstallPkgArchitecture=='ppc64'",
    ppcSLES9Nodes =>
      "InstallDistributionName=='SLES' && InstallDistributionVersion=='9' && InstallPkgArchitecture=='ppc64'",
    ppcRedHatELAS3Nodes =>
      "InstallDistributionName=='RedHatEL-AS' && InstallDistributionVersion=='3' && InstallPkgArchitecture=='ppc64'",
    ppcRedHatELAS4Nodes =>
      "InstallDistributionName=='RedHatEL-AS' && InstallDistributionVersion=='4' && InstallPkgArchitecture=='ppc64'",
    ppcRedHatELServer5Nodes =>
      "InstallDistributionName=='RedHatEL-Server' && InstallDistributionVersion=='5' && InstallPkgArchitecture=='ppc64'",
);
@::NODE_GROUP=keys %group_map_table;
#-----------------------------------------------------------------------

=head3 get_default_cluster_info

        Set default attributes for cluster info hash.

=cut

#-----------------------------------------------------------------------

sub get_default_cluster_info
{
    $::__CLUSTER_INFO_HASH{'logfile_DCcopycds'}        = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCsyncserver'}     = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCenableservices'} = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCsetupinstaller'} = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCinstallnode'}    = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCnodeset'}        = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCcleanupFailedInstall'} =
      "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCrmnode'}        = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCmonitorinstall'}        = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCrmOSInstallImage'}       = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'logfile_DCmakedhcp'}       = "/var/log/csm/DC.log";
    $::__CLUSTER_INFO_HASH{'NetworkInstallProtocol'} = "nfs";
    $::__CLUSTER_INFO_HASH{'NodeInstallTimeout'}     = "60";
    $::__CLUSTER_INFO_HASH{'TFTPPackage'}            = "";
    $::__CLUSTER_INFO_HASH{'SetupNetworkInstallProtocol'} = "1";
    $::__CLUSTER_INFO_HASH{'InstallMsgPortNum'}           = "3101";
    
    if(NodeUtils->isLinux)
	{
		$::__CLUSTER_INFO_HASH{'RemoteShell'}                 = "/usr/bin/ssh";
    	$::__CLUSTER_INFO_HASH{'RemoteCopyCmd'}                 = "/usr/bin/scp";
	}
	else
	{
		$::__CLUSTER_INFO_HASH{'RemoteShell'}                 = "/usr/bin/rsh";
    	$::__CLUSTER_INFO_HASH{'RemoteCopyCmd'}                 = "/usr/bin/rcp";
	}	
}

#---------------------------------------------------------------------------------------------

=head3  get_all_info
        
        Wrap up get_cluster_info and get_node_info. 
        All DC APIs can call it to load cluster info and node info files at one time. 

        Arguments:
        
        $cluster_info_file
        Pathname of the clusterinfo file. 
        Optional, using default values if no file given or some attributes are missing.
        
        @node_info_list
        Array containing pathnames of nodeinfo files.
        
        @is_info_list
        Array containing pathnames of nodeinfo for install servers.
        
        Returns:
        Reference to the global hash %::__CLUSTER_INFO_HASH.
        Reference to %::__TARGET_NODE_HASH.
        Reference to the hash merging %::__TARGET_IS_HASH and %::__TARGET_NS_HASH.
                        
=cut

#---------------------------------------------------------------------------------------------

sub get_all_info
{
    my ($pkg, $cluster_info_file, $node_info_list, $server_info_list,
        $skip_file_load_flag,$skip_server_info)
      = @_;

    #Wrap up get_cluster_info and get_node_info
    my $cluster_info_hash =
      DCUtils->get_cluster_info($cluster_info_file, $skip_file_load_flag);
    my ($node_info_hash, $merged_server_info_hash) =
      DCUtils->get_node_info($node_info_list, $server_info_list,
                             $skip_file_load_flag,$skip_server_info);
    return ($cluster_info_hash, $node_info_hash, $merged_server_info_hash);
}

#--------------------------------------------------------

=head3  get_cluster_info
        
        Read all global cluster data from cluster info file 
        and put all these attributes into %::__CLUSTER_INFO_HASH. 
        Return reference to global hash directly if $skip_file_load_flag is set. 

        Arguments:
        $cluster_info_file
        Pathname of the clusterinfo file. 
        Optional, using default values if no file given or some attributes are missing.
        
        Returns:
        Reference to the global hash %::__CLUSTER_INFO_HASH.
                        
=cut

#---------------------------------------------------------

sub get_cluster_info
{
    my ($pkg, $cluster_info_file, $skip_file_load_flag) = @_;

    if ($skip_file_load_flag)
    {
        return \%::__CLUSTER_INFO_HASH;
    }

    my %cluster_info_hash;
	if (defined($cluster_info_file) && -f ($cluster_info_file))
    {
        my ($attr, $value);
        open(INFO, "<$cluster_info_file")
          or
          MessageUtils->message('E2', 'EMsgCANT_READ_FILE', $cluster_info_file);
        while (<INFO>)
        {
            #Match attribute=value line
            #The value can not be null since it will overwrite the default value.
            if(/^\s*(\w+)\s*=\s*(\S+)\s*$/)
            {
                $cluster_info_hash{$1}=$2;
            }
        }
        close(INFO);
    }
    
    return DCUtils->get_cluster_info_from_hash(\%cluster_info_hash,0);
}

#--------------------------------------------------------------------------------------------------------

=head3  get_node_info
        
        Wrap up get_cluster_info and get_node_info. 
        All DC APIs can call it to load cluster info and node info files at one time. 
        Read all node attributes from node info file 
        and put them into global hash %::__TARGET_NODE_HASH,%::__TARGET_IS_HASH and %::__TARGET_NS_HASH. 
        If an optional node attribute is not set in the node info file, 
        this function will set a default value to this attribute. 
        This function will also generate @::__MS_NODELIST. 
        Return reference to global hash directly if $skip_file_load_flag is set.
        
        Arguments:
        @node_info_list
        Array containing pathnames of nodeinfo files.
        
        @is_info_list
        Array containing pathnames of nodeinfo for install servers.
        
        Returns:
        Reference to %::__TARGET_NODE_HASH.
        Reference to the hash merging %::__TARGET_IS_HASH and %::__TARGET_NS_HASH.
                        
=cut

#--------------------------------------------------------------------------------------------------------

sub get_node_info
{
    my ($pkg, $node_info_list, $server_info_list, $skip_file_load_flag,$skip_server_info) = @_;

    if ($skip_file_load_flag)
    {
        my %merged_hash = (%::__TARGET_IS_HASH, %::__TARGET_NS_HASH);
        return (\%::__TARGET_NODE_HASH, \%merged_hash);
    }

    my %target_node_hash;
    #The key in the global hash should be the "Hostname" attribute specified in the nodeinfo file.
    foreach my $node_info_file (@$node_info_list)
    {
        my %node_hash;
        my ($attr, $value);
        open(INFO, "<$node_info_file")
          or MessageUtils->message('E2', 'EMsgCANT_READ_FILE', $node_info_file);
        while (<INFO>)
        {
            #Match attribute=value line
            #The value can be null if the user do want to empty it.
            if(/^\s*(\w+)\s*=\s*(.*)\s*$/)
            {
                $node_hash{$1}=$2;
            }
        }
        close(INFO);

        #Use long resolved hostname of node as the key
        my $node_name = $node_hash{'Hostname'};

        %{$target_node_hash{$node_name}} = %node_hash;
    }

    my %all_server_hash;
    foreach my $server_info_file (@$server_info_list)
    {
        my %server_hash;
        my ($attr, $value);
        open(INFO, "<$server_info_file")
          or
          MessageUtils->message('E2', 'EMsgCANT_READ_FILE', $server_info_file);
        while (<INFO>)
        {
            #Match attribute=value line
            #The value can be null if the user do want to empty it.
            if(/^\s*(\w+)\s*=\s*(.*)\s*$/)
            {
                $server_hash{$1}=$2;
            }
        }
        close(INFO);
        my $server_name = $server_hash{'Hostname'};

        #Use install server's resolved long hostname as the key
        %{$all_server_hash{$server_name}} = %server_hash;
    }

    return DCUtils->get_node_info_from_hash(\%target_node_hash,\%all_server_hash, $skip_file_load_flag,$skip_server_info);

}
#---------------------------------------------------------------------------------------------

=head3  get_all_info_from_hash
        
        Wrap up get_cluster_info_from_hash and get_node_info_from_hash. 
        All DC APIs can call it to load cluster info and node info hashes at one time. 

        Arguments:
        
        $cluster_info_hash
        $node_info_hash
        $is_info_hash
        
        Returns:
        Reference to the global hash %::__CLUSTER_INFO_HASH.
        Reference to %::__TARGET_NODE_HASH.
        Reference to the hash merging %::__TARGET_IS_HASH and %::__TARGET_NS_HASH.
                        
=cut

#---------------------------------------------------------------------------------------------

sub get_all_info_from_hash
{
    my ($pkg, $orig_cluster_info_hash, $orig_node_info_hash, $orig_server_info_hash,
        $skip_file_load_flag,$skip_server_info)
      = @_;

	if(!$skip_file_load_flag)
	{
		#Empty all the global hash first
		%::__CLUSTER_INFO_HASH=();
		%::__TARGET_NODE_HASH=();
		%::__TARGET_IS_HASH=();
		%::__TARGET_NS_HASH=();
		@::__MS_NODELIST=();
	}	
      
    #Wrap up get_cluster_info_from_hash and get_node_info_from_hash
    my $cluster_info_hash =
      DCUtils->get_cluster_info_from_hash($orig_cluster_info_hash, $skip_file_load_flag);
    my ($node_info_hash, $merged_server_info_hash) =
      DCUtils->get_node_info_from_hash($orig_node_info_hash, $orig_server_info_hash,
                             $skip_file_load_flag,$skip_server_info);

    return ($cluster_info_hash, $node_info_hash, $merged_server_info_hash);
}

#--------------------------------------------------------

=head3  get_cluster_info_from_hash
        
        Read all global cluster data from cluster info hash 
        and put all these attributes into %::__CLUSTER_INFO_HASH. 

        Arguments:
        $cluster_info_hash
        
        Returns:
        Reference to the global hash %::__CLUSTER_INFO_HASH.
                        
=cut

#---------------------------------------------------------

sub get_cluster_info_from_hash
{
    my ($pkg, $cluster_info_hash, $skip_file_load_flag) = @_;

    if ($skip_file_load_flag)
    {
        return \%::__CLUSTER_INFO_HASH;
    }

    #Get default attributes first,it will be overwritten by $cluster_info_hash.
    &get_default_cluster_info;
    foreach my $attr (keys %$cluster_info_hash)
    {
	    
    	$::__CLUSTER_INFO_HASH{$attr}=$$cluster_info_hash{$attr};
    }
    check_cluster_info(%::__CLUSTER_INFO_HASH);
    return \%::__CLUSTER_INFO_HASH;
}

#--------------------------------------------------------------------------------------------------------

=head3  get_node_info_from_hash
        
        Wrap up get_cluster_info_from_hash and get_node_info_from_hash. 
        All DC APIs can call it to load cluster info and node info hashes at one time. 
        Read all node attributes from node info hash 
        and put them into global hash %::__TARGET_NODE_HASH,%::__TARGET_IS_HASH and %::__TARGET_NS_HASH. 
        
        Arguments:
       	$node_info_hash
	$is_info_hash

        Returns:
        Reference to %::__TARGET_NODE_HASH.
        Reference to the hash merging %::__TARGET_IS_HASH and %::__TARGET_NS_HASH.
                        
=cut

#--------------------------------------------------------------------------------------------------------

sub get_node_info_from_hash
{
    my ($pkg, $node_info_hash, $server_info_hash, $skip_file_load_flag,$skip_server_info) = @_;

    if( !defined($node_info_hash) || scalar(%$node_info_hash) == 0 ){
        MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E2', 'EMsgNO_NODES');
    }
    
    if ($skip_file_load_flag)
    {
        my %merged_hash = (%::__TARGET_IS_HASH, %::__TARGET_NS_HASH);
        return (\%::__TARGET_NODE_HASH, \%merged_hash);
    }

    #The key in the global hash should be the "Hostname" attribute specified in the nodeinfo file.
    foreach my $node (keys %$node_info_hash)
    {

	if(!defined($$node_info_hash{$node}{"Hostname"}) || $$node_info_hash{$node}{"Hostname"} eq "" )
	{	
        	MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E2', 'EMsgInvalidNodeInfo',"Hostname",$node);
        }
	if(!defined($$node_info_hash{$node}{"ManagementServer"}) || $$node_info_hash{$node}{"ManagementServer"} eq "" )
	{	
        	MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E2', 'EMsgInvalidNodeInfo',"ManagementServer",$node);
        }
	#Use long resolved hostname of node as the key
        my ($node_hostname, $node_ip) = NetworkUtils->getHost($$node_info_hash{$node}{"Hostname"});

        %{$::__TARGET_NODE_HASH{$node_hostname}} = %{$$node_info_hash{$node}};
        $::__TARGET_NODE_HASH{$node_hostname}{'Hostname'} = $node_hostname;
    }
	
	my %all_server_hash;
    foreach my $server (keys %$server_info_hash)
    {
	if(!defined($$server_info_hash{$server}{"Hostname"}) || $$server_info_hash{$server}{"Hostname"} eq "" )
	{	
        	MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E2', 'EMsgInvalidNodeInfo',"Hostname",$server);
        }
	if(!defined($$server_info_hash{$server}{"ManagementServer"}) || $$server_info_hash{$server}{"ManagementServer"} eq "" )
	{	
        	MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E2', 'EMsgInvalidNodeInfo',"ManagementServer",$server);
        }
        #Use install server's resolved long hostname as the key
        my ($server_hostname, $server_ip) = NetworkUtils->getHost($$server_info_hash{$server}{"Hostname"});
        %{$all_server_hash{$server_hostname}} = %{$$server_info_hash{$server}};
        $all_server_hash{$server_hostname}{'Hostname'} = $server_hostname;
    }

    foreach my $node (keys %::__TARGET_NODE_HASH)
    {
        my ($ms_hostname, $ms_ip) =
          NetworkUtils->getHost(
                              $::__TARGET_NODE_HASH{$node}{"ManagementServer"});

        my $has_nfs_server = 0;
        my $server_attr    = $::__TARGET_NODE_HASH{$node}{"InstallServer"};

        if (!$server_attr)
        {
            push(@::__MS_NODELIST, $node);
            $server_attr = $::__TARGET_NODE_HASH{$node}{"NFSServer"};
            if (!$server_attr)
            {
                next;
            }
            $has_nfs_server = 1;
        }

        my @server_list = split ',', $server_attr;

        # Valid InstallServer which exclude MS
        my @valid_server = ();

        #CSM nodegroup will be translated to a server list in DC
        foreach my $server (@server_list)
        {
            my ($server_name, $server_dir) = split ':', $server;
            if (!$server_name)
            {
                next;
            }
            my ($server_hostname, $server_ip);
            ($server_hostname, $server_ip) =
              NetworkUtils->getHost($server_name);

            #If install server specified by user is actually management server or node itself
            #treat it the same as node without install server
            if ($server_ip eq $ms_ip || $server_hostname eq $node)
            {
                push(@::__MS_NODELIST, $node);
                next;
            }

            if ($server_dir)
            {
                if (substr($server_dir, -1) eq "/") { chop($server_dir); }
            }
            else
            {
                $server_dir = $::IS_DEFAULT_DIR;
            }

            push @valid_server, join(":", $server_hostname, $server_dir);

            my $server_hash_ref;
            if ($has_nfs_server)
            {
                $server_hash_ref = \%::__TARGET_NS_HASH;
            }
            else
            {
                $server_hash_ref = \%::__TARGET_IS_HASH;
            }

            #Initialize server hash
            if (!defined($$server_hash_ref{$server_hostname}))
            {
                if (!defined($all_server_hash{$server_hostname}) )
                {
                    if(!$skip_server_info)
                    {
                        MessageUtils->message('E2', 'EMsgNO_IS_INFO',
                                          $server_hostname);
                    }
                    #die "The node info file of install server or nfs server $server_hostname is not provided";
                }
                else
                {
                    %{$$server_hash_ref{$server_hostname}} =
                      %{$all_server_hash{$server_hostname}};
                }
            }
            push(
                 @{$$server_hash_ref{$server_hostname}{"NodesToBeInstalled"}},
                 $node
                );
            if (defined($$server_hash_ref{$server_hostname}{"InstallDir"})
                && $$server_hash_ref{$server_hostname}{"InstallDir"} ne
                $server_dir)
            {
                MessageUtils->message('E2', 'EMsgBAD_IS_DIR', $server_hostname);
            }
            else
            {
                $$server_hash_ref{$server_hostname}{"InstallDir"} = $server_dir;
                #To be compatible with old node hash data structure
                $$server_hash_ref{$server_hostname}{"Dir"} = $server_dir;
            }
        }    #foreach @server_list

        if (@valid_server && scalar(@valid_server))
        {
            if ($has_nfs_server)
            {
                $::__TARGET_NODE_HASH{$node}{"NFSServer"} = join(",", @valid_server);
            } else {
                $::__TARGET_NODE_HASH{$node}{"InstallServer"} = join(",", @valid_server);
            }
        } else {
            # Clean the NodeHash to exclude MS
            if ($has_nfs_server)
            {
                $::__TARGET_NODE_HASH{$node}{"NFSServer"} = "";
            } else {
                $::__TARGET_NODE_HASH{$node}{"InstallServer"} = "";
            }
        }
    }    #foreach keys

    #If the target node is specified as install server of other node,remove it from target node hash
    #It also should be removed from @::__MS_NODELIST if it's served by management server,
    #or it should be removed from "NodeToBeinstalled" list in it's target server hash.
    my @remove_nodes;
    if(!$skip_server_info)
    {
        foreach my $node (keys %::__TARGET_NODE_HASH)
        {
            if (   grep(/^$node$/, keys %::__TARGET_IS_HASH)
                || grep(/^$node$/, keys %::__TARGET_NS_HASH))
            {
                push(@remove_nodes, $node);
            }
        }
    }

    return DCUtils->remove_node_from_hash(\@remove_nodes);

}
#--------------------------------------------------------------------------------------------------------

=head3  remove_node_from_hash 
        Remove nodes in %::__TARGET_NODE_HASH. 
        This function will also update this node's install servers NodesToBeInstalled attribute 
        in %::__TARGET_IS_HASH. If there isn't node that will be installed from this install server, 
        remove this install server from %::__TARGET_IS_HASH. 
        If this node doesn't use Install Server, this function will delete this node from @::__MS_NODELIST.         
        Arguments:
        @target_node
                Array of node names which will be removed from %::__TARGET_NODE_HASH.
        
        Returns:
        Reference to %::__TARGET_NODE_HASH.
        Reference to the hash merging %::__TARGET_IS_HASH and %::__TARGET_NS_HASH.
                        
=cut

#--------------------------------------------------------------------------------------------------------

sub remove_node_from_hash
{
    my ($pkg, $remove_nodes) = @_;
    my @left_nodes;

    foreach my $node (@$remove_nodes)
    {
        delete $::__TARGET_NODE_HASH{$node};
        if (!grep(/^$node$/, @::__MS_NODELIST))
        {
            push(@left_nodes, $node);
        }
    }

    #Search nodes served by mangement server
    if(scalar(@$remove_nodes))
    {
        my @ms_node_list;
        foreach my $ms_node (@::__MS_NODELIST)
        {
            if (!grep(/^$ms_node$/, @$remove_nodes))
            {
                push(@ms_node_list, $ms_node);
            }
        }
        @::__MS_NODELIST = @ms_node_list;

        #Search nodes served by install or nfs server
        @left_nodes =
            &remove_node_from_server_hash(\@left_nodes, \%::__TARGET_IS_HASH);
        @left_nodes =
            &remove_node_from_server_hash(\@left_nodes, \%::__TARGET_NS_HASH);
    }
    
    my %merged_hash = (%::__TARGET_IS_HASH, %::__TARGET_NS_HASH);
    return (\%::__TARGET_NODE_HASH, \%merged_hash);
}

#--------------------------------------------------------------------------------

=head3  remove_node_from_server_hash
        Remove node from NodesToBeInstalled list of target server hash

        Arguments:
                Reference to array contains nodes to be removed.
        Returns:
                Array contains nodes can not be removed.
=cut

#--------------------------------------------------------------------------------
sub remove_node_from_server_hash
{
    my ($remove_nodes, $server_hash) = @_;
    my @left_nodes;
    foreach my $node (@$remove_nodes)
    {
        my $found = 0;
        foreach my $server (keys %$server_hash)
        {
            my @node_list;
            foreach
              my $inst_node (@{$$server_hash{$server}{"NodesToBeInstalled"}})
            {
                if ($inst_node eq $node)
                {
                    $found = 1;
                    next;
                }
                else
                {
                    push(@node_list, $inst_node);
                }
            }
            if (@node_list)
            {
                $$server_hash{$server}{"NodesToBeInstalled"} = @node_list;
            }
            else
            {
                delete $$server_hash{$server};
            }
        }
        if (!$found)
        {
            push(@left_nodes, $node);
        }
    }
    return @left_nodes;
}

#-----------------------------------------------------------------------------------------------------

=head3  remove_server_from_hash 
        Remove a install server or nfs server in %::__TARGET_IS_HASH or %::__TARGET_NS_HASH. 
        Update the $::__TARGET_NODE_HASH accordingly.
        
        Arguments:
        @target_server
            Array of install servers(nfs servers) which should be removed from %::TARGET_IS_HASH
            or %::__TARGET_NS_HASH.
        
        Returns:
        Reference to %::__TARGET_NODE_HASH.
        Reference to the hash merging %::__TARGET_IS_HASH and %::__TARGET_NS_HASH.
                        
=cut

#-----------------------------------------------------------------------------------------------------
sub remove_server_from_hash
{
    my ($pkg, $remove_servers) = @_;
    foreach my $server (@$remove_servers)
    {
        my $server_hash_ref;
        if (defined($::__TARGET_IS_HASH{$server}))
        {
            $server_hash_ref = \%::__TARGET_IS_HASH;
        }
        else
        {
            $server_hash_ref = \%::TARGET_NS_HASH;
        }
        foreach my $node (@{$$server_hash_ref{$server}{"NodesToBeInstalled"}})
        {
            delete $::__TARGET_NODE_HASH{$node};
        }
        delete $$server_hash_ref{$server};
    }
}

#------------------------------------------------------------------

=head3  get_install_servers
        Get Install Server hash

        Returns:
                Reference to global install server hash
=cut

#------------------------------------------------------------------

sub get_install_servers
{
    return \%::__TARGET_IS_HASH;
}

#----------------------------------------------------------------

=head3  get_nfs_servers
        Get NFS server hash

        Returns 
                Reference to global nfs server hash
=cut

#----------------------------------------------------------------

sub get_nfs_servers
{
    return \%::__TARGET_NS_HASH;
}

#--------------------------------------------------------------------------------------------------

=head3  get_nodes_with_no_is
        Get node list of nodes served by management server
        
        Returns:
                Reference to global node list which contains nodes served by management server.
=cut

#--------------------------------------------------------------------------------------------------

sub get_nodes_with_no_is
{
    return \@::__MS_NODELIST;
}

#--------------------------------------------------------------------------------------------------

=head3  set_config_info
        Write node information in node hash into config file

=cut

#--------------------------------------------------------------------------------------------------

sub set_config_info
{
    my $cwd = getcwd();

    my $csm_cfg_dir=$::CSMCFGDIR;
    if (!-d $csm_cfg_dir) { mkpath($csm_cfg_dir,$::VERBOSE,0755) }
    chdir("$csm_cfg_dir");

    my $instsvr;
    foreach my $node (keys %::__TARGET_NODE_HASH)
    {
        my $info = "$node" . ".config_info";
        open(INFO, ">$info")
          or MessageUtils->message('E2', 'EMsgCANT_WRITE_FILE', $info);

        # add the node configuration information to the file
        print INFO "HostnameIP="
              . NetworkUtils->validate_ip($::__TARGET_NODE_HASH{$node}{'Hostname'})
              . "\n";
        if (defined($::__TARGET_NODE_HASH{$node}{'ManagementServer'}))
        {
            print INFO
              "ManagementServer=$::__TARGET_NODE_HASH{$node}{'ManagementServer'}\n";
            my ($ms_host, $ms_ip) =
              NetworkUtils->getHost($::__TARGET_NODE_HASH{$node}{'ManagementServer'});
            print INFO "ManagementServerIP=$ms_ip\n";
            print INFO "ManagementServerHost=$ms_host\n";
        }
        if (defined($::__TARGET_NODE_HASH{$node}{'InstallServer'})
            && ($::__TARGET_NODE_HASH{$node}{'InstallServer'} ne ""))
        {
            my $isISGrp = 0;    #whether the install server is a node group
            my $server_attr=$::__TARGET_NODE_HASH{$node}{'InstallServer'};
            my @servers = split(',',$server_attr);
            my ($server, $rep_dir) = split(':',$servers[0]);
            my ($msvr_ip, $isvr_ip);

            if (scalar(@servers)>1)
            {
                print INFO "InstallServerIsGroup=yes\n";
                $isISGrp = 1;
                $instsvr = $::__TARGET_NODE_HASH{$node}{'ManagementServer'};
            }
            else
            {
                (undef, $isvr_ip) = NetworkUtils->getHost($server);
                (undef, $msvr_ip) =
                  NetworkUtils->getHost(
                                     $::__TARGET_NODE_HASH{$node}{'ManagementServer'});
                $instsvr = $server;
            }

            if ($isvr_ip && $msvr_ip)
            {

                #not using the ms as the install server
                print INFO
                  "InstallServer=$::__TARGET_NODE_HASH{$node}{'InstallServer'}\n";
            }
        }
        elsif (defined($::__TARGET_NODE_HASH{$node}{'ManagementServer'}))
        {
            my $ms = $::__TARGET_NODE_HASH{$node}{'ManagementServer'};
            my ($hostname, $ip) = NetworkUtils->getHost($ms);

            $instsvr = $::__TARGET_NODE_HASH{$node}{'ManagementServer'};
        }
        if (defined($::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'})
            && $::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'})
        {

            # resolve the ip address of InstallAdapterHostname
            if (
                !NetworkUtils->isIpaddr(     $::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'}
                )
               )
            {

                # call dsh to resolve InstallAdapterHostname on the install server machine.
                my ($adp_hostname, $adp_hostip) =
                  ServerUtils->dshGetNodeAddrInfo(
                                $::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'},
                                $instsvr);
                if ($adp_hostip != -1)
                {

                    # write the ip address to config_info file
                    print INFO "InstallAdapterHostname=$adp_hostname\n";
                    print INFO "InstallAdapterHostnameIP=$adp_hostip\n";
                    print INFO "InstallAdapterIP=$adp_hostip\n";
                }
                else
                {
                    print INFO
                      "InstallAdapterHostname=$::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'}\n";
                    print INFO
                      "InstallAdapterHostnameIP=$::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'}\n";
                    print INFO
                      "InstallAdapterIP=$::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'}\n";
                }
            }
            else
            {
                print INFO
                  "InstallAdapterHostname=$::__TARGET_NODE_HASH{$node}{'InstallAdapterHostname'}\n";
            }
        }
        else
        {
            print INFO "InstallAdapterHostname=\n";
            print INFO "InstallAdapterHostnameIP=\n";
        }
        if (defined($::__TARGET_NODE_HASH{$node}{'InstallServerAKBNode'})
            && $::__TARGET_NODE_HASH{$node}{'InstallServerAKBNode'})
        {

            # resolve the ip address of InstallServerAKBNode
            if (
                !NetworkUtils->isIpaddr(       $::__TARGET_NODE_HASH{$node}{'InstallServerAKBNode'}
                )
               )
            {

                # call dsh to resolve InstallServerAKBNode on the install server machine.
                my ($isakb_host, $isakb_nodeip) =
                  ServerUtils->dshGetNodeAddrInfo(
                                  $::__TARGET_NODE_HASH{$node}{'InstallServerAKBNode'},
                                  $instsvr);
                if ($isakb_nodeip != -1)
                {

                    # write the ip address to config_info file
                    print INFO "InstallServerAKBNode=$isakb_host\n";
                    print INFO "InstallServerAKBNodeIP=$isakb_nodeip\n";
                }
                else
                {
                    print INFO
                      "InstallServerAKBNode=$::__TARGET_NODE_HASH{$node}{'InstallServerAKBNode'}\n";
                    print INFO
                      "InstallServerAKBNodeIP=$::__TARGET_NODE_HASH{$node}{'InstallServerAKBNode'}\n";
                }
            }
            else
            {
                print INFO
                  "InstallServerAKBNode=$::__TARGET_NODE_HASH{$node}{'InstallServerAKBNode'}\n";
            }
        }
        else
        {
            print INFO "InstallServerAKBNode=\n";
            print INFO "InstallServerAKBNodeIP=\n";
        }
        foreach my $attr (keys %{$::__TARGET_NODE_HASH{$node}})
        {
                print INFO "$attr=$::__TARGET_NODE_HASH{$node}{$attr}\n";
        }

        # add lists of scripts to run on the node
        if (defined(@{$::PREINSTSCRIPTS{$node}}))
        {
            print INFO "PREINSTSCRIPTS=@{$::PREINSTSCRIPTS{$node}}\n";
        }
        if (defined(@{$::POSTINSTSCRIPTS{$node}}))
        {
            print INFO "POSTINSTSCRIPTS=@{$::POSTINSTSCRIPTS{$node}}\n";
        }
        if (defined(@{$::UPDATESCRIPTS{$node}}))
        {
            print INFO "UPDATESCRIPTS=@{$::UPDATESCRIPTS{$node}}\n";
        }
        if (defined(@{$::DISKLESSPREBUILDSCRIPTS{$node}}))
        {
            print INFO
              "DISKLESSPREBUILDSCRIPTS=@{$::DISKLESSPREBUILDSCRIPTS{$node}}\n";
        }
        if (defined(@{$::DISKLESSBOOTSCRIPTS{$node}}))
        {
            print INFO
              "DISKLESSBOOTSCRIPTS=@{$::DISKLESSBOOTSCRIPTS{$node}}\n";
        }
        # New added, but not sure whether need them all.
        # OS_DIST_TOP,  OS_DIST_RPMS, OPEN_SRC_RPM_DIR, DRVR_TOP_DIR, PATCH_FILE_DIR
        # PREREBOOT_SCRIPT_DIR , FIRSTBOOT_SCRIPT_DIR, UPDATE_SCRIPT_DIR, DATA_SCRIPT_DIR
        # DSERVER_DIR, STATUS_FILE_DIR, CSM_PACKAGES_DIR, SMS_INSTALL_DIR, SMS_UPDATES_DIR
        my $CSMCLIENTMNTDIR = "/var/opt/csm/mnt";
        my $os_dist_top     =
          "$CSMCLIENTMNTDIR/$::__TARGET_NODE_HASH{$node}{'InstallOSName'}/$::__TARGET_NODE_HASH{$node}{'InstallDistributionName'}/$::__TARGET_NODE_HASH{$node}{'InstallDistributionVersion'}/$::__TARGET_NODE_HASH{$node}{'InstallPkgArchitecture'}";
        my $csm_pkg_dir =
          "$CSMCLIENTMNTDIR/$::__TARGET_NODE_HASH{$node}{'InstallOSName'}/$::__TARGET_NODE_HASH{$node}{'InstallDistributionName'}/csm/$::__TARGET_NODE_HASH{$node}{'InstallCSMVersion'}/packages";

        print INFO "OS_DIST_TOP=$os_dist_top\n";

        print INFO "OS_DIST_RPMS=$os_dist_top/RPMS\n";
        print INFO "OPEN_SRC_RPM_DIR=$csm_pkg_dir\n";
        print INFO "DRVR_TOP_DIR=$os_dist_top/drivers\n";
        print INFO "PATCH_FILE_DIR=$os_dist_top/patches\n";

		print INFO
		  "PREREBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/installprereboot\n";
		print INFO
		  "FIRSTBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/installpostreboot\n";
		print INFO
		  "OSUPGRADE_PREREBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/osupgradeprereboot\n";
		print INFO
		  "OSUPGRADE_FIRSTBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/osupgradepostreboot\n";

        print INFO
          "UPDATE_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/update\n";
        print INFO "DATA_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/data\n";

        print INFO "OS_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm\n";    #????
        print INFO "STATUS_FILE_DIR=$CSMCLIENTMNTDIR/csm/status\n";

        print INFO "CSM_PACKAGES_DIR=$csm_pkg_dir\n";
        print INFO "SMS_INSTALL_DIR=$os_dist_top/install\n";
        print INFO "SMS_UPDATES_DIR=$os_dist_top/updates\n";

        close(INFO);
    }

    chdir("$cwd");
}

#------------------------------------------------------------------------

=head3  node_belong2_group

    Notes:
        Determine whether the node belongs to the specified group
    Arguments:
        node name;
        group description string;    
    Returns:
           If the node matches the description of the group, return 1,
        Otherwise return 0.

 Comments:

=cut

#-------------------------------------------------------------------------
sub node_belong2_group
{
    my ($node, $grp_desc) = @_;
    my @conditions = split("&&", $grp_desc);
    foreach my $cond (@conditions)
    {
        if ($cond =~ /==/)
        {
            my ($attr, $val) = $cond =~ /^\s*(\S+)\s*==\s*(\S+)\s*$/;
            $val =~ s/^\'*//;    # Remove any leading quote
            $val =~ s/\'*$//;    # Remove any trailing quote
            if ($::NODEHASH{$node}{$attr} ne $val)
            {
                return 0;
            }
        }
        elsif ($cond =~ /like/)
        {
            my ($attr, $val) = $cond =~ /^\s*(\S+)\s*like\s*(\S+)\s*$/;
            $val =~ s/^\'*//;       # Remove any leading quote
            $val =~ s/\'*$//;       # Remove any trailing quote
            $val =~ s/\%/\\S\+/;    # Replace % with \S+
            if ($::NODEHASH{$node}{$attr} !~ /$val/)
            {
                return 0;
            }
        }
        else
        {
            return 0;
        }
    }
    return 1;
}

#--------------------------------------------------------------------------------

=head3    listNodeGroupDC

    Return the expanded member list for each of the predefined node group names.

    Arguments:
        None.
    Returns:
        reference to an array:

        Each element of the array represents one (1) row and has the format:

                    nodegroupname:|:{node1,node2,node3}


    Globals:
        none
    Error:
        none
    Example:
         my $outref =
            NodeUtils->listNodeGroupDC();
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub listNodeGroupDC
{
    my %groups;
    my @group_list;

    # initialize %groups with all the keys
    foreach my $grp (keys %group_map_table)
    {
        @{$groups{$grp}} = ();
    }

    # go through each node to figure out every group it belongs to.
    foreach my $node (keys %::NODEHASH)
    {
        foreach my $grp (keys %group_map_table)
        {
            my $desc = $group_map_table{$grp};
            if (node_belong2_group($node, $desc))
            {
                push @{$groups{$grp}}, $node;
            }

        }
    }

    # store the result in the format that exactly matches RSCT
    foreach my $grp (keys %groups)
    {
        my $nl = join(',', @{$groups{$grp}});
        my $key = $grp . ":|:{" . $nl . "}";
        push @group_list, $key;
    }

    return \@group_list;
}

sub getRemoteShell
{
    return $::__CLUSTER_INFO_HASH{'RemoteShell'};
}
sub getRemoteCopyCmd
{
    return $::__CLUSTER_INFO_HASH{'RemoteCopyCmd'};
}
sub getSetupRemoteShell
{
    return $::__CLUSTER_INFO_HASH{'SetupRemoteShell'};
}

sub makeSyncHash
{
    my $routine = "makeSyncHash";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $DestNodeHash, $InstallServerHash, $special_ref) = @_;
    my $defDir = $::IS_DEFAULT_DIR;    #/csmserver

    my %sHash;                         # return hash
    my $csm_skip_isvr;
    $::MS_INSTALL_SERVER = 0;

    # Set value to skip installserver
    if (defined $ENV{'CSM_SKIP_ISVR'} && $ENV{'CSM_SKIP_ISVR'} != 0)
    {
        $csm_skip_isvr = 1;
    }

    my %server_list_hash;
    foreach my $server (keys %$InstallServerHash)
    {
        foreach my $node (@{$$InstallServerHash{$server}{"NodesToBeInstalled"}})
        {
            push(@{$server_list_hash{$node}{"InstallServerList"}}, $server);
            $$DestNodeHash{$node}{"InstallDir"} =
              $$InstallServerHash{$server}{"InstallDir"};
        }
    }
    foreach my $node (keys %$DestNodeHash)
    {
        ($::DEBUG) && print ">>>>>NODE IN DestNodeHash:  $node\n";
        my @installservers ;
        if(defined($server_list_hash{$node}{"InstallServerList"}))
        {
            @installservers   = @{$server_list_hash{$node}{"InstallServerList"}};
        }
        else
        {
            next;
        } 
        my $dir            = $$DestNodeHash{$node}{"InstallDir"};

        if (defined($special_ref))
        {
            foreach my $iserver (@installservers)
            {
                foreach my $specialFileORDir (@$special_ref)
                {
                    push @{$sHash{$specialFileORDir}{$dir}}, $iserver;
                }
            }
            next;
        }

        if ($csm_skip_isvr)
        {
            foreach my $iserver (@installservers)
            {
                push @{$sHash{"/csminstall/csm/config"}{$dir}},      $iserver;
                if (-e "/csminstall/csm/osprereboot")
                {
                    push @{$sHash{"/csminstall/csm/osprereboot"}{$dir}}, $iserver;
                }
                if ($$DestNodeHash{$node}{'InstallOSName'} eq 'Linux')
                {
                    #template files are not valid on AIX
                    push @{$sHash{"/csminstall/csm/templates/tmpl"}{$dir}},
                    $iserver;
                }
            }
        }
        else
        {

            #find node attributes
            my $InstallOSName = $$DestNodeHash{$node}{'InstallOSName'};
            my $InstallDistributionName =
              $$DestNodeHash{$node}{'InstallDistributionName'};
            my $InstallDistributionVersion =
              $$DestNodeHash{$node}{'InstallDistributionVersion'};
            my $InstallServiceLevel =
              $$DestNodeHash{$node}{'InstallServiceLevel'};
            my $InstallPkgArchitecture =
              $$DestNodeHash{$node}{'InstallPkgArchitecture'};
            my $InstallMethod = $$DestNodeHash{$node}{'InstallMethod'};

            foreach my $iserver (@installservers)
            {
                push @{$sHash{"/csminstall/csm"}{$dir}}, $iserver;
                if ($InstallOSName eq "Linux")
                {
                    my $iInstallOSName =
                      $$InstallServerHash{$iserver}{'InstallOSName'};
                    my $iInstallDistributionName =
                      $$InstallServerHash{$iserver}{'InstallDistributionName'};
                    my $iInstallDistributionVersion =
                      $$InstallServerHash{$iserver}
                      {'InstallDistributionVersion'};
                    my $iInstallPkgArchitecture =
                      $$InstallServerHash{$iserver}{'InstallPkgArchitecture'};
                    push
                      @{$sHash{"/csminstall/Linux/$InstallDistributionName/csm"}
                          {$dir}}, $iserver;
                    push @{
                        $sHash{
                            "/csminstall/Linux/$InstallDistributionName/$InstallDistributionVersion/$InstallPkgArchitecture"
                          }{$dir}
                      },
                      $iserver;

                    if($InstallMethod eq "warewulf")
		    {
		   	push @{
				$sHash{
				"$::VNFSDIR/$InstallDistributionName$InstallDistributionVersion-$InstallServiceLevel-$InstallPkgArchitecture"
			        	}{$dir}
				},
				$iserver;
		     }		
                    if (
                        ($iInstallOSName ne $InstallOSName)
                        || ($iInstallDistributionName ne
                            $InstallDistributionName)
                        || ($iInstallDistributionVersion ne
                            $InstallDistributionVersion)
                        || ($iInstallPkgArchitecture ne $InstallPkgArchitecture)
                       )
                    {
                        push @{
                            $sHash{
                                "/csminstall/Linux/$iInstallDistributionName/csm"
                              }{$dir}
                          },
                          $iserver;
                        push @{
                            $sHash{
                                "/csminstall/Linux/$iInstallDistributionName/$iInstallDistributionVersion/$iInstallPkgArchitecture"
                              }{$dir}
                          },
                          $iserver;
                    }
                }    # if "Linux"
            }    # End of foreach my $iserver

        }    # if $csm_skip_isvr
    }
    ($::DEBUG)
      && print
      "MS_INSTALL_SERVER (Is the management server an install server?) = $::MS_INSTALL_SERVER\n";
    print "LEAVING: $routine\n" if $::DEBUG;
    return (\%sHash);
}

sub get_NetworkInstallProtocol
{
    return $::__CLUSTER_INFO_HASH{'NetworkInstallProtocol'};
}

sub get_SetupNetworkInstallProtocol
{
    return $::__CLUSTER_INFO_HASH{'SetupNetworkInstallProtocol'};
}

#-------------------------------------------------------------------------------

=head3   getMSTimeZone

   Get the managment server time zone information, the time zone information
   will be added to osprereboot


=cut

#-------------------------------------------------------------------------------
sub getMSTimeZone
{
   my $pltfrm = `uname`;
   chomp $pltfrm;
   my $zone;

   if ($pltfrm eq "AIX")
   {
        my $envfile = "/etc/environment";
        my $tz = `$::GREP "TZ=" $envfile`;
        my ($junk, $timezone) = split('=', $tz);
        chomp $timezone;
        return $timezone;
   }
   else
   {
        my $clockcfg = "/etc/sysconfig/clock";
        my $ms_distro = NodeUtils->get_DistributionName();
        if (($ms_distro =~ /SLES/) || ($ms_distro =~ /SUSE/))
        {
            $zone = `$::GREP "^TIMEZONE" $clockcfg`;
        }
        else 
        {
            $zone = `$::GREP "ZONE" $clockcfg`;
        }
        my ($junk, $timezone) = split('=', $zone);
        $timezone =~ s/"//g;
        $timezone =~ s/'//g;
        chomp $timezone;
        return $timezone;
   }
}


#-------------------------------------------------------------------------------

=head3    make_osprereboot_script

    Creates the modified osprereboot script in the /csminstall/csm directory.


=cut

#-------------------------------------------------------------------------------
sub make_osprereboot_script
{
    my ($pkg, $target_node_hash) = @_;
	
	my $script="osprereboot";
	DCUtils->embed_is_info($target_node_hash,$script);
    
	return $::OK;
}

#--------------------------------------------------------------------------------

=head3    remove_reboot_ready_file


        Arguments:
                Node hash that contains the nodes to be processed.
        Returns:
                None
        Globals:
                None
        Example:
                ServerUtils->remove_reboot_ready_file(\%::NODEHASH);

=cut

#--------------------------------------------------------------------------------
sub remove_reboot_ready_file
{
    my ($class, $ref_DestNodeHash) = @_;

    my %DestNodeHash       = %{$ref_DestNodeHash};
    my $ref_MgtNodes       = DCUtils->get_nodes_with_no_is;
    my $ref_InstallServers = DCUtils->get_install_servers;
    my %InstallServers     = %$ref_InstallServers;
    my @MgtNodes           = @$ref_MgtNodes;
    my $cmd;

    if (@MgtNodes)
    {

        # remove the reboot_ready flag file on MS for the nodes using MS as IS
        my $reboot_ready_file;
        my $status_file;
        foreach my $hostname (@MgtNodes)
        {
            $reboot_ready_file =
              $::CSMSTATUSDIR . "/" . $hostname . ".rebootready";
            $status_file =
              $::CSMSTATUSDIR . "/" . $hostname;
            unlink($reboot_ready_file);
            unlink($status_file);
        }
    }

    $ENV{DSH_LIST} = '';    #clear dsh env.

    # >>> check reachability to the Install Servers >>>
    $ENV{'DSH_REMOTE_CMD'} = DCUtils->getRemoteShell;
    my @installServers = keys %InstallServers;
    my $refISNotReachable = NetworkUtils->checkRSReachability(\@installServers);
    if ($refISNotReachable != 0)
    {
        if (scalar @$refISNotReachable > 0)
        {
            my $isvrs = join ",", @$refISNotReachable;
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgISDown', $isvrs);
        }
    }

    foreach my $is (keys %InstallServers)
    {
        my $dir      = $InstallServers{$is}{"InstallDir"};
        my @nodelist = @{$InstallServers{$is}{"NodesToBeInstalled"}};
        my @status_files;
        my $file_list;
        foreach my $hostname (@nodelist)
        {
            push @status_files,
              $dir . "/csm/status/" . $hostname . ".rebootready";
        }
        foreach my $hostname (@nodelist)
        {
            push @status_files, $dir . "/csm/status/" . $hostname;
        }
        $file_list = join " ", @status_files;
        $cmd = "/opt/csm/bin/dsh -n $is rm -f $file_list";
        NodeUtils->runcmd($cmd, -2);
    }

}

#--------------------------------------------------------------------------------

=head3  getNodeFromGroup
    

        Choose one install server from a node group by round robin for a node.
        The result will be cached.

        Arguments:
            $target_grp
        Returns:
                0; Nothing in the nodegrp or no this nodegrp
                $ref_nodelist; 
        Globals:
        
        Example:
                $node = NodeUtils->getNodeFromGroup($target_grp);
        Comments:
            none

=cut

sub getNodeFromGroup
{
    my ($class, $target, $node) = @_;

    #if already cache it , return;
    if ($DCUtils::NodegrpNode{$node})
    {
        return $DCUtils::NodegrpNode{$node};
    }

    my @targetList = split(',', $target);
    if (@targetList le 1)
    {
        return 0;
    }

    my @server_list=map { s/:.*//;$_} @targetList;	
    
    my @sorted_list = sort @targetList;
    $target = join(',', @sorted_list);

    my @glist = @sorted_list;
    if (!(defined $DCUtils::NodegrpIndex{$target}))
    {
        $DCUtils::NodegrpIndex{$target} = 0;
    }
    else
    {
        $DCUtils::NodegrpIndex{$target}++;
    }

    if ($DCUtils::NodegrpIndex{$target} >= $#glist)
    {
        $DCUtils::NodegrpIndex{$target} -= ($#glist-1);
    }

    $DCUtils::NodegrpNode{$node} = $glist[$DCUtils::NodegrpIndex{$target}];

    return $DCUtils::NodegrpNode{$node};
}

#--------------------------------------------------------------------------------

=head3  verify_ms_dhcpd

        Arguments:
                target nodes hash

        Returns:
        
        Example:
                NetworkUtils->verify_ms_dhcpd(\%DestNodeHash);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------
sub verify_ms_dhcpd
{
    my $routine = "verify_ms_dhcpd";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $nodeHashRef, $serverHashRef) = @_;
    my (@mgrattr,    %isattr, $dhcp_interface);
    my ($cmd,        @output, @lines, $line);
    my (@ms_subnets, @issvrs, @mssvrs);

    my $mgrattr_ref = DCUtils->get_nodes_with_no_is;
    my $isattr_ref  = DCUtils->get_install_servers;

    if ($mgrattr_ref) { @mgrattr = @$mgrattr_ref; }
    if ($isattr_ref)  { %isattr  = %$isattr_ref; }

    # Any target nodes will be installed through install server different than MS?
    if (!keys %isattr) { return; }

    @issvrs = keys %isattr;
    foreach my $node (keys %$nodeHashRef)
    {

        # The attr ManagementServer would be a host name or ip address
        my $mgmtsvr = $$nodeHashRef{$node}{'ManagementServer'};
        if (grep(/^$mgmtsvr$/, @issvrs))
        {
            return;
        }

        my ($ms_hostname) = NetworkUtils->getHost($mgmtsvr);
        if ($ms_hostname && grep(/^$ms_hostname$/, @issvrs))
        {
            return;
        }

    }

    # Is DHCP daemon on MS running?
    $cmd = NetworkUtils->service("dhcpd", "status");
    chomp(@output = `$cmd 2>&1`);
    @lines = split(" ", @output);
    if (!grep /running/, @output) { return; }

    # Is DHCP daemon on MS listening on same the subnet as IS?
    $cmd = "$::GREP subnet /etc/dhcpd.conf";
    chomp(@lines = `$cmd`);
    foreach $line (@lines)
    {
        chomp($line);
        $line =~ s/^\s+//g;
        $line =~ s/\s+$//g;
        $line =~ s/\{$//g;
        next if ($line =~ /^#/);

        @output = split(/\s+/, $line);
        my $subnet = join " ", @output;
        push @ms_subnets, $subnet;
    }

    if (@issvrs)
    {
        my $node_file = ServerUtils->make_node_list_file(\@issvrs);
        $ENV{'DSH_LIST'} = $node_file;
        $cmd = "$::GREP subnet /etc/dhcpd.conf";
        @output = NodeUtils->runcmd("$::DSH '$cmd'", -1);
        ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);

        foreach $line (@output)
        {
            $line =~ s/^.*://g;
            $line =~ s/^\s+//g;
            $line =~ s/\s+$//g;
            $line =~ s/\{$//g;
            next if ($line =~ /^#/);

            my @strs = split(/\s+/, $line);
            my $subnet = join " ", @strs;
            foreach (@ms_subnets)
            {
                if ($subnet =~ /$_/)
                {

                    #my $progname = NodeUtils->programName();
                    MessageUtils->messageFromCat('csmInstall.cat',
                                           $::MSGMAPPATH, 'csminstall', 'E1',
                                           'EMsgDHCPSubnetsOverlaped', $subnet);
                }
            }
        }
    }

    print "LEAVING: $routine\n" if $::DEBUG;
}

#--------------------------------------------------------------------------------

=head3    setup_pxelinux_install

    Change the pxelinux HEX file to link to <node_hostname>.install
    verify dhcp tftp and pxe 

        Notes:

=cut

#--------------------------------------------------------------------------------

sub setup_pxelinux_install
{
    my ($pkg, $ref_node_hash, $ref_node_list) = @_;
    my @sisNodes;
    my @kickstartNodes;
    my @kickstartUpgradeNodes;
    my @yastNodes;
    my @wwNodes;
    foreach my $node (@$ref_node_list)
    {
        my $InstallMethod = $$ref_node_hash{$node}{'InstallMethod'};
        if ($InstallMethod =~ "sis")
        {
            push(@sisNodes, $node);
        }

        if ($InstallMethod =~ "kickstart")
        {
            push(@kickstartNodes, $node);
        }

        if ($InstallMethod =~ "kickstart-upgrade")
        {
            push(@kickstartUpgradeNodes, $node);
        }

        if ($InstallMethod =~ "autoyast")
        {
            push(@yastNodes, $node);
        }

        if ($InstallMethod =~ "warewulf")
        {
            push(@wwNodes, $node);
        }
    }

    if ($::NOVERIFY)
    {
        $ENV{CSM_NO_VERIFY} = 1;
    }

    if ($::NOREBOOT)
    {
        $ENV{CSM_NO_REBOOT} = 1;
    }

    if (@sisNodes)
    {
        DCUtils->run_setupBoot('sis', 'onlyhex', \@sisNodes, $ref_node_hash);
    }
    if (@kickstartNodes)
    {
        DCUtils->run_setupBoot('kickstart', 'onlyhex',
                               \@kickstartNodes, $ref_node_hash
                              );
    }
    if (@kickstartUpgradeNodes)
    {
        DCUtils->run_setupBoot(
                               'kickstart-upgrade', 'onlyhex',
                               \@kickstartUpgradeNodes, $ref_node_hash
                              );
    }
    if (@yastNodes)
    {
        DCUtils->run_setupBoot(
                               'autoyast',  'onlyhex',
                               \@yastNodes, $ref_node_hash
                              );
    }
    if (@wwNodes)
    {
        DCUtils->run_setupBoot(
                               'warewulf',  'onlyhex',
                               \@wwNodes, $ref_node_hash
                              );
    }

}

#--------------------------------------------------------------------------------

=head3    full_install

        Collection of preparation work before rebooting

        Arguments:
            $ref_node_hash
            $ref_node_list
        Notes:

=cut

#--------------------------------------------------------------------------------
sub full_install
{
    my ($pkg, $ref_node_hash, $ref_is_hash,$ref_node_list,$boot_params) = @_;
    
    DCUtils->setup_pxelinux_install($ref_node_hash, $ref_node_list);
    
    foreach my $node (keys %$ref_node_hash)
    {
        my ($bootpserver, $installserver, $nfsserver, $installdir) =
          DCUtils->get_is_addrinfo_by_name($node, $ref_node_hash);
        my $client_ip =
          DCUtils->get_node_addrinfo_by_name($node, $installserver,
                                               $ref_node_hash);
        my $client_netmask = $$ref_node_hash{$node}{"InstallAdapterNetmask"};
        if (!$client_netmask)
        {
            (undef, $client_netmask) =
              NetworkUtils->getNetmask($client_ip, $installserver);
        }
    
        my $bootpserver_netmask;
        (undef, $bootpserver_netmask) =
              NetworkUtils->getNetmask($bootpserver, $installserver);

        if (
            NetworkUtils->within_same_subnet(
                                             $client_ip,   $client_netmask,
                                             $bootpserver, $bootpserver_netmask
            ) == $::NOK
           )
        {
            push @::REMOTE_SUBNET_NODES, $node;
        }
        else
        {
            my $netadapter_mac =
              $$ref_node_hash{$node}{"InstallAdapterMacaddr"};
            ServerUtils->add_permanent_arp_cache($installserver, $client_ip,
                                                 $netadapter_mac)
              if ($::ARCH eq "ppc64");
        }

            my $nw_gateway   = $::NODEHASH{$node}{"InstallAdapterGateway"};

            my $netadapter_name = $::NODEHASH{$node}{"InstallAdapterName"};
            my $netadapter_mac  = $::NODEHASH{$node}{"InstallAdapterMacaddr"};
            my $svc_level       = $::NODEHASH{$node}{"InstallServiceLevel"};
            my $install_method  = $::NODEHASH{$node}{"InstallMethod"};
            if ($bootpserver ne $nfsserver)
            {
                $installdir = "/csminstall";
            }

            my $kernelparameters;
			my $dev;
			$dev = $netadapter_name;
			$dev ||= "eth0";
		
			my $master;
			$master = $installserver;
			$master ||= $bootpserver;
			
			if( $install_method eq "warewulf")
			{
				#nw_gateway can not be NULL, will cause warewulf failure.
				$nw_gateway ||= $::ATTRS{"Gateway"};
				$nw_gateway ||= $::BOOT_ATTRIBUTES{"Gateway"};
				$nw_gateway ||= $master;
				#----------------------------------------------------
				# Get warewulf diskless group name from the node's 
				# InstallTemplate attribute.
				# If InstallTemplate is blank, we use 
				# $::DISTRO_NAME$::DISTRO_VERSION-$svclevel-$::ARCH as 
				# default name. 
				#----------------------------------------------------
				 my $DisklessNodeGroupName = $::NODEHASH{$node}{"InstallTemplate"}; 
				 $DisklessNodeGroupName = `basename $DisklessNodeGroupName`
						 if ($DisklessNodeGroupName ne "");
				 $DisklessNodeGroupName =~ s/^.*warewulf-tmpl\.//g; 
				 chomp($DisklessNodeGroupName);
				 $DisklessNodeGroupName = "$::DISTRO_NAME$::DISTRO_VERSION-$svc_level-$::ARCH"
						 if ($DisklessNodeGroupName eq "");

				$kernelparameters = 
				"root=/dev/root " .
				"warewulf=$node,$master,$nw_gateway," .
				"$DisklessNodeGroupName," .
				"$dev=$client_ip:$client_netmask" .
				" ramdisk_size=$::WW_MAX_INITRD_SIZE";
				# the --kernelparameter flag
				if (defined $ENV{KERNELPAR})
				{
					$kernelparameters .= " $ENV{KERNELPAR}";
				}
			}
			else
			{
				$kernelparameters =
              ServerUtils->build_kernel_parameters($client_ip, $bootpserver,
                          $nfsserver, $installdir, $svc_level, $netadapter_name,
                          $netadapter_mac);
			}
           $$boot_params{$node}=$kernelparameters;
           # Warewulf takes charge of building initrd and determine the size of initrd.
		   if($::pkgdef_node{$node}{'ramdisk_size'} 
			 && $::NODEHASH{$node}{'InstallDistributionName'} !~ /SLES/
			 && $install_method ne "warewulf")
		   {
               $$boot_params{$node}.= " ramdisk_size=$::pkgdef_node{$node}{'ramdisk_size'}";
           }
			# kernel parameters special case for the following hardware platform
			if((($$ref_node_hash{$node}{"HWType"} eq "9133")
				|| ($$ref_node_hash{$node}{"HWType"} eq "8844")
				|| ($$ref_node_hash{$node}{"HWType"} eq "9115")
				|| ($$ref_node_hash{$node}{"HWType"} eq "9110"))
				&& ($$ref_node_hash{$node}{"InstallDistributionName"} =~ /SLES/)
				&& ($$ref_node_hash{$node}{"InstallDistributionVersion"} eq "9")
			)
			{
				$$boot_params{$node}.= " brokenmodules=s2io";
			}

    }
    if (@::REMOTE_SUBNET_NODES)
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'W',
                                     'EMsgRequireArpInfo',
                                     join("\n\t", @::REMOTE_SUBNET_NODES)
                                    );

    }

}

#--------------------------------------------------------------------------------

=head3     run_setupBoot

    Call nodeset via dsh from install server or call setupBoot from Management Server

        Notes:

=cut

#--------------------------------------------------------------------------------
sub run_setupBoot
{
    my $routine = "run_setupBoot";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $installmethod, $hexOption, $ref_nodelist, $ref_node_hash) = @_;

    my @VALID_ATTRIBUTES = ("Netmask", "Gateway", "Nameserver", "Nameservers");
    my @dest_node_list = @$ref_nodelist;
    if (!$ref_node_hash)
    {
        $ref_node_hash = \%::NODEHASH;
    }

    my %node_hash=%$ref_node_hash;

    foreach my $node (keys %node_hash)
    {
        if (!grep /^$node$/, @dest_node_list)
        {
            delete $node_hash{$node};
        }
    }

    my ($ref_MgtNodes, $ref_InstallServers) =
          DCUtils->getInstallServers(\%node_hash);
    
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my ($cmd, $cmd2);

    #get the basename of the user specified parameters
    if (defined $ENV{CSM_USERKERNEL})
    {
        my $baseName = basename($ENV{CSM_USERKERNEL});
        $ENV{CSM_USERKERNEL} = $baseName;
    }
    if (defined $ENV{CSM_USERRAMDISK})
    {
        my $baseName = basename($ENV{CSM_USERRAMDISK});
        $ENV{CSM_USERRAMDISK} = $baseName;
    }
    if (defined $ENV{CSM_USERPXEFILE})
    {
        my $baseName = basename($ENV{CSM_USERPXEFILE});
        $ENV{CSM_USERPXEFILE} = $baseName;
    }

    if (@MgtNodes)
    {
        $ENV{CSMINSTALL_ROOT} = $::CSMINSTDIR;
        NodesetUtils->setupBoot($installmethod, $hexOption, @MgtNodes);
    }

    $ENV{DSH_LIST} = '';    #clear dsh env.

    # >>> check reachability to the Install Servers >>>
    my @installServers;
	foreach my $is (keys %InstallServers)
	{
		if ($InstallServers{$is}{isGrp})
		{
			my @group = split(',',$is);
			my @servers_without_dir=map { s/:.*//;$_} @group;	
            		push @installServers, @servers_without_dir;
		}
		else
		{
			push @installServers, $is;
		}
	}
    my $refISNotReachable = NetworkUtils->checkRSReachability(\@installServers);
    if ($refISNotReachable != 0)
    {
        if (scalar @$refISNotReachable > 0)
        {
            my $isvrs = join ",", @$refISNotReachable;
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgISDown', $isvrs);
        }
    }

    # <<< check reachability to the Install Servers <<<

    my %real_is_hash;
    my %real_is_dir;
    foreach my $is (keys %InstallServers)
    {
        my @nodelist = @{$InstallServers{$is}{"NodesToBeInstalled"}};
        if ($InstallServers{$is}{isGrp})
        {
            foreach my $node (@nodelist)
            {
                my $real_is = DCUtils->getNodeFromGroup($is, $node);
                push @{$real_is_hash{$real_is}}, $node;
        		$real_is_dir{$real_is}=$InstallServers{$is}{'InstallDir'};	
    		}
        }
        else
        {
            push @{$real_is_hash{$is}}, @nodelist;
        	$real_is_dir{$is}=$InstallServers{$is}{'InstallDir'};	
        }
    }
    foreach my $is (keys %real_is_hash)
    {
        my $dir      = $real_is_dir{$is};
        my @nodelist = @{$real_is_hash{$is}};
				
        my $node_file;
        if (scalar(@nodelist) > 500)
        {

            # for better performance: only build and use a temporary file
            # if the list is greater than 500 nodes.
            $node_file = ServerUtils->make_node_list_file(\@nodelist);
            $cmd       = "/opt/csm/bin/dcp -n $is $node_file $node_file";
            NodeUtils->runcmd($cmd, -1);
        }

        if (($hexOption eq "onlyhex") && 
			(($installmethod eq "disk") || ($installmethod eq "you")))
        {

            my @status_files;
            my $file_list;

            # Make status directory as necessary
            my $stat_dir = $dir . "/csm/status";
            $cmd  = "/opt/csm/bin/dsh -n $is mkdir -p  $stat_dir";
            $cmd2 = "/opt/csm/bin/dsh -n $is chmod 755  $stat_dir";
            NodeUtils->runcmd($cmd,  -2);
            NodeUtils->runcmd($cmd2, -2);
            
			# Create the reboot_ready flag file on IS
           	foreach my $hostname (@nodelist)
           	{
               	push @status_files,
               	$stat_dir . "/" . $hostname . ".rebootready";
           	}
			
           	$file_list = join " ", @status_files;
           	$cmd = "/opt/csm/bin/dsh -n $is touch $file_list";
           	NodeUtils->runcmd($cmd, -2);
        }
        my $rmt_cmd;
        my $rmt_var;
        if (defined $ENV{CSM_NO_SETUP_DHCP})
        {
            $rmt_var .= "export CSM_NO_SETUP_DHCP=1; ";
        }
        if (defined $ENV{CSM_NO_VERIFY})
        {
            $rmt_var .= "export CSM_NO_VERIFY=1; ";
        }
        if (defined $ENV{CSM_NO_REBOOT})
        {
            $rmt_var .= "export CSM_NO_REBOOT=1; ";
        }
        if (defined $ENV{CSM_USERKERNEL})
        {
            $rmt_var .= "export CSM_USERKERNEL=$ENV{CSM_USERKERNEL}; ";
        }
        if (defined $ENV{CSM_USERRAMDISK})
        {
            $rmt_var .= "export CSM_USERRAMDISK=$ENV{CSM_USERRAMDISK}; ";
        }
        if (defined $ENV{CSM_USERPXEFILE})
        {
            $rmt_var .= "export CSM_USERPXEFILE=$ENV{CSM_USERPXEFILE}; ";
        }
        if (defined $ENV{KERNELPAR})
        {
            $rmt_var .= "export KERNELPAR=\"$ENV{KERNELPAR}\";";
        }
        my $install_protocol = NodesetUtils->get_NetworkInstallProtocol();
        $rmt_var .= "export NETWORK_INSTALL_PROTOCOL=$install_protocol; ";

        my $setup_install_protocol =
          NodesetUtils->get_SetupNetworkInstallProtocol();
        $rmt_var .=
          "export SETUP_NETWORK_INSTALL_PROTOCOL=$setup_install_protocol; ";

        # build up attributes string including Gateway=xxx Netmask=xxx,
        # which can be identified by nodeset.utils.
        my $attrs_str;
        foreach my $key (%::ATTRS)
        {
            if (grep(/^$key$/, @VALID_ATTRIBUTES))
            {
                my $value = $::ATTRS{"$key"};
                $attrs_str .= " $key=$value " if ($value);
            }
        }

        if ($node_file)
        {
            $rmt_cmd =
              "$rmt_var export CSMINSTALL_ROOT=$dir; /opt/csm/bin/nodeset.utils -b $installmethod -P $hexOption -f $node_file $attrs_str && $::RM -f $node_file";
        }
        else
        {
            my $nodestring = join(',', @nodelist);

            $rmt_cmd =
              "$rmt_var export CSMINSTALL_ROOT=$dir; /opt/csm/bin/nodeset.utils -b $installmethod -P $hexOption -n $nodestring $attrs_str";
        }

        $cmd = "$::DSH -n $is '$rmt_cmd'";
        my $output = NodeUtils->runcmd($cmd, 1);
        $::BOOT_EXIT = 1 if $::RUNCMD_RC;
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'I', 'IMsgShow_Output',
                                     $output);

        #
        #Need analyse to get detail error info ?
        #
        if ($node_file)
        {
            ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);
        }
    }
    print "LEAVING: $routine\n" if $::DEBUG;
}

#--------------------------------------------------------------------------------

=head3    getInstallServers

        Get a cluster's Install Servers. 

        Arguments:
                Reference to a nodeName Hash;
        Returns:
        Reference to mgt nodes array and installServer Hash
        Error:
                Reference to an empty Hash - there are no install servers in the cluster.
        Example:
                 DCUtils->getInstallServers(\%DestNodeHash);
=cut

#--------------------------------------------------------------------------------
sub getInstallServers
{
    my $routine = "getInstallServers";
    print "ENTERING: $routine\n" if $::DEBUG;
    my ($class, $ref_dest_nodes_hash) = @_;

    my %dest_nodes_hash = %$ref_dest_nodes_hash;
    my %InstallServerAttr;
    my @MgrAttr;

    # for each node to be nimd
    foreach my $node (keys %dest_nodes_hash)
    {
        next
          if (   !defined $dest_nodes_hash{$node}{"ManagementServer"}
              || !defined $dest_nodes_hash{$node}{"Hostname"});

        my ($ms_hostname, $ms_ip) =
          NetworkUtils->getHost_rveg(
                                   $dest_nodes_hash{$node}{"ManagementServer"});
        if ($::GETHOST_RC != $::OK)
        {
            MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                   'NodeUtils', 'E', $::GETHOST_MSG,
                                   $dest_nodes_hash{$node}{"ManagementServer"});
            next;
        }
        my ($node_hostname, $node_ip) =
          NetworkUtils->getHost_rveg($dest_nodes_hash{$node}{"Hostname"});
        if ($::GETHOST_RC != $::OK)
        {
            MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                         'NodeUtils', 'E', $::GETHOST_MSG,
                                         $dest_nodes_hash{$node}{"Hostname"});
            next;
        }

        # skip undefined and illegal node
        my $servname = $dest_nodes_hash{$node}{"InstallServer"};

        if (!$servname)
        {

            # handle node without install server
            push(@MgrAttr, $node);
            next;
        }

        my @server_list = split ',', $servname;

        # handle node with install server
        my ($server, $dir) = split ':', $server_list[0];
        if (!$server)
        {
            push(@MgrAttr, $node);
            next;
        }

        # the IS is nodegroup
        if ($server =~ /^\+/)
        {
            my $ref_grp = NodeUtils->getNodegrp($server);
            if ($ref_grp)
            {
                @server_list = @$ref_grp;
                if (scalar @server_list > 1)
                {
                    $servname = jion(',', @server_list);
                }
                else
                {
                    $server = $server_list[0];
                }
            }
        }

        my ($hostname, $ipaddr);

        my $isISGrp = 0;
        if (@server_list > 1)
        {
            $hostname = $servname;
            $isISGrp  = 1;
        }
        else
        {
            ($hostname, $ipaddr) = NetworkUtils->getHost_rveg($server);
            if ($::GETHOST_RC != $::OK)
            {
                MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                   'NodeUtils', 'E', $::GETHOST_MSG,
                                   $dest_nodes_hash{$node}{"ManagementServer"});
                next;
            }

            if ($ipaddr eq $ms_ip)
            {

                # handle node with the same install server as management server
                # or install server is server as a managed node simultaneously
                push(@MgrAttr, $node);
                next;
            }

            if ($ms_ip eq $node_ip)
            {
                push(@MgrAttr, $node);
                next;
            }
        }

        # handle node with install server different from management server
        if ($dir)
        {

            # fix last character in $dir string
            if (substr($dir, -1) eq "/") { chop($dir); }
        }
        else
        {

            # set default $dir
            $dir = "/csmserver";
        }

        #
        # From the subroutine of NodesetUtils->run_setupBoot() we can see
        # that we cannot use the InstallServerAKBNode attribute here.
        #
        push(@{$InstallServerAttr{$hostname}{"NodesToBeInstalled"}}, $node);
        $InstallServerAttr{$hostname}{"InstallDir"} = $dir;
        if ($isISGrp)
        {
            $InstallServerAttr{$hostname}{"isGrp"} = 1;
        }

    }

    print "LEAVING: $routine\n" if $::DEBUG;
    return (\@MgrAttr, \%InstallServerAttr);
}

#--------------------------------------------------------------------------------

=head3    setInstallMsgPort 

      Arguments:
      Returns:
          Notes:
        Set InstallMsgPort used by nodestatusd daemon.
=cut

#--------------------------------------------------------------------------------
sub setInstallMsgPort
{
    my ($pkg,          $ref_node_hash)      = @_;
    my ($ref_MgtNodes, $ref_InstallServers) =
      DCUtils->getInstallServers($ref_node_hash);
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my $file           = "/etc/services";
    my $port_name      = "csm-nodemsgs";
    my @service_ports;
    my @active_ports;

    open(SERVICES, "<$file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',     $::MSGMAPPATH,
                                      'csminstall',         "E",
                                      'EMsgCANT_READ_FILE', "$file"
                                     );
    @service_ports = <SERVICES>;
    close(SERVICES);
    my $output = NodeUtils->runcmd("netstat -ta", -1, 1);
    my $ms_port = ServerUtils->findInstallMsgPort(\@service_ports, $output);

    if (!grep(/$port_name/, @service_ports))
    {
        open(SERVICES, ">>$file")
          or MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', "E", 'EMsgCANT_WRITE_FILE', "$file");
        print SERVICES "$port_name\t$ms_port\/tcp\n";
        print SERVICES "$port_name\t$ms_port\/udp\n";
        close(SERVICES);
    }

    foreach my $node (@MgtNodes)
    {
        $$ref_node_hash{$node}{'InstallMsgPort'} = $ms_port;
    }
    my %realIS;
    foreach my $is (keys %InstallServers)
    {
        my @nodelist = @{$InstallServers{$is}{"NodesToBeInstalled"}};
        if ($InstallServers{$is}{isGrp})
        {
            foreach my $node (@nodelist)
            {
                my $real_is = DCUtils->getNodeFromGroup($is, $node);
                push @{$realIS{$real_is}}, $node;
            }
        }
        else
        {
            push @{$realIS{$is}}, @nodelist;
        }
    }

    foreach my $is (keys %realIS)
    {
        my @nodelist = @{$realIS{$is}};
        my $cmd;
        my (@service_ports, @active_ports);
        $cmd = "/opt/csm/bin/dsh -n $is \"cat $file\"";
        my $output = NodeUtils->runcmd($cmd, -1, 1);
        if (!$::RUNCMD_RC)
        {
            foreach my $line (@$output)
            {
                $line =~ s/^$is:\ //g;
                push @service_ports, $line;
            }
        }
        $cmd = "/opt/csm/bin/dsh -n $is \"netstat -tn\"";
        $output = NodeUtils->runcmd($cmd, -1, 1);
        if (!$::RUNCMD_RC)
        {
            foreach my $line (@$output)
            {
                $line =~ s/^$is:\ //g;
                push @active_ports, $line;
            }
        }
        my $port =
          ServerUtils->findInstallMsgPort(\@service_ports, \@active_ports);
        foreach my $node (@nodelist)
        {
            $$ref_node_hash{$node}{'InstallMsgPort'} = $port;
        }
        if (!grep(/^$port_name/, @service_ports))
        {
            my $line = "$port_name\t$port\/tcp";
            push @service_ports, $line;
            $line = "$port_name\t$port\/udp";
            push @service_ports, $line;
            my $node_file = ServerUtils->make_node_list_file(\@service_ports);
            $cmd = "/opt/csm/bin/dcp -n $is $node_file $file";
            NodeUtils->runcmd($cmd, -2);
            ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);
        }
    }
    return $ms_port;
}

#--------------------------------------------------------------------------------

=head3    start_nodestatusd 

      Arguments:
      Returns:
          Notes:
        Start nodestatusd daemon on management server and install server.

=cut

#--------------------------------------------------------------------------------
sub start_nodestatusd
{
    my ($pkg, $ref_node_hash, $ms_port) = @_;
    my ($ref_MgtNodes, $ref_InstallServers) =
      DCUtils->getInstallServers($ref_node_hash);
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my $nodestatusd    = "/opt/csm/csmbin/nodestatusd";
    my $port;
    my $dsh = "/opt/csm/bin/dsh";
    my $dcp = "/opt/csm/bin/dcp";
    my $mgmtsvr;

    foreach my $is (keys %InstallServers)
    {
        my @nodelist = @{$InstallServers{$is}{"NodesToBeInstalled"}};
        my $dir      = $InstallServers{$is}{"InstallDir"};
        $mgmtsvr = $$ref_node_hash{$nodelist[0]}{'ManagementServer'};
        my $cmd;
        if ($InstallServers{$is}{isGrp})
        {
            my @realIS;
            my %ports;
            foreach my $node (@nodelist)
            {
                my $real_is = DCUtils->getNodeFromGroup($is, $node);
                push @realIS, $real_is;
                $ports{$real_is} = $$ref_node_hash{$node}{'InstallMsgPort'};
            }
            foreach my $is (@realIS)
            {
                my ($ms_hostname, $ms_ip) =
                  NetworkUtils->get_management_server($is);
                $cmd =
                  "$dsh -n $is \"echo $dir >/tmp/CSMINSTALL_ROOT;killall -s HUP nodestatusd 2>&1 || CSMINSTALL_ROOT=$dir MGMTSVR=$ms_ip SERVER=IS MS_PORT=$ms_port $nodestatusd -p $ports{$is}\"";
                NodeUtils->runcmd($cmd, -2);
            }
        }
        else
        {
            $port = $$ref_node_hash{$nodelist[0]}{'InstallMsgPort'};
            my ($ms_hostname, $ms_ip) =
              NetworkUtils->get_management_server($is);
            $cmd =
              "$dsh -n $is \"echo $dir >/tmp/CSMINSTALL_ROOT;killall -s HUP nodestatusd 2>&1 || CSMINSTALL_ROOT=$dir MGMTSVR=$ms_ip SERVER=IS MS_PORT=$ms_port $nodestatusd -p $port\"";
            NodeUtils->runcmd($cmd, -2);
        }
    }

    my @ms_ip;
    if (NodeUtils->isAIX)
    {
        my $cmd = "$::IFCONFIG -a|grep \"inet\ \"";
        my $outref = NodeUtils->runcmd($cmd, -2, 1);
        foreach my $ip (@$outref)
        {
            $ip =~ s/^.*inet\ //g;
            $ip =~ s/\ .*//g;
            push @ms_ip, $ip;
        }
        $mgmtsvr = join(',', @ms_ip);
        $cmd     = "ps -e -o pid,args|grep nodestatusd";
        $outref  = NodeUtils->runcmd($cmd, -2, 1);
        if (@$outref > 1)
        {
            my $cmd = "kill -HUP";
            foreach my $pid (@$outref)
            {
                chomp $pid;
                $pid =~ s/\s.*//g;
                $cmd = "$cmd $pid";
            }
            NodeUtils->runcmd($cmd, -1);
        }
        else
        {
            $cmd =
              "CSMINSTALL_ROOT=/csminstall MGMTSVR=$mgmtsvr SERVER=MS $nodestatusd -p $ms_port";
            NodeUtils->runcmd($cmd, -2);
        }
    }
    elsif (NodeUtils->isLinux)
    {
        my $cmd = "$::IFCONFIG -a|grep \"inet\ addr\"";
        my $outref = NodeUtils->runcmd($cmd, -2, 1);
        foreach my $ip (@$outref)
        {
            $ip =~ s/\s+inet\ addr://g;
            $ip =~ s/\ .*//g;
            push @ms_ip, $ip;
        }
        $mgmtsvr = join(',', @ms_ip);
        $cmd =
          "killall -s HUP nodestatusd 2>&1 || CSMINSTALL_ROOT=/csminstall MGMTSVR=$mgmtsvr SERVER=MS $nodestatusd -p $ms_port";
        NodeUtils->runcmd($cmd, -2);
    }
}

#--------------------------------------------------------------------------------

=head3    setAllowNodeMsgs 

      Arguments:
      Returns:
          Notes:
        Produce AllowNodeMsgs file on management server and install server.

=cut

#--------------------------------------------------------------------------------
sub setAllowNodeMsgs
{
    my ($pkg,          $ref_node_hash)      = @_;
    my ($ref_MgtNodes, $ref_InstallServers) =
      DCUtils->getInstallServers($ref_node_hash);
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my $dir="/var/opt/csm";
    my $allow_file     = "$dir/AllowNodeMsgs";

    if(!-d $dir)
    {
	    mkdir($dir);
    }

    my @mgtnodes;
    foreach my $node (@MgtNodes)
    {
        my ($hostname, $ip) = NetworkUtils->getHost($node);
        if ($$ref_node_hash{$node}{"InstallAdapterHostname"})
        {
            my $adapter_hostname =
              $$ref_node_hash{$node}{"InstallAdapterHostname"};
            my ($res_hostname, $res_ip) =
              NetworkUtils->getHost($adapter_hostname);
            $ip = $ip . " " . $res_ip;
        }
        push @mgtnodes, $ip;
    }
    if (@mgtnodes)
    {
        ServerUtils->make_allow_file(\@mgtnodes);
    }

    my %realIS;
    foreach my $is (keys %InstallServers)
    {
        my @nodelist = @{$InstallServers{$is}{"NodesToBeInstalled"}};
        if ($InstallServers{$is}{isGrp})
        {
            foreach my $node (@nodelist)
            {
                my $real_is = DCUtils->getNodeFromGroup($is, $node);
                my ($hostname, $ip) = NetworkUtils->getHost($node);
                push @{$realIS{$real_is}}, $ip;
            }
        }
        else
        {
            my @ip_list;
            foreach my $node (@nodelist)
            {
                if ($$ref_node_hash{$node}{"InstallAdapterHostname"})
                {
                    my $adapter_hostname =
                      $$ref_node_hash{$node}{"InstallAdapterHostname"};
                    my $ip =
                      ServerUtils->dshGetNodeAddr($adapter_hostname, $is);
                    push @ip_list, $ip;
                }
                else
                {
                    my ($hostname, $ip) = NetworkUtils->getHost($node);
                    push @ip_list, $ip;
                }
            }
            push @{$realIS{$is}}, @ip_list;
        }
    }

    my @install_servers = keys %realIS;
    my @server_list;
    foreach my $is (@install_servers)
    {
        my ($hostname, $ip) = NetworkUtils->getHost($is);
        push @server_list, $ip;
    }
    if (@server_list)
    {
        ServerUtils->make_allow_file(\@server_list);
    }
    foreach my $is (keys %realIS)
    {
        my @nodelist = @{$realIS{$is}};
        my $cmd;
        $cmd = "/opt/csm/bin/dsh -n $is \"cat $allow_file\"";
        my $output = NodeUtils->runcmd($cmd, -1, 1);
        if (!$::RUNCMD_RC && !grep(/No\ such\ file\ or\ directory/, @$output))
        {
            foreach my $line (@$output)
            {
                $line =~ s/^$is:\ //g;
                push @nodelist, $line;
            }
            @nodelist = ServerUtils->get_uniqu_arry_elemnts(@nodelist);
        }

        my $node_file = ServerUtils->make_node_list_file(\@nodelist);
        $cmd = "/opt/csm/bin/dcp -n $is $node_file $allow_file";
        NodeUtils->runcmd($cmd, -2);
        ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);
    }
}
#--------------------------------------------------------------------------------

=head3    validate_install_method

    Make sure the InstallMethod is supported for all nodes provided on the
    command line via either the -N flag or the node_list.

        Notes:

=cut

#--------------------------------------------------------------------------------

sub validate_install_method
{
    my ($pkg,$ref_node_hash,$ref_node_list)=@_;
    my ($InstallMethod);
    my (@badnodes);

    foreach my $node (@$ref_node_list)
    {
        $InstallMethod = $$ref_node_hash{$node}{'InstallMethod'};
        if ($InstallMethod eq ""
            || !grep(/^$InstallMethod$/, @::VALID_INSTALLMETHODS))
        {
            push(@badnodes, $node);
        }
    }

    # Print an error and don't install any nodes if any of them have
    # an invalid InstallMethod attribute
    if (($#badnodes + 1) > 0)
    {
        MessageUtils->message('E2', 'EMsgINVALID_INSTALL_METH',
                              join(',', @badnodes),
                              join(',', @::VALID_INSTALLMETHODS));
    }

    if ($InstallMethod =~ "autoyast")
    {
        my $distro = $::DISTRO_NAME . $::DISTRO_VERSION;
        my $rc = NodeUtils->check_valid_install_distros("autoyast", $distro);
        ($rc)
          && MessageUtils->message("E$rc", 'EMsgUNSUPPORTED_FULLINST_DISTRO',
                                   $distro, "autoyast");
    }
}
#--------------------------------------------------------------------------------

=head3    get_is_addrinfo_by_name

        Notes: 
        get address and dir of install server based on a specified node

=cut

#--------------------------------------------------------------------------------
sub get_is_addrinfo_by_name
{
	my ($class, $node,$ref_node_hash) = @_;
	my ($addr,  $isaddr, $nfsaddr, $dir);

	my $akb_node = $$ref_node_hash{$node}{"InstallServerAKBNode"};
	my $is       = $$ref_node_hash{$node}{"InstallServer"};
	my $adp_name = $$ref_node_hash{$node}{"InstallAdapterHostname"};
	my (undef, $ms_ip) =
	  NetworkUtils->getHost($$ref_node_hash{$node}{"ManagementServer"});
	if (!$adp_name)
	{
		$adp_name = $node;
	}

	my @servers=split(",",$is);
    if(@servers>1)
	{
		$is = DCUtils->getNodeFromGroup($is, $node);
	}

	if ($akb_node)
	{
		if ($is)
		{

			# split out IS addr and dir,
			# if IS is MS, default installdir to /csminstll; else to /csmserver
			($addr, $dir) = split(":", $is);
			(undef, $addr) = NetworkUtils->getHost($addr);
			$isaddr = $addr;
			if ($addr eq $ms_ip)
			{
				$dir = $::CSMINSTDIR;
			}
			if (!$dir) { $dir = $::SERVER_DIRECTORY }

			# resolve InstallServerAKBNode
			$addr = ServerUtils->dshGetNodeAddr($akb_node, $addr);
			if ($addr == -1)
			{

				# if fails to resolve InstallServerAKBNode, default it
				# through "getsourceip2target InstallAdapterHostname"
			    $addr=NetworkUtils->get_source_ip_to_target($adp_name,$is);

				if (!$addr)
				{
					(undef, $addr) = NetworkUtils->getHost($addr);
				}
			}
			
			$nfsaddr = $addr;
		}
		else
		{

			# If InstallServerAKBNode is an IP address, just use it
			if (NetworkUtils->isIpaddr($akb_node))
			{
				$addr = NetworkUtils->validate_ip($akb_node);
				if ($addr == -1) { $addr = undef; }
			}
			else
			{

				# InstallServerAKBNode is not an IP address, try to
				# resolve it.
				(undef, $addr) = NetworkUtils->getHost_rveg($akb_node);
				if ($::GETHOST_RC != $::OK)
				{

					# if InstallServerAKBNode can not be resolved, default it
					# through "getsourceip2target InstallAdapterHostname"
					$addr = NetworkUtils->get_source_ip_to_target($adp_name);
				}

			}
			$isaddr = undef;#$ms_ip;
			$nfsaddr = $addr;
			$dir = $::CSMINSTDIR;
		}
	}
	else
	{
		if ($is)
		{
			($addr, $dir) = split(":", $is);
			(undef, $addr) = NetworkUtils->getHost($addr);
			$isaddr = $addr;
			$nfsaddr = $addr;

			if ($addr eq $ms_ip)
			{
				$dir = $::CSMINSTDIR;
			}

			if (!$dir) { $dir = $::SERVER_DIRECTORY }
		}
		else
		{
			my $ns = $$ref_node_hash{$node}{"NFSServer"};
			($ns, $dir) =split(/:/,$ns);
			if($ns){
				$ns = NetworkUtils->validate_ip($ns);
				if($ns == -1){
					$ns = $ms_ip;
					$dir = $::CSMINSTDIR;
				}else{
					$dir = $::CSMINSTDIR if($ns eq $ms_ip);
					$dir = $::SERVER_DIRECTORY if(!$dir); 
				}
			}else{
				$ns = $ms_ip;
				$dir = $::CSMINSTDIR;
			}
			$addr = $ms_ip;
			$isaddr = undef;#$ms_ip;
			$nfsaddr = $ns;
		}
	}

	return ($addr, $isaddr, $nfsaddr, $dir);
}

#--------------------------------------------------------------------------------

=head3    get_node_addrinfo_by_name

        Notes: 
        get ip address for a target node

=cut

#--------------------------------------------------------------------------------
sub get_node_addrinfo_by_name
{
	my ($class, $node, $is ,$ref_node_hash) = @_;

	my $addr = undef;

	my $akb_node = $$ref_node_hash{$node}{"InstallServerAKBNode"};
	my $adp_name = $$ref_node_hash{$node}{"InstallAdapterHostname"};
	my (undef, $ms_ip) =
	  NetworkUtils->getHost($$ref_node_hash{$node}{"ManagementServer"});
	if (!$adp_name)
	{
		$adp_name = $node;
	}

	if ($is && ($is ne $ms_ip))
	{
		($is, undef) = NetworkUtils->getHost($is);
		$addr = ServerUtils->dshGetNodeAddr($adp_name, $is);
		if ($addr == -1)
		{
			(undef, $addr) = NetworkUtils->getHost($adp_name);
		}
	}
	else
	{
		(undef, $addr) = NetworkUtils->getHost($adp_name);
	}

	return $addr;
}

#-------------------------------------------------------------------------------

=head3    getDefaultInstallServerAKBNode
        
		If InstallServerAKBNode attribute is not specified with InstallServer attribute, 
		this subroutine can be used to acquire the default install server ip address 
		acknowledged by the node.

        Arguments:
                 $node
                 $ref_DestinationNodeHash
        Returns:
                Default InstallServerAKBNode of the node
                
        Globals:
                %::NODEHASH
        Error:
                -1: Can not acquire default InstallServerAKBNode of the node
        Example:
				$akbserver = ServerUtils->getDefaultInstallServerAKBNode($node,
                                                   \%DestNodeHash);
				if($akbserver != -1){
                    blah;
                }
        Comments:
			It must be guaranteed that DSH from MS to IS is reachable.
=cut

#-------------------------------------------------------------------------------

sub getDefaultInstallServerAKBNode
{
    my ($class, $node, $refNodeHash) = @_;
    if (!$refNodeHash)
    {
        $refNodeHash = \%::NODEHASH;
    }

    my $server_attr=$refNodeHash->{$node}{"InstallServer"};
    my @servers=split ",", $server_attr;
    my ($server,$rep_dir)=split ":",$servers[0];
    my $ms_used = 0;                                # MS is choosen as IS?
    if (scalar(@servers)>1)
    {
        $server = DCUtils->getNodeFromGroup($server_attr, $node);
    }

    my $adapterHostname = $node;
    if ($refNodeHash->{$node}{"InstallAdapterHostname"} && scalar(@servers) le 1)
    {
        $adapterHostname = $refNodeHash->{$node}{"InstallAdapterHostname"};
    }

    my $ip;
    if (NodeUtils->isMgmtSvr || $ENV{'DC_ENVIRONMENT'} )
    {
        $ip = NetworkUtils->get_source_ip_to_target($adapterHostname,$server);
    }
    else
    {
        $ip = NetworkUtils->get_source_ip_to_target($adapterHostname);
    }

    if (!$ip)
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'E1',
                                     'EMsgHostnameUnknownOnNode',
                                     $adapterHostname,
                                     $server
                                     );
        return -1;
    }
    return ($ip,$rep_dir);
}
#----------------------------------------------------------------
sub merge_config_info
{
    my $cwd = getcwd();
    
    chdir("$::CSMCFGDIR");
    foreach my $node (keys %::__TARGET_NODE_HASH)
    {
        my %node_hash;
        open(INFO, "<$node.config_info")
          or MessageUtils->message('E2', 'EMsgCANT_READ_FILE', "$node.config_info");
        while (<INFO>)
        {
            #Match attribute=value line
            #The value can be null if the user do want to empty it.
            if(/^\s*(\w+)\s*=\s*(.*)\s*$/)
            {
                $node_hash{$1}=$2;
            }
        }
        close(INFO);
        foreach my $attr (keys %node_hash)
        {
            #These attrbutes will be regenerated and put into config info file
            #We do not inherit them from previous config info written by csmsetup* commands 
            if(!defined($::__TARGET_NODE_HASH{$node}{$attr}) 
                && $attr ne "InstallMsgServer" 
                && $attr ne "InstallServer"
                && $attr ne "InstallServerHost"
                && $attr ne "SMS" )
            {
                $::__TARGET_NODE_HASH{$node}{$attr}=$node_hash{$attr};
            }    
                
        }
        if(!defined($::__TARGET_NODE_HASH{$node}{'RemoteShell'}))
        {
            my $remote_shell=DCUtils->getRemoteShell;
            if($remote_shell)
            {
                $::__TARGET_NODE_HASH{$node}{'RemoteShell'}=$remote_shell;
            }
        }
        open(INFO, ">$node.config_info")
          or MessageUtils->message('E2', 'EMsgCANT_WRITE_FILE', "$node.config_info");
        
        my $instsvr;
        my $server_attr;
		if (defined($::__TARGET_NODE_HASH{$node}{'InstallServer'})
            && ($::__TARGET_NODE_HASH{$node}{'InstallServer'} ne ""))
        {
            $server_attr=$::__TARGET_NODE_HASH{$node}{'InstallServer'};
            my @servers = split(',',$server_attr);
            my ($server, $rep_dir) = split(':',$servers[0]);

            if (scalar(@servers)>1)
            {
                $instsvr = $::__TARGET_NODE_HASH{$node}{'ManagementServer'};
            }
            else
            {
                $instsvr = $server;
            }
            my ($isvrakbnode_ip,$rep_dir) =
                DCUtils->getDefaultInstallServerAKBNode($node,
                                                      \%::__TARGET_NODE_HASH);
            if ($isvrakbnode_ip)
            {
                $::__TARGET_NODE_HASH{$node}{'InstallMsgServer'}=$isvrakbnode_ip;
                if ($rep_dir)
                {
                   $::__TARGET_NODE_HASH{$node}{'InstallServer'}="$isvrakbnode_ip:$rep_dir";
                }
                else
                {
                   $::__TARGET_NODE_HASH{$node}{'InstallServer'}=$isvrakbnode_ip;
                }
    
                # Resolve the ip to host name and write it to config_info file
                my ($is_hostname, undef) =
                  ServerUtils->dshGetNodeAddrInfo($isvrakbnode_ip,
                                                  $instsvr);
                if ($is_hostname != -1)
                {
                   $::__TARGET_NODE_HASH{$node}{'InstallServerHost'}=$is_hostname;
                }
            }

        }
        elsif (defined($::__TARGET_NODE_HASH{$node}{'ManagementServer'}))
        {
            $instsvr = $::__TARGET_NODE_HASH{$node}{'ManagementServer'};
            my ($hostname, $ip) = NetworkUtils->getHost($instsvr);
            $::__TARGET_NODE_HASH{$node}{'InstallMsgServer'}=$ip;
        }

        # Handle Linux-specific attributes.
        if ($::__TARGET_NODE_HASH{$node}{'InstallOSName'} eq "Linux")
        {

            # if $::NOSMS isn't set and if the node is not diskless node,
            # then perform software maintenance
            if ((!$::NOSMS)
                && ($::__TARGET_NODE_HASH{$node}{'InstallMethod'} ne "diskless"))
            {
                $::__TARGET_NODE_HASH{$node}{"SMS"}=1;
            }
            else
            {
                $::__TARGET_NODE_HASH{$node}{"SMS"}=0;
            }
        }

        
        #Set "Mode" to MinManaged if DC is used by exploiter other than CSM
        #Thus CSM special customization scripts will be skipped in csmfirstboot.
        if($ENV{'DC_ENVIRONMENT'} ne "CSM")
        {
            $::__TARGET_NODE_HASH{$node}{'Mode'}='DCInstall';
        }
        foreach my $attr (keys %{$::__TARGET_NODE_HASH{$node}})
        {
                #These attributes have been processed above
        	print INFO "$attr=$::__TARGET_NODE_HASH{$node}{$attr}\n";
        }
        close(INFO);
		#restore install server attribute
		$::__TARGET_NODE_HASH{$node}{"InstallServer"}=$server_attr;
    }
    
    chdir("$cwd");
}
#--------------------------------------------------------------------------------

=head3	common_install_method

        Test that the install method attribute is identical
	across all the nodes in the global %::NODEHASH.

        Arguments:
                uses global below
        Returns:
                a scalar value of the InstallMethod node attribute held in
		common by all nodes in the %::NODEHASH ( "kickstart" || "yast" ).
        Globals:
                %::NODEHASH - input list of nodes to check
        Error:
                Exits with an error if the InstallMethod attribute is not
                the same on all the nodes. 
        Example:
                $::INSTALL_METHOD  = DCUtils->common_install_method();
        Comments:

		May be used in conjuction with "NodeUtils->get_common_attrs()"
		but this routine exists to handle an idiosyncrasy in installnode().

		NOTE:  **** THIS SUBROUTINE IS CURRENTLY UNUSED ****

=cut

#--------------------------------------------------------------------------------

sub common_install_method
{
	my ($pkg,$ref_node_hash,$ref_node_list)=@_;

	my $const_attr = 0;                   # the InstallMethod value
	my @hosts      = @$ref_node_list;    # all hosts
	my $inst_method;
	my ($effective_distro_name) = NodeUtils->getEffectiveDistro($::DISTRO_NAME);

	if (($effective_distro_name =~ /RHEL/) || ($effective_distro_name =~ /RedHat/))
	{
		$inst_method = "kickstart";
	}else
	{
		$inst_method = "autoyast";
	}

	# go through all hosts and compare InstallMethod to $const_attr

	foreach my $host (@hosts)
	{
		# If the InstallMethod is already set to a value that doesn't
		# match the default, don't reset it.
		if ($$ref_node_hash{$host}{'InstallMethod'} ne "warewulf" 
			&& $$ref_node_hash{$host}{'InstallMethod'} ne "you")
		{
			$$ref_node_hash{$host}{'InstallMethod'}=$inst_method;
		}
	}

	return $inst_method;
}

#--------------------------------------------------------------------------------

=head3 	run_rmtCMD
	Run makedhcp command on Install Servers. If Install Server is not provided, run command 
	on Management Server.
    
    Notes:
	There is no output if $ENV{'SILENCE'} is set.	
=cut

#--------------------------------------------------------------------------------
sub run_rmtCMD
{
    my ($class, $rmt_cmd) = @_;

    my $ref_MgtNodes       = DCUtils->get_nodes_with_no_is;
    my @MgtNodes           = @$ref_MgtNodes;
    my $ref_InstallServers = DCUtils->get_install_servers;
    my %InstallServers     = %$ref_InstallServers;
    my ($command, $command2);
    my ($cmd,     $cmd2);
    my ($ms_hostname, $ms_ip, $no_dsh);
    
    if (@MgtNodes)
    {
        foreach my $hostname (@MgtNodes)
        {
            ($ms_hostname, $ms_ip) =
              NetworkUtils->getHost_rveg(
                                    $::nodehash{$hostname}{"ManagementServer"});
            if ($::GETHOST_RC != $::OK)
            {
                MessageUtils->message('I', 'IMsgFailResMS', $hostname);
                next;
            }
            $InstallServers{$ms_hostname}{"InstallDir"} = "/csminstall";
            push @{$InstallServers{$ms_hostname}{"NodesToBeInstalled"}},
              $hostname;
        }
    }

    $ENV{DSH_LIST} = '';    #clear dsh env.

    # >>> check reachability to the Install Servers >>>
    $ENV{'DSH_REMOTE_CMD'} = DCUtils->getRemoteShell;
    my @installServers    = keys %InstallServers;
    
    my @checkServers;
    foreach my $is (@installServers)
    {
		# in case IS is defined as the IP of MS.
		my ($is_hostname, $trash )= NetworkUtils->getHost_rveg($is);

        # only check reachability of non-MS ISs.
        if($is_hostname ne $ms_hostname)
        {
            push @checkServers, $is;
        }
    }
    my $refISNotReachable = NetworkUtils->checkRSReachability(\@checkServers);
    if ($refISNotReachable != 0)
    {
        if (scalar @$refISNotReachable > 0)
        {
            my $isvrs = join ",", @$refISNotReachable;
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgISDown', $isvrs);
        }
    }

    foreach my $is (keys %InstallServers)
    {
        my $dir           = $InstallServers{$is}{"InstallDir"};
        my @orig_nodelist = @{$InstallServers{$is}{"NodesToBeInstalled"}};
	
		# If CSMVersion >= 1.6.0,  we support makedhcp command with --no-dhcpd-restart 
		# and --only-dhcpd-restart option for optimization
		my $no_dhcpd_restart;
		my $only_dhcpd_restart;
		if (ArchiveUtils->testVersion($InstallServers{$is}{"InstallCSMVersion"}
										, ">=" 
										, "1.6.0")) 
		{
			$no_dhcpd_restart = "--no-dhcpd-restart";
			$only_dhcpd_restart = "--only-dhcpd-restart";
		}
		else
		{
			$no_dhcpd_restart = "";
			$only_dhcpd_restart = "";
		}

		# in case IS is defined as the IP of MS.
		my ($is_hostname, $trash )= NetworkUtils->getHost_rveg($is);
        if($is_hostname eq $ms_hostname)
        {
            $no_dsh=1;
        }
        else
        {
            $no_dsh=0;
        }

        my $divide = 100;    # max number of nodes for each division
        my @sublist;
        my @newarray;
        my @nodelist;
        my ($start_index, $end_index, $nodestring);

        my $count = 0;
        my $times = int(scalar(@orig_nodelist) / $divide);
        while ($count <= $times)
        {
            $start_index = $count * $divide;
            $end_index   =
                ((scalar(@orig_nodelist) - 1) < (($count + 1) * $divide - 1))
              ? (scalar(@orig_nodelist) - 1)
              : (($count + 1) * $divide - 1);
            @sublist  = @orig_nodelist[$start_index .. $end_index];
            @newarray = ();
            @nodelist = ();
            $command  = "";
            $command2 = "";
            $cmd      = "";
            $cmd2     = "";

            foreach my $node (@sublist)
            {
                my @vals = split ',|\s', $node;
                push @nodelist, @vals;
            }

            if ($::RUN_FOR_NODE)
            {

                if ($::MAC)
                {
                    my $nl = join(',', @nodelist);
                    $command =
                      "export CSMINSTALL_ROOT=$dir; $rmt_cmd --mac $nl $no_dhcpd_restart";
                }
                elsif ($::UUID)
                {
                    my $nl = join(',', @nodelist);
                    $command =
                      "export CSMINSTALL_ROOT=$dir; $rmt_cmd --uuid $nl $no_dhcpd_restart";
                }
                else
                {

                    # neither of --uuid and --mac option is set by user.
                    # use the method decided by node attributes.
                    my @mac_nodes  = ();
                    my @uuid_nodes = ();
                    foreach my $node (@nodelist)
                    {
                        if ($::nodehash{$node}{"InstallAdapterMacaddr"})
                        {

                            # so MAC method takes higher priority than UUID.
                            push @mac_nodes, $node;
                        }
                        elsif ($::nodehash{$node}{"UUID"})
                        {
                            push @uuid_nodes, $node;
                        }
                        else
                        {

                            # no MAC nor UUID. okay only for deleting a node.
                            if ($rmt_cmd !~ /--delete/)
                            {
                                MessageUtils->message('E', 'EMsgNoMACandUUID',
                                                      $node);
                            }
                            else
                            {

                                # when deleting node, just put it in mac nodes.
                                # logic in makedhcp will try both mac and uuid method.
                                push @mac_nodes, $node;
                            }

                        }
                    }

                    if (scalar(@mac_nodes))
                    {
                        my $nl = join(',', @mac_nodes);
                        $command =
                          "export CSMINSTALL_ROOT=$dir; $rmt_cmd --mac $nl $no_dhcpd_restart";
                    }
                    if (scalar(@uuid_nodes))
                    {
                        my $nl = join(',', @uuid_nodes);
                        $command2 =
                          "export CSMINSTALL_ROOT=$dir; $rmt_cmd --uuid $nl $no_dhcpd_restart";
                    }
                }

            }
            else
            {
                $command = "export CSMINSTALL_ROOT=$dir; $rmt_cmd";
            }

            if ($no_dsh)
            {

                # execute directly on management server.
                $cmd  = $command;
                $cmd2 = $command2;
            }
            else
            {

                # execute over dsh when install server is used.
                if ($command)
                {
                    $cmd = "$::DSH -n $is '$command'";
                }
                if ($command2)
                {
                    $cmd2 = "$::DSH -n $is '$command2'";
                }
            }

            if ($cmd)
            {
                my $output = NodeUtils->runcmd($cmd, 1);
                $::BOOT_EXIT = 1 if $::RUNCMD_RC;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'I', 'IMsgShow_Output', $output)
                  if !$ENV{'SILENCE'};
            }
            if ($cmd2)
            {
                my $output = NodeUtils->runcmd($cmd2, 1);
                $::BOOT_EXIT = 1 if $::RUNCMD_RC;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'I', 'IMsgShow_Output', $output)
                  if !$ENV{'SILENCE'};
            }

            $count++;
        }

        # restart dhcpd server only after all the nodes on this Install Server
        # have been removed if CSMVersion >=1.6.0
		if ($only_dhcpd_restart ne "")
		{
				if ($no_dsh)
				{

						# execute directly on management server.
						$cmd = "$::MAKEDHCP $only_dhcpd_restart";
				}
				else
				{

						# execute over dsh when install server is used.
						$cmd =
								"$::DSH -n $is '$::MAKEDHCP $only_dhcpd_restart'";
				}
				my $output = NodeUtils->runcmd($cmd, 1);
				$::BOOT_EXIT = 1 if $::RUNCMD_RC;
				MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
								'csminstall', 'I', 'IMsgShow_Output',
								$output)
						if !$ENV{'SILENCE'};
		}

    }
}

#--------------------------------------------------------------------------------

=head3	chkForErrors
 

	Takes in a node and a ref to a hash table.  If no hash
	is given, it takes the defualt: %::ErrorNodes.
   
	Sees if there are any errors in the status file of the 
	node.  returns 1 if there is an error and 0 if there arent.
	It puts all the error messages in a list and gives it to the 
	hashes status value for the node.  It also defines a {node} 
	{error} field so that it can be distinguished from other values
	in the hash table.

        Notes:

=cut

#--------------------------------------------------------------------------------

sub chkForErrors
{
	my $node = shift;
	my @catStatus;    # concatenated stati.
	my @status;
	my $error      = 0;
	my $statusFile = $::CSMSTATUSDIR . "/" . $node;
	if (-f $statusFile)
	{

		# Reset this, because this could be called more than once
		# and we don't need doubles.
		#if($refNodeHash->{$node}{"Error"}){
		#    # then we've done this before.
		#    print "chkForErrors: $node has been through before\n" if $::DEBUG;
		#    $refNodeHash->{$node}{"Status"} = "";
		#}
		# Get the lines that have errors in them.
		my $cmd = "$::GREP -i error $statusFile";
		chomp((@status) = `$cmd`);
		if ($#status > -1)
		{
			print "chkForErrors: $node has an Error error\n" if $::DEBUG;
			@catStatus = @status;
			$error     = 1;
		}

		# Get the lines that say status:xx where xx != 0
		my $cmd = "grep -ie 'status=[-]*[1-9][0-9]*' $statusFile";
		chomp((@status) = `$cmd`);
		print "chkForErrors: $node stat= status variable: $#status\n"
		  if $::DEBUG;
		if ($#status > -1)
		{
			print "Node $node has a status error\n" if $::DEBUG;
			push @catStatus, @status;
			$error += 1;
		}

	}

	if ($error)
	{
		$::ErrorNodes{$node}{"IStatus"} = \@catStatus;
		$::ErrorNodes{$node}{"Error"}   = "true";
	}
	return $error;
}

#--------------------------------------------------------------------------------

=head3	SpecialCases

	For now it just sees if the -i found any errors.  It will put these in
	the $nodes field if we are to print all the nodes.

        Notes:

=cut

#--------------------------------------------------------------------------------

sub SpecialCases
{
	my $nodes = shift;
	my $count;

	#First special case:
	# Nodes that said they were installed, but had
	# errors.  If we are to print all the nodes, then
	# we should print one error message of this node, so
	# that it is listed with everything else.
	# This will only happen if the default case is done.
	if ($::ALLSTATUS)
	{
		foreach (keys %::ErrorNodes)
		{
			print "Special Node: $_\n" if $::DEBUG;
			$nodes->{$_}{"Status"} = $::ErrorNodes{$_}{"Status"};
			$count++;
		}
	}
	return ($count, $nodes);
}

#--------------------------------------------------------------------------------

=head3	getLastEntry

	Takes in a node and a reference to a hash list.  Puts the value in the
	hash list and then returns 1 if it got the last entry, or 0 if it didnt.

        Notes:

=cut

#--------------------------------------------------------------------------------

sub getLastEntry
{
	my $node   = shift;
	my $status = 0;
	unless ($node) {
		die MessageUtils->message('E2', 'EMsgNoNodeDefined_error');
	}
	my $statusFile = $::CSMSTATUSDIR . "/" . $node;
	if (-f $statusFile)
	{
		chomp($status = (split(": ", `$::TAIL -n 1 $statusFile`))[1]);
		print "getLastEntry1: $node : $status\n" if $::DEBUG;

		# strip the period.
		$status =~ /(.*)\.$/;

		# if there wasn't a period in the first place, do nothing.
		if ($1) { $status = $1 }
		print "getLastEntry2: $node : $status\n" if $::DEBUG;
		MessageUtils->message('V', 'IMsgNode_status', $status);
	}
	return ($status);
}

#--------------------------------------------------------------------------------

=head3	getStatus

        Notes:

=cut

#--------------------------------------------------------------------------------

sub getStatus
{
	my ($pkg,$nodes) = @_;
	foreach my $node (keys %$nodes)
	{

		# For each node, get the last entry and if it isn't
		# "installed, then put it in the common node group.
		my $status = &getLastEntry($node);
		if ($status)
		{
			my $statusFile = $::CSMSTATUSDIR . "/" . $node;

			# This will tell us the last known install status
			$nodes->{$node}{"IStatus"} = $status;
			@{$nodes->{$node}{"History"}} = `cat $statusFile`;
		}

		# If getLastEntry returns 0, then the status file wasn't
		# found, so we can say it is not installed.
		else
		{
			my $mode = $nodes->{$node}{"Mode"};
			if ($mode eq "PreManaged")
			{    # premanaged are not installed
				$nodes->{$node}{"IStatus"} = "Not Installed";
			}
			elsif ($mode eq "Installing" || $mode eq $::CSMLITE)
			{
				$nodes->{$node}{"IStatus"} = $mode;
			}
			elsif ($mode eq "Managed")
			{

				# Node is already installed, but for some reason doesn't have
				# a status file.
				$nodes->{$node}{"IStatus"} = "Installed";
			}
			elsif($ENV{'DC_ENVIRONMENT'} ne "CSM")
			{
				$nodes->{$node}{"IStatus"} = "Not Installed";
			}		
			else
			{

				# hmmm... what's this case?  Unknown?
				print "hmmm...don't know the status...of $node\n";
			}
		}

		# check to see if there are errors even through it said it was installed.
		if (   (&chkForErrors($node))
			&& ($nodes->{$node}{'IStatus'} eq "Installed"))
		{
			my $msg = "$node said it was installed, but ";
			$msg .= "there were errors in the status file\n";
			push @::Warnings, $msg;
		}
	}
	return $nodes;
}

#--------------------------------------------------------------------------------

=head3	printStatus

	printStatus will see if there are any nodes.  If there are, it will pass
	it along to either printError (if -e is signaled), printLong (if -l is signaled),
	or printShort by default. 

        Notes:

=cut

#--------------------------------------------------------------------------------

sub printStatus
{
	my ($pkg,$current_status,$status_history) = @_;

	#unless((keys %$nodes)>0){
	#	MessageUtils->message('I','IMsgNodeNodesCategory');
	#	exit;
	#	return;
	#    }
        if ( $::ERROR)
        {
            &printError;
        }   
    
        if ( $::LONG)
        {
            &printStatusLong($current_status,$status_history);
        }
        else
        {
            &printStatusShort($current_status);
        }

}

#--------------------------------------------------------------------------------

=head3	printError

	Prints all the errors found in the status file of a node, for example:

	node leapNode

	 -------------
	Tue Feb 29 08:22:01 EDT 2002: Invalid time stamp

        Notes:

=cut

#--------------------------------------------------------------------------------

sub printError
{

	#my $nodes = shift;
	foreach my $node (keys %::ErrorNodes)
	{
		MessageUtils->message('I', 'IMsgNodeDisplay', $node);
		my $my_str1 = "-" x length($node);
		$my_str1 = $my_str1 . "------";
		MessageUtils->message('I', 'IMsgShow_Output', $my_str1);
		foreach (@{$::ErrorNodes{$node}{"IStatus"}})
		{
			MessageUtils->message('I', 'IMsgShow_Output', $_);
		}
		MessageUtils->message('I', 'IMsgShow_Output', '\n');
	}
}

#--------------------------------------------------------------------------------

=head3	printStatusLong

	Prints a long output of the status.

	Example:
		Node  c5bn03.ppd.pok.ibm.com
		----------------------------
		Fri Oct 12 17:02:04 EDT 2001: Full Install Initiated
		Fri Oct 12 17:02:12 EDT 2001: Rebooting to Install Node
		Fri Oct 12 13:12:13 EDT 2001: Kickstart RPM Installation Complete.
		
        Notes:

=cut

#--------------------------------------------------------------------------------

sub printStatusLong
{
	my $nodes = shift;
	my $nodes_history = shift;
	foreach (sort keys %$nodes)
	{
		my $node       = $_;
		my $status     = $nodes->{$node}{"IStatus"};
		my @history;
	        if( defined( $$nodes_history{$node} ) )
		{
			@history = @{$$nodes_history{$node}};
		}
		#flag used to tell if this node is to be displayed.  0: no display    1: display
		my $displayFlag = 0;    
	
		# no -i, -u, -e option is set, display all node
		# if only -e is set, donnt display any node
		if ( !$::ERROR && !$::INODES && !$::UNODES )
  	        {
  	            $displayFlag = 1
  	        }
	
		if ($::INODES)
      		{   # -i set, show nodes with status "Installed" 
       			if ( ($status eq "Installed" ) || ($status eq "$::CSMLITE" ) )
        		{
             	 		$displayFlag = 1;
          		}            
      		}	
	
		if ($::UNODES)
		{   # -u set, show only nodes WITHOUT status "Not Installed" or "Installing" or /error/i or /failed/i
		        unless (    ( $status eq "Installed") 
		                 || ( $status eq "MinManaged") 
			         ||  grep( /error/i,  $status)
		                 ||  grep( /failed/i, $status)	
                               )
                        {
                            $displayFlag = 1;
	                }
		}
	
		if ($displayFlag == 0)
		{
		    next;
	        }

		MessageUtils->message('I', 'IMsgNodeDisplay', $node);
		my $my_str1 = "-" x length($node);
		$my_str1 = $my_str1 . "------";
		MessageUtils->message('I', 'IMsgShow_Output', $my_str1);

		unless (@history)
		{
			MessageUtils->message('I', 'IMsgNoStatusFile');
			next;
		}
		else
		{
			foreach my $line (@history)
			{
				MessageUtils->message('I', 'IMsgShow_Output', $line);
			}
		}
		MessageUtils->message('I', 'IMsgShow_Output', '\n');
	}
}

#--------------------------------------------------------------------------------

=head3	printStatusShort


	prints the status of nodes in abbreviated form, taking
	the last input of the status file and putting it together.
	There is also a -c option that still has bugs, but will
	print the nodes in colors.

	Example:

	Node                    Mode          Status
	---------------------------------------------------------
	c5bn02.ppd.pok.ibm.com  Managed       Installed
	c5bn03.ppd.pok.ibm.com  Installing    Kickstart Post-Install Complete
	c5bn04.ppd.pok.ibm.com  Installing    Kickstart Post-Install Complete
	
        Notes:

=cut

#--------------------------------------------------------------------------------

sub printStatusShort
{
	my $nodes = shift;

	# more information on color changing can be found at
	# http://www.linuxgazette.com/issue65/padala.html
	my ($coOnCyan, $coOnMag, $coOnGreen, $coOnRed, $coOff);
	$coOnCyan  = sprintf("%c[%d;%d;%dm", 0x1B, 1, 36,     40);
	$coOnMag   = sprintf("%c[%d;%d;%dm", 0x1B, 0, 35,     40);
	$coOnGreen = sprintf("%c[%d;%d;%dm", 0x1B, 0, 32,     40);
	$coOnRed   = sprintf("%c[%d;%d;%dm", 0x1B, 0, 31,     40);
	$coOff     = sprintf("%c[%d;%d;%dm", 0x1B, 0, 7 + 30, 40);
	MessageUtils->message('I', 'IMsgShow_Output',
						'Node                    Mode                  Status');
	MessageUtils->message(
		'I',
		'IMsgShow_Output',
		'--------------------------------------------------------------------------'
	);

	foreach my $node (sort keys %$nodes)
	{

		my $mode   = $nodes->{$node}{"Mode"};
		my $status = $nodes->{$node}{"IStatus"};

		#flag used to tell if this node is to be displayed.  0: no display    1: display
		my $displayFlag = 0;    
	
		# no -i, -u, -e option is set, display all node
		# if only -e is set, donnt display any node
		if ( !$::ERROR && !$::INODES && !$::UNODES )
	        {
	            $displayFlag = 1
	        }
	
		if ($::INODES)
        	{   # -i set, show nodes with status "Installed" 
            		if ( ($status eq "Installed" ) || ($status eq "$::CSMLITE") )
            		{
            		        $displayFlag = 1;
           	 	}            
        	}	
	
		if ($::UNODES)
		{   # -u set, show only nodes WITHOUT status "Not Installed" or "Installing" or /error/i or /failed/i
		        unless (    ( $status eq "Installed") 
		                 || ( $status eq "MinManaged") 
			         ||  grep( /error/i,  $status)
		                 ||  grep( /failed/i, $status)	
                               )
                        {
                            $displayFlag = 1;
	                }
		}
	
		if ($displayFlag == 0)
		{
			next;
		}

		if (($status eq "Not Installed") && $::COLOR)
		{
			print $coOnRed;
		}
		if (($status eq "Installed") && $::COLOR)
		{
			print $coOnGreen;
		}

		format STDOUT =
@<<<<<<<<<<<<<<<<<<<<<< @<<<<<<<<<<<<<<<<<<<< @<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
$node,                            $mode,      $status
.
		write;
		if ($::COLOR)
		{
			print $coOff;
		}
	}
	print "\n\n";
}

#--------------------------------------------------------------------------------

=head3	printWarnings

        Notes:

=cut

#--------------------------------------------------------------------------------

sub printWarnings
{
	if ($#::Warnings > -1)
	{
		MessageUtils->message('I', 'IMsgPrintWarning');
		foreach (sort @::Warnings)
		{

			#print "$_\n";
			MessageUtils->message('I', 'IMsgShow_Output', $_);
		}
	}
}
#--------------------------------------------------------------------------------

=head3	copyStatus	

        Notes:
		Copy node status file from it's install server.

=cut

#--------------------------------------------------------------------------------

sub copyStatus
{
	my ($pkg,$nodes)=@_;
	my %nodehash=%$nodes;
	
	foreach my $node (keys %nodehash)
	{
		if($nodehash{$node}{'InstallOSName'} eq "AIX")
		{
			delete $nodehash{$node};
			next;
		}
		my $status=getLastEntry($node);
		if($status eq "Installed" || $status eq "Install - timedout." || $status eq "Install Failed" 
			|| $status eq "Updatenode is running" || $status =~ m/Updatenode\ failed/ )
		{
			delete $nodehash{$node};
		}	
	}
	
	my %InstallServers;
	if(keys %nodehash)
	{
		my ($ref_MgtNodes, $ref_InstallServers) = DCUtils->getInstallServers(\%nodehash);    
		%InstallServers = %$ref_InstallServers;
	}
	
	my @installServers;
	foreach my $is (keys %InstallServers)
	{
		if ($InstallServers{$is}{isGrp})
		{
			my @group = split(',',$is);
			my @servers_without_dir=map { s/:.*//;$_} @group;	
            push @installServers, @servers_without_dir;
		}
		else
		{
			push @installServers, $is;
		}
	}

	my @badServers;
	if(@installServers)
	{
		my $refISNotReachable = NetworkUtils->checkRSReachability(\@installServers);
		if ($refISNotReachable != 0)
		{
			@badServers=@$refISNotReachable;
		}
	}

	foreach my $is (keys %InstallServers)
	{
			my $dir=$InstallServers{$is}{"InstallDir"};
            my $cmd;
			my @nodelist = @{$InstallServers{$is}{"NodesToBeInstalled"}};
			my $statusDir="$dir/csm/status";
			if ($InstallServers{$is}{isGrp})
			{
			    my @group = split(',',$is);
			    my @servers_without_dir=map { s/:.*//;$_} @group;	
					my @filelist;
                    foreach my $node (@nodelist)
					{
						my $file="$statusDir/$node";
						push @filelist,$file;
					}
					my @servers;
					foreach my $server (@servers_without_dir)
					{
						if(!@badServers || !grep(/$server/,@badServers))
						{
							push @servers,$server;	
						}	
					}	
					if(@servers)
					{
						my $serverlist=join(',',@servers);
						$cmd="/opt/csm/bin/dcp -PR -n $serverlist $statusDir /tmp";
						NodeUtils->runcmd($cmd,-1);
						foreach my $server (@servers)
						{
							$cmd="$::MV -f /tmp/status._$server/* $::CSMSTATUSDIR";
							NodeUtils->runcmd($cmd,-1);	
							rmtree("/tmp/status._$server",$::VERBOSE,1);
						}
					}
			}
			else
			{
				if(!@badServers || !grep(/$is/,@badServers))
				{
					$cmd="/opt/csm/bin/dcp -PR -n $is $statusDir /tmp";
					NodeUtils->runcmd($cmd,-1);
					foreach my $node (@nodelist)
					{
						$cmd="$::MV -f /tmp/status._$is/$node $::CSMSTATUSDIR";
						NodeUtils->runcmd($cmd,-1);
					}
					rmtree("/tmp/status._$is",$::VERBOSE,1);
				}
		
			}
	}	
	
}

sub get_os_img_dirname
{
    my ($pkg, $hostname, $csmroot) = @_;

    my $osname = $::__TARGET_NODE_HASH{$hostname}{'InstallOSName'};
    return if ($osname eq "AIX");

    my $distro_name =
      $::__TARGET_NODE_HASH{$hostname}{'InstallDistributionName'};
    my $distro_ver =
      $::__TARGET_NODE_HASH{$hostname}{'InstallDistributionVersion'};
    my $arch     = $::__TARGET_NODE_HASH{$hostname}{'InstallPkgArchitecture'};
    my $svclevel = $::__TARGET_NODE_HASH{$hostname}{'InstallServiceLevel'};
    my ($os_img_dir, $distro_top);

    $distro_top = "$csmroot/$osname/$distro_name/$distro_ver/$arch";

    if ($distro_name =~ /SLES/)
    {
        $os_img_dir = $distro_top . "/" . $svclevel;
    }
    else
    {
        my $subdir;
        if ($svclevel eq "GA")
        {
            $subdir = $distro_name . $distro_ver . "-" . $arch;
        }
        else
        {
            $subdir = $distro_name . $distro_ver . "-" . $svclevel;
        }
        $os_img_dir = $distro_top . "/" . $subdir;
    }
    if (-e $os_img_dir)
    {
        return $os_img_dir;
    }
}

#------------------------------------------------------------------------------

=head3  check_deploy_attrs

        Check the validity of some attributes that are important to deployment 
		functions.
        Set default values if necessary.

        Arguments:

            $node_info_hash: reference to the node hash;

        Returns:

=cut

#------------------------------------------------------------------------------

sub check_deploy_attrs
{
    my ($pkg, $ref_node_hash, $skip_file_load_flag,$ref_is_hash) = @_;

    # this routine should be called only once
    return if ($::DEPLOY_ATTR_CHECKED && $skip_file_load_flag);

    # if any node is the managment server, remove it from the node hash and 
	# node list.
    my @nodelist = keys(%$ref_node_hash);
    my $ms_node  = NetworkUtils->determineMS(\@nodelist);
    if ($ms_node)
    {
        MessageUtils->message('W', 'EMsgNOT_INSTALLING_NODE', $ms_node);
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'mgmtsvr',
                                     'W', 'IMsgBOTH_ITSELF');

        delete $$ref_node_hash{$ms_node};
        @::NODELIST = keys(%$ref_node_hash);
        if (!scalar(@::NODELIST))
        {
            MessageUtils->message('E1', 'EMsgNO_NODES');
        }

    }

    # verify that the global $::ATTRS for distro version and name, arch 
	# and csmversion have appropriate values.
    NodeUtils->verify_linux_attribute_definitions();

    # now go through the node list to check the attributes of each node
    my @distro_list;
    my @attr_list;
    my @inst_method_list;
    my ($distro_name, $distro_ver, $svclevel, $arch, $csm_version, $distro,
        $inst_method, $set_default_inst_method, $attr);

    my $is_sl;
    my $ms_distro_name = NodeUtils->get_DistributionName();
    my $ms_distro_ver  = NodeUtils->get_DistributionVersion();

    my %server_list_hash;
    foreach my $server (keys %$ref_is_hash)
    {
        foreach my $node (@{$$ref_is_hash{$server}{"NodesToBeInstalled"}})
        {
            push(@{$server_list_hash{$node}{"InstallServerList"}}, $server);
        }
    }
    my %node_hash = %$ref_node_hash;
	my %inst_method_nodelist;
    foreach my $node (keys %node_hash)
    {
	$distro_name = $node_hash{$node}{InstallDistributionName};
	$distro_ver  = $node_hash{$node}{InstallDistributionVersion};
	$arch        = $node_hash{$node}{InstallPkgArchitecture};
	$svclevel    = $node_hash{$node}{InstallServiceLevel};
	$csm_version = $node_hash{$node}{InstallCSMVersion};
	$inst_method = $node_hash{$node}{InstallMethod};
	$set_default_inst_method = 0;
    #------------------------------------------------------
    # Check whether csm command match distribution or not
    # e.g. csmsetupyast can't install RedHat node.
    # e.g. csmsetupks can't install SLES node.
    #------------------------------------------------------
    my ($effective_distro_name) = NodeUtils->getEffectiveDistro($distro_name);
    if ($::command_line =~ /csmsetupks/ && $effective_distro_name =~ /SLES/)
    {
                MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                'E1', 'EMsgDistroNotMatchCmd',"csmsetupks",$effective_distro_name,"csmsetupyast");
    }
    if ($::command_line =~ /csmsetupyast/ && $effective_distro_name =~ /RedHat/)
    {
                MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                'E1', 'EMsgDistroNotMatchCmd',"csmsetupyast",$effective_distro_name,"csmsetupks");

    }

	if(!$csm_version)
	{
	    $node_hash{$node}{'InstallCSMVersion'}= NodeUtils->get_CSMVersion("csm.core");    #requires RPMCMD and LSLPP
	}

	# Use default install method if InstallMethod attr not already set.
	if (! $inst_method)
	{
	    $set_default_inst_method = 1;

	    if ($distro_name =~ /SLES/)
	    {
		$inst_method = "autoyast";
	    }
	    else
	    {
		$inst_method = "kickstart";
	    }
	}
	push @{$inst_method_nodelist{$inst_method}}, $node;

	if (!grep /$inst_method/, @inst_method_list)
	{
	    push @inst_method_list, $inst_method;
	}

	my %pkgdefs =
	ServerUtils->get_pkgdefs(
	    $node_hash{$node}{InstallOSName},
	    $node_hash{$node}{InstallDistributionName},
	    $node_hash{$node}{InstallDistributionVersion},
	    $node_hash{$node}{InstallPkgArchitecture}
	);
	$::pkgdef_node{$node} = \%pkgdefs;
	my $location = 'MgdNode';
	if (!grep(/^$location$/, @{$pkgdefs{'where_supported'}}))
	{
	    MessageUtils->message("E1",
		'EMsgUNSUPPORTED_FULLINST_DISTRO',
		$distro_name, $inst_method);

	}

	# If the InstallMethod attribute is not already set, then set it
	# to the default value based on the distribution.
	if ($set_default_inst_method)
	{	
	    $node_hash{$node}{'InstallMethod'}=$inst_method;
	}

	# now the node attributes are surely correct. load the corresponding 
	# pkgdef file.
	my $pkgdef_id = $distro_name . $distro_ver . $arch . $csm_version;
	if ($pkgdef_id ne $::PKGDEF_ID)
	{
	    $::ATTRS{"InstallServiceLevel"} = $svclevel;
	    ServerUtils->get_pkgdefs('Linux', $distro_name, $distro_ver, $arch,
		"MgdNode", $csm_version);
	    $::ATTRS{"InstallServiceLevel"} = '';
	    $::PKGDEF_ID = $pkgdef_id;
	}

	# Set default InstallServiceLevel if it's null.
	# First try to use the install server's service level.
	# If no install server,then try to use the management server's service 
	# level.
	if (!$svclevel)
	{
	    if(defined($server_list_hash{$node}{"InstallServerList"}))
	    {
		my $attr_ref_server   = $server_list_hash{$node}{"InstallServerList"}[0];
		my $is_distro_name = $$ref_is_hash{$attr_ref_server}{'InstallDistributionName'};
		my $is_distro_ver = $$ref_is_hash{$attr_ref_server}{'InstallDistributionVersion'};
		my $is_svc_level = $$ref_is_hash{$attr_ref_server}{'InstallServiceLevel'};

		#If install server and node have different distro/version, do not set
		if (   ($distro_name eq $is_distro_name)
		    && ($distro_ver eq $is_distro_ver))
		{
		    if($is_svc_level)
		    {
			$svclevel = $is_svc_level;
		    }
		}
	    }	
	    if (!$svclevel)
	    {		
		if (   ($distro_name eq $ms_distro_name)
		    && ($distro_ver eq $ms_distro_ver))
		{

		    # When the node's InstallServiceLevel is not set,
		    # use Management Server's service level to set it.
		    $svclevel = NodeUtils->get_ServiceLevel;
		}
		else
		{

		    # If MS and node have different distribution/version,
		    # the node's default Service Level should be "GA".
		    $svclevel = "GA";
		}
	    }
	    $node_hash{$node}{InstallServiceLevel} = $svclevel;
	}
    # Update InstallServiceLevel attribute of %::NODEHASH.
    if (   defined $::NODEHASH{$node}
        && $::NODEHASH{$node}{'InstallServiceLevel'} ne $svclevel)
    {
        $::NODEHASH{$node}{'InstallServiceLevel'} = $svclevel;
    }

	# setup directory structure for each distro
	$attr = $distro_name . $distro_ver . $svclevel . $arch;
	if (!grep /$attr/, @attr_list)
	{
	    ArchiveUtils->createRPMSdir("Linux", $distro_name, $distro_ver,
		$arch, $svclevel);
	    push @attr_list, $attr;
	}

    }

    my $reset_hash=0;
    if (!keys %::NODEHASH)
    {
        %::NODEHASH = %$ref_node_hash;
        @::NODELIST= keys %::NODEHASH;
        $reset_hash=1;
    }

    # set default template
    foreach $inst_method (@inst_method_list)
    {
		# Templates only exist for certain install methods
		if ($inst_method ne "you")
		{
			my %my_node_hash;
			foreach my $node (@{$inst_method_nodelist{$inst_method}})
			{
				$my_node_hash{$node}=$node_hash{$node};
			}	
			NodeUtils->setNodeTemplate($inst_method, \%my_node_hash);
			foreach my $node (@{$inst_method_nodelist{$inst_method}})
			{
				$node_hash{$node}=$my_node_hash{$node};
			}
		}
    }

    if ($reset_hash)
    {
        %::NODEHASH = %$ref_node_hash;
		@::NODELIST= keys %::NODEHASH;    
	}

    # copy templates.
    foreach $inst_method (@inst_method_list)
    {
		# Templates only exist for certain install methods
		if ($inst_method ne "you")
		{
			NodesetUtils->copy_templates($inst_method, \@{$inst_method_nodelist{$inst_method}}, $ref_is_hash);
		}

    }

    $::DEPLOY_ATTR_CHECKED = 1;

}


#--------------------------------------------------------------------------------

=head3 start_logging

	Wrap-up of MessageUtils->start_logging
=cut

#--------------------------------------------------------------------------------
sub start_logging
{
    my ($class, $logfile,$quiet_flag) = @_;
	
	if($ENV{'DC_ENVIRONMENT'} eq "CSM")
	{
		MessageUtils->append_logging($logfile,1);
	}
	else
	{
		MessageUtils->start_logging($logfile,$quiet_flag);
	}	
    
}	

#--------------------------------------------------------------------------------

=head3 stop_logging

	Wrap-up of MessageUtils->stop_logging

=cut

#--------------------------------------------------------------------------------
sub stop_logging
{
    my ($class, $quiet_flag) = @_;
	
	if($ENV{'DC_ENVIRONMENT'} eq "CSM")
	{
		return;
	}
	else
	{
		MessageUtils->stop_logging($quiet_flag);
	}	
    
}	
#--------------------------------------------------------------------------------

=head3 check_req_attrs 

	Check if all the required nodeinfo attributes is provided.
	
	Arguments:
		$node_attrs	
		$server_attrs

	Returns

=cut

#--------------------------------------------------------------------------------
sub check_req_attrs
{
	my ($pkg,$node_info_hash,$node_attrs,$server_info_hash,$server_attrs,$skip)=@_;
	if($skip || !$ENV{'DC_ENVIRONMENT'} || $ENV{'DC_ENVIRONMENT'} eq "CSM")
	{
		return;
	}	
	foreach my $node (keys %$node_info_hash)
	{
		my $type="node";
		my @missing_attrs;	
		my $missing_attrs;	
		my $hw_missing;
		foreach my $attr (@$node_attrs)
		{
			if($attr eq "InstallAdapterMacaddr" || $attr eq "UUID")
			{
				if( (!defined($$node_info_hash{$node}{'InstallAdapterMacaddr'}) 
								|| !$$node_info_hash{$node}{'InstallAdapterMacaddr'})
					&& (!defined($$node_info_hash{$node}{'UUID'}) 
								|| !$$node_info_hash{$node}{'UUID'}) )
				{
					$hw_missing=1;
				}
				next;
			}		
			if($attr eq "InstallPkgArchitecture" && $$node_info_hash{$node}{'InstallOSName'} eq "AIX")
			{
				next;
			}	
			if(!defined($$node_info_hash{$node}{$attr}) || !$$node_info_hash{$node}{$attr})
			{		
					push @missing_attrs,$attr;
			}
		}
		if($hw_missing)
		{
			push @missing_attrs,"InstallAdapterMacaddr/UUID";
		}
		if(@missing_attrs)
		{
			$missing_attrs=join ',',@missing_attrs;
			#print "The nodeinfo attributes of $type $node are missing: $missing_attrs.\n";
          	MessageUtils->message('E2', 'EMsgMissingAttr', $type ,$node,$missing_attrs);
		}		
		if($$node_info_hash{$node}{"InstallOSName"} ne "Linux" ) 
        	{
			MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 
				'E2', 'EMsgInvalidNodeInfo',"InstallOSName",$node);
		}
		if($$node_info_hash{$node}{"InstallOSName"} eq "Linux" &&
			$$node_info_hash{$node}{"InstallDistributionName"} !~ /^RedHat|^SLES/ )
		{
			MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 
				'E2', 'EMsgInvalidNodeInfo',"InstallDistributionName",$node);
		}	
		if(defined($$node_info_hash{$node}{"InstallServiceLevel"}) 
			&& $$node_info_hash{$node}{"InstallServiceLevel"} !~ /^GA$|^QU[1-9]$|^SP[1-9]$/i )
		{
			MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 
				'E2', 'EMsgInvalidNodeInfo',"InstallServiceLevel",$node);
		}	
	}		
	foreach my $server (keys %$server_info_hash)
	{
		my $type="install server";
		my @missing_attrs;	
		my $missing_attrs;	
		foreach my $attr (@$server_attrs)
		{
			if($attr eq "InstallPkgArchitecture" && $$server_info_hash{$server}{'InstallOSName'} eq "AIX")
			{
				next;
			}	
			if(!defined($$server_info_hash{$server}{$attr}) || !$$server_info_hash{$server}{$attr})
			{		
				push @missing_attrs,$attr;
			}
		}
		if(@missing_attrs)
		{
			$missing_attrs=join ',',@missing_attrs;
			#print "The nodeinfo attributes of $type $server are missing: $missing_attrs.\n";
          	MessageUtils->message('E2', 'EMsgMissingAttr', $type ,$server,$missing_attrs);
		}		
	}		
}
#--------------------------------------------------------------------------------

=head3 check_cluster_info 

	Check if the attribute of cluster info hash is valid.
	
	Arguments:
                %cluster_info_hash
	Returns
	        none

=cut

#--------------------------------------------------------------------------------
sub check_cluster_info{
    my (%cluster_info_hash) = @_;
    my $error_msg;
    if($cluster_info_hash{NetworkInstallProtocol} !~ /^nfs$|^http$/i){
        $error_msg .= "NetworkInstallProtocol=$cluster_info_hash{NetworkInstallProtocol} " ;
    }
    if(!-e $cluster_info_hash{RemoteShell}){
        $error_msg .= "RemoteShell=$cluster_info_hash{RemoteShell} ";
    }
    if(!-e $cluster_info_hash{RemoteCopyCmd}){
        $error_msg .= "RemoteCopyCmd=$cluster_info_hash{RemoteCopyCmd} ";
    }
    if($cluster_info_hash{NodeInstallTimeout} !~ /\d+/){
        $error_msg .= "NodeInstallTimeout=$cluster_info_hash{NodeInstallTimeout} ";
    }
    if($cluster_info_hash{InstallMsgPortNum} !~ /\d+/){
        $error_msg .= "InstallMsgPortNum=$cluster_info_hash{InstallMsgPortNum} ";
    }
    if($error_msg){
        MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E2', 'EMsgInvalidClusterInfo',$error_msg);

    }

}
#--------------------------------------------------------------------------------

=head3  split_IS_attr

        parse the InstallServer attribute 	
	
	Arguments:
                $attrIS: the format is node1[:/dir],node2[:/dir]        
	Returns
	        $Dir: Installation directory
                @ISName: An array contain the hostname of install server
=cut

#--------------------------------------------------------------------------------
sub split_IS_attr{
        my ($class,$attrIS) = @_;
        my @ISNameAndDir = split ",", $attrIS;
        my @ISName;
        my @ISDir;
        foreach my $NameDirPair (@ISNameAndDir){
            my ($tmpName,$tmpDir) = split ":" , $NameDirPair;
            push @ISName,$tmpName if $tmpName;
            push @ISDir, $tmpDir if $tmpDir;
        }
        my $Dir = $ISDir[0];
        if (@ISName)
        {
                $Dir ||= '/csmserver';
                return ($Dir, @ISName);
        }
        else
        {
                $Dir ||= '/csminstall';
                return ($Dir, '');
        }
}
#--------------------------------------------------------------------------------

=head3 dealwith_IS_attr 
	
       exclude invalid install server
	Arguments:
                $node_hash_ref:
                $IS_hash_ref:
                $check_reachability: if this is true, all unreachable install server will be excluded
	Returns
                $node_hash_ref:
                $IS_hash_ref:

=cut

#--------------------------------------------------------------------------------
sub dealwith_IS_attr
{
    my ($class, $node_hash_ref, $IS_hash_ref, $check_reachability) = @_;

    # find out all install servers for all nodes
    my @IS_array;
    foreach my $node (keys %$node_hash_ref)
    {
        my ($node_IS_dir, @node_IS_array) =
          DCUtils->split_IS_attr($$node_hash_ref{$node}{InstallServer});

        if (!scalar(@node_IS_array) || ($node_IS_array[0] eq ''))
        {
            next;
        }

        #remove MS from array
        while (my $ms_node = NetworkUtils->determineMS(\@node_IS_array))
        {
            @node_IS_array = grep !/^$ms_node$/, @node_IS_array;
        }

        #remove node itself from array
        @node_IS_array = grep !/^$node$/, @node_IS_array;

        # now add the Install Servers to the overall array
        foreach my $IS (@node_IS_array)
        {
            if (!grep /^$IS$/, @IS_array)
            {
                push @IS_array, $IS;
            }
        }
    }

    # now check the reachability of each install server if required
    if ($check_reachability)
    {
        if (@IS_array && $IS_array[0] ne '')
        {
            my $unreach_list = NetworkUtils->checkRSReachability(\@IS_array);
            if ($unreach_list != 0)
            {
                foreach my $unreach_IS (@$unreach_list)
                {
                    @IS_array = grep !/^$unreach_IS$/, @IS_array;
                    delete $$IS_hash_ref{$unreach_IS};
                }
            }
        }
    }

    # set proper node attributes about install server
    foreach my $node (keys %$node_hash_ref)
    {
        my ($node_IS_dir, @node_IS_array) =
          DCUtils->split_IS_attr($$node_hash_ref{$node}{InstallServer});
        foreach my $IS (@node_IS_array)
        {
            next if (!$IS);
            if (!grep /^$IS$/, @IS_array)
            {
                # remove bad Install Servers
                @node_IS_array = grep !/^$IS$/, @node_IS_array;
            }
        }
        if (@node_IS_array && $node_IS_array[0] ne '')
        {
            $$node_hash_ref{$node}{ISArrayRef} = \@node_IS_array;
            $$node_hash_ref{$node}{ISDir}      = $node_IS_dir;
            my $new_IS_attr;    #format is node1:/dir,node2:/dir
            foreach my $IS_name (@node_IS_array)
            {
                $new_IS_attr .= "$IS_name:$node_IS_dir,";
            }
            chop $new_IS_attr;
            $$node_hash_ref{$node}{InstallServer} = $new_IS_attr;
        }
        else
        {
            $$node_hash_ref{$node}{ISArrayRef}    = \@node_IS_array;
            $$node_hash_ref{$node}{ISDir}         = '/csminstall';
            $$node_hash_ref{$node}{InstallServer} = '';
        }
    }
    return ($node_hash_ref, $IS_hash_ref);
}
#-------------------------------------------------------------------------------

=head3   embed_is_info 

    embed is info hash into the scripts.


=cut

#-------------------------------------------------------------------------------
sub embed_is_info 
{
    my ($pkg, $target_node_hash,$script) = @_;
    my ($hname, $msname, $ISname, $instadaptr, $akbsvrname, $nfsserver);
    my ($ISvalue, $instname);
    my ($server, $dir, $ipaddr);
	my	$new_script= "/csminstall/csm/$script";
	my	$orig_script= "/opt/csm/csmbin/$script";
    
	my ($IS_IP, $ISAKB_IP, $MS_IP);

    # get rid of the old /csminstall/csm/osprereboot
    if (-f $new_script)
    {
    	unless (open(NEWOSPRE, "< $new_script"))
    	{

        	# Couldn't open
        	MessageUtils->messageFromCat(
                                     'csmInstall.cat',     $::MSGMAPPATH,
                                     'csminstall',         'W',
                                     'EMsgCANT_READ_FILE', $new_script
                                    );
        	return $::NOK;
    	}
        while (<NEWOSPRE>)
        {
            #Match attribute=value line
            #The value can be null if the user do want to empty it.
            if(/^\$::ISinfo{\"(.*)\"}=\"(.*)\";$/)
            {
                $::ISinfo{$1}=$2;
            }
        }
		close(NEWOSPRE);
	}

    # for each node add the InstallAdapterHostname and InstallAdapter
    #  need to include all the nodes defined in the database
    #  if this is a performance issue we might have to look at
    # using a condition/response to call this routine!!!!!

    foreach my $node (keys %$target_node_hash)
    {
		if( ( $script eq "osprereboot" && $$target_node_hash{$node}{"InstallMethod"} eq "warewulf" )
			|| ( $script eq "makedisklessnode" && $$target_node_hash{$node}{"InstallMethod"} ne "warewulf" ) )
		{
			next;
		}		
		$hname      = $$target_node_hash{$node}{"Hostname"};
        $msname     = $$target_node_hash{$node}{"ManagementServer"};
        $ISname     = $$target_node_hash{$node}{"InstallServer"};
        $akbsvrname = $$target_node_hash{$node}{"InstallServerAKBNode"};
        $instadaptr =
          $$target_node_hash{$node}{"InstallAdapterHostname"};
        $nfsserver = $$target_node_hash{$node}{"NFSServer"};

        chomp $instadaptr;

        if ($ISname eq "")
        {
            if ($nfsserver)
            {
                ($server, $dir) = split ':', $nfsserver;
                $server =~ s/^\s*//;    # delete leading white space
                $server =~ s/\s*$//;    # delete trailing spaces
                chomp $server;
                chomp $dir;
            }
            if ($nfsserver
                && NetworkUtils->validate_ip($server) ne
                NetworkUtils->validate_ip($msname))
            {
                $ipaddr = NetworkUtils->validate_ip($server);
                if ($ipaddr == -1)
                {
                    MessageUtils->messageFromCat('HCnodecmds.cat',
                                        $::MSGMAPPATH, 'common', 'E',
                                        'EMsgHostnameUnknown', $server);
                    next;
                }

                if ($dir)
                {
                    $ISvalue = $ipaddr . ":" . $dir;
                }
                else
                {
                    $ISvalue = $ipaddr . ":/csmserver";
                }

            }
            elsif ($akbsvrname ne "")
            {

                # If there is no Install Server, InstallServerAKBNode means the
                # second adapter's hostname of MS, MN should mount direcotory
                # 'csminstall' on this IP.
                $ipaddr = NetworkUtils->validate_ip($akbsvrname);
                if ($ipaddr == -1)
                {
                    MessageUtils->messageFromCat('HCnodecmds.cat',
                                    $::MSGMAPPATH, 'common', 'E',
                                    'EMsgHostnameUnknown', $akbsvrname);
                    next;
                }
                $ISvalue = $ipaddr . ":/csminstall";
            }
            else
            {
                $ipaddr = NetworkUtils->validate_ip($msname);
                if ($ipaddr == -1)
                {
                    MessageUtils->messageFromCat('HCnodecmds.cat',
                                        $::MSGMAPPATH, 'common', 'E',
                                        'EMsgHostnameUnknown', $msname);
                    next;
                }
                $ISvalue = $ipaddr . ":/csminstall";
            }

        }
        else
        {

            #maybe install server is a nodegrp
            #then akbsvrname have no effect , and choose a node from the group list.
            my $is_node = DCUtils->getNodeFromGroup($ISname, $hname);
            if ($is_node)
            {

                #do not care a akbsvrname if the install server is nodegrp
                $akbsvrname = "";
                ($server, $dir) = split ':', $is_node;
            }
            else
            {
                ($server, $dir) = split ':', $ISname;
                $server =~ s/^\s*//;    # delete leading white space
                $server =~ s/\s*$//;    # delete trailing spaces
                chomp $server;
                chomp $dir;

            }
            if ($akbsvrname)
            {
                $ipaddr =
                  ServerUtils->dshGetNodeAddr($akbsvrname, $server);
                if ($ipaddr == -1)
                {
                    MessageUtils->messageFromCat(
                             'csmInstall.cat',            $::MSGMAPPATH,
                             'csminstall',                'E',
                             'EMsgHostnameUnknownOnNode', $akbsvrname,
                             $server
                                                );
                    next;
                }
            }
            else
            {
                $ipaddr = NetworkUtils->validate_ip($server);
                if ($ipaddr == -1)
                {
                    MessageUtils->messageFromCat('HCnodecmds.cat',
                                        $::MSGMAPPATH, 'common', 'E',
                                        'EMsgHostnameUnknown', $server);
                    next;
                }
            }

            $ISAKB_IP = $ipaddr;

            if ($dir eq "")
            {
                (undef, $MS_IP) = NetworkUtils->getHost($msname);
                (undef, $IS_IP) = NetworkUtils->getHost($server);

                # if the install server is the management server
                # then use /csminstall - otherwise use /csmserver.
                if ($ISAKB_IP eq $MS_IP || $IS_IP eq $MS_IP)
                {
                    $ISvalue = $ipaddr . ":/csminstall";
                }
                else
                {
                    $ISvalue = $ipaddr . ":/csmserver";
                }

            }
            else
            {
                $ISvalue = $ipaddr . ":/$dir";
            }

        }

        $::ISinfo{$hname}=$ISvalue;
  	}	  
    # open the existing osprereboot file in /opt/csm/csmbin
    unless (open(ORIGOSPRE, "< $orig_script"))
    {

        # Couldn't open
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',     $::MSGMAPPATH,
                                     'csminstall',         'W',
                                     'EMsgCANT_READ_FILE', $orig_script
                                    );
        return $::NOK;
    }

    # create a new /csminstall/csm/osprereboot
    unless (open(NEWOSPRE, "> $new_script"))
    {

        # Couldn't open
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',      $::MSGMAPPATH,
                                     'csminstall',          'W',
                                     'EMsgCANT_WRITE_FILE', $new_script
                                    );
        return $::NOK;
    }

    # set the permissions on the new file
    #NodeUtils->runcmd("$::CHMOD 755 $new_script", 0);
    chmod 0755, $new_script;

    #  Copy the installed osprereboot, line by line, to a new
    #     osprereboot script in the /csminstall/csm directory.
    #     Insert the node name and install server information
    #     for every node in the cluster.
    while (my $line = <ORIGOSPRE>)
    {

        # if the line contains IMBEDDEDISINFO then start imbedding info
        #         after it
        if (grep /IMBEDDEDISINFO/, $line)
        {
            print NEWOSPRE $line;
         	foreach my $hname (keys %::ISinfo)
			{	
		 		# imbed the information into the osprereboot script in
                #      the /csminstall/csm directory.
                print NEWOSPRE "\$::ISinfo{\"" . $hname . "\"}" . "=" . "\""
                  . $::ISinfo{$hname} . "\"" . ";\n";
    		}    	
	 	}
        elsif (grep /IMBEDDEDTZINFO/, $line)
        {
            print NEWOSPRE $line;
            my $dt = DCUtils->getMSTimeZone;
            print NEWOSPRE "\$ENV{\'TZ\'} = \"$dt\";\n";
        }
        elsif (grep /EMBEDDEDNODEINFO/, $line)
        {
            print NEWOSPRE $line;

			foreach my $node (keys %$target_node_hash)
			{
				my $hname;
				my $ip;
				$hname       = $$target_node_hash{$node}{"Hostname"};
				(undef, $ip) = NetworkUtils->getHost($hname);
		 		# embed the information into the osprereboot script in
                #      the /csminstall/csm directory.
                print NEWOSPRE "\$::NODEINFO{\"" . $ip . "\"}" . "=" . "\""
                  . $hname . "\"" . ";\n";
			}
        }
        else
        {

            # copy the line to the new file
            print NEWOSPRE $line;
        }
    }

    close(NEWOSPRE);
    close(ORIGOSPRE);

    return $::OK;
}

#--------------------------------------------------------------------------------

=head3    you_upgrade

		Upgrade the node using you.  Do not reboot after the upgrade is
		complete.  That is up to installnode.

		Most of this code was copied from run_DSH_Commands from updatenode

        Arguments:
            $ref_node_hash
            $ref_node_list
        Notes:

=cut

#--------------------------------------------------------------------------------
sub you_upgrade
{
    my ($pkg, $ref_node_hash, $ref_is_hash, $ref_node_list) = @_;
	my %DestNodeHash = %$ref_node_hash;
	my $rc = 0;

	$::ONLINE_UPDATE = "/usr/bin/online_update";

	# Split up nodes by install server and by fanout
	my $csm_fanout = $ENV{'CSM_FANOUT'};
	if ($csm_fanout eq '') {$csm_fanout = $::DEFAULT_FANOUT;}
	if ($csm_fanout <= 0)  {$csm_fanout = scalar(keys %DestNodeHash);}
	my $fan = NetworkUtils->groupNFSnodes(\%DestNodeHash, $csm_fanout);
	
	foreach my $group (keys %$fan)
	{
		my @nodes = @{$$fan{$group}{'ALL'}};
		foreach my $server (keys %{$$fan{$group}{'Server'}})
		{
			my @snodes = @{$$fan{$group}{'Server'}{$server}};

			my $rep_dir;
			if ($server =~ m/:/)
			{	#then there is a directory in there
				my $tmp_dir;
				($server, $tmp_dir) = split ':', $server;

				if ($tmp_dir) { $rep_dir = $tmp_dir; }
				else { $rep_dir = $::SERVER_DIRECTORY; }
			}
			my $MS_Linux_dir = "/csminstall/Linux";
			my $MS_CSM_dir   = "/csminstall/csm";
			if ($rep_dir)
			{
				$MS_Linux_dir =~ s/csminstall/$rep_dir/;
				$MS_Linux_dir =~ s/\/\//\//;
				$MS_CSM_dir =~ s/csminstall/$rep_dir/;
				$MS_CSM_dir =~ s/\/\//\//;
			}
			my (@Linux_nodes, %LinuxAKBNodes);
			foreach my $n (@snodes)
			{
				if (($DestNodeHash{$n}{'InstallOSName'} eq "Linux") and
					($DestNodeHash{$n}{'InstallDistributionName'} eq "SLES") and
					($DestNodeHash{$n}{'InstallServiceLevel'} eq "GA"))
				{
					# SLES nodes cannot be upgraded to a GA level, only to a
					# service level.
					# Print a message for this node, and add the node to a
					# list of nodes that cannot be installed, or had a problem
					# during install setup.  The BADNODES list is used by 
					# installnode.
					push @::BADNODES, $n;
					MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 
										'E', 'EMsgCANT_UPGRADE_TO_SLES_GA', $n);
				}
				elsif ($DestNodeHash{$n}{'InstallOSName'} eq "Linux")
				{
					push @Linux_nodes,    $n;
					if ($DestNodeHash{$n}{'InstallServerAKBNode'})
					{
						my $akbnode = $DestNodeHash{$n}{'InstallServerAKBNode'};
						push @{$LinuxAKBNodes{$akbnode}}, $n;
					}
					else
					{
						push @{$LinuxAKBNodes{$server}}, $n;
					}
				}
				else
				{
					MessageUtils->message('E', 'EMsgUnsupportedOS', $n,
										  $DestNodeHash{$n}{'InstallOSName'});
				}
				
				# Write the monitorintsall status message "OS Upgrade Initiated"
				my $msg = MessageUtils->getMessageFromCat($MSGCAT, $MSGMAPPATH,
								$MSGSET, 'IMsgOS_UPGRADE_INITIATED');
				MessageUtils->write_status($n, $msg, $rc);

				# Write the monitorintsall status message:
				#  "Updating RPMs with online_update"
				$msg = MessageUtils->getMessageFromCat($MSGCAT, $MSGMAPPATH,
								$MSGSET, 'IMsgRUNNING_ONLINE_UPDATE');
				MessageUtils->write_status($n, $msg, $rc);
			}

			MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 
										'I', 'IMsgUPGRADING_YOU_NODES',
										join(' ', @Linux_nodes));

			if (@Linux_nodes)
			{

				foreach my $akbname (keys %LinuxAKBNodes)
				{
					my $akbIP = NetworkUtils->validate_ip($akbname);
					$akbIP = $akbname if $akbIP == -1;

					# get the OSDefs for Linux but don't set the global variables
					my %lnxOSDefs =
					  ServerUtils->get_OSDefs('Linux', '', '', '', '', 1);
					my $dsh_command = "$::DSH -t $::TIMEOUT ";
					my $node_file   =
					  ServerUtils->make_node_list_file( \@{$LinuxAKBNodes{$akbname}});
					#----------------------------------------------
					#  Mount /csminstall on the nodes in this group
					#----------------------------------------------
					$ENV{'DSH_LIST'} = $node_file;
					$dsh_command .= "\"" ;
					$dsh_command .=
					  "$::MKDIR -p $::CSMCLIENTMNTDIR/csm >/dev/null 2>&1; ";
					$dsh_command .=
					  "$::MKDIR $::CSMCLIENTMNTDIR/Linux >/dev/null 2>&1; ";
					$dsh_command .=
					  "$lnxOSDefs{MOUNT} -r $akbIP:$MS_CSM_dir $::CSMCLIENTMNTDIR/csm 2>&1; ";
					$dsh_command .=
					  "$lnxOSDefs{MOUNT} -r $akbIP:$MS_Linux_dir $::CSMCLIENTMNTDIR/Linux 2>&1; ";
					$dsh_command .= "\"" ;
					MessageUtils->messageFromCat('nodecmds.cat', 
												 $::MSGMAPPATH, 'NodeUtils', 
												 'V', 'IMsgCMD', $dsh_command);
					my @local_dsh_results = `$dsh_command`;
					$rc = $? >> 8;
					ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);

					if ($rc != 0)
					{
						# Error executing DSH.
						MessageUtils->message('E', 'EMsgNO_DSH');
					}
					chop @local_dsh_results;

					foreach my $line (@local_dsh_results)
					{
						MessageUtils->message('I', 'IMsgShow_Output', $line);
					}

					#---------------------------------------------
					# Run online_update on the nodes in this group
					#---------------------------------------------
					my $you_basedir;

					my $svclevel = lc($::SERVICE_LEVEL); #convert to lower case
					my $distro_name = lc($::DISTRO_NAME); #convert to lower case
					if ($::SERVICE_LEVEL eq "GA")
					{
						MessageUtils->message('E2', 'EMsgCANT_UPGRADE_TO_SLES_GA',
								  join (" ", @{$LinuxAKBNodes{$akbname}}));
					}
					else
					{
						$you_basedir =
							$::CSMCLIENTMNTDIR . "/Linux"
							. "/$::DISTRO_NAME"
							. "/$::DISTRO_VERSION"
							. "/$::ARCH"
							. "/$::SERVICE_LEVEL"
							. "/unitedlinux-$svclevel";
					}

					my $dsh_command = "$::DSH ";
					my $node_file   =
					  ServerUtils->make_node_list_file(\@nodes);
					$ENV{'DSH_LIST'} = $node_file;
					if ((scalar @nodes == 1) || !($::VERBOSE))
					{   #only one node, or not verbose, do streaming output
						$dsh_command .= "-s ";
					}
					$dsh_command .= "\"";
					$dsh_command .= 
						"$::ONLINE_UPDATE -V -u $you_basedir 2>&1; ";
					$dsh_command .= "\"";
					MessageUtils->messageFromCat('nodecmds.cat', 
												 $::MSGMAPPATH, 'NodeUtils', 
												 'V', 'IMsgCMD', $dsh_command);
					if ((scalar @nodes == 1) || !($::VERBOSE))
					{   #if only one node or not verbose, do streaming output
						open(STREAM, "$dsh_command 2>&1|") 
							or MessageUtils->messageFromCat($MSGCAT, 
								$MSGMAPPATH, $MSGSET, 'E', 'EMsgNO_DSH');
						while (<STREAM>)
						{
							my $line = $_;
							chomp $line;
							my $node = (split ":", $line)[0];
							MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, 
								$MSGSET, 'I', 'IMsgShow_Output', $line);
						}
						close STREAM;

						ServerUtils->close_delete_file($::NODE_LIST_FILE, 
														$node_file);
					}
					else
					{
						my @local_dsh_results = `$dsh_command`;
						$rc = $? >> 8;
						if ($rc != 0)
						{
							# Error executing DSH.
							MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, 
								$MSGSET, 'E', 'EMsgNO_DSH');
						}
						chop @local_dsh_results;
						ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);

						#  Display output, and send to the log
						foreach my $line (@local_dsh_results)
						{
							MessageUtils->message('I', 'IMsgShow_Output', $line);
						}
					}

					#--------------------------------------------
					# Run osprereboot on the nodes in this group.
					#--------------------------------------------
					#
					#
					# Note that we call osprereboot without any arguments.
					# The node's hostname is determined by osprereboot with
					# some help from the NODEINFO hash that is embedded into
					# osprereboot (see DCUtils->make_osprereboot_script and 
					# DCUtils->embed_isinfo)
					#
					foreach my $node (@nodes)
					{
						# Write the monitorinstall status message 
						# "Starting CSM post upgrade"
						my $msg = MessageUtils->getMessageFromCat($MSGCAT, 
										$MSGMAPPATH, $MSGSET, 
										'IMsgSTART_CSM_POST_UPGRADE');
						MessageUtils->write_status($node, $msg, $rc);
					}

					my $dsh_command = "$::DSH ";
					$dsh_command .= "\"";
					$dsh_command .= "$::CSMCLIENTMNTDIR/csm/osprereboot 2>&1;";
					$dsh_command .= "\"";
					MessageUtils->messageFromCat('nodecmds.cat', 
											 $::MSGMAPPATH, 'NodeUtils', 
											 'V', 'IMsgCMD', $dsh_command);
					my $node_file   =
					  ServerUtils->make_node_list_file(\@nodes);
					$ENV{'DSH_LIST'} = $node_file;
					MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
									 'NodeUtils', 'V', 'IMsgCMD', $dsh_command);
					my @local_dsh_results = `$dsh_command`;
					my $rc = $? >> 8;
					ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);

					chop @local_dsh_results;

					foreach my $line (@local_dsh_results)
					{
						MessageUtils->message('I', 'IMsgShow_Output', $line);
					}
					if ($rc != 0)
					{
						# Error executing DSH.
						MessageUtils->message('E$rc', 'EMsgNO_DSH');
					}
					

					foreach my $node (@nodes)
					{
						# Write the monitorinstall status message 
						# "Finished CSM post upgrade"
						my $msg = MessageUtils->getMessageFromCat($MSGCAT, 
										$MSGMAPPATH, $MSGSET, 
										'IMsgFINISH_CSM_POST_UPGRADE');
						MessageUtils->write_status($node, $msg, $rc);
					}

					#------------------------------------------------
					#  Unmount /csminstall on the nodes in this group
					#------------------------------------------------
					my $dsh_command = "$::DSH -t $::TIMEOUT ";
					my $node_file   =
					  ServerUtils->make_node_list_file(\@nodes);
					$ENV{'DSH_LIST'} = $node_file;
					$dsh_command .= "\"";
					$dsh_command .= "$lnxOSDefs{UNMOUNT} $::CSMCLIENTMNTDIR/csm 2>&1; ";
					$dsh_command .=
					  "$lnxOSDefs{UNMOUNT} $::CSMCLIENTMNTDIR/Linux 2>&1;";
					$dsh_command .= "\"";
					MessageUtils->messageFromCat('nodecmds.cat', 
												 $::MSGMAPPATH, 'NodeUtils', 
												 'V', 'IMsgCMD', $dsh_command);
					my @local_dsh_results = `$dsh_command`;
					$rc = $? >> 8;
					ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);

					if ($rc != 0)
					{
						# Error executing DSH.
						MessageUtils->message('E', 'EMsgNO_DSH');
					}
					chop @local_dsh_results;

					foreach my $line (@local_dsh_results)
					{
						MessageUtils->message('I', 'IMsgShow_Output', $line);
					}

				}
				# Need to sleep here, after the nodes have been upgraded, but
				# before the reboot.  Otherwise, if you reboot too quickly
				# after running online_update, the kernel will not load.
				my $sleepTime = $::ENV{"CSM_SLEEP_BEFORE_REBOOT"} ? $::ENV{"CSM_SLEEP_BEFORE_REBOOT"} : 180;
			
				MessageUtils->message('I', 'IMsgWaitingForYOU', $sleepTime);
				sleep ($sleepTime);
			}
		}
	}
}

#-------------------------------------------------------------------------------

=head3    exclude_dev_dirs

	get directory list in vnfs repository that can not be synced

=cut

#-------------------------------------------------------------------------------
sub exclude_dev_dirs 
{
		my ($pkg,$DestNodeHash)=@_;
        my @exclude_dirs;
		foreach my $node (keys %$DestNodeHash)
		{
            my $InstallDistributionName =
              $$DestNodeHash{$node}{'InstallDistributionName'};
            my $InstallDistributionVersion =
              $$DestNodeHash{$node}{'InstallDistributionVersion'};
            my $InstallServiceLevel =
              $$DestNodeHash{$node}{'InstallServiceLevel'};
            my $InstallPkgArchitecture =
              $$DestNodeHash{$node}{'InstallPkgArchitecture'};
            my $InstallMethod = $$DestNodeHash{$node}{'InstallMethod'};
			if($InstallMethod eq "warewulf")
			{
				my $dev="$::VNFSDIR/$InstallDistributionName$InstallDistributionVersion-$InstallServiceLevel-$InstallPkgArchitecture/dev";	
				if(!grep(/$dev/,@exclude_dirs))
				{		
					push(@exclude_dirs,$dev);
				}
			}		
		}
		return \@exclude_dirs;
}		
1;
