#!/bin/bash

# acuppd.sh
#
# Usage:
#    acuppd.sh [ -l LOCK_ID | --lock LOCK_ID ] [ -f FILTER_ID | --filter FILTER_ID ]
#    acuppd.sh -c LOCK_ID | --clean LOCK_ID
#    acuppd.sh -a | --clean-all
#    acuppd.sh -h | --help
#
# Description:
#    This script gathers relevant PA files, logs
#    its progress, and tars everything up, using
#    gzip for compression (-z option of tar).  It
#    may be run in two different ways, locked (run
#    from a script) and not locked (run manually).
#    Additionally, debug mode may be entered to
#    gather additional data, whether we're running
#    locked or unlocked.  Another feature of the
#    script is the ability to skip commands, as
#    specified by the --filter option.
#
#    LOCKED:  It is lockable for a given LOCK_ID.
#    In other words, if it is called again with the
#    same LOCK_ID before cleanup has been requested
#    with said LOCK_ID or globally, it will simply
#    exit with an error RC.  Naming conventions of
#    files will keep themselves seperate from other
#    locks by using the lock id in them.  For example,
#    the archive file will be named acuppd.$LOCK_ID.tgz.
#    Note that the --clean-all option is provided
#    to allow for removing all lock files created by
#    this script.
#
#    UNLOCKED:  If it is run without the --lock option,
#    as it is assumed will be done when manually run, no
#    locking is performed, --clean can not be called,
#    the temporary directories will be removed, but
#    it is the requirement of the caller to then
#    remove the archive file.  If it is manually run
#    again before the previous .tgz file is deleted,
#    the script will remove it before creating the
#    new one.  The archive file will be named
#    acuppd.tgz.
#
#    DEBUG-MODE:  Debug mode may be entered by ensuring
#    the existence of a file in $CONSOLE_PATH called
#    acuppd.debug.  The existence of this file will cause
#    the script to collect additional data.  The absence
#    of this file will cause the script to run normally.
#    The additional debug data includes the running of
#    hmcdebuginfo and collecting its resulting output,
#    as well as the entire contents of /var/log rather
#    than just a few specific files in it.  The acuppd.debug
#    file's contents do not matter, only its existence
#    matters.
#
#    FILTERING:  If it is run with the --filter option,
#    then any command (run or copy) in the configuration
#    file with one of its filters matching the passed
#    FILTER_ID will be skipped.  This is a way to configure
#    a system such that under certain call conditions,
#    less or more data will be collected in order to
#    improve efficiency.
#
# Command Line Parameters:
#    --lock LOCK_ID: locks the script from being run again with LOCK_ID before --clean is called
#    --filter FILTER_ID: skips commands that have FILTER_ID as one of their filters
#    --clean LOCK_ID: removes tar file and the PID lock for LOCK_ID
#    --clean-all: removes all lock and archive files created by this script
#    --help: displays usage information and exits
#
# Return Codes
#    0: normal script termination; no unrecoverable errors
#    1: invalid command line parameters
#    2: could not locate hmcfunctions script file
#    3: script is currently locked by another process for given lock id
#    4: temporary archive directory already exists
#    5: insufficient permissions to create lock file (locked state only)
#    6: could not locate configuration file (acuppd.config)
#
# Module History
#    00  10/15/02   J. Jenks      - Initial Release
#    01  10/16/02   J. Jenks      - Usage changes: --name=NAME is now --name NAME
#    02  11/24/03   J. Jenks      - Updated ". hmcfunctions" call to work on both
#                                   development and production environments
#    03  01/12/04   J. Jenks      - Uses queryFileLocation to find console data files
#    04  03/25/04   J. Jenks      - Added actwcud.dat to the archiving list
#    05  03/26/04   L. Brocious   - Add Unified JVM support; capture "top -bn1" output;
#                                   capture javacore files (from IBM JVM)
#    06  03/30/04   J. Jenks      - Added /tmp/hmc/ud to the archiving list
#    07  04/12/04   J. Jenks      - Removed --name parameter option; archive is always acuppd.tgz
#    08  04/12/04   J. Jenks      - Added --clean option to cleanup archiving files
#    09  04/12/04   J. Jenks      - Implemented locking mechanism so the script can not be
#                                   run multiple times concurrently.  Running the script while
#                                   locked will result in exit with RC=3.  Running the script
#                                   with the --clean parameter will remove the lock.
#    10  05/24/04   K. Schroeder  - Add collection of files in data/core directory.
#    11  11/16/04   P. Callaghan  - Add running of "showTraceBuff all" script.
#    12  11/29/04   J. Jenks      - Modified script to be lockable on a per caller basis,
#                                   or not locked at all for running manually.
#    13  11/29/04   J. Jenks      - Add collection of CONSOLE_PATH/data/rcs/rcsControl.log
#    14  03/14/05   J. Jenks      - Add --clean-all option to remove all locks.
#    15  04/22/05   K. Schroeder  - Fix core file problems.
#    16  05/18/05   K. Schroeder  - Donot run hwinfo to prevent mouse problem.
#    17  06/07/05   J. Jenks      - Add collection of actzzmnd.trm.
#    18  06/07/05   K. Schroeder  - Do not erase core files on cleanup call.
#    19  07/06/05   J. Jenks      - Add collection of bbrzspr.dat.
#    20  07/22/05   M. Jamieson   - Remove collection of iqzdtrac.trm  G7503
#    21  07/28/05   J. Jenks      - Change ps -A call to ps -Afww for parent
#                                   pid and start time collection.
#    22  09/06/05   J. Jenks      - When not in manual mode, ensures temp
#                                   dir is removed before starting.
#    23  09/10/05   J. Jenks      - Add iqyye4.log to the collection list.
#    24  10/05/05   J. Jenks      - Add /tmp/console/xrtr-query*.txt to the collection list.
#    25  10/18/05   J. Jenks      - Several collection changes based on discussions.
#                                    - Command Changes
#                                       - MOD "ps -Afww" -> "ps -AFLlww"
#                                       - ADD "iptables -L"
#                                       - MOD "netstat && netstat -l" -> "netstat -anpoee"
#                                       - ADD "df -h"
#                                    - File Changes
#                                       - ADD "`queryFileLocation iqybrst.trm`iqybrst.trm"
#                                       - ADD "`queryFileLocation actbrst.trm`actbrst.trm"
#                                       - ADD "`queryFileLocation builddate`builddate"
#                                       - ADD "/bom/image.name"
#                                       - ADD "/bom/distro_id"
#                                       - ADD "$CONSOLE_PATH/file/logfile"
#                                       - ADD "$CONSOLE_PATH/file/errorlog"
#    26  10/18/05   J. Jenks      - Add support for debug-mode (ODT H0985).
#    27  10/26/05   J. Jenks      - Set LANG=POSIX (ODT H2915).
#    28  02/17/06   M. Bartoy     - MOD "ls -R /proc" -> "ls -lR /proc"
#    29  02/22/06   J. Jenks      - Major Design Changes:
#                                    - Moved declarations of what gets collected to
#                                      a configuration file (data/acuppd.dat).  Refer
#                                      to it for format documentation!
#                                    - Implemented filtering mechanism.  This allows a
#                                      system to be configured such that acuppd.sh can
#                                      be called to maximize efficiency by skipping
#                                      execution of certain commands.  Refer to the
#                                      configuration file for details.
#    30  07/11/06   J. Jenks      - Renamed configuration file (acuppd.dat -> acuppd.config)
#    31  01/16/07   J. Jenks      - ODT I7588: No longer delete core files from this script on clean call.
#    32  03/26/07   J. Jenks      - Design Changes:
#                                    - Add "move" command support (executes a command to create a file to be moved into the archive).
#                                    - Add "Debug" field support to config file.
#                                    - Move debug commands to config file.
#    33  05/01/07   J. Jenks      - ODT K5591: Lower priority of awk and tar commands (data collection and zipping).
#

# -27
export LANG=POSIX

# -----------
# subroutines
# -----------

# subroutine for outputting help
echo_help() {
   if [ -n "$ERROR" ]; then
      echo "Error: $ERROR" >&2
      echo >&2
   fi
   echo "Usage: archives PA files into gzipped tar (tar -cz)"
   echo "   `basename $0` [ -l LOCK_ID | --lock LOCK_ID ] [ -f FILTER_ID | --filter FILTER_ID ]"
   echo "   `basename $0` -c LOCK_ID | --clean LOCK_ID"
   echo "   `basename $0` -a | --clean-all"
   echo "   `basename $0` -h | --help"
   echo
   echo "Parameters:"
   echo "   --lock LOCK_ID: locks the script from being run again before --clean is called on LOCK_ID"
   echo "   --filter FILTER_ID: skips commands that have FILTER_ID as one of their filters"
   echo "   --clean LOCK_ID: removes tar file, any owned temporary directories, and the PID lock file for LOCK_ID"
   echo "   --clean-all: removes all lock and archive files created by this script"
   echo "   --help: displays this help"
   echo
}

# subroutine for checking if there were errors while collecting data
# and writing closing data to the log file for each command
checkerrors() {
   if [ -s $TIME_FILE ]; then
      cat $TIME_FILE >>$LOG
   fi
   if [ -s $ERROR_FILE ]; then
      echo "ERRORS FOLLOW" >>$LOG
      cat $ERROR_FILE >>$LOG
   else
      echo "SUCCESS" >>$LOG
   fi
   echo >>$LOG
   rm -f $ERROR_FILE $TIME_FILE
}


# -----------------------------------------
# perform parameter collection and checking
# -----------------------------------------

# check for parameters and output help if required
until [ -z "$1" -o -n "$HELP" ]; do
   case "$1" in
      -l | --lock     )
         if [ -n "$CLEAN" ]; then
            HELP="on"
            ERROR="Only --clean, --clean-all, OR --lock may be specified!"
         else
            LOCK="on"
            shift
            LOCKID="$1"
            if [ -z "$LOCKID" ]; then
               HELP="on"
               ERROR="A lock id must be specified with the --lock option!"
            fi
         fi
      ;;
      -f | --filter   )
         shift
         FILTER="$1"
      ;;
      -c | --clean    )
         if [ -n "$LOCK" ]; then
            HELP="on"
            ERROR="Only --clean, --clean-all, OR --lock may be specified!"
         else
            CLEAN="on"
            shift
            LOCKID="$1"
            if [ -z "LOCKID" ]; then
               HELP="on"
               ERROR="A lock id must be specified with the --clean option!"
            fi
         fi
      ;;
      -a | --clean-all    )
         if [ -n "$LOCK" ]; then
            HELP="on"
            ERROR="Only --clean, --clean-all, OR --lock may be specified!"
         else
            CLEAN="on"
            LOCKID="*"
         fi
      ;;
      -h | --help     )
         HELP="on"
      ;;
      *               )
         HELP="on"
         ERROR="Invalid parameter ( $1 )!"
      ;;
   esac
   shift
done

if [ -n "$HELP" ]; then
   echo_help
   if [ -n "$ERROR" ]; then
      exit 1
   else
      exit 0
   fi
fi


# ------------------------------------
# find necessary archiving directories
# ------------------------------------

# attempt to locate hmcfunctions script
if [ -z "$CONSOLE_PATH" ]; then
   CONSOLE_PATH="$HWMCAHOME/"
fi
FOUND=
for FUNCDIR in "" "$CONSOLE_PATH/" "$CONSOLE_PATH/native/scripts/"; do
   . ${FUNCDIR}hmcfunctions >/dev/null 2>&1 && FOUND=true && break 1
done
if [ -z "$FOUND" ]; then
   echo "Error: Could not locate hmcfunctions script file!" >&2
   exit 2
fi

# find locations and names of archiving files
CONFIG_FILE="`queryFileLocation acuppd.config`acuppd.config"
if [ ! -e $CONFIG_FILE ]; then
   echo "Error: Could not locate acuppd.config configuration file!" >&2
   exit 6
fi
if [ -n "$LOCKID" ]; then
   TEMP_DIR="`queryFileLocation acuppd`acuppd.${LOCKID}"
   ARCHIVE_FILE="`queryFileLocation acuppd.tgz`acuppd.${LOCKID}.tgz"
   PID_FILE="`queryFileLocation acuppd.pid`acuppd.${LOCKID}.pid"
   ERROR_FILE="err.${LOCKID}.tmp"
   TIME_FILE="time.${LOCKID}.tmp"
else
   TEMP_DIR="`queryFileLocation acuppd`acuppd"
   ARCHIVE_FILE="`queryFileLocation acuppd.tgz`acuppd.tgz"
   PID_FILE="`queryFileLocation acuppd.pid`acuppd.pid"
   ERROR_FILE="err.tmp"
   TIME_FILE="time.tmp"
fi
TIME_BIN="`which time` -o $TIME_FILE"


# ----------------------------------
# manage script cleaning and locking
# ----------------------------------

# if this is a call to clean up old files, do so and exit
if [ -n "$CLEAN" ]; then
#-31 start
#   #-10 -15 start
#   if [ "$LOCKID" == "pa" ]; then #-18
#      rm -f $CONSOLE_PATH/core* >/dev/null 2>&1
#      rm -f "`queryFileLocation core.123`"* >/dev/null 2>&1
#   fi #-18
#   #-10 -15 end
#-31 end
   rm -f $ARCHIVE_FILE >/dev/null 2>&1
   rm -f $PID_FILE >/dev/null 2>&1
   exit 0
fi

# we are here, so the request is to archive... check for lock if necessary
if [ -n "$LOCK" ]; then
   if [ -e $PID_FILE ]; then
      # script is locked, so output error and exit
      echo "Exiting because script is already running on pid `cat $PID_FILE`!" >&2
      exit 3
   else
      # script is not locked, so attempt to lock it
      if ! echo -n "$$" 2>/dev/null >$PID_FILE; then
         echo "Could not write lock file!  Insufficient permissions (`dirname $PID_FILE`)?" >&2
         exit 5
      fi
   fi
fi

#-15 start
# Scan the console's top-level lib directory for subdirectories to add to LD_LIBRARY_PATH.
dirs=`find $CONSOLE_PATH/lib -type d -mindepth 1 2>/dev/null`
newpath=$LD_LIBRARY_PATH
for dir in $dirs; do
   newpath=$dir:$newpath
done
export LD_LIBRARY_PATH=$newpath
#-15 end

# -------------
# setup archive
# -------------

# create temp dir for storing PA files
if [ -e $TEMP_DIR ]; then
   if [ -z "$LOCKID" ]; then
      echo "Exiting because temporary archive directory already exists!  Is another manual instance running?  If not, then remove directory $TEMP_DIR prior to running again." >&2
      exit 4
   fi
   rm -rf $TEMP_DIR
fi
mkdir $TEMP_DIR
pushd $TEMP_DIR >/dev/null

# determine if we are in debug mode or not
DEBUG="off"
if [ -e "${CONSOLE_PATH}acuppd.debug" ]; then
   DEBUG="on"
fi

# set the time command output format so we don't have to specify it every time
export TIME="Elapsed Time: %E / CPU Usage: %P"

# setup log file
LOG="$TEMP_DIR/acuppd.log"
echo "acuppd log file" >$LOG
echo "---------------" >>$LOG
echo >>$LOG
echo "Started:" >>$LOG
date >>$LOG
echo >>$LOG


# -------------------------
# parse the acuppd.config file
# -------------------------
export CONSOLE_PATH
export DEBUG
export FILTER
export LOG
export ERROR_FILE
export TIME_BIN
export TIME_FILE
export -f queryFileLocation
export -f checkerrors
nice -n 19 awk '
# create global variables
BEGIN {
   CPATH = ENVIRON["CONSOLE_PATH"]
   DEBUG = ENVIRON["DEBUG"]
   FILTER = ENVIRON["FILTER"]
   LOG = ENVIRON["LOG"]
   ERRFILE = ENVIRON["ERROR_FILE"]
   TIME = ENVIRON["TIME_BIN"]
}

# define the actions to perform for "run" lines
$1 == "run" {
   # first check to be sure we have at least 5 fields
   if (NF < 5) {
      print "Error parsing run command from config file on the following line:" >>LOG
      print $0 >>LOG
      print "Less than 5 fields are found!" >>LOG
      print "" >>LOG
      next
   }

   # put together the command and its parameters
   cmd = $4
   for (i = 5; i < NF; i++)
      cmd = cmd " " $i

   # do not execute command if debug mode prevents it
   if (DEBUG == "on" && $2 == "N") {
      print "Skipping command " cmd " because Debug is ON and it is a No-Debug-Only command." >>LOG
      print "" >>LOG
      next
   }
   else if (DEBUG == "off" && $2 == "Y") {
      print "Skipping command " cmd " because Debug is OFF and it is a Debug-Only command." >>LOG
      print "" >>LOG
      next
   }

   # do not execute command if the passed filter applies
   if (FILTER != "" && match($3, FILTER)) {
      print "Skipping command " cmd " because of passed filter \"" FILTER "\"." >>LOG
      print "" >>LOG
      next
   }

   # ensure that the commands output directory exists
   if (system("test -e pa.commands") != 0) {
      if (system("mkdir pa.commands >/dev/null 2>&1") != 0) {
         print "Can not run the following command:" >>LOG
         print $0 >>LOG
         print "Could not create the output directory \"pa.commands\"!" >>LOG
         print "" >>LOG
         next
      }
   }

   # execute the command
   print "Executing command \"" cmd "\" to output file pa.commands/" $NF "." >>LOG
   system(TIME " " cmd " >pa.commands/" $NF " 2>" ERRFILE)
   system("checkerrors")
}

# define the actions to perform for "move" lines
$1 == "move" {
   # first check to be sure we have at least 7 fields
   if (NF < 7) {
      print "Error parsing move command from config file on the following line:" >>LOG
      print $0 >>LOG
      print "At least 7 fields are required!" >>LOG
      print "" >>LOG
      next
   }

   # do not execute command if debug mode prevents it
   if (DEBUG == "on" && $2 == "N") {
      print "Skipping file " $5 "/" $4 " because Debug is ON and it is a No-Debug-Only command." >>LOG
      print "" >>LOG
      next
   }
   else if (DEBUG == "off" && $2 == "Y") {
      print "Skipping file " $5 "/" $4 " because Debug is OFF and it is a Debug-Only command." >>LOG
      print "" >>LOG
      next
   }

   # do not execute command if the passed filter applies
   if (FILTER != "" && match($3, FILTER)) {
      print "Skipping file " $5 "/" $4 " because of passed filter \"" FILTER "\"." >>LOG
      print "" >>LOG
      next
   }

   # get the file input location
   if (index($5, "C") == 1) {
      file = gensub(/C/, CPATH, 1, $5) "/" $4
   }
   else if (index($5, "D") == 1) {
      query = "queryFileLocation " ( ($5 ~ /^D\/.+/) ? substr($5, 3) : $4 )
      query | getline path;
      close(query)
      file = path "/" $4
   }
   else if (index($5, "/") == 1) {
      file = $5 "/" $4
   }
   else {
      print "Error parsing move command from config file on the following line:" >>LOG
      print $0 >>LOG
      print "The \"Path\" field must begin with \"C\", \"D\", or \"/\"!" >>LOG
      print "" >>LOG
      next
   }

   # ensure that the output directory exists
   if (system("test -e " $6) != 0) {
      if (system("mkdir -p " $6 " >/dev/null 2>&1") != 0) {
         print "Error parsing move command from config file on the following line:" >>LOG
         print $0 >>LOG
         print "Could not create the output directory \"" $6 "\"!" >>LOG
         print "" >>LOG
         next
      }
   }

   # put together the command and its parameters and execute it
   cmd = $7
   for (i = 8; i <= NF; i++)
      cmd = cmd " " $i
   print "Executing pre-move command \"" cmd "\"." >>LOG
   rc = system(TIME " " cmd " >" ERRFILE " 2>&1")
   system("checkerrors")
   if (rc != 0) {
      print "Pre-move command failed, so skipping move of " file "." >>LOG
      print "" >>LOG
      next
   }

   # execute the move command
   cmd = "mv " file " " $6
   print "Executing move \"" cmd "\"." >>LOG
   system(TIME " " cmd " >" ERRFILE " 2>&1")
   system("checkerrors")
}

# define the actions to perform for "copy" lines
$1 == "copy" {
   # first check to be sure we have exactly 7 fields
   if (NF != 7) {
      print "Error parsing copy command from config file on the following line:" >>LOG
      print $0 >>LOG
      print "Exactly 7 fields are required!" >>LOG
      print "" >>LOG
      next
   }

   # do not execute command if debug mode prevents it
   if (DEBUG == "on" && $2 == "N") {
      print "Skipping file " $5 "/" $4 " because Debug is ON and it is a No-Debug-Only command." >>LOG
      print "" >>LOG
      next
   }
   else if (DEBUG == "off" && $2 == "Y") {
      print "Skipping file " $5 "/" $4 " because Debug is OFF and it is a Debug-Only command." >>LOG
      print "" >>LOG
      next
   }

   # do not execute command if the passed filter applies
   if (FILTER != "" && match($3, FILTER)) {
      print "Skipping file " $5 "/" $4 " because of passed filter \"" FILTER "\"." >>LOG
      print "" >>LOG
      next
   }

   # get the file input location
   if (index($5, "C") == 1) {
      file = gensub(/C/, CPATH, 1, $5) "/" $4
   }
   else if (index($5, "D") == 1) {
      query = "queryFileLocation " ( ($5 ~ /^D\/.+/) ? substr($5, 3) : $4 )
      query | getline path;
      close(query)
      file = path "/" $4
   }
   else if (index($5, "/") == 1) {
      file = $5 "/" $4
   }
   else {
      print "Error parsing copy command from config file on the following line:" >>LOG
      print $0 >>LOG
      print "The \"Path\" field must begin with \"C\", \"D\", or \"/\"!" >>LOG
      print "" >>LOG
      next
   }

   # get whether or not to recurse
   if ($6 == "N") {
      copyParams = ""
   }
   else if ($6 == "Y") {
      copyParams = "-R "
   }
   else {
      print "Error parsing copy command from config file on the following line:" >>LOG
      print $0 >>LOG
      print "The \"Recursive\" field must be \"Y\" or \"N\"!" >>LOG
      print "" >>LOG
      next
   }

   # ensure that the output directory exists
   if (system("test -e " $7) != 0) {
      if (system("mkdir -p " $7 " >/dev/null 2>&1") != 0) {
         print "Error parsing copy command from config file on the following line:" >>LOG
         print $0 >>LOG
         print "Could not create the output directory \"" $7 "\"!" >>LOG
         print "" >>LOG
         next
      }
   }

   # execute the copy command
   cmd = "cp " copyParams file " " $7
   print "Executing copy \"" cmd "\"." >>LOG
   system(TIME " " cmd " >" ERRFILE " 2>&1")
   system("checkerrors")
}
' $CONFIG_FILE


# --------------
# finish archive
# --------------

# finish up log file
echo "Finished:" >>$LOG
date >>$LOG

# tar and compress PA files
nice -n 19 tar -czf $ARCHIVE_FILE *
popd >/dev/null
rm -rf $TEMP_DIR >/dev/null 2>&1
