#!/bin/bash

# disketteformat.sh
#
# Usage: disketteformat.sh <device file> <filesystem> <log file> <label>
#
# device file: The device file for the diskette, usually given as /dev/fd0
# filesystem: The integer representing the filesystem type, either 1 for vfat, or 4 for ext2
# log file: The log file for recording the output of the diskette format script
# label: The optionally passed label string used to format the diskette with a label
#
# Return Codes:
# 2 - Error formating the diskette media
#
# Module History
#  00  11/21/2003  P. Provost - Changed to use /sbin/mkfs because /sbin is not in the root path
#  01  10/11/2004  P. Provost - Changed to use /sbin/mkdosfs -F 32 for FAT32 filesystems instead of /sbin/mkfs.msdos
#  02  12/15/2004  P. Provost - Put double quotes around $LABEL in case the given label contains spaces
#  03  01/10/2005  P. Provost - Added -I option to mkdosfs command, necessary to vfat format USB devices
#  04  01/10/2005  P. Provost - Added -F option to mkfs command, necessary to force ext2 format of USB devices
#  05  03/17/2005  P. Provost - Fixed tracing.
#  06  03/23/2005  P. Provost - Changed script to check for format error with write-protected diskettes
#  07  03/30/2005  P. Provost - Changed to use mkfs -t msdos for interoperability with OS/2
#  08  06/17/2005  P. Provost - Now tries to low-level format the diskette
#  09  06/21/2005  P. Provost - Added support for fdformat /dev/fd0H1440

DEVICEFILE=$1
FILESYSTEM=$2
LOGFILE=$3
LABEL=$4
TEMPFILE=$5

echo "-> disketteformat.sh" >> $LOGFILE
echo "disketteformat.sh: Formatting diskette at $DEVICEFILE for `date`" >> $LOGFILE

actzTrace "XMDA   T: -> disketteformat.sh"
actzTrace "XMDA   F: disketteformat.sh: Formatting diskette at $DEVICEFILE for `date`"

# First, try to low-level format the diskette
# this is necessary because, on the pHMC, some of the tasks use the tar command
# to tar files directly to the diskette, which essentially destroys the filesystem.

# Note, the fdformat command will probably fail with diskettes in USB diskette drives
# Currently, the Linux fdformat command does not support low-level formatting diskettes
# in USB diskette drives.

if fdformat -n $DEVICEFILE > /dev/null 2>&1; then
    echo "disketteformat.sh: Low-level formatted the diskette." >> $LOGFILE
    actzTrace "XMDA   F: disketteformat.sh: Low-level formatted the diskette."
elif fdformat -n /dev/fd0H1440 > /dev/null 2>&1; then
    echo "disketteformat.sh: Low-level formatted the diskette with /dev/fd0H1440."
    actzTrace "XMDA   F: disketteformat.sh: Low-level formatted the diskette with /dev/fd0H1440."
else
    echo "disketteformat.sh: Failed to low-level format the diskette." >> $LOGFILE
    actzTrace "XMDA   F: disketteformat.sh: Failed to low-level format the diskette."
fi

# Determine what filesystem is to be used to format the diskette
case "$FILESYSTEM" in
    # If the diskette is to be formatted with the dos filesystem
    1)
        # Check to see if the label argument exists
        if [ -z "$LABEL" ]
        then

            # Format a msdos diskette with no label
            echo "disketteformat.sh: Formatting a msdos diskette with no label." >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting a msdos diskette with no label."

            #if ! /sbin/mkdosfs -F 32 -v -I $DEVICEFILE > $TEMPFILE 2>&1; then
            if ! /sbin/mkfs -t msdos -v -I $DEVICEFILE > $TEMPFILE 2>&1; then

                if grep -q -i "unable to open" $TEMPFILE; then

                    echo "disketteformat.sh: diskette maybe write-protected or read-only" >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: diskette maybe write-protected or read-only"

                    echo "disketteformat.sh: Couldn't format a msdos diskette...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a msdos diskette...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 1

                else

                    echo "disketteformat.sh: The contents of the temp file..." >> $LOGFILE

                    echo "" >> $LOGFILE
                    cat $TEMPFILE >> $LOGFILE
                    echo "" >> $LOGFILE

                    echo "disketteformat.sh: Couldn't format a msdos diskette...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a msdos diskette...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 2

                fi

            fi

        else

            # Format a msdos diskette with a label
            echo "disketteformat.sh: Formatting a msdos diskette with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting a msdos diskette with the label: $LABEL"

            #if ! /sbin/mkdosfs -F 32 -v -I -n "$LABEL" $DEVICEFILE > $TEMPFILE 2>&1; then
            if ! /sbin/mkfs -t msdos -v -I -n "$LABEL" $DEVICEFILE > $TEMPFILE 2>&1; then

                if grep -q -i "unable to open" $TEMPFILE; then

                    echo "disketteformat.sh: diskette maybe write-protected or read-only" >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: diskette maybe write-protected or read-only"

                    echo "disketteformat.sh: Couldn't format a msdos diskette...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a msdos diskette with a label...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 1

                else

                    echo "disketteformat.sh: The contents of the temp file..." >> $LOGFILE

                    echo "" >> $LOGFILE
                    cat $TEMPFILE >> $LOGFILE
                    echo "" >> $LOGFILE

                    echo "disketteformat.sh: Couldn't format a msdos diskette...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a msdos diskette with a label...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 2

                fi

            fi

        fi
    ;;
    # If the diskette is to be formatted with the ext2 filesystem
    4)
        # Check to see if the label argument exists
        if [ -z "$LABEL" ]
        then

            # Format an ext2 diskette with no label
            echo "disketteformat.sh: Formatting an ext2 diskette with no label." >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting an ext2 diskette with no label."

            if ! /sbin/mkfs -t ext2 -v -F $DEVICEFILE > $TEMPFILE 2>&1; then

                if grep -q -i "Read-only file system while setting up superblock" $TEMPFILE; then

                    echo "disketteformat.sh: diskette maybe write-protected or read-only" >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: diskette maybe write-protected or read-only"

                    echo "disketteformat.sh: Couldn't format a ext2 diskette...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a ext2 diskette...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 1

                else

                    echo "disketteformat.sh: The contents of the temp file..." >> $LOGFILE

                    echo "" >> $LOGFILE
                    cat $TEMPFILE >> $LOGFILE
                    echo "" >> $LOGFILE

                    echo "disketteformat.sh: Couldn't format a ext2 diskette...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a ext2 diskette...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 2

                fi

            fi

        else

            # Format an ext2 diskette with a label
            echo "disketteformat.sh: Formatting an ext2 diskette with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting an ext2 diskette with the label: $LABEL"

            if ! /sbin/mkfs -t ext2 -v -F -L "$LABEL" $DEVICEFILE > $TEMPFILE 2>&1; then

                if grep -q -i "Read-only file system while setting up superblock" $TEMPFILE; then

                    echo "disketteformat.sh: diskette maybe write-protected or read-only" >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: diskette maybe write-protected or read-only"

                    echo "disketteformat.sh: Couldn't format a ext2 diskette with label $LABEL...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a ext2 diskette with label $LABEL...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 1

                else

                    echo "disketteformat.sh: The contents of the temp file..." >> $LOGFILE

                    echo "" >> $LOGFILE
                    cat $TEMPFILE >> $LOGFILE
                    echo "" >> $LOGFILE

                    echo "disketteformat.sh: Couldn't format a ext2 diskette with label $LABEL...exiting." >> $LOGFILE
                    actzTrace "XMDA   F: disketteformat.sh: Couldn't format a ext2 diskette with label $LABEL...exiting."

                    echo "<- disketteformat.sh for `date`" >> $LOGFILE
                    echo "" >> $LOGFILE

                    actzTrace "XMDA   T: <- disketteformat.sh for `date`"

                    exit 2

                fi

            fi

        fi
    ;;
esac

echo "<- disketteformat.sh for `date`" >> $LOGFILE
echo "" >> $LOGFILE

actzTrace "XMDA   T: <- disketteformat.sh for `date`"

exit 0

