// ------------------------------------------------------------------------------------------------
//
//  Methods used to construct and display context menus.
//
// External Classes:
//     ContextMenu: The context menu.
//     Menu:        Contains MenuItem objects.
//     MenuItem:    Displayed item in a Menu.
//
//
// External Methods:
//     MenuInit():
//         Description: Initializes the classes.  This must be called
//                      before using any of the classes.
//         Arguments:
//             1. The name of the function that creates the menu.
//                This function will be called each time the user right clicks.
//     ContextMenu()
//         Description: Constructor.
//         Arguments:
//             1. Menu object (optional).
//     ContextMenu.setMenu():
//         Description: Changes the main menu.
//         Arguments:
//             1. Menu object.
//     Menu()
//         Description: Constructor.
//         Arguments;
//             1. The displayed text (optional for the main menu, required for submenus).
//             2. The displayed image (optional).
//     Menu.add():
//         Description: Add a MenuItem or Menu object to the menu.
//         Arguments:
//             1. A Menu or MenuItem object.
//     Menu.addSeparator():
//         Description: Add a separator to the menu.
//         Arguments: none
//     MenuItem()
//         Description: Constructor.
//         Arguments:
//             1. Displayed text.
//             2. The name of the image (null will cause the default image to display).
//             3. Function to call when the item is selection.
//             4. This is passed to the handler when the item is selected.
//
//
// Sample useage:
//
//   function init()
//   {
//      // Initialize the menu class and tell class the name of
//      // out function that creates the menu.
//      menuInit(myMenuCreator);
//   }
//
//   function selectionHandler(userData)
//   {
//      // 'userData' is the third argument passed to the MenuItem constructor.
//   }
//
//   function myMenuCreator()
//   {
//      var nextItem = 0;
//      var mainMenu = new Menu();
//      mainMenu.add(new MenuItem("selection 1", "www/images/image1.gif", selectionHandler, nextItem++));
//      mainMenu.add(new MenuItem("selection 2", "www/images/image2.gif", selectionHandler, nextItem++));
//      mainMenu.addSeparator();
//      var subMenu = new Menu("SubMenu",  "www/images/image3.gif",);
//      subMenu.add(new MenuItem("submenu selection 1", "www/images/image4.gif", selectionHandler, nextItem++));
//      mainMenu.add(subMenu);
//      return new ContextMenu(mainMenu);
//      // return null to cause the browser's context menu to display
//   }
//
//
// ------------------------------------------------------------------------------------------------
var BLACK_ARROW_URL = "/images/ui/z/menuarrowblack.gif";
var WHITE_ARROW_URL = "/images/ui/z/menuarrowwhite.gif";
var DEFAULT_IMAGE_URL = "/res/com/ibm/hwmca/fw/images/default.gif";
var MENU_WIDTH = 250 + 4; // specifed in menu.css, +4 for border
//var BLACK_ARROW_URL = "menuarrowblack.gif";
//var WHITE_ARROW_URL = "menuarrowwhite.gif";
var contextMenu = null;
var numMenuInstances = 0;
var numMenuItemInstances = 0;
var numSeparatorInstances = 0;
var menuCreator = null;
var isIE = navigator.userAgent.indexOf("MSIE") > -1 ? true : false;
var isGecko = navigator.userAgent.indexOf("Gecko") > -1 ? true : false;

// ----------------------------------------------------------------------------
// This must be called before using any of the classes defined in this file.
//
// Arguments:
//     1. The function that creates the menu.
//
// ----------------------------------------------------------------------------
function menuInit(userMenuCreator)
{
   menuCreator = userMenuCreator;

   // ContextMenu class methods
   ContextMenu.prototype.setMenu = contextMenuSetMenu;

   // Menu class methods
   Menu.prototype.addSeparator = menuAddSeparator;
   Menu.prototype.add = menuAdd;

   document.oncontextmenu = contextMenuHandler;
}

function onClickHandler(e)
{
   parent.trace(2, "-> menu.onClickHandler()");

   var event = null;
   if (isGecko)
      event = e;
   else
      event = window.event;

   // See if the menu is displaying.
   if ((contextMenu != null) && (contextMenu.menu.divMenuElement.style.visibility == "visible"))
      if (event.button != 2)   // not right mouse button?
      {
         document.onclick = null;
         menuHide();
      }

   parent.trace(2, "<- menu.onClickHandler()");
}


// Toggle displaying/hiding the context menu.
function contextMenuHandler(e)
{
   parent.trace(2, "-> menu.contextMenuHandler()");

   var result = true;  // browser will show it's context menu
   if (parent.displayContextMenu)
   {
      result = false;
      var event = null;
      if (isGecko)
         event = e;             // NN event model
      else
         event = window.event;  // IE event model

      // If the menu is displaying then hide it.
      var menuHidden = false;
      if ((contextMenu != null) &&
          (contextMenu.menu.divMenuElement.style.visibility == "visible"))
      {
         menuHide();
         menuHidden = true;
      }

      // If the menu was not displaying then show it.
      if (menuHidden == false)
      {
         if (menuCreator != null)
            contextMenu = menuCreator(event);

         if ((contextMenu == null) || (contextMenu.menu == null))
         {
            result = true;  // browser will show it's context menu
         }
         else
         {
            if (contextMenu.menu.divMenuElement.style.visibility == "visible")
            {
               parent.trace(2, "   hide context menu");

               // Context menu is visible, hide it and all it's child submenus.
               contextMenu.menu.divMenuElement.style.visibility = "hidden";

               var children = contextMenu.menu.divMenuElement.childElements;
               for (var i = 0; i < children.length; i++)
               {
                  if (children[i].isSubMenuSelection)
                  {
                     unselectSubMenu(children[i]);
                     break;
                  }
               }
            }
            else
            {
               parent.trace(2, "   show context menu");

               var x = getX(event);   // cursor position when user right clicks.
               var y = getY(event);   // cursor position when user right clicks.
               var frameWidth = getWidth();
               if (frameWidth > x + MENU_WIDTH)
                  // left side of menu is at the cursor position
                  contextMenu.menu.divMenuElement.style.left = x;
               else
                  // right side of menu is at the cursor position
                  contextMenu.menu.divMenuElement.style.left = x - MENU_WIDTH;

               contextMenu.menu.divMenuElement.style.top = y;

               // Context menu is not visible, show it.
               contextMenu.menu.divMenuElement.style.visibility = "visible";

               // A click handler is added to the document so the menu can be removed
               // when the user left or right clicks outside of the menu.
               document.onclick = onClickHandler;
            }
         }
      }
   }

   parent.trace(2, "<- menu.contextMenuHandler()");
   return result;
}

function menuHide()
{
   parent.trace(2, "-> menu.menuHide()");

   // Context menu is visible, hide it and all it's child submenus.
   contextMenu.menu.divMenuElement.style.visibility = "hidden";

   var children = contextMenu.menu.divMenuElement.childElements;
   for (var i = 0; i < children.length; i++)
   {
      if (children[i].isSubMenuSelection)
      {
         unselectSubMenu(children[i]);
      }
   }

   parent.trace(2, "<- menu.menuHide()");
}

// ----------------------------------------------------------------------------
// ContextMenu constructor
//
// Arguments:
//     1. The main Menu object (optional).
//
// ----------------------------------------------------------------------------
function ContextMenu()
{
   this.menu = null;
   if (arguments.length > 0)
      this.setMenu(arguments[0]);
   contextMenu = this;
}

// Set the main context menu.
function contextMenuSetMenu(menu)
{
   this.menu = menu;
}

// ----------------------------------------------------------------------------
// Menu constructor.
//
// Arguments:
//     1. The displayed text (optional for the main menu, required for submenus).
//     2. The displayed image (optional).
//   The next 2 optional arguments should be passed if the use if allowed to
//   select the submenu.
//     3. The function to be called when the item is selected.
//     4. Any data that is passed to the handler when this item is selected.
//
// ----------------------------------------------------------------------------
function Menu()
{
   //parent.trace(3, "-> Menu.Menu()");
   var id = "MENU" + (++numMenuInstances);

   // The menu needs one <div> element to contain the list of selections.
   // Optionally a second <div> element is used whenn the 'text' argument
   // is passed.  The <div> for the 'text' argument is used when adding a
   // selection item to the parent menu.
   this.divMenuElement = document.createElement("div");
   this.divMenuElement.id = id;
   this.divMenuElement.className = "menu";

   // 'childElements' is needed to hide submenus.
   this.divMenuElement.childElements = new Array();

   // Create an element to contain the selection that causes this submenu
   // to display.
   if (arguments.length >= 1)
   {
      this.text = arguments[0];
      this.divSelectionElement = document.createElement("div");
      this.divSelectionElement.id = "I" + id;
      this.divSelectionElement.className = "menuitem menuitemout";
      this.divImageElement = document.createElement("img");
      if (arguments.length > 1)
         this.divImageElement.src = arguments[1];
      else
         this.divImageElement.src = DEFAULT_IMAGE_URL;
      this.divImageElement.className = "menuimage";
      this.divSelectionElement.appendChild(this.divImageElement);
      this.divSelectionElement.appendChild(document.createTextNode("   " + this.text));
      this.divSelectionElement.divMenuElement = this.divMenuElement;
      if (arguments.length > 2)
      {
         this.divSelectionElement.handler = arguments[2];
         if (arguments.length > 3)
            this.divSelectionElement.userData = arguments[3];
         if (this.divSelectionElement.handler != null)
            this.divSelectionElement.onclick = menuItemMouseClick;
      }

      // Mark this Menu as a submenu and add the arrow.
      this.divSelectionElement.isSubMenuSelection = true;
      this.divArrowElement = document.createElement("img");
      this.divArrowElement.src = BLACK_ARROW_URL;
      this.divArrowElement.className = "menuarrow";
      this.divSelectionElement.appendChild(this.divArrowElement);
      this.divSelectionElement.divArrowElement = this.divArrowElement;

      // There is no mouseout event for a submenu because the submenu stays
      // displayed until the a different select is made in the parent menu.
      this.divSelectionElement.onmouseover = menuMouseOver;

      // The submenu's parent is the selection.
      this.divMenuElement.parentDivElement = this.divSelectionElement;
   }
   document.body.appendChild(this.divMenuElement);
   //parent.trace(3, "<- Menu.Menu()");
}

function menuAddSeparator()
{
   var separator = new MenuItemSeparator();
   this.divMenuElement.appendChild(separator.divSelectionElement);
   var children = this.divMenuElement.childElements;
   children[children.length] = separator.divSelectionElement;
}

function menuAdd(item)
{
   var divElement = item.divSelectionElement;
   this.divMenuElement.appendChild(divElement);
   var children = this.divMenuElement.childElements;
   children[children.length] = divElement;
   divElement.parentDivElement = this.divMenuElement;
}

function menuMouseOver(event)
{
   // change element to selected
   this.className = "menuitem menuitemover";
   this.divArrowElement.src = WHITE_ARROW_URL;

   // The rule for menu's is to quit displaying the menu when another
   // item is selected, not when on a mouseout on the menu item.
   var children = this.parentDivElement.childElements;
   for (var i = 0; i < children.length; i++)
   {
      if (children[i].isSubMenuSelection)
      {
         if (children[i] != this)
         {
            children[i].divArrowElement.src = BLACK_ARROW_URL;
            unselectSubMenu(children[i]);
         }
      }
   }

   // display submenu
   var subMenuDivElement = document.getElementById(this.id.substring(1));
   subMenuDivElement.style.left = getSubMenuLeftPos(subMenuDivElement);
   subMenuDivElement.style.top = getSubMenuTopPos(subMenuDivElement);
   subMenuDivElement.style.visibility = "visible";
}

// ----------------------------------------------------------------------------
// MenuItem constructor
//
// Arguments:
//   1. The displayed text.
//   2. The displayed image.
//   3. The function to be called when the item is selected.
//   4. Any data that is passed to the handler when this item is selected.
//
// ----------------------------------------------------------------------------
function MenuItem()
{
   //parent.trace(3, "-> menu.MenuItem()");

   if (arguments.length >= 4)
   {
      var id = "MI" + (++numMenuItemInstances);
      this.divSelectionElement = document.createElement("div");
      this.divSelectionElement.id = id;
      this.divSelectionElement.className = "menuitem menuitemout";
      this.divImageElement = document.createElement("img");
      if (arguments.length > 1)
      {
         if (arguments[1] == null)
            this.divImageElement.src = DEFAULT_IMAGE_URL;
         else
            this.divImageElement.src = arguments[1];
      }
      else
         this.divImageElement.src = DEFAULT_IMAGE_URL;
      this.divImageElement.className = "menuimage";
      this.divSelectionElement.appendChild(this.divImageElement);
      this.divSelectionElement.appendChild(document.createTextNode("   " + arguments[0]));
      this.divSelectionElement.handler = arguments[2];
      this.divSelectionElement.userData = arguments[3];
      this.divSelectionElement.onmouseover = menuItemMouseOver;
      this.divSelectionElement.onmouseout = menuItemMouseOut;
      if (this.divSelectionElement.handler != null)
         this.divSelectionElement.onclick = menuItemMouseClick;
   }
   //parent.trace(3, "<- menu.MenuItem()");
}

function menuItemMouseOver()
{
   //parent.trace(3, "<> menu.menuItemMouseOver()");
   this.className = "menuitem menuitemover";

   // The rule for menu's is to quit displaying the menu when another
   // item is selected, not when on a mouseout on the menu item.
   var children = this.parentDivElement.childElements;
   for (var i = 0; i < children.length; i++)
   {
      if ((children[i] != this) && (children[i].isSubMenuSelection))
      {
         unselectSubMenu(children[i]);
      }
   }
}

function menuItemMouseOut()
{
   this.className = "menuitem menuitemout";
}

function menuItemMouseClick()
{
   parent.trace(3, "-> menu.menuItemMouseClick()");
   //parent.trace(3, "  this is " + this);
   //parent.trace(3, "  this.handler is " + this.handler);
   this.handler(this.userData);
   menuHide();
   parent.trace(3, "<- menu.menuItemMouseClick()");
}

function unselectSubMenu(subMenuSelection)
{
   //parent.trace(2, "-> menu.unselectSubMenu()");

   var subMenu = subMenuSelection.divMenuElement;
   if (subMenu.style.visibility == "visible")
   {
      subMenu.style.visibility = "hidden";
      subMenuSelection.className = "menuitem menuitemout";
   }

   var children = subMenu.childElements;
   for (var i = 0; i < children.length; i++)
   {
      if (children[i].isSubMenuSelection)
      {
         unselectSubMenu(children[i]);
      }
   }
   //parent.trace(2, "<- menu.unselectSubMenu()");
}

// ----------------------------------------------------------------------------
// MenuItemSeparator constructor
//
// Arguments: none
//
// ----------------------------------------------------------------------------
function MenuItemSeparator()
{
   this.id = "MIS" + (++numSeparatorInstances);
   this.divSelectionElement = document.createElement("hr");
   this.divSelectionElement.className = "menuseparator";
}

// Get the X position of the target event element.
function getX(event)
{
   if (isGecko)
      return event.clientX;
   return window.event.x;
}

// Get the Y position of the target event element.
function getY(event)
{
   if (isGecko)
      return event.clientY;
   return window.event.y;
}

// Get the left position where the submenu will be placed.
function getSubMenuLeftPos(subMenu)
{
   // The subMenu's parent is the selection and that has a menu parent.
   var parentSelection = subMenu.parentDivElement;
   var parentMenu = parentSelection.parentDivElement;
   //parent.trace(3, 'parentMenu.offsetLeft is ' + parentMenu.offsetLeft);
   //parent.trace(3, 'parentMenu.offsetWidth is ' + parentMenu.offsetWidth);
   var leftPos;
   if (getWidth() > parentMenu.offsetLeft + parentMenu.offsetWidth + MENU_WIDTH)
      // position the submenu to the right of the menu item
      leftPos = parentMenu.offsetLeft + parentMenu.offsetWidth;
   else
      // position the submenu to the left of the menu item
      leftPos = parentMenu.offsetLeft - MENU_WIDTH;
   //parent.trace(3, 'getSubMenuLeftPos() returning ' + leftPos);
   return leftPos + "px";
}

// Get the top position where the submenu will be placed.
function getSubMenuTopPos(subMenu)
{
   var parentSelection = subMenu.parentDivElement;
   var parentMenu = parentSelection.parentDivElement;
   return parentMenu.offsetTop + parentSelection.offsetTop + "px";
}

function getHeight()
{
   var height = document.body.clientHeight;
   parent.trace(3, 'getHeight() returning ' + height);
   return height;
}

function getWidth()
{
   var width = document.body.clientWidth;
   parent.trace(3, 'getWidth() returning ' + width);
   return width;
}
