#!/bin/bash
#
# This script collects network settings information from the system
# and dumps them into easy to process properties files for Java to use.
#
# INPUT:
#   Parm 1: The directory to create the output files in (eg. /tmp/console)
#   Parm 2: The ccfw network configuration data directory (eg. /console/data/network)
#
#   Additional parameters are treated as interfaces for which information
#   should be gathered (eg. eth1, tr0). Wildcards can also be specified
#   here (eg. eth*, network*)
#
# OUTPUT:
#   This script outputs the name of each file that it has written.
#
# Notes:
#   This script will create a NetworkInfo.properties file containing general
#   network settings information and a separate
#   NetworkInterfaceInformation-<ifname>.properties file for each specified
#   interface. A wildcard can be specified as the interface in which case
#   properties files will be generated for each file matching the wildcard
#   (eg. 'eth*' will generate files for eth0, eth1, eth2 etc.)
#
#   The properties listed in the files are as follows:
#
#   NetworkInfo.properties:
#       HOSTNAME                - The hostname
#       DOMAIN                  - The domain name (ie. endicott.ibm.com)
#       HOSTS_ENTRY_<n>         - Represent entries from the /etc/hosts file
#       DEFAULT_GATEWAY         - The default gateway IP address
#       DEFAULT_GATEWAY_DEV     - The default gateway device
#       ROUTE_ENTRY_<n>_DEST    - The destination for a single route entry
#       ROUTE_ENTRY_<n>_MASK    - The netmask for a single route entry
#       ROUTE_ENTRY_<n>_GATEWAY - The gateway IP for a single route entry
#       ROUTE_ENTRY_<n>_DEV     - The gateway device for a single route entry
#       DNS_SERVER_<n>          - A single DNS nameserver entry
#       DOMAIN_SUFFIXES         - list of domain suffixes
#
#   NetworkInterfaceInformation-<ifname>.properties:
#       INTERFACE_NAME          - The interface name
#       IPV4_ADDRESS            - The IPv4 address
#       IPV4_NETMASK            - The IPv4 network mask
#       IPV6_ADDRESS_<n>        - A single IPv6 address (including the prefix length)
#       MAC_ADDRESS             - The MAC address
#       IS_DHCP_CLIENT          - true if DHCP is used to configure
#       IPV6_AUTOCONFIG         - true if autoconfiguration for IPv6 is enabled
#       IPV6_USE_PRIV_EXT       - true if privacy extensiosn are used for autoconfig
#       IS_DHCP_SERVER          - true if a DHCP server runs on this interface
#       DHCP_SERVER_SUBNET      - the subnet for the DHCP server
#       DHCP_SERVER_MASK        - the network mask for the DHCP server
#       DHCP_SERVER_RANGE       - the range for the DHCP server
#       MEDIA_SPEED             - Manual setting of media speed (10|100|1000)
#
# Module History:
#   03/12/2007  M. Clark        Initial release
#   08/07/2007  M. Clark        Get the IP address from the ifcfg file if not found
#   08/07/2007  M. Clark        Get the IPv6 autoconfig/privext settings from file
#   08/21/2007  M. Clark        Set DNS_ENABLED parameter
#

# Usage: getNetworkSettings.sh <directory for output files> <ccfw networkconf directory> [network interfaces]

actzTrace "XSETNETT: getNetworkSettings.sh $*"

OUTPUT_DIR=${1%%/}
NETWORK_CONF_DIR=${2%%/}
shift
shift
INTERFACES="$*"

NETWORK_INFO_FILE="$OUTPUT_DIR/NetworkInfo.properties"
MEDIA_SPEED_FILE="$NETWORK_CONF_DIR/media_speed.conf"
IPV6_AUTOCONF_FILE="$NETWORK_CONF_DIR/ipv6_autoconf.conf"
IPV6_PRIV_EXT_FILE="$NETWORK_CONF_DIR/ipv6_privext.conf"
DHCPV6_CLIENT_FILE="$NETWORK_CONF_DIR/dhcpv6_client.conf"

##############################################################################
# Functions
##############################################################################

write_interface_info() {
    local intf=$1
    local output_file="$OUTPUT_DIR/NetworkInterfaceInformation-$intf.properties"
    local config_file="/etc/sysconfig/network/ifcfg-$intf"

    actzTrace "XSETNETT: -> write_interface_info $output_file"

    if [ -e $output_file ]; then
        actzTrace "XSETNETF: Deleting existing file $output_file"
        rm -f $output_file
    fi

    touch $output_file
    chmod 0666 $output_file

    echo "INTERFACE_NAME=$intf" >> $output_file

    # IPv4 address
    local addr="0.0.0.0"
    local mask="255.255.255.255"
    local line=$(/sbin/ifconfig $intf | grep "inet addr")
    if [ -n "$line" ]; then
        addr=$(echo $line | sed -e "s/.*inet addr\://" -e "s/\s\+Bcast.*//")
        mask=$(echo $line | sed -e "s/.*Mask\://")
    else
        actzTrace "XSETNETF: no 'inet addr' line found in ifconfig output for $intf"
        if grep -q "IPADDR=" $config_file; then
            addr=$(grep "IPADDR=" $config_file | sed -e "s|IPADDR='||" -e "s|'$||")
            mask=$(grep "NETMASK=" $config_file | sed -e "s|NETMASK='||" -e "s|'$||")
            actzTrace "XSETNETF: static ip address from config file is $addr / $mask"
        fi
    fi

    actzTrace "XSETNETF: adding IPv4 address $addr/$mask for $intf"
    echo "IPV4_ADDRESS=$addr" >> $output_file
    echo "IPV4_NETMASK=$mask" >> $output_file

    # Static IPv6 address(es)
    local i=0
    local ipv6addr=""
    for ipv6addr in $(grep "^IPADDR[[:digit:]]\+" $config_file | sed -e "s|IPADDR.\+='||" -e "s|'$||"); do
        actzTrace "XSETNETF: adding IPv6 address $ipv6addr"
        echo "IPV6_ADDRESS_$i=$ipv6addr" >> $output_file
        i=$(($i+1))
    done

    # Autoconfigured IPv6 addresses(es)
    i=0
    for ipv6addr in $(ip addr show $intf | grep inet6 | sed -e "s/^[[:space:]]*//" | cut -d' ' -f 2); do
        if ! grep -q "$ipv6addr" $config_file; then
            actzTrace "XSETNETF: found an autoconfigured address: $ipv6addr"
            echo "AUTOCONF_IPV6_ADDRESS_$i=$ipv6addr" >> $output_file
            i=$(($i+1))
        else
            actzTrace "XSETNETF: found $ipv6addr in the config file.. must be static"
        fi
    done

    # MAC address
    line=$(/sbin/ifconfig $intf | grep "HWaddr")
    if [ -n "$line" ]; then
        local mac=$(echo $line | sed -e "s/.*HWaddr.//" -e "s/\s\+$//" -e "s/://g")

        actzTrace "XSETNETF: adding MAC address $mac"
        echo "MAC_ADDRESS=$mac" >> $output_file
    else
        actzTrace "XSETNETF: no 'HWaddr' line found in ifconfig output for $intf"
    fi

    # DHCP settings
    local is_dhcp_client="0"
    if [ -e $config_file ]; then
        line=$(grep -m 1 "^BOOTPROTO" $config_file)
        if echo "$line" | grep -q "dhcp"; then
            actzTrace "XSETNETF: setting DHCP client true for $intf"
            echo "IS_DHCP_CLIENT=true" >> $output_file
            is_dhcp_client="1"
        else
            actzTrace "XSETNETF: setting DHCP client false for $intf"
            echo "IS_DHCP_CLIENT=false" >> $output_file
        fi
    else
        actzTrace "XSETNETF: no config file $config_file.. setting DHCP client false"
        echo "IS_DHCP_CLIENT=false" >> $output_file
    fi

    # IPv6 Autoconfiguration settings
    if [ -e $IPV6_AUTOCONF_FILE ]; then
        if grep -q "^$intf$" $IPV6_AUTOCONF_FILE; then
            actzTrace "XSETNETF: setting IPv6 Autoconfiguration to true"
            echo "IPV6_AUTOCONFIG=true" >> $output_file
        else
            actzTrace "XSETNETF: setting IPv6 Autoconfiguration to false"
            echo "IPV6_AUTOCONFIG=false" >> $output_file
        fi
    else
        actzTrace "XSETNETF: No conf file.. setting IPv6 Autoconfiguration to false"
        echo "IPV6_AUTOCONFIG=false" >> $output_file
    fi

    # IPv6 Privacy Extensions settings
    if [ -e $IPV6_PRIV_EXT_FILE ]; then
        if grep -q "^$intf$" $IPV6_PRIV_EXT_FILE; then
            actzTrace "XSETNETF: setting IPv6 Use privacy extensions to true"
            echo "IPV6_USE_PRIV_EXT=true" >> $output_file
        else
            actzTrace "XSETNETF: setting IPv6 Use privacy extensions to false"
            echo "IPV6_USE_PRIV_EXT=false" >> $output_file
        fi
    else
        actzTrace "XSETNETF: No conf file.. setting IPv6 Privacy ext to false"
        echo "IPV6_USE_PRIV_EXT=false" >> $output_file
    fi

    # DHCPv6 Client settings
    if [ -e $DHCPV6_CLIENT_FILE ]; then
        if grep -q "^$intf$" $DHCPV6_CLIENT_FILE; then
            actzTrace "XSETNETF: setting DHCPv6 client to true"
            echo "IS_DHCPV6_CLIENT=true" >> $output_file
        else
            actzTrace "XSETNETF: setting DHCPv6 client to false"
            echo "IS_DHCPV6_CLIENT=false" >> $output_file
        fi
    else
        actzTrace "XSETNETF: No conf file.. setting DHCPv6 client to false"
        echo "IS_DHCPV6_CLIENT=false" >> $output_file
    fi

    # Check for DHCP Server
    if [ "$is_dhcp_client" = "0" -a "$have_dhcp_server_config" = "1" ]; then
        local found_it="0"
        # seq starts at 1
        for i in $(seq ${#dhcp_server_addrs[*]}); do
            if [ "${dhcp_server_addrs[$((i-1))]}" = "$addr" ]; then
                actzTrace "XSETNETF: found our address in the dhcp server list"
                found_it="1"

                echo "IS_DHCP_SERVER=true" >> $output_file
                echo "DHCP_SERVER_SUBNET=${dhcp_server_subnets[$((i-1))]}" >> $output_file
                echo "DHCP_SERVER_NETMASK=${dhcp_server_masks[$((i-1))]}" >> $output_file
                echo "DHCP_SERVER_RANGE=${dhcp_server_ranges[$((i-1))]}" >> $output_file
                break
            fi
        done

        if [ "$found_it" = "0" ]; then
            actzTrace "XSETNETF: didn't find this address in the list of dhcp servers"
            echo "IS_DHCP_SERVER=false" >> $output_file
        fi
    else
        actzTrace "XSETNETF: this is not a dhcp server"
        echo "IS_DHCP_SERVER=false" >> $output_file
    fi

    # Find the media speed information
    if [ -e $MEDIA_SPEED_FILE ]; then
        line=$(grep "$intf" "$MEDIA_SPEED_FILE")
        if [ -n "$line" ]; then
            # Lines in this file look like this:
            #
            #/usr/sbin/ethtool -s eth<N> autoneg off speed <10|100|1000> duplex <half|full>
            #
            local autoneg=$(echo "$line" | cut -d' ' -f 5)
            if [ "$autoneg" = "off" ]; then

                local speed=$(echo "$line" | cut -d' ' -f 7)
                local duplex=$(echo "$line" | cut -d' ' -f 9)

                actzTrace "XSETNETF: speed/duplex = $speed / $duplex "
                case "$speed" in
                    10)
                        if [ "$duplex" = "half" ]; then
                            echo "MEDIA_SPEED=1" >> $output_file
                        else
                            echo "MEDIA_SPEED=2" >> $output_file
                        fi
                        ;;
                    100)
                        if [ "$duplex" = "half" ]; then
                            echo "MEDIA_SPEED=3" >> $output_file
                        else
                            echo "MEDIA_SPEED=4" >> $output_file
                        fi
                        ;;
                    1000)
                        if [ "$duplex" = "half" ]; then
                            echo "MEDIA_SPEED=5" >> $output_file
                        else
                            echo "MEDIA_SPEED=6" >> $output_file
                        fi
                        ;;
                    *) ;;
                esac
            else
                actzTrace "XSETNETF: Setting autoneg to true"
                echo "MEDIA_SPEED=0" >> $output_file
            fi

        else
            actzTrace "XSETNETF: $intf not found in $MEDIA_SPEED_FILE.. using autoneg"
            echo "MEDIA_SPEED=0" >> $output_file
        fi
    else
        actzTrace "XSETNETF: No $MEDIA_SPEED_FILE found.. using autoneg"
        echo "MEDIA_SPEED=0" >> $output_file
    fi

    # print out the name of the file we just wrote
    echo "$output_file"

    actzTrace "XSETNETT: <- write_interface_info $intf"
}

#actzTrace() {
#    echo "$*" 1>&2
#}

##############################################################################
# EXECUTION START
##############################################################################

if [ -e "$NETWORK_INFO_FILE" ]; then
    actzTrace "XSETNETF: getNetworkSettings deleting existing $NETWORK_INFO_FILE"
    rm -f $NETWORK_INFO_FILE
fi

touch $NETWORK_INFO_FILE
chmod 0666 $NETWORK_INFO_FILE

##############################################################################
# Gather the settings information used in the NetworkInfo object
##############################################################################

# hostname/domain
hostname=$(hostname -f)
if echo "$hostname" | grep -q "\."; then
    # we have a domain also
    domain=${hostname#*.}
    hostname=${hostname%%.*}
fi

echo "HOSTNAME=$hostname" >> $NETWORK_INFO_FILE
echo "DOMAIN=$domain" >> $NETWORK_INFO_FILE

# /etc/hosts entries
i=0
if [ -s "/etc/hosts" ]; then
    # this grep just gets rid of comments and empty lines
    grep -v -e "^\s*#" -e "^\s*$" /etc/hosts | while read line; do
        # XXX this code is executed in a subshell because of the pipe
        # XXX so changes made to variables inside this block will be lost

        # Filter out the localhost and hostname entries
        if [ $(expr "$line" : ".*\<localhost\>") -eq 0 ]; then
            if [ $(expr "$line" : ".*\<$hostname\>") -eq 0 ]; then
                actzTrace "XSETNETF: adding host entry $line"
                echo "HOSTS_ENTRY_$i=$line" >> $NETWORK_INFO_FILE
                i=$(($i+1))
            else
                actzTrace "XSETNETF: skipping hostname entry: $line"
            fi
        else
            actzTrace "XSETNETF: skipping localhost entry: $line"
        fi
    done
else
    actzTrace "XSETNETF: /etc/hosts didn't exist or had zero length"
fi

# Routing information
i=0
if [ -s "/etc/sysconfig/network/routes" ]; then
    # this grep just gets rid of comments and empty lines
    grep -v -e "^\s*#" -e "^\s*$" /etc/sysconfig/network/routes | while read line; do
        # XXX this code is executed in a subshell because of the pipe
        # XXX so changes made to variables inside this block will be lost
        line=$(echo "$line" | sed -e "s/\s\+/ /g")

        dest=$(echo "$line" | cut -d' ' -f 1)
        gateway=$(echo "$line" | cut -d' ' -f 2)
        mask=$(echo "$line" | cut -d' ' -f 3)
        dev=$(echo "$line" | cut -d' ' -f 4)

        actzTrace "XSETNETF: adding route entry: dest=$dest; gateway=$gateway; mask=$mask; dev=$dev"

        if [ "$dest" = "default" ]; then
            if [ -z "$default_gateway_dev" ]; then
                default_gateway_dev="$dev"
                echo "DEFAULT_GATEWAY=$gateway" >> $NETWORK_INFO_FILE
                echo "DEFAULT_GATEWAY_DEV=$dev" >> $NETWORK_INFO_FILE
            elif [ "$default_gateway_dev" == "$dev" ]; then
                actzTrace "XSETNETF: found multiple default gateways for the same device.. craziness"
            else
                echo "ROUTE_ENTRY_${i}_DEST=0.0.0.0" >> $NETWORK_INFO_FILE
                echo "ROUTE_ENTRY_${i}_MASK=0.0.0.0" >> $NETWORK_INFO_FILE
                echo "ROUTE_ENTRY_${i}_GATEWAY=$gateway" >> $NETWORK_INFO_FILE
                echo "ROUTE_ENTRY_${i}_DEV=$dev" >> $NETWORK_INFO_FILE
            fi
        else
            echo "ROUTE_ENTRY_${i}_DEST=$dest" >> $NETWORK_INFO_FILE
            echo "ROUTE_ENTRY_${i}_MASK=$mask" >> $NETWORK_INFO_FILE
            echo "ROUTE_ENTRY_${i}_GATEWAY=$gateway" >> $NETWORK_INFO_FILE
            echo "ROUTE_ENTRY_${i}_DEV=$dev" >> $NETWORK_INFO_FILE
            i=$(($i+1))
        fi
    done

else
    actzTrace "XSETNETF: /etc/sysconfig/network/routes didn't exist or had zero length"
fi

# DNS information
i=0
if [ -s "/etc/resolv.conf" ]; then
    echo "DNS_ENABLED=true" >> $NETWORK_INFO_FILE
    suffixes=$(grep -m 1 -e "^search " /etc/resolv.conf)
    if [ -n "$suffixes" ]; then
        actzTrace "XSETNETF: Adding domain suffixes $suffixes"
        echo "DOMAIN_SUFFIXES=${suffixes#search }" >> $NETWORK_INFO_FILE
    else
        actzTrace "XSETNETF: No DNS Suffixes found in resolv.conf"
    fi

    grep "^nameserver " /etc/resolv.conf | while read line; do
        # XXX this code is executed in a subshell because of the pipe
        # XXX so changes made to variables inside this block will be lost

        nameserver=${line#nameserver }
        actzTrace "XSETNETF: Adding nameserver $nameserver"
        echo "DNS_SERVER_$i=$nameserver" >> $NETWORK_INFO_FILE
        i=$(($i+1))
    done
else
    actzTrace "XSETNETF: /etc/resolv.conf didn't exist or had zero length"
    echo "DNS_ENABLED=false" >> $NETWORK_INFO_FILE
fi

##############################################################################
# Collect the DHCP server info (if available) for later use
##############################################################################

if [ -e "/etc/dhcpd.conf" ]; then
    if grep -q "Licensed Internal Code - Property of IBM" "/etc/dhcpd.conf"; then
        actzTrace "XSETNETF: found a valid dhcpd.conf file.. parsing"

        # Entries in the dhcpd.conf file must look like this:
        #
        # subnet 10.255.128.0 netmask 255.255.240.0{
        # # option routers 10.255.128.1
        #   range 10.255.128.2 10.255.143.254
        # }
        #
        # The inclusion of the '# option routers' comment is a hack to include
        # the ip address of the interface in this file

        i=0
        for subnet in $(grep "^subnet " /etc/dhcpd.conf | sed -e "s/^subnet //" -e "s/ netmask.*//"); do
           dhcp_server_subnets[$i]="$subnet"
           actzTrace "XSETNETF: dhcp server subnet = ${dhcp_server_subnets[$i]}"
           i=$(($i+1))
        done

        i=0
        for mask in $(grep "^subnet " "/etc/dhcpd.conf" | sed -e "s/.*netmask //" -e "s/\s*{\s*//"); do
           dhcp_server_masks[$i]="$mask"
           actzTrace "XSETNETF: dhcp server netmask = ${dhcp_server_masks[$i]}"
           i=$(($i+1))
        done

        i=0
        for addr in $(grep "^# option routers " /etc/dhcpd.conf | sed -e "s/.*routers //" -e "s/;.*$//"); do
            dhcp_server_addrs[$i]="$addr"
            actzTrace "XSETNETF: dhcp server address = ${dhcp_server_addrs[$i]}"
            i=$(($i+1))
        done

        i=0
        # a for loop won't work here since there is whitespace between the range values
        grep "^[[:space:]]*range " /etc/dhcpd.conf | sed -e "s/.*range //" -e "s/;.*$//" > "/tmp/dhcp_server_ranges"
        while read range; do
            dhcp_server_ranges[$i]="$range"
            actzTrace "XSETNETF: dhcp server range = ${dhcp_server_ranges[$i]}"
            i=$(($i+1))
        done < /tmp/dhcp_server_ranges
        rm -f /tmp/dhcp_server_ranges

        have_dhcp_server_config="1"
        if [ "${#dhcp_server_subnets[*]}" != "${#dhcp_server_masks[*]}" ]; then
            actzTrace "XSETNETF: found a different number of subnets than masks.. invalid dhcp server config"
            have_dhcp_server_config="0"
        elif [ "${#dhcp_server_subnets[*]}" != "${#dhcp_server_addrs[*]}" ]; then
            actzTrace "XSETNETF: found a different number of subnets than addrs.. invalid dhcp server config"
            have_dhcp_server_config="0"
        elif [ "${#dhcp_server_subnets[*]}" != "${#dhcp_server_ranges[*]}" ]; then
            actzTrace "XSETNETF: found a different number of subnets than ranges.. invalid dhcp server config"
            have_dhcp_server_config="0"
        fi
    else
        actzTrace "XSETNETF: found a bootleg dhcpd.conf file.. ignoring"
    fi
else
    actzTrace "XSETNETF: no dhcpd.conf file found"
fi


##############################################################################
# Gather the settings information for each network interface
##############################################################################

if [ -n "$INTERFACES" ]; then
    for intf in $INTERFACES; do
        actzTrace "XSETNETF: gathering info for interface $intf"

        if echo "$intf" | grep -q "\*$"; then
            actzTrace "XSETNETF: processing wildcard interfaces"

            interfaces=$(ip link show label $intf | grep "^[0-9]\+:" | sed -e "s/^[0-9]\+: \(.\+\):.*/\1/")
            if [ -n "$interfaces" ]; then
                for intf in $interfaces; do
                    actzTrace "XSETNETF: found wildcard interface $intf"
                    write_interface_info $intf
                done
            else
                actzTrace "XSETNETF: didn't find any interfaces from wildcard $intf"
            fi
        else
            if /sbin/ifconfig $intf >/dev/null 2>&1; then
                write_interface_info $intf
            else
                actzTrace "XSETNETF: $intf does not exist"
            fi
        fi
    done
fi

