#!/bin/bash

# mountmedia.sh
#
# Usage: mountmedia.sh <mount point> <temp file> <log file> <read-only option> \
#                      <option0> <option1> <userid> <remount> <keepmount> <perms> <device> <filesystem>
#
# mount point: the mount point for the media
#
# Return Codes:
# 0 - Media mounted successfully, and in read-write mode
# 1 - General failure, cause unknown
# 2 - Media already mounted, attempt to unmount failed
# 3 - Media mounted successfully, is write protected, which is not necessarily a problem
# 4 - Media device failure
# 5 - No media in the device
# 6 - Filesystem problem, probably resulting from unformatted media
# 7 - Attempted to explicitly mount read-only media (or perhaps on a read-only device) as read-write
#
# 01 05/28/2002  P. Provost - Added chown -R command after mounting media
# 02 03/17/2005  P. Provost - Fixed tracing
# 03 04/18/2005  P. Provost - Made modifications with mounting to improve performace (perhaps)
# 04 06/09/2005  P. Provost - Changed chmod -R 777 to chmod -R 755
# 05 08/17/2005  S. Nichols - Add retry logic
# 06 10/03/2005  J. Kunz    - check for read-only "(ro," pipe tempfile into logfile, add REMOUNT support
# 07 05/03/2006  J. Kunz    - fix check for read-only to "(ro"
# 08 10/11/2006  J. Kunz    - Add support to just return if a keep mounted device is already mounted
#    04/11/2007  J. Kunz    - Add support to pass in device permissions
#    05/22/2007  J. Kunz    - Add support to specify filesystem
# 09 11/20/2007  J. Kunz    - remove lsof .. takes 2-3 mins on unmount of unplugged USB

MOUNTPOINT=$1
TEMPFILE=$2
LOGFILE=$3
RO=$4
OPTION0=$5
OPTION1=$6
USERID=$7
REMOUNT=$8
KEEPMOUNT=$9
PERMS=${10}
DEVICE=${11}
FILESYSTEM=${12}

echo "-> mountmedia.sh" >> $LOGFILE
echo "mountmedia.sh: called with parms $* for `date`" >> $LOGFILE

actzTrace "XMDA   T: -> mountmedia.sh"
actzTrace "XMDA   F: mountmedia.sh: called with parms $* for `date`"

################################################################################
# Check to see if media is already mounted if not doing a remount
################################################################################

if [ -z "$REMOUNT" ]
then

   echo "mountmedia.sh: contents of /etc/mtab = " >> $LOGFILE

   echo "" >> $LOGFILE
   cat /etc/mtab >> $LOGFILE
   echo "" >> $LOGFILE

   if grep -q $MOUNTPOINT /etc/mtab; then # Media is already mounted

       echo "mountmedia.sh: Media is already mounted at $MOUNTPOINT" >> $LOGFILE
       actzTrace "XMDA   F: mountmedia.sh: Media is already mounted at $MOUNTPOINT"

       if [ -n "$KEEPMOUNT" ]
       then

           # check that the device isn't mount with root root permissions
           if ! ls -ald $MOUNTPOINT 2>&1 | grep -q -i 'root *root'; then # permissions not root root
	           # exit rc=3 if mounted ro, otherwise exit rc=0
			   if grep $MOUNTPOINT /etc/mtab 2>&1 | grep -q -i ' ro'; then # Media is mounted ro
	              echo "mountmedia.sh: Media is designated keep mounted, exiting rc=3." >> $LOGFILE
	              actzTrace "XMDA   F: mountmedia.sh: Media is designated keep mounted, exiting rc=3."
	              echo "<- mountmedia.sh for `date`" >> $LOGFILE
	              echo "" >> $LOGFILE
	
	              actzTrace "XMDA   T: <- mountmedia.sh for `date`"
			      exit 3
			   else
	              echo "mountmedia.sh: Media is designated keep mounted, exiting rc=0." >> $LOGFILE
	              actzTrace "XMDA   F: mountmedia.sh: Media is designated keep mounted, exiting rc=0."
	              echo "<- mountmedia.sh for `date`" >> $LOGFILE
	              echo "" >> $LOGFILE
	
	              actzTrace "XMDA   T: <- mountmedia.sh for `date`"
			      exit 0
			   fi
		   else
		   	  echo "mountmedia.sh: Media is designated keep mounted, but is mounted with root permissions, unmount then mount anyway" >> $LOGFILE
		   	  actzTrace "XMDA   F: mountmedia.sh: Media is designated keep mounted, but is mounted with root permissions, unmount then mount anyway"
		   fi
       fi

       echo "mountmedia.sh: Unmounting media at $MOUNTPOINT" >> $LOGFILE
       actzTrace "XMDA   F: mountmedia.sh: Umounting media at $MOUNTPOINT"

       if ! umount -vf $MOUNTPOINT >> $LOGFILE 2>&1; then
          	
           echo "mountmedia.sh: Couldn't unmount the media at $MOUNTPOINT... sleep then retry" >> $LOGFILE
           actzTrace "XMDA   F: mountmedia.sh: Couldn't unmount the media at $MOUNTPOINT...sleep then retry."

           # Sleep for a few seconds to ensure that Java garbage collection occurs
           # before the unmount is attemped. Otherwise, there may be some Java objects
           # hanging around that reference paths to the media that can cause the umount
           # command to fail with device busy errors. See MediaDevice.unmount()
           # for the explicit call to Java garbage collector.
           sleep 10
           if ! umount -vf $MOUNTPOINT >> $LOGFILE 2>&1; then

			   #echo "mountmedia.sh: Still couldn't unmount the media at $MOUNTPOINT... put lsof info to $LOGFILE" >> $LOGFILE
			   #actzTrace "XMDA   F: mountmedia.sh: Still couldn't unmount the media at $MOUNTPOINT...put lsof info to $LOGFILE."
			   #echo "lsof info begin" >> $LOGFILE
			   #lsof | grep $MOUNTPOINT >> $LOGFILE 2>&1
			   #echo "lsof info end" >> $LOGFILE

               echo "<- mountmedia.sh for `date`" >> $LOGFILE
               echo "" >> $LOGFILE

               actzTrace "XMDA   T: <- mountmedia.sh for `date`"
              	
               exit 2

           else # unmount was successful

               echo "mountmedia.sh: Unmounted the media at $MOUNTPOINT" >> $LOGFILE
               actzTrace "XMDA   F: mountmedia.sh: Unmounted the media at $MOUNTPOINT"

           fi
       else # unmount was successful

           echo "mountmedia.sh: Unmounted the media at $MOUNTPOINT" >> $LOGFILE
           actzTrace "XMDA   F: mountmedia.sh: Unmounted the media at $MOUNTPOINT"

       fi

   fi

fi

################################################################################
# Perform the actual mount
################################################################################

echo "mountmedia.sh: Mounting the media at $MOUNTPOINT" >> $LOGFILE
actzTrace "XMDA   F: mountmedia.sh: Mounting the media at $MOUNTPOINT"

if mount -v $RO $FILESYSTEM $DEVICE $MOUNTPOINT $OPTION0 $OPTION1 $REMOUNT > $TEMPFILE 2>&1; then

    echo "mountmedia.sh: mount -v $RO $FILESYSTEM $DEVICE $MOUNTPOINT $OPTION0 $OPTION1 $REMOUNT" >> $LOGFILE
    actzTrace "XMDA   F: mountmedia.sh: mount -v $RO $MOUNTPOINT $OPTION0 $OPTION1 $REMOUNT"

    echo "mountmedia.sh: Mounted the media at $MOUNTPOINT" >> $LOGFILE
    actzTrace "XMDA   F: mountmedia.sh: Mounted the media at $MOUNTPOINT"

    echo "***$TEMPFILE***" >> $LOGFILE
    cat $TEMPFILE >> $LOGFILE
    echo "***$TEMPFILE***" >> $LOGFILE

    if grep -q -i "(ro" $TEMPFILE; then

        echo "mountmedia.sh: The media is read-only, which may not be a problem...setting appropriate return code" >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: The media is read-only, which may not be a problem...setting appropriate return code"

        echo "<- mountmedia.sh for `date`" >> $LOGFILE
        echo "" >> $LOGFILE

        actzTrace "XMDA   T: <- mountmedia.sh for `date`"
       	
        exit 3

    else

        echo "mountmedia.sh: The media is not read-only." >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: The media is not read-only."

        # Check to see if the userid was passed to the script
        if [ -z $USERID ]
        then
            # Userid was not passed, don't try to change the onwer of the filesystem

            echo "mountmedia.sh: No userid passed. Did not change the owner of the media" >> $LOGFILE
            actzTrace "XMDA   F: mountmedia.sh: No userid passed. Did not change the owner of the media"

            if [ -z $R0 ]
            then
                if ! chmod -Rv $PERMS $MOUNTPOINT >> $LOGFILE 2>&1; then

                    echo "mountmedia.sh: Did not change the permissions of the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Did not change the permissions of the filesystem on the media"

                else

                    echo "mountmedia.sh: Changed the permissions of the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Changed the permissions of the filesystem on the media"

                fi
            fi

        else

            if [ -z $RO ]
            then

                # Userid was passed, try to change the owner of the filesystem
                if ! chown -Rv $USERID $MOUNTPOINT >> $LOGFILE 2>&1; then

                    echo "mountmedia.sh: Userid was passed but did not change the owner of the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Userid was passed but did not change the owner of the media at $MOUNTPOINT"

                else

                    echo "mountmedia.sh: Userid was passed and changed the owner of the media to $USERID" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Userid was passed and changed the owner of the media at $MOUNTPOINT"

                fi

                if ! chmod -Rv $PERMS $MOUNTPOINT >> $LOGFILE 2>&1; then

                    echo "mountmedia.sh: Did not change the permissions of the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Did not change the permissions of the filesystem on the media"

                else

                    echo "mountmedia.sh: Changed the permissions on the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Changed the permissions of the filesystem on the media"

                fi
            fi

        fi

    fi


elif mount -v $RO $FILESYSTEM $DEVICE $MOUNTPOINT $REMOUNT > $TEMPFILE 2>&1; then

    echo "mountmedia.sh: mount -v $RO $FILESYSTEM $DEVICE $MOUNTPOINT $REMOUNT" >> $LOGFILE
    actzTrace "XMDA   F: mountmedia.sh: mount -v $RO $MOUNTPOINT $REMOUNT"

    echo "mountmedia.sh: Mounted the media at $MOUNTPOINT" >> $LOGFILE
    actzTrace "XMDA   F: mountmedia.sh: Mounted the media at $MOUNTPOINT"

   echo "***$TEMPFILE***" >> $LOGFILE
   cat $TEMPFILE >> $LOGFILE
   echo "***$TEMPFILE***" >> $LOGFILE

   if grep -q -i "(ro" $TEMPFILE; then

        echo "mountmedia.sh: The media is read-only, which may not be a problem...setting appropriate return code" >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: The media is read-only, which may not be a problem...setting appropriate return code"

        echo "<- mountmedia.sh for `date`" >> $LOGFILE
        echo "" >> $LOGFILE

        actzTrace "XMDA   T: <- mountmedia.sh for `date`"
       	
        exit 3

    else

        echo "mountmedia.sh: The media is not read-only." >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: The media is not read-only."

        # Check to see if the userid was passed to the script
        if [ -z $USERID ]
        then
            # Userid was not passed, don't try to change the onwer of the filesystem

            echo "mountmedia.sh: No userid passed. Did not change the owner of the media" >> $LOGFILE
            actzTrace "XMDA   F: mountmedia.sh: No userid passed. Did not change the owner of the media"

            if [ -z $R0 ]
            then

                if ! chmod -Rv $PERMS $MOUNTPOINT >> $LOGFILE 2>&1; then

                    echo "mountmedia.sh: Did not change the permissions of the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Did not change the permissions of the filesystem on the media"

                else

                    echo "mountmedia.sh: Changed the permissions of the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Changed the permissions of the filesystem on the media"

                fi
            fi

        else  # Userid was passed, try to change the owner of the filesystem

            if [ -z $RO ]
            then

                if ! chown -Rv $USERID $MOUNTPOINT >> $LOGFILE 2>&1; then

                    echo "mountmedia.sh: Userid was passed but did not change the owner of the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Userid was passed but did not change the owner of the media at $MOUNTPOINT"


                else

                    echo "mountmedia.sh: Userid was passed and changed the owner of the media to $USERID" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Userid was passed and changed the owner of the media at $MOUNTPOINT"

                fi

                if ! chmod -Rv $PERMS $MOUNTPOINT >> $LOGFILE 2>&1; then

                    echo "mountmedia.sh: Did not change the permissions of the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Did not change the permissions of the filesystem on the media"

                else

                    echo "mountmedia.sh: Changed the permissions on the filesystem on the media" >> $LOGFILE
                    actzTrace "XMDA   F: mountmedia.sh: Changed the permissions of the filesystem on the media"

                fi

            fi

        fi

    fi

else # mount failed

    echo "mountmedia.sh: Couldn't mount the media at $MOUNTPOINT" >> $LOGFILE
    actzTrace "XMDA   F: mountmedia.sh: Couldn't mount the media at $MOUNTPOINT"

    echo "mountmedia.sh: The contents of the temp file..." >> $LOGFILE

    echo "***$TEMPFILE***" >> $LOGFILE
    cat $TEMPFILE >> $LOGFILE
    echo "***$TEMPFILE***" >> $LOGFILE

    if grep -q -i "wrong major or minor number" $TEMPFILE; then

       	echo "mountmedia.sh: Hardware failure on the media drive...exiting." >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: Hardware failure on the media drive...exiting."

        echo "<- mountmedia.sh for `date`" >> $LOGFILE
        echo "" >> $LOGFILE

        actzTrace "XMDA   T: <- mountmedia.sh for `date`"

        exit 4

    elif (grep -q -i "No medium found" $TEMPFILE) || (grep -q -i "is not a valid block device" $TEMPFILE); then

       	echo "mountmedia.sh: No media in the device...exiting." >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: No media in the device...exiting."

        echo "<- mountmedia.sh for `date`" >> $LOGFILE
        echo "" >> $LOGFILE

        actzTrace "XMDA   T: <- mountmedia.sh for `date`"

        exit 5

    elif (grep -q -i "wrong fs type" $TEMPFILE) || (grep -q -i "you must specify the filesystem type" $TEMPFILE); then

       	echo "mountmedia.sh: Wrong filesystem type, or unformatted media, or wrong media type...exiting." >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: Wrong filesystem type, or unformatted media, or wrong media type...exiting."

        echo "<- mountmedia.sh for `date`" >> $LOGFILE
        echo "" >> $LOGFILE

        actzTrace "XMDA   T: <- mountmedia.sh for `date`"

        exit 6

    elif grep -q -i "is write-protected but explicit" $TEMPFILE; then

       	echo "mountmedia.sh: Attempted to mount read-only media as read-write...exiting." >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: Attempted to mount read-only media as read-write...exiting."

        echo "<- mountmedia.sh for `date`" >> $LOGFILE
        echo "" >> $LOGFILE

        actzTrace "XMDA   T: <- mountmedia.sh for `date`"

        exit 7

    else
       	
        echo "mountmedia.sh: Unknown error...exiting." >> $LOGFILE
        actzTrace "XMDA   F: mountmedia.sh: Unknown error...exiting."

        echo "<- mountmedia.sh for `date`" >> $LOGFILE
        echo "" >> $LOGFILE

        actzTrace "XMDA   T: <- mountmedia.sh for `date`"

    	exit 1
    fi
fi

echo "<- mountmedia.sh for `date`" >> $LOGFILE
echo "" >> $LOGFILE

actzTrace "XMDA   T: <- mountmedia.sh for `date`"

exit 0

