package OSInstall::AIX_Remote;

use strict;
use File::Copy;
use File::Path;
use File::Find;
use Cwd 'abs_path';
use IO::Socket;
use Sys::Hostname;
use OSInstall::Remote_Resource;
use OSInstall::Common;
#use OSInstall::Platform_Tools;
use Carp;

use base ("OSInstall::Remote_Resource");

sub define
{
	my $class= shift;
	my $this= $class->SUPER::define(@_);

	# now that we have an Remote_Resource object, bless the reference
	# into an AIX_Remote object so that we can perform the steps
	# to verify the remote server and install resources using the
	# overridden methods
	bless $this, $class;

	# verify the remote resource, but no nim check (only in allocate)
	unless ($this->verify(1))
	{
		confess __("Failed to validate Remote_Resource because the remote server or resource is unavailable or incomplete\n");
		return 0;
	}

	$this->{ready} = 1;
	
	$this;
}

sub allocate
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my $rc= 1;
	my $remote_rc= 1;
	my $if1_str;

	my @clientdef;
	my $ref_clientdef= \@clientdef;
	
	# make sure our argument is a reference to the right type
	unless (ref($client_ref) =~ /^(?:OSInstall::)?Client$/)
	{
		log_entry(__("Invalid reference type: %s  Expected: %s\n", ref($client_ref), 'OSInstall::Client'), 1);
		return 0;
	}

	# verify the remote resource, including nim check
	unless ($this->verify())
	{
		log_entry __("Failed to validate Remote_Resource because the remote server or resource is unavailable or incomplete\n");
		return 0;
	}

	# NIM mac_addr is without ":" separators
	my $nim_mac_addr= $client_ref->mac_addr;
	$nim_mac_addr=~ s/://g;

	# commands
	my $SSH= find_command('ssh');

	unless ($OSInstall::ssh_keyfile eq "") {
		# must include the SSH key file, if found
		$SSH .= " -i $OSInstall::ssh_keyfile";
	}

	my $lsnim_client_cmd= "lsnim -l ".$client_ref->name;

	# netboot_kernel=64 is used for now, but may be problematic on AIX 5.3
	# problems have been encountered with 10GB RAM and netboot_kernel 32/mp
	my $nim_define_cmd= "nim -o define -t standalone -a platform=chrp -a netboot_kernel=64 -a if1=\\'find_net ".$client_ref->ip_addr.' '.$nim_mac_addr."\\' -a net_definition=\\'ent ".$client_ref->subnet_mask.' '.$client_ref->gateway."\\' ".$client_ref->name;

	my $nim_allocate_cmd= "nim -o allocate -a group=".$this->remote_identifier.' '.$client_ref->name;
	my $nim_reset_cmd= "nim -o reset -a force=yes ".$client_ref->name;
	my $nim_deallocate_cmd= "nim -Fo deallocate -a subclass=all ".$client_ref->name;
	# accept_licenses=yes is specified, but not explicitly asked of the user
	# it may be required, at some point, to explicitly ask for acceptance
	my $nim_bosinst_lpp_cmd= "nim -o bos_inst -a source=rte -a boot_client=no -a accept_licenses=yes -a installp_flags=-agX ".$client_ref->name;
	my $nim_bosinst_mksysb_cmd= "nim -o bos_inst -a source=mksysb -a boot_client=no -a accept_licenses=yes -a installp_flags=-agX ".$client_ref->name;
	my $nim_remove_cmd= "nim -o remove ".$client_ref->name;


	my $ssh_client_hostname_cmd= "$SSH root@".$this->server." host ";

	my $ssh_client_cmd= "$SSH root@".$this->server.' '.$lsnim_client_cmd;
	my $ssh_define_cmd= "$SSH root@".$this->server.' '.$nim_define_cmd;
	my $ssh_allocate_cmd= "$SSH root@".$this->server.' '.$nim_allocate_cmd;
	my $ssh_reset_cmd= "$SSH root@".$this->server.' '.$nim_reset_cmd;
	my $ssh_deallocate_cmd= "$SSH root@".$this->server.' '.$nim_deallocate_cmd;
	my $ssh_bosinst_lpp_cmd= "$SSH root@".$this->server.' '.$nim_bosinst_lpp_cmd;
	my $ssh_bosinst_mksysb_cmd= "$SSH root@".$this->server.' '.$nim_bosinst_mksysb_cmd;
	my $ssh_remove_cmd= "$SSH root@".$this->server.' '.$nim_remove_cmd;


	# check if client already exists on the remote server and
	# obtain long listing
	if (@clientdef= `$ssh_client_cmd`)
	{
		# check that the object is a standalone machine
		my $is_machine= 1;
		$is_machine= 0 unless ($ref_clientdef->[1] =~ /class\s*=\s*machines/);
		$is_machine= 0 unless ($ref_clientdef->[2] =~ /type\s*=\s*standalone/);

		if ($is_machine == 0)
		{
			# just log a message and return error
			log_entry(__("NIM object %s already exists on remote resource server %s, but is not a standalone machine\n", $client_ref->name, $this->server), 1);
			return 0;
		}
		else
		{
			# check same IP/MAC addresses as client machine and
			# the NIM object machine is "ready for a NIM operation"
			# the MAC address is allowed to be zero (not using DHCP)
			# the IP address is allowed to be the client host name
			my $each_def_line;
			my $if1line;
			my $Cstateline;
			my $ent;
			foreach $each_def_line (@clientdef)
			{
				$if1line = $each_def_line if ($each_def_line =~ /if1*\s*=\s*/);
				$Cstateline = $each_def_line if ($each_def_line =~ /Cstate*\s*=\s*/);
			}
			unless ($if1line =~ /if1\s*=\s*/)
			{
				log_entry(__("NIM object machine %s already exists on remote resource server %s, but has no network configuration if1\n", $client_ref->name, $this->server), 1);
				return 0;
			}

			my ($if1, $eq, $net, $ipaddr, $macaddr, $ent)= split ' ', $if1line;
			unless (($macaddr == 0) || ($macaddr eq $nim_mac_addr) || ((lc $macaddr) eq (lc $nim_mac_addr)) )
			{
				log_entry(__("NIM object machine %s already exists on remote resource server %s, but has different MAC address: expected %s, found %s; IP address is %s\n", $client_ref->name, $this->server, $nim_mac_addr, $macaddr, $ipaddr), 1);
				return 0;
			}
			unless ($ipaddr eq $client_ref->ip_addr)
			{
			    # the client ipaddr might be the correct host name
			    # check using the host command on resource server
			    # e.g. "host corrals"  produces
			    #      corrals.<domain> has address <ip_addr>
			    # if ipaddr is an unknown IP address, then command
			    #      "host ipaddr"  produces
			    #      <inverted_ipaddr>.in-addr.arpa domain
			    #                     name pointer <name>.<domain>
			    $ssh_client_hostname_cmd .= $ipaddr;
			    my $client_hostname= `$ssh_client_hostname_cmd`;
			    # check host command returned our client ip_addr
			    unless ($client_hostname =~ $client_ref->ip_addr)
			    {
				log_entry(__("NIM object machine %s already exists on remote resource server %s, but has different IP address: expected %s, found %s; MAC address is %s\n", $client_ref->name, $this->server, $client_ref->ip_addr, $ipaddr, $macaddr), 1);
				return 0;
			    }
			}

			# check client is ready for a NIM operation
			#
			# if client is ready, there may already be allocations,
			# but that is allowed
			#
			# if client is not ready, this probably indicates a
			# failed previous bos_inst (or worse), so have to
			# reset client state to ready and eliminate existing
			# allocations, if any

			unless ($Cstateline =~ /Cstate\s*=\s*ready for a NIM operation/)
			{
				# if the force flag is NOT specified, exit
				unless ($OSInstall::force_flag) {
					log_entry(__("NIM object machine %s already exists on remote resource server %s, but Cstate is not ready, use option -F to force a NIM object reset\n", $client_ref->name, $this->server), 1);
					return 0;
				}
				# reset client state
				$remote_rc= `$ssh_reset_cmd`;
				unless ($? == 0)
				{
					log_entry(__("NIM object machine %s already exists on remote resource server %s, but cannot reset Cstate to ready :\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
					return 0;
				}
				# deallocate resources from client, if any
				$remote_rc= `$ssh_deallocate_cmd`;
				unless ($? == 0)
				{
					log_entry(__("NIM object machine %s already exists on remote resource server %s, but cannot deallocate all resources\n%s\n", $client_ref->name, $this->server,  $remote_rc), 1);
					return 0;
				}
			}
		# pre-existing NIM client machine is now ready for allocation
		}
	}
	else
	{
		# define client on NIM remote resource server
		$remote_rc= `$ssh_define_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to define client %s on the AIX NIM master remote resource server %s\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
			return 0;
		}
	}


	# allocate resource group to client
	$remote_rc= `$ssh_allocate_cmd`;
	if ($? != 0)
	{
		# if allocation fails, log error and remove NIM client
		log_entry(__("Failed to allocate remote resource group %s to client %s\n%s\n", $this->remote_identifier, $client_ref->name, $remote_rc), 1);
		$remote_rc= `$ssh_deallocate_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to deallocate all resources from NIM client %s on remote resource server %s\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
		}
		$remote_rc= `$ssh_remove_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to remove NIM client %s on remote resource server %s\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
		}
		return 0;
	}


	my $bos_inst_ok= 0;
	# recheck client on the remote server to see if mksysb or lpp resource
	# (mksysb has precedence over lpp) and run the NIM bos_inst command
	@clientdef= `$ssh_client_cmd`;
	if (grep /mksysb\s*=/, @clientdef)
	{
		$remote_rc= `$ssh_bosinst_mksysb_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to bos_inst mksysb NIM remote resource client %s\n%s\n", $client_ref->name, $remote_rc), 1);
		}
		else
		{
			$bos_inst_ok= 1;
		}
	}
	elsif (grep /lpp_source\s*=/, @clientdef)
	{
		$remote_rc= `$ssh_bosinst_lpp_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to bos_inst lpp_source NIM remote resource client %s\n%s\n", $client_ref->name, $remote_rc), 1);
		}
		else
		{
			$bos_inst_ok= 1;
		}
	}

	unless ($bos_inst_ok)
	{
		# if lsnim or bos_inst fails, deallocate and remove NIM client
		$remote_rc= `$ssh_deallocate_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to deallocate all resources from NIM client %s on remote resource server %s\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
		}
		$remote_rc= `$ssh_remove_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to remove NIM client %s on remote resource server %s\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
		}
		return 0;
	}


	# add the client to the list of allocated clients, if not there already
	$this->clients_allocated->{$client_ref->{name}}= $client_ref;
	
	$client_ref->allocation($this) if ($rc);

	$rc;
}

sub deallocate
{
	# shift args
	my $this= shift;
	my $client_ref= shift;
	my %tmp_clients_allocated= ();

	# local data
	my $rc= 1;
	my $remote_rc= 1;
	
	my @resalloc;
	my $ref_resalloc= \@resalloc;

	# make sure our argument is a reference to the right type
	unless (ref($client_ref) =~ /^(?:OSInstall::)?Client$/)
	{
		log_entry(__("Invalid reference type: %s  Expected: %s\n", ref($client_ref), 'OSInstall::Client'), 1);
		return 0;
	}

	# commands
	my $SSH= find_command('ssh');

	unless ($OSInstall::ssh_keyfile eq "") {
		# must include the SSH key file, if found
		$SSH .= " -i $OSInstall::ssh_keyfile";
	}

	my $lsnim_client_cmd= "lsnim -l ".$client_ref->name;
	my $lsnim_clientres_cmd= "lsnim -c resources ".$client_ref->name;
	my $nim_deallocate_cmd= "nim -Fo deallocate -a subclass=all ".$client_ref->name;
	my $nim_reset_cmd= "nim -o reset -a force=yes ".$client_ref->name;
	my $nim_remove_cmd= "nim -o remove ".$client_ref->name;

	my $ssh_client_cmd= "$SSH root@".$this->server.' '.$lsnim_client_cmd;
	my $ssh_clientres_cmd= "$SSH root@".$this->server.' '.$lsnim_clientres_cmd;
	my $ssh_deallocate_cmd= "$SSH root@".$this->server.' '.$nim_deallocate_cmd;
	my $ssh_reset_cmd= "$SSH root@".$this->server.' '.$nim_reset_cmd;
	my $ssh_remove_cmd= "$SSH root@".$this->server.' '.$nim_remove_cmd;

	# obtain long listing for the client on the remote server
	$remote_rc= `$ssh_client_cmd`;
	unless ($? == 0)
	{
		log_entry(__("Failed to obtain status for client %s on remote resource server %s\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
		# this may not be sufficient reason to prevent successful
		# deallocation in the OS_install data base,
		# but the design calls for an error return
		return 0;
	}

	# obtain long listing of client's allocated resources on the
	# remote server
	@resalloc= `$ssh_clientres_cmd`;
	unless ($? == 0)
	{
		log_entry(__("Failed to obtain allocated resource list for client %s on remote resource server %s\n%s\n", $client_ref->name, $this->server, @resalloc), 1);
		# this may not be sufficient reason to prevent successful
		# deallocation in NIM and the OS_install data base,
		# but this NIM command should have worked, return an error
		return 0;
	}

	# if the client has no allocated resources on the remote server
	# do not consider it an error, just log the fact and continue
	unless ($ref_resalloc->[0] ne "")
	{
		log_entry(__("No resources allocated for client %s on remote resource server %s\n", $client_ref->name, $this->server), 0);
	}
	else
	{
		# could check that the resources correspond to group
		# resource members, but suppose not required (for now)
		# the "-a subclass=all" deallocate command deallocates
		# all resources and does not return an error if none
		# were allocated
		# reset first in case the Cstate is not "ready"

		$remote_rc= `$ssh_reset_cmd`;
		$remote_rc= `$ssh_deallocate_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to deallocate remote resource %s from client %s\n%s\n", $this->name, $client_ref->name, $remote_rc), 1);
			return 0;
		}
	}

	# because of naming problems in LDW sysplans, remove the NIM client
	# IF the OS_install Client name is not the IP address hostname

	# find if Client ip_addr can be resolved to a hostname
	my $temp_iaddr = inet_aton($client_ref->ip_addr);
	my $temp_client_name  = gethostbyaddr($temp_iaddr, AF_INET);

	if (($?) || ($temp_client_name eq ""))
	{
		log_entry(__("gethostbyaddr() did not find hostname for Client IP address %s, removing NIM client definition\n", $client_ref->ip_addr), 0);
		$temp_client_name = "";
	}

	else
	{
        	# find short name
        	$temp_client_name =~  s/\..*$// ;
		log_entry(__("gethostbyaddr() found hostname %s for Client IP address %s\n", $temp_client_name, $client_ref->ip_addr), 0);
	}

	# OS_install Client name is not IP address hostname, remove NIM client
	if ($temp_client_name ne $client_ref->name)
	{
		$remote_rc= `$ssh_remove_cmd`;
		unless ($? == 0)
		{
			log_entry(__("Failed to remove NIM client %s on remote resource server %s\n%s\n", $client_ref->name, $this->server, $remote_rc), 1);
		}
	}

	# remove the client from the list of allocated clients
	%tmp_clients_allocated= %{$this->clients_allocated};
	delete $tmp_clients_allocated{$client_ref->name};
	$this->clients_allocated(\%tmp_clients_allocated);

	$rc;
}

sub verify
{
	# shift args
	my $this= shift;
	my $skipnimcheck= shift;

	# local data
	my $i= 0;
	my $missing= 0;
	my $res_spot= 0;
	my $res_lpp= 0;
	my $res_mksysb= 0;
	my $res_bosinst= 0;
	my $remote_rc= 1;

	my @resgrp;
	my $ref_resgrp= \@resgrp;

	# commands
	my $SSH= find_command('ssh');

	unless ($OSInstall::ssh_keyfile eq "") {
		# must include the SSH key file, if found
		$SSH .= " -i $OSInstall::ssh_keyfile";
	}

	my $lsnim_resgrp_cmd= "lsnim -g ".$this->remote_identifier;
	my $lsnim_objops_cmd= "lsnim -O ";
	my $nim_check_cmd= "nim -o check ";

	my $ssh_verify_cmd= "$SSH root@".$this->server.' '.$lsnim_resgrp_cmd;
	my $ssh_objops_cmd= "$SSH root@".$this->server.' '.$lsnim_objops_cmd;
	my $ssh_check_cmd= "$SSH root@".$this->server.' '.$nim_check_cmd;

	my $ssh_test_cmd= "$SSH -o StrictHostKeyChecking=no -o ConnectTimeout=10 -o PasswordAuthentication=no root@".$this->server." ls";


	# perform some general ssh checks when defining a remote resource
	# check local ssh command is available
	unless ($SSH)
	{
		log_entry(__("Local ssh command not found\n"), 1);
		return 0;
	}

	# check using ls command that SSH service configured with remote server
	# using PasswordAuthentication=no and ConnectTimeout=10 (seconds)
	# and StrictHostKeyChecking=no to automatically add server RSA key
	# fingerprint to the list of known hosts (this latter does not work
	# if there is a stale key fingerprint already in the list)
	$remote_rc= `$ssh_test_cmd`;
	unless ($? == 0)
	{
		log_entry(__("NIM master remote resource server %s, ssh connection error : %s\n", $this->server, $remote_rc), 1);
		return 0;
	}


	# obtain long listing for the resource group on the remote server
	@resgrp= `$ssh_verify_cmd`;

	# check that the resource group exists on the remote server
	$missing= 1 unless ($ref_resgrp->[0] =~ "^".$this->remote_identifier.":");
	$missing= 1 unless ($ref_resgrp->[1] =~ /class\s*=\s*groups/);
	$missing= 1 unless ($ref_resgrp->[2] =~ /type\s*=\s*res_group/);

	if ($missing) {
		log_entry(__("The remote resource group %s is not available on server %s\n", $this->remote_identifier, $this->server), 1);
		return 0;
	}

	# check that the resource group contains expected resource members
	my $member_objops_cmd;
	my $member_check_cmd;
	my $member_name;
	for ( $i = 3 ; $i < @$ref_resgrp ; $i++ )
	{
	    $res_spot= 1 if ($ref_resgrp->[$i] =~ /member\d*\s*=.*spot;\s*ready for use;/);
	    $res_lpp= 1 if ($ref_resgrp->[$i] =~ /member\d*\s*=.*lpp_source;\s*ready for use;/);
	    $res_mksysb= 1 if ($ref_resgrp->[$i] =~ /member\d*\s*=.*mksysb;\s*ready for use;/);
	    $res_bosinst= 1 if ($ref_resgrp->[$i] =~ /member\d*\s*=.*bosinst_data;\s*ready for use;/);

	    # only perform nim check if called for allocate operation
	    unless ($skipnimcheck == 1)
	    {
		$member_name= $ref_resgrp->[$i];
		# be careful, resource names may not be just alphanumeric
		$member_name=~ s/^\s*member\d*\s*=\s*(\S+);.*$/$1/;
		if ($member_name eq $ref_resgrp->[$i])
		{
		    log_entry(__("Cannot extract resource name from resource member %s in the remote resource group %s on server %s\n", ($ref_resgrp->[$i], $this->remote_identifier, $this->server), 1));
		    $missing= 1;
		    next;
		}
		$member_objops_cmd= "$ssh_objops_cmd".$member_name;
		$remote_rc= `$member_objops_cmd`;
		# nim -o check cannot be run for all resource types
		if ($remote_rc =~ /check\s+=\s+check the status of a NIM object/)
		{
		    # check resource
		    $member_check_cmd= "$ssh_check_cmd".$member_name;
		    $remote_rc= `$member_check_cmd`;
		    # return codes 0 and 1=>allocated ($?==256) are success
		    unless (($? == 0) || ($? == 256)) {
			log_entry(__("The resource %s in the remote resource group %s on server %s is not ready for use\n", ($ref_resgrp->[$i], $this->remote_identifier, $this->server), 1));
			$missing= 1;
		    }
		}
	    }
	}

	unless ($res_spot) {
		log_entry(__("No spot resource in the remote resource group %s on server %s\n", $this->remote_identifier, $this->server), 1);
		$missing= 1;
	}

	unless ($res_bosinst) {
		log_entry(__("No bosinst_data resource in the remote resource group %s on server %s\n", $this->remote_identifier, $this->server), 1);
		$missing= 1;
	}

	unless ($res_lpp || $res_mksysb) {
		log_entry(__("No lpp_source or mksysb resource in the remote resource group %s on server %s\n", $this->remote_identifier, $this->server), 1);
		$missing= 1;
	}

	return 0 if ($missing);

	return 1;
}

sub get_install_status
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	return $this->monitor($client_ref);
}


sub monitor
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my @new_status= ();

	# commands
	my $SSH= find_command('ssh');

	unless ($OSInstall::ssh_keyfile eq "") {
		# must include the SSH key file, if found
		$SSH .= " -i $OSInstall::ssh_keyfile";
	}

	my $lsnim_info_cmd= "lsnim -a info ".$client_ref->name;
	my $ssh_monitor_cmd= "$SSH root@".$this->server.' '.$lsnim_info_cmd;
	my $lsnim_Cstate_cmd= "lsnim -l ".$client_ref->name." | grep Cstate";
	my $ssh_Cstate_cmd= "$SSH root@".$this->server.' '.$lsnim_Cstate_cmd;

	return 0 unless (ref($client_ref) =~ /^(?:OSInstall::)Client/);

	my $infoline= `$ssh_monitor_cmd`;
	$infoline=~ s/^\s*info\s*=//;
	push @new_status, $infoline;

	# return a string indicating the installation is finished
	# if the server no longer reports an "info" line
	if (grep /^$/, @new_status)
	{
		my $Cstateline= `$ssh_Cstate_cmd`;
		if (($Cstateline=~ /^\s*Cstate\s*=\s+BOS installation/) ||
		    ($Cstateline=~ /^\s*Cstate\s*=\s+Base Operating System installation is being performed/))
		{
			# the info line is not yet available, but Cstate
			# indicates that a bos_inst operation had been run
			push @new_status, $Cstateline;
		}
		else
		{
			# no info line available, this is probably a user
			# command line run outside of an install context
			push @new_status, "No installation context information available\n";
			push @new_status, $Cstateline;
			push @new_status, 'FINISHED';
		}
	}

	# add a string indicating the installation is finished
	# if the server reported 100% BOS install complete
	push @new_status, 'FINISHED' if (grep /100\%/, @new_status);

	# returning status every 5 seconds in Client.pm is
	# probably too much on "slow machines/lpars", add 5 more
	sleep 5;

	\@new_status;
}

sub destroy
{
	# shift args
	my $this= shift;
	my $resource= shift;

	# the NIM master resource group is not to be modified
	1;
}

1;
__END__


