package OSInstall::Control_Host;

use strict;
use warnings;
use IO::Socket;
use Sys::Hostname;
use OSInstall::Client;
use OSInstall::Common;
use OSInstall::XMLHandler;
use Carp;
our $AUTOLOAD;

my %modifiable_fields= (
		name => "",
		communication_method => "ssh",
		host_name => "",
		type => "",
	);

my %constant_fields= ();

sub new
{
	my $class_name= shift;
	my $this; #what we will bless into an object
	
	#The first 3 args are required
	confess "usage OSInstall::Control_Host->new(name, host_name, type [,communication_method])\n" unless (@_ >= 3);

	$modifiable_fields{name}= shift;
	$modifiable_fields{host_name}= shift;
	$modifiable_fields{type}= shift;
	$modifiable_fields{communication_method}= shift if (@_);

	# turn type and comm_method into lowercase
	$modifiable_fields{type}= lc $modifiable_fields{type};
	$modifiable_fields{communication_method}= lc $modifiable_fields{communication_method};

	# see AUTOLOAD below for what _modifiable does- it's not meant for direct external use
	$this= {
		_modifiable => \%modifiable_fields,
		%modifiable_fields,
		%constant_fields,
	};

	bless $this, $class_name;
	return $this;
}

sub AUTOLOAD
{
	my $this= shift; #ref to the class
	my $class_name= ref($this);
	my $param= shift; 

	#must take a reference to this class
	croak "Invalid reference type\n" unless ($class_name =~ /^(?:OSInstall::)?Control_Host$/);

	my $attr= $AUTOLOAD; #the field we're autoloading
	$attr =~ s/.*://; #strip out full package name

	#error if we don't have this field
	croak "The field \"$attr\" does not exist\n" unless (exists $this->{$attr});

	# if we have a parameter to set the attr to, make sure it's a modifiable attr
	if ($param)
	{
		# croak if this field is not modifiable
		croak "This field cannot be modified\n" unless ($this->{_modifiable}->{$attr});

		$this->{$attr}= $param; #damage is done
	}

	return $this->{$attr};
}

sub netboot
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my $rc= 1;
	my $lpar_netboot_cmd= 'lpar_netboot -i [arg1] [arg11] -t ent -m [arg2] -D -s [arg3] -d [arg4] -S [arg5] -G [arg6] -C [arg7] -K "[arg12]" "[arg8]" "[arg9]" "[arg10]"';
	my @data_params= qw(debug mac_addr speed duplex server_addr gateway client_addr lpar profile managed_sys boot_params subnet_mask);
	my $lpar_netboot_debug= $ENV{OSINSTALL_BOOT_DEBUG} ? '-v -x' : '';  # lpar_netboot debug mode (lots of extra output)
	my $mac_addr_param= $client_ref->mac_addr;
	my %data_hash= ();
	my $boot_params= $client_ref->boot_params ? "-g \"".$client_ref->boot_params."\"" : "";

	# commands
	my $SSH= find_command('ssh');

	unless ($OSInstall::ssh_keyfile eq "") {
		# must include the SSH key file, if found
		$SSH .= " -i $OSInstall::ssh_keyfile";
	}



	return 0 unless (ref($client_ref) =~ /^(?:OSInstall::)Client$/);

	# strip out any :'s from the MAC address
	$mac_addr_param =~ s/://g;

	# make sure this client has p5 lpar attributes set
	unless ($client_ref->lpar && $client_ref->profile && $client_ref->managed_sys)
	{
		log_entry(__("The client %s does not appear to be an LPAR capable system and cannot be netbooted\n", $client_ref->name), 1);
		return 0;
	}

	# check for net adapter information
	unless ($client_ref->speed && $client_ref->duplex)
	{
		log_entry(__("The client %s cannot be netbooted because its network adapter setings have not been configured\n", $client_ref->name), 1);
		return 0;
	}

	# fill in the hash containing the arg data, then build the cmd-line string
	%data_hash= (
		debug => $lpar_netboot_debug,
		mac_addr => $mac_addr_param,
		speed=> $client_ref->speed,
		duplex => $client_ref->duplex,
		# FIXME: server_addr should be ip of resource server, not control host
		# setting value to hostname of machine that osinstall is running on
		server_addr => inet_ntoa(inet_aton(hostname)),
		gateway => $client_ref->gateway,
		client_addr => $client_ref->ip_addr,
		subnet_mask => $client_ref->subnet_mask,
		lpar => $client_ref->lpar,
		profile => $client_ref->profile,
		managed_sys => $client_ref->managed_sys,
		boot_params => $boot_params,
	);

	# check for an OS resource allocation, OS_install -netboot does not
	# check for one and neither does the Client class netboot method
	# if allocation is a Remote_Resource resource, get ip for server_addr
	if ($client_ref->allocation)
	{
	    # instantiate client->allocation
	    $client_ref->allocation(XH_read_object($client_ref->allocation));

	    if (ref($client_ref->allocation) =~ /^(?:OSInstall::)?\w+_Remote$/)
	    {
		my $ser_nam = $client_ref->allocation->server;
		my $ser_addr = gethostbyname("$ser_nam");
		$data_hash{"server_addr"} = inet_ntoa($ser_addr);
	    }
	}

	$lpar_netboot_cmd= fill_in_args($lpar_netboot_cmd, build_param_list(\%data_hash, \@data_params));

	my $ssh_user;    # ssh username
	my $prefix_path; # path to prefix lpar_netboot command for local executions

	if ( $this->type =~ /^[Hh][Mm][Cc]/ ) {
		$ssh_user    = "hscroot";
		$prefix_path = "/opt/hsc/bin";
	} elsif ( $this->type =~ /^[Ii][Vv][Mm]/ ) {
		$ssh_user    = "padmin";
		$prefix_path = "/opt/ibm/sysmgt/dsm/dsmbin";
	} else {
		log_entry(__("Control Host type '%s' is invalid\n", $this->type), 1);
		return 0;
	}

	unless ($this->host_name =~ /^localhost/)
	{
		# lpar_netboot is accessed through ioscli on the IVM
		$lpar_netboot_cmd = "ioscli ".$lpar_netboot_cmd if ($this->type =~ /^[Ii][Vv][Mm]/);

		# enclose everything in '' in case any wierd characters come in somehow (like through boot_params)
		$lpar_netboot_cmd= "'".$lpar_netboot_cmd."'";

		$lpar_netboot_cmd= "$SSH $ssh_user@".$this->host_name.' '.$lpar_netboot_cmd;
	}
	else
	{
		# if localhost is the Control_Host, then we are running on an HMC or IVM
		$lpar_netboot_cmd = $prefix_path."/".$lpar_netboot_cmd;
	}

	$rc= log_cmd($lpar_netboot_cmd);

	$rc;
}

1;

