package OSInstall::Platform_Tools;

use strict;
use warnings;
use Carp;
use FileHandle;
use Fcntl qw(:DEFAULT :flock);
use OSInstall::Common;
use OSInstall::OS_Resource;
use OSInstall::Client;
use File::Copy;
use IO::Socket;
use Sys::Hostname;

require Exporter;
our @ISA = qw(Exporter);

our @EXPORT_OK = ();

our @EXPORT = qw(
$SERVICE_STATE_DIR
$CONFIG_FILES_DIR
$CONFIG_DATA_DIR
$UNDO_CMD_STACK
set_config_rules
start_daemons
refresh_daemons
config_bootp
export_resource
unexport_resource
unconfig_bootp
stop_daemons
get_system_os
enforce_firewall_rules
flush_firewall_rules
);

my @services= qw(nfs bootp tftp inetd);

# declaration of constants
my $COMMENT_BEGIN_TAG= "__BEGIN_OSINSTALL_TAG__";
my $COMMENT_END_TAG= "__END_OSINSTALL_TAG__";
our $SERVICE_STATE_DIR= "/var/osinstall/service_state";
our $CONFIG_FILES_DIR= "$SERVICE_STATE_DIR/cfg_save";
our $UNDO_CMD_STACK= "$SERVICE_STATE_DIR/undo_cmd_stack";
our $CONFIG_DATA_DIR= "$SERVICE_STATE_DIR/cfg_data";
my $CONFIG_DELIMITOR= ":-:-:";
my $HOSTNAME=  (gethostbyname(hostname))[0];


#BEGIN
#{
#	# constants
#	$COMMENT_BEGIN_TAG= "__BEGIN_OSINSTALL_TAG__";
#	$COMMENT_END_TAG= "__END_OSINSTALL_TAG__";
#	$SERVICE_STATE_DIR= "/var/osinstall/service_state";
#	$CONFIG_FILES_DIR= "$SERVICE_STATE_DIR/cfg_save";
#	$UNDO_CMD_STACK= "$SERVICE_STATE_DIR/undo_cmd_stack";
#	$CONFIG_DATA_DIR= "$SERVICE_STATE_DIR/cfg_data";
#	$CONFIG_DELIMITOR= "|*|";

	# create needed directories
#	unless (-d $SERVICE_STATE_DIR)
#	{
#		mkdir $SERVICE_STATE_DIR, 0700 or die "$SERVICE_STATE_DIR : $!\n";
#	}
#
#	unless (-d $CONFIG_FILES_DIR)
#	{
#		mkdir $CONFIG_FILES_DIR, 0700 or die "$CONFIG_FILES_DIR : $!\n";
#	}
#}

# we're going to build hashes from the rules file... this one is just an example
my @linux_rules=(
	{
		name => "nfs",
		start_cmd => "/etc/init.d/portmap start && /etc/init.d/nfsserver start",
		stop_cmd => "/etc/init.d/nfsserver stop && /etc/init.d/portmap stop",
		refresh_cmd => "/etc/init.d/nfsserver restart",
		status_cmd => "/etc/init.d/portmap status && /etc/init.d/nfsserver status",
		config_file => "/etc/exports",
		comment_pattern => "# [arg1]",
		add_element_pattern => "[arg1]	[arg2]([arg3],insecure,no_root_squash)",
		data_params => ['directory', 'host', 'permissions'],
		started => 0,
	},

	{
		name => "bootp",
		start_cmd => "touch /etc/nimoltab && /usr/sbin/nimol_bootreplyd -s $HOSTNAME -l -d",
		status_cmd => "[[ -s /var/nimol_bootreplyd/pid && -d /proc/`cat /var/nimol_bootreplyd/pid` ]]",
		stop_cmd => "kill `cat /var/nimol_bootreplyd/pid`",
		refresh_cmd => "kill -HUP `cat /var/nimol_bootreplyd/pid`",
		add_element_pattern => "[arg1]:ip=[arg2]:ht=ethernet:gw=[arg3]:sm=[arg4]:bf=[arg5]:sa=[arg6]:ha=[arg7]",
		comment_pattern => "# [arg1]",
		config_file => "/etc/nimoltab",
		mac_addr_strip => 1,
		data_params => ['hostname', 'client_addr', 'gateway', 'subnet_mask', 'filename', 'server_addr', 'mac_addr'],
		started => 0,
	},

	{
		name => "tftp",
		start_cmd => "/usr/sbin/in.tftpd",
		inetd_name => "tftp",
		started => 0,
	},

	# inetd rules- this hash governs how inetd is configured on the system
	# inetd can have a single config file (as is the case with AIX), or
	# - can have a config file for each service it supports in a directory
	#   (as is the case with xinetd on most Linux boxes)
	# if this rule exists, it will receive special processing (since other daemons will be started through it)
	{
		name => 'inetd',
		start_cmd => '/etc/init.d/xinetd start',
		stop_cmd => '/etc/init.d/xinetd stop',
		refresh_cmd => '/etc/init.d/xinetd restart',
		status_cmd => '/etc/init.d/xinetd status',
		config_dir => '/etc/xinetd.d',
		comment_pattern => '# [arg1]',
		add_element_cmd => "perl -e 'while (<>){ s/(disable\\s*=\\s*)yes/\${1}no/;\$new.=\$_ } open(F,\">[arg1]\")or die \"[arg1] : \$!\n\"; print F \$new;close F;' [arg1]",
		rm_element_cmd => "perl -e 'while (<>){ s/(disable\\s*=\\s*)no/\${1}yes/;\$new.=\$_ } open(F,\">[arg1]\")or die \"[arg1] : \$!\n\"; print F \$new;close F;' [arg1]",
		data_params => ['filename',],
		started => 0,
	},

	{
		name => "syslogd",
		start_cmd => "/etc/init.d/syslog start",
		status_cmd => "/etc/init.d/syslog status",
		stop_cmd => "/etc/init.d/syslog stop",
		refresh_cmd => "/etc/init.d/syslog restart",
		add_element_cmd => "perl -e 'while (<>){ s/local2,//;\$new.=\$_ } \$new.=\"local2.*\t\t\t$INSTALL_LOG\n\"; open(F ,\">/etc/syslog.conf\") or die \"/etc/syslog.conf\"; print F \$new' /etc/syslog.conf; sed 's/SYSLOGD_PARAMS=\"\"/SYSLOGD_PARAMS=\"-r\"/' /etc/sysconfig/syslog >/tmp/syslog-sysconfig && mv /tmp/syslog-sysconfig /etc/sysconfig/syslog",
		rm_element_cmd => "perl -e 'while(<>) { s/local3/local2,local3/, \$new.=\$_ unless (/osinstall/) } open(F ,\">/etc/syslog.conf\") or die \"/etc/syslog.conf\"; print F \$new' /etc/syslog.conf; sed 's/SYSLOGD_PARAMS=\"-r\"/SYSLOGD_PARAMS=\"\"/' /etc/sysconfig/syslog >/tmp/syslog-sysconfig && mv /tmp/syslog-sysconfig /etc/sysconfig/syslog",
		started => 0,
	},
	
	);

my @aix_rules= (
	{
		name => "nfs",
		start_cmd => "startsrc -g nfs",
		stop_cmd => "stopsrc -g nfs",
		refresh_cmd => "refresh -g nfs",
		status_cmd => "lssrc -s nfsd | grep active",
		add_element_cmd => "mknfsexp -d [arg1] -t [arg2] -c [arg3] -r [arg3]",
		rm_element_cmd => 'rmnfsexp -d [arg1]',
		data_params => ['directory', 'permissions', 'access_list'],
		host_list => 1,
		started => 0,
	},

	{
		name => "bootp",
		start_cmd => "touch /etc/nimoltab && /usr/sbin/nimol_bootreplyd -s $HOSTNAME -l -d",
		status_cmd => "[[ -s /var/nimol_bootreplyd/pid && -d /proc/`cat /var/nimol_bootreplyd/pid` ]]",
		stop_cmd => "kill `cat /var/nimol_bootreplyd/pid`",
		refresh_cmd => "kill -HUP `cat /var/nimol_bootreplyd/pid`",
		add_element_pattern => "[arg1]:ip=[arg2]:ht=ethernet:gw=[arg3]:sm=[arg4]:bf=[arg5]:sa=[arg6]:ha=[arg7]",
		comment_pattern => "# [arg1]",
		config_file => "/etc/nimoltab",
		mac_addr_strip => 1,
		data_params => ['hostname', 'client_addr', 'gateway', 'subnet_mask', 'filename', 'server_addr', 'mac_addr'],
		started => 0,
	},

	{
		name => "tftp",
		start_cmd => "/usr/sbin/tftpd",
		inetd_name => "tftp",
		started => 0,
	},

	{
		name => 'inetd',
		start_cmd => 'startsrc -s inetd',
		stop_cmd => 'stopsrc -s inetd',
		refresh_cmd => 'refresh -s inetd',
		status_cmd => 'lssrc -s inetd | grep active',
		add_element_cmd => "sed 's/#*[arg1]/[arg1]/' /etc/inetd.conf > /tmp/new-inetd && mv /tmp/new-inetd /etc/inetd.conf",
		rm_element_cmd => "sed 's/^[arg1]/#[arg1]/' /etc/inetd.conf > /tmp/new-inetd && mv /tmp/new-inetd /etc/inetd.conf",
		data_params => ['service_name'],
		started => 0,
	},

	{
		name => "syslogd",
		start_cmd => "startsrc -s syslogd",
		status_cmd => "lssrc -s syslogd | grep active",
		stop_cmd => "stopsrc -s syslogd",
		refresh_cmd => "stopsrc -s syslogd; startsrc -s syslogd",
		add_element_pattern => "local2.debug $INSTALL_LOG rotate",
		comment_pattern => "# [arg1]",
		config_file => "/etc/syslog.conf",
		started => 0,
	},
	);

my %aix_commands= (
	image_copy => {
		cmd => "NIM_METHODS=/usr/lpp/bos.sysmgt/nim/methods . \$NIM_METHODS/c_sh_lib && /usr/sbin/gencopy -X -b \"qv\" -d [arg1] -t [arg2] \$SIMAGES_OPTIONS",
		data_params => ['source', 'target'],
		},
	);

my %linux_commands= (
	image_copy => {
		cmd => "(cd [arg2] && [arg1] -cf - [arg4]) | (cd [arg3] && [arg1] -xf -)",
		data_params => ['command', 'source', 'target', 'copy_list'],
		},
	);

my @rules= @aix_rules;
my %platform_commands= %aix_commands;
my $system_os= 'AIX';
my $is_hmc= 0;

# Below are non-hot-swappable-rule daemons (we'll call them transient).
# This means we cannot ever change the rules for this daemon once they are loaded.
# This is useful for services like NFS, for which it wouldn't be a good idea to 
# momentarily unconfigure all its exports while clients are installing.  Also,
# it's very uncommon (if not impossible) to have multiple implementations of NFS 
# on one system, so chances are we will never need to change NFS's rules.
my %transient_services= ( nfs => 1 );

sub set_config_rules
{
	# hopefully, we only ever need to call this once and never change rules
	# if we have to, for whatever reason, then we can do it here
	if (-e '/usr/sbin/gencopy')
	{
		$system_os= 'AIX';
		@rules= @aix_rules;
	}
	else
	{
		$system_os= 'Linux';
		@rules= @linux_rules;
	}

	$is_hmc= 1 if (-e '/usr/hmcrbin/lssyscfg');

	1;
}

#----------------------------------------------------------------------
=head3 start_daemons

	start any daemons that are not already running

	Arguments:
		none

	Returns:
        	0:        successful
        	non-zero: failed	
		
	

=cut
#----------------------------------------------------------------------



# this function will start any daemons that are not already running
sub start_daemons
{
	my $each_daemon;
	my $config_lock_fh; # FileHandle to the lockfile
	my $inetd_rule_ref; # ref to the inetd rule hash
	my $refresh_inetd_now= 0;
	my %temp_params;
	my $ordered_params;
	my $rc= 1;

	log_entry(__("Starting daemons:\n"));

	# obtain lock for system configuration
	if ($config_lock_fh= lock_file('platform_tools_sysconfig', "exclusive"))
	{
		# find and save the inetd reference before configuring any other daemons
		$inetd_rule_ref= &get_rule_ref('inetd');
	
		# loop through each daemon in @rules
		foreach $each_daemon (@rules)
		{
			# skip the inetd reference
			next if ($each_daemon->{name} eq 'inetd');

			# IF the daemon is managed by inetd
			if ($each_daemon->{inetd_name})
			{
				# error if we have no inetd rule
				unless ($inetd_rule_ref)
				{
					log_entry(__("Cannot configure an inetd service without inetd rules.\n"), 1);
					$rc= 0;
					last;
				}

				# call add_config to add the new inetd configuration with the params in the proper order
				$temp_params{filename}= $inetd_rule_ref->{config_dir}.'/'.$each_daemon->{name} unless ($inetd_rule_ref->{config_file});
				$temp_params{command_name}= $each_daemon->{start_cmd};
				$temp_params{service_name}= $each_daemon->{name};

				# obtain the properly ordered param list
				$ordered_params= build_param_list(\%temp_params, $inetd_rule_ref->{data_params});

				# make sure inetd isn't already configured to run this daemon
				next if (&get_configured_data($inetd_rule_ref, $ordered_params));
				
				if (&add_config($inetd_rule_ref, $ordered_params))
				{
					# refresh/start inetd later
					$refresh_inetd_now= 1;
					$each_daemon->{started}= 1;
				}
				else
				{
					log_entry(__("Error encountered while attempting to configure the %s service.\n", $inetd_rule_ref->{name}), 1);
					$rc= 0;
				}
			}
			# configure syslogd now, if present
			elsif ($each_daemon->{name} eq 'syslogd')
			{
				$ordered_params= []; # none needed for syslogd

				unless (&add_config($each_daemon, $ordered_params))
				{
					log_entry(__("Error encountered while attempting to configure the %s service.\n", 'syslogd'), 1);
					$rc= 0;
				}
				else
				{
					$each_daemon->{started}= 1;
				}
			}
			else
			{
				# run the daemon's start_cmd if it's not already running (told by its status_cmd)
				unless (log_cmd($each_daemon->{status_cmd}, 1))
				{
					unless (log_cmd($each_daemon->{start_cmd}))
					{
						log_entry(__("Error encountered while attempting to start the %s service.\n", $each_daemon->{name}));
						$rc= 0;
					}
					else
					{
						$each_daemon->{started}= 1;
					}
				}
			}
		}

		# if we messed with inetd, refresh it now
		if ($refresh_inetd_now)
		{
			# is inetd even running to begin with? start it if not
			if (log_cmd($inetd_rule_ref->{status_cmd}, 1))
			{
				# inetd is already running- refresh it so it picks up the new config
				unless (log_cmd($inetd_rule_ref->{refresh_cmd}))
				{
					log_entry(__("Error encountered while attempting to refresh the %s service.\n", $inetd_rule_ref->{name}));
					$rc= 0;
				}
			}
			else 
			{
				# inetd is not running- start it now
				unless (log_cmd($inetd_rule_ref->{start_cmd}))
				{
					log_entry(__("Error encountered while attempting to start the %s service.\n", $inetd_rule_ref->{name}));
					$rc= 0;
				}
				else
				{
					$inetd_rule_ref->{started}= 1;
				}
			}	
		}

		# unlock config lockfile
		close $config_lock_fh or confess "$!\n";
	}
	else
	{
		$rc= 0;
	}

	$rc;
}

# this function will call start_daemons and will then issue refresh commands on daemons that were
sub refresh_daemons
{
	my $config_lock_fh; # FileHandle to the lockfile
	my $each_daemon;
	my $inetd_ref= "";
	my $refesh_inetd= 0;
	my $rc= 1;

	log_entry(__("Refreshing daemons:\n"));

	# obtain lock for system configuration
	if ($config_lock_fh= lock_file('platform_tools_sysconfig', "exclusive"))
	{
		# loop through each daemon in @rules
		foreach $each_daemon (@rules)
		{
			if ($each_daemon->{inetd_name})
			{
				# this daemon doesn't need to be refreshed because refreshing
				# inetd will take care of that
				next;
			}
			elsif ($each_daemon->{refresh_cmd})
			{
				# run the daemon's refresh_cmd
				$rc= 0 unless (log_cmd($each_daemon->{refresh_cmd}));
			}
		}

		# unlock config lockfile
		close $config_lock_fh or confess "$!\n";
	}
}

sub config_bootp
{
	# shift args
	my $os_res_ref= shift;
	my $client_ref= shift;
	my $cmd= shift;

	# local data
	my $rc= 1;
	my $config_lock_fh;
	my %args_hash;
	my $bootp_rule_ref= &get_rule_ref('bootp');
	my $mac_addr_param= $client_ref->mac_addr;

	# make sure we've received an OS_Resource and Client object
	return 0 unless (ref($os_res_ref) =~ /^(?:OSInstall::)?\w+_Resource$/ && ref($client_ref) =~ /^(?:OSInstall)?::Client$/);

	# make sure we have a bootp rule
	unless ($bootp_rule_ref)
	{
		log_entry(__("The service %s is not configurable by this application.\n", 'bootp'), 1);
		return 0;
	}

	# strip :'s out of the mac addr if we need to
	$mac_addr_param =~ s/://g if ($bootp_rule_ref->{mac_addr_strip});

	# populate the hash to pass to the add/rm_config functions in the following order:
	# ['hostname', 'client_addr', 'server_addr', 'filename', 'mac_addr', 'gateway']
	%args_hash= (
		hostname => $client_ref->hostname,
		client_addr => $client_ref->ip_addr,
		# if we're calling this function, this host is definitely the install server
		server_addr => inet_ntoa((gethostbyname(hostname))[4]),
		filename => $client_ref->hostname,
		mac_addr => $mac_addr_param,
		gateway => $client_ref->gateway,
		subnet_mask => $client_ref->subnet_mask,
		);
	
	if ($config_lock_fh= lock_file('platform_tools_sysconfig', "exclusive"))
	{
		# check whether we are configuring or unconfiguring a client in bootp 
		if ($cmd && $cmd eq 'rm')
		{
			$rc= &rm_config($bootp_rule_ref, build_param_list(\%args_hash, $bootp_rule_ref->{data_params}));
		}
		else
		{
			$rc= &add_config($bootp_rule_ref, build_param_list(\%args_hash, $bootp_rule_ref->{data_params}));
		}

		close $config_lock_fh;
	}
	else
	{
		$rc= 0;
	}

	$rc;
}

sub export_resource
{
	# shift args
	my $os_res_ref= shift;
	my $client_ref= shift;
	my $client_ref_list= shift;
	my $cmd= shift;

	# local data
	my $rc= 1;
	my $config_lock_fh;
	my %args_hash;
	my $nfs_rule_ref= &get_rule_ref('nfs');
	my $each_client;
	my $access_list= "";

	# make sure we have an nfs rule
	unless ($nfs_rule_ref)
	{
		log_entry(__("The service %s is not configurable by this application.\n", 'nfs'), 1);
		return 0;
	}

	# make sure we've received an OS_Resource and Client object
	return 0 unless (  ref($os_res_ref) =~ /^(?:OSInstall::)?\w+_Resource$/ &&
		( ($client_ref_list && ref($client_ref_list) eq 'ARRAY') || 
			($client_ref && ref($client_ref) =~ /^(?:OSInstall::)?Client$/) )  
		);

	# populate the hash to pass to the add/rm_config function in the following order:
	# ['directory', 'permissions', 'access_list']
	%args_hash= (
		directory => $os_res_ref->image_location,
		permissions => 'ro',
		);

	if ($nfs_rule_ref->{host_list})
	{
		foreach $each_client (values %{$os_res_ref->clients_allocated})
		{
			return 0 unless (ref($each_client) eq 'OSInstall::Client');
			$access_list.= $each_client->hostname.",";
		}

		# chop the trailing ,
		chop $access_list;

		$args_hash{access_list}= $access_list
	}
	else
	{	
		$args_hash{host}= $client_ref->hostname;
	}
	
	if ($config_lock_fh= lock_file('platform_tools_sysconfig', "exclusive"))
	{	
		# check whether we are configuring or unconfiguring an export 
		if ($cmd && $cmd eq 'rm')
		{
			$rc= &rm_config($nfs_rule_ref, build_param_list(\%args_hash, $nfs_rule_ref->{data_params}));
			# Remove the client from the resource's clients_allocated hash
			my %tmp_clients_allocated= %{$os_res_ref->clients_allocated};
			delete $tmp_clients_allocated{$client_ref->name};
			$os_res_ref->clients_allocated(\%tmp_clients_allocated);

			# Now rebuild the access list, minus client to be allocated
			$access_list = "";
			if ($nfs_rule_ref->{host_list})
			{
				foreach $each_client (values %{$os_res_ref->clients_allocated})
				{
					return 0 unless (ref($each_client) eq 'OSInstall::Client');
					$access_list.= $each_client->hostname.",";
				}

				# chop the trailing ,
				chop $access_list;

				$args_hash{access_list}= $access_list
			}
			else
			{	
				$args_hash{host}= $client_ref->hostname;
			}

			# re-export the NFS directory if there's still clients allocated
			$rc= &add_config($nfs_rule_ref, build_param_list(\%args_hash, $nfs_rule_ref->{data_params})) if ($rc && $access_list && $nfs_rule_ref->{host_list});
		}
		else
		{
			$rc= &add_config($nfs_rule_ref, build_param_list(\%args_hash, $nfs_rule_ref->{data_params}));
		}

		close $config_lock_fh;
	}
	else
	{
		$rc= 0;
	}

	$rc;
}

sub unexport_resource
{
	# shift args
	my $os_res_ref= shift;
	my $client_ref= shift;
	my $client_ref_list= shift;

	&export_resource($os_res_ref, $client_ref, $client_ref_list, 'rm');
}

sub unconfig_bootp
{
	# shift args
	my $os_res_ref= shift;
	my $client_ref= shift;

	&config_bootp($os_res_ref, $client_ref, 'rm');
}

sub stop_daemons
{
	my $stop_everything= shift;
	my $inetd_rule_ref= &get_rule_ref('inetd');
	my $config_lock_fh;
	my %temp_params;
	my $ordered_params;
	my $refresh_inetd_now= 0;
	my $each_daemon;
	my $rc= 1;

	log_entry(__("Stopping daemons:\n"));

	# obtain lock for system configuration
	return 0 unless ($config_lock_fh= lock_file('platform_tools_sysconfig', "exclusive"));

	my $UMOUNT = find_command('umount');
	my $MOUNT  = find_command('mount');
	my $GREP   = find_command('grep');

	# unmount cd/dvd drive if mounted
	my $mountcmd = "$MOUNT | $GREP " . $OSInstall::Common::MOUNT_POINT;
	log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($stop_everything  && (`$mountcmd`));

	# turn off each service if we turned it on (with a couple of exceptions)
	foreach $each_daemon (@rules)
	{
		# here, we check if we want to stop everything, or just what this invocation 
		# of OS_install started through an allocation
		next unless ($stop_everything || $each_daemon->{started});

		if ($each_daemon->{inetd_name})
		{
			# error if we have no inetd rule
			unless ($inetd_rule_ref)
			{
				log_entry(__("Cannot configure an inetd service without inetd rules.\n"), 1);
				$rc= 0;
				last;
			}

			# call add_config to remove the inetd configuration with the params in the proper order
			$temp_params{filename}= $inetd_rule_ref->{config_dir}.'/'.$each_daemon->{name} unless ($inetd_rule_ref->{config_file});
			$temp_params{command_name}= $each_daemon->{start_cmd};
			$temp_params{service_name}= $each_daemon->{name};

			# obtain the properly ordered param list
			$ordered_params= build_param_list(\%temp_params, $inetd_rule_ref->{data_params});

			# make sure inetd is configured to run this daemon (so we can undo it)
			next unless (&get_configured_data($inetd_rule_ref, $ordered_params));
			
			if (&rm_config($inetd_rule_ref, $ordered_params))
			{
				# refresh inetd later
				$refresh_inetd_now= 1;
			}
			else
			{
				log_entry(__("Error encountered while attempting to stop the %s service.\n", $inetd_rule_ref->{name}), 1);
				$rc= 0;
			}
		}
		elsif ($each_daemon->{name} eq 'syslogd')
		{
			unless (&rm_config($each_daemon, []))
			{
				log_entry(__("Error encountered while attempting to stop the %s service.\n", $each_daemon->{name}), 1);
				$rc= 0;
			}
		}
		else
		{
			unless (log_cmd($each_daemon->{stop_cmd}))
			{
				log_entry(__("Error encountered while attempting to stop the %s service.\n", $each_daemon->{name}), 1);
				$rc= 0;
			}
		}
	}

	if ($refresh_inetd_now)
	{
		# is inetd even running to begin with? if not, then ignore- we've already modified the config
		if (log_cmd($inetd_rule_ref->{status_cmd}, 1))
		{
			# inetd is already running- refresh it so it picks up the new config
			unless (log_cmd($inetd_rule_ref->{refresh_cmd}))
			{
				log_entry(__("Error encountered while attempting to refresh the %s service.\n", $inetd_rule_ref->{name}));
				$rc= 0;
			}
		}
	}

	close $config_lock_fh;
	$rc;
}

sub get_system_os
{
	return $system_os;
}

sub enforce_firewall_rules
{
	# shift args
	my $client_host= shift;

	&flush_firewall_rules($client_host, 1);  # call the flush firewall function with the enforce flag on
}

sub flush_firewall_rules
{
	#only mess with the firewall if we're running on an HMC
	return 1 unless ($is_hmc);

	# shift args
	my $client_host= shift;
	my $enforce= shift;

	# local data
	my $firewall_cmd= '[arg1] [arg2] INPUT -s [arg3] -j ACCEPT'; # -I accepts all packets
	my $flag= $enforce ? '-D' : '-I';
	
	# commands
	my $IPTABLES= find_command('iptables');
	
	log_cmd(fill_in_args($firewall_cmd, [$IPTABLES, $flag, $client_host]));
}

#########################################
# Private functions called by the above #
#########################################

sub get_rule_ref
{
	#shift args
	my $service_name= shift;

	# local data
	my $each_rule;
	my $rule_ref;

	foreach $each_rule (@rules)
	{
		$rule_ref= $each_rule if ($each_rule->{name} eq $service_name);
	}

	$rule_ref;
}

# This function takes as a parameter a rule hash and either modifies its
# config file or runs a command to modify the state of its associated daemon.
# this function is not reentrant!!! call it after locking the config file only
#
# Parameters:
#	SCALAR: Reference to the rule entry for the daemon
#	SCALAR: Reference to a list of add data (must be properly ordered by now)
sub add_config
{
	# shift the args
	my $rule_ref= shift;
	my $data_ref= shift;	# the data passed here must already be properly ordered using the build_param_list function

	#local data
	my $add_data;
	my $i; 		# loop counter
	my $each_data;
	my $rc= 1;

	# if we've already configured this exact setting, just return success
	return $rc if (&get_configured_data($rule_ref, $data_ref));

	# build the add_data string- insert the arguments into the [arg#] fields of the command or 
	# data string
	$add_data= fill_in_args($rule_ref->{add_element_pattern} ? $rule_ref->{add_element_pattern} : $rule_ref->{add_element_cmd}, $data_ref);

	# first save the original state of this daemon so we can restore it later if we need to
	$rc= &save_undo_data($rule_ref, $data_ref);
	return $rc unless ($rc);

	# if we're adding text to the config file
	if ($rule_ref->{add_element_pattern})
	{
		# actually modify the config file using the format string we've just filled in
		$rc= &add_config_to_file($rule_ref->{config_file}, $add_data, $rule_ref->{comment_pattern});

		# append and prepend the $add_data with the comment tags so we can properly save in the cfg_save directory
		# add begin tag to the data
		$each_data= fill_in_args($rule_ref->{comment_pattern}, [$COMMENT_BEGIN_TAG]);
		$add_data= $each_data."\n".$add_data;

		# add the end tag to the data
		$each_data= fill_in_args($rule_ref->{comment_pattern}, [$COMMENT_END_TAG]);
		$add_data= $add_data."\n".$each_data;
	}
	# in this case, the service's state is changed by executing a command
	elsif ($rule_ref->{add_element_cmd})
	{
		# build the argument list 
		$rc= log_cmd($add_data);
	}

	# save the actual data (command or data appended to a config file)
	&save_configured_data($rule_ref, $add_data) if ($rc);

	$rc;
}

# Function takes a hash ref of a @rules entry, and a list ref of parameters 
# Function is not thread-safe! Make sure you've locked the config lockfile before calling it!
# ARGS:
# 	SCALAR: file to modify
#	SCALAR: add_data
#	SCALAR: comment_pattern (optional)
sub add_config_to_file
{
	# shift the args
	my $filename= shift;
	my $add_data= shift;
	my $comment_pattern= shift;

	# local data
	my $original_file= "";
	my $new_data= "";
	my $each_line;

	if (-f $filename)
	{
		# first read in the file
		open(FILE, $filename) or confess "$filename : $!";
		$original_file= join '', <FILE>;
		close FILE;
	}

	# we're just adding a block of text to the end of the file
	# just add the line with the begin/end tag comments so we can
	# easily remove it later
	
	#add begin tag to the data
	$new_data= fill_in_args($comment_pattern, [$COMMENT_BEGIN_TAG]);
	$new_data.= "\n";

	#insert the config data
	$new_data.= $add_data;
	$new_data.= "\n";

	#add the end tag to the data
	$new_data.= fill_in_args($comment_pattern, [$COMMENT_END_TAG]);
	$new_data.= "\n";

	$original_file.= $new_data;

	# now $original_file contains the new data to copy into the real file
	# we can clobber the config file because we saved a copy above in save_undo_data
	open(FILE, "> $filename") or confess "$filename : $!\n";
	print FILE $original_file;
	close FILE;
}


# This function will save the current state of a daemon if its state is about to be changed.
# If it's a config file that's being modified, save the initial version
# If a command is being run, construct the counterpart remove command and save that
# 
# Parameters:
#	SCALAR: Reference to the rule entry for the daemon
#	SCALAR: Reference to a list of add data (optional- only needed if using an add_element_cmd)
sub save_undo_data
{
	# shift args
	my $rule_ref= shift;
        my $data_ref= shift;

	# we will maintain a stack (conceptually) of changes to system services...
	# in reality, there will be a directory holding the following:
	# a directory called cfg_save looking like this:
	#	cfg_save/
	#		cfgfile1
	#		cfgfile2
	#		...
	#	svc_cmd_list
	#
	# basically, for services that are managed by config files, we will save the original form of that config file
	# inside the cfg_save directory as soon as an edit occurs.
	#
	# for services that are modified by commands, we will build an undo command for each change command that is run
	# (we will do this building an rm_element_command for each add_element_cmd with the same arguments)

	#local data
	my $rc= 1;
	my $copy_dst;
	my $rm_element_cmd;
	my @undo_cmd_stack= ();
	my %undo_tmp_hash= ();
	
	# if we're modifying a config file
	if ($rule_ref->{add_element_pattern})
	{
		$copy_dst= "$CONFIG_FILES_DIR/".$rule_ref->{name};
		
		# first, make sure we don't already have a saved config file
		return 1 unless (-f $copy_dst);
	
		# not much we can do if you want to edit a non-existant config file
		unless (-f $rule_ref->{config_file})
		{
			carp __("The file %s does not exist.\n", $rule_ref->{config_file});
			$rc= 0;
			return $rc;
		}

		# copy the original copy of the file to the save directory here
		$rc= copy($rule_ref->{config_file}, $copy_dst);
	}
	elsif ($rule_ref->{add_element_cmd})
	{
		# add this command's rm_element counterpart to the unconfig command stack
		$rm_element_cmd= $rule_ref->{rm_element_cmd};
		return 0 unless ($rm_element_cmd); # not much we can do if there's no way to undo the service change
		$rm_element_cmd = fill_in_args($rm_element_cmd, $data_ref);
		$rm_element_cmd.= "\n" unless ($rm_element_cmd =~ /.*\n$/s);


		# read in the existing commands on the undo stack
		if (open(STACK, "$UNDO_CMD_STACK"))
		{
			@undo_cmd_stack= <STACK>;
			close STACK;
		}
		else
		{
			confess "$UNDO_CMD_STACK : $!\n";
		}

		# we will not push duplicate commands onto the stack
		foreach (@undo_cmd_stack) 
		{
			$undo_tmp_hash{$_}= 1; 
		}
		return 1 if ($undo_tmp_hash{$rm_element_cmd});

		# push the cmd onto the stack
		unshift(@undo_cmd_stack, $rm_element_cmd);

		# now write the stack back to file
		open(STACK, "> $UNDO_CMD_STACK") or confess "$UNDO_CMD_STACK : $!\n";
		print STACK join '', @undo_cmd_stack;
		close STACK;
	}
	else
	{
		carp __("Malformed rule supplied.\n");
		$rc= 0;
	}

	$rc;
}

# This function will save a string of configuration data being used to modify the state of a daemon  
#
# Parameters:
#	SCALAR: Reference to the rule entry for the daemon
#	SCALAR: String of configuration data to save
sub save_configured_data
{
	# shift args
	my $rule_ref= shift;
	my $save_str= shift;

	# local data
	my $rc= 1;
	my $filename= $CONFIG_DATA_DIR.'/'.$rule_ref->{name};
	my @file_contents= ();
	my %file_contents_hash= ();
	my $each_str;

	# eliminate new lines from the save str
	$save_str=~ s/\n/$CONFIG_DELIMITOR/gs; 
	$save_str.= "\n";

	# first, make sure we aren't saving a duplicate configuration
	if (open(CFG_SAVE, "$filename"))
	{
		@file_contents= <CFG_SAVE>;
		foreach $each_str (@file_contents) { $file_contents_hash{$each_str}= 1 }
		close CFG_SAVE;
	}
	
	return 1 if ($file_contents_hash{$save_str});

	# since this one's not a duplicate, save it now
	unshift @file_contents, $save_str;
	open(CFG_SAVE, ">$filename") or confess "$filename : $!\n";
	print CFG_SAVE join '', @file_contents;
	close CFG_SAVE;

	$rc;
}

# This function will locate data for which a service has been configured to react for a given 
# set of input data
#
# Parameters:
#	SCALAR: Reference to the rule entry for the daemon
#	SCALAR: Reference to an ordered list of configuration data
sub get_configured_data
{
	# shift args
	my $rule_ref= shift;
	my $data_list= shift;

	# local variables
	my $filename= $CONFIG_DATA_DIR.'/'.$rule_ref->{name};
	my @file_contents;
	my %file_contents_hash;
	my $each_str;
	my $query_str;
	my $template= $rule_ref->{add_element_pattern} ? $rule_ref->{add_element_pattern} : $rule_ref->{add_element_cmd}; 
	
	# obtain the saved service deltas
	open(CFG_SAVE, "$filename") or return 0;
	@file_contents= <CFG_SAVE>;
	foreach $each_str (@file_contents) { $file_contents_hash{$each_str}= 1 }
	close CFG_SAVE;

	# build the service configuration delta we are looking for
	$query_str= fill_in_args($template, $data_list);

	$file_contents_hash{$query_str};
}

# This function will reverse a change previously made to a service through add_config
#
# Parameters:
#	SCALAR: Reference to the rule entry for the daemon
#	SCALAR: Reference to an ordered list of configuration data
sub rm_config
{
	# shift the args
	my $rule_ref= shift;
	my $rm_args= shift; #must already be in order

	# local data
	my $rc= 1;
	my $target_str;
	my $file_contents= "";
	my $config_save_file= $CONFIG_DATA_DIR.'/'.$rule_ref->{name};
	my $config_save_data= "";

	# IF there's an add_element_pattern, then we have to remove the string we added 
	# to the config
	if ($rule_ref->{add_element_pattern})
	{
		# construct the target string (what we will remove from the config file)
		$target_str= fill_in_args($rule_ref->{comment_pattern}, [$COMMENT_BEGIN_TAG]);
		$target_str.= "\n";
		$target_str.= fill_in_args($rule_ref->{add_element_pattern}, $rm_args);
		$target_str.= "\n";
		$target_str.= fill_in_args($rule_ref->{comment_pattern}, [$COMMENT_END_TAG]);

		# read in the contents of the config file
		open(CONFIG_FILE, $rule_ref->{config_file}) or confess $rule_ref->{config_file}." : $!\n"; 
		while (<CONFIG_FILE>) { $file_contents.= $_ }
		close CONFIG_FILE;

		my $quoted_target_str= quotemeta $target_str;

		# now sub out the string we want to remove from the config file
		if ($file_contents =~ s/$quoted_target_str//gs)
		{
			# write the config file back out
			open(CONFIG_FILE, "> ".$rule_ref->{config_file}) or confess $rule_ref->{config_file}." : $!\n";
			print CONFIG_FILE $file_contents;
			close CONFIG_FILE;
		}
		else
		{
			$rc= 0;
		}

		# we're not gonna bother to check whether we've removed all changes to the config file
		# and see if we should remove the original config file from the undo data... if we shut
		# down all the services and revert back to the original state, it won't hurt to just 
		# copy that file back
	}
	elsif ($rule_ref->{rm_element_cmd})
	{
		# call the rm_config_cmd 
		$target_str= fill_in_args($rule_ref->{rm_element_cmd}, $rm_args);
		$rc= log_cmd($target_str);
		
		# remove the command from the undo stack
		if ($rc)
		{
			open(STACK, "$UNDO_CMD_STACK") or confess "$UNDO_CMD_STACK : $!\n";
			while (<STACK>) { $file_contents.= $_ }
			close STACK;

			if ($file_contents =~ s/$target_str//gs)
			{
				# write the config file back out
				open(STACK, "> $UNDO_CMD_STACK") or confess "$UNDO_CMD_STACK : $!\n";
				print STACK $file_contents;
				close STACK;
			}

			# now make $target_str the add_element_cmd so we can remove it from the config save file
			$target_str= fill_in_args($rule_ref->{add_element_cmd}, $rm_args);
		}
	}

	# remove the configuration delta from the saved configuration data list
	if ($rc)
	{
		# read the contents of the config_save_file
		open(CFG_SAVE, $config_save_file) or confess "$config_save_file : $!\n";
		while (<CFG_SAVE>) { $config_save_data.= $_ }
		close CFG_SAVE;

		$target_str =~ s/\n/$CONFIG_DELIMITOR/gs;
		$target_str= quotemeta $target_str;

		# blow out the config delta
		if ($config_save_data =~ s/$target_str//gs)
		{
			open(CFG_SAVE, ">$config_save_file") or confess "$config_save_file : $!\n";
			print CFG_SAVE $config_save_data;
			close CFG_SAVE;
		}
		else
		{
			$rc= 0;
		}
	}

	$rc;
}

1;
__END__

