#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2003,2008 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
# @(#)30   1.82.1.158   src/csm/core/pm/ArchiveUtils.pm.perl, setup, csm_rgar2h, rgar2hs001a 1/21/08 04:49:05
#####################################################################

package ArchiveUtils;

use strict;

#            locale tells perl to honor locale for sorting, dates, etc.
#            More info about locale.pm and perl handling of locales can be found in
#            /usr/lib/perl5/5.6.0/locale.pm and the man page for perllocale.
use locale;
use Socket;
use File::Glob ':glob';
use File::Basename;
use File::Path;    # Provides mkpath() and rmtree()

my $msgs;
my $distro;
my $useTranslatedMsg;
my %catHashes;
my $csmroot;
my ($MSGCAT, $MSGMAPPATH, $MSGSET);
my $NO_NODERANGES;
my $NODEGROUPEXPMEM_WARNING = 1;

# $NodeUtils::NO_MESSAGES;    # Set this to 1 if you do not want NodeUtils to
# print any error msgs

# $NodeUtils::errno;          # Will be set if an error occurs.  You must zero
# this out before calling a NodeUtils function,
# if you want to check it afterwards.

BEGIN
{

	#    This enables us to redirect where it looks for other CSM files during development
	$csmroot = $ENV{'CSM_ROOT'} ? $ENV{'CSM_ROOT'} : '/opt/csm';

	$MSGCAT = 'nodecmds.cat';
	if (defined $::MSGMAPPATH)
	{
		$MSGMAPPATH = $ENV{'CSM_ROOT'} ? "$csmroot/msgmaps" : $::MSGMAPPATH;
	}
	else
	{
		$MSGMAPPATH = "$csmroot/msgmaps";
	}
	$MSGSET = 'NodeUtils';
}

umask(0022);   #  This sets umask for all CSM files so that group and world only
               #  have read permissions.
               #  To change it, simply use the umask call in your script, after
               #  the "use ArchiveUtils;" line

#--------------------------------------------------------------------------------

=head1    ArchiveUtils

=head2    Package Description

This program module file, supports the CSM/install archiving and open source
dependencies.  

If adding to this file, please take a moment to ensure that:

    1.  Your contrib has a readable pod header describing the purpose and use of
         your contrib.

    2. Your contrib is under the correct heading and is in alphabetical order
    under that heading.

    3. You test your contribution by running from the command line:  

       pod2html  --verbose --title=ArchiveUtils ArchiveUtils.pm.perl --outfile=ArchiveUtils.html
       
       and examining the ./ArchiveUtils.html file in a browser.


=cut

#--------------------------------------------------------------------------------

=head2    Package Dependancies

    use strict;
    use Fcntl qw(:flock);
    use File::Basename;
    use File::Find;
    use File::Path;    # Provides mkpath()

=cut

#-------------------------------------------------------------------------------

=head1    Subroutines by Functional Group

=head2    License Support

=cut

#-------------------------------------------------------------------------------

=head3 check_for_display_opensrcLicense

	Check if the open source license is required to be displayed.
	This routine runs on the management server and calls the
	check_gpl_packages command remotely (using dsh) on each install
	server to see if any of the GPL packages will need to be installed.
	Then display a single message and prompt if any of the install 
	servers will need to install any of the GPL packages.  Exit if 
	the user answers no, allow the install to continue if the user 
	answers yes.

        Arguments:
                @install_servers
        Returns:
                0 if the license was not displayed
                1 if the license was displayed
        Globals:
		$::CHECK_GPL_PACKAGES - command name to run on install servers
        Error:
                none
        Example:
                ArchiveUtils->check_for_display_opensrcLicense(\@install_servers);
        Comments:
                none

=cut

#-------------------------------------------------------------------------------

sub check_for_display_opensrcLicense
{
	my $routine = "check_for_display_opensrcLicense";
	print "ENTERING: $routine\n" if $::DEBUG;

	my ($class, $ref_install_servers) = @_;
	if(-f "/etc/HmcBase")
	{
		return;
	}
	my @gpl_packages_to_display;
	if($ref_install_servers)
	{
		my @install_servers = @$ref_install_servers;
	
		my $cmd = "$::CHECK_GPL_PACKAGES";
	
		my %options_api = ();
		$options_api{'command'} = $cmd;
                $options_api{'nodes'} = join(',', @install_servers);
		my @dshresult = NodeUtils->runDsh(\%options_api, -2);    # Exit on errors
		my $rc;
		$rc        = $::RUNCMD_RC;
	
		foreach my $line (@dshresult)
		{
	
			# The format of the @dshresult is:
			#   hostname: rpm
			# Where hostname is the csm node on which the dsh was run.
			# Where rpm is the name of any GPL packages that need to be
			# be installed (one per line).  The rpm is empty if there is
			# are GPL packages to install.
	
			chomp $line;
			my ($csmnode, $rpm) = split(" ", $line);
			$csmnode =~ s/://g;
			$rpm     =~ s/://g;
			($::DEBUG) && print "GPL package to install on $csmnode:  $rpm\n";
			if (defined($rpm))
			{
				if (!grep(/^$rpm$/, @gpl_packages_to_display))
				{
					push(@gpl_packages_to_display, $rpm);
				}
			}
		}
	}
	else
	{

		my $cmd = "$::CHECK_GPL_PACKAGES";
	
		#put all dest nodes in tmp file for dsh to use
		my @result = NodeUtils->runcmd("$cmd", -2);    # Exit on errors
		my $rc        = $? >> 8;
		$rc        = $::RUNCMD_RC;
	
		foreach my $line (@result)
		{
	
			# The format of the @dshresult is:
			#   hostname: rpm
			# Where hostname is the csm node on which the dsh was run.
			# Where rpm is the name of any GPL packages that need to be
			# be installed (one per line).  The rpm is empty if there is
			# are GPL packages to install.
	
			chomp $line;
			my $rpm=$line;
			($::DEBUG) && print "GPL package to install on ms:  $rpm\n";
			if (defined($rpm))
			{
				if (!grep(/^$rpm$/, @gpl_packages_to_display))
				{
					push(@gpl_packages_to_display, $rpm);
				}
			}
		}
	}
	if (@gpl_packages_to_display)
	{
		($::DEBUG)
		  && print "GPL packages to install:  @gpl_packages_to_display\n";
		ArchiveUtils->display_opensrcLicense(@gpl_packages_to_display);
	}
	print "LEAVING: $routine\n" if $::DEBUG;
	return (0);
}

#--------------------------------------------------------------------------------

=head3 display_opensrcLicense

        Display the open source product licence

        Arguments:
                rpm list
        Returns:
                none
        Globals:
                $::PREREQS_ATTR
        Error:
                none
        Example:
                ArchiveUtils->display_opensrcLicense( \@rpm_list );
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub display_opensrcLicense
{
	my ($class, $ref_rpmlist) = @_;
	my @rpmlist = $ref_rpmlist;

	# If this is one of the GPL open source packages, display a license
	# agreement.
	if ($::PREREQS_ATTR{'PkgArchitecture'} =~ /ppc64/)
	{

		# There are no GPL packages for PPC64, so nothing to do.
		# If we ever have to start installing GPL packages for PPC64,
		# this will have to be reinstated.
		# MessageUtils->message('I', 'IMsgGPL_LICENSE_PPC64');
		# MessageUtils->message('P', 'IMsgPRESS_ENTER_TO_INSTALL_PPC64');
	}
	else
	{
		MessageUtils->message('I', 'IMsgGPL_LICENSE');
		MessageUtils->message('P', 'IMsgPRESS_ENTER_TO_INSTALL');
		my $t = <STDIN>;
		chop($t);
		if (length($t) > 0)
		{
			MessageUtils->message('I', 'IMsgCSM_NOT_INSTALLING');
			exit 1;
		}
	}
}

#--------------------------------------------------------------------------------

=head3    invokeLAP

        Invoke the LAP tool and display the CSM license agreements.
        The LAP tool will be invoked and copy the LA and LI files to

            /var/opt/csm/<VRnumber>/<type>/license/<lang>.txt
        Where:
                <VRnumber> = CSM Version (1.3, etc)
                <type>     = tryandbuy or full, depending upon the license.
                <lang>     = one of the 17 supported LAP languages.

        Arguments:
                A string - either: "tab" or "full"

        Returns: one of [ 3 | 9 | 127 ]
                If everything ran successful, then LAP will return 9,
        otherwise, if the user returns 3, the license was not
        accepted.  If something went wrong with the LAP tool,
        then an error of 127 will be returned.
        Error:
                127

        Example: Doesn't appear to be used in csm/install

        Comments: none

=cut

#--------------------------------------------------------------------------------

sub invokeLAP
{
	my ($class, $licType) = @_;
	my ($csmVersion, $laDir, $strCmd, $rc);
	my $laptool = "/opt/csm/install/lapapp.jar";

	# make sure that $licType is either tab or full
	if ($licType eq "tab")
	{
		$licType = "tryandbuy";
	}
	elsif ($licType ne "full")
	{

		#print "don't know..\n";
		return 127;
	}

	# get the CSM version:
	$csmVersion = NodeUtils->get_CSMVersion("csm.server");
	$csmVersion = sprintf "%.1f", $csmVersion;               # e.g. "1.3"

	# check that the LA and LI files are in place:
	if (NodeUtils->isLinux())
	{
		$laDir = "/opt/csm/install/$csmVersion/Linux/license/$licType";
	}
	else                                                     # AIX
	{
		$laDir = "/opt/csm/install/$csmVersion/AIX/license/$licType";
	}

	unless (-f "$laDir/LA_en")
	{
		MessageUtils->messageFromCat(
									 'csmInstall.cat',     $::MSGMAPPATH,
									 'csminstall',         'E127',
									 'EMsgCANT_READ_FILE', "$laDir/LA_en"
									);
	}

	# check that the LAP tool is in place:
	unless (-f "$laptool")
	{
		MessageUtils->messageFromCat(
									 'csmInstall.cat',     $::MSGMAPPATH,
									 'csminstall',         'E127',
									 'EMsgCANT_READ_FILE', $laptool
									);
	}

	# Everything should be in place now, so let's call LAP and  construct the command:

    my $found_java = 0;
    my $java;
    my @output = NodeUtils->runcmd("$::LS -1d $::JAVA_LOCATION");

    foreach my $dir (@output) {
        $dir =~ s/^\s*//;    # Remove any leading whitespace
        $dir =~ s/\s*$//;    # Remove any trailing whitespace
        if (-e "$dir/jre/bin/java") {
            my @version = NodeUtils->runcmd("$dir/jre/bin/java -version");
            $version[0] =~ /(\d+\.\d+\.\d+)/;
            $found_java = grep(/IBM/,@version) if ($1 ge $::JAVA_VERSION);
            if ($found_java) {
                $java = "$dir/jre/bin/java";
                last;
            }
        }
    }

    if($found_java){
        $strCmd = "$java -cp /opt/csm/install/lapapp.jar:";
        $strCmd .= " com.ibm.lex.lapapp.LAP ";
        $strCmd .= "-l $laDir ";
        $strCmd .= "-s /var/opt/csm/$csmVersion/$licType ";
        $strCmd .= "-text_only";
        $rc = system("$strCmd");
        $rc = $rc >> 8;
    } else {
        my $string = "$::JAVA_NAME, $::JAVA_VERSION";
        MessageUtils->messageFromCat(
            'csmInstall.cat',     $::MSGMAPPATH,
            'csminstall',         'E127',
            'EMsgMissingPackage', $string);
    }

	# kill processes if they exist:
	if (NodeUtils->isLinux)
	{
		foreach (`ps -efl --columns 300 | grep lapapp | awk '{print \$4}'`)
		{
			chomp $_;
			`kill -9 $_ 2>/dev/null`;
		}
	}

	return $rc;
}

#--------------------------------------------------------------------------------

=head2    RPM Support

=cut

#--------------------------------------------------------------------------------

=head3    checkDistroCopied

        Determine if there are more than 100 files in /csminstall/RPMS
    directory

        Arguments:
                $rpmDirectory

        Returns:
                1 - more than 100 files in /csminstall/RPMS
                0 - eveything okay
        Globals:
                none
        Error:
                1
        Example:
                never used
        Comments:
                never used???

=cut

#--------------------------------------------------------------------------------

sub checkDistroCopied
{
	my ($class, $rpmdir) = @_;

	if (!-e "$rpmdir")
	{
		return 0;
	}
	my @output = `$::LS -l $rpmdir/ | /usr/bin/wc -l`;
	chomp @output;
	my $lines = $output[0];
	$lines =~ s/\s+//;
	if ($lines > 100)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

#--------------------------------------------------------------------------------

=head3    checkForRpmTool

        Exits E2 if the rpm tool is not installed on the local host.

=cut

#--------------------------------------------------------------------------------

sub checkForRpmTool
{
	my $ret_code;

	$ret_code = NodeUtils->get_CSMVersion("rpm");
	if (length($ret_code) == 0)
	{
		MessageUtils->message('E2', 'EMsgNO_RPM');
	}
}

#--------------------------------------------------------------------------------

=head3    suppressRpmSigWarning

        Suppress the NOKEY signature warning starting from RHEL3 QU3 and SLES9 SP1.

=cut

#--------------------------------------------------------------------------------

sub suppressRpmSigWarning
{
	if ($::VERBOSE)
	{

		# No need to suppress NOKEY warning on verbose mode.
		return;
	}

	my ($name, $ver, $version);
	my $cmd    = "/bin/rpm --version";
	my $output = NodeUtils->runcmd($cmd);
	chomp($output);
	if (grep /RPM version/, $output)
	{
		($name, $ver, $version) = split(/\s/, $output);
	}
	my ($a, $b, $c) = split(/\./, $version);

	my $suppress = 0;
	my $distro   = NodeUtils->get_DistributionName();
    my ($effective_distro_name) = NodeUtils->getEffectiveDistro($distro);
	if ($effective_distro_name =~ /RedHat/)
	{
		if (($a >= 4) && ($b >= 2) && ($c >= 3))
		{
			$suppress = 1;
		}
	}
	elsif ($effective_distro_name =~ /SLES/)
	{
		if (($a >= 4) && ($b >= 1) && ($c >= 1))
		{
			$suppress = 1;
		}
	}
	if ($suppress)
	{
		$::RPMCMD = "/bin/rpm --nosignature";
	}
}

#--------------------------------------------------------------------------------

=head3    checkRPMInstall

        Checks if a given rpm is installed. The full pathname of the
        rpm must be given.


        Arguments:
                $rpmName


        Returns:
                0 - not installed
                1 - installed

        Globals:
                $::PLTFRM
                $::rpmShortnames
                $::rpmBasenames
        Error:
                0

        Example:
                 my $installed =
                        ArchiveUtils->checkRPMInstall($rpmFileName);

        Comments:
                checkRPMInstall must be called before this function is used.

=cut

#--------------------------------------------------------------------------------

sub checkRPMInstall
{
	my ($class, $filename) = @_;    #rpm name
	my $file;
	if ($::PLTFRM eq "AIX")
	{
		my @output =
		  `$::RPMCMD -q --qf %{NAME}-%{VERSION}-%{RELEASE} -p $filename`;
		chomp @output;
		$file = $output[0];
	}
	else
	{
		my @path = split '/', $filename;
		$file = pop @path;
	}

	#determine if the given rpm is installed on the system
	#we need to check to see if this rpm is already installed
	my ( $shortname, $basename) = ArchiveUtils->getShortName( $file);
	if (!exists $::rpmShortnames{$shortname})
	{

		#so check the basename
		if (!exists $::rpmBasenames{$basename})
		{

			#no version of this rpm is installed
			#push @rpms, $wholename;
			return 0;
		}
		else
		{

			# If the required os package is installed,  CSM need not install them
			# This is because the cooperation between different  the Service Level
			foreach my $rpm (@::linux_os_prereqs)
			{
				if ($rpm =~ /^(.*)\*$/)
				{
					$rpm = $1;
				}
				my $rpmname = substr($shortname, 0, length $rpm);
				if ($rpmname eq $rpm)
				{

					MessageUtils->messageFromCat('csmInstall.cat',
											$::MSGMAPPATH, 'csminstall', 'V',
											'IMsgRPM_ALREADY_INSTALLED', $file);
					return 1;
				}
			}

			# a different version of this rpm is installed
			#test to see if this rpm  is installed
			# Check if the RPM is already installed at the correct version
			my $cmd = "$::RPMCMD -U --test $filename";
			MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
										 'NodeUtils', 'V', 'IMsgCMD', $cmd);
			my @cmdout = `LANG=C $cmd 2>&1`;
			my $rc     = $? >> 8;
			my $cmdout = join('', @cmdout);    # Put output into a single string
			if ($cmdout !~ /package .* is already installed/)
			{

				#not installed
				return 0;
			}
		}
	}
	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
								 'V', 'IMsgRPM_ALREADY_INSTALLED', $file);
	return 1;                                  #it is installed
}

#--------------------------------------------------------------------------------

=head3	getServiceLevel

        Notes: This routine just a temperary solution for RedHatEL-AS3

=cut

#--------------------------------------------------------------------------------
sub getServiceLevel
{
	my $readme =
	  "/csminstall/Linux/$::DISTRO_NAME/$::DISTRO_VERSION/$::ARCH/README";
	my $sp = undef;

	if (-l "$readme")
	{
		my $target = readlink "$readme";
		if ($target =~ /RedHatEL\-AS3\-ppc64/)
		{
			$sp = "QU1";
		}
		elsif ($target =~ /RedHatEL\-AS3\-QU2/)
		{
			$sp = "QU2";
		}
	}

	return $sp;
}

#
# Compatible for original routine
#
sub copyDistroCD
{
	my ($class, $platform, $arch, $distro_top, $distro_name, $distro_version, $nomountcd,
		$sl, $extracds)
	  = @_;
	my $orig_dir;
	my @service_level = @$sl;

	if (scalar @service_level == 0)
	{
		print "InstallServiceLevel is not set for node. Exiting.\n";
		return 1;
	}
	else
	{
		foreach my $svl (@service_level)
		{
			if (
				!ArchiveUtils->copyDistroPkgs(   $platform,    $arch,           $distro_top,
									 $distro_name, $distro_version, $svl, $extracds, $nomountcd)
			   )
			{

				# Copy packages error.
				exit 1;
			}
			ArchiveUtils->setupBootOrder($arch, $distro_top, $distro_name,
										 $distro_version, $svl, $::MEDIATYPE);
		}
	}

	return 0;
}

sub copyDistroPKGPATH
{
	my ($class, $platform, $arch, $distro_top, $distro_name, $distro_version, $nomountcd,
		$sl, $extracds)
	  = @_;
	my $orig_dir;
	my @service_level = @$sl;

	if (scalar @service_level == 0)
	{
		print "InstallServiceLevel is not set for node. Exiting.\n";
		return 1;
	}
	else
	{
		foreach my $svl (@service_level)
		{
			if (
				!ArchiveUtils->copyDistroPkgs(   $platform,    $arch,           $distro_top,
									 $distro_name, $distro_version, $svl, $extracds, $nomountcd)
			   )
			{

				# Copy packages error.
				exit 1;
			}
			ArchiveUtils->setupBootOrder($arch, $distro_top, $distro_name,
										 $distro_version, $svl);
		}
	}

	return 0;
}

sub foundPkgPath
{
	my ($class, $suffix) = @_;
	my @files = ();

	use DirHandle;

	$suffix = lc($suffix);
	my $err_dir;

	foreach my $dir (@::PKGPATH)
	{
		if (!-d $dir)
		{
			$err_dir .= "\'$dir\' ";
			next;
		}

		push @files, $dir;

		my $d = new DirHandle "$dir";
		if (defined $d)
		{
			while (defined($_ = $d->read))
			{
				next if ($_ eq "." || $_ eq "..");
				next if ($_ !~ /$suffix$/);
				my $tmp_filenam = "$dir" . "/" . "$_";
				next if (!-f $tmp_filenam);
				push @files, $tmp_filenam;
			}
		}
		undef $d;
	}
	if ($err_dir)
	{
		MessageUtils->messageFromCat(
									 'csmInstall.cat',          $::MSGMAPPATH,
									 'csminstall',              'I',
									 'IMsgDIRECTORY_NOT_FOUND', $err_dir
									);
	}

	return @files;
}

sub probeImages
{
	my ($class, $r_pkgpath, $r_disks) = @_;
	my @pkgpath = @$r_pkgpath;
	my @disks   = @$r_disks;
	my $images  = {};
	my ($mntpoint, $ismounted);

	foreach my $diskref (@disks)
	{
		my $diskname = $diskref->{name};
		my $disktype = $diskref->{type} || "CD";
		my $disknum  = $diskref->{num};

		foreach my $path (@pkgpath)
		{
			$ismounted = 0;

			if (-f $path && $path =~ /iso$/)
			{

				# AIX does not support iso image
				next if ($::PLTFRM eq "AIX");

				$mntpoint = ArchiveUtils->mount_iso_image($path);
				if ($mntpoint)
				{
					$ismounted = 1;

					#$path = $mntpoint;
				}
				else
				{
					MessageUtils->messageFromCat('csmInstall.cat',
							 $::MSGMAPPATH, 'csminstall', 'I', 'IMsgCANT_MOUNT',
							 $path);
					next;
				}
			}
			elsif (-d $path)
			{
				$mntpoint = $path;
			}

			# valid the disk
			my $progref = $diskref->{diskid_searchprog};
			my $progarg = $diskref->{diskid_searchargs};
			if (&$progref($mntpoint, $progarg))
			{

				# yes , this is the required disk, save it
				$images->{$diskname}->{$disktype}->{$disknum} = $path;
			}

			if ($ismounted)
			{
				ArchiveUtils->unmount_iso_image($mntpoint);
				$ismounted = 0;
			}
		}
	}

	return $images;
}

sub is_req_disks_copied
{
	my ($class, $distro_top, $ref_disks) = @_;
	my @disks = @$ref_disks;

	if (!-e $distro_top) { return 0; }

	foreach my $diskref (@disks)
	{
		next
		  if (defined($diskref->{not_for_full_install})
			  && $diskref->{not_for_full_install} eq 'TRUE');

		if (defined $diskref->{diskid_target})
		{
			my $target = $diskref->{diskid_target};
			if ($target ne "")
			{
				if (!-e "$distro_top/$target") { return 0; }
			}
		}
	}

	return 1;
}

#--------------------------------------------------------------------------------

=head3   mkLinks 

        Arguments:
		$destdir
		$strFiles
        Returns:
		none
        Globals:
                none
        Error:
		none
        Example:
		none
        Comments:
              mkLinks is used to make links in for a boudle of files  a dest dir 
              if the link/file already exists, it will be removed. 

=cut

#--------------------------------------------------------------------------------

sub mkLinks
{
	my ($class, $destdir, $strFiles) = @_;
    use Cwd;
    my $oldDir = getcwd;
	chdir $destdir;
	my @filelist = bsd_glob($strFiles);
	my ($filename, $file);
	foreach $file (@filelist)
	{
		$filename = basename $file;

		# overwrite for a link file, but ignore for a normal file.
		# The reason is because it is not very good to overwrite the RPMs copied by installms
		# need add a special parameter to control it?
		if (-l $filename)
		{
			unlink $filename;
		}
		symlink $file, $filename;
	}
    chdir $oldDir;
}

#-------------------------------------------------------------------------------

=head3   remove_all_links

         Arguments:
         $basedir
         Returns:
           none
         Globals:
            none
         Error:
            none
	     Example:
		    none
       	 Comments:
		    none

=cut

#-------------------------------------------------------------------------------
sub remove_all_links
{
	my ($class, $basedir) = @_;

	my $d = new DirHandle "$basedir";
	if (defined $d)
	{
		while (defined($_ = $d->read))
		{
			next if ($_ eq "." || $_ eq "..");
			my $tmp_filenam = "$basedir" . "/" . "$_";
			if (-l "$tmp_filenam")
			{
				unlink($tmp_filenam);
			}
		}
	}
	undef $d;
}

# only to be used internally in ArchiveUtils during CD copying.
# run the copy command and check the dir space.
sub run_cmd_and_check_space
{
    my ($class, $cmd) = @_;
    my @output = NodeUtils->runcmd($cmd, -1);
    if($::PLTFRM eq "Linux")
    {
        if (grep /No space left on device/, @output)
        {
            return 0;  # error!
        }
    }
    else
    {
        if (grep /not enough space/, @output)
        {
            return 0; # error!
        }
    }
    return 1;
}

sub setupBootOrder
{
	my ($class, $arch, $distro_top, $distro_name, $distro_version, $sp) = @_;
	my $media_type = $::MEDIATYPE;
	undef $::MEDIATYPE;

	my ($cmd,   $rc,    $output);
	my (@origdirs,      @spdirs);
	my ($first_origdir, $first_spdir);

	if ($arch =~ /i.86/) { $arch = "i386"; }

	my @orig_disks = @{$::pkgdefs{'distro_disks'}};
	if ($media_type eq "DVD")
	{
		@orig_disks = @{$::pkgdefs{'distro_dvd_disks'}};
	}
    my ($effective_distro_name, $effective_distro_ver, $effective_svl_level) =
        NodeUtils->getEffectiveDistro($distro_name, $distro_version, $sp);

	# On SLES, no matter this is GA or any other service level, firstly make sure
	# the GA disks were copied.
	if ($effective_distro_name =~ /SLES/ && $effective_distro_ver <= 9
		&& !ArchiveUtils->is_req_disks_copied($distro_top . "/GA/",
											  \@orig_disks))
	{
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									 'csminstall', 'E1', 'EMsgDisksMissing');
	}

	if (($effective_distro_name =~ /SLES/) && ($effective_distro_ver <= 9)
		&&($effective_svl_level ne "GA"))
	{

		# For service levels other than GA, create the links into the GA dir.
		foreach my $diskref (@orig_disks)
		{
			my $filename = $diskref->{diskid_target};
			if ($filename ne "")
			{
				my $linkfile = $distro_top . "/" . $sp . "/" . $filename;
				if (-l $linkfile)
				{
					unlink $linkfile;
				}
				symlink "../GA/" . $filename, $linkfile;
			}
		}

	}

	foreach my $diskref (@orig_disks)
	{
		if (!grep(/$diskref->{diskid_target}/, @origdirs))
		{
			push(@origdirs, $diskref->{diskid_target});
		}
	}

	$first_origdir = $orig_disks[0]->{diskid_target};

	if ($effective_svl_level && ($effective_distro_ver <= 9)
		&& ($effective_svl_level ne "GA"))
	{
		if (defined $::pkgdefs{'sp_disks'}{$sp})
		{
			my @sp_disks = @{$::pkgdefs{'sp_disks'}{$sp}};
			if ($media_type eq "DVD")
			{
				@sp_disks = @{$::pkgdefs{'sp_dvd_disks'}{$sp}};
			}
			foreach my $diskref (@sp_disks)
			{
				if (!grep(/$diskref->{diskid_target}/, @spdirs))
				{
					push(@spdirs, $diskref->{diskid_target});
				}
			}

			$first_spdir = $sp_disks[0]->{diskid_target};

			if ($effective_distro_name =~ /SLES/)
			{
				if (
					!ArchiveUtils->is_req_disks_copied(     $distro_top . "/" . $sp, \@sp_disks
					)
				   )
				{
					MessageUtils->messageFromCat('csmInstall.cat',
						 $::MSGMAPPATH, 'csminstall', 'E1', 'EMsgDisksMissing');
				}
			}
			else
			{

				# For RHEL, check the copy of CDs in a different way.
				my $my_dirname;
				if ($sp eq "GA")
				{
					$my_dirname = $distro_name . $distro_version . "-" . $arch;
				}
				else
				{
					$my_dirname = $distro_name . $distro_version . "-" . $sp;
				}

				if (!-e "$distro_top/$my_dirname")
				{
					MessageUtils->messageFromCat('csmInstall.cat',
						 $::MSGMAPPATH, 'csminstall', 'E1', 'EMsgDisksMissing');
				}
			}

		}
		else
		{

			# whl InstallServiceLevel is not found in the pkgdefs file, user-defined service level.
			if (   (($distro_name =~ /SLES/) && ($distro_version eq 8.1))
				|| (($distro_name =~ /SLES/) && ($distro_version eq 9))
				|| (($distro_name =~ /SLES/) && ($distro_version eq '10')))
			{
				my $directory_name;

				# Use different directory name for SLES8.1 and SLES9
				if ($distro_version eq 8.1)
				{
					$directory_name = "sles8";
				}
				elsif ($directory_name eq "9")
				{
					$directory_name = "sles9";
				}
				elsif ($directory_name eq "10")
				{
				    $directory_name = "sles10";
				}
				$first_spdir = $directory_name . "-" . $sp;
			}
			else
			{
				$first_spdir = $sp;
			}
		}

	}

	if ($effective_distro_name =~ /SLES/)
	{

		my $realRPMS = "$distro_top/$distro_name/RPMS";
		my $spRPMS   = "$distro_top/$sp/RPMS";

		if (   (($distro_name =~ /SLES/) && ($distro_version eq 8.1))
			|| (($distro_name =~ /SLES/) && ($distro_version eq 9))
			|| (($distro_name =~ /SLES/) && ($distro_version eq "10")))
		{
			my $directory_name;
			my $disk_name;

			# Use different directory name for SLES8.1 and SLES9
			if ($distro_version eq 8.1)
			{
				$directory_name = "sles8";
				$disk_name      = "ul";
			}
			elsif ($distro_version eq '9')
			{
				$directory_name = "sles9";
				$disk_name      = "core9";
			}
			elsif ($distro_version eq '10')
			{
			    $directory_name = "sles10";
			    $disk_name	    = "core10";
			}

			foreach my $subpath (@origdirs)
			{
				ArchiveUtils->mkLinks($realRPMS,
									  "../../$sp/$subpath/*/*/*.rpm");
				ArchiveUtils->mkLinks($spRPMS, "../$subpath/*/*/*.rpm");
			}

			if (($effective_svl_level) && ($effective_svl_level ne "GA"))
			{
				if (!defined $::pkgdefs{'sp_disks'}{$sp})
				{
					ArchiveUtils->mkLinks($realRPMS,
									 "../../$sp/$directory_name-$sp/*/*/*.rpm");
					ArchiveUtils->mkLinks($spRPMS,
										  "../$directory_name-$sp/*/*/*.rpm");
				}
				else
				{
					foreach my $subpath (@spdirs)
					{
						ArchiveUtils->mkLinks($realRPMS,
											  "../../$sp/$subpath/*/*/*.rpm");
						ArchiveUtils->mkLinks($spRPMS, "../$subpath/*/*/*.rpm");
					}
				}
			}

			# all "cp -a" has been replaced with "cp -r" due to "-a" is not supported by AIX.
			# but don't know whether it works fine for all linux distribution.
			$cmd =
			  "$::COPY -r $distro_top/$sp/$first_origdir/yast $distro_top/$sp/$first_origdir/boot $distro_top/$sp/$first_origdir/content $distro_top/$sp/$first_origdir/media.1 $distro_top/$sp/";
			$rc = ArchiveUtils->run_cmd_and_check_space($cmd);
			return $rc if(!$rc);

			if ($arch =~ /ppc64/)
			{

				$cmd = "$::MKDIR $distro_top/$sp/yast";
				$rc = ArchiveUtils->run_cmd_and_check_space($cmd);
				return $rc if(!$rc);

				if (($sp) && ($sp ne "GA") && ($effective_distro_ver <= 9))
				{
					$cmd = 
					  "$::MKDIR -p $distro_top/$sp/yast/$directory_name/suse/setup/descr && $::COPY -r $distro_top/$sp/$first_origdir/suse/setup/descr $distro_top/$sp/yast/$directory_name/suse/setup && $::COPY -r $distro_top/$sp/$first_origdir/content $distro_top/$sp/$first_origdir/media.1 $distro_top/$sp/yast/$directory_name && $::COPY -r $distro_top/$sp/$first_spdir/media.1 $distro_top/$sp/$first_spdir/driverupdate $distro_top/$sp/$first_spdir/boot $distro_top/$sp/ && $::MKDIR -p $distro_top/$sp/updates/kernel ; $::COPY $distro_top/$sp/$first_spdir/suse/ppc/kernel-ppc64*.ppc.rpm $distro_top/$sp/updates/kernel";
					$rc = ArchiveUtils->run_cmd_and_check_space($cmd);
					return $rc if (!$rc);

					# Link all the service pack rpm directories to the "update"
					# directories so that "you" OS Upgrade can find all the RPMs
					# it needs.
					my $sp_lower = lc($sp);
					my $_topdir = "$distro_top/$sp/unitedlinux-$sp_lower/ppc/update/SUSE-CORE/$distro_version/rpm";
					if (-d "$_topdir/ppc")
					{
						if (! -d "$_topdir/ppc.orig")
						{
							# Save off the original directory, only the 1st time
							rename ("$_topdir/ppc", "$_topdir/ppc.orig")
						}
						else
						{
							# Clean up before linking if there's already a .orig dir
							rmtree("$_topdir/ppc", $::VERBOSE, 1);
						}
						symlink "../../../../../suse/ppc", "$_topdir/ppc";
					}
					if (-d "$_topdir/ppc64")
					{
						if (! -d "$_topdir/ppc64.orig")
						{
							# Save off the original directory, only the 1st time
							rename ("$_topdir/ppc64", "$_topdir/ppc64.orig")
						}
						else
						{
							# Clean up before linking if there's already a .orig dir
							rmtree("$_topdir/ppc64", $::VERBOSE, 1);
						}
						symlink "../../../../../suse/ppc64", "$_topdir/ppc64";
					}
					if (-d "$_topdir/noarch") 
					{
						if (! -d "$_topdir/noarch.orig")
						{
							# Save off the original directory, only the 1st time
							rename ("$_topdir/noarch", "$_topdir/noarch.orig")
						}
						else
						{
							# Clean up before linking if there's already a .orig dir
							rmtree("$_topdir/noarch", $::VERBOSE, 1);
						}
						symlink "../../../../../suse/noarch", "$_topdir/noarch";
					}
				}

			}
			else
			{
				$cmd =
				  "$::MKDIR -p $distro_top/$sp/yast/$directory_name/suse/setup/descr && $::COPY -r $distro_top/$sp/$first_origdir/suse/setup/descr $distro_top/$sp/yast/$directory_name/suse/setup && $::COPY -r $distro_top/$sp/$first_origdir/content $distro_top/$sp/$first_origdir/media.1 $distro_top/$sp/yast/$directory_name";
				$rc=ArchiveUtils->run_cmd_and_check_space($cmd);
				return $rc if(!$rc);

				if (($sp) && ($sp ne "GA") && ($effective_distro_ver <= 9))
				{
					$cmd =
					  "$::COPY -r $distro_top/$sp/$first_spdir/linux $distro_top/$sp/$first_spdir/driverupdate $distro_top/$sp/ && $::COPY -r $distro_top/$sp/$first_spdir/boot/loader/linux $distro_top/$sp/boot/loader/ && $::COPY -r $distro_top/$sp/$first_spdir/boot/loader/initrd $distro_top/$sp/boot/loader/";
					$rc=ArchiveUtils->run_cmd_and_check_space($cmd);
					return $rc if(!$rc);
				}
			}

			my $order     = "$distro_top/$sp/yast/order";
			my $instorder = "$distro_top/$sp/yast/instorder";

			if ($arch =~ /ppc64/)
			{
				if (open(FILE, ">$instorder"))
				{
					if ($sp ne "GA")
					{
						print FILE "/yast/$directory_name\n";
						print FILE "/\n";
					}
					else
					{
						print FILE "$first_origdir\t$first_origdir\n";
					}

					# for SLES9, add path for the CORE CDs
					if ($distro_version eq 9)
					{
						my $sec_origdir = $orig_disks[1]->{diskid_target};
						print FILE "$sec_origdir\t$sec_origdir\n";
					}

					close(FILE);
				}
			}

			if (open(FILE, ">$order"))
			{
				if ($arch =~ /ppc64/)
				{
					if ($sp ne "GA")
					{
						print FILE "/$first_spdir\t$first_spdir\n";
						print FILE "/yast/$directory_name\t$first_origdir\n";
					}
					else
					{
						print FILE "$first_origdir\t$first_origdir\n";
					}

					# For SLES9, Add path for CORE CDs
					if ($distro_version eq 9)
					{
						my $sec_origdir = $orig_disks[1]->{diskid_target};
						print FILE "/yast/$disk_name\t$sec_origdir\n";
					}
					# FIXME sles10
				}
				else
				{
					if ($sp ne "GA")
					{
						print FILE "/$first_spdir\t$first_spdir\n";
					}

					my $sec_origdir = $orig_disks[1]->{diskid_target};
					print FILE "/yast/$directory_name\t$first_origdir\n";
					print FILE "/yast/$disk_name\t$sec_origdir\n";
				}
				close(FILE);
			}

			# end of SLES8.1 support
		}
	}    # end of SLES
	elsif ($effective_distro_name =~ /RedHat/)
	{
		my $my_dirname;
		if ($sp eq "GA")
		{
			$my_dirname = $distro_name . $distro_version . "-" . $arch;
		}
		else
		{
			$my_dirname = $distro_name . $distro_version . "-" . $sp;
		}
		my $realRPMS = "$distro_top/$::pkgdefs{DISTRO_RPMDIR}";

        # rhel 5 put rpms in 'Server' or 'Client' or 'Workstation' ...
        if (($effective_distro_ver == 5) || ($effective_distro_ver =~ /5\./)) {

            # make all rpms could be access through 'Client/' for 
            # rhel5 client
            if ($effective_distro_name =~ /Client/) {
                if (-e "$distro_top/$my_dirname/Workstation") {
                    ArchiveUtils->mkLinks( 
                        "$distro_top/$my_dirname/$::pkgdefs{DISTRO_RPMDIR}",
                        "../Workstation/*.rpm"
                        );
                }

                if (-e "$distro_top/$my_dirname/Desktop") {
                    ArchiveUtils->mkLinks( 
                        "$distro_top/$my_dirname/$::pkgdefs{DISTRO_RPMDIR}",
                        "../Desktop/*.rpm"
                    );
                }

                if (-e "$distro_top/$my_dirname/VT") {
                    ArchiveUtils->mkLinks( 
                        "$distro_top/$my_dirname/$::pkgdefs{DISTRO_RPMDIR}",
                        "../VT/*.rpm"
                    );
                }
            }

            if ($effective_distro_name =~ /Server/) {
                if (-e "$distro_top/$my_dirname/Cluster") {
                    ArchiveUtils->mkLinks( 
                        "$distro_top/$my_dirname/$::pkgdefs{DISTRO_RPMDIR}",
                        "../Cluster/*.rpm"
                    );
                }

                if (-e "$distro_top/$my_dirname/ClusterStorage") {
                    ArchiveUtils->mkLinks( 
                        "$distro_top/$my_dirname/$::pkgdefs{DISTRO_RPMDIR}",
                        "../ClusterStorage/*.rpm"
                    );
                }

                if (-e "$distro_top/$my_dirname/VT") {
                    ArchiveUtils->mkLinks( 
                        "$distro_top/$my_dirname/$::pkgdefs{DISTRO_RPMDIR}",
                        "../VT/*.rpm"
                    );
                }
            }
            if (-e "$realRPMS/../$my_dirname/$::pkgdefs{DISTRO_UPDATEDIR}") {
		        ArchiveUtils->mkLinks(
                        $realRPMS,
                        "../$my_dirname/$::pkgdefs{DISTRO_RPMDIR}/*.rpm"
                        );
            }
        }
        else {
            if (-e "$realRPMS/../../$my_dirname/$::pkgdefs{DISTRO_UPDATEDIR}") {
		    	ArchiveUtils->mkLinks(
                        $realRPMS,
                        "../../$my_dirname/$::pkgdefs{DISTRO_UPDATEDIR}/*.rpm"
                        );
		    }
        }
	}

	if ($effective_distro_name =~ /SLES/)
	{
		if (-f "$distro_top/$sp/media.1/info.txt")
		{
			unlink("$distro_top/$sp/media.1/info.txt");
		}
	}

	return 1;
}

# If the "-p dirname" is used to specify the distribution iso content directory, return dirname, otherwise if
# "dirname" is used to specify where distribution iso images are held, return the .iso file directly.
sub resolveUserPkgPath
{
	my ($class, $suffix) = @_;
	my @files = ();

	use DirHandle;

	$suffix = lc($suffix);
	my $err_dir;

	foreach my $dir (@::PKGPATH)
	{
		if (!-d $dir)
		{
			$err_dir .= "\'$dir\' ";
			next;
		}

		my $d = new DirHandle "$dir";
		if (defined $d)
		{
			while (defined($_ = $d->read))
			{
				next if ($_ eq "." || $_ eq "..");
				if ($_ =~ /$suffix$/)
				{

					# We can find .iso in this directory. Add each .iso file under this directory into @files.
					my $tmp_filenam = "$dir" . "/" . "$_";
					next if (!-f $tmp_filenam);
					push @files, $tmp_filenam;
				}
				elsif (($_ =~ "GPL") || ($_ =~ "content"))
				{

					# This is a RHEL or SLES content directory. Just put the whole directory name into @files and stop browsing the directory.
					push @files, $dir;
					last;
				}
				else
				{
					next;
				}
			}
		}
		undef $d;
	}
	if ($err_dir)
	{
		MessageUtils->messageFromCat(
									 'csmInstall.cat',          $::MSGMAPPATH,
									 'csminstall',              'I',
									 'IMsgDIRECTORY_NOT_FOUND', $err_dir
									);
	}

	return @files;
}

#---------------------------------------------------------------------

=head3    copyAndShowProgress

        Copy one dir to another and show the progress

        Arguments:
            $srcDir
            $dstDir

        Returns:
            none

=cut

#---------------------------------------------------------------------

sub copyAndShowProgress
{
	my ($class, $srcDir, $dstDir) = @_;
	my ($oldDir, $nFilesToCopy, $cmd, $nFilesCopied, $percent);
	my $return_value = 1; # success

	use Cwd;
	$oldDir = getcwd;
	chdir($srcDir);

	$cmd = "find . -print | wc -l";
	$nFilesToCopy = NodeUtils->runcmd("$cmd", 1);
	chomp($nFilesToCopy);
	$nFilesToCopy |= 0;
	if ($::PLTFRM eq "Linux")
	{
		if(-f "/etc/HmcBase")
        {    
            $cmd = " cp -av . $dstDir --reply=yes";
	    }
        else
        {
            $cmd = "find . -print | cpio -vdump --quiet $dstDir";
	    }
    }
	else
	{

		# for AIX and other platform (not sure this command can work fine on other platform)
		$cmd = "find . -print | cpio -pdlvu $dstDir 2>/dev/null";

	}
	open(PIPE, "$cmd 2>&1 |")
	  || MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									  'csminstall', 'E',
									  'EMsgERROR_RUNNING_COMMAND', $cmd);
	$nFilesCopied = 0;

	my $save = $|;
	$| = 1;    # turn on autoflush
	if ($::PLTFRM eq "Linux")
	{
		my $lastmsg;
		while (<PIPE>)
		{
			$lastmsg = $_;
			next if /^cpio:/;    # skip some cpio info about hard links
			$percent = $nFilesCopied / $nFilesToCopy;
                        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                                     'csminstall', 'l', 'IMsgPERCENT_COMPLETE',(sprintf "%0.2f%%", $percent *100) );
                        print "\r";
			#printf "%0.2f%% complete \r", $percent * 100;
			++$nFilesCopied;
		}
		$| = $save;
		close(PIPE);
		if (grep /No space left on device/, $lastmsg)
		{
			MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									   'csminstall', 'E', 'EMsgCsminstallFull');
			$return_value=0;   # failure						   
		}
		else
		{
			#print "100% complete        \n";
                        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                                     'csminstall', 'l', 'IMsgPERCENT_COMPLETE','100%' );
                        print "\n";
		}
	}
	else
	{

		# cpio on AIX doesn't abort when the destination directory is full,
		# so have to grep in real-time. This block is seperated from above for
		# the efficency of Linux package copy.
		while (<PIPE>)
		{
			if (grep /not enough space/, $_)
			{
				MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									   'csminstall', 'E', 'EMsgCsminstallFull');
				$return_value=0;   # failure
				last;
			}
			next if /^cpio:/;    # skip some cpio info about hard links
			$percent = $nFilesCopied / $nFilesToCopy;
			#printf "%0.2f%% complete \r", $percent * 100;
                        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                                     'csminstall', 'l', 'IMsgPERCENT_COMPLETE',(sprintf "%0.2f%%", $percent *100) );
                        print "\r";
			++$nFilesCopied;
		}
		$| = $save;
		close(PIPE);
		if($return_value)
		{
			#print "100% complete        \n";
                        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                                     'csminstall', 'l', 'IMsgPERCENT_COMPLETE','100%' );
                        print "\n";
		}	
	}
	chdir $oldDir;
	return $return_value;
}

# Called by copyDistroPkgs to deal with user-defined service level disk copy.
# Return 1 on success, 0 on failure.
sub copyUserDistroPkgs
{
	my ($class, $platform, $arch, $distro_top, $distro_name, $distro_version,
		$sl_top)
	  = @_;
	my ($mntpoint, $ismounted, $target, $pkghome, $cmd, $diskname, $disknum);
	my $rc = 1;

	my @pkgpath = ArchiveUtils->resolveUserPkgPath("iso");
	if (!@pkgpath)
	{

		# print "Must use \"-p\" option when using customized service level.\n";
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									 'csminstall', 'I', 'IMsgOptionP');
		return 0;
	}

	$disknum = "0";
	foreach my $path (@pkgpath)
	{
		$ismounted = 0;

		if (-f $path && $path =~ /iso$/)
		{

			# AIX does not support iso image
			if ($::PLTFRM eq "AIX")
			{

				#print out some message
			}

			$mntpoint = ArchiveUtils->mount_iso_image($path);
			if ($mntpoint)
			{
				$ismounted = 1;
			}
			else
			{
				MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									'csminstall', 'I', 'IMsgCANT_MOUNT', $path);
				next;
			}
		}
		elsif (-d $path)
		{
			$mntpoint = $path;
		}
		else
		{
			return 0;
		}

		# Now begin copy the contents of Distro disk from $mntpoint to /csminstall
		$target = $mntpoint;

		if ($platform eq "Linux")
		{
			if (
				(($distro_name =~ /SLES/) && ($distro_version eq 8.1))
				|| (   ($distro_name =~ /SLES/)
					&& ($distro_version eq 9))
			   )
			{
				my $directory_name;

				# Use different directory name for SLES8.1 and SLES9
				if ($distro_version eq 8.1)
				{
					$directory_name = "sles8";
				}
				else
				{
					$directory_name = "sles9";
				}
				$pkghome =
				    $distro_top . "/" . $sl_top . "/"
				  . $directory_name . "-"
				  . $sl_top;
			}
			else
			{

				# Don't need to create sub-dir further for RHEL or other distributions.
				my $my_dirname;
				if ($sl_top eq "GA")
				{
					$my_dirname = $distro_name . $distro_version . "-" . $arch;
				}
				else
				{
					$my_dirname =
					  $distro_name . $distro_version . "-" . $sl_top;
				}
				$pkghome = $distro_top . "/" . $my_dirname;
			}

			if (!-d $pkghome)
			{
				$cmd = "$::MKDIR -p $pkghome";
				NodeUtils->runcmd($cmd, -1);
			}
		}

		# This message should be changed in future.
		$diskname = $path;
		$disknum++;
		MessageUtils->messageFromCat(
									 'csmInstall.cat',   $::MSGMAPPATH,
									 'csminstall',       'I',
									 'IMsgCOPYING_DISK', $diskname,
									 'CD', $disknum
									);

		$rc = ArchiveUtils->copyAndShowProgress($target, $pkghome);

		if ($ismounted)
		{

			# Umount disk
			ArchiveUtils->unmount_iso_image($mntpoint);
			$ismounted = 0;
		}
		
		if(!$rc)
		{
			last; # error during copy, skip the rest of path.
		}

	}
	return $rc;
}

#---------------------------------------------------------------------

=head3	

	Move directory from $distro_top to service level subdirectory if exists,
	in order to support migration from Amenia.

=cut

#---------------------------------------------------------------------
sub migrateDir
{
	my ($class, $distro_top, $distro_name, $distro_version, $sp) = @_;
	my @disks;
	my $flag      = 0;
	my $migrating = 0;
	my $svl;
    my ($effective_distro_name, $effective_distro_ver, $effective_svl_level) =
        NodeUtils->getEffectiveDistro($distro_name, $distro_version, $sp);

	if ($effective_distro_name !~ /SLES/)
	{
		# No migration needed for RHEL.
		return;
	}

	if (($effective_svl_level ne "GA") && (!defined $::pkgdefs{'sp_disks'}{$sp}))
	{
		# No migration issue for customized service level.
		return;
	}

	$svl   = "GA";
	@disks = @{$::pkgdefs{'distro_disks'}};

  migration_check_again:
	foreach my $diskref (@disks)
	{
		if (defined $diskref->{diskid_target})
		{
			my $target = $diskref->{diskid_target};
			if ($target ne "")
			{
				if (-e "$distro_top/$target")
				{

					# Move directory into the service level sub-directory.
					my $pkghome = $distro_top . "/" . $svl;
					if (!-d $pkghome)
					{
						my $cmd = "$::MKDIR -p $pkghome";
						NodeUtils->runcmd($cmd, -1);
					}

					if (-e "$pkghome/$target") { next; }

					my $mv_cmd = "/bin/mv $distro_top/$target $pkghome/";
					MessageUtils->message('I', "IMsgMigrateDir", $distro_top,
										  $target, $pkghome);
					NodeUtils->runcmd($mv_cmd, 0);

					# Now make symbolic link for backward compatibility.
					my $ln_cmd =
					  "cd $distro_top; $::LN -s $svl/$target $target";
					NodeUtils->runcmd($ln_cmd, 0);
					$migrating = 1;
				}
			}
		}
	}
	if ($migrating)
	{
		ArchiveUtils->setupBootOrder($::ARCH, $distro_top, $distro_name,
									 $distro_version, $svl);
		$migrating = 0;
	}
	if (($effective_svl_level ne "GA") && (!$flag))
	{
		$flag  = 1;
		$svl   = $sp;
		@disks = @{$::pkgdefs{'sp_disks'}{$sp}};
		goto migration_check_again;
	}
	return;
}

#--------------------------------------------------------------------------------

=head3   TrimDisksArray 

        Trim disk hash according to the mediatype,
		For example, remove all the CD disk entries
		from the hash if mediatype is "DVD" 

        Arguments:
                $ref_disk - the distribution disks hash
				$mediatype - media type, can be "CD" or "DVD"

        Returns:
                The array that contains all the valid disk entries 
        Globals:
                N/A 
        Error:
                N/A 

        Example:
				@disks = ArchiveUtils->TrimDisksArray(\@disks, $::MEDIATYPE);

        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub TrimDisksArray()
{
	my ($class, $ref_disks, $mediatype) = @_;

	my @disks = @$ref_disks;

	my @res_disk = ();
	foreach my $ref (@disks)
	{
		my $disktype = $ref->{type} || "CD";
		if ($disktype eq $mediatype)
		{
			push @res_disk, $ref;
		}
	}

	return @res_disk;
}

#--------------------------------------------------------------------------------

=head3   getMediaTypeFromImages 

        Determine the $::MEDIATYPE through the images hash 

        Arguments:
                $images - images hash
                $disk_ref - the distribution disks hash

        Returns:
                N/A
        Globals:
                $::MEDIATYPE 
        Error:
                N/A 

        Example:
			ArchiveUtils->getMediaTypeFromImages($images, \@disks);			
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub getMediaTypeFromImages()
{
	my ($class, $images, $disk_ref) = @_;
	my @disks = @$disk_ref;

	# Flags to indicate whether CD/DVD images were found
	my $dvd_iso_found = 0;
    my $cd_iso_found = 0;

	# Whether there is any DVD/CD ISO in the path
	foreach my $diskref (@disks)
	{
		my $diskname = $diskref->{name};
		my $disknum  = $diskref->{num};
		my $disktype = $diskref->{type} || "CD";
		if ($images->{$diskname}->{$disktype}->{$disknum} && ($disktype eq "DVD"))
		{
			$dvd_iso_found = 1;
		}
		elsif ($images->{$diskname}->{$disktype}->{$disknum} && $disktype eq "CD") 
		{
			$cd_iso_found = 1;
		}
	}

	# DVDs take precedence over CDs
	if ($dvd_iso_found)
	{
		($::DEBUG) && print "DVD ISOs are found.\n";
		# return the mdia type
		$::MEDIATYPE = "DVD";
	}
	elsif ($cd_iso_found && !$dvd_iso_found) 
	{
		($::DEBUG) && print "CD ISOs are found.\n";
		# only found the CD ISO, remove the DVD entries ...
		$::MEDIATYPE = "CD";
	}
}

#--------------------------------------------------------------------------------

=head3   getDisksWithSameNum 

        Determine the $::MEDIATYPE through the images hash 

        Arguments:
                \@disks - disks reference
                $disk_num - the disk number

        Returns:
                @disk_with_same_num - The array that contains all the disk with same disk number
        Globals:
                N/A 
        Error:
                N/A 

        Example:
			my @disk_with_same_num = ArchiveUtils->getDisksWithSameNum($disks_ref, $disknum);			
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub getDisksWithSameNum()
{
	my ($class, $disks_ref, $disknum) = @_;
	
	my @disks = @$disks_ref;
	my @disk_with_same_num = ();
	
	foreach my $disk (@disks)
	{
		if ($disk->{num} eq $disknum)
		{
			push @disk_with_same_num, $disk;
		}
	}
	return @disk_with_same_num;
}

#--------------------------------------------------------------------------------

=head3   copyDistroPkgs 

        Copy distribution packages from CD/DVD disks or ISO files 

        Arguments:
				$platform - platform name
				$arch     - architecture
				$distro_top - distribution top directory
				$distro_name - distribution name 
				$distro_version - distribution version

        Returns:
                1 - successful
				0 - fail 
        Globals:
                $::MEDIATYPE 
				$::DEVICEDRV
        Error:
                N/A 

        Example:
				if (!ArchiveUtils->copyDistroPkgs($platform,$arch,$distro_top, $distro_name, $distro_version, $svl, $extracds, $nomountcd)

        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub copyDistroPkgs
{
	my ($class, $platform, $arch, $distro_top, $distro_name, $distro_version,
		$sp, $extracds, $nomountcd)
	  = @_;
	my ($cmd, $output, $orig_dir, $target, $ismounted, $pkghome, $rc);

	#my $platform = $::PLTFRM;

	if ($arch =~ /i.86/) { $arch = "i386"; }

	$orig_dir = getcwd();

	# Create the directory
	if (!(-e $distro_top))
	{
		$cmd = "$::MKDIR -p $distro_top;";
		$output = NodeUtils->runcmd($cmd, 0);
	}

	my @disks = ();
	if (!defined($::pkgdefs{'distro_disks'}))
	{
		# set the global variable for get_pkgdefs
		$::ATTRS{"InstallServiceLevel"} = $sp;
		ServerUtils->get_pkgdefs($platform, $distro_name, $distro_version,
								 $arch);
		$::ATTRS{"InstallServiceLevel"} = '';
	}

	my @orig_disks = @{$::pkgdefs{'distro_disks'}};
	# Add dvd disks into list
	if ($::pkgdefs{'distro_dvd_disks'})
	{
		push @orig_disks, @{$::pkgdefs{'distro_dvd_disks'}};
	}
    my ($effective_distro_name, $effective_distro_ver, $effective_svl_level) =
        NodeUtils->getEffectiveDistro($distro_name, $distro_version, $sp);

	if ($effective_svl_level eq "GA")
	{
		@disks = @orig_disks;
	}
	else
	{
		if (
			$effective_distro_name =~ /SLES/ && $effective_distro_ver <= 9
			&& !ArchiveUtils->is_req_disks_copied(
												  $distro_top . "/GA",
												  \@orig_disks
												 )
		   )
		{

			# print "The CDs of base distribution must be copied before service level, please run copycds with InstallServiceLevel paratermer set to \"GA\" first. \n";
			MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
										 'csminstall', 'I', 'IMsgGAFirst');
			$rc = 0;
			goto done;
		}
		if (defined $::pkgdefs{'sp_disks'}{$sp})
		{
			@disks = @{$::pkgdefs{'sp_disks'}{$sp}};
			# Add dvd disks into list
			if ($::pkgdefs{'sp_dvd_disks'}{$sp})
			{
				push @disks, @{$::pkgdefs{'sp_dvd_disks'}{$sp}};
			}
		}
		else
		{

			# If the service level is not defined in pkgdefs file, take it as user defined service level.
			# Don't prompt for disks. Just copy the entire contents of each -p path to the target location in /csminstall
			$rc =
			  ArchiveUtils->copyUserDistroPkgs($platform, $arch, $distro_top,
											$distro_name, $distro_version, $sp);
			goto done;
		}
	}

	if (scalar(@disks) <= 0)
	{
		$rc = 0;
		goto done;
	}

	my @pkgpath = ArchiveUtils->foundPkgPath("iso");

	if ($::PLTFRM eq 'AIX')
	{
		my @isoFile = grep /\.iso$/, @pkgpath;
		if (@isoFile)
		{
			my $grepStr = join "|", @isoFile;
			@pkgpath = grep !/$grepStr/, @pkgpath if @isoFile;
			MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
				'csminstall', 'W', 'EMsgAIXNotSupportISO', join("\n", @isoFile))
			  if !$::AIXISOWARN;
			$::AIXISOWARN = 1;
		}
	}
	my $images = ArchiveUtils->probeImages(\@pkgpath, \@disks);


	if (scalar(%$images))
	{
		# The $::MEDIATYPE will be in the images
		ArchiveUtils->getMediaTypeFromImages($images, \@disks);
		@disks = ArchiveUtils->TrimDisksArray(\@disks, $::MEDIATYPE);
	}

	#Determine the cd/dvd device list
	my @devarray = ArchiveUtils->determineDeviceList();
	
	foreach my $diskref (@disks)
	{

		next if (defined($diskref->{not_for_full_install}) && !$extracds);
		next if (defined($diskref->{is_extracd})           && !$extracds);
		
		my ($realdisk_ref, $ismounted, $target) = ArchiveUtils->MountMedia($diskref, \@disks, $images, \@devarray, $nomountcd);
		@disks = ArchiveUtils->TrimDisksArray(\@disks, $::MEDIATYPE);
	

		# Copy the contents of the Distro disk to /csminstall
		# Handle SLES 8.1
		if ($platform eq "Linux")
		{
			if ($distro_name =~ /SLES/)
			{
				$pkghome =
				  $distro_top . "/" . $sp . "/" . $realdisk_ref->{diskid_target};
			}
			else
			{
				if (   ($effective_distro_name =~ /RedHat/)
					&& ($effective_distro_ver eq "3")
					&& ($arch           eq "ppc64")
					&& (($effective_svl_level eq "GA") ||
                        ($effective_svl_level eq "QU1"))
					&& (grep /extra/, $realdisk_ref->{diskid_target}))
				{
					$pkghome = $distro_top . "/" . $realdisk_ref->{diskid_target};
				}
				else
				{
					my $my_dirname;
					if ($sp eq "GA")
					{
						$my_dirname =
						  $distro_name . $distro_version . "-" . $arch;
					}
					else
					{
						$my_dirname =
						  $distro_name . $distro_version . "-" . $sp;
					}
					$pkghome = $distro_top . "/" . $my_dirname;
				}
			}
			if (!-d $pkghome)
			{
				$cmd = "$::MKDIR -p $pkghome";
				NodeUtils->runcmd($cmd, -1);
			}
		}

		MessageUtils->messageFromCat(
									 'csmInstall.cat',   $::MSGMAPPATH,
									 'csminstall',       'I',
									 'IMsgCOPYING_DISK', $realdisk_ref->{name},
									 $::MEDIATYPE, $realdisk_ref->{num}
									);

		$rc = ArchiveUtils->copyAndShowProgress($target, $pkghome);

		# Unmount the CD-ROM if necessary
		if ($ismounted)
		{
			if ($platform eq "AIX")
			{
				`$::UNMOUNT $target 2>/dev/null 1>/dev/null`;
			}
			else
			{
				$cmd = "$::UNMOUNT $target > /dev/null 2>&1";
				NodeUtils->runcmd("$cmd", -1);
			}
		}

		if(!$rc)
		{
			goto done; # error occured during ISO copying, skip the rest of CDs.
		}
	}    # end foreach my $diskref...

	$rc = 1;    # successfully :)

  done:
	chdir($orig_dir);
	return $rc;
}

#--------------------------------------------------------------------------------

=head3    createRPMSdir

        Creates the proper /csminstall/.../RPMS directory structure 

        Arguments:
                $OS
                $distro_name
                $distro_version
                $arch

        Returns:
                undefined
        Globals:
                none
        Error:
                undefined

        Example:
                ServerUtils->createRPMSdir (   "Linux",
                                                $::DISTRO_NAME,
                                                $::DISTRO_VERSION,
                                                $::ARCH);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub createRPMSdir
{
	my ($class, $OS, $dist, $distver, $arch, $svclevel) = @_;
	use File::Path;    # Provides mkpath()

	my $base = ServerUtils->getBaseDir($OS, $dist, $distver, $arch, $svclevel);
	my $SMS_RPMS = ArchiveUtils->getRPMSDir($OS, $dist, $distver, $arch);
	my $svclevelRPMS =
	  ArchiveUtils->getSvcLevelRPMSDir($OS, $dist, $distver, $arch, $svclevel);
	my $SMS_updates = ServerUtils->getUpdateDir($OS, $dist, $distver, $arch);
	my $svclevel_updates =
	  ServerUtils->getUpdateDir($OS, $dist, $distver, $arch, $svclevel);
	my $SMS_install = ServerUtils->getInstallDir($OS, $dist, $distver, $arch);
	my $svclevel_install =
	  ServerUtils->getInstallDir($OS, $dist, $distver, $arch, $svclevel);
	my $realRPMS = ArchiveUtils->getRealRPMSDir($OS, $dist, $distver, $arch);
    my $install_images = 
	  ServerUtils->getInstallImagesDir($OS, $dist, $distver, $arch);
    my $svclevel_install_images = 
      ServerUtils->getInstallImagesDir($OS, $dist, $distver, $arch, $svclevel);
    my $csmAddon = ServerUtils->getCSMAddonDir($OS, $dist, $distver, $arch);
    my $svclevel_csmAddon = ServerUtils->getCSMAddonDir($OS, $dist, $distver, $arch, $svclevel);

    my ($effective_distro_name, $effective_distro_ver, $effective_svl_level) =
        NodeUtils->getEffectiveDistro($dist, $distver, $svclevel);
	if (!-e "$base")
	{
		my $mig = 0;

		# Check for migration
		if (   ($effective_distro_name =~ /RedHat/)
			&& ($effective_distro_ver eq "3")
			&& ($arch =~ /ppc64/)
			&& ($effective_svl_level eq "QU1"))
		{
			my $gatop =
			  ServerUtils->getBaseDir($OS, $dist, $distver, $arch, "GA");
			my @junks=split("/",$gatop);
			my $opp_ga=pop(@junks);
			if (-e $gatop)
			{
				my $cmd = "/bin/ln -s $opp_ga $base";
				NodeUtils->runcmd($cmd, 0);
				$mig = 1;
			}
		}
		if (!$mig)
		{
			mkpath($base, $::VERBOSE, 0755);
		}
	}
	if (!-e "$install_images")
	{
		mkpath($install_images, $::VERBOSE, 0755);
	}
    if (!-e "$svclevel_install_images")
    {
        mkpath($svclevel_install_images, $::VERBOSE, 0755);
    }
    if (!-e "$csmAddon")
    {
        mkpath($csmAddon, $::VERBOSE, 0755);
    }
    if (!-e "$svclevel_csmAddon")
    {
        mkpath($svclevel_csmAddon, $::VERBOSE, 0755);
    }

	if (!-e "$realRPMS")
	{
		mkpath($realRPMS, $::VERBOSE, 0755);
	}
	if (!-e "$SMS_updates")
	{
		mkpath($SMS_updates, $::VERBOSE, 0755);
	}
	if (!-e "$SMS_install")
	{
		mkpath($SMS_install, $::VERBOSE, 0755);
	}
	if (!-e "$svclevelRPMS")
	{
		mkpath($svclevelRPMS, $::VERBOSE, 0755);
	}
	if (!-e "$svclevel_updates")
	{
		mkpath($svclevel_updates, $::VERBOSE, 0755);
	}
	if (!-e "$svclevel_install")
	{
		mkpath($svclevel_install, $::VERBOSE, 0755);
	}

	if ($::command_line =~ /installms/)
	{

		# To migrate from Amenia, installms should move all the rpms from Amenia's
		# /csminstall/.../SLES/RPMS/ to /csminstall/.../GA/RPMS/ to avoid prompting
		# for CDs to copy os_prereq rpms. Symbolic link files must NOT be moved
		# to avoid links to other service level's rpms being moved into the new
		# service level sub-directory.
		my $move_flg = 1;
		my $d        = new DirHandle "$svclevelRPMS";
		if (defined $d)
		{
			while (defined($_ = $d->read))
			{
				next if ($_ eq "." || $_ eq "..");
				if ($_ =~ /\.rpm$/)
				{

					# No need to migrate if service level sub-dir already contains rpm files.
					$move_flg = 0;
					last;
				}

			}
		}
		if ($move_flg)
		{
			$d = new DirHandle "$realRPMS";
			if (defined $d)
			{
				my @mv_files;
				while (defined($_ = $d->read))
				{
					next if ($_ eq "." || $_ eq ".." || $_ !~ /\.rpm$/);
					my $tmp_filenam = "$realRPMS" . "/" . "$_";
					if (-l "$tmp_filenam")
					{
						next;
					}
					push @mv_files, $tmp_filenam;
				}
				if (scalar(@mv_files) >= 1)
				{
					my $mv_file_list = join(" ", @mv_files);
					my $mv_cmd = "/bin/mv $mv_file_list $svclevelRPMS/";
					NodeUtils->runcmd($mv_cmd, 0);
				}
			}
		}
	}
	if ($effective_distro_name =~ /RedHat/)
	{

		# Make RPMS symbolic link inside each service level's own directory.
		# i.e. $distro_top/$svclevel/RPMS ->  $distro_top/$svclevel/RedHat/RPMS.
        # changed for supporting rhel5. rhel5 puts rpms in "Server", 
        # not RedHat/RPMS.
        # When rhel5, this will link $distro_top/$svclevel/RPMS to
        # $distro_top/$svclevel/Server
		my (undef,$srcd) = split("$base/", $svclevelRPMS);
		my $targetdir = "$base/RPMS";
		my $link = "RPMS";
		unlink $targetdir;
		rmdir $targetdir;
		my $orig_dir = Cwd::getcwd();
		chdir $base;
		symlink $srcd, $link;
		chdir $orig_dir;

		# this is a symbolic link to ->RedHat/RPMS
		# force the change to make the symbolic link
		# relative, instead of full path.
		if (-d $SMS_RPMS)
		{
			unlink($SMS_RPMS);
		}

        # changed for supporting rhel5
        # When rhel5, this would link,
        # ../<distro_name>/<version>/<arch>/RPMS to
        # ../<distro_name>/<version>/<arch>/Server.
		my ($basedir, undef) = split /\/$::pkgdefs{DISTRO_RPMDIR}/, $realRPMS;
		$srcd    = $::pkgdefs{DISTRO_RPMDIR};
        my $target  = "RPMS";
		chdir $basedir;
		symlink $srcd, $target;
		chdir $orig_dir;
	}

	#SLES Changes
	elsif ($effective_distro_name =~ /SLES/)
	{

		# There is no need for SLES to create the RPMS symbolic link inside service level directory.
		# But still need to create the top level symbolic link.
		if (-d $SMS_RPMS)
		{
			unlink($SMS_RPMS);
		}

		my $basedir = "$realRPMS/../../";
		my @pa      = split("/", $realRPMS);
		my $srcd    = pop @pa;
		$srcd    = pop(@pa) . "/" . $srcd;
		$basedir = join("/", @pa);
		my (undef, $target) = split("$basedir/", $SMS_RPMS);
		my $orig_dir = Cwd::getcwd();
		chdir $basedir;
		symlink $srcd, $target;
		chdir $orig_dir;

		#End of SLES Changes
	}
	else
	{
		mkpath($SMS_RPMS, $::VERBOSE, 0755);
	}
}

#--------------------------------------------------------------------------------

=head3    createRPMList

        Creates  a global hash of installed RPMS.


        Arguments:
                none


        Returns:
                none


        Globals:
                $::rpmShortnames
                $::rpmBasenames
        Error:
                undefined


        Example:
                ArchiveUtils->createRPMList();


        Comments:
                Subroutine must be called before checkRPMInstall.

=cut

#--------------------------------------------------------------------------------

sub createRPMList
{

	#first run rpm -qa to get a list of all rpms installed
	my @output = `$::RPMCMD -qa`;
	foreach my $line (@output)
	{
		chomp $line;
		$::rpmShortnames{$line} = 1;

		#now create the basename
		my @parts = split /-/, $line;
		my $basename = shift @parts;
		my $p;
	  BNAME: while ($p = shift @parts)
		{
			if (($p !~ /^\d+\.[\d\.]*/) && ($p !~ /^\d+\*$/))
			{    #if it doesn't start with a digit
				$basename .= "-$p";
			}
			else
			{
				unshift @parts, $p;
				last BNAME;
			}
		}
		$::rpmBasenames{$basename} = 1;
	}

}

#--------------------------------------------------------------------------------

=head3  createBatch  

        Groups the given nodes according to the fanout value.

        Arguments:
                $nodeList 
                $batchNodes 
                $fanout

        Returns:
                Fills $batchNodes.
        Globals:
                none    
        Error:
                undefined

        Example:
               my (@nodeList) = @_;
               my @batch = ();
               my $fanout = $ENV{'CSM_FANOUT'};
               if ($fanout eq '') { $fanout = $::DEFAULT_FANOUT; }
               ArchiveUtils->createBatch(\@nodeList, \@batch, $fanout); 

        Comments:

=cut

#--------------------------------------------------------------------------------
sub createBatch
{
	my ($class, $nodeList, $batchNodes, $fanout) = @_;
	my $length = @$nodeList;
	if ($length <= $fanout)
	{
		push(@$batchNodes, $nodeList);
	}
	else
	{

		#group them by the size of fanout value
		my $size = $fanout;
		my @batch;
		foreach my $node (@$nodeList)
		{
			push(@batch, $node);
			$size--;
			unless ($size)
			{
				my @newBatch = @batch;
				push(@$batchNodes, \@newBatch);
				$size  = $fanout;
				@batch = ();
			}
		}    #end of foreach
		if (@batch)
		{

			#last batch
			push(@$batchNodes, \@batch);
		}

	}    # end of else
}

#--------------------------------------------------------------------------------

=head3    findImagesInPath

        Find the images  in the given path.

        Arguments:
                $image
                @path

        Returns:
                @array of found $images in $path
        Globals:
                none    
        Error:
                undefined

        Example:
                 my @images =
                        ServerUtils->findImagesInPath($file, @::PKGPATH);

        Comments:
                The $image parameter can have a *, i.e. the routine
                can find more than one $image in the $path.

=cut

#--------------------------------------------------------------------------------

sub findImagesInPath
{

	# the $image variable may have a *, so we may find more than one that
	# matches.

	my ($class, $image, @path) = @_;
	my (@tmpImages, @returnImages, $tmpImage, $realImage);

	foreach my $p (@path)
	{
		my $filename = "$p/$image";

		# use the ls command here instead of the -e operator
		# so we can handle *s in the names

		if (`/bin/ls $filename 2> /dev/null`)
		{

			# put all of the images we just got into @tmpImages array
			chomp(@tmpImages = `/bin/ls $filename`);

			# Go through each tmpImage and see if the base name already
			# exists in the returnImages array.  If it does exist, don't
			# put it in, otherwise, put it in!

			foreach $tmpImage (@tmpImages)
			{

				# get the base name of the test image.

				my $tstImageBaseName;
				my $defined = 0;
				($tstImageBaseName = $tmpImage) =~ s:^.*/::g;

				foreach $realImage (@returnImages)
				{

					# get the base name of the real image

					my $realImageBaseName;
					($realImageBaseName = $realImage) =~ s:^.*/::g;

					if ($realImageBaseName eq $tstImageBaseName)
					{
						$defined = "true";
						last;
					}
				}

				unless ($defined)
				{
					push @returnImages, $tmpImage;
				}
				else
				{

					print "\n\nnot copying $tmpImage\n\n";
				}
			}
		}
	}
	return @returnImages;
}

#--------------------------------------------------------------------------------

=head3    findRpmInPath

         Find the rpm package in the given path

        Arguments:
                $rpmPackageName
                @path

        Returns:
                Absolute path to rpm package. Package name is not
                returned with path value.

        Globals:
                none
        Error:
                none
        Example:
                $path =
                   ArchiveUtils->findRpmInPath($rpmfile, @my_path);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub findRpmInPath
{
	my ($class, $rpmfile, @path) = @_;

	foreach my $p (@path)
	{
		my $filename = "$p/$rpmfile";

		# use the ls command here instead of the -e operator so we can handle *s in the names
		($::DEBUG) && print "looking for $filename: ", `/bin/ls $filename`,
		  "\n";
		if (`/bin/ls $filename 2> /dev/null`)
		{

			# Special case for Java RPM.
			# Need to handle differently depending on the Vendor attribute.
			if ($rpmfile =~ /IBMJava2-JRE/)
			{

				# process the Java rpm name for the packager...
				my @outout =
				  NodeUtils->runcmd(
					"$::RPMCMD -qp --qf '%{Name}::%{Version}::%{Release}::%{Arch}::%{Vendor}\n' $filename",
					-1
				  );
				foreach my $i (@outout)
				{
					if (!($i =~ /.*::.*::.*::.*::.*/)) { next; }
					my ($name, $version, $release, $arch, $packager_name) =
					  split(/::/, $i);

					# The IBMJava2-JRE RPM is only valid if the vendor is IBM or UnitedLinux
					if (   ($packager_name =~ /International Business Machines/)
						|| ($packager_name =~ /UnitedLinux/))
					{
						MessageUtils->messageFromCat('csmInstall.cat',
							  $::MSGMAPPATH, 'csminstall', 'V', 'IMsgFOUND_RPM',
							  $rpmfile, $p);
						return $p;
					}
					else { next; }
				}
			}
			else
			{

				MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
							  'csminstall', 'V', 'IMsgFOUND_RPM', $rpmfile, $p);
				return $p;
			}
		}
	}

	return undef;
}

#---------------------------------------------------------------------------------

=head3		isOneFileInList
		Check if a file is in the list.
		Arguments:
				file: the file need to be checked
				fListRef: a reference for a file name array. 
						  The file name can end with "*", like aaa-* ,aaa*,
						  aaa-3.*, or can only contain the short name, like "aaa".

		Returns:
				1: this file is in the list
				0: this file is not in the list
        Globals:
				none
        Error:
                none

        Example:
                ArchiveUtils->isOneFileInList();

        Comments:
                none

=cut

#---------------------------------------------------------------------------------

sub isOneFileInList
{
	my ( $class, $file, $fListRef) = @_;
	if ( !defined( $fListRef) || scalar( @$fListRef) == 0 )
	{
		return 0;
	}
	my @fList = @$fListRef;
	for my $f ( @fList)
	{
		if ( $f =~ /^(.*)\*$/)
		{
			$f = $1;
		}
		if ( $file =~ /^\Q$f\E.*/)
		{
			return 1;
		}
	}
	return 0;
}

#-------------------------------------------------------------------------------

=head3	getPackageList

	get packages lists

    Arguments:
        none
    Returns:
        # Lists of Open Source Packages that are required on the node
        @::opensrc_rpm_prereqs

        # Lists of packages that are part of each CSM component
        @::csm_packages

	# rsct client packages
        @::rsct_packages

        # Operating System Prereqs for Managed Nodes
        @::os_prereq

        # Apache packages needed for the install server
        @::apache_packages

	# Optional Packages to copy for Managed Nodes
	@::optional_copy_packages

	# Optional Packages to install on Managed Nodes
	@::optional_install_packages

	@ package version name
        $::PackageVersionName

    Globals:
        see returns
    Error:
        multiple E1 msgs
    Example:
        ArchiveUtils->getPackageList();
    Comments:
        none

=cut

#-------------------------------------------------------------------------------

sub getPackageList
{
	my ($class, $shouldCopyAll) = @_;
	%::pkgdefs =
	  ServerUtils->get_pkgdefs(
							   $::ATTRS{'InstallOSName'},
							   $::ATTRS{'InstallDistributionName'},
							   $::ATTRS{'InstallDistributionVersion'},
							   $::ATTRS{'InstallPkgArchitecture'},
							   'MgdNode',
							   $::ATTRS{'InstallCSMVersion'}
							  );

	# $::ATTRS{'InstallServiceLevel'} can be determined only after ServerUtils->get_pkgdefs.
	if (!($::ATTRS{'InstallServiceLevel'}))
	{
		if (($::ATTRS{'InstallOSName'} eq "Linux") && ($::PLTFRM eq "Linux"))
		{
			my $msdistro    = NodeUtils->get_DistributionName();
			my $msdistrover = NodeUtils->get_DistributionVersion();
			if (   ($msdistro eq $::ATTRS{'InstallDistributionName'})
				&& ($msdistrover eq $::ATTRS{'InstallDistributionVersion'}))
			{

				# Use Install Server's service level as the default value for node.
				$::ATTRS{'InstallServiceLevel'} = NodeUtils->get_ServiceLevel;
			}
			else
			{

				# If MS and node have different distribution/version, the node's default Service Level should be "GA".
				$::ATTRS{'InstallServiceLevel'} = "GA";
			}
		}
		elsif (($::ATTRS{'InstallOSName'} eq "Linux") && ($::PLTFRM eq "AIX"))
		{

			# When InstallServiceLevel attribute is not specified, and MS is not Linux,
			# use "GA" as default value.
			$::ATTRS{'InstallServiceLevel'} = "GA";
		}
	}

	if ($::ATTRS{'InstallOSName'} eq "AIX")
	{
		$::osTypeVersion =
		  $::ATTRS{'InstallOSName'} . $::ATTRS{'InstallDistributionVersion'};
		$::INSTALLDIR       = "/csminstall";
		$::INSTALLDIR_CSM   = $::INSTALLDIR . "/csm";
		$::INSTALLDIR_CSMVR =
		  $::INSTALLDIR_CSM . "/" . $::ATTRS{'InstallCSMVersion'};
		$::INSTALLDIR_CSMVRBIN = $::INSTALLDIR_CSMVR . "/bin";

		$::INSTALLDIR_OS    = $::INSTALLDIR . "/" . $::ATTRS{'InstallOSName'};
		$::INSTALLDIR_OSVER =
		  $::INSTALLDIR_OS . "/" . $::ATTRS{InstallDistributionVersion};
		$::INSTALLDIR_OSCSM = $::INSTALLDIR_OS . "/csm";

		$::INSTALLDIR_CSMVER =
		  $::INSTALLDIR_OSCSM . "/" . $::ATTRS{'InstallCSMVersion'};

		$::INSTALLDIR_CSMBIN    = $::INSTALLDIR_CSMVER . "/bin";
		$::INSTALLDIR_CSMPKG    = $::INSTALLDIR_CSMVER . "/packages";
		$::INSTALLDIR_CSMRPM    = $::INSTALLDIR_CSMPKG . "/rpms";
		$::INSTALLDIR_CSMINSTLP = $::INSTALLDIR_CSMPKG . "/installp";


	}
	elsif ($::ATTRS{'InstallOSName'} eq "Linux")
	{
		$::osTypeVersion =
		    $::ATTRS{'InstallOSName'}
		  . $::ATTRS{'InstallDistributionName'}
		  . $::ATTRS{'InstallDistributionVersion'};
		$::INSTALLDIR       = "/csminstall";
		$::INSTALLDIR_CSM   = $::INSTALLDIR . "/csm";
		$::INSTALLDIR_CSMVR =
		  $::INSTALLDIR_CSM . "/" . $::ATTRS{InstallCSMVersion};
		$::INSTALLDIR_CSMVRBIN = $::INSTALLDIR_CSMVR . "/bin";

		$::INSTALLDIR_OS =
		    $::INSTALLDIR . "/"
		  . $::ATTRS{"InstallOSName"} . "/"
		  . $::ATTRS{'InstallDistributionName'};

		#/csminstall/Linux/RedHatEL-AS
		$::INSTALLDIR_OSVER =
		  $::INSTALLDIR_OS . "/" . $::ATTRS{InstallDistributionVersion};
		$::INSTALLDIR_OSVER_ARCH =
		  $::INSTALLDIR_OSVER . "/" . $::ATTRS{'InstallPkgArchitecture'};

		$::INSTALLDIR_OSCSM  = $::INSTALLDIR_OS . "/csm";
		$::INSTALLDIR_CSMVER =
		  $::INSTALLDIR_OSCSM . "/" . $::ATTRS{InstallCSMVersion};

		my $INSTALLDIR_OS_NAME = $::PREREQS_ATTR{DistributionName};
		if ($::PREREQS_ATTR{DistributionName} =~ /RedHat/)
		{
			$INSTALLDIR_OS_NAME = "RedHat";
		}
        my ($effective_distro_name) =
            NodeUtils->getEffectiveDistro($::ATTRS{'InstallDistributionName'});

		# /csminstall/Linux/RedHatEL-AS/3/i386/RedHatEL-AS3-QU2/RedHat/RPMS for RHEL,
		# or /csminstall/Linux/SLES/8.1/i386/SP3/RPMS for SLES
		if ($effective_distro_name =~ /RedHat/)
		{
			my $my_dirname;
			if ($::ATTRS{'InstallServiceLevel'} eq "GA")
			{
				$my_dirname =
				    $::ATTRS{'InstallDistributionName'}
				  . $::ATTRS{InstallDistributionVersion} . "-"
				  . $::ATTRS{'InstallPkgArchitecture'};
			}
			else
			{
				$my_dirname =
				    $::ATTRS{'InstallDistributionName'}
				  . $::ATTRS{InstallDistributionVersion} . "-"
				  . $::ATTRS{'InstallServiceLevel'};
			}
			$::INSTALLDIR_OS_NAME_RPMS =
			  $::INSTALLDIR_OSVER_ARCH . "/" . $my_dirname .
              "/". $::pkgdefs{DISTRO_RPMDIR};
		}
		else
		{
			$::INSTALLDIR_OS_NAME_RPMS =
			    $::INSTALLDIR_OSVER_ARCH . "/"
			  . $::ATTRS{'InstallServiceLevel'} . "/RPMS";
		}

		$::INSTALLDIR_CSMBIN = $::INSTALLDIR_CSMVER . "/bin";
		$::INSTALLDIR_CSMPKG = $::INSTALLDIR_CSMVER . "/packages";

	}
	else
	{
		MessageUtils->message("E2", 'EMsgINVALID_OSTYPE');
	}
    my @rpmLists;
    push @rpmLists, $::pkgdefs{'pkgs'}{'MgdNode'};
    push @rpmLists, $::pkgdefs{'pkgs'}{'InstallServer'};
    push @rpmLists, $::pkgdefs{'pkgs'}{'MgmtServer'} if ($shouldCopyAll);
    return \@rpmLists;         
}

#--------------------------------------------------------------------------------

=head3    get_PkgArchitecture

        Linux only routine.

        Returns the rpm package architure for the OS in $::PLTFRM
        i.e. i386 for linux

        Arguments:
                $output,
                $osarch
        Returns:
                none
        Globals:
                $::PLTFRM
        Error:
                messageFromCat E2

        Example:
                $::ATTRS{'InstallPkgArchitecture'} =
                        ServerUtils->get_PkgArchitecture;

        Comments:
                Works for Linux.  Returns null values $::PLTFRM
                is AIX, and exits with E2 for any other $::PLTFRM. 

=cut

#--------------------------------------------------------------------------------

sub get_PkgArchitecture
{
	my ($cmd, $output, $osarch);
	if ($::PLTFRM eq "Linux")
	{
		$cmd    = "$::UNAME -m";
		$output = NodeUtils->runcmd($cmd);
		chomp($output);

		$osarch = $output;

		return (length($osarch) ? $osarch : undef);
	}
	elsif ($::PLTFRM eq "AIX")
	{
		$osarch = "";
		return (length($osarch) ? $osarch : undef);
	}
	else
	{
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									 'csminstall', 'E2', 'EMsgINVALID_OSTYPE');
	}
}

#--------------------------------------------------------------------------------

=head3    getRealRPMSDir

        Get the RPMS directory for the install attributes parameters

        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur

        Returns:
                appropriate path name for parameter values
        Globals:
                none
        Error:
                none
        Example:
                 my $realRPMS =
                    ArchiveUtils->getRealRPMSDir($OS, $dist, $distver, $arch);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getRealRPMSDir
{
	my ($class, $os, $distributor, $version, $arch) = @_;
	my $distro_dir = $distributor;
	my $base_install = $::CSMINSTDIR;
	my ($effective_distro_name) = NodeUtils->getEffectiveDistro($distributor);
	if ($distributor =~ /RedHat/) { $distro_dir = "RedHat"; }
	if ($arch        =~ /i.86/)   { $arch       = "i386"; }
	if ($effective_distro_name =~ /RedHat/)
	{
		return "$base_install/$os/$distributor/$version/$arch/$::pkgdefs{DISTRO_RPMDIR}";
	}else
	{
		return "$base_install/$os/$distributor/$version/$arch/$distro_dir/RPMS";
	}
}

#--------------------------------------------------------------------------------

=head3    getSvcLevelRPMSDir

        Get the service level sub directory from the install attributes parameters

        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur
                $InstallServiceLevel

        Returns:
                appropriate path name for parameter values
        Globals:
                none
        Error:
                none
        Example:
                 my $realRPMS =
                    ArchiveUtils->getSvcLevelRPMSDir($OS, $dist, $distver, $arch, $svclevel);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getSvcLevelRPMSDir
{
	my ($class, $os, $distributor, $version, $arch, $svclevel) = @_;
	my $targetdir;
	my $base_install = $::CSMINSTDIR;
	my ($effective_distro_name) = NodeUtils->getEffectiveDistro($distributor);

	if ($arch =~ /i.86/) { $arch = "i386"; }

	if ($os eq "Linux")
	{
		if ($effective_distro_name =~ /SLES/)
		{
			$targetdir =
			  "$base_install/$os/$distributor/$version/$arch/$svclevel/RPMS";
		}
		else
		{
			my $my_dirname;
			if ($svclevel eq "GA")
			{
				$my_dirname = $distributor . $version . "-" . $arch;
			}
			else
			{
				$my_dirname = $distributor . $version . "-" . $svclevel;
			}
			$targetdir =
			  "$base_install/$os/$distributor/$version/$arch/$my_dirname/$::pkgdefs{DISTRO_RPMDIR}";
		}
	}

	return $targetdir;
}

#--------------------------------------------------------------------------------

=head3    getRecentVersionRpm

        Get the most recent version of the rpm argument.

        Arguments:
                $rpmName
        Returns:
                Recent RPM as a string
        Globals:
                none
        Error:
                none
        Example:
                $rpmfile =
                    ArchiveUtils->getRecentVersionRpm($path, $rpmFileName);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getRecentVersionRpm
{
	my ($class, $path, $rpm_name) = @_;

	my %localversions_hash = ();
	my %file_name_hash     = ();
	my %packver;
	my $rpm_basename;
	my $test_rpm_name;
	my $test_rpm_name_short;
	my @output = NodeUtils->runcmd("$::LS  -1 $path/$rpm_name");
	chomp @output;

	# Determine the basename of this rpm (strip off .arch.rpm)

	# %::PackageVersionName hash.  This code can handle an rpm
	# provided in these formats:
	#    rpmname
	#    rpmname*
	#    rpmname-*
	#    rpmname-**
	my (@test_rpm_split) = split(/\./, $rpm_name);
	my $splitpos = $#test_rpm_split - 2;
	$test_rpm_name = join('.', @test_rpm_split[0 .. $splitpos]);

	# Replace two *'s with one * (in case the rpm in pkgdefs already has a *)
	$test_rpm_name =~ s/\*\*$/*/;

	# Also, try stripping off the "*" or "-*" at the end
	$test_rpm_name_short = $test_rpm_name;
	$test_rpm_name_short =~ s/\*$//;
	$test_rpm_name_short =~ s/-$//;

	foreach my $r (keys %::PackageVersionName)
	{
		my $fullname = $::PackageVersionName{$r}{'FullName'};
		if (   ($fullname eq $test_rpm_name)
			or ($fullname eq $test_rpm_name_short))
		{
			$rpm_basename = $r;
			last;
		}
	}

=head3 mmm - getRecentVersionRpm

	-----------------
	This disconnected block of code is being preserved in the file, rather than
	being removed.  It is a part of a change which is pending, which, due to the
	scale of the change, is being defered to a bac or brahms, where it will be
	evaluated and accepted or rejected. 
	-----------------

    # it might be in an iso file

    if (!$rpm_basename)
    {
        #print "getRecentVersionRpm: rpm_basename not found using defs_filter.\n";

        my $hash_ref = RpmIndex->defs_filter ( $rpm_name );
        my %ret_hash = %$hash_ref;
	my $key;

	foreach my $arch ( keys %ret_hash ) 
	{
	     my @key_array = @ret_hash { $arch };
	     foreach $key ( @key_array )
	     {
                #print "found $key\n";
             }
            if (RpmIndex->in_iso_image($key) == $::OK )
            {
                #print "found $key to be in an iso  - returning it\n";
	        #$rpm_basename = $key;
		return $key;
            }
        }
    }

=cut

	if (!$rpm_basename)
	{

		#print "CANNOT FIND BASENAME FOR $rpm_name\n";
		MessageUtils->messageFromCat(
									 'csmInstall.cat',       $::MSGMAPPATH,
									 'csminstall',           'E1',
									 'EMsgPREREQ_NOT_FOUND', $rpm_name,
									 $path
									);
		exit 1;
	}

	# if there is only one rpm with the same basename, then return that rpm name
	if (scalar(@output) == 1)
	{
		my ($fn) = $output[0] =~ m|^.*?([^/]+)$|;    # get rid of the directory.
		      # Get the basename of the rpm found in the path or on the CD.
		my ($ipkg, $VERSION, $RELEASE) = $fn =~ /^(.+)-(\d.*)-(\d.*)\..+\..+$/;

		# The name of the RPM must match the basename of the rpm in pkgdefs
		if ($rpm_basename ne $ipkg)
		{

			#print "$rpm_basename DOES NOT MATCH $ipkg\n";
			MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
					  'csminstall', 'E1', 'EMsgPREREQ_NOT_FOUND', $rpm_basename,
					  $path);
			exit 1;
		}
		return $output[0];
	}

	# If there are multiple RPMs with the same basename, then compare their
	# versions to see which is the latest.
	my $i = 0;
	foreach my $rpm1 (@output)
	{

		if ($rpm1 =~ /openCIMOM/)
		{

			# For Linux, copy and install openCIMOM-0.8-1.noarch.rpm,
			# For AIX, copy and install openCIMOM-0.8-1.aix5.2.noarch.rpm.
			my $platform;
			if ($::command_line =~ /installms/)
			{
				$platform = $::PLTFRM;
			}
			else
			{
				$platform = $::ATTRS{"InstallOSName"};
			}

			if ($platform eq "Linux")
			{
				next if ($rpm1 =~ /aix/);
			}
			else
			{
				next if ($rpm1 !~ /aix/);
			}
		}

		####jcran changes start
		#### --getting the version from the output of LS, instead of running rpm multiple times

		# my $VERSION=NodeUtils->runcmd("$::RPMCMD -qp --qf '%{Version}' $rpm1");
		#my $VERSION=`rpm -qp --qf '%{Version}' $rpm1`;
		my ($fn) = $rpm1 =~ m|^.*?([^/]+)$|;  # get rid of the directory, if any
		                                      #print "fn=$fn\n";
		      #my ($ipkg, $VERSION) = $fn =~ /^(.+)-(\d.*)-.*$/;
		my ($ipkg, $VERSION, $RELEASE) = $fn =~ /^(.+)-(\d.*)-(\d.*)\..+\..+$/;

		####jcran changes end

		chomp $VERSION;
		chomp $RELEASE;

		# The basename of the rpm must match the name given in pkgdefs
		if ($rpm_basename eq $ipkg)
		{
			$localversions_hash{$i}{'VERSION'}  = $VERSION;
			$localversions_hash{$i}{'RELEASE'}  = $RELEASE;
			$file_name_hash{$VERSION}{$RELEASE} = $rpm1;
			$i++;
		}
	}
	if ($i == 0)
	{

		#print "BASENAME DOES NOT MATCH NAME IN PKGDEFS FOR $rpm_basename\n";
		MessageUtils->messageFromCat(
									 'csmInstall.cat',       $::MSGMAPPATH,
									 'csminstall',           'E1',
									 'EMsgPREREQ_NOT_FOUND', $rpm_basename,
									 $path
									);
		exit 1;

	}
	my $versionout = "";
	my $releaseout = "";
	$i = 0;
	foreach my $k (keys %localversions_hash)
	{
		if ($i == 0)
		{
			$versionout = $localversions_hash{$k}{'VERSION'};
			$releaseout = $localversions_hash{$k}{'RELEASE'};
		}
		if (
			ArchiveUtils->testVersion(
									  $localversions_hash{$k}{'VERSION'},
									  ">",
									  $versionout,
									  $localversions_hash{$k}{'RELEASE'},
									  $releaseout
									 )
		   )
		{
			$versionout = $localversions_hash{$k}{'VERSION'};
			$releaseout = $localversions_hash{$k}{'RELEASE'};
		}
		$i++;

	}

	return ($file_name_hash{$versionout}{$releaseout});
}

#--------------------------------------------------------------------------------

=head3    getReqsNotSatisfied

        Return the rpms (and versions) that the given rpm requires that
        are not already on the system. This only works for RHEL. For
        SLES it always returns null even if there are rpm dependencies.

        Arguments:
                $pathname
        Returns:
                Posibly empty array of rpm package names.
        Globals:
                none
        Error:
                none
        Example:
                 my @reqs =
                    ArchiveUtils->getReqsNotSatisfied($rpm);

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getReqsNotSatisfied
{
	my ($class, $pathname) = @_;
	my (@requires, @output);
	my $suggest_flag = 0;

	@output = `LANG=C $::RPMCMD -i --test $pathname 2>&1`;
	my $RC = $? >> 8;
	if ($RC == 0) { return @requires; }    # all dependencies were satisfied
	if ($RC > 1)  { return @requires; }
	my $firstline = shift @output;
	if ($firstline =~ /warning:/) { $firstline = shift @output; }
	my $secondline = $output[0];
	if ($firstline  =~ /is already installed/) { return @requires; }
	if ($secondline =~ /conflicts with/)       { return @requires; }
	if (!($firstline =~ /failed dependencies/i)) { return @requires; }

	if ($::PREREQS_ATTR{'DistributionName'} =~ /SLES/)
	{
		foreach my $o (@output)
		{
			my $ver = $o;
                        $ver =~ s/^\s*.*?\s([\d\.]*)\s.*$/$1/;
                        $o =~ s/^\s*(.*?)\s.*$/$1/;
                        chomp($o);
                        
			if ($o)
			{
				if ($ver)
				{
					$o .="-$ver";
				}
				push(@requires, $o);
			}
		}
	}
	else
	{
		foreach my $o (@output)
		{
			if ($suggest_flag == 0)
			{
				if (!($o =~ /Suggested resolutions:/))
				{
					next;
				}
				else
				{
					$suggest_flag = 1;
					next;
				}
			}
			$o =~ s/\s+//g;

			#take the rpm name behind "/" for RedHat AS3 QU5/QU6
			my @junks = split(/\//,$o);
			$o=pop(@junks);
			chomp($o);
			if ($o)
			{
				push(@requires, $o);
			}
		}
		if ($suggest_flag == 0)
		{
			foreach my $o (@output)
			{
				my $ver = $o;
                        	$ver =~ s/^\s*.*?\s([\d\.]*)\s.*$/$1/;
                            	$o =~ s/^\s*(.*?)\s.*$/$1/;
                   	        chomp($o);
                   	        
				if ($o)
				{
					if ($ver)
					{
						$o .="-$ver";
					}
					push(@requires, $o);
				}
			}
		}
	}
	return @requires;
}

#--------------------------------------------------------------------------------

=head3    getRPMSDir

         Returns the RPMS directory (for SMS) for input params

        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur
                $InstallServiceLevel

        Returns:
                appropriate pathname for parameters
        Globals:
                none
        Error:
                none
        Example:
                 my $SMS_RPMS =
                    ArchiveUtils->getRPMSDir($OS, $dist, $distver, $arch, $sl);

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getRPMSDir
{
	my ($class, $os, $distributor, $version, $arch, $sl) = @_;
    
	if ($arch =~ /i.86/) { $arch = "i386"; }
	my $base_install = $::CSMINSTDIR;
	my ($effective_distro_name) = NodeUtils->getEffectiveDistro($distributor);
	if (!$sl)
	{
		return "$base_install/$os/$distributor/$version/$arch/RPMS";
	}
	else
	{
		if ($effective_distro_name =~ /SLES/)
		{
			return "$base_install/$os/$distributor/$version/$arch/$sl/RPMS";
		}
		else
		{
			my $dir;
			if ($sl eq "GA")
			{
				$dir = $distributor . $version . "-" . $arch;
			}
			else
			{
				$dir = $distributor . $version . "-" . $sl;
			}
			return "$base_install/$os/$distributor/$version/$arch/$dir/RPMS";
		}
	}
}

#--------------------------------------------------------------------------------

=head3    getRPMVersionRelease

         Get RPM version and release if aviable.

        Arguments:
                $rpmPackageName
                $productFlag

        Returns:
                Rpm version and release as an array of two strings.
        Globals:
                none
        Error:
                E1 from csmInstallCat
        Example:
                (@temp_array) =
                   ArchiveUtils->getRPMVersionRelease($conserrpmfile, 1);

        Comments:
                $productFlag values are:
                   0 = already installed rpm or default
                   1 = product rpm

=cut

#--------------------------------------------------------------------------------

sub getRPMVersionRelease
{
	my (
		$cmd,   $version,  $release,      $my_str, $my_cmd,
		$class, $RPM_NAME, $product_flag, @my_array
	   );
	@my_array     = ();
	$product_flag = 0;
	($class, $RPM_NAME, $product_flag) = @_;
	if ($product_flag == 0)
	{
		$my_cmd = "$::RPMCMD -q --qf '%{Version}:|:%{Release}' $RPM_NAME";
		$my_str = NodeUtils->runcmd($my_cmd, -1);
	}
	else
	{
		$my_cmd = "$::RPMCMD -qp --qf '%{Version}:|:%{Release}' $RPM_NAME";
		$my_str = NodeUtils->runcmd($my_cmd, -1);
	}
	if ($::RUNCMD_RC != 0)
	{
		MessageUtils->messageFromCat(
									 'csmInstall.cat', $::MSGMAPPATH,
									 'csminstall',     'E1',
									 'EMsgCANT_RUN',   $my_cmd,
									 $::RUNCMD_RC
									);
	}
	($version, $release) = split(/:\|:/, $my_str);
	push(@my_array, $version);
	push(@my_array, $release);
	return (@my_array);
}

#--------------------------------------------------------------------------------

=head3  mount_iso_image

	Tries to mount a file parameter as an ISO image file - a CD image as a
	file. The file is mounted to a predetermined mount point "/tmp_loop_mnt".

        Arguments:
                ISO image file including path from root.
        Returns:
                $::OK if mount worked
		$::NOK if not
        Globals:
                none
        Error:
		none
        Example:
                 my $rc = ArchiveUtils->mount_iso_image($file};
        Comments:
		$file parameter must include the path from root.
                

=cut

#--------------------------------------------------------------------------------

sub mount_iso_image
{
	my ($class, $file) = @_;

	# uses predefined mount point
	#my $mnt_point = "/tmp_loop_mnt";
	my $mnt_point = $::MNTCDROM;

	# futz with the mount point

	if (!(stat($mnt_point))[2])
	{

		# need to make the mount point
		my $cmd = "$::MKDIR -m 777 -p $mnt_point";
		NodeUtils->runcmd($cmd, -1);    # no exit on error
		my $rc = $::RUNCMD_RC;          # no exit on error
		if (!($rc == 0)) { return 0; }
	}

	my $cmd = "$::MOUNT -o loop $file $mnt_point";
	NodeUtils->runcmd($cmd, -1);        # no exit on error

	my $rc = $::RUNCMD_RC;

	if ($rc == 0) { return $mnt_point; }

	# umount in case the mount op worked, but cmd failed for some reason.
	$cmd = "$::UMOUNT $mnt_point";
	NodeUtils->runcmd($cmd, -1);        # no exit on error

	return 0;
}

#--------------------------------------------------------------------------------

=head3    parseRequire

        Parse a single require string, distinguishing between rpms
        and individual libraries of executables.

        Arguments:
                $requiredRpmString
        Returns:
                ($package, $operator, $version);
        Globals:
                none
        Error:
                none

        Example:
                my ($pkg, $operator, $version) =
                        ArchiveUtils->parseRequire($requiredRpmString);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub parseRequire
{
	my ($class, $line) = @_;
	$line =~ s/ is needed by .*$//;
	my ($pkg, $operator, $version, $therest) = split(' ', $line);
	if ($pkg =~ /^rpmlib/ || $pkg =~ m|/| || $pkg =~ /\.so/) { return undef; }
	else { return ($pkg, $operator, $version); }
}

#-------------------------------------------------------------------------------

=head3 get_pkgpath

	Return the array containing the PKGPATH.  Depending on the situation,
	PKGPATH may contain just $::PKGPATH, or may have /reqs and 
	/reqs/<distro> added to each path element.

        Arguments:
		$append_reqs
                 1 (Linux): Append /reqs and /reqs/<distro> to each path element
                 1 (AIX): Append /reqs to each path element.
                 0 (Linux or AIX): Do not alter PKGPATH.
        Returns:
                @pkgpath : array containing path elements to search
        Globals:
                %::PREREQS_ATTR
                $::ATTRS
                @::PKGPATH
        Error:
                none
        Example:
                ArchiveUtils->get_pkgpath();
                ArchiveUtils->get_pkgpath(1);
        Comments:
                none

=cut

#-------------------------------------------------------------------------------
sub get_pkgpath()
{
	my ($class, $append_reqs) = @_;
	my @pkgpath;

	if ($append_reqs)
	{
		if ($::PREREQS_ATTR{'OSName'} eq "AIX")
		{

			# Append /reqs to each PKGPATH element to search for pkgs
			@pkgpath = @::PKGPATH;
			map { s/$/\/reqs/ } @pkgpath;
		}
		else    # Linux
		{

			# Append /reqs and /reqs/<distro> to each PKGPATH element
			# to search for pkgs
			my $distroname = (
							    $::PREREQS_ATTR{'DistributionName'}
							  ? $::PREREQS_ATTR{'DistributionName'}
							  : $::ATTRS{'InstallDistributionName'}
							 );
			my $distroversion = (
								   $::PREREQS_ATTR{'DistributionVersion'}
								 ? $::PREREQS_ATTR{'DistributionVersion'}
								 : $::ATTRS{'InstallDistributionVersion'}
								);
			my $distro = $distroname . $distroversion;

			# Assemble the pkgpath as follows:
			#  path1/reqs/distro:path1/reqs:path1:path2/reqs/distro/path2/reqs:path2
			foreach my $p (@::PKGPATH)
			{
				push(@pkgpath, "$p/reqs/$distro");
				push(@pkgpath, "$p/reqs");
				push(@pkgpath, "$p");
			}
		}
	}

	# Do not append the reqs or reqs/<distro>
	else
	{
		@pkgpath = @::PKGPATH;
	}

	return @pkgpath;
}

#-------------------------------------------------------------------------------

=head3 search_rpms_Linux

	Search for RPMs in the PKGPATH and populate the 
        %::Rpm_name_path and %::temp_CmdLinePathVersionName hashes.
	The %::temp_CmdLinePathVersionName hash is used later in
	create_rpm_list_to_copy.

        If an rpm is not found, add it to the %::RpmsNotFoundInCmdPath hash.

        Arguments:
		$which_rpms:
			"CSM"      - Search for csm and rsct packages
	    			   - Before calling search_rpms_Linux with this
				     argument, the following arrays must be 
				     populated: @::csm_packages, 
				     @::rsct_packages

			"OPENSRC"  - Search for open source prereqs
	    			   - Before calling search_rpms_Linux with this
				     argument, the following arrays must be 
				     populated: @::opensrc_rpm_prereqs

			"OPTIONAL" - Search for optional packages
	    			   - Before calling search_rpms_Linux with this
				     argument, the following arrays must be 
				     populated: @::optional_copy_packages

        Returns:
                undefined
        Globals:
                @::csm_packages
                @::rsct_packages
                @::opensrc_rpm_prereqs
                @::optional_packages
                $::Rpm_name_path
                %::RpmsNotFoundInCmdPath
                $::PREREQS_ATTR (used by get_pkgpath)
                $::ATTRS (used by get_pkgpath)
                @::PKGPATH (used by get_pkgpath)
                $::INSTALLDIR_CSMPKG
                $::COPY_OPT_FLAG
		@::mgmtserver_open_srcrpms
		@::isvr_open_srcrpms
        Error:
                message E1 (cannot find at least one non-optional rpm)
        Example:
                ServerUtils->search_rpms_Linux("CSM");
                ServerUtils->search_rpms_Linux("OPENSRC");
                ServerUtils->search_rpms_Linux("OPTIONAL");

        Comments:
                none

=cut

#-------------------------------------------------------------------------------
sub search_rpms_Linux
{
	my ($class, $which_rpms) = @_;

	my @rpmlist;
	my $append_reqs_to_pkgpath = 0;
	my $targets;

	# Create the rpmlist based on the $which_rpms parameter.
	if ($which_rpms =~ /CSM/)
	{
		@rpmlist                = (@::csm_packages, @::rsct_packages);
		$append_reqs_to_pkgpath = 0;
		$targets                = "CSM and RSCT";
	}
	elsif ($which_rpms =~ /OPENSRC/)
	{
		@rpmlist                = (@::opensrc_rpm_prereqs);
		$append_reqs_to_pkgpath = 1;
		$targets                = "Open Source";
	}
	elsif ($which_rpms =~ /OPTIONAL/)
	{
		@rpmlist                = (@::optional_copy_packages);
		$append_reqs_to_pkgpath = 1;
		$targets                = "Optional";
	}
	else
	{
		@rpmlist = ();
	}

	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
								 'I', 'IMsgSearchingfor', $targets);

	my ($rpmfile, $rpm, $path);

	# Get the list of rpm architectures to search for based on target
	# machine's architecture.
	my @archs = ArchiveUtils->get_archs_to_search();

	my @pkgpath = ArchiveUtils->get_pkgpath($append_reqs_to_pkgpath);

	foreach $rpm (@rpmlist)
	{
		$rpmfile = '';
		$path    = '';

		# First search for RPM-* for all architectures
		foreach my $a (@archs)
		{
			if (!($path))
			{
				$rpmfile = "${rpm}-*.$a.rpm";
				$path = ArchiveUtils->findRpmInPath($rpmfile, @pkgpath);
			}
			else { last; }
		}

		# If not found, search for RPM* for all architectures (no dash)
		if (!($path))
		{

			# we take out the dash because the images, may not have them.
			foreach my $a (@archs)
			{
				if (!($path))
				{
					$rpmfile = "${rpm}*.$a.rpm";
					$path = ArchiveUtils->findRpmInPath($rpmfile, @pkgpath);
				}
				if ($path) { last; }
			}
		}

		if (defined($path))
		{

			# Now that the rpm pattern was found in the path, get the
			# specific RPM filename.  If there is more than one RPM that
			# matches the pattern, take the most recent one (based on version)
			$rpmfile = ArchiveUtils->getRecentVersionRpm($path, $rpmfile);

			chomp $rpmfile;
			my $rpmfile_basename;
			($rpmfile_basename = $rpmfile) =~ s:^.*/::g;

			#find the shortname
			my $shortname = ArchiveUtils->getShortName( $rpmfile_basename);

			# Add the rpm file information into %::temp_CmdLinePathVersionName
			# for later use by ArchiveUtils->create_rpm_list_to_copy().
			push(@{$::temp_CmdLinePathVersionName{$path}}, $rpmfile_basename);
			$::Rpm_name_path{$shortname}{'BaseName'}    = $rpmfile_basename;
			$::Rpm_name_path{$shortname}{'Location'}    = $path;
			$::Rpm_name_path{$shortname}{'Destination'} = $::INSTALLDIR_CSMPKG;
		}

		# The RPM pattern was not found in the path.
		else
		{

			#get the shortname of the rpm
			my $shortname = ArchiveUtils->getShortName( $rpm);
			my $tmp_var1;
			my $my_rindex = rindex($shortname, "*");
			if ($my_rindex != -1)
			{
				$tmp_var1 = substr($shortname, 0, rindex($shortname, "*"));
			}
			else { $tmp_var1 = $shortname; }
			$my_rindex = rindex($tmp_var1, "-");
			if (($my_rindex + 1) == 0) { $shortname = $tmp_var1; }
			elsif (($my_rindex + 1) == length($tmp_var1))
			{
				$shortname = substr($tmp_var1, 0, rindex($tmp_var1, "*"));
			}
			else { $shortname = $tmp_var1; }

			# Ignore packages if they were not found but have already
			# been installed.  Add them to @::RpmsNotBeCopied, which gets
			# used later in install_rpm_Linux.
			# Check only for open source rpms that are supposed to be
			# installed on the management server or install server.
			if (
				grep (/^$rpm$/,
					  (@::mgmtserver_open_srcrpms, @::isvr_open_srcrpms)))
			{
				my $tmp_rpm = $rpm;
				$tmp_rpm =~ s/\*//;
				my $cmd = "$::RPMCMD -q $shortname|$::GREP $tmp_rpm";

				NodeUtils->runcmd("$cmd", -1);
				if ($::RUNCMD_RC == 0)
				{
					push(@::RpmsNotBeCopied, $rpm);
					next;
				}
			}

			push(@::RpmsNotFoundInCmdPath, $shortname);
		}
	}

	# If the -c flag was provided, all the non-optional RPMs must be copied,
	# even if they have already been copied before.
	# Display messages for any missing RPMs.  Exit with an error unless all
	# missing RPMs are on the optional list.
	if ($::COPY_OPT_FLAG)
	{
		if ($which_rpms =~ /OPTIONAL/)
		{

			# If this subroutine was called for the OPTIONAL rpms, then
			# it is OK if some of them do not exist.  Just print a warning
			# message for the missing ones.
			my @rpms_ignored = @::RpmsNotFoundInCmdPath;
			@::RpmsNotFoundInCmdPath = ();

			# Print info message about ignored RPMs.
			if (scalar(@rpms_ignored) > 0)
			{
				MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
								   'csminstall', 'I', 'IMsgMissingOptionalRPMs',
								   join(", ", @rpms_ignored));
			}
		}
		else
		{

			# If this subroutine was not called for the OPTIONAL rpms, then
			# display an error message for anything that does not exist.

			# Exit with error and message listing missing RPMs.
			if (scalar(@::RpmsNotFoundInCmdPath) > 0)
			{
				MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
										  'csminstall', 'E1', 'EMsgNoRpmFound',
										  join(", ", @::RpmsNotFoundInCmdPath));
			}
		}
	}
}

#-------------------------------------------------------------------------------

=head3 search_copy_rpms_AIX

	Search for RPMs in the PKGPATH and copy them to 
	/csminstall/Linux/<distro>/csm/packages

        Arguments:
		$which_rpms:
			"CSM"      - Does nothing, CSM rpms not copied for AIX
			"OPENSRC"  - Search for open source prereqs
			"OPTIONAL" - Search for optional packages
        Returns:
                undefined
        Globals:
	        $::INSTALLDIR_CSMRPM
                @::opensrc_rpm_prereqs
                @::optional_packages
                @::PKGPATH
        Error:
                message E1 (cannot find at least one non-optional rpm)
        Example:
                ServerUtils->search_rpms_AIX("OPENSRC");
                ServerUtils->search_rpms_AIX("OPTIONAL");

        Comments:
                none

=cut

#-------------------------------------------------------------------------------
sub search_rpms_AIX
{
	my ($class, $which_rpms) = @_;

	my @rpmlist;
	my $append_reqs_to_pkgpath = 0;

	if ($which_rpms =~ /CSM/)
	{

		# CSM for AIX does not come in RPM format.
		@rpmlist                = ();
		$append_reqs_to_pkgpath = 0;
		return 0;
	}
	elsif ($which_rpms =~ /OPENSRC/)
	{
		@rpmlist                = (@::opensrc_rpm_prereqs);
		$append_reqs_to_pkgpath = 1;
	}
	elsif ($which_rpms =~ /OPTIONAL/)
	{
		@rpmlist                = (@::optional_packages);
		$append_reqs_to_pkgpath = 1;
	}
	else
	{
		@rpmlist = ();
	}

	my ($rpmfile, $rpm, $path);

	my @archs = ('ppc', 'noarch');

	my @pkgpath = ArchiveUtils->get_pkgpath($append_reqs_to_pkgpath);

	# Search open source prereqs from CSM distribution CD-ROM or
	# directory structure
	foreach my $rpm (@rpmlist)
	{
		$rpmfile = '';
		$path    = '';

		# First search for RPM-* for all architectures
		foreach my $a (@archs)
		{
			if (!($path))
			{
				$rpmfile = "${rpm}-*.$a.rpm";
				$path = ArchiveUtils->findRpmInPath($rpmfile, @pkgpath);
			}
			else { last; }
		}

		# If not found, search for RPM* for all architectures (no dash)
		if (!($path))
		{

			# we take out the dash because the images, may not have them.
			foreach my $a (@archs)
			{
				if (!($path))
				{
					$rpmfile = "${rpm}*.$a.rpm";
					$path = ArchiveUtils->findRpmInPath($rpmfile, @pkgpath);
				}
				if ($path) { last; }
			}
		}

		# Copy the RPM if it was found, otherwise exit.
		if (defined($path))
		{

			# Now that the rpm pattern was found in the path, get the
			# specific RPM filename.  If there is more than one RPM that
			# matches the pattern, take the most recent one (based on version).
			$rpmfile = ArchiveUtils->getRecentVersionRpm($path, $rpmfile);

			chomp $rpmfile;
			my $rpmfile_basename;
			($rpmfile_basename = $rpmfile) =~ s:^.*/::g;
			my @files = ("$path/$rpmfile_basename");
			NodesetUtils->copyfile(\@files, $::INSTALLDIR_CSMRPM);
		}

		# The RPM pattern was not found in the path, so exit.
		else
		{
			MessageUtils->message(         'E1', 'EMsgCANT_FIND',
								  $rpm, join(':', @::PKGPATH));
		}

	}
}

#--------------------------------------------------------------------------------

=head3    testVersion

        Compare version1 and version2 according to the operator and
        return True or False.

        Arguments:
                $version1
                $operator
                $version2
                $release1
                $release2
        Returns:
                True or False
        Globals:
                none
        Error:
                messageFromCat E11

        Example:
                if (ArchiveUtils->testVersion ( $ins_ver,
                                                "<",
                                                $req_ver,
                                                $ins_rel,
                                                $req_rel)){ blah; }

        Comments:
                The return value is generated with the Require query of the
                rpm command.

=cut

#--------------------------------------------------------------------------------

sub testVersion
{
	my ($class, $version1, $operator, $version2, $release1, $release2) = @_;

	# Note: perl seems to be able to handle really big integers, but if we run into a limit, we
	#       can use the Math::BigInt pkg

	my @a1 = split(/\./, $version1);
	my @a2 = split(/\./, $version2);
	my $len = (scalar(@a1) > scalar(@a2) ? scalar(@a1) : scalar(@a2));
	$#a1 = $len - 1;  # make the arrays the same length before appending release
	$#a2 = $len - 1;
	push @a1, split(/\./, $release1);
	push @a2, split(/\./, $release2);
	$len = (scalar(@a1) > scalar(@a2) ? scalar(@a1) : scalar(@a2));
	my $num1 = '';
	my $num2 = '';

	for (my $i = 0 ; $i < $len ; $i++)
	{
		my ($d1) = $a1[$i] =~ /^(\d*)/;    # remove any non-numbers on the end
		my ($d2) = $a2[$i] =~ /^(\d*)/;

		my $diff = length($d1) - length($d2);
		if ($diff > 0)                     # pad d2
		{
			$num1 .= $d1;
			$num2 .= ('0' x $diff) . $d2;
		}
		elsif ($diff < 0)                  # pad d1
		{
			$num1 .= ('0' x abs($diff)) . $d1;
			$num2 .= $d2;
		}
		else                               # they are the same length
		{
			$num1 .= $d1;
			$num2 .= $d2;
		}
	}

	# Remove the leading 0s or perl will interpret the numbers as octal
	$num1 =~ s/^0+//;
	$num2 =~ s/^0+//;

	#SLES Changes
	# if $num1="", the "eval '$num1 $operator $num2'" will fail. So MUSTBE be sure that $num1 is not a "".
	if (length($num1) == 0) { $num1 = 0; }
	if (length($num2) == 0) { $num2 = 0; }

	#End of SLES Changes

	if (length($release1))
	{
		$release1 = "-$release1";
	}    # this is just for error msgs
	if (length($release2))
	{
		$release2 = "-$release2";
	}    # this is just for error msgs

	if ($operator eq '=') { $operator = '=='; }
	my $bool = eval "$num1 $operator $num2";
	if (length($@))
	{
		MessageUtils->messageFromCat(
							  'csmInstall.cat',
							  $::MSGMAPPATH,
							  'csminstall',
							  'W',
							  'EMsgVERSION_COMPARE_PROBLEM',
							  "$version1$release1 $operator $version2$release2",
							  $@
		);
		return undef;
	}
	if ($ENV{LANG} =~ /^(C|POSIX|en_US)/)
	{    #this is a work around for NLS problems
		MessageUtils->messageFromCat(
							  'csmInstall.cat',
							  $::MSGMAPPATH,
							  'csminstall',
							  'V',
							  'IMsgCOMPARING_VERSION',
							  "$version1$release1 $operator $version2$release2",
							  ($bool ? 'true' : 'false')
		);
	}
	return $bool;
}

#--------------------------------------------------------------------------------

=head3  unmount_iso_image

	Tries to unmount a file as an ISO image (a CD image as a file).

        Arguments:
                mount point (usually "/tmp_loop_mnt")
        Returns:
                $::OK if umount worked
		$::NOK if not
        Globals:
                none
        Error:
		none
        Example:
                 my $rc = ArchiveUtils->unmount_iso_image($mount_pt};
        Comments:
		none

=cut

#--------------------------------------------------------------------------------

sub unmount_iso_image
{
	my ($class, $mount_point) = @_;

	my $cmd = "$::UMOUNT $mount_point";

	NodeUtils->runcmd($cmd, -1);    # no exit on error
	my $rc = $::RUNCMD_RC;

	if ($rc == 0) { return $::OK; }

	return $::NOK;
}

#--------------------------------------------------------------------------------

=head2    installp Support

=cut

#--------------------------------------------------------------------------------

=head3    get_pkgVersion

        AIX only routine.

        Returns the installp image version from Installp command which,
        looks into the .toc file.

        Arguments:
                $installpPackage

        Returns:
                AIX installp image version as a string
        Globals:
                $::PLTFRM
                $::INSTALLPCMD

        Error:
                messageFromCat E2
        Example:
                not used outside ServerUtils

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub get_pkgVersion
{
	my ($class, $pkg) = @_;
	my ($cmd, $output, $version);

	if ($::PLTFRM eq "Linux")
	{
		$version = "";
		return (length($version) ? $version : undef);
	}
	elsif ($::PLTFRM eq "AIX")
	{
		$cmd    = qq($::INSTALLPCMD -L -d . | grep $pkg 2>&1);
		$output = NodeUtils->runcmd($cmd);
		chomp($output);
		(undef, undef, $version, undef) = split(/:/, $output);
		return (length($version) ? $version : undef);
	}
	else
	{
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
									 'csminstall', 'E2', 'EMsgINVALID_OSTYPE');
	}

}

#--------------------------------------------------------------------------------

=head3   getShortName 

        Get the short name and base name of an rpm file

        Arguments:
                $rpm_name


        Returns:
                If not want array, return short name of a rpm package. 
                If want array, return short name and base name. Here the 
                        short name is different than the one above.
				        This short name include a part of version number. 
                        Here the base name is same as the short name above.

        Globals:
                N/A
        Error:
                0

        Example:
                 my $shortname =
                        ArchiveUtils->getShortName($rpmFileName);
                 my ($shortname, basename) = 
                        ArchiveUtils->getShortName($rpmFileName);

        Comments:
				N/A
=cut

#--------------------------------------------------------------------------------
sub getShortName
{
	my ($class, $rpm_name) = @_;
	
	my @parts = split /-/, $rpm_name;
	my $shortname = shift @parts;
	my $p;
  SNAME: while ($p = shift @parts)
	{
		if (($p !~ /^\d+\.[\d\.]*/) && ($p !~ /^\d+\*$/))
		{    #if it doesn't start with a digit
			$shortname .= "-$p";
		}
		else { unshift @parts, $p; last SNAME; }
	}
	if ( $rpm_name =~ /^termcap-32bit/)
	{
		$shortname = "termcap-32bit";
	}
	
	if ( !wantarray)
	{
		return $shortname;
	}
	my $basename = $shortname;
	if ($basename =~ /rsct.64bit/)
	{
		$basename = $shortname = "rsct.64bit";
	}
	$shortname .= "-" . $parts[0] . "-" . (split /\./, $parts[1])[0];

	return ( $shortname, $basename);
}

#--------------------------------------------------------------------------------

=head3   promptInsertCDs 

        Prompt message to users to insert disk into device drive 

        Arguments:
				$diskname - distribution name
				$disktype - disk type, CD/DVD 
				$disknum - disk number


        Returns:
				N/A

        Globals:
                N/A

        Example:
               ArchiveUtils->promptInsertCDs($diskname, $disktype, $disknum);

        Comments:
				N/A
=cut

#--------------------------------------------------------------------------------
sub promptInsertCDs()
{
	my ($class, $diskname, $disktype, $disknum) = @_;

	$disktype = $disktype || "CD";

	my $junk;
	if (($::command_line =~ /installms/) || ($::command_line =~ /copycsmpkgs/))
	{
		MessageUtils->messageFromCat(
									'csmInstall.cat', $::MSGMAPPATH,
								 	'csminstall', 'P', 
									'IMsgPRESS_ENTER_OR_QUIT', $diskname, 
									$disktype, $disknum
									);
	 }
	else
	{
		# Prompt insert disk
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
				 'csminstall', 'I', 'IMsgINSERT_DISK', $diskname, $disktype, $disknum);
		MessageUtils->messageFromCat(
									'csmInstall.cat', $::MSGMAPPATH,
									'csminstall', 'P', 
									'IMsgPRESS_ENTER'
									);
	}

	system "stty", '-icanon', 'eol', "\001";
	while (1) 
	{
		$junk = getc;
		#read the disk content
		if ($junk eq "\n")
		{
			return;
		}
		if (($::command_line =~ /copycsmpkgs/) 
		|| ($::command_line =~ /installms/))
		{
			#quit
			if (($junk eq "q") || ($junk eq "Q")) 
			{
				print "\n";
				system "stty", 'icanon', 'eol', '^@';
				if ($::command_line =~ /installms/)
				{
					MessageUtils->messageFromCat(
									'csmInstall.cat', $::MSGMAPPATH,
                      				'csminstall', 'P',
                      				'IMsgInstallmsQuit' 
                      				)
				}
				else
				{
                   	MessageUtils->messageFromCat(
               				'csmInstall.cat', $::MSGMAPPATH,
               				'csminstall', 'P',
               				'IMsgCopycsmpkgsQuit' 
               				)
				}
				exit 1;
			} # end if ($junk eq "q...
		} # end if (($::command_line =
	} # end while (1)
	system "stty", 'icanon', 'eol', '^@';
}

#--------------------------------------------------------------------------------

=head3   forceUseDisk 

        Prompt message to users, about whether force the use of a disk in the device drive.

        Arguments:
				$diskname - distribution name
				$disktype - disk type, CD/DVD 
				$disknum - disk number


        Returns:
				1 - force use the disk
				0 - do not force use the disk

        Globals:
                N/A

        Example:
               ArchiveUtils->forceUseDisk($diskname, $disktype, $disknum);

        Comments:
				N/A
=cut

#--------------------------------------------------------------------------------
sub forceUseDisk()
{
	my ($class, $diskname, $disktype, $disknum) = @_;

	$disktype = $disktype || "CD";
	my $junk;
	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
		  'csminstall', 'W', 'EMsgWRONG_DISK', $diskname, $disktype, $disknum);
	if (($::command_line =~ /copycsmpkgs/)
		|| ($::command_line =~ /installms/))
	{
		my $progname = NodeUtils->programName();
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
						 'csminstall', 'I', 'IMsgPRESS_ENTER_OR_FORCE_OR_QUIT',
						$progname
						);
	}
	else
	{
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
						 'csminstall', 'I', 'IMsgPRESS_ENTER_OR_FORCE'
						);
	}

	# Get user input

	system "stty", '-icanon', 'eol', "\001";
	while (1)
	{
		$junk = getc;
		if ($junk eq "F" or $junk eq "f")
		{
			return 1;
		}
		if (($::command_line =~ /copycsmpkgs/)
       		|| ($::command_line =~ /installms/))
		{
			if (($junk eq "q") || ($junk eq "Q")) 
			{
				print "\n";
				system "stty", 'icanon', 'eol', '^@';
				if ($::command_line =~ /installms/)
				{
               		MessageUtils->messageFromCat(
						'csmInstall.cat', $::MSGMAPPATH,
   						'csminstall', 'P',
   						'IMsgInstallmsQuit' 
   						)
				}
				else
				{
               		MessageUtils->messageFromCat(
   						'csmInstall.cat', $::MSGMAPPATH,
   						'csminstall', 'P',
   						'IMsgCopycsmpkgsQuit' 
   						)
				}
				exit 1;
			}
		}
		if ($junk eq "\n")
		{
			return 0;
		}
		
	}
	system "stty", 'icanon', 'eol', '^@';
	print "\n";

}

#-------------------------------------------------------------------------------

=head3   MountMedia 

	Mount cdrom or iso image
	Support both AIX and Linux
	Used by installms/copycsmpkgs or copycds/setupks/setupyast command

	Arguments:
	$diskref - The reference hash for current disk
	$diskname - The reference hash for all disks
	$images - the ISO images hash
	$devarray_ref - the device drive list array
	$nomountcd - not mount cd flag

	Returns:
	$realdisk_ref - the real disk reference
	$ismounted - whether mounted flag, 0 - no mounted, 1 - mounted
	$target - mount target    

	Globals used:
	
=cut

#-------------------------------------------------------------------------------
sub MountMedia()
{	
	my ($class, $diskref, $disks_ref, $images, $devarray_ref, $nomountcd) = @_;
	my @disks = @$disks_ref;
	my @devarray = @$devarray_ref;
	my ($ismounted, $target);
	
	my $diskname = $diskref->{name};
	my $disknum  = $diskref->{num};
	my $disktype = $diskref->{type} || "CD";
	
	my $realdisk_ref = $diskref;
	if (!defined $images->{$diskname}->{$disktype}->{$disknum})
	{
		if ($nomountcd)
		{
			$::GLOBAL_EXIT = 1;
			exit 1;	
		}	
		my @diskWithSameNum = ArchiveUtils->getDisksWithSameNum(\@disks, $disknum);
		my @typearray;
		foreach my $ref (@diskWithSameNum)
		{
			my $disktype = $ref->{type} || "CD";
			# Avoid duplicate disktype, this might happen for SLESx SPx
			if (!grep /^$disktype/, @typearray)
			{
				push @typearray, $disktype;
			}
		}
		my $type = $::MEDIATYPE || join("/", @typearray);
	
		while (1)
		{
			ArchiveUtils->promptInsertCDs($diskWithSameNum[0]->{name}, $type, $diskWithSameNum[0]->{num});
			
			# Device drive alreay determined, use it
			if ($::DEVICEDRV)
			{
				@devarray = ("$::DEVICEDRV");
			}

			MOUNT_ALL_DEV:
			my $diskfound = 0;
			my $force_use_disk = 0;
			my $i = 0;
			my $dev_with_disk;

			# no device in the devarray at all,
			# need users to mount the device manually
			if (scalar(@devarray) == 0)
			{
				$target = $::MNTCDROM;
				opendir DIR, $target;
				my @content = readdir(DIR);
				closedir(DIR);

				# No files in target directory 
				# No disk is inserted into this device
				if (scalar @content <= 2)
				{
					# This is the last dev
					# no disk inserted at all
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,'csminstall', 'W','IMsgNO_DISK_INSERTED',$diskname, $type, $disknum);
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,'csminstall', 'I','IMsgPRESS_ENTER_OR_EXIT');
					system "stty", '-icanon', 'eol', "\001";
					my $junk = getc;
					if ($junk eq "\n")
					{
						system "stty", 'icanon', 'eol', '^@';
						goto MOUNT_ALL_DEV;
					}
					if (($junk eq "q") || ($junk eq "Q"))
					{
						system "stty", 'icanon', 'eol', '^@';
						print "\n";
						exit;
					}
				}
				foreach my $ref (@diskWithSameNum)
				{
					my $progref = $ref->{diskid_searchprog};
					my $progarg = $ref->{diskid_searchargs};
					if (&$progref($target, $progarg))
					{
						# the disk reference should be updated
						$realdisk_ref = $ref;
						
						$::MEDIATYPE = $ref->{type} || "CD";
						$diskfound = 1;
						$ismounted = 1;
						last;
					}
				}
				if ($diskfound)
				{
					last;
				}
				else
				{
					if (ArchiveUtils->forceUseDisk($diskWithSameNum[0]->{name}, $type, $diskWithSameNum[0]->{num}))
					{
						$::MEDIATYPE = "CD";
						$ismounted = 1;
						$force_use_disk = 1;
						last;
					}
				}
			}
			if ($diskfound || $force_use_disk)
			{
				last;
			}

			foreach my $dev (@devarray)
			{
				$i++;
				($ismounted, $target) =
  					ArchiveUtils->MountDevice($dev, $nomountcd);
				opendir DIR, $target;
				my @content = readdir(DIR);
				closedir(DIR);

				# No files in target directory 
				# No disk is inserted into this device
				if (scalar @content <= 2)
				{
					# This is the last dev
					if ($i eq scalar(@devarray))
					{
						if (!$dev_with_disk)
						{
							# no disk inserted at all
							MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,'csminstall', 'W','IMsgNO_DISK_INSERTED',$diskname, $type, $disknum);
							MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,'csminstall', 'I','IMsgPRESS_ENTER_OR_EXIT');
							system "stty", '-icanon', 'eol', "\001";
							my $junk = getc;
							if ($junk eq "\n")
							{
								system "stty", 'icanon', 'eol', '^@';
								goto MOUNT_ALL_DEV;
							}
							if (($junk eq "q") || ($junk eq "Q"))
							{
								system "stty", 'icanon', 'eol', '^@';
								print "\n";
								exit;
							}

						}
						else
						{
							# the last device is empty,
							# but the previous one is not empty
							`$::UNMOUNT $target 2>/dev/null 1>/dev/null`;
							($ismounted, $target) =
  								ArchiveUtils->MountDevice($dev_with_disk, $nomountcd);
						}
					}
					else
					{
						# not the last device, go to next one
						next;
					}
				}
				else
				{
					# can be used if the last dev is empty
					$dev_with_disk = $dev;
				}

				foreach my $ref (@diskWithSameNum)
				{
					my $progref = $ref->{diskid_searchprog};
					my $progarg = $ref->{diskid_searchargs};
					if (&$progref($target, $progarg))
					{
						# the disk reference should be updated
						$realdisk_ref = $ref;
						
						$::MEDIATYPE = $ref->{type} || "CD";
						$::DEVICEDRV = $dev;
						$diskfound = 1;
						($::DEBUG) && print "Found $::MEDIATYPE disk number 1 in $dev\n";
						last;
					}
				}
				if ($diskfound)
				{
					last;
				}
				else
				{
					# This is the last device, force use it?
					if ($i eq scalar(@devarray))
					{
						if (ArchiveUtils->forceUseDisk($diskWithSameNum[0]->{name}, $type, $diskWithSameNum[0]->{num}))
						{
							# default the $::MEDIATYPE to "CD"
							$::MEDIATYPE = "CD";
							$::DEVICEDRV = $dev;
							$force_use_disk = 1;
							last;
						}
						else
						{
						        # Umount it because we need to free the last dev
						        `$::UNMOUNT $target 2>/dev/null 1>/dev/null`;
							last;
						}
					}
					else
					{
						# Umount it because it is not a correct disk
						`$::UNMOUNT $target 2>/dev/null 1>/dev/null`;
					}
				}
			}
			if ($diskfound || $force_use_disk)
			{
				last;
			}
		}
	}
	else
	{
		my $path = $images->{$diskname}->{$disktype}->{$disknum};
		if (-f $path && $path =~ /iso$/i)
		{
			$ismounted = 1;
			$target = ArchiveUtils->mount_iso_image($path);
			if (!$target)
			{
				($::DEBUG) && print "Mount ISO $path failed, exiting...\n";
				exit 1;
			}
		}
		else
		{
			# the path is a directory instead of ISO file,
			# copy the content of this directory
			$ismounted = 0;
			$target = $path;
		}
	}
	
	return ($realdisk_ref, $ismounted, $target);

}
#-------------------------------------------------------------------------------

=head3   MountDevice 

	Mount cdrom or DVD rom
	Support both AIX and Linux
	Used by installms/copycsmpkgs or copycds/setupks/setupyast/csmsetupinstall command

	Arguments:
	$device - device name
	$nomountcd - no mount cd flag

	Returns:
	$ismounted - whether mount succeed, 1 - succeed, 0 - Failed    

	Globals used:
	$::MNTCDROM	
=cut

#-------------------------------------------------------------------------------
sub MountDevice
{
	my ($class, $dev, $nomountcd) = @_;
	my ($wrong_disk, $force_use_disk, $ismounted);

	my $mountpoint = $::MNTCDROM;

	if ($nomountcd)
	{
		$ismounted = 0;
		return ($ismounted, $mountpoint);
	}
		
	# Mount each device
	if (-d $mountpoint)
	{
		if ($::PLTFRM eq "AIX")
		{
			sleep(5); #this is needed for AIX and wont hurt Linux (too much)
		`$::UNMOUNT $mountpoint 2>/dev/null 1>/dev/null`;
		}
		else
		{
			my $cmd = "$::MOUNT | $::GREP \"$mountpoint \"";
			NodeUtils->runcmd("$cmd", -1);
			if (!$::RUNCMD_RC)
			{
				if ($::DEVPATH)
				{
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,'csminstall', 'I','IMsgAlreadyMounted', $mountpoint);
				}
				$ismounted = 1;
				return ($ismounted, $mountpoint);
			}
		}
	}

	use File::Path;    # Provides mkpath()

	if (!(-d $mountpoint))
	{
		mkpath($mountpoint, $::VERBOSE, 0755);
	}
	
	my $cmd;
	if ($::PLTFRM eq "AIX")
	{
		$cmd = "$::MOUNT -rvcdrfs $dev $mountpoint";
	}
	else
	{
		$cmd = "$::MOUNT $dev $mountpoint";
	}

	# Do not exit with errors, we need to use 
	# the diskid_searchprog to verify the disk anyway
	NodeUtils->runcmd($cmd, -1);
	
	$ismounted = 1;
	return ($ismounted, $mountpoint);
}

#-------------------------------------------------------------------------------

=head3   getPkgPath 

	get the pkgpaths from the command line -p argument

	Arguments:
	$paths - the paths specified by command line -p argument

	Returns:
	@pkgpath - the pkgpath array    

	Globals Vars:
	$::DEVPATH - the first block device specified by the command line,
				 which will be used in the subsequent operations	

=cut

#-------------------------------------------------------------------------------
sub getPkgPath()
{
	my ($class, $paths) = @_;

	my @origpath = split(":", $paths);

	my @pkgpath = ();

	foreach my $dir (@origpath)
	{
		if(-b $dir)
		{
			# block device, treat it as a mountable device,
			# and remove it from the list. only the first one
			# in the list will be used
			if(!$::DEVPATH)
			{
				$::DEVPATH = $dir;
			}
		}
		else
		{
			if (! -d $dir)
			{
				MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall', 
											 'I', 'IMsgDIRECTORY_NOT_FOUND', $dir);
				exit 1;
			}
			push @pkgpath, $dir;
		}
	}

	return @pkgpath;
}

################################################################################
#
# determineDeviceList 
#
#  Determine the CD/DVD device list 
#        that can be used to mount disks 
#
#   Arguments:
#		N/A
#	
#   Return:
#       @devarray - Array that contains the device list 
#
#	Global Variable:
#		@::PKGPATH - package paths
#		$::PLTFRM - Platform name
#		$::AIX_MNT_DEV - default device name for AIX platform
#		$::DEVPATH - the device name specified in the command line
#
#   Example:
#		my @devarray = ArchiveUtils->determineDeviceList();
#   
################################################################################
sub determineDeviceList()
{
	my ($class) = @_;

	my @devarray = ();
	
	# devce specified in cmd line first
	if ($::DEVPATH)
	{
		push @devarray, $::DEVPATH;
		goto done;
	}
	
	# Linux: the devices in /etc/fstab
	if ($::PLTFRM eq "Linux")
	{
		my $fstab = "/etc/fstab";
		if (-f $fstab)
		{
			open(FSTAB,$fstab)
			or 
			MessageUtils->messageFromCat(
											'csmInstall.cat', $::MSGMAPPATH,
											'csminstall', 'E1',
											'EMsgCANT_OPEN', $fstab
										);
			my $line;
			while ($line = <FSTAB>)
			{
				chomp $line;
				if (($line !~ /\s*#/)
					&& (($line =~ /\s*(.*?)\s+\/media\/cdrom/)
						|| ($line =~ /\s*(.*?)\s+\/media\/dvd/)))
				{
					my $dev = $1;
					if (!grep(/^$dev$/,@devarray))
					{
						push @devarray, $dev;
					}	
				}
			} # end while
			close(FSTAB);
		} # end if (-f $fstab) 
		if (@devarray)
		{
			goto done;
		}
	} #end if ($::PLTFRM eq "Linux")

	# default device list:
	#   Linux: /dev/cdrom, /dev/dvd
	#	AIX: $::AIX_MNT_DEV or /dev/cd0
	if ($::PLTFRM eq "Linux")
	{
		if (-b "/dev/cdrom")
		{
			push @devarray, "/dev/cdrom";
		}
		if (-b "/dev/dvd")
		{
			push @devarray, "/dev/dvd";
		}
	}
	else
	{
		if ($::AIX_MNT_DEV)
		{
			push @devarray, $::AIX_MNT_DEV;
		}
		else
		{
			push @devarray, "/dev/cd0";
		}
	}

	done:
		return @devarray;
}

################################################################################
#
# isVersionInRange
#
#   judge whether one package version is in one range or not
#
#   Arguments:
#       $ver    target package version
#       $lver   low allowed package version, $ver should not be smaller than it
#       $hver   high range, $ver should not be EQUAL or bigger than it
#
#   Return:
#       1       $ver is allowed
#       0       $ver is not allowed
#
#   Example:
#       if (!ArchiveUtils->isVersionInRange(
#           $res_val_buf{$id_name},$low_version, $high_ver))
#       {
#           next;
#       }
#   
################################################################################
sub isVersionInRange
{
    my ($class, $ver,$lver,$hver) = @_;

    if (!$ver || (!$lver && !$hver))
    {
        return 0;
    }
    if ( $ver eq $lver)
    {
        return 1;
    }
    if ($ver eq $hver || $lver eq $hver)
    {
        return 0;
    }
    my ($version, $release) = split /-/, $ver;
    my ($lversion, $lrelease) = split /-/, $lver;
    my ($hversion, $hrelease) = split /-/, $hver;
    if ( $lver && ArchiveUtils->testVersion( $version, '<', $lversion, $release, $lrelease))
    {
        return 0;
    }
    if ( $hver && ArchiveUtils->testVersion( $version, '>', $hversion, $release, $hrelease))
    {
        return 0;
    }
    return 1;
}

################################################################################
#
# searchAndCopyPkgs
#
#   search some rpm lists from some directories,
#   and copy found rpm lists to destination directory
#
#   Arguements:
#       $rmpLists           the reference to one array that contains rpm lists
#       $srcDirs            the reference to one array that contains all source
#                           directories
#       $notCopyFlag        1, NOT to copy packages; 0 or undef, to copy packages
#       $cdIds              the reference to one array that contans ISO IDs
#       $isoFiles           one hash, which contains where usefull ISO files are
#       $my_DISTRO_RPMDIR   where rpms are placed in ISO files
#       $baseDir            the base dir name for csm insallation e.g. /csminstall
#       $InstallOSName      
#       $InstallDistributionName 
#       $InstallDistributionVersion
#       $InstallPkgArchitectur
#       $InstallServiceLevel
#       $csmVer
#   Return:
#       undef
#
#   Note:
#       This subroutine will exit and give error message,
#       if some Mandatory rpms can not be found
#
#   Example:
#       ArchiveUtils->searchAndCopyPkgs(
#           \@osProvidedRpmLists,
#           \@dirsToSearch,
#           $notCopyFlag,
#           \@cdIds,
#           $correctImages,
#           $pkgdefs->{ 'DISTRO_RPMDIR'},
#           $nomountcd,
#           $baseDir,
#           "Linux",
#           $distro{'distro_name'},
#           $distro{'distro_version'},
#           $distro{'arch'},
#           $distro{'svc_level'},
#           $csmVer
#       );
#
################################################################################
sub searchAndCopyPkgs
{
    my ($class, $rpmLists, $srcDirs, $notCopyFlag, $cdIds, $isoFiles, 
           $my_DISTRO_RPMDIR, $nomountcd , $baseDir,
        $os, $distributor, $version, $arch, $sl , $csmVer, $forceCopyFlag) = @_;
    
    my @notFoundOsProvidedRpmLists;
    my @notFoundOsProvidedRpms;
    for my $rpmList (@$rpmLists)
    {
        my $destDir = $rpmList->getDestDir($baseDir, $os, $distributor, $version, $arch, $sl, $csmVer);
        print $rpmList->getDescription() . "= $destDir\n" if ($::DEBUG);
        $rpmList->searchAllRpmsInDirList( $srcDirs,$destDir);
        if ( ! $rpmList->isAllFound() && $rpmList->isContainedInCds())
        {
            push @notFoundOsProvidedRpmLists, $rpmList;
            push @notFoundOsProvidedRpms, $rpmList->getUnfoundRpmDetails();
        }
        if ( !$notCopyFlag)
        {
            $rpmList->copyAllFoundRpms( $destDir, $forceCopyFlag);
        }
    }
   
    # search the rpms in CDs 
    if ( scalar( @notFoundOsProvidedRpmLists) > 0 && $cdIds && !$notCopyFlag)
    {
	# only show the packages want to search when no iso files be found/specified
	if (!%{$isoFiles})
	{
	    @notFoundOsProvidedRpms = ServerUtils->get_uniqu_arry_elemnts( @notFoundOsProvidedRpms);
	    MessageUtils->message( 'I', 'IMsgNOT_FOUND_RPMS', join ", ", @notFoundOsProvidedRpms);
	    if ( $nomountcd == 1){ exit 1;}
	}

	if (scalar(%{$isoFiles}))
	{
		# The $::MEDIATYPE will be in the images
		ArchiveUtils->getMediaTypeFromImages($isoFiles, $cdIds);
		@$cdIds = ArchiveUtils->TrimDisksArray($cdIds, $::MEDIATYPE);
	}

	#Determine the cd/dvd device list
	my @devarray = ArchiveUtils->determineDeviceList();
        for my $cdId ( @$cdIds)
        {
		next if (defined $cdId->{is_extracd});
		my ($realdisk_ref, $ismounted, $target) = ArchiveUtils->MountMedia($cdId, $cdIds, $isoFiles, \@devarray, $nomountcd);
		@$cdIds = ArchiveUtils->TrimDisksArray(\@$cdIds, $::MEDIATYPE);
	    
		for my $rpmList ( @notFoundOsProvidedRpmLists)
		{
			my @cdDirs;
			for (@{$my_DISTRO_RPMDIR}) {
				push @cdDirs, "$::MNTCDROM/$_";
			}

                my $destDir = $rpmList->getDestDir($baseDir,$os, $distributor, $version, $arch, $sl, $csmVer);
                print $rpmList->getDescription() . "$destDir\n" if ($::DEBUG);
                $rpmList->searchAllRpmsInDirList( \@cdDirs);
                $rpmList->copyAllFoundRpms( $destDir, $forceCopyFlag);
            }
            if ( $ismounted)
            {
                NodeUtils->runcmd( "$::UNMOUNT $target > /dev/null 2>&1");
            }
        }
    }

    my @notFoundPrereq;
    my @notFoundOptional;
    for my $rpmList ( @$rpmLists)
    {
        if ( ! $rpmList->isAllFound())
        {
            if ( $rpmList->isMandatoryRpmList())
            {
                push @notFoundPrereq, $rpmList->getUnfoundRpmShortNames();
            }
            else
            {
                push @notFoundOptional, $rpmList->getUnfoundRpmShortNames()
                    if ( $rpmList->getUpdatingStyle() eq "UPDATE_OR_INSTALL");
            }
        }
    }
    if ( scalar( @notFoundPrereq))
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                'csminstall', 'E1', 'EMsgPREREQ_NOT_FOUND',
                join(", ", @notFoundPrereq));
    }
    if ( scalar( @notFoundOptional))
    {
        
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                'csminstall', 'I', 'IMsgMissingOptionalRPMs',
                join(", ", @notFoundOptional));
    }
}

################################################################################
#
# getAllDistroDisks 
#
#    Get all the distribution disk hash, including CD disks and DVD disks
#
#   Arguments:
#       $distro_ref   distro_id reference 
#       $pkgdefs   pkgdefs reference
#       $svcLevel  service level 
#
#   Return:
#		@disks - the disk items reference
#
#	Global Variable:
#		N/A
#
#   Example:
#		my @cdIds = ArchiveUtils->getAllDistroDisks(\%distro, $pkgdefs, $svcLevel);    
#       }
#   
################################################################################
sub getAllDistroDisks()
{
	my ($class, $distro_ref, $pkgdefs, $svcLevel) = @_;

	my %distro = %$distro_ref;
	my @cdIds = ();

    if ( exists $pkgdefs->{'distro_disks'} && 
                ( $distro{'effective_svc_level'} eq 'GA' || 
                  ($distro{'effective_distro_name'} eq 'SLES' && $distro{'effective_distro_ver'} <= 9)))
    {
        push @cdIds, @{$pkgdefs->{'distro_disks'}};
    }
    if ( exists $pkgdefs->{'distro_dvd_disks'} && 
                ( $distro{'effective_svc_level'} eq 'GA' || 
                  ($distro{'effective_distro_name'} eq 'SLES' && $distro{'effective_distro_ver'} <= 9)))
    {
        push @cdIds, @{$pkgdefs->{'distro_dvd_disks'}};
    }

    if ( exists $pkgdefs->{'sp_disks'} && $distro{'effective_svc_level'} ne 'GA')
    {
        for my $svc_level ( keys %{$pkgdefs->{'sp_disks'}})
        {
            if ( $svcLevel eq $svc_level)
            {
                push @cdIds, @{$pkgdefs->{'sp_disks'}->{$svcLevel}};
            }
        }
    }
    if ( exists $pkgdefs->{'sp_dvd_disks'} && $distro{'effective_svc_level'} ne 'GA')
    {
        for my $svc_level ( keys %{$pkgdefs->{'sp_dvd_disks'}})
        {
            if ( $svcLevel eq $svc_level)
            {
                push @cdIds, @{$pkgdefs->{'sp_dvd_disks'}->{$svcLevel}};
            }
        }
    }
	return @cdIds;
}
################################################################################
#
# update_packages_to_csminstall
#
#   create needed directory structure and copy needed packages
#   It will call ArchiveUtils->searchAndCopyPkgs().
#   It can handle only one platform each time.
#
#   Arguments:
#       $distroName     distribution name
#       $distroVer      distribution version
#       $svcLevel       service level
#       $arch           arch
#       $csmVer         CSM version
#       $rpmListHashes  the reference to one array that contains rpm lists
#       $baseDir        base directory where the packages should copy to
#       $notCopyFlag    1, NOT to copy pakcages; 0 or undef, copy packages
#       $pkgdefs        one hash that contains platform information
#
#   Return:
#       undef
#
#   Globals:
#       @::PKGPATH
#
#   Note:
#       Until now, three types of base directories are supported,
#       /csminstall on management server,
#       /csmserver on install server,
#       and /var/opt/csm/mnt on nodes
#
#   Example:
#       ArchiveUtils->update_packages_to_csminstall(
#           $::PREREQS_ATTR{'DistributionName'},
#           $::PREREQS_ATTR{'DistributionVersion'},
#           $::PREREQS_ATTR{'InstallServiceLevel'},
#           $::PREREQS_ATTR{'PkgArchitecture'},
#           $::PREREQS_ATTR{'CsmCoreVersion'},
#           '/csminstall',
#           $::DONT_COPY_RPMS,
#           \%::Svrpkgdefs
#       );
#
################################################################################
sub update_packages_to_csminstall
{
    my ($class, $distroName, $distroVer, $svcLevel, $arch, $csmVer, $rpmListHashes,
          $baseDir, $notCopyFlag, $pkgdefs, $nomountcd, $forceCopyFlag) = @_;
   
    my @dirs = @::PKGPATH;
    my %distro = NodeUtils->getDistroID( $distroName,
                                         $distroVer,
                                         $svcLevel,
                                         $arch
    );
    if (!$distro{'distro_name'})
    {
	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
	    'csminstall', 'E1', 'EMsgUNSUPPORTED_DISTRO_SVC_ARCH', join(' ', $distroName,
		$distroVer, $svcLevel, $arch) )
    }
    my $csmRpmDir = "$baseDir/Linux/$distro{'distro_name'}/csm/$csmVer/packages";
    my $csmAddonRpmDir = "$baseDir/Linux/$distro{'distro_name'}/$distro{'distro_version'}/$distro{'arch'}/csm_addon";
    my $osRpmDir = ArchiveUtils->getRPMSDir(
        "Linux",
        $distro{'distro_name'},
        $distro{'distro_version'},
        $distro{'arch'},
        $distro{'svc_level'}
    );

    # get real directory for the operations on remote machines
    $osRpmDir =~ s/\/csminstall/$baseDir/g;
    # Remove symbol links in <osRpmDir>/<csm_repo> if it exists
    # Otherwise it result in copy the same file from sourec to target
    if (-d "$osRpmDir/csmrepo")
    {
        NodeUtils->runcmd("/usr/bin/find  $osRpmDir/csmrepo -name '*.rpm' -exec $::RM -f {} \\;");
    }
                                 
    my ( @osProvidedRpmLists, @notOsProvidedRpmLists);

    for my $rpmListHash ( @$rpmListHashes)
    {
        for my $rpmList ( values %$rpmListHash)
        {
            next if ( !$rpmList);
            if ( $rpmList->isContainedInCds())
            {
                push @osProvidedRpmLists, $rpmList;
            }
            else
            {
                push @notOsProvidedRpmLists, $rpmList;
            }
        }
    }
    
	my @cdIds = ArchiveUtils->getAllDistroDisks(\%distro, $pkgdefs, $svcLevel);    

    my @isoFiles = NodeUtils->getFilePathListInDirList( "*.iso", @dirs);
    my $correctImages = ArchiveUtils->probeImages( \@isoFiles, \@cdIds);

	if (scalar(%$correctImages))
	{
		ArchiveUtils->getMediaTypeFromImages($correctImages, \@cdIds);
		@cdIds = ArchiveUtils->TrimDisksArray(\@cdIds, $::MEDIATYPE);
	}

    my (@distroRpmDirs, @csmReqsDirs);
    @distroRpmDirs = @dirs;
    for my $dir (@dirs)
    {
	# make sure the path as "suse/*" be added.
    # The <current_dir>/reqs and <current_dir>/<DISTRO_RPMDIR> 
    # also should be added into search paths.
	my $rpmdir = $pkgdefs->{ 'DISTRO_RPMDIR' };
	$rpmdir =~ s/\*$//g;

        if (-d "$dir/".$rpmdir)
        {
            push @distroRpmDirs, "$dir/$pkgdefs->{ 'DISTRO_RPMDIR'}";
        }
        if (-d "$dir/reqs")
        {
            push @csmReqsDirs, "$dir/reqs";
        }
    }
    my @dirsToSearch = ( $osRpmDir, @distroRpmDirs);
    # from rhel5, there are several dirs on the CD contain rpms ...
    # for installms find the rpms on redhatel 5 client
    my @rpm_dirs;
    if ($pkgdefs->{'DISTRO_NAME'} =~ /red.*client/i) {
        @rpm_dirs = ($pkgdefs->{'DISTRO_RPMDIR'}, 'Workstation');
    }
    else {
        @rpm_dirs = ($pkgdefs->{'DISTRO_RPMDIR'});
    }
    ArchiveUtils->searchAndCopyPkgs( \@osProvidedRpmLists, 
                                    \@dirsToSearch, 
                                    $notCopyFlag,
                                    \@cdIds,
                                    $correctImages,
                                    \@rpm_dirs,
				                    $nomountcd,
                                    $baseDir,
                                    "Linux",
                                    $distro{'distro_name'},
                                    $distro{'distro_version'},
                                    $distro{'arch'},
                                    $distro{'svc_level'},
                                    $csmVer,
                                    $forceCopyFlag);
    @dirsToSearch = (  $csmRpmDir, $csmAddonRpmDir, @distroRpmDirs, @csmReqsDirs);
    ArchiveUtils->searchAndCopyPkgs( \@notOsProvidedRpmLists, 
                                    \@dirsToSearch, 
                                    $notCopyFlag,
                                    undef,
                                    undef,
                                    undef,
                				    $nomountcd,
                                    $baseDir,
                                    "Linux",
                                    $distro{'distro_name'},
                                    $distro{'distro_version'},
                                    $distro{'arch'},
                                    $distro{'svc_level'},
                                    $csmVer,
                                    $forceCopyFlag);

}

################################################################################
#
# install_packages_Linux
#
#   install Linux rpm packages
#
#   Arguments:
#       $rpmListsHashes the reference to one array that contains rpm lists
#       $installFlag    the option rpm command will use
#
#   Return:
#       1   succeed
#       0   fail
#
#   Example:
#       ArchiveUtils->install_packages_Linux(\@rpmListsToBeInstalled, $installFlag);
#
################################################################################
sub install_packages_Linux
{
    my ($class, $rpmListsHashes, $installFlag, $forceInstallFlag) = @_;
    
    return 0 if ( scalar( @$rpmListsHashes) <= 0);
    if ( $forceInstallFlag)
    {
        if ( $installFlag !~ /--force/)
        {
            $installFlag .= ' --force';
        }
    }
    for my $rpmListsHash ( @$rpmListsHashes)
    {
        for my $installSeq ( sort keys( %$rpmListsHash))
        {
            my $rpmList = $rpmListsHash->{ $installSeq};
            if ( $forceInstallFlag)
            {
                $rpmList->setAllInstalled ( $::FALSE);
            }
            if ( $rpmList ) 
            {
                my $rc = $rpmList->installAll( $installFlag);
                return $::FALSE if ( $rc == $::FALSE);
            }
        }
    }
    return $::TRUE;
}

################################################################################
#
# uninstall_packages_Linux
#
#   uninstall Linux rpm packages
#
#   Arguments:
#       $rpmListsHashes the reference to one array that contains rpm lists
#       $uninstallFlag  the option rpm command will use
#
#   Return:
#       1   succeed
#       0   fail
#
################################################################################
sub uninstall_packages_Linux
{
    my ($class, $rpmListsHashes, $uninstallFlag) = @_;
    
    return $::TRUE if ( scalar( @$rpmListsHashes) <= 0);
    my $rc = $::TRUE;
    for my $rpmListsHash ( @$rpmListsHashes)
    {
        for my $installSeq ( sort {$b<=>$a} keys( %$rpmListsHash))
        {
            my $rpmList = $rpmListsHash->{ $installSeq};
            if ( $rpmList->needToBeUninstall())
            {
                my $isSuccessful = $rpmList->uninstallAll( $uninstallFlag);
                $rc = $::FALSE if ( !$isSuccessful);
            }
        }
    }
    return $rc;
}

################################################################################
#
#   isFSEntryInFstab
#
#       search file system entry in fstab.
#
#   Arguments:
#       $entry  - the file system entry name
#
#   Return:
#       1   succeed
#       0   fail
#
################################################################################
sub isFSEntryInFstab
{
    my ($class, $entry) = @_;
    
    my $fstab = "/etc/fstab";
    my $line;
    my $isfound = 0;
    if (!-e "$fstab")
    {
        return 0;
    }

    open(FSTAB, "<$fstab")
        or return 0;
    while ($line = <FSTAB>)
    {
        #it is not comment and it is not blank
        if (($line !~ m'^#') && ($line =~ m/\w/))
        {
            if ($line =~ /$entry/)
            {
                $isfound = 1;
                last;
             }
        }
    }
    close FSTAB;
    return $isfound; 
}

1;

