# @(#)72   1.28   src/csm/core/pm/Rpm/Rpm.pm.perl, csmcore, csm_rgar2h, rgar2hs001a 1/28/08 04:19:10
# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2006,2008 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
=head1 Rpm

=head2 Class Description
    The class used to deal with all RPM related operations.

Attributes:

=cut

package Rpm;
use strict;
use File::Basename;

my ($MSGCAT, $MSGMAPPATH, $MSGSET);
BEGIN
{

    $::licenseMsgHash{ 'OpenSource'} = 'IMsgGPL_LICENSE';
    $::LS = "/bin/ls";

	# the order of array items can not be changed at will
	# those at the beginning have high priority than at the end
    $::COMPATIBLE_ARCH = {  'i386'   => [ 'i.86', 'noarch'],
                            'x86_64' => [ 'i.86', 'x86_64', 'noarch'],
                            'ppc64'  => [ 'ppc', 'ppc64', 'noarch']
                         };
    $::GREATER = 1;
    $::EQUAL = 0;
    $::LOWER = -1;
    $::TRUE = 1;
    $::FALSE = 0;
    $::UNKNOWN = 'UNKNOWN';

    $MSGCAT = 'csmInstall.cat';
    $MSGMAPPATH = $ENV{'CSM_MSGMAPS'} ? $ENV{'CSM_MSGMAPS'} : '/opt/csm/msgmaps';
    $MSGSET = 'csminstall';
    
}

umask(0022);   #  This sets umask for all CSM files so that group and world only
               #  have read permissions.
               #  To change it, simply use the umask call in your script, after
               #  the "use Rpm;" line

use CSMDefs;
use NodeUtils;
#--------------------------------------------------------------------------------
=head3    new
    Create a new object of class Rpm

    Arguments:
        $shortName: the name of rpm package without version number
        $versionRange: the version range of the rpm package
        $arch: the architecture of the rpm package
        $os: the OS the rpm package belong to
        $license: the license of the rpm package
    Returns:
        the reference of the Rpm object be created
    Globals:
        $::pkgdefs $::UNKNOWN $::FALSE
    Error:

    Example:
        my $perl_pkg = new Rpm('perl', '5');
        my $ospr_ref{'InstallServer'} = new RpmList(
                'Management Server Operating System Prerequisites Packages',
                $perl_pkg
                );

    Comments:
=cut
#--------------------------------------------------------------------------------

sub new
{
    my ( $class, $shortName, $versionRange, $arch, $os, $license) = @_;
    my $self = {};
    bless $self, $class;
    $self->setShortName( $shortName);
    $self->setVersionRange( $versionRange);
    $self->setArch( $arch);
    $self->setOS ( $os);
    $self->setLicense( $license);
    $self->setCompatibleArchList( $::pkgdefs{ "arch"});
    $self->setCompatibleOSList( $::pkgdefs{ "OS"});
    $self->{ "_fullName"} = "";
    $self->{ "_absDirName"} = "";
    $self->{ "_addtionalInfo"} = "";
    $self->{ "_isInstalled"} = $::UNKNOWN;
    $self->{ "_isOtherVerInstalled"} = $::UNKNOWN;
    $self->{ "_isFound"} = $::FALSE;
    $self->{ "_foundVersion"} = "";
    $self->{ "_foundArch"}    = "";
    $self->{ "_foundOs"}    = "";
    $self->{ "_isFoundInstalled"} = $::UNKNOWN;
    $self->{ "_installedVersion"} = "";
    $self->{ "_installedArch"}    = "";
    $self->{ "_installedOs"}    = "";
    $self->{ "_platform"} = "";
    $self->{ '_preinstall'} = $::UNKNOWN;
    $self->{ '_postinstall'} = $::UNKNOWN;
#We have to store the information for the RPM package that was found not in the destination dir
#We may also need to store the information such as OS and Arch of this RPM(out of dest) in future,
# but till now, the "outOfDestOS and outofDestArch" are not needed by current CSM.
    $self->{ '_hasBeenFoundOutOfDest'} = $::FALSE;
    $self->{ '_outOfDestVersion'} = "";
    $self->{ '_outOfDestAbsDir'} = "";
    $self->{ '_outOfDestFullName'} = "";
    $self->{ '_fastParsing'} = $::FALSE;
    return $self;
}
#--------------------------------------------------------------------------------
=head3    searchHighestVersionInFileList
	Search the package represented by current Rpm object in a files list,
	to find the highest version one.
	Set some current Rpm object's attributes when it succeed.

    Arguments:
        a list of files to be searched
    Returns:
        true: success
        false: failure
    Globals:
    Error:
        none

    Example:
        if ( ! $rpm->searchHighestVersionInFileList( @files))
        {
            push @notFoundRpms, $rpm;
        }
    Comments:
=cut
#--------------------------------------------------------------------------------

sub searchHighestVersionInFileList
{
    my $self = shift;
    my $fileListRef = shift;
    my $destAbsDir = shift;
    my $needCompareVersionWithDest = shift;

    my @fileList = @$fileListRef;
    my $highestVersion;
    my $hasFoundInOutOfDest = $::FLASE;
    my $foundOutOfDestVersion;

    my $shortName = $self->{ "_shortName"};
    my @workableFileList = grep /\Q$shortName\E/, @fileList;
    for my $fullPathAndName ( @workableFileList)
    {
        my $fileName = basename( $fullPathAndName);
        next if ($fileName !~ /^\Q$shortName\E/);
        next if ($fileName =~ /src\.rpm$/);
        my $dirName  = NodeUtils->getAbsolutePath( dirname($fullPathAndName));
        my ( $isMatched, $shortname, $version, $arch, $os) = 
            $self->parseRpmFile( $fullPathAndName);
        if ( $isMatched)
        {
            my $compRes = $self->compareVersion( $highestVersion, $version);
            # an undef version/release is the lowest
            # if the found RPM has a higher version,
            # or the version is equal but has a lower arch index
            # for example, the arch index of ppc is lower than ppc64,
            # we choose this RPM
            if (    !$highestVersion ||
                    $compRes == $::LOWER ||
                    ( $compRes == $::EQUAL && 
                          $self->getArchIndex($arch) < $self->getArchIndex( $self->{ "_foundArch"})
                    )
               )
            {
                $highestVersion = $version;
                $self->{ "_fullName"} = $fileName;
                $self->{ "_absDirName"} = $dirName;
                $self->{ "_foundVersion"} = $version;
                $self->{ "_foundArch"}    = $arch;
                $self->{ "_foundOs"}      = $os;
                $self->{ "_isFound"} = $::TRUE;
            }

            if ( $dirName ne $destAbsDir)
            {
                $self->{ '_hasBeenFoundOutOfDest'} = $::TRUE;
                my $outOfDestCompRes = $self->compareVersion( $foundOutOfDestVersion, $version);
                if ( !$self->{ '_outOfDestVersion'} || $outOfDestCompRes == $::LOWER)
                {
                    $self->{ '_outOfDestVersion'} = $version;
                    $self->{ '_outOfDestAbsDir'} = $dirName;
                    $self->{ '_outOfDestFullName'} = $fileName;
                }
            }
        }
    }
    if ( $self->isFound())
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                'csminstall', 'V', 'IMsgFOUND_RPM', $self->{ "_shortName"}, $self->{ "_absDirName"}); 
        if ( $needCompareVersionWithDest && 
                $self->{ '_hasBeenFoundOutOfDest'} && 
                $self->{ "_absDirName"} eq $destAbsDir &&
                $self->{ '_outOfDestVersion'} ne $self->{ "_foundVersion"})
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                    'csminstall', 'I', 'IMsgHigherVersionRpmExistInDest', $self->{ "_shortName"}, $destAbsDir);
        }
    }
    return $self->isFound();
}
#--------------------------------------------------------------------------------
=head3    install
	Install the rpm package repesented by current Rpm object 
	and set relevant attributes.

    Arguments:
        $option:	rpm command option
        $notRunPreAndPost:	whether run pre and post function
    Returns:
        true: success
        false: failure
    Globals:
    Error:
    Example:
        my $cmdReturnCode = $rpm->install('--nodeps');
    Comments:
=cut
#--------------------------------------------------------------------------------
sub install
{
	my $self = shift;
	my $option = shift;
	my $notRunPreAndPost = shift;
	my $rpmName = $self->getShortName();
	if ( !$self->isFound())
	{
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'W', 'EMsgInstallRpmNotFound', $rpmName);
        return $::FALSE;
	}
	my $rpm = $self->getFullPath();
	
	my $returnCode = $::TRUE;
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
        'csminstall', 'I', 'IMsgInstallingRpm', $rpmName);
    if ( !$notRunPreAndPost && !$self->runPreinstall())
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'W', 'EMsgRpmPreinstall', $rpmName);
        $returnCode = $::FALSE;
    }
    
	my $rpmCmdOutput = NodeUtils->runcmd("$::RPMCMD $option $rpm", -1);
	if ( $::RUNCMD_RC == $::OK)
	{
		$self->{ "_isInstalled"} = 'YES';
		$self->{ "_isFoundInstalled"} = 'YES';
	}
	else
	{
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'W', 'EMsgFailedtoInstallRpm', $rpmName);
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'I', 'IMsgShow_Output', "Command rpm printed out:\n\t$rpmCmdOutput");
		$returnCode = $::FALSE;
	}
	
    if ( !$notRunPreAndPost && !$self->runPostinstall())
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'W', 'EMsgRpmPostinstall', $rpmName);
        $returnCode = $::FALSE;
    }
    
    return $returnCode;
}
#--------------------------------------------------------------------------------
=head3    uninstall
	Uninstall the rpm package repesented by current Rpm object
	and set relevant attributes.

    Arguments:
        $options: rpm command options
    Returns:
        true: success
        false: failure
    Globals:
    Error:
    Example:
        my $cmdReturnCode = $rpm->uninstall('--force');
    Comments:
=cut
#--------------------------------------------------------------------------------
sub uninstall()
{
    my $self = shift;
    my $options = shift;
    my $rpm_uninst = $self->getShortName() . "-" . $self->getVersion();

    return $::TRUE if (!$self->isInstalled);

    my $rpmCmdOutput = NodeUtils->runcmd("$::RPMCMD $options $rpm_uninst", -1);
    if ($::RUNCMD_RC == $::OK)
    {
        $self->{"_isInstalled"} = 'NO';
        $self->{"_isFoundInstalled"} = $::UNKNOWN;
        return $::TRUE;
    }
    else
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'W', 'EMsgRemoveRpm', $rpm_uninst);
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'I', 'IMsgShow_Output', "Command rpm printed out:\n\t$rpmCmdOutput");
        return $::FALSE;
    }
}
#--------------------------------------------------------------------------------
=head3    checkInstallStatus
	Check whether the rpm package repesented by current Rpm object 
	has been installed, and set relevant attributes.

    Arguments:
        none
    Returns:
        true: installed
        false: not installed
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub checkInstallStatus
{
    my $self = shift;
    my @cmdRes = NodeUtils->runcmd( 
            qq($::RPMCMD -q --qf '%{Name}::%{Version}-%{Release}::%{Arch}::%{OS}\n' $self->{'_shortName'}), -1);
    if( $::RUNCMD_RC != $::OK)
    {
        $self->{ "_isInstalled"} = 'NO';
        return $::FALSE ;
    }
    for my $cmdResEntry (@cmdRes)
    {
        my ( $name, $version, $arch, $os) = ( $cmdResEntry =~ /(.*)::(.*)::(.*)::(.*)/);
        if ( $name eq $self->{ "_shortName"} && 
                $self->isCorrectArch( $arch) && 
                $self->isCorrectOS( $os))
        {
            if ( $self->isInRange($version, $self->{ "_versionRange"}))
            {
                $self->{ "_isInstalled"} = 'YES';
                $self->{ "_installedVersion"} = $version;
                $self->{ "_installedArch"}    = $arch;
                $self->{ "_installedOs"}	  = $os;
                return $::TRUE;
            }
            else
            {
                $self->{ "_isOtherVerInstalled"} = 'YES';
            }
        }
    }
    $self->{ "_isInstalled"} = 'NO';
    $self->{ "_isOtherVerInstalled"} = 'NO' if ( $self->{ "_isOtherVerInstalled"} eq $::UNKNOWN);
    return $::FALSE;
}
#--------------------------------------------------------------------------------
=head3    isInstalled
	Check whether the rpm package repesented by current Rpm object 
	has been installed, call checkInstallStatus

    Arguments:
        none
    Returns:
        true: installed
        false: not installed
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub isInstalled
{
	my $self = shift;
	if ( $self->{ "_isInstalled"} eq $::UNKNOWN)
	{
		$self->checkInstallStatus();
	}
	return $::TRUE if ( $self->{ "_isInstalled"} eq 'YES');
    return $::FLASE;
}
#--------------------------------------------------------------------------------
=head3    isOtherVerInstalled
	Check whether the rpm package with other version (incorrect version)
	has been installed, call checkInstallStatus

    Arguments:
        none
    Returns:
        true: installed
        false: not installed
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub isOtherVerInstalled
{
	my $self = shift;
	if ( $self->{ "_isOtherVerInstalled"} eq $::UNKNOWN)
	{
		$self->checkInstallStatus();
	}
	return $::TRUE if ( $self->{ "_isOtherVerInstalled"} eq 'YES');
    return $::FLASE;
}
#--------------------------------------------------------------------------------
=head3   isFoundInstalled 
	Check whether the rpm package repesented by current Rpm object 
	has been installed and found, call checkFoundRpmInstallationStatus

    Arguments:
        none
    Returns:
        true: the package has been found and installed
        false: the package has not been found or installed
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub isFoundInstalled
{
    my $self = shift;
    if ( $self->{"_isFoundInstalled"} eq $::UNKNOWN)
    {
        $self->checkFoundRpmInstallationStatus();
    }
    return $::TRUE if ( $self->{"_isFoundInstalled"} eq 'YES');
    return $::FLASE;
}
#--------------------------------------------------------------------------------
=head3   isFound 
	Check whether the rpm package repesented by current Rpm object 
	has been found.

    Arguments:
        none
    Returns:
        true: the package has been found 
        false: the package has not been found
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub isFound
{
    my $self = shift;
    return $self->{ "_isFound"};
}
#--------------------------------------------------------------------------------
=head3   setCompatibleArchList 
	Set the compatible architectures of current rpm package.
	The compatible architectures specify which arch the package could be on.

    Arguments:
        $arch: 'i386' 'x86_64' or 'ppc64'
    Returns:
        none
    Globals:
        $::COMPATIBLE_ARCH
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setCompatibleArchList
{
    my $self = shift;
    my $arch = shift;
    $self->{ '_compatibleArchList'} = $::COMPATIBLE_ARCH->{$arch};
}
#--------------------------------------------------------------------------------
=head3   getCompatibleArchList 
	Get the compatible architectures list of current rpm package.

        Arguments:
	    none
        Returns:
	    none
        Globals:
        Error:
        Example:
        Comments:
=cut
#--------------------------------------------------------------------------------
sub getCompatibleArchList
{
	my $self = shift;
	return @{$self->{ '_compatibleArchList'}};
}

#--------------------------------------------------------------------------------
=head3   setShortName 
	Set the short name of current rpm package

    Arguments:
        the name of rpm package without version number, such as 'perl'
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setShortName
{
	my $self = shift;
	$self->{ "_shortName"} = NodeUtils->clearBlank( shift);
}
#--------------------------------------------------------------------------------
=head3   getShortName 
	Get the short name of current rpm package

    Arguments:
        none
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub getShortName
{
	my $self = shift;
	return $self->{ "_shortName"};
}

#--------------------------------------------------------------------------------
=head3   setVersionRange
	Set the version range that current rpm package must be fit.

    Arguments:
	    the version range. 
	    should be 1) 'n,m', means higher than n but lower than m
	    	  2) 'n,', means higher than n
	    	  3) ',m', means lower than m
	    	  4) 'n', means exactly n
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setVersionRange
{
	my $self = shift;
	my $versionRange = shift;
	if ( $versionRange =~ /(.*),(.*)/)
	{
		my $min = $1;
		my $max = $2;
		$min = NodeUtils->clearBlank( $min);
		$max = NodeUtils->clearBlank( $max);
		if ( ! $max)
		{
			$self->{ "_versionRange"} = $min;
		}
		else
		{
			$self->{ "_versionRange"} = "$min,$max";
		}
	}
	else
	{
		$self->{ "_versionRange"} = NodeUtils->clearBlank( $versionRange);
	}
}
#--------------------------------------------------------------------------------
=head3   getVersionRange 
	Get the demanded version range for current package

    Arguments:
        none
    Returns:
        version range
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub getVersionRange
{
	my $self = shift;
	return $self->{ "_versionRange"};
}

# add by review
#--------------------------------------------------------------------------------
=head3   getVersion 
	Get the version of current package which has been installed

    Arguments:
        none
    Returns:
        version
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub getVersion
{
    my $self = shift;
    return $self->{"_installedVersion"};
}
#--------------------------------------------------------------------------------
=head3   setArch 
	Set the architecture of current package

    Arguments:
        one of 'i386' 'x86_64' 'ppc64' 
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setArch
{
	my $self = shift;
	$self->{ "_arch"} = NodeUtils->clearBlank( shift);
}
#--------------------------------------------------------------------------------
=head3   getArch 
	Get the architecture of current package

        Arguments:
	    none
        Returns:
	    architecture of current package
        Globals:
        Error:
        Example:
        Comments:
=cut
#--------------------------------------------------------------------------------
sub getArch
{
	my $self = shift;
	return $self->{ "_arch"};
}
#--------------------------------------------------------------------------------
=head3   setOS 
	Set the OS name which currently be used on

    Arguments:
        OS name such as "Linux" "AIX"
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setOS
{
	my $self = shift;
	$self->{ "_os"} = NodeUtils->clearBlank( shift);
}
#--------------------------------------------------------------------------------
=head3   getOS
	Get the OS name which current be used on

    Arguments:
        none
    Returns:
        OS name
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub getOS
{
	my $self = shift;
	return $self->{ "_os"};
}
#--------------------------------------------------------------------------------
=head3   setLicense
	Set the license name current package controlled by

    Arguments:
        license name such as "GPL"
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setLicense
{
	my $self = shift;
	$self->{ "_license"} = NodeUtils->clearBlank( shift);
}
#--------------------------------------------------------------------------------
=head3   getLicense
	Get the license name current package controlled by

    Arguments:
        none
    Returns:
        license name
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub getLicense
{
	my $self = shift;
	return $self->{ "_license"};
}

#=====================new function========================================
#--------------------------------------------------------------------------------
=head3   parseRpmFile
	Parse the given rpm file to see if it is the package repesented
	by current object

    Arguments:
        a full path of a rpm package
    Returns:
        a array with 5 elements. the first one indicate whether
        the argument is the package repesented by current object.
        the other 4 is the package's name, version, architecture 
        and OS.
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub parseRpmFile
{
    my $self = shift;
    my $fullPath = shift;
    my $cmdRes;
    if ( -f "$fullPath")
    {
        if ( ! $::__RPM_INFO_BUFFER{ $fullPath})
        {
            if ( $self->{ '_fastParsing'})
            {
               my $fileName = basename( $fullPath);
               if ( $fileName =~ /^\Q$self->{ "_shortName"}\E-(.*)-(.*)\.(.*)\.rpm$/ )
               {
                  $::__RPM_INFO_BUFFER{ $fullPath} = $self->{ "_shortName"} . "::$1-$2::$3::linux";
               } 
            }
            
            if ( ! $::__RPM_INFO_BUFFER{ $fullPath}) #if not fast parsing, or fast parsing failed
            {
                $cmdRes = NodeUtils->runcmd( 
                    "$::RPMCMD -qp --qf '%{Name}::%{Version}-%{Release}::%{Arch}::%{OS}\n' $fullPath", -1);
                if ( ! $::RUNCMD_RC)
                {
                    $::__RPM_INFO_BUFFER{ $fullPath} = $cmdRes;
                }
            }

            if ( !$::__RPM_INFO_BUFFER{ $fullPath}) #if there is no way to parse this file
            {
                if (!$::__ALREADY_ERR_TO_PARSE_FILE{$fullPath})
                {
                    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                            'csminstall', 'V', 'IMsgParseRPMFailed', $fullPath, $::__RPM_INFO_BUFFER{ $fullPath});
                }
                $::__ALREADY_ERR_TO_PARSE_FILE{$fullPath} = 1;
            }
        }
                
        my ( $name, $version, $arch, $os) = ( $::__RPM_INFO_BUFFER{ $fullPath} =~ /(.*)::(.*)::(.*)::(.*)/);
        if ( $name eq $self->{ "_shortName"} &&
                $self->isInRange( $version, $self->{ "_versionRange"}) &&
                $self->isCorrectArch( $arch) &&
                $self->isCorrectOS( $os))
        {
            return ( $::TRUE, $name, $version, $arch, $os);
        }
        else
        {
            return ( $::FALSE, $name, $version, $arch, $os);
        }
    }
    else
    {
        return ( $::FALSE, undef, undef, undef, undef);
    }
}
#--------------------------------------------------------------------------------
=head3   isInRange
	Check whether a given version between the given version range

        Arguments:
	    n		a version such as "1.8.5"
	    range	a version range such as "1.1,2.0"
        Returns:
	    true	given version between given range
	    false	given version not between given range
        Globals:
        Error:
        Example:
        Comments:
=cut
#--------------------------------------------------------------------------------
sub isInRange
{
    my $self = shift;
    my $n = shift;
    my $range = shift;
    return $::TRUE if ( !$range);
    my ( $minN, $maxN);
    if ( $range =~ /(.*),(.*)/ )
    {
        $minN = $1;
        $maxN = $2;
        return ($self->compareVersion($minN,$n) != $::GREATER && $self->compareVersion($n,$maxN) != $::GREATER);
    }
    else
    {
        $minN = $range;
	return ($self->compareVersion($minN,$n) != $::GREATER);
    }
}
#--------------------------------------------------------------------------------
=head3   compareVersion 
	Compare the given two versions

    Arguments:
        two version, version1 and version2
    Returns:
        $::LOWER if version1 lower than version2
        $::EQUAL if version1 equal to version2
        $::GREATER if version1 greater than version2
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub compareVersion
{
    my $self = shift;
    my ( $version1, $version2) = @_; # here version1 and version2 contain version 
                                     # number and release number, for an example: "1.2.3-45.6".
    my ( $v1, $r1, $v2, $r2); # v1 will be the version number of $version1, for an example: "1.2.3"
                           # r1 will be the release number of $version1, for an example: "45.6"
                           # v2 will be the version number of $version2
                           # r2 will be the release number of $version2
    $version1 = '0' if ( ! $version1);
    $version2 = '0' if ( ! $version2);
    if ( $version1 =~ /^(.+)-(.*)$/)
    {
        $v1 = $1;
        $r1 = $2;
    }
    else
    {
        $v1 = $version1;
        $r1 = undef;
    }
    
    if ( $version2 =~ /^(.+)-(.*)$/)
    {
        $v2 = $1;
        $r2 = $2;
    }
    else
    {
        $v2 = $version2;
        $r2 = undef;
    }
    
	return $::LOWER if $self->testVersion($v1,'<',$v2,$r1,$r2);
	return $::EQUAL if $self->testVersion($v1,'=',$v2,$r1,$r2);
	return $::GREATER;
}
#--------------------------------------------------------------------------------
=head3   isCorrectArch 
	Check whether the given architecture is the same as current 
	package's architecture, or is one of the compatible architectures.

    Arguments:
        arch	the given architecture, such as 'i386' 'x86_64' 'ppc64'
    Returns:
        true	same or be one of the compatible architectures.
        false	not same and ont one of the compatible architectures.
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub isCorrectArch
{
    my $self = shift;
    my $arch = shift;
    if ( $self->{ "_arch"})
    {
        return $::TRUE if $self->{ "_arch"} eq $arch;
        return $::FALSE;
    }
    else
    {
        for my $compatArch (@{$self->{ "_compatibleArchList"}})
        {
            return $::TRUE if $arch =~ /^$compatArch$/;
        }
                
        return $::FALSE;
    }
}
#--------------------------------------------------------------------------------
=head3   isCorrectOS
	Check whether the given OS is same to the package's OS, or one of
	the compatible OS.

    Arguments:
        OS name such as "Linux" "AIX"
    Returns:
        true    same or be one of the compatible OS
        falase  not same and not be one of the compatible OS
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub isCorrectOS
{
    my $self = shift;
    my $os = shift;
    if ( $self->{ "_os"})
    {
        return $::TRUE if $os eq $self->{ "_os"};
        return $::FALSE;
    }
    else
    {
        for my $compatOS ( @{$self->{ "_compatibleOSList"}})
        {
            return $::TRUE if $os =~ /^$compatOS$/;
        }
        return $::FALSE;
    }
}
#--------------------------------------------------------------------------------
=head3   setCompatibleOSList 
	Set the compatible OS for current package

    Arguments:
        OS name such as "Linux" "AIX"
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
        Only set the '_compatibleOSList' for linux.
=cut
#--------------------------------------------------------------------------------
sub setCompatibleOSList
{
	my $self = shift;
	my $os = shift;
	if ( $os eq 'linux')
	{
		$self->{ '_compatibleOSList'} = ['linux', 'Linux'];
	}
}
#--------------------------------------------------------------------------------
=head3   getAbsDirName 
	Return the current package's absolute path but without its name.

    Arguments:
        none
    Returns:
        absolute path without package name, or ""
    Globals:
    Error:
    Example:
    Comments:
        return "/csmserver/Linux/RedHatEL-AS/csm/1.6.0/packages/"
        but not "/csmserver/Linux/RedHatEL-AS/csm/1.6.0/packages/rsct.basic-2.4.6.0-06213.ppc.rpm"
=cut
#--------------------------------------------------------------------------------
sub getAbsDirName
{
    my $self = shift;
    return $self->{ "_absDirName"};
}
#--------------------------------------------------------------------------------
=head3    checkFoundRpmInstallationStatus
	Check whether the rpm package repesented by current Rpm object 
	has been installed and found, and set relevant attributes.

    Arguments:
        none
    Returns:
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub checkFoundRpmInstallationStatus
{
    my $self = shift;
    $self->checkInstallStatus();
    if ( $self->isFound() && $self->isInstalled() &&
         $self->{ "_installedVersion"} eq $self->{ "_foundVersion"} &&
         $self->{ "_installedArch"} eq $self->{ "_foundArch"} &&
         $self->{ "_installedOs"} eq $self->{ "_foundOs"}
        )
    {
        $self->{ "_isFoundInstalled"} = 'YES';
    }
}
#--------------------------------------------------------------------------------
=head3   getFullPath 
	Return the current package's absolute path

    Arguments:
        none
    Returns:
        absolute path of current package
    Globals:
    Error:
    Example:
    Comments:
        return such as 
        "/csmserver/Linux/RedHatEL-AS/csm/1.6.0/packages/rsct.basic-2.4.6.0-06213.ppc.rpm"
=cut
#--------------------------------------------------------------------------------
sub getFullPath
{
    my $self = shift;
    return $self->{ "_absDirName"} . '/' . $self->{ "_fullName"};
}
#--------------------------------------------------------------------------------
=head3   copy 
	Copy current package to the specified directory if the package 
	has been found and the destination isn't the same as the '_absDirName'

    Arguments:
        destination directory
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub copy
{
    my $self = shift;
    my $destDir = shift;
    my $forceCopyFlag = shift;
    $destDir = NodeUtils->getAbsolutePath( $destDir);
    if ( $self->isFound())
    {
        my $srcFile;
        my $destFile;
        my $needToBeCopied = $::FALSE;
        if ( $destDir ne $self->{ '_absDirName'})
        {
            $srcFile = "$self->{ '_absDirName'}/$self->{ '_fullName'}";
            $destFile = "$destDir/$self->{ '_fullName'}";
            $self->{ '_absDirName'} = $destDir;
            $needToBeCopied = $::TRUE;
        }
        elsif ( $forceCopyFlag && $self->{ '_hasBeenFoundOutOfDest'})
        {
            $srcFile = "$self->{ '_outOfDestAbsDir'}/$self->{ '_outOfDestFullName'}";
            $destFile = "$destDir/$self->{ '_outOfDestFullName'}";
            $needToBeCopied = $::TRUE;
        }

        if ( $needToBeCopied)
        {
		MessageUtils->messageFromCat(
							  'csmInstall.cat',
							  $::MSGMAPPATH,
							  'csminstall',
							  'V',
							  'IMsgCOPYING',
							  'RPM package ' .  $self->getShortName()
		);

            my $cmd = "$::COPY $srcFile $destDir";
            NodeUtils->runcmd("$cmd", -2);
            $cmd = "$::CHMOD +r $destFile";
            NodeUtils->runcmd("$cmd", -2);
        }
    }
}
#=======================================================================
#--------------------------------------------------------------------------------

=head3    testVersion

        Compare version1 and version2 according to the operator and
        return True or False.

        Arguments:
                $version1
                $operator
                $version2
                $release1
                $release2
        Returns:
                True or False
        Globals:
                none
        Error:
                messageFromCat E11

        Example:
                if (ArchiveUtils->testVersion ( $ins_ver,
                                                "<",
                                                $req_ver,
                                                $ins_rel,
                                                $req_rel)){ blah; }

        Comments:
                The return value is generated with the Require query of the
                rpm command.

=cut

#--------------------------------------------------------------------------------

sub testVersion
{
    my ($self, $version1, $operator, $version2, $release1, $release2) = @_;

# Note: perl seems to be able to handle really big integers, but if we run into a limit, we
#       can use the Math::BigInt pkg

    my @a1 = split(/\./, $version1);
    my @a2 = split(/\./, $version2);
#We should make version1 and version2 have a same version precision, i.e.
#when we compare versions 1.6.0 with 1.6.0.10, we should change 1.6.0.10 to 1.6.0
#first. It's because of a human common sense, if someone ask for CSM 1.6.0 packages,
#then 1.6.0.10 should be correct packages for him. 
    my ($len,$num1,$num2);
    if ((defined $release1) && (defined $release2)){
        $len = (scalar(@a1) > scalar(@a2) ? scalar(@a1) : scalar(@a2));
    }else{
        $len = (scalar(@a1) < scalar(@a2) ? scalar(@a1) : scalar(@a2));
    }
    $#a1 = $len - 1;  # make the arrays the same length before appending release
    $#a2 = $len - 1;
    push @a1, split(/\./, $release1);
    push @a2, split(/\./, $release2);
    $len = (scalar(@a1) < scalar(@a2) ? scalar(@a1) : scalar(@a2));
    $num1 = '';
    $num2 = '';

    for (my $i = 0 ; $i < $len ; $i++)
    {
        my ($d1,$w1) = $a1[$i] =~ /^(\d*)(\w*)/;   
        my ($d2,$w2) = $a2[$i] =~ /^(\d*)(\w*)/;


        my $diff = length($d1) - length($d2);
        if ($diff > 0)                     # pad d2
        {
            $num1 .= $d1;
            $num2 .= ('0' x $diff) . $d2;
        }
        elsif ($diff < 0)                  # pad d1
        {
            $num1 .= ('0' x abs($diff)) . $d1;
            $num2 .= $d2;
        }
        else                               # they are the same length
        {
            $num1 .= $d1;
            $num2 .= $d2;
        }

        if ( $w1 && $w2)
        {
            my ($w_to_d1, $w_to_d2) = $self->compWord( $w1, $w2);
            $num1 .= $w_to_d1;
            $num2 .= $w_to_d2;
        }

    }

# Remove the leading 0s or perl will interpret the numbers as octal
    $num1 =~ s/^0+//;
    $num2 =~ s/^0+//;

#SuSE Changes
# if $num1="", the "eval '$num1 $operator $num2'" will fail. So MUSTBE be sure that $num1 is not a "".
    if (length($num1) == 0) { $num1 = 0; }
    if (length($num2) == 0) { $num2 = 0; }

#End of SuSE Changes

    if (length($release1))
    {
        $release1 = "-$release1";
    }    # this is just for error msgs
    if (length($release2))
    {
        $release2 = "-$release2";
    }    # this is just for error msgs

    if ($operator eq '=') { $operator = '=='; }
    my $bool = eval "$num1 $operator $num2";
    if (length($@))
    {
        MessageUtils->messageFromCat(
                'csmInstall.cat',
                $::MSGMAPPATH,
                'csminstall',
                'W',
                'EMsgVERSION_COMPARE_PROBLEM',
                "$version1$release1 $operator $version2$release2",
                $@
                );
        return undef;
    }
    if ($ENV{LANG} =~ /^(C|POSIX|en_US)/)
    {    #this is a work around for NLS problems
        MessageUtils->messageFromCat(
                'csmInstall.cat',
                $::MSGMAPPATH,
                'csminstall',
                'V',
                'IMsgCOMPARING_VERSION',
                "$version1$release1 $operator $version2$release2",
                ($bool ? 'true' : 'false')
                );
    }
    return $bool;
}
#--------------------------------------------------------------------------------
=head3   setPreinstall
	Set the function to be run before installing the package.

    Arguments:
        reference of a function
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setPreinstall
{
    my $self = shift;
    my $func = shift;
    if ( !$func)
    {
        $self->{ '_preinstall'} = $func;
    }
}
#--------------------------------------------------------------------------------
=head3   runPreinstall
	Call the function be set to run before installing the package

    Arguments:
        none
    Returns:
        the result of the called function or true if there is no 
        such function
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub runPreinstall
{
    my $self = shift;
    if ( $self->{ '_preinstall'} ne $::UNKNOWN)
    {
        &{$self->{ '_preinstall'}};
    }
	else
	{
			return $::TRUE;
	}
}
#--------------------------------------------------------------------------------
=head3   setPostinstall
	Set the function to be run after installing the package.

    Arguments:
        reference of a function
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setPostinstall
{
    my $self = shift;
    my $func = shift;
    if ( $func)
    {
        $self->{ '_postinstall'} = $func;
    }
}
#--------------------------------------------------------------------------------
=head3   runPostinstall
	Call the function be set to run after installing the package

    Arguments:
        none
    Returns:
        the result of the called function or true if there is no 
        such function
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub runPostinstall
{
    my $self = shift;
    if ( $self->{ '_postinstall'} ne $::UNKNOWN)
    {
        &{$self->{ '_postinstall'}};
    }
	else
	{
		return $::TRUE;
	}
}
#--------------------------------------------------------------------------------
=head3   isFoundRequiredToBeInstalled
	Check whether should install the found package represented by
	current object.

    Arguments:
        none
    Returns:
        true    should install the found package
        false   shouldn't install the found package
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub isFoundRequiredToBeInstalled
{
	my $self = shift;
	return $::FALSE if ( $self->isFoundInstalled());
	return $::TRUE if ( !$self->isInstalled());
	my $result = $self->compareVersion( $self->{ '_foundVersion'}, $self->{ '_installedVersion'});
	return ( $result == $::GREATER);
}
#--------------------------------------------------------------------------------
=head3   getArchIndex
	Get the index of given architecture in the '_compatibleOSList'

    Arguments:
        arch    such as 'i386' 'ppc64' 'x86_64'
    Returns:
        the index of the architecture in the '_compatibleOSList' or
        a very big number identify false.
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub getArchIndex
{
	my $self = shift;
	my $arch = shift;
	for ( my $i =0; $i< scalar(@{$self->{ "_compatibleArchList"}}); $i++)
	{
		$a = ${$self->{ "_compatibleArchList"}}[$i];
		return $i if ( $arch eq $a);
	}
	return 99999; # big enough
}
#--------------------------------------------------------------------------------
=head3   getRpmDetails
	Get the package name pattern of current package represented by
	the object

    Arguments:
    Returns:
        package name pattern such as 'csm.core.*.i386.rpm'
    Globals:
    Error:
    Example:
    Comments:
        be used when try to get the not found package
=cut
#--------------------------------------------------------------------------------
sub getRpmDetails
{
    my $self = shift;
    my $info = $self->getShortName() . '*';
                   
    if ( $self->{ '_arch'})
    {
        $info .= '.' . $self->{ '_arch'};
    }
    
    $info .= '.rpm';
    return $info;
}

#-------------------------------------------------------------------------------
=head3   setInstalled
	Set the install flag of this Rpm object

    Arguments:
        $::TRUE or $::FALSE or $::UNKNOWN
    Returns:
        none
    Globals:
    Error:
    Example:
    Comments:
=cut
#--------------------------------------------------------------------------------
sub setInstalled
{
    my $self = shift;
    $self->{ '_isInstalled'} = shift;
    $self->{ '_isFoundInstalled'} = $self->{ '_isInstalled'};
}

#--------------------------------------------------------------------------------

=head3    compWord

        Compare version1 word and version2 word. This subroutine can only be used to compare
        one section in version number, and this section cannot start with number.

        Arguments:
                $w1:
                $w2
        Returns:
                if $w1 > $w2, return (1,0)
                if $w1 < $w2, return (0,1)
                if $w1 == $w2, return (undef, undef)
        Globals:
                none

        Example:
                if ($self->compWord ( "adfadsfa","acc2")
                return (0,1)
        Comments:
                the version word cannot contain ".", and cannot start with number.
                For examples, following version words cannot be compared by this subroutine:
                (123.321, 123.322)  You need use subroutine testVersion to do the version comparision
                (123abc,12bcd)      You need use subroutine testVersion to do the version comparision

=cut

#--------------------------------------------------------------------------------

sub compWord
{
    my $self = shift;

    my ($w1,$w2) = @_;

    return (undef,undef) if (!$w1 || !$w2);

    my @strList1 = unpack "C*", $w1;
    my @strList2 = unpack "C*", $w2;

    my $len = scalar(@strList1) < scalar(@strList2) ? scalar(@strList1) : scalar(@strList2);

    for ( my $i = 0; $i < $len; $i++)
    {
        next if ( $strList1[$i] == $strList2[$i]);
        return ( 0, 1) if ( $strList1[$i] < $strList2[$i]);
        return ( 1, 0);
    }
    return (undef,undef);

}

#--------------------------------------------------------------------------------

=head3    setFastParsing

        Set fast parsing attribute for this Rpm object.

        Arguments:
                $fastParsing: true of false
        Returns:
                none
        Globals:
                none

        Example:
                if ($self->setFastParsing ( $::TRUE)
        Comments:
                The attribute _fastParsing will be used in subroutine parseRpmFile.
=cut

#--------------------------------------------------------------------------------

sub setFastParsing
{
    my $self = shift;
    my $fastParsing = shift;

    if ( $fastParsing)
    {
        $self->{ '_fastParsing'} = $::TRUE;
    }
    else
    {
        $self->{ '_fastParsing'} = $::FALSE;
    }
}

return 1;
