#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2003,2008 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
# "@(#)32   1.56.1.314   src/csm/core/pm/ServerUtils.pm.perl, setup, csm_rgar2h, rgar2hs001a 12/3/07 03:22:13"
#
#####################################################################

package ServerUtils;

use strict;

#            locale tells perl to honor locale for sorting, dates, etc.
#            More info about locale.pm and perl handling of locales can be found in
#            /usr/lib/perl5/5.6.0/locale.pm and the man page for perllocale.
use locale;
use Socket;

use Fcntl qw(:flock);
use File::Glob ':glob';
use File::Basename;
use File::Find;
use File::Copy;
use File::Path;    # Provides mkpath() and rmtree()
use IO::Socket;

use Cwd;

my $distro;
my $csmroot;
my ($MSGCAT, $MSGMAPPATH, $MSGSET);
my $NODEGROUPEXPMEM_WARNING = 1;

# Clear dsh environment variables
if(!$ENV{'RUNNING_DSH'}){
    delete $ENV{'DSH_NODE_LIST'};
    delete $ENV{'WCOLL'};
    delete $ENV{'DSH_DEVICE_LIST'};
}
# $NodeUtils::NO_MESSAGES;    # Set this to 1 if you do not want NodeUtils to
# print any error msgs

# $NodeUtils::errno;          # Will be set if an error occurs.  You must zero
# this out before calling a NodeUtils function,
# if you want to check it afterwards.

BEGIN
{

    #    This enables us to redirect where it looks for other CSM files during development
    $csmroot = $ENV{'CSM_ROOT'} ? $ENV{'CSM_ROOT'} : '/opt/csm';

    $MSGCAT = 'csmInstall.cat';
    if (defined $::MSGMAPPATH)
    {
        $MSGMAPPATH = $ENV{'CSM_ROOT'} ? "$csmroot/msgmaps" : $::MSGMAPPATH;
    }
    else
    {
        $MSGMAPPATH = "$csmroot/msgmaps";
    }
    $MSGSET = 'csminstall';
}

if (-e "$csmroot/pm/DisklessUtils.pm")
{
    require DisklessUtils;
    $::DISKLESS_PM = 1;
}

umask(0022);   #  This sets umask for all CSM files so that group and world only
               #  have read permissions.
               #  To change it, simply use the umask call in your script, after
               #  the "use ServerUtils;" line

#--------------------------------------------------------------------------------

=head1    ServerUtils

=head2    Package Description

This program module file, supports the CSM/install server-side dependencies.

If adding to this file, please take a moment to ensure that:

    1.  Your contrib has a readable pod header describing the purpose and use of
         your contrib.

    2. Your contrib is under the correct heading and is in alphabetical order
    under that heading.

    3. You test your contribution by running from the command line:  

       pod2html  --verbose --title=ServerUtils ServerUtils.pm.perl --outfile=ServerUtils.html
       
       and examining the ./ServerUtils.html file in a browser.


=cut

#--------------------------------------------------------------------------------

=head2    Package Dependancies

    use strict;
    use Fcntl qw(:flock);
    use File::Basename;
    use File::Find;
    use File::Path;    # Provides mkpath()

=cut

#--------------------------------------------------------------------------------

=head2    Misc Tools

=cut

#--------------------------------------------------------------------------------

=head3    close_delete_file

        Closes and deletes the file specified by the two input parameters.

        Arguments:
                $file handle
                $file name
        Returns:
                null
        Error:
                none
        Example:
                ServerUtils->close_delete_file($::NODE_LIST_FILE, $nodelist_file);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub close_delete_file
{
    my ($class, $file_handle, $file_name) = @_;
    close $file_handle;

    unlink($file_name);
}

#--------------------------------------------------------------------------------

=head3    copy_exceptions_rveg

        A subroutine to hold some file copy hacks for the vega release.

	The rotuine has the release name rveg attached to it, to motiveate those who
	come later to fix the underlying problem for the next release.

        Arguments:
		n/a
        Returns:
                null
        Error:
                none
        Example:
                n/a
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub copy_exceptions_rveg
{
    my ($class,
        $install_method,    # eg kickstart or autoyast
        $filename        # the name of the file to test.
    ) = @_;

    my $server = $::PLTFRM;    # the managment server doing the install

    # This hack relates to an interoprability issue between AIX and Linux.
    # When AIX runs copycsmpkgs on a Linux Distro, it tries to copy a few too many files...

    if (
        ($server eq "AIX")
        && (   ($install_method eq 'kickstart')
            || ($install_method eq 'kickstart-upgrade')
            || ($install_method eq 'autoyast'))
      )
    {

        # just test against the filenames one at a time
        #if ($filename eq "/opt/csm/csmbin/nodestatus.client")	{ return $::NOK; }
        #if ($filename eq "/opt/csm/csmbin/setbootdisk")	{ return $::NOK; }
        if ($filename eq "/usr/lib/syslinux/pxelinux.0") { return $::NOK; }
    }

    return $::OK;
}

#--------------------------------------------------------------------------------

=head3    get_lock

        get ownership of a non-blocking semiphore / file

        Arguments:
                $fileName
        Returns:
                2  if it couldn't open $fileName
                3  if it couldn't lock $fileName
        Error:
                messageFromCat E1
        Example:
                ServerUtils->get_lock("/tmp/updateisvr.sem");_
        Comments:
                get_lock() and release_lock() implement semiphore locking for csm comnands
                such as updatenode.perl.  The devleoper obtains a lock at the beginning
                of execution and hold it until the script completes. It is
                guaranteed that two or more instances of the script will NOT be able to 
                run concurrently.

                This is not a blocking lock.  If a script calls get_lock while another
                instance of the same script has not released the lock, the calling script
                will fail. If the script exits for any reason, the lock is released.

=cut

#--------------------------------------------------------------------------------

sub get_lock
{
    my ($class, $file) = @_;
    my $return_code = 0;
    my $program     = NodeUtils->programName();
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 "V", 'IMsgGet_Lock', $file, $program);
    if (!open(SEM, ">$file"))
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',      $::MSGMAPPATH,
                                     'csminstall',          "E1",
                                     'EMsgSEMAPHORE_ERROR', $file,
                                     "$? $!"
                                     );
    }
    if (!flock(SEM, LOCK_EX | LOCK_NB))
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall',     "E1",
                                     'EMsgLOCK_ERROR', $program,
                                     $file,            "$? $!"
                                     );
    }
}

#--------------------------------------------------------------------------------

=head3    get_uniqu_arry_elemnts

         Returns the unique elements in an array

        Arguments:
                @array_to_examined
        Returns:
                @array_of_unique_ele
        Globals:
                none
        Error:
                undefined
        Example:
                 @::os_prereq =
                        ServerUtils->get_uniqu_arry_elemnts( @::os_prereq );
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub get_uniqu_arry_elemnts
{
    my ($class, @my_arry) = @_;
    my %my_hsh = ();
    my @uniq   = ();
    my $itm;
    foreach $itm (@my_arry)
    {
        $my_hsh{$itm}++;
    }
    @uniq = keys %my_hsh;
    return @uniq;
}

#--------------------------------------------------------------------------------

=head3    getNodehwithAutoupdateInstalled

        Checks to see if the Autoupdate RPM is installed on nodes

        Arguments:
        	%nodeHash	Node Hash
        Returns:
            %nodehash	Target NodeHash which have autoupdate RPM installed
        Globals:
                none
        Error:
                none
        Example:
         none
        Comments:
                Will DSH to nodes to check if autoupdate RPM is installed.

=cut

#--------------------------------------------------------------------------------
sub getNodewithAutoupdateInstalled
{
    my ($class, $ref_nodehash) = @_;
    my %nodehash = %$ref_nodehash;

    my (@installNodes, @badNodes);
    my $timeout  = 60;
    my @nodelist = keys %nodehash;
    foreach my $node (keys %nodehash)
    {
        my $os      = $nodehash{$node}{'InstallOSName'};
        my $distro  = $nodehash{$node}{'InstallDistributionName'};
        my $default_dir = "/csminstall"; 
        my $dir = "$default_dir/$os/$distro/csm/";

        my $cmd = "find $dir |grep autoupdate";
        my @result = NodeUtils->runcmd($cmd, -1);
        my $exit = $::RUNCMD_RC;
        if (($exit != 0) && ($exit != 1)){
            MessageUtils->message('E255', 'EMsgCANT_RUN', $cmd,"@result $exit");
        }
        else{
            if ($exit == 0){
                push @installNodes, $node;
            }
            else{
                my %options_api = ();
                $options_api{'command'} = "$::RPMCMD -q autoupdate";
                $options_api{'timeout'} = $timeout;
                $options_api{'nodes'} = $node;
                my $result = NodeUtils->runDsh(\%options_api, -1);

                # Sucessful nodes;
                if ($result !~ /not installed/ && $result =~ /autoupdate/){
                    push @installNodes, $node;
                }
           }
        }
     }  

    # Delete node without autoupdate installed fron %nodehash
    foreach my $node (@nodelist)
    {
        if (!grep(/^$node$/, @installNodes))
        {
            push @badNodes, $node;
            delete $nodehash{$node};
        }
    }

    # Print warning Messages
    if (@badNodes)
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'I',
                                     'IMsgNo_Autoupdate_Installed',
                                     join(',', @badNodes)
                                     );
    }
    return \%nodehash;
}

#--------------------------------------------------------------------------------

=head3    isAutoupdateInstalled

        Checks to see if the Autoupdate RPM is installed on a Management Server

        Arguments:
        nodeHash
        Returns:
                returns 1 if Autoupdate is installed
            0 otherwise
        Globals:
                none
        Error:
                none
        Example:
         none
        Comments:
                This make sense only on a managment server.

=cut

#--------------------------------------------------------------------------------

sub isAutoupdateInstalled
{
    my ($class, $destnodekey) = @_;
    my $exit = 1;    #assume it is installed
      #Autoupdate should be in /csminstall/Linux/<InstallDistributionName>/csm/<InstallCSMVersion>/packages/
      #EX: /csminstall/Linux/RedHat/csm/1.3.0/packages/autoupdate-4.0.1-1.noarch.rpm

    #determine possible directory structures
    #If there is no rpm in installDir & updateDir, Autoupdate won't be checked.
    my %directories = ();
    my %rpm_directories = ();
    my ($dir,$ostype,$distro,$version,$csmversion,$insmethod,$arch,$sl,$update_dir,$base_dir,$install_dir);
    foreach my $node (keys %$destnodekey)
    {
	    if ($$destnodekey{$node}{'InstallOSName'} eq "Linux")
	    {
		    $ostype  = $$destnodekey{$node}{'InstallOSName'};
		    $distro  = $$destnodekey{$node}{'InstallDistributionName'};
		    $version = $$destnodekey{$node}{'InstallDistributionVersion'};
		    $arch    = $$destnodekey{$node}{'InstallPkgArchitecture'};
		    $sl      = $$destnodekey{$node}{'InstallServiceLevel'};
		    if ($insmethod =~ /warewulf/)
		    {
#Do not print the Autoupdate missing message for diskless nodes
			    return 0;
		    }
		    $update_dir =
		    	ServerUtils->getUpdateDir($ostype, $distro, $version,$arch, $sl);
		    $rpm_directories{$update_dir} = 1;

		    $base_dir =
		    	ServerUtils->getBaseDir($ostype, $distro, $version,$arch, $sl);
		    $install_dir = "$base_dir/install/"; 
		    $rpm_directories{$install_dir} = 1;
	    }
	    my $notfound = 1;	
	    foreach my $dir (keys %rpm_directories)
	    {
		    if (!-d $dir || (!-f glob("$dir/*.rpm") && !-f glob("$dir/*/*.rpm")))
		    {
			    my ( $nothin, $csminstall, $ostype, $osdistr, $osversion,
				    $arch,   $sl_dir, $end_dir)= split "/", $dir;
			    $dir = "/$csminstall/$ostype/$osdistr/$osversion/$arch/$end_dir";
			    if ((-f glob("$dir/*.rpm")) || (-f glob("$dir/*/*.rpm")))
			    {
				    $notfound = 0;
			    }
		    }else
		    {
			    $notfound = 0;
		    }
	    }

	    if ($notfound)
	    {
		    return 0;
	    }
        
	
	if ($$destnodekey{$node}{'InstallOSName'} eq "Linux")
        {
            $distro  = $$destnodekey{$node}{'InstallDistributionName'};
            $csmversion = $$destnodekey{$node}{'InstallCSMVersion'};
            $insmethod = $$destnodekey{$node}{'InstallMethod'};
            if ($insmethod =~ /warewulf/)
            {
                #Do not print the Autoupdate missing message for diskless nodes
                return 0;
            }
            if ($csmversion >= 1.3)
            {
                $dir = "/csminstall/Linux/$distro/csm/$csmversion/packages/";
                $directories{$dir} = 1;
            }
        }
    }

    #check if there is an autoupdate rpm in the directory
    foreach my $dir (keys %directories)
    {
        `/bin/ls $dir/autoupdate*.rpm 2>/dev/null`;
        my $rc = $? >> 8;
        if ($rc != 0)
        {    #autoupdate isn't there
            MessageUtils->messageFromCat(
                      'csmInstall.cat',                           $::MSGMAPPATH,
                      'csminstall',                               'E',
                      'EMsgNoAutoupdate',                         $dir,
                      "http://freshmeat.net/projects/autoupdate", $dir
                      );
            $exit = 0;
        }
    }
    return $exit;
}

#--------------------------------------------------------------------------------

=head3 release_lock

        Releases a non-blocking semiphore /  file

        Arguments:
                $fileName
        Returns:
                2  if it couldn't open $fileName
                3  if it couldn't lock $fileName
        Error:
                messageFromCat E1
        Example:
                ServerUtils->get_lock("/tmp/updateisvr.sem");_
        Comments:
                get_lock() and release_lock() implement semiphore locking for csm comnands
                such as updatenode.  The develeoper obtains a lock at the beginning
                of execution and hold it until the script completes. It is
                guaranteed that two or more instances of the script will NOT be able to 
                run concurrently.

                This is not a blocking lock.  If a script calls get_lock while another
                instance of the same script has posession of the lock, the calling script
                will fail. If a script exits for any reason, the lock will be released.

=cut

#--------------------------------------------------------------------------------

sub release_lock
{
    my ($class, $file) = @_;

    close(SEM);
    if (-e "$file")
    {
        system("/bin/rm -f $file");
    }
}

#--------------------------------------------------------------------------------

=head2    Cluster Layer

=cut

#--------------------------------------------------------------------------------

=head3    makeSyncHash

        Returns a hash of which directories to distribute to which 
	Install Servers.  
	
	Assigns a value to the global: 
	%::INSTALL_SERVERS - this is an array of the host names of all 
	the install servers in the cluster.                                              

        Arguments:
                %DestNodeHash (where InstallServer is one of the attributes).
                $InstallServerHash
                $special_ref : If this argument is specified, only files and directories included in @$special_ref 
                               will be synced to InstallServer)
        Returns:
                Hash of directory name arrays by install server keys
        Globals:
                %::INSTALL_SERVERS
        Error:
                undefined
        Example:
                 my $sHash =
			ServerUtils->makeSyncHash($DestNodeHash,$InstallServerHash,$special_ref);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub makeSyncHash
{
    my $routine = "makeSyncHash";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $DestNodeHash, $InstallServerHash, $special_ref) = @_;
    my $defDir = $::IS_DEFAULT_DIR;    #/csmserver

    my %sHash;                         # return hash
    my $csm_skip_isvr;
    $::MS_INSTALL_SERVER = 0;

    # Set value to skip installserver
    if (defined $ENV{'CSM_SKIP_ISVR'} && $ENV{'CSM_SKIP_ISVR'} != 0)
    {
        $csm_skip_isvr = 1;
    }

    foreach my $node (keys %$DestNodeHash)
    {
        ($::DEBUG) && print ">>>>>NODE IN DestNodeHash:  $node\n";
        my @installservers;

        my ($server, $nfs, $dir) =
          NodeUtils->getInstallServerAttribute($$DestNodeHash{$node});

        # No IS/NFSServer defined, remove the MS_INSTALL_SERVER in future
        if (!$server and !$nfs)
        {
            $::MS_INSTALL_SERVER = 1;
            next;
        }
        if (!$dir)
        {
            $dir = $defDir;
        }

        if ($server)
        {
            my $ref_grp = NodeUtils->getNodegrp($server);
            if ($ref_grp)
            {
                foreach my $iserver (@$ref_grp)
                {
                    if (grep(/$iserver/, keys %$InstallServerHash))
                    {
                        push @installservers, $iserver;
                    }
                }
            }
            else
            {
                @installservers = ($server);
            }
        }
        else
        {
            @installservers = ($nfs);
        }

        if (defined($special_ref))
        {
            foreach my $iserver (@installservers)
            {
                foreach my $specialFileORDir (@$special_ref)
                {
                    push @{$sHash{$specialFileORDir}{$dir}}, $iserver;
                }
            }
            next;
        }

        if ($csm_skip_isvr)
        {
            foreach my $iserver (@installservers)
            {
                push @{$sHash{"/csminstall/csm/config"}{$dir}},      $iserver;
                push @{$sHash{"/csminstall/csm/osprereboot"}{$dir}}, $iserver;
                if ($$DestNodeHash{$node}{'InstallOSName'} eq 'Linux')
                {
                    #template files are not valid on AIX
                    push @{$sHash{"/csminstall/csm/templates/tmpl"}{$dir}},
                    $iserver;
                }
            }
        }
        else
        {

            #find node attributes
            my $InstallOSName = $$DestNodeHash{$node}{'InstallOSName'};
            my $InstallDistributionName =
              $$DestNodeHash{$node}{'InstallDistributionName'};
            my $InstallDistributionVersion =
              $$DestNodeHash{$node}{'InstallDistributionVersion'};
            my $InstallServiceLevel =
              $$DestNodeHash{$node}{'InstallServiceLevel'};
            my $InstallPkgArchitecture =
              $$DestNodeHash{$node}{'InstallPkgArchitecture'};
            my $InstallMethod = $$DestNodeHash{$node}{'InstallMethod'};

            foreach my $iserver (@installservers)
            {
                push @{$sHash{"/csminstall/csm"}{$dir}}, $iserver;
                if ($InstallOSName eq "Linux")
                {
                    my $iInstallOSName =
                      $$InstallServerHash{$iserver}{'InstallOSName'};
                    my $iInstallDistributionName =
                      $$InstallServerHash{$iserver}{'InstallDistributionName'};
                    my $iInstallDistributionVersion =
                      $$InstallServerHash{$iserver}
                      {'InstallDistributionVersion'};
                    my $iInstallPkgArchitecture =
                      $$InstallServerHash{$iserver}{'InstallPkgArchitecture'};
                    push
                      @{$sHash{"/csminstall/Linux/$InstallDistributionName/csm"}
                          {$dir}}, $iserver;
                    push @{
                        $sHash{
                            "/csminstall/Linux/$InstallDistributionName/$InstallDistributionVersion/$InstallPkgArchitecture"
                          }{$dir}
                      },
                      $iserver;
                    if($InstallMethod eq "warewulf")
					{
#					   	push @{
#							$sHash{
#								"$::VNFSDIR/$InstallDistributionName$InstallDistributionVersion-$InstallServiceLevel-$InstallPkgArchitecture"
#							}{$dir}
#							},
#							$iserver;
					}
                    if ($InstallMethod eq "sis")
                    {
                        my $image = $$DestNodeHash{$node}{'InstallImage'}; 
                        my $image_dir = "/csminstall/sis/images/" . "$image";
                        push @{$sHash{$image_dir}{$dir}}, $iserver;
                    }
                    
                    if (
                        ($iInstallOSName ne $InstallOSName)
                        || ($iInstallDistributionName ne
                            $InstallDistributionName)
                        || ($iInstallDistributionVersion ne
                            $InstallDistributionVersion)
                        || ($iInstallPkgArchitecture ne $InstallPkgArchitecture)
                      )
                    {
                        push @{
                            $sHash{
                                "/csminstall/Linux/$iInstallDistributionName/csm"
                              }{$dir}
                          },
                          $iserver;
                        push @{
                            $sHash{
                                "/csminstall/Linux/$iInstallDistributionName/$iInstallDistributionVersion/$iInstallPkgArchitecture"
                              }{$dir}
                          },
                          $iserver;
                    }
                }    # if "Linux"
            }    # End of foreach my $iserver

            #transfer files for diskless node
            if (   $::DISKLESS_PM
                && ($InstallMethod eq "diskless")
                && !$csm_skip_isvr)
            {
                my $disklessAttr = DisklessUtils->getDisklessAttrs($node);
                my %da           = %$disklessAttr;
                push @{
                    $sHash{
                        "/csminstall/diskless/$InstallOSName/$InstallDistributionName-$InstallDistributionVersion-$InstallPkgArchitecture/images/readonly"
                      }{$dir}
                  },
                  $server;
                push @{
                    $sHash{
                        "/csminstall/diskless/$InstallOSName/$InstallDistributionName-$InstallDistributionVersion-$InstallPkgArchitecture/images/$da{Component}"
                      }{$dir}
                  },
                  $server;
                push @{
                    $sHash{
                        "/tftpboot/zImage.$InstallDistributionName-$InstallDistributionVersion-$InstallPkgArchitecture"
                      }{"/tftpboot"}
                  },
                  $server;
            }
        }    # if $csm_skip_isvr
    }
    ($::DEBUG)
      && print
      "MS_INSTALL_SERVER (Is the management server an install server?) = $::MS_INSTALL_SERVER\n";
    print "LEAVING: $routine\n" if $::DEBUG;
    return (\%sHash);
}

#--------------------------------------------------------------------------------

=head3    createSyncHash

        Returns a hash of which directories to distribute to which 
	Install Servers.  
	

        Arguments:
                %DestNodeHash (where InstallServer is one of the attributes).
        Returns:
        Globals:
        Error:
                undefined
        Example:
                 my $sHash =
                   ServerUtils->createSyncHash($DestNodeHash);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub createSyncHash
{
    my $routine = "createSyncHash";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $ref_DestNodeHash) = @_;

    return \%::NODE_ISVR_HASH if (%::NODE_ISVR_HASH);

    my %DestNodeHash   = %$ref_DestNodeHash;
    my %node_isvr_hash = %DestNodeHash;
    my @nodes          = keys %node_isvr_hash;
    my @is = ServerUtils->getInstallNFSServer(\@nodes, \%node_isvr_hash);

    if (!scalar(@is))
    {
        print "LEAVING: $routine\n" if $::DEBUG;
        return \%node_isvr_hash;
    }

    my $lsnode_attr =
      "Hostname,ManagementServer,InstallOSName,InstallCSMVersion,InstallDistributionVersion,InstallDistributionName,InstallPkgArchitecture,Mode,InstallAdapterMacaddr,InstallAdapterType,InstallMethod,ConsoleSerialDevice,Status,PowerStatus,InstallServer,NFSServer";

    my $ref_lsnode_info = NodeUtils->listNodeAttrs(\@is, $lsnode_attr);
    my $check_delim = ':\|:';    # Used when splitting the line apart
    foreach my $line (@$ref_lsnode_info)
    {    #this only has node names in it from the cluster

        my @attributes = split /$check_delim/, $line;
        my $hostname = $attributes[0];
        if (grep (/^$hostname$/, @nodes))
        {

            # Already in list;
            next;
        }

        chomp $line;

        my @attr_names = split ',', $lsnode_attr;
        my $index = 0;
        foreach my $attr (@attr_names)
        {
            $node_isvr_hash{$hostname}{$attr} = $attributes[$index];
            $index++;
        }
    }

    print "LEAVING: $routine\n" if $::DEBUG;

    # Cache %::NODE_ISVR_HASH
    if (!%::NODE_ISVR_HASH) { %::NODE_ISVR_HASH = %node_isvr_hash }

    return \%node_isvr_hash;

}

#--------------------------------------------------------------------------------

=head3    seperateNFSServerFromInstallServer

        Seperate  NFSServer from InstallServer Hash

        Arguments:
			%nodehash:	the target node hash
			%ishash:	InstallServer Hash
        Returns:
			@non_isnode:	the node which don't have InstallServer
			@isnode:		the node which have InstallServer
        Globals:
			None
        Error:
                undefined
        Example:
			my ($ref_non_isnode, $ref_isserver) = 
				seperateNFSServerFromInstallServer(\%NodeHash, \%ISHash);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub seperateNFSServerFromInstallServer
{
    my ($class, $ref_nodehash, $ref_ishash) = @_;
    my (@non_isnode, @isnode);
    my @installservers = keys %$ref_ishash;

    # Get InstallServer first in case if some nodes are both InstallServer and NFSServer
    foreach my $node (keys %$ref_nodehash)
    {
        my ($is, $nfs, $dir) =
          NodeUtils->getInstallServerAttribute($$ref_nodehash{$node});
        if ($is)
        {
            my $ref_grp = NodeUtils->getNodegrp($is);
            if ($ref_grp)
            {
                foreach my $iserver (@$ref_grp)
                {
                    if (grep(/$iserver/, @installservers))
                    {
                        push @isnode, $iserver;
                    }
                }
            }
            else
            {
                push @isnode, $is;
            }
        }
        else
        {
            push @non_isnode, $node;
        }
    }
    return (\@non_isnode, \@isnode);
}

#-------------------------------------------------------------------------------

=head3    enable_services

        Installs and enables network and rdm services for the nodes' install
	servers.  The nodes are listed in $ref_nodehash.  The install servers
	are listed in $ref_ishash.

        Arguments:
                 Reference to a nodeName Hash;
				 Reference to Node's InstallServer hash;
        Returns:
                0 on error
        Globals:
        Error:
                0 - there are no install servers in the cluster.
        Example:
                 ServerUtils->enable_services(\%DestNodeHash, \%ISHash);
        Comments:

=cut

#-------------------------------------------------------------------------------

sub enable_services
{
    my $routine = "enable_services";
    print "ENTERING: $routine\n" if $::DEBUG;
    my ($class, $ref_nodehash, $ref_ishash) = @_;

    my ($non_isnode, $is_nodes) =
      ServerUtils->seperateNFSServerFromInstallServer($ref_nodehash,
                                                      $ref_ishash);
    my %nodehash = %$ref_nodehash;
    my %ishash   = %$ref_ishash;

    my @AIX_servers;
    my @Linux_servers;
    my @PXE_servers;
    my @BOOTP_servers;
    my @InstallServers = keys %ishash;

    if ($::DEBUG)
    {
        foreach my $s (@InstallServers)
        {
            print "INSTALL SERVER: $s\n";
        }
    }

    # is it possible that we have multiple mgmtsvr? Migration?
    my @mgmtsvr_hostnames;
    if (@$non_isnode)
    {
        foreach my $node (@$non_isnode)
        {
            my $mgmtsvr = $nodehash{$node}{'ManagementServer'};
            if (!grep /^$mgmtsvr/, @mgmtsvr_hostnames)
            {
                push(@mgmtsvr_hostnames, $mgmtsvr);
            }
        }
    }

    # If there are install servers other than the management server,
    # add them to the appropriate *_servers list(s).
    if (@$is_nodes or @$non_isnode)
    {
        foreach my $server (@$is_nodes)
        {
            my $os     = $ishash{$server}{'InstallOSName'};
            my $distro = $ishash{$server}{'InstallDistributionName'};
            my $arch   = $ishash{$server}{'InstallPkgArchitecture'};
            my $dir    = $ishash{$server}{'Dir'};

            if ($os =~ /AIX/)
            {
                push @AIX_servers, $server;
            }
            elsif ($os eq "Linux")
            {
                push @Linux_servers, $server;
            }
            else
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgINVALID_OSTYPE');
            }
        }

        # Check if any GPL code needs to be installed on the Linux install
        # servers.  If so, display a license and prompt the user to accept it.
        my @check_servers;
        push(@check_servers, @mgmtsvr_hostnames) if (@mgmtsvr_hostnames);
        if (scalar(@check_servers) > 0)
        {
            ArchiveUtils->check_for_display_opensrcLicense(\@check_servers);
        }

        # Determine the value of the TFTPpackage attribute
        my $mgmtsvr;
        my $dir;
        my $TFTPpackage =
          `CT_MANAGEMENT_SCOPE=1 lsrsrc-api -i -s IBM.DmsCtrl::::TFTPpackage 2>/dev/null`;
        chomp $TFTPpackage;

        # Determine whether an HTTP server should be set up on the
        # install servers.  Set $SETUP_HTTP_SERVER_FLAG if
        # SetupNetworkInstallProtocol=1 and NetworkInstallProtocol=HTTP
        my $SETUP_HTTP_SERVER_FLAG = "";
        if (    (NodesetUtils->get_NetworkInstallProtocol =~ /http/i)
            and (NodesetUtils->get_SetupNetworkInstallProtocol =~ /1|yes/i))
        {
            $SETUP_HTTP_SERVER_FLAG = "-H";
        }

        # Set the verbose flag depending on how this command was called
        my $verboseflag = "";
        $verboseflag = "-v" if ($::VERBOSE);

        # Call syncServers.client on the management server if the MS is
        # an install server.
        if (@mgmtsvr_hostnames)
        {

            # The management server might be known by different hostnames
            # on different nodes.
            foreach my $mgmtsvr (@mgmtsvr_hostnames)
            {
                $dir = $::CSMINSTDIR;
				#If undefined $TFTPpackage attribute, it means its value is blank.
				#Remove this attribute from command string
                my $cmd;
				if (!(defined $TFTPpackage) || ($TFTPpackage eq ""))
				{
	           	   $cmd =
                  "LANG=C $::SYNCSERVERS_CLIENT -m $mgmtsvr -i $mgmtsvr -p $dir $SETUP_HTTP_SERVER_FLAG $verboseflag";
				
				} 
				else
				{
                	$cmd =
                  "LANG=C $::SYNCSERVERS_CLIENT -m $mgmtsvr -i $mgmtsvr -p $dir -t $TFTPpackage $SETUP_HTTP_SERVER_FLAG $verboseflag";
				}
                MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                             'NodeUtils', 'V', 'IMsgCMD', $cmd);
                open(DSHCMD, "$cmd|")
                  || MessageUtils->messageFromCat('csmInstall.cat',
                          $::MSGMAPPATH, 'csminstall', 'E2', 'EMsgCANT_RUN_CMD',
                          $cmd);

                while (<DSHCMD>)
                {
                    print $_;
                }
                close(DSHCMD);
            }
        }

        # Call syncServers.client on the Linux install servers
        foreach my $installsvr (@Linux_servers)
        {
            chomp $installsvr;

            # Get the hostname of the management server as known by
            # the install server.
            if ($ishash{$installsvr}{'ManagementServer'})
            {
                $mgmtsvr = $ishash{$installsvr}{'ManagementServer'};
            }
            else
            {

                # Determine the name of the management server as known by the
                # install server.
                $mgmtsvr = NetworkUtils->get_management_server($installsvr);
            }
            chomp $mgmtsvr;

            $dir = $ishash{$installsvr}{'Dir'};

			#If undefined $TFTPpackage attribute, it means its value is blank.
			#Remove this attribute from command string
            my $cmd;
			if (!(defined $TFTPpackage) || ($TFTPpackage eq ""))
			{
               $cmd =
              "LANG=C $::DSH -s -n $installsvr \"$::SYNCSERVERS_CLIENT -m $mgmtsvr -i $installsvr -p $dir $SETUP_HTTP_SERVER_FLAG $verboseflag\"";
	
			}
			else
			{
               $cmd =
              "LANG=C $::DSH -s -n $installsvr \"$::SYNCSERVERS_CLIENT -m $mgmtsvr -i $installsvr -p $dir -t $TFTPpackage $SETUP_HTTP_SERVER_FLAG $verboseflag\"";
			}
           $cmd .= ' 2>&1';
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cmd);
            open(DSHCMD, "$cmd|") || die 
            MessageUtils->messageFromCat('csmInstall.cat', 
                    $::MSGMAPPATH, 'csminstall', 'E1', 
                    'EMsgCANT_RUN_CMD', $cmd);

            while (<DSHCMD>)
            {
                print $_;
            }
            close(DSHCMD);
        }
    }
    print "LEAVING: $routine\n" if $::DEBUG;
}

#-------------------------------------------------------------------------------

=head3    syncServers

        Syncronize a cluster's Install Servers. 

        Arguments:
				Reference to a nodeName Hash; 
				Reference to node's InstallServer Hash;
				Reference to Dir need to be excluded;
				Reference to Dirs or files that  need sync;
        Returns:
                0 on error
        Globals:
                None
        Error:
                0 - there are no install servers in the cluster.
        Example:
                 ServerUtils->syncServers(\%DestNodeHash, \%ISHash,\@exclude,\@sepcial);
        Comments:
                 The caller of this function must use CFM. 

=cut

#-------------------------------------------------------------------------------

sub syncServers
{

    #use CFM -- all commands that call this function need to use CFM
    my $routine = "syncServers";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $ref_nodehash, $ref_ishash, $exclude_ref, $special_ref) = @_;
    my (@exclude_dirs, $dir_str, @ex_dirs);

    my @is = keys %$ref_ishash;
    if (!@is)
    {    # no install servers
        print "LEAVING: $routine\n" if $::DEBUG;
        return 0;
    }

    # Check InstallServer's reachability
    my $unreachable = NetworkUtils->checkRSReachability(\@is);
    if ($unreachable != 0)
    {
        if (scalar @$unreachable >= 1)
        {

            #there are unreachable InstallServers
            my $isvrs = join ",", @$unreachable;
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgISDown', $isvrs);
        }
    }

    my $sHash =
      ServerUtils->makeSyncHash($ref_nodehash, $ref_ishash, $special_ref);

    # Allow the user to set the CSM_SKIP_ISVR environment variable
    # in order to skip the transfer of files to the install server.
    if (!defined $ENV{'CSM_SKIP_ISVR'})
    {

        # Exclude the autoyast and kickstart config file and their links
        my $ref_ex_dirs = ServerUtils->exclude_config_dir();
        if ($ref_ex_dirs)
        {
            push @exclude_dirs, @$ref_ex_dirs;
        }

        # Exclude the dir passed in
        if ($exclude_ref) { push @exclude_dirs, @$exclude_ref; }
    }

	my $ref_ex_dirs = DCUtils->exclude_dev_dirs($ref_nodehash);
	if(@$ref_ex_dirs)
	{
		push @exclude_dirs, @$ref_ex_dirs;	
	}		

    # For pRedHat, there might have a link "/csminstall/csm/RedHatEL-AS*-drivers"
    # Exclude them but create the link in syncServer.client
    if (-l "/csminstall/csm/RedHatEL-AS3-drivers")
    {
        push @exclude_dirs, "/csminstall/csm/RedHatEL-AS3-drivers";
    }
    if (-l "/csminstall/csm/RedHatEL-AS4-drivers")
    {
        push @exclude_dirs, "/csminstall/csm/RedHatEL-AS4-drivers";
    }

    my $distro       = NodeUtils->get_DistributionName();
    my $distlocation = "/tmp/csm_servers";
    CFM->createGenDistfile($sHash, $distlocation, \@exclude_dirs);
    CFM->distGenDistfile($distlocation);
    unlink $distlocation;

    print "LEAVING: $routine\n" if $::DEBUG;
}

#--------------------------------------------------------------------------------

=head3    checkCrossArchIS

        Give error message when a node and its Install Server have different InstallPkgArchitecture
        attributes. 

        Arguments:
                Reference to NodeHash, reference to InstallServer Hash;
        Returns:
		None.
        Error:
                None.
        Example:
                 ServerUtils->checkCrossArch(\%NodeHash, \%ISHash);
=cut

#--------------------------------------------------------------------------------
sub checkCrossArchIS()
{
    my ($class, $DestNodeHash, $ISHash) = @_;
    my @nodelist = keys %$DestNodeHash;

    foreach my $node (@nodelist)
    {
        my ($InstallServer) = split ":", $$DestNodeHash{$node}{'InstallServer'};

        my $InstallPkgArchitecture =
          $$DestNodeHash{$node}{'InstallPkgArchitecture'};
        my $iInstallPkgArchitecture;
        if ($InstallServer)
        {
            my $ref_grp = NodeUtils->getNodegrp($InstallServer);
            if ($ref_grp)
            {

                # By design all Install Servers have the same architecture.
                $InstallServer = $$ref_grp[0];
            }
            $iInstallPkgArchitecture =
              $$ISHash{$InstallServer}{'InstallPkgArchitecture'};
            if(!$iInstallPkgArchitecture)
            {
                my ($IS_full, $IS_ip)=NetworkUtils->getHost($InstallServer);
                $iInstallPkgArchitecture =
                        $$ISHash{$IS_full}{'InstallPkgArchitecture'};
            }
        }
        else
        {
            $iInstallPkgArchitecture = ArchiveUtils->get_PkgArchitecture;
        }

		# get_PkgArchitecture() return undef if MS is AIX
		if (!defined($iInstallPkgArchitecture)) {
			# this subroutine only called by csmsetupks and csmsetupyast
			# so linux to be installed.
			# AIX couldn't install linux, you should make a IS
			MessageUtils->messageFromCat(
				       'csmInstall.cat',			$::MSGMAPPATH,
				       'csminstall',				'E1',
				       'EMsgCrossArchNotSupported',	$node,
				       'Linux',						'AIX'
				       );
		}	

        if (   ($iInstallPkgArchitecture ne "ppc64")
            && ($InstallPkgArchitecture eq "ppc64"))
        {

            # xSeries IS to install pSeries node, only allowed for p-blade.
            my $PowerMethod = $$DestNodeHash{$node}{'PowerMethod'};
            if ($PowerMethod ne "blade")
            {
                MessageUtils->messageFromCat(
                           'csmInstall.cat',            $::MSGMAPPATH,
                           'csminstall',                'E1',
                           'EMsgCrossArchNotSupported', $node,
                           $InstallPkgArchitecture,     $iInstallPkgArchitecture
                           );
            }

        }
        elsif (   ($iInstallPkgArchitecture eq "ppc64")
               && ($InstallPkgArchitecture ne "ppc64"))
        {
            MessageUtils->messageFromCat(
                           'csmInstall.cat',            $::MSGMAPPATH,
                           'csminstall',                'E1',
                           'EMsgCrossArchNotSupported', $node,
                           $InstallPkgArchitecture,     $iInstallPkgArchitecture
                           );
        }

    }
}
#--------------------------------------------------------------------------------

=head3    checkCrossDistroIS

        Give error message when a node and its Install Server have different distro
        name or version.
        Note that this subroutine should be called only for Linux diskless
                   installation scenario.
        Arguments:
                Reference to NodeHash, reference to InstallServer Hash;
        Returns:
		None.
        Error:
                None.
        Comments:
            # Check cross diskless installation legal for linux diskless installation.
            # So far, the cross arch and cross distribution version installation are not supported 
            # at most situations. 
            # But there is a special case for RedHatEL-Server5 and RedHatEL-Client5. 
            # They can be installed crossly.

        Example:
                 ServerUtils->checkCrossDistroIS(\%NodeHash, \%ISHash);
=cut

#--------------------------------------------------------------------------------
sub checkCrossDistroIS()
{
    my ($class, $DestNodeHash, $ISHash) = @_;
    my @nodelist = keys %$DestNodeHash;

    foreach my $node (@nodelist)
    {
        my ($InstallServer) = split ":", $$DestNodeHash{$node}{'InstallServer'};

        my $InstallPkgArchitecture =
          $$DestNodeHash{$node}{'InstallPkgArchitecture'};
        my $InstallDistributionName = 
          $$DestNodeHash{$node}{'InstallDistributionName'};
        my $InstallDistributionVersion =
          $$DestNodeHash{$node}{'InstallDistributionVersion'};
        my ($iInstallPkgArchitecture, $iInstallDistributionName, $iInstallDistributionVersion);

        if ($InstallServer)
        {
            my $ref_grp = NodeUtils->getNodegrp($InstallServer);
            if ($ref_grp)
            {

                # By design all Install Servers have the same architecture.
                $InstallServer = $$ref_grp[0];
            }
            $iInstallPkgArchitecture =
              $$ISHash{$InstallServer}{'InstallPkgArchitecture'};
            $iInstallDistributionName = 
              $$ISHash{$InstallServer}{'InstallDistributionName'};
            $iInstallDistributionVersion =
              $$ISHash{$InstallServer}{'InstallDistributionVersion'};
            if(!$iInstallPkgArchitecture)
            {
                my ($IS_full, $IS_ip)=NetworkUtils->getHost($InstallServer);
                $iInstallPkgArchitecture =
                        $$ISHash{$IS_full}{'InstallPkgArchitecture'};
            }
        }
        else
        {
            $iInstallPkgArchitecture = ArchiveUtils->get_PkgArchitecture;
            $iInstallDistributionName = NodeUtils->get_DistributionName();
            $iInstallDistributionVersion = NodeUtils->get_DistributionVersion();
        }
        # Check cross distribution name
        # Special case: allow RedHatEL-Client5 and RedHatEL-Server5 cross installation
        # RedHatEL-Client5 <=> RedHatEL-Server5
        if (  (    $iInstallDistributionName =~ /RedHatEL-Server/ && $iInstallDistributionVersion =~ /^5/ 
               &&  $InstallDistributionName  =~ /RedHatEL-Client/ && $InstallDistributionVersion =~ /^5/ 
              )
            ||(    $iInstallDistributionName =~ /RedHatEL-Client/ && $iInstallDistributionVersion =~ /^5/ 
               &&  $InstallDistributionName  =~ /RedHatEL-Server/ && $InstallDistributionVersion =~ /^5/
              )
          )
          {
              return;
          }
          # Check other scenario
          elsif ( $iInstallDistributionName ne  $InstallDistributionName)
          {
            MessageUtils->messageFromCat(
                           'csmInstall.cat',            $::MSGMAPPATH,
                           'csminstall',                'E1',
                           'EMsgCrossDistroNotSupported', $node,
                           $InstallDistributionName,     $iInstallDistributionName
                           );

          }
          # If user supplied images, Cross distribution version is permitted.
          elsif ( $::NODEHASH{$node}{"DisklessType"} =~ /user-ramdisk/)
          {
              return;
          }
          # Truncate distribution version to integer to deal with '.' version e.g. RHEL4.5,RHEL5.1
          elsif ( int($iInstallDistributionVersion) ne int($InstallDistributionVersion))
          {
            MessageUtils->messageFromCat(
                           'csmInstall.cat',            $::MSGMAPPATH,
                           'csminstall',                'E1',
                           'EMsgCrossDistroVerNotSupported', $node,
                           $InstallDistributionName,     $InstallDistributionVersion,
                           $iInstallDistributionName,    $iInstallDistributionVersion
                           );

          }
           

    }
}

#--------------------------------------------------------------------------------

=head3    getInstallServers

        Get a cluster's Install Servers. 

        Arguments:
                Reference to a nodeName Hash;
        Returns:
		Reference to mgt nodes array and installServer Hash
        Error:
                Reference to an empty Hash - there are no install servers in the cluster.
        Example:
                 ServerUtils->getInstallServers(\%DestNodeHash);
=cut

#--------------------------------------------------------------------------------

sub getInstallServers
{
    my $routine = "getInstallServers";
    print "ENTERING: $routine\n" if $::DEBUG;
    my ($class, $ref_dest_nodes_hash) = @_;

    my %dest_nodes_hash = %$ref_dest_nodes_hash;
    my %InstallServerAttr;
    my @MgrAttr;

    # for each node to be nimd
    foreach my $node (keys %dest_nodes_hash)
    {
        next
          if (   !defined $dest_nodes_hash{$node}{"ManagementServer"}
              || !defined $dest_nodes_hash{$node}{"Hostname"});

        my ($ms_hostname, $ms_ip) =
          NetworkUtils->getHost_rveg(
                                   $dest_nodes_hash{$node}{"ManagementServer"});
        if ($::GETHOST_RC != $::OK)
        {
            MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                   'NodeUtils', 'E', $::GETHOST_MSG,
                                   $dest_nodes_hash{$node}{"ManagementServer"});
            next;
        }
        my ($node_hostname, $node_ip) =
          NetworkUtils->getHost_rveg($dest_nodes_hash{$node}{"Hostname"});
        if ($::GETHOST_RC != $::OK)
        {
            MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                         'NodeUtils', 'E', $::GETHOST_MSG,
                                         $dest_nodes_hash{$node}{"Hostname"});
            next;
        }

        # skip undefined and illegal node
        my $servname = $dest_nodes_hash{$node}{"InstallServer"};

        if (!$servname)
        {

            # handle node without install server
            push(@MgrAttr, $node);
            next;
        }

        # handle node with install server
        my ($server, $dir) = split ':', $servname;
        if (!$server)
        {
            push(@MgrAttr, $node);
            next;
        }
        my $ref_grp;
        my $isISGrp = 0;
        $ref_grp = NodeUtils->getNodegrp($server);
        if ($ref_grp)
        {
            $isISGrp = 1;
        }

        my ($hostname, $ipaddr);
        if ($isISGrp)
        {
            $hostname = $server;
            $InstallServerAttr{$hostname}{isGrp} = 1;
        }
        else
        {
            ($hostname, $ipaddr) = NetworkUtils->getHost_rveg($server);
            if ($::GETHOST_RC != $::OK)
            {
                MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                   'NodeUtils', 'E', $::GETHOST_MSG,
                                   $dest_nodes_hash{$node}{"ManagementServer"});
                next;
            }

            if ($ipaddr eq $ms_ip)
            {

                # handle node with the same install server as management server
                # or install server is server as a managed node simultaneously
                push(@MgrAttr, $node);
                next;
            }

            if ($ms_ip eq $node_ip)
            {
                push(@MgrAttr, $node);
                next;
            }
        }

        # handle node with install server different from management server
        if ($dir)
        {

            # fix last character in $dir string
            if (substr($dir, -1) eq "/") { chop($dir); }
        }
        else
        {

            # set default $dir
            $dir = "/csmserver";
        }

        #
        # From the subroutine of NodesetUtils->run_setupBoot() we can see
        # that we cannot use the InstallServerAKBNode attribute here.
        #
        push(@{$InstallServerAttr{$hostname}{nodelist}}, $node);
        push(@{$InstallServerAttr{$hostname}{dir}{$dir}}, $node);

    }

    print "LEAVING: $routine\n" if $::DEBUG;
    return (\@MgrAttr, \%InstallServerAttr);
}

#--------------------------------------------------------------------------------

=head3    getNFSServer

        Get a cluster's NFS Servers.

        Arguments:
                Reference to a nodeName Hash;
        Returns:
                Reference to mgt nodes array and NFSServer Hash
        Error:
                Reference to an empty Hash - there are no NFS servers in the cluster.
        Example:
                 ServerUtils->getNFSServers(\%DestNodeHash);
=cut

#--------------------------------------------------------------------------------

sub getNFSServers
{
    my $routine = "getNFSServers";
    print "ENTERING: $routine\n" if $::DEBUG;
    my ($class, $ref_dest_nodes_hash) = @_;

    my %dest_nodes_hash = %$ref_dest_nodes_hash;
    my %NFSServerAttr;
    my @MgrAttr;

    # for each node to be nimd
    foreach my $node (keys %dest_nodes_hash)
    {
        next
          if (   !defined $dest_nodes_hash{$node}{"ManagementServer"}
              || !defined $dest_nodes_hash{$node}{"Hostname"});

        my ($ms_hostname, $ms_ip) =
          NetworkUtils->getHost_rveg(
                                   $dest_nodes_hash{$node}{"ManagementServer"});
        if ($::GETHOST_RC != $::OK)
        {
            MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                   'NodeUtils', 'E', $::GETHOST_MSG,
                                   $dest_nodes_hash{$node}{"ManagementServer"});
            next;
        }
        my ($node_hostname, $node_ip) =
          NetworkUtils->getHost_rveg($dest_nodes_hash{$node}{"Hostname"});
        if ($::GETHOST_RC != $::OK)
        {
            MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                         'NodeUtils', 'E', $::GETHOST_MSG,
                                         $dest_nodes_hash{$node}{"Hostname"});
            next;
        }

        # skip undefined and illegal node
        my $servname = $dest_nodes_hash{$node}{"NFSServer"};

        if (!$servname && !$dest_nodes_hash{$node}{"InstallServer"})
        {

            # handle node without NFS server
            push(@MgrAttr, $node);
            next;
        }

        # handle node with NFS server
        my ($server, $dir) = split ':', $servname;
        if (!$server)
        {
            push(@MgrAttr, $node);
            next;
        }
        my $ref_grp;
        my $isISGrp = 0;
        $ref_grp = NodeUtils->getNodegrp($server);
        if ($ref_grp)
        {
            $isISGrp = 1;
        }

        my ($hostname, $ipaddr);
        if ($isISGrp)
        {
            $hostname = $server;
            $NFSServerAttr{$hostname}{isGrp} = 1;
        }
        else
        {
            ($hostname, $ipaddr) = NetworkUtils->getHost_rveg($server);
            if ($::GETHOST_RC != $::OK)
            {
                MessageUtils->messageFromCat('nodecmds.cat', '/opt/csm/msgmaps',
                                   'NodeUtils', 'E', $::GETHOST_MSG,
                                   $dest_nodes_hash{$node}{"ManagementServer"});
                next;
            }

            if ($ipaddr eq $ms_ip)
            {

                # handle node with the same NFS server as management server
                # or NFS server is server as a managed node simultaneously
                push(@MgrAttr, $node);
                next;
            }

            if ($ms_ip eq $node_ip)
            {
                push(@MgrAttr, $node);
                next;
            }
        }

        # handle node with NFS server different from management server
        if ($dir)
        {

            # fix last character in $dir string
            if (substr($dir, -1) eq "/") { chop($dir); }
        }
        else
        {

            # set default $dir
            $dir = "/csmserver";
        }

        #
        # From the subroutine of NodesetUtils->run_setupBoot() we can see
        # that we cannot use the InstallServerAKBNode attribute here.
        #
        push(@{$NFSServerAttr{$hostname}{nodelist}}, $node);
        push(@{$NFSServerAttr{$hostname}{dir}{$dir}}, $node);

    }

    print "LEAVING: $routine\n" if $::DEBUG;
    return (\@MgrAttr, \%NFSServerAttr);
}

#--------------------------------------------------------------------------------

=head2    Filesystem Layer

=cut

#--------------------------------------------------------------------------------

=head3 copyBinaries

        This function takes functionality from installms so that the commands
        updatenode, rmnode, and csmsetupnim can ensure that the latest binaries
        are located in the /csminstall/csm directory.

        This function does the following:
                1. create the /csminstall/csm directory if it doesn't exist 
                2. create the /csminstall/csm/script directories 
                3. copy the binaries into /csminstall/csm, including the defs and pkdefs
                   directory, if the timestamp of the files in /csminstall/csm are older
                   than those installed on the system.
                4. copies updateglibc._RedHat80Nodes for RedHat
                5. copies /tftpboot files for Linux 
        Arguments:
                none
        Returns:
                none
        Globals:
                $::PREINSTALLDIR
                $::POSTINSTALLDIR
                $::PREOSUPGRADEDIR
                $::POSTOSUPGRADEDIR
                $::UPDATEDIR
                $::SCRIPTDATADIR
                $::DISKLESSPREBUILDDIR
                $::DISKLESSBOOTDIR
        Error:
                none
        Example:
                ServerUtils->copyBinaries;
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub copyBinaries
{
    my (
        $class,
        $osname,            # eg Linux or AIX (case sensitive)
        $distro_name,       # eg RedHat or SLES (case sensitive)
        $distro_version,    # eg 3 or 5.2
        $hw_arch,           # eg i386, ppc
        $csm_version,       # eg 1.3.0
    ) = @_;

    ServerUtils->copy_csm_scripts(0);    #first copy customization scripts

    my $INSTALLDIR_CSM = "$::CSMINSTDIR/csm";

    #
    # Get information about the OS, distribution, and package
    #
    my %pkginfo;
    ($osname) and $pkginfo{'OSName'} = $osname;
    (!$osname)
      and $pkginfo{'OSName'} = NodeUtils->get_OSName;    #AIX/Linux

    ($distro_name) and $pkginfo{'DistributionName'} = $distro_name;
    (!$distro_name)
      and $pkginfo{'DistributionName'} = NodeUtils->get_DistributionName;

    ($distro_version) and $pkginfo{'DistributionVersion'} = $distro_version;
    (!$distro_version)
      and $pkginfo{'DistributionVersion'} = NodeUtils->get_DistributionVersion;

    ($hw_arch) and $pkginfo{'PkgArchitecture'} = $hw_arch;
    (!$hw_arch)
      and $pkginfo{'PkgArchitecture'} = NodeUtils->get_PkgArchitecture;

    if ($pkginfo{'PkgArchitecture'} =~ /i.86/)
    {
        $pkginfo{'PkgArchitecture'} = "i386";
    }

    ($csm_version) and $pkginfo{'CsmCoreVersion'} = $csm_version;
    (!$csm_version)
      and $pkginfo{'CsmCoreVersion'} = NodeUtils->get_CSMVersion("csm\.core");

    if (length($pkginfo{'CsmCoreVersion'}) == 0)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'E2', 'EMsgNO_CORE');
    }

    #
    # Create the /csminstall/csm directory if it doesn't exist
    # NOTE: This function will not create any operating system specific
    #       directories, such as /csminstall/csm/AIX
    if (!-d "$INSTALLDIR_CSM")
    {
        mkpath($INSTALLDIR_CSM, $::VERBOSE, 0755);
    }

    # make the script sub-directories if they don't exist
    if (!-d "$::PREINSTALLDIR")
    {
        mkpath($::PREINSTALLDIR, $::VERBOSE, 0755);
    }
    if (!-d "$::POSTINSTALLDIR")
    {
        mkpath($::POSTINSTALLDIR, $::VERBOSE, 0755);
    }
    if (!-d "$::PREOSUPGRADEDIR")
    {
        mkpath($::PREOSUPGRADEDIR, $::VERBOSE, 0755);
    }
    if (!-d "$::POSTOSUPGRADEDIR")
    {
        mkpath($::POSTOSUPGRADEDIR, $::VERBOSE, 0755);
    }
    if (!-d "$::UPDATEDIR")
    {
        mkpath($::UPDATEDIR, $::VERBOSE, 0755);
    }
    if (!-d "$::SCRIPTDATADIR")
    {
        mkpath($::SCRIPTDATADIR, $::VERBOSE, 0755);
    }
    if (!-d "$::DISKLESSPREBUILDDIR")
    {
        mkpath($::DISKLESSPREBUILDDIR, $::VERBOSE, 0755);
    }
    if (!-d "$::DISKLESSBOOTDIR")
    {
        mkpath($::DISKLESSBOOTDIR, $::VERBOSE, 0755);
    }
    if (!-d "$INSTALLDIR_CSM/lib")
    {
        mkpath("$INSTALLDIR_CSM/lib", $::VERBOSE, 0755);
    }
    $::DISKLESSDIR =
      "$::CSMINSTDIR/diskless";
    if (!-d "$::DISKLESSDIR")
    {
        mkpath($::DISKLESSDIR, $::VERBOSE, 0755);
    }

    #
    # Store the pkgdefs into a hash
    #
    my %Svrpkgdefs =
      ServerUtils->get_pkgdefs(
                  $pkginfo{'OSName'},              $pkginfo{'DistributionName'},
                  $pkginfo{'DistributionVersion'}, $pkginfo{'PkgArchitecture'},
                  "MgmtServer",                    $pkginfo{'CsmCoreVersion'}
                  );

    #
    # Get the list of binaries to copy
    #
    my @csm_bin_file_csm;

    if (defined $Svrpkgdefs{csm_bin_copy_csm})
    {
        @csm_bin_file_csm = (@{$Svrpkgdefs{csm_bin_copy_csm}});
    }

    foreach my $bin_file (@csm_bin_file_csm)
    {
        my @temper_val  = ();
        my $temper_file = ();
        @temper_val  = split(/\//, $bin_file);
        $temper_file = $temper_val[$#temper_val];
        $temper_file = $INSTALLDIR_CSM . "\/" . $temper_file;
        if (-e $temper_file)
        {
            my $exception = ServerUtils->copy_exceptions_rveg($Svrpkgdefs{install_method}, $temper_file);

            if ($exception == $::OK)
            {
                unlink($temper_file);
                NodeUtils->runcmd("$::COPY -pr $bin_file $INSTALLDIR_CSM", -2);
            }
        }
        else
        {
            my $exception = ServerUtils->copy_exceptions_rveg($Svrpkgdefs{install_method},$bin_file);
            
            if ($exception == $::OK)
            {
                NodeUtils->runcmd("$::COPY -pr $bin_file $INSTALLDIR_CSM", -2);
            }
        }

    }

    #
    # copy the files in /opt/csm/install/sample_scripts to /csminstall/csm/scripts
    #
    NodeUtils->runcmd(
                    "$::COPY -p /opt/csm/install/sample_scripts/* $::SCRIPTDIR",
                    0);

    # copy driver customization perl modules to /csminstall/csm/install/drivers
    NodeUtils->runcmd(
            "$::COPY -rp /opt/csm/install/drivers/pm $::CSMDRIVERDIR", 0);

}

#-------------------------------------------------------------------------------

=head3 populate_tftpboot

        This function copies all the required files into /tftpboot on the
	install server.

        Arguments:
                \%pkgdefs:  

        Returns:
                $::OK
                $::NOK
        Globals:
		none
        Error:
		none
        Example:
		ServerUtils->populate_tftpboot(\%pkgdefs);
        Comments:
                none

=cut

#-------------------------------------------------------------------------------
sub populate_tftpboot
{
    my $routine = "populate_tftpboot";
    print "ENTERING: $routine\n" if $::DEBUG;
    my ($self, $ref_pkgdefs, $sl) = @_;
    my %pkgdefs = %$ref_pkgdefs;

    #
    # Get the list of files to copy
    #
    my @csm_bin_file_tftpboot = (@{$pkgdefs{csm_bin_copy_tftpboot}});

    my $MS_Arch = `/bin/uname -m`;
    if (($MS_Arch =~ /ppc/) && (($pkgdefs{'pkgdefs_hw_arch'} =~ /i.86/) || ($pkgdefs{'pkgdefs_hw_arch'} =~ /x86_64/)))
    {
        my $osname         = $pkgdefs{'pkgdefs_osname'};
        my $distro_name    = $pkgdefs{'pkgdefs_distro_name'};
        my $distro_version = $pkgdefs{'pkgdefs_distro_version'};
        my $hw_arch        = $pkgdefs{'pkgdefs_hw_arch'};
        my ($effective_distro_name, $effective_distro_ver) = 
            NodeUtils->getEffectiveDistro($distro_name, $distro_version);
        
        my $rpmdir = ArchiveUtils->getRPMSDir(
                        $osname,            # eg Linux or AIX (case sensitive)
                        $distro_name,       # eg RedHat or SLES (case sensitive)
                        $distro_version,    # eg 3 or 5.2
                        $hw_arch,           # eg i386, ppc
			$sl,                # eg sp3, qu2
                        );      #"Linux", $distname, $distvers, $arch, $sl;

        $::GREP = "/bin/grep";
        my $cmd;
        if ($effective_distro_name =~ /RedHat/)
        {
		$cmd = "$::FIND $rpmdir/ -name 'syslinux*.rpm' ! -name 'syslinux*.src.rpm'|sort|tail -1";
        }
        else
        {
		$cmd = "$::FIND $rpmdir/ -name 'syslinux*.rpm' ! -name 'syslinux*.src.rpm'|sort|tail -1";
        }
        my $res = `$cmd`;
        chomp $res;

        if ($res ne "")
        {
            $res =
              `rm -fr /tmp/dir4syslinux; mkdir -p /tmp/dir4syslinux; cd /tmp/dir4syslinux; $::RPM2CPIO $res | cpio -idu --quiet;`;
            my $cmd =
		"$::FIND /tmp/dir4syslinux | $::GREP \'pxelinux\.0\'";
            my $res = `$cmd`;
            chomp $res;
            @csm_bin_file_tftpboot = ($res);
        }
    }

    # Files to copy to /tftpboot
    if ($MS_Arch !~ /ppc/)
    {    
    	my $cmd     = "$::RPMCMD -qa | $::GREP syslinux";
    	my $res     = `$cmd`;
    	my $version = (split("-", $res))[1];
    	if ($version > "2.04")
    	{
        	$cmd = "$::RPMCMD -ql syslinux | $::GREP \'pxelinux\.0\'";
        	$res = `$cmd`;
        	chomp $res;
        	pop @csm_bin_file_tftpboot;
        	@csm_bin_file_tftpboot = ($res);
    	}
    }

    # Prevent the situation where /tftpboot does not exist at all.
    if (!-d "/tftpboot")
    {
        mkpath("/tftpboot");
    }

    # HOHO
    foreach my $bin_file (@csm_bin_file_tftpboot)
    {
        NodeUtils->runcmd("$::COPY -p $bin_file /tftpboot", -2);
    }

    # Link files into RDM subdirectories
    my $tftpboot_rdm_top = "opt/IBM/RDM/repository/environment";
    mkpath("/tftpboot/$tftpboot_rdm_top", $::VERBOSE, 0110);

    foreach my $tftpboot_rdm_file ("csm", "pxelinux.0", "pxelinux.cfg")
    {

        # Create the symlink if it doesn't already exist
        my $link = "/tftpboot/$tftpboot_rdm_top/$tftpboot_rdm_file";
        if (!-e $link)
        {
            NodeUtils->runcmd(
                "cd /tftpboot/$tftpboot_rdm_top; $::LN -sf ../../../../../$tftpboot_rdm_file .",
                -2
                );
        }
    }

    # after files are copied, change the permission:
    NetworkUtils->secureFilePermissions('/tftpboot', '0110', '0440');
    print "LEAVING: $routine\n" if $::DEBUG;
}

#--------------------------------------------------------------------------------

=head3 copy_csm_scripts

        This function copies customization scripts from /opt/csm/install/scripts/
        to /csminstall/csm/scripts.                                                            

        If the extension of the script matches a defined node group or there is
        no extension, then copy the script.

        If the $verify parameter is 1, this function checks the target dir if the
        scripts exist, if no, it will copy the scripts, otherwise, it will do
        nothing.  If the $verify parameter is 0,this function will always copy the
        scripts.

        Arguments:
                $verify  - either 1 or 0

        Returns:
                $::OK
                $::NOK
        Globals:
                $::NODEGRP
                $::PREINSTALLDIR
                $::POSTINSTALLDIR
                $::PREOSUPGRADEDIR
                $::POSTOSUPGRADEDIR
                $::UPDATEDIR
                $::DISKLESSPREBUILDDIR
                $::DISKLESSBOOTDIR
				$::SCRIPTDATADIR
        Error:
                csmInstall.cat 'E'_
        Example:
                ServerUtils->copy_csm_scripts(1);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub copy_csm_scripts
{

    my $INSTALLPREREBOOT  = "/opt/csm/install/scripts/installprereboot";
    my $INSTALLPOSTREBOOT = "/opt/csm/install/scripts/installpostreboot";
    my $OSUPGRADEPREREBOOT  = "/opt/csm/install/scripts/osupgradeprereboot";
    my $OSUPGRADEPOSTREBOOT = "/opt/csm/install/scripts/osupgradepostreboot";
    my $UPDATE            = "/opt/csm/install/scripts/update";
    my $DISKLESSPREBUILD = "/opt/csm/install/scripts/disklessprebuild";
    my $DISKLESSBOOT      = "/opt/csm/install/scripts/disklessboot";
    my $SCRIPTS           = "/opt/csm/install/scripts";
    my $METHODS           = "/opt/csm/install/methods";

    my @groupname;
    my $targetdir;
    my $filename;
    my $basename;
    my $extension;
    my $junk;
    my $cmd;

	if(!defined($ENV{'DC_ENVIRONMENT'}) || $ENV{'DC_ENVIRONMENT'} eq "CSM")
	{
        $cmd = "/usr/bin/lsrsrc-api -i -D ':|:' -s IBM.NodeGroup::::Name";
        @groupname = NodeUtils->runcmd($cmd, -1);
        if ($::RUNCMD_RC)
        {
            MessageUtils->messageFromCat(
                                     'csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall',     "E",
                                     'EMsgCANT_RUN',   $cmd,
                                     $::RUNCMD_RC
                                     );
            return $::NOK;
        }
	}
	else
	{
		@groupname=@::NODE_GROUP;
	}

    foreach my $dirname ($INSTALLPREREBOOT, $INSTALLPOSTREBOOT, 
                         $OSUPGRADEPREREBOOT, $OSUPGRADEPOSTREBOOT, $UPDATE,
                         $DISKLESSPREBUILD, $DISKLESSBOOT, $METHODS)
    {
        if ($dirname eq $INSTALLPREREBOOT)
        {
            $targetdir = $::PREINSTALLDIR;
        }
        elsif ($dirname eq $INSTALLPOSTREBOOT)
        {
            $targetdir = $::POSTINSTALLDIR;
        }
        if ($dirname eq $OSUPGRADEPREREBOOT)
        {
            $targetdir = $::PREOSUPGRADEDIR;
        }
        elsif ($dirname eq $OSUPGRADEPOSTREBOOT)
        {
            $targetdir = $::POSTOSUPGRADEDIR;
        }
        elsif ($dirname eq $UPDATE)
        {
            $targetdir = $::UPDATEDIR;
        }
        elsif ($dirname eq $DISKLESSPREBUILD)
        {
            $targetdir = $::DISKLESSPREBUILDDIR;
        }
        elsif ($dirname eq $DISKLESSBOOT)
        {
            $targetdir = $::DISKLESSBOOTDIR;
        }
        elsif ($dirname eq $METHODS)
        {
            # copy adapter configuration methods to the scripts/data dir
            $targetdir = $::SCRIPTDATADIR;
        }

        if (-d $dirname)
        {
            if (!-d $targetdir)
            {
                mkpath($targetdir, $::VERBOSE, 0755);
            }

            my @filelist;
            my $file;
            my $DIR;
            opendir($DIR, $dirname) or return ();
            while ($file = readdir($DIR))
            {
                if ($file ne '.' and $file ne '..')
                {
                    push @filelist, $file;
                }
            }
            closedir($DIR) or return ();

            foreach $filename (sort @filelist)
            {
                chomp $filename;
                if (-d "$dirname/$filename")
                {
                    MessageUtils->messageFromCat('csmInstall.cat',
                             $::MSGMAPPATH, 'csminstall', "V", 'IMsgSkipSubDir',
                             "$dirname/$filename");
                    next;
                }

                if (!-x "$dirname/$filename")
                {
                    MessageUtils->messageFromCat('csmInstall.cat',
                             $::MSGMAPPATH, 'csminstall', "E", 'EMsgSkipScript',
                             "$dirname/$filename");
                    next;
                }
                ($basename, $extension, $junk) = split(/\._/, $filename);
                if (defined($junk)) { next; }
                if (!defined($extension) || grep(/^$extension$/, @groupname))
                {
                    $cmd = "$::COPY -pf $dirname/$filename $targetdir";
                    NodeUtils->runcmd($cmd, -2);
                }
            }
        }

    }

    # copy 2nd adapter script to update, installprereboot and 
	# osupgradeprereboot directories
    my $adapterscript = "006CSM_adapter_config._AllNodes";
    if (-e "$SCRIPTS/$adapterscript")
    {
        $targetdir = $::UPDATEDIR;
        if (!-d $targetdir)
        {
            mkpath($targetdir, $::VERBOSE, 0755);
        }
        $cmd = "$::COPY -pf $SCRIPTS/$adapterscript $targetdir";
        NodeUtils->runcmd($cmd, -2);

        $targetdir = $::PREINSTALLDIR;
        if (!-d $targetdir)
        {
            mkpath($targetdir, $::VERBOSE, 0755);
        }
        $cmd = "$::COPY -pf $SCRIPTS/$adapterscript $targetdir";
        NodeUtils->runcmd($cmd, -2);

        $targetdir = $::PREOSUPGRADEDIR;
        if (!-d $targetdir)
        {
            mkpath($targetdir, $::VERBOSE, 0755);
        }
        $cmd = "$::COPY -pf $SCRIPTS/$adapterscript $targetdir";
        NodeUtils->runcmd($cmd, -2);

        $targetdir = $::DISKLESSBOOTDIR;
        if (!-d $targetdir)
        {
            mkpath($targetdir, $::VERBOSE, 0755);
        }
        $cmd = "$::COPY -pf $SCRIPTS/$adapterscript $targetdir";
        NodeUtils->runcmd($cmd, -2);
    }

    return $::OK;
}

#-------------------------------------------------------------------------------

=head3    create_directory_structure

       Creates the csminstall directory sturctuire on the Managment Server.

	Globls:

		Relies on $::PREREQS_ATTR variable for directory names

=cut

#-------------------------------------------------------------------------------

sub create_directory_structure
{

    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'I',
                                 'IMsgCreatingDirectories');
    mkpath($::INSTALLDIR_CSMVRBIN,  $::VERBOSE, 0755);
    mkpath($::INSTALLDIR_CSMBIN,    $::VERBOSE, 0755);
    mkpath($::PREINSTALLDIR,        $::VERBOSE, 0755);
    mkpath($::POSTINSTALLDIR,       $::VERBOSE, 0755);
    mkpath($::PREOSUPGRADEDIR,      $::VERBOSE, 0755);
    mkpath($::POSTOSUPGRADEDIR,     $::VERBOSE, 0755);
    mkpath($::UPDATEDIR,            $::VERBOSE, 0755);
    mkpath($::SCRIPTDATADIR,        $::VERBOSE, 0755);
    mkpath($::INSTALLDIR_CSMPKG,    $::VERBOSE, 0755);
    mkpath($::DISKLESSPREBUILDDIR, $::VERBOSE, 0755);
    mkpath($::DISKLESSBOOTDIR,      $::VERBOSE, 0755);
    mkpath($::CSMDRIVERDIR,         $::VERBOSE, 0755);

    ArchiveUtils->createRPMSdir(
                                $::PREREQS_ATTR{'OSName'},
                                $::PREREQS_ATTR{'DistributionName'},
                                $::PREREQS_ATTR{'DistributionVersion'},
                                $::PREREQS_ATTR{'PkgArchitecture'},
                                $::PREREQS_ATTR{"InstallServiceLevel"}
                                );

    # So far ServerUtils->create_directory_structure() is only used by installms,
    # copycsmpkgs uses its own implementation. So create top RPMS symbolic link in
    # $distro_top to $distro_top/$svclevel/RPMS for installms here is proper.
    my $TOP_RPMS =
      ArchiveUtils->getRPMSDir(
                               $::PREREQS_ATTR{'OSName'},
                               $::PREREQS_ATTR{'DistributionName'},
                               $::PREREQS_ATTR{'DistributionVersion'},
                               $::PREREQS_ATTR{'PkgArchitecture'}
                               );
    if (-d $TOP_RPMS)
    {

        #NodeUtils->runcmd("rm $TOP_RPMS");
        unlink($TOP_RPMS);
    }
    my @pa = split("/", $TOP_RPMS);
    pop @pa;
    my $basedir = join("/", @pa);

    my $svclevelRPMS =
      ArchiveUtils->getSvcLevelRPMSDir(
                                       $::PREREQS_ATTR{'OSName'},
                                       $::PREREQS_ATTR{'DistributionName'},
                                       $::PREREQS_ATTR{'DistributionVersion'},
                                       $::PREREQS_ATTR{'PkgArchitecture'},
                                       $::PREREQS_ATTR{"InstallServiceLevel"}
                                       );
    my ($srcd);
    my ($effective_distro_name, $effective_distro_ver) = 
            NodeUtils->getEffectiveDistro(
                    $::PREREQS_ATTR{'DistributionName'},
                    $::PREREQS_ATTR{'DistributionVersion'}
                    );

    if ($effective_distro_name =~ /SLES/)
    {
        @pa = split("/", $svclevelRPMS);
        pop @pa;
        $srcd = pop @pa;
    }
    else
    {
        #@pa = split("/", $svclevelRPMS);
        #pop @pa;
        #pop @pa;
        #$srcd = pop @pa;
        # support rhel5: rhel5's packages be put at Server, not RedHat/RPMS.
        (undef, $srcd) = split /$basedir/,$svclevelRPMS;
        $srcd =~ s/$::pkgdefs{DISTRO_RPMDIR}$//;
        $srcd =~ s{/}{}g;
    }

    my $targetRPMS = "$srcd/RPMS";
    my $ln_cmd     = "cd $basedir; $::LN -s $targetRPMS $TOP_RPMS";
    NodeUtils->runcmd($ln_cmd, 0);
}

#--------------------------------------------------------------------------------

=head3    getBaseDir

        Get the base directory for the install
        attribute parameters

        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur
                $InstallServiceLevel
        Returns:
                approparite base directory for parameters
        Globals:
                none
        Error:
                undefined
        Example:
                $::DISTRO_TOP = ServerUtils->getBaseDir("Linux",
                                                        $::DISTRO_NAME,
                                                        $::DISTRO_VERSION,
                                                        $::ARCH, $::SERVICE_LEVEL);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getBaseDir
{
    my ($class, $os, $distributor, $version, $arch, $sl) = @_;
    if ($arch =~ /i.86/) { $arch = "i386"; }
    my $default_dir = $::CSMINSTDIR;
    if ($ENV{'CSMINSTALL_ROOT'}) { $default_dir = $ENV{'CSMINSTALL_ROOT'}; }
    my ($effective_distro_name, $effective_distro_ver) = 
            NodeUtils->getEffectiveDistro($distributor, $version);

    if (!$sl)
    {
        return "$default_dir/$os/$distributor/$version/$arch";
    }
    else
    {
        if ($effective_distro_name =~ /SLES/)
        {
            return "$default_dir/$os/$distributor/$version/$arch/$sl";
        }
        else
        {
            my $dir;
            if ($sl eq "GA")
            {
                $dir = $distributor . $version . "-" . $arch;
            }
            else
            {
                $dir = $distributor . $version . "-" . $sl;
            }
            return "$default_dir/$os/$distributor/$version/$arch/$dir";
        }
    }
}

#--------------------------------------------------------------------------------

=head3    getInstallDir

        Returns the install directory path name for the Install Attribute
        parameters.

        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur
                $InstallServiceLevel
        Returns:
                Appropriate path name 
        Globals:
                none
        Error:
                undefined
        Example:
                unused
        Comments:
                unused??

=cut

#--------------------------------------------------------------------------------

sub getInstallDir
{
    my ($class, $os, $distributor, $version, $arch, $sl) = @_;
    if ($arch =~ /i.86/) { $arch = "i386"; }
    my $base_install = $::CSMINSTDIR;
    if (!defined $sl)
    {
        return "$base_install/$os/$distributor/$version/$arch/install";
    }
    else
    {
        if ($distributor =~ /SLES/)
        {
            return "$base_install/$os/$distributor/$version/$arch/$sl/install";
        }
        else
        {
            my $dir;
            if ($sl eq "GA")
            {
                $dir = $distributor . $version . "-" . $arch;
            }
            else
            {
                $dir = $distributor . $version . "-" . $sl;
            }
            return "$base_install/$os/$distributor/$version/$arch/$dir/install";
        }
    }
}

#--------------------------------------------------------------------------------

=head3    getUpdateDir

        Returns the update directory for the install
        attritbute parameters

        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur
                $InstallServiceLevel

        Returns:
                Appropriate path name of the attribute input
        Globals:
                none
        Error:
                undefined
        Example:
                my $updateDir =
                    ServerUtils->getUpdateDir($OS, $dist, $distver, $arch, $sl);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getUpdateDir
{
    my ($class, $os, $distributor, $version, $arch, $sl) = @_;
    if ($arch =~ /i.86/) { $arch = "i386"; }
    my $base_install = $::CSMINSTDIR;
    if (!defined $sl)
    {
        return "$base_install/$os/$distributor/$version/$arch/updates";
    }
    else
    {
        if ($distributor =~ /SLES/)
        {
            return "$base_install/$os/$distributor/$version/$arch/$sl/updates";
        }
        else
        {
            my $dir;
            if ($sl eq "GA")
            {
                $dir = $distributor . $version . "-" . $arch;
            }
            else
            {
                $dir = $distributor . $version . "-" . $sl;
            }
            return "$base_install/$os/$distributor/$version/$arch/$dir/updates";
        }
    }
}

#--------------------------------------------------------------------------------

=head3    getCSMAddonDir

        Returns the csm additional directory for the install
        attritbute parameters.
		Note that this directory have additional packages 
		distribution dependent.  

        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur
                $InstallServiceLevel

        Returns:
                Appropriate path name of the attribute input
        Globals:
                none
        Error:
                undefined
        Example:
                my $csmAddonDir =
                    ServerUtils->getCSMAddonDir($OS, $dist, $distver, $arch, $sl);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getCSMAddonDir
{
    my ($class, $os, $distributor, $version, $arch, $sl) = @_;
    if ($arch =~ /i.86/) { $arch = "i386"; }
    my $base_install = $::CSMINSTDIR;
    if (!defined $sl)
    {
        return "$base_install/$os/$distributor/$version/$arch/csm_addon";
    }
    else
    {
        if ($distributor =~ /SLES/)
        {
            return "$base_install/$os/$distributor/$version/$arch/$sl/csm_addon";
        }
        else
        {
            my $dir;
            if ($sl eq "GA")
            {
                $dir = $distributor . $version . "-" . $arch;
            }
            else
            {
                $dir = $distributor . $version . "-" . $sl;
            }
            return "$base_install/$os/$distributor/$version/$arch/$dir/csm_addon";
        }
    }

}
#--------------------------------------------------------------------------------

=head3    getInstallImagesDir

        Returns the images directory for the install
        attritbute parameters.
		Note that this directory is used to put user supplied 
		images for warewulf install method.
        Arguments:
                $InstallOSName
                $InstallDistributionName
                $InstallDistributionVersion
                $InstallPkgArchitectur
                $InstallServiceLevel

        Returns:
                Appropriate path name of the attribute input
        Globals:
                none
        Error:
                undefined
        Example:
                my $updateDir =
                    ServerUtils->getInstallImagesDir($OS, $dist, $distver, $arch, $sl);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getInstallImagesDir
{
    my ($class, $os, $distributor, $version, $arch, $sl) = @_;
    if ($arch =~ /i.86/) { $arch = "i386"; }
    my $base_install = $::CSMINSTDIR;
    if (!defined $sl)
    {
        return "$base_install/$os/$distributor/$version/$arch/installimages";
    }
    else
    {
        if ($distributor =~ /SLES/)
        {
            return "$base_install/$os/$distributor/$version/$arch/$sl/installimages";
        }
        else
        {
            my $dir;
            if ($sl eq "GA")
            {
                $dir = $distributor . $version . "-" . $arch;
            }
            else
            {
                $dir = $distributor . $version . "-" . $sl;
            }
            return "$base_install/$os/$distributor/$version/$arch/$dir/installimages";
        }
    }
}

#--------------------------------------------------------------------------------

=head3    returnSubdirectory

        Returns all the sub directories of the input parameter.

        Arguments:
                $directoryName
                $validDir - (optional) value of 1 if defined
        Returns:
                array of subdirectory names     
        Globals:
                $::PLTFRM
        Error:
                undefined
        Example:
                 push   @::PKGPATH,
                        (ServerUtils->returnSubdirectory($p));
        Comments:
                the $validDir parameter, when set to 1 checks if the 
                parent directory to the $directoryName parmeter is valid. 

=cut

#--------------------------------------------------------------------------------

sub returnSubdirectory
{
    my ($class, $dir, $valid_dir) = @_;
    my $file;
    my @subdirs       = ();
    my @returndirs    = ();
    my @valid_subdirs = ();    #these are ones whose children we can look at
    if ($::PLTFRM eq "AIX")
    {
        if ($dir =~ m/(installp.?\z)|(usr.?\z)|(RPMS.?\z)/)
        {                      #check to see if the parent is valid
            $valid_dir = 1;
        }
    }
    opendir(DIRECTORY, $dir) or return ();
    while ($file = readdir(DIRECTORY))
    {
        if ($file ne '.' and $file ne '..')
        {
            my $filename;
            if ($dir =~ /\/\z/)
            {                  #if it ends with a slash
                $filename = "$dir$file";
            }
            else
            {                  #no slash
                $filename = "$dir/$file";
            }
            if ((-d "$filename") && ($::PLTFRM eq "AIX"))
            {
                if (!$valid_dir)
                {              #the parent wasn't an valid directory
                    if (   ($file eq "installp")
                        || ($file eq "usr")
                        || ($file eq "RPMS"))
                    {
                        push @valid_subdirs, $filename;
                    }
                }
                else
                {    #a valid parent, so therefore its children are valid too
                    push @valid_subdirs, $filename;
                }

            }
        }
    }
    closedir(DIRECTORY);
    push @returndirs, @valid_subdirs;
    push @returndirs, @subdirs;
    foreach my $sub (@valid_subdirs)
    {                #these are valid children
        my @tmpdirs = ServerUtils->returnSubdirectory($sub, 1);
        push @returndirs, @tmpdirs;
    }
    return @returndirs;
}

#--------------------------------------------------------------------------------

=head2    File Layer

=cut

#--------------------------------------------------------------------------------

=head3    CreateRandomName

        Create a randome file name.

        Arguments:
                Prefix of name
        Returns:
                Prefix with 8 random letters appended
        Error:
                none
        Example:
                $file = ServerUtils->CreateRandomName($namePrefix);
        Comments:
                None

=cut

#--------------------------------------------------------------------------------

sub CreateRandomName
{

    my ($class, $name) = @_;

    my $nI;
    for ($nI = 0 ; $nI < 8 ; $nI++)
    {
        my $char = ('a' .. 'z', 'A' .. 'Z')[int(rand(52)) + 1];
        $name .= $char;
    }
    $name;
}

#--------------------------------------------------------------------------------

=head3    create_nodemap_file

        Create /csminstall/csm/config/nodemap file that contains a mapping
        of ManagedNode hostnames as defined in the CSM database, and node hostnames
        as returned by the hostname command. 

        Arguments:
                 $ref_nodeList
                 $ref_nodeInfo
                 $ref_DestNodeHash
        Returns:
                $::OK
                $::NOK
        Globals:
                $::NODE_LIST_FILE
        Error:
                $::NOK
        Example:
                if( ServerUtils->create_nodemap_file ( \@::NODELIST,
                                                        \@::LSNODE_INFO,
                                                        \%::NODEHASH) != 0){
                    blah;
                }
        Comments:
                File format -  csm_hostname<space>node_hostname    

=cut

#--------------------------------------------------------------------------------

sub create_nodemap_file
{
    my ($class, $ref_nodelist, $ref_nodeinfo, $ref_DestNodeHash) = @_;
    my @nodelist = @$ref_nodelist;
    my @nodeinfo = @$ref_nodeinfo;

    #  DestNodeHash is a hash containing attribute values for each node
    #    that was provided on the command line.
    my %DestNodeHash = %$ref_DestNodeHash;
    %main::dsh_failed = ();
    my ($cmd,          @dshout,  $rc);
    my ($csm_nodename, $csmnode, $hostname);
    my $csmcfgdir    = "/csminstall/csm/config";
    my $nodemap_file = "/csminstall/csm/config/nodemap";
    my $node;
    my $dshresult = "";

    #
    #  Check and/or create the /csminstall/csm/config directory
    #
    if (!-d $csmcfgdir)
    {
        $cmd = "$::MKDIR -m 755 -p $csmcfgdir";
        $rc  = system("$cmd");
        if ($rc >> 8)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', "E", 'EMsgNO_CreateDir', $csmcfgdir);
            return ($::NOK);
        }
    }

    # Remove any old occurences of the nodemap file
    if (-f $nodemap_file)
    {
        unlink($nodemap_file);
    }

    # open the map file
    unless (open(NODEMAP, ">$nodemap_file"))
    {

        # print error message
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',      $::MSGMAPPATH,
                                     'csminstall',          "E",
                                     'EMsgCANT_WRITE_FILE', "$nodemap_file"
                                     );
        return $::NOK;
    }
    my @AIX;
    my @Linux;
    my %found = ();
    my %dups  = ();

    # for each node in the node list
    foreach $node (keys %DestNodeHash)
    {

        # set the correct command based on the InstallOSName
        if (defined($DestNodeHash{$node}{'InstallOSName'}))
        {
            if ($DestNodeHash{$node}{'InstallOSName'} eq "AIX")
            {

                #$dshcmd = "$::DSH -n$node hostname";
                push @AIX, $node;
            }
            elsif ($DestNodeHash{$node}{'InstallOSName'} eq "Linux")
            {

                #$dshcmd = "$::DSH -n$node hostname -f";
                push @Linux, $node;
            }
        }
    }
    if (@Linux)
    {
	my %options_api = ();
	$options_api{'command'} = "hostname -f";
        $options_api{'nodes'} = join(',', @Linux);
        my @dshresult = NodeUtils->runDsh(\%options_api, -1);

	if ($::RUNCMD_RC != 0){
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
			'csminstall', 'E', 'EMsgCMD_FAILED_RC', $options_api{'command'}, $::RUNCMD_RC);
	}
        foreach my $line (@dshresult)
        {
            my ($csmnode, $hostname) = split(" ", $line);
            $hostname =~ s/://g;
            $csmnode  =~ s/://g;
            if (defined($csmnode) && defined($hostname))
            {

                # make sure the format is consistent with the name that
                #     was used to create the configinfo file
                #$csm_nodename = NodeUtils->tryHost($csmnode);
                $found{$csmnode} = 1;
                $dups{$hostname}{$csmnode} = 1;

                # put the result in the nodemap file
                unless (print NODEMAP "$csmnode $hostname\n")
                {
                    MessageUtils->messageFromCat('csmInstall.cat',
                                       $::MSGMAPPATH, 'csminstall', 'E',
                                       'EMsgCANT_WRITEtoFILE', "$nodemap_file");
                }
            }
        }
    }
    if (@AIX)
    {
	my %options_api = ();
	$options_api{'command'} = "hostname";
        $options_api{'nodes'} = join(',', @AIX);
        my @dshresult = NodeUtils->runDsh(\%options_api, -1);

	if ($::RUNCMD_RC != 0){
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
			'csminstall', 'E', 'EMsgCMD_FAILED_RC', $options_api{'command'}, $::RUNCMD_RC);
	}
        foreach my $line (@dshresult)
        {
            my ($csmnode, $hostname) = split(" ", $line);
            $hostname =~ s/://g;
            $csmnode  =~ s/://g;
            if (defined($csmnode) && defined($hostname))
            {

                # make sure the format is consistent with the name that
                #     was used to create the configinfo file
                #$csm_nodename = NodeUtils->tryHost($csmnode);
                $found{$csmnode} = 1;
                $dups{$hostname}{$csmnode} = 1;

                # put the result in the nodemap file
                unless (print NODEMAP "$csmnode $hostname\n")
                {
                    MessageUtils->messageFromCat('csmInstall.cat',
                                       $::MSGMAPPATH, 'csminstall', 'E',
                                       'EMsgCANT_WRITEtoFILE', "$nodemap_file");
                }
            }
        }
    }

    foreach my $n (keys %dups)
    {    #get rid of any nodes that map to the same hostname
        if (scalar(keys %{$dups{$n}}) > 1)
        {
            my @dupNodes = keys %{$dups{$n}};
            my $dNstring = join ',', @dupNodes;
            my $hostname = $n;
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                    'csminstall', 'E', 'EMsgHostnameMap', $hostname, $dNstring);
            foreach my $d (@dupNodes)
            {
                $main::dsh_failed{$d} = 1;
            }
        }
    }
    close(NODEMAP);
    if (!@AIX && !@Linux)
    {
        return $::NOK;
    }
    return $::OK;
}

#--------------------------------------------------------------------------------

=head3    createSMSConfig

        Put SMS node group info into the file:
        /csminstall/csm/var/smsupdate.info

        Arguments:
                none
        Returns:
                reference to a hash of all groups
        Globals:
                none
        Error:
                NodeUtil message E
        Example:
                 #create the SMS nodegroup config file
                 ServerUtils->createSMSConfig();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub createSMSConfig
{
    my %all_groups = ();
    my %node_index;
    my @empty_array = ();
    if (-e "/csminstall")
    {
        if (!-e "/csminstall/csm")
        {
            mkdir "/csminstall/csm",        0077;
            mkdir "/csminstall/csm/config", 0077;
        }
    }
    my %groups;
    if (defined %ServerUtils::AllGroupsAndMembers)
    {
        %groups = %ServerUtils::AllGroupsAndMembers;
    }
    else
    {
        my @empty_array = ();
        my $outref      = NodeUtils->listNodeGroupExpMem(@empty_array);

        foreach my $row (@$outref)
        {
            my ($groupname, $nodelist) = $row =~ /^(.+):\|:\{(.*)\}$/;
            my @nodes = split(/,/, $nodelist);
            @{$groups{$groupname}} = @nodes;
        }
        %ServerUtils::AllGroupsAndMembers = %groups;    #cache
        undef $outref, @empty_array;
    }

    foreach my $groupname (keys %groups)
    {
        my @nodes = @{$groups{$groupname}};
        foreach my $node (@nodes)
        {
            push @{$node_index{$node}},      $groupname;
            push @{$all_groups{$groupname}}, $node;

        }
    }
    system("$::MKDIR -p /csminstall/csm/config/");
    open(GROUPFILE, ">/csminstall/csm/config/sms.info")
      or MessageUtils->messageFromCat(
                                      'csmSMS.cat',
                                      $MSGMAPPATH,
                                      'smsupdatenode',
                                      'E',
                                      'EMsgCANT_OpenFile',
                                      "/csminstall/csm/config/sms.info",
                                      "$? $!"
                                      );
    foreach my $node (keys %node_index)
    {
        print GROUPFILE "$node: @{$node_index{$node}} \n";
    }
    close GROUPFILE;
    return \%all_groups;
}

#--------------------------------------------------------------------------------

=head3 get_script_list

        For a given directory get a list of scripts and associated list
        of nodes to run it on.

        Arguments:
                $directoryName
                 @nodeList
        Returns:
                $::OK
                $::NOK 
        Globals:
                @::scriptlist
                @{$::scriptnodes{$fname}}
                
        Error:
                $::NOK 
        Example:
                $rc = ServerUtils->get_script_list($dirname,@nodelist);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub get_script_list
{
    my ($junk, $dirname, @nlist) = @_;
    my %basename;
    my $target;
    my $filename;
    my @filelist;
    my $cmd;

    @::scriptlist = ();

    #get the list of groups for name2list

    my %groups;
    if (defined %ServerUtils::AllGroupsAndMembers)
    {
        %groups = %ServerUtils::AllGroupsAndMembers;
    }
    else
    {
        my @empty_array = ();
        my $outref;

	if ($ENV{'DC_ENVIRONMENT'} eq 1)
        {
            $outref = DCUtils->listNodeGroupDC(@empty_array);
        }
        else
        {
            $outref = NodeUtils->listNodeGroupExpMem(@empty_array);
        }

        foreach my $row (@$outref)
        {
            my ($groupname, $nodelist) = $row =~ /^(.+):\|:\{(.*)\}$/;
            my @nodes = split(/,/, $nodelist);
            @{$groups{$groupname}} = @nodes;
        }
        # Append Warewulf node group node list.
        $outref = ServerUtils->listWarewulfNodeGroup();
        foreach my $row (@$outref)
        {
            my ($groupname, $nodelist) = $row =~ /^(.+):\|:\{(.*)\}$/;
            my @nodes = split(/,/, $nodelist);
            @{$groups{$groupname}} = @nodes;
        }

        %ServerUtils::AllGroupsAndMembers = %groups;    #cache
        undef $outref, @empty_array;
    }
    if (-d $dirname)
    {
        my $file;
        my $DIR;
        opendir($DIR, $dirname) or return $::NOK;
        while ($file = readdir($DIR))
        {
            if ($file ne '.' and $file ne '..')
            {
                push @filelist, $file;
            }
        }
        closedir($DIR) or return $::NOK;
    }
    else
    {

        # msg - Warning: The \'$dirname\' does not exist.
        # Skipping to the next directory.
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', "E", 'EMsgSkipDir',
                                     $dirname);
        return $::NOK;
    }

    # check each script name in this directory
    # - add the filename to the script list
    # - get the initial list of node to run the script on
    # - save the basename for each script
    foreach $filename (sort @filelist)
    {
        chomp $filename;

        #  make sure this is not a subdir
        if (-d "$dirname/$filename")
        {

            # Skipping subdirectory $dirname/$filename.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                     'csminstall', "V", 'IMsgSkipSubDir', "$dirname/$filename");
            next;
        }

        # figure out what nodes to run the script on by
        # deciphering the script name
        @{$::scriptnodes{$filename}} =
          ServerUtils->name2list($dirname, $filename, \%groups, \@nlist);
        if (${$::scriptnodes{$filename}}[0] eq $::NOK)
        {

            # msg - Skipping to the next script name.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                         'csminstall', "E", 'EMsgSkipScript');
            next;
        }
        push @::scriptlist, $filename;
        (@basename{$filename}, $target) = split(/\._/, $filename);
    }

    # now check each script that has no extension (target) and see
    #  if there are any scripts with the same basename that have targets
    # if there are - remove any nodes in the targeted script list from the
    #  list of nodes for the script with no extensions(target)
    # ex.  say script foo goes to all nodes and script foo._grpa goes
    # to node1 & node2.  In this case I need to remove node1 & node2
    # from the list of nodes for the script foo.

    foreach my $fname (sort @::scriptlist)
    {
        chomp $fname;

        # if the script name is the same as the basename then
        #   the list is all the nodes passed in on the command line.
        if ($fname eq @basename{$fname})
        {

            # check the rest of the scripts to see if any
            # have the same basename - if we find one then redo
            # the list for the basename-only script
            foreach my $file (@::scriptlist)
            {
                chomp $file;

                #  see if the basenames are the same
                if (@basename{$fname} eq @basename{$file})
                {

                    # don't remove itself
                    if ($file ne $fname)
                    {
                        my @tmplist = ();
                        foreach my $node (@{$::scriptnodes{$fname}})
                        {

                            # if the node is not in the list for $file
                            #  then add it to a temp list
                            if (!grep /^$node/, @{$::scriptnodes{$file}})
                            {
                                push @tmplist, $node;
                            }
                        }
                        @{$::scriptnodes{$fname}} = @tmplist;
                    }
                }
            }
        }
    }
    return $::OK;
}

#-----------------------------------------------------------------------------

=head3    cleanup_adapter_files

        Remove any unused adapter stanza files from the 
         /csminstall/csm/config/adapters directory.  They were copied 
         there by make_config_file()

    Arguments:
           none
    Returns:
           $::OK
           $::NOK
    Globals:
           None

    Called By:
           getadapters

    Example:
        if (ServerUtils->cleanup_adapter_files != 0 ) {  }

=cut

#------------------------------------------------------------------------------
sub cleanup_adapter_files
{
	my $dirname = "/csminstall/csm/config/adapters";
    my $filelist;

	# nothing to do if the directory doesn't exist yet!
	if (!-d $dirname)
	{
		return $::OK;
	}

	#  get a list of all adapter file being used from the node definitions
	my $filelist = NodeUtils->runrmccmd('lsrsrc-api',"-i", "-s IBM.ManagedNode::::AdapterStanzaFile");
    if ($::RUNCMD_RC)
    {
		# could not run command
        return $::NOK;
    }

	#  get a list of the files contained in the directory

	my $cmd = "cd $dirname; $::FIND . -type f -print | cut -f 2- -d /";
	my @outref = NodeUtils->runcmd($cmd, -1);

	if ($::RUNCMD_RC == 0)
    {
        foreach my $line (@outref)
        {
			#  check to see if each file in the directory is being used
			if ( !grep /$line/, @$filelist)
			{
				# if not then remove the file
				my $name = "$dirname/$line";
				NodeUtils->runcmd("$::RM -f $name", -1);
			}
		}
	}
	else
	{
		#  could not run command
		MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall', 'E', 'EMsgCMD_FAILED_RC', $::FIND, $::RUNCMD_RC);
		return $::NOK;
	}
	return $::OK;
}

#-----------------------------------------------------------------------------

=head3    set_stanza_file

		Check an adapter stanza file and save the name in the 
		AdapterStanzaFile attribute.

    Arguments:
           $stanzafile
    Returns:
           $::OK
           $::NOK
    Globals:
		   None

    Called By:
           getadapters

    Example:
		if (ServerUtils->set_stanza_file($::STANZAFILE) != 0 ) {  }

=cut

#------------------------------------------------------------------------------
sub set_stanza_file
{
    my ($class, $stanza_file) = @_;
    my @NODELIST = ();
    my $line;
    my $hostname;
    my $junk;
    my $junk2;
    my $hostname;
    my $look_for_colon;
    my $attr;
    my $val;
    my $value;
    my $foundstartifcfg = 0;
    my @stanzas         = ();
    my $linenum;
    my $stanzafile;

    my @VALID_ATTRIBUTES = (
                            "machine_type",         "network_type",
                            "MAC_address",          "location",
                            "netaddr",              "subnet_mask",
                            "ping_status",          "adapter_speed",
                            "adapter_duplex",       "install_server",
                            "install_gateway",      "interface_name",
                            "attributes",           "secondary_hostname",
                            "cable_type",           "media_speed",
                            "interface_type",       "hostaddr",
                            "interface_attributes", "adapter_attributes",
                            "multiple_physloc",     "comments",
                            "bos_preconfig",        "cust_preconfig",
                            "device_driver",        "route",
                            "adapter_type",         "ib_adapter",
							"ib_port",              "mtu",
							"p_key",                "srq_size"
                            );

    chomp $stanza_file;

	#  convert stanza file name to full path name - if needed
	$stanzafile = NodeUtils->get_file_realpath($stanza_file);

    $::n = -1;    # stanza counter.  Start at -1 so it gets
                  #      incremented to 0 the first time.

    if ($stanzafile eq 0)  # file does not exist.
    {
        # error - file doesn't exist
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',     $::MSGMAPPATH,
                                     'csminstall',         'E',
                                     'EMsgFILE_NOT_FOUND', $stanza_file
                                     );
        return $::NOK;
    }

    # Make sure the file exists
    if (!-f $stanzafile)
    {

        # error - file doesn't exist
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',     $::MSGMAPPATH,
                                     'csminstall',         'E',
                                     'EMsgFILE_NOT_FOUND', $stanzafile
                                     );
        return $::NOK;
    }

    # open the file to read
    unless (open(STANZADEF, "<$stanzafile"))
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',     $::MSGMAPPATH,
                                     'csminstall',         'E',
                                     'EMsgCANT_READ_FILE', $stanzafile
                                     );
        return $::NOK;
    }

    #  read through the stanza file and
    #     -  get a list of nodes that have stanzas in the file
    #     -  check for errors in the file
    #     - create a list of stanzas that can be used to check for valid attrs
    while ($line = <STANZADEF>)
    {

        $linenum++;

        # keep track of whether the START_IFCFG tag has been set
        if ($line =~ /START_IFCFG/)
        {
            $foundstartifcfg = 1;
        }
        elsif ($line =~ /END_IFCFG/)
        {
            $foundstartifcfg = 0;
        }

        (grep(/^\s*#/, $line)) && next;    # Skip comment lines

        next if ($line =~ /^\s*$/);        # Next if empty blank line

        if (grep(/:\s*$/, $line))
        {                                  # see if it's a stanza name

            $look_for_colon = 0;
            ($hostname, $junk, $junk2) = split(/:/, $line);

            # check for second : or = sign
            #    if $junk2 is defined then there was a second :
            if (defined($junk2) || grep(/=/, $junk))
            {

                # error - invalid header $line in node definition file
                #         skipping to next node stanza
                #  Stanza file contains bad format
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                         'csminstall', 'E', 'EMsgADAPDEF_BAD_FORMAT', $linenum);

                $look_for_colon++;
                next;
            }

            $hostname =~ s/^\s*//;    # Remove any leading whitespace
            $hostname =~ s/\s*$//;    # Remove any trailing whitespace

            if ($hostname ne "default")
            {

                #  make a unique list of nodes
                if (!grep /^$hostname/, @NODELIST)
                {
                    push(@NODELIST, $hostname);
                }
            }

            $::n++;

            # add hostname to stanza list
            #     - used to check for valid values etc.
            $::stanzas[$::n]{'hostname'}  = $hostname;
            $::linenums[$::n]{'hostname'} = $linenum;

        }
        elsif (($line =~ /^\s*(\w+)\s*=\s*(.*)\s*/) && (!$look_for_colon))
        {

            # look at attr=val values
            $attr  = $1;
            $value = $2;
            $attr =~ s/^\s*//;    # Remove any leading whitespace
            $attr =~ s/\s*$//;    # Remove any trailing whitespace

            #remove any trailing comments
            ($val, $junk) = split(/#/, $value);

            $val =~ s/^\s*//;
            $val =~ s/\s*$//;

            # remove any surrounding quotes
            $val =~ s/^(\'|\")(.*)(\"|\')$/$2/;

            # check for error in format - no hostname set
            if ($hostname eq "")
            {
                chomp $line;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                         'csminstall', 'E', 'EMsgADAPDEF_BAD_FORMAT', $linenum);

                $look_for_colon++;
                next;
            }

            #  Either it's on the CSM list or the START_IFCFG value must be set
            if ((!grep(/^$attr$/, @VALID_ATTRIBUTES)) && !$foundstartifcfg)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                       'csminstall', 'W', 'EMsgADAPDEF_INVALID_ATTR', $linenum);
                next;
            }

            #  # add attr values to stanza list
            #     - used to check for valid values etc. - see below
            $::stanzas[$::n]{$attr}  = $val;
            $::linenums[$::n]{$attr} = $linenum;
        }
    }

    #
    #  get the InstallOSName and InstallDistributionName attr for each node
    #
    my $outref =
      NodeUtils->runrmccmd(
        'lsrsrc-api',
        "-i",
        "-s IBM.ManagedNode::::Hostname::InstallOSName::InstallDistributionName"
        );
    if ($::RUNCMD_RC)
    {
        $outref = [];
    }

    my (%OSName, %EffectiveDistroName);
    foreach my $l (@$outref)
    {
        my ($hname, $osname, $distroname) = split(/::/, $l);
        chomp $osname;
        my $short_hname = NodeUtils->getShorthost($hname);
        $OSName{$short_hname}     = $osname;
        ($EffectiveDistroName{$short_hname}) =
           NodeUtils->getEffectiveDistro($distroname); 
    }

    #
    # check the stanzas for valid attributes.  If error, just give
    # a warning and continue processing the other attributes.
    #

    my $size     = $::n;
    my $index    = 0;
    my %defaults = ();

    while ($index <= $size)
    {

        # check the default stanza attributes also
        #   save the values so they can be checked when looking at the
        #   hostname stanzas below
        if ($::stanzas[$index]{'hostname'} eq "default")
        {

            #  set default values - then check for attr or default below

	        # save values in default hash
            foreach my $attr (keys %{$::stanzas[$index]})
            {
                if ($::stanzas[$index]{$attr} ne "")
                {
                        $defaults{$attr} = $::stanzas[$index]{$attr};
                }
            }

            #  machine_type
            if ($::stanzas[$index]{'machine_type'} ne "")
            {
                my $mtype = $::stanzas[$index]{'machine_type'};
                if (   ($mtype ne "secondary")
                    && ($mtype ne "install")
                    && ($mtype ne "etherchannel"))
                {

                    # The value for the machine_type attribute, $mtype, on
                    #   line $::linenums[$index]{'machine_type'} should be either
                    #   secondary, install, or etherchannel.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_BadMachType', $mtype,
                        $::linenums[$index]{'machine_type'}
                        );
                }
            }

            # adapter_type
            if ($::stanzas[$index]{'adapter_type'} ne "")
            {
                my $atype = $::stanzas[$index]{'adapter_type'};
                if (   ($atype ne "ent")
                    && ($atype ne "sni")
					&& ($atype ne "iba")
                    && ($atype ne "ml")
                    && ($atype ne "mlt")
                    && ($atype ne "tok")
                    && ($atype ne "fiddi"))
                {

                    # The value for the \'adapter_type\' attribute, (\'$atype\'), on
                    #   line $::linenums[$index]{'adapter_type'} is not supported
                    #   by CSM or AIX.  A user-provided method must be used for
                    #   this adapter type.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_BadAdapType', $atype,
                        $::linenums[$index]{'adapter_type'}
                        );
                }
            }

            # subnet_mask
            if ($::stanzas[$index]{'subnet_mask'} ne "")
            {
                if (!NetworkUtils->isIpaddr($::stanzas[$index]{'subnet_mask'}))
                {

                    # The subnet_mask value $::stanzas[$index]{'subnet_mask'}
                    #   on line $::linenums[$index]{'netaddr'} is not a valid IP
                    #   address.
                    MessageUtils->messageFromCat(
                                              'csmInstall.cat',
                                              $::MSGMAPPATH,
                                              'csminstall',
                                              'W',
                                              'EMsgADAPDEF_BadIP',
                                              'subnet_mask',
                                              $::stanzas[$index]{'subnet_mask'},
                                              $::linenums[$index]{'subnet_mask'}
                                              );

                }
            }

            # NETMASK
            if ($::stanzas[$index]{'NETMASK'} ne "")
            {
                if (!NetworkUtils->isIpaddr($::stanzas[$index]{'NETMASK'}))
                {

                    # is it valid IP
                    # The NETMASK value $::stanzas[$index]{'NETMASK'} on line
                    #   $::linenums[$index]{'NETMASK'} is not a valid IP address
                    MessageUtils->messageFromCat(
                                                 'csmInstall.cat',
                                                 $::MSGMAPPATH,
                                                 'csminstall',
                                                 'W',
                                                 'EMsgADAPDEF_BadIP',
                                                 'NETMASK',
                                                 $::stanzas[$index]{'NETMASK'},
                                                 $::linenums[$index]{'NETMASK'}
                                                 );
                }
            }


            $index++;
            next;
        }

        #
        # checks that apply to all stanzas
        #

        # skip this stanza if it's for an install adapter
        if ($::stanzas[$index]{'machine_type'} ne "")
        {
            if ($::stanzas[$index]{'machine_type'} eq "install")
            {
                $index++;
                next;
            }
        }
        elsif ($defaults{"machine_type"} eq "install")
        {
            $index++;
            next;
        }

        #   check if hostname is resolvable
        my $tmpaddr = NetworkUtils->validate_ip($::stanzas[$index]{'hostname'});
        if ($tmpaddr == -1)
        {

            # Could not resolve hostname $::stanzas[$index]{'hostname'}
            #   on line $::linenums[$index]{'hostname'} in stanza file.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'W', 'EMsgADAPDEF_NoResHost',
                                     $::stanzas[$index]{'hostname'}, $linenum);
        }

        # check for machine_type
        if ($::stanzas[$index]{'machine_type'} ne "")
        {
            my $mtype = $::stanzas[$index]{'machine_type'};
            if (   ($mtype ne "secondary")
                && ($mtype ne "install")
                && ($mtype ne "etherchannel"))
            {

                # The value for the machine_type attribute, $mtype, on
                #   line $::linenums[$index]{'machine_type'} should be either
                #   secondary, install, or etherchannel.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_BadMachType', $mtype,
                    $::linenums[$index]{'machine_type'}
                    );
            }
        }
        elsif ($defaults{"machine_type"} eq "")
        {

            # Missing required attribute machine_type in the stanza
            #   starting on line $::linenums[$index]{'hostname'}
            MessageUtils->messageFromCat(
                'csmInstall.cat',          $::MSGMAPPATH,
                'csminstall',              'W',
                'EMsgADAPDEF_MissingAttr', 'machine_type',
                $::linenums[$index]{'hostname'}
                );
        }

        # check for adapter_type
		my $atype;
        if ($::stanzas[$index]{'adapter_type'} ne "")
        {
            $atype = $::stanzas[$index]{'adapter_type'};
            if (   ($atype ne "ent")
                && ($atype ne "sni")
				&& ($atype ne "iba")
                && ($atype ne "ml")
                && ($atype ne "mlt")
                && ($atype ne "tok")
                && ($atype ne "fiddi"))
            {

                # The value for the \'adapter_type\' attribute, (\'$atype\'), on
                #   line $::linenums[$index]{'adapter_type'} is not supported
                #   by CSM or AIX.  A user-provided method must be used for
                #   this adapter type.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_BadAdapType', $atype,
                    $::linenums[$index]{'adapter_type'}
                    );
            }
        }
        elsif ($defaults{"adapter_type"} ne "")
        {
			$atype = $defaults{"adapter_type"};
		}
		else
		{

            # Missing required attribute adapter_type in the stanza
            #	starting on line $::linenums[$index]{'hostname'}.
            MessageUtils->messageFromCat(
                'csmInstall.cat',          $::MSGMAPPATH,
                'csminstall',              'W',
                'EMsgADAPDEF_MissingAttr', 'adapter_type',
                $::linenums[$index]{'hostname'}
                );
        }

        #  need short hostname for %OSName
        my $shortHost = NodeUtils->getShorthost($::stanzas[$index]{'hostname'});

        if ($OSName{$shortHost} eq "AIX")
        {

            #
            #  AIX checks
            #

			#  if IB adapter then check for required attrs
			#        (ib_adapter, ib_port, interface_name )
			if ($atype eq "iba")

			{

				# check for ib_adapter
				if ( ($::stanzas[$index]{'ib_adapter'} eq "") && ($defaults{'ib_adapter'} eq ""))
				{

                	#  The attribute is not defined for the stanza
                	#   starting on line $::linenums[$index]{'hostname'}.
                	MessageUtils->messageFromCat(
                    	'csmInstall.cat',          $::MSGMAPPATH,
                    	'csminstall',              'W',
                    	'EMsgADAPDEF_MissingAttr', 'ib_adapter',
                    	$::linenums[$index]{'hostname'}
                                            );
            	}

				# check for ib_port
		if ( ($::stanzas[$index]{'ib_port'} eq "") && ($defaults{'ib_port'} eq ""))
            	{

                	#  The netaddr attribute is not defined for the stanza
                	#   starting on line $::linenums[$index]{'hostname'}.
                	MessageUtils->messageFromCat(
                    	'csmInstall.cat',          $::MSGMAPPATH,
                    	'csminstall',              'W',
                    	'EMsgADAPDEF_MissingAttr', 'ib_port',
                    	$::linenums[$index]{'hostname'}
                                            );
            	}

				# check for interface_name
		if (($::stanzas[$index]{'interface_name'} eq "") && ($defaults{'interface_name'} eq "") )
            	{
                	# Missing required attribute interface_name in the stanza
                	#   starting on line $::linenums[$index]{'hostname'}.
                	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall', 'W', 'EMsgADAPDEF_MissingAttr', 'interface_name', $::linenums[$index]{'hostname'} );

            	}

			}

            #  check for netaddr
            if ($::stanzas[$index]{'netaddr'} eq "")
            {

                #  The netaddr attribute is not defined for the stanza
                #	starting on line $::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_MissingAttr', 'netaddr',
                    $::linenums[$index]{'hostname'}
                    );

            }
            elsif (!NetworkUtils->isIpaddr($::stanzas[$index]{'netaddr'}))
            {

                # is it valid IP
                # The netaddr value of $::stanzas[$index]{'netaddr'}
                #	on line $::linenums[$index]{'netaddr'} is not a valid IP
                #	address.
                MessageUtils->messageFromCat(
                                             'csmInstall.cat',
                                             $::MSGMAPPATH,
                                             'csminstall',
                                             'W',
                                             'EMsgADAPDEF_BadIP',
                                             'netaddr',
                                             $::stanzas[$index]{'netaddr'},
                                             $::linenums[$index]{'netaddr'}
                                             );
            }

            #  check for subnet_mask
            if ($::stanzas[$index]{'subnet_mask'} ne "")
            {
                if (!NetworkUtils->isIpaddr($::stanzas[$index]{'subnet_mask'}))
                {

                    # The subnet_mask value $::stanzas[$index]{'subnet_mask'}
                    #   on line $::linenums[$index]{'netaddr'} is not a valid IP
                    #   address.
                    MessageUtils->messageFromCat(
                                              'csmInstall.cat',
                                              $::MSGMAPPATH,
                                              'csminstall',
                                              'W',
                                              'EMsgADAPDEF_BadIP',
                                              'subnet_mask',
                                              $::stanzas[$index]{'subnet_mask'},
                                              $::linenums[$index]{'subnet_mask'}
                                              );

                }

            }
            elsif ( ($defaults{'subnet_mask'} eq "") && ($atype ne "iba"))
            {

                # The subnet_mask attribute is not defined for the stanza
                #   starting on line $::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_MissingAttr', 'subnet_mask',
                    $::linenums[$index]{'hostname'}
                    );
            }

            # check for either interface_type or network_type
            #  need check for possible values!!!!
            my $found_type = "";
            if ($::stanzas[$index]{'interface_type'} ne "")
            {
                $found_type = $::stanzas[$index]{'interface_type'};
            }
            elsif ($defaults{'interface_type'} ne "")
            {
                $found_type = $defaults{'interface_type'};
            }
            if ($::stanzas[$index]{'network_type'} ne "")
            {
                $found_type = $::stanzas[$index]{'network_type'};
            }
            elsif ($defaults{'network_type'} ne "")
            {
                $found_type = $defaults{'network_type'};
            }
			if (($found_type eq "") && ($atype ne "iba") )
            {

                # Neither the interface_type nor the network_type attribute
                #	was set for the stanza starting on line
                #	$::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'W', 'EMsgADAPDEF_MissingType',
                                 , $::linenums[$index]{'hostname'});
            }

        }
        elsif ($OSName{$shortHost} eq "Linux")
        {

            #
            #  Linux checks
            #

            # required for CSM - (interface_name and MAC_address)

            # check for interface_name
            if (($::stanzas[$index]{'interface_name'} eq ""))
            {

                # Missing required attribute interface_name in the stanza
                #   starting on line $::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_MissingAttr', 'interface_name',
                    $::linenums[$index]{'hostname'}
                    );

            }

            # check for MAC_address
            if (($::stanzas[$index]{'MAC_address'} eq "") && ($atype ne "iba"))
            {

                # Missing required attribute 'MAC_address' in the stanza
                #   starting on line $::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_MissingAttr', 'MAC_address',
                    $::linenums[$index]{'hostname'}
                    );
            }

            #  used in ifcfg file - (IPADDR, NETMASK, BOOTPROTO)
            #    - these are the only attrs that seem to be used in all
            #        cases - SLES and RedHat
            #

            #  check for IPADDR
            if ($::stanzas[$index]{'IPADDR'} eq "")
            {

                # The IPADDR attribute is not defined for the stanza starting
                #	on line $::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_MissingAttr', 'IPADDR',
                    $::linenums[$index]{'hostname'}
                    );

            }
            elsif (!NetworkUtils->isIpaddr($::stanzas[$index]{'IPADDR'}))
            {

                # The IPADDR value of $::stanzas[$index]{'IPADDR'} on
                #	line $::linenums[$index]{'IPADDR'} is not a valid IP
                #	address.
                MessageUtils->messageFromCat(
                                             'csmInstall.cat',
                                             $::MSGMAPPATH,
                                             'csminstall',
                                             'W',
                                             'EMsgADAPDEF_BadIP',
                                             'IPADDR',
                                             $::stanzas[$index]{'IPADDR'},
                                             $::linenums[$index]{'IPADDR'}
                                             );

            }

            #  check for NETMASK
            if ($::stanzas[$index]{'NETMASK'} ne "")
            {
                if (!NetworkUtils->isIpaddr($::stanzas[$index]{'NETMASK'}))
                {

                    # is it valid IP
                    # The NETMASK value $::stanzas[$index]{'NETMASK'} on line
                    #   $::linenums[$index]{'NETMASK'} is not a valid IP address
                    MessageUtils->messageFromCat(
                                                 'csmInstall.cat',
                                                 $::MSGMAPPATH,
                                                 'csminstall',
                                                 'W',
                                                 'EMsgADAPDEF_BadIP',
                                                 'NETMASK',
                                                 $::stanzas[$index]{'NETMASK'},
                                                 $::linenums[$index]{'NETMASK'}
                                                 );
                }
            }
            elsif ($defaults{'NETMASK'} eq "")
            {

                # The NETMASK attribute is not defined for the stanza starting
                #	on line $::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_MissingAttr', 'NETMASK',
                    $::linenums[$index]{'hostname'}
                    );

            }

            #  check for BOOTPROTO
            if (   ($::stanzas[$index]{'BOOTPROTO'} eq "")
                && ($defaults{'BOOTPROTO'} eq ""))
            {

                # The BOOTPROTO attribute is not defined for the stanza
                #	starting on line $::linenums[$index]{'hostname'}.
                MessageUtils->messageFromCat(
                    'csmInstall.cat',          $::MSGMAPPATH,
                    'csminstall',              'W',
                    'EMsgADAPDEF_MissingAttr', 'BOOTPROTO',
                    $::linenums[$index]{'hostname'}
                    );
            }

			#
            # check for SLES attrs
			#
            if ($EffectiveDistroName{$shortHost} =~ /SLES/)
            {

                #  check for BROADCAST
                if (   ($::stanzas[$index]{'BROADCAST'} eq "")
                    && ($defaults{'BROADCAST'} eq ""))
                {

                    # The BROADCAST attribute is not defined for the stanza
                    #	starting on line $::linenums[$index]{'hostname'}.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_MissingAttr', 'BROADCAST',
                        $::linenums[$index]{'hostname'}
                        );
                }

                #  check for STARTMODE
                if (   ($::stanzas[$index]{'STARTMODE'} eq "")
                    && ($defaults{'STARTMODE'} eq ""))
                {

                    # The STARTMODE attribute is not defined for the stanza
                    #	starting on line $::linenums[$index]{'hostname'}.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_MissingAttr', 'STARTMODE',
                        $::linenums[$index]{'hostname'}
                        );
                }
            }

			#
            # check for RedHat attrs
			#
            if ($EffectiveDistroName{$shortHost} =~ /RedHat/)
            {

                #  check for HWADDR
                if ($::stanzas[$index]{'HWADDR'} eq "")
                {

                    # The HWADDR attribute is not defined for the stanza
                    #	starting on line $::linenums[$index]{'hostname'}.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_MissingAttr', 'HWADDR',
                        $::linenums[$index]{'hostname'}
                        );
                }

                #  check for DEVICE
                if (   ($::stanzas[$index]{'DEVICE'} eq "")
                    && ($defaults{'DEVICE'} eq ""))
                {

                    # The DEVICE attribute is not defined for the stanza
                    #	starting on line $::linenums[$index]{'hostname'}.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_MissingAttr', 'DEVICE',
                        $::linenums[$index]{'hostname'}
                        );
                }
				elsif ( $::stanzas[$index]{'DEVICE'} ne $::stanzas[$index]{'interface_name'})
				{
					# The DEVICE and interface_name attributes must be the same
					#  in the stanza starting on line xx
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall', 'W', 'EMsgADAPDEF_DeviceMismatch', $::linenums[$index]{'hostname'});
				}

                #  check for TYPE
                if (   ($::stanzas[$index]{'TYPE'} eq "")
                    && ($defaults{'TYPE'} eq ""))
                {

                    # The TYPE attribute is not defined for the stanza
                    #	starting on line $::linenums[$index]{'hostname'}.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_MissingAttr', 'TYPE',
                        $::linenums[$index]{'hostname'}
                        );
                }

                #  check for ONBOOT
                if (   ($::stanzas[$index]{'ONBOOT'} eq "")
                    && ($defaults{'ONBOOT'} eq ""))
                {

                    # The ONBOOT attribute is not defined for the stanza
                    #	starting on line $::linenums[$index]{'hostname'}.
                    MessageUtils->messageFromCat(
                        'csmInstall.cat',          $::MSGMAPPATH,
                        'csminstall',              'W',
                        'EMsgADAPDEF_MissingAttr', 'ONBOOT',
                        $::linenums[$index]{'hostname'}
                        );
                }
            }
        }

        $index++;
    }

    #  for each node add the name of the file to AdapterStanzaFile
    my $error = 0;
    foreach my $node (@NODELIST)
    {
        if (defined($stanzafile) && ($stanzafile ne ""))
        {

            NodeUtils->runrmccmd(
                'chrsrc-api',
                '-i',
                qq(-s IBM.ManagedNode::"Hostname='$node'"::AdapterStanzaFile::"$stanzafile"),
                0
                );

            if ($::RUNCMD_RC)
            {

                # An error occurred when attempting to set the
                #	AdapterStanzaFile attribute for node \'$node\'.
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                           'csminstall', 'W', 'EMsgADAPDEF_CantSetAttr', $node);

                $error++;
            }
        }
    }
    if ($error)
    {
        return $::NOK;
    }
    else
    {
        return $::OK;
    }
}

#-------------------------------------------------------------------------------

=head3    make_config_file

        Create configuration information files for each node provided
        in the nodelist reference argument.

        Arguments:
                 $ref_nodeList
                 $ref_nodeinfo
                 $ref_DestinationNodeHash
        Returns:
                $::OK
                $::NOK
        Globals:
                $::SETUP_REMOTE_SHELL
                $::NOSMS
        Error:
                $::NOK
        Example:
                if(ServerUtils->make_config_file( \@DestNode,
                                                   \@lsnode_info,
                                                   \%DestNodeHash) != 0) {
                    blah;
                }
        Comments:
                Requires $::REMOTE_SHELL and $::SETUP_REMOTE_SHELL to be set optionally,
                $::NOSMS can be set to instruct makenode to not perform software maintenance
                on Linux nodes

=cut

#--------------------------------------------------------------------------------

sub make_config_file
{

    my ($class, $ref_nl, $ref_nodeinfo, $ref_DestNodeHash) = @_;
    my @nl       = @$ref_nl;
    my @nodeinfo = @$ref_nodeinfo;

    #  DestNodeHash is a hash containing attribute values for each node
    #    that was provided on the command line.
    my %DestNodeHash = %$ref_DestNodeHash;

    my ($hname, $short, $rc);
    my $csmcfgdir = "/csminstall/csm/config";
    my $cmd;
    my $output;
    my ($host, $ipaddr, $shorthost);
    my @copystanzafiles;    # list of stanza files to copy
    my $instsvr;
	#These attributes can be print into config file directly if defined
	my @omit_attrs=(
			 "AllowManageRequest","ChangedAttributes","ConfigChanged","Name","NodeNameList",
			 "UpdatenodeFailed","InstallStatus","UniversalId");
   	my @special_attrs=(
			"AdapterStanzaFile","ManagementServer","InstallServer","InstallAdapterHostname",
			"InstallServerAKBNode","IsBlade","InstallMsgPort","CFM");
	#
    #  Check and/or create the /csminstall/csm/config directory
    #
    if (!-d $csmcfgdir)
    {
        $cmd = "$::MKDIR -m 755 -p $csmcfgdir";
        $rc  = system("$cmd");
        if ($rc >> 8)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', "E", 'EMsgNO_CreateDir', $csmcfgdir);
            return ($::NOK);
        }
    }

    # get the lists of user-provided cstomization script
    #   that we need to run on the nodes
    if (ServerUtils->make_script_lists(@nl))
    {

        # msg  -Could not get lists of customization scripts
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', "E", 'EMsgNoCustScripts');

        # continue
    }

    # get the lists of user-provided configuration methods
    #   that we need to run on the nodes - continue if errors
    ServerUtils->make_method_lists(@nl);

    my $cmd = "/usr/bin/lscondresp";
    my $outref = NodeUtils->runcmd($cmd, -1, 1);
    if ($::RUNCMD_RC == 0)
    {
        foreach my $line (@$outref)
        {
            if ($line =~ m/NodeFullInstallComplete.*SetupSSHAndRunCFM.*Active/)
            {
                foreach my $node (keys %DestNodeHash)
                {
                    $DestNodeHash{$node}{'CFM'} = 1;
                }
            }
        }
    }

    #
    # create the config_info file
    #

    #  go through each node in the list
    #   (the keys are the hostnames as returned from tryHost() !!)
    foreach $hname (keys %DestNodeHash)
    {

        # remove the old config file if any
        if (-e "$csmcfgdir/$hname.config_info")
        {
            unlink("$csmcfgdir/$hname.config_info");
        }

        # create the new file
        unless (open(CONFIG_OUT, ">$csmcfgdir/$hname.config_info"))
        {
            my $fn = "$csmcfgdir/$hname.config_info";
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                               'csminstall', "E", 'EMsgCANT_WRITE_FILE', "$fn");
            return $::NOK;
        }

        # Add cluster-wide configuration information to the file
        if (defined($::REMOTE_SHELL))
        {
            print CONFIG_OUT "RemoteShell=$::REMOTE_SHELL\n";
        }
        if (defined($::SETUP_REMOTE_SHELL))
        {
            print CONFIG_OUT "SetupRemoteShell=$::SETUP_REMOTE_SHELL\n";
        }

        # if make_config_file was called by csmsetup* cmds or updatenode -c
        #      - then we need to add the CONFIGADAPTERS attr
        my $csmsetupcmds = 0;
        my $progname     = NodeUtils->programName();

        if (   $progname =~ m/csmsetupks/
            || $progname =~ m/csmsetupnim/
            || $progname =~ m/csmsetupyast/
            || $progname =~ m/csmsetupinstall/
            || $progname =~ m/installnode/)
        {
            $csmsetupcmds = 1;
        }

        if (defined($::UPDATENODE_C)
            && !defined($DestNodeHash{$hname}{'AdapterStanzaFile'}))
        {

            # give a warning - they want to configure adapters but there is no
            #  stanza file for this node.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'W', 'EMsgNoStanzaFile', $hname);
        }

        if (defined($::UPDATENODE_C) || $csmsetupcmds)
        {

            # If the node's AdapterStanzaFile attribute is set then add the attr
            if (defined($DestNodeHash{$hname}{'AdapterStanzaFile'}) && $DestNodeHash{$hname}{'AdapterStanzaFile'} )
            {

                # add the filename to the list of files to copy and do it
                # at the end of the loop below to avoid multiple copies
                #  (- do this now to make sure we have the latest version)
                my $sfile = $DestNodeHash{$hname}{'AdapterStanzaFile'};
                if (!grep (/^$sfile$/, @copystanzafiles))
                {
                    push @copystanzafiles,
                      $DestNodeHash{$hname}{'AdapterStanzaFile'};
                }

				#  use the full pathname in the config_info file
				my $stanzafilename = $DestNodeHash{$hname}{'AdapterStanzaFile'};

                # write it to the file
                print CONFIG_OUT "CONFIGADAPTERS=$stanzafilename\n";

                # also add list of user methods if any
                if (defined(@{$::CONFIGMETHODS{$hname}}))
                {
                    print CONFIG_OUT
                      "CONFIGMETHODS=@{$::CONFIGMETHODS{$hname}}\n";
                }
            }
        }

        # add the node configuration information to the file
        if (defined($hname))
        {
            print CONFIG_OUT "Hostname=$hname\n";
            print CONFIG_OUT "HostnameIP="
              . NetworkUtils->validate_ip($DestNodeHash{$hname}{'Hostname'})
              . "\n";
        }
        if (defined($DestNodeHash{$hname}{'ManagementServer'}))
        {
            print CONFIG_OUT
              "ManagementServer=$DestNodeHash{$hname}{'ManagementServer'}\n";
            my ($ms_host, $ms_ip) =
              NetworkUtils->getHost($DestNodeHash{$hname}{'ManagementServer'});
            print CONFIG_OUT "ManagementServerIP=$ms_ip\n";
            print CONFIG_OUT "ManagementServerHost=$ms_host\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallServer'})
            && ($DestNodeHash{$hname}{'InstallServer'} ne ""))
        {
            my $isISGrp = 0;    #whether the install server is a node group
            my ($server, $rep_dir) = split ':',
              $DestNodeHash{$hname}{'InstallServer'};
            my $ref_grp;
            my ($msvr_ip, $isvr_ip);

            $ref_grp = NodeUtils->getNodegrp($server);
            if ($ref_grp)
            {
                print CONFIG_OUT "InstallServerIsGroup=yes\n";
                $isISGrp = 1;
                $instsvr = $DestNodeHash{$hname}{'ManagementServer'};
            }
            else
            {
                (undef, $isvr_ip) = NetworkUtils->getHost($server);
                (undef, $msvr_ip) =
                  NetworkUtils->getHost(
                                     $DestNodeHash{$hname}{'ManagementServer'});
                $instsvr = $server;
            }

            my $isvrakbnode_ip =
              ServerUtils->getDefaultInstallServerAKBNode($hname,
                                                          \%DestNodeHash);
            if ($isvrakbnode_ip)
            {
                print CONFIG_OUT "InstallMsgServer=$isvrakbnode_ip\n";
            }
            my $progname = NodeUtils->programName();
            if ($progname =~ m/installnode/ || $progname =~ m/csmsetupnim/)
            {
                if ($isvrakbnode_ip)
                {
                    if ($rep_dir)
                    {
                        print CONFIG_OUT
                          "InstallServer=$isvrakbnode_ip:$rep_dir\n";
                    }
                    else
                    {
                        print CONFIG_OUT "InstallServer=$isvrakbnode_ip\n";
                    }

                    # Resolve the ip to host name and write it to config_info file
                    my ($is_hostname, undef) =
                      ServerUtils->dshGetNodeAddrInfo($isvrakbnode_ip,
                                                      $instsvr);
                    if ($is_hostname != -1)
                    {
                        print CONFIG_OUT "InstallServerHost=$is_hostname\n";
                    }
                }
            }
            elsif ($isvr_ip && $msvr_ip)
            {

                #not using the ms as the install server
                print CONFIG_OUT
                  "InstallServer=$DestNodeHash{$hname}{'InstallServer'}\n";
            }
        }
        elsif (defined($DestNodeHash{$hname}{'ManagementServer'}))
        {
            my $ms = $DestNodeHash{$hname}{'ManagementServer'};
            my ($hostname, $ip) = NetworkUtils->getHost($ms);
            print CONFIG_OUT "InstallMsgServer=$ip\n";

            $instsvr = $DestNodeHash{$hname}{'ManagementServer'};
        }
        # Handle Linux-specific attributes.
        if ($DestNodeHash{$hname}{'InstallOSName'} eq "Linux")
        {

            # if $::NOSMS isn't set and if the node is not diskless node,
            # then perform software maintenance
            if (   (!$::NOSMS)
                && ($DestNodeHash{$hname}{'InstallMethod'} ne "diskless"))
            {
                print CONFIG_OUT "SMS=1\n";
            }
        }


        # put in the SetupKRB5 attribute value
        if (!defined $::KRB5SETUP)
        {

            $::KRB5SETUP = NodeUtils->getSetupKRB5;
        }
        print CONFIG_OUT "SetupKRB5=$::KRB5SETUP\n";

        # put in what would be the KRB5 principal name for ManagementServer
        if (!defined $::MSNAME)
        {
            $::MSNAME = NetworkUtils->getRSCThostname;
        }
        my $msnamelower = lc($::MSNAME);
        print CONFIG_OUT "CSMKRB5Principal=ctsrcmdp.$msnamelower";

        # put in what would be the KRB5 principal name for HA ManagementServer
        if (!defined $::HAMODE)
        {
            my $outref =
              NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                   "-s IBM.DmsCtrl::::HAMode");
            $::HAMODE = $$outref[0];
        }
        if ($::HAMODE == 1)
        {    #on an active HAMS management server
            require HAMS;
            my $prom = NodeUtils->programName();
            if ($prom =~ m/csmsetupks/)
            {
                delete $ENV{'POSIXLY_CORRECT'};
            }
            my $inactivems      = HAMS->getInactiveRSCTHostname(); # get HA name
            my $inactivemslower = lc($inactivems);
            print CONFIG_OUT ",ctsrcmdp.$inactivemslower\n";

        }
        else
        {
            print CONFIG_OUT "\n";
        }

        if (($::SETUP_REMOTE_SHELL == 1) && ($::REMOTE_SHELL =~ m/rsh/))
        {
            if (!defined $::HAMODE)
            {
                my $outref =
                  NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                       "-s IBM.DmsCtrl::::HAMode");
                $::HAMODE = $$outref[0];
            }
            if ($::HAMODE == 1)
            {    #on an active HAMS management server
                    #get management server attribute for node on other ms
                require HAMS;
                my $other_ms = HAMS->getManagementServerAttr($hname);
                if ($other_ms)
                {
                    print CONFIG_OUT "InActiveMSAttr=$other_ms\n";
                }
            }
        }

        #
        # add the /etc/hosts info for the management server - just AIX for now
        #
        if ($DestNodeHash{$hname}{'InstallOSName'} eq "AIX")
        {
            ($host, $ipaddr) =
              NetworkUtils->getHost($DestNodeHash{$hname}{'ManagementServer'});
            if ($ipaddr && $ipaddr ne $host)
            {
                ($shorthost = $host) =~ s/\..*$//;
                print CONFIG_OUT "ETCHOSTSMS=$ipaddr $host $shorthost\n";
            }

        }

        print CONFIG_OUT "ManagementProduct=CSM\n";

        if (defined($DestNodeHash{$hname}{'InstallAdapterHostname'})
            && $DestNodeHash{$hname}{'InstallAdapterHostname'})
        {

            # resolve the ip address of InstallAdapterHostname
            if (
                !NetworkUtils->isIpaddr(
                                 $DestNodeHash{$hname}{'InstallAdapterHostname'}
                )
              )
            {

                # call dsh to resolve InstallAdapterHostname on the install server machine.
                my ($adp_hostname, $adp_hostip) =
                  ServerUtils->dshGetNodeAddrInfo(
                                $DestNodeHash{$hname}{'InstallAdapterHostname'},
                                $instsvr);
                if ($adp_hostip != -1)
                {

                    # write the ip address to config_info file
                    print CONFIG_OUT "InstallAdapterHostname=$adp_hostname\n";
                    print CONFIG_OUT "InstallAdapterHostnameIP=$adp_hostip\n";
                    print CONFIG_OUT "InstallAdapterIP=$adp_hostip\n";
                }
                else
                {
                    print CONFIG_OUT
                      "InstallAdapterHostname=$DestNodeHash{$hname}{'InstallAdapterHostname'}\n";
                    print CONFIG_OUT
                      "InstallAdapterHostnameIP=$DestNodeHash{$hname}{'InstallAdapterHostname'}\n";
                    print CONFIG_OUT
                      "InstallAdapterIP=$DestNodeHash{$hname}{'InstallAdapterHostname'}\n";
                }
            }
            else
            {
                print CONFIG_OUT
                "InstallAdapterHostname=$DestNodeHash{$hname}{'InstallAdapterHostname'}\n";
            }
        }
        else
        {
            print CONFIG_OUT "InstallAdapterHostname=\n";
            print CONFIG_OUT "InstallAdapterHostnameIP=\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallServerAKBNode'})
            && $DestNodeHash{$hname}{'InstallServerAKBNode'})
        {

            # resolve the ip address of InstallServerAKBNode
            if (
                !NetworkUtils->isIpaddr(
                                   $DestNodeHash{$hname}{'InstallServerAKBNode'}
                )
              )
            {

                # call dsh to resolve InstallServerAKBNode on the install server machine.
                my ($isakb_host, $isakb_nodeip) =
                  ServerUtils->dshGetNodeAddrInfo(
                                  $DestNodeHash{$hname}{'InstallServerAKBNode'},
                                  $instsvr);
                if ($isakb_nodeip != -1)
                {

                    # write the ip address to config_info file
                    print CONFIG_OUT "InstallServerAKBNode=$isakb_host\n";
                    print CONFIG_OUT "InstallServerAKBNodeIP=$isakb_nodeip\n";
                }
                else
                {
                    print CONFIG_OUT
                      "InstallServerAKBNode=$DestNodeHash{$hname}{'InstallServerAKBNode'}\n";
                    print CONFIG_OUT
                      "InstallServerAKBNodeIP=$DestNodeHash{$hname}{'InstallServerAKBNode'}\n";
                }
            }
            else
            {
                   print CONFIG_OUT
                   "InstallServerAKBNode=$DestNodeHash{$hname}{'InstallServerAKBNode'}\n";
            }
        }
        else
        {
            print CONFIG_OUT "InstallServerAKBNode=\n";
            print CONFIG_OUT "InstallServerAKBNodeIP=\n";
        }


        if (defined($DestNodeHash{$hname}{'IsBlade'}))
        {
            print CONFIG_OUT "IsBlade=$DestNodeHash{$hname}{'IsBlade'}\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallMsgPort'}))
        {
            print CONFIG_OUT
              "InstallMsgPort=$DestNodeHash{$hname}{'InstallMsgPort'}\n";
        }
        if (defined($DestNodeHash{$hname}{'CFM'}))
        {
            print CONFIG_OUT "CFM=1\n";
        }

		foreach my $attr (keys %{$DestNodeHash{$hname}})
		{
        	if (!grep(/^$attr$/,@omit_attrs) && !grep(/^$attr$/,@special_attrs))
        	{
            	print CONFIG_OUT
              "$attr=$DestNodeHash{$hname}{$attr}\n";
        	}
			
		}		
        # add lists of scripts to run on the node
        if (defined(@{$::PREINSTSCRIPTS{$hname}}))
        {
            print CONFIG_OUT "PREINSTSCRIPTS=@{$::PREINSTSCRIPTS{$hname}}\n";
        }
        if (defined(@{$::POSTINSTSCRIPTS{$hname}}))
        {
            print CONFIG_OUT "POSTINSTSCRIPTS=@{$::POSTINSTSCRIPTS{$hname}}\n";
        }
        if (defined(@{$::PREOSUPGRADESCRIPTS{$hname}}))
        {
            print CONFIG_OUT "PREOSUPGRADESCRIPTS=@{$::PREOSUPGRADESCRIPTS{$hname}}\n";
        }
        if (defined(@{$::POSTOSUPGRADESCRIPTS{$hname}}))
        {
            print CONFIG_OUT "POSTOSUPGRADESCRIPTS=@{$::POSTOSUPGRADESCRIPTS{$hname}}\n";
        }
        if (defined(@{$::UPDATESCRIPTS{$hname}}))
        {
            print CONFIG_OUT "UPDATESCRIPTS=@{$::UPDATESCRIPTS{$hname}}\n";
        }
        if (defined(@{$::DISKLESSPREBUILDSCRIPTS{$hname}}))
        {
            print CONFIG_OUT
              "DISKLESSPREBUILDSCRIPTS=@{$::DISKLESSPREBUILDSCRIPTS{$hname}}\n";
        }
        if (defined(@{$::DISKLESSBOOTSCRIPTS{$hname}}))
        {
            print CONFIG_OUT
              "DISKLESSBOOTSCRIPTS=@{$::DISKLESSBOOTSCRIPTS{$hname}}\n";
        }

        # New added, but not sure whether need them all.
        # OS_DIST_TOP,  OS_DIST_RPMS, OPEN_SRC_RPM_DIR, DRVR_TOP_DIR, PATCH_FILE_DIR
        # PREREBOOT_SCRIPT_DIR , FIRSTBOOT_SCRIPT_DIR, UPDATE_SCRIPT_DIR, DATA_SCRIPT_DIR
        # DSERVER_DIR, STATUS_FILE_DIR, CSM_PACKAGES_DIR, SMS_INSTALL_DIR, SMS_UPDATES_DIR
        my $CSMCLIENTMNTDIR = "/var/opt/csm/mnt";
        my $os_dist_top     =
          "$CSMCLIENTMNTDIR/$DestNodeHash{$hname}{'InstallOSName'}/$DestNodeHash{$hname}{'InstallDistributionName'}/$DestNodeHash{$hname}{'InstallDistributionVersion'}/$DestNodeHash{$hname}{'InstallPkgArchitecture'}";
        my $csm_pkg_dir =
          "$CSMCLIENTMNTDIR/$DestNodeHash{$hname}{'InstallOSName'}/$DestNodeHash{$hname}{'InstallDistributionName'}/$DestNodeHash{$hname}{'InstallDistributionVersion'}/csm/$DestNodeHash{$hname}{'InstallCSMVersion'}/packages";
        my ($effectiveDistroName, $effectiveDistroVer) =
            NodeUtils->getEffectiveDistro(
                   $DestNodeHash{$hname}{'InstallDistributionName'},
                   $DestNodeHash{$hname}{'InstallDistributionVersion'}
                   ); 

        print CONFIG_OUT "OS_DIST_TOP=$os_dist_top\n";
        print CONFIG_OUT "OS_DIST_RPMS=$os_dist_top/RPMS\n";
        print CONFIG_OUT "OPEN_SRC_RPM_DIR=$csm_pkg_dir\n";
        print CONFIG_OUT "DRVR_TOP_DIR=$os_dist_top/drivers\n";
        print CONFIG_OUT "PATCH_FILE_DIR=$os_dist_top/patches\n";

        print CONFIG_OUT
          "PREREBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/installprereboot\n";
        print CONFIG_OUT
          "FIRSTBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/installpostreboot\n";
        print CONFIG_OUT
          "OSUPGRADE_PREREBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/osupgradeprereboot\n";
        print CONFIG_OUT
          "OSUPGRADE_FIRSTBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/osupgradepostreboot\n";
        print CONFIG_OUT
          "UPDATE_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/update\n";
        print CONFIG_OUT "DATA_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/data\n";

        print CONFIG_OUT "OS_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm\n";    #????
        print CONFIG_OUT "STATUS_FILE_DIR=$CSMCLIENTMNTDIR/csm/status\n";

        print CONFIG_OUT "CSM_PACKAGES_DIR=$csm_pkg_dir\n";
        print CONFIG_OUT "SMS_INSTALL_DIR=$os_dist_top/install\n";
        print CONFIG_OUT "SMS_UPDATES_DIR=$os_dist_top/updates\n";
        print CONFIG_OUT "RPMDir=$::pkgdefs{DISTRO_RPMDIR}\n";
        print CONFIG_OUT "EffectiveDistroName=$effectiveDistroName\n";
		# Warewulf relevant attributes
		print CONFIG_OUT "InstallImage=$DestNodeHash{$hname}{'InstallImage'}\n";
		if (defined $DestNodeHash{$hname}{'DisklessType'})
		{
			print CONFIG_OUT "DisklessType=$DestNodeHash{$hname}{'DisklessType'}\n";
		}
		if (defined $DestNodeHash{$hname}{'DisklessKernel'})
		{
			print CONFIG_OUT "DisklessKernel=$DestNodeHash{$hname}{'DisklessKernel'}\n";
		}
		if (defined $DestNodeHash{$hname}{'DisklessInitrd'})
		{
			print CONFIG_OUT "DisklessInitrd=$DestNodeHash{$hname}{'DisklessInitrd'}\n";
		}
		if (defined $DestNodeHash{$hname}{'DisklessZimage'})
		{
			print CONFIG_OUT "DisklessZimage=$DestNodeHash{$hname}{'DisklessZimage'}\n";
		}

        if (defined $ENV{'CSM_NOSTATUS'} && $ENV{'CSM_NOSTATUS'} != 0)
        {
            print CONFIG_OUT "CSM_NOSTATUS=1\n";
        }
        close(CONFIG_OUT);

	    MessageUtils->messageFromCat('csmInstall.cat', 
                                     $::MSGMAPPATH,
                                     'csminstall','V',
                                     'IMsgCREATE_CONFIG_INFO_FILE',"$csmcfgdir/$hname.config_info");
        chmod 0644, "$csmcfgdir/$hname.config_info";

        #
        # Add symbolic links
        #   Hopefully the scripts that run on the
        #   node will recognize one of these names!!
        #
        my $short_hname = NodeUtils->getShorthost($hname);
        my $node_ip     = NetworkUtils->validate_ip($hname);

        foreach my $linkName ($short_hname, $node_ip)
        {
            if ($linkName ne $hname)
            {
                $cmd =
                  "cd $csmcfgdir; $::LN -fs $hname.config_info $linkName.config_info";
                $rc = system("$cmd");
                if ($rc >> 8)
                {
                    MessageUtils->messageFromCat('csmInstall.cat',
                          $::MSGMAPPATH, 'csminstall', 'E', 'EMsgCMD_FAILED_RC',
                          $::LN, $rc);
                }
            }
        }

    }    # go do the next node

    #  copy the adapter stanza files to /csminstall/csm/config/adapters
    #  do this now just in case the file has been changed recently
    if (@copystanzafiles)
    {

		my $tmpstanzadir = "/csminstall/csm/config/adapters";

		foreach my $file (@copystanzafiles)
		{
			my $scriptdir = dirname($file);
			my $targetdir = $tmpstanzadir . $scriptdir;

			if (!-d $targetdir)
			{
				mkpath($targetdir, $::VERBOSE, 0755);
			}

			my $cmd = "$::COPY -p $file $targetdir";
			NodeUtils->runcmd($cmd, 0);
		}

    }

    return $::OK;
}

#-------------------------------------------------------------------------------

=head3    dshGetNodeAddr

      This translate the hostname to ip address on install server side.
      
=cut

#-------------------------------------------------------------------------------
sub dshGetNodeAddr
{
    my ($class, $node, $instserv) = @_;
    my ($host, $hostip) = ServerUtils->dshGetNodeAddrInfo($node, $instserv);

    return $hostip;
}

#-------------------------------------------------------------------------------

=head3    dshGetNodeHost

      This resolves the the host name to FQDN and IP on install server side
      
=cut

#-------------------------------------------------------------------------------
sub dshGetNodeAddrInfo
{
	my ($class,$node,$instserv) = @_;
	my ($cmd, $output, $addr,$host);
        my %options_api = ();
	$options_api{'nodes'} = $instserv;
	if(NetworkUtils->isIpaddr($node)){
		$options_api{'command'} = "echo \"use Net::hostent; use Socket; print inet_ntoa(gethost('$node')->addr); print '::::';  print gethostbyaddr(gethost('$node')->addr)->name \" | perl";
	}else{
		$options_api{'command'} = "echo \"use Net::hostent; use Socket; print inet_ntoa(gethost('$node')->addr); print '::::';  print gethost('$node')->name; \" | perl";
	}

	chomp($output = NodeUtils->runDsh(\%options_api, -1));
	if ($::RUNCMD_RC || $output =~ /(\d+\.\d+\.\d+\.\d+)::::(.*)/){
		return NetworkUtils->getHost($node);
	}else{
		$host = $2;
		$addr = $1;
		return ($host,$addr);
	}	
}

#-------------------------------------------------------------------------------

=head3    getDefaultInstallServerAKBNode
        
		If InstallServerAKBNode attribute is not specified with InstallServer attribute, 
		this subroutine can be used to acquire the default install server ip address 
		acknowledged by the node.

        Arguments:
                 $node
                 $ref_DestinationNodeHash
        Returns:
                Default InstallServerAKBNode of the node
                
        Globals:
                %::NODEHASH
        Error:
                -1: Can not acquire default InstallServerAKBNode of the node
        Example:
				$akbserver = ServerUtils->getDefaultInstallServerAKBNode($node,
                                                   \%DestNodeHash);
				if($akbserver != -1){
                    blah;
                }
        Comments:
			It must be guaranteed that DSH from MS to IS is reachable.
=cut

#-------------------------------------------------------------------------------

sub getDefaultInstallServerAKBNode
{
    my ($class, $node, $refNodeHash) = @_;
    if (!$refNodeHash)
    {
        $refNodeHash = \%::NODEHASH;
    }

    my ($server) = split ":", $refNodeHash->{$node}{"InstallServer"};

    my $ms_used = 0;                                # MS is choosen as IS?
    my $ref_grp = NodeUtils->getNodegrp($server);
    if ($ref_grp)
    {
        $server = NodeUtils->getNodeFromGroup($server, $node);
    }

    my $adapterHostname = $node;
    if ($refNodeHash->{$node}{"InstallAdapterHostname"} && !$ref_grp)
    {
        $adapterHostname = $refNodeHash->{$node}{"InstallAdapterHostname"};
    }

    my $ip;
    if (NodeUtils->isMgmtSvr)
    {
        my $cmd = "";
	my %options_api = ();
	$options_api{'nodes'} = $server;
	$options_api{'command'} = "$::GETSOURCEIP2TARGET $adapterHostname";
        $ip = NodeUtils->runDsh(\%options_api, -1);
        if ($::RUNCMD_RC)
        {

            # error running the command. reformat the output.
            if ($ip =~ /no such host found/)
            {
                $ip = "Can not resolve $adapterHostname on $server";
            }
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                   'NodeUtils', "E$::RUNCMD_RC", 'EMsgCMD_FAILED', $::RUNCMD_RC,
                   $cmd, $ip);

        }
        chomp($ip);
        ($ip) = $ip =~ /:\s*(\d+\.\d+\.\d+\.\d+)$/;

    }
    else
    {
        $ip = NodeUtils->runcmd("$::GETSOURCEIP2TARGET $adapterHostname");
        chomp($ip);
        ($ip) = $ip =~ /^(\d+\.\d+\.\d+\.\d+)$/;
    }

    if (!$ip || $::RUNCMD_RC)
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'E1',
                                     'EMsgHostnameUnknownOnNode',
                                     $adapterHostname,
                                     $server
                                     );
        return -1;
    }
    return $ip;
}

#-------------------------------------------------------------------------------

=head3    make_osprereboot_script

    Creates the modified osprereboot script in the /csminstall/csm directory.


=cut

#-------------------------------------------------------------------------------
sub make_osprereboot_script
{

    require DCUtils;
    my ($hname, $msname, $ISname, $instadaptr, $akbsvrname, $nfsserver);
    my ($ISvalue, $instname);
    my ($server, $dir, $ipaddr);
    my $newosprereboot  = "/csminstall/csm/osprereboot";
    my $origosprereboot = "/opt/csm/csmbin/osprereboot";
    my ($IS_IP, $ISAKB_IP, $MS_IP);

    # get rid of the old /csminstall/csm/osprereboot
    if (-f $newosprereboot)
    {
        unlink($newosprereboot);
    }

    # open the existing osprereboot file in /opt/csm/csmbin
    unless (open(ORIGOSPRE, "< $origosprereboot"))
    {

        # Couldn't open
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',     $::MSGMAPPATH,
                                     'csminstall',         'W',
                                     'EMsgCANT_READ_FILE', $origosprereboot
                                     );
        return $::NOK;
    }

    # create a new /csminstall/csm/osprereboot
    unless (open(NEWOSPRE, ">> $newosprereboot"))
    {

        # Couldn't open
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',      $::MSGMAPPATH,
                                     'csminstall',          'W',
                                     'EMsgCANT_WRITE_FILE', $newosprereboot
                                     );
        return $::NOK;
    }

    # set the permissions on the new file
    #NodeUtils->runcmd("$::CHMOD 755 $newosprereboot", 0);
    chmod 0755, $newosprereboot;

    # get node data for every node
    my $outref =
      NodeUtils->runrmccmd(
        'lsrsrc-api',
        "-i",
        "-s IBM.ManagedNode::::Hostname::ManagementServer::InstallServer::InstallServerAKBNode::InstallAdapterHostname::NFSServer"
        );

    if ($::RUNCMD_RC)
    {
        $outref = [];
    }

    #  Copy the installed osprereboot, line by line, to a new
    #     osprereboot script in the /csminstall/csm directory.
    #     Insert the node name and install server information
    #     for every node in the cluster.
    while (my $line = <ORIGOSPRE>)
    {

        # if the line contains IMBEDDEDISINFO then start imbedding info
        #         after it
        if (grep /IMBEDDEDISINFO/, $line)
        {
            print NEWOSPRE $line;

            # for each node add the InstallAdapterHostname and InstallAdapter
            #  need to include all the nodes defined in the database
            #  if this is a performance issue we might have to look at
            # using a condition/response to call this routine!!!!!

            foreach my $l (@$outref)
            {
                ($hname, $msname, $ISname, $akbsvrname, $instadaptr, $nfsserver)
                  = split(/::/, $l);

                chomp $instadaptr;

                if ($ISname eq "")
                {
                    if ( $nfsserver)
					{
                        ($server, $dir) = split ':', $nfsserver;
                        $server =~ s/^\s*//;    # delete leading white space
                        $server =~ s/\s*$//;    # delete trailing spaces
                        chomp $server;
                        chomp $dir;
                    }
                    if ($nfsserver && NetworkUtils->validate_ip( $server) ne NetworkUtils->validate_ip( $msname))
                    {
                        $ipaddr = NetworkUtils->validate_ip($server);
                        if ($ipaddr == -1)
                        {
                            MessageUtils->messageFromCat('HCnodecmds.cat',
                                                $::MSGMAPPATH, 'common', 'E',
                                                'EMsgHostnameUnknown', $server);
                            next;
                        }

                        if ($dir)
                        {
                            $ISvalue = $ipaddr . ":" . $dir;
                        }
                        else
                        {
                            $ISvalue = $ipaddr . ":/csmserver";
                        }

                    }
                    elsif ($akbsvrname ne "")
                    {

                        # If there is no Install Server, InstallServerAKBNode means the
                        # second adapter's hostname of MS, MN should mount direcotory
                        # 'csminstall' on this IP.
                        $ipaddr = NetworkUtils->validate_ip($akbsvrname);
                        if ($ipaddr == -1)
                        {
                            MessageUtils->messageFromCat('HCnodecmds.cat',
                                            $::MSGMAPPATH, 'common', 'E',
                                            'EMsgHostnameUnknown', $akbsvrname);
                            next;
                        }
                        $ISvalue = $ipaddr . ":/csminstall";
                    }
                    else
                    {
                        # In case ManagementServer attribute is blank
						if (!$msname)
                        {
                            $msname = NetworkUtils->get_MSip($hname);
                        }

                        $ipaddr = NetworkUtils->validate_ip($msname);
                        if ($ipaddr == -1)
                        {
                            MessageUtils->messageFromCat('HCnodecmds.cat',
                                                $::MSGMAPPATH, 'common', 'E',
                                                'EMsgHostnameUnknown', $msname);
                            next;
                        }
                        $ISvalue = $ipaddr . ":/csminstall";
                    }

                }
                else
                {
                    ($server, $dir) = split ':', $ISname;
                    $server =~ s/^\s*//;    # delete leading white space
                    $server =~ s/\s*$//;    # delete trailing spaces
                    chomp $server;
                    chomp $dir;

                    #maybe install server is a nodegrp
                    #then akbsvrname have no effect , and choose a node from the group list.
                    my $is_node = NodeUtils->getNodeFromGroup($server, $hname);
                    if ($is_node)
                    {

                        #do not care a akbsvrname if the install server is nodegrp
                        $akbsvrname = "";
                        $server     = $is_node;
                    }

                    if ($akbsvrname)
                    {
                        $ipaddr =
                          ServerUtils->dshGetNodeAddr($akbsvrname, $server);
                        if ($ipaddr == -1)
                        {
                            MessageUtils->messageFromCat(
                                     'csmInstall.cat',            $::MSGMAPPATH,
                                     'csminstall',                'E',
                                     'EMsgHostnameUnknownOnNode', $akbsvrname,
                                     $server
                                     );
                            next;
                        }
                    }
                    else
                    {
                        $ipaddr = NetworkUtils->validate_ip($server);
                        if ($ipaddr == -1)
                        {
                            MessageUtils->messageFromCat('HCnodecmds.cat',
                                                $::MSGMAPPATH, 'common', 'E',
                                                'EMsgHostnameUnknown', $server);
                            next;
                        }
                    }

                    $ISAKB_IP = $ipaddr;

                    if ($dir eq "")
                    {
                        (undef, $MS_IP) = NetworkUtils->getHost($msname);
                        (undef, $IS_IP) = NetworkUtils->getHost($server);

                        # if the install server is the management server
                        # then use /csminstall - otherwise use /csmserver.
                        if ($ISAKB_IP eq $MS_IP || $IS_IP eq $MS_IP)
                        {
                            $ISvalue = $ipaddr . ":/csminstall";
                        }
                        else
                        {
                            $ISvalue = $ipaddr . ":/csmserver";
                        }

                    }
                    else
                    {
                        $ISvalue = $ipaddr . ":/$dir";
                    }

                }

                # imbed the information into the osprereboot script in
                #      the /csminstall/csm directory.
                print NEWOSPRE "\$::ISinfo{\"" . $hname . "\"}" . "=" . "\""
                  . $ISvalue . "\"" . ";\n";

            }
        }
        elsif (grep /IMBEDDEDTZINFO/, $line)
        {
            print NEWOSPRE $line;
            my $dt = DCUtils->getMSTimeZone;
            print NEWOSPRE "\$ENV{\'TZ\'} = \"$dt\";\n";
        }
        else
        {

            # copy the line to the new file
            print NEWOSPRE $line;
        }
    }

    close(NEWOSPRE);
    close(ORIGOSPRE);

    return $::OK;
}

#-------------------------------------------------------------------------------

=head3    make_node_list_file

        Makes a node list file.  

        Arguments:
                (\@list_of_nodes) - reference to an arrary of nodes.
        Returns:
                $file_name and sets the global var: $::NODE_LIST_FILE
        Globals:
                the ENV vars: DSH_LIST,  RPOWER_LIST,  RCONSOLE_LIST
        Error:
                None documented
        Example:
                ServerUtils->make_node_list_file(\@nodelist); 

        Comments:
                IMPORTANT:
          Make sure to cleanup afterwards with:

                         ServerUtils->close_delete_file($file_handle, $file_name)

=cut

#--------------------------------------------------------------------------------

sub make_node_list_file
{
    my ($class, $ref_node_list) = @_;
    my @node_list = @$ref_node_list;
    srand(time | $$);    #random number generator start

    my $file = "/tmp/csm_$$";
    while (-e $file)
    {
        $file = ServerUtils->CreateRandomName($file);
    }

    open($::NODE_LIST_FILE, ">$file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',      $::MSGMAPPATH,
                                      'csminstall',          "E",
                                      'EMsgCANT_WRITE_FILE', "$file"
                                      );
    foreach my $node (@node_list)
    {
        print $::NODE_LIST_FILE "$node\n";
    }
    return $file;
}

#---------------------------------------------------------------------------

=head3    make_method_lists

			Create a list of user-provided adapter configuration methods
			for each node.  The user methods may be found in the directory
			/csminstall/csm/scripts/data

        Arguments:
                @nodelist
        Returns:
                $::OK   - on success
                1       - on error
        Globals:
                %::CONFIGMETHODS
        Example:
                if( ServerUtils->make_method_lists(@nodelist) ) { --- }_
        Comments:
        		none

=cut

#-------------------------------------------------------------------------------
sub make_method_lists
{

    my ($junk, @nodel) = @_;   # the list of nodes passed in on the command line
    my $dirname;               #  directory to check for methods
    my $method;                #  unique method name
    my $cmd;
    my $rc;

    $dirname = "/csminstall/csm/scripts/data";

    $rc = ServerUtils->get_script_list($dirname, @nodel);
    if ($rc == $::NOK)
    {

        #could not get list of user-provided adapter configuration methods.
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', "W", 'EMsgNoUserMethods');
        return $::NOK;
    }

    # check each method name in this directory
    foreach $method (sort @::scriptlist)
    {
        chomp $method;

        #  see if name starts with "config_adapters"
        my $methodstring = "config_adapters";
        if (!(grep /^$methodstring/, $method))
        {
            next;
        }

        # see if there are any nodes in the list
        if (@{$::scriptnodes{$method}}[0] eq $::NOK)
        {
            next;
        }

        # see if the script is executable
        if (!(-x "$dirname/$method"))
        {

            # msg - Script $script is not executable.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                       'csminstall', "W", 'EMsgExecMethod', "$dirname/$method");

            next;
        }

        #  add the script to the correct method list for the node
        foreach my $node (@{$::scriptnodes{$method}})
        {
            chomp $node;
            push @{$::CONFIGMETHODS{$node}}, $method;
        }
    }
    return $::OK;
}

#--------------------------------------------------------------------------------

=head3    make_script_lists

        Create lists of pre-install, post-install and update scrips
        for each node and put them in the following global hashes:

        %::PREINSTSCRIPTS - scripts get run from csmprereboot
                        during full install before the first reboot.
        %::POSTINSTSCRIPTS - scripts get run from csmfirstboot
                        after the first reboot of the system.
        %::PREOSUPGRADESCRIPTS - scripts get run from csmprereboot
                        after os upgrade and before the reboot.
        %::POSTOSUPGRADESCRIPTS - scripts get run from csmfirstboot
                        after os upgrade and after the reboot.
        %::UPDATESCRIPTS - scripts get run from updatenode.client
                        after a node has been updated.

        Arguments:
                @nodelist
        Returns:
                $::OK   - on success
                1       - on error
        Globals:
                %::PREINSTSCRIPTS
                %::POSTINSTSCRIPTS 
                %::PREOSUPGRADESCRIPTS
                %::POSTOSUPGRADESCRIPTS 
                %::UPDATESCRIPTS
        Example:
                if( ServerUtils->make_script_lists(@nodelist) ) { blah; }_
        Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub make_script_lists
{

    my ($junk, @nodel) = @_;   # the list of nodes passed in on the command line
    my $dirname;               #  directory to check for scripts
    my $script;                #  unique script name
    my $cmd;
    my $rc;

    @::scriptlist;     #  list of scripts found in a directory
    %::scriptnodes;    #  lists of nodes to run each script on

    # get the list of all nodes so we can check
    # for valid node names later
    if (!defined($ENV{'DC_ENVIRONMENT'}) || $ENV{'DC_ENVIRONMENT'} eq "CSM")
    {
        my $cmd = "/usr/bin/lsrsrc-api";
        $cmd .= " -D ':|:' -i -s IBM.ManagedNode::::Hostname";
        @::allnodes = NodeUtils->runcmd($cmd, -1);
        if ($::RUNCMD_RC)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                         'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $::RUNCMD_RC);
            return $::NOK;
        }
    }
    else
    {
        # rsct not available, possibly in DC environment
        foreach my $node (@nodel)
        {
            my ($hostname, $ipaddr) = NetworkUtils->tryHost($node);
            push @::allnodes, $hostname;
        }
    }

    my @scriptdirlist;
    if ($::UPDATESCRIPTSONLY)
    {
        @scriptdirlist = ($::UPDATEDIR);
    }
    else
    {
        @scriptdirlist = ($::PREINSTALLDIR, 
                          $::POSTINSTALLDIR, 
                          $::PREOSUPGRADEDIR,
                          $::POSTOSUPGRADEDIR,
                          $::DISKLESSPREBUILDDIR, 
                          $::DISKLESSBOOTDIR);
    }

    # check each subdirectory for user scripts
    foreach $dirname (@scriptdirlist)
    {
        $rc = ServerUtils->get_script_list($dirname, @nodel);

        if ($rc == $::NOK)
        {
            next;
        }

        # check each script name in this directory
        foreach $script (sort @::scriptlist)
        {
            chomp $script;
            # see if there are any nodes in the list
            if (@{$::scriptnodes{$script}}[0] eq $::NOK)
            {
                # msg - Skipping to the next script name.
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                           'csminstall', "E", 'EMsgSkipScript');
                next;
            }

            # see if the script is executable
            if (!(-x "$dirname/$script"))
            {
                # msg - Script $script is not executable.
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                       'csminstall', "E", 'EMsgExecScript', "$dirname/$script");

                # msg - Skipping to the next script name.
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                           'csminstall', "E", 'EMsgSkipScript');
                next;
            }

            #  add the file to the correct script list for the node
            foreach my $node (@{$::scriptnodes{$script}})
            {
                chomp $node;
                if ($dirname eq $::PREINSTALLDIR)
                {
                    push @{$::PREINSTSCRIPTS{$node}}, $script;
                }
                elsif ($dirname eq $::POSTINSTALLDIR)
                {
                    push @{$::POSTINSTSCRIPTS{$node}}, $script;
                }
                elsif ($dirname eq $::PREOSUPGRADEDIR)
                {
                    push @{$::PREOSUPGRADESCRIPTS{$node}}, $script;
                }
                elsif ($dirname eq $::POSTOSUPGRADEDIR)
                {
                    push @{$::POSTOSUPGRADESCRIPTS{$node}}, $script;
                }
                elsif ($dirname eq $::UPDATEDIR)
                {
                    push @{$::UPDATESCRIPTS{$node}}, $script;
                }
                elsif ($dirname eq $::DISKLESSPREBUILDDIR)
                {
                    push @{$::DISKLESSPREBUILDSCRIPTS{$node}}, $script;
                }
                elsif ($dirname eq $::DISKLESSBOOTDIR)
                {
                    push @{$::DISKLESSBOOTSCRIPTS{$node}}, $script;
                }
            }
        }
    }

    return $::OK;
}

#--------------------------------------------------------------------------------

=head3    name2list

        Convert a script file name to a list of nodes that the
        script can be run on.

        Arguments:
                $dirname,
                $filename
                $grouplist
                $nlist
        Returns:
                array of node names.
        Error:
                1 - An error occured.                                   #
        Globals:
                
                Requires @::allnodes be set by caller
        Example:
                @{$::scriptnodes{$filename}}=
                        ServerUtils->name2list($dirname,$filename,@nlist);_
        Comments:
                The file name format is <basename>._<node or group>.
                If there is just a basename then it applies to all nodes.
                If there is a node name then it just applies to that node.
                If there is a group name then it applies to the members of
                the group.


=cut

#--------------------------------------------------------------------------------

sub name2list
{
    my ($junk, $dirname, $filename, $groups, $nlist) = @_;
    my @nodelist = ();
    my ($base,  $name,   @glist);
    my ($short, $target, $cmd);
    my $junk2;

    chomp $filename;

    # the second part of the file name tells us
    #    what nodes to run it on
    # also check if additional target was used - error
    ($base, $target, $junk2) = split(/\._/, $filename);

    if (defined($junk2))
    {

        #  \'$filename\' is not a valid customization script name.
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     "E",
                                     'EMsgBadScriptName',
                                     "$dirname/$filename"
                                     );
        return $::NOK;
    }

    if (!defined($target))
    {

        # do on all nodes the calling
        #    command was asked to process
        return @$nlist;

    }
    else
    {

        # see if target is a valid group name
        #$cmd = "$::NODEGRP -p $target";
        #@glist = NodeUtils->runcmd($cmd, -1);
        if (defined $$groups{$target})
        {
            my @nodes = @{$$groups{$target}};
            foreach my $name (@nodes)
            {

                # we only need a node if it was specified
                #    on the command line
                if (grep(/^$name/, @$nlist))
                {
                    push(@nodelist, $name);
                }
            }
            return @nodelist;
        }

	# see if the target is a valid node name
        my ($hostname, $ipaddr) = NetworkUtils->tryHost($target);
        if (grep(/^$hostname/, @::allnodes))
        {

            # see if the target is one of the nodes
            #     specified on the command line
            if (grep(/^$hostname/, @$nlist))
            {
                chomp(@nodelist = ($hostname));
            }
            return @nodelist;
        }

        else
        {

            # msg - An incorrect file name format was used for the script $filename. '$target' is neither a node name nor a group name.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                       'csminstall', "E", 'EMsgBadScript', "$dirname/$filename",
                       $target);
            return $::NOK;

        }
    }
    return $::NOK;
}

#--------------------------------------------------------------------------------

=head2    PkgDefs Support

=cut

#--------------------------------------------------------------------------------

=head3    get_OSDefs

        Determine which definition files need to be loaded and
        load them.  Read the set of OSDefs variables into a hash
        and assigns them each to global variables.  Calls get_osdefs_files
        to read the files in /opt/csm/install/defs.  The correct file(s)
        are chosen based on the parameters passed in to get_OSDefs.

        If no parameters are provided, default values based on the current
        machine's configuration are filled in.

        Arguments:
                $argsProvided.
        Returns:
                Array of defintion file names
        Globals:
                none
        Error:
                undefined
        Example:
                my %osdefs =
                   ServerUtils->get_OSDefs();  # Get defs for this host
                print "CHMOD = $osdefs{'CHMOD'}\n";

                # You can also use a global variable:
                my %osdefs =
                   ServerUtils->get_OSDefs();  # Get for current machine
                print "CHMOD = $::CHMOD\n";
        Comments:
                The pathnames must be defined in CSMDefs.pm.

=cut

#--------------------------------------------------------------------------------

sub get_OSDefs
{
    my $class         = shift;
    my $args_provided = (@_);

    my (
        $osname,            # eg Linux or AIX (case sensitive)
        $distro_name,       # eg RedHat or SLES (case sensitive)
        $distro_version,    # eg 3 or 5.2
        $hw_arch,           # eg i386, ppc
        $csm_version,       # eg 1.3.0
        $no_globals,        # Don't set global variables if set to 1
    ) = @_;

    # Default to setting global variables
    $no_globals = 0 if (!$no_globals);

    # Set defaults if no arguments were provided
    # Note below that some of these require command pathnames to be defined.
    # Therefore, the pathnames must be defined in CSMDefs.pm.
    if (!$args_provided)
    {
        $osname         = $::PLTFRM;
        $distro_name    = NodeUtils->get_DistributionName;     #requires no cmds
        $distro_version = NodeUtils->get_DistributionVersion;  #requires LSLPP
        $hw_arch     = NodeUtils->get_PkgArchitecture; #requires UNAME for Linux
        $csm_version =
          NodeUtils->get_CSMVersion("csm.core");    #requires RPMCMD and LSLPP
    }
    my $munge = join(
                     "|::|",
                     (
                      $osname,  $distro_name, $distro_version,
                      $hw_arch, $csm_version
                     )
                     );
    if ($::all_OSDefs{$munge})
    {
        ($::DEBUG) && print "Found $munge in OSDefs hash\n";
        return (%{$::all_OSDefs{$munge}});
    }

    ServerUtils->get_osdefs_files($osname, $distro_name, $distro_version,
                                  $hw_arch, $csm_version);

    # Assign each hash element to a global variable unless $no_globals is set
    unless ($no_globals)
    {
        foreach my $key (keys %::OSDefs)
        {
            my $value = $::OSDefs{$key};

            # Assign each hash element to a global variable.
            # For example, the following has the effect of setting
            #    $::GREP = $value
            # In order for this to work, we need to temporarily turn off
            # strict type checking, until the end of this block.
            no strict 'refs';
            ${"main::$key"} = $value;
        }
    }

    # Store the attributes that were used to obtain this OSdefs
    $::OSDefs{OSDefs_osname}         = $osname;
    $::OSDefs{OSDefs_distro_name}    = $distro_name;
    $::OSDefs{OSDefs_distro_version} = $distro_version;
    $::OSDefs{OSDefs_hw_arch}        = $hw_arch;
    $::OSDefs{OSDefs_csm_version}    = $csm_version;

    # Make a local copy of OSDefs
    my %OSDefs = %::OSDefs;

    # Add it to all_pkgdefs
    $::all_OSDefs{$munge} = \%OSDefs;

    return %OSDefs;

}

#--------------------------------------------------------------------------------

=head3    get_osdefs_files

        Determine which definition files need to be loaded and load them.

        Arguments:
                $class,
                $osname,
                $distro_name,
                $distro_version,
                $hw_arch,
                $csm_version
        Returns:
                Array of defintion file names
        Globals:
                none
        Error:
                undefined
        Example:
                 ServerUtils->get_osdefs_files($osname,
                                                $distro_name,
                                                $distro_version,
                                                $hw_arch,
                                                $csm_version);
        Comments:
                Search order on Linux
                        Linux-RedHatEL-AS3-i386.pm, Linux-RedHatEL-AS3.pm,
                        Linux-RedHat.pm, Linux.pm
                Search order on AIX:
                        AIX-5.2.pm, AIX.pm

                Search for the files in either <pwd>/defs or $OSDEFS

=cut

#--------------------------------------------------------------------------------

sub get_osdefs_files
{
    my ($class, $osname, $distro_name, $distro_version, $hw_arch, $csm_version,)
      = @_;

    my $isAIX = ($osname eq "AIX");

    # Set these global variables to load the correct values from osdefs.
    $::csm_version = $csm_version;
    my ($effectiveDistroName) = NodeUtils->getEffectiveDistro($distro_name);

    my $distro   = "${effectiveDistroName}${distro_version}";
    my $osdistro = "${osname}-${distro}";

    my @defsfiles;
    push(@defsfiles, "${osdistro}-${hw_arch}.pm") if (${hw_arch});
    push(@defsfiles, "${osdistro}.0-${hw_arch}.pm") if (${hw_arch} and $isAIX);
    push(@defsfiles, "${osdistro}.pm");
    push(@defsfiles, "${osdistro}.0.pm") if ($isAIX);
    push(@defsfiles, "${osname}-${effectiveDistroName}.pm") if (${effectiveDistroName});
    push(@defsfiles, "${osname}.pm");

    my $progdir = dirname($0);

    foreach my $file (@defsfiles)
    {
        if (defined $::ENV{'CSM_MOUNT_POINT_ON_NODE'})
        {
            $::DIR_ON_NODE = "$::ENV{'CSM_MOUNT_POINT_ON_NODE'}/csm/defs";
        }
        else
        {
            $::DIR_ON_NODE = "/var/opt/csm/mnt/csm/defs";
        }
        foreach my $dir ("$progdir/defs", $::OSDEFS, $::DIR_ON_NODE)
        {
            my $defsfile = "$dir/$file";
            ($::DEBUG)
              && MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                              'csminstall', 'V', 'IMsgChecking_For', $defsfile);
            if (-f $defsfile)
            {

                # Using "do" rather than "require" because require only loads
                # a file once.  We might need to load it again with a different
                # value for csm_version.
                #require "$defsfile";
                do "$defsfile";
                return $defsfile;
            }
        }
    }

    # If we get here, then there was nothing loaded.
	my $defsfiles = join(' ', @defsfiles);
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 'E1', 'EMsgNoDefsFiles', $defsfiles);

    #die "Error:  No OS Definition files to load in $::OSDEFS\n";
}

#--------------------------------------------------------------------------------

=head3    get_pkgdefs

        Read the set of pkgdefs variables into a hash.

        Calls get_pkgdefs_files to read the files in
        /opt/csm/install/pkgdefs.  The correct file(s) are
        chosen based on the parameters passed in to get_pkgdefs.

        If no parameters are provided, default values based on the
        current machine's configuration are filled in.

        Arguments: - optional
                $osname,                # eg Linux or AIX (case sensitive)
                $distro_name,           # eg RedHat or SLES (case sensitive)
                $distro_version,        # eg 3 or 5.2
                $hw_arch,               # eg i386, ppc
                $mgmt_type,             # Either MgmtServer or MgdNode
                $csm_version,   
        Returns:
                Hash of pkgdef values by pkgdef variable name
        Globals:
                $::all_pkgdefs
                $::pkgdefs
                $::mgmt_type
                $::csm_version
        Error:
                undefined
        Example:
                my %pkgdefs =
                        ServerUtils->get_pkgdefs();            # Get for current machine

                print "DISTRO_NAME = $pkgdefs{'DISTRO_NAME'}\n";        # string example

                print "opensrc_prereqs = " .                            # array example
                join(" ", @{$pkgdefs{'opensrc_prereqs'}}) ."\n";

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub get_pkgdefs
{
    require Rpm::RpmList;
    require Rpm::Child::RsctRpmList;
    require Rpm::Child::HAMSRpmList;

    my $class         = shift;
    my $args_provided = (@_);

    my (
        $osname,            # eg Linux or AIX (case sensitive)
        $distro_name,       # eg RedHat or SLES (case sensitive)
        $distro_version,    # eg 3 or 5.2
        $hw_arch,           # eg i386, ppc
        $mgmt_type,         # Either MgmtServer or MgdNode
        $csm_version,       # eg 1.3.0
    ) = @_;

    # Set defaults if no arguments were provided
    # Note below that some of these require command pathnames to be defined.
    # Therefore, the pathnames must be defined in CSMDefs.pm.
    if (!$args_provided)
    {
        $osname         = $::PLTFRM;
        $distro_name    = NodeUtils->get_DistributionName;     #requires no cmds
        $distro_version = NodeUtils->get_DistributionVersion;  #requires LSLPP
        $hw_arch     = NodeUtils->get_PkgArchitecture; #requires UNAME for Linux
        $mgmt_type   = "MgmtServer";
        $csm_version =
          NodeUtils->get_CSMVersion("csm.core");    #requires RPMCMD and LSLPP
    }

    my $munge = join(
                     "|::|",
                     (
                      $osname,  $distro_name, $distro_version,
                      $hw_arch, $mgmt_type,   $csm_version
                     )
                     );
    if ($::all_pkgdefs{$munge})
    {
        ($::DEBUG) && print "Found $munge in pkgdefs hash\n";
        return (%{$::all_pkgdefs{$munge}});
    }

    # Set these global variables to load the correct values from pkgdefs.
    $::mgmt_type   = $mgmt_type;
    $::csm_version = $csm_version;

    ServerUtils->get_pkgdefs_files($osname, $distro_name, $distro_version,
                                   $hw_arch, $mgmt_type, $csm_version);

    # Store the attributes that were used to obtain this pkgdefs
    $::pkgdefs{pkgdefs_osname}         = $osname;
    $::pkgdefs{pkgdefs_distro_name}    = $distro_name;
    $::pkgdefs{pkgdefs_distro_version} = $distro_version;
    $::pkgdefs{pkgdefs_hw_arch}        = $hw_arch;
    $::pkgdefs{pkgdefs_mgmt_type}      = $mgmt_type;
    $::pkgdefs{pkgdefs_csm_version}    = $csm_version;

    # Make a local copy of pkgdefs
    my %pkgdefs = %::pkgdefs;

    # Add it to all_pkgdefs
    $::all_pkgdefs{$munge} = \%pkgdefs;

    return %pkgdefs;
}

#--------------------------------------------------------------------------------

=head3    get_pkgdefs_files

        Determine which definition files need to be loaded and load them.

        Arguments:
                $osname,
                $distro_name,
                $distro_version,
                $hw_arch,
                $mgmt_type,
                $csm_version
        Returns:
                Array of defintion file names
        Globals:
                $::mgmt_type
                $::csm_version
        Error:
                csmInstallCat E1
        Example:
                 ServerUtils->get_pkgdefs_files($osname,
                                                $distro_name,
                                                $distro_version,
                                                $hw_arch,
                                                $csm_version);
        Comments:
                Search order on Linux
                        Linux-RedHatEL-AS3-i386.pm, Linux-RedHatEL-AS3.pm,
                        Linux-RedHat.pm, Linux.pm
                Search order on AIX:
                        AIX-5.2.pm, AIX.pm
                Search for the files in either <pwd>/pkgdefs or $PKGDEFS

=cut

#--------------------------------------------------------------------------------

sub get_pkgdefs_files
{
    my ($class, $osname, $distro_name, $distro_version, $hw_arch, $mgmt_type,
        $csm_version,)
      = @_;

    my $isAIX = ($osname eq "AIX");

    # Set these global variables to load the correct values from pkgdefs.
    $::mgmt_type   = $mgmt_type;
    $::csm_version = $csm_version;

    my $distro   = "${distro_name}${distro_version}";
    my $osdistro = "${osname}-${distro}";
    my $progdir  = dirname($0);

    my ($file, $defsfile);
    if ($isAIX)
    {
        my (@defsfiles,$a,$b,$c);
        ($a,$b,$c) = split('\.', $distro);
        push(@defsfiles, "${osdistro}-${hw_arch}.pm")   if (${hw_arch});
        push(@defsfiles, "${osdistro}.0-${hw_arch}.pm") if (${hw_arch});
        push(@defsfiles, "${osdistro}.pm");
        push(@defsfiles, "${osdistro}.0.pm");
        push(@defsfiles, "${osname}-$a.$b.pm");
        push(@defsfiles, "${osname}-$a.pm");
        push(@defsfiles, "${osname}-${distro_name}.pm") if (${distro_name});
        push(@defsfiles, "${osname}.pm");
        foreach $file (@defsfiles)
        {

            foreach my $dir ("$progdir/pkgdefs", $::PKGDEFS)
            {
                $defsfile = "$dir/$file";
                ($::DEBUG)
                  && MessageUtils->messageFromCat('csmInstall.cat',
                           $::MSGMAPPATH, 'csminstall', 'V', 'IMsgChecking_For',
                           $defsfile);

                last if (-f $defsfile);
            }
            last if (-f $defsfile);
        }
    }
    else
    {
        if (${hw_arch} =~ /i.86/)
        {
            $file = "${osdistro}.pm";
        }
        else
        {
            $file = "${osdistro}-${hw_arch}.pm";
        }

        $defsfile = "$::PKGDEFS/$file";
        if (-f "$progdir/pkgdefs/$file")
        {
            $defsfile = "$progdir/pkgdefs/$file";
        }

        ($::DEBUG)
          && MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                              'csminstall', 'V', 'IMsgChecking_For', $defsfile);
    }

    if (-f $defsfile)
    {
	# clean the original %::pkgdefs
	undef %::pkgdefs;
        do "$defsfile";
        return $defsfile;
    }
    else
    {

        # If we get here, then there was nothing loaded.
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',  $::MSGMAPPATH,
                                     'csminstall',      'E1',
                                     'EMsgNoDefsFiles', $defsfile,
                                     );
        exit 1;
    }

}

#--------------------------------------------------------------------------------

=head3    getNodesInfo

        Get nodes info from config_info files.

        Arguments:
                (\@ARGV) either from @ARGV 
                $::opt_n ($node group string) 
                $::opt_N $file containing node list 1 -- skips the check for not
                        found nodes
        Returns:
                \@DestNode      = reference to list of all target nodes
                \%DestNodeHash  = reference to a hash of hostnames and attributes
                                  (foreach node in @Destnode):
                                  $DestNodeHash{$hostname}{$attribute_name} = "attribute value"
                                  For the list of attributes see $lsnode_attr
        Error:
                None documented
        Example
                my ($ref_DestNode, $ref_DestNodeHash) =
                        NodeUtils->getNodesInfo(\@ARGV, $::NODE_GRPS, $::FILE);
        Comments:
                Common method for getting node info from config_info files.

                Not complete yet. But could be used for testing purpose.

=cut

#--------------------------------------------------------------------------------

sub getNodesInfo
{
    my ($class, $ref_nodelist) = @_;
    my @nodelist   = @$ref_nodelist;
    my $csmdefault = "/csminstall";
    if (defined $ENV{'CSMINSTALL_ROOT'})
    {
        $csmdefault = $ENV{'CSMINSTALL_ROOT'};
    }
    my $infopath = "$csmdefault/csm/config";    #get from where?
    my (%DestNodeHash, @DestNode);

    my ($attr, $value);

    my @real_nodelist;    #split the nodes if there are commas

    if (!(-d $infopath))
    {
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',          $::MSGMAPPATH,
                                     'csminstall',              'I',
                                     'IMsgDIRECTORY_NOT_FOUND', $infopath
                                     );
    }
    foreach my $n (@nodelist)
    {
        if ($n =~ m/,/)
        {
            my @tmp = split ',', $n;
            foreach my $t (@tmp)
            {
                push @real_nodelist, $t;
            }
        }
        else
        {
            push @real_nodelist, $n;
        }
    }

    foreach my $node (@real_nodelist)
    {
        my %nodehash;
        my $nodename;
        my $nodeinfo = "$infopath/$node.config_info";
        unless (open(CFGINFO, $nodeinfo))
        {
            print "error reading $nodeinfo \n";
            next;
        }
        while (<CFGINFO>)
        {
            ($attr, $value) = split('=');
            chomp($value);
            $nodehash{$attr} = $value;
            if ($attr eq "Hostname")
            {
                $nodename = $value;
                push @DestNode, $nodename;
            }
        }

        foreach $attr (keys %nodehash)
        {
            $DestNodeHash{$nodename}{$attr} = $nodehash{$attr};
        }

        close(CFGINFO);

        #if installserver is a group, InstallServerAKBNode have no effect, it will get from the getSourceIP2Target
        #
        if ($DestNodeHash{$nodename}{'InstallServerIsGroup'} eq "yes")
        {
            my $akbnode = $nodename;
            if ($DestNodeHash{$nodename}{'InstallAdapterHostname'} ne "")
            {
                $akbnode = $DestNodeHash{$nodename}{'InstallAdapterHostname'};
            }
            my $hostname =
              NodeUtils->runcmd("/opt/csm/csmbin/getSourceIP2Target $akbnode",
                                -1);
            chomp($hostname);
            $DestNodeHash{$nodename}{'InstallServerAKBNode'} = "";
            my ($instserver, $dir) = split ':',
              $DestNodeHash{$nodename}{'InstallServer'};
            $DestNodeHash{$nodename}{'InstallServer'} = "$hostname:$dir";

        }
		# If InstallMethod=warewulf , set DisklessNodeGrpName attribute.
		if ($DestNodeHash{$nodename}{'InstallMethod'} eq "warewulf")
		{
			if ($DestNodeHash{$nodename}{'InstallTemplate'} eq "")
			{
				# In default
				$DestNodeHash{$nodename}{'DisklessNodeGrpName'} = 
				"$DestNodeHash{$nodename}{'InstallDistributionName'}".
				"$DestNodeHash{$nodename}{'InstallDistributionVersion'}-".
				"$DestNodeHash{$nodename}{'InstallServiceLevel'}-".
				"$DestNodeHash{$nodename}{'InstallPkgArchitecture'}";
			}
			else
			{
				 # Customization name
            	$DestNodeHash{$nodename}{'DisklessNodeGrpName'} = 
					`$::BASENAME $DestNodeHash{$nodename}{'InstallTemplate'}`;
            	chomp($DestNodeHash{$nodename}{'DisklessNodeGrpName'});
				$DestNodeHash{$nodename}{'DisklessNodeGrpName'} =~ s/^.*warewulf-tmpl\.//g;
			}
		} 
    }
    return (\@DestNode, \%DestNodeHash);

}

#-------------------------------------------------------------------------------

=head3    clear_old_pxeconfig

        Clear dynamic uuid/mac entries saved in rdm-dserver  

        Arguments:
              $csminstall_dir:
        Returns:
        Error:
        Example
        Comments:
=cut

#-------------------------------------------------------------------------------

sub clear_old_pxeconfig
{
    my $routine = "clear_old_pxeconfig";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class) = @_;

    # Create the temporary pxeconfig_update.xml file.
    srand(time | $$);    #random number generator start
    my $rdmconfig_file = "/tmp/csm_$$";
    while (-e $rdmconfig_file)
    {
        $rdmconfig_file = ServerUtils->CreateRandomName($rdmconfig_file);
    }

    open(RDMCONFIG, ">$rdmconfig_file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',
                                      $::MSGMAPPATH,
                                      'csminstall',
                                      "E",
                                      'EMsgCANT_WRITE_FILE',
                                      "$rdmconfig_file"
                                      );

    print RDMCONFIG "<pxeconfig>\n";
    print RDMCONFIG "  <pxe>\n";

    my $pxecfg_cache_file = "/var/opt/csm/csm_rdmconfig.cache";

    my @content = NodeUtils->readFile($pxecfg_cache_file);
    foreach my $line (@content)
    {
        chomp($line);
        next if (!$line);
        my ($type, $key) = split(/\s+/, $line);
        next if ($type ne "uuid" && $type ne "mac");
        next if (!$key);

        print RDMCONFIG
          "    <pxeclient type=\"$type\" action=\"respond\" op=\"remove\">\n";
        print RDMCONFIG "      <target>$key</target>\n";
        print RDMCONFIG "    </pxeclient>\n";
    }

    # Write the rdmconfig file footer
    print RDMCONFIG "  </pxe>\n";
    print RDMCONFIG "</pxeconfig>\n";

    close(RDMCONFIG);

    # Now pass the created file to updateRDMdserver
    my $cmd    = "$::UPDATERDMDSERVER $rdmconfig_file\n";
    my $output = NodeUtils->runcmd($cmd);
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 'V', 'IMsgShow_Output', $output);

    # Give updateRDMdserver time to complete
    sleep(1);

    # Delete the temporary file
    unlink($rdmconfig_file);

    # Delete the pxeconfig cache file
    unlink($pxecfg_cache_file);

    print "LEAVING: $routine\n" if $::DEBUG;
}

#-------------------------------------------------------------------------------

=head3    RDMdserver_add

        Update RDM dserver configuration

        Arguments:
		$bootMethod  - "disk","getmacs","autoyast","kickstart"
                (\%nodeHash) - reference to a hash of nodes.
        Returns:
        Error:
        Example
        Comments:
=cut

#-------------------------------------------------------------------------------
sub RDMdserver_add
{
    my $routine = "RDMdserver_add";
    print "ENTERING: $routine\n" if $::DEBUG;

    # Get a list of which nodes use which install servers,
    # and a list of which nodes use the manasgement server.
    my ($class,        $bootMethod)         = @_;
    my ($ref_MgtNodes, $ref_InstallServers) =
      ServerUtils->getInstallServers(\%::NODEHASH);
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;

    my $cmd;
    my $output;

    # Create a pxeconfig_update.xml file for each install server.
    # Gather the list of nodes that use each install server, and the
    # list of nodes that use the management server.

    # This part of the code runs when this routine is called from the
    # management server.
    if (@MgtNodes)
    {
        ($::DEBUG)
          && print "NODE LIST FOR MANAGEMENT SERVER = "
          . join(",", @MgtNodes) . "\n";

        if ($bootMethod eq "getmacs")
        {
            ServerUtils->clear_old_pxeconfig;
        }

        my $rdmconfig_file =
          ServerUtils->create_pxeconfig_update_file($bootMethod, \@MgtNodes);

        # Now pass the created file to updateRDMdserver
        $cmd    = "$::UPDATERDMDSERVER $rdmconfig_file\n";
        $output = NodeUtils->runcmd($cmd);
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'V', 'IMsgShow_Output',
                                     $output);

        # Give updateRDMdserver time to complete
        sleep(1);

        # Delete the temporary file
        unlink($rdmconfig_file);

    }

    # This part of the code runs when this routine is called from an
    # install server.  It runs on the install server.
    # The logic might not be quite right here.  I don't know what happens
    # if there are multiple install servers defined in the cluster.
    foreach my $is (keys %InstallServers)
    {
        if ($bootMethod eq "getmacs")
        {
            ServerUtils->clear_old_pxeconfig;
        }

        foreach my $dir (keys %{$InstallServers{$is}{dir}})
        {

            # Get the list of nodes that use this install server ($is)
            # and this directory ($dir).
            my @node_list = @{$InstallServers{$is}{dir}{$dir}};
            ($::DEBUG)
              && print "NODE LIST FOR $is:$dir = "
              . join(",", @node_list) . "\n";
            my $rdmconfig_file =
              ServerUtils->create_pxeconfig_update_file($bootMethod,
                                                        \@node_list);

            # Now pass the created file to updateRDMdserver
            $cmd    = "$::UPDATERDMDSERVER $rdmconfig_file\n";
            $output = NodeUtils->runcmd($cmd);
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'V', 'IMsgShow_Output', $output);

            # Give updateRDMdserver time to complete
            sleep(1);

            # Delete the temporary file
            unlink($rdmconfig_file);
        }
    }

    print "LEAVING: $routine\n" if $::DEBUG;
}

#-------------------------------------------------------------------------------

=head3    create_pxeconfig_update_file

        Create an RDM dserver configuration file for a list of nodes.

        Arguments:
                (\@node_list) - reference to a list of nodes.
        Returns:
		$filename - the filename that was created
        Error:
        Example
		ServerUtils->create_pxeconfig_update_file(\@nodelist);
        Comments:
	     SAMPLE PXECONFIG_UPDATE.XML FILE:
                <pxeconfig>
                   <pxe>
                   <pxeclient type="mac" action="respond" op="add">
                     <target>0002559C0633</target>
                     <pxe2bstrap32>pxelinux.0</pxe2bstrap32>
                     <bootdhcpopt type = "data" tag = "209">"xblade09.clusters.com.install"</bootdhcpopt>
                     <bootdhcpopt type = "data" tag = "210">"pxelinux.cfg"</bootdhcpopt>
                   </pxeclient>
                   </pxe>
                </pxeconfig>
=cut

#-------------------------------------------------------------------------------
sub create_pxeconfig_update_file
{
    my $routine = "create_pxeconfig_update_file";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $bootMethod, $ref_node_list, $dir) = @_;
    my @node_list = @$ref_node_list;

    my %nodeHash = %::NODEHASH;
    my %pxecfgCached;

    # Create the temporary pxeconfig_update.xml file.
    srand(time | $$);    #random number generator start
    my $rdmconfig_file = "/tmp/csm_$$";
    while (-e $rdmconfig_file)
    {
        $rdmconfig_file = ServerUtils->CreateRandomName($rdmconfig_file);
    }

    open(RDMCONFIG, ">$rdmconfig_file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',
                                      $::MSGMAPPATH,
                                      'csminstall',
                                      "E",
                                      'EMsgCANT_WRITE_FILE',
                                      "$rdmconfig_file"
                                      );

    # Write the rdmconfig file header
    print RDMCONFIG "<pxeconfig>\n";
    print RDMCONFIG "  <pxe>\n";

    # A single (temporary) pxeconfig_update.xml file will be created for
    # all the nodes in @node_list.  A stanza will be created for each node.
    if ($bootMethod eq "kickstart" || 
		$bootMethod eq "kickstart-upgrade" || 
		$bootMethod eq "autoyast")
    {

        # for kickstart/autoyast method, valid type is either "uuid" or "mac"
        my %hex = NetworkUtils->genHex(keys %nodeHash);
        foreach my $nodename (keys %nodeHash)
        {
            my $host = $nodeHash{$nodename}{"Hostname"};
            if ($nodeHash{$nodename}{"InstallAdapterHostname"})
            {
                $host = $nodeHash{$nodename}{"InstallAdapterHostname"};
            }

            # Determine the "type" (uuid or mac)
            my $type;
            my $target;
            my $uuid = $nodeHash{$nodename}{'UUID'};
            my $mac  = $nodeHash{$nodename}{'InstallAdapterMacaddr'};
            if ($mac)
            {
                $type = "mac";
                ($target = $mac) =~ s/://g;    # Strip colons (:) from Macaddr

            }
            elsif ($uuid)
            {
                $type = "uuid";
                ($target = $uuid) =~ s/ //g;    # Strip spaces from UUID
            }
            else
            {
                MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                     "E1", 'EMsgNO_MAC_OR_UUID_DEFINED', $host);
            }

            my $hexFile = $hex{$nodename};

            print RDMCONFIG
              "    <pxeclient type=\"$type\" action=\"respond\" op=\"add\">\n";
            print RDMCONFIG "      <target>$target</target>\n";
            print RDMCONFIG "      <pxe2bstrap32>pxelinux.0</pxe2bstrap32>\n";
            print RDMCONFIG
              "      <bootdhcpopt type = \"data\" tag = \"209\">\"pxelinux.cfg/$hexFile\"</bootdhcpopt>\n";

            print RDMCONFIG "    </pxeclient>\n";

            # cache the pxeconfig
            push(@{$pxecfgCached{$type}}, $target);
        }
    }
    elsif ($bootMethod eq "getmacs")
    {

        my @subnets;

        foreach my $nodename (keys %nodeHash)
        {
            my ($bc, $netmask) = NetworkUtils->getNetmask($nodename,\%nodeHash);
            if (!defined $netmask)
            {
                MessageUtils->message("W", 'EMsgINVALID_NETMASK', $nodename);
                next;
            }
            my $subnet = NetworkUtils->subnet($nodename, $netmask);
            my $iamask = unpack('N', inet_aton($netmask));
            if (!$iamask)
            {
                MessageUtils->message("W", 'EMsgINVALID_NETMASK', $nodename);
                next;
            }

            my $shift = 0;

            while ((($iamask >> $shift) & 1) == 0 && $shift < 32)
            {
                ++$shift;
            }

            if (!grep(/^$subnet\/$shift$/, @subnets))
            {
                push(@subnets, "$subnet/$shift");
            }
        }

        foreach my $subnet (@subnets)
        {
            print RDMCONFIG
              "    <pxeclient type=\"subnet\" action=\"respond\" op=\"add\">\n";
            print RDMCONFIG "      <target>$subnet</target>\n";
            print RDMCONFIG "      <pxe2bstrap32>pxelinux.0</pxe2bstrap32>\n";
            print RDMCONFIG
              "      <bootdhcpopt type = \"data\" tag = \"209\">\"pxelinux.cfg/getmacs.default\"</bootdhcpopt>\n";
            print RDMCONFIG "    </pxeclient>\n";

        }
    }
    elsif ($bootMethod eq "disk")
    {

        # Dynamic subnet entries added by pxeboot_mac method will be automatically removed.
        my @subnets;

        foreach my $nodename (keys %nodeHash)
        {

            # get nodemast by given node
            my ($bc, $netmask) = NetworkUtils->getNetmask($nodename, \%nodeHash);
            if (!defined $netmask)
            {
                MessageUtils->message("W", 'EMsgINVALID_NETMASK', $nodename);
                next;
            }

            # get subnet by given node
            my $subnet = NetworkUtils->subnet($nodename, $netmask);
            my $iamask = unpack('N', inet_aton($netmask));
            if (!$iamask)
            {
                MessageUtils->message("W", 'EMsgINVALID_NETMASK', $nodename);
                next;
            }

            my $shift = 0;

            while ((($iamask >> $shift) & 1) == 0 && $shift < 32)
            {
                ++$shift;
            }

            if (!grep(/^$subnet\/$shift$/, @subnets))
            {
                push(@subnets, "$subnet/$shift");
            }
        }

        # write the xml stanza for removing subnet entries
        foreach my $subnet (@subnets)
        {
            print RDMCONFIG
              "    <pxeclient type=\"subnet\" action=\"respond\" op=\"remove\">\n";
            print RDMCONFIG "      <target>$subnet</target>\n";
            print RDMCONFIG "    </pxeclient>\n";

        }

    }

    # Write the rdmconfig file footer
    print RDMCONFIG "  </pxe>\n";
    print RDMCONFIG "</pxeconfig>\n";

    close(RDMCONFIG);

    # cache the fresh dynamic entries
    my $pxecfg_cache_dir  = "/var/opt/csm/";
    my $pxecfg_cache_file = "csm_rdmconfig.cache";

    if (!-d $pxecfg_cache_dir)
    {
        mkpath($pxecfg_cache_dir, $::VERBOSE, 0755);
    }

    open(PXECACHE, ">>$pxecfg_cache_dir/$pxecfg_cache_file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',
                                      $::MSGMAPPATH,
                                      'csminstall',
                                      "E",
                                      'EMsgCANT_WRITE_FILE',
                                      "$pxecfg_cache_dir/$pxecfg_cache_file"
                                      );

    if (defined $pxecfgCached{uuid} || defined $pxecfgCached{mac})
    {
        foreach my $type (keys %pxecfgCached)
        {
            foreach my $key (@{$pxecfgCached{$type}})
            {
                print PXECACHE "$type $key\n";
            }
        }
    }
    close(PXECACHE);

    print "RDM CONFIG FILE:\n" . `cat $rdmconfig_file` . "\n" if $::DEBUG;
    print "LEAVING: $routine\n"                               if $::DEBUG;

    return ($rdmconfig_file);
}

#-------------------------------------------------------------------------------

=head3   migrate_install_servers

    This subroutine is used when migrating from a pre 1.4.1 level of CSM. 
	It converts any InstallServer attribute values to NFSServer attributes.
	It also sets the IBM.DmsCtrl MigrationInfo attribute to the installed 
	version of csm.server.

=cut

#-------------------------------------------------------------------------------

sub migrate_install_servers
{

    $::converted = 0;

    # get the previously installed version of CSM
    my $preVersion =
      NodeUtils->runcmd("/usr/bin/lsrsrc-api -s IBM.DmsCtrl::::MigrationInfo",
                        -1);

    # don't try to do the conversions if we don't know what we're coming from
    if ($::RUNCMD_RC || $preVersion eq "")
    {
        return $::OK;
    }
    chomp $preVersion;

    #  if it is earlier than 1.4.1 than we might have to convert
    #     the InstallServer attributes to NFSServer
    if (ArchiveUtils->testVersion($preVersion, "<", "1.4.1.0"))
    {

        # get node data for every node
        my $outref =
          NodeUtils->runrmccmd('lsrsrc-api', "-i",
                               "-s IBM.ManagedNode::::Hostname::InstallServer");
        if ($::RUNCMD_RC)
        {
            $outref = [];
        }

        # if the node definition has the InstallServer attr set then
        #   we have to move it to the NFSServer attr and blank out
        #   the InstallServer attr.
        foreach my $l (@$outref)
        {
            my ($hname, $instserver) = split(/::/, $l);
            chomp $instserver;
            if (defined($instserver) && ($instserver ne ""))
            {
                NodeUtils->runrmccmd(
                    'chrsrc-api',
                    '-i',
                    qq(-s IBM.ManagedNode::"Hostname='$hname'"::NFSServer::"$instserver"::InstallServer::"''")
                    );
                if ($::RUNCMD_RC)
                {

                    # Could not move InstallServer attribute values to
                    #      corresponding NFSServer attributes.
                    MessageUtils->messageFromCat('csmInstall.cat',
                         $::MSGMAPPATH, 'csminstall', 'W', 'EMsgNo_IS_TO_NFSS');
                    return $::NOK;
                }
                $::converted++;
            }
        }

        # if any conversions were done give an info message
        if ($::converted)
        {

            # InstallServer attribute values were moved to the
            # corresponding NFSServer attributes.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                         'csminstall', 'I', 'IMsgIS_TO_NFSS');
        }
    }

    #
    # set the value of MigrationInfo to the current level of csm.server
    #

    # make sure $::PLTFRM is set
    if (NodeUtils->isLinux())
    {
        $::PLTFRM = "Linux";
    }
    else
    {
        $::PLTFRM = "AIX";
    }

    #  get the version of csm.server currently installed
    #     - this uses RPMCMD or LSLPP based on value of PLTFRM
    my $csm_version = NodeUtils->get_CSMVersion("csm.server", 1);

    chomp $csm_version;

    # set the value of MigrationInfo
    NodeUtils->runcmd(
            "/usr/bin/chrsrc-api -s IBM.DmsCtrl::::MigrationInfo::$csm_version",
            -1);

    return $::OK;
}

#--------------------------------------------------------------------------------

=head3    get_conserver_verion

        Return the version of the installed conserver daemon

        Arguments:
                None
        Returns:
                The version string as reported by conserver, or
                "-1" if conserver is not installed
        Globals:
                None
        Example:
                ServerUtils->get_conserver_version();

=cut

#--------------------------------------------------------------------------------

sub get_conserver_version
{
    my $conserver_vers_ref;
    my $ret_val;

    if (!-s "/opt/conserver/bin/conserver")
    {
        $ret_val = -1;
    }
    else
    {
        $conserver_vers_ref =
          NodeUtils->runcmd("/opt/conserver/bin/conserver -V", -1, 1);
        $ret_val = $$conserver_vers_ref[0];

        $ret_val =~ s/\D+(\d+\.\d+\.\d+)\D*/$1/g;
    }

    return $ret_val;
}

#--------------------------------------------------------------------------------

=head3    determineNFSThreadCount


        Arguments:
		
        Returns:
				None
        Globals:
                None
        Example:
                ServerUtils->determineNFSThreadCount($ISvr,$ref_ISvrNodes,$ref_DestNodeHash);

=cut

#--------------------------------------------------------------------------------

sub determineNFSThreadCount
{
    my ($class, $ISvr, $ref_ISvrNodes, $ref_DestNodeHash) = @_;

    my @ISvrNodes    = @{$ref_ISvrNodes};
    my %DestNodeHash = %{$ref_DestNodeHash};
    my $thread_count = 8;

    return if (!defined %{$DestNodeHash{$ISvr}});

    if (scalar(@ISvrNodes) >= 100)
    {
        $thread_count = 32;
    }
    elsif (scalar(@ISvrNodes) >= 50)
    {
        $thread_count = 16;
    }

    my $nfs_cfg_file       = "/cfmroot/etc/sysconfig/nfs";
    my $nfs_restart_script = $nfs_cfg_file . ".post";
    if (!-f "$nfs_cfg_file")
    {
        my $distro = $DestNodeHash{$ISvr}{InstallDistributionName};
        my ($effectiveDistroName) = NodeUtils->getEffectiveDistro($distro);

        my $cfg_key_word;
        if ($effectiveDistroName =~ /RedHat/)
        {
            $cfg_key_word = "RPCNFSDCOUNT";
        }
        elsif ($effectiveDistroName =~ /SLES/)
        {
            $cfg_key_word = "USE_KERNEL_NFSD_NUMBER";
        }
        else
        {
            return;
        }

        mkpath("/cfmroot/etc/sysconfig");

        unless (open(CFG_FILE, ">$nfs_cfg_file"))
        {
            MessageUtils->messageFromCat('HCnodecmds.cat', $::MSGMAPPATH,
                          'common', 'E', 'EMsgErrorOpeningFile', $nfs_cfg_file);
        }
        else
        {
            print CFG_FILE "$cfg_key_word=\"$thread_count\"\n";
            close(CFG_FILE);
        }

        #$nfs_restart_script= $nfs_cfg_file . ".post";
        if (!-f "$nfs_restart_script")
        {
            unless (open(SCRIPT_FILE, ">$nfs_restart_script"))
            {
                MessageUtils->messageFromCat('HCnodecmds.cat', $MSGMAPPATH,
                                         'common', 'E1', 'EMsgErrorOpeningFile',
                                         $nfs_restart_script);
            }
            else
            {
                print SCRIPT_FILE "#!/bin/sh\n";
                print SCRIPT_FILE "n=\`showmount | wc -l\`\n";
                print SCRIPT_FILE "[ \"\$n\" != 1 ] && exit\n";
                print SCRIPT_FILE
                  "/etc/init.d/nfsserver restart || /etc/init.d/nfs restart\n";
                close(SCRIPT_FILE);
                chmod 0755, $nfs_restart_script;
            }
        }
        NodeUtils->runcmd("$::CSMBIN/cfmupdatenode -n $ISvr",        -1);
        NodeUtils->runcmd("rm -f $nfs_restart_script $nfs_cfg_file", -1);
    }
    else
    {
        NodeUtils->runcmd("$::CSMBIN/cfmupdatenode -n $ISvr", -1);
    }
}

#--------------------------------------------------------------------------------

=head3    createAutoprovidesDB


        Arguments:
		hash nodes with their attributes
        Returns:
		none
        Globals:
                none
        Example:
                ServerUtils->createAutoprovidesDB();

=cut

#--------------------------------------------------------------------------------
sub createAutoprovidesDB
{

    my ($class, $nodehash) = @_;

    my %donedirs;
    if (!-e "/usr/sbin/autoupd")
    {
        return 0;
    }    #return if autoupdate is not installed

    foreach my $node (keys %$nodehash)
    {

        #each node corresponds to a possible directory structure
        my $os = $$nodehash{$node}{'InstallOSName'};
        if ($os ne "Linux") { next; }
        my $distro       = $$nodehash{$node}{'InstallDistributionName'};
        my $version      = $$nodehash{$node}{'InstallDistributionVersion'};
        my $arch         = $$nodehash{$node}{'InstallPkgArchitecture'};
        my $servicelevel = $$nodehash{$node}{'InstallServiceLevel'};
        if (!defined $servicelevel) { $servicelevel = "GA"; }
        my $dir =
          ServerUtils->getBaseDir("Linux", $distro, $version, $arch,
                                  $servicelevel);
        $dir = "$dir/RPMS";
        my $min = 1;

        if (!defined $donedirs{$dir})
        {

            #create the autoprovides database
            NodeUtils->runcmd(
                "echo \"/usr/sbin/autoupd --noupdate --addtodb $dir/*.rpm --database $dir/autoprovides.db 1>/dev/null 2>/dev/null\" | at now + $min minutes"
                );
            $min++;
            $donedirs{$dir} = 1;
        }
        return 0;
    }
}

#--------------------------------------------------------------------------------

=head3    remove_reboot_ready_file


        Arguments:
				Node hash that contains the nodes to be processed.
        Returns:
                None
        Globals:
                None
        Example:
				ServerUtils->remove_reboot_ready_file(\%::NODEHASH);

=cut

#--------------------------------------------------------------------------------

sub remove_reboot_ready_file
{
    my ($class, $ref_DestNodeHash) = @_;

    my %DestNodeHash = %{$ref_DestNodeHash};
    my ($ref_MgtNodes, $ref_InstallServers) =
      ServerUtils->getInstallServers(\%::NODEHASH);
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my $cmd;

    if (@MgtNodes)
    {

        # remove the reboot_ready flag file on MS for the nodes using MS as IS
        my $reboot_ready_file;
        foreach my $hostname (@MgtNodes)
        {
            $reboot_ready_file =
              $::CSMSTATUSDIR . "/" . $hostname . ".rebootready";
            NodeUtils->runcmd("rm -f $reboot_ready_file");
        }
    }

    $ENV{DSH_LIST} = '';    #clear dsh env.

    # >>> check reachability to the Install Servers >>>
    my @installServers;
    foreach my $is (keys %InstallServers)
    {
        if ($InstallServers{$is}{isGrp})
        {
            my $refGroup = NodeUtils->getNodegrp($is);
            if ($refGroup)
            {
                push @installServers, @$refGroup;
            }
        }
        else
        {
            push @installServers, $is;
        }
    }
    my $refISNotReachable = NetworkUtils->checkRSReachability(\@installServers);
    if ($refISNotReachable != 0)
    {
        if (scalar @$refISNotReachable > 0)
        {
            my $isvrs = join ",", @$refISNotReachable;
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgISDown', $isvrs);
        }
    }

    foreach my $is (keys %InstallServers)
    {
        my $isISGrp = $InstallServers{$is}{isGrp};
        my $realis  = $is;
        if ($isISGrp)
        {
            $realis =~ s/^\+//;
        }

        foreach my $dir (keys %{$InstallServers{$is}{dir}})
        {
            my @nodelist = @{$InstallServers{$is}{dir}{$dir}};
            my @status_files;
            my $file_list;
            foreach my $hostname (@nodelist)
            {
                push @status_files,
                  $dir . "/csm/status/" . $hostname . ".rebootready";
            }
            $file_list = join " ", @status_files;
            if ($isISGrp)
            {
                $cmd = "/opt/csm/bin/dsh -N $realis rm -f $file_list";
            }
            else
            {
                $cmd = "/opt/csm/bin/dsh -n $is rm -f $file_list";
            }
            NodeUtils->runcmd($cmd, -2);
            foreach my $hostname (@nodelist)
            {
                push @status_files, $dir . "/csm/status/" . $hostname;
            }
            $file_list = join " ", @status_files;
            if ($isISGrp)
            {
                $cmd = "/opt/csm/bin/dsh -N $realis rm -f $file_list";
            }
            else
            {
                $cmd = "/opt/csm/bin/dsh -n $is rm -f $file_list";
            }
            NodeUtils->runcmd($cmd, -2);
        }
    }

}

#--------------------------------------------------------------------------------

=head3   RDMdserver_installed 


        Arguments:
        
        Returns:
                1 - rdm dserver has been installed on the server
                0 - rdm dserver has not been installed on the server
        Globals:
                None
        Example:
				ServerUtils->RDMDserver_installed;

=cut

#--------------------------------------------------------------------------------
sub RDMdserver_installed
{
    my ($class) = @_;
    my $req_ver = "4.20";
    my $req_rel = "1";
    my ($curr_ver, $curr_rel);

    $::RDM_TOP       = "/opt/IBM/RDM";
    $::RDM_LOCAL_DIR = "$::RDM_TOP/local";

    if (defined $::rdm_dserver{status})
    {
        return $::rdm_dserver{status};
    }

    if (NodeUtils->isAIX)
    {
        $::rdm_dserver{status} = 0;
        return $::rdm_dserver{status};
    }

    if (NodeUtils->isPLinux)
    {
        $::rdm_dserver{status} = 0;
        return $::rdm_dserver{status};
    }

    my ($cmd, $output);

    $cmd    = "$::RPMCMD -q rdm-dserver --qf '%{Version}:|:%{Release}' ";
    $output = NodeUtils->runcmd("$cmd", -1);

    if ($::RUNCMD_RC)
    {

        # can not find rdm-dserver package
        $::rdm_dserver{status} = 0;
        return $::rdm_dserver{status};
    }

    ($curr_ver, $curr_rel) = split(/:\|:/, $output);
    if (
        ArchiveUtils->testVersion($curr_ver, "<", $req_ver, $curr_rel, $req_rel)
      )
    {

        #rdm-dserver package version must be at least 4.20-1
        $::rdm_dserver{status} = 0;
        return $::rdm_dserver{status};
    }

    if (-f "$::RDM_LOCAL_DIR/dserver.ini")
    {
        $cmd    = NetworkUtils->service("RDMdserver", "whoami");
        $output = NodeUtils->runcmd("$cmd",           -1);
        if ($::RUNCMD_RC)
        {
            MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                              "E$::RUNCMD_RC", 'EMsgCMD_FAILED', $cmd, $output);
        }

        if (   $output =~ /master D-Server/
            or $output =~ /remote D-Server/)
        {
            $::rdm_dserver{status} = 1;
        }
        else
        {
            $::rdm_dserver{status} = 0;
        }

        return $::rdm_dserver{status};
    }
    else
    {
        $::rdm_dserver{status} = 0;
        return $::rdm_dserver{status};
    }
}

#------------------------------------------------------------------------------

=head3   getInstallNFSServer

		Get the InstallServer and NFSServer of target nodes(@nodelist)

        Arguments:
        $ref_nl     Input Node List
        $ref_nodehash   Node Hash
        Returns:
                Install Server and NFS Server
        Globals:
                 None
        Example:
		ServerUtils->getInstallNFSServer(\@nodelist,\%nodehash);

=cut

#-------------------------------------------------------------------------------
sub getInstallNFSServer
{
    my ($class, $ref_nl, $ref_nodehash) = @_;
    my @installserver;
    my @nodelist     = @{$ref_nl};
    my %DestNodeHash = %{$ref_nodehash};
    my $is;

    # Get InstallServer or NFSServer
    foreach my $node (@nodelist)
    {
        if ($DestNodeHash{$node}{'InstallServer'})
        {
            $is = $DestNodeHash{$node}{'InstallServer'};
        }
        elsif ($DestNodeHash{$node}{'NFSServer'})
        {
            $is = $DestNodeHash{$node}{'NFSServer'};
        }
        else
        {
            next;
        }
        my ($server, $rep_dir) = split ':', $is;

        if ($server)
        {
            my @iserver = ();

            # Get Server from NodeGroup
            my $ref_grp = NodeUtils->getNodegrp($server);
            if ($ref_grp)
            {
                push @iserver, @$ref_grp;
            }
            else
            {
                push @iserver, $server;
            }
            foreach $server (@iserver)
            {
                my ($hostname, $ip) = NetworkUtils->getHost($server);
                push @installserver, $hostname;
            }
        }
    }
    return @installserver;
}

#------------------------------------------------------------------------------

=head3   getInstallNodeList 


        Arguments:
		$ref_nl		Input Node List
		$ref_nodehash	Node Hash
        Returns:
                Node List, extract the Install Server
        Globals:
                None
        Example:
		ServerUtils->getInstallNodeList(\@::NODELIST, \%::NODEHASH);

=cut

#-------------------------------------------------------------------------------
sub getInstallNodeList
{
    my ($class, $ref_nl, $ref_nodehash) = @_;

    my @nodelist     = @{$ref_nl};
    my %DestNodeHash = %{$ref_nodehash};
    my @return_list  = ();
    my @bad_list     = ();

    my @installserver =
      ServerUtils->getInstallNFSServer(\@nodelist, \%DestNodeHash);

    #Extract the InstallServer or NFSServer from nodelist and nodehash
    foreach my $node (@nodelist)
    {
        if (!grep(/$node$/, @installserver))
        {
            push @return_list, $node;
        }
        else
        {
            push @bad_list, $node;
            delete ${$ref_nodehash}{$node};
        }
    }
    if (scalar(@bad_list))
    {
        my $host = join(",", @bad_list);
        MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'I',
                                     'IMsgSkipInstallServer', $host);
    }

    return @return_list;
}

#--------------------------------------------------------------------------------

=head3    add_permanent_arp_cache

        Notes: 
        just a trick to successfully install a ppc node over network.

=cut

#--------------------------------------------------------------------------------
sub add_permanent_arp_cache
{
    my ($class, $is, $client_ip, $mac) = @_;
    my ($cmd,   $rc);

    if ($is)
    {
        my @nodes     = ($is);

        $cmd = "/sbin/arp -s $client_ip $mac";
	my %options_api = ();
	$options_api{'command'} = "/sbin/arp -s $client_ip $mac";
        $options_api{'nodes'} = join(',', @nodes);
        NodeUtils->runDsh(\%options_api);
    }
    else
    {
        if($::PLTFRM eq "AIX")
        {
            $cmd = "/usr/sbin/arp -s ether $client_ip $mac";
        }
        else
        {
            $cmd = "/sbin/arp -s $client_ip $mac";
        }
        NodeUtils->runcmd($cmd);
    }
}

#--------------------------------------------------------------------------------

=head3    build_kernel_parameters

        Notes: 
        build kernel parameters to install a pLinux node over network

=cut

#--------------------------------------------------------------------------------
sub build_kernel_parameters
{
    my ($class, $client_ip, $installserver, $nfsserver, $installdir, $svc_level,
        $netadapter_name, $netadapter_mac, $client_netmask, $gateway)
      = @_;

    my $kernelparameters;
    my $install_source_header;

    # Get the install protocol.  get_NetworkInstallProtocol() ensures
    # only "nfs" and "http".
    my $install_protocol = NodesetUtils->get_NetworkInstallProtocol();

    if ($::DISTRO_NAME =~ /RedHat/i)
    {
        if ($install_protocol =~ /nfs/i)
        {
            $install_source_header = "nfs\:$installserver\:$installdir";
        }
        elsif ($install_protocol =~ /http/i)
        {
            $install_source_header = "http\://$installserver$installdir";
        }

        $kernelparameters =
            "ks=$install_source_header/csm/"
          . $::DISTRO_NAME
          . $::DISTRO_VERSION . "/";

        if ($client_ip)
        {
            $kernelparameters .= "$client_ip";
        }

        if ($::DISTRO_VERSION == 3
            && ($svc_level eq "QU1" || $svc_level eq "GA"))
        {
            $kernelparameters =
                $kernelparameters
              . " dd=$install_source_header/csm/"
              . $::DISTRO_NAME
              . $::DISTRO_VERSION
              . "-drivers/iprdd.img";
        }

        my $ksdevice;

        #AS3 QU4 and AS4 support mac address as kernel parameters "ksdevice"
        if ($svc_level ge "QU4" || $::DISTRO_VERSION >= 4)
        {
            $ksdevice = $netadapter_mac;
        }
        else
        {
            if (!$netadapter_name)
            {
                $ksdevice = "link";
            }
            else
            {
                $ksdevice = $netadapter_name;
            }
        }
        $kernelparameters = $kernelparameters . " ksdevice=$ksdevice";

        # On HMC and within DC environment, we need to skip bootp request when node netbooting.
        # So, we have to pass ip and netmask boot parameters to notify lpar ks info 
        # instead of using DHCP/bootp services.

	    if ( NodeUtils->isHMC() 
	       && ( defined $ENV{'DC_ENVIRONMENT'} && $ENV{'DC_ENVIRONMENT'} ne 'CSM')
               )
	    {
			$kernelparameters = $kernelparameters . " ip=$client_ip netmask=$client_netmask";
                        if ($gateway)
                        {
                            $kernelparameters = $kernelparameters . " gateway=$gateway";
                        }
	    }
    }
    else
    {
        if ($install_protocol =~ /nfs/i)
        {
            $install_source_header = "nfs://$installserver/$installdir";
        }
        else
        {
            $install_source_header = "http://$installserver$installdir";
        }

        $kernelparameters =
            "autoyast=$install_source_header/csm/"
          . $::DISTRO_NAME
          . $::DISTRO_VERSION . "/";
        if ($client_ip)
        {
            my (undef, $hex) = NetworkUtils->genHex( $client_ip);
            $kernelparameters .= $hex; 
        }

	# disable this line for boot parameters too long on js20
	#$kernelparameters = $kernelparameters . " server=$nfsserver";

        if (   ($::KERNEL_VERSION > 19)
            || ($::DETAIL_KERNEL_VERSION > 215)
            || ($::DISTRO_VERSION >= 9))
        {
            $kernelparameters = $kernelparameters . " dhcptimeout=150";
        }
        ## This adds a netdevice like eth0 or eth1 so that the install
        ## only contacts the correct adapter.
        ## If InstallAdapterName is not set, netdevice will not be set and
        ## yast will automatically search adapter.
        if ($netadapter_name)
        {
            $kernelparameters =
              $kernelparameters . " netdevice=$netadapter_name";
        }
	# special code for sles10
        if ($::DISTRO_VERSION eq '10')
        {       
	    my $install_src = " install=$install_source_header/" . "Linux/$::DISTRO_NAME/$::DISTRO_VERSION/ppc64/$svc_level/CD1";
	    $kernelparameters = $kernelparameters . $install_src;
        }
	elsif ($::DISTRO_VERSION eq '9')
	{
            my $install_src = " install=$install_source_header/" . "Linux/$::DISTRO_NAME/$::DISTRO_VERSION/ppc64/$svc_level"
;
            $kernelparameters = $kernelparameters . $install_src;
	}

        # On HMC and within DC environment, we need to skip bootp request when node netbooting.
        # So, we have to pass ip and netmask boot parameters to notify lpar ks info 
        # instead of using DHCP/bootp services.
    
        if (  NodeUtils->isHMC()
           && ( defined $ENV{'DC_ENVIRONMENT'} && $ENV{'DC_ENVIRONMENT'} ne 'CSM')
           )
        {
    		$kernelparameters = $kernelparameters . " hostip=$client_ip netmask=$client_netmask";
                if ($gateway)
                {
                    $kernelparameters = $kernelparameters . " gateway=$gateway";
                }
        }
    }
    $kernelparameters .= " $ENV{KERNELPAR}";
    return $kernelparameters;
}


#-------------------------------------------------------------------------                                        

=head3  common_node_check_correct                                                                       

	Notes:
		Check and correct the Node attribute for some kinds of 
	applications.                                                                                                    
	Arguments:
		$node_hash_ref - Nodes List that will be checked and corrected
		$check_opt_ref - Options for what it should check for.                                                                                             
	Returns:
		$node_hash_ref - Nodes List that has been removed the invalid nodes
		%Except_Node   - Invalid nodes array

=cut

#-------------------------------------------------------------------------
sub common_node_check_correct
{
    my ($class, $node_hash_ref, $check_opt_ref) = @_;

    my %check_opt = %$check_opt_ref;

    my %Except_Node;
    
    
    if ($check_opt{'Rmnode'})
    {
        foreach my $node (keys %$node_hash_ref)
        {
            my ($rc, $ms_ip, $node_ip);
	     ($rc, undef, $ms_ip) = 
	         NetworkUtils->getHost_noexit($$node_hash_ref{$node}{'ManagementServer'});
	     if ($rc)
	     {
	         push @{$Except_Node{'Rmnode'}}, $node;
	         delete $$node_hash_ref{$node};
	         next;
	     }
	     ($rc, undef, $node_ip) = NetworkUtils->getHost_noexit($node);
            if ($rc)
            {
                push @{$Except_Node{'Rmnode'}}, $node;
                delete $$node_hash_ref{$node};
                next;
            }

            # MS is a node, don't need IS for MS
            if ($ms_ip eq $node_ip)
            {
                NodeUtils->setInstallServerAttribute($$node_hash_ref{$node}, "");
            }
        }
    }
#    elsif ($check_opt{'installserver'})
#    {       
#    }
#    elsif ($check_opt{'Attribute'})
#    {
#    
#    }

    return ($node_hash_ref, \%Except_Node);

}


#-------------------------------------------------------------------------

=head3  getHashInstallServerToNodeInCluster

	Notes:
		Get a HASH that store the node list for Install Server. In the HASH, the Key is 
		the IS, the value for this key is the nodes list that use the IS as the Install Server.
		If the $ref_IS_list is available, this subroutine only search the node list for the 
		ISs in this list. Or all of the IS that exist in this cluster will be searched.
		
	Arguments:
		$ref_IS_list - IS list that will be searched
                
	Returns:
		%IStoNode - Key (IS) => Value (Node list that use the IS as the install server)

=cut

#-------------------------------------------------------------------------
sub getHashInstallServerToNodeInCluster
{
    my ($class, $ref_IS_list) = @_;

    my @IS_list = @$ref_IS_list;
    my %IStoNode;

    #Get Hostname::InstallServer attributes for  all of the nodes in the cluster
    my $resclass = 'IBM.ManagedNode';
    my $attrs = 'Hostname::InstallServer';
    my $outputFormat = "-D ':|:'";
    my $exitcode = -2;
    my $outref   = NodeUtils->runrmccmd('lsrsrc-api', "$outputFormat -i", "-s ${resclass}::::$attrs", $exitcode);

    my ($name, $IS);
    foreach my $line (@$outref)
    {
        my @IS4node;
        ($name, $IS) = split(/:\|:/, $line, 2);

        # In case the $IS is something like "ISname:dir"
        ($IS, my $dir) = split (":", $IS);
        
        #Put all the ISs into the @IS4node for a node
        if ($IS)
        {
            my $IS_grp = NodeUtils->getNodegrp($IS);
            if ($IS_grp)
            {
                # Handle the nodegrp
                foreach my $isvr (@$IS_grp)
                {
                    push @IS4node, $isvr;
                }
            }
            else
            {
                push @IS4node, $IS;
            }
        }
        else
        {
            next;
        }

        #For each IS, push the Hostname of current node
        foreach my $isvr1 (@IS4node)
        {
            $isvr1 = NetworkUtils->tryHost($isvr1);
            if (@IS_list)
            {
                if (grep (/^$isvr1$/, @IS_list))
                {
                    push @{$IStoNode{$isvr1}}, $name;
                }
            }
            else
            {
                push @{$IStoNode{$isvr1}}, $name;
            }
        }
    }

    return \%IStoNode;
}



#-------------------------------------------------------------------------

=head3	getInstallServerHashFromNode

	Notes: 
	Arguments:
		$ref_nodehash - NodeHash
                $skip_unavailable_IS - For an unavailable IS: If this argument is TRUE, 
                    this subroutine will report error messages and exit; Otherwise, this subroutine just reports
                    a warning message.

	Returns:
		%rc			-	Error code. 
						101-Cannot resolve ManagementServer IP;
						102-Cannot resolve node itself;
						103-Get hostname for IS failed;
						104-Node's IS is itself;
						105-IS is unavailable;
		%NodeHash	-	Target NodeHash, the InstallServer/NFSServer attribute of the nodes are set
						to the InstallServer/NFSServer's Hostname attribute. And delete the nodes 
						which are the InstallServer of other target nodes to NodeAndISHash.
		%ISHash		-	Node's InstallServer Hash
		%ISToNodeHash -	A new hash which key is the InstallServer, the value is Node Hostname 

=cut

#-------------------------------------------------------------------------
sub getInstallServerHashFromNode
{
    my ($class, $ref_nodehash, $skip_unavailable_IS) = @_;
    my (%NodeHash, %ISHash, %ISToNodeHash);

    my @unavailable_IS;
    my @grp_installserver;
    my %handled_installserver;
    my $is;
    my $nfs;
    my $dir;
    my $default_dir = $::IS_DEFAULT_DIR;    #/csmserver
    my $ms_ip;
    my $check_delim = ':\|:';               # Used when splitting the line apart
    my $lsnode_attr =
      "Hostname,ManagementServer,InstallOSName,InstallCSMVersion,InstallDistributionVersion,InstallDistributionName,InstallServiceLevel,InstallPkgArchitecture,Mode,InstallAdapterMacaddr,InstallAdapterType,InstallMethod,ConsoleSerialDevice,Status,InstallServer,NFSServer";
    my $rc;
    my $err_code;

    %NodeHash = %$ref_nodehash;

    foreach my $node (keys %NodeHash)
    {

        # get InstallServer or NFSServer, otherwise, delete this node from nodehash;
        if (!$ms_ip)
        {
            ($rc, undef, $ms_ip) =
              NetworkUtils->getHost_noexit($NodeHash{$node}{'ManagementServer'});
            if ($rc)
            {
                $err_code = 101;
                return ($err_code);
            }
        }
        my $node_ip;	
        ($rc, undef, $node_ip) = NetworkUtils->getHost_noexit($node);
        if ($rc)
        {
            $err_code = 102;
            return ($err_code);
        }
        
        # MS is a node, don't need IS for MS
        if ($ms_ip eq $node_ip)
        {
            NodeUtils->setInstallServerAttribute($NodeHash{$node}, "");
            next;
        }

        # get IS and Dir
        ($is, $nfs, $dir) =
          NodeUtils->getInstallServerAttribute($NodeHash{$node});
        if (!$is && !$nfs)
        {
            next;
        }

        # NFSServer cannot be a Nodegrp
        if ($is)
        {
            my $ref_grp = NodeUtils->getNodegrp($is);
            if ($ref_grp)
            {

                # Handle the nodegrp in last
                foreach my $isvr (@$ref_grp)
                {
                    push @grp_installserver, $isvr;
                }
                next;
            }
        }
        else
        {
            $is = $nfs;
        }

        #Unify the InstallServer/NFSServer
        my ($real_is, $is_ip);
	if ($skip_unavailable_IS)
        {
            ($real_is, $is_ip) = NetworkUtils->getHost_rveg($is);
            if (!$real_is || !$is_ip)
            {
                NodeUtils->setInstallServerAttribute($NodeHash{$node}, "");
                push @unavailable_IS, $node;
                next;
            }
        }
        else
        {
            ($rc, $real_is, $is_ip) = NetworkUtils->getHost_noexit($is);
            if ($rc)
            {
                $err_code = 103;
                return ($err_code);
            }
        }

        # the IS is the ManagementServer
        if ($is_ip eq $ms_ip)
        {
            NodeUtils->setInstallServerAttribute($NodeHash{$node}, "");
            next;
        }

        # Node's IS is itself
        if ($is_ip eq $node_ip)
        {
            if ($skip_unavailable_IS)
            {
                NodeUtils->setInstallServerAttribute($NodeHash{$node}, "");
                push @unavailable_IS, $node;
                next;
            }
            else
            {
                MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E',
                                         'EMsgInvalidInstallServer', $node);
                $err_code = 104;
                return ($err_code);
            }
        }

        # in case the IS already in NodeHash
        if (grep(/^$real_is$/, keys %handled_installserver))
        {

            # Format the InstallServer/NFSServer Attribute to avoid repeated resolution
            if ($dir)
            {
                $is = $handled_installserver{$real_is} . ":" . $dir;
            }
            else
            {
                $is = $handled_installserver{$real_is};
            }
            NodeUtils->setInstallServerAttribute($NodeHash{$node}, $is);
            next;
        }

        # Get the installserver hash and set the hash
        my @nodelist = ($real_is);
        my %is_hash = NodeUtils->setNodeHash(\@nodelist, $lsnode_attr);
        if (!%is_hash)
        {
            if ($skip_unavailable_IS)
            {
                NodeUtils->setInstallServerAttribute($NodeHash{$node}, "");
                push @unavailable_IS, $node;
                next;
            }
            else
            {
                MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E',
                                         'EMsgInvalidInstallServer', $node);
                $err_code = 105;
                return ($err_code);
            }
        }

        # Make sure the Key is IS's Hostname
        my @is_hostname = keys %is_hash;

        %ISHash = (%ISHash, %is_hash);

        # Change the InstallServer/NFSServer of dest_node_hash to the keys of the installserver_hash
        if ($dir)
        {
            $is = $is_hostname[0] . ":" . $dir;
        }
        else
        {
            $is = $is_hostname[0];
        }
        $handled_installserver{$real_is} = $is_hostname[0];
        NodeUtils->setInstallServerAttribute($NodeHash{$node}, $is);
    }

    if (@unavailable_IS)
    {
        MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'I',
                                         'IMsgUnavailableIS', join(',', @unavailable_IS));
    }
    # Handle the installserver group
    my @installserver       = keys %ISHash;
    my @other_installserver = ();
    foreach my $isvr (@grp_installserver)
    {
        if (   grep(/^$isvr$/, @installserver)
            || grep(/^$isvr$/, @other_installserver))
        {
            next;
        }
        push @other_installserver, $isvr;
    }
    if (@other_installserver)
    {
        my %is_hash =
          NodeUtils->setNodeHash(\@other_installserver, $lsnode_attr);
        %ISHash = (%ISHash, %is_hash);
    }
    return (0, \%NodeHash, \%ISHash);
}

#-------------------------------------------------------------------------

=head3	excludeInstallServerFromNode

	Notes: 
	Arguments:
		%NodeHash
		%ISHash

	Returns:
		%NodeHash	-	Target NodeHash, the InstallServer/NFSServer attribute of the nodes are set
						to the InstallServer/NFSServer's Hostname attribute. And delete the nodes 
						which are the InstallServer of other target nodes to NodeAndISHash.
		%ISHash		-	Node's InstallServer Hash
		%ISToNodeHash -	A new hash which key is the InstallServer, the value is Node Hostname 
		%NodeAndISHash-	The NodeHash which is splited from %NodeHash, the node in the hash is the IS
						of %NodeHash

=cut

#-------------------------------------------------------------------------
sub excludeInstallServerFromNodeHash
{
    my ($class, $ref_nodehash, $ref_ishash) = @_;

    my (%NodeHash, %ISHash, %ISToNodeHash, %NodeAndISHash);
    my $default_dir = $::IS_DEFAULT_DIR;

    %NodeHash = %$ref_nodehash;
    %ISHash   = %$ref_ishash;

    # Since the InstallServer has been formated to its Hostname, don't need to handle it
    foreach my $is (keys %ISHash)
    {

        # Node is one of the InstallServer or NFSServer
        if (grep /^$is$/, keys %NodeHash)
        {
            %{$NodeAndISHash{$is}} = %{$NodeHash{$is}};
            delete $NodeHash{$is};
        }
    }

    # Cleanup the ISHash and generate ISToNodehash
    foreach my $node (keys %NodeHash)
    {
        my ($is, $nfs, $dir) =
          NodeUtils->getInstallServerAttribute($NodeHash{$node});
        if (!$is and !$nfs)
        {
            next;
        }
        if (!$dir)
        {
            $dir = $default_dir;
        }
        if ($is)
        {
            my $ref_grp = NodeUtils->getNodegrp($is);
            if ($ref_grp)
            {
                foreach my $isvr (@$ref_grp)
                {

                    # in case InstallServer not in ISHash
                    if (grep(/$isvr/, keys %ISHash))
                    {
                        push @{$ISToNodeHash{$isvr}}, $node;
                        $ISHash{$isvr}{'Dir'} = $dir;
                    }
                }
                next;
            }
        }
        else
        {
            $is = $nfs;
        }
        push @{$ISToNodeHash{$is}}, $node;
        $ISHash{$is}{'Dir'} = $dir;
    }

    # Delete the IS which have no node using it
    foreach my $is (keys %ISHash)
    {
        if (!(grep /$is/, keys %ISToNodeHash))
        {
            delete $ISHash{$is};
        }
    }
    return (\%NodeHash, \%ISHash, \%ISToNodeHash, \%NodeAndISHash);
}

#-------------------------------------------------------------------------

=head3	getIStoNodeHASHFromNodeHASH

	Notes: 
	Arguments:
		$ref_nodehash - Node Hash

	Returns:
		%ISToNodeHash -	A new hash which key is the InstallServer, the value is Node Hostname
		%ISDir - The directory that is used to store the installation files in the Install Server. 
		The default one is /csmserver.

=cut

#-------------------------------------------------------------------------
sub getIStoNodeHASHFromNodeHASH
{
    my ($class, $ref_nodehash) = @_;
    my %NodeHash = %$ref_nodehash;

    my (%ISToNodeHash, %ISDir);
    # Generate the ISToNodehash
    foreach my $node (keys %NodeHash)
    {
        my ($is, $nfs, $dir) =
          NodeUtils->getInstallServerAttribute($NodeHash{$node});
        if (!$is and !$nfs)
        {
            next;
        }
        if (!$dir)
        {
            $dir = $::IS_DEFAULT_DIR;
        }
        if ($is)
        {
            my $ref_grp = NodeUtils->getNodegrp($is);
            if ($ref_grp)
            {
                foreach my $isvr (@$ref_grp)
                {
                    if (!grep (/^\Q$node\E$/, @{$ISToNodeHash{$isvr}}))
                    {
                        push @{$ISToNodeHash{$isvr}}, $node;
                        $ISDir{$isvr}{'Dir'} = $dir;
                    }
                }
                next;
            }
        }
        else
        {
            $is = $nfs;
        }
        if (!grep (/^\Q$node\E$/, @{$ISToNodeHash{$is}}))
        {
            push @{$ISToNodeHash{$is}}, $node;
            $ISDir{$is}{'Dir'} = $dir;
        }
    }

    return (\%ISToNodeHash, \%ISDir);
}

#-------------------------------------------------------------------------

=head3	getNodeHashFromInstallServer

	Notes: 
	Arguments:
		$ref_ishash - InstallServer Hash

	Returns:
		%NodeHash	-	Target NodeHash, the InstallServer/NFSServer attribute of the nodes are set
						to the InstallServer/NFSServer's Hostname attribute. And delete the nodes 
						which are the InstallServer of other target nodes to NodeAndISHash.
		%ISHash		-	Node's InstallServer Hash
		%ISToNodeHash -	A new hash which key is the InstallServer, the value is Node Hostname 

=cut

#-------------------------------------------------------------------------
sub getNodeHashFromInstallServer
{
    my ($class, $ref_nodehash) = @_;
    my %ISHash = %$ref_nodehash;
    my %NodeHash;
    my @empty = ();
    my @installserver_ip;
    my ($attr_installserver, $attr_nfsserver);
    my ($is,                 $dir);
    my %is_ip_hostname;

    my $lsnode_attr =
      "Hostname,ManagementServer,InstallOSName,InstallCSMVersion,InstallDistributionVersion,InstallDistributionName,InstallPkgArchitecture,Mode,InstallAdapterMacaddr,InstallAdapterType,InstallMethod,ConsoleSerialDevice,Status,InstallServer,NFSServer";

    #Unify the installserver for search
    foreach my $node (keys %ISHash)
    {
        my (undef, $ip) = NetworkUtils->getHost($node);
        $is_ip_hostname{$ip} = $node;
    }

    # Get all nodes attributes
    my @installserver = keys %ISHash;
    my $lsnode_info = NodeUtils->listNodeAttrs(\@empty, $lsnode_attr);
    my $check_delim = ':\|:';    # Used when splitting the line apart
    foreach my $line (@$lsnode_info)
    {    #this only has node names in it from the cluster
        my @attributes = split /$check_delim/, $line;
        my $hostname = $attributes[0];
        chomp $line;
        my @attr_names = split ',', $lsnode_attr;
        my $index = 0;
        $attr_installserver = "";
        $attr_nfsserver     = "";

        # Get the InstallServer/NFSServer for nodes
        # Could be optimized
        my $is_attr = 0;
        foreach my $attr (@attr_names)
        {
            if ($attr eq "InstallServer" && $attributes[$index])
            {
                $attr_installserver = $attributes[$index];
                $is_attr            = 1;
                last;
            }
            if ($attr eq "NFSServer" && $attributes[$index])
            {
                $attr_nfsserver = $attributes[$index];
                if ($is_attr)
                {
                    last;
                }
            }
            $index++;
        }
        if ($attr_installserver)
        {
            ($is, $dir) = split(":", $attr_installserver);
        }
        elsif ($attr_nfsserver)
        {
            ($is, $dir) = split(":", $attr_nfsserver);
        }
        else
        {
            next;
        }
        if (!$is)
        {
            next;
        }

        my $ref_grp = NodeUtils->getNodegrp($is);
        if ($ref_grp)
        {

            # Add nodes into dest_node_hash if any nodes in the group exists in installserver hash
            foreach my $node (@$ref_grp)
            {
                if (grep(/$node/, keys %ISHash))
                {
                    $index = 0;
                    foreach my $attr (@attr_names)
                    {
                        $NodeHash{$hostname}{$attr} = $attributes[$index];
                        $index++;
                    }
                    last;
                }
            }
        }
        else
        {
            my (undef, $is_ip) = NetworkUtils->getHost($is);
            if (grep(/$is_ip/, keys %is_ip_hostname))
            {

                # Save the node attribute to $dest_node_hash
                $index = 0;
                foreach my $attr (@attr_names)
                {
                    $NodeHash{$hostname}{$attr} = $attributes[$index];
                    $index++;
                }
            }

            # Change the InstallServer/NFSServer of dest_node_hash to the keys of the installserver_hash
            if ($dir)
            {
                $is = $is_ip_hostname{$is_ip} . ":" . $dir;
            }
            else
            {
                $is = $is_ip_hostname{$is_ip};
            }
            NodeUtils->setInstallServerAttribute($NodeHash{$hostname}, $is);
        }
    }
    return (\%NodeHash, \%ISHash);
}

#-----------------------------------------------------------------------------------------------

=head3	getTargetNodeHashandInstallServerHash

	Notes: 
	Arguments:
		$ref_nodehash - NodeServer Hash

	Returns:
		%NodeHash	-	Target NodeHash, the InstallServer/NFSServer attribute of the nodes are set
						to the InstallServer/NFSServer's Hostname attribute. And delete the nodes 
						which are the InstallServer of other target nodes to NodeAndISHash.
		%ISHash		-	Node's InstallServer Hash
		%ISToNodeHash -	A new hash which key is the InstallServer, the value is Node Hostname 

=cut

#------------------------------------------------------------------------------------------------
sub getTargetNodeHashandInstallServerHash
{
    my ($class, $ref_nodehash) = @_;
    my ($ref_targetnodehash, $ref_ishash, $ref_istonodehash,
        $ref_nodeandishash);
    my $return_code;
    ($return_code, $ref_targetnodehash, $ref_ishash) =
      ServerUtils->getInstallServerHashFromNode($ref_nodehash);
    if ($return_code)
    {
        exit 1;
    }

    # If there are InstallServer
    if (keys %$ref_ishash)
    {
        (
         $ref_targetnodehash, $ref_ishash,
         $ref_istonodehash,   $ref_nodeandishash
          )
          = ServerUtils->excludeInstallServerFromNodeHash($ref_targetnodehash,
                                                          $ref_ishash);
    }
    if (keys %$ref_nodeandishash)
    {
        my $host = join(",", keys %$ref_nodeandishash);
        MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'I',
                                     'IMsgSkipInstallServer', $host);
    }

    #Check whether there are Nodes which served by an UnManaged InstallServer
    my @NodewithUnManagedIS;
    foreach my $is (keys %$ref_ishash)
    {
        if ($$ref_ishash{$is}{'Mode'} ne 'Managed')
        {
            foreach my $node (@{$$ref_istonodehash{$is}})
            {
                if (!grep(/^\Q$node\E$/, @NodewithUnManagedIS))
                {
                    push @NodewithUnManagedIS, $node;
                }
            }
        }
    }
    if (@NodewithUnManagedIS)
    {
        MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET, 'E1',
            'EMsgNodewithUnManagedIS', join(',', @NodewithUnManagedIS));
    }
    
    return ($ref_targetnodehash, $ref_ishash, $ref_istonodehash);
}

#-----------------------------------------------------------------------------------------------

=head3	get_installing_nodes

	Notes: 
	Arguments:

	Returns:
		@nodelist - List of installing nodes

=cut

#------------------------------------------------------------------------------------------------
sub get_installing_nodes
{
    my @empty_array;
    my $lsnode_attr = "Hostname";
    my @nodelist;
    my @lsnode_info;

    my $select_string   = "Mode=\"Installing\"";
    my %opthash         = (WhereStr => $select_string);
    my $ref_lsnode_info =
      NodeUtils->listNodeAttrs(\@empty_array, $lsnode_attr, \%opthash);
    push(@lsnode_info, @$ref_lsnode_info);

    $select_string   = "Mode=\"MinManaged-Installing\"";
    %opthash         = (WhereStr => $select_string);
    $ref_lsnode_info =
      NodeUtils->listNodeAttrs(\@empty_array, $lsnode_attr, \%opthash);
    push(@lsnode_info, @$ref_lsnode_info);

    foreach my $line (@lsnode_info)
    {
        chomp $line;
        push(@nodelist, $line);
    }
    return @nodelist;

}

#--------------------------------------------------------------------------------

=head3     validate_power_method

          Validate nodes' PowerMethod. If a node is a blade node, this node's Attribute
          "IsBlade" will be set to "yes". If not, nothing will be done.
          To reduce the run time of command "lshwstat", only when this node is a SLES9 
          GA node ,and it's PowerMethod is xseries, "lshwstat" will be executed, 
          otherwise this node will be ignored.
          If command 'lshwstat' failed, there were 2 results for this node:
                1. If noreboot option flag/env var is set, give a warning message.
                2. If noreboot option flag/env var is NOT set, give a warning message
                and remove this node from installing queue. 

          Notes:

=cut

#--------------------------------------------------------------------------------
sub validate_power_method
{
    my ($class, $ref_nodehash) = @_;
    my %nodehash = %{$ref_nodehash};
    my @nodelist;

    foreach my $node (keys %nodehash)
    {
        if (   $nodehash{$node}{InstallDistributionName} eq 'SLES'
            && $nodehash{$node}{InstallDistributionVersion} eq '9'
            && $nodehash{$node}{InstallServiceLevel}        eq 'GA'
            && $nodehash{$node}{PowerMethod}                eq 'xseries')
        {
            push @nodelist, $node;
        }

        # When UUID and HTTP install method meets blade targets, special disposal
        # is required. Save the "IsBlade" attribute to config_info file so that it
        # can be used on the install server side.
        if ($nodehash{$node}{UUID} && !$nodehash{$node}{InstallAdapterMacaddr})
        {
            push @nodelist, $node;
        }
    }

    if (@nodelist)
    {
        my $nodefile = ServerUtils->make_node_list_file(\@nodelist);
        my @cmdres   =
          NodeUtils->runcmd("/opt/csm/bin/lshwstat -f $nodefile  blade", -1);
        foreach my $line (@cmdres)
        {
            chomp($line);
            next if ($line !~ /(.*):\s+(.*)/);
            my $node = $1;
            ($node, undef) = NetworkUtils->getHost($node);
            my $status = $2;
            $nodehash{$node}{IsBlade} = "yes" if ($status eq 'yes');
        }
        ServerUtils->close_delete_file($::NODE_LIST_FILE, $nodefile)
          if ($nodefile);
    }

    return \%nodehash;

}

#--------------------------------------------------------------------------------

=head3    make_allow_file 

	  Arguments:
		$nodelist - node list
	  Returns:
          Notes:

=cut

#--------------------------------------------------------------------------------
sub make_allow_file
{
    my ($class, $nodelist) = @_;
    
    my $allow_file = "/var/opt/csm/AllowNodeMsgs";
    my @old_nodelist;

    if (-e $allow_file)
    {
        open(ALLOWFILE, "<$allow_file")
          or MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                        'csminstall', "E", 'EMsgCANT_READ_FILE', "$allow_file");
        @old_nodelist = <ALLOWFILE>;
        close(ALLOWFILE);
    }

    open(ALLOWFILE, ">>$allow_file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',      $::MSGMAPPATH,
                                      'csminstall',          "E",
                                      'EMsgCANT_WRITE_FILE', "$allow_file"
                                      );
    foreach my $node (@$nodelist)
    {
        if (grep(/^$node$/, @old_nodelist)) { next; }
        print ALLOWFILE "$node\n";
    }
    close(ALLOWFILE);

}

#--------------------------------------------------------------------------------

=head3    setAllowNodeMsgs 

	  Arguments:
	  Returns:
          Notes:
		Produce AllowNodeMsgs file on management server and install server.

=cut

#--------------------------------------------------------------------------------
sub setAllowNodeMsgs
{
	my ($ref_MgtNodes, $ref_InstallServers) =
	  ServerUtils->getInstallServers(\%::NODEHASH);    
	my %InstallServers = %$ref_InstallServers;
	my @MgtNodes       = @$ref_MgtNodes;
	my $allow_file     = "/var/opt/csm/AllowNodeMsgs";

	my @mgtnodes;
	foreach my $node (@MgtNodes)
	{
		my ($hostname,$ip) = NetworkUtils->getHost($node);
		if ($::NODEHASH{$node}{"InstallAdapterHostname"})
		{
			my $adapter_hostname=$::NODEHASH{$node}{"InstallAdapterHostname"};
			my ($res_hostname,$res_ip)=NetworkUtils->getHost($adapter_hostname);
			$ip=$ip . " " . $res_ip;
		}
		push @mgtnodes, $ip;
	}
	if (@mgtnodes)
	{
		ServerUtils->make_allow_file(\@mgtnodes);
	}

	my %realIS;
	foreach my $is (keys %InstallServers)
	{
		foreach my $dir (keys %{$InstallServers{$is}{dir}})
		{
			my @nodelist = @{$InstallServers{$is}{dir}{$dir}};
			if ($InstallServers{$is}{isGrp})
			{
				my $refGroup = NodeUtils->getNodegrp($is);
				if ($refGroup)
				{
					foreach my $node (@nodelist)
					{
						my $real_is = NodeUtils->getNodeFromGroup($is, $node);
						my ($hostname, $ip) = NetworkUtils->getHost($node);
						push @{$realIS{$real_is}}, $ip;
					}
				}
			}
			else
			{
				my @ip_list;
				foreach my $node (@nodelist)
				{
					if ($::NODEHASH{$node}{"InstallAdapterHostname"})
					{
						my $adapter_hostname =
						  $::NODEHASH{$node}{"InstallAdapterHostname"};
						my $ip =
						  ServerUtils->dshGetNodeAddr($adapter_hostname, $is);
						push @ip_list, $ip;
					}
					else
					{
						my ($hostname, $ip) = NetworkUtils->getHost($node);
						push @ip_list, $ip;
					}
				}
				push @{$realIS{$is}}, @ip_list;
			}
		}
	}

	my @install_servers = keys %realIS;
	my @server_list;
	foreach my $is (@install_servers)
	{
		my ($hostname, $ip) = NetworkUtils->getHost($is);
		push @server_list, $ip;
	}
	if (@server_list)
	{
		ServerUtils->make_allow_file(\@server_list);
	}
	foreach my $is (keys %realIS)
	{
		my @nodelist = @{$realIS{$is}};
		my $cmd;
		$cmd = "/opt/csm/bin/dsh -n $is \"cat $allow_file\"";
		my $output = NodeUtils->runcmd($cmd, -1, 1);
		if (!$::RUNCMD_RC && !grep(/No\ such\ file\ or\ directory/, @$output))
		{
			foreach my $line (@$output)
			{
				$line =~ s/^$is:\ //g;
				push @nodelist, $line;
			}
			@nodelist = ServerUtils->get_uniqu_arry_elemnts(@nodelist);
		}

		my $node_file = ServerUtils->make_node_list_file(\@nodelist);
		$cmd = "/opt/csm/bin/dcp -n $is $node_file $allow_file";
		NodeUtils->runcmd($cmd, -2);
		ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);
	}
}


#--------------------------------------------------------------------------------

=head3    start_nodestatusd 

	  Arguments:
	  Returns:
          Notes:
		Start nodestatusd daemon on management server and install server.

=cut

#--------------------------------------------------------------------------------
sub start_nodestatusd
{
    my ($ref_MgtNodes, $ref_InstallServers) =
      ServerUtils->getInstallServers(\%::NODEHASH);
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my $nodestatusd    = "/opt/csm/csmbin/nodestatusd";
    my $port;
    my $dsh = "/opt/csm/bin/dsh";
    my $dcp = "/opt/csm/bin/dcp";
    my $mgmtsvr;

    foreach my $is (keys %InstallServers)
    {
        foreach my $dir (keys %{$InstallServers{$is}{dir}})
        {
            my @nodelist = @{$InstallServers{$is}{dir}{$dir}};
            $mgmtsvr = $::NODEHASH{$nodelist[0]}{'ManagementServer'};
            my $cmd;
            if ($InstallServers{$is}{isGrp})
            {
                my $refGroup = NodeUtils->getNodegrp($is);
                if ($refGroup)
                {
                    my @realIS;
                    my %ports;
                    foreach my $node (@nodelist)
                    {
                        my $real_is = NodeUtils->getNodeFromGroup($is, $node);
                        push @realIS, $real_is;
                        $ports{$real_is} = $::NODEHASH{$node}{'InstallMsgPort'};
                    }
                    foreach my $is (@realIS)
                    {
                        my ($ms_hostname, $ms_ip) =
                          NetworkUtils->get_management_server($is);
                        # To prevent killall cmd with different syntax,
                        # Use different syntax if the first one fails.
                        $cmd =
                          "$dsh -n $is \"echo $dir >/tmp/CSMINSTALL_ROOT;killall -s HUP nodestatusd 2>&1 || killall -HUP nodestatusd 2>&1 || CSMINSTALL_ROOT=$dir MGMTSVR=$ms_ip SERVER=IS MS_PORT=$::MS_PORT $nodestatusd -p $ports{$is}\"";
                        NodeUtils->runcmd($cmd, -2);
                    }
                }
            }
            else
            {
                $port = $::NODEHASH{$nodelist[0]}{'InstallMsgPort'};
                my ($ms_hostname, $ms_ip) =
                  NetworkUtils->get_management_server($is);
                # To prevent killall cmd with different syntax,
                # Use different syntax if first one fails.
                $cmd =
                  "$dsh -n $is \"echo $dir >/tmp/CSMINSTALL_ROOT;killall -s HUP nodestatusd 2>&1 || killall -HUP nodestatusd 2>&1 || CSMINSTALL_ROOT=$dir MGMTSVR=$ms_ip SERVER=IS MS_PORT=$::MS_PORT $nodestatusd -p $port\"";
                NodeUtils->runcmd($cmd, -2);
            }
        }
    }

    my @ms_ip;
    if (NodeUtils->isAIX)
    {
        my $cmd = "$::IFCONFIG -a|grep \"inet\ \"";
        my $outref = NodeUtils->runcmd($cmd, -2, 1);
        foreach my $ip (@$outref)
        {
            $ip =~ s/^.*inet\ //g;
            $ip =~ s/\ .*//g;
            push @ms_ip, $ip;
        }
        $mgmtsvr = join(',', @ms_ip);
        $cmd     = "ps -e -o pid,args|grep nodestatusd";
        $outref  = NodeUtils->runcmd($cmd, -2, 1);
        if (@$outref > 1)
        {
            my $cmd = "kill -HUP";
            foreach my $pid (@$outref)
            {
                chomp $pid;
                $pid =~ s/\s.*//g;
                $cmd = "$cmd $pid";
            }
            NodeUtils->runcmd($cmd, -1);
        }
        else
        {
            $cmd =
              "CSMINSTALL_ROOT=/csminstall MGMTSVR=$mgmtsvr SERVER=MS $nodestatusd -p $::MS_PORT";
            NodeUtils->runcmd($cmd, -2);
        }
    }
    elsif (NodeUtils->isLinux)
    {
        my $cmd = "$::IFCONFIG -a|grep \"inet\ addr\"";
        my $outref = NodeUtils->runcmd($cmd, -2, 1);
        foreach my $ip (@$outref)
        {
            $ip =~ s/\s+inet\ addr://g;
            $ip =~ s/\ .*//g;
            push @ms_ip, $ip;
        }
        $mgmtsvr = join(',', @ms_ip);
        # To prevent killall cmd with different syntax,
        # Use different syntax if the first one fails.
        $cmd =
          "killall -s HUP nodestatusd 2>&1 || killall -HUP nodestatusd 2>&1 || CSMINSTALL_ROOT=/csminstall MGMTSVR=$mgmtsvr SERVER=MS $nodestatusd -p $::MS_PORT";
        NodeUtils->runcmd($cmd, -2);
    }
}

#--------------------------------------------------------------------------------

=head3    findInstallMsgPort 

	  Arguments:
	  	$service_ports - ports declared in /etc/service file.
		$active_ports - ports in use.
	  Returns:
          Notes:
		
=cut

#--------------------------------------------------------------------------------
sub findInstallMsgPort
{
    my ($pkg, $service_ports, $active_ports) = @_;
    my $port      = 3101;
    my $port_name = "csm-nodemsgs";

    if (grep(/^$port_name/, @$service_ports))
    {
        foreach my $line (@$service_ports)
        {
            if (grep(/$port_name/, $line))
            {
                chomp $line;
                $line =~ s /^$port_name\s+//g;
                $line =~ s /\/tcp.*$//g;
                $line =~ s /\/udp.*$//g;
                $port = $line;
                last;
            }
        }
    }
    else
    {
        while (1)
        {
            if (
                !(    grep(/$port/, @$service_ports)
                   || grep(/$port/, @$active_ports))
              )
            {
                last;
            }
            else { $port++; }
        }
    }

    return $port;
}

#--------------------------------------------------------------------------------

=head3    setInstallMsgPort 

	  Arguments:
	  Returns:
          Notes:
		Set InstallMsgPort used by nodestatusd daemon.
=cut

#--------------------------------------------------------------------------------
sub setInstallMsgPort
{
    my ($ref_MgtNodes, $ref_InstallServers) =
      ServerUtils->getInstallServers(\%::NODEHASH);    #
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my $file           = "/etc/services";
    my $port_name      = "csm-nodemsgs";
    my @service_ports;
    my @active_ports;

    open(SERVICES, "<$file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',     $::MSGMAPPATH,
                                      'csminstall',         "E",
                                      'EMsgCANT_READ_FILE', "$file"
                                      );
    @service_ports = <SERVICES>;
    close(SERVICES);
    my $output = NodeUtils->runcmd("netstat -ta", -1, 1);
    $::MS_PORT = ServerUtils->findInstallMsgPort(\@service_ports, $output);

    if (!grep(/$port_name/, @service_ports))
    {
        open(SERVICES, ">>$file")
          or MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', "E", 'EMsgCANT_WRITE_FILE', "$file");
        print SERVICES "$port_name\t$::MS_PORT\/tcp\n";
        print SERVICES "$port_name\t$::MS_PORT\/udp\n";
        close(SERVICES);
    }

    foreach my $node (@MgtNodes)
    {
        $::NODEHASH{$node}{'InstallMsgPort'} = $::MS_PORT;
    }
    my %realIS;
    foreach my $is (keys %InstallServers)
    {
        foreach my $dir (keys %{$InstallServers{$is}{dir}})
        {
            my @nodelist = @{$InstallServers{$is}{dir}{$dir}};
            if ($InstallServers{$is}{isGrp})
            {
                my $refGroup = NodeUtils->getNodegrp($is);
                if ($refGroup)
                {
                    foreach my $node (@nodelist)
                    {
                        my $real_is = NodeUtils->getNodeFromGroup($is, $node);
                        push @{$realIS{$real_is}}, $node;
                    }
                }
            }
            else
            {
                push @{$realIS{$is}}, @nodelist;
            }
        }
    }

    foreach my $is (keys %realIS)
    {
        my @nodelist = @{$realIS{$is}};
        my $cmd;
        my (@service_ports,@active_ports);
        $cmd = "/opt/csm/bin/dsh -n $is \"cat $file\"";
        my $output = NodeUtils->runcmd($cmd, -1, 1);
        if (!$::RUNCMD_RC)
        {
            foreach my $line (@$output)
            {
                $line =~ s/^$is:\ //g;
                push @service_ports, $line;
            }
        }
        $cmd = "/opt/csm/bin/dsh -n $is \"netstat -tn\"";
        $output = NodeUtils->runcmd($cmd, -1, 1);
        if (!$::RUNCMD_RC)
        {
            foreach my $line (@$output)
            {
                $line =~ s/^$is:\ //g;
                push @active_ports, $line;
            }
        }
        my $port =
          ServerUtils->findInstallMsgPort(\@service_ports, \@active_ports);
        foreach my $node (@nodelist)
        {
            $::NODEHASH{$node}{'InstallMsgPort'} = $port;
        }
        if (!grep(/^$port_name/, @service_ports))
        {
            my $line = "$port_name\t$port\/tcp";
            push @service_ports, $line;
            $line = "$port_name\t$port\/udp";
            push @service_ports, $line;
            my $node_file = ServerUtils->make_node_list_file(\@service_ports);
            $cmd = "/opt/csm/bin/dcp -n $is $node_file $file";
            NodeUtils->runcmd($cmd, -2);
            ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file);
        }
    }
}

#--------------------------------------------------------------------------------

=head3    check_csm_version 

	  Arguments: $nodelist -  The nodes will be checked.
	  Returns:
          Notes:
		Check if CSM version on install server is lower than 1.5.

=cut

#--------------------------------------------------------------------------------
sub check_csm_version
{
    my ($class, $nodelist) = @_;

    my @dest_nodes = @$nodelist;
    
    my $query_cmd = "rpm -q csm.core --qf %{VERSION}";
    
    my @csm_version_error_list;
    my @csm_not_install_list;
    
    #Check the DSH reachability for all the nodes
    my $unReachable = NetworkUtils->checkRSReachability(\@dest_nodes);
    if ($unReachable != 0)
    {
        if (scalar @$unReachable > 0)
        {
            MessageUtils->message("E1", "EMsgISDown", join(',', @$unReachable));
        }
    }

    #Use DSH to get the CSM version information from the nodes 
    my $dshcmd = $query_cmd;
    my %options_api = ();
    $options_api{'nodes'} = join(',', @dest_nodes);
    $options_api{'command'} = $dshcmd;
    my @output = NodeUtils->runDsh(\%options_api, -1);

    #Parse the DSH output to get the nodes that have issue
    my $version;
    foreach my $node (@dest_nodes)
    {
        my ($ver_find, $ver_ok);
        foreach my $out_line (@output)
        {
            if ($out_line =~ /\s*$node:.*?(\d)\.(\d)\./)
            {
                $ver_find = 1;
                if ($1 > 1 || ($1 == 1 && $2 >= 5))
                {
                    $ver_ok = 1;
                }
            }
        }
        if ($ver_find)
        {
            if(!$ver_ok)
            {
                #CSM has a version le 1.5
                if (!grep(/^\Q$node\E$/, @csm_version_error_list))
                {
                    push @csm_version_error_list, $node;
                }
                
            }
        }
        else
        {
            #Maybe CSM has not been installed 
            if (!grep(/^\Q$node\E$/, @csm_not_install_list))
            {
                push @csm_not_install_list, $node;
            }
        }
    }

    if (@csm_not_install_list)
    {
        MessageUtils->message("E1", "EMsgNotInstallCsmOnIS", join(',', @csm_not_install_list));
    }
    if (@csm_version_error_list)
    {
        MessageUtils->message("E1", "EMsgBAD_CSMVER_ON_IS", join(',', @csm_version_error_list));
    }
}

#--------------------------------------------------------------------------------

=head3    exclude_config_dir

	  Arguments:
	  Returns:
          \@config_dirs		kickstart and autoyast config dir in /csminstall/csm

	    Get the config dir which are excluded when running syncServers. They are:
		
		/csminstall/csm/<distro>:						#link to the config dir
		/csminstall/csm/<csmversion>/autoyast.<distro>
		/csminstall/csm/<csmversion>/kickstart.<distro>
=cut

#--------------------------------------------------------------------------------
sub exclude_config_dir
{
    my @config_dirs  = ();
    my @install_type = ('kickstart', 'autoyast');

    foreach my $type (@install_type)
    {
        my $dir_str = "/csminstall/csm/*/$type*";
        my @ex_dirs = bsd_glob($dir_str);
        push @config_dirs, @ex_dirs;

        # There are links in /csminstall/csm link to /csminstall/csm/*/kickstart.distro
        # or /csminstall/csm/*/autoyast.distro; Do not transfer these link to install
        # server either.
        foreach my $dir (@ex_dirs)
        {
            my ($path, $distro) = split("$type.", $dir);
            chomp($distro);
            my $distro_link = "/csminstall/csm/$distro";
            if (-l $distro_link)
            {
                if (!grep(/^$distro_link$/, @config_dirs))
                {
                    push @config_dirs, $distro_link;
                }
            }
        }
    }
    return \@config_dirs;
}

#--------------------------------------------------------------------------------

=head3    remove_cfmcomplete_file 

	  Arguments: $refNodehash - reference to node hash
          Notes:
		Remove cfmcomplete file.
=cut

#--------------------------------------------------------------------------------
sub remove_cfmcomplete_file
{
    my ($pkg,          $refNodehash)        = @_;
    my ($ref_MgtNodes, $ref_InstallServers) =
      ServerUtils->getInstallServers($refNodehash);
    my %InstallServers = %$ref_InstallServers;
    my @MgtNodes       = @$ref_MgtNodes;
    my $cmd;

    if (@MgtNodes)
    {

        # remove the reboot_ready flag file on MS for the nodes using MS as IS
        my $cfmcomplete_file;
        foreach my $hostname (@MgtNodes)
        {
            $cfmcomplete_file =
              "/csminstall/csm/status" . "/" . $hostname . ".cfmcomplete";
            NodeUtils->runcmd("rm -f $cfmcomplete_file");
        }
    }

    foreach my $is (keys %InstallServers)
    {
        my $isISGrp = $InstallServers{$is}{isGrp};
        my $realis  = $is;
        if ($isISGrp)
        {
            $realis =~ s/^\+//;
        }

        foreach my $dir (keys %{$InstallServers{$is}{dir}})
        {
            my @nodelist = @{$InstallServers{$is}{dir}{$dir}};
            my @cfmcomplete_files;
            my $file_list;
            foreach my $hostname (@nodelist)
            {
                push @cfmcomplete_files,
                  $dir . "/csm/status/" . $hostname . ".cfmcomplete";
            }
            $file_list = join " ", @cfmcomplete_files;
            if ($isISGrp)
            {
                $cmd = "/opt/csm/bin/dsh -N $realis rm -f $file_list";
            }
            else
            {
                $cmd = "/opt/csm/bin/dsh -n $is rm -f $file_list";
            }
            NodeUtils->runcmd($cmd, -2);
        }
    }
}

#--------------------------------------------------------------------------------

=head3   check_invalid_options 

	  Arguments:		@options - processed arguments
	  Returns:
=cut

#--------------------------------------------------------------------------------
sub check_invalid_options
{
    my ($pkg, $options) = @_;
    if (scalar(@$options) > 0)
    {
        my @invalid_opts;
        foreach my $opt (@$options)
        {
            if ($opt =~ m/^-/)
            {
                push @invalid_opts, $opt;
            }
        }
        if (@invalid_opts)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'E', 'EMsgInvalidOptions',
                                  join(',', @invalid_opts));
            return 1;
        }
    }
    return 0;
}
#--------------------------------------------------------------------------------

=head3  nodestatus_stop_logging

    Arguments:
        $node_hash_ref
    Returns:
        none
=cut

#--------------------------------------------------------------------------------
sub nodestatus_stop_logging{
    my ($class, $node_hash_ref) = @_;

    foreach my $node (keys %$node_hash_ref){
        my $adapter_hostname;
        if($$node_hash_ref{$node}{"InstallAdapterHostname"})
        {
            $adapter_hostname=$$node_hash_ref{$node}{"InstallAdapterHostname"};
        }
        my $port = $$node_hash_ref{$node}{'InstallMsgPort'};
        my $server   = $$node_hash_ref{$node}{'InstallMsgServer'};

        my ($hostname,$ip);
        if($adapter_hostname)
        {
            $ip=ServerUtils->dshGetNodeAddr($adapter_hostname,$server);
        }
        else
        {
            ($hostname,$ip)=NetworkUtils->getHost($node);
        }
        my $sock = new IO::Socket::INET ( PeerAddr => $server ,
                                          PeerPort => $port ,
                                          Proto    => 'tcp'
                                        );
        die "Socket could not be created.Reason:$!\n" unless $sock;
        syswrite($sock,"$node CSM_start_session\n");
        my $read;
        my $bytes_to_read=6;
        sysread($sock,$read,$bytes_to_read);
        chomp $read;
        if ( $read eq "ready" )
        {
            syswrite($sock, "$node CSM_stop_logging $ip\n");
        }
        $bytes_to_read=5;
        sysread($sock,$read,$bytes_to_read);
        chomp $read;
        close ($sock);
    }
}

#--------------------------------------------------------------------------------

=head3   set_default_svclevel

      Check the service level of each node. Make sure every node has InstallServiceLevel set properly.

      Arguments:        %node_hash
      Returns: reference to an array containing all the different service levels.
=cut

#--------------------------------------------------------------------------------
sub set_default_svclevel
{
    my ($class, $distro_name, $distro_ver, $ref_node_hash) = @_;
    my %node_hash = %$ref_node_hash;
    my @svclevels;
    my $svclevel;

    my $is_sl;
    my $ms_distro_name = NodeUtils->get_DistributionName();
    my $ms_distro_ver  = NodeUtils->get_DistributionVersion();

    foreach my $node (keys %node_hash)
    {
        $svclevel = $node_hash{$node}{InstallServiceLevel};
        if (!$svclevel)
        {

            # The previous call of InstallUtils->get_common_attrs() has made sure each node
            # has the same $::DISTRO_NAME and $::DISTRO_VERSION.
            if (   ($distro_name eq $ms_distro_name)
                && ($distro_ver eq $ms_distro_ver))
            {

                # When the node's InstallServiceLevel is not set,
                # use Management Server's service level to set it.
                $svclevel = NodeUtils->get_ServiceLevel;
            }
            else
            {

                # If MS and node have different distribution/version,
                # the node's default Service Level should be "GA".
                $svclevel = "GA";
            }
            $node_hash{$node}{InstallServiceLevel} = $svclevel;
        }
        if (!grep($svclevel, @svclevels))
        {
            push @svclevels, $svclevel;
        }
    }
    return \@svclevels;
}

#--------------------------------------------------------------------------------

=head3   do_copycds

        copy the contents of either the distribution CD-ROMs
        or the contents of PKGPATH to /csminstall

      Arguments:        %node_hash
      Returns: reference to an array containing all the different service levels.
=cut

#--------------------------------------------------------------------------------
sub do_copycds
{
    my ($class, $distro_name, $distro_ver, $arch, $csm_version,
        $ref_svclevel_list, $nomountcd, $extracds)
      = @_;
    my @service_level = @$ref_svclevel_list;
    my ($distro_top, $rc, $effective_distro_name);
    ($effective_distro_name) = NodeUtils->getEffectiveDistro($distro_name);

    if (($effective_distro_name =~ /RedHat/) || ($effective_distro_name =~ /RHEL/))
    {
        $distro_top = $::REDHAT_TOP;
    }
    else
    {
        $distro_top = $::SUSE_TOP;
    }

    my $pkgpath = join(':', @::PKGPATH);
    if ($pkgpath eq $::DEFAULT_PKGPATH)
    {
        $rc =
          ArchiveUtils->copyDistroCD("Linux", $arch, $distro_top, $distro_name,
                                     $distro_ver, $nomountcd, \@service_level, $extracds);
    }
    else
    {
        $rc =
          ArchiveUtils->copyDistroPKGPATH("Linux", $arch, $distro_top,
                                    $distro_name, $distro_ver, $nomountcd, \@service_level, $extracds);
    }

    if ($rc)
    {
        MessageUtils->message("E$rc", 'EMsgERROR_COPYING_RPMS', $distro_name);
    }
}

sub copy_iprdd_driver
{
    my ($class, $svclevel) = @_;
    my $subdir;

    if ($svclevel eq "QU1")
    {
        return if ($::QU1_DONE);
        $subdir = $::DISTRO_NAME . $::DISTRO_VERSION . "-" . $svclevel;
    }
    elsif ($svclevel eq "GA")
    {
        return if ($::GA_DONE);
        $subdir = $::DISTRO_NAME . $::DISTRO_VERSION . "-ppc64";
    }
    else
    {
        return;
    }

    $::GREP = "/bin/grep";
    my $ipr_dir =
      "/csminstall/Linux/$::DISTRO_NAME/$::DISTRO_VERSION/$::ARCH/$subdir/updates/drivers";
    unless (-d $ipr_dir)
    {
        mkpath("$ipr_dir", $::VERBOSE, 0755);
    }
    my $rpms_dir =
      "/csminstall/Linux/$::DISTRO_NAME/$::DISTRO_VERSION/$::ARCH/RedHatEL-AS3-extra/";
    my $cmd = "$::FIND $rpms_dir -name \"iprdd.img\"";
    my $res = `$cmd`;
    chomp $res;
    if ($res ne "")
    {
        $cmd = "$::COPY -f $res $ipr_dir";
        my $output = NodeUtils->runcmd($cmd, 0);
        if ($::RUNCMD_RC)
        {
            MessageUtils->message('E', 'EMsgNO_CopyFile', $res, $ipr_dir);
        }
        if ($svclevel eq "QU1")
        {
            $::QU1_DONE = 1;
        }
        else
        {
            $::GA_DONE = 1;
        }
    }
    else
    {
        MessageUtils->message('E', 'EMsgCANT_FIND', "iprdd.img", $rpms_dir);
    }

}

#####################################################################
#
# addCronJob
#
#   add one job into cron tab
#
#   Arguments:
#       $job    one scalar, job name
#
#   Returns:
#       $::OK
#       $::NOK
#
#   Error:
#       $::NOK
#
#   Example:
#       if (ServerUtils->addCronJob("cfmupdatenode")){
#           next;
#       }
#
#   Comments:
#       If the required job has been added,
#       this routine will return $::OK without doing anything.
#       Until now, this routine only supports two jobs,
#       cfmupdatenode and cleanup.logs.csp.
#
#####################################################################
sub addCronJob
{
    my ($class, $job) = @_;

    # set variables
    my $cronRoot;
    if (NodeUtils->isLinux())
    {
        my $distrib = NodeUtils->get_DistributionName();
        if ($distrib eq "SLES")
        {
            $cronRoot = "/var/spool/cron/tabs/root";
        }else
        {
            $cronRoot = "/var/spool/cron/root";
        }            
    }elsif (NodeUtils->isAIX())
    {
        $cronRoot = "/var/spool/cron/crontabs/root";
    }else
    {
        # unknown OS
        return $::NOK;
    }

    my ($cronCmd, $errorLog);
    if ($job eq "cfmupdatenode")
    {
        $cronCmd = "/opt/csm/bin/cfmupdatenode -a";
        $errorLog = "/var/log/csm/cfmerror.log";
    }elsif ($job eq "cleanup.logs.csp" && NodeUtils->isAIX())
    {
        $cronCmd = "/opt/csm/csmbin/cleanup.logs.csp";
        $errorLog = "/var/log/csm/csperror.log";
    }elsif ($job eq "cleanup.logs.csp" && NodeUtils->isLinux())
    {
        # this job does not exist on Linux
        return $::OK;
    }else
    {
        # unknown job
        return $::NOK;
    }

    # check whether the required job has been added
    if (-e $cronRoot)
    {
        NodeUtils->runcmd("/usr/bin/crontab -l | $::GREP $job", -1);
        if (!$::RUNCMD_RC)
        {
            # the entry has been there
            return $::OK;
        }
    }

    # add required job
    my $error = 0;
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
        'csminstall', 'I', 'IMsgAddingCronJob', $job);
    if (NodeUtils->isLinux())
    {
        my $sedCmd = "
\$ a \\
0 0 * * * $cronCmd 1>/dev/null 2>/dev/null
1,3d";
        if (-e $cronRoot)
        {
            NodeUtils->runcmd("/usr/bin/crontab -l | /bin/sed \"$sedCmd\" - | /usr/bin/crontab -", 0);
        }else
        {
            NodeUtils->runcmd("/bin/echo \" 0 0 * * * $cronCmd 1>/dev/null 2>/dev/null\" | /usr/bin/crontab -", 0);
        }
        if ($::RUNCMD_RC)
        {
            $error = 1;
        }
    }else
    {
        # on AIX
        my $tmpEntry = "/tmp/CSM_cron_entry";
        if (!-e $tmpEntry)
        {
            if (-e $cronRoot)
            {
                NodeUtils->runcmd("/usr/bin/crontab -l > $tmpEntry", 0);
            }
            NodeUtils->runcmd("echo \"0 0 * * * $cronCmd 1>>$errorLog 2>>$errorLog\" >> $tmpEntry", 0);
            NodeUtils->runcmd("/usr/bin/crontab $tmpEntry", 0);
            if ($::RUNCMD_RC)
            {
                $error = 1;
            }
            unlink $tmpEntry;
        }else
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                'csminstall', 'W', 'IMsgTmpCronEntryExist', $job, $tmpEntry);
        }
    }

    if ($error)
    {
        return $::NOK;
    }
    return $::OK;
}
#--------------------------------------------------------------------------------

=head3    set_cluster_info 
            Produce cluster info file for DC API.
            
        Arguments:
            $info_hash
            Reference to a hash which contains command specific attributes
        Returns:
        
        Notes:
            If this file is not really used, ensure to remove it now.
            For example
                my $file = ServerUtils->set_cluster_info(\%clusterInfoHash);
                unlink $file;
=cut

#--------------------------------------------------------------------------------
sub set_cluster_info
{
    my ($pkg, $info_hash) = @_;
    srand(time | $$);    #random number generator start

    my $file = "/tmp/cluster_info_$$";
    while (-e $file)
    {
        $file = ServerUtils->CreateRandomName($file);
    }

    open(INFO, ">$file")
      or MessageUtils->messageFromCat(
                                      'csmInstall.cat',      $::MSGMAPPATH,
                                      'csminstall',          "E",
                                      'EMsgCANT_WRITE_FILE', "$file"
                                     );

    foreach my $attr (keys %$info_hash)
    {
        print INFO "$attr=$$info_hash{$attr}\n";
    }
#    my ($NetworkInstallProtocol, $TFTPPackage, $SetupNetworkInstallProtocol,
#       $RemoteShell) = ServerUtils->get_cluster_attr;

    my $NetworkInstallProtocol = NodesetUtils->get_NetworkInstallProtocol;
    print INFO "NetworkInstallProtocol=$NetworkInstallProtocol\n";
    if(!$::TFTPPACKAGE)
    {
       ServerUtils->get_cluster_attr;
    }
    my $TFTPPackage = $::TFTPPACKAGE;
    chomp $TFTPPackage;
    print INFO "TFTPPackage=$TFTPPackage\n";

    my $SetupNetworkInstallProtocol =
      NodesetUtils->get_SetupNetworkInstallProtocol;
    print INFO "SetupNetworkInstallProtocol=$SetupNetworkInstallProtocol\n";

    my $RemoteShell=NetworkUtils->getRemoteShell;
    print INFO "RemoteShell=$RemoteShell\n";
        
    if($::SETUP_REMOTE_SHELL)
    {
        print INFO "SetupRemoteShell=$::SETUP_REMOTE_SHELL\n";
    }

    if($::REMOTECOPYCMD)
    {
        print INFO "RemoteCopyCmd=$::REMOTECOPYCMD\n";
    }
    close(INFO);
    return $file;
}
#--------------------------------------------------------------------------------

=head3    set_cluster_info_hash 

        Arguments:
            $info_hash
            Reference to a hash which contains command specific attributes
        Returns:
        
        Notes:
            Produce cluster info hash for DC API.

=cut

#--------------------------------------------------------------------------------
sub set_cluster_info_hash
{
    my ($pkg, $info_hash) = @_;


    my $NetworkInstallProtocol = NodesetUtils->get_NetworkInstallProtocol;
    $$info_hash{"NetworkInstallProtocol"}=$NetworkInstallProtocol;

    if(!$::TFTPPACKAGE)
    {
    #my $TFTPPackage =
    #  `CT_MANAGEMENT_SCOPE=1 lsrsrc-api -i -s IBM.DmsCtrl::::TFTPpackage 2>/dev/null `;
    #chomp $TFTPPackage;
        ServerUtils->get_cluster_attr;
    }
    $$info_hash{"TFTPPackage"}=$::TFTPPACKAGE;

    my $SetupNetworkInstallProtocol =
      NodesetUtils->get_SetupNetworkInstallProtocol;
    $$info_hash{"SetupNetworkInstallProtocol"}=$SetupNetworkInstallProtocol;

    my $RemoteShell=NetworkUtils->getRemoteShell;
    $$info_hash{"RemoteShell"}=$RemoteShell;

    if($::SETUP_REMOTE_SHELL)
    {
        $$info_hash{"SetupRemoteShell"}=$::SETUP_REMOTE_SHELL;
    }

    if($::REMOTECOPYCMD)
    {
        $$info_hash{"RemoteCopyCmd"}=$::REMOTECOPYCMD;
    }
    #testprint %{$info_hash};
    #print "\n";
}
#--------------------------------------------------------------------------------

=head3    get_cluster_attr

        Get the value of the common clster info from IBM.DmsCtrl.
        The first time the DmsCtrl attribute is obtained, it is cached, so
        that the next time function is called, the cached
        copy is returned. 
        The function runs lsrsrc-api command once for several common attributes,
        it is more effective rather than call ecah subroutine for lsrsrc-api once.  
        Arguments:
            none
        Globals:
                sets $::REMOTE_SHELL,$::SETUP_REMOTE_SHELL,$::REMOTECOPYCMD
              $::__SETUP_NETWORK_INSTALL_PROTOCOL,$::__NETWORK_INSTALL_PROTOCOL
              $::TFTPPACKAGE
        Returns:
        Notes:
            Produce cluster info for NodesetUtils->get_SetupNetworkInstallProtocol,
            NodesetUtils->get_NetworkInstallProtocol and NetworkUtils->getRemoteShell.
        Comments:
            None
=cut

#--------------------------------------------------------------------------------

sub get_cluster_attr
{
    #Generate lsrsrc-api command string
    my $DmsCtrlAttrString= "::SetupNetworkInstallProtocol"
                          ."::NetworkInstallProtocol"
                          ."::RemoteShell::SetupRemoteShell::RemoteCopyCmd"
                          ."::TFTPpackage";
    my @output = split "::",NodeUtils->runcmd(
            "CT_MANAGEMENT_SCOPE=1 /usr/sbin/rsct/bin/lsrsrc-api -i -s IBM.DmsCtrl::".$DmsCtrlAttrString
            );
    ($::__SETUP_NETWORK_INSTALL_PROTOCOL, $::__NETWORK_INSTALL_PROTOCOL,
     $::REMOTE_SHELL, $::SETUP_REMOTE_SHELL, $::REMOTECOPYCMD,
     $::TFTPPACKAGE) = @output;
    #test print @output;
    #print "\n";
    chomp $::TFTPPackage;
    return ;
} 
#-------------------------------------------------------------------------------

=head3    make_nodeinfo

        Create nodeinfo files for each node provided
        in the nodelist reference argument.

        Arguments:
                 $ref_DestinationNodeHash
        Returns:
                $ref_nodeinfo_list
        Globals:
                $::SETUP_REMOTE_SHELL
                $::NOSMS
        Error:
                $::NOK
        Example:
                if(ServerUtils->make_nodeinfo(\%DestNodeHash) != 0) {
                    blah;
                }
        Comments:
                Requires $::REMOTE_SHELL and $::SETUP_REMOTE_SHELL to be set optionally,
                $::NOSMS can be set to instruct makenode to not perform software maintenance
                on Linux nodes

=cut

#--------------------------------------------------------------------------------

sub make_nodeinfo
{
    my ($class, $ref_DestNodeHash) = @_;

    #  DestNodeHash is a hash containing attribute values for each node
    #    that was provided on the command line.
    my %DestNodeHash = %$ref_DestNodeHash;
    my @nl           = (keys %DestNodeHash);

    my ($hname, $short, $rc);
    my $csm_nodeinfo_dir = "/tmp/csm_$$";
    while (-e $csm_nodeinfo_dir)
    {
        $csm_nodeinfo_dir = ServerUtils->CreateRandomName($csm_nodeinfo_dir);
    }

    my $cmd;
    my $output;
    my ($host, $ipaddr, $shorthost);
    my @copystanzafiles;    # list of stanza files to copy
    my $instsvr;
    my @nodeinfo_list;

    #
    #  Check and/or create the /csminstall/csm/config directory
    #
    if (!-d $csm_nodeinfo_dir)
    {
        $cmd = "$::MKDIR -m 755 -p $csm_nodeinfo_dir";
        $rc  = system("$cmd");
        if ($rc >> 8)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                      'csminstall', "E", 'EMsgNO_CreateDir', $csm_nodeinfo_dir);
            return ($::NOK);
        }
    }

    # get the lists of user-provided cstomization script
    #   that we need to run on the nodes
    if (ServerUtils->make_script_lists(@nl))
    {

        # msg  -Could not get lists of customization scripts
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', "E", 'EMsgNoCustScripts');

        # continue
    }

    # get the lists of user-provided configuration methods
    #   that we need to run on the nodes - continue if errors
    ServerUtils->make_method_lists(@nl);

    my $cmd = "/usr/bin/lscondresp";
    my $outref = NodeUtils->runcmd($cmd, -1, 1);
    if ($::RUNCMD_RC == 0)
    {
        foreach my $line (@$outref)
        {
            if ($line =~ m/NodeFullInstallComplete.*SetupSSHAndRunCFM.*Active/)
            {
                foreach my $node (keys %DestNodeHash)
                {
                    $DestNodeHash{$node}{'CFM'} = 1;
                }
            }
        }
    }

    if(!defined($::SETUP_REMOTE_SHELL))
    {
        #The $::SETUP_REMOTE_SHELL variable will be set in this subroutine
        NetworkUtils->getRemoteShell;
    }
    #
    # create the config_info file
    #

    #  go through each node in the list
    #   (the keys are the hostnames as returned from tryHost() !!)
    foreach $hname (keys %DestNodeHash)
    {
        my $nodeinfo_file = "$csm_nodeinfo_dir/$hname.nodeinfo";

        # remove the old config file if any
        if (-e $nodeinfo_file)
        {
            unlink($nodeinfo_file);
        }

        # create the new file
        unless (open(CONFIG_OUT, ">$nodeinfo_file"))
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                    'csminstall', "E", 'EMsgCANT_WRITE_FILE', "$nodeinfo_file");
            return $::NOK;
        }
        push @nodeinfo_list, $nodeinfo_file;

        if (defined($::SETUP_REMOTE_SHELL))
        {
            print CONFIG_OUT "SetupRemoteShell=$::SETUP_REMOTE_SHELL\n";
        }
        # if make_config_file was called by csmsetup* cmds or updatenode -c
        #      - then we need to add the CONFIGADAPTERS attr
        my $csmsetupcmds = 0;
        my $progname     = NodeUtils->programName();

        if (   $progname =~ m/csmsetupks/
            || $progname =~ m/csmsetupnim/
            || $progname =~ m/csmsetupyast/
            || $progname =~ m/installnode/)
        {
            $csmsetupcmds = 1;
        }

        if (defined($::UPDATENODE_C)
            && !defined($DestNodeHash{$hname}{'AdapterStanzaFile'}))
        {

            # give a warning - they want to configure adapters but there is no
            #  stanza file for this node.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'W', 'EMsgNoStanzaFile', $hname);
        }

        if (defined($::UPDATENODE_C) || $csmsetupcmds)
        {

            # If the node's AdapterStanzaFile attribute is set then add the attr
            if (defined($DestNodeHash{$hname}{'AdapterStanzaFile'}) && $DestNodeHash{$hname}{'AdapterStanzaFile'} )
            {

                # add the filename to the list of files to copy and do it
                # at the end of the loop below to avoid multiple copies
                #  (- do this now to make sure we have the latest version)
                my $sfile = $DestNodeHash{$hname}{'AdapterStanzaFile'};
                if (!grep (/^$sfile$/, @copystanzafiles))
                {
                    push @copystanzafiles,
                      $DestNodeHash{$hname}{'AdapterStanzaFile'};
                }

				#  use the full pathname in the config_info file
				my $stanzafilename = $DestNodeHash{$hname}{'AdapterStanzaFile'};

                # write it to the file
                print CONFIG_OUT "CONFIGADAPTERS=$stanzafilename\n";

                # also add list of user methods if any
                if (defined(@{$::CONFIGMETHODS{$hname}}))
                {
                    print CONFIG_OUT
                      "CONFIGMETHODS=@{$::CONFIGMETHODS{$hname}}\n";
                }
            }
        }

        # add the node configuration information to the file
        if (defined($hname))
        {
            print CONFIG_OUT "Hostname=$hname\n";
        }
        if (defined($DestNodeHash{$hname}{'ManagementServer'}))
        {
            print CONFIG_OUT
              "ManagementServer=$DestNodeHash{$hname}{'ManagementServer'}\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallOSName'}))
        {
            print CONFIG_OUT
              "InstallOSName=$DestNodeHash{$hname}{'InstallOSName'}\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallCSMVersion'}))
        {
            print CONFIG_OUT
              "InstallCSMVersion=$DestNodeHash{$hname}{'InstallCSMVersion'}\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallDistributionVersion'}))
        {
            print CONFIG_OUT
              "InstallDistributionVersion=$DestNodeHash{$hname}{'InstallDistributionVersion'}\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallDistributionName'}))
        {
            print CONFIG_OUT
              "InstallDistributionName=$DestNodeHash{$hname}{'InstallDistributionName'}\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallServer'}))
        {
            my ($server, $rep_dir) = split ':',
              $DestNodeHash{$hname}{'InstallServer'};
            
            my $ref_grp = NodeUtils->getNodegrp($server);
            if ($ref_grp)
            {
                my $server_str;
                if($rep_dir)
                {
                    my @servers_with_dir=map { "$_:$rep_dir" } @$ref_grp;
                    $server_str=join(',',@servers_with_dir);
                }
                else
                {
                    $server_str=join(',',@$ref_grp);
                }    
                print CONFIG_OUT
                "InstallServer=$server_str\n";
            }
            else
            {  
                print CONFIG_OUT
                "InstallServer=$DestNodeHash{$hname}{'InstallServer'}\n";
            }
        }
        if (defined($DestNodeHash{$hname}{'NFSServer'}))
        {
            print CONFIG_OUT "NFSServer=$DestNodeHash{$hname}{'NFSServer'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallPkgArchitecture'}))
        {
            print CONFIG_OUT
              "InstallPkgArchitecture=$DestNodeHash{$hname}{'InstallPkgArchitecture'}\n";
        }
        if (defined($DestNodeHash{$hname}{'Mode'}))
        {
            print CONFIG_OUT "Mode=$DestNodeHash{$hname}{'Mode'}\n";
        }
        if (defined($DestNodeHash{$hname}{'ConsoleSerialDevice'}))
        {
            print CONFIG_OUT
              "ConsoleSerialDevice=$DestNodeHash{$hname}{'ConsoleSerialDevice'}\n";
        }
        if (defined($DestNodeHash{$hname}{'PowerMethod'}))
        {
            print CONFIG_OUT
              "PowerMethod=$DestNodeHash{$hname}{'PowerMethod'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallAdapterMacaddr'}))
        {
            print CONFIG_OUT
              "InstallAdapterMacaddr=$DestNodeHash{$hname}{'InstallAdapterMacaddr'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallAdapterName'}))
        {
            print CONFIG_OUT
              "InstallAdapterName=$DestNodeHash{$hname}{'InstallAdapterName'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallAdapterDuplex'}))
        {
            print CONFIG_OUT
              "InstallAdapterDuplex=$DestNodeHash{$hname}{'InstallAdapterDuplex'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallAdapterGateway'}))
        {
            print CONFIG_OUT
              "InstallAdapterGateway=$DestNodeHash{$hname}{'InstallAdapterGateway'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallAdapterNetmask'}))
        {
            print CONFIG_OUT
              "InstallAdapterNetmask=$DestNodeHash{$hname}{'InstallAdapterNetmask'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallAdapterSpeed'}))
        {
            print CONFIG_OUT
              "InstallAdapterSpeed=$DestNodeHash{$hname}{'InstallAdapterSpeed'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallAdapterType'}))
        {
            print CONFIG_OUT
              "InstallAdapterType=$DestNodeHash{$hname}{'InstallAdapterType'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallDiskType'}))
        {
            print CONFIG_OUT
              "InstallDiskType=$DestNodeHash{$hname}{'InstallDiskType'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallDisk'}))
        {
            print CONFIG_OUT
              "InstallDisk=$DestNodeHash{$hname}{'InstallDisk'}\n";
        }

        # Handle Linux-specific attributes.
        if ($DestNodeHash{$hname}{'InstallOSName'} eq "Linux")
        {

            # if $::NOSMS isn't set and if the node is not diskless node,
            # then perform software maintenance
            if (   (!$::NOSMS)
                && ($DestNodeHash{$hname}{'InstallMethod'} ne "diskless"))
            {
                print CONFIG_OUT "SMS=1\n";
            }
        }

        if (defined($DestNodeHash{$hname}{'ConsoleSerialSpeed'}))
        {
            print CONFIG_OUT
              "ConsoleSerialSpeed=$DestNodeHash{$hname}{'ConsoleSerialSpeed'}\n";
        }

        if (defined($DestNodeHash{$hname}{'InstallMethod'}))
        {
            print CONFIG_OUT
              "InstallMethod=$DestNodeHash{$hname}{'InstallMethod'}\n";
        }

        # put in the SetupKRB5 attribute value
        if (!defined $::KRB5SETUP)
        {

            $::KRB5SETUP = NodeUtils->getSetupKRB5;
        }
        print CONFIG_OUT "SetupKRB5=$::KRB5SETUP\n";

        # put in what would be the KRB5 principal name for ManagementServer
        if (!defined $::MSNAME)
        {
            $::MSNAME = NetworkUtils->getRSCThostname;
        }
        my $msnamelower = lc($::MSNAME);
        print CONFIG_OUT "CSMKRB5Principal=ctsrcmdp.$msnamelower";

        # put in what would be the KRB5 principal name for HA ManagementServer
        if (!defined $::HAMODE)
        {
            my $outref =
              NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                   "-s IBM.DmsCtrl::::HAMode");
            $::HAMODE = $$outref[0];
        }
        if ($::HAMODE == 1)
        {    #on an active HAMS management server
            require HAMS;
            my $prom = NodeUtils->programName();
            if ($prom =~ m/csmsetupks/)
            {
                delete $ENV{'POSIXLY_CORRECT'};
            }
            my $inactivems      = HAMS->getInactiveRSCTHostname(); # get HA name
            my $inactivemslower = lc($inactivems);
            print CONFIG_OUT ",ctsrcmdp.$inactivemslower\n";

        }
        else
        {
            print CONFIG_OUT "\n";
        }

        if (($::SETUP_REMOTE_SHELL == 1) && ($::REMOTE_SHELL =~ m/rsh/))
        {
            if (!defined $::HAMODE)
            {
                my $outref =
                  NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                       "-s IBM.DmsCtrl::::HAMode");
                $::HAMODE = $$outref[0];
            }
            if ($::HAMODE == 1)
            {    #on an active HAMS management server
                    #get management server attribute for node on other ms
                require HAMS;
                my $other_ms = HAMS->getManagementServerAttr($hname);
                if ($other_ms)
                {
                    print CONFIG_OUT "InActiveMSAttr=$other_ms\n";
                }
            }
        }

        #
        # add the /etc/hosts info for the management server - just AIX for now
        #
        if ($DestNodeHash{$hname}{'InstallOSName'} eq "AIX")
        {
            ($host, $ipaddr) =
              NetworkUtils->getHost($DestNodeHash{$hname}{'ManagementServer'});
            if ($ipaddr && $ipaddr ne $host)
            {
                ($shorthost = $host) =~ s/\..*$//;
                print CONFIG_OUT "ETCHOSTSMS=$ipaddr $host $shorthost\n";
            }

        }

        print CONFIG_OUT "ManagementProduct=CSM\n";


        if (defined($DestNodeHash{$hname}{'InstallServiceLevel'}))
        {
            print CONFIG_OUT
              "InstallServiceLevel=$DestNodeHash{$hname}{'InstallServiceLevel'}\n";
        }

        if (defined($DestNodeHash{$hname}{'UUID'}))
        {
            print CONFIG_OUT "UUID=$DestNodeHash{$hname}{'UUID'}\n";
        }

        if (defined($DestNodeHash{$hname}{'HWType'}))
        {
            print CONFIG_OUT "HWType=$DestNodeHash{$hname}{'HWType'}\n";
        }

        if (defined($DestNodeHash{$hname}{'IsBlade'}))
        {
            print CONFIG_OUT "IsBlade=$DestNodeHash{$hname}{'IsBlade'}\n";
        }
        if (defined($DestNodeHash{$hname}{'InstallMsgPort'}))
        {
            print CONFIG_OUT
              "InstallMsgPort=$DestNodeHash{$hname}{'InstallMsgPort'}\n";
        }
        if (defined($DestNodeHash{$hname}{'CFM'}))
        {
            print CONFIG_OUT "CFM=1\n";
        }

        # add lists of scripts to run on the node
        if (defined(@{$::PREINSTSCRIPTS{$hname}}))
        {
            print CONFIG_OUT "PREINSTSCRIPTS=@{$::PREINSTSCRIPTS{$hname}}\n";
        }
        if (defined(@{$::POSTINSTSCRIPTS{$hname}}))
        {
            print CONFIG_OUT "POSTINSTSCRIPTS=@{$::POSTINSTSCRIPTS{$hname}}\n";
        }
        if (defined(@{$::PREOSUPGRADESCRIPTS{$hname}}))
        {
            print CONFIG_OUT "PREOSUPGRADESCRIPTS=@{$::PREOSUPGRADESCRIPTS{$hname}}\n";
        }
        if (defined(@{$::POSTOSUPGRADESCRIPTS{$hname}}))
        {
            print CONFIG_OUT "POSTOSUPGRADESCRIPTS=@{$::POSTOSUPGRADESCRIPTS{$hname}}\n";
        }
        if (defined(@{$::UPDATESCRIPTS{$hname}}))
        {
            print CONFIG_OUT "UPDATESCRIPTS=@{$::UPDATESCRIPTS{$hname}}\n";
        }
        if (defined(@{$::DISKLESSPREBUILDSCRIPTS{$hname}}))
        {
            print CONFIG_OUT
              "DISKLESSPREBUILDSCRIPTS=@{$::DISKLESSPREBUILDSCRIPTS{$hname}}\n";
        }
        if (defined(@{$::DISKLESSBOOTSCRIPTS{$hname}}))
        {
            print CONFIG_OUT
              "DISKLESSBOOTSCRIPTS=@{$::DISKLESSBOOTSCRIPTS{$hname}}\n";
        }

        # New added, but not sure whether need them all.
        # OS_DIST_TOP,  OS_DIST_RPMS, OPEN_SRC_RPM_DIR, DRVR_TOP_DIR, PATCH_FILE_DIR
        # PREREBOOT_SCRIPT_DIR , FIRSTBOOT_SCRIPT_DIR, UPDATE_SCRIPT_DIR, DATA_SCRIPT_DIR
        # DSERVER_DIR, STATUS_FILE_DIR, CSM_PACKAGES_DIR, SMS_INSTALL_DIR, SMS_UPDATES_DIR
        my $CSMCLIENTMNTDIR = "/var/opt/csm/mnt";
        my $os_dist_top     =
          "$CSMCLIENTMNTDIR/$DestNodeHash{$hname}{'InstallOSName'}/$DestNodeHash{$hname}{'InstallDistributionName'}/$DestNodeHash{$hname}{'InstallDistributionVersion'}/$DestNodeHash{$hname}{'InstallPkgArchitecture'}";
        my $csm_pkg_dir =
          "$CSMCLIENTMNTDIR/$DestNodeHash{$hname}{'InstallOSName'}/$DestNodeHash{$hname}{'InstallDistributionName'}/$DestNodeHash{$hname}{'InstallDistributionVersion'}/csm/$DestNodeHash{$hname}{'InstallCSMVersion'}/packages";

        print CONFIG_OUT "OS_DIST_TOP=$os_dist_top\n";

        print CONFIG_OUT "OS_DIST_RPMS=$os_dist_top/RPMS\n";
        print CONFIG_OUT "OPEN_SRC_RPM_DIR=$csm_pkg_dir\n";
        print CONFIG_OUT "DRVR_TOP_DIR=$os_dist_top/drivers\n";
        print CONFIG_OUT "PATCH_FILE_DIR=$os_dist_top/patches\n";

		print CONFIG_OUT
		  "PREREBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/installprereboot\n";
		print CONFIG_OUT
		  "FIRSTBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/installpostreboot\n";
		print CONFIG_OUT
		  "OSUPGRADE_PREREBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/osupgradeprereboot\n";
		print CONFIG_OUT
		  "OSUPGRADE_FIRSTBOOT_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/osupgradepostreboot\n";

        print CONFIG_OUT
          "UPDATE_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/update\n";
        print CONFIG_OUT "DATA_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm/scripts/data\n";

        print CONFIG_OUT "OS_SCRIPT_DIR=$CSMCLIENTMNTDIR/csm\n";    #????
        print CONFIG_OUT "STATUS_FILE_DIR=$CSMCLIENTMNTDIR/csm/status\n";

        print CONFIG_OUT "CSM_PACKAGES_DIR=$csm_pkg_dir\n";
        print CONFIG_OUT "SMS_INSTALL_DIR=$os_dist_top/install\n";
        print CONFIG_OUT "SMS_UPDATES_DIR=$os_dist_top/updates\n";

        if (defined $ENV{'CSM_NOSTATUS'} && $ENV{'CSM_NOSTATUS'} != 0)
        {
            print CONFIG_OUT "CSM_NOSTATUS=1\n";
        }

        close(CONFIG_OUT);

        chmod 0644, "$nodeinfo_file";

        #
        # Add symbolic links
        #   Hopefully the scripts that run on the
        #   node will recognize one of these names!!
        #
        my $short_hname = NodeUtils->getShorthost($hname);
        my $node_ip     = NetworkUtils->validate_ip($hname);

        foreach my $linkName ($short_hname, $node_ip)
        {
            if ($linkName ne $hname)
            {
                $cmd =
                  "cd $csm_nodeinfo_dir; $::LN -fs $hname.nodeinfo $linkName.nodeinfo";
                $rc = system("$cmd");
                if ($rc >> 8)
                {
                    MessageUtils->messageFromCat('csmInstall.cat',
                          $::MSGMAPPATH, 'csminstall', 'E', 'EMsgCMD_FAILED_RC',
                          $::LN, $rc);
                }
            }
        }
    }    # go do the next node


    #  copy the adapter stanza files to /csminstall/csm/config/adapters
    #  do this now just in case the file has been changed recently
    if (@copystanzafiles)
    {

		my $tmpstanzadir = "/csminstall/csm/config/adapters";

		foreach my $file (@copystanzafiles)
		{
			my $scriptdir = dirname($file);
			my $targetdir = $tmpstanzadir . $scriptdir;

			if (!-d $targetdir)
			{
				mkpath($targetdir, $::VERBOSE, 0755);
			}

			my $cmd = "$::COPY -p $file $targetdir";
			NodeUtils->runcmd($cmd, 0);
		}

    }

    return \@nodeinfo_list;
}

#-------------------------------------------------------------------------------

=head3    make_nodeinfo_hash

        Create nodeinfo hash for each node provided
        in the nodelist reference argument.

        Arguments:
                 $ref_DestinationNodeHash
                 $skip_genarate_scripts_list
        Returns:
                $ref_nodeinfo_hash
        Globals:
                $::SETUP_REMOTE_SHELL
                $::NOSMS
        Error:
                $::NOK
        Example:
                if(ServerUtils->make_nodeinfo(\%DestNodeHash,$skip_genarate_scripts_list) != 0) {
                    blah;
                }
        Comments:
                Requires $::REMOTE_SHELL and $::SETUP_REMOTE_SHELL to be set optionally,
                $::NOSMS can be set to instruct makenode to not perform software maintenance
                on Linux nodes

=cut

#--------------------------------------------------------------------------------

sub make_nodeinfo_hash
{
    my ($class, $ref_DestNodeHash,$skip_genarate_scripts_list) = @_;

    #  DestNodeHash is a hash containing attribute values for each node
    #    that was provided on the command line.
    my %DestNodeHash;
   	foreach my $node (keys %$ref_DestNodeHash)
	{
		foreach my $attr (keys %{$$ref_DestNodeHash{$node}})
		{
			$DestNodeHash{$node}{$attr}=$$ref_DestNodeHash{$node}{$attr};
		}
	}		

    my @nl           = (keys %DestNodeHash);

    my ($hname, $short, $rc);

    my $cmd;
    my $output;
    my ($host, $ipaddr, $shorthost);
    my @copystanzafiles;    # list of stanza files to copy
    my $instsvr;
    my @nodeinfo_list;

    if(! $skip_genarate_scripts_list){
        # get the lists of user-provided cstomization script
        #   that we need to run on the nodes
        if (ServerUtils->make_script_lists(@nl))
        {

            # msg  -Could not get lists of customization scripts
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', "E", 'EMsgNoCustScripts');

            # continue
        }

        # get the lists of user-provided configuration methods
        #   that we need to run on the nodes - continue if errors
        ServerUtils->make_method_lists(@nl);
    }
    my $cmd = "/usr/bin/lscondresp";
    my $outref = NodeUtils->runcmd($cmd, -1, 1);
    if ($::RUNCMD_RC == 0)
    {
        foreach my $line (@$outref)
        {
            if ($line =~ m/NodeFullInstallComplete.*SetupSSHAndRunCFM.*Active/)
            {
                foreach my $node (keys %DestNodeHash)
                {
                    $DestNodeHash{$node}{'CFM'} = 1;
                }
            }
        }
    }

    if(!defined($::SETUP_REMOTE_SHELL))
    {
        #The $::SETUP_REMOTE_SHELL variable will be set in this subroutine
        NetworkUtils->getRemoteShell;
    }

    #  go through each node in the list
    #   (the keys are the hostnames as returned from tryHost() !!)
    foreach $hname (keys %DestNodeHash)
    {
        if (defined($::SETUP_REMOTE_SHELL))
        {
            $DestNodeHash{$hname}{"SetupRemoteShell"}=$::SETUP_REMOTE_SHELL;
        }
        # if make_config_file was called by csmsetup* cmds or updatenode -c
        #      - then we need to add the CONFIGADAPTERS attr
        my $csmsetupcmds = 0;
        my $progname     = NodeUtils->programName();

        if (   $progname =~ m/csmsetupks/
            || $progname =~ m/csmsetupnim/
            || $progname =~ m/csmsetupyast/
            || $progname =~ m/installnode/)
        {
            $csmsetupcmds = 1;
        }

        if (defined($::UPDATENODE_C)
            && !defined($DestNodeHash{$hname}{'AdapterStanzaFile'}))
        {

            # give a warning - they want to configure adapters but there is no
            #  stanza file for this node.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'W', 'EMsgNoStanzaFile', $hname);
        }

        if (defined($::UPDATENODE_C) || $csmsetupcmds)
        {

            # If the node's AdapterStanzaFile attribute is set then add the attr
            if (defined($DestNodeHash{$hname}{'AdapterStanzaFile'}) && $DestNodeHash{$hname}{'AdapterStanzaFile'} )
            {

                # add the filename to the list of files to copy and do it
                # at the end of the loop below to avoid multiple copies
                #  (- do this now to make sure we have the latest version)
                my $sfile = $DestNodeHash{$hname}{'AdapterStanzaFile'};
                if (!grep (/^$sfile$/, @copystanzafiles))
                {
                    push @copystanzafiles,
                      $DestNodeHash{$hname}{'AdapterStanzaFile'};
                }

				#  use the full pathname in the config_info file
				my $stanzafilename = $DestNodeHash{$hname}{'AdapterStanzaFile'};

                # write it to the file
                $DestNodeHash{$hname}{"CONFIGADAPTERS"}=$stanzafilename;

                # also add list of user methods if any
                if (defined(@{$::CONFIGMETHODS{$hname}}))
                {
                    $DestNodeHash{$hname}{"CONFIGMETHODS"}=join(' ',@{$::CONFIGMETHODS{$hname}});
                }
            }
        }

        if (defined($DestNodeHash{$hname}{'InstallServer'}))
        {
            my ($server, $rep_dir) = split ':',
              $DestNodeHash{$hname}{'InstallServer'};
            
            my $ref_grp = NodeUtils->getNodegrp($server);
            if ($ref_grp)
            {
                my $server_str;
                if($rep_dir)
                {
                    my @servers_with_dir=map { "$_:$rep_dir" } @$ref_grp;
                    $server_str=join(',',@servers_with_dir);
                }
                else
                {
                    $server_str=join(',',@$ref_grp);
                }    
                $DestNodeHash{$hname}{"InstallServer"}=$server_str;
            }
        }

        # Handle Linux-specific attributes.
        if ($DestNodeHash{$hname}{'InstallOSName'} eq "Linux")
        {

            # if $::NOSMS isn't set and if the node is not diskless node,
            # then perform software maintenance
            if (   (!$::NOSMS)
                && ($DestNodeHash{$hname}{'InstallMethod'} ne "diskless"))
            {
                $DestNodeHash{$hname}{"SMS"}=1;
            }
        }


        # put in the SetupKRB5 attribute value
        if (!defined $::KRB5SETUP)
        {

            $::KRB5SETUP = NodeUtils->getSetupKRB5;
        }
        $DestNodeHash{$hname}{"SetupKRB5"}=$::KRB5SETUP;

        # put in what would be the KRB5 principal name for ManagementServer
        if (!defined $::MSNAME)
        {
            $::MSNAME = NetworkUtils->getRSCThostname;
        }
        my $msnamelower = lc($::MSNAME);
        $DestNodeHash{$hname}{"CSMKRB5Principal"}="ctsrcmdp.$msnamelower";

        # put in what would be the KRB5 principal name for HA ManagementServer
        if (!defined $::HAMODE)
        {
            my $outref =
              NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                   "-s IBM.DmsCtrl::::HAMode");
            $::HAMODE = $$outref[0];
        }
        if ($::HAMODE == 1)
        {    #on an active HAMS management server
            require HAMS;
            my $prom = NodeUtils->programName();
            if ($prom =~ m/csmsetupks/)
            {
                delete $ENV{'POSIXLY_CORRECT'};
            }
            my $inactivems      = HAMS->getInactiveRSCTHostname(); # get HA name
            my $inactivemslower = lc($inactivems);
            $DestNodeHash{$hname}{"CSMKRB5Principal"}="ctsrcmdp.$msnamelower,ctsrcmdp.$inactivemslower";

        }

        if (($::SETUP_REMOTE_SHELL == 1) && ($::REMOTE_SHELL =~ m/rsh/))
        {
            if (!defined $::HAMODE)
            {
                my $outref =
                  NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                       "-s IBM.DmsCtrl::::HAMode");
                $::HAMODE = $$outref[0];
            }
            if ($::HAMODE == 1)
            {    #on an active HAMS management server
                    #get management server attribute for node on other ms
                require HAMS;
                my $other_ms = HAMS->getManagementServerAttr($hname);
                if ($other_ms)
                {
                    $DestNodeHash{$hname}{"InActiveMSAttr"}=$other_ms;
                }
            }
        }

        #
        # add the /etc/hosts info for the management server - just AIX for now
        #
        if ($DestNodeHash{$hname}{'InstallOSName'} eq "AIX")
        {
            ($host, $ipaddr) =
              NetworkUtils->getHost($DestNodeHash{$hname}{'ManagementServer'});
            if ($ipaddr && $ipaddr ne $host)
            {
                ($shorthost = $host) =~ s/\..*$//;
                $DestNodeHash{$hname}{"ETCHOSTSMS"}="$ipaddr $host $shorthost";
            }

        }

        $DestNodeHash{$hname}{"ManagementProduct"}="CSM";



        # add lists of scripts to run on the node
        if (defined(@{$::PREINSTSCRIPTS{$hname}}))
        {
            $DestNodeHash{$hname}{"PREINSTSCRIPTS"}=join(' ',@{$::PREINSTSCRIPTS{$hname}});
        }
        if (defined(@{$::POSTINSTSCRIPTS{$hname}}))
        {
            $DestNodeHash{$hname}{"POSTINSTSCRIPTS"}=join(' ',@{$::POSTINSTSCRIPTS{$hname}});
        }
        if (defined(@{$::PREOSUPGRADESCRIPTS{$hname}}))
        {
            $DestNodeHash{$hname}{"PREOSUPGRADESCRIPTS"}=join(' ',@{$::PREOSUPGRADESCRIPTS{$hname}});
        }
        if (defined(@{$::POSTOSUPGRADESCRIPTS{$hname}}))
        {
            $DestNodeHash{$hname}{"POSTOSUPGRADESCRIPTS"}=join(' ',@{$::POSTOSUPGRADESCRIPTS{$hname}});
        }
        if (defined(@{$::UPDATESCRIPTS{$hname}}))
        {
            $DestNodeHash{$hname}{"UPDATESCRIPTS"}=join(' ',@{$::UPDATESCRIPTS{$hname}});
        }
        if (defined(@{$::DISKLESSPREBUILDSCRIPTS{$hname}}))
        {
            $DestNodeHash{$hname}{"DISKLESSPREBUILDSCRIPTS"}=join(' ',@{$::DISKLESSPREBUILDSCRIPTS{$hname}});
        }
        if (defined(@{$::DISKLESSBOOTSCRIPTS{$hname}}))
        {
            $DestNodeHash{$hname}{"DISKLESSBOOTSCRIPTS"}=join(' ',@{$::DISKLESSBOOTSCRIPTS{$hname}});
        }

        # New added, but not sure whether need them all.
        # OS_DIST_TOP,  OS_DIST_RPMS, OPEN_SRC_RPM_DIR, DRVR_TOP_DIR, PATCH_FILE_DIR
        # PREREBOOT_SCRIPT_DIR , FIRSTBOOT_SCRIPT_DIR, UPDATE_SCRIPT_DIR, DATA_SCRIPT_DIR
        # DSERVER_DIR, STATUS_FILE_DIR, CSM_PACKAGES_DIR, SMS_INSTALL_DIR, SMS_UPDATES_DIR
        my $CSMCLIENTMNTDIR = "/var/opt/csm/mnt";
        my $os_dist_top     =
          "$CSMCLIENTMNTDIR/$DestNodeHash{$hname}{'InstallOSName'}/$DestNodeHash{$hname}{'InstallDistributionName'}/$DestNodeHash{$hname}{'InstallDistributionVersion'}/$DestNodeHash{$hname}{'InstallPkgArchitecture'}";
        my $csm_pkg_dir =
          "$CSMCLIENTMNTDIR/$DestNodeHash{$hname}{'InstallOSName'}/$DestNodeHash{$hname}{'InstallDistributionName'}/csm/$DestNodeHash{$hname}{'InstallCSMVersion'}/packages";

        $DestNodeHash{$hname}{"OS_DIST_TOP"}=$os_dist_top;

        $DestNodeHash{$hname}{"OS_DIST_RPMS"}="$os_dist_top/RPMS";
        $DestNodeHash{$hname}{"OPEN_SRC_RPM_DIR"}=$csm_pkg_dir;
        $DestNodeHash{$hname}{"DRVR_TOP_DIR"}="$os_dist_top/drivers";
        $DestNodeHash{$hname}{"PATCH_FILE_DIR"}="$os_dist_top/patches";

		$DestNodeHash{$hname}{"PREREBOOT_SCRIPT_DIR"}="$CSMCLIENTMNTDIR/csm/scripts/installprereboot";
		$DestNodeHash{$hname}{"FIRSTBOOT_SCRIPT_DIR"}="$CSMCLIENTMNTDIR/csm/scripts/installpostreboot";
		$DestNodeHash{$hname}{"OSUPGRADE_PREREBOOT_SCRIPT_DIR"}="$CSMCLIENTMNTDIR/csm/scripts/osupgradeprereboot";
		$DestNodeHash{$hname}{"OSUPGRADE_FIRSTBOOT_SCRIPT_DIR"}="$CSMCLIENTMNTDIR/csm/scripts/osupgradepostreboot";

        $DestNodeHash{$hname}{"UPDATE_SCRIPT_DIR"}="$CSMCLIENTMNTDIR/csm/scripts/update";
        $DestNodeHash{$hname}{"DATA_SCRIPT_DIR"}="$CSMCLIENTMNTDIR/csm/scripts/data";

        $DestNodeHash{$hname}{"OS_SCRIPT_DIR"}="$CSMCLIENTMNTDIR/csm";    #????
        $DestNodeHash{$hname}{"STATUS_FILE_DIR"}="$CSMCLIENTMNTDIR/csm/status";

        $DestNodeHash{$hname}{"CSM_PACKAGES_DIR"}=$csm_pkg_dir;
        $DestNodeHash{$hname}{"SMS_INSTALL_DIR"}="$os_dist_top/install";
        $DestNodeHash{$hname}{"SMS_UPDATES_DIR"}="$os_dist_top/updates";

        if (defined $ENV{'CSM_NOSTATUS'} && $ENV{'CSM_NOSTATUS'} != 0)
        {
            $DestNodeHash{$hname}{"CSM_NOSTATUS"}=1;
        }
        my $svclevel = $DestNodeHash{$hname}{'InstallServiceLevel'};
        # Consider all service level whose distro version contains '.' as default value 'GA'.
        if (  $DestNodeHash{$hname}{'InstallMethod'} eq "warewulf"
            && $DestNodeHash{$hname}{'InstallDistributionName'} =~ /RedHatEL/
            && $DestNodeHash{$hname}{'InstallDistributionVersion'} =~ /\./
            && $svclevel ne "GA"
        )
        {
            $DestNodeHash{$hname}{'InstallServiceLevel'} = "GA";
        }

    }    # go do the next node


    #  copy the adapter stanza files to /csminstall/csm/config/adapters
    #  do this now just in case the file has been changed recently
    if (@copystanzafiles)
    {

		my $tmpstanzadir = "/csminstall/csm/config/adapters";

		foreach my $file (@copystanzafiles)
		{
			my $scriptdir = dirname($file);
			my $targetdir = $tmpstanzadir . $scriptdir;

			if (!-d $targetdir)
			{
				mkpath($targetdir, $::VERBOSE, 0755);
			}

			my $cmd = "$::COPY -p $file $targetdir";
			NodeUtils->runcmd($cmd, 0);
		}

    }

    return \%DestNodeHash;
}
#-------------------------------------------------------------------------------

=head3   getCSMAttr 

        Reads the requested CSM Attribute from IBM.DMSCtrl
		and returns the value.
        
        Arguments: $::CSMAttrHash  
                 $Attribute_name 
        Returns:
                1 on failure
				value of attribute on success
        Example:
		%::CSMAttrHash; defined in caller ( see cfmupdatenode)
		my $attrvalue = ServerUtils->getCSMAttr("Attribute");
		my $CSMAdminID = ServerUtils->getCSMAttr("CSMAdminID");
        Comments:

=cut

#--------------------------------------------------------------------------------

sub  getCSMAttr
{
    my ($class, $attrname) = @_;
	if (exists($::CSMAttrHash{$attrname})) { # already hash'd it
	   return $::CSMAttrHash{$attrname};
    } else {  # get the value
	  # set local scope because we only want classes on the mgmt svr
	  if (NodeUtils->isMgmtSvr()) {  # only on Management Server
	    $ENV{'CT_MANAGEMENT_SCOPE'} = 1;
        $ENV{'CT_SESSION_SCOPE'} = 1;
	    my $outref =
		  NodeUtils->runrmccmd('lsrsrc-api', "-i",
			 "-s IBM.DmsCtrl::::$attrname");
        $::CSMAttrHash{$attrname} = $$outref[0]; 
	    return $::CSMAttrHash{$attrname};
	  } else {  # error
        return 1;
	  }
    }
}

#--------------------------------------------------------------------------------

=head2    VNFS utilities

=cut

#--------------------------------------------------------------------------------

#-------------------------------------------------------------------------------

=head3   setupVNFS 

	create, build and customize the VNFS repository for warewulf install

        Arguments:
		
		Global variable: $::WAREWULF_TMPL_DIR warewulf template directory
        
	Returns:
        	zero on success
		non-zero on failure
	Example:
        
	Comments:

=cut

#--------------------------------------------------------------------------------

sub setupVNFS
{
	if ($ENV{'SKIP_BUILDING_VNFSIMAGE'})
	{
		return $::OK;
	}
	my ($pkg,$distro_name,$distro_ver,$arch,$csm_ver,$svc_level , $minmgd, $vnfsname)=@_;
    
    ##############################################################################################
	# VNFS means virtual node file system, which is mounted by disklessnode as network file system. 
	# The following three conditions determine whether to rebuild VNFS.
    # 1 . The csmsetupinstall creates a file in 
	#     <vnfs-dir>/<WarewulfNodeGroupName>/var/opt/csm/vnfs/vnfs-build-successful or 
	#														  minmgd-vnfs-build-successful
	#		if the vnfs or MinManaged vnfs is build successfully. 
	# 	  Use this file as an indicator if we should skip the vnfs build or not. 
	#	  We should not use the vnfs directory as an indicator for build vnfs any more.
	# 2 . If the BuildDisklessVNFS==skip, this is the default case, the csmsetupinstall should check 
	#	  if the vnfs-build-successful or minmgd-vnfs-build-successful file exist. 
	#	  If these file exists
	#     		If file  minmgd-vnfs-build-successful is found and current is normal VNFS, 
	#													report warning message and exit.
	#	  		else
	#			If file vnfs-build-successful is found and current is MinManaged VNFS,
	#													report warning message and exit. 
	#			else
	#			skip building VNFS
	#	  else skip the vnfs build.  	
	# 3 . If the BuildDisklessVNFS==build,  build the vnfs. Notice in this case,  
	#	  the vnfs-build-successfully is removed when csmsetupinstall removes the old vnfs 
	#	  before build or rebuild new vnfs.
 	##############################################################################################
	
	###############################################
	#  Global variable declaration
	###############################################
	my $skip_build; # skip build flag
	my @dirlist;    # Directory list touched
    
	my $temp;
        my $FIND = $::FIND;
	
	# In case CSMINSTALL_ROOT environment variable exist 
	if ($ENV{'CSMINSTALL_ROOT'}) 
	{  
		$temp = $::CSMINSTDIR;
		$::CSMINSTDIR = $ENV{'CSMINSTALL_ROOT'}; 
	}
	@dirlist = @{$::pkgdefs{lx_diskless_dirs}};
	$::VNFSDIR = NodeUtils->GetVnfsDIRFromWarewulf();
	my $vnfs_repo="$::VNFSDIR/$vnfsname";

	###############################################
	# Determine whether to rebuild VNFS or not
	###############################################
	my ($vnfs_build_success, $minmgd_build_success);
	$vnfs_build_success = (-e "$vnfs_repo/var/opt/csm/vnfs/vnfs-build-successful") ?
						  $::OK : undef;
	$minmgd_build_success = (-e "$vnfs_repo/var/opt/csm/vnfs/minmgd-vnfs-build-successful") ?
						  $::OK : undef;	
	if ($::BOOT_ATTRIBUTES{"BuildDisklessVNFS"} eq "build" 
	 || $NodesetUtils::BOOT_ATTRIBUTES{"BuildDisklessVNFS"} eq "build")
	{
		$skip_build = 0;
		$::SKIP_VNFS_BUILD = 0; # This value will be used by Disklessprebuild scripts
	}
	
	elsif (!(defined $vnfs_build_success || 
			defined $minmgd_build_success)) { # in case last time illegal quit
	
		$skip_build = 0;
		$::SKIP_VNFS_BUILD = 0; # This value will be used by Disklessprebuild scripts
	}
			 # last VNFS type           Current type
	elsif ( ( defined $vnfs_build_success   && defined $minmgd   ) 
		  ||( defined $minmgd_build_success && (!defined $minmgd)  )
		  )
   {
		my $last_time_type = $vnfs_build_success eq $::OK ?
							 "Non-MinManaged VNFS" : "MinManaged VNFS";
		my $current_type = $minmgd eq $::OK ?
							 "MinManaged VNFS" : "Non-MinManaged VNFS";
		MessageUtils->messageFromCat(
				  'csmInstall.cat',
				  $::MSGMAPPATH,
				  'csminstall',
				  'E1',
				  'EMsgVNFSTypeNotMatch',
				  $last_time_type,
				  $current_type	 	
				  );
   }
   else
   {
  		$skip_build = 1;
  		$::SKIP_VNFS_BUILD = 1; # This value will be used by Disklessprebuild scripts

   }
	
	##############################################
	# House keeping work in VNFS
	##############################################
     
	my $distro_dir;
	$distro_dir = ServerUtils->getBaseDir("Linux",
			                               $distro_name,
										   $distro_ver,
										   $arch,
										   $svc_level);
    my %old_pkgdefs    = %::pkgdefs;
    my %pkgdefs =
          ServerUtils->get_pkgdefs("Linux", 
                                   $distro_name, 
                                   $distro_ver,
                                   $arch,
                                   "MgdNode",  
                                   $csm_ver);
	
	# FIXME for SLES, there are some bugs right now
    $distro_dir .= ($distro_name =~ /SLES/) ? "/RPMS": "/$pkgdefs{DISTRO_RPMDIR}";
    %::pkgdefs = %old_pkgdefs;
	
	# If we not skip building vnfs process
    if (!$skip_build)
	{
		# Remove original VNFS 
		rmtree($vnfs_repo,0,1);
		
		##############################################
		# Do some preparation work for building VNFS 
		# Touch directory needed by building VNFS 
		##############################################
		
		NodeUtils->touchDirList(\@dirlist,$vnfs_repo);
	 
		NodeUtils->touchFile("$vnfs_repo/fastboot");
    	NodeUtils->touchFile("$vnfs_repo/var/run/hotplug/coldplug"); # hotplug service need it
        NodeUtils->touchFile("$vnfs_repo/var/log/lastlog");
		# Initialize rpm db
		NodeUtils->runcmd("$::RPMCMD --initdb -dbpath $vnfs_repo/var/lib/rpm");
		NodeUtils->runcmd("mknod $vnfs_repo/dev/null c 1 3		>/dev/null 2>&1",-1);
    	# In SLES9/10 x86_64, yum need PYTHONPATH environment variable 
		# We need to add special case .
		
		if (  $distro_name =~ /SLES/ 
		   && $distro_ver eq '9'
		   && $arch =~ /x86_64/)
		{
			$ENV{'PYTHONPATH'} ="/usr/lib/python2.3/site-packages/"; 
		}
		if (  $distro_name =~ /SLES/ 
		   && $distro_ver eq '10'
		   && $arch =~ /x86_64/)
		{
			$ENV{'PYTHONPATH'} ="/usr/lib/python2.4/site-packages/"; 
		}

        # For RHEL5 , OS also ship package repository ,
        # To avoid conflict with it, we create repo in the subdirectory
        # of OS rpm packages directory and create symbol link for every rpm file.
        if (  $distro_name =~ /RedHat/i
           && $distro_ver >= 5 )
        {
            $distro_dir .= "/csmrepo";
            mkpath($distro_dir, $::VERBOSE, 0755);
        }

		##############################################
		# Customize yum configuration 
		##############################################
        
		my $yumconf_ref = {
					#  keyword        			vaule
					'#CSMVAR:DistroVerPkg#' => $distro_name =~ /SLES/ ? 'sles-release' : 'redhat-release',
					# distribution rpm repository 
					'#CSMVAR:BaseName#' => "$distro_name$distro_ver-$svc_level-$arch",
					'#CSMVAR:BaseUrl#'	=> "file://$distro_dir",
					# csm and warewulf/yum package repository
					'#CSMVAR:CSMName#'	=> "CSM $csm_ver",
					'#CSMVAR:CSMUrl#'	=> "file://$::CSMINSTDIR/Linux/$distro_name/csm/$csm_ver/packages/",
					# additional package repository at distro_ver level
					'#CSMVAR:CSMAddonVUrl#' => "file://" . 
												ServerUtils->getCSMAddonDir("Linux",
 																			$distro_name,
										   									$distro_ver,
										   									$arch),
                    # additional package repository at service-level level
    	    		'#CSMVAR:CSMAddonSUrl#' => "file://" . 
												ServerUtils->getCSMAddonDir("Linux",
 																			$distro_name,
										   									$distro_ver,
										   									$arch,
										  									 $svc_level)
					}; 
		if ( ! -d "$vnfs_repo/var/opt/csm/yum/")
        {
            mkpath("$vnfs_repo/var/opt/csm/yum/", $::VERBOSE, 0755);
        }
		NodeUtils->substFile($::WAREWULF_TMPL_DIR . "warewulf-tmpl." .$vnfsname ."/yum.conf", 
							 "$vnfs_repo/var/opt/csm/yum/yum.conf", 
							 $yumconf_ref);
			
		##############################################
		# Create yum repository data base
		##############################################
		my $yumconf = NodeUtils->get_config("$vnfs_repo/var/opt/csm/yum/yum.conf"); # In case user customize
		delete $$yumconf{'main'}; # remove yum own conf section
		undef @dirlist;
        my %repoHash;
        %repoHash = ();
        # e.g. %repoHash = ( 
        #                   "/csminstall/Linux/SLES/10/i386/suse/RPMS" => 
        #                      { output => "/csminstall/Linux/SLES/10/i386/suse/RPMS/csmrepo"},
        #                   "/csminstall/Linux/SLES/csm/1.7.0/packages" =>
        #                      undef
        #                  );
        # Note: undef means pkgdir equals to output dir.
		
        # Filter yum config hash and extract dirlist
		foreach my $section (keys %$yumconf)	{
			foreach my $entry (keys %{ $$yumconf{$section} }  ) {
				if ($entry eq "baseurl")
				{
					my $value = $$yumconf{$section}{$entry};
					$value =~ s/file:\/\///g;
                    if ($value =~ /csmrepo/)
                    {
                        chomp($value = NodeUtils->runcmd("$::DIRNAME $value",-1) );
                        $repoHash{$value}{output} = "$value/csmrepo";
                    }
                    else
                    {
					    $repoHash{$value} = undef;
                    }
				}
			}
		}
        my $repo_out;
		foreach my $repo (keys %repoHash)
        {
            $repo_out = undef;
            $repo_out = $repoHash{$repo}{output} || $repo_out; 
            
		    ServerUtils->createrepodatabase($repo, $repo_out);
        }
        if (  $distro_name =~ /RedHat/i
           && $distro_ver >= 5 )
        {
            mkpath($distro_dir, $::VERBOSE, 0755);  # make sure it exists
            NodeUtils->runcmd("cd $distro_dir;$::LN -fs ../*.rpm ./",-1);
        }
		##############################################
		# Get rpm pakcage to be installed in VNFS
		##############################################

		my ($rpmlistfile , $rpmlist);
		$rpmlistfile = "$::WAREWULF_TMPL_DIR/warewulf-tmpl.$vnfsname/vnfsrpm";
		
		#Get then rpm list strings .(default include operating system prerequisites ,RSCT and CSM packages )
		#But if MinManaged Mode attribute is set, CSM ,RSCT related packages will not installed into VNFS.

		$rpmlist = ServerUtils->getDisklessNodeRpmList( $distro_name, $distro_ver, $arch, $csm_ver , $rpmlistfile,$minmgd);
		# prompt starting to build VNFS	
   		MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'I',
                                     'IMsgBuildingVNFS',
                                     );

	    ##############################################
		# Build VNFS 
		##############################################
    	my $output = 
			NodeUtils->runcmd("export INUCLIENTS=1; yum -y -t -d 2 -c $vnfs_repo/var/opt/csm/yum/yum.conf --installroot $vnfs_repo install $rpmlist", -1);
		if ($::RUNCMD_RC)
		{
    		MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'V',
                                     'IMsgShow_Output',
                                     $output
                                     );
			
    		MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'E1',
                                     'EMsgDETECT_YUM_ERRORS'
                                     );
		}
	
		##############################################
		# Do some configurations in VNFS
		##############################################
			
		# SLES special	
		if ($distro_name =~ /SLES/)
		{
			#In SLES,  some dependencies package or binary name required by wulfd is different from RedHat.
			#So we install it directly via rpm.  

			my ($installwulfdfile,$csm_package_dir, $additional_dir);
			$csm_package_dir = "$::CSMINSTDIR/Linux/$distro_name/csm/$csm_ver/packages/";
			# try under additional directory
			$additional_dir = ServerUtils->getCSMAddonDir("Linux",
                                                           $distro_name,
                                                           $distro_ver,
                                                           $arch);
            $additional_dir = join(" ",$additional_dir, 
                                       ServerUtils->getCSMAddonDir("Linux",
                                                           $distro_name,
                                                           $distro_ver,
                                                           $arch,
                                                           $svc_level) 
                                  );
                                                    
            $installwulfdfile = `$FIND $csm_package_dir $additional_dir -name 'warewulf-wulfd*.rpm' | $::HEAD -n1;`;
			chomp $installwulfdfile;
			if (!-e $installwulfdfile)
			{
				MessageUtils->messageFromCat(
                                     'csmInstall.cat',
                                     $::MSGMAPPATH,
                                     'csminstall',
                                     'E1',
                                     'EMsgNotFoundWulfd',
									 $csm_package_dir,
									 $additional_dir	
                                     );

			}
			NodeUtils->runcmd("$::RPMCMD --root $vnfs_repo -Uvh --nodeps --noscripts $installwulfdfile",-2);
		}
		else
		{
			# RedHat special
			NodeUtils->runcmd("echo \"NETWORKING=yes\"  > $vnfs_repo/etc/sysconfig/network");
		}
		NodeUtils->runcmd("echo \"rsh\"		>> $vnfs_repo/etc/securetty");
		NodeUtils->runcmd("echo \"rlogin\"		>> $vnfs_repo/etc/securetty");
		NodeUtils->runcmd("echo \"rexec\"		>> $vnfs_repo/etc/securetty");
		NodeUtils->runcmd("echo \"127.0.0.1		localhost localhost.localdomain\"  > $vnfs_repo/etc/hosts");
	
		# For MinManaged disklessnode, CSM and RSCT packages will not be installed.
		# But makedisklessnode scripts depend on some files in csm.core package
		# So we extract some files from csm.core into vnfs.
		if (defined $minmgd)
		{
			my $csmcorefile;
			$csmcorefile = `$FIND "$::CSMINSTDIR/Linux/$distro_name/csm/$csm_ver/packages/" -name 'csm.core*.rpm' | $::TAIL -n1`;
			chomp($csmcorefile);
	        my $dir = Cwd::getcwd();
            chdir "$vnfs_repo";
			NodeUtils->runcmd("$::RPM2CPIO $csmcorefile | $::CPIO -idu --quiet",-1);
			# Remove some files for MS and avoiding make user confusing
			NodeUtils->runcmd("$::RM -rf $vnfs_repo/opt/csm/man",-1);
			NodeUtils->runcmd("$::RM -rf $vnfs_repo/opt/csm/samples",-1);
			my @rmlist = 
			 		("$vnfs_repo/opt/csm/bin/installms",
					 "$vnfs_repo/opt/csm/bin/uninstallms",
					 "$vnfs_repo/opt/csm/bin/copycsmpkgs" , 
					 "$vnfs_repo/opt/csm/bin/lscmtms");
			unlink @rmlist;
            chdir "$dir";
		}
		#my @service_list;
		# Post operations in VNFS
		NodeUtils->runcmd("$::LN -s ../bin/true $vnfs_repo/sbin/detect    >/dev/null 2>&1",-1);
		NodeUtils->runcmd("$::LN -s ../bin/true $vnfs_repo/sbin/nash      >/dev/null 2>&1",-1);
		if ( ! -c "$vnfs_repo/dev/random" )
		{
        	NodeUtils->runcmd("$::MKNOD $vnfs_repo/dev/random c 1 8");
		}
		if ( ! -c "$vnfs_repo/dev/urandom" )
		{
			NodeUtils->runcmd("$::MKNOD $vnfs_repo/dev/urandom c 1 9");
		}
		NodeUtils->runcmd("$::CHOWN root.sys $vnfs_repo/dev/random $vnfs_repo/dev/urandom");
		NodeUtils->runcmd("$::CHMOD 0644 $vnfs_repo/dev/random $vnfs_repo/dev/urandom");
		NodeUtils->runcmd("$::CHROOT $vnfs_repo umount /proc	>/dev/null 2>&1",-1);
		NodeUtils->runcmd("$::RM -rf $vnfs_repo/var/cache/yum/*");
		NodeUtils->runcmd("$::CP /etc/resolv.conf $vnfs_repo/etc/");
		# write fstab file
		ServerUtils->hardcode("fstab","$vnfs_repo/etc/fstab");
		
		if ($distro_name =~ /SLES/)
		{
			# SLES special
			# FIXME
			# Warewulf 2.6 don't provide specific wulfd rc script for SLES . So, we hack it.
			# we need to hardcode for SLES.
			ServerUtils->hardcode('warewulf-wulfd',"$vnfs_repo/etc/init.d/wulfd");
			chmod 0755, "$vnfs_repo/etc/init.d/wulfd";
		
			# nfs hack
			NodeUtils->touchFile("$vnfs_repo/var/lib/nfs/state");

		}
		else
		{
			# RHEL special
			# we need disable Selinux in RHEL5 ,
			# since there is not appropriate security policy for warewulf
   			if ($distro_ver >= 5 
				&& -f "$vnfs_repo/etc/selinux/config")
			{
				# Make sure "SELINUX=disabled" line existed in /etc/selinux/config 
				NodeUtils->runcmd("$::SED -i 's/^SELINUX=.*\$/SELINUX=disabled/g' $vnfs_repo/etc/selinux/config" );
			}
            # Setting up bash promptt if RHEL5   
   			if ($distro_ver >= 4.6)
            { 

                # Set PROMPT_COMMAND and PS1 environment in <VNFS>/root/.bash_profile
                my $prompt_command = <<'EOF_PROMPT';
PROMPT_COMMAND='echo -ne "\033]0;${USER}@${HOSTNAME%%.*}:${PWD/#$HOME/~}\007"'
EOF_PROMPT
                my $ps1 = <<'EOF_PS1';
[ "$PS1" = "\\s-\\v\\\$ " ] && PS1="[\u@\h \W]\\$ "
EOF_PS1
                my @content = NodeUtils->readFile("$vnfs_repo/root/.bash_profile");
                open(PROFILE, "> $vnfs_repo/root/.bash_profile") 
                  || MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                                  'csminstall', 'E2', 'EMsgCANT_WRITE_FILE', "$vnfs_repo/root/.bash_profile");
                push @content, "$prompt_command\n";
                push @content, "$ps1\n";
                print PROFILE @content;
                close(PROFILE);
            }

		}
		####################################################
		# Enable necessary system services chrooting in VNFS 
		####################################################
		ServerUtils->chSysServiceInChrtDir(
                                           $distro_name,
                                           $distro_ver,
 	                                       $arch,
                                           $svc_level,
										   $minmgd,	
										   $vnfs_repo);

        ###########################################
        # Copy time zone definition files into VNFS
        ###########################################
        my $timezonefile = "/etc/sysconfig/clock";
        my $timezonedata = "/etc/localtime";
        NodesetUtils->copyfile($timezonefile, "$vnfs_repo/$timezonefile") if ( -e $timezonefile);
        NodesetUtils->copyfile($timezonedata, "$vnfs_repo/$timezonedata") if ( -e $timezonedata);
	}
	
	######################################################
	# Enable SSH services in VNFS
	######################################################
	
	ServerUtils->setupVNFSSSH($vnfs_repo);
	
	######################################################
	# Create root and admin's accounts in VNFS
	######################################################
	
	ServerUtils->setupVNFSAccount($vnfs_repo,$distro_name);
	
	######################################################
	# Config console in VNFS
	######################################################
	
	ServerUtils->setupVNFSConsole($vnfs_repo,$distro_name,$arch);
	
	######################################################
	# Misc work in VNFS
	######################################################

	# delent following directories so RSCT can create new node id for each
	# diskless node at node's boot time
	if ( -d "$vnfs_repo/var/ct" )
	{
		rmtree("$vnfs_repo/var/ct",$::VERBOSE,1);
	}
	
	if ( -d "$vnfs_repo/var/adm" )
	{
		rmtree("$vnfs_repo/var/adm",$::VERBOSE,1);
	}
	######################################################
	# Add makedisklessnode into /etc/inittab in VNFS	
	######################################################
	my	@output=`$::GREP makedisklessnode $vnfs_repo/etc/inittab`;
	if(!@output)
	{
		open(INITTAB,">>$vnfs_repo/etc/inittab")
	  		|| MessageUtils->messageFromCat(
									  'csmInstall.cat',      $::MSGMAPPATH,
									  'csminstall',          'E1',
									  'EMsgCANT_WRITE_FILE', "$vnfs_repo/etc/inittab"
									 );

		print INITTAB "csm:345:wait:/opt/csm/install/makedisklessnode\n";
		close(INITTAB);
	}	

	# If we reach here, it means VNFS has been already built successfully
	# Touch successful flag file.
	mkpath("$vnfs_repo/var/opt/csm/vnfs/", $::VERBOSE, 0755);  # In case it doesn't exist
	NodeUtils->touchFile("$vnfs_repo/var/opt/csm/vnfs/"
						.(defined $minmgd ? "minmgd-" : "")
						."vnfs-build-successful"); 
	# Clean up environment
	$::CSMINSTDIR = $temp; # Assign original value.
}



#-------------------------------------------------------------------------------

=head3   createrepodatabase 
     Create rpm package repository database
 
    Arguments:
          $pkgdir      - directory
          $outputdir   - repository output directory 
          	
	Returns:
        	zero on success
		non-zero on failure
	Example:
        
	Comments:

=cut

#--------------------------------------------------------------------------------

sub createrepodatabase
{
	my ($class, $pkgdir, $outputdir) = @_;
        my $FIND = $::FIND;
	if ( -f "$::CREATEREPO" )
	{
	    	MessageUtils->messageFromCat(
                                     	'csmInstall.cat',
                                     	$::MSGMAPPATH,
                                     	'csminstall',
                                     	'I',
                                     	'IMsgCreaterepo',
										$outputdir || $pkgdir
                                     	);
		
            # Make sure directory exists
            mkpath($pkgdir, $::VERBOSE, 0755);
            # Remove .olddata cache firstly
			NodeUtils->runcmd("$::RM -rf $pkgdir/.olddata",-1);
		    if ($outputdir)
            {
                # Remove symbol link firstly
                #NodeUtils->runcmd("$::RM -rf $outputdir/*.rpm");
                NodeUtils->runcmd("$FIND $outputdir -name '*.rpm' -exec $::RM -f {} \\;");
                NodeUtils->runcmd("$::CREATEREPO -o $outputdir $pkgdir");
            }
            else
            {
			    NodeUtils->runcmd("$::CREATEREPO $pkgdir" );
            }
            return $::RUNCMD_RC;

	}
	else 
	{
		MessageUtils->messageFromCat(
									  'csmInstall.cat',
									  $::MSGMAPPATH,
									  'csminstall',
									  'E1',
									  'EMsgNoinstallCreaterepo',
									);
        return $::NOK;
	
	}
}	
#-------------------------------------------------------------------------------

=head3   getDisklessNodeRpmList 

	Get diskless node installation rpm list from vnfsrpm definition file.
	OS prerequisite , RSCT and CSM package are required in order to 
	support diskless node managed by CSM.
    
	If distro is SLES, remove warewulfd package from rpmlist .
    Since warewulfd's postscripts have some issues on SLES, 
	it can't be installed by yum, so we install it by "rpm --nodeps"   
        Arguments:
          $osdistr      e.g. SLES or RedHatEL-AS
          $osversion    e.g. 9 or 4
		  $arch			e.g. ppc64 or i386
		  $csmvers      e.g. 1.6.0 or 1.5.1
		  $rpmlistfle   vnfsrpmlist template definition file e.g. vnfsrpm.SLES9-SP3-i386		
	Returns:
        	zero on success
		non-zero on failure
	Example:
        
	Comments:

=cut

#--------------------------------------------------------------------------------

sub getDisklessNodeRpmList
{
	my ($class, $osdistr, $osversion, $arch, $csmvers ,$rpmlistfile, $minmgd) = @_;
    # get the appropriate lists of filesets/packages
	my %pkgdefs =
      ServerUtils->get_pkgdefs("Linux", $osdistr, $osversion,
                               $arch,   "MgdNode",  $csmvers);
    my @rpmLists;
    push @rpmLists, $::pkgdefs{'pkgs'}{'MgdNode'};
    my (@shortnamelist,@shortosnamelist, @shortcsmnamelist);
    my $rpmliststring;
	
    for my $rpmListsHash ( @rpmLists)
    {
            for my $installSeq ( sort keys( %$rpmListsHash))
           {
                my $rpmList = $rpmListsHash->{ $installSeq};
                my $rpmListDescription = $rpmList->{ '_description'};
                #Only get rpmshortnamelist from Managed Node Operating System Prerequisites Packages
				#						   		Managed Node RSCT Packages
				#						   		Managed Node CSM Packages
				if (  
					 $rpmListDescription =~ /CSM/
					|| $rpmListDescription =~ /RSCT/
				   )
				{  
					push @shortnamelist , $rpmList->getAllRpmShortNames();
					push @shortcsmnamelist , $rpmList->getAllRpmShortNames();
                	#print "rpmlist".$rpmList->{ '_description'}."=@shortnamelist\n";
				}
				elsif ($rpmListDescription =~ /Operating.*Prerequisites/)
				{
					push @shortosnamelist, $rpmList->getAllRpmShortNames();
				}
				
            }
    }
    #print "rpmlist=@shortnamelist\n";
	open(RPMLIST,"<$rpmlistfile")
      	|| MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
				                        'csminstall', 'E1',				
						  				'EMsgCANT_CREATE', $rpmlistfile
								    	);
	while (<RPMLIST>) {
			chomp;
			if ( $_ =~ /^#.*$/ ) {
					next;
			}
			#$rpmliststring = $rpmliststring ." " . $_;
			push @shortnamelist , $_;
			 
	}
    close(RPMLIST);
    # If MinManaged Mode is defined, remove CSM and RSCT rpm package from rpmlist
	# Otherwise, add OS prerequisites into rpmlist 
	my %temphash;
	if (defined $minmgd)
	{
		foreach my $name (@shortnamelist)
		{
			my $patternstring = $name; 
			$patternstring =~ s/\*[^\*]*$//g;
			$patternstring = quotemeta($patternstring);
			if (grep(/$patternstring/, @shortcsmnamelist))
			{
				print "deleteing $name\n" if ($::DEBUG);
			}
			else
			{
				$temphash{$name} =1 if (defined $name);
			}
			
		}  
		print "Remove csm! @shortcsmnamelist\n" if ($::DEBUG);
		@shortnamelist = keys %temphash;
		undef %temphash;
	}
	else
	{
		push @shortnamelist, @shortosnamelist;
	}
	
	#Remove repeated entry from namelist
	foreach my $rpmname (@shortnamelist)
	{
	    if ($osdistr =~ /SLES/)
		{
			#remove warewulf-wulfd package from rpmlist
        	if ($rpmname =~/warewulf-wulfd/)
			{
				next;
			}
		}	
		$temphash{$rpmname}=1 if (defined $rpmname);
	}

    @shortnamelist = keys %temphash;
	$rpmliststring = join(" ",@shortnamelist);
    print "rpmliststring = $rpmliststring \n" if $::DEBUG; 
	return $rpmliststring;
}

#-------------------------------------------------------------------------------

=head3   hardcode 

	Hack file needed to harcode. It is often a workaround in most cases.
	Current hardcode id:
			fstab   		-  hardcode diskless VNFS /etc/fstab file.
			warewulf-wulfd	-  hardcode wulfd boot script.
			autoconsole     -  hardcode boot script to enable console daemon
    Arguments:
          $id      	  - identifier	
		  $filename   - file path	
	Returns:
        	zero on success
		non-zero on failure
	Example:
        ServerUtils->hardcode("fstab","/etc/fstab");
	Comments:

=cut

#--------------------------------------------------------------------------------
sub hardcode
{
	my ($pkg, $id, $filename) = @_;
	my $fstab_codes = <<"EOF_FSTAB";
# Don't touch the following macro unless you really know what your doing!
%{root entry}
none            /dev/pts                devpts  gid=5,mode=620  0 0
none            /dev/shm                tmpfs   defaults        0 0
none            /proc                   proc    defaults        0 0

# To use a local disk on the nodes (make sure the mountpoints exist in
# the VNFS!!!)
#/dev/hda2      /scratch                ext3    defaults        0 0
#/dev/hda1      none                    swap    defaults        0 0

# The following lines are macros for the master's shared NFS volumes. You can
# comment out or remove if you have other plans for sharing the users data.
%{vnfs entry}
EOF_FSTAB
	my $wulfd_codes = <<'EOF_WULFD';
#!/bin/bash
#
# chkconfig: 345 95 05
# description: WULFD watches the status of the slave nodes on the cluster
#              and keeps them configured properly and available to cluster
#              users.
# processname: wulfd
# pidfile: /var/run/wulfd.pid
### BEGIN INIT INFO
# Provides:       wulfd
# Required-Start: network
# Required-Stop:  network
# Default-Start:  2 3 5
# Default-Stop:
# Description:    Start the warewulfd daemons
### END INIT INFO

WULFD=/usr/sbin/wulfd


# Source SuSE config
if [ -r /etc/rc.config ]
then
	. /etc/rc.config
fi

test -x $WULFD || exit 5

# shell functions from /etc/rc.status
. /etc/rc.status
. /etc/sysconfig/wulfd

# reset status of the service
rc_reset


case "$1" in
    start)
      	echo -n "Starting wulfd: "
	if [ -f /var/run/wulfd.pid ]
	then
		checkproc $WULFD
		rc_status -v
	else
      		startproc $WULFD -m $ADMIN_MASTER $WULFD_OPTIONS </dev/null
      		rc_status -v
      		touch /var/lock/subsys/wulfd
      		echo READY > /tmp/nodestatus
	fi
	;;

    stop) 
      	echo -n $"Shutting down wulfd: "
      	killproc $WULFD
      	RETVAL=$?
	rc_status -v
   	test -f /var/lock/subsys/wulfd && rm -f /var/lock/subsys/wulfd
   	test -f /var/run/wulfd.pid && rm -f /var/run/wulfd.pid
	;;

    stats|status)
	echo -n "Checking for wulfd: "
      	checkproc $WULFD
      	rc_status -v
	;;
    reload|reset|restart)
	$0 stop
	$0 start
	;;
    *)
	echo "Usage: $0 {start|stop|stats|status|reload|restart}"
	exit 1
	;;
esac
rc_exit
EOF_WULFD
	my $autoconsole_codes = <<'EOF_CONSOLE';
cat /proc/cpuinfo | grep POWER >/dev/null 2>&1
	TTY=hvc0
	mknod -m 620 /dev/hvc0 c 229 0 >/dev/null 2>&1
	chgrp tty /dev/hvc0
	BPS=9600

echo "##autoconsole" >>/etc/inittab
echo "hvc0:1235:respawn:/sbin/agetty -L $BPS $TTY vt320" >>/etc/inittab
echo $TTY >>/etc/securetty

init q
EOF_CONSOLE
	my %codes = (
				# id  					codes
				'fstab'			 =>		$fstab_codes,
				'warewulf-wulfd' =>		$wulfd_codes,
				'autoconsole'	 =>		$autoconsole_codes
				);
	open(FILE,">$filename")
	  		|| MessageUtils->messageFromCat(
									  'csmInstall.cat',      $::MSGMAPPATH,
									  'csminstall',          'E1',
									  'EMsgCANT_WRITE_FILE', "$filename"
									 );

	print FILE "$codes{$id}";
	close(FILE);

	#NodeUtils->runcmd("$::CAT ".$codes{$id}." > $filename");
}
#-------------------------------------------------------------------------------

=head3   chSysServiceInChrtDir 

	Enable or disable system service in a chroot directory
    Arguments:
          $distro_name 		- distribution name	
		  $distro_ver  		- distribution version
		  $arch				- distribution architecture
		  $svc_level		- distribution service level
		  $minmgd			- whether MinManaged. 1 - yes 0 - no
		  $chroot_dir 		- chroot directory				
	Returns:
        	zero on success
		non-zero on failure
	Example:
       	ServerUtils->chSysServiceInChrtDir(
                                         'SLES',
                                         '9',
 	                                     'ppc64',
                                         'GA',
										 $::OK,	
										 '/csminstall/diskless/vnfs/SLES9-GA-ppc64');
	Comments:

=cut

#--------------------------------------------------------------------------------
sub chSysServiceInChrtDir
{
	my ($pkg, $distro_name, $distro_ver, $arch, $svc_level, $minmgd, $chroot_dir)
		= @_;
	if ($distro_name =~ /SLES/)
	{
		# SLES special
		if ( -x "$chroot_dir/sbin/chkconfig" )
		{
			#-------------------------------------------------------
			# SLES 10 have some services different from SLES 9 
			# Enable some specific services
			#-------------------------------------------------------
			if (!(  $distro_name =~ /SLES/ 
			   && $distro_ver eq '10'
			   ))
			{
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 boot.shm on >/dev/null 2>&1");
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345  boot.rootfsck on >/dev/null 2>&1");
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345  boot.localfs on >/dev/null 2>&1");

			}
			else
			{
					# These services should be enabled regarding the specific order
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG   boot.rootfsck on >/dev/null 2>&1");
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG   boot.localfs on >/dev/null 2>&1");
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG   boot.cleanup on >/dev/null 2>&1");
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG   boot.localnet on >/dev/null 2>&1");
					NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG   haldaemon on >/dev/null 2>&1");
					
					# Don't set hostname in boot.localnet service again
					# Warewulf has already done it for us.
					NodeUtils->runcmd("sed -i \"s/\\\ start)/\\\ start)\\\ exit\\\ 0/g\" $chroot_dir/etc/init.d/boot.localnet",-2);
					# Comment the last three lines in /etc/pam.d/login to aviod login issue 
					NodeUtils->runcmd("$::HEAD -n -3 $chroot_dir/etc/pam.d/login > $chroot_dir/tmp/pamlogin;$::TAIL -n 3 $chroot_dir/etc/pam.d/login | sed -e \"s/^\\\(.*\\\)\$/#\\1/g\" >> $chroot_dir/tmp/pamlogin",-2);
					NodeUtils->runcmd("$::MV -f $chroot_dir/tmp/pamlogin $chroot_dir/etc/pam.d/login",-2);
			}
		
			
			#If MinManaged, rsct package is not installed.So ctrmc service don't exist.
			if (!defined $minmgd)
			{
				NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 ctrmc off >/dev/null 2>&1");
			}
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 network on >/dev/null 2>&1");
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 sshd on >/dev/null 2>&1");
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 syslog on >/dev/null 2>&1"); 
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 portmap on >/dev/null 2>&1");
			# SLES10 has no nfslock service.
			if ($distro_ver ne '10')
			{ 
				NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 nfslock on >/dev/null 2>&1");
			}
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 nfs on >/dev/null 2>&1");
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 xinetd on >/dev/null 2>&1");
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG rsh on >/dev/null 2>&1");
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG rlogin on >/dev/null 2>&1");
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG rexec on >/dev/null 2>&1");
			if ($distro_ver eq '9' && $arch !~ /ppc64/)
			{
					NodeUtils->runcmd("[ -f $chroot_dir/etc/init.d/hwscan ] && $::CHROOT $chroot_dir $::CHKCONFIG hwscan off", -1);
			}
			NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG --level 345 wulfd on >/dev/null 2>&1");
	  
		}
		else
		{
				 MessageUtils->messageFromCat(
										  'csmInstall.cat', $::MSGMAPPATH,
										  'csminstall',     "E1",
										  'EMsgLACK_CMD', "chkconfig",
										  "aaa_base"
										  );
		}

	}
	else
	{
		# RedHat special
		NodeUtils->runcmd("$::CHROOT $chroot_dir $::CHKCONFIG xinetd on >/dev/null 2>&1");
	}
}
#-------------------------------------------------------------------------------

=head3    setupVNFSSSH

	Set up ssh service in VNFS
    Arguments:
          $vnfs_repo    	- VNFS directory	
	Returns:
        	zero on success
		non-zero on failure
	Example:
        ServerUtils->setupVNFSSSH("/csminstall/diskless/vnfs/SLES9-GA-ppc64");
	Comments:

=cut

#--------------------------------------------------------------------------------
sub setupVNFSSSH
{
	my ($pkg, $vnfs_repo) = @_;
	# setup SSH
	NodeUtils->runcmd("$::CP -r /root/.ssh $vnfs_repo/root",-1);
	mkpath("$vnfs_repo/var/empty/sshd", $::VERBOSE, 0755); 
    unlink "$vnfs_repo/etc/ssh/ssh_host_key";
	NodeUtils->runcmd("$::CHROOT $vnfs_repo /usr/bin/ssh-keygen -t rsa1 -f /etc/ssh/ssh_host_key -C '' -N ''");
	NodeUtils->runcmd("$::CHROOT $vnfs_repo chmod 600 /etc/ssh/ssh_host_key");
	NodeUtils->runcmd("$::CHROOT $vnfs_repo chmod 644 /etc/ssh/ssh_host_key.pub");
	unlink "$vnfs_repo/etc/ssh/ssh_host_rsa_key";
	NodeUtils->runcmd("$::CHROOT $vnfs_repo /usr/bin/ssh-keygen -t rsa -f /etc/ssh/ssh_host_rsa_key -C '' -N ''");
	NodeUtils->runcmd("$::CHROOT $vnfs_repo chmod 600 /etc/ssh/ssh_host_rsa_key");
	NodeUtils->runcmd("$::CHROOT $vnfs_repo chmod 644 /etc/ssh/ssh_host_rsa_key.pub");
	unlink "$vnfs_repo/etc/ssh/ssh_host_dsa_key";
	NodeUtils->runcmd("$::CHROOT $vnfs_repo /usr/bin/ssh-keygen -t dsa -f /etc/ssh/ssh_host_dsa_key -C '' -N ''");
	NodeUtils->runcmd("$::CHROOT $vnfs_repo chmod 600 /etc/ssh/ssh_host_dsa_key");
	NodeUtils->runcmd("$::CHROOT $vnfs_repo chmod 644 /etc/ssh/ssh_host_dsa_key.pub");
	# SSH done

}
#-------------------------------------------------------------------------------

=head3    setupVNFSAccount

	Create root and admin account and password in VNFS
    Arguments:
          $vnfs_repo    	- VNFS directory	
          $distro_name      - DistributionName
	Returns:
        	zero on success
		non-zero on failure
	Example:
        ServerUtils->setupVNFSAccount("/csminstall/diskless/vnfs/SLES9-GA-ppc64","SLES");
	Comments:

=cut

#--------------------------------------------------------------------------------
sub setupVNFSAccount
{
	my ($pkg, $vnfs_repo,$distro_name) = @_;
	# convert to shadow account
	if ( -x "$vnfs_repo/usr/sbin/pwconv" )
	{
	   NodeUtils->runcmd("$::CHROOT $vnfs_repo /usr/sbin/pwconv >/dev/null 2>&1||:");
	}

    # setup password
    NodeUtils->runcmd("$::CHROOT $vnfs_repo chmod 644 /etc/passwd");
    # add admin accout if it don't exist
    my @output;
    @output=`$::GREP ^admin: $vnfs_repo/etc/passwd`;
    if(!@output)
    {
        #NodeUtils->runcmd("$::CHROOT $vnfs_repo useradd admin");
		# Account admin's userid and groupid in vnfs should be consistent with 
		# in install server
		my $userid = `$::GREP ^admin: /etc/passwd | $::CUT -d ':' -f 3 | $::TR -d '\n'`;
		# in case admin account doesn't exist in install server.
		if ($userid)
		{
			NodeUtils->runcmd("$::CHROOT $vnfs_repo $::USERADD admin -u $userid -o");
		}
		else
		{
			NodeUtils->runcmd("$::CHROOT $vnfs_repo $::USERADD admin");
		}

    }

    # done passeord

    # add system root and admin password to the nodes
    if ($distro_name =~ /SLES/)
    {
        NodeUtils->runcmd("echo \"root:cluster\" | $::CHPASSWD -P $vnfs_repo/etc");
        NodeUtils->runcmd("echo \"admin:cluster\" | $::CHPASSWD -P $vnfs_repo/etc");
    }
    else
    {
        NodeUtils->runcmd("echo \"root:cluster\" | $::CHROOT  $vnfs_repo $::CHPASSWD");
        NodeUtils->runcmd("echo \"admin:cluster\" |  $::CHROOT $vnfs_repo $::CHPASSWD");
    }
    #Secure shadow file in VNFS
    NetworkUtils->secureFilePermissions("$vnfs_repo/etc/shadow", "0110", "0644");
    # Set owner and group to /etc/shadow.
    NodeUtils->runcmd("$::CHOWN root $vnfs_repo/etc/shadow",-1);
    NodeUtils->runcmd("$::CHGRP shadow $vnfs_repo/etc/shadow",-1);
	
}

#-------------------------------------------------------------------------------

=head3    setupVNFSConsole

	Configure console in VNFS
    Arguments:
          $vnfs_repo    	- VNFS directory	
		  $distro_name 		- distribution name
		  $arch				- distribution architecture		
	Returns:
        	zero on success
		non-zero on failure
	Example:
        ServerUtils->setupVNFSConsole("/csminstall/diskless/vnfs/SLES9-GA-ppc64");
	Comments:

=cut

#--------------------------------------------------------------------------------
sub setupVNFSConsole
{
	my ($pkg, $vnfs_repo, $distro_name, $arch) = @_;
	my @output;


    #serial console
	my $boot_script = ($distro_name =~ /SLES/) ? 
								"$vnfs_repo/etc/init.d/boot.local"
							   :"$vnfs_repo/etc/rc.d/rc.local"; 
    if ($arch =~ /ppc64/)
	 {
			@output = `$::GREP autoconsole $boot_script`;
			if (!@output)
			{
				   ServerUtils->hardcode("autoconsole", "$boot_script");	
			}
	}
}
#--------------------------------------------------------------------------------

=head3    listWarewulfNodeGroup


    Return the expanded member list for each of the warewulf node group names.
    Note that each warewulf node group equals to unique VNFS.
    Arguments:
        None.
    Returns:
        reference to an array:

        Each element of the array represents one (1) row and has the format:

                    nodegroupname:|:{node1,node2,node3}


    Globals:
        none
    Error:
        none
    Example:
         my $outref =
            NodeUtils->listWarewulfNodeGroup();
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub listWarewulfNodeGroup
{
    my %groups;
    my @group_list;
    my %vnfs_map_table;
    # go through the node list and sort them based on unique
    #    DisklessNodeGroupName, namely InstallTemplate
    foreach my $node (keys %::NODEHASH)
    {
        if ( $::NODEHASH{$node}{'InstallMethod'} =~/warewulf/)
        {
            my $hashkey = NodeUtils->getDisklessNodeGrpName(\%::NODEHASH,$node);
            push @{$vnfs_map_table{$hashkey}}, $node;
        }
    }    
    # initialize %groups with all the keys
    foreach my $grp (keys %vnfs_map_table)
    {
        @{$groups{$grp}} = ();
        push @{$groups{$grp}}, @{$vnfs_map_table{$grp}};
    }


    # store the result in the format that exactly matches RSCT
    foreach my $grp (keys %groups)
    {
        my $nl = join(',', @{$groups{$grp}});
        my $key = $grp . ":|:{" . $nl . "}";
        push @group_list, $key;
    }

    return \@group_list;
}

1;
