#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2003,2008 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
#
# @(#)31   1.35.7.38   src/csm/core/pm/NetworkUtils.pm.perl, setup, csm_rgar2h, rgar2hs001a 11/13/07 23:24:05
#
#####################################################################

package NetworkUtils;

require ServerUtils;
require MessageUtils;
use strict;

#            locale tells perl to honor locale for sorting, dates, etc.
#            More info about locale.pm and perl handling of locales can be found in
#            /usr/lib/perl5/5.6.0/locale.pm and the man page for perllocale.

use locale;
use Socket;
use File::Find;

my $msgs;
my $distro;
my $useTranslatedMsg;
my %catHashes;
my $csmroot;
my ($MSGCAT, $MSGMAPPATH, $MSGSET);
my $NO_NODERANGES;
my $NODEGROUPEXPMEM_WARNING = 1;

# Clear dsh environment variables
if(!$ENV{'RUNNING_DSH'}){
    delete $ENV{'DSH_NODE_LIST'};
    delete $ENV{'WCOLL'};
    delete $ENV{'DSH_DEVICE_LIST'};
}
# $NodeUtils::errno;          # Will be set if an error occurs.  You must zero
# this out before calling a NodeUtils function,
# if you want to check it afterwards.

BEGIN
{

    #    This enables us to redirect where it looks for other CSM files during development
    $csmroot = $ENV{'CSM_ROOT'} ? $ENV{'CSM_ROOT'} : '/opt/csm';

    $MSGCAT = 'nodecmds.cat';
    if (defined $::MSGMAPPATH)
    {
        $MSGMAPPATH = $ENV{'CSM_ROOT'} ? "$csmroot/msgmaps" : $::MSGMAPPATH;
    }
    else
    {
        $MSGMAPPATH = "$csmroot/msgmaps";
    }
    $MSGSET = 'NodeUtils';
}

umask(0022);   #  This sets umask for all CSM files so that group and world only
               #  have read permissions.
               #  To change it, simply use the umask call in your script, after
               #  the "use NetworkUtils;" line

#--------------------------------------------------------------------------------

=head1    NetworkUtils


This program module file, supports the CSM/install networking dependencies;  
from datalink to session layers.



If adding to this file, please take a moment to ensure that:

    1.  Your contrib has a readable pod header describing the purpose and use of
         the subroutine.

    2. Your contrib is under the correct heading and is in alphabetical order
    under that heading.

    3. You  have run tidypod on your this file and copied NetworkUtils.pm.perl.tdy to
       NetworkUtils.pm.perl before checking in

       you can examine the pod output in  NetworkUtils.pm.perl.html file in a browser.

    4. Copy the new ./NetworkUtils.pm.perl.html to /project/csm/www, to update the pod.
       for the team.

=cut

#--------------------------------------------------------------------------------

=head2    DHCP Support

=cut

#--------------------------------------------------------------------------------

=head3    verify_dhcp

        Check that dhcp is up and running.

        Arguments:
                none
        Returns:
                0 - dhcp is running
                numErrors from attempt to start dhcp
        Globals:
                none
        Error:
                numErrors from attempt to start dhcp
        Example:
                $rc = NetworkUtils->verify_dhcp();
        Comments:
                Right now we don't check the /etc/dhcpd.conf file, since that
                should have  been verified when it was created.  This function
                is just for extremely paranoid peopole.

=cut

#--------------------------------------------------------------------------------

sub verify_dhcp
{
    my $errors = 0;
    my @output;
    my $cmd;
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 'V', 'IMsgDhcpCheck');

    if ($::PLTFRM eq "AIX")
    {

        # if the DHCP service is not up running, start it and check again.
        my $cmd = "$::LSSRC -s dhcpsd";
        @output = NodeUtils->runcmd($cmd, -1);
        my ($subsys, $group, $pid, $status) = split(' ', $output[1]);
        if (!defined($status) || $status ne 'active')
        {

            # start it to check again later
            $cmd = "$::STARTSRC -s dhcpsd";
            NodeUtils->runcmd($cmd, -1);
        }
        else
        {

            # dhcp is okay
            return $errors;
        }
        $cmd = "$::LSSRC -s dhcpsd";
        @output = NodeUtils->runcmd($cmd, -1);
        ($subsys, $group, $pid, $status) = split(' ', $output[1]);
        if (!defined($status) || $status ne 'active')
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                       'csminstall', 'I', 'IMsgDhcpNotWorking');
            $errors++;

        }
        return $errors;
    }

    # following logic is for Linux only
    # FIXME rhel5 put the lease in new dir
    unless (-f "/var/lib/dhcp/dhcpd.leases" || -f
        "/var/lib/dhcpd/dhcpd.leases")
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'I', 'IMsgDhcpMissing');
        unless (-d "/var/lib/dhcp" || -d "/var/lib/dhcpd")
        {
            NodeUtils->runcmd("$::MKDIR -p /var/lib/dhcp;$::MKDIR -p
                /var/lib/dhcpd");
        }
        NodeUtils->runcmd("$::TOUCH /var/lib/dhcp/dhcpd.leases;$::TOUCH
            /var/lib/dhcpd/dhcpd.leases");
    }

    # SLES Change
    #$cmd = "$::SERVICE dhcpd status";
    $cmd = NetworkUtils->service("dhcpd", "status");

    chomp(@output = `$cmd`);
    my @line = split(" ", @output);
    if (grep /running/, @output)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'V', 'IMsgDhcpOutput',
                                     @output);
    }
    else
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'V', 'IMsgDhcpOutput',
                                     @output);
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',         $::MSGMAPPATH,
                                     'csminstall',             'V',
                                     'IMsgDhcpProblemRestart', @output
                                     );

        # SLES Change
        #`$::SERVICE dhcpd start`;
        #$cmd = "$::SERVICE dhcpd status";
        $cmd = NetworkUtils->service("dhcpd", "start");
        `$cmd`;
        $cmd = NetworkUtils->service("dhcpd", "status");

        chomp(@output = `$cmd`);
        if (grep /running/, @output)
        {

            #print "dhcp: @output\n" if $::VERBOSE;
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'V', 'IMsgDhcpOutput', @output);
        }
        else
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                       'csminstall', 'I', 'IMsgDhcpNotWorking');
            $errors++;
        }
    }
    return $errors;
}

#--------------------------------------------------------------------------------

=head3    verify_atd

	check for atd status. If it is not running, start it. 

        Arguments:

        Returns:
                $::RUNCMD_RC = 0 atd is running		
                $::RUNCMD_RC > 0 atd is not running
        Globals:
                none
        Example:
                my $rc = NetworkUtils->verify_atd;
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub verify_atd
{
    my ($class) = @_;
    my ($cmd);

    $cmd = NetworkUtils->service("atd", "status");
    NodeUtils->runcmd($cmd, -1);
    if ($::RUNCMD_RC)
    {
        $cmd = NetworkUtils->service("atd", "start");
        NodeUtils->runcmd($cmd, -1);
        if (($::RUNCMD_RC) && ($::VERBOSE))
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'I', 'IMsgSTART_ATD_FAILED');
        }
        elsif (!$::RUNCMD_RC)
        {
            sleep(1);
            if ($::VERBOSE)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                    'csminstall', 'I', 'IMsgSTART_ATD_SUCCESS');
            }
        }
    }
    else
    {
        if ($::VERBOSE)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                    'csminstall', 'I', 'IMsgSTART_ATD_SUCCESS');
        }
    }
    return $::RUNCMD_RC;
}

#--------------------------------------------------------------------------------

=head2    Ethernet Support

=cut

#--------------------------------------------------------------------------------

=head3    chkethup

         Ensure that the eth0 is up and running 

        Arguments:
                none
        Returns:
                none
        Globals:
                none
        Error:
                messageFromCat E1
        Example:
                NetworkUtils->chkethup();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub chkethup
{
    my $cmd = "$::NETSTAT -rn | $::GREP eth0";
    unless (`$cmd`)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'E1', 'EMsgEth0NotUp');
    }
    return;
}

#--------------------------------------------------------------------------------

=head2    IP Address Support

=cut

#--------------------------------------------------------------------------------

=head3    genHex

        Generates a hex representation of an ip address:
            for example, 9.114.133.193 --> 097285C1

        Takes in a list of hosts and returns a hash where the key is
        the hostname and the value is the IP in hex form. 

        Arguments:
                @hostNames
        Returns:
                Hash of hostName / IP address (in hex) pairs.
        Globals:
                none
        Error:
                undefined
        Example:
                %hex = NetworkUtils->genHex(@::nodes);
        Comments:
			It is STRONGLY advised that before calling this function
			@hostNames should contain the node names that have
			already been resolved to long hostname by calling
			NetworkUtils->getHost().

=cut

#--------------------------------------------------------------------------------

sub genHex
{
    shift;
    my $host;
    my $uno;
    my $dos;
    my $tres;
    my $cuatro;
    my $hex;
    my %hex;

    foreach $host (@_)
    {
        my ($hostname, $ip) = NetworkUtils->getHost($host);    # Get ip address
        ($uno, $dos, $tres, $cuatro) = split(/\./, $ip);    # Split by the .
        $hex = sprintf("%02X", $uno);    # put formatted into hex
        $hex .= sprintf("%02X", $dos);
        $hex .= sprintf("%02X", $tres);
        $hex .= sprintf("%02X", $cuatro);
        $hex{$hostname} = $hex;
    }
    return %hex;
}

#--------------------------------------------------------------------------------

=head3    inc_ip

        Increment an IP address by 1, carrying to the previous octet as required.
        This code originally appeared in the LUI product (luiapi.pm)

        Arguments:
                $IPaddress
        Returns:
                IP address as a string
        Globals:
                none
        Error:
                csmInstall.cat E
        Example:
                $node = NetworkUtils->inc_ip($ipaddr); 
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub inc_ip
{
    my ($class, $ip) = @_;
    
    my ($a, $b, $c, $d);
    ($a, $b, $c, $d) = split(/\./, $ip);
    $d++;
    if ($d > 255)
    {
        $d = $d - 256;
        $c++;
        if ($c > 255)
        {
            $c = $c - 256;
            $b++;
            if ($b > 255)
            {
                $b = $b - 256;
                $a++;
                if ($a > 255)
                {
                    MessageUtils->messageFromCat('csmInstall.cat',
                           $::MSGMAPPATH, 'csminstall', 'E', 'EMsgIPADDR_ERROR',
                           "$a.$b.$c.$d");

                    #printf "Error - ip address out of valid range.\n";
                    return 1;
                }
            }
        }
    }
    return $a . "\." . $b . "\." . $c . "\." . $d;

}

#--------------------------------------------------------------------------------

=head3    isIpaddr

    returns 1 if parameter is has a valid IP address form.

    Arguments:
        dot qulaified IP address: e.g. 1.2.3.4
    Returns:
        1 - if legal IP address
        0 - if not legal IP address.
    Globals:
        none
    Error:
        none
    Example:
         if (NetworkUtils->isIpaddr( $ipAddr )) { blah; }
    Comments:
        Doesn't test if the IP address is on the network,
        just tests its form.

=cut

#--------------------------------------------------------------------------------

sub isIpaddr
{
    my ($class, $addr) = @_;

    #print "addr=$addr\n";
    if ($addr !~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/)
    {
        return 0;
    }

    if ($1 > 255 || $1 == 0 || $2 > 255 || $3 > 255 || $4 > 255)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

#--------------------------------------------------------------------------------

=head3    subnet

        Returns the subnet in dot format

        Arguments:
                $hostName
                $netmask
        Returns:
                subnet for $hostName
        Globals:
                none
        Error:
                none
        Example:
                never used.
        Comments:
                never used???

=cut

#--------------------------------------------------------------------------------

sub subnet
{
    my ($class, $host, $netmask) = @_;

    #$host = `hostname` if (!$host);
    #$netmask = '255.255.255.0' if (!$netmask);

    chomp $host;
    chomp $netmask;

    #print "INPUT:  $host $netmask\n";

    my ($hostname, $ipaddr) = NetworkUtils->getHost($host);

    my ($ia, $ib, $ic, $id) = split('\.', $ipaddr);
    my ($na, $nb, $nc, $nd) = split('\.', $netmask);

    # Convert to integers so the bitwise and (&) works correctly.
    int $ia;
    int $ib;
    int $ic;
    int $id;
    int $na;
    int $nb;
    int $nc;
    int $nd;

    my $sa     = ($ia & $na);
    my $sb     = ($ib & $nb);
    my $sc     = ($ic & $nc);
    my $sd     = ($id & $nd);
    my $subnet = "$sa.$sb.$sc.$sd";

    #print "IPADDR = $ia.$ib.$ic.$id\n";
    #print "NETMSK = $na.$nb.$nc.$nd\n";
    #print "SUBNET = $subnet\n";

    return $subnet;
}
#--------------------------------------------------------------------------------

=head3    broadcast

        Returns the broadcast in dot format

        Arguments:
                $hostName or ip
                $netmask
        Returns:
                broadcast for $hostName
        Globals:
                none
        Error:
                none
        Example:
                $broadcast = NetworkUtils->broadcast($node_ip).
        Comments:

=cut

#--------------------------------------------------------------------------------

sub broadcast
{
    my ($class, $host, $netmask) = @_;

    #$host = `hostname` if (!$host);
    #$netmask = '255.255.255.0' if (!$netmask);

    chomp $host;
    chomp $netmask;

    #print "INPUT:  $host $netmask\n";

    my ($hostname, $ipaddr) = NetworkUtils->getHost($host);

    my ($ia, $ib, $ic, $id) = split('\.', $ipaddr);
    my ($na, $nb, $nc, $nd) = split('\.', $netmask);

    # Convert to integers so the bitwise and (&) works correctly.
    int $ia;
    int $ib;
    int $ic;
    int $id;
    int $na;
    int $nb;
    int $nc;
    int $nd;

    my $sa     = ($ia |  ($na ^ 255) );
    my $sb     = ($ib |  ($nb ^ 255) );
    my $sc     = ($ic |  ($nc ^ 255) );
    my $sd     = ($id |  ($nd ^ 255) );
    my $broadcast = "$sa.$sb.$sc.$sd";

    #print "IPADDR = $ia.$ib.$ic.$id\n";
    #print "NETMSK = $na.$nb.$nc.$nd\n";
    #print "broadcast = $broadcast\n";

    return $broadcast;
}

#--------------------------------------------------------------------------------

=head2    MAC Address Support

=cut

#--------------------------------------------------------------------------------

=head3    getmacs_via_dsh

        Gathers MAC addresses for an array of node Names.

        Arguments:
                referance to an array of Node Names
        Returns:
                Array of unreachable node names
        Globals:
                none
        Error:
                none
        Example:
                unused

        Comments:
                unused???

=cut

#--------------------------------------------------------------------------------

sub getmacs_via_dsh
{
    my ($class, $ref_nodelist) = @_;
    my @nodelist = @$ref_nodelist;
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 'I', 'IMsgGETTING_MACS_VIA_DSH');
    my $dshcmd =
      '/sbin/ifconfig eth0 | /bin/grep HWaddr | /bin/awk "{print \$5}"';
    my $RemoteShell = NetworkUtils->getRemoteShell();
    my @new_getmacslist;
    my @unreach_nodes = ();
    my @reach         = ();

    my $node_file =
      ServerUtils->make_node_list_file(\@nodelist)
      ;    #put all dest nodes in tmp file
    my $cmd = "$::REMOTESHELL_EXPECT -d $node_file";
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 "V", 'IMsgRUN_CMD', $cmd);
    my @output = `$cmd 2>&1`;
    ServerUtils->close_delete_file($::NODE_LIST_FILE, $node_file)
      ;    #delete node file
    my %out;

    foreach my $line (@output)
    {
        chomp $line;
        my @parts = split ":", $line;
        if ($parts[1] =~ m/test.success/)
        {    #this node is reachable
            $out{$parts[0]} = 1;
        }
    }
    foreach my $n (@nodelist)
    {
        if (!$out{$n})
        {
            push @unreach_nodes, $n;
        }
        else
        {    #it is reachable
            push @reach, $n;
        }
    }
    if (@unreach_nodes)
    {
        my $node_str = join ', ', @unreach_nodes;
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',       $::MSGMAPPATH,
                                     'csminstall',           'W',
                                     'EMsgDSHCannotConnect', $RemoteShell,
                                     $node_str
                                     );
        push @new_getmacslist, @unreach_nodes;
    }
    if (@reach)
    {

        #we can theoretically reach it with DSH
	my %options_api = ();
	$options_api{'command'} = $dshcmd;
        $options_api{'nodes'} = join(',', @reach);
        my $output = NodeUtils->runDsh(\%options_api, -1);
        my %macs;
        foreach my $line (@$output)
        {
            chomp $line;
            my ($nodename, $macaddr) = split(': ', $line);
            if ($macaddr =~ /(..:..:..:..:..:..)/)
            {
                $macs{$nodename} = $macaddr;
            }
        }
        foreach my $node (@reach)
        {
            if ($macs{$node})
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'I', 'IMsgFOUND_MACADDR', $node,
                                  $macs{$node});
                NodeUtils->runcmd(
                          "$::CHNODE $node InstallAdapterMacaddr=$macs{$node}");
            }
            else
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'I', 'IMsgCANT_DSH_NODE', $node);
                push(@new_getmacslist, $node);
            }
        }
    }
    return (@new_getmacslist);
}

#--------------------------------------------------------------------------------

=head2    NFS Support

=cut

#--------------------------------------------------------------------------------

=head3   cacheEtcHosts

    This subroutine caches /etc/hosts to improve performance for large clusters.
    This routine is only called if more than 15 nodes are used.

=cut

#--------------------------------------------------------------------------------

sub cacheEtcHosts
{
    if (defined $NodeUtils::cacheetchosts || defined $ENV{'CSM_NO_NAME_RES'})
    {
        return;
    }
    else
    {
        $NodeUtils::cacheetchosts = 1;    #so we don't do this again
    }
    my $cache = 0;
    if (NodeUtils->isLinux())
    {

        #On linux you can tell if /etc/hosts is used by looking at the /etc/host.conf
        my $file = "/etc/host.conf";
        if (open(CONF, $file))
        {
            while (<CONF>)
            {
                my $line = $_;
                chomp $line;
                if ($line =~ m/^\s*order\s+hosts/)
                {
                    $cache = 1;
                    last;
                }
            }
            close(CONF);
        }
        else { return; }

    }
    elsif (NodeUtils->isAIX())
    {

        #there is an environement variable
        if (defined $ENV{'NSORDER'} && length($ENV{'NSORDER'}))
        {
            if ($ENV{'NSORDER'} =~ m/^local/) { $cache = 1; }
            else { $cache = 0; }
        }
        else
        {    #On AIX -- look at /etc/netsvc.conf
            my $file = "/etc/netsvc.conf";
            if (-e "/etc/irs.conf" && !-e "/etc/netsvc.conf")
            {
                $file = "/etc/irs.conf";
            }
            if (open(CONF, $file))
            {
                my $line;
                while ($line = <CONF>)
                {
                    chomp $line;
                    if ($line !~ m/^\#/ && $line =~ m/^\s*hosts/)
                    {    #if it doesn't start with a commnet
                        if ($line =~ m/^\s*hosts\s*=\s*local/)
                        {
                            $cache = 1;
                        }
                        last;
                    }
                }
                close(CONF);
            }
        }

    }
    if ($cache == 0) { return; }
    my $file = "/etc/hosts";
    if (open(HOSTS, $file))
    {

        #print "CACHING ETC HOSTS\n";
        my $line;
        while ($line = <HOSTS>)
        {
            chomp $line;
            my ($before, $after) = split "#", $line;
            if ($before)
            {
                my ($ip, $longhost, $shorthost) = split /\s+/, $before;

                #print "IP =|$ip|\tlonghost =|$longhost|\tshorthost =|$shorthost|\n";
                if (length($longhost) && length($ip) && $ip =~ /127\.0\.0/)
                {
                    $NodeUtils::Hosts{'Hostname'}{$longhost}{'Name'} =
                      $longhost;
                    $NodeUtils::Hosts{'Hostname'}{$longhost}{'IP'} = $ip;
                    $NodeUtils::Hosts{'IP'}{$ip} = $longhost;
                    if (length($shorthost))
                    {
                        $NodeUtils::Hosts{'Hostname'}{$shorthost}{'Name'} =
                          $longhost;
                        $NodeUtils::Hosts{'Hostname'}{$shorthost}{'IP'} = $ip;
                    }
                }

            }

        }
        close(HOSTS);
    }

}

#--------------------------------------------------------------------------------

=head3    determineMS

        Get the name of the localhost from an array of node names.

        Arguments:
        reference to a list of nodeNames
        Returns:
                Name of the local host
        Globals:
                $::PLTFRM
        Error:
                undefined
        Example:
        :$::MS_NODE = NetworkUtils->determineMS(\@::NODELIST);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub determineMS
{
    my ($class, $refnodes) = @_;
    delete $ENV{'CSM_NO_NAME_RES'};    #so that tryHost works
    my @nodes = @$refnodes;
    my (%localhostnames, $localnode);

    my @ipaddrs;
    my $cmd;
    if($::PLTFRM eq "AIX")
    {
        $cmd=
            "$::IFCONFIG -a  | $::GREP \"inet \" | $::GREP -v 127.0.0.1 | /bin/awk '{print \$2}'";
    }
    else
    {
        $cmd=
            "$::IP addr | $::GREP \"inet \" | $::GREP -v 127.0.0.1 | $::TR '/' ' ' | $::AWK '{print \$2}'";
    }
    chomp(@ipaddrs =NodeUtils->runcmd($cmd));

    foreach my $addr (@ipaddrs)
    {

        #see what it resolves to
        $localhostnames{NetworkUtils->tryHost($addr)} = 1;
    }
  FINDMS: foreach my $name (@nodes)
    {
        my $rn = NetworkUtils->tryHost($name);
        if ($localhostnames{$rn})
        {

            #there should only be one ManagedNode that maps to the localhost
            $localnode = $name;
            last FINDMS;
        }
    }
    return $localnode;
}
#--------------------------------------------------------------------------------

=head3    export_home

        NFS Export the /home directory on the Management Server for diskless node. 
        
        This subroutine is used when InstallMethod is warewulf.
        Arguments:
				$ref_nodehash:		Node Hash
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
        Error:
                $::NOK
        Example:
                if(NetworkUtils->export_home(\%nodehash) != 0) {
                    blah;
                }
        Comments:
                In order to share data between diskless nodes, the /home needs 
                to be exported on management server. 
                Also, we should add a entry like this 
                "/home network/netmask(rw,root_squash,async)" where the network
                is CSM cluster network.
                E.g, "/home 192.168.6.0/255.255.255.0(rw,root_squash,async)"
=cut

#--------------------------------------------------------------------------------

sub export_home
{
	my ($class, $ref_nodehash) = @_;
	my ($cmd, $rc);

	my (%nodehash);
	my @mgtnodes = ();
	# Default exported dir
	my $targetdir = "/home";

	%nodehash = %$ref_nodehash if ($ref_nodehash);

	$rc = 0;

	# The nodes which use MS as management server
	@mgtnodes = NodeUtils->getUsingMSNodes(\%nodehash) if ($ref_nodehash);

	# Trying to export MS's dir
	foreach my $nodename (keys %nodehash)
    {
        if ($nodehash{$nodename}{'InstallMethod'} =~ /warewulf/)
        {   
            my ($bc, $netmask) = NetworkUtils->getNetmask($nodename, \%nodehash);
            if (!defined $netmask)
            {
                MessageUtils->message("W", 'EMsgINVALID_NETMASK', $nodename);
                next;
            }
            my $subnet = NetworkUtils->subnet($nodename, $netmask);
		if ( $::PLTFRM eq "AIX")
		{
                 # The /home directory already exists in /etc/exports?   
                 # Extract hostname list have access 
                 my $output = `$::GREP  "^/home" /etc/exports | $::AWK "{print \$2}" | $::CUT -d ',' -f 1 | $::CUT -d "=" -f 2`;
                 my $hostlist = $subnet;
                 chomp($output);
                 # If entries already exists
                 if ($output)
                 {
                     chomp($output);
                     $hostlist = $hostlist . ':' . $output;
                 }
				 $cmd = "$::EXPORTFS -i -o access=$hostlist,root=$hostlist $targetdir";
                 # Add a entry into /etc/exports in case nfs server restart
                 my $export_entry = "/home -access=$hostlist,root=$hostlist";
                 # If no entry exists
                 if (!$output)
                 { 
                     NodeUtils->runcmd("$::ECHO '$export_entry' >> /etc/exports");
                 }
                 else
                 {
                     my @content = NodeUtils->readFile("/etc/exports");
                     open(EXPORT, ">/etc/exports") 
                          || MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                                         'csminstall', 'E2', 'EMsgCANT_WRITE_FILE', "/etc/exports");
                     foreach my $line (@content)
                     {
                          $line =~ s/^\/home.*$/\/$export_entry/g;
                          print EXPORT $line;
                     }
                     close(EXPORT);
                 }
			
		} else # For Linux Only
		{
			
				$cmd = "$::EXPORTFS -o sync,rw,root_squash,insecure $subnet/$netmask:$targetdir"  ;
	            # Add entry into /etc/export in case nfs server restart
                my $output = `$::GREP  "^/home.*$subnet" /etc/exports`;
                if (!$output)
                { 
                    NodeUtils->runcmd("$::ECHO '/home $subnet/$netmask(rw,root_squash,async,insecure)' >> /etc/exports");
                }

		}

		$rc = NodeUtils->run_cmd($cmd, "W", "ignore_errors");
		if ($rc)
		{
			MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
					'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
		}
	}
}

	return $rc;
}

#--------------------------------------------------------------------------------

=head3    export_csminstall

        NFS Export the csminstall directory on the Management Server and Install Server

        Arguments:
				$ref_nodehash:		Node Hash
				$ref_ishash:		nodehash's InstallServer/NFSServer Hash
                $ref_target_dir:	Reference to exported target dir array 
				$unexport:			unexport or export
					- 0 or null : export csminstall
					- 1: unexport csminstall
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
        Error:
                $::NOK
        Example:
                if(NetworkUtils->export_csminstall(\%nodehash, \%ishash) != 0) {
                    blah;
                }
        Comments:
			If no %nodehash or %ishash, default to export MS's dir
=cut

#--------------------------------------------------------------------------------

sub export_csminstall
{
	my ($class, $ref_nodehash, $ref_ishash, $ref_target_dir, $unexport) = @_;
	my ($cmd, $rc);

	my (%nodehash, %ishash, %nfshash);
	my @mgtnodes = ();
	# Default exported dir
	my @targetdir = ("/csminstall/Linux", "/csminstall/csm");

	if ($ref_target_dir) 
	{
		@targetdir = @$ref_target_dir;
	}

    # Record all export directories by CSM
    if ( ! $unexport)
    {
        open(EXPORTS, ">$::CSM_EXPORT_DIRS")
        || MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
            'csminstall', 'E', 'EMsgCANT_READ_FILE', $::CSM_EXPORT_DIRS);
        foreach my $dir (@targetdir)
        {
            print EXPORTS "$dir\n";
        }
        close(EXPORTS);
    }
	%nodehash = %$ref_nodehash if ($ref_nodehash);
	%ishash = %$ref_ishash if ($ref_ishash);

	$rc = 0;
	my (@installing_MgtNodes, @installing_installservers);

	#Get installing nodes list and their install servers.
	#Do NOT unexport the directories which are still used by installing nodes.
	if ($unexport)
	{
		my @installing_nodes = ServerUtils->get_installing_nodes;
		if (@installing_nodes)
		{
			my ($ref_DestNode, $ref_lsnode_info, $ref_DestNodeHash) =
				NodeUtils->get_target_nodes(\@installing_nodes);
			my ($ref_Nodes, $ref_InstallServers) =
				ServerUtils->getInstallServers($ref_DestNodeHash);
			my (undef, $ref_NFSServers) =
				ServerUtils->getNFSServers($ref_DestNodeHash);
			@installing_MgtNodes       = @$ref_Nodes;
			@installing_installservers = keys %$ref_InstallServers;
			push @installing_installservers, keys %$ref_NFSServers;
		}
	}

	# The nodes which use MS as management server
	@mgtnodes = NodeUtils->getUsingMSNodes(\%nodehash) if ($ref_nodehash);

	# Trying to export MS's dir
	if (((@mgtnodes && !@installing_MgtNodes) || ! $ref_nodehash)
        	&& !($unexport && @installing_MgtNodes))
	{
		my @exportdir = ();

		# if all target nodes are AIX machines, do not export /csminstall/Linux
		if ( $::PLTFRM eq "AIX")
		{
			# Check if there are Linux Nodes
			my $have_linux_nodes = 0;
			foreach my $ms_node (@mgtnodes)
			{
				if ($nodehash{$ms_node}{InstallOSName} eq "Linux")
				{
					$have_linux_nodes = 1;
					next;
				}
			}
			foreach my $dir (@targetdir)
			{
				if (! $have_linux_nodes && $dir eq "/csminstall/Linux" )
				{
					next;
				}
				if (-d $dir)
				{
					push @exportdir, $dir;
				} else
				{
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
							'csminstall', 'W', 'EMsgNo_Dir_Export', $dir);
				}
			}
			if ($unexport)
			{
				$cmd = "$::EXPORTFS -i -u ". join " ", @exportdir;
			} else
			{
				 $cmd = "$::EXPORTFS -i -o anon=0 " . join " ",  @exportdir;
			}
		} else # For Linux Only
		{
			foreach my $dir (@targetdir)
			{
				if (-d $dir)
				{
					push @exportdir, "*:" . $dir;
				} else
				{
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
							'csminstall', 'W', 'EMsgNo_Dir_Export', $dir);
				}
			}
			if ($unexport)
			{
				$cmd = "$::EXPORTFS -u " . join " ", @exportdir;
			} else
			{
				$cmd = "$::EXPORTFS -o sync,ro,no_root_squash,insecure " . join " ",  @exportdir;
			}
		    
			if (NodeUtils->isHMC() && ($ENV{'DC_ENVIRONMENT'} ne 1))
			{
				if ($unexport)
				{
					my $cmd = "/usr/sbin/rsct/bin/runlpcmd nfs_start_stop stop";
					NodeUtils->runcmd($cmd, -1);
					return $::RUNCMD_RC;
				}
				return 0;
			}
			if ($unexport)
			{
				# we have to unexport twice on Linux,
				# to fix one Linux kernel bug
				$rc = NodeUtils->run_cmd($cmd, "W", "ignore_errors");
				if ($rc)
				{
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
						'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
				}
			}
		}

		$rc = NodeUtils->run_cmd($cmd, "W", "ignore_errors");
		if ($rc)
		{
			MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
					'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
		}
	}

	#Special for install servers
	foreach my $is (keys %ishash)
	{
		next if ( $unexport and (grep /^$is$/, @installing_installservers));
		my @exportdir = ();
		my $os = $ishash{$is}{InstallOSName};
		if ($os ne "AIX" && $os ne "Linux")
		{
			MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
					'csminstall', 'E1', 'EMsgINVALID_OSTYPE');
		}
		if ($os eq "Linux")
		{
			push @exportdir, "*:" . $ishash{$is}{Dir} . "/Linux";
			push @exportdir, "*:" . $ishash{$is}{Dir} . "/csm";
		} else
		{
			push @exportdir, $ishash{$is}{Dir} . "/csm";
		}

		my @nodes = ($is);

		my $real_dir = join " ", @exportdir;
		my %options_api = ();
                $options_api{'nodes'} = join(',', @nodes);
		if ($os eq "AIX")
		{
			if ($unexport)
			{
				$cmd = "$::EXPORTFS -i -u $real_dir";
				$options_api{'command'} = $cmd;
			} else
			{
				$cmd = "$::EXPORTFS -i -o ro $real_dir";
				$options_api{'command'} = $cmd;
			}
		} else
		{
			if ($unexport)
			{
				$cmd = "$::EXPORTFS -u $real_dir";
				$options_api{'command'} = $cmd;
			} else
			{
				$cmd = "$::EXPORTFS -o sync,ro,no_root_squash,insecure $real_dir";
				$options_api{'command'} = $cmd;
			}
		}
		NodeUtils->runDsh(\%options_api, 0);
		$rc = $::RUNCMD_RC;
	}
	return $rc;
}

#--------------------------------------------------------------------------------

=head3    getHost

    Return primary hostname and ip address for the given hostname or
    ip address and die if hostname resolution fails.

    Arguments:
        A string: either an Ip Address or a HostName.
    Returns:
         ($nodename, $nodeip) as strings
    Globals:
        $NodeUtils::errno
        $NodeUtils::Hosts
    Error:
        returns 0;
        sets NodeUtils::errno.
    Example:
        my ($nodename, $nodeip) = NetworkUtils->getHost($node);
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub getHost
{
    my ($class, $arg, $live) = @_;

    #print "arg=$arg\n";
    my ($hostname, $ipaddr);
    if (!$NodeUtils::gethost{$arg})
    {    #not cached
        if (NetworkUtils->isIpaddr($arg))
        {
            if (defined $NodeUtils::Hosts{'IP'}{$arg})
            {
                $ipaddr = $arg;

                #cached from /etc/hosts
                $hostname = $NodeUtils::Hosts{'IP'}{$arg};

                #print "using cached IP: $arg\n";
            }
            else
            {
                $ipaddr = $arg;
                my $packedaddr = inet_aton($ipaddr);
                $hostname = gethostbyaddr($packedaddr, AF_INET);
                if (!$hostname)
                {
                    $hostname = $ipaddr;
                }

                #print "hostname=$hostname\n";
            }
        }
        else    # they specified a hostname
        {
            if (defined $NodeUtils::Hosts{'Hostname'}{$arg})
            {
                $hostname = $NodeUtils::Hosts{'Hostname'}{$arg}{'Name'};
                $ipaddr   = $NodeUtils::Hosts{'Hostname'}{$arg}{'IP'};

                #print "using cached Hostname: $arg\n";
            }
            else
            {
                $hostname = $arg;    # this may be a short hostname
                my ($name, $aliases, $addrtype, $length, @addrs) =
                  gethostbyname($hostname);
                if (!length($name) && $live)
                {
                    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                          "E28", 'EMsgBAD_HOSTNAME', $hostname);
                }
                if (!length($name))
                {
                    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                          "E28", 'EMsgBAD_HOSTNAME', $hostname);
                }

                #print "name=$name, # of addrs=$#addrs.\n";
                my $packedaddr = $addrs[0];
                $ipaddr   = inet_ntoa($packedaddr);
                $hostname =
                  $name; # they may have specified a shorter or non-primary name
                         #print "ipaddr=$ipaddr, hostname=$hostname\n";
            }
        }
        $NodeUtils::gethost{$arg}{'Name'} = $hostname;    #cache this for later
        $NodeUtils::gethost{$arg}{'IP'}   = $ipaddr;      #cache this for later
    }
    else
    {                                                     #cached
        $hostname = $NodeUtils::gethost{$arg}{'Name'};
        $ipaddr   = $NodeUtils::gethost{$arg}{'IP'};
    }
    MessageUtils->messageFromCat(
                              'csmInstall.cat',
                              $::MSGMAPPATH,
                              'csminstall',
                              "E1",
                              'EMsgHostnameShouldNotBeResolveToLoopbackAddress',
                              $hostname,
                              $ipaddr
      )
      if $ipaddr =~ /127\.0\.0/;
    return ($hostname, $ipaddr);
}


#--------------------------------------------------------------------------------

=head3    getHost_noexit
    Get the primary hostname and ip address for the given hostname or ip address.

    Arguments:
        $arg: either an Ip Address or a HostName.
    Returns:
        $err_code - error code. 101-cannot resolve; 102-get a loopback interface
        $Hostname
        $IP
    Globals:
        $::FORCE_RMNODE
        $NodeUtils::Hosts
   
    Example:
        my ($rc, $nodename, $nodeip) = NetworkUtils->getHost($node);
=cut

#--------------------------------------------------------------------------------

sub getHost_noexit
{
    my ($class, $arg) = @_;

    my $err_code;
    my ($hostname, $ipaddr);
    
    if (!$NodeUtils::gethost{$arg})
    {    #not cached
        if (NetworkUtils->isIpaddr($arg))
        {
            if (defined $NodeUtils::Hosts{'IP'}{$arg})
            {
                $ipaddr = $arg;

                #cached from /etc/hosts
                $hostname = $NodeUtils::Hosts{'IP'}{$arg};
            }
            else
            {
                $ipaddr = $arg;
                my $packedaddr = inet_aton($ipaddr);
                $hostname = gethostbyaddr($packedaddr, AF_INET);
                if (!$hostname)
                {
                    $hostname = $ipaddr;
                }
            }
        }
        else    # they specified a hostname
        {
            if (defined $NodeUtils::Hosts{'Hostname'}{$arg})
            {
                #using cached Hostname
                $hostname = $NodeUtils::Hosts{'Hostname'}{$arg}{'Name'};
                $ipaddr   = $NodeUtils::Hosts{'Hostname'}{$arg}{'IP'};
            }
            else
            {
                $hostname = $arg;    # this may be a short hostname
                my ($name, $aliases, $addrtype, $length, @addrs) =
                  gethostbyname($hostname);
                if (!length($name))
                {
                    $err_code = 101;
                    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                          "E", 'EMsgBAD_HOSTNAME', $hostname);
                    return ($err_code);
                }

                my $packedaddr = $addrs[0];
                $ipaddr   = inet_ntoa($packedaddr);
                $hostname = $name; # they may have specified a shorter or non-primary name
            }
        }
        $NodeUtils::gethost{$arg}{'Name'} = $hostname;    #cache this for later
        $NodeUtils::gethost{$arg}{'IP'}   = $ipaddr;      #cache this for later
    }
    else
    {                                                     #cached
        $hostname = $NodeUtils::gethost{$arg}{'Name'};
        $ipaddr   = $NodeUtils::gethost{$arg}{'IP'};
    }

    if ($ipaddr =~ /127\.0\.0/)
    {
        MessageUtils->messageFromCat(
                              'csmInstall.cat',
                              $::MSGMAPPATH,
                              'csminstall',
                              "E",
                              'EMsgHostnameShouldNotBeResolveToLoopbackAddress',
                              $hostname,
                              $ipaddr);
        $err_code = 102;
        return ($err_code);
    }
    
    return (0, $hostname, $ipaddr);
}



#--------------------------------------------------------------------------------

=head3    getHost_rveg

	NOTE:  This is a last minute correction for the rveg release,
	which will be distributed to CSM as a whole during rbac.

    Description:

	Returns primary hostname and ip address for a node.
	Either an IP address or a Hostname will be accepted as
	the single parameter.

	Uses a runcmd style error code and message string to return status.

	Status should be checked before accessing the return values, 
	which will both be undef if the command failed.


    Arguments:
        One scalar parameter:  either an Ip Address or a HostName.

    Returns:
	($host_name, $ip_addr)	- the hostname and ip address for parameter

	$::GETHOST_RC		- $::OK on success - this is zero
	$::GETHOST_MSG		- could contain a warning message on success
                                  but currently doesn't.

    Globals:
	$::GETHOST_RC
	@::GETHOST_MSG

    Error:
        Sets $::GETHOST_RC to:

		"E27" bad ip address
		"E28" bad host_name

		or $::OK  on success - this is zero

	Sets @::GETHOST_MSG to one of:

		'EMsgBAD_IPADDR'
		'EMsgBAD_HOSTNAME'
	
    Example:

        my $node = $someting; # either ip addres or hostname

        my ($host_name, $ip_address) = NetworkUtils->getHost($node);

	#  if you're not concerned with status and just want to know
	#  if things worked out, use something like this:

        # retun values are undef on error
        if ( $::GETHOST_RC != $::OK )
	{
	    next;
	}
 	# otherwise the $nodename and $nodeip are good.

    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub getHost_rveg
{
    my ($class, $arg) = @_;
    my ($hostname, $ipaddr, $packedaddr);

    # glob status
    $::GETHOST_RC  = $::OK;    # reassigned on error
    $::GETHOST_MSG = undef;    # reassigned on error

    if ($NodeUtils::gethost{$arg})
    {

        # print "returning cached values \n";
        $hostname = $NodeUtils::gethost{$arg}{'Name'};
        $ipaddr   = $NodeUtils::gethost{$arg}{'IP'};

        return ($hostname, $ipaddr);
    }

    # do the IP address parameter

    if (NetworkUtils->isIpaddr($arg))
    {
        $ipaddr = $arg;

        if (defined $NodeUtils::Hosts{'IP'}{$arg})
        {

            # print "returning cached IP: $ipaddr\n";
            $hostname = $NodeUtils::Hosts{'IP'}{$ipaddr};

            return ($hostname, $ipaddr);

        }

        # IP not cached
        $packedaddr = inet_aton($ipaddr);
        $hostname   = gethostbyaddr($packedaddr, AF_INET);

        #if ip cann't be resolved to hostname,then ip will be the hostname.
        if (!$hostname)
        {
            $hostname = $ipaddr;
        }

        # cache the  values
        $NodeUtils::gethost{$arg}{'Name'} = $hostname;
        $NodeUtils::gethost{$arg}{'IP'}   = $ipaddr;

        return ($hostname, $ipaddr);
    }

    # do the hostname parameter

    $hostname = $arg;

    if (defined $NodeUtils::Hosts{'Hostname'}{$hostname})
    {

        # print "using cached Hostname: $hostname\n";
        return ($hostname, $ipaddr);
    }

    # high overhead call...
    my ($name, $aliases, $addrtype, $length, @addrs) = gethostbyname($hostname);

    if (!$name && !$ENV{'HAMS_DEFINE_NODE'})    # return on error
    {
        $::GETHOST_RC  = 28;
        $::GETHOST_MSG = 'EMsgBAD_HOSTNAME';
        return (undef, undef);
    }
    elsif ($ENV{'HAMS_DEFINE_NODE'})
    {

        # when HA MS is running, node can be defined even though it can not be resolved
        return ($hostname, $ipaddr);
    }

    # print "found name=$name, and addrs=$#addrs.\n";

    $packedaddr = $addrs[0];
    $ipaddr     = inet_ntoa($packedaddr);
    $hostname   = $name;                    # may be a short name

    # cache the  values
    $NodeUtils::gethost{$hostname}{'Name'} = $hostname;
    $NodeUtils::gethost{$hostname}{'IP'}   = $ipaddr;
    MessageUtils->messageFromCat(
                              'csmInstall.cat',
                              $::MSGMAPPATH,
                              'csminstall',
                              "E1",
                              'EMsgHostnameShouldNotBeResolveToLoopbackAddress',
                              $hostname,
                              $ipaddr
      )
      if $ipaddr =~ /127\.0\.0/;
    return ($hostname, $ipaddr);

}

#--------------------------------------------------------------------------------

=head3    get_management_server

        Get the hostname of the management server.

        Arguments:
                $node_hostname
        Returns:
                ($hostname, $ipaddr) or $hostname, depending on the value of
                wantarray (?)
        Globals:
                $::GETSOURCEIP2TARGET
        Error:
                undefined
        Example:
                ($ms_hostname, $ipaddr) = NetworkUtils->get_management_server_by_hostname;
        Comments:
                Since the nodes might need to contact the management server through a
                different address than its primary hostname, we cannot just use the output
                of the hostname command here.  Use the getSourceIP2Target command instead.

                If we cannot determine the route to the node via getSourceIP2Target
                just return nothing.  This indicates that there is no way to get to the node.

                Note:  caller is reponsible for verifying return values.

=cut

#--------------------------------------------------------------------------------

sub get_management_server
{
    my ($class, $node_hostname) = @_;
    my $ipaddr   = "";
    my $hostname = "";
    delete $ENV{'CSM_NO_NAME_RES'};    #so that tryHost works

    # Move this to InstallDefs.pm.perl
    $::GETSOURCEIP2TARGET = "/opt/csm/csmbin/getSourceIP2Target";

    my $cmd    = "$::GETSOURCEIP2TARGET $node_hostname";
    my @output = NodeUtils->runcmd($cmd, 0);
    my $rc     = $::RUNCMD_RC;
    if (!$rc)                          # If no errors
    {
        $ipaddr = $output[0];

        if ($ipaddr)
        {
            $hostname = NetworkUtils->tryHost($ipaddr);
        }
    }

    if (wantarray) { return ($hostname, $ipaddr); }
    else { return $hostname; }
}

#--------------------------------------------------------------------------------

=head3    getRSCThostname

        This routine is used to always return the same hostname regardless 
        of hostname resolution settings. It returns the RSCT node id of the 
        current machine.

        Since the RSCT node id is unchangable by the user, the only way
        this could change is if RSCT is uninstalled, and reinstalled.
 

    Arguments:
        none
    Returns:
        $ref from lsrsrc-api call.
    Globals:
        none
    Error:
        none
    Example:
        none
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub getRSCThostname
{

    #my $outref =   NodeUtils->runcmd('CT_MANAGEMENT_SCOPE=1 /usr/bin/lsrsrc-api -s IBM.Host::::NodeNameList');
    #$outref =~ s/{|}//g;
    #now using the RSCT node id because the NodeNameList changes when the output of hostname changes
    my $outref = NodeUtils->runcmd("/usr/sbin/rsct/bin/lsnodeid");
    return $outref;
}

#--------------------------------------------------------------------------------

=head3    groupNFSnodes

        I wonder ...

        Arguments:
        \%DestNodeHash    - InstallServer must be an attr of each node in hash.
          Fanout value
        \@nodeArray    - If caller is the only dsh to a select group
        Returns:
                I wonder...
        Globals:
                none
        Error:
                none
        Example:
         my $fan = NetworkUtils->groupNFSnodes( \%nodeHash, $csmFanout );
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub groupNFSnodes
{
    my ($class, $DestNodeHash, $fanout, $nodes, $skip_nfs_server) = @_;

    #put nodes into hashes based on install server
    my %is;
    my @norm;
    my @slist;
    if ($nodes)
    {
        foreach my $node (@$nodes)
        {
            my $server = $$DestNodeHash{$node}{'InstallServer'};
            if (!$server && !$skip_nfs_server)
            {
                $server = $$DestNodeHash{$node}{'NFSServer'};
            }

            if ($server)
            {    #there is an install server for this node
                    #$is{$server}{$node}=1;
                if (!exists $is{$server})
                {    #haven't scene this one before
                    push @slist, $server;
                }
                push @{$is{$server}}, $node;
            }
            else
            {        #just uses the ms as an install server
                     #$norm{$node}=1;
                push @norm, $node;

                #print "pushed node $node onto norm\n";
            }
        }
    }
    else
    {                #using the entire hash of nodes
        foreach my $node (keys %$DestNodeHash)
        {
            my $server = $$DestNodeHash{$node}{'InstallServer'};
            if (!$server && !$skip_nfs_server)
            {
                $server = $$DestNodeHash{$node}{'NFSServer'};
            }

            if ($server)
            {        #there is an install server for this node
                     #$is{$server}{$node}=1;
                if (!exists $is{$server})
                {    #haven't scene this one before
                    push @slist, $server;
                }
                push @{$is{$server}}, $node;
            }
            else
            {        #just uses the ms as an install server
                     #$norm{$node}=1;
                push @norm, $node;

                #print "pushed node $node onto norm\n";
            }
        }
    }
    my %dsh;
    my $num_servers = (scalar @slist) + 1;
    my $num_nodes;
    if ($nodes) { $num_nodes = scalar @$nodes; }
    else { $num_nodes = scalar keys %$DestNodeHash; }

    #print "the number of install servers is $num_servers -1\nthe number of nodes is $num_nodes\nthe fanout is $fanout\n";
    my $count    = 0;
    my $group    = 0;
    my $fancount = 0;
  GROUPNODES: while ($num_nodes)
    {

        #print "in GROUPNODES list: count=$count\tgroup=$group\tnum_nodes=$num_nodes\n";
        if ($fancount == $fanout)
        {    #we are done with one list
            $fancount = 0;
            $group++;
        }
        if ($count >= $num_servers)
        {
            $count = 0;
        }
        if ($count == 0)
        {    #grab one from the normal list if any are left
            if (@norm)
            {

                #print "pushing onto group $group\n";
                push @{$dsh{$group}}, shift @norm;
                $num_nodes--;
                $fancount++;
                $count++;
                next GROUPNODES;
            }
        }
        while (!@{$is{$slist[$count - 1]}})
        {    #none in this array
                #print "server $slist[$count] is not defined\n";
                #print "incrementing count $count\n";
            $count++;
            if ($count == $num_servers || !($slist[$count - 1]))
            {
                next GROUPNODES;
            }
        }
        push @{$dsh{$group}}, shift @{$is{$slist[$count - 1]}};
        $fancount++;
        $num_nodes--;
        $count++;
    }
    my %fan = ();
    my $ms_ip;
    foreach my $g (keys %dsh)
    {
        @{$fan{$g}{'ALL'}} = @{$dsh{$g}};
        foreach my $node (@{$dsh{$g}})
        {
            my $server = $$DestNodeHash{$node}{'InstallServer'};
            if (!$server && !$skip_nfs_server)
            {
                $server = $$DestNodeHash{$node}{'NFSServer'};
            }
            my ($s_hostname, $dir) = split(/:/, $server);
            if (!$server)
            {
                $server = $$DestNodeHash{$node}{'ManagementServer'};
            }
            else
            {    #server exists
                my $serverFromGroup =
                  NodeUtils->getNodeFromGroup($s_hostname, $node);
                if ($serverFromGroup)
                {
                    $s_hostname = $serverFromGroup;
                }

                if (!$dir)
                {
                    $dir = $::SERVER_DIRECTORY;
                }
                $server = $s_hostname . ":" . $dir;

                my (undef, $node_ip) =
                  NetworkUtils->getHost_rveg($$DestNodeHash{$node}{'Hostname'});
                my (undef, $is_ip) = NetworkUtils->getHost_rveg($s_hostname);
                if (!$ms_ip)
                {
                    (undef, $ms_ip) =
                      NetworkUtils->getHost_rveg(
                                     $$DestNodeHash{$node}{'ManagementServer'});
                }
                if ($node_ip eq $ms_ip)
                {

                    #this node is also the MS
                    $server = $$DestNodeHash{$node}{'ManagementServer'};
                }
                elsif ($is_ip eq $ms_ip)
                {
                    $server = $$DestNodeHash{$node}{'ManagementServer'};
                }
            }
            push @{$fan{$g}{'Server'}{$server}}, $node;
        }
        my @s = keys %{$fan{$g}{'Server'}};

        #print "group $g has nodes @{$dsh{$g}}\tand servers @s \n";
    }
    return (\%fan);
}

#--------------------------------------------------------------------------------

=head3    nfsprobe

         Probe NFS status like the "nfs probe" command on redhat

        Arguments:
                none
        Returns:
                NFS service command (e.g. start, restart or reload).
        Globals:
                none
        Error:
                retCode from NodeUils->service("nfs","probe");
        Example:
                $serviceCmd = NetworkUtils->nfsprobe();

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub nfsprobe
{
    my $CHECKPROC = "/sbin/checkproc";
    my $PIDOF     = "/sbin/pidof";
    my $NFSD      = "nfsd";
    my $MOUNTD    = "/usr/sbin/rpc.mountd";
    my $EXPORTS   = "/etc/exports";
    my $ETAB      = "/var/lib/nfs/etab";
    my ($nfsd_status, $mountd_status);
    my $cmd = NetworkUtils->service("nfs", "probe");

    #my $output=NodeUtils->runcmd($cmd,0);
    my $output = `$cmd 2>&1`;
    if (!$?)
    {
        return $output;
    }

    NodeUtils->runcmd("$CHECKPROC -n $NFSD", -1);
    $nfsd_status = $::RUNCMD_RC;

    NodeUtils->runcmd("$CHECKPROC $MOUNTD", -1);
    $mountd_status = $::RUNCMD_RC;

    #either nfsd and mountd are not start
    if ($nfsd_status || $mountd_status)
    {
        return "start";
    }

    NodeUtils->runcmd("$PIDOF $NFSD", -1);
    $nfsd_status = $::RUNCMD_RC;

    NodeUtils->runcmd("$PIDOF $MOUNTD", -1);
    $mountd_status = $::RUNCMD_RC;

    if ($nfsd_status == 1 || $mountd_status == 1)
    {
        return "restart";
    }

    my $exports_mtime = (stat($EXPORTS))[9];
    my $etab_mtime    = (stat($ETAB))[9];

    if ($exports_mtime > $etab_mtime)
    {
        return "reload";
    }

    return "status";
}

#--------------------------------------------------------------------------------

=head3    service

    Send a service request to an init script.

    Arguments:
        $service  - a service name such as 'inetd','xinetd'
        $svcarg   - arguments for the service such as 'start',
                    'stop' or 'status'. 
    Returns:
        A full cli for the service script.
    Globals:
        none
    Error:
        none
    Example:
        none
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub service
{
    my ($class, $service, $svcarg) = @_;

    my $cmd;
    my $SVCCLI = "/sbin/service";
    my $SVCDIR = "/etc/init.d";
    my $distro = NodeUtils->distribution();

    #  On SLES, nfs server script is "nfsserver".
    if (($distro =~ /SLES/ || NodeUtils->isHMC()) && $service eq "nfs")
    {
        $service = "nfsserver";
    }

    if (-f $SVCCLI)
    {
        $cmd = "$SVCCLI $service $svcarg ";
    }
    else
    {
        $cmd = "$SVCDIR/$service $svcarg";
    }

    return $cmd;
}

#--------------------------------------------------------------------------------

=head3    start_nfs_AIX

        Starts NFS services on AIX Management Server

        Arguments:
                none
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
                $::SERVICE
        Error:
                $::NOK
        Example:
                if (NetworkUtils->start_nfs_AIX != 0) { blah; }_
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub start_nfs_AIX
{
    my $rc            = 0;
    my $rpc_mountd_up = 0;
    my $nfsd_up       = 0;
    my @output = NodeUtils->runcmd("LANG=C /usr/bin/lssrc -s rpc.mountd", -1);
    if (!$::RUNCMD_RC)
    {
        my ($subsys, $group, $pid, $status) = split(' ', $output[1]);
        if (defined($status) && $status eq 'active')
        {
            $rpc_mountd_up = 1;
        }
    }

    @output = NodeUtils->runcmd("LANG=C /usr/bin/lssrc -s nfsd", -1);
    if (!$::RUNCMD_RC)
    {
        my ($subsys, $group, $pid, $status) = split(' ', $output[1]);
        if (defined($status) && $status eq 'active')
        {
            $nfsd_up = 1;
        }
    }

    # Only call mknfs if rpc.mountd or nfsd are not active
    if ($rpc_mountd_up == 0 || $nfsd_up == 0)
    {
        my $cmd = "$::MKNFS -N  >/dev/null 2>&1";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        $rc = system($cmd) >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
    }
    return $rc;
}

#--------------------------------------------------------------------------------

=head3    start_nfs

        Starts NFS services on the Linux ManagementServer

        Arguments:
                none
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
                $::SERVICE
        Error:
                $::NOK
        Example:
                if (NetworkUtils->start_nfs_Linux != 0) { blah; }_
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub start_nfs_Linux
{
    # Initial $::RUNCMD_RC in case the $output return null;
    $::RUNCMD_RC = 0;
    # SLES Change
    my $output = NetworkUtils->nfsprobe();

    if ($output)
    {
        #start portmap when on HMC
        if(NodeUtils->isHMC())
	    {
		    my $cmd = NetworkUtils->service("portmap", $output);

        	MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        	NodeUtils->runcmd($cmd, -1);
        	if ($::RUNCMD_RC)
        	{
            	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCommandFailed', $cmd, $::RUNCMD_RC);
        	}
        }
        # SLES Change
        my $cmd = NetworkUtils->service("nfs", $output);

        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        NodeUtils->runcmd($cmd, -1);
        if ($::RUNCMD_RC)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCommandFailed', $cmd, $::RUNCMD_RC);
        }
        else
        {

            #wait for nfsserver to start on SLES7
            sleep(1);
        }
    }
    return $::RUNCMD_RC;
}

#--------------------------------------------------------------------------------

=head3    start_nfs

        Starts NFS services on the managment server

        Arguments:
				%NodeHash
				%ISHash
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
                $::SERVICE
        Error:
                $::NOK
        Example:
                if (NetworkUtils->start_nfs != 0) { blah; }_
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub start_nfs
{
    my ($class, $ref_nodehash, $ref_ishash) = @_;
    my ($cmd, $output);
    my $rc      = 0;
    my $exports = "/etc/exports";

    my %nodehash = %$ref_nodehash;
    my %ishash   = %$ref_ishash;

    if ($::MS_INSTALL_SERVER || (!keys %ishash))
    {

        #the MS is serving NFS files
        if (!-s $exports)
        {

            # nfs will only start if /etc/exports has a size > 0
            open(EXPORTS, ">$exports")
              || MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCANT_READ_FILE', $exports);
            print EXPORTS "\n";
            close(EXPORTS);
        }

        # Start nfs if needed.
        if ($::PLTFRM eq "AIX")
        {
            $rc = NetworkUtils->start_nfs_AIX();
        }
        else
        {
            if (NodeUtils->isHMC() && ($ENV{'DC_ENVIRONMENT'} ne 1))
            {
                my $cmd = "/usr/sbin/rsct/bin/runlpcmd nfs_start_stop start";
                NodeUtils->runcmd($cmd, -1);
            }
            else
            {
                $rc = NetworkUtils->start_nfs_Linux();
            }
        }
    }

    if (defined %ishash)
    {
        my @make_ex        = ();
        my @AIX_servers    = ();
        my @RedHat_servers = ();
        my @SUSE_servers   = ();
        my %service;

        my @servers = keys %ishash;
        foreach my $is (@servers)
        {
            my $os     = $ishash{$is}{'InstallOSName'};
            my $distro = $ishash{$is}{'InstallDistributionName'};
            my ($effective_distro_name) = NodeUtils->getEffectiveDistro($distro);

            if ($os =~ /AIX/)
            {
                push @AIX_servers, $is;
            }
            elsif ($os eq "Linux")
            {
                if ($effective_distro_name =~ m/RedHat/)
                {
                    push @RedHat_servers, $is;
                }
                elsif ($effective_distro_name =~ m/SLES/)
                {
                    push @SUSE_servers, $is;
                }
            }
        }
        if (@RedHat_servers)
        {
            my $list = join ',', @RedHat_servers;
            my $cmd =
              "LANG=C -n $list \"$::LS -s $exports; /sbin/service nfs probe;\"";
	    my $ENV_OLD = $ENV{'LANG'};
	    $ENV{'LANG'} = "C";
	    my %options_api = ();
	    $options_api{'nodes'} = $list;
	    $options_api{'command'} = "$::LS -s $exports; /sbin/service nfs probe;";
            my @output = NodeUtils->runDsh(\%options_api, -1);
            $ENV{'LANG'} = $ENV_OLD;
            foreach my $line (@output)
            {
                chomp $line;
                my ($server, $data) = split ":", $line, 2;
                if ($line =~ m/$exports/)
                {    #this is the ls line
                    if ($line !~ m/"does not exist"/ || $line !~ m/"not found"/)
                    {
                        $data =~ s/^\s+//;    #remove any leading whitespace
                        my ($size, $file) = split /\s+/, $data, 2;
                        if ($size == 0)
                        {
                            push @make_ex, $server;
                        }
                    }
                    else
                    {                         #/ext/exports does not exist
                        push @make_ex, $server;
                    }
                }
                elsif (   ($data =~ m/start/)
                       || ($data =~ m/restart/)
                       || ($data =~ m/reload/))
                {
                    $data =~ s/^\s+//;        #remove any leading whitespace
                    push @{$service{$data}}, $server;
                }
            }
        }    #RedHatServers

        if (@AIX_servers || @SUSE_servers)
        {
            my $list = join ',', @SUSE_servers, @AIX_servers;

            #make sure /exports is larger than 0
            my $cmd = "LANG=C -n $list \"$::LS -s $exports;\" 2>&1";
            my $ENV_OLD = $ENV{'LANG'};
            $ENV{'LANG'} = "C";
            my %options_api = ();
            $options_api{'nodes'} = $list;
            $options_api{'command'} = "$::LS -s $exports;";
            my @output = NodeUtils->runDsh(\%options_api, -1);
            $ENV{'LANG'} = $ENV_OLD;
            foreach my $line (@output)
            {
                chomp $line;
                my ($server, $data) = split ":", $line, 2;
                if ($line =~ m/$exports/)
                {    #this is the ls line
                    if ($line !~ m/"does not exist"/ || $line !~ m/"not found"/)
                    {

                        #then /etc/exports exists
                        $data =~ s/^\s+//;    #remove any leading whitespace
                        my ($size, $file) = split /\s+/, $data, 2;
                        if ($size == 0)
                        {
                            push @make_ex, $server;
                        }
                    }
                    else
                    {                         #no file
                        push @make_ex, $server;
                    }
                }
            }
        }

        if (@make_ex)
        {
            my $ex_list = join ",", @make_ex;
            $cmd = "";
	    my %options_api = ();
	    $options_api{'nodes'} = $ex_list;
	    $options_api{'command'} = "/bin/echo ' ' >> $exports";
            NodeUtils->runDsh(\%options_api, 0);
        }

        if (@RedHat_servers)
        {
            my $remoteshell = NetworkUtils->getRemoteShell();
            (my $cmd_basename = $remoteshell) =~ s:^.*/::;
            foreach my $value (keys %service)
            {
                my $list = join ",", @{$service{$value}};
		my %options_api = ();
		$options_api{'nodes'} = $list;
                if (($cmd_basename eq "rsh") && 
                    (($value eq "start") || ($value eq "restart")))
                {
                    #To workaround a rsh bug
                    $options_api{'command'} = "/sbin/service nfs $value < /dev/null";
                }
                else
                {
                    $options_api{'command'} = "/sbin/service nfs $value";
                }
                NodeUtils->runDsh(\%options_api, 0);
            }
        }

        if (@AIX_servers)
        {
            my $list = join ',', @AIX_servers;
	    my %options_api = ();
	    $options_api{'nodes'} = $list;
	    $options_api{'command'} = "$::MKNFS -B >/dev/null";
            NodeUtils->runDsh(\%options_api, 0);
        }

        if (@SUSE_servers)
        {

            #going to need to run sed command to edit /etc/rc.config on SUSE server
            my $list      = join ',', @SUSE_servers;
            my $RCCONFIG  = "/etc/rc.config";
            my $CHECKPROC = "/sbin/checkproc";
            my $PIDOF     = "/sbin/pidof";
            my $NFSD      = "nfsd";
            my $MOUNTD    = "/usr/sbin/rpc.mountd";
            my $EXPORTS   = "/etc/exports";
            my $ETAB      = "/var/lib/nfs/etab";
            my %options_api = ();
	    $options_api{'nodes'} = $list;
            $cmd =
              qq($CHECKPROC -n $NFSD; echo \"nfsd proc rc=\$?\"; $CHECKPROC $MOUNTD; /bin/echo \"mountd proc rc=\$?\"; $PIDOF $NFSD; echo \"pidof nfsd rc=\$?\"; $PIDOF $MOUNTD; echo \"pidof mountd rc=\$?\";);
	    $options_api{'command'} = $cmd;
            my @output = NodeUtils->runDsh(\%options_api, -1);
            my %service_hash;
            my %suse_service = ();
            foreach my $line (@output)
            {
                chomp $line;
                my $hostname;
                ($hostname, $line) = split ':', $line, 2;
                my $rc = (split '=', $line)[1];
                if ($line =~ m/nfsd proc/)
                {
                    $service_hash{$hostname}{'NFSD'}{'PROC'} = $rc;
                }
                elsif ($line =~ m/mountd proc/)
                {
                    $service_hash{$hostname}{'MOUNTD'}{'PROC'} = $rc;
                }
                elsif ($line =~ m/pidof nfsd/)
                {
                    $service_hash{$hostname}{'NFSD'}{'PIDOF'} = $rc;
                }
                elsif ($line =~ m/pidof mountd/)
                {
                    $service_hash{$hostname}{'MOUNTD'}{'PIDOF'} = $rc;
                }
                elsif ($line =~ m/$EXPORTS/)
                {
                    $service_hash{$hostname}{'EXPORTS'} = $rc;
                }
                elsif ($line =~ m/$ETAB/)
                {
                    $service_hash{$hostname}{'ETAB'} = $rc;
                }
            }
            foreach my $hostname (keys %service_hash)
            {
                my $value;
                if (   $service_hash{$hostname}{'NFSD'}{'PROC'}
                    || $service_hash{$hostname}{'MOUNTD'}{'PROC'})
                {
                    $value = "start";
                }
                elsif (   $service_hash{$hostname}{'NFSD'}{'PIDOF'} == 1
                       || $service_hash{$hostname}{'MOUNTD'}{'PIDOF'} == 1)
                {
                    $value = "restart";
                }
                elsif ($service_hash{$hostname}{'EXPORTS'} >
                       $service_hash{$hostname}{'ETAB'})
                {
                    $value = "reload";
                }
                else
                {
                    $value = "status";
                }

                push @{$suse_service{$value}}, $hostname;
            }
            foreach my $value (keys %suse_service)
            {
                my $list = join ',', @{$suse_service{$value}};
		my %options_api = ();
		$options_api{'nodes'} = $list;
		$options_api{'command'} = "PATH=\$PATH:/sbin ; /etc/init.d/nfsserver $value;";
                NodeUtils->runDsh(\%options_api, 0);
            }
        }
    }
    return ($rc);
}

#--------------------------------------------------------------------------------

=head3    tryHost

    Try to return the primary hostname for the given hostname or
    ip address.  If resolution fails, return the argument passed in.
        

    Arguments:
        hostname and/or ip address as a string
    Returns:
        primary hostname as a string or given argument if
        the hostname can's be resolved
 
    Globals:
        $NodeUtils::Hosts
        
    Error:
        undefined
    Example:
         my $hostname =
            NetworkUtils->tryHost($node, \$resolved);
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub tryHost
{
    my ($class, $arg, $resolvedref) = @_;
    if (defined $ENV{'CSM_NO_NAME_RES'})
    {    #skip name res
        $resolvedref && ($$resolvedref = 1);
        return $arg;
    }

    if (!defined $NodeUtils::tryhost{$arg})
    {
        my $host;
        $resolvedref && ($$resolvedref = 0);
        my $res = 0;    #for caching purposes
        if (NetworkUtils->isIpaddr($arg))
        {
            if (defined $NodeUtils::Hosts{'IP'}{$arg})
            {

                #cached from /etc/hosts
                #print "using cached IP: $arg\n";
                $host = $NodeUtils::Hosts{'IP'}{$arg};
            }
            else
            {
                my $packedaddr = inet_aton($arg);
                $host = gethostbyaddr($packedaddr, AF_INET);
                if (!length($host)) { $host = $arg; }
            }
            $resolvedref && ($$resolvedref = 1);
            $res = 1;
        }
        else    # they specified a hostname, but it may be a short name
        {
            if (defined $NodeUtils::Hosts{'Hostname'}{$arg})
            {

                #cached from /etc/hosts
                #print "using cached hostname: $arg\n";
                $host = $NodeUtils::Hosts{'Hostname'}{$arg}{'Name'};
                $resolvedref && ($$resolvedref = 1);
                $res = 1;
            }
            else
            {
                my ($hostname, $aliases, $addrtype, $length, @addrs) =
                  gethostbyname($arg);
                if (length($hostname))
                {
                    $host = $hostname;
                    $resolvedref && ($$resolvedref = 1);
                    $res = 1;
                }
                else { $host = $arg; }
            }
        }
        $NodeUtils::tryhost{$arg}{'Name'} = $host;    #cache this for later
        if ($resolvedref)
        {
            $NodeUtils::tryhost{$arg}{'resolved'} = $$resolvedref;
        }
        else { $NodeUtils::tryhost{$arg}{'resolved'} = $res; }
        return $host;
    }
    else
    {
        $resolvedref && ($$resolvedref = $NodeUtils::tryhost{$arg}{'resolved'});
        return $NodeUtils::tryhost{$arg}{'Name'};     #use cached version
    }
}

#--------------------------------------------------------------------------------

=head3    unexport_csminstall

        NFS unexport the csminstall directory on the Management Server and Install Server

        Arguments:
				$ref_nodehash:		Node Hash
				$ref_ishash:		nodehash's InstallServer/NFSServer Hash
                $ref_target_dir:	Reference to unexported target dir array 
        Returns:
                $::OK
                $::NOK
        Error:
                $::NOK
        Example:
                if(NetworkUtils->unexport_csminstall(\%nodehash, \%ishash) != 0) {
                    blah;
                }
        Comments:
			If no %nodehash or %ishash, default to unexport MS's dir

=cut

#--------------------------------------------------------------------------------

sub unexport_csminstall
{
	my ($class, $ref_nodehash, $ref_ishash, $ref_target_dir) = @_;

	return NetworkUtils->export_csminstall($ref_nodehash, $ref_ishash, $ref_target_dir, 1);
}

#--------------------------------------------------------------------------------

=head2    Remote Shells - RSH, SSH 

=cut

#--------------------------------------------------------------------------------

=head3    checkHOMEforSSHperm

        Checks attributes of root home directory.

        Arguments:
                none
        Returns:
                1 - if $HOME has the wrong permissions
                0 - if permissions are correct.
        Globals:
                none
        Error:
                1 - wrong permissions on root's  $HOME
        Example:
                $rc = NetworkUtils->checkHOMEforSSHperm();
        Comments:
                If the permessions on $HOME are incorrect, then SSH
                should not be setup.

=cut

#--------------------------------------------------------------------------------

sub checkHOMEforSSHperm
{
    my $home      = NetworkUtils->getRootHomeDir();
    my $mode      = (stat($home))[2];
    my $real_mode = sprintf "%04o", $mode & 07777;
    my @perm      = split '', $real_mode;
    if (   ($perm[2] == 2)
        || ($perm[2] == 6)
        || ($perm[2] == 7)
        || ($perm[3] == 2)
        || ($perm[3] == 6)
        || ($perm[3] == 7))
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'E',
                                     'EMsgWrongHomePermissions', $home);
        return 1;
    }
    else
    {
        return 0;
    }
}

#--------------------------------------------------------------------------------

=head3    checkRSReachability

        Checks a list of machines (they do not need to be CSM nodes) to make
        sure they can be reached via dsh.

        Arguments:
                reference to an array of hostnames
        Returns:
                reference to an array of unreachable nodes
        Globals:
                $::REMOTE_SHELL
        Error:
                none
        Example:
                 my $unreachable =
                   NetworkUtils->checkRSReachability(\@install_servers);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub checkRSReachability
{

    my $routine = "checkRSReachability";
    print "ENTERING: $routine\n" if $::DEBUG;
    my ($class, $machines, $device_flag, $unaccess_nodes) = @_;

    #print "CHECKING MACHINES: @$machines\n";
    $::REMOTE_SHELL = NetworkUtils->getRemoteShell();

    #put all dest nodes in tmp file
    my $file = ServerUtils->make_node_list_file($machines);
    my $machinesstr = join(",", @$machines);

    my $cmd;
    (my $cmd_basename = $::REMOTE_SHELL) =~ s:^.*/::;
    if ($cmd_basename ne "rsh" && (NetworkUtils->supportedSSHversion()))
    {    #can do SSH with options
        if ($device_flag)
        {
            #remoteshell.expect -dsd now accepts machines list
            $cmd = "$::REMOTESHELL_EXPECT -dsd $machinesstr";
        }
        else
        {
            $cmd = "$::REMOTESHELL_EXPECT -ds $file";
        }
    }
    else
    {
        if ($device_flag)
        {
            $cmd = "$::REMOTESHELL_EXPECT -dd $machinesstr";
        }
        else
        {
            $cmd = "$::REMOTESHELL_EXPECT -d $file";
        }
    }
    my $value = NetworkUtils->supportedSSHversion();
    MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                 'V', 'IMsgCMD', $cmd);
    my @output = `$cmd 2>&1`;
    ServerUtils->close_delete_file($::NODE_LIST_FILE, $file);  #delete node file
    my %reachable;
    my @unreachable;
    my @success_nodes;

    foreach my $line (@output)
    {
        chomp $line;
        my @parts = split ":", $line;

        my ($fqdn, $ip) = NetworkUtils->getHost_rveg($parts[0]);
        if ($::GETHOST_RC != $::OK)
        {
            next;
        }
        if ($parts[1] =~ m/test.success/)
        {    #this node is reachable
            $reachable{$ip} = 1;
            push @success_nodes, $fqdn;

        }

        # For down Nodes
        elsif (
              $unaccess_nodes
              and (  $line =~ m/not responding/
                  || ($line =~ m/Connection refused/ and $cmd_basename ne "rsh")
                  || $line =~ m/Timed out/)
          )
        {
            $reachable{$ip} = -1;
        }
        else
        {
            $reachable{$ip} = 0;
        }
    }

    foreach my $n (@$machines)
    {
        if (grep(/^$n$/, @success_nodes))
        {
            next;
        }
        my ($fqdn, $ip) = NetworkUtils->getHost_rveg($n);
        if ($::GETHOST_RC != $::OK)
        {
            push @unreachable, $n;
            if ($unaccess_nodes)
            {
                push @$unaccess_nodes, $n;
            }
            next;
        }

        if (defined $reachable{$ip} and $reachable{$ip} == 0)
        {
            push @unreachable, $n;
        }
        elsif ($reachable{$ip} != 1)
        {
            push @unreachable, $n;
            if ($unaccess_nodes)
            {
                push @$unaccess_nodes, $n;
            }
        }
    }
    if (@unreachable)
    {
        print "LEAVING: $routine\n" if $::DEBUG;
        return (\@unreachable);
    }
    print "LEAVING: $routine\n" if $::DEBUG;
    return (0);
}

#--------------------------------------------------------------------------------

=head3    cpSSHFiles

        Returns the locations of authorized keys in /csminstall.

        Arguments:
                none
        Returns:
                ($authorizedKeys1, $authorizedKeys2)
        Globals:
                none
        Error:
                returns 1 or exits on message E value
        Example:
                $rc = NetworkUtils->cpSSHFiles();

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub cpSSHFiles
{
    my ($cmd, $rc);
    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 'V', 'IMsgCopySSHKeys');
    my $SSHdir = "/csminstall/csm/config/.ssh";
    if (!(-d "/csminstall/csm/config"))
    {
        $cmd = "$::MKDIR /csminstall/csm/config";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->message(
                                  'E',
                                  'EMsgNO_CreateDir',
                                  "/csminstall/csm/config"
                                  );
        }
    }
    if (!(-d "$SSHdir"))
    {
        $cmd = "$::MKDIR $SSHdir";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                'csminstall', "E", 'EMsgNO_CreateDir', $SSHdir);
        }
    }
    my $home             = NetworkUtils->getRootHomeDir();
    my $authorized_keys  = "$SSHdir/authorized_keys";
    my $authorized_keys2 = "$SSHdir/authorized_keys2";
    if (   !(-e "$home/.ssh/identity.pub")
        || !(-e "$home/.ssh/id_rsa.pub")
        || !(-e "$home/.ssh/id_dsa.pub"))
    {
        return 1;
    }

    $cmd = "$::COPY $home/.ssh/identity.pub $authorized_keys";
    MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                 'V', 'IMsgCMD', $cmd);
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                     $rc);
    }

    $cmd = "$::COPY $home/.ssh/id_rsa.pub $authorized_keys2";
    MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                 'V', 'IMsgCMD', $cmd);
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                     $rc);
    }

    $cmd = "$::CAT $home/.ssh/id_dsa.pub >> $authorized_keys2";
    MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                 'V', 'IMsgCMD', $cmd);
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                     $rc);
    }

    if (!defined $::HAMODE)
    {
        my $outref =
          NodeUtils->runrmccmd('lsrsrc-api', "-i", "-s IBM.DmsCtrl::::HAMode");
        $::HAMODE = $$outref[0];
    }
    if ($::HAMODE == 1)
    {    #on an active HAMS management server
            #get management server attribute for node on other ms
        require HAMS;
        my $inactivems = HAMS->determineInactive();
        if ($inactivems ne "")
        {
            NodeUtils->runcmd(
                "/opt/csm/bin/dsh -v -n $inactivems /opt/csm/csmbin/remoteshell.expect -k",0);  # inactive ms may be down
          if ($::RUNCMD_RC == 0){  # if not down then use keys

            #my $auth_key = "$SSHdir/inactivems_auth";
            #my $auth_key2 = "$SSHdir/inactivems_auth2";
            if (!defined $::REMOTECOPYCMD)
            {
                NetworkUtils->getRemoteShell();
            }    #sets REMOTECOPYCMD
            NodeUtils->runcmd(
                          "$::REMOTECOPYCMD $inactivems:.ssh/\*.pub $SSHdir/.");
            NodeUtils->runcmd(
                             "$::CAT $SSHdir/identity.pub >> $authorized_keys");
            NodeUtils->runcmd("$::CAT $SSHdir/id_rsa.pub >> $authorized_keys2");
            NodeUtils->runcmd("$::CAT $SSHdir/id_dsa.pub >> $authorized_keys2");
            NodeUtils->runcmd("$::RM -f $SSHdir/\*.pub");
          }
        }
    }

    if (!(-e "$authorized_keys") || !(-e "$authorized_keys2"))
    {
        return 1;
    }
    return ($authorized_keys, $authorized_keys2);
}

#--------------------------------------------------------------------------------

=head3    getRemoteShell

        Get the path of the remote shell if avialble.

        Arguments:
                none
        Returns:
                path of the remote shell
        Globals:
                sets $::REMOTE_SHELL, $::SETUP_REMOTE_SHELL, $::REMOTECOPYCMD
        Error:
                1 - couldn't connect to remote shell
        Example:
                $::REMOTE_SHELL = NetworkUtils->getRemoteShell();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getRemoteShell
{
    require DCUtils;
    if(DCUtils->getRemoteShell)
    {
        return DCUtils->getRemoteShell;
    }    
    if (NodeUtils->isMgmtSvr())
    {    #only defined on managment server
         #If need to get an attribute from IBM.DmsCtrl
         #we get all cluster attributes once
         #    using ServerUtils->get_cluster_attr function
         #and set all global variables
         #avoiding muticall lsrsrc-api command in each "get" subroutine.
         ServerUtils->get_cluster_attr;
         #$::REMOTE_SHELL, $::SETUP_REMOTE_SHELL, $::REMOTECOPYCMD
         #have been set in ServerUtils->get_cluster_attr

        if (
            !(
                  $ENV{'DSH_NODE_RSH'}
               || $ENV{'DSH_REMOTE_SHELL'}
               || $ENV{'DSH_REMOTE_CMD'}
            )
          )
        {
            return $::REMOTE_SHELL;
        }
        else
        {
            return $ENV{'DSH_NODE_RSH'}
              || $ENV{'DSH_REMOTE_SHELL'}
              || $ENV{'DSH_REMOTE_CMD'};
        }
    }
    return 1;
}

#--------------------------------------------------------------------------------

=head3    getRootHomeDir

        Get the path the root user home directory from /etc/passwd.

        Arguments:
                none
        Returns:
                path to root user home directory.
        Globals:
                none
        Error:
                none
        Example:
                $myHome = NetworkUtils->getRootHomeDir();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getRootHomeDir
{
    my $username = "root";
    if (NodeUtils->isHMC() && ($ENV{'DC_ENVIRONMENT'} ne 1))
    {
        $username = $ENV{'USER'} ? $ENV{'USER'} : "hscroot";
    }
    my @root = split ':', (`/bin/grep ^$username /etc/passwd 2>&1`);
    my $home = $root[5];
    return $home;
}

#--------------------------------------------------------------------------------

=head3    getSetupRemoteShell

        Get the value of SetupRemoteShell in CSM database.

        Arguments:
                none
        Returns:
                value from database
        Globals:
                none
        Error:
                undefined
        Example:
                $::SETUP_REMOTE_SHELL =
                    NetworkUtils->getSetupRemoteShell();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getSetupRemoteShell
{

    # set local scope because we only want classes on the mgmt svr
    $ENV{'CT_MANAGEMENT_SCOPE'} = 1;

    # todo: remove when lsrsrc-api converts to above
    $ENV{'CT_SESSION_SCOPE'} = 1;
    if (NodeUtils->isMgmtSvr())
    {    #only defined on managment server
        my $outref =
          NodeUtils->runrmccmd('lsrsrc-api', "-i",
                               "-s IBM.DmsCtrl::::SetupRemoteShell");
        return $$outref[0];
    }
}

#--------------------------------------------------------------------------------

=head3    installSSHFiles

        Installs SSH files

        Arguments:
                $filename_of_authorized_keys
                $filename_of_authorized_keys2
        Returns:
                $::OK
        Globals:
                none
        Error:
                undefined
        Example:
                NetworkUtils->installSSHFiles ( $authorized_keys,
                                                $authorized_keys2);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub installSSHFiles
{
    my ($class, $authorized_keys, $authorized_keys2) = @_;
    my ($cmd, $rc);
    my $home                  = NetworkUtils->getRootHomeDir();
    my $dest_dir              = "$home/.ssh";
    my $dest_authorized_keys  = "$dest_dir/authorized_keys";
    my $dest_authorized_keys2 = "$dest_dir/authorized_keys2";

    umask(0077);
    if (!-d "$dest_dir")
    {

        # create a local directory
        $cmd = "$::MKDIR -p $dest_dir";
        $rc  = system("$cmd");
        if ($rc >> 8)
        {

            # Could not create $::CSMINSTDIR directory.
            MessageUtils->message('E', 'EMsgNO_CreateDir', $dest_dir);
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                              'csminstall', "E", 'EMsgNO_CreateDir', $dest_dir);
            return $::NOK;
        }
        else
        {
            chmod 0700, $dest_dir;
        }
    }

    umask(0133);
    if (-e "$dest_authorized_keys")
    {
        $cmd = "$::CAT $authorized_keys >> $dest_authorized_keys";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
    }
    else
    {

        #just copy the keys
        my $cmd = "$::COPY $authorized_keys $dest_authorized_keys";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
        chmod 0644, $dest_authorized_keys;
    }
    if (-e "$dest_authorized_keys2")
    {
        $cmd = "$::CAT $authorized_keys2 >> $dest_authorized_keys2";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
    }
    else
    {

        #just copy the keys
        my $cmd = "$::COPY $authorized_keys2 $dest_authorized_keys2";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
        chmod 0644, $dest_authorized_keys2;
    }
}

#--------------------------------------------------------------------------------

=head3    rmNodesKnownHosts

        Removes SSH known hosts

        Arguments:
                \@node_hostnames
                 file name (optional)
        Returns:
                1 - error
                0 - success
        Globals:
                none
        Error:
                1
        Example:
                NetworkUtils->rmNodesKnownHosts(\@nodes);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub rmNodesKnownHosts
{
    my ($class, $ref_nodes, $tmpKnownHosts) = @_;
    my @node_hostnames = @$ref_nodes;
    my $home           = NetworkUtils->getRootHomeDir();
    my @all_names;
    my $return = 0;
    foreach my $n (@node_hostnames)
    {
        my @parts = split /\./, $n;
        my $name = shift @parts;
        push @all_names, $name;
        foreach my $ext (@parts)
        {
            $name = "$name.$ext";
            push @all_names, $name;
        }

        #get the IP
        my $ip = inet_ntoa(inet_aton($n));
        push @all_names, $ip;
    }

    #create the sed command
    my $sed = "/bin/sed -e ";
    $sed .= "\"";
    foreach my $n (@all_names)
    {
        $sed .= "/^$n\[,| ]/d; ";
    }
    chop $sed;    #get rid of last space
    $sed .= "\"";
    my $sed2 = $sed;
    my $file = "/tmp/$$";
    while (-e $file)
    {
        $file = ServerUtils->CreateRandomName($file);
    }
    if (defined $tmpKnownHosts)
    {             #use a temporary known hosts
        if (-e $tmpKnownHosts)
        {
            $sed .= " $tmpKnownHosts";
            $sed .= " > $file";
            my $printsed = $sed;
            $printsed =~ s/"//g;    #"
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                        'NodeUtils', 'V', 'IMsgCMD', $printsed);
            `$sed 2>&1`;
            my $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCANT_RUN', $printsed, $rc);
            }
            # Do not use cp or mv, it may not work on HMC,
            # Becaused the .ssh dir is owned by root
            my $cp = "$::CAT $file > $tmpKnownHosts";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cp);
            `$cp 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                   'csminstall', 'E', 'EMsgCANT_RUN', $cp, $rc);
            }

            # system("$::RM -f $file");
        }
    }
    else
    {    #use real known hosts
        if (-e "$home/.ssh/known_hosts")
        {
            $sed .= " $home/.ssh/known_hosts";
            $sed .= " > $file";
            my $printsed = $sed;
            $printsed =~ s/"//g;    #"
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                        'NodeUtils', 'V', 'IMsgCMD', $printsed);
            `$sed 2>&1`;
            my $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCANT_RUN', $printsed, $rc);
            }
            my $cp = "$::CAT $file > $home/.ssh/known_hosts";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cp);
            `$cp 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                   'csminstall', 'E', 'EMsgCANT_RUN', $cp, $rc);
            }

            #system("$::RM -f $file");
        }
        if (-e "$home/.ssh/known_hosts2")
        {
            $sed2 .= " $home/.ssh/known_hosts2";
            $sed2 .= " > $file";
            my $printsed2 = $sed2;
            $printsed2 =~ s/"//g;    #"
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                       'NodeUtils', 'V', 'IMsgCMD', $printsed2);
            `$sed2 2>&1`;
            my $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                            'csminstall', 'E', 'EMsgCANT_RUN', $printsed2, $rc);
            }
            my $cp = "$::CAT $file > $home/.ssh/known_hosts2";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cp);
            `$cp 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                   'csminstall', 'E', 'EMsgCANT_RUN', $cp, $rc);
            }

            #system("$::RM -f $file");
        }
    }
    NodeUtils->runcmd("$::RM -f $file", -1);
    return $return;
}

#--------------------------------------------------------------------------------

=head3    setupAIXRSH

        Setup RSH on AIX Managment Server

        Arguments:
                $fullHostNameofManagmentServer
        Returns:
                none
        Globals:
                none
        Error:
                none
        Example:
                NetworkUtils->setupAIXRSH($::csm_server);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub setupAIXRSH
{
    my ($class, $ms, $inactivems) = @_;
    umask(0177);
    my $home = NetworkUtils->getRootHomeDir();
    my ($cmd, $rc);
    $cmd = "/bin/grep \"^$ms root\" $home/.rhosts";
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        $cmd = "/bin/echo $ms root >> $home/.rhosts";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;

        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
    }
    if (defined $inactivems)
    {
        $cmd = "/bin/grep \"^$inactivems root\" $home/.rhosts";
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            $cmd = "/bin/echo $inactivems root >> $home/.rhosts";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
        }
    }
    chmod 0600, "$home/.rhosts";
}

#--------------------------------------------------------------------------------

=head3    setupLinuxRSH

        Setup RSH on Linux Mangement Server

        Arguments:
                $fullHostNameofManagmentServer
        Returns:
        none
        Globals:
                none
        Error:
                undefined
        Example:
                NetworkUtils->setupLinuxRSH($::csm_server);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub setupLinuxRSH
{
    my ($class, $ms, $inactivems) = @_;
    umask(0177);
    my $home = NetworkUtils->getRootHomeDir();
    my ($cmd, $rc);

    # SLES Changes
    # Setup rsh in inetd on SLES
    my $distroname = NodeUtils->get_DistributionName;
    if ($distroname =~ /SLES/)
    {
        `/bin/rpm -q inetd 2>&1`;
        if ($? >> 8)
        {    #setup rsh by xinetd
            $cmd =
              "/bin/sed -e \"s/disable.*=.*yes/disable = no/\" /etc/xinetd.d/rsh";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cmd);
            my @rsh = `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
            open(RSH, ">/etc/xinetd.d/rsh")
              or MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                 'csminstall', "E", 'EMsgCANT_WRITE_FILE', "/etc/xinetd.d/rsh");
            print RSH @rsh;
            close RSH;
            chmod 0600, "/etc/xinetd.d/rsh";
            $cmd = "/etc/init.d/xinetd restart";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;

            if ($rc)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
            if (-e "/etc/securetty")
            {
                $cmd = "/bin/grep rsh /etc/securetty";
                `$cmd 2>&1`;
                $rc = $? >> 8;
                if ($rc)
                {
                    `/bin/echo rsh >> /etc/securetty`;
                }
            }
        }
        else
        {
            $cmd = "/bin/sed -e \"s/#.*shell/shell/\" /etc/inetd.conf";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cmd);
            my @rsh = `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
            open(RSH, ">/etc/inetd.conf")
              or MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                   'csminstall', "E", 'EMsgCANT_WRITE_FILE', "/etc/inetd.conf");
            print RSH @rsh;
            close RSH;
            chmod 0600, "/etc/inetd.conf";
            $cmd = "/etc/rc.d/inetd restart";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;

            if ($rc)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
        }
    }
    else
    {
        $cmd =
          "/bin/sed -e \"s/disable.*=.*yes/disable = no/\" /etc/xinetd.d/rsh";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        my @rsh = `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
        open(RSH, ">/etc/xinetd.d/rsh")
          or MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                 'csminstall', "E", 'EMsgCANT_WRITE_FILE', "/etc/xinetd.d/rsh");
        print RSH @rsh;
        close RSH;
        chmod 0600, "/etc/xinetd.d/rsh";
        $cmd = "/sbin/service xinetd restart";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;

        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
        if (-e "/etc/securetty")
        {
            $cmd = "/bin/grep rsh /etc/securetty";
            `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                `/bin/echo rsh >> /etc/securetty`;
            }
        }
    }

    # End of SLES change

    $cmd = "/bin/grep \"^$ms root\" $home/.rhosts";
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        $cmd = "/bin/echo $ms root >> $home/.rhosts";
        MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
        }
    }
    if (defined $inactivems)
    {
        $cmd = "/bin/grep \"^$inactivems root\" $home/.rhosts";
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            $cmd = "/bin/echo $inactivems root >> $home/.rhosts";
            MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                         'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
        }
    }
    chmod 0600, "$home/.rhosts";
}

#--------------------------------------------------------------------------------

=head3    setupRemoteShellforFullInstall

        Sets up the remote shell across a list of nodes.

        Arguments:
                Reference to an array of nodeNames
        Returns:
                1 - error
                0 - success
        Globals:
                $::REMOTE_SHELL         - shell to configure
                $SETUP_REMOTE_SHELL     - bool: 1 == setup; 0 = don't setup
        Error:
                1
        Example:
                if (NetworkUtils->setupRemoteShellforFullInstall(\@::NODELIST) != 0) {
                    blah;
                }
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub setupRemoteShellforFullInstall
{
    my $rc = 0;

    my ($class, $ref_nodes) = @_;
    my @node_hostnames = @$ref_nodes;

    # SETUP_REMOTE_SHELL controls whether or not to set up the remote shell
    # on the nodes.  It is used when creating the config_file.
    #     0=do not set up remote shell
    #     1=set up whichever remote shell is listed in $::REMOTE_SHELL
    $::SETUP_REMOTE_SHELL = NetworkUtils->getSetupRemoteShell();

    # If the SetupRemoteShell attribute is set to 0 (or null) in DMSCTRL,
    # don't do any remote shell setup.
    if (!$::SETUP_REMOTE_SHELL)
    {
        return $rc;
    }

    $::REMOTE_SHELL = NetworkUtils->getRemoteShell();

    # If the REMOTE_SHELL is set to 1, there was an error getting it from
    # the dmsctrl.  Don't bother setting up the remote shell.
    if ($::REMOTE_SHELL == 1)
    {
        return $::NOK;
    }

    (my $cmd_basename = $::REMOTE_SHELL) =~ s:^.*/::;
    if ($cmd_basename eq "rsh")
    {
        $::SETUP_REMOTE_SHELL = 1;
    }
    else    # Attempt to set up ssh
    {
        if (NetworkUtils->supportedSSHversion())
        {

            # Check permissions on home directory
            $rc = NetworkUtils->checkHOMEforSSHperm();
            if ($rc)
            {
                $::SETUP_REMOTE_SHELL = 0;
                return $rc;
            }

            # Setup ssh keys
            NodeUtils->runcmd("/opt/csm/csmbin/remoteshell.expect -k", 0);

            # Copy ssh keys to /csminstall
            $rc = NetworkUtils->cpSSHFiles();
            if ($rc == 1)
            {
                $::SETUP_REMOTE_SHELL = 0;
                return $rc;
            }

            # Clean up the Known Hosts file for each node so that when the
            # nodes are reinstalled, the hosts files will be regenerated.
            #$rc = NetworkUtils->rmNodesKnownHosts(\@node_hostnames);
            #the rmNodesKnownHosts step was moved to GatherSSHHostKeys

            # ssh was set up properly, so set it up on the nodes.
            $::SETUP_REMOTE_SHELL = 1;

            # Add RemoteShell to config_file
        }
    }

    return $rc;
}

#--------------------------------------------------------------------------------

=head3    supportedSSHversion

        Check if SSH version is supported by CSM install.

        Arguments:
                Requires proper valued in $::REMOTE_SHELL
        Returns:
                1 - remote shell is a supported version 
                0 - unsupported vesion
        Globals:
                $::REMOTE_SHELL
        Error:
                0
        Example:
                my $sshOk = 
                    NetworkUtils->supportedSSHversion();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub supportedSSHversion
{

    #check if its a supported version of OpenSSH
    my $ssh_check = "$::REMOTE_SHELL -V";
    my @ssh_res   = `$ssh_check 2>&1`;
    my $ssh_rc    = $? >> 8;
    if (!$ssh_rc)
    {    #successful
        if ($ssh_res[0] =~ m/OpenSSH.(\d)\.\d/)
        {    #we can set this up
            if ($1 >= 2)
            {
                if (wantarray)
                {
                    return (1, $1);
                }
                else
                {
                    return 1;
                }
            }
        }
        else
        {
            return 0;
        }
    }
}

#--------------------------------------------------------------------------------

=head2    TFTP USER Support

=cut

#--------------------------------------------------------------------------------

=head3    check4InstalledTftpServers

	Checks the current install for tftp rpms

        Arguments:
                none
        Returns:
	       $is_TFTPpackage_installed - is the TFTPpackage RPM installed?
               @tftpRpms  - array of rpms found.
        Globals:
                $::TFTPpackage
		$::default_tftp_server_name
        Error:
                error count
        Example:
                 NetworkUtils->check4InstalledTftpServers();
        Comments:
                none


=cut

#--------------------------------------------------------------------------------

sub check4InstalledTftpServers
{

    # check if there are any tftp rpms installed
    my ($class, $stars) = @_;

    my $is_TFTPpackage_installed = 0;
    my @unknown_tftp_rpms;

    # check and see if any tftp rpms are installed:
    my $cmd = "rpm -qa | grep tftp";
    my @tftpRpms = NodeUtils->runcmd($cmd, -1);
    if ($::RUNCMD_RC eq 0)
    {

        # first check:  If our TFTPpackage RPM is installed, then
        # we can just skip:
        foreach (@tftpRpms)
        {

            # Figure out the basename of the rpm, minus the version and release
            my $rpm_fullname = $_;
            my @parts        = split /-/, $rpm_fullname;
            my $pos          = 0;
            my @basename_parts;
            foreach my $part (@parts)
            {
                if ($pos < ($#parts - 1))
                {
                    push(@basename_parts, $part);
                }
                $pos++;
            }
            my $rpm_basename = join('-', @basename_parts);
            if (grep (/^$rpm_basename$/, $::TFTPpackage))
            {

                # the default was found:
                MessageUtils->message('V', 'IMsgRPM_ALREADY_INSTALLED',
                                      $::TFTPpackage);

                # Indicate that TFTPpackage is already installed.
                $is_TFTPpackage_installed = 1;
            }
            else
            {
                push(@unknown_tftp_rpms, $rpm_fullname);
            }
        }

        if (@unknown_tftp_rpms)
        {

            # Display any unknown tftp RPMs that are installed.
            #MessageUtils->message('V', 'IMsgRPM_ALREADY_INSTALLED', "tftp");
            print "One or more unknown tftp RPMs are already installed:\n";
            foreach (@unknown_tftp_rpms) { print "\t$_\n"; }
        }
    }
    return ($is_TFTPpackage_installed, @tftpRpms);
}

#--------------------------------------------------------------------------------

=head3    chTftpUserPermissions

        Change permissions on TFTP user files.
	Support routine for calls to find.

        Arguments:
                none
        Returns:
               none
        Globals:
               $::EXISTTFTPUSER
        Error:
                error count
        Example:
		 find(\&chTftpUserPermissions, $dir);
        Comments:
 		none

=cut

#--------------------------------------------------------------------------------

sub chTftpUserPermissions
{
    my $user  = "tftpd";
    my $group = "tftpd";

    # Indicate if TFTP User already exists
    $::EXISTTFTPUSER;

    if (!$::EXISTTFTPUSER)
    {
        if (NetworkUtils->testTftpUser($user) == 0)
        {
            NetworkUtils->createTftpUser($user);
            $::EXISTTFTPUSER = 1;
        }
        else
        {
            $::EXISTTFTPUSER = 1;
        }
    }

    my $filename = $File::Find::name;

    if (-f $filename)
    {
        chmod oct( $::filePermissions), $filename;
    }
    elsif (-d $filename)
    {
        chmod oct( $::dirPermissions), $filename;
    }
    elsif (-l $filename)
    {
    }
    else
    {
#it's possible a file is not the one listed above (user may create a special file by mknod),
#but csm should ignore such files
        return ;
    }

    # change owner and group:
    my ($login, $pass, $uid, $gid) = getpwnam($user);
    chown $uid, $gid, $filename;

}

#--------------------------------------------------------------------------------

=head3    configureTftp

        Configure the default tftp server 

        Arguments:
                none
        Returns:
                error count or 0 (success)
        Globals:
                none
        Error:
                error count
        Example:
                 my $eCount = NetworkUtils->configureTftp();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub configureTftp
{
    if ($::PLTFRM eq "AIX")
    {
        my $inetderr = NetworkUtils->configureTftp_inetd();
        return $inetderr;
    }
    else
    {
        my $inetd        = "/etc/init.d/inetd";
        my $xinetd       = "/etc/init.d/xinetd";
        my $tftpConf     = "/etc/xinetd.d/tftp";
        my $xinetdbackup = "/tmp/tftp.bak";
        my $errors       = 0;
        my ($cmd, $output, $rc);

        unless (-f $xinetd)
        {
            if (-f $inetd)
            {
                my $inetderr = NetworkUtils->configureTftp_inetd();
                return $inetderr;
            }
            else
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                            'csminstall', "E", 'EMsgXINETD_MISSING', "$xinetd");
                $errors++;
            }
        }
        # save old file to tmp
        if (-f "$tftpConf")
        {
            $cmd    = "$::COPY -f $tftpConf $xinetdbackup";
            $output = NodeUtils->runcmd($cmd, 0);
            $rc     = $::RUNCMD_RC;
            if ($rc)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'I', 'IMsgShow_Output', $output);
                $errors++;
            }
        }

        # Now, configure our file:
        open(FILE, ">$tftpConf")
          or die MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                          'csminstall', 'E1', 'EMsgCANT_WRITE_FILE', $tftpConf);
        my @fC = <<EOF;
# description: The tftp server serves files using the trivial file transfer \
#       protocol.  The tftp protocol is often used to boot diskless \
#       workstations, download configuration files to network-aware printers, \
#       and to start the installation process for some operating systems.
service tftp
{
        socket_type             = dgram
        protocol                = udp
        wait                    = yes
        user                    = root
        server                  = /usr/sbin/in.tftpd
        server_args             = -vvv -r blksize -p -u tftpd -s /tftpboot
}
EOF
	        foreach (@fC) { print FILE "$_"; }
        close(FILE);
        $errors =
          NetworkUtils->restartTftp($errors, "xinetd");   # restart tftp server.
        return $errors;
    }
}

#--------------------------------------------------------------------------------

=head3    configureTftp_inetd

        Modify /etc/inetd.conf for tftp

        Arguments:
                none
        Returns:
                error count
        Globals:
                none
        Error:
                error count or 0 (success)
        Example:
                 my $inetderr = configureTftp_inetd();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub configureTftp_inetd
{

    my $inetdconf   = "/etc/inetd.conf";
    my $inetdbackup = "/tmp/tftp_inetd.bak";
    my $errors      = 0;
    my ($cmd, $output, $rc, $s);

    unless (-f "$inetdconf") { return ($errors++); }

    # backup inetd.conf
    $cmd    = "$::COPY -f $inetdconf $inetdbackup ";
    $output = NodeUtils->runcmd($cmd, 0);
    $rc     = $::RUNCMD_RC;
    if ($rc)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                     'csminstall', 'I', 'IMsgShow_Output',
                                     $output);
        $errors++;
    }

    # now edit the inetd.conf file:
    $s = NodeUtils->readFile($inetdconf);

    if ($::PLTFRM eq "AIX")
    {
        $s =~ s/\n#*?\s*?(tftp\s+.*?\n)/\n/s;
        $s = $s
          . "tftp     dgram  udp6    SRC     nobody  /usr/sbin/tftpd         tftpd -n\n";
    }
    else
    {
        $s =~ s/^\s*(tftp\s+[\s\S]*$)//m;
        $s = $s
          . "tftp    dgram   udp     wait    root    /usr/sbin/in.tftpd in.tftpd -r blksize -vvv -p -u tftpd -s /tftpboot\n";
    }

    if (open(FILE, ">$inetdconf"))
    {
        print FILE ($s);
        close(FILE);
    }
    else { exit; }

    # Restart inetd to pick up the changes
    $errors =
      NetworkUtils->restartTftp($errors, "inetd");    # restart tftp server.
    return $errors;
}

#--------------------------------------------------------------------------------

=head3    createTftpUser

        Create the TFTP user to secure system directories

        Arguments:
                $tftpuser  - TFTP User Name
        Returns:
                none
        Globals:
                none
        Error:
                V & I Err mesages
        Example:
                 NetworkUtils->createTftpUser();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub createTftpUser
{
    my ($class, $tftpuser) = @_;
    my $tftpgrp    = $tftpuser;
    my $passwdFile = "/etc/passwd";
    my $grpFile    = "/etc/group";

    # determine whether or not a tftpd user already exists:
    my $file = NodeUtils->readFile($passwdFile);
    if (grep(/$tftpuser/, $file))
    {

        #print "tftpd user has already been created.\n" if $::VERBOSE;
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',         $::MSGMAPPATH,
                                     'csminstall',             "V",
                                     'IMsgUserAlreadyCreated', $tftpuser
                                     );
    }
    else
    {

        #print "Creating default tftpd user for the tftp server.\n";
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',       $::MSGMAPPATH,
                                     'csminstall',           "I",
                                     'IMsgCreatingTFTPUser', $tftpuser
                                     );

        # create tftpd user
        # If the last line of passwd file is "+::::::",we must add tftp user before the last line.
        if (grep (/\n\+:*\n/, $file))
        {
            $file =~
              s/\n\+/\n$tftpuser:x:6969:6969::\/dev\/null:\/dev\/null\n\+/;
            open(PASSWD, ">$passwdFile")
              or
              MessageUtils->messageFromCat('csmInstall.cat', 
                      $::MSGMAPPATH, 'csminstall', 'E1', 
                      'EMsgCANT_OPEN', $passwdFile);
            print PASSWD $file;
        }
        else
        {
            open(PASSWD, ">>$passwdFile")
              or
              MessageUtils->messageFromCat('csmInstall.cat', 
                      $::MSGMAPPATH, 'csminstall', 'E1', 
                      'EMsgCANT_OPEN', $passwdFile);
            print PASSWD "$tftpuser:x:6969:6969::/dev/null:/dev/null";
        }
        close(PASSWD);
    }
    $file = NodeUtils->readFile($grpFile);
    if (grep(/$tftpgrp/, $file))
    {

        #print "tftpd group has already been created.\n" if $::VERBOSE;
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',          $::MSGMAPPATH,
                                     'csminstall',              "V",
                                     'IMsgGroupAlreadyCreated', $tftpgrp
                                     );
    }
    else
    {

        #print "Creating default tftpd group for the tftp server.\n";
        MessageUtils->messageFromCat(
                                     'csmInstall.cat',        $::MSGMAPPATH,
                                     'csminstall',            "I",
                                     'IMsgCreatingTFTPGroup', $tftpgrp
                                     );

        # If the last line of group file is "+:::",we must add tftp group before the last line.
        if (grep (/\n\+:*\n/, $file))
        {
            $file =~ s/\n\+/\n$tftpgrp:x:6969:\n\+/;
            open(GROUP, ">$grpFile")
              or
              MessageUtils->messageFromCat('csmInstall.cat', 
                      $::MSGMAPPATH, 'csminstall', 'E1', 
                      'EMsgCANT_OPEN', $grpFile);
            print GROUP $file;
        }
        else
        {
            open(GROUP, ">>$grpFile")
              or
              MessageUtils->messageFromCat('csmInstall.cat', 
                      $::MSGMAPPATH, 'csminstall', 'E1', 
                      'EMsgCANT_OPEN', $grpFile);
            print GROUP "$tftpgrp:x:6969:";
        }
        close(GROUP);
    }
}

#--------------------------------------------------------------------------------

=head3    process_tftp

	Handles the special-case installation of the tftp-hpa rpm 
	(or an alternative tftp rpm).

        Arguments:
                none
        Returns:
                $installTftp - 1 means TFTPpackage must be installed
                               0 means TFTPpackage should not be installed
        Globals:
                $::TFTPpackage
                $::default_tftp_server_name
        Error:
                V & I Err mesages
        Example:
                NetworkUtils->process_tftp();
	Comments:

	The TFTPpackage attribute in the Dmsctrl (csmconfig) class is set
	by default to the same value as $::default_tftp_server_name.  The
	value is the name of the TFTP RPM package (tftp-hpa) to install.
	The user may change it to another RPM name if desired.

	If $::TFTPpackage is the same as $::default_tftp_server_name, then 
	CSM will install and configure it on the install server, and 
	possibly remove any existing tftp services.  
	
	If TFTPpackage is set to something other than default_tftp_server_name,
	then CSM will just ensure that the specified tftp RPM is installed
	on the install server, or that a tftp process (maybe installed from a
	tarball) is running.  If nothing is installed or running, then inform
	the user to install tftp and exit.

	Special Cases for TFTP:
	1.    IF (TFTPpackage installed)
	    THEN do nothing   (set installTftp=0)

	2. ELSIF (TFTPpackage = defaultTFTP)
	     AND (TFTPpackage is not installed)
	    THEN Install TFTPpackage l   (set installTftp=1)

	3. ELSIF (TFTPpackage != defaultTFTP) 
	     AND (TFTPpackage is not installed)
	     AND (something unknown (a tarball?) is running)
	    THEN Just print message and do nothing.  (set installTftp=0)
		    (hopefully unknown running tftp will work)

	4. ELSIF (TFTPpackage != defaultTFTP)
	     AND (TFTPpackage is not installed)
	     AND (nothing is running)
	    THEN print message and exit. (user needs to install tftp)
=cut

#--------------------------------------------------------------------------------

sub process_tftp
{
    my $installTftp = 0;                  # 0 means no, 1 means install it.
    my @vn          = ($::TFTPpackage);
    my ($cmd, $output, $rc, $returncode);
    $returncode = 0;
    my $ans = "";

    my ($is_TFTPpackage_installed, @tftpRpms) =
      NetworkUtils->check4InstalledTftpServers();

    if ($is_TFTPpackage_installed)
    {

        # CASE 1
        # The TFTPpackage RPM is alrady installed.
        # There is no need to install it again, so set installTftp to 0
        # so that TFTPpackage does not get installed.
        ($::DEBUG) && print "TFTPpackage ($::TFTPpackage) already installed\n";
        $installTftp = 0;
    }
    elsif ($::TFTPpackage eq $::default_tftp_server_name)
    {

        # CASE 2
        # The TFTPpackage RPM is not already installed, and we know how to
        # install it since it is the same as the default_tftp_server_name.

        # If any conflicting tftp RPMs are already installed, then remove
        # them before installing TFTPpackage.
        if ($#tftpRpms ne -1)
        {
            ($::DEBUG) && print "Removing conflicting tftp rpms: @tftpRpms\n";
            NetworkUtils->rmTftpRpms(@tftpRpms);
        }

        # Set installTftp to 1 so that TFTPpackage gets installed.
        ($::DEBUG) && print "TFTPpackage ($::TFTPpackage) not installed\n";
        $installTftp = 1;
    }
    elsif ($::default_tftp_server_name ne $::TFTPpackage)
    {

        # The TFTPpackage is not installed, and the user has specified
        # a diferent TFTPpackage attribute than the default, so we do not
        # know how to install it.

        # Check if there is a TFTP process running
        my $running = NetworkUtils->verifyTftp();
        if ($running)
        {

            # CASE 3
            print "An unknown tftp server is running, but the $::TFTPpackage\n"
              . "RPM is not installed.  Hopefully the running tftp server\n"
              . "will work OK!\n";
            MessageUtils->message('I', 'IMsgChangeTftpUser', 'tftpd');
            $installTftp = 0;
        }
        else
        {

            # CASE 4
            print "The $::TFTPpackage RPM is not installed and no tftp\n"
              . "server could be found on the system.\n"
              . "A TFTP server must be installed for CSM to function\n";
            MessageUtils->message('I', 'IMsgChangeTftpUser', 'tftpd');
            exit 1;
        }
    }
    else
    {

        # if we got here something went wrong
        print "internal error 127";
        exit;
    }

    #print "INSTALLTFTP = $installTftp\n";
    return $installTftp;

    #############################
    # OLD CODE BELOW THIS POINT #
    #############################
    #    elsif ($#tftpRpms eq -1)
    #    {
    #        # no tftp servers were found.  BUT, perhaps there is a
    #        # tarball installed.  We should check for that!
    #        my $running = NetworkUtils->verifyTftp();
    #        if ($running)
    #        {
    #	    # An unknown tftp server is running, but no tftp server RPM
    #	    # can be found.  Install the tftp package listed in TFTPpackage.
    #
    #	    ($::DEBUG) &&
    #	    print "An unknown tftp server was running, but no tftp server\n" .
    #	          "RPM could be found.  Installing $::TFTPpackage\n";
    #	    $installTftp = 1;
    #
    #        }    # there were no tftp processes running.
    #        else
    #        {
    #	    ($::DEBUG) &&
    #	    print "No tftp rpm installed or any tftp processes running.\n" .
    #	          "Installing $::TFTPpackage\n";
    #            # so we need to install!
    #            $installTftp = 1;
    #        }
    #    }
    #    elsif ($#tftpRpms ne -1 && !$isDefault)
    #    {
    #        MessageUtils->message('I', 'IMsgTFTPInfo', @vn, @vn);
    #
    #        while (1)
    #        {
    #            MessageUtils->message('I', 'IMsgTFTPOptions', @vn, @vn, @vn, @vn);
    #            $|   = 1;         # force a flush
    #            $ans = <STDIN>;
    #            chomp($ans);
    #            if ($ans eq 1 || $ans eq 2 || $ans eq 3 || $ans eq 4) { last; }
    #            print "\n\n";
    #            print "*" x 80 . "\n";
    #
    #        }
    #
    #        # warn them if they do not install tftp...
    #        unless ($ans eq 1 || $ans eq 2 || $ans eq 4)
    #        {
    #            MessageUtils->message('I', 'IMsgChangeTftpUser', 'tftpd');
    #            my $d = <STDIN>;
    #        }
    #
    #        if ($ans eq 1)
    #        {
    #
    #            # remove tftp rpms and install $::TFTPpackage
    #            NetworkUtils->rmTftpRpms(@tftpRpms);
    #            $installTftp = 1;    # install it!
    #        }
    #        elsif ($ans eq 2)
    #        {
    #
    #            # install $::TFTPpackage
    #            $installTftp = 1;    # install tftp
    #        }
    #        elsif ($ans eq 3)
    #        {
    #
    #            # do nothing and leave.
    #            MessageUtils->message('I', 'IMsgWILL_NOT_INSTALL', $::TFTPpackage);
    #        }
    #        elsif ($ans eq 4)
    #        {
    #            exit;
    #        }
    #        else
    #        {
    #
    #            # if we got here something went wrong
    #            print "internal error 127";
    #            exit;
    #        }
    #    }
    #    elsif (!$isDefault)
    #    {
    #
    #        # no tftp server is installed, so install the tftp server:
    #        $installTftp = 1;
    #    }
    #
    #    # the last case is if the default tftp server was already installed.
    #    # if this is true then nothing needs to be returned (0) because we
    #    # do not wish to install it again.
    #    return $installTftp;
    #    #############################
    #    # END OF OLD CODE           #
    #    #############################

}

#--------------------------------------------------------------------------------

=head3    restartTftp

    Restart xinetd to pick up changes.

        Arguments:
        	$errors
        	$service  - e.g. "inetd" or "xinetd"
        Returns:
                Number of errors
        Globals:
                none
        Error:
               Number of errors as a scalar.
        Example:
                $errors = NetworkUtils->restartTftp($errors, "inetd" );
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub restartTftp
{
    my ($class, $errors, $service) = @_;
    my $rc;

    if ($::PLTFRM eq "AIX")
    {
        my $cmd = "/usr/bin/startsrc -s tftpd";
        NodeUtils->runcmd($cmd, 0);
        if ($::RUNCMD_RC) { $errors++ }
    }
    else
    {

        # Restart xinetd to pick up the changes
        my $cmd = NetworkUtils->service($service, "start");
        NodeUtils->runcmd($cmd, 0);
        if ($rc) { $errors++ }

        $cmd = NetworkUtils->service($service, "reload");
        NodeUtils->runcmd($cmd, 0);
        if ($rc) { $errors++ }
    }

    return $errors;
}

#--------------------------------------------------------------------------------

=head3    rmTftpRpms

        Removes uses rpm to remove TFTP RPMs

=cut

#--------------------------------------------------------------------------------

sub rmTftpRpms
{
    my ($class, @tftpRpms) = @_;

    my ($output, $returncode, $ans);
    $returncode = 1;    # signal an error by default, unless things run right.

    foreach my $installedTftp (@tftpRpms)
    {
        MessageUtils->message("I", 'IMsgUNINSTALLING', $installedTftp);
        my $cmd = "$::RPMCMD -e $installedTftp --nodeps";
        $output = NodeUtils->runcmd($cmd, 0);
        $returncode = $::RUNCMD_RC if ($::RUNCMD_RC > $returncode);
    }
    return ($returncode);
}

#--------------------------------------------------------------------------------

=head3    secureFilePermissions

        Configure the default tftp server 

        Arguments:
                $directory   - the root directory
                $dir-perm    - permissions for dirs and subdirs
                $file-perm   - permissions for files
                
        Returns:
                none
        Globals:
                none
        Error:
                none
        Example:
                 NetworkUtils->secureFilePermissions ( $directory,
                                                       $dir-perm,
                                                       $file-perm);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub secureFilePermissions
{
    my ($class, $dir, $dirPerm, $filePerm, $disable_recurtive) = @_;

    # set these variables global, since &wanted does not
    # take parameters (see find routine in perl docs).

    $::dirPermissions  = $dirPerm;
    $::filePermissions = $filePerm;

    if ($disable_recurtive == 0)
    {
        find(\&chTftpUserPermissions, $dir);
    }
    else
    {
        $File::Find::name = $dir;
        NetworkUtils->chTftpUserPermissions;
    }

}

#--------------------------------------------------------------------------------

=head3   testTftpUser

        Tests that the tftp user exists. 

        Arguments:
                $tftpuser - TFTP User Name
        Returns:
                0 = does not exit
                1 = does exits
                errors
        Globals:
                none
        Error:
                none
        Example:
                if ( NetworkUtils->testTftpUser ) { blah;};
        Comments:
                ATTN:
                set $tftpuser and $passwdFile in this sub.
                the tftp stuff might be better in a security package.

=cut

#--------------------------------------------------------------------------------

sub testTftpUser
{
    my ($class, $tftpuser) = @_;
    my $passwdFile = "/etc/passwd";

    # determine whether or not a tftpd user already exists:
    my $file = NodeUtils->readFile($passwdFile);
    if (grep(/$tftpuser/, $file))
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

#--------------------------------------------------------------------------------

=head3    verifyTftp

        Check that tftp server is running.  If a value of 1 is passed
        in as an argument, then the routine will try to restart tftp if
        it isn't running.

        Arguments:
                optional argument "1"

        Returns:
                0 = not running
                1 = running
                errors
        Globals:
                none
        Error:
                return value of restart routine
        Example:
                my $running = NetworkUtils->verifyTftp(1);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub verifyTftp
{

    # if the value 1 is passed in then we should try to restart tftp if
    # it is not running.
    my ($class, $restart) = @_;

    my $errors  = 0;
    my $running = 0;
    my $cmd;

    MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                 'V', 'IMsgCheckingTftp');

    if ($::PLTFRM eq "AIX")
    {
        my @output = NodeUtils->runcmd("LANG=C /usr/bin/lssrc -s tftpd", -1);
        if (!$::RUNCMD_RC)
        {
            my ($subsys, $group, $pid, $status) = split(' ', $output[1]);
            if (defined($status) && $status eq 'active')
            {

                # A tftp server is up and running.
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'V', 'IMsgRunning', 'tftp');
                $running = 1;
            }
        }
        if (!$running)
        {

            # There is no tftp server running on this machine
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                               'csminstall', 'I', 'IMsgTftpNotRunning', 'tftp');
        }

        # try to restart it
        if ($restart && !$running)
        {

            # trying to restart tftp server
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'I', 'IMsgRestart', 'tftp');

            $errors = NetworkUtils->restartTftp($errors, "inetd");

            my @output =
              NodeUtils->runcmd("LANG=C /usr/bin/lssrc -s tftpd", -1);
            if (!$::RUNCMD_RC)
            {
                my ($subsys, $group, $pid, $status) = split(' ', $output[1]);
                if (defined($status) && $status eq 'active')
                {

                    # A tftp server is up and running.
                    MessageUtils->messageFromCat('csmInstall.cat',
                                $::MSGMAPPATH, 'csminstall', 'V', 'IMsgRunning',
                                'tftp');
                    $running = 1;
                }
            }

            if (!$running)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'I', 'IMsgStillCantRestart', 'tftp');

            }
        }
    }
    else
    {
        unless ($::LSOF) { $::LSOF = '/usr/sbin/lsof' }
        $cmd = "$::LSOF -i UDP:tftp";
        NodeUtils->runcmd($cmd, -1);

        if ($::RUNCMD_RC)
        {

            # There is no tftp server running on this machine
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                               'csminstall', 'I', 'IMsgTftpNotRunning', 'tftp');
        }
        else
        {

            # A tftp server is up and running.
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'V', 'IMsgRunning', 'tftp');
            $running = 1;
        }
        # try to restart it
        if ($restart && $::RUNCMD_RC)
        {

            # trying to restart tftp server
            MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'I', 'IMsgRestart', 'tftp');
            my $distro = NodeUtils->distribution();

            # restart tftp server.
            my $inetd  = "/etc/init.d/inetd";
            my $xinetd = "/etc/init.d/xinetd";

            #       if (($distro =~ /SLES/)&&(!($distro =~ /SLES 9/)))
            if (-f $xinetd)
            {
                $errors = NetworkUtils->restartTftp($errors, "xinetd");
            }
            elsif (-f $inetd)
            {
                $errors = NetworkUtils->restartTftp($errors, "inetd");
            }
            $cmd = "$::LSOF -i UDP:tftp";
            NodeUtils->runcmd($cmd, -1);
            if ($::RUNCMD_RC)
            {
                MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'I', 'IMsgStillCantRestart', 'tftp');

            }
            else
            {
                $running = 1;
            }
        }
    }
    return $running;
}

#--------------------------------------------------------------------------------

=head3    validate_ip

        Check that tftp server is running.  If a value of 1 is passed
        in as an argument, then the routine will try to restart tftp if
        it isn't running.

        Arguments:
                hostname

        Returns:
                -1 = errors
                ipaddress 
        Globals:
                none
        Error:
                return ip address for the hostname
        Example:
                my $ipaddr = NetworkUtils->validate_ip($hostname);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------

sub validate_ip
{
    my ($class, $host) = @_;
    
    my ($work, $res_ip_addr);

    $work = inet_aton($host);
    if (!defined($work))
    {
        $res_ip_addr = -1;
    }
    else
    {
        $res_ip_addr = inet_ntoa($work);
    }
    MessageUtils->messageFromCat(
                              'csmInstall.cat',
                              $::MSGMAPPATH,
                              'csminstall',
                              "E1",
                              'EMsgHostnameShouldNotBeResolveToLoopbackAddress',
                              $host,
                              $res_ip_addr
      )
      if $res_ip_addr =~ /127\.0\.0/;
    return $res_ip_addr;
}

#--------------------------------------------------------------------------------

=head3    ssh_keyscan


        Arguments:
                node name

        Returns:
                @keys array

        Globals:
                none
        Error:
              @keys is empty  
        Example:
                my @keys = NetworkUtils->ssh_keyscan($hostname);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------

sub ssh_keyscan
{
    my ($class, $node) = @_;
    require Socket;
    delete $ENV{'CSM_NO_NAME_RES'};    #so that tryHost works later
    my $keyscan = "/usr/bin/ssh-keyscan";
    my @names;
    my @keys;
    my %k;
    my @parts;

    my $res = 0;                       #used to see if hostname is resolvable

    #get the IP first to check if the passed in node is an IP.
    my $ip = Socket::inet_ntoa(Socket::inet_aton($node));
    push @names, $ip;

    # If the passed in node is an IP address then try to convert the ip
    # to a name. Then split the name to determine all combinations of the
    # hostname to use for short hostnames.
    if ( $ip eq $node )
    {
        my $converted_name = NetworkUtils->tryHost($node, \$res);
        @parts = split /\./, $converted_name;
    }
    else 
    {
        @parts = split /\./, $node;
    }

    # determine short names
    my $name = shift @parts;
    NetworkUtils->tryHost($name, \$res);
    if ($res)
    {
        push @names, $name;
    }
    foreach my $ext (@parts)
    {
        $name = "$name.$ext";
        NetworkUtils->tryHost($name, \$res);
        if ($res)
        {
            push @names, $name;
        }
    }

    #real name
    NetworkUtils->tryHost($node, \$res)
      ;    #tryHost sets $resolved to zero at startup
    if ($res)
    {
        push @names, $node;
    }

    $NodeUtils::NO_STDERR_REDIRECT = 1; #so stderr isn't redirected to @otherips
    my @otherips =
      NodeUtils->runcmd(
        "CT_MANAGEMENT_SCOPE=3 /usr/bin/lsrsrc-api -i -s IBM.NetworkInterface::\"NodeNameList IN ('$node')\"::IPAddress 2>/dev/null",
        -1
        );
    $NodeUtils::NO_STDERR_REDIRECT = 0;    #unset
    if (scalar @otherips > 1)
    {    #we were able to contact the node over RSCT

        foreach my $oip (@otherips)
        {
            push @names, $oip;
            my $host = NetworkUtils->tryHost($oip);
            if ($host ne $oip)
            {
                push @names, $host;
            }
        }
    }

    #make the name list
    my %nhash;
    foreach my $n (@names) { $nhash{$n} = 1; }    #remove duplicate names
    my $namestring = join ",", (keys %nhash);

    #get the hostkeys
    my $string = join " ", (keys %nhash);
    $::upd_debug && print "Debug running: $keyscan -t rsa1,rsa,dsa $string\n";
    my @output = `$keyscan -t rsa1,rsa,dsa $string 2>/dev/null`;
    chomp @output;

    foreach my $line (@output)
    {
        if ($line !~ m/^\#/)
        {
            my ($hostname, $rest) = split " ", $line, 2;
            $k{$rest}{$hostname} = 1;
        }
    }

    foreach my $i (keys %k)
    {    #this is the ssh host key
        my $goodkey = 0;
        foreach my $name (keys %{$k{$i}})
        {
            if (!$goodkey && defined $nhash{$name})
            {
                $goodkey = 1;
            }    #this is a hostname we are looking for
        }
        if ($goodkey) { push @keys, "$namestring $i\n"; }
    }

    return @keys;
}

#--------------------------------------------------------------------------------

=head3  update_known_hosts


        Arguments:
                array of nodes, optional argument 1 to skip ssh check
                mainly for devices

        Returns:


        Globals:
                $::SETUP_REMOTE_SHELL $::REMOTE_SHELL
        Error:

        Example:
                NetworkUtils->update_known_hosts(\@nodes);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------

sub update_known_hosts
{
    my ($class, $machines, $skipSSHCheck) = @_;

    my @nodes = @$machines;
    my ($supported, $sshversion);
    if (!defined $skipSSHCheck) { $skipSSHCheck = 0; }

    if ($skipSSHCheck != 1)
    {
        if ($::REMOTE_SHELL !~ /ssh/)
        {
            return 1;
        }
        ($supported, $sshversion) = NetworkUtils->supportedSSHversion();
        if (!$supported)
        {
            exit 0;
        }
    }

    if ($sshversion < 3 && $skipSSHCheck != 1)
    {
        NetworkUtils->rmNodesKnownHosts(\@nodes);
        foreach my $n (@nodes)
        {
            my $cmd = "/opt/csm/csmbin/remoteshell.expect -n $n";
            system($cmd);
        }
    }
    else
    {    #ssh 3 or higher, use ssh-keyscan
        my $home        = NetworkUtils->getRootHomeDir();
        my $known_hosts = "$home/.ssh/known_hosts";
        if (!-d "$home/.ssh") { mkdir "$home/.ssh", 0700; }
        if (-e "/var/opt/csm/known_hosts_debug") { $::DEBUG = 1; }
        if ($::DEBUG)
        {

            #DEBUG
            if (!-e "/tmp/known_hosts_debug/")
            {
                mkdir("/tmp/known_hosts_debug");
            }
            my $time      = time();
            my $orig_file = "/tmp/known_hosts_debug/$time";
            my $file      = $orig_file;
            my $count     = 1;
            while (-e $file)
            {
                $file = $orig_file . "_$count";    #append count
                $count++;
            }
            open(DEBUG_FILE, ">$file")
              or die MessageUtils->messageFromCat('csmInstall.cat', 
                                $::MSGMAPPATH, 'csminstall', 'E1', 
                                'EMsgCANT_WRITE_FILE', $file);
            print DEBUG_FILE "gatherSSHHostkeys\n\n";

            #END DEBUG
        }

        #run ssh-keyscan
        my @newkeys;
        my @goodnodes;    #nodes that are reachable and have host keys
        foreach my $n (@nodes)
        {
            my @keys = NetworkUtils->ssh_keyscan($n);
            if (scalar @keys > 0)
            {
                push @newkeys,   @keys;
                push @goodnodes, $n;
            }
        }
        if (scalar @newkeys > 0)
        {

            # First it creates an exclusive, blocking kernel lock on a file.
            # This will only let one execution run at a time, the rest will have to wait until its finished.
            use Fcntl qw(:flock);
            my $file =
              "/var/opt/csm/.CSMgatherSSHHostkeys_lock"
              ;    #lock a file in /var so it doesn't need to be removed
            if (!open(SEM, ">$file"))
            {

                #die "cannot open file $file";
                MessageUtils->messageFromCat('csmInstall.cat', 
                        $::MSGMAPPATH, 'csminstall', 'E1', 
                        'EMsgCANT_OPEN', $file);
            }

            #use an alarm to timeout if we cannot get the lock in 10 minutes
            my $cantlock = 0;
            eval {
                local $SIG{ALRM} = sub { die "alarm clock restart" };
                alarm 600;    # schedule alarm in 600 seconds
                eval {
                    if (!flock(SEM, LOCK_EX))
                    {
                        $cantlock = 1;
                    }
                };
                alarm 0;      # cancel the alarm
            };
            alarm 0;          # race condition protection
            die if $@ && $@ !~ /alarm clock restart/;    # reraise
            if ($cantlock == 1) { 
                    die MessageUtils->messageFromCat('csmInstall.cat', 
                            $::MSGMAPPATH, 'csminstall', 'E1', 
                            'EMsgCANT_LOCK_FILE', $file); 
            }

            #remove old entries from known_hosts
            $::DEBUG
              and print DEBUG_FILE
              "Removing nodes from known_hosts: @goodnodes\n\n";    #DEBUG
            NetworkUtils->rmNodesKnownHosts(\@goodnodes);

            #append new entries to known_hosts
            $::DEBUG
              and print DEBUG_FILE
              "\n\nAdding the following lines to known_hosts:\n@newkeys"; #DEBUG
            if (-e $known_hosts)
            {
                open(HOSTS, ">>$known_hosts")
                  or 
                  MessageUtils->messageFromCat('csmInstall.cat', 
                          $::MSGMAPPATH,'csminstall', 'E1', 
                          'EMsgCANT_OPEN', $known_hosts);
            }
            else
            {
                open(HOSTS, ">$known_hosts")
                  or 
                  MessageUtils->messageFromCat('csmInstall.cat', 
                          $::MSGMAPPATH,'csminstall', 'E1', 
                          'EMsgCANT_OPEN', $known_hosts);
            }
            print HOSTS @newkeys;
            close HOSTS
              or
              MessageUtils->message("E2", 'EMsgCANT_CLOSE_FILE', $known_hosts,
                                    $!, ($? >> 8));

            close(SEM);
        }
        else
        {
            $::DEBUG and print DEBUG_FILE "Nothing in the array newkeys\n";
        }    #DEBUG
        $::DEBUG and close DEBUG_FILE;
    }

    #update the known_hosts file on backup ms
    if (!defined $ENV{'CSM_HAMS_DO_NOT_REPEAT'})
    {
        if (!defined $::HAMODE)
        {
            my $outref =
              NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                   "-s IBM.DmsCtrl::::HAMode");
            $::HAMODE = $$outref[0];
        }
        if ($::HAMODE == 1)
        {    #on an active HAMS management server
            require HAMS;
            my $inactive_ms = HAMS->determineInactive();
            if ($inactive_ms =~ m/\w/)
            {    #if its a hostname
                my $node_string = join " ", @$machines;
		my %options_api = ();
		my $ENV_OLD_1 = $ENV{'CSM_HAMS_CONTROL'};
		my $ENV_OLD_2 = $ENV{'CSM_HAMS_DO_NOT_REPEAT'};
		$ENV{'CSM_HAMS_CONTROL'} = 1;
		$ENV{'CSM_HAMS_DO_NOT_REPEAT'} = 1;
		$options_api{'vefify'} = 1;
		$options_api{'nodes'} = $inactive_ms;
		$options_api{'command'} = "/opt/csm/csmbin/gatherSSHHostkeys $node_string";
                NodeUtils->runDsh(\%options_api, 0);
		$ENV{'CSM_HAMS_CONTROL'} = $ENV_OLD_1;
		$ENV{'CSM_HAMS_DO_NOT_REPEAT'} = $ENV_OLD_2;
            }
        }
    }
}

#--------------------------------------------------------------------------------

=head3  getNetmask

        Arguments:
                $node: node ip or name 
                $ref_nodehash: reference to node hash global variable 

        Returns:
        	  Default Netmask and BroadCast of the node

        Example:
                ($bcast, $mask)=NetworkUtils->getNetmask($node,\$::NODE_HASH);
        Comments:
                The following five steps determine node's netmask value.
                0. If $ref_nodehash is blank, just return the broadcast and netmask of its ip subnet. 
                1. If node's InstallAdapterNetmask attribute is set , return it.
                2. Call NetworkUtils::get_source_ip_to_target to get IS'ip (*1) close to node.
                3. If MS is IS and *1 <> ManagementServer'IP, print warning message 
                    and use ManagementServer attr(*2)  to search subnet of node linked.
                4. Parse netmask value from ifconfig output via subnet (*1) or (*2).
                   (Note: AIX and Linux have different logic for this step.)
=cut

#--------------------------------------------------------------------------------
sub getNetmask
{
    my ($class, $node, $ref_nodehash, $installserver,$is_os) = @_;
    my ($node_hostname, $node_ip) = NetworkUtils->getHost($node);
    my $HasIS = 0;
    my ($msip, $parse_ip, $ip_link_to_node);
    my %nodehash; 
    if ($ref_nodehash)
    {
        %nodehash = %$ref_nodehash;
        # Step 1 : Check node's InstallAdapterNetmask attribute
        if ($nodehash{$node_hostname}{'InstallAdapterNetmask'})
        {
            my $netmask = $nodehash{$node_hostname}{'InstallAdapterNetmask'};
            my $broadcast = NetworkUtils->broadcast($node_ip,$netmask);
            return ($broadcast,$netmask);
        }
        (undef, $msip) = 
            NetworkUtils->getHost($$ref_nodehash{$node_hostname}{'ManagementServer'});

        (undef, $installserver) =
            NodeUtils->get_is_addrinfo_by_name($node_hostname, $ref_nodehash);
        if (! $installserver)
        {
            $installserver = $msip;
        }
        if ($nodehash{$node_hostname}{'InstallServer'})
        {
            $HasIS = 1;
        }
        # Step 2: Get the netcard's ip through which the IS can
        #         reach the target node.
        if ($HasIS)
        {
            $ip_link_to_node = NetworkUtils->get_source_ip_to_target($node_hostname,$installserver);
        }
        else
        {
            $ip_link_to_node = NetworkUtils->get_source_ip_to_target($node_hostname);
        }
        # In case $ref_nodehash don't include IS hash.
        #($ref_nodehash, undef, undef) =
        #ServerUtils->getTargetNodeHashandInstallServerHash(\%nodehash);
        if ($HasIS)
        {
            # Ensure $installserver is not a key of $ref_nodehash
            my @target_nodes = (keys %$ref_nodehash);
            my ($is_name, $is_ip) = NetworkUtils->getHost($installserver);
           
            if (grep /^$is_name/, @target_nodes)
            {
                $is_os = $$ref_nodehash{$is_name}{'InstallOSName'};
            }
        }

        if (! $is_os)
        {
            $is_os = NodeUtils->get_OSName;
        }
        # Step 3 Compare ManagementServer attribute with get_source_ip_to_target return value
        if (  (! $HasIS)
            && ($ip_link_to_node ne $msip)
        )
        {
            # Don't need to prompt warning message, makedhcp has done it.
            #MessageUtils->messageFromCat('csmInstall.cat',
            #    $::MSGMAPPATH, 'makedhcp', 'I',
            #    'IMsgManagementServerAttrNotconsistNode', $node_hostname, $ip_link_to_node);
            # Honour node's ManagementServer attribute set by users
            $parse_ip = $msip;
        }
        else
        {
            $parse_ip = $ip_link_to_node;
        }
    }
    else
    {
        $parse_ip = $node_ip;
        if ($installserver)
        {
            $HasIS = 1;
        }
        $is_os |= NodeUtils->get_OSName;
    }


    # Step 4 Parse ifconfig output.
    # if IS is AIX 
    my $cmd;
    if ($is_os eq "AIX")
    {
        $cmd = "$::IFCONFIG -a|$::GREP $parse_ip";
        my @result;
        if ($HasIS)
        {
            
            @result = NodeUtils->dshOnNodes($installserver, $cmd , $::TIMEOUT | 60 , -1);
        }
        else
        {
            @result = NodeUtils->runcmd($cmd, -1); 
        }
        # Give a default netmask in case of system command error
        if ($::RUNCMD_RC != 0)
        {
            return (undef, "255.255.255.0");
        }

        if ($result[0] =~ /inet (.*) netmask (.*) broadcast (.*)$/)
        {
            my $netmask   = $2;
            my $broadcast = $3;
            $netmask = NetworkUtils->hex2dec($netmask);
            return ($broadcast, $netmask);
        }
    }
    else
    {
        # logic for Linux is different than AIX.
        my $cmd = $::IFCONFIG;
        $cmd = $cmd . "|$::GREP `/opt/csm/csmbin/getSourceIP2Target $parse_ip`";
        my @result;
        if ($HasIS)
        {
            @result =  NodeUtils->dshOnNodes($installserver, $cmd , $::TIMEOUT | 60 , -1);
        }
        else
        {
            @result = NodeUtils->runcmd($cmd, -1);
        }

        # Give a default netmask in case of system command error
        if ($::RUNCMD_RC != 0)
        {
            return (undef, "255.255.255.0");
        }

        if ($result[0] =~ /inet addr:(.*)  Bcast:(.*)  Mask:(.*)$/)
        {
            return ($2, $3);
        }
    }

    return (undef, "255.255.255.0");
}

#--------------------------------------------------------------------------------

=head3  verify_ms_dhcpd

        Arguments:
                target nodes hash

        Returns:
		
        Example:
                NetworkUtils->verify_ms_dhcpd(\%DestNodeHash);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------
sub verify_ms_dhcpd
{
    my $routine = "verify_ms_dhcpd";
    print "ENTERING: $routine\n" if $::DEBUG;

    my ($class, $nodehash_ref) = @_;
    my (@mgrattr,    %isattr, $dhcp_interface);
    my ($cmd,        @output, @lines, $line);
    my (@ms_subnets, @issvrs, @mssvrs);

    my ($mgrattr_ref, $isattr_ref) =
      ServerUtils->getInstallServers($nodehash_ref);

    if ($mgrattr_ref) { @mgrattr = @$mgrattr_ref; }
    if ($isattr_ref)  { %isattr  = %$isattr_ref; }

    # Any target nodes will be installed through install server different than MS?
    if (!keys %isattr) { return; }

    # MS is serving as an IS
    foreach my $is (keys %isattr)
    {
        if ($isattr{$is}{isGrp})
        {
            my $refGroup = NodeUtils->getNodegrp($is);
            if ($refGroup)
            {
                push @issvrs, @$refGroup;
            }
        }
        else
        {
            push @issvrs, $is;
        }
    }

    foreach my $node (keys %::NODEHASH)
    {

        # The attr ManagementServer would be a host name or ip address
        my $mgmtsvr = $::NODEHASH{$node}{'ManagementServer'};
        if (grep(/^$mgmtsvr$/, @issvrs))
        {
            return;
        }

        my ($ms_hostname) = NetworkUtils->getHost($mgmtsvr);
        if ($ms_hostname && grep(/^$ms_hostname$/, @issvrs))
        {
            return;
        }

    }

    # Is DHCP daemon on MS running?
    $cmd = NetworkUtils->service("dhcpd", "status");
    chomp(@output = `$cmd 2>&1`);
    @lines = split(" ", @output);
    if (!grep /running/, @output) { return; }

    # Is DHCP daemon on MS listening on same the subnet as IS?
    $cmd = "$::GREP subnet /etc/dhcpd.conf";
    chomp(@lines = `$cmd`);
    foreach $line (@lines)
    {
        chomp($line);
        $line =~ s/^\s+//g;
        $line =~ s/\s+$//g;
        $line =~ s/\{$//g;
        next if ($line =~ /^#/);

        @output = split(/\s+/, $line);
        my $subnet = join " ", @output;
        push @ms_subnets, $subnet;
    }

    if (@issvrs)
    {
        $cmd = "$::GREP subnet /etc/dhcpd.conf";
	my %options_api = ();
	$options_api{'command'} = $cmd;
        $options_api{'nodes'} = join(',', @issvrs);
        @output = NodeUtils->runDsh(\%options_api, -1);

        foreach $line (@output)
        {
            $line =~ s/^.*://g;
            $line =~ s/^\s+//g;
            $line =~ s/\s+$//g;
            $line =~ s/\{$//g;
            next if ($line =~ /^#/);

            my @strs = split(/\s+/, $line);
            my $subnet = join " ", @strs;
            foreach (@ms_subnets)
            {
                if ($subnet =~ /$_/)
                {

                    #my $progname = NodeUtils->programName();
                    MessageUtils->messageFromCat('csmInstall.cat',
                                           $::MSGMAPPATH, 'csminstall', 'E1',
                                           'EMsgDHCPSubnetsOverlaped', $subnet);
                }
            }
        }
    }

    print "LEAVING: $routine\n" if $::DEBUG;
}

#--------------------------------------------------------------------------------

=head3    within_same_subnet

        Notes: 
        check whether two hosts are locating in same subnet
        

=cut

#--------------------------------------------------------------------------------
sub within_same_subnet
{
    my ($class, $client_ip, $client_netmask, $bootpserver, $bootpserver_netmask)
      = @_;

    my ($client_subnet, $is_subnet);

    if (!$client_netmask)
    {
		# take a default value for the netmask to prevent the unpredictable error
		$client_netmask = "255.255.255.0";
    }
    $client_subnet = NetworkUtils->subnet($client_ip, $client_netmask);

    if (!$bootpserver_netmask)
    {
		# take a default value for the netmask to prevent the unpredictable error
		$bootpserver_netmask = "255.255.255.0";
    }
    $is_subnet = NetworkUtils->subnet($bootpserver, $bootpserver_netmask);

    if ($is_subnet ne $client_subnet)
    {
        return $::NOK;
    }

    return $::OK;
}

#--------------------------------------------------------------------------------

=head3    get_source_ip_to_target

        Notes: 
          get the IP address of the IS's netcard through which the IS can 
          reach the target node. 
          Works on both Linux and AIX.

=cut

#--------------------------------------------------------------------------------
sub get_source_ip_to_target
{
    my ($class,$node,$server)=@_;

    my $source_ip;

    if (!$server)
    {
        socket(SOCK, PF_INET, SOCK_DGRAM, 'tcp');
        my $sin = sockaddr_in(2006, inet_aton("$node"));
        connect(SOCK, $sin);
        my ($port, $iaddr) = sockaddr_in(getsockname(SOCK));
        $source_ip = inet_ntoa($iaddr);
        close SOCK;
    }
    else
    {
        my $exeFile = "/tmp/exeFile$$";
        open(EXEFILE, ">$exeFile");
        chmod 0755, $exeFile;
        print EXEFILE <<perlfile;
#!/usr/bin/perl
use Socket;
socket(SOCK,PF_INET,SOCK_DGRAM,"tcp");
my \$sin = sockaddr_in(2006,inet_aton("$node"));
connect(SOCK,\$sin);
my (\$port,\$iaddr)= sockaddr_in(getsockname(SOCK));
my \$source_ip = inet_ntoa(\$iaddr);
close SOCK;
print \$source_ip;
perlfile
        close EXEFILE;
	my %options_api = ();
	$options_api{'nodes'} = $server;
	$options_api{'execute'} = 1;
	$options_api{'command'} = $exeFile;
        my $output = NodeUtils->runDsh(\%options_api, -1);
        unlink $exeFile;

        if ($::RUNCMD_RC || !($output =~ /.*:\s*(\d+\.\d+\.\d+\.\d+)$/))
        {
            (undef, $source_ip) = NetworkUtils->getHost($node);
        }
        else
        {
            $source_ip = $1;
        }
    }

    return $source_ip;
}

#--------------------------------------------------------------------------------

=head3    hex2dec

        Generates a dotted decimal format of an ip address from the hex format:
            for example, 097285C1 -> 9.114.133.193
        This is generally used on AIX.

        Arguments:
                $ip in hex format
        Returns:
                $ip in dotted decimal format
        Globals:
                none
        Error:
                undefined
        Example:
                my $mask = NetworkUtils->hex2dec("0xffffff00");
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub hex2dec
{
    my ($class, $hexaddr) = @_;

    # get rid of the possible preceding 0x
    $hexaddr =~ /0x(\S+)/;
    $hexaddr = $1;

    # split it
    my $a1 = substr $hexaddr, 0, 2;
    my $a2 = substr $hexaddr, 2, 2;
    my $a3 = substr $hexaddr, 4, 2;
    my $a4 = substr $hexaddr, 6, 2;

    # transfer to decimal
    my $h1 = hex($a1);
    my $h2 = hex($a2);
    my $h3 = hex($a3);
    my $h4 = hex($a4);

    my $number = "$h1.$h2.$h3.$h4";
    return $number;
}

#--------------------------------------------------------------------------------

=head3    setDefaultFilePermission

        set default file permissions for tftp server

        Arguments:
                $filename   - the directory or file's name
        Returns:
                none
        Globals:
                none
        Error:
                none
        Example:
                 NetworkUtils->setDefaultFilePermission ( $filename );

        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub setDefaultFilePermission
{
	my ($class, $filename, $disable_recurtive) = @_;
	if($::PLTFRM eq "AIX")
	{
		# for AIX, set file mode to 444, dir mode to 110.
		if (-f $filename)
		{
			chmod oct("0444"), $filename;
		}
		elsif (-d $filename)
		{
			chmod oct("0110"), $filename;
		}	
		else
		{
			return;
		}
	}
	else
	{
		# for Linux, set file mode to 440, dir mode to 110, and set tftpd
		# to be the file owner.
		NetworkUtils->secureFilePermissions($filename, "0110", "0440", $disable_recurtive);
	}	
}
#--------------------------------------------------------------------------------

=head3    reorderHosts

        Reorder /etc/hosts after warewulf rewrite /etc/hosts.
        Avoid to resolve ip into wrong hostname that is not identified by CSM DMS.

        # Reorder /etc/hosts, advance clusters.com hostname entry.
        # "# Node 'cdlf4ih02.clusters.com' configuration (group: RedHatEL-Server5-GA-ppc64)
        # 192.168.0.10    cdlf4ih02.clusters.com-admin
        # 192.168.0.10    cdlf4ih02.clusters.com cdlf4ih02.clusters.com-cluster
        # 192.168.0.10    cdlf4ih02.clusters.com-sharedfs"
        # should be 
        # "# Node 'cdlf4ih02.clusters.com' configuration (group: RedHatEL-Server5-GA-ppc64)
        # 192.168.0.10    cdlf4ih02.clusters.com cdlf4ih02.clusters.com-cluster
        # 192.168.0.10    cdlf4ih02.clusters.com-sharedfs
        # 192.168.0.10    cdlf4ih02.clusters.com-admin
        # "
      
        Arguments:
        Returns:
                none
        Globals:
                none
        Error:
                none
        Example:
                 NetworkUtils->reorderHosts();

        Comments:
                reference to defect 140990

=cut

#--------------------------------------------------------------------------------

sub reorderHosts
{
    my $class = @_;
    my @lines;
    open(HOSTS,"< /etc/hosts") 
        ||   MessageUtils->messageFromCat('csmInstall.cat',
            $::MSGMAPPATH,'csminstall', 'E1',
            'EMsgCANT_OPEN', '/etc/hosts');

    @lines = <HOSTS>;
    close HOSTS;
    my $start_warewulf = 0;
    my $ins_admin = 0;
    my ($ip_cache, $hostname_cache);
    open(HOSTS, "> /etc/hosts")
        ||  MessageUtils->messageFromCat('csmInstall.cat',
            $::MSGMAPPATH,'csminstall', 'E1',
            'EMsgCANT_OPEN', '/etc/hosts');

    foreach my $line (@lines)
    {
        if ( $line =~ /# Warewulf nodes #/)
        {
            $start_warewulf = 1;
        }
        if ( $start_warewulf)
        {
            my ($ip , $hostname) = split(/\s+/, $line);
            if ( $hostname =~ /-admin/)
            {
                # Reserve it
                ($ip_cache, $hostname_cache) = ($ip , $hostname);
                $ins_admin = 1;

            }
            else
            {
                # Insert admin entry if reach section bottom.
                if ( $ip ne $ip_cache
                    && $ins_admin)
                {
                    print HOSTS "$ip_cache\t$hostname_cache\n";
                    $ins_admin = 0;
                }
                print HOSTS "$line";
            }
        }
        else
        {
            print HOSTS "$line";
        }
    }
    close HOSTS;

}

sub start_warewulf
{
    my ($class, $ref_nodehash, $ref_ishash) = @_;
    my $rc      = 0;

    my %nodehash = %$ref_nodehash;
    my %ishash   = %$ref_ishash;
    my $warewulfd = "/etc/init.d/warewulf";
	my $vnfsd = "/etc/init.d/vnfsd";
    my %options_api = ();

    # Export /home directory on MS.
    # This directory will be shared by all diskless node to reserve common data.
    NetworkUtils->export_home($ref_nodehash);
    if ($::MS_INSTALL_SERVER || (!keys %ishash))
    {

        # Start nfs if needed.
        if ($::PLTFRM eq "AIX")
        {
        	return;
        }
        else
        {
            # Check warewulf environment.
            if (!-f "/var/tmp/warewulf-ready")
            {
                $::GLOBAL_EXIT = 1;
                return;
            }
			my $cmd;
            $cmd = "$warewulfd restart";
        	MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        	$rc = NodeUtils->runcmd($cmd,-2);# system($cmd) >> 8;
        	if ($rc)
        	{
            	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'I', 'IMsgShow_Output',  $rc);
        	}
            # Reorder /etc/hosts, advance clusters.com hostname entry.
            # "# Node 'cdlf4ih02.clusters.com' configuration (group: RedHatEL-Server5-GA-ppc64)
            # 192.168.0.10    cdlf4ih02.clusters.com-admin
            # 192.168.0.10    cdlf4ih02.clusters.com cdlf4ih02.clusters.com-cluster
            # 192.168.0.10    cdlf4ih02.clusters.com-sharedfs"
            # should be 
            # "# Node 'cdlf4ih02.clusters.com' configuration (group: RedHatEL-Server5-GA-ppc64)
            # 192.168.0.10    cdlf4ih02.clusters.com cdlf4ih02.clusters.com-cluster
            # 192.168.0.10    cdlf4ih02.clusters.com-sharedfs
            # 192.168.0.10    cdlf4ih02.clusters.com-admin
            # "
            NetworkUtils->reorderHosts();

			#my $cmd = NetworkUtils->service("vnfsd", "restart");
            $cmd = "$vnfsd restart";
        	MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                     'V', 'IMsgCMD', $cmd);
        	$rc = NodeUtils->runcmd($cmd,-2);#system($cmd) >> 8;
        	if ($rc)
        	{
            	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'I', 'IMsgShow_Output',  $rc);
        	}
            #NodeUtils->runcmd("wwinit --exports");
        }
	}
    if (defined %ishash)
    {
			my $dsh = "/opt/csm/bin/dsh";
			my $dcp = "/opt/csm/bin/dcp";
			foreach my $is (keys %ishash)
			{
				my ($cmd, $outref);
				my $isISGrp = $ishash{$is}{isGrp};
				my $realis = $is;
				if ($isISGrp)
				{
				   $realis =~ s/^\+//;
				}
				if ($isISGrp)
				{
					%options_api = ();
	                $options_api{'nodegroups'} = $realis;
                    $options_api{'command'} = "$::LS /var/tmp/warewulf-ready";
	                $options_api{'execute'} = 0;
                    # Check warewulf environment exists
                    NodeUtils->runDsh(\%options_api,-1);
                    if ($::RUNCMD_RC)
                    {
                        $::GLOBAL_EXIT = 1;
                        return;
                    }
                    %options_api= ();
                    $options_api{'nodegroups'} = $realis;
                    $options_api{'command'} = "$warewulfd restart";
	                $options_api{'execute'} = 0;
                    
					$outref = NodeUtils->runDsh(\%options_api, -2);
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
												'I', 'IMsgShow_Output', $outref);

					%options_api = ();
	                $options_api{'nodegroups'} = $realis;
					$options_api{'command'} = "$vnfsd restart";
                    $options_api{'execute'} = 0;
					$outref = NodeUtils->runDsh(\%options_api, -2);
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
												'I', 'IMsgShow_Output', $outref);
                    #NodeUtils->runcmd("$dsh -N $realis /usr/sbin/wwinit --exports",-2);
             		#For SLES, we need restart nfs server service.
                    #if ($ishash{$is}{InstallDistributionName} eq "SLES")
                    #{ 
                    #	NodeUtils->runcmd("$dsh -N $realis /etc/init.d/nfsserver restart",-2);
                    #}

				}
				else
				{
					%options_api= ();
	                $options_api{'nodes'} = $is;
                    $options_api{'command'} = "$::LS /var/tmp/warewulf-ready";
	                $options_api{'execute'} = 0;
                    # Check warewulf environment exists
                    NodeUtils->runDsh(\%options_api,-1);
                    if ($::RUNCMD_RC)
                    {
                        $::GLOBAL_EXIT = 1;
                        return;
                    }
                    %options_api= ();
                    $options_api{'nodes'} = $is;
                    $options_api{'command'} = "$warewulfd restart";
                    $options_api{'execute'} = 0;
					$outref = NodeUtils->runDsh(\%options_api, -2);

					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
												'I', 'IMsgShow_Output', $outref);

                    
					%options_api= ();
					$options_api{'nodes'} = $is;
                    $options_api{'command'} = "$vnfsd restart";
					$options_api{'execute'} = 0;
                    $outref = NodeUtils->runDsh(\%options_api, -2);
					
					MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
												'I', 'IMsgShow_Output', $outref);
                    #NodeUtils->runcmd("$dsh -n $is /usr/sbin/wwinit --exports",-2);
             		#For SLES, we need restart nfs server service.
                    #if ($ishash{$is}{InstallDistributionName} eq "SLES")
                    #{ 
                    #	NodeUtils->runcmd("$dsh -n $is /etc/init.d/nfsserver restart",-2);
                    #}

				}
			}
	}
    return ($rc);
}


#--------------------------------------------------------------------------------

=head3    stop_nfs_Linux

        Stops NFS services on the Linux ManagementServer

        Arguments:
                none
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
                $::SERVICE
        Error:
                $::NOK
        Example:
                if (NetworkUtils->stop_nfs_Linux != 0) { blah; }_
        Comments:
                none

=cut

#--------------------------------------------------------------------------------
sub stop_nfs_Linux
{
    my $rc = 0;

    #stop nfs service
    my $cmd = NetworkUtils->service("nfs", "stop");

    MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                 'V', 'IMsgCMD', $cmd);
    $rc = system($cmd) >> 8;
    if ($rc)
    {
        MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                         'csminstall', 'E', 'EMsgCommandFailed', $cmd, $rc);
    }
    #stop portmap when on HMC
    if(NodeUtils->isHMC())
    {
	    my $cmd = NetworkUtils->service("portmap", "stop");

       	MessageUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                    'V', 'IMsgCMD', $cmd);
       	$rc = system($cmd) >> 8;
       	if ($rc)
       	{
           	MessageUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                            'csminstall', 'E', 'EMsgCommandFailed', $cmd, $rc);
       	}
	}
    return $rc;
}

#-------------------------------------------------------------------------------

=head3    get_MSip

          Get the IP of the management server.

        Development notes:

=cut

#-------------------------------------------------------------------------------
sub get_MSip
{
    my ($class, $node_hostname) = @_;
    my $ms_ipaddr     = "";

    $::GETSOURCEIP2TARGET = "/opt/csm/csmbin/getSourceIP2Target";

    my $cmd    = "$::GETSOURCEIP2TARGET $node_hostname";
    my @output = NodeUtils->runcmd($cmd, 0);
    my $rc     = $::RUNCMD_RC;
    if (!$rc) { $ms_ipaddr = $output[0]; }    # If no errors
    else { $ms_ipaddr = -1; }

    return $ms_ipaddr;
}

1;

