#!/usr/bin/perl

package VPD_List;

BEGIN { unshift (@INC, "/var/adm/invscout"); }
use VPD;

# --------------------------------------------------------------------------------+
# -- Constructor --                                                               |
# --------------------------------------------------------------------------------+
sub new
{
  my $this = shift;
  my $class = ref($this) || $this;

  my ($inFileName) = @_;
  my $self = [];
  bless $self, $class;

  $self->_readFile($inFileName);

  return $self;
}



# --------------------------------------------------------------------------------+
# -- Name     : _readFile                                                         |
# -- Function : Private function used by constructor                              |
# --            This function parses the input file (lsvpd-output file-format)    |
# --              and convert them into Array (this object) of VPD objects        |
# -- Input    : File Name or command                                              |
# -- Output   : none                                                              |
# --------------------------------------------------------------------------------+
sub _readFile($inFileName)
{
  my $this = shift;
  my ($inFileName) = @_;

  my ($line, $index, $key, $value) = ();
  open ( fIN, $inFileName );

  $index = -1;   # -- FC is the "start" separator, therefore the first index should be 0 --

  # -- Parse the VPD input and convert it to List of VPD objects --
  while ( $line = <fIN> )
  {
    if ($line =~ /^\*FC/)      # ----- FC Separator -----
    {
      $index += 1;
      $this->[$index] =  VPD->new();
    }
    else                       # ----- Subsequent Key&Value Pairs -----
    {
      return if ( $index < 0 );
      $this->[$index]->add_lsvpdString($line);
    }
  }
  close ( fIN );
}


# --------------------------------------------------------------------------------+
# -- Name     : _display                                                          |
# -- Function : Prints the List of VPD objects (Debugging Purposes)               |
# --              This function uses the "_display" method of the VPD object      |
# -- Input    : None (optional: list of indexes to be printed)                    |
# -- Output   : Printout of the VPD_List object                                   |
# --------------------------------------------------------------------------------+
sub _display
{
  my $this = shift;
  my @inIndexes = @_ ? @_ : ( 0 .. $#{$this} );

  my ($index) = ();

  foreach $index (@inIndexes)
  {
    print "--------------------VPD # $index:--------------------\n";
    $this->[$index]->_display();
  }
}


# --------------------------------------------------------------------------------+
# -- Name     : VPD_List_IsEmpty                                                  |
# -- Function : Checks whether this object contains an empty list                 |
# -- Input    : None                                                              |
# -- Output   : Returns true (1) if the List is Empty, False (0) otherwise        |
# --------------------------------------------------------------------------------+
sub VPD_List_IsEmpty
{
  my $this = shift;
  if ( $#{$this} <= 0 )
  {
    # -- if This List has no Element, then it's empty
    return(1);
  }
  else
  {
    return(0);
  }
}



# --------------------------------------------------------------------------------+
# -- Name     : get_VPD_Object                                                    |
# -- Function : Returns the VPD Object stored in this VPD_List object             |
# -- Input    : Index                                                             |
# -- Output   : VPD Object                                                        |
# --------------------------------------------------------------------------------+
sub get_VPD_Object
{
  my $this = shift;
  my ($inIndex) = @_;
  return ( $this->[$inIndex] );
}



# --------------------------------------------------------------------------------+
# -- Name     : get_VPDsWithKey                                                   |
# -- Function : Searches the whole Array for a VPD with given list of Keys        |
# -- Input    : Key (one Key)                                                     |
# -- Output   : List (Array) of indexes of this VPD_List object that contains     |
# --             the requested Key                                                |
# --------------------------------------------------------------------------------+
sub get_VPDsWithKey
{
  my $this = shift;
  my ( @inKeys ) = @_;

  my ($index, @retVal) = ();

  for ($index = 0; $index <= $#{$this}; $index++)
  {
    my ($vpd) = $this->[$index];
    if ( $vpd->has_Keys(@inKeys) )
    {
      push (@retVal, $index);
    }
  }

  return (@retVal);
}



# --------------------------------------------------------------------------------+
# -- Name     : insert_VPD_Objects                                                |
# -- Function : Add / Inserts VPD objects to the list                             |
# -- Input    : 1. Index of insertion                                             |
# --            2. The list of VPD objects                                        |
# -- Output   : none                                                              |
# --------------------------------------------------------------------------------+
sub insert_VPD_Objects
{
  my $this = shift;
  my ( $inIndex, @inVPD_Objects ) = @_;

  # -- Must trust the caller: assume @inVPDObjects are list of VPD objects
  splice (@{$this}, $inIndex, 0, @inVPD_Objects);
}



# --------------------------------------------------------------------------------+
# -- Name     : write_Object                                                      |
# -- Function : Writes the whole VPD_List Object into the specified Destination   |
# -- Input    : Output_Handle = Destination                                       |
# -- Output   :                                                                   |
# --              Note: To write to screen, use "\*STDOUT"                        |
# --                    For a file, open a file for writing, then pass the handle |
# --------------------------------------------------------------------------------+
sub write_Object
{
  my $this = shift;
  my ( $inIO_Handle ) = shift;  # -- Where the output will be written to --

  my ($vpd);
  foreach $vpd (@$this)
  {
    $vpd->write_Object($inIO_Handle);
  }
}





1;




