#!/usr/bin/perl -w
##
## actmUtil.pl
##
## (c)  Copyright  IBM Corp.  2004, 2007  All Rights Reserved
##
## US Government Users Restricted Rights - Use, duplication or
## disclosure restricted by GSA ADP Schedule Contract with IBM Corp
##
## Module History:
##  -00 03/05/2004 John Thornton   LM28 initial release
##  -01 03/08/2004 John Thornton   LM29 delete support
##  -02 03/10/2004 John Thornton   LM29 "is directory" support
##  -03 11/01/2004 John Thornton   D61C E6831 - File permission processing
##  -04 02/18/2005 John Thornton   D61N Add copy and execute command functions
##  -05 02/25/2005 John Thornton   D61N Make some error messages go away
##  -06 03/29/2005 John Thornton   D61Q Link processing
##  -07 04/19/2005 John Thornton   D61T Unable to load shared objects during
##                                      execute function
##  -08 05/20/2005 John Thornton   D61Y Patch test fixes
##  -09 08/06/2005 John Thornton   D63H G8651 - '$' files pose problems
##  -10 09/04/2005 John Thornton   D63J H0491 - file permissions lost
##  -11 06/09/2006 John Thornton   D66G I1098 Change file timestamp when
##                                        changing file permissions
##  -12 08/28/2006 John Thornton   D66Q hmcfunction; setupEnv no longer needed
##  -13 03/16/2007 John Thornton   D71H Support patch tracing
##  -14 06/14/2007 John Thornton   D71S RPM for MCF eCLipz line item
##  -15 06/20/2007 John Thornton   D71U Adding use of --force and --nodeps
##  -16 06/22/2007 John Thornton   D71U Tracing not working correctly
##

#
# Includes
#
use Getopt::Long;

################################################################################
######                          GLOBAL VARIABLES                          ######
################################################################################

# Command Constants
$UTIL_CMD_NONE          = 0;
$UTIL_CMD_EXISTS        = 1;
$UTIL_CMD_MKDIRS        = 2;
$UTIL_CMD_DELETE        = 3;                                                    #-01
$UTIL_CMD_ISDIR         = 4;                                                    #-02
$UTIL_CMD_PERMS         = 5;                                                    #-03
$UTIL_CMD_COPY          = 6;                                                    #-04
$UTIL_CMD_EXECUTE       = 7;                                                    #-04
$UTIL_CMD_LINK          = 8;                                                    #-06
$UTIL_CMD_RPM_INSTALL   = 9;                                                    #-14-16
$UTIL_CMD_RPM_LIST      = 10;                                                   #-14-16

# Trace Constants
$TRC_T = "XMCLSHLT:";                                                           #-03
$TRC_D = "XMCLSHLD:";
$TRC_F = "XMCLSHLF:";

# Error Constants
$ERROR_ALL_NOERROR            = 0;
$ERROR_ALL_UNKNOWN_COMMAND    = 255;
                                                                                #-01
$ERROR_DELETE_NOTFOUND        = 1;                                              #-01
$ERROR_DELETE_ERROR           = 3;                                              #-01

$ERROR_EXISTS_NOTFOUND        = 1;

$ERROR_ISDIR_NOTFOUND         = 1;                                              #-02

$ERROR_MKDIRS_EXISTS          = 1;
$ERROR_MKDIRS_ISFILE          = 2;
$ERROR_MKDIRS_ERROR           = 3;

$ERROR_PERMS_CHMOD            = 1;                                              #-03
$ERROR_PERMS_CHOWN            = 2;                                              #-03
$ERROR_PERMS_NO_FILE          = 3;                                              #-03
$ERROR_PERMS_FILE_NOT_FOUND   = 4;                                              #-03
$ERROR_PERMS_TOUCH            = 5;                                              #-11
                                                                                #-03
$ERROR_COPY_NO_SOURCE_FILE           = 1;                                       #-04
$ERROR_COPY_NO_TARGET_FILE           = 2;                                       #-04
$ERROR_COPY_SOURCE_FILE_NOT_FOUND    = 3;                                       #-04
$ERROR_COPY_EXECUTE                  = 4;                                       #-04
                                                                                #-04
$ERROR_EXEC_NO_COMMAND               = 1;                                       #-04
$ERROR_EXEC_EXECUTE                  = 2;                                       #-04
                                                                                #-04
$ERROR_LINK_NO_TARGET                = 1;                                       #-06
$ERROR_LINK_NO_ALIAS                 = 2;                                       #-06
$ERROR_LINK_TARGET_NOTFOUND          = 3;                                       #-06
$ERROR_LINK_ALIAS_EXISTS             = 4;                                       #-06
$ERROR_LINK_FAILS                    = 5;                                       #-06
$ERROR_LINK_BAD_LINK_TYPE            = 6;                                       #-06
                                                                                #-06
# start -14
$ERROR_RPM_LIST_NO_RPM_FILE_PARM         = 1;
$ERROR_RPM_LIST_NO_INVENTORY_FILE_PARM   = 2;
$ERROR_RPM_LIST_NO_RPM_FILE              = 3;
$ERROR_RPM_LIST                          = 4;
$ERROR_RPM_NO_DIRS                       = 5;

$ERROR_RPM_INSTALL_NO_RPM_FILE_PARM         = 1;
$ERROR_RPM_INSTALL_NO_RPM_FILE              = 2;
$ERROR_RPM_INSTALL_QUERY_NAME               = 3;
$ERROR_RPM_INSTALL_PACKAGE_NAME             = 4;
$ERROR_RPM_INSTALL                          = 5;
# end -14

# Program Data
$me = $0;
$me =~ s/^(.*)\///;
$me =~ s/.pl$//;
$cmd = $UTIL_CMD_NONE;
$traceCmd = "actmTrace";                                                        #-13

################################################################################
######                          SHARED METHODS                            ######
################################################################################

#
#
#
sub trace
{
    my $text = shift;
    chomp($text);

    if ("actmTrace" eq "$traceCmd")                                             #-13
    {
       system(("$traceCmd", "$text"));
    }
    else
    {
       print(STDOUT "$text\n");
    }
}

#
# usage
#
sub usage
{
    print(<<EOM);
Usage: actmUtil [-help] command=cmd [source=sfile] [target=tfile] [...]
   cmd   an integer value that represent a command
            1 - exists()
            2 - mkdirs()
   any other arguments are optional and will vary from command to command
EOM
}

################################################################################
######                          UTILITY METHODS                           ######
################################################################################

# start -01
#
#
sub doDelete
{
   trace("$TRC_D -> $me.doDelete()");
   my $file = shift;

   trace("$TRC_T -> $me.doDelete() - [$file]");                                 #-08

   my $rc = $ERROR_DELETE_NOTFOUND;
   if ( (-e "$file") || (-l "$file") )                                          #-08
   {
      my $error = `rm -f '$file' 2>&1`;                                         #-08-09
      unless (defined($error)) { $error = ""; }                                 #-05
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doDelete() - rm error is \"$error\"");
         $rc = $ERROR_DELETE_ERROR
      }
   }
   trace("$TRC_D <- $me.doDelete() - rc = $rc");
   return($rc);
}
# end -01

#
#
#
sub doExists
{
   trace("$TRC_D -> $me.doExists()");
   my $file = shift;

   trace("$TRC_T -> $me.doExists() - [$file]");                                 #-08

   my $rc = $ERROR_EXISTS_NOTFOUND;
   # Does the path directory exist?                                             #-02
   if (-e "$file")
   {
      $rc = $ERROR_ALL_NOERROR;
   }
   trace("$TRC_D <- $me.doExists() - rc = $rc");
   return($rc);
}

# start -02
#
#
sub doIsDir
{
   trace("$TRC_D -> $me.doIsDir()");
   my $file = shift;

   trace("$TRC_T -> $me.doIsDir() - [$file]");                                  #-08

   my $rc = $ERROR_ISDIR_NOTFOUND;
   # Does a directory exist?
   if (-d "$file")
   {
      $rc = $ERROR_ALL_NOERROR;
   }
   trace("$TRC_D <- $me.doIsDir() - rc = $rc");
   return($rc);
}
# end -02

#
#
#
sub doMkDirs
{
   trace("$TRC_D -> $me.doMkDirs()");
   my $file = shift;

   trace("$TRC_T -> $me.doMkDirs() - [$file]");                                 #-08

   my $rc = $ERROR_ALL_NOERROR;
   # Does it already exist?
   my $exists = (-e "$file");
   # Is it a directory?
   my $isDir = (-d "$file");
   if ($exists)
   {
      if ($isDir)
      {
         $rc = $ERROR_MKDIRS_EXISTS;
      }
      else
      {
         $rc = $ERROR_MKDIRS_ISFILE;
      }
   }
   else
   {
      my $error = `mkdir -p '$file' 2>&1`;                                      #-08-09
      unless (defined($error)) { $error = ""; }                                 #-05
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doMkDirs() - mkdir error is \"$error\"");
         $rc = $ERROR_MKDIRS_ERROR
      }
   }
   trace("$TRC_D <- $me.doMkDirs() - rc = $rc");
   return($rc);
}

# start -03
#
#
#
sub doPerms
{
   my ($target, $owner, $group, $flags) = @_;                                   #-08

   unless (defined($target)) { $target = ""; }                                  #-08
   unless (defined($owner))  { $owner  = ""; }                                  #-08
   unless (defined($group))  { $group  = ""; }                                  #-08
   unless (defined($flags))  { $flags  = ""; }                                  #-08

   trace("$TRC_T -> $me.doPerms() - [$target], [$owner], [$group], [$flags]");  #-08

   my $rc = $ERROR_ALL_NOERROR;

   # File must be specified
   if ($target eq "")                                                           #-08
   {
      trace("$TRC_F    $me.doPerms() - file argument missing");
      $rc = $ERROR_PERMS_NO_FILE;
   }

   # See if the file/path exists (if no previous error).
   if ($ERROR_ALL_NOERROR == $rc)
   {
      unless ( -e $target )                                                     #-08
      {
         trace("$TRC_F    $me.doPerms() - file [$target] does not exist");      #-08
         $rc = $ERROR_PERMS_FILE_NOT_FOUND;
      }
   }

   # Do the flags portion, if necessary, and if no previous error.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($flags ne "")
      {
         my $error = `chmod '$flags' '$target' 2>&1`;                           #-09
         unless (defined($error)) { $error = ""; }                              #-05
         chop $error;
         $rc = $? >> 8;
         if ($rc)
         {
            trace("$TRC_F    $me.doPerms() - chmod error is \"$error\"");
            $rc = $ERROR_PERMS_CHMOD;
         }
      }
   }

   # Do the ownership portion, if necessary, and if no previous error.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ( ($owner ne "") || ($group ne "") )
      {
         my $error = `chown '$owner:$group' '$target' 2>&1`;                    #-09
         unless (defined($error)) { $error = ""; }                              #-05
         chop $error;
         $rc = $? >> 8;
         if ($rc)
         {
            trace("$TRC_F    $me.doPerms() - chown error is \"$error\"");
            $rc = $ERROR_PERMS_CHOWN;
         }
      }
   }

   # start -11
   # Update the file's timestamp so mirroring and backup will pick up the
   # change
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `touch -m '$target' 2>&1`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doPerms() - touch error is \"$error\"");
         $rc = $ERROR_PERMS_TOUCH;
      }
   }
   # end -11

   trace("$TRC_T <- $me.doPerms() - rc = $rc");
   return($rc);
}
# end -03

# start -04
#
#
#
sub doCopy
{
   my ($source,$target) = @_;

   unless (defined($source)) { $source = ""; }                                  #-05
   unless (defined($target)) { $target = ""; }                                  #-05

   trace("$TRC_T -> $me.doCopy() - [$source], [$target]");

   my $rc = $ERROR_ALL_NOERROR;

   # Source file must be specified
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($source eq "")
      {
         trace("$TRC_F    $me.doCopy() - source file argument missing");
         $rc = $ERROR_COPY_NO_SOURCE_FILE;
      }
   }

   # Target file must be specified
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($target eq "")
      {
         trace("$TRC_F    $me.doCopy() - target file argument missing");
         $rc = $ERROR_COPY_NO_TARGET_FILE;
      }
   }

   # See if the source file/path exists
   if ($ERROR_ALL_NOERROR == $rc)
   {
      unless ( -e $source )
      {
         trace("$TRC_F    $me.doCopy() - source file does not exist");
         $rc = $ERROR_COPY_SOURCE_FILE_NOT_FOUND;
      }
   }

   # Attempt the copy.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `cp -v --preserve=timestamps '$source' '$target' 2>&1`;       #-09-10
      unless (defined($error)) { $error = ""; }                                 #-05
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doCopy() - cp error is \"$error\"");
         $rc = $ERROR_COPY_EXECUTE;
      }
   }

   trace("$TRC_T <- $me.doCopy() - rc = $rc");
   return($rc);
}

#
#
#
sub doExecute
{
   my ($commandString) = @_;

   unless (defined($commandString)) { $commandString = ""; }                    #-05

   trace("$TRC_T -> $me.doExecute() - [$commandString]");

   my $rc = $ERROR_ALL_NOERROR;

   # Command string must be specified
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($commandString eq "")
      {
         trace("$TRC_F    $me.doExecute() - command string argument missing");
         $rc = $ERROR_EXEC_NO_COMMAND;
      }
   }

   # Attempt the command.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `$commandString 2>&1`;
      unless (defined($error)) { $error = ""; }                                 #-05
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doExecute() - command rc = $rc - execution error is \"$error\"");  #-07
         $rc = $ERROR_EXEC_EXECUTE;
      }
   }

   trace("$TRC_T <- $me.doExecute() - rc = $rc");
   return($rc);
}
# end -04

# start -06
#
#
#
sub doLink
{
   my ($alias,$target,$symbolic) = @_;

   unless (defined($alias))    { $alias    = ""; }
   unless (defined($target))   { $target   = ""; }
   unless (defined($symbolic)) { $symbolic = ""; }

   trace("$TRC_T -> $me.doLink() - [$alias], [$target]");

   my $rc = $ERROR_ALL_NOERROR;

   # Alias file must be specified
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($alias eq "")
      {
         trace("$TRC_F    $me.doLink() - alias name argument missing");
         $rc = $ERROR_LINK_NO_ALIAS;
      }
   }

   # Target file must be specified
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($target eq "")
      {
         trace("$TRC_F    $me.doLink() - target file argument missing");
         $rc = $ERROR_LINK_NO_TARGET;
      }
   }

   # Create the proper value for the symbolic link option
   my $linkType = "";
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($symbolic eq "")
      {
         $linkType = "";
      }
      elsif ($symbolic eq "true")
      {
         $linkType = "-s";
      }
      else
      {
         trace("$TRC_F    $me.doLink() - unrecognized link type argument: $symbolic");
         $rc = $ERROR_LINK_BAD_LINK_TYPE;
      }
   }

   # See if the target file/path exists
   if ($ERROR_ALL_NOERROR == $rc)
   {
      unless ( -e $target )
      {
         trace("$TRC_F    $me.doLink() - link target file does not exist");
         $rc = $ERROR_LINK_TARGET_NOTFOUND;
      }
   }

   # See if there is already a file where the alias is being created
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ( -e $alias )
      {
         trace("$TRC_F    $me.doLink() - there is a file where the link was to be created");
         $rc = $ERROR_LINK_ALIAS_EXISTS;
      }
   }

   # Attempt the link.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `ln $linkType '$target' '$alias' 2>&1`;                       #-09
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doLink() - cp error is \"$error\"");
         $rc = $ERROR_LINK_FAILS;
      }
   }

   trace("$TRC_T <- $me.doLink() - rc = $rc");
   return($rc);
}
# end -06

# start -14
#
#
#
sub doRpmGetInventory
{
   my ($source, $target) = @_;

   unless (defined($source)) { $source = ""; }
   unless (defined($target)) { $target = ""; }

   trace("$TRC_T -> $me.doRpmGetInventory() - [$source], [$target]");

   my $rc = $ERROR_ALL_NOERROR;

   # RPM file must be specified
   if ($source eq "")
   {
      trace("$TRC_F    $me.doRpmGetInventory() - RPM file argument missing");
      $rc = $ERROR_RPM_LIST_NO_RPM_FILE_PARM;
   }

   # Inventory file must be specified
   if ($target eq "")
   {
      trace("$TRC_F    $me.doRpmGetInventory() - inventory file argument missing");
      $rc = $ERROR_RPM_LIST_NO_INVENTORY_FILE_PARM;
   }

   # Ensure the RPM file exists (if no previous error).
   if ($ERROR_ALL_NOERROR == $rc)
   {
      unless ( -f $source )
      {
         trace("$TRC_F    $me.doRpmGetInventory() - RPM file [$source] does not exist");
         $rc = $ERROR_RPM_LIST_NO_RPM_FILE;
      }
   }

   # Query the RPM file to obtain the inventory.
   my $workFile = "/tmp/iqzm.rpm.workfile.trm";
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `rpm -qpl --dump '$source' 2>&1 | tee $workFile`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doRpmGetInventory() - rc is $rc");
         trace("$TRC_F    $me.doRpmGetInventory() - error is \"$error\"");
         $rc = $ERROR_RPM_LIST;
      }
   }

   # Eliminate directories and reduce to just file names.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `cat $workFile | grep -v '^.* 4096 .* 00000000000000000000000000000000 ' | sed 's/ .*\$//' > $target`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doRpmGetInventory() - rc is $rc");
         trace("$TRC_F    $me.doRpmGetInventory() - error is \"$error\"");
         $rc = $ERROR_RPM_NO_DIRS;
      }
   }

   # Merge in catalog files.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $catDir = "/var/lib/rpm/";
      my $error = `ls -Al '$catDir' | grep -v '^total' | sed 's:^.* :\/var\/lib\/rpm\/:' >> $target`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doRpmGetInventory() - rc is $rc");
         trace("$TRC_F    $me.doRpmGetInventory() - error is \"$error\"");
         $rc = $ERROR_RPM_NO_DIRS;
      }
   }

   trace("$TRC_T <- $me.doRpmGetInventory() - rc = $rc");
   return($rc);
}

#
#
#
sub doRpmInstall
{
   my ($source) = @_;

   unless (defined($source)) { $source = ""; }

   trace("$TRC_T -> $me.doRpmInstall() - [$source]");

   my $rc = $ERROR_ALL_NOERROR;

   my $fileData = "";
   my $packageName = "";
   my $packageData = "";

   # RPM file must be specified
   if ($source eq "")
   {
      trace("$TRC_F    $me.doRpmInstall() - RPM file argument missing");
      $rc = $ERROR_RPM_INSTALL_NO_RPM_FILE_PARM;
   }

   # Ensure the RPM file exists (if no previous error).
   if ($ERROR_ALL_NOERROR == $rc)
   {
      unless ( -f $source )
      {
         trace("$TRC_F    $me.doRpmInstall() - RPM file [$source] does not exist");
         $rc = $ERROR_RPM_INSTALL_NO_RPM_FILE;
      }
   }

   # Query the RPM file to determine the package name.  Take
   # only the first 3 lines for later comparison of RPM installation level.
   # The 4th line contains the install date and will differ.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `rpm -qip '$source' 2>&1 | head -n 3`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doRpmInstall() - rpm query error is \"$error\"");
         $rc = $ERROR_RPM_INSTALL_QUERY_NAME;
      }
      else
      {
         $fileData = $error;
         my $nameData = $fileData;
         # Find "Name" and throw away up to the colon, then the colon and any
         # blanks before the package name
         $nameData =~ s/^.*Name[^:]*:[ ]*//;
         # Now get only the first word
         my @words = split(/ /, $nameData);
         $packageName = shift @words;
         unless (defined($packageName)) { $packageName = ""; }
         trace("$TRC_F    $me.doRpmInstall() - the package name is \"$packageName\"");
      }
   }

   # Ensure the package name we determined is not "empty".
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($packageName eq "")
      {
         trace("$TRC_F    $me.doRpmInstall() - could not determine package name");
         $rc = $ERROR_RPM_INSTALL_PACKAGE_NAME;
      }
   }

   # Query the RPM database to determine the package is installed. Take
   # only the first 3 lines for later comparison of RPM installation level.
   # The 4th line contains the install date and will differ.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      my $error = `rpm -qi $packageName 2>&1 | head -n 3`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doRpmInstall() - the RPM is not installed");
      }
      else
      {
         $packageData = $error;
      }
   }

   # See if the RPM file info matches the installed package info. If it does
   # we are installed at this level. If not, go ahead and install the RPM.
   if ($ERROR_ALL_NOERROR == $rc)
   {
      if ($packageData eq $fileData)
      {
         trace("$TRC_F    $me.doRpmInstall() - package is already installed");
      }
      else
      {
         my $error = `rpm -Uhv --force --nodeps --excludedocs '$source' 2>&1`;  #-15
         unless (defined($error)) { $error = ""; }
         chop $error;
         $rc = $? >> 8;
         if ($rc)
         {
            trace("$TRC_F    $me.doRpmInstall() - rc is $rc");
            trace("$TRC_F    $me.doRpmInstall() - rpm error is \"$error\"");
            $rc = $ERROR_RPM_INSTALL;
         }
      }
   }

   trace("$TRC_T <- $me.doRpmInstall() - rc = $rc");
   return($rc);
}
# end -14

################################################################################
######                           SCRIPT CODE                              ######
################################################################################

# Process the arguments
GetOptions(
      'help'            =>\$help,
      'command=i'       =>\$cmd,
      'tracecommand:s'  =>\$traceCmd,
      'target:s'        =>\$target,
# start -04
      'source:s'        =>\$source,
      'commandString:s' =>\$commandString,
# end -04
# start -06
      'alias:s'         =>\$alias,
      'symbolic:s'      =>\$symbolic,
# end -06
# start -03
      'owner:s'         =>\$owner,
      'group:s'         =>\$group,
      'flags:s'         =>\$flags
# end -03
);

unless (defined($traceCmd )) { $traceCmd = "actmTrace"; }                       #-16

trace("$TRC_T -> $me(@ARGV)");

unless($> == 0) {
   trace("$TRC_F    This must be run as root.");
}

if ($help)
{
   usage();
   exit;
}

my $rc = $ERROR_ALL_NOERROR;

# Route the command
if ($cmd == $UTIL_CMD_EXISTS)
{
   $rc = doExists($target);
}
elsif ($cmd == $UTIL_CMD_MKDIRS)
{
   $rc = doMkDirs($target);
}
elsif ($cmd == $UTIL_CMD_DELETE)                                                #01
{                                                                               #01
   $rc = doDelete($target);                                                     #01
}                                                                               #01
elsif ($cmd == $UTIL_CMD_ISDIR)                                                 #02
{                                                                               #02
   $rc = doIsDir($target);                                                      #02
}                                                                               #02
elsif ($cmd == $UTIL_CMD_PERMS)                                                 #03
{                                                                               #03
   $rc = doPerms($target,$owner,$group,$flags);                                 #03
}                                                                               #03
elsif ($cmd == $UTIL_CMD_COPY)                                                  #04
{                                                                               #04
   $rc = doCopy($source,$target);                                               #04
}                                                                               #04
elsif ($cmd == $UTIL_CMD_EXECUTE)                                               #04
{                                                                               #04
   $rc = doExecute($commandString);                                             #04
}                                                                               #04
elsif ($cmd == $UTIL_CMD_LINK)                                                  #06
{                                                                               #06
   $rc = doLink($alias,$target,$symbolic);                                      #06
}                                                                               #06
elsif ($cmd == $UTIL_CMD_RPM_LIST)                                              #14
{                                                                               #14
   $rc = doRpmGetInventory($source,$target);                                    #14
}                                                                               #14
elsif ($cmd == $UTIL_CMD_RPM_INSTALL)                                           #14
{                                                                               #14
   $rc = doRpmInstall($source);                                                 #14
}                                                                               #14
else
{
   # Unknown command
   $rc = $ERROR_ALL_UNKNOWN_COMMAND;
}

trace("$TRC_T <- $me() - rc = $rc");

exit($rc);
