#!/usr/bin/perl -w

#
# 12/21/04 J.Stapels  Added a check for UUCP lock on /dev/modem
# 02/09/06 J.Stapels  Removed the flock from the modem since it's not a valid way of locking a modem.
# 06/27/06 S.Lyons    Modem moved to /dev/stty14 (H9136)
# 06/28/06 S.Lyons    initial stty commands fail on MCP 4  (I2035)
# 08/30/08 S.Lyons    MM: New modem support MT9234MU  (USB modem)
# 10/30/08 S.Lyons    Z: T6804 New internal modem MT9234ZPX
#

use Getopt::Long;
use Fcntl qw(:DEFAULT :flock);
use POSIX qw(:termios_h);
#use POSIX qw(tcflush);

# CONSTANTS
$DEBUG = 0;
( undef, $me ) = $0 =~ /^(?:(.*)\/)?(.*)$/;

# Timeout (in second) for device to become available.
$WRITE_TIMEOUT = 2;
$READ_TIMEOUT = 2;

# Options used to setup device.
#STTY_OPT = "5:0:800014b1:0:3:1c:7f:15:4:0:1:0:11:13:1a:0:12:f:17:16:0:0:0:0:0:0:0:0:0:0:0:0:0:0:0:0";
$STTY_OPT = "57600 time 5 " .
            "-parenb  -parodd cs8     -hupcl  -cstopb  cread    clocal " .
            "crtscts  ignbrk  -brkint -ignpar -parmrk  -inpck   -istrip " .
            "-inlcr   -igncr  -icrnl  -ixon    -ixoff   -iuclc   -ixany " .
            "-imaxbel -opost  -olcuc  -ocrnl  -onlcr   -onocr   -onlret " .
            "-ofill   -ofdel  nl0     cr0     tab0     bs0      vt0 ".
            "ff0      -isig   -icanon -iexten -echo    -echoe   -echok " .
            "-echonl  -noflsh -xcase  -tostop -echoprt -echoctl -echoke";

$DEBUG and print( "DEBUG: base - $STTY_OPT\n" );
# Device to check for a modem on.
@DEVICES = (
    "/dev/modem" ,
    "/dev/ttyS0" ,
    "/dev/ttyS1" ,
    "/dev/ttyS14",
    "/dev/ttyS2" ,
    "/dev/ttyS13",
    "/dev/ttyS15",
    "/dev/ttyS3" ,
    "/dev/ttyS4" ,
    "/dev/ttyS5" ,
    "/dev/ttyS6" ,
    "/dev/ttyS7" ,
    "/dev/ttyS8" ,
    "/dev/ttyS9" ,
    "/dev/ttyS10",
    "/dev/ttyS11",
    "/dev/ttyS12",
    "/dev/ttyS16",
    "/dev/ttyUSB0",
    "/dev/ttyUSB1",
    "/dev/ttyUSB2",
    "/dev/ttyUSB3",
    "/dev/ttyUSB4",
    "/dev/ttyUSB5",
    "/dev/ttyUSB6",
    "/dev/ttyUSB7",
    );

# The modems this script will recognize.
@MODEM_DB = (
  { type   => 2,
    desc   => "MultiTech MT5634ZPX-PCI-U Series 56k",
    query  => "ATI0",
    key    => "MT5634" },

  { type   => 3,
    desc   => "MultiTech MT5600 Series 56k",
    query  => "ATI0",
    key    => "MT5600",
    secure => [ 'ATE0', 'AT#CBP01=dsq2mkl', 'AT#CBN01=-', 'AT#DB1' ] },

  { type => 4,
    desc   => "MultiTech MT2834 Series 33k",
    query  => "ATI2",
    key    => "MT2834",
    secure => [ 'ATE0', 'AT#IMULTI-TECH', 'AT#SMODEMSETUP', 'AT#CBN0dsq2mkl', 'AT-DTN0', 'AT#DB2', 'Y' ] },

  { type => 5,
    desc   => "Microcom DeskPorte",
    query  => "ATI2",
    key    => "eskPort",
    secure => [ 'AT%U0*E0%I0;;dsq2m', 'AT$M2', 'AT', 'AT&W0*W0' ] },

  { type => 6,
    desc   => "MultiTech USB 9234MU Series",
    query  => "ATI0",
    key    => "MT9234MU",},

  { type => 7,
    desc   => "MultiTech internal 9234ZPX Series",
    query  => "ATI0",
    key    => "MT9234ZPX",},
  { type => 1,
    desc   => "Generic Modem",
    query  => "ATZ",
    key    => "OK" },

    );

$ERR_MODEM_NOT_FOUND = 0;
$ERR_MODEM_NOT_SECURE = 101;
$ERR_MODEM_LOCKED = 201;

#
###
#

#
# openModem
#
# Opens the specified serial device after using the stty command to get all the
# parameters.
#
sub openModem
{
    my $device = shift;

    # Get original serial settings
    chomp( $origSettings = `stty -F $device -g` );

    $DEBUG and print( "DEBUG: old  - $origSettings\n" );

    # Set our serial settings
    system( "stty -F $device $STTY_OPT" )
        and die( "Unable to setup $device: $!" );

    # Open modem device
    sysopen( MODEM, "$device", O_RDWR | O_NONBLOCK )
        or die( "Unable to open $device: $!" );

    # Lock modem while we use it.
    #flock( MODEM, LOCK_EX | LOCK_NB )
    #    or exit( $ERR_MODEM_LOCKED );
        #or die( "Couldn't obtain lock on $device: $!" );

    # Set autoflush
    select( ( select( MODEM ), $| = 1 )[0] );
}

#
# closeModem
#
# Closes the specified serial device. If the second parameter evaluates to
# true, then we will not reset the port.
#
sub closeModem
{
    my $device = shift;
    my $no_reset = shift;

    # Unlock the modem.
    flock( MODEM, LOCK_UN )
        or die( "Couldn't obtain unlock on $device: $!" );

    # Flush the serial port otherwise we won't be able to close it if data is
    # buffered.
    tcflush( fileno( MODEM ), TCIOFLUSH );
    close( MODEM );

    return if( $no_reset );

    # Restore original serial settings.
    system( "stty -F $device $origSettings 2>/dev/null 1>&2"),
    system( "stty -F $device $origSettings 2>/dev/null 1>&2")
        and die( "Unable to setup $device: $!" );
}

#
# sendCommand
#
# Sends the specific AT command to the opened port and returns the result.
#
sub sendCommand
{
    my $command = shift;

    # DEBUG STUFF
    $DEBUG and print( "DEBUG: cmd - $command\n" );

    # Replace multiple carriage return with only one.
    $command =~ s/\r*$/\r/;

    my $chars;
    my $win = "";
    vec( $win, fileno( MODEM ), 1 ) = 1;

    # Wait until device becomes ready for writing.
    select( undef, $win, undef, $WRITE_TIMEOUT )
        or die( "Timed out waiting to write to the modem.\n" );

    # Send the string to the modem.
    $chars = syswrite( MODEM, $command, length( $command ) );
    if( ! defined( $chars ) or $chars < length( $command ) ) {
        die "Couldn't write to the modem: $!";
    }


    my $rin = "";
    vec( $rin, fileno( MODEM ), 1 ) = 1;
    my $response;
    my $success = 0;

    # Loop until timeout, or an OK is received.
    while( 1 )
    {
        $chars = select( $rin, undef, undef, $READ_TIMEOUT );
        last unless( $chars );

        $chars = sysread( MODEM, $data, 1024 );

        die( "Could not read from modem: $!" )
            unless( defined $chars and $chars );

        $response .= $data;
    }

    # No response means no modem.
    return( undef )
        unless( $response );

    # Cleanup linefeeds and newlines.
    $response =~ s/\r/\n/g;
    $response =~ s/\n+/\n/g;
    $response =~ s/^\n//;

    # DEBUG STUFF
    if( $DEBUG ) {
        my @debug = split( /\n/, $response );
        $line = 0; for( @debug ) { printf( "DEBUG: (%-2d) %s\n", ++$line, $_ ); }
    }

    return( $response );
}

#
# queryModem
#
# Tries to determine what kind of modem is on the device.
#
sub queryModem
{
    my $device = shift;

    # If stty fails to query the device, then we don't even want to try to
    # open it.
    if( system( "stty -F $device 2> /dev/null 1>&2" ) ) {
        return;
    }

    # Open device.
    openModem( $device );

    my $modemFound;
    for( @MODEM_DB )
    {
        $ret = sendCommand( $_->{"query"} ) ;

        next unless( $ret );

        if( $ret =~ m/$_->{"key"}/ ) {
            $modemFound = $_;

            if( $reset ) {
                sendCommand( $reset );
            }

            last;
        }
    }

    # Close the device.
    closeModem( $device, ($setup && $modemFound) );

    return( $modemFound );
}

#
# secureModem
#
# Attempt to secure the modem by setting a call-in password. Specific support
# for MegaMouth.
#
sub secureModem
{
    my $device = shift;
    my $modemRef = queryModem( $device );

    if( ! $modemRef ) {
        print( "$device - Unable to determine modem.\n" );
        return( 0 );
    }

    print( "Found: $modemRef->{desc}\n" );

    if( ! defined( $modemRef->{'secure'} ) ) {
        print( "$device - Do not know how to secure modem.\n" );
        return( 0 );
    }

    openModem( $device );

    my $success = 1;
    for $cmd ( @{ $modemRef->{"secure"} } )
    {
        unless( sendCommand( $cmd ) ) {
            $success = 0;
            print( "Secure failed on sending $cmd\n" );

            # Hack for MT2834 modem :(.
            if( $cmd eq "Y" ) { $success = 2; }
        }
    }

    closeModem( $device );

    return( $success );
}

sub usage
{
    print <<EOM;
Usage:
    $me [--device DEVICE] --query
    $me [--device DEVICE] --setup
    $me [--device DEVICE] --secure

Description:
    TODO

Command Line Parameters:
    TODO

EOM
}

# Quick and dirty UUCP lock check
sub checkUUCP
{
    my $device = shift;

    # Step 1: Follow the symlink back to it's source.
    while( -l $device )
    {
        $device = readlink( $device );
    }

    # Step 2: Get the basename of the device (Obvious problem with devfs systems)
    my ($lockName) = $device =~ m{^(?:.*/)?(.*)$};
    $lockName = "/var/lock/LCK..$lockName";

    # Step 3: Check to see if the lock exists
    if( -f $lockName )
    {
        # Step 4a: Make sure the process ID is valid.
        open( LCK, "$lockName" );
        my $lockPid = <LCK>;
        close( LCK );
        chomp( $lockPid );

        if( kill( 0, $lockPid ) )
        {
            # Lock is good, leave the modem alone
            return( 1 );
        }

        # Step 4b: Remove the stale lock.
        unlink( $lockName )
            or print( "Failed to remove stale lock on $lockName\n" );
    }

    return( 0 );
}


GetOptions(
    "device=s"  => \$device,
    "query"     => \$query,
    "setup"     => \$setup,
    "secure"    => \$secure,
    "reset=s"   => \$reset,
    "spawn"     => \$spawn
    );

if( $spawn )
{
    $pid = fork;

    if( $pid )
    {
        print( "Spawned into background.\n" );
        exit( 0 );
    }
}

if( ! $device ) {
    @devices = @DEVICES;
} else {
    $devices[0] = $device;
}


# Added a timeout in the event that a system call hangs
$TIMEOUT = 60*5;
eval {
    local $SIG{ALRM} = sub { die "alarm\n" }; # NB : \n required
    alarm $TIMEOUT;

    # Check for UUCP lock
    my $locked = 0;
    for( @devices )
    {
        if( checkUUCP( $_ ) )
        {
            print( "$_ - LOCKED\n" );
            $locked++;
        }
    }

    if( $locked )
    {
        exit( $ERR_MODEM_LOCKED );
    }

    if( $query )
    {
        for $dev ( @devices )
        {
            if( $ret = queryModem( $dev ) )
            {
                $modemFound = 1;
                print( "$dev - $ret->{desc}\n" );
                exit( $ret->{'type'} );
            }
        }

        exit( $ERR_MODEM_NOT_FOUND );
    }

    if( $setup )
    {
        $modemFound = 0;
        for $dev ( @devices )
        {
            if( $ret = queryModem( $dev ) )
            {
                $modemFound = 1;
                print( "$dev - $ret->{desc}\n" );

                unless( $dev eq "/dev/modem" ) {
                    system( "ln -sf $dev /dev/modem" );
                }
                exit( $ret->{'type'} );
            }
        }

        exit( $ERR_MODEM_NOT_FOUND );
    }

    if( $secure )
    {
        $modemSecure = 0;
        for $dev ( @devices )
        {
            $ret = secureModem( $dev );

            if( $ret == 1 ) {
                $modemSecure = 1;
                print( "$dev - Modem Secure\n" );
            }

            # Hack for MT2834 modem :(.
            if( $ret == 2 ) {
                $modemSecure = 1;
                print( "$dev - Modem Secure (or already secure)\n" );
            }
        }

        if( ! $modemSecure ) {
            exit( $ERR_MODEM_NOT_SECURE );
        }

        exit( 0 );
    }

    alarm 0;
};
if ($@) {
    die unless $@ eq "alarm\n";

    print( "ERROR! Timed out waiting for program to complete." );
    exit( -1 );
}

usage;
exit( 0 );
