#!/bin/bash
#
# This script reads settings from properties files written by the CCFW
# Network Settings java code (com.ibm.hwmca.base.settings.network.NetworkSettings)
# and updates the OS' network configuration files.
#
# INPUT:
#   Parm 1: The directory location of the properties files (eg. /tmp/console)
#   Parm 2: The ccfw network configuration data directory (eg. /console/data/network)
#
#   Additional parameters are treated as interface specific properties files
#   (eg. NetworkInterfaceInformation-eth0.properties)
#
# OUTPUT:
#   This script produces no output.
#
# Notes:
#   This script creates/modifies the following system network configuration files:
#
#   /etc/HOSTNAME
#   /etc/hosts
#   /etc/resolv.conf
#   /etc/sysconfig/network/config
#   /etc/sysconfig/network/routes
#   /etc/sysconfig/network/ifcfg-eth*
#   /etc/sysconfig/network/ifcfg-tr*
#   /etc/dhcpd.conf
#   <ccfw networkconf directory>/media_speed.conf
#   <ccfw networkconf directory>/ipv6_autoconf.conf
#   <ccfw networkconf directory>/ipv6_privext.conf
#   <ccfw networkconf directory>/dhcpv6_client.conf
#
# Module History:
#   05/16/2007  Mike Clark          Initial release
#   06/29/2007  Mike Clark          Added dynamic resolv.conf updating control
#   08/07/2007  Mike Clark          Added STARTMODE variable
#   08/07/2007  Mike Clark          Store IPv6 autoconfig/privext settings in networkconf directory
#   08/21/2007  Mike Clark          Check DNS_ENABLED before writing /etc/resolv.conf
#   09/18/2007  Mike Clark      -01 Unset IPv6 address variables after using
#   10/06/2007  Mike Clark          Write entries for autonegotiate media speeds
#   11/28/2007  Mike Clark          Add DHCPv6 Client support
#   12/11/2007  Mike Clark      -02 Remove 1000/half duplex setting.
#   08/07/2008  Mike Clark      -03 Allow modified MTU's to persist
#

# Usage: updateNetworkSettings.sh <directory> <ccfw networkconf directory> [Network Interface properties files]

#actzTrace() {
#    echo "$*" 1>&2
#}

##############################################################################
# EXECUTION START
##############################################################################

actzTrace "XSETNETT: updateNetworkSettings.sh $*"

INPUT_DIR=${1%%/}
NETWORK_CONF_DIR=${2%%/}
shift
shift
NET_IF_INFO_FILES="$*"
NETWORK_INFO_FILE="$INPUT_DIR/NetworkInfo.properties"
BACKUP_TARBALL="$NETWORK_CONF_DIR/old_network_config.tgz"
MEDIA_SPEED_FILE="$NETWORK_CONF_DIR/media_speed.conf"
IPV6_AUTOCONF_FILE="$NETWORK_CONF_DIR/ipv6_autoconf.conf"
IPV6_PRIVEXT_FILE="$NETWORK_CONF_DIR/ipv6_privext.conf"
DHCPV6_CLIENT_FILE="$NETWORK_CONF_DIR/dhcpv6_client.conf"

##############################################################################
# Backup all the files we are about to change
##############################################################################
tar czvf $BACKUP_TARBALL /etc/HOSTNAME /etc/hosts /etc/resolv.conf /etc/sysconfig/network/config /etc/sysconfig/network/routes /etc/sysconfig/network/ifcfg-eth* /etc/sysconfig/network/ifcfg-tr* /etc/dhcpd.conf $MEDIA_SPEED_FILE $IPV6_AUTOCONF_FILE $IPV6_PRIVEXT_FILE $DHCPV6_CLIENT_FILE

rm -f $MEDIA_SPEED_FILE
rm -f $IPV6_AUTOCONF_FILE
rm -f $IPV6_PRIVEXT_FILE
rm -f $DHCPV6_CLIENT_FILE

##############################################################################
# Update files from NetworkInfo.properties
##############################################################################

# source the properties file to load all the variables
. $NETWORK_INFO_FILE

# hostname/domain
if [ -z "$DOMAIN" ]; then
    echo "$HOSTNAME" > /etc/HOSTNAME
else
    echo "$HOSTNAME.$DOMAIN" > /etc/HOSTNAME
fi

# /etc/hosts entries
i=0
rm -f /etc/hosts
echo "127.0.0.1 localhost.localdomain localhost" > /etc/hosts
while true; do
    var="HOSTS_ENTRY_$i"
    eval entry=\$$var

    if [ -z "$entry" ]; then
        break;
    fi

    actzTrace "XSETNETF: adding host entry: $entry"
    echo "$entry" >> /etc/hosts
    i=$((i+1))
done

# Check if we are set to get DNS settings from DHCP
dynamic_update=$(grep "MODIFY_RESOLV_CONF_DYNAMICALLY=" /etc/sysconfig/network/config | cut -d'=' -f 2)
if [ "$USE_DHCP_DNS_SETTINGS" = "false" ]; then
    if [ "$dynamic_update" = '"yes"' ]; then
        actzTrace "XSETNETF: disabling dynamic resolv.conf updating"
        sed -e 's/MODIFY_RESOLV_CONF_DYNAMICALLY=.*$/MODIFY_RESOLV_CONF_DYNAMICALLY="no"/' /etc/sysconfig/network/config > /tmp/_tmp_network_config
        mv -f /tmp/_tmp_network_config /etc/sysconfig/network/config
    else
        actzTrace "XSETNETF: dynamic resolv.conf updating already disabled"
    fi
elif [ "$USE_DHCP_DNS_SETTINGS" = "true" ]; then
    if [ "$dynamic_update" = '"no"' ]; then
        actzTrace "XSETNETF: enabling dynamic resolv.conf updating"
        sed -e 's/MODIFY_RESOLV_CONF_DYNAMICALLY=.*$/MODIFY_RESOLV_CONF_DYNAMICALLY="yes"/' /etc/sysconfig/network/config > /tmp/_tmp_network_config
        mv -f /tmp/_tmp_network_config /etc/sysconfig/network/config
    else
        actzTrace "XSETNETF: dynamic resolv.conf updating already enabled"
    fi
else
    actzTrace "XSETNETF: invalid value for USE_DHCP_DNS_SETTINGS: $dynamic_update"
fi

# DNS information in /etc/resolv.conf
rm -f /etc/resolv.conf
if [ "$DNS_ENABLED" = "true" ]; then
    if [ -n "$DOMAIN_SUFFIXES" ]; then
        echo "search $DOMAIN_SUFFIXES" > /etc/resolv.conf
    fi

    i=0
    while true; do
        var="DNS_SERVER_$i"
        eval entry=\$$var

        if [ -z "$entry" ]; then
            break;
        fi

        actzTrace "XSETNETF: adding $entry to resolv.conf"
        echo "nameserver $entry" >> /etc/resolv.conf
        i=$((i+1))
    done
else
    actzTrace "XSETNETF: DNS is disabled"
fi

# Routing information in /etc/sysconfig/network/routes
rm -f /etc/sysconfig/network/routes
if [ -n "$DEFAULT_GATEWAY" ]; then
    actzTrace "XSETNETF: setting default gateway to $DEFAULT_GATEWAY $DEFAULT_GATEWAY_DEV"
    echo "default $DEFAULT_GATEWAY 0.0.0.0 $DEFAULT_GATEWAY_DEV" > /etc/sysconfig/network/routes
fi
i=0
while true; do
    var="ROUTE_ENTRY_${i}_DEST"
    eval dest=\$$var

    if [ -z "$dest" ]; then
        break;
    fi

    var="ROUTE_ENTRY_${i}_MASK"
    eval mask=\$$var
    var="ROUTE_ENTRY_${i}_GATEWAY"
    eval gateway=\$$var
    var="ROUTE_ENTRY_${i}_DEV"
    eval dev=\$$var

    actzTrace "XSETNETF: adding route: $dest $gateway $mask $dev"
    echo "$dest $gateway $mask $dev" >> /etc/sysconfig/network/routes
    i=$((i+1))
done

##############################################################################
# Write the config files from information in NetworkInterfaceInformation-<ifname>.properties files
##############################################################################
etc_hosts_hostname_addr=""
if [ -n "$NET_IF_INFO_FILES" ]; then

    have_dhcp_server=0
    dhcp_server_index=0
    for file in $NET_IF_INFO_FILES; do
        . "$INPUT_DIR/$file"

        actzTrace "XSETNETF: processing $file"

        config_file="/etc/sysconfig/network/ifcfg-$INTERFACE_NAME"

        #-03 start
        # Try to keep the same MTU value
        mtu_line=$(grep -m 1 "MTU" $config_file)
        if [ -z "$mtu_line" ]; then
            mtu_line="MTU='1500'"
        fi
        #-03 end

        if [ "$IS_DHCP_CLIENT" = "true" ]; then
            actzTrace "XSETNETF: setting DHCP"
            echo "BOOTPROTO='dhcp'" > $config_file
        else
            actzTrace "XSETNETF: setting static address: $IPV4_ADDRESS/$IPV4_NETMASK"
            echo "BOOTPROTO='static'" > $config_file
            echo "IPADDR='$IPV4_ADDRESS'" >> $config_file
            echo "NETMASK='$IPV4_NETMASK'" >> $config_file
            if [ "$INTERFACE_NAME" = "$DEFAULT_GATEWAY_DEV" ]; then
                if [ "$IPV4_ADDRESS" != "0.0.0.0" -a "$IPV4_ADDRESS" != "127.0.0.1" ]; then
                    actzTrace "XSETNETF: Found a static address for the gateway interface.. using it in /etc/hosts"
                    etc_hosts_hostname_addr="$IPV4_ADDRESS"
                else
                    actzTrace "XSETNETF: The gateway interface address was $IPV4_ADDRESS.. not using"
                fi
            elif [ -z "$etc_hosts_hostname_addr" ]; then
                if [ "$IPV4_ADDRESS" != "0.0.0.0" -a "$IPV4_ADDRESS" != "127.0.0.1" ]; then
                    actzTrace "XSETNETF: Found a static address, non-gateway.. might use it in /etc/hosts"
                    etc_hosts_hostname_addr="$IPV4_ADDRESS"
                else
                    actzTrace "XSETNETF: Static address was $IPV4_ADDRESS.. not using"
                fi
            else
                actzTrace "XSETNETF: Already had a decent hostname address for /etc/hosts.. not using $IPV4_ADDRESS"
            fi
        fi
        #-03 start
        echo "$mtu_line" >> $config_file
        #-03 end
        if [ -n "$STARTMODE" ]; then
            actzTrace "XSETNETF: setting startmode to $STARTMODE"
            echo "STARTMODE='$STARTMODE'" >> $config_file
        else
            actzTrace "XSETNETF: setting default startmode of onboot"
            echo "STARTMODE='onboot'" >> $config_file
        fi

        # Add any IPv6 addresses
        i=0
        while true; do
            var="IPV6_ADDRESS_$i"
            eval addr=\$$var

            if [ -z "$addr" ]; then
                break;
            fi

            actzTrace "XSETNETF: adding IPv6 address: $addr"
            echo "IPADDR$i='$addr'" >> $config_file
            unset $var #-01
            i=$((i+1))
        done

        # Check for IPv6 Autoconf and Privacy Extensions
        if [ "$IPV6_AUTOCONFIG" = "true" ]; then
            actzTrace "XSETNETF: ipv6 autoconf enabled for $INTERFACE_NAME"
            echo "$INTERFACE_NAME" >> $IPV6_AUTOCONF_FILE
        fi

        if [ "$IPV6_USE_PRIV_EXT" = "true" ]; then
            actzTrace "XSETNETF: ipv6 privacy extensiosn enabled for $INTERFACE_NAME"
            echo "$INTERFACE_NAME" >> $IPV6_PRIVEXT_FILE
        fi

        # Check for DHCPv6 support
        if [ "$IS_DHCPV6_CLIENT" = "true" ]; then
            actzTrace "XSETNETF: DHCPv6 client enabled for $INTERFACE_NAME"
            echo "$INTERFACE_NAME" >> $DHCPV6_CLIENT_FILE
        fi

        # Update the media speed file
        case "$MEDIA_SPEED" in
            0)
                actzTrace "XSETNETF: media speed for $INTERFACE_NAME was autoneg"
                echo "/usr/sbin/ethtool -s $INTERFACE_NAME autoneg on" >> $MEDIA_SPEED_FILE
                ;;
            1)
                actzTrace "XSETNETF: media speed for $INTERFACE_NAME was 10/half"
                echo "/usr/sbin/ethtool -s $INTERFACE_NAME autoneg off speed 10 duplex half" >> $MEDIA_SPEED_FILE
                ;;
            2)
                actzTrace "XSETNETF: media speed for $INTERFACE_NAME was 10/full"
                echo "/usr/sbin/ethtool -s $INTERFACE_NAME autoneg off speed 10 duplex full" >> $MEDIA_SPEED_FILE
                ;;
            3)
                actzTrace "XSETNETF: media speed for $INTERFACE_NAME was 100/half"
                echo "/usr/sbin/ethtool -s $INTERFACE_NAME autoneg off speed 100 duplex half" >> $MEDIA_SPEED_FILE
                ;;
            4)
                actzTrace "XSETNETF: media speed for $INTERFACE_NAME was 100/full"
                echo "/usr/sbin/ethtool -s $INTERFACE_NAME autoneg off speed 100 duplex full" >> $MEDIA_SPEED_FILE
                ;;
            5)
                #-02 start
                actzTrace "XSETNETF: media speed for $INTERFACE_NAME was 1000/full"
                echo "/usr/sbin/ethtool -s $INTERFACE_NAME autoneg off speed 1000 duplex full" >> $MEDIA_SPEED_FILE
                #-02 end
                ;;
            6)
                actzTrace "XSETNETF: media speed for $INTERFACE_NAME was 1000/full"
                echo "/usr/sbin/ethtool -s $INTERFACE_NAME autoneg off speed 1000 duplex full" >> $MEDIA_SPEED_FILE
                ;;
            *)
                actzTrace "XSETNETF: invalid media speed for $INTERFACE_NAME : $MEDIA_SPEED"
                ;;
        esac

        # check for dhcp server info
        if [ "$IS_DHCP_SERVER" = "true" ]; then
            have_dhcp_server=1
            dhcp_server_addrs[$dhcp_server_index]="$IPV4_ADDRESS"
            dhcp_server_subnets[$dhcp_server_index]="$DHCP_SERVER_SUBNET"
            dhcp_server_masks[$dhcp_server_index]="$DHCP_SERVER_MASK"
            dhcp_server_ranges[$dhcp_server_index]="$DHCP_SERVER_RANGE"
            dhcp_server_index=$((dhcp_server_index+1))
        fi

        # check for post up script to set inteface tso, add default if doesn't exists
        ifup_script="/etc/sysconfig/network/if-up.d/$INTERFACE_NAME-settso"
        if [ ! -e  $ifup_script ]; then
            # create a default script file to turn off tso for the interface
            touch $ifup_script
            echo "ethtool -K $INTERFACE_NAME tso off" > $ifup_script
            chmod 755 $ifup_script
        fi

    done

    # Update /etc/dhcpd.conf if we need to
    if [ "$have_dhcp_server" = "1" ]; then
        actzTrace "XSETNETF: some interface is a dhcp server"

        # Add the header
        cat > /etc/dhcpd.conf <<EOF
# Licensed Internal Code - Property of IBM
#    A sample dhcpd.conf file
#  (Copy this to /etc/dhcpd.conf)
#
#authoritative;
ddns-update-style none;
default-lease-time 600;
max-lease-time infinite;

EOF

        # Add entries for each dhcp server interface we have
        # seq starts at 1
        for i in $(seq ${#dhcp_server_addrs[*]}); do
            addr="${dhcp_server_addrs[$((i-1))]}"
            subnet="${dhcp_server_subnets[$((i-1))]}"
            mask="${dhcp_server_masks[$((i-1))]}"
            range="${dhcp_server_ranges[$((i-1))]}"

            actzTrace "XSETNETF: adding dhcp server config: addr=$addr, subnet=$subnet, mask=$mask, range=$range"

            cat >> /etc/dhcpd.conf <<EOF
subnet $subnet netmask $mask {
# option routers $addr;
  range $range;
}

EOF
        done
    fi
fi

# Add an entry to /etc/hosts for the hostname of this machine
if [ -z "$etc_hosts_hostname_addr" ]; then
    actzTrace "XSETNETF: didn't find a good address to associate with the hostname.. using 127.0.0.1"
    etc_hosts_hostname_addr="127.0.0.1"
fi

if [ -n "$DOMAIN" ]; then
    echo "$etc_hosts_hostname_addr $HOSTNAME.$DOMAIN $HOSTNAME" >> /etc/hosts
else
    echo "$etc_hosts_hostname_addr $HOSTNAME" >> /etc/hosts
fi

