package OSInstall;

use strict;
use warnings;

use OSInstall::Common;
use OSInstall::Platform_Tools;
use OSInstall::XMLHandler;
use File::Path;
require Exporter;

our @ISA = qw(Exporter);

our @EXPORT = qw(
init
); 
our $destroy_flag;
our $force_flag;
our $ssh_keyfile = "";

sub init
{
	my $osi_dir = shift;

	unless (-d $SERVICE_STATE_DIR)
	{
		mkpath $SERVICE_STATE_DIR, 0700 or die "$SERVICE_STATE_DIR : $!\n";
	}

	unless (-d $CONFIG_FILES_DIR)
	{
		mkpath $CONFIG_FILES_DIR, 0700 or die "$CONFIG_FILES_DIR : $!\n";
	}

	unless (-d $CONFIG_DATA_DIR)
	{
		mkpath $CONFIG_DATA_DIR, 0700 or die "$CONFIG_DATA_DIR : $!\n";
	}

	# Make lock dir
	unless (-d $LOCK_DIR) {
		mkpath $LOCK_DIR, 0700 or die "$LOCK_DIR : $!\n";
	}

	# touch our event log
	unless (-f $EVENT_LOG) {
		open TOUCH, ">$EVENT_LOG" or die "$EVENT_LOG : $!\n";
		close TOUCH; 
	}

	# touch our install log
	unless (-f $INSTALL_LOG) {
		open TOUCH, ">$INSTALL_LOG" or die "$INSTALL_LOG : $!\n";
		close TOUCH;
	}

	# touch our undo command stack file
	unless (-f $UNDO_CMD_STACK) {
		open TOUCH, ">$UNDO_CMD_STACK" or die "$UNDO_CMD_STACK : $!\n";
		close TOUCH; 
	}

	OSInstall::Platform_Tools::set_config_rules(); 
	OSInstall::XMLHandler::init($osi_dir); 
}
1;
package OSInstall::CLIHandler;
use OSInstall::Common;
use OSInstall::XMLHandler;
use Getopt::Std;
use Carp;
# CLIHandler package
# Parses the command line, passing parameters to correct OS_install operation subroutine
# OS_install operation subroutines:
# define_client
# define_remote_resource
# define_resource
# define_ctrl_host
# allocate
# deallocate
# netboot
# monitor_installation
# remove

# Aggregate operations:
# install_sysplan
# install_client
# define_sysplan
# delete_sysplan

# subroutine return code constants
use constant E_SYNTAX      => 11;
use constant E_INVALID_XML => 12;
use constant E_INVALID_ATTR => 13;
use constant E_XMLHANDLER => 14;
use constant E_SYSPLAN => 15;
use constant E_NETBOOT => 16;
use constant E_MONITOR => 17;
use constant E_INVALID_VALUE => 18;
use constant E_INVALID_ALLOCATION => 19;
use constant E_INVALID_OBJ_NAME => 20;
use constant E_CTRL_HOST => 21;
use constant E_ALLOCATE => 22;

sub invoke_osinstall 
{ 
	my %cli_opts;
	my @cli_argv = @ARGV;
	my $rc = 0;

	# default HMC SSH key file
	my $HMC_ssh_keyfile = "/home/hscroot/ssh_keys";

	# default IVM SSH key file
	my $IVM_ssh_keyfile = "/home/padmin/.ssh/id_rsa";

	# Join @ARGV array into single string
	my $argv_str = join(' ', @ARGV);

	log_entry(__("=" x 47 . "\n"), 0, 1);
	log_entry(__("OS_install v%s executed with the following parameters: '%s'\n", $OSInstall::Common::VERSION, $argv_str), 0, 1);

	getopts('USFK:lt:vi:o:x:a:d', \%cli_opts);

	die(__("OS_install v$OSInstall::Common::VERSION\nUsage:\nOS_install [-K keyfile_path_name] -i sysplan -x <niml_sysplan.xml> [-d] [-F]\nOS_install -o define_client -a attr=value -a attr=value... object_name\nOS_install [-K keyfile_path_name] -o define_remote_resource -a attr=value... object_name\nOS_install -o define_resource -a attr=value... object_name\nOS_install -o define_ctrl_host -a attr=value... object_name\nOS_install [-K keyfile_path_name] -o allocate [-F] -a {install_resource=<resource_name> | os_resource=<resource_name> | remote_resource=<resource_name>} <client_name>\nOS_install [-K keyfile_path_name] -o netboot <client_name>\nOS_install [-K keyfile_path_name] -o monitor_installation <client_name>\nOS_install [-K keyfile_path_name] -o deallocate <client_name>\nOS_install [-K keyfile_path_name] -o remove <object_name>\nOS_install -l [-v] [-t object_type | object_name]\nOS_install -S\nOS_install -U\n")) unless ($cli_opts{'i'} || $cli_opts{'o'} || $cli_opts{'l'} ||
							$cli_opts{'U'} || $cli_opts{'S'});

	# -U stops all daemons, no questions asked... if stuff is allocated, it remains allocated
	if ($cli_opts{'U'})
	{
		my $rc= OSInstall::Platform_Tools::stop_daemons(1);

		if ($rc)
		{
			log_entry(__("System services have been stopped. Run OS_install -S before attempting to install with any allocated resources, or use OS_install -o allocate again to reactivate system services.\n"));
		}

		exit $rc;
	}

	# this is a good time to initialize our signal handlers
	$SIG{HUP}= 'IGNORE';
	$SIG{INT}= $SIG{QUIT}= $SIG{TERM}= $SIG{STOP}= $SIG{ABRT}= sub {
		log_entry(__("OS_install has been interrupted by a fatal signal!\n"), 1);
		die "\n";
	};

	# start all the daemons (to do installs after OS_install -U)
	if ($cli_opts{'S'})
	{
		my $rc= OSInstall::Platform_Tools::start_daemons();

		log_entry(__("System services have been started\n")) if ($rc);

		exit $rc;
	}

	# OSInstall operations
	# Check if we have a -K operation
	if($cli_opts{'K'}) {
		$ssh_keyfile = $cli_opts{'K'};
		$ssh_keyfile =~ s/^\s*(\w+)\s*$/$1/;
		# check key file exists, is a plain file and can be read
		unless((-f $ssh_keyfile) && (-r $ssh_keyfile)) {
			print __("Invalid -K keyfile_path_name: $ssh_keyfile\n");
			return E_SYNTAX;
		}
	}
	else {
		# no -K keyfile_path_name option
		# check if HMC default key file /home/hscroot/ssh_keys exists
		# if not, $ssh_keyfile remains null and the standard pathnames
		# of SSH key files, in $HOME/.ssh, must be accessible to the
		# OS_install command process
		if((-f $HMC_ssh_keyfile) && (-r $HMC_ssh_keyfile)) {
			$ssh_keyfile = $HMC_ssh_keyfile;
		}
		elsif((-f $IVM_ssh_keyfile) && (-r $IVM_ssh_keyfile)) {
			$ssh_keyfile = $IVM_ssh_keyfile;
		}
	}

	# OSInstall operations
	# Check if we have a -o operation
	if($cli_opts{'o'}) {
		my $operation = $cli_opts{'o'};
		$operation =~ s/^\s*(\w+)\s*$/$1/;
		unless($operation =~ /^define_client$|^define_remote_resource$|^define_resource$|^define_ctrl_host$|^allocate$|^deallocate$|^netboot$|^monitor_installation$|^remove$/) {
			print __("Invalid -o operation: $operation\nSupported OS_install operations:\ndefine_client define_remote_resource define_resource define_ctrl_host allocate deallocate netboot monitor_installation remove\n");
			return E_SYNTAX;
		}
		$rc = eval("$operation(\$argv_str)");
		if($@) { log_entry(__("Error in $operation: $@\n"), 1); }

		# note that eval returns an undefined value if there is a
		# syntax error or runtime error, or a die statement is executed
		# If there was no error, $@  is guaranteed to be a null string
		if(($rc) && ($rc eq E_SYNTAX)) {
			print (__(usage_msg($operation)));
		}
		return $rc;
	}

	if ( $cli_opts{'i'} ) {
		if ( $cli_opts{'i'} eq "sysplan" ) {
			chomp($argv_str);
			unless($argv_str =~ /-i\s+sysplan\s+-x\s+\S+\s*(-d)?\s*(-F)?\s*(-d)?$/) {
				print __("Usage: OS_install -i sysplan -x <niml_sysplan.xml> [-d] [-F]\n");
				return E_SYNTAX;
			}
	
			$destroy_flag= $1 ? 1 : ($3 ? 1 : 0);
			$force_flag= $2 ? 1 : 0;
 	 
			$rc = install_sysplan($cli_opts{'x'});
			return $rc;
		} elsif ( $cli_opts{'i'} eq "client" ) { install_client(@ARGV); }
		else { }
	}

	if($cli_opts{'l'}) {
		$rc = list_repository($argv_str, $cli_opts{'t'}, $cli_opts{'v'});
		return $rc;
	}

	die(__("Usage:\nOS_install [-K keyfile_path_name] -i sysplan -x <niml_sysplan.xml> [-d] [-F]\nOS_install -o define_client -a attr=value -a attr=value... object_name\nOS_install [-K keyfile_path_name] -o define_remote_resource -a attr=value... object_name\nOS_install -o define_resource -a attr=value... object_name\nOS_install -o define_ctrl_host -a attr=value... object_name\nOS_install [-K keyfile_path_name] -o allocate [-F] -a {install_resource=<resource_name> | os_resource=<resource_name> | remote_resource=<resource_name>} <client_name>\nOS_install [-K keyfile_path_name] -o netboot <client_name>\nOS_install [-K keyfile_path_name] -o monitor_installation <client_name>\nOS_install [-K keyfile_path_name] -o deallocate <client_name>\nOS_install [-K keyfile_path_name] -o remove <object_name>\nOS_install -l [-v] [-t object_type | object_name]\nOS_install -S\nOS_install -U\n"));
}

sub usage_msg
{
	my $usage_key = shift;
	my %usage_msg_hash = (
		define_client => "OS_install -o define_client -a attr=value -a attr=value... object_name\nRequired attributes: ip_addr mac_addr gateway subnet_mask\nOptional attributes: adapter_speed adapter_duplex lpar profile managed_system disk_location ctrl_host\n",
		define_remote_resource => "OS_install [-K keyfile_path_name] -o define_remote_resource -a attr=value attr=value... object_name\nRequired attributes: server type remote_identifier\nOptional attribute: communication_method\n",
		define_resource => "OS_install -o define_resource -a attr=value attr=value... object_name\nRequired attributes: type version\nOptional attributes: source location configfile\n",
		define_ctrl_host => "OS_install -o define_ctrl_host -a attr=value ... object_name\nRequired attributes: communication_method hostname type\n",
		allocate => "OS_install [-K keyfile_path_name] -o allocate [-F] -a attr=<resource_name> attr=value <client_name>\nRequired attribute: install_resource|os_resource|remote_resource\nOptional attribute for OS_Resouce type: config_file\n",
		deallocate => "OS_install [-K keyfile_path_name] -o deallocate <client_name>\n",
		netboot => "OS_install [-K keyfile_path_name] -o netboot <client_name>\n",
		monitor_installation => "OS_install [-K keyfile_path_name] -o monitor_installation <client_name> [-d]\n",
		remove => "OS_install [-K keyfile_path_name] -o remove <object_name>\n",
	);

	return $usage_msg_hash{$usage_key};
}

sub install_sysplan
{
    my $sysplan_niml_doc = shift;
    my %sysplan_allocations;
	my %sysplan_objects;
	my $rc;
	my $client_key;

	# Consume sysplan, return (by reference) sysplan allocations and hash of sysplan objects
	if(XH_consume_sysplan($sysplan_niml_doc, \%sysplan_allocations, \%sysplan_objects)) {
		open(SYSPLAN, "<$sysplan_niml_doc") or die "$sysplan_niml_doc : $!\n";
		log_entry(__("System plan successfully processed:\n" . join('',<SYSPLAN>)));
		close(SYSPLAN);
	}
	else {
		log_entry(__("System plan is invalid\n"), 1);
		return E_SYSPLAN; 
	}
 
	# Make sure sysplan allocations > 0
	unless(keys(%sysplan_allocations)) {
		log_entry(__("System plan contains no allocations. Nothing to install\n"), 1);
		return E_SYSPLAN;
	}

    # Write sysplan objects to repository
	unless(XH_write_objects(\%sysplan_objects)) {
		log_entry(__("Failed to write sysplan objects to repository\n"), 1);
	}

	# make sure that we always delete the system plan and stop the daemons before exiting if the destroy flag is set
	END {
		my $original_rc = $?;
		delete_sysplan($sysplan_niml_doc, \%sysplan_allocations) if ($destroy_flag);
		OSInstall::Platform_Tools::stop_daemons() if ($destroy_flag);
		$? = $original_rc;
	}

	# Attempt to allocate a resource to each client in our allocations hash
	foreach $client_key (keys %sysplan_allocations) {
		log_entry(__("Installing client $client_key with install_resource $sysplan_allocations{$client_key}...\n"));
		# construct command-line-type string to pass to alloc_resource subroutine
		my $alloc_str = "-o allocate -a install_resource=$sysplan_allocations{$client_key} $client_key";
		$alloc_str= "-F ".$alloc_str if ($force_flag);
		# cleanup if allocate fails?
		if($rc = allocate($alloc_str)) {
			return E_ALLOCATE;
		}
    }

	# Now that install_resources have been allocated to all clients, netboot client and 
	# monitor each install
	foreach $client_key (keys %sysplan_allocations) {
		# cleanup if netboot fails?
		if($rc = netboot("-o netboot $client_key")) {
			return E_NETBOOT;
		}

		# Attempt to monitor install of client
		# Instantiate client->allocation and all of it's references
		if($rc = monitor_installation("-o monitor_installation $client_key")) {
			return E_MONITOR;
		}
	}

	return 0;
}

sub install_client
{
    # Construct hash of attributes from argument array
    my %attrs = make_attr_hash(@_);
    #parse command line, if -a client=name 
    unless($attrs{"client"}) { XH_define_client(%attrs); }
    unless($attrs{"os_resource"} ) { create_resource(%attrs); }
# Something missing here ? Where is create_resource() ?

    #alloc_resource
}

# Required attributes: name ip_addr mac_addr gateway subnet_mask
# Optional attributes: adapter_speed adapter_duplex lpar profile managed_system disk_location ctrl_host
sub define_client
{
	my $argv_str = shift;
	chomp($argv_str);
	my $client_ref;
	my %attr_hash; # Hash to copy attributes from command line

	# Check that our command line syntax is valid
	unless($argv_str =~ /-o\s+define_client\s+(-a\s+\w+=\S+\s+)+(\w+)$/) {
		log_entry(__("define_client command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	# Regex stores last argument (object name) in $2
	$attr_hash{name} = $2;

	# Take command line argument string and convert any attributes to attribute->value hash
	make_attr_hash($argv_str, \%attr_hash);

	# Return a syntax error if attributes are not valid for a client
	foreach my $attr_key (keys %attr_hash) {
		unless($attr_key =~ /^name$|^ip_addr$|^mac_addr$|^gateway$|^subnet_mask$|^adapter_speed$|^adapter_duplex$|^lpar$|^profile$|^managed_system$|^disk_location$|^ctrl_host$/) {
			return E_SYNTAX;
		}
	}

	#we need to check that all required attributes are defined
	unless(defined($attr_hash{name}) && defined($attr_hash{ip_addr}) 
&& defined($attr_hash{mac_addr}) && defined($attr_hash{gateway})  
&& defined($attr_hash{subnet_mask})) {
			log_entry(__("The define_client operation is missing required attributes\n"), 1);
			return E_SYNTAX;
	}

	# Instantiate object from attr/value hash
	$client_ref = OSInstall::Client->new($attr_hash{name}, $attr_hash{ip_addr}, $attr_hash{gateway}, $attr_hash{mac_addr}, $attr_hash{adapter_speed}, $attr_hash{adapter_duplex}, $attr_hash{subnet_mask}, $attr_hash{disk_location}, $attr_hash{ctrl_host}, $attr_hash{lpar}, $attr_hash{profile}, $attr_hash{managed_system});
	
	unless($client_ref) { log_entry(__("Unable to instantiate Client object\n"), 1);
	}


	# Make sure XML_Handler returned a Client ref
	unless(ref($client_ref) =~ /^(?:OSInstall::)?Client$/) {
		return E_XMLHANDLER;
	}

	XH_write_object($client_ref);

	return 0;
}

sub define_remote_resource
{
	my $argv_str = shift;
	chomp($argv_str);
	my $resource_ref;
	my $ready_flag = 0; # Set ready field to 0 when we initially instantiate
	my %attr_hash; # Hash to copy attributes from command line

	# Check that our command line syntax is valid
	unless($argv_str =~ /-o\s+define_remote_resource\s+(-a\s+\w+=(.*?)\s+)+(\w+)$/) {
		log_entry(__("Command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	# Regex stores last argument (object name) in $2
	$attr_hash{name} = $3;

	# Take command line argument string and convert any attributes to attribute->value hash
	make_attr_hash($argv_str, \%attr_hash);

	# Return a syntax error if attributes are not valid for remote resource
	foreach my $attr_key (keys %attr_hash) {
		unless($attr_key =~ /^name$|^server$|^type$|^remote_identifier$|^communication_method$/) {
			log_entry(__("$attr_key is an invalid attribute for the define_remote_resource operation\nValid attributes: server type remote_identifier communication_method\n"), 1);
			return E_INVALID_ATTR;
		}
	}

	#we need to check that all required attributes are defined
	unless(defined($attr_hash{name}) && defined($attr_hash{server}) && defined($attr_hash{type}) && defined($attr_hash{remote_identifier})) {
		log_entry(__("The define_remote_resource operation is missing a required attribute\n"), 1);
		return E_SYNTAX;
	}

	unless($attr_hash{type} =~ /^AIX$|^Linux$/) {
		log_entry(__("%s is not a valid type attribute\n", $attr_hash{type}), 1);
		return E_INVALID_VALUE;
	}

	# Currently, only secure shell is supported
	unless(defined($attr_hash{communication_method})) {
		$attr_hash{communication_method} = "ssh";
	}

	# If a Linux remote resource, return error until implemented
	if($attr_hash{type} =~ /^Linux$/) {
		log_entry(__("Linux remote resources are not yet implemented\n"), 1);
		return E_INVALID_VALUE;
	}

	# Instantiate object from attr/value hash
	my $constructor_str = "OSInstall::$attr_hash{type}_Remote->define(\$attr_hash{name}, \$attr_hash{server}, \$attr_hash{type}, \$attr_hash{remote_identifier}, \$attr_hash{communication_method}, \$ready_flag)";
	$resource_ref = eval $constructor_str;
	
	unless($resource_ref) {
		log_entry(__("Unable to create Remote_Resource object\n"), 1);
	}

	# Make sure XML_Handler returned a Remote_Resource ref
	unless(ref($resource_ref) =~ /^(?:OSInstall::)?\w+_Remote$/) {
		return E_XMLHANDLER;
	}

	XH_write_object($resource_ref);

	return 0;

}

sub define_resource
{
	my $argv_str = shift;
	chomp($argv_str);
	my $resource_ref;
	my $distro_str;
	my $ready_flag = 0; # Set ready field to 0 when we initially instantiate
	my %attr_hash; # Hash to copy attributes from command line

	# Check that our command line syntax is valid
	unless($argv_str =~ /-o\s+define_resource\s+(-a\s+\w+=(.*?)\s+)+(\w+)$/) {
		log_entry(__("Command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	# Regex stores last argument (object name) in $2
	$attr_hash{name} = $3;

	# Take command line argument string and convert any attributes to attribute->value hash
	make_attr_hash($argv_str, \%attr_hash);

	# Return a syntax error if attributes are not valid for an os_resource
	foreach my $attr_key (keys %attr_hash) {
		unless($attr_key =~ /^name$|^type$|^version$|^source$|^location$|^configfile$|^resource_server$/) {
			log_entry(__("$attr_key is an invalid attribute for the define_resource operation\nValid attributes: type version source location configfile\n"), 1);
			return E_INVALID_ATTR;
		}
	}

	#we need to check that all required attributes are defined
	unless(defined($attr_hash{name}) && defined($attr_hash{type}) && defined($attr_hash{version})) {
		log_entry(__("The define_resource operation is missing a required attribute\n"), 1);
		return E_SYNTAX;
	}

	unless($attr_hash{type} =~ /^AIX$|^Linux$|^VIOS$/) {
		log_entry(__("%s is not a valid type attribute\n", $attr_hash{type}), 1);
		return E_INVALID_VALUE;
	}

	# If a Linux resource, extract the distribution from the version string
	if($attr_hash{type} =~ /^Linux$/) {
		if($attr_hash{version} =~ s/^(\S+)\s+(\S+\s+\S+)\s*$/$2/) {
			$distro_str = $1;
		}
		else {
			log_entry(__("%s is not a valid version attribute for Linux resources\n", $attr_hash{version}), 1);
			return E_INVALID_VALUE;
		}
	}

	# Determine type of OS_Resource to instantiate
	my $inst_type; #prefix to '_Resource' for class name
	if ( $attr_hash{type} eq 'VIOS' ) {
		$inst_type = 'AIX';
	}
	else {
		$inst_type = $attr_hash{type};
	}

	# Instantiate object from attr/value hash
	my $constructor_str = "OSInstall::${inst_type}_Resource->new(\$attr_hash{name}, \$attr_hash{source}, \$attr_hash{type}, \$attr_hash{version}, \$attr_hash{location}, \$attr_hash{configfile}, \$attr_hash{resource_server}, \$distro_str, \$ready_flag)";
	$resource_ref = eval $constructor_str;
	
	unless($resource_ref) {
		log_entry(__("Unable to create OS_Resource object\n"), 1);
	}

	# Make sure XML_Handler returned a OS_Resource ref
	unless(ref($resource_ref) =~ /^(?:OSInstall::)?\w+_Resource$/) {
		return E_XMLHANDLER;
	}

	XH_write_object($resource_ref);

	return 0;

}

sub define_ctrl_host
{
	my $argv_str = shift;
	chomp($argv_str);
	my $ctrl_host_ref;
	my %attr_hash; # Hash to copy attributes from command line

	# Check that our command line syntax is valid
	unless($argv_str =~ /-o\s+define_ctrl_host\s+(-a\s+\w+=\S+\s+)+(\w+)$/) {
		log_entry(__("define_ctrl_host command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	# Regex stores last argument (object name) in $2
	$attr_hash{name} = $2;

	# Take command line argument string and convert any attributes to attribute->value hash
	make_attr_hash($argv_str, \%attr_hash);

	# Return a syntax error if attributes are not valid for a ctrl_host
	foreach my $attr_key (keys %attr_hash) {
		unless($attr_key =~ /^name$|^communication_method$|^hostname$|^type$/) {
			return E_SYNTAX;
		}
	}

	#we need to check that all required attributes are defined
	unless(defined($attr_hash{name}) && defined($attr_hash{communication_method}) 
&& defined($attr_hash{hostname}) && defined($attr_hash{type})) {
			log_entry(__("The define_ctrl_host operation is missing required attributes\n"), 1);
			return E_SYNTAX;
	}

	# Instantiate object from attr/value hash
	$ctrl_host_ref = OSInstall::Control_Host->new($attr_hash{name}, $attr_hash{hostname}, $attr_hash{type}, $attr_hash{communication_method});
	
	unless($ctrl_host_ref) { log_entry(__("Unable to instantiate Control_Host object\n"), 1);
	}


	# Make sure XML_Handler returned a Control_Host ref
	unless(ref($ctrl_host_ref) =~ /^(?:OSInstall::)?Control_Host$/) {
		return E_XMLHANDLER;
	}

	XH_write_object($ctrl_host_ref);

	return 0;

}

sub allocate
{
	my $argv_str = shift;
	chomp($argv_str);

	my $client_str;
	my $resource_str;
	my %attr_hash;

	# Check that our command line syntax is valid
	unless($argv_str =~ /(-F)?\s*-o\s+allocate\s+(-F)?\s*(-a\s+\w+=\S+\s+)+(\w+)$/) {
		log_entry(__("allocate command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	# Regex stores last argument (object name) in $4
	$attr_hash{name} = $4;

	# initialise force flag
	$force_flag= $1 ? 1 : ($2 ? 1 : 0);

	# Take command line argument string and convert any attributes to attribute->value hash
	make_attr_hash($argv_str, \%attr_hash);

	# Return a syntax error if attributes are not valid for a client
	my $resource_count = 0;
	foreach my $attr_key (keys %attr_hash) {
		unless($attr_key =~ /^name$|^os_resource$|remote_resource|install_resource|config_file^$/) {
			return E_SYNTAX;
		}
		unless($attr_key =~ /^name$|config_file^$/) {
			$resource_count++;
		}
	}

	# Check that only one resource attribute is defined
	if ($resource_count != 1) {
			log_entry(__("The allocate operation requires only one resource attribute\n"), 1);
			return E_SYNTAX;
	}

	# Check that all required attributes are defined
	unless(defined($attr_hash{name}) && (defined($attr_hash{os_resource}) || defined($attr_hash{remote_resource}) || defined($attr_hash{install_resource}))) {
			log_entry(__("The allocate operation is missing required attributes\n"), 1);
			return E_SYNTAX;
	}

	if (defined($attr_hash{os_resource})) {
		$resource_str = $attr_hash{os_resource};
	}
	if (defined($attr_hash{remote_resource})) {
		$resource_str = $attr_hash{remote_resource};
	}
	if (defined($attr_hash{install_resource})) {
		$resource_str = $attr_hash{install_resource};
	}
	$client_str = $attr_hash{name};

	my $client_ref = XH_read_object($client_str);
	my $resource_ref = XH_read_object($resource_str);

	unless($client_ref) {
		log_entry(__("Unable to read Client object $client_str from repository\n"), 1);
		return E_INVALID_OBJ_NAME; 
	}

	unless($resource_ref) {
		log_entry(__("Unable to read OS_Resource or Remote_Resource object $resource_str from repository\n"), 1);
		return E_INVALID_OBJ_NAME;
	}

	# Check that the client does not have anything allocated to it
	if($client_ref->allocation) {
		log_entry(__("The OS_Resource or Remote_Resource %s is already allocated to Client %s\n", $client_ref->allocation, $client_ref->name), 1);
		return E_INVALID_ALLOCATION;
	}

	# Check that no config_file is defined with a remote resource
	if (ref($resource_ref) =~ /^(?:OSInstall::)?\w+_Remote$/) {
		if (defined($attr_hash{config_file})) {
			log_entry(__("The remote resource allocate operation does not require a config_file attribute\n"), 1);
			return E_SYNTAX;
		}
	}

	# Instantiate existing client references that this resource is allocated to
	foreach my $client_name (keys %{($resource_ref->clients_allocated)}) {
		($resource_ref->clients_allocated)->{$client_name} = XH_read_object($client_name);
	}

	if($resource_ref->allocate($client_ref, $attr_hash{config_file})) {
		unless(XH_alloc_resource($resource_ref->name, $client_ref->name, $client_ref->boot_params)) {
			return E_XMLHANDLER;
		}
	}
	else {
		log_entry(__("An error occurred attempting to allocate %s to %s\n", $resource_ref->name, $client_ref->name), 1);
		return E_INVALID_ALLOCATION;
	}

	return 0;
}

sub deallocate
{
	my $argv_str = shift;
	chomp($argv_str);

	my $client_str;

	unless($argv_str =~ /-o\s+deallocate\s+(\w+)\s*$/) {
		log_entry(__("deallocate command line syntax is invalid\n"), 1);
		return E_SYNTAX;
	}

	$client_str = $1;

	my $client_ref = XH_read_object($client_str);

	unless($client_ref) {
		log_entry(__("Unable to read Client object $client_str from repository\n"), 1);
		return E_INVALID_OBJ_NAME; 
	}

	unless($client_ref->allocation) {
		log_entry(__("Client object $client_str has no allocation\n"), 1);
		return E_INVALID_ALLOCATION; 
	}

	# instantiate client->allocation
	$client_ref->allocation(XH_read_object($client_ref->allocation));

	unless (ref($client_ref->allocation) =~ /^(?:OSInstall::)?\w+_Resource$/) {
	    unless (ref($client_ref->allocation) =~ /^(?:OSInstall::)?\w+_Remote$/) {
		log_entry(__("Unable to instantiate allocation for %s\n", $client_ref->name), 1);
		return E_INVALID_ALLOCATION; 
	    }
	}

	# instantiate client->allocation->client_allocated{}
	my %other_clients_alloc = %{$client_ref->allocation->clients_allocated};
	foreach my $oca (keys %other_clients_alloc) {
		$other_clients_alloc{$oca} = XH_read_object($oca);
	}
	# set the clients_allocated hash for the os_resource or remote_resource
	# to be deallocated
	$client_ref->allocation->clients_allocated(\%other_clients_alloc);

	# call deallocate for the client's resource allocation, passing the client object
	$client_ref->allocation->deallocate($client_ref);
	# remove allocation reference from repository
	unless(XH_deallocate($client_ref->name, $client_ref->allocation->name)) {
		log_entry(__("Error removing allocation reference in repository for client %s\n", $client_ref->allocation->name), 1);
		return E_XMLHANDLER;
	}


	return 0;
} 

sub netboot
{
	my $argv_str = shift;
	chomp($argv_str);
	my $client_name;
	my $client_ref;
	
	unless($argv_str =~ /-o\s+netboot\s+(\w+)\s*$/) {
		log_entry(__("netboot command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	$client_name = $1;
	log_entry(__("Netbooting client $client_name\n"));

	unless($client_ref = XH_read_object($client_name)) {
		log_entry(__("Unable to read Client object $client_name from repository\n"), 1);
		return E_INVALID_OBJ_NAME;
	}
	
	# Determine if control_host is a reference to a ctrl_host object
	# If not, read contrl_host object defined by client and set field for client
	unless(ref($client_ref->control_host)) {
		# Make sure control host field is defined
		if($client_ref->control_host) {
			# Read ctrl_host ref from repository
			my $ctrl_host_ref = XH_read_object($client_ref->control_host);
			# Set client's control_host field to our ref
			if($ctrl_host_ref) {
				$client_ref->control_host($ctrl_host_ref);
			}
			else {
				log_entry(__("The control_host object ".$client_ref->control_host." associated with $client_name does not exist\n"));
				return E_CTRL_HOST;
			}
		}
		else {
			log_entry(__("Client object $client_name has no control host defined\n"));
		}
	}

	# Attempt to netboot client
	unless($client_ref->netboot()) {
		log_entry(__("netboot failed for client $client_name\n"), 1);
		return E_NETBOOT;
	}

	return 0;
}

sub monitor_installation
{
	my $argv_str = shift;
	chomp($argv_str);
	my $client_name;
	my $client_ref;
	
	unless($argv_str =~ /-o\s+monitor_installation\s+(\w+)\s*$/) {
		log_entry(__("monitor_installation command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	$client_name = $1;
	log_entry(__("Monitoring $client_name install\n"));

	unless($client_ref = XH_read_object($client_name)) {
		log_entry(__("Unable to read Client object $client_name from repository\n"), 1);
		return E_INVALID_OBJ_NAME;
	}

	# Attempt to monitor install of client
	$client_ref->allocation(XH_read_object($client_ref->allocation));

	unless($client_ref->monitor_installation()) {
		log_entry(__("Installation of client %s has failed, opening a virtual terminal will provide more details\n", $client_name), 1);
		return E_MONITOR;
	}

	return 0;
}

sub remove
{
	my $argv_str = shift;
	chomp($argv_str);
	my $object_name;
	my $object_ref;
	
	unless($argv_str =~ /-o\s+remove\s+(\w+)\s*$/) {
		log_entry(__("remove command line syntax is invalid\n"), 1);
		return E_SYNTAX; 
	}

	$object_name = $1;
	log_entry(__("Removing object $object_name\n"));

	# instantiate object in case we need to call any remove methods before
	# we remove repository entry
	$object_ref = XH_read_object($object_name);
	if(ref($object_ref) =~ /^(?:OSInstall::)?\w+_Resource$/) {
		# if this is an OS_Resource, call rm_images and rm_bootfiles
		log_entry(__("Removing resource images for $object_name\n"));
		$object_ref->rm_images();
	}

	if(ref($object_ref) =~ /^(?:OSInstall::)?\w+_Remote$/) {
		# if this is a Remote_Resource, call destroy (initially a no-op)
		log_entry(__("Removing remote resource for $object_name\n"));
		$object_ref->destroy();
	}

	# Remove object from repository
	unless(XH_remove_object($object_name)) {
		log_entry(__("Error removing object $object_name from repository\n"), 1);
		return E_XMLHANDLER;
	}

	return 0;
}

sub delete_sysplan
{
	my $sysplan_doc = shift;
	my $sysplan_allocations= shift;
	
	# deallocate any current allocations in sysplan
	foreach my $each_client (keys %$sysplan_allocations)
	{
		deallocate("-o deallocate $each_client");
	}

	# remove all osinstall objects in sysplan
	XH_delete_sysplan($sysplan_doc);
}

sub list_repository
{
	my $argv_str = shift;
	my $obj_type = shift;
	my $verbose = shift;
	my $obj_name;

	my %arg_hash;
	chomp($argv_str);

	# Check that our command line syntax is valid
#	unless($argv_str =~ /-lv?(-t\s+\w+)|(\w+)$/) {
	if(!$obj_type && $argv_str =~ /\s+(\w+)\s*$/) {
		$obj_name = $1;
	}

	unless(XH_list_objects($obj_type, $obj_name, $verbose)) {
		return 1;		
	}

	return 0;
}

# Takes a command line array and constructs a key->value hash for each -a attribute
sub make_attr_hash
{
    my $arg_str = shift;
	my $attr_hash_ref = shift;

#   Arg string looks like this:
#   -o some_operation [-F] -a attr1=foo -a attr2=bar -a someattr=somevalue object_name
#   Save key/value pairs for each -a flag into our hash to return by reference
    while($arg_str =~ s/-a\s*(\w+)=(.*?)(-a|\w+$)/$3/)
    {
		my $attr_name = $1;
		my $attr_value = $2;
		$attr_value =~ s/\s+$//;
        ${$attr_hash_ref}{$attr_name}= $attr_value;
    }
}

1;

__END__
