package OSInstall::AIX_Resource;

use strict;
use File::Copy;
use File::Path;
use File::Find;
use Cwd 'abs_path';
use IO::Socket;
use Sys::Hostname;
use OSInstall::OS_Resource;
use OSInstall::Common;
use OSInstall::Platform_Tools;
use Carp;
use Cwd;

use constant RPM_LOC => "RPMS/linux/aix-res-*.rpm";
use constant AIX_RES_LOCATION => "/opt/aix-res";
use constant TFTPBOOT => "/tftpboot";

#our @ISA = qw(OS_Resource);
use base ("OSInstall::OS_Resource");

our @SIMAGES= qw(
./installp/ppc/bos*
./installp/ppc/csm.client*
./installp/ppc/csm.core*
./installp/ppc/csm.deploy*
./installp/ppc/csm.diagnostics*
./installp/ppc/csm.dsh*
./installp/ppc/csm.gui.dcem*
./installp/ppc/csm.msg.en_US*
./installp/ppc/csm.msg.EN_US*
./installp/ppc/devices*
./installp/ppc/ICU4C.rte*
./installp/ppc/ifor_ls.base*
./installp/ppc/ifor_ls.html.en_US*
./installp/ppc/infocenter*
./installp/ppc/invscout.com*
./installp/ppc/invscout.ldb*
./installp/ppc/invscout.rte*
./installp/ppc/Java14.sdk*
./installp/ppc/Java5_64.sdk*
./installp/ppc/Java5.sdk*
./installp/ppc/lum.base*
./installp/ppc/lum.msg.en_US*
./installp/ppc/lwi*
./installp/ppc/perfagent.tools*
./installp/ppc/perl.libext*
./installp/ppc/perl.rte*
./installp/ppc/printers.rte*
./installp/ppc/rpm.rte*
./installp/ppc/rsct.core*
./installp/ppc/rsct.msg.en_US*
./installp/ppc/rsct.msg.EN_US*
./installp/ppc/sysmgt.help.en_US*
./installp/ppc/sysmgt.help.EN_US*
./installp/ppc/sysmgt.help.msg.en_US*
./installp/ppc/sysmgt.help.msg.EN_US*
./installp/ppc/sysmgtlib.framework*
./installp/ppc/sysmgtlib.libraries*
./installp/ppc/sysmgt.pconsole*
./installp/ppc/sysmgt.sguide*
./installp/ppc/sysmgt.websm*
./installp/ppc/Tivoli_Management_Agent.client*
./installp/ppc/X11*
./installp/ppc/xlC.aix*
./installp/ppc/xlC.cpp*
./installp/ppc/xlC.msg.en_US.cpp*
./installp/ppc/xlC.msg.EN_US.cpp*
./installp/ppc/xlC.rte*
./installp/ppc/xlC.sup*
./RPMS/ppc/cdrecord*.ppc.rpm
./RPMS/ppc/mkisofs*.ppc.rpm
);

my %CONFIG_FILES= (
	bosinst_data => 'bosinst.data',
	image_data => 'image.data',
	cust_script => 'scripts/cust.sh',
	secondary_adapters => 'scripts/nimadapters',
	resolv_conf => 'resolv.conf',
);

BEGIN
{
	unless (-d $OSInstall::Common::MOUNT_POINT)
	{
		mkdir($OSInstall::Common::MOUNT_POINT, 0755) or die $OSInstall::Common::MOUNT_POINT." : $!\n";
	}
}

sub new
{
	my $class= shift;
	my $this= $class->SUPER::new(@_);

	# add the SIMAGES list to the data of the class
	$this->{SIMAGES}= \@SIMAGES;
	$this->{CONFIG}= \%CONFIG_FILES;

	# now that we have an OS_Resource object, bless the reference into an AIX_Resource
	# object so that we can perform the steps to copy the install images using the overridden
	# methods
	bless $this, $class;

	unless ($this->verify_resource())
	{
		unless($this->image_source)
		{
			log_entry(__("The 'source' attribute is undefined; unable to copy operating system images for this resource\n"), 1);
			return 0;
		}

		# create an AIX lpp_source
		unless ($this->copy_images())
		{
			log_entry(__("Failed to copy operating system images from %s to %s\n", $this->image_source, $this->image_location), 1);
			return 0;
		}

		# copy the AIX configuration files into the resource location
		unless ($this->gen_config_file())
		{
			log_entry(__("Failed to set up configuration files for resource\n"), 1);
			return 0;
		}
	}
	else
	{
		# If our resource already exists, copy the boot image to /tftpboot
		unless ((-d TFTPBOOT || mkdir(TFTPBOOT, 0755)) && copy($this->image_location.'/booti.chrp.mp.ent', TFTPBOOT.'/'.$this->name.'.chrp.mp.ent'))
		{
			log_entry(__("Failed to copy files from %s to %s\n", $this->image_location.'/booti.chrp.mp.ent', TFTPBOOT.'/'.$this->name.'.chrp.mp.ent'), 1);
			return 0;
		}
	}

	confess __("Failed to validate OS_Resource %s because it is missing something which is required\n") unless ($this->verify_resource(1));

	$this->{ready} = 1;
	
	$this;
}

sub gen_config_file
{
	# shift args
	my $this= shift;

	# local data
	my $rc= 1;
	my @cfg_file_contents;
	my $cfg_file= $this->config_file;
	my $each_cfg_line;
	my $aix_std_cfg= \%CONFIG_FILES;
	my $each_cfg_type;
	my $each_cfg_file;
	
	# commands
	my $CAT= find_command('cat');

	# if they didn't specify a config file, then just return- they'll use the defaults
	return 1 unless ($cfg_file);

	@cfg_file_contents= `$CAT $cfg_file 2>/dev/null`;

	foreach $each_cfg_line (@cfg_file_contents)
	{
		chomp $each_cfg_line;

		($each_cfg_type, $each_cfg_file)= split /:/, $each_cfg_line;
		next unless ($aix_std_cfg->{$each_cfg_type}); # skip unknown configuration files

		# copy the config file into the location of the resource
		$rc= copy($each_cfg_file, $this->image_location.'/'.$aix_std_cfg->{$each_cfg_type});
		last unless ($rc);
	}

	$rc;
}

sub copy_images
{
	# shift args
	my $this= shift;

	# local data
	my $rc= 1;
	my $image_dir= $this->image_source;
	my $mounted= 0;
	my $from_cd= 0;
	my $from_tar= 0;
	my $from_vios= 0;
	my $new_aix_rpm;
	my $aix_release;
	my $simages_list= join " ", @SIMAGES;
	my $mount_flags= "";
	my $image_copy_cmd;
	my $lock_fh;
	my $bos_usr_location;
	my $mksysb_loc;
	my $mksysb1;
	my $mksysb2;

	# commands
	my $TAR= find_command('tar');
	my $MOUNT= find_command('mount');
	my $UMOUNT= find_command('umount');
	my $RPM= find_command('rpm');
	my $GREP= find_command('grep');
	my $GZIP= find_command('gzip');
	my $GENCOPY= find_command('gencopy');
	my $CP= find_command('cp');
	my $CAT= find_command('cat');

	# create the directory that will contain the resource
	unless (-d $this->image_location)
	{
		unless (mkpath $this->image_location, 0755)
		{
			log_entry(__("Failed to create the directory %s\n", $this->image_location), 1);
			return 0;
		}
	}

	# determine what the image source really is

	# first, mount this if it's an NFS-exported directory or a CD
	# save the mount point as a temporary image_source
	if ($this->image_source =~ m|^/+dev/+.*| || $this->image_source =~ m|^[\w\.]+:(?:/+\w*)+|)
	{
		$from_cd= $this->image_source =~ m|^/+dev/+.*|; 
		$from_tar= $this->image_source =~ /.*\.tar$/;
	
		$mount_flags= "-oro -vcdrfs" if (OSInstall::Platform_Tools::get_system_os() eq 'AIX' && $from_cd);
		
		$rc= log_cmd("$MOUNT $mount_flags $image_dir ".$OSInstall::Common::MOUNT_POINT);
		
		if ($rc)
		{
			$image_dir= $OSInstall::Common::MOUNT_POINT;
			$mounted= 1;
		}
		else
		{
			log_entry(__("Failed to mount %s over %s\n", $this->image_source, $OSInstall::Common::MOUNT_POINT), 1);
			return 0;
		}

		if (-d $image_dir."/nimol/ioserver_res")
		{
			$from_vios= 1;
			$image_dir.= "/nimol/ioserver_res";
		}
	}

	# treat this as a CD
	if ($from_cd && ! $from_vios)
	{
		# locate and install the aix-res RPM (override vs. media)
		$new_aix_rpm = get_rpm_path($image_dir);

		unless (-f $new_aix_rpm)
		{
			log_entry(__("The file %s cannot be found\n", $image_dir.'/'.RPM_LOC), 1);
			log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);
			return 0;
		}

		if ($lock_fh= lock_file('aix_res_rpm', 'exclusive'))
		{
			# perform the actual installation of the RPM (does nothing if it's already installed)
			log_cmd("$RPM -Uvh --force --nodeps --ignoreos $new_aix_rpm", 1);
			close $lock_fh;
		}
		# ELSE error out
		
		# extract the release name from the rpm name (for example: in aix-res-5300-03-1.0-1.rpm the release name is 5300-03)
		$new_aix_rpm =~ /.*\/aix-res-(\d+-\d+)-.*/;
		$aix_release= $1;

		# copy in the resources from the AIX_RES_LOCATION
		log_entry(__("Copying files from %s to %s ...\n", AIX_RES_LOCATION."/$aix_release", $this->image_location));
		if ($lock_fh= lock_file('aix_res_rpm', 'shared'))
		{
			unless (log_cmd("(cd ".AIX_RES_LOCATION."/$aix_release && $TAR -cf - ./) | (cd ".$this->image_location." && $TAR -xf -)"))
			{
				log_entry(__("Failed to copy files from %s to %s\n", AIX_RES_LOCATION."/$aix_release", $this->image_location), 1);
				log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);
				return 0;
			}
			close $lock_fh;
		}

		# unzip the SPOT and boot image
		unless (log_cmd("$GZIP -d ".$this->image_location."/*.Z") &&
			log_cmd("(cd ".$this->image_location." && tar -xf *.tar)") &&
			unlink glob($this->image_location."/*.tar"))
		{
			log_entry(__("Failed to unpack resources in %s from %s\n", $this->image_location, AIX_RES_LOCATION), 1);
			log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);
			return 0;
		}

		# copy installp/ppc contents to the lpp_source directory
		unless (-d $this->image_location."/lpp_source" || mkdir($this->image_location."/lpp_source", 0755))
		{
			log_entry(__("Failed to create the directory %s : %s\n", $this->image_location."/lpp_source", $!), 1);
			log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);
			return 0;
		}
		
		# we won't fail altogether if the copy command gives a bad return code... some commands (gencopy)
		# give a bad return code if just one of the packages is missing- we don't care if this is the case
		log_entry(__("Copying files from %s to %s ...\n", $image_dir, $this->image_location.'/lpp_source'));

		# if we're running on an AIX system, make use of the gencopy command, since it's cleaner than the tar stuff
#		if ($GENCOPY)
#		{
#			$ENV{NIM_METHODS}= '/usr/lpp/bos.sysmgt/nim/methods';
#			$image_copy_cmd= ". \$NIM_METHODS/c_sh_lib && $GENCOPY -X -b \"qv\" -d $image_dir -t ".$this->image_location."/lpp_source \$SIMAGES_OPTIONS";
#		}
#		else
#		{
#		$image_copy_cmd = "(cd $image_dir && $TAR -cf - $simages_list) | (cd ".$this->image_location."/lpp_source && $TAR -xf -)";
#		}

		# Use our own copy subroutine instead of tar 
		&copy_dir($image_dir, $this->image_location."/lpp_source", \@SIMAGES);
				
#		log_cmd($image_copy_cmd, 1);

		# so we've got all the images in place- generate index file
		open(INDEX, ">".$this->image_location."/index") or confess $this->image_location."/index"." : $!\n";
		print INDEX "lpp_source:lpp_source\n";
		print INDEX "spot:SPOT\n";
		print INDEX "bosinst_data:bosinst.data\n";
		print INDEX "ent.bi:booti.chrp.mp.ent\n";
		close INDEX;
	}
	# a system backup generated by bootpkg was supplied
	elsif ($from_tar)
	{
		# extract the contents of the tar file into the resource location
		unless (log_cmd("cd ".$this->image_location." && $TAR -xf $image_dir"))
		{
			log_entry(__("Failed to unpack resources in %s from %s\n", $this->image_location, $image_dir), 1);
			return 0;
		}
	}
	# a plain directory was supplied
	else
	{
		# copy files into the resource location using tar
		unless (log_cmd("(cd $image_dir && $TAR -cf - ./) | (cd ".$this->image_location." && $TAR -xf -)"))
		{
			log_entry(__("Failed to copy files from %s to %s\n", $image_dir, $this->image_location), 1);
			return 0;
		}

		# if this is from the VIOS media, do an explicit copy of the mksysb since it is a symlink and tar doesn't
		# follow symlinks the way it's being called above
		if ($from_vios)
		{
			unlink $this->image_location."/mksysb";

			# defect 641979 split up the mksysb image on the VIOS 1.5.2 media
			# into two files in <media_mount>/usr/sys/inst.images and a
			# corresponding symbolic link in <media_mount>/nimol/ioserver_res
			# was not created for the second image. defect 653117 should create
			# this symbolic link but it might not get included in the build
			# therefore we must use the direct mksysb_image paths.

			$mksysb1 = $image_dir . "/../../usr/sys/inst.images/mksysb_image";
			$mksysb2 = $image_dir . "/../../usr/sys/inst.images/mksysb_image2";
			if ( (-f $mksysb1) && (-f $mksysb2) )
			{
				unless (log_cmd("$CAT $mksysb1 $mksysb2 > ".$this->image_location."/mksysb"))
				{
					log_entry(__("Failed to copy files from %s to %s\n", $image_dir, $this->image_location), 1);
					return 0;
				}
			}
			else
			{
				unless (log_cmd("$CP $image_dir/mksysb ".$this->image_location."/mksysb"))
				{
					log_entry(__("Failed to copy files from %s to %s\n", $image_dir, $this->image_location), 1);
					return 0;
				}
			}
		}

		# untar and unzip any compressed files that may be present
		if (glob($this->image_location."/*.Z"))
		{
			unless (log_cmd("$GZIP -d ".$this->image_location."/*.Z"))
			{
				log_entry(__("Failed to unpack resources in %s from %s\n", $this->image_location, $image_dir), 1);
				return 0;
			}
		}

		if (glob($this->image_location."/*.gz"))
		{
			unless (log_cmd("$GZIP -d ".$this->image_location."/*.gz"))
			{
				log_entry(__("Failed to unpack resources in %s from %s\n", $this->image_location, $image_dir), 1);
				return 0;
			}
		}

		if (glob($this->image_location."/*.tar"))
		{
			unless (log_cmd("(cd ".$this->image_location." && tar -xf *.tar)"))
			{
				log_entry(__("Failed to unpack resources in %s from %s\n", $this->image_location, $image_dir), 1);
				return 0;
			}

			unlink glob($this->image_location."/*.tar");
		}

		# so we've got all the images in place- generate index file
		open(INDEX, ">".$this->image_location."/index") or confess $this->image_location."/index"." : $!\n";
		print INDEX "lpp_source:lpp_source\n" if (-d $this->image_location."/lpp_source");
		print INDEX "spot:SPOT\n";
		print INDEX "bosinst_data:bosinst.data\n";
		print INDEX "ent.bi:booti.chrp.mp.ent\n";

		$mksysb_loc= glob($this->image_location."/*mksysb*");
		$mksysb_loc =~ s/.*\///; # get only the relative filename
		print INDEX "mksysb:$mksysb_loc" if ($mksysb_loc);
		
		close INDEX;
	}

	# rename bos.usr.* to just bos in the lpp_source
	$bos_usr_location= glob($this->image_location."/lpp_source/installp/ppc/bos.usr*");
	if (-e $bos_usr_location)
	{
		unless (rename $bos_usr_location, $this->image_location."/lpp_source/installp/ppc/bos")
		{
			log_entry(__("Failed to rename file %s to %s\n", $bos_usr_location, $this->image_location."/lpp_source/installp/ppc/bos"), 1);
			log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);
			return 0;
		}
	}

	# now that everything's been copied, put the boot image in /tftpboot
	unless ((-d TFTPBOOT || mkdir(TFTPBOOT, 0755)) && copy($this->image_location.'/booti.chrp.mp.ent', TFTPBOOT.'/'.$this->name.'.chrp.mp.ent'))
	{
		log_entry(__("Failed to copy files from %s to %s\n", $this->image_location.'/booti.chrp.mp.ent', TFTPBOOT.'/'.$this->name.'.chrp.mp.ent'), 1);
		log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);
		return 0;
	}

	# create the scripts directory
	unless (mkdir($this->image_location.'/scripts', 0755))
	{
		log_entry(__("Failed to create the directory %s : %s\n", $this->image_location."/lpp_source", $!), 1);
		log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);
		return 0;
	}
	
	# chmod the entire directory
	find(sub {if (-l $File::Find::directory) { $File::Find::prune= 1 } else { chmod 0755, $File::Find::name unless (-l $File::Find::name) }}, $this->image_location);

	log_cmd("$UMOUNT ".$OSInstall::Common::MOUNT_POINT) if ($mounted);

	$rc;
}

sub rm_images
{
	# shift args
	my $this= shift;

	# just blow out the whole resources directory
	rmtree $this->image_location;
	# remove the boot image from /tftpboot
	unlink TFTPBOOT.'/'.$this->name.'.chrp.mp.ent';
}

sub copy_bootfiles
{
	# shift args
	my $this= shift;
	my $client_ref= shift; # ref to client allocating the resource

	# local data
	my $rc= 1;
	my $bi_target= $this->name.'.chrp.mp.ent';
	my $boot_image= TFTPBOOT.'/'.$bi_target;
	my $bi_link= TFTPBOOT.'/'.$client_ref->hostname;
	my $niminfo_file= TFTPBOOT.'/'.$client_ref->hostname.'.info';
	my $server_hostname= (gethostbyname(hostname))[0];
	my $client_hostname= $client_ref->hostname;
	my $client_name= $client_ref->name;
	my $server_ip= inet_ntoa((gethostbyname(hostname))[4]);
	my $client_ip= $client_ref->ip_addr;
	my $client_gateway= $client_ref-> gateway;
	my $res_location= $this->image_location;
	my $template_str;
	my $each_field;
	my $niminfo_contents= "";
	my $bosinst_data_file= $this->image_location.'/bosinst.data';
	my $target_disk= $client_ref->target_disk;
	my $bosinst_data_contents;
	my $client_files; # hash ref to client files
	my $niminfo_template= <<TEMPLATE_END;
#------------------ Network Install Manager ---------------
# warning - this file contains NIM configuration information
#       and should only be updated by NIM
export NIM_SERVER_TYPE=linux
export NIM_SYSLOG_PORT=514
export NIM_SYSLOG_FACILITY=local2
export NIM_NAME=$client_hostname
export NIM_HOSTNAME=$client_hostname
export NIM_CONFIGURATION=standalone
export NIM_MASTER_HOSTNAME=$server_hostname
export REMAIN_NIM_CLIENT=[remain_nim_client]
export RC_CONFIG=rc.bos_inst
export NIM_BOSINST_ENV="/../SPOT/usr/lpp/bos.sysmgt/nim/methods/c_bosinst_env"
export NIM_BOSINST_RECOVER="/../SPOT/usr/lpp/bos.sysmgt/nim/methods/c_bosinst_env -a hostname=$client_hostname"
[image_data]
export NIM_BOSINST_DATA=/NIM_BOSINST_DATA
export SPOT=[spot]
export NIM_CUSTOM="/../SPOT/usr/lpp/bos.sysmgt/nim/methods/c_script -a location=[cust_script_location]"
[nim_bos_image]
export NIM_HOSTS="$client_ip:$client_hostname $server_ip:$server_hostname"
export NIM_MOUNTS="[nim_mounts]"
TEMPLATE_END

	# clear out ROUTES if the client's gateway is non-existant (0.0.0.0)
	if ( $client_gateway =~ /^\s*0\.0\.0\.0\s*$/ )
	{
		$niminfo_template .= "export ROUTES=\"\"\n";
	}
	else
	{
		$niminfo_template .= "export ROUTES=\" default:0:$client_gateway \"\n";
	}

	# commands
	my $LN= find_command('ln');
	my $CAT= find_command('cat');

	# make sure the boot image is there
	return 0 unless (-f $boot_image);

	# process any client-specific allocation data
	$client_files= $this->apply_client_cust($client_ref);
	unless ($client_files)
	{
		log_entry(__("Error applying customization for client %s\n", $client_ref->name), 1);
		return 0;
	}

	# make a symlink to the resource's boot image with the client's hostname
	unless (log_cmd("$LN -fs $bi_target $bi_link"))
	{
		log_entry(__("Failed to link the file %s to file %s\n", $bi_link, $boot_image), 1);
		return 0;
	}

	$bosinst_data_file= $client_files->{bosinst_data} if ($client_files->{bosinst_data});

	# set the target disk
	if ($client_ref->target_disk)
	{
		$bosinst_data_contents= `$CAT $bosinst_data_file`;

		if ($bosinst_data_contents =~ /PHYSICAL_LOCATION/)
		{
			$bosinst_data_contents =~ s/PHYSICAL_LOCATION\s*=\s*\n/PHYSICAL_LOCATION = $target_disk\n/gs;
		}
		else
		{
			$bosinst_data_contents =~ s/target_disk_data:\n/target_disk_data:\n    PHYSICAL_LOCATION = $target_disk\n/gs;
		}
		
		$bosinst_data_file= $this->image_location.'/'.$client_ref->name."_cust/$client_hostname-bosinst.data" unless ($client_files->{bosinst_data});
		
		unless (open(NEW_BOSINST_DATA, ">$bosinst_data_file"))
		{
			log_entry(__("Failed to create the file %s : %s\n", $bosinst_data_file, $!), 1);
			return 0;
		}
		print NEW_BOSINST_DATA $bosinst_data_contents;
		close NEW_BOSINST_DATA;
	}

	# fill in the niminfo template with stuff
	
	# remain_nim_client (i.e. config IP?)
	$template_str= $client_ref->config_ip ? 'yes' : 'no';
	$niminfo_template =~ s/\[remain_nim_client\]/$template_str/gs;

	# SPOT mount point
	$template_str= "\"$server_hostname:".$this->image_location."/SPOT/usr\"";
	$niminfo_template =~ s/\[spot\]/$template_str/gs;

	# image_data?
	$template_str= (-f $this->image_location.'/image.data') ? 'export NIM_IMAGE_DATA=/NIM_IMAGE_DATA' : '';
	$niminfo_template =~ s/\[image_data\]/$template_str/gs;

	# nim_bos_image mount point
	$template_str= (-f $this->image_location.'/mksysb') ? "export NIM_BOS_IMAGE=/NIM_BOS_IMAGE\nexport NIM_BOS_FORMAT=mksysb" : 
		"export NIM_BOS_IMAGE=/SPOT/usr/sys/inst.images/installp/ppc/bos\nexport NIM_BOS_FORMAT=rte";
	$niminfo_template =~ s/\[nim_bos_image\]/$template_str/gs;

	# nim_mounts 
	$template_str= "";

	# BOS image? or lpp_source?
	$template_str.= "$server_hostname:".$this->image_location.'/mksysb:/NIM_BOS_IMAGE:file ' if (-e $this->image_location.'/mksysb');
	$template_str.= "$server_hostname:".$this->image_location.'/lpp_source:/SPOT/usr/sys/inst.images:dir ' if (-e $this->image_location.'/lpp_source');
	$template_str.= "$server_hostname:$bosinst_data_file:/NIM_BOSINST_DATA:file " if (-e $bosinst_data_file);
	$template_str.= "$server_hostname:".$this->image_location.'/image.data:/NIM_IMAGE_DATA:file ' if (-e $this->image_location.'/image.data');
	$niminfo_template =~ s/\[nim_mounts\]/$template_str/gs;

	# set path to customization script
	$template_str= $server_hostname.':'.$client_files->{nim_script};
	$niminfo_template =~ s/\[cust_script_location\]/$template_str/gs;

	# take out all the emtpy lines from the niminfo_template
	foreach $each_field (split /\n/, $niminfo_template)
	{
		$niminfo_contents.= "$each_field\n" if ($each_field);
	}

	$niminfo_contents.= "\n";

	# create the niminfo file
	unless (open(NIMINFO, "> $niminfo_file"))
	{
		log_entry(__("Failed to create the file %s : %s\n", $niminfo_file, $!), 1);
		return 0;
	}

	print NIMINFO $niminfo_contents;
	close NIMINFO;

	$rc;
}

sub rm_bootfiles
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my $rc= 1;
	my $bi_link= TFTPBOOT.'/'.$client_ref->hostname;
	my $special_bidata= $this->image_location.'/'.$client_ref->hostname.'-bosinst.data';
	my $niminfo= TFTPBOOT.'/'.$client_ref->hostname.'.info';
	
	# remove the symlink to the boot image
	unless (-f $bi_link && unlink $bi_link)
	{
		log_entry(__("Failed to delete file %s : %s\n", $bi_link, $!), 1);
	}

	# remove any customized bosinst.data file for this client
	if(-f $special_bidata) 
	{
		unless (unlink $special_bidata)
		{
			log_entry(__("Failed to delete file %s : %s\n", $special_bidata, $!), 1);
		}
	}

	#remove the client's info file
	unless (-f $niminfo && unlink $niminfo)
	{
		log_entry(__("Failed to delete file %s : %s\n", $niminfo, $!), 1);
	}

	unless ($this->rm_client_cust($client_ref))
	{
		log_entry(__("Failed to delete client customization for client %s\n", $client_ref->name), 1);
	}

	$rc;
}

sub verify_resource
{
	# shift args
	my $this= shift;
	my $verbose= shift;

	# local data
	my $rc= 1;
	my $CAT= find_command("cat");
	my $index_file= $this->image_location."/index";
	my @index= `$CAT $index_file 2>/dev/null`;
	my $each_line;
	my $each_res;
	my $each_file;
	my @missing= ();

	foreach $each_line (@index)
	{
		chomp $each_line;
		($each_res, $each_file)= split /:/, $each_line;
		$each_file= $this->image_location."/$each_file";
	
		unless (-e $each_file)
		{
			push @missing, $each_res;
			$rc= 0;
		}
	}

	$rc= 0 unless (@index);

	log_entry(__("The resource %s is missing the following: %s\n", $this->name, join ', ', @missing), 1) if (@missing && $verbose);

	$rc;
}

sub get_install_status
{
	# shift args
	my $this= shift;
	my $client_ref= shift;
	my $prev_status= shift;

	# local data
	my $syslog_file= $INSTALL_LOG;
	my @file_contents;
	my @host_contents;
	my @new_status= ();
	my $client_hostname_exp= $client_ref->hostname;
	my %prev_status_hash= ();
	my $each;

	# commands
	my $CAT= find_command('cat');

	return 0 unless (ref($client_ref) =~ /^(?:OSInstall::)Client/);

	# build hash out of @$prev_status
	foreach $each (@$prev_status) { $prev_status_hash{$each}= 1 }

	# build regex to grep through the log for the client hostname
	$client_hostname_exp =~ s/^(\w+?)(\..*)$/$1(?:$2)?/; # basically means do not require fully qualified hostname
	$client_hostname_exp =~ s/\./\\./g;

	@file_contents= `$CAT $syslog_file`;
	@host_contents= grep /$client_hostname_exp/, @file_contents;

	# now build only the status messages we have not yet seen
	foreach $each (@host_contents) { push @new_status, $each unless ($prev_status_hash{$each}) }

	# add a string indicating the installation is finished if the client reported 100% BOS install complete
	push @new_status, "FINISHED\n" if (grep /100\%/, @new_status);

	\@new_status;
}

sub apply_client_cust
{
	my $this= shift;  # reference to AIX_Resource instance
	my $client_ref= shift; # reference to allocated client

	# make sure params are good
	return 0 unless ($this && $client_ref && ref($client_ref) =~ /^(?:OSInstall::)Client/);
	
	my $client_cust_dir= $this->image_location."/".$client_ref->name."_cust";
	my $cust_script_file= $client_cust_dir.'/'.$client_ref->name.'.script';
	my $client_files;

	# create the directory inside the resource's location
	unless (-d $client_cust_dir || mkdir($client_cust_dir, 0755))
	{
		log_entry(__("Failed to create the directory %s : %s\n", $client_cust_dir, $!), 1);
		return 0;
	}

	# if the client has a config_file, call the copy_client_config_file() sub
	if ($client_ref->config_file)
	{
		$client_files= $this->copy_client_config_file($client_ref);
		unless ($client_files)
		{
			log_entry(__("Failed to copy configuration data for client %s\n", $client_ref->name), 1);
			$this->rm_client_cust($client_ref);
			return 0;
		}
	}

	# now do the default cust script
	my $cust_script_contents= $this->gen_cust_script($client_ref, $client_files);

	unless ($cust_script_contents)
	{
		log_entry(__("Failed to generate default customization script for client %s\n", $client_ref->name), 1);
		return 0;
	}

	# copy the cust script into the client's cust directory
	unless (open (CUST_SCRIPT, ">$cust_script_file"))
	{
		log_entry(__("Failed to open file %s for writing: %s\n", $cust_script_file, $!), 1);
		return 0;
	}

	print CUST_SCRIPT $cust_script_contents;
	close CUST_SCRIPT;

	# set execute permissions to the cust script
	unless (chmod 0755, $cust_script_file)
	{
		log_entry(__("Failed to change permissions for file %s\n", $cust_script_file), 1);
		return 0;
	}

	$client_files->{nim_script}= $cust_script_file;

	$client_files;
}

# per allocation, a client is allowed a customization file.  After doing the AIX_Resource
# allocation, do this client-specific resource copying, if there's a config file for the 
# client in this allocation.
sub copy_client_config_file
{
	my $this= shift;  # reference to AIX_Resource instance
	my $client_ref= shift; # reference to allocated client

	my $config_file= $client_ref->config_file;
	my $client_cust_dir= $this->image_location."/".$client_ref->name."_cust";
	my %client_files= ();

	return 0 unless ($this && $client_ref && -f $config_file);

	# %client_files looks like:
	#	<resource name> => <path>
	# path is relative to $this->image_location

	unless (open (CONFIG_FILE, $config_file))
	{
		log_entry(__("Error opening file %s : %s\n", $config_file, $!), 1);
		return 0;
	}

	# extract fields out of the config_file
	while(<CONFIG_FILE>)
	{
		my $each_line= $_;
		chomp $each_line;

		my @fields= split /:/, $each_line;
		$client_files{$fields[0]}= $fields[1];
	}

	# copy each file for each field into the client_cust_dir 
	foreach my $each_field (keys %client_files)
	{
		# extract the filename out of its absolute path
		$client_files{$each_field} =~ /.*\/(.*)/;
		my $relative_name= $1;
		my $new_name= $client_cust_dir.'/'.$relative_name;
	
		unless (File::Copy::copy($client_files{$each_field}, $new_name))
		{
			log_entry(__("Failed to copy file from %s to %s : %s\n", $client_files{$each_field}, $new_name, $!), 1);
			return 0;
		}

		# make the path relative to image location
		$client_files{$each_field}= $client_ref->name.'_cust/'.$relative_name;
	}

	return \%client_files;
}

# for a given client and resource, generates a customization script, which is capable of
# performing IP configuration, running another user-supplied cust script
sub gen_cust_script
{
	my $this= shift;  # reference to AIX_Resource instance
	my $client_ref= shift; # reference to allocated client
	my $client_files= shift; # reference to the stuff in the client's config_file, if anything

	my $server_hostname= (gethostbyname(hostname))[0];
	my $client_cust_dir= $this->image_location."/".$client_ref->name."_cust";
	my $cust_script_template= <<END_CUST_SCRIPT;
#!/bin/ksh
# nim_script for $client_ref->hostname
result=success
export NIM_NON_41_MASTER=yes
export NIM_SCRIPT=yes
[mk_nim_client]

[resolv_conf]

[cust_script]

[[ \${result} = success ]] && exit 0 || exit 1
END_CUST_SCRIPT

	my $client_hostname= $client_ref->hostname;
	my $client_ip_addr= $client_ref->ip_addr;
	my $client_subnet_mask= $client_ref->subnet_mask;
	my $mk_nim_client= <<END_MK_NIM_CLIENT;
# NIM client initialization
/../SPOT/usr/lpp/bos.sysmgt/nim/methods/c_mk_nimclient \\
	-ahostname=$client_hostname \\
	-aip=$client_ip_addr \\
	-acable_type=N/A \\
	-asnm=$client_subnet_mask
[[ \$? != 0 ]] && result=failure
END_MK_NIM_CLIENT

	my $resolv_conf= <<END_RESOLV_CONF;
# /etc/resolv.conf
/../SPOT/usr/lpp/bos.sysmgt/nim/methods/c_cp_resolv \\
	-a location="[resolv_conf_location]"
END_RESOLV_CONF

	my $extra_cust_script= <<END_CUST_SCRIPT;
# optional customization script
/../SPOT/usr/lpp/bos.sysmgt/nim/methods/c_script \\
	-alocation="[cust_script_location]"
[[ \$? != 0 ]] && result=failure
END_CUST_SCRIPT

	# if we want to configure this client's IP address after the installation, then call c_mk_nimclient
	if ($client_ref->config_ip)
	{
		$cust_script_template=~ s/\[mk_nim_client\]/$mk_nim_client/gs;
	}
	else
	{
		$cust_script_template=~ s/\[mk_nim_client\]//gs;
	}

	# do we have a resolv_conf?
	if ($client_files->{resolv_conf} || -f $this->image_location.'/resolv_conf')
	{
		# fill in the path to mount the resolv_conf from the install server
		my $resolv_conf_path=  -f $this->image_location.'/resolv_conf' ? 
			$this->image_location.'/resolv_conf' : 
			$client_cust_dir.'/resolv_conf';
		
		$resolv_conf=~ s/\[resolv_conf_location\]/$server_hostname:$resolv_conf_path/gs;
		
		$cust_script_template=~ s/\[resolv_conf\]/$resolv_conf/gs;
	}
	else
	{
		$cust_script_template=~ s/\[resolv_conf\]//gs;
	}

	# do we have a cust_script to fill in?
	if ($client_files->{cust_script})
	{
		$extra_cust_script=~ s/\[cust_script_location\]/$server_hostname:$client_files->{cust_script}/gs;
		$cust_script_template=~ s/\[cust_script\]/$extra_cust_script/gs;
	}
	else
	{
		$cust_script_template=~ s/\[cust_script\]//gs;
	}

	$cust_script_template;
}

# undoes any client-specific allocation stuff
sub rm_client_cust
{
	my $this= shift;  # reference to AIX_Resource instance
	my $client_ref= shift; # reference to allocated client

	my $client_cust_dir= $this->image_location."/".$client_ref->name."_cust";

	return rmtree $client_cust_dir;
}

sub copy_dir
{
	my $src_dir= shift;
	my $dst_dir= shift;
	my $arg_list= shift;
	my %copy_list= ();
	my $first_run= 1;
	my $each_arg;
	my $each_glob;
	
	my $rc= 1;
	
	my $src_dir_re = qr/$src_dir/;
	
	return 0 unless (-d $src_dir && $dst_dir);
	
	# resolve all globs in the copy list, and store it in a local hash
	my $before_chdir = getcwd;
	chdir $src_dir;
	foreach $each_arg (@$arg_list)
	{
		foreach $each_glob (glob $each_arg)
		{
			$copy_list{$each_glob}= 1;
		}
	}
	
	mkpath $dst_dir unless (-d $dst_dir);
	$dst_dir= abs_path($dst_dir);
	
	$rc= find(sub
	{
		if ($first_run)
		{
			# if we're here, we're processing the rot directory itself
			# in the destinsation, the root's name will be different
			$first_run= 0;
		}
		else
		{
			my $tgt_dir= $File::Find::dir;
			my $each_file= $File::Find::name;
	
	
			# this if/else strips out the top level directory from the current directory
			if ($tgt_dir !~ /\//)
			{
				$tgt_dir= "";
			}
			else
			{
			#                               $tgt_dir =~ s/.*?\///;
				#strip out leading $src_dir/
				$tgt_dir =~ s/$src_dir_re\/*//;
			}
		
			# now set the current target directory to create or write to
			$tgt_dir= $dst_dir."/".$tgt_dir;
			$tgt_dir =~ s/(.*)\/$/$1/; # eliminate trailing /
		
			# replace $src_dir wth ./ before we compare to copy_list
			$each_file =~ s/$src_dir_re\/*/.\//;
		
			# return if we have a file list and this file isn't present in it
			return if ($arg_list && !$copy_list{$each_file});
		
			# strip entire leading path from our file name
			unless(-d $each_file) {
				$each_file =~ s/.*\///;
			}
			unless(-d $tgt_dir) {
				mkpath $tgt_dir;
			}
		
			if (-d $each_file)
			{
				mkpath $tgt_dir."/".$each_file;
			}
			elsif (-l $each_file)
			{
				symlink abs_path($each_file), $tgt_dir."/".$each_file;
			}
			else
			{
				File::Copy::copy($each_file, $tgt_dir."/".$each_file);
			}
		}
	}, $src_dir);

	chdir $before_chdir;

	$rc;
}

# locate and install the aix-res RPM (override vs. media)
sub get_rpm_path {
	my $image_dir = shift;
	my $release;
	my $level;
	my $file;

	# get release + level from media mount
	for $file ( glob($image_dir . '/*') ) {
		($release, $level) = ($file =~ /^$image_dir\/(\d+)-(\d+)$/);
		last if ($release && $level);
	}
	return undef unless ($release && $level);

	# search for an rpm file to override one on media
	for $file ( glob(AIX_RES_LOCATION . '/*') ) {
		if ( $file =~ /.*\/aix-res-(\d+)-(\d+)-.*\.rpm$/ ) {
			return $file if ( ($1 eq $release) && ($2 eq $level) )
		}
	}

	# no override rpm file found, return one on media
	return ( glob($image_dir . '/' . RPM_LOC) );
}


1;
__END__


