package OSInstall::Common;

use strict;
use Fcntl;
use FileHandle;
use Fcntl qw(:DEFAULT :flock);
use IPC::Open3;
use IO::Select;
use English;
use Carp;

require Exporter;

our @ISA = qw(Exporter);

our @EXPORT = qw(
	$VAR_DIR
	$LOCK_DIR
	$EVENT_LOG
	$INSTALL_LOG
	log_entry
	log_cmd
	lock_file
	fill_in_args
	check_credentials
	find_command
	build_param_list
	__
);

our $VERSION = '1.2';

# if we're on an HMC, we want to mount the media at /media/cdrom
if ( -f "/opt/hsc/bin/commandcontrol" ) {
  our $MOUNT_POINT = "/media/cdrom";
} else {
  our $MOUNT_POINT = "/opt/osinstall/src_mnt";
}

# lockfile register- all lockfiles throughout osinstall will have an entry here
# hash contents are:
# 	lock_name => relative_filename
our %lockfile_register= (
		event_log => "event_log_lockfile",
		service_config => "service_config_lockfile",
		xml_repos => "xml_repos_lockfile",
		platform_tools_sysconfig => "platform_tools_sysconfig_lockfile",
		aix_res_rpm => "aix_res_rpm_lockfile",
	);

# list of directories in which we want to search for commands
our @additional_search_path= qw(
	/bin
	/sbin
	/usr/bin
	/usr/sbin
	/usr/local/bin
	/usr/local/sbin
	/opt/bin
	/opt/sbin
);

# Constants common to all OSInstall APIs
our $VAR_DIR= '/var/osinstall';
our $LOCK_DIR= "${VAR_DIR}/lock";
our $EVENT_LOG= "${VAR_DIR}/event_log";
our $INSTALL_LOG= "/var/log/osinstall.log";

# here we will touch needed files if they don't exist, initialize global variables
#BEGIN
#{
#	#initialize the constants
#	$VAR_DIR= '/var/osinstall';
#	$LOCK_DIR= "${VAR_DIR}/lock";
#	$EVENT_LOG= "${VAR_DIR}/event_log";
#
#	# osinstall var directory
#	unless (-d $VAR_DIR)
#	{
#		mkdir $VAR_DIR, 0700 or die "$VAR_DIR: $!\n";
#	}
#	
#	#directory containing lock files
#	unless (-d $LOCK_DIR)
#	{
#		mkdir $LOCK_DIR, 0700 or die "$LOCK_DIR: $!\n";
#	}
#
#	# osinstall event log
#	unless (-f $EVENT_LOG)
#	{
#		open TOUCH, ">$EVENT_LOG" or die "$EVENT_LOG : $!\n";
#		close TOUCH;
#	}
#} # end BEGIN initialization

#########################################################################
# common functions for osinstall APIs                                   #
# add stuff here that all the other OSInstall modules could make use of #
#########################################################################

# --------------------------------------------------------------------------- #
# NAME:         __
#
# FUNCTION:     Translates the provided text, using that text as the key for
#		the translated message (that is, when this thing is actually 
#		translated).
#
# PARAMETERS:   SCALAR: Text to be translated, as a printf formatted string
#		SCALAR...SCALAR: Parameters to pass into the printf formatted 
#				 string
#
# RETURNS:     	The translated string.
# --------------------------------------------------------------------------- #
sub __
{
	my $key;

        $key= shift;
        $key= sprintf $key, @_;

	$key;
}

# --------------------------------------------------------------------------- #
# NAME:         log_cmd
#it puts
# FUNCTION:     Runs a shell command and saves its output to the osinstall event 
#		logs.
#
# PARAMETERS:   SCALAR: Shell command to run with arguments like this: ... :arg1: ... :arg2: ...
#		SCALAR...SCALAR: the arguments to insert into the command
#
# RETURNS:     	Return code of the invoked shell command
# --------------------------------------------------------------------------- #
sub log_cmd
{
	my $rc= 1;
	my $i= 1;
	my $cmd_rc;
	my $my_cmd= shift;
	my $suppress_error= shift;
	my $cmd_name= (split /\s+/, $my_cmd)[0];
	local (*CMD_STDIN, *CMD_STDOUT, *CMD_STDERR);
	my $select;
	my @handles;
	my $each_handle;
	my $each_line;
	my $pid;

	# catch SIGPIPE because open3 won't return if there's an error
	local $SIG{PIPE}= sub { &log_entry(__("An error has occurred while executing %s : %s\n", $my_cmd, $!), 1); croak; };

	return 0 unless (&check_credentials); # make sure we're authorized to exec commands
	
	$my_cmd= &fill_in_args($my_cmd, \@_) if (@_);
	
	&log_entry(__("Executing: %s  \n", $my_cmd));

	if ($pid= open3(\*CMD_STDIN, \*CMD_STDOUT, \*CMD_STDERR, "$my_cmd"))
	{
		# we're not using stdin
		close CMD_STDIN;
	
		# initialize the select object
		$select= new IO::Select();
		$select->add(\*CMD_STDOUT);
		$select->add(\*CMD_STDERR);
	
		while (@handles= $select->can_read()) 
		{
			foreach $each_handle (@handles)
			{
				if (fileno($each_handle) == fileno(CMD_STDOUT) && ($each_line= <CMD_STDOUT>))
				{
					&log_entry(__("STDOUT (%s): %s", $cmd_name, $each_line));
				}
				elsif (fileno($each_handle) == fileno(CMD_STDERR) && ($each_line= <CMD_STDERR>))
				{
					&log_entry(__("STDERR (%s): %s", $cmd_name, $each_line));
				}

				# remove the handle if we've read all of it
				$select->remove($each_handle) if (eof($each_handle));
			}
		}
		close CMD_STDOUT;
		close CMD_STDERR;
	
		# reap the child so we canget the return code
		waitpid($pid, 0);
		$cmd_rc= $?;

		&log_entry(__("rc for %s = %d\n", $cmd_name, $cmd_rc));

		&log_entry(__("Execution of %s failed.\n", $cmd_name), 1) if ($cmd_rc && ! $suppress_error);
		$rc= 0 if ($cmd_rc);
	}
	else
	{
		# this will never run- that's why it's in the SIGPIPE handler above
		$rc= 0;
		&log_entry(__("An error has occurred while executing %s : %s\n", $my_cmd, $!), 1);
	}

	$rc;
}

# --------------------------------------------------------------------------- #
# NAME:         log_entry
#
# FUNCTION:     Prints text to the osinstall event log.
#
# PARAMETERS:   SCALAR: The translated text containing \n to log (this function will not translate messages)
#		SCALAR: Is this error output? (denoted by true/false value)
#		SCALAR: Disable printing to standard output/error? (denoted by true/false value)
#
# RETURNS:      nothing
# --------------------------------------------------------------------------- #
sub log_entry
{
	my $msg= shift;
	my $error= shift;
	my $no_std_print= shift;
	my $lock; #filehandle for the lockfile

	# obtain lock to write to the log synchronously
	if ($lock= lock_file('event_log', 'exclusive', 'blocking'))
	{
		#open log, create if it doesn't exist
		open (LOG, ">>$EVENT_LOG") or die "$EVENT_LOG : $!\n";

		#prepend error banner if this is an error message
		$msg= __('ERROR:')." ".$msg if ($error);

		#prepend date, time and process id to message
		my @lt = localtime;
		my $prefix = sprintf("[%02d/%02d/%4d][%02d:%02d:%02d][%d]: ",
			$lt[4] + 1, $lt[3], $lt[5] + 1900, $lt[2], $lt[1], $lt[0], $$);
		$msg = $prefix . $msg;

		print LOG $msg;
		unless ($no_std_print)
		{
			if ($error)
			{
				print STDERR $msg;
			}
			else
			{
				print STDOUT $msg;
			}
		}

		close LOG;

		close $lock; # release our write lock
	}
}

# --------------------------------------------------------------------------- #
# NAME:         lock_file
#
# FUNCTION:     This function will serve to synchronize reentrant osinstall code
#		through the use of lock files.  Locks that can be obtained on these
#		may be shared or exclusive, and the function may be blocking or 
#		non-blocking.  
#
# PARAMETERS:   SCALAR: Lock identifier (string)
#		SCALAR: Lock type ("shared", "exclusive") 
#		SCALAR: Lock attempt behavior ("blocking", "nonblocking")
#
# RETURNS:     	typeglob of the locked file, or undef if failure
# --------------------------------------------------------------------------- #
sub lock_file
{
	my $lock_id= shift;
	my $lock_type= shift;
	my $lock_behavior= shift;
	my $lockfile= $LOCK_DIR;
	my $flock_operation= 0; # we will OR the proper values in here to pass to flock
	my $fh; # FileHandle object reference we will use and return 

	# bail if we don't know this lockfile id or any of the arguments
	return undef unless ($lockfile_register{$lock_id} && ($lock_type eq 'shared' || $lock_type eq 'exclusive') && (! $lock_behavior || $lock_behavior eq 'nonblocking' || $lock_behavior eq 'blocking'));

	# obtain the actual path to the lockfile
	$lockfile.= '/'.$lockfile_register{$lock_id};

	# create the lock file if it doesn't exist
	sysopen(TMP, $lockfile, O_WRONLY | O_CREAT) or confess "$lockfile : $!\n";
	close TMP;

	# OR into the flock_operation the proper flags
	if ($lock_type eq 'shared')
	{
		$flock_operation|= Fcntl::LOCK_SH; # shared read lock
	}
	elsif ($lock_type eq 'exclusive')
	{
		$flock_operation|= Fcntl::LOCK_EX; # exclusive read/write lock
	}

	# set non-blocking flag if needed
	$flock_operation|= Fcntl::LOCK_NB if ($lock_behavior eq 'nonblocking');

	# open the filehandle for the lock file
	$fh= new FileHandle $lockfile, O_RDWR | O_CREAT;

	# now, do the actual task of locking the file with the specified options
	unless (flock($fh, $flock_operation))
	{
		# if we fail the flock for some reason, close the fh and return undef
		print STDERR "$lockfile : $!\n" unless ($lock_behavior eq "nonblocking"); # only report error if blocking
		close $fh;
		return undef;
	}

	# return the filehandle
	return $fh;
}

# --------------------------------------------------------------------------- #
# NAME:         fill_in_args
#
# FUNCTION:     This function fills in the arguments of a string denoted by the
#		format "... [arg1] ... [arg2] ... [argN]" (the args need not be
#		in ascending order) with the elements supplied in the list param
#
# PARAMETERS:   SCALAR: Format string governed by the above rules
#		SCALAR: Reference to a list of arguments to fill in to the 
#			format string
#
# RETURNS:     	true/false
# --------------------------------------------------------------------------- #
sub fill_in_args
{
	#shift args
	my $str= shift;
	my $args= shift;

	# local data
	my $i;
	my $each_data;

	# error checking
	confess __("Invalid or insufficient arguments\n") unless ($str && $args);

	for ($i= 1; $i <= @$args; $i++)
	{
		$each_data= $args->[$i - 1];
		$str =~ s/\[arg$i\]/$each_data/g;
	}

	$str;
}

# --------------------------------------------------------------------------- #
# NAME:         check_credentials
#
# FUNCTION:     This function tells whether the uid of the current invoker of
#		perl is authorized to execute a command.
#
# PARAMETERS:   none
#
# RETURNS:     	true/false
# --------------------------------------------------------------------------- #
sub check_credentials
{
	my $rc= 1;

	unless ($UID == 0)
	{
		carp __("You are not authorized to execute commands using this module.\n");
		$rc= 0;
	}

	$rc; 
}

# --------------------------------------------------------------------------- #
# NAME:         find_command
#
# FUNCTION:     This function attempts to find the full path to a command by
#		searching through the PATH variable found in the environment
#		and through looking through some specified directories.
#
# PARAMETERS:   none
#
# RETURNS:     	true/false
# --------------------------------------------------------------------------- #
sub find_command
{
	# shift args
	my $cmd= shift;

	# local data
	my @search_path= split /:/, $ENV{PATH};
	my $each_dir;
	my $full_cmd_path;

	# first search in the invoker's PATH env variable
	foreach $each_dir (@search_path)
	{
		if (-x "$each_dir/$cmd" && ! -d "$each_dir/$cmd")
		{
			$full_cmd_path= "$each_dir/$cmd";
			last;
		}
	}

	unless ($full_cmd_path)
	{
		# if we couldn't find anything in the invoker's path, search our own path for the command
		foreach $each_dir (@additional_search_path)
		{
			if (-x "$each_dir/$cmd" && ! -d "$each_dir/$cmd")
			{
				$full_cmd_path= "$each_dir/$cmd";
				last;
			}
		}
	}

	$full_cmd_path;
}

# This function will take an unordered input hash of parameters and an ordered list of metadata from
# a service rule and will supply the parameters ordered in the same way as the metadata
#
# Parameters:
#	SCALAR: Reference to a data input hash
#	SCALAR: Reference to a data_param list from one of the service rules
sub build_param_list
{
	# shift the args
	my $data_hash= shift;
	my $ordered_params= shift;

	#local data
	my @ordered_data= ();
	my $each_param;

	confess __("Invalid or insufficient arguments\n") unless ($data_hash && $ordered_params);

	# the list @ordered_data will contain all the values in order according to a rule's data_params field
	# any items in the input data not present in the data_params field are not inserted into the ordered 
	# add data
	foreach $each_param (@$ordered_params)
	{
		if ($data_hash->{$each_param})
		{
			push @ordered_data, $data_hash->{$each_param};
		}
		else
		{
			push @ordered_data, '';
		}
	}

	\@ordered_data;
}


1;
__END__

