package OSInstall::Linux_Resource;

use strict;
use warnings;
use lib '/opt/csm/pm';

use OSInstall::Common;
use OSInstall::Platform_Tools;
use OSInstall::OS_Resource;
use IO::Socket;
use Sys::Hostname;
use DCAPI;

#our @ISA = qw(OS_Resource);
use base ("OSInstall::OS_Resource");

my %modifiable_fields= ();

my %constant_fields= ();

sub new
{
	# shift args
	my $class= shift;
	my $this= $class->SUPER::new(@_);  # get an OS_Resource object

	# indicate external usage of CSM's DCAPI
	$ENV{'DC_ENVIRONMENT'} = 1;

	# local data
	my $rc= 1;
	my $server_hostname= (gethostbyname(hostname))[0];
	my ($distro_version, $service_level)= split /\s+/, $this->version;
	
	# bless into a Linux_resource object
	bless $this, $class;
	
	my %server_nodeinfo= (
		$server_hostname => {
			InstallOSName => 'Linux', 
			InstallDistributionName => $this->distribution,
			InstallPkgArchitecture => 'ppc64',
			InstallDistributionVersion => $distro_version,
			InstallServiceLevel => $service_level,
			ManagementServer => $server_hostname,
			InstallServer => $server_hostname,
			Hostname => $server_hostname,
		},
	);

	# call DCcopycds
	unless ($this->distribution)
	{
		log_entry(__("A distribution type must be supplied for this Linux resource"), 1);
		return 0;
	}

	# if this CSM distro is already defined, we won't call DCcopycds again
	unless ($this->ready)
	{
		if (DCAPI->DCcopycds(undef, \%server_nodeinfo, 0, 1, 0, $this->image_source))
		{
			log_entry(__("Failed to copy Linux resources from %s\n", $this->image_source), 1);
			return 0;
		}
		$this->{ready} = 1;
	}

	$this;
}

sub allocate
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my $rc= 1;
	my $server_hostname= (gethostbyname(hostname))[0];
	my ($distro_version, $service_level)= split /\s+/, $this->version;
	my %bootparams= ();
	
	my %client_nodeinfo= (
		$client_ref->hostname => {
			InstallOSName => 'Linux', 
			InstallDistributionName => $this->distribution,
			InstallPkgArchitecture => 'ppc64',
			InstallDistributionVersion => $distro_version,
			InstallServiceLevel => $service_level,
			InstallAdapterMacaddr => $client_ref->mac_addr,
			InstallAdapterNetmask => $client_ref->subnet_mask,
			InstallAdapterGateway => $client_ref->gateway,
			ManagementServer => $server_hostname,
			InstallServer => $server_hostname,
			Hostname => $client_ref->hostname
		},
	);

	unless (OSInstall::Platform_Tools::flush_firewall_rules($client_ref->hostname))
	{
		log_entry(__("Could not flush firewall rules for client %s\n", $client_ref->name));
	}

	if (DCAPI->DCenableservices(undef, \%client_nodeinfo, undef, 0, 1))
	{
		log_entry(__("Failed to enable system services for Linux network installation\n"), 1);
		return 0;

	}

	if (DCAPI->DCsetupinstaller(undef, \%client_nodeinfo,  undef, 0, 1))
	{
		log_entry(__("Failed to configure system services for Linux network installation of client %s\n", $client_ref->name), 1);
		return 0;
	}

	if (DCAPI->DCinstallnode({TFTPPackage => 'tftp', NetworkInstallProtocol => 'nfs'}, \%client_nodeinfo, undef, 0, 1, \%bootparams, '', 0))
	{
		log_entry(__("Failed to configure installation environment for Linux network installation of client %s\n", $client_ref->name), 1);
		return 0;
	}

	# fill in the firmware boot parameters for this client if DCAPI returned any, also print them on the screen
	if ($bootparams{$client_ref->hostname})
	{
		$client_ref->boot_params($bootparams{$client_ref->hostname});
		log_entry(__("The firmware command needed to install Linux on this client is: 'boot %s'\n", $client_ref->boot_params));
	}

	$rc;
}

sub deallocate
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my $rc= 1;
	my $server_hostname= (gethostbyname(hostname))[0];
	my ($distro_version, $service_level)= split /\s+/, $this->version;
	
	my %client_nodeinfo= (
		$client_ref->hostname => {
			InstallOSName => 'Linux', 
			InstallDistributionName => $this->distribution,
			InstallPkgArchitecture => 'ppc64',
			InstallDistributionVersion => $distro_version,
			InstallServiceLevel => $service_level,
			InstallAdapterMacaddr => $client_ref->mac_addr,
			InstallAdapterNetmask => $client_ref->subnet_mask,
			InstallAdapterGateway => $client_ref->gateway,
			ManagementServer => $server_hostname,
			InstallServer => $server_hostname,
			Hostname => $client_ref->hostname,
		},
	);

	if (DCAPI->DCrmnode(undef, \%client_nodeinfo,  undef, 0, 1))
	{
		log_entry(__("Could not deallocate client %s from resource %s\n", $client_ref->name, $this->name), 1);
		return 0;
	}

	# remove firmware boot parameters from this client's definition if we had any before
	$client_ref->boot_params("") if ($client_ref->boot_params);

	# restore the firewall rules for this client (if we're on an HMC)
	OSInstall::Platform_Tools::enforce_firewall_rules($client_ref->hostname) or $rc= 0;

	$rc;
}

sub rm_images
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my $rc= 1;
	my $server_hostname= (gethostbyname(hostname))[0];
	my ($distro_version, $service_level)= split /\s+/, $this->version;

	my %server_nodeinfo= (
		$server_hostname => {
			InstallOSName => 'Linux', 
			InstallDistributionName => $this->distribution,
			InstallPkgArchitecture => 'ppc64',
			InstallDistributionVersion => $distro_version,
			InstallServiceLevel => $service_level,
			ManagementServer => $server_hostname,
			InstallServer => $server_hostname,
			Hostname => $server_hostname,
		},
	);

	if (DCAPI->DCrmOSInstallImage(undef, \%server_nodeinfo, undef, 0, 0))
	{
		log_entry(__("Failed to destroy resource %s\n", $this->name), 1);
		return 0;
	}

	$rc;
}

sub get_install_status
{
	# shift args
	my $this= shift;
	my $client_ref= shift;
	my $status_list_ref= shift;

	# local data
	my $rc= 1;
	my $server_hostname= (gethostbyname(hostname))[0];
	my ($distro_version, $service_level)= split /\s+/, $this->version;
	my %status= ();
	
	my %client_nodeinfo= (
		$client_ref->hostname => {
			InstallOSName => 'Linux', 
			InstallDistributionName => $this->distribution,
			InstallPkgArchitecture => 'ppc64',
			InstallDistributionVersion => $distro_version,
			InstallServiceLevel => $service_level,
			InstallAdapterMacaddr => $client_ref->mac_addr,
			InstallAdapterNetmask => $client_ref->subnet_mask,
			InstallAdapterGateway => $client_ref->gateway,
			ManagementServer => $server_hostname,
			InstallServer => $server_hostname,
			Hostname => $client_ref->hostname,
		},
	);

	if (DCAPI->DCmonitorinstall(undef, \%client_nodeinfo, undef, 0, 0, \%status)) {
		log_entry(__("Failed to get install status for client %s\n", $client_ref->name), 1);
		return 0;
	}

	# nothing is done with this status for the first call to this function
	return ["Installation monitoring has begun\n"] if (@$status_list_ref == 0);

	my $status = "CSM: " . $status{$client_ref->hostname}->{IStatus} . "\n";
	if ($status_list_ref->[-1] ne $status) {
		return [$status] 
	}
	else {
		return [];
	}
}

1;

