package OSInstall::XMLHandler;

use strict;
use warnings;

our @ISA = qw(Exporter);

use XML::LibXML;
use Carp;

# OSInstall classes
use OSInstall;
use OSInstall::Client;
use OSInstall::OS_Resource;
use OSInstall::AIX_Resource;
use OSInstall::Linux_Resource;
use OSInstall::Common;
use OSInstall::Remote_Resource;
use OSInstall::AIX_Remote;
# This may not be ready for initial version - use OSInstall::Linux_Remote;

our @EXPORT = qw(
XH_consume_sysplan 
XH_delete_sysplan
XH_read_object
XH_remove_object
XH_write_objects
XH_write_object
XH_list_objects
XH_alloc_resource
XH_deallocate
);

my $XML_REPOS = $OSInstall::Common::VAR_DIR."/xml_repos";

my $xmlschema;

# Initialize our schema object
sub init
{
	my $osi_dir = shift;
	use File::Find;
	my $schema_path;
    find (sub{if ($_ eq "niml.xsd")
        { $schema_path = $File::Find::name; }},"$osi_dir/schema"); 
    # Create an instance of the NIML schema 
    $xmlschema = XML::LibXML::Schema->new(location=>${schema_path});

}

# XH_consume_sysplan
# Parses a NIML sysplan document and returns a hash of client/os_resource and
# client/remote_resource pairs (local and remote resources)
#
# Returns by reference a hash of client/os_resource and client/remote_resource
# (local and remote) pairs, and a hash of OS_install objects
sub XH_consume_sysplan
{
    my $niml_inst_doc = shift;
	my $alloc_hash_ref = shift;
	my $obj_hash_ref = shift;
    my $parser = XML::LibXML->new;
    my $doc_tree = $parser->parse_file(${niml_inst_doc});
	
    my %allocations = (); # Hash to return allocations by reference
	my %sysplan_objects = (); # Hash to return instantiated sysplan objects

    # Validate here
    eval {
        $xmlschema->validate($doc_tree);
    };

	if($@) {
    	log_entry(__("Sysplan file failed validation: $@\n"), 1);
		return 0;
	}

    # Get XML document object
    my $doc_root = $doc_tree->getDocumentElement;

    # Give our default namespace a prefix
    $doc_root->setNamespace('http://www.ibm.com/niml','niml',1);


	# Parse client, os_resource and remote_resource entries to
	# populate allocations hash 

	# Check os_resource nodes for client references
	my @os_resources = $doc_root->findnodes("/niml:osinstall_config/niml:os_resource");
	# Check each os_resource for client references
	foreach my $os_xml (@os_resources) {
		my $os_str = $os_xml->findvalue('@name');
		foreach my $client_ref ($os_xml->findnodes('niml:ref_client')) {
			my $client_str = $client_ref->findvalue('@client_name');
			if($allocations{$client_str}) {
				log_entry(__("Client %s is referenced by multiple OS_Resources\n", $client_str), 1);
				return 0;
			}
			# Save each client reference to our allocations hash
			$allocations{$client_str} = $os_str;
		}
	}


	# Check remote_resource nodes for client references
	my @remote_resource = $doc_root->findnodes("/niml:osinstall_config/niml:remote_resource");
	# Check each remote_resource for client references
	foreach my $os_xml (@remote_resource) {
		my $os_str = $os_xml->findvalue('@name');
		foreach my $client_ref ($os_xml->findnodes('niml:ref_client')) {
			my $client_str = $client_ref->findvalue('@client_name');
			if($allocations{$client_str}) {
				log_entry(__("Client %s is referenced by multiple OS_Resources or Remote_Resources\n", $client_str), 1);
				return 0;
			}
			# Save each client reference to our allocations hash
			$allocations{$client_str} = $os_str;
		}
	}


	# Now check each client for os_resource/remote_resource reference,
	# checking for inconsistent references
	my @clients = $doc_root->findnodes("/niml:osinstall_config/niml:client");
	# Check each client for os_resource/remote_resource reference
	foreach my $client_xml (@clients) {
		my $client_str = $client_xml->findvalue('@name');
		if(my $os_str = $client_xml->findvalue('niml:ref_os_resource/@os_resource_name')) {

			# Check that references are consistent
			if(my $old_os_str = $allocations{$client_str}) {
				unless($old_os_str eq $os_str) {
					log_entry(__("Client %s contains reference to %s but is already referenced by %s\n", $client_str, $os_str, $old_os_str), 1);
					return 0;
				}
			} 
			# Save each ref_os_resource (local or remote) reference
			# to our allocations hash
			$allocations{$client_str} = $os_str;
		}
	}
	# Finished populating allocations hash, now instantiate objects in sysplan

    # Return all nodes in XML sysplan 
    my @all_nodes = $doc_root->findnodes("/niml:osinstall_config/*");
    
    # For each node in our sysplan, instantiate object and save reference in objects hash
    foreach my $niml_node (@all_nodes) {
		my $obj_ref = create_object($niml_node);
        unless($obj_ref) {
			log_entry(__("Could not instantiate sysplan object\n"), 1);
			return 0;
		}
		my $key_string = $niml_node->findvalue('@name');
		$sysplan_objects{$key_string} = $obj_ref
    }

	%{$alloc_hash_ref} = %allocations;
	%{$obj_hash_ref} = %sysplan_objects;
    return 1;
}

# Remove sysplan objects from XML repository
sub XH_delete_sysplan
{
	my $niml_inst_doc = shift;
	my %sysplan_nodes_hash;
	my $parser = XML::LibXML->new;
    my $doc_tree;

	$doc_tree = $parser->parse_file(${niml_inst_doc});

	# Validate sysplan
    eval {
        $xmlschema->validate($doc_tree);
    };

	if($@) {
		log_entry(__("delete_sysplan failed to validate: $@\n"), 1);
		return 0;
	}

    # Get XML document object
    my $sysplan_doc_root = $doc_tree->getDocumentElement;

    # Give our default namespace a prefix
    $sysplan_doc_root->setNamespace('http://www.ibm.com/niml','niml',1);

	# Save all nodes from the sysplan document into hash
	my @sysplan_nodes = $sysplan_doc_root->findnodes("/niml:osinstall_config/*");
	foreach my $sp_node (@sysplan_nodes) {
		my $key_string = $sp_node->nodeName.":".$sp_node->findvalue('@name');
		$sysplan_nodes_hash{$key_string} = 1;
	}

	# Get writer lock for repository
	if (my $lock = lock_file('xml_repos', 'exclusive', 'blocking'))
    {
		# Create empty repository document
		my $xml_repos_doc = new XML::LibXML::Document->new();
		# Make NIML root element for new repository
		my $new_root = $xml_repos_doc->createElementNS("http://www.ibm.com/niml",
					"osinstall_config");
		$new_root->setAttribute('niml_version', "1.0");
		$xml_repos_doc->setDocumentElement($new_root);

		#read in current xml repository if it exists
		if(-e $XML_REPOS) {
			my $parser = XML::LibXML->new;
			my $repos_tree = $parser->parse_file($XML_REPOS);

		    # Validate existing repository
		    eval {
		        $xmlschema->validate($repos_tree);
		    };

			if($@) {
		    	log_entry(__("XML repository failed validation: $@\n"), 1);
				return 0;
			}

		    # Get XML document object 
		    my $repos_root = $repos_tree->getDocumentElement;

		    # Give our default namespace a prefix
	   		$repos_root->setNamespace('http://www.ibm.com/niml','niml',1);

		    # Set XPath search string to return all nodes of obj_type
		    my @repos_nodes = $repos_root->findnodes("/niml:osinstall_config/*");

			# Loop through repository and save nodes unless in sysplan hash
		    foreach my $repos_node (@repos_nodes) {
				my $key_string = $repos_node->nodeName.":".$repos_node->findvalue('@name');
				unless(exists($sysplan_nodes_hash{$key_string})) {
					#Not found in sysplan, save $repos_node back to repository
					$new_root->addChild($repos_node);
				}
			}

		}
		else {
			# Repository doesn't exist
			log_entry(__("OSInstall repository has not been initialized!\n"), 1);
			close $lock; # release our write lock before we return
			return 0;
		}

		# Open XML_REPOS file in overwrite mode
		open (XML_FILE, ">$XML_REPOS") or die "$XML_REPOS : $!\n";

		# print repository object to repository file
		print XML_FILE $xml_repos_doc->toString;

		close XML_FILE;

		close $lock; # release our write lock

	} #End of lock block
	else {
		log_entry(__("Couldn't get repository lock file\n"), 1);
		return 0;
	}

	return 1;
}

# Find osinstall object specified by name/type pair in repository and return
# reference
# Parameters: type and name of object to read
sub XH_read_object
{
    my $obj_name = shift;
	my @obj_node;
	my $obj_ref;

	# Get reader lock for repository
	if (my $lock = lock_file('xml_repos', 'shared', 'blocking'))
    {
		#read in current xml repository if it exists
		if(-e $XML_REPOS) {
			my $parser = XML::LibXML->new;
			my $repos_tree = $parser->parse_file($XML_REPOS);

		    # Validate existing repository
		    eval {
		        $xmlschema->validate($repos_tree);
		    };

			if($@) {
		    	log_entry(__("XML repository failed validation: $@\n"), 1);
				return 0;
			}

		    # Get XML document object
		    my $doc_root = $repos_tree->getDocumentElement;

		    # Give our default namespace a prefix
	   		$doc_root->setNamespace('http://www.ibm.com/niml','niml',1);

			# Return the NIML node for this object
			@obj_node = $doc_root->findnodes("/niml:osinstall_config/*[\@name='$obj_name']");
		}
		else {
			# Repository doesn't exist
			log_entry(__("osinstall repository has not been initialized!\n"), 1);
		}

		close $lock; # release our read lock

	} #End of lock block
	else {
		log_entry(__("Couldn't get repository lock file\n"), 1);
		return 0;
	}

	# If we were able to read object node from repository, instantiate OSInstall
	# object and return reference
	# Will return undef if xml_to_obj fails
	if($obj_node[0]){
		# Instantiate OSInstall object
	    $obj_ref = xml_to_obj($obj_node[0]);
	}
	else {
		log_entry(__("The object $obj_name does not exist in the repository\n"), 1);
	}
    return $obj_ref;
}

sub XH_remove_object
{
	my $obj_name = shift;
	my @obj_node;
	my $repos_doc;

	# Get writer lock for repository
	if (my $lock = lock_file('xml_repos', 'exclusive', 'blocking'))
    {
		#read in current xml repository if it exists
		if(-e $XML_REPOS) {
			my $parser = XML::LibXML->new;
			$repos_doc = $parser->parse_file($XML_REPOS);

		    # Validate existing repository
		    eval {
		        $xmlschema->validate($repos_doc);
		    };

			if($@) {
		    	log_entry(__("XML repository failed validation: $@\n"), 1);
				return 0;
			}

		    # Get XML document object
		    my $repos_root = $repos_doc->getDocumentElement;

		    # Give our default namespace a prefix
	   		$repos_root->setNamespace('http://www.ibm.com/niml','niml',1);

		    # Set XPath search string to return node that matches $obj_name
			@obj_node = $repos_root->findnodes("/niml:osinstall_config/*[\@name='$obj_name']");
			if($obj_node[0]) {
				unless($repos_root->removeChild($obj_node[0])) {
					log_entry(__("Error removing object $obj_name from repository\n"), 1);
					return 0;
				}
			}
			else {
				log_entry(__("Object not found in repository: $obj_name\n"), 1);
				return 0;
			} 

		}
		else {
			# Repository doesn't exist
			log_entry(__("OSInstall repository has not been initialized!\n"), 1);
			close $lock; # release our write lock before we return
			return 0;
		}

		# Open XML_REPOS file in overwrite mode
		open (XML_FILE, ">$XML_REPOS") or die "$XML_REPOS : $!\n";

		# print repository object to repository file
		print XML_FILE $repos_doc->toString;

		close XML_FILE;

		close $lock; # release our write lock

	} #End of lock block
	else {
		log_entry(__("Couldn't get repository lock file\n"), 1);
		return 0;
	}

	# Log results of removal
	log_entry(__("Removed $obj_name from repository\n"));
	return 1;
}

# Converts each object in hash to NIML and writes to repository
# Parameter: A hash of OSInstall objects keyed by type:name
# TODO: validate XML against schema before writing to repository
sub XH_write_objects
{
	my $objects_hash_ref = shift;
	my %objects = %{$objects_hash_ref};

	# Get writer lock for repository
	if (my $lock = lock_file('xml_repos', 'exclusive', 'blocking'))
    {
		# Create empty repository document
		my $xml_repos_doc = new XML::LibXML::Document->new();
		# Make NIML root element 
		my $root = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 
					"osinstall_config");
		$root->setAttribute('niml_version', "1.0");
		$xml_repos_doc->setDocumentElement($root);

		#read in current xml repository if it exists
		if(-e $XML_REPOS) {
			my $parser = XML::LibXML->new;
			my $repos_tree = $parser->parse_file($XML_REPOS);

		    # Validate existing repository
		    eval {
		        $xmlschema->validate($repos_tree);
		    };

			if($@) {
		    	log_entry(__("XML repository failed validation: $@\n"), 1);
				return 0;
			}

		    # Get XML document object
		    my $doc_root = $repos_tree->getDocumentElement;

			# Give our default namespace a prefix
	   		$doc_root->setNamespace('http://www.ibm.com/niml','niml',1);

		    # Set XPath search string to return all nodes
		    my @nodes = $doc_root->findnodes("/niml:osinstall_config/*");

		    # Check repository for existing entry for object
			# This check is performed when object is first instantiated in create_object
			# but the repository may have been changed since then by another osinstall process
		    foreach my $niml_node (@nodes) {
				my $object_key = $niml_node->findvalue('@name');
				if($objects{$object_key}) {
					# Found object, write message to log
					log_entry(__("Object '" . $niml_node->findvalue('@name') . "' of type '" . $niml_node->nodeName . "' already exists in repository\n"), 1);
				}
				else {
					#Save $niml_node back to repository
					# TODO: validate here
					$root->addChild($niml_node);
				}
			}

		}
		else {
			# Repository doesn't exist
			log_entry(__("OSInstall repository not found, create new\n"));
		}

		# Create a NIML node for each object in hash
		foreach my $obj_key (keys (%objects)) {
			if(my $obj_ref = $objects{$obj_key}) {
				log_entry(__("Writing $obj_key to repository\n"));
				# Determine the type of node to create
				my $obj_type;
				if(ref($obj_ref) =~ /^(?:OSInstall::)?Client$/) {
					$obj_type = "client";
				}
				elsif(ref($obj_ref) =~ /^(?:OSInstall::)?\w+_Resource$/) {
					$obj_type = "os_resource";
				}
				elsif(ref($obj_ref) =~ /^(?:OSInstall::)?\w+_Remote$/) {
					$obj_type = "remote_resource";
				}
				elsif(ref($obj_ref) =~ /^(?:OSInstall::)?Control_Host$/) {
					$obj_type = "ctrl_host";
				}
				else {
					log_entry(__("Unknown object reference\n"));
				}

				my $niml_node = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", $obj_type);
				
				$niml_node->setAttribute('name', $obj_ref->{name});

				# Translate object fields to NIML elements
				if($niml_node->nodeName eq 'client') {
					# Check that required object fields are defined
					unless($obj_ref->{ip_addr} && $obj_ref->{mac_addr} && $obj_ref->{gateway} && $obj_ref->{speed} && $obj_ref->{duplex}) {
						log_entry(__("Invalid object, cannot write to repository\n"), 1);
						return 0;
					}

					# Create and add object fields to our NIML node
					# Check that optional fields exist
					my $ip_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "ip_addr");
					$ip_el->addChild(XML::LibXML::Text->new($obj_ref->{ip_addr}));
					my $mac_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "mac_addr");
					$mac_el->addChild(XML::LibXML::Text->new($obj_ref->{mac_addr}));
					my $gw_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "gateway");
					$gw_el->addChild(XML::LibXML::Text->new($obj_ref->{gateway}));
					my $speed_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "adapter_speed");
					$speed_el->addChild(XML::LibXML::Text->new($obj_ref->{speed}));
					my $duplex_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "adapter_duplex");
					$duplex_el->addChild(XML::LibXML::Text->new($obj_ref->{duplex}));

					$niml_node->addChild($ip_el);
					$niml_node->addChild($mac_el);
					$niml_node->addChild($gw_el);
					if($obj_ref->{subnet_mask}) {
						my $subnet_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "subnet_mask");
						$subnet_el->addChild(XML::LibXML::Text->new($obj_ref->{subnet_mask}));
						$niml_node->addChild($subnet_el);
					}
					$niml_node->addChild($speed_el);
					$niml_node->addChild($duplex_el);
					
					if($obj_ref->{lpar} && $obj_ref->{profile} && $obj_ref->{managed_sys} ) {
						my $partition_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "partition_info");
						$partition_el->setAttribute("lpar", $obj_ref->{lpar});
						$partition_el->setAttribute("profile", $obj_ref->{profile});
						$partition_el->setAttribute("managed_system", $obj_ref->{managed_sys});
						$niml_node->addChild($partition_el);
					}

					if($obj_ref->{target_disk}) {
						my $disk_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "disk_location");
						$disk_el->addChild(XML::LibXML::Text->new($obj_ref->{target_disk}));
						$niml_node->addChild($disk_el);
					}

					if($obj_ref->{control_host}) {
						my $ctrl_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", "ref_ctrl_host");
						$ctrl_el->setAttribute('ctrl_host_name', $obj_ref->{control_host});
						$niml_node->addChild($ctrl_el);
					}

				}
				elsif($niml_node->nodeName eq 'os_resource') {
					# Check that required object fields are defined
					unless($obj_ref->{type} && $obj_ref->{version} && $obj_ref->{ready}) {
						log_entry(__("Invalid object, cannot write to repository\n"), 1);
						return 0;
					}

					# Set the ready attribute
					$niml_node->setAttribute('ready', $obj_ref->{ready});
					

					# prepend the distribution name to the version if this is a Linux resource
					if ($obj_ref->{type} =~ /[Ll][Ii][Nn][Uu][Xx]/)
					{
						$obj_ref->version($obj_ref->{distribution}." ".$obj_ref->{version});
					}

					# Create and add required object fields to our NIML node
					my $os_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'os');
					$os_el->setAttribute('type', $obj_ref->{type});
					$os_el->setAttribute('version', $obj_ref->{version});
					$niml_node->addChild($os_el);

					my $imgsrc_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'image_source');
					if($obj_ref->{image_source}) {
						$imgsrc_el->addChild(XML::LibXML::Text->new($obj_ref->{image_source}));
						$niml_node->addChild($imgsrc_el);
					}

					# Check for optional object fields, add to node if they exist
					if($obj_ref->{image_location}) {
						my $imgloc_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'image_location');
						$imgloc_el->addChild(XML::LibXML::Text->new($obj_ref->{image_location}));
						$niml_node->addChild($imgloc_el);
					}
					if($obj_ref->{config_file}) {
						my $cfg_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'config_file');
						$cfg_el->addChild(XML::LibXML::Text->new($obj_ref->{config_file}));
						$niml_node->addChild($cfg_el);
					}
					if($obj_ref->{resource_server}) {
						my $ressrv_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'ref_resource_server');
						$ressrv_el->setAttribute('resource_server_name', $obj_ref->{resource_server});
						$niml_node->addChild($ressrv_el);
					}
				}
				elsif($niml_node->nodeName eq 'remote_resource') {
					# Check that required object fields are defined
					unless($obj_ref->{server} && $obj_ref->{type} && $obj_ref->{remote_identifier} && $obj_ref->{ready}) {
						log_entry(__("Invalid object, cannot write to repository\n"), 1);
						return 0;
					}

					# Set the ready attribute
					$niml_node->setAttribute('ready', $obj_ref->{ready});
					

					# Create and add required object fields to our NIML node
					my $server_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'server');
					$server_el->addChild(XML::LibXML::Text->new($obj_ref->{server}));
					$niml_node->addChild($server_el);

					my $type_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'type');
					$type_el->addChild(XML::LibXML::Text->new($obj_ref->{type}));
					$niml_node->addChild($type_el);

					if($obj_ref->{communication_method}) {
						my $commo_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'commo_method');
						$commo_el->addChild(XML::LibXML::Text->new($obj_ref->{communication_method}));
						$niml_node->addChild($commo_el);

					my $remoteid_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'remote_identifier');
					$remoteid_el->addChild(XML::LibXML::Text->new($obj_ref->{remote_identifier}));
					$niml_node->addChild($remoteid_el);

					}
				}
				elsif($niml_node->nodeName eq 'ctrl_host') {
					# Check that required object fields are defined
					unless($obj_ref->{host_name} && $obj_ref->{type} && $obj_ref->{communication_method}) {
						log_entry(__("Invalid object, cannot write to repository\n"), 1);
						return 0;
					}
					# Create and add required object fields to our NIML node
					my $hostname_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'hostname');
					$hostname_el->addChild(XML::LibXML::Text->new($obj_ref->{host_name}));
					my $type_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'type');
					$type_el->addChild(XML::LibXML::Text->new($obj_ref->{type}));
					my $commo_el = $xml_repos_doc->createElementNS("http://www.ibm.com/niml", 'commo_method');
					$commo_el->addChild(XML::LibXML::Text->new($obj_ref->{communication_method}));

					$niml_node->addChild($hostname_el);
					$niml_node->addChild($type_el);
					$niml_node->addChild($commo_el);
				}
				else {
					log_entry(__("Don't know how to write ".$niml_node->nodeName." object\n"), 1);
				}
				# Add this node to repository document
				# TODO: validate here
				$root->addChild($niml_node);
			}
			else {
				log_entry(__("$obj_key object does not exist in hash\n"), 1);
			}
		}
###
		# Need to sort nodes in $root to pass validation
		# Give the default namespace a prefix
	    $root->setNamespace('http://www.ibm.com/niml','niml',1);

		my @clients = $root->findnodes("/niml:osinstall_config/niml:client");
		my @os_resources = $root->findnodes("/niml:osinstall_config/niml:os_resource");
		my @remote_resource = $root->findnodes("/niml:osinstall_config/niml:remote_resource");
		my @ctrl_hosts = $root->findnodes("/niml:osinstall_config/niml:ctrl_host");

		# Need a new sorted XML document to pass validation
		# Create empty repository document
		my $xml_repos_doc_sort = new XML::LibXML::Document->new();
		my $sorted_root = $xml_repos_doc_sort->createElementNS("http://www.ibm.com/niml", "osinstall_config");
		$xml_repos_doc_sort->setDocumentElement($sorted_root);
		$sorted_root->setAttribute('niml_version', "1.0");

		my $tmp_nn;
		foreach $tmp_nn (@clients) {
			$sorted_root->addChild($tmp_nn);
		}
		foreach $tmp_nn (@os_resources) {
			$sorted_root->addChild($tmp_nn);
		}
		foreach $tmp_nn (@remote_resource) {
			$sorted_root->addChild($tmp_nn);
		}
		foreach $tmp_nn (@ctrl_hosts) {
			$sorted_root->addChild($tmp_nn);
		}

###

		# Validate existing repository
	    eval {
	        $xmlschema->validate($xml_repos_doc_sort);
	    };

		if($@) {
	    	log_entry(__("Can't add new objects to XML repository: $@\n"), 1);
			close $lock;
			return 0;
		}

		# Open XML_REPOS file in overwrite mode
		open (XML_FILE, ">$XML_REPOS") or die "$XML_REPOS : $!\n";

		# print repository object to repository file
		print XML_FILE $xml_repos_doc_sort->toString;

		close XML_FILE;

		close $lock; # release our write lock

	} #End of lock block
	else {
		log_entry(__("Couldn't get repository lock file\n"), 1);
		return 0;
	}

	return 1;
}

sub XH_write_object
{
	my $client_ref = shift;
	my %client_hash;
	# Store Client ref in hash to pass to XH_write_objects
#	my $key_str = "client:".$client_ref->name;
	$client_hash{$client_ref->name} = $client_ref;
	# Write object to XML repository
	XH_write_objects(\%client_hash);
}

sub XH_list_objects
{
	my $obj_type = shift;
	my $obj_name = shift;
	my $verbose = shift;

	unless(-e $XML_REPOS) {
		print "No objects in repository\n";
		return 0;
	}

	if (my $lock = lock_file('xml_repos', 'shared', 'blocking')) {
		my $parser = XML::LibXML->new;
		my $repos_tree = $parser->parse_file($XML_REPOS);

	    # Validate existing repository
	    eval {
	        $xmlschema->validate($repos_tree);
	    };

		if($@) {
	    	log_entry(__("XML repository failed validation: $@\n"), 1);
			return 0;
		}

	    # Get XML document object
	    my $doc_root = $repos_tree->getDocumentElement;

		# Give our default namespace a prefix
   		$doc_root->setNamespace('http://www.ibm.com/niml','niml',1);

		my @nodes;
	    # Find list of nodes bases on our list options
		if($obj_type) {
			# Find all nodes of a certain type
		    @nodes = $doc_root->findnodes("/niml:osinstall_config/niml:$obj_type");
		}
		elsif($obj_name) {
			# Find a single node by name
			@nodes = $doc_root->findnodes("/niml:osinstall_config/*[\@name='$obj_name']");
		}
		else {
			# Return all nodes
			@nodes = $doc_root->findnodes("/niml:osinstall_config/*");
		}

		# Return an error if we found nothing to list
		unless($nodes[0]) {
			if($obj_name) {
				log_entry(__("Object $obj_name not found in repository\n"), 1);
			}
			elsif($obj_type) {
				log_entry(__("No objects of type $obj_type found in repository\n"), 1);
			}
			return 0;
		}

		# Print out list
	    foreach my $niml_node (@nodes) {
			print $niml_node->findvalue('@name')."\t\t".$niml_node->nodeName."\n";
			if($verbose) {
				my $node_str = $niml_node->toString;
				print "$node_str\n";
			}
		}
		close $lock;
		return 1;
	}

}

sub XH_alloc_resource
{
    my ($res_name, $client_name, $boot_params, $config_file) = @_;

	my @client_node;
	my @res_node;

	# Get reader lock for repository
	if (my $lock = lock_file('xml_repos', 'exclusive', 'blocking'))
    {
		my $parser = XML::LibXML->new;
		my $repos_doc = $parser->parse_file($XML_REPOS);

	    # Validate existing repository
	    eval {
	        $xmlschema->validate($repos_doc);
	    };

		if($@) {
	    	log_entry(__("XML repository failed validation: $@\n"), 1);
			return 0;
		}

	    # Get XML document object
	    my $doc_root = $repos_doc->getDocumentElement;

	    # Give our default namespace a prefix
   		$doc_root->setNamespace('http://www.ibm.com/niml','niml',1);

		# Get NIML nodes for the client and resource to update below
		@client_node = $doc_root->findnodes("/niml:osinstall_config/*[\@name='$client_name']");
		@res_node = $doc_root->findnodes("/niml:osinstall_config/*[\@name='$res_name']");

		# Create/add allocation references for the client and resource nodes
		my $res_alloc_ref_el = $repos_doc->createElementNS("http://www.ibm.com/niml", 'ref_os_resource');
		$res_alloc_ref_el->setAttribute('os_resource_name', $res_name);
		$client_node[0]->addChild($res_alloc_ref_el);

		# Add parameters to the "boot" openfirmware command if we need any
 		if ($boot_params) {
			my $bootparams_el= $repos_doc->createElementNS("http://www.ibm.com/niml", "boot_params");
			$bootparams_el->addChild(XML::LibXML::Text->new($boot_params));
			$client_node[0]->addChild($bootparams_el);
		}

		# if this Client has a config file associated through an allocation, save it here
		# This is not the case for a remote_resource allocation
		if ($config_file)
		{
			my $config_file_el= $repos_doc->createElementNS("http://www.ibm.com/niml", "config_file");
			$config_file_el->addChild(XML::LibXML::Text->new($config_file));
			$client_node[0]->addChild($config_file_el);
		}

		my $client_alloc_ref_el = $repos_doc->createElementNS("http://www.ibm.com/niml", 'ref_client');
		$client_alloc_ref_el->setAttribute('client_name', $client_name);
		$res_node[0]->addChild($client_alloc_ref_el);

		# Update our repository
		# Open XML_REPOS file in overwrite mode
		open (XML_FILE, ">$XML_REPOS") or die "$XML_REPOS : $!\n";

		# print repository object to repository file
		print XML_FILE $repos_doc->toString;

		close XML_FILE;

		close $lock; # release our write lock

	} #End of lock block
	else {
		log_entry(__("Couldn't get repository lock file\n"), 1);
		return 0;
	}

	return 1;
}

sub XH_deallocate
{
	my ($client_name, $res_name) = @_;

	# Get reader lock for repository
	if (my $lock = lock_file('xml_repos', 'exclusive', 'blocking'))
    {
		my $parser = XML::LibXML->new;
		my $repos_doc = $parser->parse_file($XML_REPOS);

	    # Validate existing repository
	    eval {
	        $xmlschema->validate($repos_doc);
	    };

		if($@) {
	    	log_entry(__("XML repository failed validation: $@\n"), 1);
			return 0;
		}

	    # Get XML document object
	    my $doc_root = $repos_doc->getDocumentElement;

	    # Give our default namespace a prefix
   		$doc_root->setNamespace('http://www.ibm.com/niml','niml',1);

		my (@client_node, @res_node, @res_ref_node, @client_ref_node, @boot_params_node, @config_file_node);

		# Get client node and res_ref child 
		@client_node = $doc_root->findnodes("/niml:osinstall_config/niml:client[\@name='$client_name']");
		@res_ref_node = $doc_root->findnodes("/niml:osinstall_config/niml:client[\@name='$client_name']/niml:ref_os_resource");
		@boot_params_node= $doc_root->findnodes("/niml:osinstall_config/niml:client[\@name='$client_name']/niml:boot_params");
		@config_file_node= $doc_root->findnodes("/niml:osinstall_config/niml:client[\@name='$client_name']/niml:config_file");

		# Get resource node and client_ref child
		@res_node = $doc_root->findnodes("/niml:osinstall_config/*[\@name='$res_name']");
		@client_ref_node = $doc_root->findnodes("/niml:osinstall_config/*[\@name='$res_name']/niml:ref_client[\@client_name='$client_name']");

		# Remove allocation references for the client and resource nodes
		$client_node[0]->removeChild($res_ref_node[0]);
		$res_node[0]->removeChild($client_ref_node[0]);

		# remove boot params attribute
		$client_node[0]->removeChild($boot_params_node[0]) if (@boot_params_node);

		# remove the config file attribute
		$client_node[0]->removeChild($config_file_node[0]) if (@config_file_node);

		# Update our repository
		# Open XML_REPOS file in overwrite mode
		open (XML_FILE, ">$XML_REPOS") or die "$XML_REPOS : $!\n";

		# print repository object to repository file
		print XML_FILE $repos_doc->toString;

		close XML_FILE;

		close $lock; # release our write lock

	} #End of lock block
	else {
		log_entry(__("Couldn't get repository lock file\n"), 1);
		return 0;
	}

	return 1;

}

# create_object($obj_data)
# Check OSInstall repository for existing object of same type/name, return 0 if found
# Attempt to instantiate the object, return a reference to object if we can
#
# Parameters 
# $obj_data: OSInstall object descriptor in either NIML format or a hash of 
# attributes that describes the object
#

sub create_object
{

    my $obj_data = shift;
	my $lock;
	my $repos_tree;

    if(ref($obj_data) eq "HASH") {
		#convert hash to XML object to write to repository
		$obj_data = hash_to_xml($obj_data); 
    }

    # Add this object to osinstall repository
	# get xml_repos lock
    if ($lock = lock_file('xml_repos', 'shared', 'blocking'))
    {
		#read in current xml repository if it exists
		if(-e $XML_REPOS) {
			my $parser = XML::LibXML->new;
			$repos_tree = $parser->parse_file($XML_REPOS);

		    # Validate existing repository
		    eval {
		        $xmlschema->validate($repos_tree);
		    };

			if($@) {
		    	log_entry(__("XML repository failed validation: $@\n"), 1);
				return 0;
			}

		    # Get XML document object
		    my $doc_root = $repos_tree->getDocumentElement;

		    # Give our default namespace a prefix
	   		$doc_root->setNamespace('http://www.ibm.com/niml','niml',1);

		    # Set XPath search string to return all nodes
		    my @nodes = $doc_root->findnodes("/niml:osinstall_config/*");
 
		    # Check for existing object of type/name in repository, return error if found
		    foreach my $niml_node (@nodes) {
				if($niml_node->findvalue('@name') eq 
					   $obj_data->findvalue('@name')) {
					log_entry(__("Object '" . $niml_node->findvalue('@name') . "' of type '" . $niml_node->nodeName . "' already exists in repository\n"), 1);
					close $lock;
					return 0;
				}
			}

		}
		close $lock; # release our write lock
		# Object is not currently defined in repository, attempt to instantiate it 
		return xml_to_obj($obj_data);

	} #End of lock block
	else {
		log_entry(__("Could not get repository lock\n"), 1);
		return 0;
	}
 
}

# xml_to_obj
# Private subroutine to parse a single NIML node and instantiate an OSInstall
# object, returning the reference.
# XML should have been previously validated against the NIML schema
sub xml_to_obj
{
    my $xml_obj = shift;
    my $obj_ref;

    if($xml_obj->nodeName eq "client")
    { 
		my $name =     $xml_obj->findvalue('@name');
        my $ip_addr =  $xml_obj->findvalue('niml:ip_addr');
        my $gateway =  $xml_obj->findvalue('niml:gateway');
        my $mac_addr = $xml_obj->findvalue('niml:mac_addr');
        my $subnet   = $xml_obj->findvalue('niml:subnet_mask');
        my $speed =    $xml_obj->findvalue('niml:adapter_speed');
        my $duplex =   $xml_obj->findvalue('niml:adapter_duplex');
        my $lpar =     $xml_obj->findvalue('niml:partition_info/@lpar');
        my $profile =  $xml_obj->findvalue('niml:partition_info/@profile');
        my $mng_sys =  $xml_obj->findvalue('niml:partition_info/@managed_system');
	my $bootparams=$xml_obj->findvalue('niml:boot_params');
	my $cfg_file = $xml_obj->findvalue('niml:config_file');
        my $disks_array_ref= $xml_obj->findvalue('niml:disk_location');
        my $ctrl_host_ref =   $xml_obj->findvalue('niml:ref_ctrl_host/@ctrl_host_name');
		my $resource_allocated = $xml_obj->findvalue('niml:ref_os_resource/@os_resource_name');

        $obj_ref = OSInstall::Client->new($name, $ip_addr, $gateway, $mac_addr, $speed, $duplex, $subnet, $disks_array_ref, $ctrl_host_ref, $lpar, $profile, $mng_sys, $resource_allocated);
		unless($obj_ref) {
			log_entry(__("Unable to instantiate Client object\n"), 1);
		}
	$obj_ref->boot_params($bootparams);
	$obj_ref->config_file($cfg_file);
    }

    elsif($xml_obj->nodeName eq "os_resource")
    {
		my $name =         $xml_obj->findvalue('@name');
        my $type =         $xml_obj->findvalue('niml:os/@type');
        my $version =      $xml_obj->findvalue('niml:os/@version');
		my $image_src =    $xml_obj->findvalue('niml:image_source');
        my $image_loc =    $xml_obj->findvalue('niml:image_location');
        my $config_file =  $xml_obj->findvalue('niml:config_file');
        my $res_server_ref =   $xml_obj->findvalue('niml:ref_resource_server/@resource_server_name');
		my @client_ref_nodes = $xml_obj->findnodes('niml:ref_client');

		my $ready_flag = $xml_obj->findvalue('@ready');
		my %allocated_clients;

		foreach my $node (@client_ref_nodes) {
			$allocated_clients{$node->findvalue('@client_name')} = $node->findvalue('@client_name');
		}

		# extract the distribution from the version string
		my $distro_str;
		if($version =~ s/^(\S+)\s+(\S+\s+\S+\s*)$/$2/) {
			$distro_str = $1;
		}

		# Determine type of OS_Resource to instantiate
		my $inst_type; #prefix to '_Resource' for class name
		if ( $type eq 'VIOS' ) {
			$inst_type = 'AIX';
		}
		else {
			$inst_type = $type;
		}

		my $constructor_str = "OSInstall::${inst_type}_Resource->new(\$name, \$image_src, \$type, \$version, \$image_loc, \$config_file, \$res_server_ref, \$distro_str, \\\%allocated_clients, \$ready_flag)";
			$obj_ref = eval $constructor_str;
			if($@) {print "ERROR--".$@;}
			unless($obj_ref) {
				log_entry(__("Unable to instantiate ${inst_type}_Resource object\n"), 1);
			}
    }

    elsif($xml_obj->nodeName eq "remote_resource")
    {
		my $name =        $xml_obj->findvalue('@name');
	my $server =      $xml_obj->findvalue('niml:server');
	my $type =        $xml_obj->findvalue('niml:type');
	my $remoteid =    $xml_obj->findvalue('niml:remote_identifier');
	my $commo =       $xml_obj->findvalue('niml:commo_method');
		my @client_ref_nodes = $xml_obj->findnodes('niml:ref_client');

		my $ready_flag = $xml_obj->findvalue('@ready');
		my %allocated_clients;

		foreach my $node (@client_ref_nodes) {
			$allocated_clients{$node->findvalue('@client_name')} = $node->findvalue('@client_name');
		}

		my $constructor_str = "OSInstall::${type}_Remote->define(\$name, \$server, \$type, \$remoteid, \$commo, \\\%allocated_clients, \$ready_flag)";
			$obj_ref = eval $constructor_str;
			if($@) {print "ERROR--".$@;}
			unless($obj_ref) {
				log_entry(__("Unable to instantiate ${type}_Remote object\n"), 1);
			}
    }

	elsif($xml_obj->nodeName eq "ctrl_host") {
		my $name =     $xml_obj->findvalue('@name');
		my $hostname = $xml_obj->findvalue('niml:hostname'); 
		my $type =     $xml_obj->findvalue('niml:type');
		my $commo =    $xml_obj->findvalue('niml:commo_method');

		$obj_ref = OSInstall::Control_Host->new($name, $hostname, $type, $commo);
		unless($obj_ref) {
			log_entry(__("Unable to instantiate Control_Host object\n"), 1);
		}
	}

    else  { 
		print "Unknown object:".$xml_obj->nodeName."\n"; 
    }
     
    return $obj_ref; 
}


1;
__END__
