#!/bin/bash 
#CONFIDENTIAL
#
# LICENSED INTERNAL CODE SOURCE MATERIALS
#
# TS4500 LICENSED INTERNAL CODE
#
# (C) COPYRIGHT IBM CORP. 2014
#
# THE SOURCE CODE FOR THIS PROGRAM IS NOT PUBLISHED OR
# OTHERWISE DIVESTED OF ITS TRADE SECRETS, IRRESPECTIVE
# OF WHAT HAS BEEN DEPOSITED WITH THE U.S. COPYRIGHT OFFICE.
#
################################################################################

################################################################################
#
# Filename: library_backup.sh
#
# DESCRIPTION: 
#    This utility is called by the primary LCC to back up all important library 
#    data (i.e. all databases and certificates).  
#    This script and the dbm_restore.sh are be bundled with CLI tool.
#    The final back up file name is: TS4500_Backup_LibSerial_TimeStamp.dbz"
#    and it is stored in /mnt/scratch/ directory
################################################################################

################################################################################
#
# CHANGE ACTIVITY:
#
#
# MM/DD/YYYY INI    CQ ID      Change Description
# ---------- ---   --------   --------------------------------------------------
# 08/12/2014 kqiu              Creation 
# 09/04/2014 kqiu              Added -l option for scp to limit bandwidth
################################################################################
VERSION=2.0
############################################################################
# DEFINED CONSTANTS HERE
############################################################################
num_of_cmd_param=$#
_DEBUG="ON"  # Internal debug flag
USER_PARAMS="$@"

DB_DIR="/mnt/persist/db"

CAL_DB="calibration.db"
CONFIG_DB="configuration.db"
INV_DB="inventory.db"
MISC_DB="misc.db"
STAT_DB="statistic.db"

SCRATCH_DIR="/mnt/scratch"
LIB_BACKUP_DIR="$SCRATCH_DIR"/backup
LIB_BACKUP_PRE_FIX="TS4500"
LIB_BACKUP_EXT="dbz"

FWNODE_TABLE_NAME="$SCRATCH_DIR"/FWNodeTable.csv
LCC_STATE_FILE="$SCRATCH_DIR"/LCC_State.csv

DB_BACK_TOOL="/fw/tools/db_backup"

LOG_FILE="/var/log/fw/lib_backup.log"
# If the current log file size is greater than 100KB,
# we will back up the current one and open a new file. 
# We only keep 2 copies of log files, which should be more than enough.
LOG_FILE_SIZE_LIMIT=102400
NUM_OF_LOG_FILES=1   # This will keep up to 2 back up log files

TimeStamp=$(date '+%Y%m%d%H%M%S')

TimeStamp_Nice=$(date '+%Y-%m-%d %H:%M:%S')
BackupSignature="TS4500 Backup File"
LibSerialNum=""
LibName=""
Lib_FW_Version=""
TempBackupFile="backup.tgz"
LIB_BACKUP_FILE=""

THIS_FILE=/fw/tools/$(basename $0)

OTHER_LCC_IPS=$(cat $FWNODE_TABLE_NAME | cut -d ',' -f2)

LOCAL_PRIMARY=""
PRIMARY_LCC_IP=""
PRIMARY_LCC_NODE=""

###############################################################################
# DEFINE FUNCTIONS HERE
###############################################################################

#------------------------------------------------------------------------------
# print_usage:
#   Prints the usage of this script.
#------------------------------------------------------------------------------
print_usage() 
{
   echo "Current Version: $VERSION"
   echo "Usage: $0 "
   echo "This script backs up all important library files."
   echo "OPTIONS:"
   echo "   -h                 : Show this message"
   echo

}

###############################################################################
# DEFINE FUNCTIONS HERE
###############################################################################
DEBUG()
{
   # If _Debug flag is turned on, run the commands in $@ (i.e. ehco "hello")
   if [ "$_DEBUG" == "ON" ]; then
      echo $(date +%m-%d-%y" "%H:%M:%S) $@ >> $LOG_FILE 
   fi
}

RUN_CMD()
{
   # This function will 
   # 1. Log the current command being executed with time stamp
   # 2. Execute the shell command passed in and all the stdout and stderr will
   #    be redirected to the log file $LOG_FILE
   ########################################################################
   # WARNING: This function cannot be used to run cascaded commands or
   #          any command that needs to be redirected.
   #          i.e. cat /proc/mtd | grep A-Boot
   #          i.e. cat /dev/mtd1 > outputfile
   ########################################################################
   echo $(date +%m-%d-%y" "%H:%M:%S) "Running command: " $@ >> $LOG_FILE
   $@ >> $LOG_FILE 2>&1
}

#-----------------------------------------------------------------------------
# back_up_log_file
#    This function keeps up to NUM_OF_LOG_FILES copies of the drivedump.log 
#-----------------------------------------------------------------------------

back_up_log_file()
{
   local currentLogFileSize=""

	if [ ! -f $LOG_FILE ]; then
		touch $LOG_FILE
	else
		currentLogFileSize=$(stat -c%s $LOG_FILE)
		# echo currentLogFileSize = $currentLogFileSize
		# If the current log file size is greater than the limit,
		# rotate it out. Otherwise, keep using the current log file
		if [ "$currentLogFileSize" -gt "$LOG_FILE_SIZE_LIMIT" ]; then
			for ((i=$NUM_OF_LOG_FILES; i>=1 ; i--))
			do
				if [ -f $LOG_FILE$i ]; then
					mv $LOG_FILE$i  $LOG_FILE$((i+1))
				fi

			done
			if [ -f $LOG_FILE ]; then
				mv "$LOG_FILE" "$LOG_FILE"1
			fi

		fi
	fi

	echo "--------------------------------------------------------" >> $LOG_FILE
	echo "Start running $0 $USER_PARAMS" >> $LOG_FILE

}

Prepare_Backup()
{
   DEBUG "Preparing library back up ..."
   #Remove the back up directory first
	RUN_CMD rm -rf $LIB_BACKUP_DIR
   #Create a new directory
	RUN_CMD mkdir $LIB_BACKUP_DIR
   # Remove any old back up files 
   RUN_CMD rm -f "$SCRATCH_DIR"/"$LIB_BACKUP_PRE_FIX"*."$LIB_BACKUP_EXT"
}

Get_Primary_LCC_Info()
{
	local nodeID=""
	local lccState=""
	DEBUG "Getting primary LCC info ..."

	#Check to see if this local LCC is the primary first before we start searching for remote LCCs
   lccState=$(cat $LCC_STATE_FILE)
	if [[ $lccState =~ ^"A" ]]; then  # If lccState start with A, it is a primary
		DEBUG "This local LCC is the primary ($lccState)"
		LOCAL_PRIMARY="TRUE"
	else
      DEBUG "This local LCC is NOT the primary ($lccState)"
		LOCAL_PRIMARY="FALSE"
		# Start copying to other LCC
		for thisIP in $OTHER_LCC_IPS
		do
			nodeID=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f1)
			Interface=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f3)
			if [ -z "$Interface" ]; then   # If no interface is found, default to interface eth0.103
				Interface="eth0.103"
			fi
			lccState=$(ssh -6 -o ConnectTimeOut=2 root@$thisIP%$Interface "cat $LCC_STATE_FILE")
			if [[ $lccState =~ ^"A" ]]; then  # If lccState start with A, it is a primary
				DEBUG "LCC state ($lccState) in node $nodeID at $thisIP is primary"
				DEBUG "Primary LCC found, stop searching ..."
				PRIMARY_LCC_IP="$thisIP"
				PRIMARY_LCC_NODE="$nodeID"
				break
			else
				DEBUG "LCC state ($lccState) in node $nodeID at $thisIP is cache, continue searching ..."
			fi
			rc=$?
			if [ $rc != "0" ]; then
				DEBUG "Failed to find primary at node $nodeID via $thisIP%$Interface"
				# Don't exit here, continue to copy to other LCC
			fi
		done
	fi

}
##########################################################################
# Backup_DBs
# The function backs up all the DBs
##########################################################################
Backup_DBs()
{

   mkdir -p "$LIB_BACKUP_DIR"/db

   DEBUG "Backing up $CAL_DB ..."
   RUN_CMD "$DB_BACK_TOOL" -s "$DB_DIR"/"$CAL_DB" -t "$LIB_BACKUP_DIR"/db/"$CAL_DB"
   rc=$?
   if [ $rc != "0" ]; then
	   DEBUG "Failed to back up $CAL_DB to $LIB_BACKUP_DIR"
      exit 1
   else
      DEBUG "Successfully backed up $CAL_DB to $LIB_BACKUP_DIR"
   fi

   DEBUG "Backing up $CONFIG_DB ..."
   RUN_CMD "$DB_BACK_TOOL" -s "$DB_DIR"/"$CONFIG_DB" -t "$LIB_BACKUP_DIR"/db/"$CONFIG_DB"
   rc=$?
   if [ $rc != "0" ]; then
	   DEBUG "Failed to back up $CONFIG_DB to $LIB_BACKUP_DIR"
      exit 1
   else
      DEBUG "Successfully backed up $CONFIG_DB to $LIB_BACKUP_DIR"
   fi

   DEBUG "Backing up $INV_DB ..."
   RUN_CMD "$DB_BACK_TOOL" -s "$DB_DIR"/"$INV_DB" -t "$LIB_BACKUP_DIR"/db/"$INV_DB"
   rc=$?
   if [ $rc != "0" ]; then
	   DEBUG "Failed to back up $INV_DB to $LIB_BACKUP_DIR"
      exit 1
   else
      DEBUG "Successfully backed up $INV_DB to $LIB_BACKUP_DIR"
   fi

   DEBUG "Backing up $MISC_DB ..."
   RUN_CMD "$DB_BACK_TOOL" -s "$DB_DIR"/"$MISC_DB" -t "$LIB_BACKUP_DIR"/db/"$MISC_DB"
   rc=$?
   if [ $rc != "0" ]; then
	   DEBUG "Failed to back up $MISC_DB to $LIB_BACKUP_DIR"
      exit 1
   else
      DEBUG "Successfully backed up $MISC_DB to $LIB_BACKUP_DIR"
   fi

   DEBUG "Backing up $STAT_DB ..."
   RUN_CMD "$DB_BACK_TOOL" -s "$DB_DIR"/"$STAT_DB" -t "$LIB_BACKUP_DIR"/db/"$STAT_DB"
   rc=$?
   if [ $rc != "0" ]; then
	   DEBUG "Failed to back up $STAT_DB to $LIB_BACKUP_DIR"
      exit 1
   else
      DEBUG "Successfully backed up $STAT_DB to $LIB_BACKUP_DIR"
   fi

}

##########################################################################
# Backup_Certificates
# The function backs up all the certificates
#
##########################################################################
Backup_Certificates()
{
   DEBUG "Backing up certificates ..."
   cp -r /mnt/certificates/  $LIB_BACKUP_DIR
}

##########################################################################
# Print_Header_Info
# The function prints the back up header file info to a tmp file first.
#
##########################################################################
Print_Header_Info()
{
   DEBUG "Printing header info ..."

   LibSerialNum=$(sqlite3 "$DB_DIR"/"$CONFIG_DB" "SELECT SerialNumber FROM FRAME_CONFIG WHERE FrameIdx = (SELECT L_Frame FROM LIBRARY_CONFIG);")
   LibName=$(sqlite3 "$DB_DIR"/"$CONFIG_DB" "SELECT SysName FROM RAS_DATA;")
   Lib_FW_Version=$(/fw/tools/fw_version)

   LIB_BACKUP_FILE="$LIB_BACKUP_PRE_FIX"_"$LibSerialNum"_"$TimeStamp"."$LIB_BACKUP_EXT"

   DEBUG "Signature=$BackupSignature"
   echo "Signature=$BackupSignature" > $SCRATCH_DIR/$LIB_BACKUP_FILE

   DEBUG "BackupVersion=$VERSION"
   echo "BackupVersion=$VERSION" >> $SCRATCH_DIR/$LIB_BACKUP_FILE
 
   DEBUG "FirmwareVersion=$Lib_FW_Version"
   echo "FirmwareVersion=$Lib_FW_Version" >> $SCRATCH_DIR/$LIB_BACKUP_FILE

   DEBUG "LibrarySerialNumber=$LibSerialNum"
   echo "LibrarySerialNumber=$LibSerialNum" >> $SCRATCH_DIR/$LIB_BACKUP_FILE

   DEBUG "LibraryName=$LibName"
   echo "LibraryName=$LibName" >> $SCRATCH_DIR/$LIB_BACKUP_FILE

   DEBUG "TimeStamp=$TimeStamp_Nice"
   echo "TimeStamp=$TimeStamp_Nice" >> $SCRATCH_DIR/$LIB_BACKUP_FILE

   echo "" >> $SCRATCH_DIR/$LIB_BACKUP_FILE
  
}

##########################################################################
# Finalize_backup
# The function finalizes the library back up by taring up all the file
# and protected with a key and finally append the header file to the tarball
# At this point, assume all the files we need to back up are stored in
# LIB_BACKUP_DIR.
##########################################################################
Finalize_backup()
{
   DEBUG "Finalizing library back up ..."
   #Generate a hash key based on the timestamp to protect the tarball
   backup_key=$(echo $TimeStamp_Nice | shasum | cut -f1 -d ' ')
	pushd $LIB_BACKUP_DIR
   tar -zcvf - . | openssl des3 -salt -k $backup_key | dd of=$SCRATCH_DIR/$TempBackupFile
	rc=$?
   popd
	if [ $rc != "0" ]; then
		DEBUG "Failed to tar up back up file"
   else
		DEBUG "Adding header file to tarball ..."
		#Append the tarball to the header file
		cat $SCRATCH_DIR/$TempBackupFile  >> $SCRATCH_DIR/$LIB_BACKUP_FILE
		rc=$?
		if [ $rc != "0" ]; then
			DEBUG "Failed to create back up file"
		else
			DEBUG "Successfully created back up file: $SCRATCH_DIR/$LIB_BACKUP_FILE"
		fi
   fi

   # Do some clean up first
   rm $SCRATCH_DIR/$TempBackupFile
   rm -rf $LIB_BACKUP_DIR

   if [ $rc != "0" ]; then
     exit 2
   fi
  
}

##########################################################################
# Distribute_backup_script
# Before we start back up library files, we need to distribute this tool
# to all other LCC to make sure the same version of tool is in all LCC.
# Only the primary LCC will perform the library back up
#
##########################################################################
Distribute_backup_script()
{
	local nodeID=""

	# Start copying to other LCC
	for thisIP in $OTHER_LCC_IPS
	do
		nodeID=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f1)
		Interface=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f3)
		if [ -z "$Interface" ]; then   # If no interface is found, default to interface eth0.103
		   Interface="eth0.103"
		fi
		DEBUG "Copying $THIS_FILE to node $nodeID"
		scp -6 -l 14588 -o ConnectTimeOut=2 "$THIS_FILE" root@[$thisIP%$Interface]:"/fw/tools/"
		rc=$?
		if [ $rc != "0" ]; then
			DEBUG "Failed to scp $THIS_FILE to node $nodeID via $thisIP%$Interface"
			# Don't exit here, continue to copy to other LCC
		fi
	done

}

##########################################################################
# Execute_backup_script
# This function execute the back up script.  If there is only one LCC
# in the library, it executes locally.  If there are multiple LCC in the 
# library, it will first check to see if the local LCC is the primary.
# If it is the primary, it will run it on local then exit.
# Otherwise, keep finding the primary LCC to execute the script.
#
##########################################################################
Execute_backup_script()
{
	local Interface=""

   # only need to do the following from the local LCC that initiates the library back up
	if [ "$remoteOnly" != "TRUE" ]; then
		# Get the primary info first
      Get_Primary_LCC_Info
		if [ -z $OTHER_LCC_IPS ]; then 
			DEBUG "Only one LCC, running back up locally ..."  
			run_local=TRUE
		elif [ $LOCAL_PRIMARY == "TRUE" ]; then
			DEBUG "This local LCC is the primary, running back up locally ..."  
			run_local=TRUE
		else
			DEBUG "Multiple LCCs found, start distributing the back up script to other LCC ..."
			run_local=FALSE
			Distribute_backup_script
			Interface=$(grep "$PRIMARY_LCC_IP" $FWNODE_TABLE_NAME | cut -d  ',' -f3)
			DEBUG "Requesting the primary LCC $PRIMARY_LCC_NODE via $PRIMARY_LCC_IP%$Interface to perform the backup ..."
			
			RUN_CMD ssh -6 -o ConnectTimeOut=2 root@$PRIMARY_LCC_IP%$Interface "$THIS_FILE -r"
			rc=$?
			if [ $rc != 0 ]; then
				DEBUG "Failed to back up library files remotely in node $PRIMARY_LCC_NODE"
				exit 3
			else
				DEBUG "Successfully backed up library files remotely in node $PRIMARY_LCC_NODE"
				# Remove any old back up files 
				RUN_CMD rm -f "$SCRATCH_DIR"/"$LIB_BACKUP_PRE_FIX"*."$LIB_BACKUP_EXT"

				DEBUG "Copying the file back to local LCC ..."
				RUN_CMD scp -6 -l 14588 -o ConnectTimeOut=2 root@[$PRIMARY_LCC_IP%$Interface]:"$SCRATCH_DIR"/*."$LIB_BACKUP_EXT" $SCRATCH_DIR
				rc=$?
				if [ $rc != 0 ]; then
					DEBUG "Failed to transfer $SCRATCH_DIR"/*"$LIB_BACKUP_EXT to local LCC"
					exit 4
				else
					DEBUG "Successfully transferred $SCRATCH_DIR"/*"$LIB_BACKUP_EXT to local LCC"
					exit 0
				fi
			fi 
		fi
   fi

   # If we get to this far, we need to back up the files
   if [ "$run_local" == "TRUE" ] || [ "$remoteOnly" == "TRUE" ]; then
		Prepare_Backup
		Backup_DBs
		Backup_Certificates
      Print_Header_Info 
		Finalize_backup
   fi
}
###########################################################################
# Start of main
###########################################################################
echo "Running $0"
back_up_log_file

while getopts "hr" OPTION
do
   case $OPTION in
      h)
        print_usage
        exit 0
        ;;
      r)
        remoteOnly="TRUE"  # This flag is used internally by the script
        ;;
      ?)
        print_usage
        exit 1
        ;;
   esac

done

Execute_backup_script

#if we get this far, we are good
exit 0





 







