#!/bin/bash

##############################################################################
#                                                                           
# IBM CONFIDENTIAL                                                          
#                                                                           
# LICENSED INTERNAL CODE SOURCE MATERIALS                                   
#                                                                           
# ANACONDA LICENSED INTERNAL CODE                                           
#                                                                           
# (C) COPYRIGHT IBM CORP. 2012                                  
#                                                                           
# THE SOURCE CODE FOR THIS PROGRAM IS NOT PUBLISHED OR                      
# OTHERWISE DIVESTED OF ITS TRADE SECRETS, IRRESPECTIVE                     
# OF WHAT HAS BEEN DEPOSITED WITH THE U.S. COPYRIGHT OFFICE.                
#                                                                           
##############################################################################
# Purpose: This script is used to migrate live DB to the latest DB level 
#          according to the latest blank DBs stored in /fw/db/blankDBs 
#          before the library firmware processes are started.
#
# 
# CHANGE ACTIVITY:
#
#
# MM/DD/YY INI   CQID     Change Description
# -------- ---- --------  -------------------------------------------------------
# 10/24/12 DAG            Initial creation      
# 04/17/13 kqiu  S1044872 LCC4 and MCP6.1 integration
# 05/06/13 DAG   S1046884 Data from new tables and columns now imported with 
#                         default values set in dbcreate.sh.
# 09/16/13 DAG            Fixed issue where columns added in alphabetical order
#                         rather then in order shown in schema.
# 10/11/13 kqiu  S1046884 Added code to back up current DB before any migration
#                         Added code to support DB clean up
# 10/21/13 kqiu  S1046884 Added code to restore Ethernet config data
# 12/01/13 kqiu  S1050924 2nd phase DB clean up
# 04/01/14 kqiu  S1053410 Added support to reset mfg default settings
# 04/24/14 kqiu  S1053410 Updated DB_WIPE_OUT to preserve statistic data
# 04/28/14 kqiu  S1055423 Updated DB_WIPE_OUT to preserve WWNN
# 04/29/14 kqiu  S1055449 Added support to clear statistic DB
# 05/01/14 kqiu  S1055498 Fixed db corruption problem after DB wipeout
# 05/27/14 kqiu  S1053722 Fixed a DB restore problem
# 08/14/14 kqiu           Added library restore support
# 03/25/15 kqiu           Improved logging
# 07/23/15 kqiu  S1059170 Removed unnecessary record update after alter table
#                         to add a new column.
# 09/10/15 kqiu  S1060314 Added more robust check in DB version check to make sure
#                         DB_INFO exists before checking the DB version
###############################################################################

###############################################################################
# COMMON VARIABLES ACROSS SCRIPT
###############################################################################
VERSION=1.8

#Directory locations on card
DEFAULT_CUSTOMER_DB_PATH='/mnt/persist/db'
DEFAULT_DB_FROM_FW_IMAGE_PATH='/fw/db/blankDBs'
BACKUP_DB_DIR='/mnt/persist/db.bak'

LIB_RESTORE_DIR="/mnt/persist/restore"


LOG_FILE="/var/log/fw/db_migration.log"

TMP_FILE="/mnt/scratch/table_insertions_temp.sql"
TMP_DATA_FILE="/mnt/scratch/table_insertions_temp_data.txt"


BACKUP_ENET_DATA="/mnt/scratch/enet_data.dmp"

# PENDING_RESTORE_SCRIPT is a script creating by library_restore.sh as 
# part of library restore process.
PENDING_RESTORE_SCRIPT=/mnt/persist/pending_lib_restore.sh
PENDING_MFG_RESET_FILE="pending_db_mfg_reset"
PENDING_DB_WIPEOUT_FILE="pending_db_wipeout"
PENDING_DB_CLR_STAT_FILE="pending_clear_statistic"


timeInitial=$(date +%s.%N)

# As part of DB clean up, if live DB and blank DB version is different 
# and the blank DB version is the following version, the live DB will 
# be replaced by the blank DB.
DB_CLEAN_UP_VERSION="0.41"  

###############################################################################
# FUNCTIONS
###############################################################################
echo "Start running version $VERSION of $0 ..."

#-----------------------------------------------------------------------------
# Bail_On_Error:
#    This function exits the script when encounters an error
#-----------------------------------------------------------------------------
Bail_On_Error()
{
   local rc="$1"
   local msg="$2"

   if [ $rc -ne "0" ]; then
      echo "$0 failed with $msg"
	   echo "$0 failed with $msg" >> $LOG_FILE
      exit 1
   fi
}

#-----------------------------------------------------------------------------
# Compare_db_version:
#    Compares the db version stored in PRAGMA user_version;
# Return Code
# 1: Means DB needs to be upgrade
# 2. Means DB needs to replaced
#-----------------------------------------------------------------------------
Compare_db_version()
{
	local dataBase=$1
	local blank_db_version=""
	local live_db_version=""
	local output=""

	#If a DB doesn't exist, sqlite will create a blank DB, we don't want that, so we need to check if the live DB exist. 
	if [ -f $DEFAULT_CUSTOMER_DB_PATH/$dataBase ]; then
		blank_db_version=$(sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase "SELECT Version from DB_INFO")
		blankDBRC=$?
		live_db_version=$(sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase "SELECT Version from DB_INFO")
		liveDBRC=$?

		#If both DB version tables are valid, then compare the DB version
		if [[ "$blankDBRC" == "0" && "$liveDBRC" == "0" ]]; then
			echo "$DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase version = $blank_db_version" >> "$LOG_FILE"
			echo "$DEFAULT_CUSTOMER_DB_PATH/$dataBase version = $live_db_version" >> "$LOG_FILE"

			# IF versions match no update is needed
			# If live_db_version < blank_db_version update to fw_version
			# IF live_db_version > blank_db_version do nothing
			if (( $(echo "$live_db_version $blank_db_version" | awk '{print ($1 < $2)}') )); then
				echo "Newer database is found, update is needed"  >> "$LOG_FILE"
				return 1
			else
				echo "No update is needed"  >> "$LOG_FILE"
				return 0
			fi	
		else
			echo "$DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase version = $blank_db_version" >> "$LOG_FILE"
			echo "$DEFAULT_CUSTOMER_DB_PATH/$dataBase version = $live_db_version" >> "$LOG_FILE"
			if [ "$blankDBRC" != "0" ]; then
				# The blank DB is invalid, do not migrate				
				echo "$DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase is NOT valid! rc = $blankDBRC" >> "$LOG_FILE"
				sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase "SELECT Version from DB_INFO" >> "$LOG_FILE"	2>&1
				# Print out the DB schema to help figure out what the problem is
				sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase ".schema" >> "$LOG_FILE"	2>&1
				return 0
			elif [ "$liveDBRC" != "0" ]; then
				# The live DB is invalid, need to figure out what the return code is
				echo "$DEFAULT_CUSTOMER_DB_PATH/$dataBase is NOT valid! rc = $liveDBRC" >> "$LOG_FILE"				
				sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase "SELECT Version from DB_INFO" >> "$LOG_FILE"	2>&1
				# Print out the DB schema to help figure out what the problem is
				sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase ".schema" >> "$LOG_FILE"	2>&1
				if [[ "$liveDBRC" == "11" || "$liveDBRC" == "14" || "$liveDBRC" == "16" || "$liveDBRC" == "24" || "$liveDBRC" == "26" ]]; then  
					# RC: 11 = The database disk image is malformed,  most likely DB is corrupted
					# RC: 14 = Unable to open the database file
					# RC: 16 = Database is empty
					# RC: 24 = Auxiliary database format error
					# RC: 26 = File opened that is not a database file, most likely DB is corrupted
					echo "$DEFAULT_CUSTOMER_DB_PATH/$dataBase is corrupted" >> "$LOG_FILE"
					logger "$DEFAULT_CUSTOMER_DB_PATH/$dataBase is corrupted!"
					if [[ "$dataBase" == "statistic.db" ]]; then
						echo "$DEFAULT_CUSTOMER_DB_PATH/$dataBase will be cleaned up" >> "$LOG_FILE"
					   return 2
					else
						echo "$DEFAULT_CUSTOMER_DB_PATH/$dataBase will NOT be cleaned up" >> "$LOG_FILE"
						return 0
					fi
				elif [ "$liveDBRC" == "1" ]; then 
					# 1 = SQL error or missing database (i.e. missing the DB_INFO table)
					return 1
				else  # For other errors, we dont want to mess around with the DB
					return 0
				fi

			fi
		fi
	else
		echo "$DEFAULT_CUSTOMER_DB_PATH/$dataBase does NOT exist! Migration needed!" >> "$LOG_FILE"	
		return 1
	fi
}

#-----------------------------------------------------------------------------
# Update_db_version:
#    Updates the db_version after succesfully modifing the db.
#-----------------------------------------------------------------------------
Update_db_version()
{
	local dataBase=$1
	local blank_db_version=$(sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase "SELECT Version from DB_INFO")
   echo "Update $dataBase version to $blank_db_version"  >> "$LOG_FILE"
	sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase "UPDATE DB_INFO SET Version='$blank_db_version'"
}


#-----------------------------------------------------------------------------
# Copy_new_db:
#    Copies the databases in blanckDbs to persist directory
#-----------------------------------------------------------------------------
Copy_new_db()
{
	local copyDB=0
	# Parse input parameters
	# If input '>' then next input is db name to copy
	for arg in "$@"
	do 
		if [[  $copyDB -eq 1 ]]; then
			# Copy this database to DEFAULT_CUSTOMER_DB_PATH
			echo "Copying $arg to $DEFAULT_CUSTOMER_DB_PATH"     >> "$LOG_FILE"
			cp $DEFAULT_DB_FROM_FW_IMAGE_PATH/$arg $DEFAULT_CUSTOMER_DB_PATH/$arg
			copyDB=0
		fi
		
		if [[ "$arg" == '>' ]]; then
			copyDB=1
		fi
	done
}

#-----------------------------------------------------------------------------
# Get_column_names:
#-----------------------------------------------------------------------------
Get_column_names()
{
	#Arguments
	dataBase=$1
	table=$2
	
	#Get column names from pragma table_info
	table_info=$(sqlite3 $dataBase "pragma table_info('$table')" )
	OLDIFS=$IFS
	IFS='
'
	for line in $table_info
	do
		tmp=${line#*|}      # Remove the characters before the first |
		column=${tmp%%|*}     # Remove the characters after the second |
		echo $column 
	done
	OLDIFS=$IFS
}

#-----------------------------------------------------------------------------
# Get_column_definition_alter_table:
#-----------------------------------------------------------------------------
Get_column_definition_alter_table()
{
	#Arguments
	dataBase=$1
	table=$2
	column_name=$3
	
	#Table Schema
	table_info=$(sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase ".schema '$table'")
	
	#Seperate table_info by lines
	OLDIFS=$IFS
	IFS='
'
	array=( $table_info )
	
	#Search for column name
	found=0
	for ((i=1; i<${#array[@]}; ++i)); #skip first line
	do
		tmp=${array[$i]#*[}      # Remove the characters before the first [
		name=${tmp%%]*}     # Remove the characters after the second ]
    	#echo "line $i: ${array[$i]}";
    	if  [[ "$name" == "$column_name" ]]; then
    		#remove the comma if found
    		array[$i]=${array[$i]%%,*}
    		found=1
			break
		fi
	done
	OLDIFS=$IFS
	
	#ALTER TABLE ADD COLUMN for line found
	if [[ $found -eq 1 ]]; then
		sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase "ALTER TABLE ${table} ADD COLUMN ${array[$i]}" 
      rc=$?
		if [ "$rc" == "0" ]; then
        echo "      sqlite3 ${DEFAULT_CUSTOMER_DB_PATH}${dataBase} ALTER TABLE ${table} ADD COLUMN ${array[$i]}"
		  echo "      sqlite3 ${DEFAULT_CUSTOMER_DB_PATH}${dataBase} ALTER TABLE ${table} ADD COLUMN ${array[$i]}" >> $LOG_FILE
      else
        echo "      Failed to add  COLUMN ${array[$i]} to ${table}"
        echo "      Failed to add  COLUMN ${array[$i]} to ${table}"  >> $LOG_FILE
        Bail_On_Error $rc "Failed to add  COLUMN ${array[$i]} to ${table}"
      fi
	fi	

}

###############################################################################
# MAIN
###############################################################################
echo "-------------------------------------------------------" >> "$LOG_FILE"
echo "$(date): Starting running $0 "                           >> "$LOG_FILE"

# Create a back up DB directory if it doesn't exist to back up the db before migration
# Only save one copy before the migration
mkdir -p "$BACKUP_DB_DIR"

################################################################################
# Check to see if pending_db_restore exists in /mnt/persist/db/ directory.
# If there is one, that means we need to replace the current configuration.db
# with the backup version. pending_db_restore file contains the file to be restored
################################################################################
pushd $DEFAULT_CUSTOMER_DB_PATH

if [ -f $PENDING_RESTORE_SCRIPT ]; then
   chmod 755 $PENDING_RESTORE_SCRIPT
	echo "Running $PENDING_RESTORE_SCRIPT to restore db from $LIB_RESTORE_DIR" >> "$LOG_FILE"
	cat $PENDING_RESTORE_SCRIPT                                                >> "$LOG_FILE"
	echo "$(ls -l $LIB_RESTORE_DIR/*)"                                         >> "$LOG_FILE"
   # Run the restore script
   $PENDING_RESTORE_SCRIPT   >> "$LOG_FILE" 2>&1
   # Remove the restore script
   rm $PENDING_RESTORE_SCRIPT
   echo "Finished restoring $LIB_RESTORE_DIR" >> "$LOG_FILE"
   sync  # make sure it is written in the CF
elif [ -f $PENDING_MFG_RESET_FILE ]; then
   echo "$PENDING_MFG_RESET_FILE exists, resetting all DB to default MFG settings" >> "$LOG_FILE"
   # Need to add more code here later

elif [ -f $PENDING_DB_CLR_STAT_FILE ]; then
   echo "$PENDING_DB_CLR_STAT_FILE exists, deleting current statistic DB" >> "$LOG_FILE"
   rm $PENDING_DB_CLR_STAT_FILE
   # Move all statistic DB files to back up dir
   mv $DEFAULT_CUSTOMER_DB_PATH/statistic* $BACKUP_DB_DIR/
   # Let the rest of DB migration to take care the rest.
elif [ -f $PENDING_DB_WIPEOUT_FILE ]; then
   echo "$PENDING_DB_WIPEOUT_FILE exists, deleting all current DB" >> "$LOG_FILE"

   # Move all current DB file to back up dir
   mv $DEFAULT_CUSTOMER_DB_PATH/* $BACKUP_DB_DIR/

   # Need to copy the blank configuration.db first before we can back up the WWNN
   echo "Copying blank databases from $DEFAULT_DB_FROM_FW_IMAGE_PATH to DEFAULT_CUSTOMER_DB_PATH ..." >> "$LOG_FILE"
   cp $DEFAULT_DB_FROM_FW_IMAGE_PATH/* $DEFAULT_CUSTOMER_DB_PATH/

   # Then copy back the statistic database to preserve the statistic data
   # Let the DB migration below to handle the rest
   cp $BACKUP_DB_DIR/statistic*  $DEFAULT_CUSTOMER_DB_PATH/

   # Preserve the VPD_First_WWNN in Library_Config
   WWNN=$(sqlite3 $BACKUP_DB_DIR/configuration.db "Select VPD_First_WWNN from LIBRARY_CONFIG;")
   sqlite3 $DEFAULT_CUSTOMER_DB_PATH/configuration.db  "UPDATE LIBRARY_CONFIG set VPD_First_WWNN = \"$WWNN\";"
   if [ "$?" == "0" ];then
      echo "WWNN $WWNN is backed up successfully during DB wipe out" >> "$LOG_FILE"
   else
      echo "Failed to back up WWNN $WWNN during DB wipe out" >> "$LOG_FILE"
   fi
 
   # Remove the existing server.xml symloblic link to make sure it points to the
   # non-ssl in case ssl was enabled before the wipe out.
	rm -f /etc/tomcat6/server.xml
   ln -s /etc/tomcat6/server_no_ssl.xml  /etc/tomcat6/server.xml
   # Lastly, remove the pending_db_wipeout file
   rm $PENDING_DB_WIPEOUT_FILE
fi

popd

#################################
#DataBases in each directory
#################################
cd $DEFAULT_CUSTOMER_DB_PATH #old
FILES_Update=$(ls *.db)

cd $DEFAULT_DB_FROM_FW_IMAGE_PATH #new
FILES_Current=$(ls *.db)

#Print out db files in each directoy
echo "Database files in $DEFAULT_CUSTOMER_DB_PATH" >> "$LOG_FILE"
echo "$(ls -l $DEFAULT_CUSTOMER_DB_PATH)"          >> "$LOG_FILE"

echo "Database files in $DEFAULT_DB_FROM_FW_IMAGE_PATH" >> "$LOG_FILE"
echo "$(ls -l $DEFAULT_DB_FROM_FW_IMAGE_PATH)"          >> "$LOG_FILE"

###################################
#Compare Differences of directories
###################################

differences=$(diff  <(echo "$FILES_Update" ) <(echo "$FILES_Current")) 

#Call Function to parse differences and copy new db needed
Copy_new_db $differences

###################################
#Compare each db for modifications
###################################

# Compare each DB table
# FILES_Current contains list of DBs to check tables
for dataBase in $FILES_Current
do
   echo "Checking DB schema for database: $dataBase ..."
   echo "Checking DB schema for database: $dataBase ..." >> $LOG_FILE
   dbMigrationStartTime=$(date +%s.%N)

	Compare_db_version $dataBase
   compareDB=$?
	if [[ "$compareDB" == "1" || "$compareDB" == "2" ]]; then
      echo "Database: $dataBase needs to be updated, backing up current database ..."
      echo "Database: $dataBase needs to be updated, backing up current database ..." >> $LOG_FILE
		# Before we do any migration, save the current db first
		cp $DEFAULT_CUSTOMER_DB_PATH/$dataBase $BACKUP_DB_DIR

		# If DB version mismatched and the blank DB version is one of the DB_CLEANUP_VERSION
      # the current DB will be replaced by blank DB
		dbVersion=$(sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase "SELECT Version from DB_INFO")
		if [[ "$compareDB" == "2" || "$dbVersion" < "$DB_CLEAN_UP_VERSION" ]]; then
			# Replace the current DB with the blank one
         echo "Cleaning up databases, replacing $DEFAULT_CUSTOMER_DB_PATH/$dataBase with $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase"
			echo "Cleaning up databases, replacing $DEFAULT_CUSTOMER_DB_PATH/$dataBase with $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase" >> $LOG_FILE
			cp $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase $DEFAULT_CUSTOMER_DB_PATH/$dataBase
			if [[ "$dataBase" == "configuration.db" ]]; then
				echo "Backing Ethernet Config Data" >> $LOG_FILE
				# Dump the Ethernet_Config_Data to a file
				sqlite3 $BACKUP_DB_DIR/configuration.db ".dump 'ETHERNET_CONFIG_DATA'" > $BACKUP_ENET_DATA 
				# Drop the Ethernet_Config_Data table in the new DB and restore the data from the old DB
				sqlite3 $DEFAULT_CUSTOMER_DB_PATH/configuration.db "DROP TABLE ETHERNET_CONFIG_DATA"
				echo "Restoring Ethernet Config data" >> $LOG_FILE
				# Restore the Ethernet_Config_Data from the old table
				sqlite3 $DEFAULT_CUSTOMER_DB_PATH/configuration.db < $BACKUP_ENET_DATA
				# Remove the back up data
				rm $BACKUP_ENET_DATA
			fi

			if [[ "$dataBase" == "statistic.db" ]]; then
             # Backup the STAT_DATA table
             TEMP_DATA="tmp.sql"
             echo "Restoring up stat_data table" >> $LOG_FILE
             echo "ATTACH '/mnt/persist/db.bak/statistic.db' as backup;" > $TEMP_DATA
				 echo "BEGIN;" >> $TEMP_DATA
				 echo "DELETE FROM STAT_DATA;" >>  $TEMP_DATA
				 echo "INSERT INTO STAT_DATA SELECT * FROM backup.STAT_DATA;" >> $TEMP_DATA
				 echo "COMMIT;" >> $TEMP_DATA
             echo "DETACH backup;" >>  $TEMP_DATA
           
				 sqlite3 $DEFAULT_CUSTOMER_DB_PATH/statistic.db   < ./$TEMP_DATA
             rm $TEMP_DATA

			fi

		else
			# Get Tables for each DataBase
			echo "Updating $dataBase ..." >> $LOG_FILE

			#DB_TABLES_NEW=$(sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase .tables | tr ' ' '\n'|sort)
			#DB_TABLES_OLD=$(sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase .tables | tr ' ' '\n'|sort)
		
			DB_TABLES_NEW=$(sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase "SELECT NAME FROM SQLITE_MASTER WHERE TYPE='table' ORDER BY NAME ASC")
			DB_TABLES_OLD=$(sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase "SELECT NAME FROM SQLITE_MASTER WHERE TYPE='table' ORDER BY NAME ASC")

			tables_to_add=$(comm <(echo "$DB_TABLES_NEW") <(echo "$DB_TABLES_OLD") -3) 
		
			#Add new tables to existing customer db
			#for table in $tables_to_add 
			for table in $tables_to_add
			do
            echo "   Adding new table: $table to $dataBase ..."
            echo "   Adding new table: $table to $dataBase ..." >> $LOG_FILE
            tableUpdateStartTime=$(date +%s.%N)

				table_schema=$(sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase ".schema '$table'")
				sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase "$table_schema"	
            rc=$?
				if [ "$rc" == "0" ]; then
				  echo "      sqlite3 ${DEFAULT_CUSTOMER_DB_PATH}${dataBase} ${table_schema}" >> $LOG_FILE	
				else
				  echo "      Failed to add  table ${table_schema} to ${dataBase}"  >> $LOG_FILE
				  Bail_On_Error $rc "Failed to add  table ${table_schema} to ${dataBase}"
				fi
					
				#^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
				# Copy records into /mnt/persist from blank DBs.
				#
				# Algorithm:
				#	Dump the database where it has insert into table
				#   like:
				#		sqlite3 misc.db .dump | grep '^INSERT INTO "TEST_1"'
				# 
				#   Append this to a tmp file called table_insertions.sql
				#   
				#   table_insertions (Example):
				#	BEGIN TRANSACTION;
				#	INSERT INTO "TEST_1" VALUES(1,2,3);
				#	INSERT INTO "TEST_1" VALUES(2,2,3);
				#	INSERT INTO "TEST_1" VALUES(3,2,3);
				#	INSERT INTO "TEST_1" VALUES(4,2,3);
				#	INSERT INTO "TEST_1" VALUES(5,2,3);
				#   COMMIT;
				#
				#
				#   Execute the SQL into the DB
				#   remove the tmp file.
				#^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
				#echo "table to add records-> $table"
				echo "BEGIN TRANSACTION;" > $TMP_FILE
				sqlite3 $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase .dump | grep '^INSERT INTO "'$table'"' >> $TMP_FILE
				echo "COMMIT;" >> $TMP_FILE
				sqlite3 $DEFAULT_CUSTOMER_DB_PATH/$dataBase < $TMP_FILE
				rm $TMP_FILE
            tableUpdateEndTime=$(date +%s.%N)
            timeDiff=$(echo "$tableUpdateEndTime $tableUpdateStartTime" | awk '{print ($1 - $2)}')
            printf "   Took %.3F seconds to add $table to $dataBase\n" $timeDiff
            printf "   Took %.3F seconds to add $table to $dataBase\n" $timeDiff >> "$LOG_FILE"

		   done
	
			#Add new fields into tables
			echo "Checking tables for modifications" >> $LOG_FILE
			for table in $DB_TABLES_OLD
			do
            echo "   Checking table: $table in $dataBase ..." 
				echo "   Checking table: $table in $dataBase ..."  >> $LOG_FILE
            tableUpdateStartTime=$(date +%s.%N)

				table_columns_old=$(Get_column_names $DEFAULT_CUSTOMER_DB_PATH/$dataBase $table | tr ' ' '\n')
				table_columns_new=$(Get_column_names $DEFAULT_DB_FROM_FW_IMAGE_PATH/$dataBase $table | tr ' ' '\n')
				columns_to_add=$(comm <(echo -e "$table_columns_new") <(echo "$table_columns_old") -3 )
				
				#if columns_to_add is set then go inside if
				if [ ! -z "$columns_to_add" ]; then
					for column in $columns_to_add
					do
                  echo "      Adding new column: " $column " to table: " $table
						echo "      Adding new column: " $column " to table: " $table >> $LOG_FILE
						Get_column_definition_alter_table $dataBase $table $column 
					done
               tableUpdateEndTime=$(date +%s.%N)
               timeDiff=$(echo "$tableUpdateEndTime $tableUpdateStartTime" | awk '{print ($1 - $2)}')
               printf "   Took %.3F seconds to update $table in $dataBase\n" $timeDiff
               printf "   Took %.3F seconds to update $table in $dataBase\n" $timeDiff >> "$LOG_FILE"
				else
               echo "   No update needed for table: $table" >> $LOG_FILE
            fi
			done
			Update_db_version $dataBase
		fi
	fi
	

done


timeComplete=$(date +%s.%N)
timeDiff=$(echo "$timeComplete $timeInitial" | awk '{print ($1 - $2)}')

echo "$(date) Database migration completed"   >> "$LOG_FILE"
printf "Total time: %.3F seconds\n" $timeDiff >> "$LOG_FILE"
echo "Exiting $0 ..."  



