#!/bin/bash 
#CONFIDENTIAL
#
# LICENSED INTERNAL CODE SOURCE MATERIALS
#
# ANACONDA LICENSED INTERNAL CODE
#
# (C) COPYRIGHT IBM CORP. 2014
#
# THE SOURCE CODE FOR THIS PROGRAM IS NOT PUBLISHED OR
# OTHERWISE DIVESTED OF ITS TRADE SECRETS, IRRESPECTIVE
# OF WHAT HAS BEEN DEPOSITED WITH THE U.S. COPYRIGHT OFFICE.
#
################################################################################

################################################################################
#
# Filename: library_restore.sh
#
# DESCRIPTION: 
#    This utility is used to restore important library files (i.e. databases and 
#    certificates).
################################################################################

################################################################################
#
# CHANGE ACTIVITY:
#
#
# MM/DD/YYYY INI    CQ ID      Change Description
# ---------- ---   --------   --------------------------------------------------
# 02/20/2014 kqiu              Creation 
# 07/30/2014 kqiu   S1056793   Fixed a problem where CLI restore configuration 
#                              fails if restoring to a new CF with blank DB   
# 08/12/2014 kqiu              Fixed a problem where not all DBs are restored   
# 09/04/2014 kqiu              Added -l option for scp to limit bandwidth
# 11/10/2014 kqiu   S1057987   Added code to remove the back up directories before
#                              the restore
################################################################################
VERSION=1.1
############################################################################
# DEFINED CONSTANTS HERE
############################################################################
num_of_cmd_param=$#
_DEBUG="ON"         # Internal debug flag
USER_PARAMS="$@"

PERSIST_DIR="/mnt/persist"
DB_DIR="$PERSIST_DIR/db"
LIB_RESTORE_DIR="$PERSIST_DIR/restore"

CAL_DB="calibration.db"
CONFIG_DB="configuration.db"
INV_DB="inventory.db"
MISC_DB="misc.db"
STAT_DB="statistic.db"

SCRATCH_DIR="/mnt/scratch"

CURRENT_CAL_DB="$DB_DIR"/"$CAL_DB"
RESTORE_CAL_DB="$LIB_RESTORE_DIR"/db/"$CAL_DB"

CURRENT_CONFIG_DB="$DB_DIR"/"$CONFIG_DB"
RESTORE_CONFIG_DB="$LIB_RESTORE_DIR"/db/"$CONFIG_DB"

CURRENT_INV_DB="$DB_DIR"/"$INV_DB"
RESTORE_INV_DB="$LIB_RESTORE_DIR"/db/"$INV_DB"

CURRENT_MISC_DB="$DB_DIR"/"$MISC_DB"
RESTORE_MISC_DB="$LIB_RESTORE_DIR"/db/"$MISC_DB"

CURRENT_STAT_DB="$DB_DIR"/"$STAT_DB"
RESTORE_STAT_DB="$LIB_RESTORE_DIR"/db/"$STAT_DB"

FWNODE_TABLE_NAME="/mnt/scratch/FWNodeTable.csv"

LOG_FILE="/var/log/fw/lib_restore.log"
# If the current log file size is greater than 100KB,
# we will back up the current one and open a new file. 
# We only keep 2 copies of log files, which should be more than enough.
LOG_FILE_SIZE_LIMIT=102400
NUM_OF_LOG_FILES=1   # This will keep up to 2 back up log files

PENDING_RESTORE_FILE="$PERSIST_DIR"/pending_lib_restore.sh

LIB_BACKUP_FILE=""

OTHER_LCC_IPS=""

LIB_BACKUP_HEADER="/mnt/scratch/back_header.tmp"

###############################################################################
# DEFINE FUNCTIONS HERE
###############################################################################

#------------------------------------------------------------------------------
# print_usage:
#   Prints the usage of this script.
#------------------------------------------------------------------------------
print_usage() 
{
   echo "Current Version: $VERSION"
   echo "Usage: $0 "
   echo "This script restores the important library files previously saved."
   echo "OPTIONS:"
   echo "   -h                 : Show this message"
   echo "   -p                 : Full path where the library back up file is located"
   echo

}

###############################################################################
# DEFINE FUNCTIONS HERE
###############################################################################
DEBUG()
{
   # If _Debug flag is turned on, run the commands in $@ (i.e. ehco "hello")
   if [ "$_DEBUG" == "ON" ]; then
      echo $(date +%m-%d-%y" "%H:%M:%S) $@ >> $LOG_FILE 
   fi
}

RUN_CMD()
{
   # This function will 
   # 1. Log the current command being executed with time stamp
   # 2. Execute the shell command passed in and all the stdout and stderr will
   #    be redirected to the log file $LOG_FILE
   ########################################################################
   # WARNING: This function cannot be used to run cascaded commands or
   #          any command that needs to be redirected.
   #          i.e. cat /proc/mtd | grep A-Boot
   #          i.e. cat /dev/mtd1 > outputfile
   ########################################################################
   echo $(date +%m-%d-%y" "%H:%M:%S) "Running command: " $@ >> $LOG_FILE
   $@ >> $LOG_FILE 2>&1
}

#-----------------------------------------------------------------------------
# back_up_log_file
#    This function keeps up to NUM_OF_LOG_FILES copies of the log 
#-----------------------------------------------------------------------------

back_up_log_file()
{
   local currentLogFileSize=""

	if [ ! -f $LOG_FILE ]; then
		touch $LOG_FILE
	else
		currentLogFileSize=$(stat -c%s $LOG_FILE)
		# echo currentLogFileSize = $currentLogFileSize
		# If the current log file size is greater than the limit,
		# rotate it out. Otherwise, keep using the current log file
		if [ "$currentLogFileSize" -gt "$LOG_FILE_SIZE_LIMIT" ]; then
			for ((i=$NUM_OF_LOG_FILES; i>=1 ; i--))
			do
				if [ -f $LOG_FILE$i ]; then
					mv $LOG_FILE$i  $LOG_FILE$((i+1))
				fi

			done
			if [ -f $LOG_FILE ]; then
				mv "$LOG_FILE" "$LOG_FILE"1
			fi

		fi
	fi

	echo "--------------------------------------------------------" >> $LOG_FILE
	echo "Start running $0 $USER_PARAMS" >> $LOG_FILE

}

#-----------------------------------------------------------------------------
# Prepare_Restore
#    This function prepares to restore the library files 
#-----------------------------------------------------------------------------
Prepare_Restore()
{
   DEBUG "Preparing library restore ..."
   #Remove the back up directory first
	rm -rf $LIB_RESTORE_DIR
   #Create a new directory
	mkdir $LIB_RESTORE_DIR

}

#-----------------------------------------------------------------------------
# Extract_Backup_Files
#    This function extracts the back up file to the restore directory
#-----------------------------------------------------------------------------
Extract_Backup_Files()
{
   local tmp_tarball_name="restore.dbz"
   local backup_key=""

	DEBUG "Getting library backup header from $LIB_BACKUP_FILE ..."
	head -7 $LIB_BACKUP_FILE > $LIB_BACKUP_HEADER

   Signature=$(cat $LIB_BACKUP_HEADER | grep Signature | cut -f2 -d '=')
   BackupVersion=$(cat $LIB_BACKUP_HEADER | grep BackupVersion | cut -f2 -d '=')
   FW_Version=$(cat $LIB_BACKUP_HEADER | grep FirmwareVersion | cut -f2 -d '=')
   LibrarySerialNumber=$(cat $LIB_BACKUP_HEADER | grep LibrarySerialNumber | cut -f2 -d '=')
   LibraryName=$(cat $LIB_BACKUP_HEADER | grep LibraryName | cut -f2 -d '=')
   TimeStamp=$(cat $LIB_BACKUP_HEADER | grep TimeStamp | cut -f2 -d '=')

   DEBUG "Signature: $Signature"
   DEBUG "BackupVersion: $BackupVersion"
   DEBUG "FW_Version: $FW_Version"
   DEBUG "LibrarySerialNumber: $LibrarySerialNumber"
   DEBUG "LibraryName: $LibraryName"
   DEBUG "TimeStamp: $TimeStamp"
   
   rm -f $LIB_BACKUP_HEADER
   # First need to strip off the backup header to create a restore.dbz
   # Use the tail command to start from 8th line
	DEBUG "Stripping off the backup header ..."
   tail -n +8 "$LIB_BACKUP_FILE" > "$SCRATCH_DIR"/"$tmp_tarball_name"
   rc=$?
   if [ "$rc" -ne "0" ]; then
      DEBUG "Failed to create $SCRATCH_DIR/$tmp_tarball_name rc = $rc"
   else
		DEBUG "Start extracting the back up tarball ..."
      # the key is generated from the timestamp in the header
      backup_key=$(echo $TimeStamp | shasum | cut -f1 -d ' ')
      # go in to the restore directory to extract the tarball  there.  It will have
      # directories like db, certificates 
      pushd $LIB_RESTORE_DIR
      dd if="$SCRATCH_DIR/$tmp_tarball_name" | openssl des3 -d -k "$backup_key" | tar -zxvf -
      rc=$?
      popd
      if [ "$rc" -ne "0" ]; then
         DEBUG "Failed to extract $tmp_tarball_name, rc = $rc"
         rm -rf $LIB_RESTORE_DIR
         exit 1
		else
         DEBUG "Successfully extracted backup files"
         rm -rf $SCRATCH_DIR/$tmp_tarball_name 
		fi      
	fi
}

##########################################################################
# Valid_Restore_DB
# The function validates if the restore DB is good enough to be used
# Here are the requirements:
# 1. Library serial number must match the current library serial number
# 2. The restore version must have same or higher DB version than the current one
# 3. 
##########################################################################
Valid_Restore_DB()
{
	local currentLibSerialNum=""
	local restoreLibSerialNum=""
	local currentConfigDBVersion=""
	local restoreConfigDBVersion=""
	local rc=0

	DEBUG "Validating $RESTORE_CONFIG_DB ..."
	# First validate the configuration DB
	currentLibSerialNum=$(sqlite3 "$CURRENT_CONFIG_DB" "SELECT SerialNumber FROM FRAME_CONFIG WHERE FrameIdx = (SELECT L_Frame FROM LIBRARY_CONFIG);")
	restoreLibSerialNum=$(sqlite3 "$RESTORE_CONFIG_DB" "SELECT SerialNumber FROM FRAME_CONFIG WHERE FrameIdx = (SELECT L_Frame FROM LIBRARY_CONFIG);")

	currentConfigDBVersion=$(sqlite3 "$CURRENT_CONFIG_DB" "select Version from db_info;")
	restoreConfigDBVersion=$(sqlite3 "$RESTORE_CONFIG_DB" "select Version from db_info;")

	DEBUG "Current Library Serial Number is $currentLibSerialNum"
	DEBUG "Backup Library Serial Number is $restoreLibSerialNum"
	DEBUG "Current configuration.db version is $currentConfigDBVersion"
	DEBUG "Backup configuration.db version is $restoreConfigDBVersion"

   # If current library serial number is not empty and compare to the back up library serial number
	if [ -n "$currentLibSerialNum" ] && [ "$currentLibSerialNum" != "$restoreLibSerialNum" ]; then
		DEBUG "Library Serial Number mismatch, can NOT restore the configuration DB!"
      rm -rf $LIB_RESTORE_DIR
		exit 3
	else
		if awk "BEGIN {exit ! ($restoreConfigDBVersion < $currentConfigDBVersion)}" 
	      then  # Need to use awk to compare decimal in bash
		   DEBUG "Backup configuration.db version is lower than current version, can NOT restore the configuration DB"
		   rm -rf $LIB_RESTORE_DIR
		   exit 4
      else
         DEBUG "$RESTORE_CONFIG_DB is validated!"
      fi
	fi

   if [ -z "$currentLibSerialNum" ]; then
      DEBUG "Current Library Serial Number is empty, continue the restore process ..."
   fi

   # Validate the inventory.db by checking the DB version
	currentInvDBVersion=$(sqlite3 "$CURRENT_INV_DB" "select Version from db_info;")
	restoreInvDBVersion=$(sqlite3 "$RESTORE_INV_DB" "select Version from db_info;")
   
   DEBUG "$CURRENT_INV_DB version: $currentInvDBVersion"
   DEBUG "$RESTORE_INV_DB version: $restoreInvDBVersion"
   if [ -n "$restoreInvDBVersion" ]; then
		if awk "BEGIN {exit ! ($restoreInvDBVersion < $currentInvDBVersion)}" 
			then  # Need to use awk to compare decimal in bash
			DEBUG "RESTORE_INV_DB version is lower than current version, can NOT restore library"
			rm -rf $LIB_RESTORE_DIR
			exit 4
		else
			DEBUG "$RESTORE_INV_DB is validated" 
		fi
   else
      DEBUG "Invalid DB version for $RESTORE_INV_DB"
      exit 5
   fi

   # Validate the calibration.db by checking the DB version
	currentCalDBVersion=$(sqlite3 "$CURRENT_CAL_DB" "select Version from db_info;")
	restoreCalDBVersion=$(sqlite3 "$RESTORE_CAL_DB" "select Version from db_info;")
   DEBUG "$CURRENT_CAL_DB version: $currentCalDBVersion"
   DEBUG "$RESTORE_CAL_DB version: $restoreCalDBVersion"

   if [ -n "$restoreCalDBVersion" ]; then
		if awk "BEGIN {exit ! ($restoreCalDBVersion < $currentCalDBVersion)}" 
			then  # Need to use awk to compare decimal in bash
			DEBUG "$RESTORE_CAL_DB version is lower than current version, can NOT restore library"
			rm -rf $LIB_RESTORE_DIR
			exit 4
		else
			DEBUG "$RESTORE_CAL_DB is validated" 
		fi
   else
      DEBUG "Invalid DB version for $RESTORE_CAL_DB"
      exit 5
   fi

   # Validate the misc.db by checking the DB version
	currentMiscDBVersion=$(sqlite3 "$CURRENT_MISC_DB" "select Version from db_info;")
	restoreMiscDBVersion=$(sqlite3 "$RESTORE_MISC_DB" "select Version from db_info;")
   DEBUG "$CURRENT_MISC_DB version: $currentMiscDBVersion"
   DEBUG "$RESTORE_MISC_DB version: $restoreMiscDBVersion"

   if [ -n "$restoreMiscDBVersion" ]; then
		if awk "BEGIN {exit ! ($restoreMiscDBVersion < $currentMiscDBVersion)}" 
			then  # Need to use awk to compare decimal in bash
			DEBUG "$RESTORE_MISC_DB version is lower than current version, can NOT restore library"
			rm -rf $LIB_RESTORE_DIR
			exit 4
		else
			DEBUG "$RESTORE_MISC_DB is validated" 
		fi
   else
      DEBUG "Invalid DB version for $RESTORE_MISC_DB"
      exit 5
   fi
   # Validate the statistic.db by checking the DB version
	currentStatDBVersion=$(sqlite3 "$CURRENT_STAT_DB" "select Version from db_info;")
	restoreStatDBVersion=$(sqlite3 "$RESTORE_STAT_DB" "select Version from db_info;")
   DEBUG "$CURRENT_STAT_DB version: $currentStatDBVersion"
   DEBUG "$RESTORE_STAT_DB version: $restoreStatDBVersion"

   if [ -n "$restoreStatDBVersion" ]; then
		if awk "BEGIN {exit ! ($restoreStatDBVersion < $currentStatDBVersion)}" 
			then  # Need to use awk to compare decimal in bash
			DEBUG "$RESTORE_STAT_DB version is lower than current version, can NOT restore library"
			rm -rf $LIB_RESTORE_DIR
			exit 4
		else
			DEBUG "$RESTORE_STAT_DB is validated" 
		fi
   else
      DEBUG "Invalid DB version for $RESTORE_STAT_DB"
      exit 5
   fi

	# If we make it this far, we are good
   DEBUG "All DB are validated!"

	# Get the ip address for other LCC out there
	OTHER_LCC_IPS=$(cat $FWNODE_TABLE_NAME | cut -d ',' -f2)
}

##########################################################################
# Distribute_restore_files
# Send the entire restore directory to other LCC in the library
#
##########################################################################
Distribute_restore_files()
{
	local nodeID=""

	DEBUG "Copying restore files to other LCCs: $OTHER_LCC_IPS ..."
	# Start copying to other LCC
	for thisIP in $OTHER_LCC_IPS
	do
		nodeID=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f1)
		Interface=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f3)
		if [ -z "$Interface" ]; then   # If no interface is found, default to interface eth0.103
		   Interface="eth0.103"
		fi
		DEBUG "Copying $LIB_RESTORE_DIR to node $nodeID"
		RUN_CMD scp -6 -l 14588 -o ConnectTimeOut=2 -r "$LIB_RESTORE_DIR" root@[$thisIP%$Interface]:"$PERSIST_DIR"
		rc=$?
		if [ $rc != "0" ]; then
			DEBUG "Failed to scp $LIB_RESTORE_DIR to node $nodeID via $thisIP%$Interface"
			# Don't exit here, continue to copy to other LCC
		fi
	done

}


##########################################################################
# Finalize_Restore
# This function first distribute the pending restore script to all LCCs
# and restart the library
#
##########################################################################
Finalize_Restore()
{
	local nodeID=""

	# Create a pending restore script to be executed by start up script
   # Do NOT use any symbols or macros in this pending restore script
   # because it is a stand alone script that gets executed at start up
   echo "#!/bin/bash" > $PENDING_RESTORE_FILE

   echo "echo Start restoring library files ..." >> $PENDING_RESTORE_FILE 
   echo "rm -rf /mnt/persist/db.bak" >> $PENDING_RESTORE_FILE
   echo "mv /mnt/persist/db /mnt/persist/db.bak"  >> $PENDING_RESTORE_FILE 

   echo "echo Restoring databases ..."  >> $PENDING_RESTORE_FILE
   echo "rm -rf /mnt/persist/db" >> $PENDING_RESTORE_FILE
   echo "mv /mnt/persist/restore/db /mnt/persist/db" >> $PENDING_RESTORE_FILE

   echo "echo Restoring certificates ..."  >> $PENDING_RESTORE_FILE
   echo "rm -rf /mnt/certificates.bak" >> $PENDING_RESTORE_FILE
   echo "mv /mnt/certificates /mnt/certificates.bak" >> $PENDING_RESTORE_FILE
   echo "mv /mnt/persist/restore/certificates  /mnt/certificates" >> $PENDING_RESTORE_FILE

   echo "echo Removing /mnt/persist/restore directory ..." >> $PENDING_RESTORE_FILE
   echo "rm -rf /mnt/persist/restore" >> $PENDING_RESTORE_FILE

   # Make sure it has the correct permission
   chmod 755 $PENDING_RESTORE_FILE

	# Start copying to other LCC
	for thisIP in $OTHER_LCC_IPS
	do
		nodeID=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f1)
		Interface=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f3)
		if [ -z "$Interface" ]; then   # If no interface is found, default to interface eth0.103
		   Interface="eth0.103"
		fi
		DEBUG "Copying $PENDING_RESTORE_FILE to node $nodeID"
		RUN_CMD scp -6 -l 14588 -o ConnectTimeOut=2 "$PENDING_RESTORE_FILE" root@[$thisIP%$Interface]:"$PENDING_RESTORE_FILE"
		rc=$?
		if [ $rc != "0" ]; then
			DEBUG "Failed to scp $PENDING_RESTORE_FILE to node $nodeID via $thisIP%$Interface"
			# Don't exit here, continue to copy to other LCC, the primary DB manager will sync to
			# the one that failed to copy after reset
		fi
	done

	# After distributing the pending restore script to all LCCs, the script will 
   # issue a restart on all LCC
	for thisIP in $OTHER_LCC_IPS
	do
		nodeID=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f1)
		Interface=$(grep "$thisIP" $FWNODE_TABLE_NAME | cut -d  ',' -f3)
		if [ -z "$Interface" ]; then   # If no interface is found, default to interface eth0.103
		   Interface="eth0.103"
		fi
		DEBUG "Restarting LCC on node $nodeID"
		RUN_CMD ssh -6 -o ConnectTimeOut=2 root@$thisIP%$Interface "shutdown -r now"
		rc=$?
		if [ $rc != "0" ]; then
			DEBUG "Failed to shutdown down node $nodeID via $thisIP%$Interface"
			# Don't exit here, continue to shutdown other LCC as there is not much
         # we can do at this point
		fi
	done

   # Finally we can restart the local LCC
   DEBUG "Restarting local LCC ..."
   # do one extra sync to de-stage the log file
   sync
   shutdown -r now

}

###########################################################################
# Start of main
###########################################################################
echo "Running $0"
back_up_log_file

# First process the user input parameters
if [ $num_of_cmd_param -lt 1 ]; then
  echo "Number of param enter: $num_of_cmd_param"
  echo "Please enter at least one option"
  print_usage
  exit 1
fi

while getopts "p:hr" OPTION
do
   case $OPTION in
      h)
        print_usage
        exit 0
        ;;
      p)
        LIB_BACKUP_FILE="$OPTARG"
		  echo LIB_BACKUP_FILE = $LIB_BACKUP_FILE
        ;;
      r)
        remoteOnly="TRUE"  # This flag is used internally by the script
        ;;
      ?)
        print_usage
        exit 1
        ;;
   esac

done

Prepare_Restore
Extract_Backup_Files
Valid_Restore_DB
Distribute_restore_files
Finalize_Restore

#if we get this far, we are good
exit 0





 







