package java.lang;

import java.io.*;
import java.security.CodeSource;
import java.security.PermissionCollection;
import java.security.Policy;
import java.security.ProtectionDomain;
import com.ibm.oti.vm.AbstractClassLoader;
import java.net.URL;
import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Hashtable;
import java.security.cert.Certificate;
import java.security.AccessController;
import java.security.PrivilegedAction;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

/**
 * ClassLoaders are used to dynamically load, link and install
 * classes into a running image.
 *
 * @author		OTI
 * @version		initial
 */
public abstract class ClassLoader {

	/**
	 * The default protection domain for classes loaded by
	 * this classloader which are defined without a specific
	 * protection domain.
	 */
	static private ProtectionDomain defaultProtectionDomain;

	static final ClassLoader systemClassLoader;

	private static ClassLoader applicationClassLoader;

	private int vmRef;
	ClassLoader parent;

	private Hashtable packages = new Hashtable();
	private Hashtable classSigners = null; // initialized if needed
	private Hashtable packageSigners = new Hashtable();
	private Certificate[] emptyCertificates = new Certificate[0];

	static {
		ClassLoader sysTemp = null;
		// Proper initialization requires BootstrapLoader is the first loader instantiated
		String systemLoaderString = System.getProperty("systemClassLoader");
		if(null == systemLoaderString) {
			sysTemp = com.ibm.oti.vm.BootstrapClassLoader.singleton();
		} else {
			try {
				sysTemp = (ClassLoader)Class.forName(systemLoaderString,true,null).newInstance();
			} catch(Throwable x) {
				x.printStackTrace();
				sysTemp = null; /* stop compiler from complaining */
				System.exit(0);
			}
		}
		systemClassLoader = sysTemp;
		AbstractClassLoader.setBootstrapClassLoader(systemClassLoader);
		applicationClassLoader = systemClassLoader;

		applicationClassLoader = sun.misc.Launcher.getLauncher().getClassLoader();
	}

/**
 * Constructs a new instance of this class with the system
 * class loader as its parent.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	SecurityException
 *					if a security manager exists and it does not
 *					allow the creation of new ClassLoaders.
 */
protected ClassLoader() {
	this(applicationClassLoader);
}

/**
 * Constructs a new instance of this class with the given
 * class loader as its parent.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		parent ClassLoader
 *					the ClassLoader to use as the new class
 *					loaders parent.
 * @exception	SecurityException
 *					if a security manager exists and it does not
 *					allow the creation of new ClassLoaders.
 * @exception	NullPointerException
 *					if the parent is null.
 */
protected ClassLoader(ClassLoader parentLoader) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkCreateClassLoader();

	// VM Critical: must set parent before calling initializeInternal()
	parent = parentLoader;
	if (systemClassLoader != null)
		com.ibm.oti.vm.VM.initializeClassLoader(this, false);
}

/**
 * Constructs a new class from an array of bytes containing a
 * class definition in class file format.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		classRep byte[]
 *					a memory image of a class file.
 * @param 		offset int
 *					the offset into the classRep.
 * @param 		length int
 *					the length of the class file.
 * @deprecated Use defineClass(String, byte[], int, int)
 */
protected final Class defineClass (byte [] classRep, int offset, int length) throws ClassFormatError {
	return defineClass ((String) null, classRep, offset, length);
}

/**
 * Constructs a new class from an array of bytes containing a
 * class definition in class file format.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		className java.lang.String
 *					the name of the new class
 * @param 		classRep byte[]
 *					a memory image of a class file
 * @param 		offset int
 *					the offset into the classRep
 * @param 		length int
 *					the length of the class file
 */
protected final Class defineClass(String className, byte[] classRep, int offset, int length) throws ClassFormatError {
	return defineClass(className, classRep, offset, length, null);
}

private String checkClassName(String className) {
	int index;
	if((index = className.lastIndexOf('.')) >= 0) {
		String packageName = className.substring(0, index);
		if (className.startsWith("java.")) {
			throw new SecurityException(com.ibm.oti.util.Msg.getString("K01d2", packageName));
		}
		return packageName;
	}
	return "";
}

/**
 * Constructs a new class from an array of bytes containing a
 * class definition in class file format and assigns the new
 * class to the specified protection domain.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		className java.lang.String
 *					the name of the new class.
 * @param 		classRep byte[]
 *					a memory image of a class file.
 * @param 		offset int
 *					the offset into the classRep.
 * @param 		length int
 *					the length of the class file.
 * @param 		protectionDomain ProtectionDomain
 *					the protection domain this class should
 *					belongs to.
 */
protected final Class defineClass (
	String className,
	byte[] classRep,
	int offset,
	int length,
	ProtectionDomain protectionDomain)
	throws java.lang.ClassFormatError
{
	Certificate[] certs = null;
	if (protectionDomain != null) {
		CodeSource cs = protectionDomain.getCodeSource();
		if (cs != null) certs = cs.getCertificates();
	}
	if (className != null) {
		String packageName = checkClassName(className);
		checkPackageSigners(packageName, packageName, certs);
	}

	Class answer = defineClassImpl(className, classRep, offset, length);
	if (protectionDomain == null)
		answer.setPDImpl(getDefaultProtectionDomain());
	else {
		answer.setPDImpl(protectionDomain);
	}
	if (certs != null) setSigners(answer, certs);
	return answer;
}

private void checkPackageSigners(String packageName, String className, Certificate[] classCerts) {
	Certificate[] packageCerts = (Certificate[])packageSigners.get(packageName);
	if (packageCerts == null) {
		if (classCerts == null) packageSigners.put(packageName, emptyCertificates);
		else packageSigners.put(packageName, classCerts);
	} else {
		if ((classCerts == null && packageCerts.length == 0) || classCerts == packageCerts)
			return;
		if (classCerts != null && classCerts.length == packageCerts.length) {
			boolean foundMatch = true;
			test: for (int i=0; i<classCerts.length; i++) {
				if (classCerts[i] == packageCerts[i]) continue;
				if (classCerts[i].equals(packageCerts[i])) continue;
				for (int j=0; j<packageCerts.length; j++) {
					if (j == i) continue;
					if (classCerts[i] == packageCerts[j]) continue test;
					if (classCerts[i].equals(packageCerts[j])) continue test;
				}
				foundMatch = false;
				break;
			}
			if (foundMatch) return;
		}
		throw new SecurityException(com.ibm.oti.util.Msg.getString("K01d1", className));
	}
}

/**
 * Gets the current default protection domain. If there isn't
 * one, it attempts to construct one based on the currently
 * in place security policy.
 * <p>
 * If the default protection domain can not be determined,
 * answers null.
 * <p>
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		ProtectionDomain or null
 *					the default protection domain.
 */
static final ProtectionDomain getDefaultProtectionDomain () {
	if (defaultProtectionDomain == null) {
		Policy policy = (Policy)AccessController.doPrivileged(new PrivilegedAction() {
			public Object run() {
				return Policy.getPolicy();
			}});
		if (policy != null) {
			CodeSource nullCodeSource = new CodeSource(null, null);
			PermissionCollection permissions = policy.getPermissions(nullCodeSource);
			defaultProtectionDomain = new ProtectionDomain(nullCodeSource, permissions);
		}
	}
	return defaultProtectionDomain;
}

/**
 * VM level support for constructing a new class. Should not
 * be called by subclasses.
 *
 * @author		OTI
 * @version		initial
 */
private final native Class defineClassImpl (String className, byte [] classRep, int offset, int length);

/**
 * Overridden by subclasses, by default throws ClassNotFoundException.
 * This method is called by loadClass() after the parent ClassLoader
 * has failed to find a loaded class of the same name.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.Class
 *					the class or null.
 * @param 		className String
 *					the name of the class to search for.
 * @exception	ClassNotFoundException
 *					always, unless overridden.
 */
protected Class findClass (String className) throws ClassNotFoundException {
	throw new ClassNotFoundException();
}

/**
 * Attempts to find and return a class which has already
 * been loaded by the virtual machine. Note that the class
 * may not have been linked and the caller should call
 * resolveClass() on the result if necessary.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.Class
 *					the class or null.
 * @param 		className String
 *					the name of the class to search for.
 */
protected final native Class findLoadedClass (String className);

/**
 * Attempts to load a class using the system class loader.
 * Note that the class has already been been linked.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.Class
 *					the class which was loaded.
 * @param 		className String
 *					the name of the class to search for.
 * @exception	ClassNotFoundException
 *					if the class can not be found.
 */
protected final Class findSystemClass (String className) throws ClassNotFoundException {
	return applicationClassLoader.loadClass(className);
}

/**
 * Returns the specified ClassLoader's parent.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.ClassLoader
 *					the class or null.
 * @exception	SecurityException
 *					if a security manager exists and it does not
 *					allow the parent loader to be retrieved.
 */
public final ClassLoader getParent() {
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
		ClassLoader callersClassLoader = com.ibm.oti.vm.VM.callerClassLoader();
		if (callersClassLoader != null && callersClassLoader != this
			&& !callersClassLoader.isAncestorOf(this))
				security.checkPermission(
					RuntimePermission.permissionToGetClassLoader);
	}
	return parent;
}

/**
 * Answers an URL which can be used to access the resource
 * described by resName, using the class loader's resource lookup
 * algorithm. The default behavior is just to return null.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		URL
 *					the location of the resource.
 * @param		resName String
 *					the name of the resource to find.
 *
 * @see			Class#getResource
 */
public URL getResource (String resName) {
	URL url = parent == null
		? systemClassLoader.findResource(resName)
		: parent.getResource(resName);
	if (url != null) return url;

	return findResource(resName);
}

/**
 * Answers an Enumeration of URL which can be used to access the resources
 * described by resName, using the class loader's resource lookup
 * algorithm. The default behavior is just to return an empty Enumeration.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Enumeration
 *					the locations of the resources.
 * @param		resName String
 *					the name of the resource to find.
 */
public final Enumeration getResources (String resName) throws IOException {
	ClassLoader up = this;
	final Vector resources = new Vector();
	do {
		Enumeration e = up.findResources(resName);
		if (e.hasMoreElements()) resources.addElement(e);
		if (up == systemClassLoader) break;
		up = up.parent;
		if (up == null) up = systemClassLoader;
	} while (true);
	return new Enumeration() {
		int index = resources.size() - 1;
		public boolean hasMoreElements() {
			while (index >= 0) {
				if (((Enumeration)resources.elementAt(index)).hasMoreElements())
					return true;
				index--;
			}
			return false;
		}
		public Object nextElement() {
			while (index >= 0) {
				Enumeration e = (Enumeration)resources.elementAt(index);
				if (e.hasMoreElements()) return e.nextElement();
				index--;
			}
			throw new NoSuchElementException();
		}
	};
}

/**
 * Answers a stream on a resource found by looking up
 * resName using the class loader's resource lookup
 * algorithm. The default behavior is just to return null.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		InputStream
 *					a stream on the resource or null.
 * @param		resName	String
 *					the name of the resource to find.
 *
 * @see			Class#getResourceAsStream
 */
public InputStream getResourceAsStream (String resName) {
	URL url = getResource(resName);
	try {
		if (url != null) return url.openStream();
	} catch (IOException e){}
	return null;
}

/**
 * Returns the system class loader.  This is the parent
 * for new ClassLoader instances, and is typically the
 * class loader used to start the application.
 *
 * If a security manager is present, and the caller's
 * class loader is not null and the caller's class loader
 * is not the same as or an ancestor of the system class loader,
 * then this method calls the security manager's checkPermission
 * method with a RuntimePermission("getClassLoader") permission
 * to ensure it's ok to access the system class loader.
 * If not, a SecurityException will be thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.ClassLoader
 *					the system classLoader.
 * @exception	SecurityException
 *					if a security manager exists and it does not
 *					allow access to the system class loader.
 */
public static ClassLoader getSystemClassLoader () {
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
		ClassLoader callersClassLoader = com.ibm.oti.vm.VM.callerClassLoader();
		if (callersClassLoader != null &&
			callersClassLoader != applicationClassLoader &&
			!callersClassLoader.isAncestorOf(applicationClassLoader))
				security.checkPermission(
					RuntimePermission.permissionToGetClassLoader);
	}
	return applicationClassLoader;
}

/**
 * Answers an URL specifing a resource which can be found by
 * looking up resName using the system class loader's resource
 * lookup algorithm.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		URL
 *					a URL specifying a system resource or null.
 * @param		resName String
 *					the name of the resource to find.
 *
 * @see			Class#getResource
 */
public static URL getSystemResource(String resName) {
	return getSystemClassLoader().getResource(resName);
}

/**
 * Answers an Emuneration of URL containing all resources which can be
 * found by looking up resName using the system class loader's resource
 * lookup algorithm.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Enumeration
 *					an Enumeration of URL containing the system resources
 * @param		resName String
 *					the name of the resource to find.
 */
public static Enumeration getSystemResources(String resName) throws IOException {
	return getSystemClassLoader().getResources(resName);
}

/**
 * Answers a stream on a resource found by looking up
 * resName using the system class loader's resource lookup
 * algorithm. Basically, the contents of the java.class.path
 * are searched in order, looking for a path which matches
 * the specified resource.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a stream on the resource or null.
 * @param		resName		the name of the resource to find.
 *
 * @see			Class#getResourceAsStream
 */
public static InputStream getSystemResourceAsStream(String resName) {
	return getSystemClassLoader().getResourceAsStream(resName);
}

/**
 * Invoked by the Virtual Machine when resolving class references.
 * Equivalent to loadClass(className, false);
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.Class
 *					the Class object.
 * @param 		className String
 *					the name of the class to search for.
 * @exception	ClassNotFoundException
 *					If the class could not be found.
 */
public Class loadClass (String className) throws ClassNotFoundException {
	return loadClass(className, false);
}

/**
 * Loads the class with the specified name, optionally linking the class
 * after load.
 *
 * Steps are:
 *	1) Call findLoadedClass(className) to determine if class is loaded
 *	2) Call loadClass(className, resolveClass) on the parent loader.
 *	3) Call findClass(className) to find the class
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.Class
 *					the Class object.
 * @param 		className String
 *					the name of the class to search for.
 * @param 		resolveClass boolean
 *					indicates if class should be resolved after loading.
 * @exception	ClassNotFoundException
 *					If the class could not be found.
 */
protected synchronized Class loadClass(String className, boolean resolveClass) throws ClassNotFoundException {
	// Ask the VM to look in its cache.
	Class loadedClass = findLoadedClass(className);

	// search in parent if not found
	if (loadedClass == null) {
		try {
			if (parent == null) {
				loadedClass = systemClassLoader.loadClass(className);
			} else
				loadedClass = parent.loadClass(className, resolveClass);
		} catch (ClassNotFoundException e) {
			// don't do anything.  Catching this exception is the normal protocol for
			// parent classloaders telling use they couldn't find a class.
		}

		// not findLoadedClass or by parent.loadClass, try locally
		if (loadedClass == null) loadedClass = findClass(className);
	}

	// resolve if required
	if (resolveClass) resolveClass(loadedClass);
	return loadedClass;
}

/**
 * Forces a class to be linked (initialized).  If the class has
 * already been linked this operation has no effect.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		clazz Class
 *					the Class to link.
 * @exception	NullPointerException
 *					if clazz is null.
 *
 * @see			Class#getResource
 */
protected final void resolveClass(Class clazz) {
	if (clazz != null) clazz.verify();
	else throw new NullPointerException();
}

/**
 * Forces the parent of a classloader instance to be newParent
 *
 * @author		OTI
 * @version		initial
 *
 * @param		newParent ClassLoader
 *					the ClassLoader to make the parent.
 *
 * @see			ClassLoader#<clinit>
 */
private void setParent(ClassLoader newParent) {
	parent = newParent;
}

/**
 * Answers true if the receiver is a system class loader.
 * <p>
 * Note that this method has package visibility only. It is
 * defined here to avoid the security manager check in
 * getSystemClassLoader, which would be required to implement
 * this method anywhere else.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		boolean
 *					true if the receiver is a system class loader
 */
final boolean isSystemClassLoader() {
	if (this == systemClassLoader) return true;
	ClassLoader cl = applicationClassLoader;
	while (cl != null) {
		if (this == cl) return true;
		cl = cl.parent;
	}
	return false;
}

/**
 * Answers true if the receiver is ancestor of another class loader.
 * <p>
 * Note that this method has package visibility only. It is
 * defined here to avoid the security manager check in
 * getParent, which would be required to implement
 * this method anywhere else.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		child	ClassLoader, a child candidate
 *
 * @return		boolean
 *					true if the receiver is ancestor of the parameter
 */
final boolean isAncestorOf (ClassLoader child) {
	if (child == null) return false;
	if (this == systemClassLoader) return true;
	ClassLoader cl = child.parent;
	while (cl != null) {
		if (this == cl) return true;
		cl = cl.parent;
	}
	return false;
}

protected URL findResource (String resName) {
	return null;
}

protected Enumeration findResources (String resName) throws IOException {
	return new Vector().elements();
}

/**
 * Answers the absolute path of the file containing the library
 * associated with the given name, or null. If null is answered,
 * the system searches the directories specified by the system
 * property "java.library.path".
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *					the library file name or null.
 * @param		libName	String
 *					the name of the library to find.
 */
protected String findLibrary(String libName) {
	return null;
}

/**
 * Attempt to locate the requested package. If no package information
 * can be located, null is returned.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name		The name of the package to find
 * @return		The package requested, or null
 */
protected Package getPackage(String name) {
	if (this != systemClassLoader) {
		ClassLoader parent = this.parent;
		if (parent == null)
			parent = systemClassLoader;
		Package pkg = parent.getPackage(name);
		if (pkg != null) return pkg;
	}
	return (Package) packages.get(name);
}

/**
 * Return all the packages known to this class loader.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		All the packages known to this classloader
 */
protected Package[] getPackages() {

	Package[] ancestorsPackages = null;
	if (parent == null) {
		if (this != systemClassLoader)
			ancestorsPackages = systemClassLoader.getPackages();
	} else
		ancestorsPackages = parent.getPackages();

	int resultSize = packages.size();
	if (ancestorsPackages != null)
		resultSize += ancestorsPackages.length;
	Package[] result = new Package[resultSize];
	Enumeration myPkgs = packages.elements();
	int i = 0;
	if (ancestorsPackages != null)
		for (; i < ancestorsPackages.length; i++)
			result[i] = ancestorsPackages[i];
	while (myPkgs.hasMoreElements())
		result[i++] = (Package) myPkgs.nextElement();
	return result;
}

/**
 * Define a new Package using the specified information.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name		The name of the package
 * @param		specTitle	The title of the specification for the Package
 * @param		specVersion	The version of the specification for the Package
 * @param		specVendor	The vendor of the specification for the Package
 * @param		implTitle	The implementation title of the Package
 * @param		implVersion	The implementation version of the Package
 * @param		implVendor	The specification vendor of the Package
 * @return		The Package created
 *
 * @exception	IllegalArgumentException if the Package already exists
 */
protected Package definePackage(
	String name, String specTitle,
	String specVersion, String specVendor,
	String implTitle, String implVersion,
	String implVendor, URL sealBase)
	throws IllegalArgumentException
{
	synchronized(packages) {
		if (getPackage(name) == null) {
			Package newPackage = new Package(name, specTitle, specVersion,
				specVendor, implTitle, implVersion, implVendor, sealBase);
			packages.put(name, newPackage);
			return newPackage;
		} else throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0053", name));
	}
}

/**
 * Gets the signers of a class.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		c		The Class object
 * @return		signers	The signers for the class
 */
final Object[] getSigners(Class c) {
	if (classSigners == null) return null;
	try {
		Object result = classSigners.get(c);
		if (result != null)
			return (Object[]) result.clone();
	} catch (CloneNotSupportedException e) {}
	return null;
}

/**
 * Sets the signers of a class.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		c		The Class object
 * @param		signers	The signers for the class
 */
protected final void setSigners(Class c, Object[] signers) {
	if (c.getClassLoader() == this) {
		if (signers == null) {
			if (classSigners != null)
				classSigners.remove(c);
		} else {
			if (classSigners == null) classSigners = new Hashtable();
			classSigners.put(c, signers);
		}
	} else c.getClassLoader().setSigners(c, signers);

}

static ClassLoader getCallerClassLoader() {
	ClassLoader loader = com.ibm.oti.vm.VM.getStackClassLoader(2);
	if (loader == systemClassLoader) return null;
	return loader;
}

/**
 * Loads and links the library specified by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		libName		the name of the library to load
 * @param		loader		the classloader in which to load the library
 *
 * @exception	UnsatisfiedLinkError
 *							if the library could not be loaded
 * @exception	SecurityException
 *							if the library was not allowed to be loaded
 */
static synchronized void loadLibraryWithClassLoader(String libName, ClassLoader loader) {
	SecurityManager smngr = System.getSecurityManager();
	if (smngr != null)
		smngr.checkLink(libName);
	if (loader != null) {
		String realLibName = loader.findLibrary(libName);

		if (realLibName != null) {
			loadLibrary(realLibName, loader, null);
			return;
		}
	}
	loadLibrary(
		libName,
		loader,
		System.internalGetProperties().getProperty(
			loader == null
				? "com.ibm.oti.vm.bootstrap.library.path"
				: "java.library.path"
		)
	);
}

/**
 * Loads and links the library specified by the argument.
 * No security check is done.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		libName			the name of the library to load
 * @param		loader			the classloader in which to load the library
 * @param		libraryPath		the library path to search, or NULL'
 *
 * @exception	UnsatisfiedLinkError
 *							if the library could not be loaded
 */
static void loadLibrary(String libName, ClassLoader loader, String libraryPath) {
	byte[] message = ClassLoader.loadLibraryWithPath(com.ibm.oti.util.Util.getBytes(libName), loader, libraryPath == null ? null : com.ibm.oti.util.Util.getBytes(libraryPath));
	if (message != null)
		throw new UnsatisfiedLinkError(com.ibm.oti.util.Util.toString(message));
}

static native byte[] loadLibraryWithPath(byte[] libName, ClassLoader loader, byte[] libraryPath);

static void loadLibrary(Class caller, String name, boolean fullPath) {
	if (fullPath)
		loadLibrary(name, caller.getClassLoaderImpl(), null);
	else
		loadLibraryWithClassLoader(name, caller.getClassLoaderImpl());
}

}
