package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

import java.io.*;
import java.util.Properties;
import java.util.PropertyPermission;

/**
 * Class System provides a standard place for programs
 * to find system related information. All System API
 * is static.
 *
 * @author		OTI
 * @version		initial
 */

public final class System {

	// The standard input, output, and error streams.
	// Typically, these are connected to the shell which
	// ran the Java program.
	public static final InputStream in;
	public static final PrintStream out, err;

	//Get a ref to the Runtime instance for faster lookup
	private static final Runtime RUNTIME = Runtime.getRuntime();
	private static Properties systemProperties;

	private static SecurityManager security;

	// Initialize all the slots in System on first use.
	static {
		// Fill in the properties from the VM information.
		ensureProperties();
		// Set up standard in, out, and err.
		in = new BufferedInputStream(new FileInputStream(FileDescriptor.in));
		out = new PrintStream(new BufferedOutputStream(new FileOutputStream(FileDescriptor.out)), true);
		err = new PrintStream(new BufferedOutputStream(new FileOutputStream(FileDescriptor.err)), true);
	}

/**
 * Sets the value of the static slot "in" in the receiver
 * to the passed in argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		newIn InputStream
 *					the new value for in.
 */
public static void setIn(InputStream newIn) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToSetIO);
	setFieldImpl("in", newIn);
}

/**
 * Sets the value of the static slot "out" in the receiver
 * to the passed in argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		newOut PrintStream
 *					the new value for out.
 */
public static void setOut(java.io.PrintStream newOut) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToSetIO);
	setFieldImpl("out", newOut);
}

/**
 * Sets the value of the static slot "err" in the receiver
 * to the passed in argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		newErr PrintStream
 *					the new value for err.
 */
public static void setErr(java.io.PrintStream newErr) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToSetIO);
	setFieldImpl("err", newErr);
}

/**
 * Prevents this class from being instantiated.
 */
private System() {
}

/**
 * Copies the contents of <code>array1</code> starting at offset <code>start1</code>
 * into <code>array2</code> starting at offset <code>start2</code> for
 * <code>length</code> elements.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		array1
 *					the array to copy out of
 * @param		start1
 *					the starting index in array1
 * @param		array2
 *					the array to copy into
 * @param		start2
 *					the starting index in array2
 * @param		length
 *					the number of elements in the array to copy
 *
 */
public static native void arraycopy(Object array1, int start1, Object array2, int start2, int length);
/**
 * Answers the current time expressed as milliseconds since
 * the time 00:00:00 UTC on January 1, 1970.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		long		the time in milliseconds.
 */
public static native long currentTimeMillis();

/**
 * If systemProperties is unset, then create a
 * new one based on the values provided by the
 * virtual machine.
 *
 * @author		OTI
 * @version		initial
 */
private static void ensureProperties() {
	systemProperties = new Properties();

	systemProperties.put("file.encoding", getFileEncoding());

	systemProperties.put("java.version", "1.3.1");
	systemProperties.put("java.specification.version", "1.3");

	systemProperties.put("com.ibm.oti.configuration", "scar");

	String[] list = getPropertyList();
	for (int i=0; i<list.length; i+=2)
		systemProperties.put(list[i], list[i+1]);
}
/**
 * Causes the virtual machine to stop running, and the
 * program to exit. If runFinalizersOnExit(true) has been
 * invoked, then all finalizers will be run first.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		code		the return code.
 *
 * @exception	SecurityException
 *							if the running thread is not allowed
 *							to cause the vm to exit.
 *
 * @see			SecurityManager#checkExit
 */
public static void exit(int code) {
	RUNTIME.exit(code);
}

/**
 * Indicate to the virtual machine that it would be a
 * good time to collect available memory. Note that, this
 * is a hint only.
 *
 * @author		OTI
 * @version		initial
 */
public static void gc() {
	RUNTIME.gc();
}

/*
 * @deprecated Use System.getProperty()
 */
public static String getenv(String var) {
	throw new Error();
}

/**
 * Answers the system properties. Note that this is
 * not a copy, so that changes made to the returned
 * Properties object will be reflected in subsequent
 * calls to getProperty and getProperties.
 * <p>
 * Security managers should restrict access to this
 * API if possible.
 *
 * @author		OTI
 * @version		initial
 *
 * @return	Propties	the system properties
 */
public static Properties getProperties() {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPropertiesAccess();
	return systemProperties;
}

/**
 * Answers the system properties without any security
 * checks. This is used for access from within java.lang.
 *
 * @author		OTI
 * @version		initial
 *
 * @return	Properties	the system properties
 */
static Properties internalGetProperties() {
	return systemProperties;
}
/**
 * Answers the value of a particular system property.
 * Answers null if no such property exists,
 * <p>
 * The properties currently provided by the virtual
 * machine are:
 * <pre>
 *     java.vendor.url
 *     java.class.path
 *     user.home
 *     java.class.version
 *     os.version
 *     java.vendor
 *     user.dir
 *     user.timezone
 *     path.separator
 *     os.name
 *     os.arch
 *     line.separator
 *     file.separator
 *     user.name
 *     java.version
 *     java.home
 * </pre>
 *
 * @author		OTI
 * @version		initial
 *
 * @param	prop	the system property to look up
 * @return	the value of the specified system property, or
 *			null if the property doesn't exist
 */
public static String getProperty(String prop) {
	return getProperty(prop, null);
}

/**
 * Answers the value of a particular system property.
 * If no such property is found, answers the defaultValue.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	prop	the system property to look up
 * @return	the value of the specified system property, or
 *			defaultValue if the property doesn't exist
 */
public static String getProperty(String prop, String defaultValue) {
	if (prop.length() == 0) throw new IllegalArgumentException();
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPropertyAccess(prop);
	return systemProperties.getProperty(prop, defaultValue);
}

/**
 * Sets the value of a particular system property.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	prop	the system property to change
 * @param	value	the value to associate with prop
 * @return	the old value of the property, or null
 */
public static String setProperty(String prop, String value) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			new PropertyPermission(prop, "write"));
	return (String)systemProperties.setProperty(prop, value);
}

/**
 * Answers an array of Strings containing key..value pairs
 * (in consecutive array elements) which represent the
 * starting values for the system properties as provided
 * by the virtual machine.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the default values for the system properties.
 */
private static native String [] getPropertyList();

/**
 * Return the system file.encoding.
 *
 * @author		OTI
 * @version		initial
 */
private static native String getFileEncoding();

/**
 * Answers the active security manager.
 *
 * @author		OTI
 * @version		initial
 *
 * @return	SecurityManager
 *				the system security manager object.
 */
public static SecurityManager getSecurityManager() {
	return security;
}

/**
 * Answers an integer hash code for the parameter.
 * The hash code returned is the same one that would
 * be returned by java.lang.Object.hashCode(), whether
 * or not the object's class has overridden hashCode().
 * The hash code for null is 0.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		anObject	the object
 * @return		int			the hash code for the object
 *
 * @see			java.lang.Object#hashCode
 */
public static native int identityHashCode(Object anObject);

public static void load(String pathName) {
	SecurityManager smngr = System.getSecurityManager();
	if (smngr != null)
		smngr.checkLink(pathName);
	ClassLoader.loadLibrary(pathName, com.ibm.oti.vm.VM.callerClassLoader(), null);
}

/**
 * Loads and links the library specified by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		libName		the name of the library to load
 *
 * @exception	UnsatisfiedLinkError
 *							if the library could not be loaded
 * @exception	SecurityException
 *							if the library was not allowed to be loaded
 */
public static void loadLibrary(String libName) {
	ClassLoader.loadLibraryWithClassLoader(libName, com.ibm.oti.vm.VM.callerClassLoader());
}

/**
 * Provides a hint to the virtual machine that it would
 * be useful to attempt to perform any outstanding
 * object finalizations.
 *
 * @author		OTI
 * @version		initial
 */
public static void runFinalization() {
	RUNTIME.runFinalization();
}

/**
 * Ensure that, when the virtual machine is about to exit,
 * all objects are finalized. Note that all finalization
 * which occurs when the system is exiting is performed
 * after all running threads have been terminated.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		flag 	true means finalize all on exit.
 *
 * @deprecated This method is unsafe.
 */
public static void runFinalizersOnExit(boolean flag) {
	RUNTIME.runFinalizersOnExit(flag);
}

/**
 * Answers the system properties. Note that the object
 * which is passed in not copied, so that subsequent
 * changes made to the object will be reflected in
 * calls to getProperty and getProperties.
 * <p>
 * Security managers should restrict access to this
 * API if possible.
 *
 * @author		OTI
 * @version		initial
 */
public static void setProperties(Properties p) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPropertiesAccess();
	if (p == null) ensureProperties();
	else systemProperties = p;
}

/**
 * Sets the active security manager. Note that once
 * the security manager has been set, it can not be
 * changed. Attempts to do so will cause a security
 * exception.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	SecurityException
 *					if the security manager has already been set.
 */
public static void setSecurityManager(final SecurityManager s) {
	try {
		// Preload classes used for checkPackageAccess(),
		// otherwise we could go recursive
		s.checkPackageAccess("java.lang");
	} catch (Exception e) {}
	if (security != null) {
		security.checkPermission(
			RuntimePermission.permissionToSetSecurityManager);
	}
	security = s;
}

/**
 * Answers the platform specific file name format for the shared
 * library named by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *					the platform specific filename for the library
 * @param		userLibName String
 *					the name of the library to look up.
 */
public static native String mapLibraryName(String userLibName);

/**
 * Sets the value of the named static field in the receiver
 * to the passed in argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		fieldName the name of the field to set, one of in, out, or err
 * @param		stream the new value of the field
 */
private static native void setFieldImpl(String fieldName, Object stream);

static Class getCallerClass() {
	return com.ibm.oti.vm.VM.getStackClasses(2, false)[1];
}
}
