
package java.lang;

import java.io.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

/**
 * This class is the superclass of all classes which
 * can be thrown by the virtual machine. The two direct
 * subclasses represent recoverable exceptions (Exception)
 * and unrecoverable errors (Error). This class provides
 * common methods for accessing a string message which
 * provides extra information about the circumstances in
 * which the throwable was created, and for filling in a
 * walkback (i.e. a record of the call stack at a
 * particular point in time) which can be printed later.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Error
 * @see			Exception
 * @see			RuntimeException
 */
public class Throwable implements java.io.Serializable {

	static final long serialVersionUID = -3042686055658047285L;

	/**
	 * The message provided when the exception was created.
	 */
	private String detailMessage;

	/**
	 * An object which describes the walkback.
	 */
	private transient Object walkback;

/**
 * Constructs a new instance of this class with its
 * walkback filled in.
 *
 * @author		OTI
 * @version		initial
 */
public Throwable () {
	super ();
	fillInStackTrace();
}

/**
 * Constructs a new instance of this class with its
 * walkback and message filled in.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		detailMessage String
 *				The detail message for the exception.
 */
public Throwable (String detailMessage) {
	this ();
	this.detailMessage = detailMessage;
}

/**
 * Record in the receiver a walkback from the point
 * where this message was sent. The message is
 * public so that code which catches a throwable and
 * then <em>re-throws</em> it can adjust the walkback
 * to represent the location where the exception was
 * re-thrown.
 *
 * @author		OTI
 * @version		initial
 */
public native Throwable fillInStackTrace();

/**
 * Answers the extra information message which was provided
 * when the throwable was created. If no message was provided
 * at creation time, then answer null.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *				The receiver's message.
 */
public String getMessage() {
	return detailMessage;
}

/**
 * Answers the extra information message which was provided
 * when the throwable was created. If no message was provided
 * at creation time, then answer null. Subclasses may override
 * this method to answer localized text for the message.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *				The receiver's message.
 */
public String getLocalizedMessage() {
	return getMessage();
}

/**
 * Answers an array of StackTraceElement. Each StackTraceElement represents
 * a entry on the stack.
 *
 * @author		OTI
 * @version		initial
 */
private native StackTraceElement[] getStackTraceImpl();

/**
 * Outputs a printable representation of the receiver's
 * walkback on the System.err stream.
 *
 * @author		OTI
 * @version		initial
 */
public void printStackTrace () {
	printStackTrace(System.err);
}

/**
 * Outputs a printable representation of the receiver's
 * walkback on the stream specified by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		err PrintStream
 *				The stream to write the walkback on.
 */
public void printStackTrace (PrintStream err) {
	err.println(toString());
	StackTraceElement[] stack = getStackTraceImpl();
	for (int i=0; i<stack.length; i++)
		err.println(stack[i]);

}

/**
 * Outputs a printable representation of the receiver's
 * walkback on the writer specified by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		err PrintWriter
 *				The writer to write the walkback on.
 */
public void printStackTrace(PrintWriter err) {
	err.println(toString());
	StackTraceElement[] stack = getStackTraceImpl();
	for (int i=0; i<stack.length; i++)
		err.println(stack[i]);

}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *				a printable representation for the receiver.
 */
public String toString () {
	String msg = getMessage();
	if (msg == null)
		return this.getClass().getName();
	else
		return this.getClass().getName() + ": " + msg;

}

}
