package java.lang.ref;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

/**
 * ReferenceQueue is the container on which reference objects
 * are enqueued when their reachability type is detected for
 * the referent.
 *
 * @author		OTI
 * @version		initial
 * @since		JDK1.2
 */
public class ReferenceQueue extends Object {
	private Reference[] references;
	private int head, tail;
	private boolean empty;

	static private final int DEFAULT_QUEUE_SIZE = 128;

/**
 * Returns the next available reference from the queue
 * if one is enqueued, null otherwise.  Does not wait
 * for a reference to become available.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Reference
 *					next available Reference or NULL.
 */
public Reference poll () {
	Reference ref;

	synchronized(this) {
		if(empty) {
			return null;
		}
		ref = references[head++];
		ref.dequeue();
		if(head == references.length) {
			head = 0;
		}
		if(head == tail) {
			empty = true;
		}
	}
	return ref;
}

/**
 * Return the next available enqueued reference on the queue, blocking
 * indefinately until one is available.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Reference
 *					a Reference object if one is available,
 *                  null otherwise.
 * @exception	InterruptedException
 *					to interrupt the wait.
 */
public Reference remove() throws InterruptedException {
	return remove(0L);
}

/**
 * Return the next available enqueued reference on the queue, blocking
 * up to the time given until one is available.  Return null if no
 * reference became available.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		timeout
 *					maximum time spent waiting for a reference object
 *					to become available.
 * @return		Reference
 *					a Reference object if one is available,
 *                  null otherwise.
 * @exception	IllegalArgumentException
 *					if the wait period is negative.
 * 				InterruptedException
 *					to interrupt the wait.
 */
public Reference remove(long timeout) throws IllegalArgumentException, InterruptedException {
	if (timeout < 0) throw new IllegalArgumentException();

	Reference ref;
	synchronized(this) {
		if(empty) {
			wait(timeout);
			if(empty) return null;
		}
		ref = references[head++];
		ref.dequeue();
		if(head == references.length) {
			head = 0;
		}
		if(head == tail) {
			empty = true;
		} else {
			notifyAll();
		}
	}
	return ref;
}

/**
 * Enqueue the reference object on the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		reference
 *					reference object to be enqueued.
 * @return		boolean
 *					true if reference is enqueued.
 *					false if reference failed to enqueue.
 */
boolean enqueue (Reference reference) {
	synchronized(this) {
		if(!empty && head == tail) {
			/* Queue is full - grow */
			int newQueueSize = (int)(references.length * 1.10);
			Reference newQueue[] = new Reference[newQueueSize];
			System.arraycopy(references, head, newQueue, 0, references.length - head);
			if(tail > 0) {
				System.arraycopy(references, 0, newQueue, references.length - head, tail);
			}
			head = 0;
			tail = references.length;
			references = newQueue;
		}
		references[tail++] = reference;
		if(tail == references.length) {
			tail = 0;
		}
		empty = false;
		notifyAll();
	}
	return true;
}

/**
 * Constructs a new instance of this class.
 *
 * @author		OTI
 * @version		initial
 */
public ReferenceQueue() {
	references = new Reference[DEFAULT_QUEUE_SIZE];
	head = 0;
	tail = 0;
	empty = true;
}

}
