package java.lang.reflect;

/*
	AccessibleObject definition for BB/J9.
*/

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

/**
 * This class is the superclass of all member reflect classes
 * (Field, Constructor, Method). AccessibleObject provides
 * the ability to toggle access checks for these objects.
 * By default accessing a member (for example, setting a field
 * or invoking a method) checks the validity of the access
 * (for example, invoking a private method from outside the
 * defining class is prohibited) and throws IllegalAccessException
 * if the operation is not permitted. If the accessible flag is set
 * to true, these checks are omitted. This allows privileged
 * applications such as Java Object Serialization, inspectors, and
 * debuggers to have complete access to objects.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Field
 * @see			Constructor
 * @see			Method
 * @see			ReflectPermission
 */
public class AccessibleObject {
	private static final ReflectPermission suppressAccessChecksPermission =
		new ReflectPermission("suppressAccessChecks");

	static final Object[] emptyArgs = new Object[0];

	private int vm1;

/**
 * AccessibleObject constructor.
 * AccessibleObjects can only be created by the
 * Virtual Machine.
 */
protected AccessibleObject() {
	super();
}
/**
 * Returns the value of the accessible flag.
 * This is false if access checks are performed,
 * true if they are skipped.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		the value of the accessible flag
 */
public boolean isAccessible() {
	return getAccessibleImpl();
}
/**
 * Attempts to set the value of the accessible flag
 * for all the objects in the array provided. Only
 * one security check is performed.
 *
 * Setting this flag to false will enable access checks,
 * setting to true will disable them.
 *
 * If there is a security manager, checkPermission is
 * called with a ReflectPermission("suppressAccessChecks").
 *
 * @author		OTI
 * @version		initial
 *
 * @param	array	the accessible objects
 * @param	flag	the new value for the accessible flag
 *
 * @see		setAccessible(boolean)
 * @see		ReflectPermissions
 */
public static void setAccessible(AccessibleObject[] objects, boolean flag) throws SecurityException {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(suppressAccessChecksPermission);

	for(int i = 0; i < objects.length; i++)
		objects[i].setAccessibleImpl(flag);
}
/**
 * Attempts to set the value of the accessible flag.
 * Setting this flag to false will enable access checks,
 * setting to true will disable them.
 *
 * If there is a security manager, checkPermission is
 * called with a ReflectPermission("suppressAccessChecks").
 *
 * @author		OTI
 * @version		initial
 *
 * @param	flag	the new value for the accessible flag
 *
 * @see		ReflectPermissions
 */
public void setAccessible(boolean flag) throws SecurityException
{
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(suppressAccessChecksPermission);
	setAccessibleImpl(flag);
}

static Object[] marshallArguments(Class[] parameterTypes, Object[] args) throws IllegalArgumentException {
	if(parameterTypes.length != args.length) {
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01b3"));
	}

	Object[] marshalledArgs = args;
	for (int i = args.length - 1; i >= 0; i--) {
		Object arg = args[i];
		Class type = parameterTypes[i];

		Object marshalledArg = marshallArgument(arg, type);

		if (marshalledArg != arg) {
			if (marshalledArgs == args) marshalledArgs = (Object[])args.clone();
			marshalledArgs[i] = marshalledArg;
		}
	}

	return marshalledArgs;
}

private static Object marshallArgument(Object arg, Class type) throws IllegalArgumentException {
	if (type.isPrimitive()) {
		if (arg instanceof Boolean) {
			if (type == Boolean.TYPE) return arg;
		} else if (arg instanceof Character) {
			if (type == Character.TYPE) return arg;
			char charValue = ((Character)arg).charValue();
			if (type == Integer.TYPE) {
				return new Integer((int)charValue);
			} else if (type == Long.TYPE) {
				return new Long((long)charValue);
			} else if (type == Double.TYPE) {
				return new Double((double)charValue);
			} else if (type == Float.TYPE) {
				return new Float((float)charValue);
			}
		} else if (arg instanceof Byte) {
			if (type == Byte.TYPE) return arg;
			if (type == Short.TYPE) return arg;
			if (type == Integer.TYPE) return arg;
			Number result = coerce( (Number)arg, type);
			if (result != null) return result;
		} else if (arg instanceof Short) {
			if (type == Short.TYPE) return arg;
			if (type == Integer.TYPE) return arg;
			Number result = coerce( (Number)arg, type);
			if (result != null) return result;
		} else if (arg instanceof Integer) {
			if (type == Integer.TYPE) return arg;
			Number result = coerce( (Number)arg, type);
			if (result != null) return result;
		} else if (arg instanceof Long) {
			if (type == Long.TYPE) return arg;
			Number result = coerce( (Number)arg, type);
			if (result != null) return result;
		} else if (arg instanceof Float) {
			if (type == Float.TYPE) return arg;
			if (type == Double.TYPE) {
				return new Double( ((Number)arg).doubleValue());
			}
		} else if (arg instanceof Double) {
			if (type == Double.TYPE) return arg;
		} else if (arg == null) {
			throw new NullPointerException();
		}
	} else {
		if (arg == null || type.isInstance(arg)) return arg;
	}

	throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01b4", arg.getClass().getName(), type.getName()));
}

private static Number coerce(Number source, Class primitiveType) {
	if (primitiveType == Double.TYPE) {
		return new Double(source.doubleValue());
	} else if (primitiveType == Float.TYPE) {
		return new Float(source.floatValue());
	} else if (primitiveType == Long.TYPE) {
		return new Long(source.longValue());
	}
	return null;
}

void invokeV(Object receiver, Object args[]) throws InvocationTargetException {
	try {
		invokeImpl(receiver, args, (Void) null);
	} catch (Throwable t) {
		throw new InvocationTargetException(t);
	}
}

Object invokeL(Object receiver, Object args[]) throws InvocationTargetException {
	try {
		return invokeImpl(receiver, args, (Object) null);
	} catch (Throwable t) {
		throw new InvocationTargetException(t);
	}
}

int invokeI(Object receiver, Object args[]) throws InvocationTargetException {
	try {
		return invokeImpl(receiver, args, (Integer) null);
	} catch (Throwable t) {
		throw new InvocationTargetException(t);
	}
}

long invokeJ(Object receiver, Object args[]) throws InvocationTargetException {
	try {
		return invokeImpl(receiver, args, (Long) null);
	} catch (Throwable t) {
		throw new InvocationTargetException(t);
	}
}

float invokeF(Object receiver, Object args[]) throws InvocationTargetException {
	try {
		return invokeImpl(receiver, args, (Float) null);
	} catch (Throwable t) {
		throw new InvocationTargetException(t);
	}
}

double invokeD(Object receiver, Object args[]) throws InvocationTargetException {
	try {
		return invokeImpl(receiver, args, (Double) null);
	} catch (Throwable t) {
		throw new InvocationTargetException(t);
	}
}

private native boolean getAccessibleImpl();
private native void setAccessibleImpl(boolean flag);
native Class[] getParameterTypesImpl();
native int getModifiers();
native Class[] getExceptionTypesImpl();
native String getSignature();
native boolean checkAccessibility(Class senderClass, Object receiver);
private native Object invokeImpl(Object receiver, Object args[], Object dummy);
private native double invokeImpl(Object receiver, Object args[], Double dummy);
private native float invokeImpl(Object receiver, Object args[], Float dummy);
private native int invokeImpl(Object receiver, Object args[], Integer dummy);
private native long invokeImpl(Object receiver, Object args[], Long dummy);
private native void invokeImpl(Object receiver, Object args[], Void dummy);
static native void initializeClass(Class clazz);
}
