package java.lang.reflect;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

/**
 * This class models a constructor. Information about the constructor
 * can be accessed, and the constructor can be invoked dynamically.
 *
 * @author		OTI
 * @version		initial
 */
public final class Constructor extends AccessibleObject implements Member
{
	private Class declaringClass;
	private Class[] parameterTypes;
	private Class[] exceptionTypes;
	private Class returnType;
	private String name;
	private int vm1;

/**
 * Prevent this class from being instantiated.
 */
private Constructor() {}

/**
 * Compares the specified object to this Constructor and answer if they
 * are equal. The object must be an instance of Constructor with the same
 * defining class and parameter types.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		object	the object to compare
 * @return		true if the specified object is equal to this Constructor, false otherwise
 *
 * @see			#hashCode
 */
public boolean equals(Object object)
{
	Class[] array1, array2;
	Constructor method;
	int i;

	if(this == object) return true;
	if(!(object instanceof Constructor)) return false;
	method = (Constructor)object;
	if(getDeclaringClass() != method.getDeclaringClass()) return false;

	array1 = getParameterTypes();
	array2 = method.getParameterTypes();
	if(array1.length != array2.length) return false;
	for(i = 0; i < array1.length; i++)
		if(array1[i] != array2[i]) return false;

	return true;
}

/**
 * Return the java.lang.Class associated with the class that defined
 * this constructor.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the declaring class
 */
public Class getDeclaringClass()
{
	return declaringClass;
}

/**
 * Return an array of the java.lang.Class objects associated with the
 * exceptions declared to be thrown by this constructor. If the
 * constructor was not declared to throw any exceptions, the array
 * returned will be empty.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the declared exception classes
 */
public Class[] getExceptionTypes()
{
	if(exceptionTypes == null) getExceptionTypesImpl();
	return (Class[])exceptionTypes.clone();
}

/**
 * Return the modifiers for the modelled constructor.
 * The Modifier class should be used to decode the result.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the modifiers
 * @see			java.lang.reflect.Modifier
 */
public int getModifiers() {
	return super.getModifiers();
}

/**
 * Return the name of the modelled constructor.
 * This is the name of the declaring class.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the name
 */
public String getName()
{
	if(name != null) return name;
	name = getDeclaringClass().getName();
	return name;
}

/**
 * Return an array of the java.lang.Class objects associated with the
 * parameter types of this constructor. If the constructor was declared
 * with no parameters, the array returned will be empty.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the parameter types
 */
public Class[] getParameterTypes()
{
	if(parameterTypes == null) getParameterTypesImpl();
	return (Class[])parameterTypes.clone();
}

/**
 * Answers an integer hash code for the receiver. Objects which are
 * equal answer the same value for this method.
 *
 * The hash code for a Constructor is the hash code of the declaring
 * class' name.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode()
{
	return getName().hashCode();
}

/**
 * Return a new instance of the declaring class, initialized by dynamically invoking
 * the modelled constructor. This reproduces the effect of
 *		<code>new declaringClass(arg1, arg2, ... , argN)</code>
 *
 * This method performs the following:
 * <ul>
 * <li>A new instance of the declaring class is created. If the declaring class cannot be instantiated
 *	(i.e. abstract class, an interface, an array type, or a base type) then an InstantiationException
 *	is thrown.</li>
 * <li>If this Constructor object is enforcing access control (see AccessibleObject) and the modelled
 *	constructor is not accessible from the current context, an IllegalAccessException is thrown.</li>
 * <li>If the number of arguments passed and the number of parameters do not match, an
 *	IllegalArgumentException is thrown.</li>
 * <li>For each argument passed:
 * <ul>
 * <li>If the corresponding parameter type is a base type, the argument is unwrapped. If the unwrapping
 *	fails, an IllegalArgumentException is thrown.</li>
 * <li>If the resulting argument cannot be converted to the parameter type via a widening conversion,
 *	an IllegalArgumentException is thrown.</li>
 * </ul>
 * <li>The modelled constructor is then invoked. If an exception is thrown during the invocation,
 *	it is caught and wrapped in an InvocationTargetException. This exception is then thrown. If
 *	the invocation completes normally, the newly initialized object is returned.
 * </ul>
 *
 * @author		OTI
 * @version		initial
 *
 * @param		args	the arguments to the constructor
 * @return		the new, initialized, object
 * @exception	java.lang.InstantiationException	if the class cannot be instantiated
 * @exception	java.lang.IllegalAccessException	if the modelled constructor is not accessible
 * @exception	java.lang.IllegalArgumentException	if an incorrect number of arguments are passed, or an argument could not be converted by a widening conversion
 * @exception	java.lang.reflect.InvocationTargetException	if an exception was thrown by the invoked constructor
 * @see			java.lang.reflect.AccessibleObject
 */
public Object newInstance(Object args[])
	throws InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException
{
	if (args == null) args = emptyArgs;

	/* according to the spec we're supposed to check the number
	 * of arguments before we instantiate the class */

	Object newInstance = newInstanceImpl();
	if (newInstance == null) throw new InstantiationException();

	/* check accessibility */
	if (!isAccessible()) {
		Class callerClass = com.ibm.oti.vm.VM.getStackClass(1);
		if (!checkAccessibility(callerClass, newInstance)) {
			throw new IllegalAccessException();
		}
	}

	/* ensure that the parameter types are loaded */
	if(parameterTypes == null) getParameterTypesImpl();
	args = marshallArguments(parameterTypes, args);

	initializeClass(getDeclaringClass());

	invokeV(newInstance, args);

	return newInstance;
}

/* creates a new, uninitialized object */
private native Object newInstanceImpl();

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * The format of the string is
 *		modifiers (if any)
 *		declaring class name
 *		'('
 *		parameter types, separated by ','
 *		')'
 * If the constructor throws exceptions,
 *		' throws '
 *		exception types, separated by ','
 *
 * For example:
 *		<code>public String(byte[],String) throws UnsupportedEncodingException</code>
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver
 */
public String toString()
{
	StringBuffer buf;
	String mods;
	Class[] types;
	Class current;
	int i, arity;

	buf = new StringBuffer();
	mods = Modifier.toString(getModifiers());
	if(mods.length() != 0)
	{
		buf.append(mods);
		buf.append(" ");
	}
	buf.append(getName());
	buf.append("(");
	types = getParameterTypes();
	for(i = 0; i < types.length; i++)
	{
		current = types[i];
		arity = 0;
		while(current.isArray())
		{
			current = current.getComponentType();
			arity++;
		}
		buf.append(current.getName());
		for(;arity > 0; arity--) buf.append("[]");
		if(i != (types.length - 1))
			buf.append(",");
	}
	buf.append(")");
	types = getExceptionTypes();
	if(types.length > 0)
	{
		buf.append(" throws ");
		for(i = 0; i < types.length; i++)
		{
			current = types[i];
			buf.append(current.getName());
			if(i != (types.length - 1))
				buf.append(",");
		}
	}

	return buf.toString();
}
}
