package java.lang.reflect;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

/**
 * This class models a field. Information about the field
 * can be accessed, and the field's value can be accessed dynamically.
 *
 * @author		OTI
 * @version		initial
 */
public final class Field extends AccessibleObject implements Member
{
	private Class declaringClass;
	private Class type;
	private String name;
	private int vm1;
	private int vm2;

/**
 * Prevent this class from being instantiated.
 */
private Field() {}

/**
 * Compares the specified object to this Field and answer if they
 * are equal. The object must be an instance of Field with the same
 * defining class and name.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		object	the object to compare
 * @return		true if the specified object is equal to this Field, false otherwise
 *
 * @see			#hashCode
 */
public boolean equals(Object object)
{
	Field field;

	if(this == object) return true;
	if(!(object instanceof Field)) return false;
	field = (Field)object;
	if(!getName().equals(field.getName())) return false;
	if(getDeclaringClass() != field.getDeclaringClass()) return false;
	if(getType() != field.getType()) return false;
	return true;
}

/**
 * Return the value of the field in the specified object.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * The value of the field is returned. If the type of this field is a base type, the field
 * value is automatically wrapped.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value, possibly wrapped
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native Object get(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the value of the field in the specified object as a boolean.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native boolean getBoolean(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the value of the field in the specified object as a byte.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native byte getByte(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the value of the field in the specified object as a char.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native char getChar(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the java.lang.Class associated with the class that defined
 * this field.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the declaring class
 */
public Class getDeclaringClass()
{
	return declaringClass;
}

/**
 * Return the value of the field in the specified object as a double.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native double getDouble(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the value of the field in the specified object as a float.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native float getFloat(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the value of the field in the specified object as an int.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native int getInt(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the value of the field in the specified object as a long.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native long getLong(Object object) throws IllegalAccessException, IllegalArgumentException;

/**
 * Return the modifiers for the modelled field.
 * The Modifier class should be used to decode the result.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the modifiers
 * @see			java.lang.reflect.Modifier
 */
public native int getModifiers();

/**
 * Return the name of the modelled field.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the name
 */
public String getName()
{
	if(name != null) return name;
	return getNameImpl();
}

private native String getNameImpl();

/**
 * Return the value of the field in the specified object as a short.
 * This reproduces the effect of <code>object.fieldName</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @return	the field value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native short getShort(Object object) throws IllegalAccessException, IllegalArgumentException;

native String getSignature();

/**
 * Return the java.lang.Class associated with the type of
 * this field.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the type
 */
public Class getType()
{
	if(type != null) return type;
	return getTypeImpl();
}

private native Class getTypeImpl();

/**
 * Answers an integer hash code for the receiver. Objects which are
 * equal answer the same value for this method.
 *
 * The hash code for a Field is the hash code of the field's name.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode()
{
	return getName().hashCode();
}

/**
 * Set the value of the field in the specified object to the boolean value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the field type is a base type, the value is automatically unwrapped. If the unwrap fails,
 * an IllegalArgumentException is thrown. If the value cannot be converted to the field type via
 * a widening conversion, an IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void set(Object object, Object value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the boolean value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setBoolean(Object object, boolean value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the byte value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setByte(Object object, byte value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the char value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setChar(Object object, char value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the double value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setDouble(Object object, double value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the float value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setFloat(Object object, float value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the int value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setInt(Object object, int value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the long value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setLong(Object object, long value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Set the value of the field in the specified object to the short value.
 * This reproduces the effect of <code>object.fieldName = value</code>
 *
 * If the modelled field is static, the object argument is ignored.
 * Otherwise, if the object is null, a NullPointerException is thrown.
 * If the object is not an instance of the declaring class of the method, an
 * IllegalArgumentException is thrown.
 *
 * If this Field object is enforcing access control (see AccessibleObject) and the modelled
 * field is not accessible from the current context, an IllegalAccessException is thrown.
 *
 * If the value cannot be converted to the field type via a widening conversion, an
 * IllegalArgumentException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	object	the object to access
 * @param	value	the new value
 * @exception	java.lang.NullPointerException
 *					if the object is null and the field is non-static
 * @exception	java.lang.IllegalArgumentException
 *					if the object is not compatible with the declaring class, or the value could not be converted to the field type via a widening conversion
 * @exception	java.lang.IllegalAccessException
 *					if modelled field is not accessible
 */
public native void setShort(Object object, short value) throws IllegalAccessException, IllegalArgumentException;

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * The format of the string is
 *		modifiers (if any)
 *		return type
 *		declaring class name
 *		'.'
 *		field name
 *
 * For example:
 *		<code>public static java.io.InputStream java.lang.System.in</code>
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver
 */
public String toString()
{
	StringBuffer buf;
	Class current;
	int arity = 0;

	buf = new StringBuffer();
	String modifier = Modifier.toString(getModifiers());
	if (modifier.length() > 0) {
		buf.append(modifier);
		buf.append(" ");
	}

	current = getType();
	while(current.isArray())
	{
		current = current.getComponentType();
		arity++;
	}
	buf.append(current.getName());
	for(;arity > 0; arity--) buf.append("[]");

	buf.append(" ");
	buf.append(getDeclaringClass().getName());
	buf.append(".");
	buf.append(getName());
	return buf.toString();
}
}
