package java.lang.reflect;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

/**
 * This class models a method. Information about the method
 * can be accessed, and the method can be invoked dynamically.
 *
 * @author		OTI
 * @version		initial
 */
public final class Method extends AccessibleObject implements Member
{
	private Class declaringClass;
	private Class[] parameterTypes;
	private Class[] exceptionTypes;
	private Class returnType;
	private String name;
	private int vm1;

	/**
 * Prevent this class from being instantiated.
 */
private Method() {}

/**
 * Compares the specified object to this Method and answer if they
 * are equal. The object must be an instance of Method with the same
 * defining class and parameter types.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		object	the object to compare
 * @return		true if the specified object is equal to this Method, false otherwise
 *
 * @see			#hashCode
 */
public boolean equals(Object object)
{
	Class[] array1, array2;
	Method method;
	int i;

	if(this == object) return true;
	if(!(object instanceof Method)) return false;
	method = (Method)object;
	if(!getName().equals(method.getName())) return false;
	if(getDeclaringClass() != method.getDeclaringClass()) return false;

	array1 = getParameterTypes();
	array2 = method.getParameterTypes();
	if(array1.length != array2.length) return false;
	for(i = 0; i < array1.length; i++)
		if(array1[i] != array2[i]) return false;

	return true;
}

/**
 * Return the java.lang.Class associated with the class that defined
 * this constructor.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the declaring class
 */
public Class getDeclaringClass()
{
	return declaringClass;
}

/**
 * Return an array of the java.lang.Class objects associated with the
 * exceptions declared to be thrown by this method. If the method was
 * not declared to throw any exceptions, the array returned will be
 * empty.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the declared exception classes
 */
public Class[] getExceptionTypes()
{
	if(exceptionTypes == null) getExceptionTypesImpl();
	return (Class[])exceptionTypes.clone();
}

/**
 * Return the modifiers for the modelled constructor.
 * The Modifier class should be used to decode the result.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the modifiers
 * @see			java.lang.reflect.Modifier
 */
public int getModifiers() {
	return super.getModifiers();
}

/**
 * Return the name of the modelled method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the name
 */
public String getName()
{
	if(name != null) return name;
	return getNameImpl();
}

private native String getNameImpl();

/**
 * Return an array of the java.lang.Class objects associated with the
 * parameter types of this method. If the method was declared with no
 * parameters, the array returned will be empty.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the parameter types
 */
public Class[] getParameterTypes()
{
	if(parameterTypes == null) getParameterTypesImpl();
	return (Class[])parameterTypes.clone();
}

/**
 * Return the java.lang.Class associated with the return type of
 * this method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the return type
 */
public Class getReturnType()
{
	if(returnType != null) return returnType;
	return getReturnTypeImpl();
}

private native Class getReturnTypeImpl();

/**
 * Answers an integer hash code for the receiver. Objects which are
 * equal answer the same value for this method.
 *
 * The hash code for a Method is the hash code of the method's name.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode()
{
	return getName().hashCode();
}

/**
 * Return the result of dynamically invoking the modelled method.
 * This reproduces the effect of
 *		<code>receiver.methodName(arg1, arg2, ... , argN)</code>
 *
 * This method performs the following:
 * <ul>
 * <li>If the modelled method is static, the receiver argument is ignored.</li>
 * <li>Otherwise, if the receiver is null, a NullPointerException is thrown.</li>
 * If the receiver is not an instance of the declaring class of the method, an
 *	IllegalArgumentException is thrown.
 * <li>If this Method object is enforcing access control (see AccessibleObject) and the modelled
 *	method is not accessible from the current context, an IllegalAccessException is thrown.</li>
 * <li>If the number of arguments passed and the number of parameters do not match, an
 *	IllegalArgumentException is thrown.</li>
 * <li>For each argument passed:
 * <ul>
 * <li>If the corresponding parameter type is a base type, the argument is unwrapped. If the unwrapping
 *	fails, an IllegalArgumentException is thrown.</li>
 * <li>If the resulting argument cannot be converted to the parameter type via a widening conversion,
 *	an IllegalArgumentException is thrown.</li>
 * </ul>
 * <li> If the modelled method is static, it is invoked directly. If it is non-static, the modelled
 *	method and the receiver are then used to perform a standard dynamic method lookup. The resulting
 *	method is then invoked.</li>
 * <li>If an exception is thrown during the invocation it is caught and wrapped in an
 *	InvocationTargetException. This exception is then thrown.</li>
 * <li>If the invocation completes normally, the return value is itself returned. If the method
 *	is declared to return a base type, the return value is first wrapped. If the return type is
 *	void, null is returned.</li>
 * </ul>
 *
 * @author		OTI
 * @version		initial
 *
 * @param		args	the arguments to the constructor
 * @return		the new, initialized, object
 * @exception	java.lang.NullPointerException		if the receiver is null for a non-static method
 * @exception	java.lang.IllegalAccessException	if the modelled method is not accessible
 * @exception	java.lang.IllegalArgumentException	if an incorrect number of arguments are passed, the receiver is incompatible with the declaring class, or an argument could not be converted by a widening conversion
 * @exception	java.lang.reflect.InvocationTargetException	if an exception was thrown by the invoked constructor
 * @see			java.lang.reflect.AccessibleObject
 */
public Object invoke(Object receiver, Object args[])
	throws IllegalAccessException, IllegalArgumentException, InvocationTargetException
{
	Class declaringClass = getDeclaringClass();

	if ((getModifiers() & Modifier.STATIC) == 0) {
		if (receiver == null) throw new NullPointerException();
		if (!declaringClass.isInstance(receiver)) {
			throw new IllegalArgumentException("invalid receiver");
		}
	} else {
		receiver = null;
	}

	if (args == null) args = emptyArgs;

	/* check accessibility */
	if (!isAccessible()) {
		Class callerClass = com.ibm.oti.vm.VM.getStackClass(1);
		if (!checkAccessibility(callerClass, receiver)) {
			throw new IllegalAccessException();
		}
	}

	/* ensure that the parameter types are loaded */
	if(parameterTypes == null) getParameterTypesImpl();
	args = marshallArguments(parameterTypes, args);

	/* need to initialize the static receiver here */
	if (receiver == null) initializeClass(declaringClass);

	if(returnType == null) getReturnTypeImpl();

	if (returnType == Void.TYPE) {
		invokeV(receiver, args);
		return null;
	} else if (returnType == Boolean.TYPE) {
		return new Boolean(invokeI(receiver, args) != 0);
	} else if (returnType == Character.TYPE) {
		return new Character((char)invokeI(receiver, args));
	} else if (returnType == Byte.TYPE) {
		return new Byte((byte)invokeI(receiver, args));
	} else if (returnType == Short.TYPE) {
		return new Short((short)invokeI(receiver, args));
	} else if (returnType == Integer.TYPE) {
		return new Integer(invokeI(receiver, args));
	} else if (returnType == Long.TYPE) {
		return new Long(invokeJ(receiver, args));
	} else if (returnType == Float.TYPE) {
		return new Float(invokeF(receiver, args));
	} else if (returnType == Double.TYPE) {
		return new Double(invokeD(receiver, args));
	} else {
		return invokeL(receiver, args);
	}
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * The format of the string is
 *		modifiers (if any)
 *		return type
 *		declaring class name
 *		'.'
 *		method name
 *		'('
 *		parameter types, separated by ','
 *		')'
 * If the method throws exceptions,
 *		' throws '
 *		exception types, separated by ','
 *
 * For example:
 *		<code>public native Object java.lang.Method.invoke(Object,Object) throws IllegalAccessException,IllegalArgumentException,InvocationTargetException</code>
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver
 */
public String toString()
{
	StringBuffer buf;
	String mods;
	Class[] types;
	Class current;
	int i, arity;

	buf = new StringBuffer();
	mods = Modifier.toString(getModifiers());
	if(mods.length() != 0)
	{
		buf.append(mods);
		buf.append(" ");
	}

	current = getReturnType();
	arity = 0;
	while(current.isArray())
	{
		current = current.getComponentType();
		arity++;
	}
	buf.append(current.getName());
	for(;arity > 0; arity--) buf.append("[]");

	buf.append(" ");
	buf.append(getDeclaringClass().getName());
	buf.append(".");
	buf.append(getName());
	buf.append("(");
	types = getParameterTypes();
	for(i = 0; i < types.length; i++)
	{
		current = types[i];
		arity = 0;
		while(current.isArray())
		{
			current = current.getComponentType();
			arity++;
		}
		buf.append(current.getName());
		for(;arity > 0; arity--) buf.append("[]");
		if(i != (types.length - 1))
			buf.append(",");
	}
	buf.append(")");
	types = getExceptionTypes();
	if(types.length > 0)
	{
		buf.append(" throws ");
		for(i = 0; i < types.length; i++)
		{
			current = types[i];
			buf.append(current.getName());
			if(i != (types.length - 1))
				buf.append(",");
		}
	}

	return buf.toString();
}
}
