/*
 * @(#)src/classes/sov/java/lang/ref/Finalizer.java, jvm, hs131, 20031015 1.11.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM Java(tm)2 SDK, Standard Edition, v 1.3.1
 * (C) Copyright IBM Corp. 1998, 2001. All Rights Reserved
 * ===========================================================================
 */

/*
 *
 * ===========================================================================
 *
 * Copyright 1997-2000 Sun Microsystems, Inc. All Rights Reserved.
 * ===========================================================================
 */

/*
 * @(#)Finalizer.java   1.16 99/12/04
 *
 */

package java.lang.ref;

import java.security.PrivilegedAction;
import java.security.AccessController;


final class Finalizer extends FinalReference { /* Package-private; must be in
                                                  same package as the Reference
                                                  class */

    /* A native method that invokes an arbitrary object's finalize method is
       required since the finalize method is protected
     */
    static native void invokeFinalizeMethod(Object o) throws Throwable;
    private static native void initForReset();           /*ibm@22667*/

    static private ReferenceQueue queue = new ReferenceQueue();
    static private Finalizer unfinalized = null;
    static private Object lock = new Object();
    static private Finalizer processing = null;                       /*ibm@5724*/
    /* List of Application objects to be finalized in ResetGC       */
    static private Finalizer unfinalizedApp = null;                   /*ibm@5724*/

    private Finalizer
        nextf = null,                                                 /*ibm@5724*/
        prevf = null;                                                 /*ibm@5724*/

    private boolean hasBeenFinalized() {
        return (nextf == this);                                       /*ibm@5724*/
    }

    private void add() {
        synchronized (lock) {
            if (unfinalized != null) {
                this.nextf = unfinalized;                             /*ibm@5724*/
                unfinalized.prevf = this;                             /*ibm@5724*/
            }
            unfinalized = this;
        }
    }

    private void remove() {
        synchronized (lock) {
            if (unfinalized == this) {
                if (this.nextf != null) {                             /*ibm@5724*/
                    unfinalized = this.nextf;                         /*ibm@5724*/
                } else {
                    unfinalized = this.prevf;                         /*ibm@5724*/
                }
            }
            if (this.nextf != null) {                                 /*ibm@5724*/
                this.nextf.prevf = this.prevf;                        /*ibm@5724*/
            }
            if (this.prevf != null) {                                 /*ibm@5724*/
                this.prevf.nextf = this.nextf;                        /*ibm@5724*/
            }
            this.nextf = this;   /* Indicates that this has been finalized *//*ibm@5724*/
            this.prevf = this;                                        /*ibm@5724*/
        }
    }

    private Finalizer(Object finalizee) {
        super(finalizee, queue);
        add();
    }

    /* Invoked by VM */
    static void register(Object finalizee) {
        new Finalizer(finalizee);
    }

    private void runFinalizer() {
        synchronized (this) {
            if (hasBeenFinalized()) return;
            remove();
        }
        try {
            Object finalizee = this.get();
            if (finalizee != null) {
                invokeFinalizeMethod(finalizee);
                /* Clear stack slot containing this variable, to decrease
                   the chances of false retention with a conservative GC */
                finalizee = null;
            }
        } catch (Throwable x) { }
        super.clear();
    }

    /* Additional functions for finalization of App objects during  */
    /* resetTransientHeap                                 ibm@5724  */
    static void runAppFinalizer(Finalizer f) {
           f.runFinalizerNoRemove();
    }

    static void runAppFinalizers() {
        for (Finalizer f = unfinalizedApp;f != null; f=f.nextf)       /*ibm@5724*/
         runAppFinalizer(f);

    unfinalizedApp = null;
    }
    

    private void runFinalizerNoRemove() {
        synchronized (this) {
            if (hasBeenFinalized()) return;
        }
        try {
            Object finalizee = this.get();
            if (finalizee != null) {
                    invokeFinalizeMethod(finalizee);
            /* Clear stack slot containing this variable, to decrease
               the chances of false retention with a conservative GC */
            finalizee = null;
            }
        } catch (Throwable x) { }
        super.clear();
    }

    /* Create a privileged secondary finalizer thread in the system thread
       group for the given Runnable, and wait for it to complete.

       This method is used by both runFinalization and runFinalizersOnExit.
       The former method invokes all pending finalizers, while the latter
       invokes all uninvoked finalizers if on-exit finalization has been
       enabled.

       These two methods could have been implemented by offloading their work
       to the regular finalizer thread and waiting for that thread to finish.
       The advantage of creating a fresh thread, however, is that it insulates
       invokers of these methods from a stalled or deadlocked finalizer thread.
     */
    private static void forkSecondaryFinalizer(final Runnable proc) {
        PrivilegedAction pa = new PrivilegedAction() {
            public Object run() {
                ThreadGroup tg = Thread.currentThread().getThreadGroup();
                for (ThreadGroup tgn = tg;
                     tgn != null;
                     tg = tgn, tgn = tg.getParent());
                Thread sft = new Thread(tg, proc, "Secondary finalizer");
                sft.start();
                try {
                    sft.join();
                } catch (InterruptedException x) {
                    /* Ignore */
                }
                return null;
            }};
        AccessController.doPrivileged(pa);
    }

    /* Called by Runtime.runFinalization() */
    static void runFinalization() {
        forkSecondaryFinalizer(new Runnable() {
            public void run() {
                for (;;) {
                    Finalizer f = (Finalizer)queue.poll();
                    if (f == null) break;
                    f.runFinalizer();
                }
            }
        });
    }

    /* Invoked by java.lang.Shutdown */
    static void runAllFinalizers() {
        forkSecondaryFinalizer(new Runnable() {
            public void run() {
                for (;;) {
                    Finalizer f;
                    synchronized (lock) {
                        f = unfinalized;
                        if (f == null) break;
                        unfinalized = f.nextf;                        /*ibm@5724*/
                    }
                    f.runFinalizer();
                }}});
    }

    private static class FinalizerThread extends Thread {
        FinalizerThread(ThreadGroup g) {
            super(g, "Finalizer");
        }
        public void run() {
            initForReset();                                       /*ibm@22667*/
            for (;;) {
                try {
                    processing = (Finalizer) queue.poll(); /*ibm@5724,ibm@12945,ibm@13717*//*ibm@26216*/
                    if (processing == null) {              /*ibm@26216*/
                        processing = (Finalizer) queue.remove();/*ibm@5724,ibm@12945*//*ibm@22667*/
                    }                                      /*ibm@26216*/
                    processing.runFinalizer();                        /*ibm@22667*/
                 } catch (InterruptedException x) {
                    continue;
                } catch (Throwable x) {                               /*ibm@13717*/                
                }                                                     /*ibm@13717*/
            }
        }
    }

    static {
        ThreadGroup tg = Thread.currentThread().getThreadGroup();
        for (ThreadGroup tgn = tg;
             tgn != null;
             tg = tgn, tgn = tg.getParent());
        Thread finalizer = new FinalizerThread(tg);
        finalizer.setPriority(Thread.MAX_PRIORITY - 2);
        finalizer.setDaemon(true);
        finalizer.start();
    }

}
