/*
 * @(#)src/classes/sov/javax/swing/plaf/basic/ComplexAreaView.java, as131, as131, 20031014 1.5.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM Java(tm)2 SDK, Standard Edition, v 1.3.1
 * (C) Copyright IBM Corp. 1998, 2001. All Rights Reserved
 * ===========================================================================
 */

/* 
 *
 * ===========================================================================
 *
 *
 *
 * ===========================================================================
 * Change activity:
 *
 * Reason  Date   Origin  Description
 * ------  ----   ------  ---------------------------------------------------- 
 *         270799         New file.
 *         280999         Update.
 *
 * ===========================================================================
 * Module Information:
 *		
 * DESCRIPTION: Hindi, Thai, and Bidi enhancements.
 * ===========================================================================
 */

//ibm.2722

/*
 * @(#)ComplexAreaView.java	1.7 99/10/05
 *
 * Complex unwrapped plain text area view
 *
 *
 * This file requires Java1.2.
 */ 

package javax.swing.plaf.basic;

import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;
import java.text.*;
import java.util.HashMap;
import java.util.Map;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

// swing's version is package private and in another package
import sun.awt.font.Bidi;

import javax.swing.plaf.basic.TabLayout.FixedTabs;

class ComplexAreaView extends View {
    FixedTabs tabs = new FixedTabs();    // tabs for most recently fetched line
    float lx, ly;      // origin for most recently fetched line
    boolean ltr;       // direction of most recently fetched line
    int lpos;          // pos in model of start of most recently fetched line

    Font cacheFont;
    FontRenderContext cacheFRC;
    float cacheTabWidth;

    boolean defaultIsLTR;
    boolean contextual;

    HashMap layoutCache = new HashMap(11);

    private final Segment lineBuffer = new Segment();

    float cacheLongWidth;
    Element cacheLongElement;

    int lineAscent = -1;
    int lineHeight = -1;

    static final float HPAD = 4; // horizontal padding around text for caret flag

    ComplexAreaView(Element e) {
        super(e);

        defaultIsLTR = !Boolean.FALSE.equals(getDocument().getProperty("ltr orientation"));
        contextual = Boolean.TRUE.equals(getDocument().getProperty("contextual line direction"));
    }

    /**
     * Returns the tab size set for the document, defaulting to 8.
     *
     * @return the tab size
     */
    int getTabSize() {
        Integer i = (Integer) getDocument().getProperty(PlainDocument.tabSizeAttribute);
        int size = (i != null) ? i.intValue() : 8;
        return size;
    }

    TabLayout createLayout(Element line) {
        Document doc = getDocument();

        int p0 = line.getStartOffset();
        int p1 = line.getEndOffset();
        p1 = Math.min(p1, doc.getLength());

        boolean hasStrong = false;
        boolean isLTR = defaultIsLTR;
        
        if (contextual) {
            try {
                doc.getText(p0, p1 - p0, lineBuffer);
                isLTR = Bidi.defaultIsLTR(lineBuffer.array, 
                                          lineBuffer.offset, 
                                          lineBuffer.offset + lineBuffer.count, 
                                          defaultIsLTR);
            } 
            catch (BadLocationException e) {
            }
        }

        Map attributes = new HashMap();
        attributes.put(TextAttribute.FONT, cacheFont);
        attributes.put(TextAttribute.RUN_DIRECTION, 
                       isLTR ? TextAttribute.RUN_DIRECTION_LTR : TextAttribute.RUN_DIRECTION_RTL);

        AttributedCharacterIterator aci = ComplexWrappedAreaView.getElementACI(line, attributes);

        return TabLayout.create(aci, cacheFRC);
    }
    
    // --- layout cache -----------------------------------------------------------

    TabLayout getLayout(Element line) {
        TabLayout layout = (TabLayout)layoutCache.get(line);
        if (layout == null) {
            layout = createLayout(line);
            layoutCache.put(line, layout);

            // !!! todo: limit size of cache
        }
        return layout;
    }

    void clearLayoutCache() {
        layoutCache.clear();
    }

    void putLayoutCache(Element line, TabLayout layout) {
        layoutCache.put(line, layout);
    }

    void removeLayoutCache(Element line) {
        layoutCache.remove(line);
    }

    // --------------------------------------------------------------

    TabLayout getLayoutInfo(Element line) {
        TabLayout layout = getLayout(line);

        lpos = line.getStartOffset();
        lx = 0;
        ly = lineAscent;
        tabs.tabBase = 0;
        tabs.tabWidth = layout.isLeftToRight() ? cacheTabWidth : -cacheTabWidth;

        return layout;
    }

    float getLayoutWidth(Element line) {
        TabLayout layout = getLayoutInfo(line);
        return Math.abs(layout.getAdvance(lx, ly, tabs));
    }

    TabLayout getLayoutInfo(int lineIndex, Rectangle2D alloc) {
            
        Element line = getElement().getElement(lineIndex);

        TabLayout l = getLayoutInfo(line);

        if (alloc != null) {
            if (l.isLeftToRight()) {
                lx = (float)alloc.getX();
            } else {
                lx = (float)(alloc.getX() + alloc.getWidth());
            }
            tabs.tabBase = lx;
            ly = (float)(lineIndex * lineHeight + lineAscent);
        }

        return l;
    }

    private static final FontRenderContext defaultFRC = new FontRenderContext(null, false, false);

    void updateMetrics() {
        Component host = getContainer();
        Font f = host.getFont();
        if (f != cacheFont) { // swing holds its fonts so it can do identity comparison
            cacheFont = f;
            Graphics g = host.getGraphics();
            cacheFRC = g == null ? defaultFRC : ((Graphics2D)g).getFontRenderContext();
            cacheTabWidth = (float)(getTabSize() * cacheFont.getStringBounds("m", cacheFRC).getWidth());
            FontMetrics fm = host.getFontMetrics(cacheFont);
            lineHeight = fm.getHeight();
            lineAscent = fm.getAscent();

            clearLayoutCache();
            recomputeLongElement();
        }
    }

    void recomputeLongElement() {
        cacheLongElement = null;
        cacheLongWidth = -1;

        Element map = getElement();
        int count = map.getElementCount();
        for (int i = 0; i < count; ++i) {
            Element line = map.getElement(i);
            float width = getLayoutWidth(line);
            if (width > cacheLongWidth) {
                cacheLongWidth = width;
                cacheLongElement = line;
            }
        }
    }
            
    // ---- View methods ----------------------------------------------------

    public float getPreferredSpan(int axis) {
        updateMetrics();

        switch (axis) {
        case View.X_AXIS:
            return cacheLongWidth + HPAD * 2;

        case View.Y_AXIS:
            int count = getElement().getElementCount();
            return count * lineHeight;

        default:
            throw new IllegalArgumentException("Invalid axis: " + axis);
        }
    }

    public void paint(Graphics g, Shape a) {
        updateMetrics();

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        JTextComponent host = (JTextComponent) getContainer();
        int sel0 = host.getSelectionStart();
        int sel1 = host.getSelectionEnd();
        boolean haveSelection = (sel0 != sel1) && host.getCaret().isSelectionVisible();

        Color unselected = (host.isEnabled()) ? host.getForeground() : host.getDisabledTextColor();
        Color selected = haveSelection ? host.getSelectedTextColor() : unselected;
        boolean doSelection = !selected.equals(unselected);

        Highlighter h = host.getHighlighter();
        LayeredHighlighter dh = (h instanceof LayeredHighlighter) ?
            (LayeredHighlighter)h : null;

        Rectangle2D clipRect = g.getClip().getBounds2D();
        float heightBelow = (float)(alloc.getMaxY() - clipRect.getMaxY());
        int linesBelow = Math.max(0, (int)(heightBelow / lineHeight));
        float heightAbove = (float)(clipRect.getY() - alloc.getY());
        int linesAbove = Math.max(0, (int)(heightAbove / lineHeight));
        int linesTotal = (int)(alloc.getHeight() / lineHeight);
        if (alloc.getHeight() > linesTotal * lineHeight) {
            linesTotal++;
        }

        Element map = getElement();
        int lineCount = map.getElementCount();
        int endLine = Math.min(lineCount, linesTotal - linesBelow);

        lineCount--; // adjust for later test

        Graphics2D g2d = (Graphics2D)g;

        for (int line = linesAbove; line < endLine; line++) {
            Element lineElement = map.getElement(line);
            int p0 = lineElement.getStartOffset();
            int p1 = lineElement.getEndOffset();

            if (dh != null) {
                if (line == lineCount) {
                    dh.paintLayeredHighlights(g, p0, p1, a, host, this);
                } else {
                    dh.paintLayeredHighlights(g, p0, p1 - 1, a, host, this);
                }
            }

            TabLayout layout = getLayoutInfo(line, alloc);

            if (haveSelection && sel0 <= p0 && sel1 >= p1) {
                // all selected
                g2d.setColor(selected);
                layout.draw(g2d, lx, ly, tabs);
            } else {
                // some or all unselected
                g2d.setColor(unselected);
                layout.draw(g2d, lx, ly, tabs);

                if (doSelection && (sel0 < p1 || sel1 > p0)) {
                    // some selected
                    Shape clip = g2d.getClip();
                    g2d.setClip(layout.getLogicalHighlightShape(lx, ly, tabs, sel0 - p0, sel1 - p0, alloc));
                    g2d.setColor(selected);
                    layout.draw(g2d, lx, ly, tabs);
                    g2d.setClip(clip);
                }
            }
        }
    }

    public int getNextVisualPositionFrom(int pos, Position.Bias b, Shape a, 
                                         int direction, Position.Bias[] biasRet) 
        throws BadLocationException {

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        boolean fwd = b != Position.Bias.Backward;

        Element map = getElement();
        int count = map.getElementCount();
        int lineIndex = map.getElementIndex(pos);

        switch (direction) {
        case NORTH:
        case SOUTH:
            if ((direction == NORTH && lineIndex > 0) || (direction == SOUTH && lineIndex < count - 1)) {
                JTextComponent host = (JTextComponent)getContainer();
                Caret caret = host.getCaret();
                Point p = caret != null ? caret.getMagicCaretPosition() : null;
                if (p == null) {
                    TabLayout layout = getLayoutInfo(lineIndex, alloc);
                    TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos - lpos) : TextHitInfo.beforeOffset(pos - lpos);
                    float x = layout.position(lx, ly, tabs, hit);
                    p = new Point((int)x, (int)ly);
                }
                
                int newIndex = direction == NORTH ? lineIndex - 1 : lineIndex + 1;
                TabLayout layout = getLayoutInfo(newIndex, alloc);
                TextHitInfo hit = layout.hitTest(lx, ly, tabs, p.x, ly);
                int ix = hit.getInsertionIndex();
                if (ix == layout.getCharacterCount()) {
                    fwd = true;
                    pos = lpos + ix - 1;
                } else {
                    fwd = hit.isLeadingEdge();
                    pos = lpos + ix;
                }
            }
            break;

        case WEST:
        case EAST:
            TabLayout layout = getLayoutInfo(lineIndex, alloc);
            TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos - lpos) : TextHitInfo.beforeOffset(pos - lpos);
            hit = layout.getNextHit(hit, direction == EAST);
            if (hit == null || hit.getInsertionIndex() == layout.getCharacterCount()) {
                if (layout.isLeftToRight() == (direction == EAST)) {
                    if (lineIndex < count - 1) {
                        fwd = false;
                        pos = lpos + layout.getCharacterCount();
                    }
                } else {
                    if (lineIndex > 0) {
                        pos = lpos - 1;
                        fwd = true;
                    }
                }
                break;
            }
            pos = lpos + hit.getInsertionIndex();
            fwd = hit.isLeadingEdge();
            break;

        default:
            throw new IllegalArgumentException("Bad direction: " + direction);
        }

        biasRet[0] = fwd ? Position.Bias.Forward : Position.Bias.Backward;

        return pos;
    }

    public Shape modelToView(int pos, Shape a, Position.Bias b) throws BadLocationException {
        updateMetrics();

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        Element elem = getElement();
        int line = elem.getElementIndex(pos);
        TabLayout l = getLayoutInfo(line, alloc);

        boolean fwd = b != Position.Bias.Backward;
        TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos - lpos) : TextHitInfo.beforeOffset(pos - lpos);
        float x = l.position(lx, ly, tabs, hit);
        Rectangle2D r = new Rectangle2D.Float(x, ly - lineAscent, 1, lineHeight);

        return r;
    }

    public Shape modelToView(int p0, Position.Bias b0, int p1, Position.Bias b1, Shape a) throws BadLocationException {
        updateMetrics();
        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        Rectangle2D.Float bounds = new Rectangle2D.Float((float)alloc.getX(),
                                                         (float) 0,
                                                         (float)alloc.getWidth(),
                                                         lineHeight);

        Element elem = getElement();
        int startLine = elem.getElementIndex(p0);
        int finishLine = elem.getElementIndex(p1);
        
        if (startLine == finishLine) {
            TabLayout layout = getLayoutInfo(startLine, alloc);
            bounds.y = ly - lineAscent;
            return layout.getLogicalHighlightShape(lx, ly, tabs, p0 - lpos, p1 - lpos, bounds);
        } else {
            TabLayout layout1 = getLayoutInfo(startLine, alloc);
            bounds.y = ly - lineAscent;
            Shape s1 = layout1.getLogicalHighlightShape(lx, ly, tabs, p0 - lpos, p1 - lpos, bounds);
            float bottom = bounds.y + lineHeight;

            TabLayout layout2 = getLayoutInfo(finishLine, alloc);
            bounds.y = ly - lineAscent;
            Shape s2 = layout2.getLogicalHighlightShape(lx, ly, tabs, p0 - lpos, p1 - lpos, bounds);

            GeneralPath gp = new GeneralPath(s1);
            gp.append(s2, false);

            if (startLine < finishLine - 1) {
                bounds.height = bounds.y - bottom;
                bounds.y = bottom;
                gp.append(bounds, false);
            }

            return gp;
        }           
    }

    public int viewToModel(float x, float y, Shape a, Position.Bias[] biasReturn) {
        updateMetrics();

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        int count = getElement().getElementCount();
        int line = (int)Math.floor((y - alloc.getY()) / lineHeight);
        if (line < 0) {
            biasReturn[0] = Position.Bias.Backward;
            return getStartOffset();
        } else if (line >= count) {
            biasReturn[0] = Position.Bias.Forward;
            return getEndOffset() - 1; 
        } else {
            TabLayout l = getLayoutInfo(line, alloc);
            TextHitInfo hit = l.hitTest(lx, ly, tabs, x, y);
            int ix = hit.getInsertionIndex();
            if (ix == l.getCharacterCount()) {
                biasReturn[0] = Position.Bias.Forward;
                return lpos + ix - 1;
            } else {
                biasReturn[0] = hit.isLeadingEdge() ? Position.Bias.Forward : Position.Bias.Backward;
                return lpos + ix;
            }
        }
    }

    public void insertUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        Shape aa = adjustAllocation(a);
        super.insertUpdate(e, aa, f);
        updateDamage(e, aa, f);
    }

    public void removeUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        Shape aa = adjustAllocation(a);
        super.removeUpdate(e, aa, f);
        updateDamage(e, aa, f);
    }

    public void changedUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        Shape aa = adjustAllocation(a);
        super.changedUpdate(e, aa, f);
        updateDamage(e, aa, f);
    }

    void updateDamage(DocumentEvent event, Shape aa, ViewFactory f) {

        Component host = getContainer();
        if (host.isShowing()) {
            updateMetrics();

            Element map = getElement();
            DocumentEvent.ElementChange ec = event.getChange(map);
            
            Element[] added = (ec != null) ? ec.getChildrenAdded() : null;
            Element[] removed = (ec != null) ? ec.getChildrenRemoved() : null;
            if (((added != null) && (added.length > 0)) || 
                ((removed != null) && (removed.length > 0))) {

                boolean heightChanged = added.length != removed.length;
                boolean widthChanged = false;

                if (added != null) {
                    for (int i = 0; i < added.length; i++) {
                        float width = getLayoutWidth(added[i]);
                        if (width > cacheLongWidth) {
                            cacheLongWidth = width;
                            cacheLongElement = added[i];
                            widthChanged = true;
                        }
                    }
                }

                if (removed != null) {
                    for (int i = 0; i < removed.length; i++) {
                        removeLayoutCache(removed[i]);
                        if (removed[i] == cacheLongElement) { // will only happen once
                            recomputeLongElement();
                            widthChanged = true;
                        }
                    }
                }
                preferenceChanged(null, widthChanged, heightChanged);
                host.repaint();
            } else {
                int lineIndex = map.getElementIndex(event.getOffset());
                Element line = map.getElement(lineIndex);
                putLayoutCache(line, createLayout(line));

                if (event.getType() == DocumentEvent.EventType.INSERT) {
                    float width = getLayoutWidth(line);
                    if (line == cacheLongElement) {
                        cacheLongWidth = width;
                        preferenceChanged(null, true, false);
                    } else if (width > cacheLongWidth) {
                        cacheLongWidth = width;
                        cacheLongElement = line;
                        preferenceChanged(null, true, false);
                    }
                } else if (event.getType() == DocumentEvent.EventType.REMOVE) {
                    if (line == cacheLongElement) {
                        recomputeLongElement();
                        preferenceChanged(null, true, false);
                    }                   
                }

                if (aa != null) {
                    Rectangle2D alloc = aa.getBounds2D();
                    float top = (float)(alloc.getY() + lineIndex * lineHeight);
                    host.repaint((int)alloc.getX(),
                                 (int)top,
                                 (int)Math.ceil(alloc.getMaxX()),
                                 (int)Math.ceil(top + lineHeight));
                }
            }
        }
    }

    protected Shape adjustAllocation(Shape a) {
        if (a != null) {
            Rectangle bounds = a.getBounds();
            bounds.x += HPAD;
            bounds.width -= HPAD * 2;
            return bounds;
        }
        return a;
    }
}
