/*
 * @(#)src/classes/sov/javax/swing/plaf/basic/ComplexFieldView.java, as131, as131, 20031014 1.6.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM Java(tm)2 SDK, Standard Edition, v 1.3.1
 * (C) Copyright IBM Corp. 1998, 2001. All Rights Reserved
 * ===========================================================================
 */

/*
 *
 * ===========================================================================
 *
 * Copyright 1997-1999 by Sun Microsystems, Inc.,
 * ===========================================================================
 */
 
package javax.swing.plaf.basic;

import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;
import java.text.*;
import java.util.HashMap;
import java.util.Map;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

// swing's version is package private and in another package
import sun.awt.font.Bidi;

class ComplexFieldView extends View {
    TextLayout layout;
    Font cacheFont;
    FontRenderContext cacheFRC;
    boolean defaultIsLTR;
    boolean contextual;

    ComplexFieldView(Element e) {
        super(e);
        defaultIsLTR = !Boolean.FALSE.equals(getDocument().getProperty("ltr orientation"));
        contextual = Boolean.TRUE.equals(getDocument().getProperty("contextual line direction"));
    }

    final int HPAD = 4;

    public float getPreferredSpan(int axis) {
        TextLayout l = getLayout(null);

        switch (axis) {
        case View.X_AXIS:
            return l.getAdvance() + HPAD * 2;
        case View.Y_AXIS:
            return l.getAscent() + l.getDescent() + l.getLeading();
        default:
            throw new IllegalArgumentException("Invalid axis: " + axis);
        }
    }

    public void paint(Graphics g, Shape allocation) {
        Shape aa = adjustAllocation(allocation);
        TextLayout l = getLayout(g);

        Rectangle r = (Rectangle)aa;
        float x = (float)r.getX();
        float y = (float)r.getY() + l.getAscent();

        int p0 = getStartOffset();
        int p1 = Math.min(getDocument().getLength(), getEndOffset());

        JTextComponent host = (JTextComponent)getContainer();

        try {
            LayeredHighlighter h = (LayeredHighlighter)host.getHighlighter();
            if (h != null) {
                h.paintLayeredHighlights(g, p0, p1, allocation, host, this);
            }
        }
        catch (ClassCastException e) {
        }

        Graphics2D g2d = (Graphics2D)g;
        Color unselected = host.isEnabled() ? host.getForeground() : host.getDisabledTextColor();
        g2d.setColor(unselected);
        g2d.translate(x, y);
        l.draw(g2d, 0, 0);

        int sel0 = host.getSelectionStart();
        int sel1 = host.getSelectionEnd();
        if (sel0 != sel1) {
            Color selected = host.getCaret().isSelectionVisible() ? host.getSelectedTextColor() : unselected;
            if (!selected.equals(unselected) && (sel0 < p1) && (sel1 > p0)) {
                Shape clip = g2d.getClip();
                g2d.setClip(l.getLogicalHighlightShape(sel0 - p0, sel1 - p0));
                g2d.setColor(selected);
                l.draw(g2d, 0, 0);
                g2d.setClip(clip);
            }
        }
        g2d.translate(-x, -y);
    }

    /*
     * Not sure exactly how this should work.
     * General swing caret idea is that the caret flag points in the direction of the character it is attached to.
     * At directional boundaries the caret visual position doesn't change, but instead the flag flips.  
     * For example:
     *
     * >abDC - 0 before / -1 trailing
     * a>bDC - 1 before / 0 trailing
     * ab>DC - 2 before / 1 trailing
     * ab<DC - 4 before / 3 trailing
     * abD<C - 3 before / 2 trailing
     * abDC< - 2 after  / 2 leading
     * abDC> - 4 after  / 4 leading
     *
     * The movement is either with or against the line direction.  The caret is either at a directional
     * boundary or not (before or after the movement).  The rules for picking hits are as follows:
     * - if with line direction
     *   - if hit at direction boundary
     *     - if other hit insertion point > hit insertion point
     *       - return other hit
     *   - get next hit (l or r)
     *       - if next hit at direction boundary
     *         - if next other hit insertion point < next hit insertion point
     *           - return other next hit
     *   - return next hit
     *
     * 'with line directon' only affects the comparisons (note that == is not a problem since at a direction
     * boundary the ips are always !=.
     */

    public int getNextVisualPositionFrom(int pos, Position.Bias b, Shape a, 
                                         int direction, Position.Bias[] biasRet) 
        throws BadLocationException {

        // default bias backward
        boolean biasf = false;

        switch (direction) {
        case NORTH:
        case SOUTH:
            break;

        case WEST:
        case EAST:
            TextLayout l = getLayout(null);
            boolean ltr = l.isLeftToRight();
            boolean right = direction == EAST;
            boolean wld = ltr == right;

            if (pos == -1) {
                if (wld) { // 'wrap around'
                    pos = getStartOffset();
                } else {
                    pos = Math.min(getDocument().getLength(), getEndOffset());
                }
            } else {
                boolean fwd = b != Position.Bias.Backward;
                TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos) : TextHitInfo.beforeOffset(pos);
                TextHitInfo other = l.getVisualOtherHit(hit);
                
                int ix = pos;
                boolean ib = fwd;
                boolean iltr = (l.getCharacterLevel(hit.getCharIndex()) & 0x1) == 0;

                int ox = other.getInsertionIndex();
                boolean ob = other.isLeadingEdge();

                if (ix != ox) {
                    // at directional boundary, do we cross, or toggle?
                    // toggle if we're on the left and moving right, or vice-versa
                    // we're on the left if we're following an ltr char, or
                    // leading an rtl char, i.e. ib != iltr
                    if ((ib != iltr) == right) {
                        pos = ox;
                        biasf = ob;
                        break;
                    }
                }

                hit = right ? l.getNextRightHit(hit) : l.getNextLeftHit(hit);
                if (hit != null) {
                    other = l.getVisualOtherHit(hit);
                    ix = hit.getInsertionIndex();
                    ib = hit.isLeadingEdge();
                    iltr = (l.getCharacterLevel(hit.getCharIndex()) & 0x1) == 0;

                    ox = other.getInsertionIndex();
                    ob = other.isLeadingEdge();

                    if (ix != ox) {
                        // at directional boundary again
                        // toggle if we're on the left and *not* moving right, etc.
                        if ((ib != iltr) != right) {
                            pos = ox;
                            biasf = ob;
                            break;
                        }
                    }

                    pos = ix;
                    biasf = ib;
                } else {
                    if (wld) { // 'pin'
                        pos = Math.min(getDocument().getLength(), getEndOffset());
                    } else {
                        pos = getStartOffset();
                    }
                }
            }
            break;

        default:
            throw new IllegalArgumentException("Bad direction: " + direction);
        }

        biasRet[0] = biasf ? Position.Bias.Forward : Position.Bias.Backward;
        return pos;
    }

    public Shape modelToView(int pos, Shape a, Position.Bias b) throws BadLocationException {
        Shape aa = adjustAllocation(a);
        TextLayout l = getLayout(null);
        boolean fwd = b != Position.Bias.Backward;
        TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos) : TextHitInfo.beforeOffset(pos);
        try {
            Rectangle2D lr = l.getCaretShape(hit).getBounds2D();
            Rectangle2D r = aa.getBounds2D();
            r.setRect(lr.getX() + r.getX(), lr.getY() + r.getY() + l.getAscent(), lr.getWidth(), lr.getHeight());
            return r;
        }
        catch (IllegalArgumentException e) {
        }
        Rectangle2D r = aa.getBounds2D();
        return r;
    }

    public Shape modelToView(int p0, Position.Bias b0, int p1, Position.Bias b1, Shape a) throws BadLocationException {
        Shape aa = adjustAllocation(a);
        TextLayout l = getLayout(null);
        int pos0 = getStartOffset();
        Shape s = l.getLogicalHighlightShape(p0 - pos0, p1 - pos0);
        Rectangle2D r = aa.getBounds2D();
        AffineTransform at = AffineTransform.getTranslateInstance(r.getX(), r.getY() + l.getAscent());
        return at.createTransformedShape(s);
    }

    public int viewToModel(float x, float y, Shape a, Position.Bias[] biasReturn) {
        Shape aa = adjustAllocation(a);
        TextLayout l = getLayout(null);
        boolean ltr = l.isLeftToRight();

        Rectangle2D r = aa.getBounds2D();
        int pos = -1;
        boolean biasForward = false;
        if (y < r.getY()) {
            // do nothing
        } else if (y > r.getMaxY()) {
            pos = -2;
        } else if (x < r.getMinX()) { 
            pos = ltr ? -1 : -2;
        } else if (x > r.getMaxX()) {
            pos = ltr ? -2 : -1;
        } else {
            x -= r.getX();
            y -= r.getY() + l.getAscent();
            TextHitInfo hit = l.hitTestChar(x, y);
            pos = hit.getInsertionIndex();
            biasForward = hit.isLeadingEdge();
        }

        switch (pos) {
        case -1: 
            pos = getStartOffset();
            biasReturn[0] = Position.Bias.Backward;
            return pos;
        case -2:
            pos = Math.min(getDocument().getLength(), getEndOffset());
            biasReturn[0] = Position.Bias.Forward;
            return pos;
        default:
            biasReturn[0] = biasForward ? Position.Bias.Forward : Position.Bias.Backward;
            return getStartOffset() + pos;
        }
    }

    private TextLayout getLayout(Graphics g) {
        Component host = getContainer();
        Font f = host.getFont();
        FontRenderContext frc = g == null ? defaultFRC : ((Graphics2D)g).getFontRenderContext();
        if (layout == null ||
            !f.equals(cacheFont) ||
            (frc != null && !frc.equals(cacheFRC))) {

            cacheFont = f;
            cacheFRC = frc;

            Document doc = getDocument();
            Element elem = getElement();

            Map attributes = new HashMap();
            attributes.put(TextAttribute.FONT, cacheFont);

            boolean isLTR = defaultIsLTR;
            if (contextual) {
                int p0 = elem.getStartOffset();
                int p1 = elem.getEndOffset();
                p1 = Math.min(p1, doc.getLength()); // ???
                try {
                    char[] chars = doc.getText(p0, p1 - p0).toCharArray();
                    isLTR = Bidi.defaultIsLTR(chars, 
                                              0, 
                                              chars.length, 
                                              defaultIsLTR);
                } 
                catch (BadLocationException e) {
                }
            }

            attributes.put(TextAttribute.RUN_DIRECTION, 
                           isLTR ? TextAttribute.RUN_DIRECTION_LTR : 
                           TextAttribute.RUN_DIRECTION_RTL);

            AttributedCharacterIterator aci = ComplexWrappedAreaView.getElementACI(elem, attributes);

            layout = new TextLayout(aci, frc);
        }

        return layout;
    }

    static final FontRenderContext defaultFRC = new FontRenderContext(null, false, false);

    public void insertUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        super.insertUpdate(e, adjustAllocation(a), f);
        updateDamage(e, a, f);
    }

    public void removeUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        super.removeUpdate(e, adjustAllocation(a), f);
        updateDamage(e, a, f);
    }

    public void changedUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        super.changedUpdate(e, adjustAllocation(a), f);
        updateDamage(e, a, f);
    }

    void updateDamage(DocumentEvent e, Shape a, ViewFactory f) {
        layout = null;
        Component host = getContainer();
        if (host.isShowing()) {
            preferenceChanged(this, true, false);
            host.repaint();
        }

        updateVisibilityModel();
    }

    void updateVisibilityModel() {
        Component c = getContainer();
        if (c instanceof JTextField) {
            JTextField field = (JTextField) c;
            BoundedRangeModel vis = field.getHorizontalVisibility();
            int hspan = (int) getPreferredSpan(X_AXIS);
            int extent = vis.getExtent();
            int maximum = Math.max(hspan, extent);
            extent = (extent == 0) ? maximum : extent;
            int value = maximum - extent;
            int oldValue = vis.getValue();
            if ((oldValue + extent) > maximum) {
                oldValue = maximum - extent;
            }
            value = Math.max(0, Math.min(value, oldValue));
            vis.setRangeProperties(value, extent, 0, maximum, false);
        }
    }

    protected Shape adjustAllocation(Shape a) {
        if (a != null) {
            Rectangle bounds = a.getBounds();
            int vspan = (int) getPreferredSpan(Y_AXIS);
            int hspan = (int) getPreferredSpan(X_AXIS);
            if (bounds.height != vspan) {
                int slop = bounds.height - vspan;
                bounds.y += slop / 2;
                bounds.height -= slop;
            }

            // horizontal adjustments
            Component c = getContainer();
            if (c instanceof JTextField) {
                JTextField field = (JTextField) c;
                BoundedRangeModel vis = field.getHorizontalVisibility();
                int max = Math.max(hspan, bounds.width);
                int value = vis.getValue();
                int extent = Math.min(max, bounds.width - 1);
                if ((value + extent) > max) {
                    value = max - extent;
                }
                vis.setRangeProperties(value, extent, vis.getMinimum(),
                                       max, false);
                if (hspan < bounds.width) {
                    // horizontally align the interior
                    int slop = bounds.width - 1 - hspan;

                    int align = ((JTextField)c).getHorizontalAlignment();
		    if (align == LEADING || align == TRAILING) {                  //ibm.3526
			boolean ltr = contextual ?                                       //ibm.3526
			    getLayout(null).isLeftToRight() :                            //ibm.3526
			    c.getComponentOrientation().isLeftToRight();                 //ibm.3526

                    if (align == LEADING) {
                        align = ltr ? LEFT : RIGHT;
			} else {                                                         //ibm.3526
                        align = ltr ? RIGHT : LEFT;
                    }
		    }                                                             //ibm.3526
                            
                    switch (align) {
                    case SwingConstants.CENTER:
                        bounds.x += slop / 2;
                        bounds.width -= slop;
                        break;
                    case SwingConstants.RIGHT:
                        bounds.x += slop;
                        bounds.width -= slop;
                        break;
                    }
                } else {
                    // adjust the allocation to match the bounded range.
                    bounds.width = hspan;
                    bounds.x -= vis.getValue();
                }
            }
            
            bounds.x += HPAD;
            bounds.width -= HPAD * 2;
            return bounds;
        }
        return null;
    }
}
