/*
 * @(#)src/classes/sov/javax/swing/plaf/basic/ComplexWrappedAreaView.java, as131, as131, 20031014 1.6.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM Java(tm)2 SDK, Standard Edition, v 1.3.1
 * (C) Copyright IBM Corp. 1998, 2001. All Rights Reserved
 * ===========================================================================
 */

/*
 *
 * ===========================================================================
 *
 * Copyright 1997-1999 by Sun Microsystems, Inc.,
 * ===========================================================================
 */
 
package javax.swing.plaf.basic;

import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;
import java.text.*;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

// swing's version is package private and in another package
import sun.awt.font.Bidi;

import javax.swing.plaf.basic.TabLayout.FixedTabs;

class ComplexWrappedAreaView extends View {
    // data for most recently fetched line/paragraph
    FixedTabs tLineTabs = new FixedTabs();
    float tLinex;
    float tLiney;
    int tLinePos;

    // data for most recently fetched paragraph
    int tParaLineIndex;

    Font cacheFont;
    FontRenderContext cacheFRC;
    float cacheTabWidth;
    float width;

    boolean contextual;
    boolean defaultIsLTR;
    boolean wordWrap;

    HashMap paragraphCache = new HashMap(11);

    private final Segment lineBuffer = new Segment();

    int lineAscent = -1;
    int lineHeight = -1;
    int lineCount = -1;

    static final float HPAD = 4; // horizontal padding around text for caret flag

    ComplexWrappedAreaView(Element e, boolean wordWrap) {
        super(e);

        this.wordWrap = wordWrap;
        defaultIsLTR = !Boolean.FALSE.equals(getDocument().getProperty("ltr orientation"));
	contextual = Boolean.TRUE.equals(getDocument().getProperty("contextual line direction"));
    }

    /**
     * Returns the tab size set for the document, defaulting to 8.
     *
     * @return the tab size
     */
    int getTabSize() {
        Integer i = (Integer) getDocument().getProperty(PlainDocument.tabSizeAttribute);
        int size = (i != null) ? i.intValue() : 8;
        return size;
    }

    // --- DamageInfo -------------------------------------------------------------

    // Used when updating paragraph

    static final class DamageInfo {
        int lineStart; // first line that needs redrawing
        int linesRemoved; // number of lines deleted
        int linesAdded; // number of lines added

        public DamageInfo(int lineStart, int linesRemoved, int linesAdded) {
            this.lineStart = lineStart;
            this.linesRemoved = linesRemoved;
            this.linesAdded = linesAdded;
        }
    }

    // --- paragraph --------------------------------------------------------------

    class Paragraph {
        Element elem;
        TabLayout[] layouts;
        int cc;

        Paragraph(Element elem, TabLayout[] layouts) {
            this.elem = elem;
            this.layouts = layouts;

            this.cc = elem.getEndOffset() - elem.getStartOffset();
        }

        int getLineCount() {
            return layouts.length;
        }

        TabLayout getLayout(int localLineIndex) {
            return layouts[localLineIndex];
        }

        int getPos() {
            return elem.getStartOffset();
        }

        int getLinePos(int localLineIndex) {
            int pos = elem.getStartOffset();
            for (int i = 0; i < localLineIndex; ++i) {
                pos += layouts[i].getCharacterCount();
            }
            return pos;
        }

        int getCharacterCount() {
            return cc;
        }

        // save ourselves from overhead of rebuilding short paragraphs that don't need it.
        boolean affectedByWidthChange(float oldWidth, float newWidth) {
            if (layouts.length > 1) {
                return true;
            }
           
            TabLayout l = layouts[0];
            if (l.isLeftToRight() != (tLineTabs.tabWidth >= 0)) {
                tLineTabs.tabWidth = -tLineTabs.tabWidth;
            }
            
            return Math.abs(l.getAdvance(0, 0, tLineTabs)) > newWidth;
        }

        int getLineIndex(int localOffset) {
            for (int i = 0; i < layouts.length; ++i) {
                int lcc = layouts[i].getCharacterCount();
                if (localOffset < lcc) {
                    return i;
                }
                localOffset -= lcc;
            }

            return layouts.length;
        }

        // call with paragraph-relative start, inserted and deleted text must be contained
        // entirely within this paragraph and not cause new paragraphs
        DamageInfo update(int start, int inserted, int deleted) {
            char[] chars = getParagraphText(elem);
            boolean ltr = contextual ?
		Bidi.defaultIsLTR(chars, 0, chars.length, defaultIsLTR) :
		defaultIsLTR;
            AttributedCharacterIterator aci = getParagraphACI(elem, chars, ltr);
            
            // base direction change forces rebuild of entire paragraph
            if (ltr != layouts[0].isLeftToRight()) {
                start = 0;
                inserted = aci.getEndIndex() - aci.getBeginIndex();
                deleted = cc;
            }

            DamageInfo damage = updateLayouts(aci, start, inserted, deleted, this);

            cc += inserted - deleted;

            return damage;
        }
    }

    // --- paragraph cache --------------------------------------------------------

    Paragraph getParagraph(Element elem) {
        Paragraph paragraph = (Paragraph)paragraphCache.get(elem);
        if (paragraph == null) {
            paragraph = createParagraph(elem);
            paragraphCache.put(elem, paragraph);

            // !!! todo: limit size of cache
        }
        return paragraph;
    }

    void clearParagraphCache() {
        paragraphCache.clear();
    }

    void putParagraphCache(Element elem, Paragraph paragraph) {
        paragraphCache.put(elem, paragraph);
    }

    void removeParagraphCache(Element elem) {
        paragraphCache.remove(elem);
    }

    char[] getParagraphText(Element para) {
        Document doc = getDocument();

        int p0 = para.getStartOffset();
        int p1 = para.getEndOffset();
        int docLength = doc.getLength();
        p1 = Math.min(p1, docLength);

        int len = p1 - p0;
        char[] result = null;
        try {
            if (p1 == docLength) {
                len += 1;
            }
            result = new char[len];
            if (p1 > p0) {
                doc.getText(p0, p1-p0, lineBuffer);
                System.arraycopy(lineBuffer.array, lineBuffer.offset, result, 0, lineBuffer.count);
            }
            if (p1 == docLength) {
                result[len - 1] = '\n';
            }
        }
        catch (BadLocationException e) {
        }

        return result;
    }

    AttributedCharacterIterator getParagraphACI(Element para, char[] chars, boolean ltr) {

        Map attributes = new HashMap();
	if (cacheFont == null) {                                           //ibm.3526
	    System.out.println("cacheFont is null");                       //ibm.3526
	    new Exception().printStackTrace();                             //ibm.3526
	}                                                                  //ibm.3526
        attributes.put(TextAttribute.FONT, cacheFont);
        attributes.put(TextAttribute.RUN_DIRECTION, 
                       ltr ? TextAttribute.RUN_DIRECTION_LTR : TextAttribute.RUN_DIRECTION_RTL);

        AttributedCharacterIterator aci = getElementACI(para, attributes);

        return aci;
    }

    Paragraph createParagraph(Element para) {

        char[] chars = getParagraphText(para);
        boolean ltr = contextual ?
	    Bidi.defaultIsLTR(chars, 0, chars.length, defaultIsLTR) :
	    defaultIsLTR;
        AttributedCharacterIterator aci = getParagraphACI(para, chars, ltr);

        Paragraph p = new Paragraph(para, new TabLayout[] {});

        updateLayouts(aci, 0, aci.getEndIndex() - aci.getBeginIndex(), 0, p);

        return p;
    }

    // synching lines
    // get the origin of the edit
    // get the first line affected by the edit
    // set the measurer to the start of that line
    // generate a line
    // if we haven't dealt with all the new text, continue
    // otherwise we're in the old text again.  does our line end match
    // the line end of one of our original lines.  if so we're done.
    
    DamageInfo updateLayouts(AttributedCharacterIterator aci, 
                             int offset, int added, int removed, 
                             Paragraph p) {

        int cc = aci.getEndIndex() - aci.getBeginIndex();
        int tabCount = 0;
        char[] chars = new char[cc];
        {
            int i = 0;
            char ch = aci.first();
            while (ch != aci.DONE) {
                if (ch == '\t') {
                    ++tabCount;
                }
                chars[i++] = ch;

                ch = aci.next();
            }
        }

        BreakIterator breakIter = wordWrap ? BreakIterator.getLineInstance() : BreakIterator.getCharacterInstance();

        int lenm1 = cc - 1;
        LineBreakMeasurer measurer = new LineBreakMeasurer(aci, breakIter, cacheFRC);
        Vector layoutv = new Vector();
        Vector linev = new Vector();
        int tabLimit = tabCount == 0 ? cc : -1;

        float wrapWidth = width - 2 * HPAD;
        if (wrapWidth < 0) {
            throw new InternalError("wrapWidth < 0");
        }
        float tw = Math.abs(tLineTabs.tabWidth);

        // synch up to position of first changed line
        // pos is the position within the paragraph text
        // lineindex is the index of the line we need to regenerate
        int start = 0;
        int lineIndex = 0;
        for (; lineIndex < p.layouts.length; ++lineIndex) {
            int llen = p.layouts[lineIndex].getCharacterCount();
            if (start + llen > offset) {
                // !!! must adjust for word wrap back to of previous line
                // for now always start at line before the actual edit

                if (wordWrap && lineIndex > 0) {
                    --lineIndex;
                    start -= p.layouts[lineIndex].getCharacterCount();
                }
                break;
            }
            start += llen;
        }
        measurer.setPosition(start);

        int oldLineIndex = lineIndex;
        int oldstart = start;
        int delta = added - removed;

        lineloop:
        while (measurer.getPosition() != cc) {
            // init width;
            float pos = 0;
            // init vector for new line
            layoutv.clear();

            TabLayout line = null;

        layoutloop:
            while (true) {
                // collect layouts for line

                int textpos = measurer.getPosition();
                if (tabLimit <= textpos) {
                    tabLimit = textpos;
                    while (tabLimit < lenm1 && chars[tabLimit] != '\t') {
                        ++tabLimit;
                    }
                    ++tabLimit;
                }

                TextLayout layout = measurer.nextLayout(wrapWidth - pos, tabLimit, layoutv.size() != 0);

                if (layout != null) {
                    pos += layout.getAdvance();
                    pos = TabLayout.FixedTabs.gridAway(pos, 0, tw);
                    layoutv.addElement(layout);

                    if (pos < width) {
                        continue layoutloop;
                    }
                }

                TextLayout[] layouts = new TextLayout[layoutv.size()];
                for (int i = 0; i < layouts.length; ++i) {
                    layouts[i] = (TextLayout)layoutv.elementAt(i);
                }

                line = TabLayout.create(layouts, chars[measurer.getPosition()-1] == '\t');

                // check to see if we need the first line
                if (linev.size() == 0 
                    && lineIndex < p.layouts.length
                    && line.getCharacterCount() == p.layouts[lineIndex].getCharacterCount()
                    && start + line.getCharacterCount() <= offset) {

                    // didn't need that line after all
                    ++lineIndex;
                    ++oldLineIndex;
                    start += line.getCharacterCount();
                    oldstart = start;
                    continue lineloop;
                } else {
                    // we need it
                    linev.addElement(line);
                }

                break;
            }

            // see if we can synch up with existing lines.  If so, we stop.
            start += line.getCharacterCount();
            if (start >= offset + added) {
                // handled all the new text
                while (oldLineIndex < p.layouts.length && oldstart + delta < start) {
                    oldstart += p.layouts[oldLineIndex++].getCharacterCount();
                }

                if (oldstart + delta == start) {
                    break;
                }
            }           
        }

        // the first line replaced is at lineIndex
        // the number of lines deleted is at oldLineIndex - lineIndex
        // the number of lines added is in linev.size()
        int nlines = lineIndex + linev.size() + (p.layouts.length - oldLineIndex);
        TabLayout[] lines = new TabLayout[nlines];
        int n = 0;
        for (int i = 0; i < lineIndex; ++i) {
            lines[n++] = p.layouts[i];
        }
        for (int i = 0; i < linev.size(); ++i) {
            lines[n++] = (TabLayout)linev.elementAt(i);
        }
        for (int i = oldLineIndex; i < p.layouts.length; ++i) {
            lines[n++] = p.layouts[i];
        }

        p.layouts = lines;

        return new DamageInfo(lineIndex, oldLineIndex - lineIndex, linev.size());
    }

    int getLineIndex(int offset) {
        Element map = getElement();
        int count = map.getElementCount();
        int index = 0;
        for (int i = 0; i < count; ++i) {
            Element elem = map.getElement(i);
            Paragraph p = getParagraph(elem);
            int cc = p.getCharacterCount();
            if (offset < cc) {
                return index + p.getLineIndex(offset);
            }
            index += p.getLineCount();
            offset -= cc;
        }

        return index;
    }
            
    Paragraph getParagraphInfoForLine(int lineIndex) {
        // side effects:
        //   tParaLineIndex is the line index at the start of the returned paragraph

        Element e = getElement();
        int count = e.getElementCount();
        
        tParaLineIndex = 0;
        for (int i = 0; i < count; ++i) {
            Element paraElem = e.getElement(i);
            Paragraph p = getParagraph(paraElem);
            int lineCount = p.getLineCount();
            if (lineIndex < tParaLineIndex + lineCount) {
                return p;
            }
            tParaLineIndex += lineCount;
        }

        return null;
    }

    Paragraph getParagraphInfoForIndex(int paraIndex) {
        // side effects:
        //   tParaLineIndex is the line index at the start of the returned paragraph
        Element e = getElement();

        tParaLineIndex = 0;
        for (int i = 0; i < paraIndex; ++i) {
            Element paraElem = e.getElement(i);
            Paragraph p = getParagraph(paraElem);
            tParaLineIndex += p.getLineCount();
        }

        return getParagraph(e.getElement(paraIndex));
    }

    // --- layout -----------------------------------------------------------------

    TabLayout getLayoutInfo(int lineIndex, Rectangle2D alloc) {
        Paragraph p = getParagraphInfoForLine(lineIndex);
        
        int localLineIndex = lineIndex - tParaLineIndex;
        TabLayout l = p.getLayout(localLineIndex);

        tLinePos = p.getLinePos(localLineIndex);
        tLinex = 0;
        tLiney = lineAscent;
        tLineTabs.tabBase = 0;
        tLineTabs.tabWidth = l.isLeftToRight() ? cacheTabWidth : -cacheTabWidth;

        if (alloc != null) {
            if (l.isLeftToRight()) {
                tLinex = (float)alloc.getX();
            } else {
                tLinex = (float)(alloc.getX() + alloc.getWidth());
            }
            tLineTabs.tabBase = tLinex;
            tLiney = (float)(lineIndex * lineHeight + lineAscent);
        }

        return l;
    }

    static AttributedCharacterIterator getElementACI(Element e, Map a) {
        if (e.isLeaf()) {
            return getLeafElementACI(e, a);
        } else {
            int count = e.getElementCount();
            switch (count) {
            case 0: 
                return getLeafElementACI(e, a); // shouldn't happen???
            case 1:
                return getElementACI(e.getElement(0), a);
            default:
                int start = 0;
                int limit = e.getEndOffset() - e.getStartOffset();
                AttributedCharacterIterator[] iters = new AttributedCharacterIterator[count];
                for (int i = 0; i < count; ++i) {
                    Element temp = e.getElement(i);
                    iters[i] = getElementACI(e.getElement(i), a);
                }
                
                return new MultiACI(start, limit, iters);
            }
        }
    }

    static AttributedCharacterIterator getLeafElementACI(Element elem, Map a) {
        AttributedString as = null;

        Document doc = elem.getDocument();
        int p0 = elem.getStartOffset();
        int p1 = elem.getEndOffset();
        int docLength = doc.getLength();
        p1 = Math.min(p1, docLength);

        AttributeSet attr = elem.getAttributes();
        if (attr != null && attr.isDefined(StyleConstants.ComposedTextAttribute)) {
            as = (AttributedString)attr.getAttribute(StyleConstants.ComposedTextAttribute);
            // there is no public api to get the length of an attributed string
            // the only public api to set multiple styles at once on an attributed string
            // requires a range.
            // so you have to know the range separately from the attributed string
            // thank you norbert
            // let's just assume the range is good
            if (p1 > p0) {
                as.addAttributes(a, 0, p1 - p0);
            }
        } else {
            String s = "\n";
            if (p1 != p0) {
                try {
                    s = doc.getText(p0, p1-p0);
                    // have to figure out this stuff
                    if (p1 == docLength && s.charAt(s.length() - 1) != '\n') {
                        s = s + "\n";
                    }
                } catch (BadLocationException e) {
                }
            }
            as = new AttributedString(s);
            as.addAttributes(a, 0, s.length());
        }

        return as.getIterator();
    }

    private static final FontRenderContext defaultFRC = new FontRenderContext(null, false, false);

    void updateMetrics() {
        Component host = getContainer();
        Font f = host.getFont();
        if (f != cacheFont) { // swing holds its fonts so it can do identity comparison
            cacheFont = f;
            Graphics g = host.getGraphics();
            cacheFRC = g == null ? defaultFRC : ((Graphics2D)g).getFontRenderContext();
            cacheTabWidth = (float)(getTabSize() * cacheFont.getStringBounds("m", cacheFRC).getWidth());
            FontMetrics fm = host.getFontMetrics(cacheFont);
            lineHeight = fm.getHeight();
            lineAscent = fm.getAscent();

            clearParagraphCache();

            // we can be asked to update before the view has set our size.  If our size has not been set,
            // don't bother with this computation.
            if (width > 0) {
                recomputeLineCount(); // rebuilds all paragraphs
            }

            preferenceChanged(null, false, true);
        }
    }

    void recomputeLineCount() {
        lineCount = 0;

        Element map = getElement();
        int count = map.getElementCount();
        for (int i = 0; i < count; ++i) {
            Element paraElem = map.getElement(i);
            lineCount += getParagraph(paraElem).getLineCount();
        }
    }
            
    // ---- View methods ----------------------------------------------------

    public float getPreferredSpan(int axis) {
        updateMetrics();

        switch (axis) {
        case View.X_AXIS:
            return width == 0 ? 200 : width;

        case View.Y_AXIS:
            int count = lineCount < 0 ? getElement().getElementCount() : lineCount;
            float height = count * lineHeight;
            return height;

        default:
            throw new IllegalArgumentException("Invalid axis: " + axis);
        }
    }

    public void paint(Graphics g, Shape a) {
        updateMetrics();

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        JTextComponent host = (JTextComponent) getContainer();
        int sel0 = host.getSelectionStart();
        int sel1 = host.getSelectionEnd();
        boolean haveSelection = (sel0 != sel1) && host.getCaret().isSelectionVisible();

        Color unselected = (host.isEnabled()) ? host.getForeground() : host.getDisabledTextColor();
        Color selected = haveSelection ? host.getSelectedTextColor() : unselected;
        boolean doSelection = !selected.equals(unselected);

        Highlighter h = host.getHighlighter();
        LayeredHighlighter dh = (h instanceof LayeredHighlighter) ?
            (LayeredHighlighter)h : null;

        Rectangle2D clipRect = g.getClip().getBounds2D();
        float heightBelow = (float)(alloc.getMaxY() - clipRect.getMaxY());
        int linesBelow = Math.max(0, (int)(heightBelow / lineHeight));
        float heightAbove = (float)(clipRect.getY() - alloc.getY());
        int linesAbove = Math.max(0, (int)(heightAbove / lineHeight));
        int linesTotal = (int)(alloc.getHeight() / lineHeight);
        if (alloc.getHeight() > linesTotal * lineHeight) {
            linesTotal++;
        }

        Element map = getElement();

        int endLine = Math.min(lineCount, linesTotal - linesBelow);

        Graphics2D g2d = (Graphics2D)g;

        for (int line = linesAbove; line < endLine; line++) {
            TabLayout layout = getLayoutInfo(line, alloc);

            int p0 = tLinePos;
            int p1 = p0 + layout.getCharacterCount();

            if (dh != null) {
                if (line == lineCount - 1) {
                    dh.paintLayeredHighlights(g, p0, p1, a, host, this);
                } else {
                    dh.paintLayeredHighlights(g, p0, p1 - 1, a, host, this);
                }
            }

            if (haveSelection && sel0 <= p0 && sel1 >= p1) {
                // all selected
                g2d.setColor(selected);
                layout.draw(g2d, tLinex, tLiney, tLineTabs);
            } else {
                // some or all unselected

                g2d.setColor(unselected);
                layout.draw(g2d, tLinex, tLiney, tLineTabs);

                if (doSelection && (sel0 < p1 || sel1 > p0)) {
                    // some selected
                    Shape clip = g2d.getClip();
                    g2d.setClip(layout.getLogicalHighlightShape(tLinex, tLiney, tLineTabs, sel0 - p0, sel1 - p0, alloc));
                    g2d.setColor(selected);
                    layout.draw(g2d, tLinex, tLiney, tLineTabs);
                    g2d.setClip(clip);
                }
            }
        }
    }

    public int getNextVisualPositionFrom(int pos, Position.Bias b, Shape a, 
                                         int direction, Position.Bias[] biasRet) 
        throws BadLocationException {

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        boolean fwd = b != Position.Bias.Backward;

        int lineIndex = getLineIndex(pos);

        switch (direction) {
        case NORTH:
        case SOUTH:
            if ((direction == NORTH && lineIndex > 0) || (direction == SOUTH && lineIndex < lineCount - 1)) {
                JTextComponent host = (JTextComponent)getContainer();
                Caret caret = host.getCaret();
                Point p = caret != null ? caret.getMagicCaretPosition() : null;
                if (p == null) {
                    TabLayout layout = getLayoutInfo(lineIndex, alloc);
                    TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos - tLinePos) : 
                        TextHitInfo.beforeOffset(pos - tLinePos);
                    float x = layout.position(tLinex, tLiney, tLineTabs, hit);
                    p = new Point((int)x, (int)tLiney);
                }
                
                int newIndex = direction == NORTH ? lineIndex - 1 : lineIndex + 1;
                TabLayout layout = getLayoutInfo(newIndex, alloc);
                TextHitInfo hit = layout.hitTest(tLinex, tLiney, tLineTabs, p.x, tLiney);
                int ix = hit.getInsertionIndex();
                if (ix == layout.getCharacterCount()) {
                    fwd = true;
                    pos = tLinePos + ix - 1;
                } else {
                    fwd = hit.isLeadingEdge();
                    pos = tLinePos + ix;
                }
            }
            break;

        case WEST:
        case EAST:
            TabLayout layout = getLayoutInfo(lineIndex, alloc);
            TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos - tLinePos) : TextHitInfo.beforeOffset(pos - tLinePos);
            hit = layout.getNextHit(hit, direction == EAST);
            if (hit == null || hit.getInsertionIndex() == layout.getCharacterCount()) {
                if (layout.isLeftToRight() == (direction == EAST)) {
                    if (lineIndex < lineCount - 1) {
                        fwd = false;
                        pos = tLinePos + layout.getCharacterCount();
                    }
                } else {
		    if (lineIndex > 0) {
			pos = tLinePos - 1;
			fwd = true;
		    }
                }
                break;
            }
            pos = tLinePos + hit.getInsertionIndex();
            fwd = hit.isLeadingEdge();
            break;

        default:
            throw new IllegalArgumentException("Bad direction: " + direction);
        }

        biasRet[0] = fwd ? Position.Bias.Forward : Position.Bias.Backward;

        return pos;
    }

    public Shape modelToView(int pos, Shape a, Position.Bias b) throws BadLocationException {
        updateMetrics();

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        int lineIndex = getLineIndex(pos);
        TabLayout l = getLayoutInfo(lineIndex, alloc);

        boolean fwd = b != Position.Bias.Backward;
        TextHitInfo hit = fwd ? TextHitInfo.afterOffset(pos - tLinePos) : TextHitInfo.beforeOffset(pos - tLinePos);
        float x = l.position(tLinex, tLiney, tLineTabs, hit);

        // pin cursor to allocation bounds so we don't autoscroll
        if (x < alloc.getMinX()) {
            x = (float)alloc.getMinX();
        } else if (x > alloc.getMaxX()) {
            x = (float)alloc.getMaxX();
        }

        Rectangle2D r = new Rectangle2D.Float(x, tLiney - lineAscent, 1, lineHeight);

        return r;
    }

    public Shape modelToView(int p0, Position.Bias b0, int p1, Position.Bias b1, Shape a) throws BadLocationException {
        updateMetrics();
        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        Rectangle2D.Float bounds = new Rectangle2D.Float((float)alloc.getX(),
                                                         (float) 0,
                                                         (float)alloc.getWidth(),
                                                         lineHeight);

        int startLine = getLineIndex(p0);
        int finishLine = getLineIndex(p1);
        
        if (startLine == finishLine) {
            TabLayout layout = getLayoutInfo(startLine, alloc);
            bounds.y = tLiney - lineAscent;
            return layout.getLogicalHighlightShape(tLinex, tLiney, tLineTabs, p0 - tLinePos, p1 - tLinePos, bounds);
        } else {
            TabLayout layout1 = getLayoutInfo(startLine, alloc);
            bounds.y = tLiney - lineAscent;
            Shape s1 = layout1.getLogicalHighlightShape(tLinex, tLiney, tLineTabs, p0 - tLinePos, p1 - tLinePos, bounds);
            float bottom = bounds.y + lineHeight;

            TabLayout layout2 = getLayoutInfo(finishLine, alloc);
            bounds.y = tLiney - lineAscent;
            Shape s2 = layout2.getLogicalHighlightShape(tLinex, tLiney, tLineTabs, p0 - tLinePos, p1 - tLinePos, bounds);

            GeneralPath gp = new GeneralPath(s1);
            gp.append(s2, false);

            if (startLine < finishLine - 1) {
                bounds.height = bounds.y - bottom;
                bounds.y = bottom;
                gp.append(bounds, false);
            }

            return gp;
        }           
    }

    public int viewToModel(float x, float y, Shape a, Position.Bias[] biasReturn) {
        updateMetrics();

        Shape aa = adjustAllocation(a);
        Rectangle2D alloc = aa.getBounds2D();

        int line = (int)Math.floor((y - alloc.getY()) / lineHeight);
        if (line < 0) {
            biasReturn[0] = Position.Bias.Backward;
            return getStartOffset();
        } else if (line >= lineCount) {
            biasReturn[0] = Position.Bias.Forward;
            return getEndOffset() - 1;
        } else {
            TabLayout l = getLayoutInfo(line, alloc);
            TextHitInfo hit = l.hitTest(tLinex, tLiney, tLineTabs, x, y);
            int ix = hit.getInsertionIndex();
            if (ix == l.getCharacterCount()) {
                biasReturn[0] = Position.Bias.Forward;
                return tLinePos + ix - 1;
            } else {
                biasReturn[0] = hit.isLeadingEdge() ? Position.Bias.Forward : Position.Bias.Backward;
                return tLinePos + ix;
            }
        }
    }

    public void insertUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        Shape aa = adjustAllocation(a);
        super.insertUpdate(e, aa, f);

        updateDamage(e, aa, f, e.getLength(), 0);
    }

    public void removeUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        Shape aa = adjustAllocation(a);
        super.removeUpdate(e, aa, f);
        updateDamage(e, aa, f, 0, e.getLength());
    }

    public void changedUpdate(DocumentEvent e, Shape a, ViewFactory f) {
        Shape aa = adjustAllocation(a);
        super.changedUpdate(e, aa, f);
        updateDamage(e, aa, f, e.getLength(), e.getLength());
    }

    void updateDamage(DocumentEvent event, Shape aa, ViewFactory f, int addedChars, int removedChars) {

        Component host = getContainer();
        if (host.isShowing() && width > 0) {
            updateMetrics();

            Element map = getElement();
            DocumentEvent.ElementChange ec = event.getChange(map);
            
            Rectangle2D alloc = aa != null ? aa.getBounds2D() : null;

            Element[] added = (ec != null) ? ec.getChildrenAdded() : null;
            Element[] removed = (ec != null) ? ec.getChildrenRemoved() : null;
            if (((added != null) && (added.length > 0)) || 
                ((removed != null) && (removed.length > 0))) {

                int addedLines = 0;
                if (added != null) {
                    for (int i = 0; i < added.length; i++) {
                        addedLines += getParagraph(added[i]).getLineCount();
                    }
                }

                int removedLines = 0;
                if (removed != null) {
                    for (int i = 0; i < removed.length; i++) {
                        removedLines += getParagraph(removed[i]).getLineCount();
                        removeParagraphCache(removed[i]);
                    }
                }

                lineCount += addedLines - removedLines;
                preferenceChanged(null, false, addedLines != removedLines);
                host.repaint();
            } else {
                int paraIndex = map.getElementIndex(event.getOffset());
                Paragraph p = getParagraphInfoForIndex(paraIndex);

                int localOffset = event.getOffset() - p.getPos();

                DamageInfo damage = p.update(localOffset, addedChars, removedChars);

                if (damage.linesAdded != damage.linesRemoved) {
                    lineCount += damage.linesAdded - damage.linesRemoved;
                    preferenceChanged(null, false, true);
                }

                // repaint starting at first line changed.
                // if number of lines changed, paint to bottom of view, otherwise just
                // repaint changed lines
                
                if (alloc != null) {
                    int firstLine = tParaLineIndex + damage.lineStart;
                    int lastLine = damage.linesAdded == damage.linesRemoved ? 
                        firstLine + damage.linesAdded :
                        lineCount;

                    float top = (float)(alloc.getY() + firstLine * lineHeight);
                    float bottom = (float)(alloc.getY() + lastLine * lineHeight);

                    host.repaint((int)alloc.getX(),
                                 (int)top,
                                 (int)Math.ceil(alloc.getMaxX()),
                                 (int)Math.ceil(bottom));
                }
            }
        }
    }

    public void setSize(float width, float height) {
        if (this.width != width) {

            float oldWidth = this.width;
            this.width = width;

            // remove all paragraphs from cache that were affected
	    if (cacheFont == null) {                                       //ibm.3526
		updateMetrics();                                                  //ibm.3526
	    } else {                                                       //ibm.3526
            boolean paragraphsAffected = false;
            Element map = getElement();
            int count = map.getElementCount();
            for (int i = 0; i < count; ++i) {
                Element para = map.getElement(i);
		    //Paragraph p = (Paragraph)paragraphCache.get(para);          //ibm.3526
		    //if (p != null && p.affectedByWidthChange(oldWidth, width)) {    //ibm.3526
                Paragraph p = getParagraph(para);
                if (p.affectedByWidthChange(oldWidth, width)) {
                    removeParagraphCache(para);
                    paragraphsAffected = true;
                }
            }

            // if any affected (likely) recompute the line count
            if (paragraphsAffected) {
                int oldLineCount = lineCount;
                recomputeLineCount();
                preferenceChanged (null, false, oldLineCount != lineCount);
            }
            }                                                       //ibm.3526
            
            getContainer().invalidate();
        }
    }

    protected Shape adjustAllocation(Shape a) {
        if (a != null) {
            Rectangle bounds = a.getBounds();
            bounds.x += HPAD;
            bounds.width -= HPAD * 2;
            return bounds;
        }
        return a;
    }
}
