/*
 * @(#)src/classes/sov/javax/swing/plaf/basic/TabLayout.java, as131, as131, 20031014 1.5.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM Java(tm)2 SDK, Standard Edition, v 1.3.1
 * (C) Copyright IBM Corp. 1998, 2001. All Rights Reserved
 * ===========================================================================
 */

/* 
 *
 * ===========================================================================
 *
 *
 *
 * ===========================================================================
 * Change activity:
 *
 * Reason  Date   Origin  Description
 * ------  ----   ------  ---------------------------------------------------- 
 *         270799         New file.
 *         280999         Update.
 *
 * ===========================================================================
 * Module Information:
 *		
 * DESCRIPTION: Hindi, Thai, and Bidi enhancements.
 * ===========================================================================
 */

//ibm.2722

/*
 * @(#)TabLayout.java	1.3 99/10/05
 *
 * Tabbed text line using TextLayout.
 *
 *
 * This file requires Java1.2.
 */ 

package javax.swing.plaf.basic;

import java.awt.*;
import java.awt.event.*;
import java.awt.font.*;
import java.awt.geom.*;
import java.text.*;

class TabLayout {
    TextLayout[] multi;
    int cc;
    boolean finalTab;

    static interface Tabs {

        /**
         * Return next tab position.  Input and output are not line relative.
         */
        public float nextTab(float x);
    }

    // simple tabs implementation

    /*
     * The tab base is not line relative.
     * The tab width is negative for rtl lines.
     */
    static class FixedTabs implements Tabs {
        float tabBase;
        float tabWidth;

        FixedTabs() {
            tabBase = 0;
            tabWidth = 50;
        }

        FixedTabs(float tabBase, float tabWidth) {
            this.tabBase = tabBase;
            this.tabWidth = tabWidth;
        }

        public float nextTab(float x) {
            return gridAway(x, tabBase, tabWidth);
        }

        static final float gridAway(float x, float base, float width) {
            return base + ((int)((x - base) / width) + 1) * width;
        }

        public String toString() {
            return "tb: " + tabBase + " tw: " + tabWidth;
        }
    }

    static TabLayout create(AttributedCharacterIterator aci, FontRenderContext frc) {

        int cc = aci.getEndIndex() - aci.getBeginIndex();

        int segCount = 1;
        char[] chars = new char[cc];
        {
            int i = 0;
            char ch = aci.first();
            while (ch != aci.DONE) {
                if (ch == '\t') {
                    ++segCount;
                }
                chars[i++] = ch;

                ch = aci.next();
            }
        }

        int lenm1 = cc - 1;
        boolean finalTab = chars[lenm1] == '\t';
        if (finalTab) {
            --segCount;
        }

        TextLayout[] layouts = new TextLayout[segCount];
        if (segCount == 1) {
            layouts[0] = new TextLayout(aci, frc);
        } else {
            LineBreakMeasurer m = new LineBreakMeasurer(aci, frc);

            for (int i = 0, limit = 0; i < segCount; ++i) {
                while (limit < lenm1 && chars[limit] != '\t') {
                    ++limit;
                }
                ++limit;

                layouts[i] = m.nextLayout(Float.MAX_VALUE, limit, i == 0);
            }
        }

        return create(layouts, finalTab);
    }

    static TabLayout create(TextLayout[] layouts, boolean finalTab) {
        TabLayout result = new TabLayout();

        result.multi = layouts;
        result.finalTab = finalTab;
        result.cc = 0;
        for (int i = 0; i < layouts.length; ++i) {
            result.cc += layouts[i].getCharacterCount();
        }
        
        return result;
    }

    boolean isLeftToRight() {
        return multi[0].isLeftToRight();
    }

    float getAscent() {
        return multi[0].getAscent();
    }

    float getDescent() {
        return multi[0].getDescent();
    }

    float getLeading() {
        return multi[0].getLeading();
    }

    float getHeight() {
        return getAscent() + getDescent() + getLeading();
    }

    int getCharacterCount() {
        return cc;
    }

    /*
     * X and Y are the origin of the line.  On RTL lines, x is the right of the line.
     */
    void draw(Graphics g, float x, float y, Tabs tabs) {
        Graphics2D g2d = (Graphics2D)g;
        boolean ltr = isLeftToRight();
        for (int i = 0; i < multi.length; ++i) {
            TextLayout l = multi[i];
            float adv = l.getAdvance();
            if (ltr ) {
                l.draw(g2d, x, y);
                x += adv;
            } else {
                x -= adv;
                l.draw(g2d, x, y);
            }
            x = nextTab(tabs, x);
        }
    }

    /*
     * Horizontal only.
     * X and Y are the origin of the line.  On RTL lines, x is the right of the line.
     * Result is negative for rtl lines.
     */
    float getAdvance(float x, float y, Tabs tabs) {
        boolean ltr = isLeftToRight();
        float initialX = x;
        for (int i = 0; i < multi.length; ++i) {
            TextLayout l = multi[i];
            float adv = l.getAdvance();

            x = ltr ? x + adv : x - adv;
            if (i < multi.length - 1 || finalTab) {
                x = nextTab(tabs, x);
            }
        }

        return x - initialX;
    }

    /*
     * Horizontal only.
     * X and Y are the origin of the line in the component coordinate system.  hitX and hitY are the
     * hit locations, also in this coordinate system (not relative to line).
     */
    TextHitInfo hitTest(float x, float y, Tabs tabs, float hitX, float hitY) {
        int pos = 0;
        boolean fwd = false;
        boolean miss = true;
        boolean ltr = isLeftToRight();

        float lineX = x;

        if (hitX == lineX || (ltr == (hitX > lineX))) {
            for (int i = 0; i < multi.length; ++i) {
                TextLayout l = multi[i];
                float adv = l.getAdvance();
                int cc = l.getCharacterCount();
            
                float leftX;
                float rightX;
                float tabX;
                if (ltr) {
                    leftX = lineX;
                    rightX = lineX + adv;
                    if (i < multi.length - 1 || finalTab) {
                        tabX = nextTab(tabs, rightX);
                    } else {
                        tabX = rightX;
                    }
                    if (hitX < tabX) {
                        if (hitX < rightX) {
                            TextHitInfo hit = l.hitTestChar(hitX - leftX, 0);
                            pos += hit.getInsertionIndex();
                            fwd = hit.isLeadingEdge();
                        } else {
                            if (hitX - rightX < tabX - hitX) {
                                pos += cc - 1;
                                fwd = true;
                            } else {
                                pos += cc;
                                fwd = false;
                            }
                        }
                        miss = false;
                        break;
                    }

                } else {
                    leftX = lineX - adv;
                    rightX = lineX;
                    if (i < multi.length - 1 || finalTab) {
                        tabX = nextTab(tabs, leftX);
                    } else {
                        tabX = leftX;
                    }
                    if (hitX > tabX) {
                        if (hitX > leftX) {
                            TextHitInfo hit = l.hitTestChar(hitX - leftX, 0);
                            pos += hit.getInsertionIndex();
                            fwd = hit.isLeadingEdge();
                        } else {
                            if (leftX - hitX < hitX - tabX) {
                                pos += cc - 1;
                                fwd = true;
                            } else {
                                pos += cc;
                                fwd = false;
                            }
                        }
                        miss = false;
                        break;
                    }
                }

                pos += cc;
                lineX = tabX;
            }
        } else {
            miss = false; // before first char, so keep !fwd
        }
        if (miss) {
            fwd = true;
        }

        return fwd ? TextHitInfo.afterOffset(pos) : TextHitInfo.beforeOffset(pos);
    }

    /*
     * Return the position of the offset/bias.  Horizontal only.
     * X and Y are the origin of the line.  Result is not relative to line.
     */
    float position(float x, float y, Tabs tabs, TextHitInfo hit) {
        
        int pos = hit.getInsertionIndex();
        boolean bias = hit.isLeadingEdge();

        boolean ltr = isLeftToRight();

        for (int i = 0; i < multi.length; ++i) {
            TextLayout l = multi[i];
            float adv = l.getAdvance();
            int cc = l.getCharacterCount();

            if (pos < cc) {
                TextHitInfo nhit = bias ? TextHitInfo.afterOffset(pos) : TextHitInfo.beforeOffset(pos);
                x += l.getCaretInfo(nhit)[0];
                if (!ltr) {
                    x -= l.getAdvance();
                }
                break;
            }

            pos -= cc;
            x = ltr ? x + adv : x - adv;
            if (i < multi.length - 1 || finalTab) {
                x = nextTab(tabs, x);
            }
        }

        return x;
    }

    /*
     * Implements swing-style 'flag' carets.
     * returns null if walk off either end of line.
     */
    TextHitInfo getNextHit(TextHitInfo hit, boolean right) {
        boolean ltr = isLeftToRight();
        boolean wld = ltr == right;

        TextHitInfo result = null;

        int chpos = hit.getCharIndex();
        int delta = 0;
        for (int i = 0; i < multi.length; ++i) {
            TextLayout l = multi[i];
            int cc = l.getCharacterCount();
            if ((chpos >= delta + cc) && (i < multi.length - 1)) {
                delta += cc;
            } else {
                result = getNextSwingHit(i, delta, hit, right);
                break;
            }
        }
        
        return result;
    }

    /**
     * i is the index of the current layout
     * delta is index of the character that starts the current layout
     * hit is the 'global' hit
     * right is true if the caret is moving right
     */
    TextHitInfo getNextSwingHit(int i, int delta, TextHitInfo hit, boolean right) {

        TextLayout l = multi[i];
        TextHitInfo localHit = hit.getOffsetHit(-delta);

        boolean ltr = l.isLeftToRight();
        boolean wld = ltr == right;

        int ix = localHit.getInsertionIndex();
        boolean ib = localHit.isLeadingEdge();

        int ox;
        boolean ob;
        boolean oltr;

        boolean done = false;

        TextHitInfo other = l.getVisualOtherHit(localHit);
        int ocx = other.getCharIndex();
        if ((ocx == -1) && (i > 0)) {
            TextLayout ol = multi[i-1];
            int olcc = ol.getCharacterCount();
            other = ol.getVisualOtherHit(TextHitInfo.afterOffset(olcc));
            ox = other.getInsertionIndex() - olcc;
            ob = other.isLeadingEdge();
            oltr = (ol.getCharacterLevel(other.getCharIndex()) & 0x1) == 0;
        } else if ((ocx == l.getCharacterCount()) && (i < (multi.length - 1))) {
            TextLayout ol = multi[i+1];
            int cc = l.getCharacterCount();
            other = ol.getVisualOtherHit(TextHitInfo.beforeOffset(0));
            ox = other.getInsertionIndex() + cc;
            ob = other.isLeadingEdge();
            oltr = (ol.getCharacterLevel(other.getCharIndex()) & 0x1) == 0;
        } else {
            ox = other.getInsertionIndex();
            ob = other.isLeadingEdge();
            oltr = (l.getCharacterLevel(ocx) & 0x1) == 0; // -1, cc are ok values for gCL
        }

        if (ix != ox) {
            // at directional boundary, do we cross, or toggle?
            // toggle if we're on the left and moving right, or vice-versa
            // we're on the left if other is on the right
            // other is on the right if it's leading an ltr char, or
            // trailing an rtl char, i.e. ob == oltr

            if ((ob == oltr) == right) {
                ix = ox;
                ib = ob;
                done = true;
            }
        }

        if (!done) {
            TextHitInfo newHit = right ? l.getNextRightHit(localHit) : l.getNextLeftHit(localHit);
            if (newHit == null) {
                if (wld && (i < multi.length - 1)) {
                    int cc = l.getCharacterCount();
                    delta += cc;
                    ix -= cc;
                    l = multi[++i];
                    localHit = TextHitInfo.beforeOffset(0);
                    newHit = right ? l.getNextRightHit(localHit) : l.getNextLeftHit(localHit);
                } else if (!wld && (i > 0)) {
                    l = multi[--i];
                    int cc = l.getCharacterCount();
                    delta -= cc;
                    ix += cc;
                    localHit = TextHitInfo.afterOffset(cc);
                    newHit = right ? l.getNextRightHit(localHit) : l.getNextLeftHit(localHit);
                } else {
                    return null;
                }
            }

            ocx = newHit.getCharIndex();
            if ((ocx == -1) && (i > 0)) {
                TextLayout ol = multi[i-1];
                int olcc = ol.getCharacterCount();
                other = ol.getVisualOtherHit(TextHitInfo.afterOffset(olcc));
                ix = other.getInsertionIndex() - olcc;
                ib = other.isLeadingEdge();
            } else if ((ocx == l.getCharacterCount()) && (i < (multi.length - 1))) {
                TextLayout ol = multi[i+1];
                int cc = l.getCharacterCount();
                other = ol.getVisualOtherHit(TextHitInfo.beforeOffset(0));
                ix = other.getInsertionIndex() + cc;
                ib = other.isLeadingEdge();
            } else {
                ix = newHit.getInsertionIndex();
                ib = newHit.isLeadingEdge();
            }

            other = l.getVisualOtherHit(newHit);
            ocx = other.getCharIndex();
            if ((ocx == -1) && (i > 0)) {
                TextLayout ol = multi[i-1];
                int olcc = ol.getCharacterCount();
                other = ol.getVisualOtherHit(TextHitInfo.afterOffset(olcc));
                ox = other.getInsertionIndex() - olcc;
                ob = other.isLeadingEdge();
                oltr = (ol.getCharacterLevel(other.getCharIndex()) & 0x1) == 0;
            } else if ((ocx == l.getCharacterCount()) && (i < (multi.length - 1))) {
                TextLayout ol = multi[i+1];
                int cc = l.getCharacterCount();
                other = ol.getVisualOtherHit(TextHitInfo.beforeOffset(0));
                ox = other.getInsertionIndex() + cc;
                ob = other.isLeadingEdge();
                oltr = (ol.getCharacterLevel(other.getCharIndex()) & 0x1) == 0;
            } else {
                ox = other.getInsertionIndex();
                ob = other.isLeadingEdge();
                oltr = (l.getCharacterLevel(ocx) & 0x1) == 0;
            }

            if (ix != ox) {
                // at directional boundary again
                // toggle if we're on the left and *not* moving right, or vice-versa
                // we're on the left if other is on the right
                // other is on the right if it's leading an ltr char, or
                // trailing an rtl char, i.e. ob == oltr
                
                if ((ob == oltr) != right) {
                    ix = ox;
                    ib = ob;
                    done = true;
                }
            }
        }

        ix += delta;

        return ib ? TextHitInfo.afterOffset(ix) : TextHitInfo.beforeOffset(ix);
    }

    Shape getLogicalHighlightShape(float x, float y, Tabs tabs, int start, int limit, Rectangle2D bounds) {
        if (start < 0) {
            start = 0;
        }
        if (limit > cc) {
            limit = cc;
        }
        if (start >= limit) {
            return new Rectangle(0, 0, 0, 0);
        }

        Rectangle2D.Float b = null;
        if (bounds != null) {
            b = new Rectangle2D.Float((float)bounds.getX() - x, 
                                      (float)bounds.getY() - y, 
                                      (float)bounds.getWidth(), 
                                      (float)bounds.getHeight());
        } else {
            b = new Rectangle2D.Float(-20000, 
                                      -getAscent(), 
                                      40000, 
                                      getAscent() + getDescent());
        }

        boolean ltr = isLeftToRight();
        GeneralPath gp = null;
        int pos = 0;

        float left = x;
        float right = x;
        float tab = x;

        for (int i = 0; i < multi.length && pos < limit; ++i) {
            TextLayout l = multi[i];

            float adv = l.getAdvance();
            if (ltr) {
                left = tab;
                right = tab + adv;
            } else {
                right = tab;
                left = tab - adv;
            }
            if (i < multi.length - 1 || finalTab) {
                if (ltr) {
                    tab = nextTab(tabs, right);
                } else {
                    tab = nextTab(tabs, left);
                }
            }

            int lcc = l.getCharacterCount();
            if (pos + lcc > start) {
                int st = Math.max(start - pos, 0);
                int fn = Math.min(limit - pos, lcc);
                float locall = Math.min(left, tab);
                float localr = Math.max(right, tab);
                b.x = locall - left;
                b.width = localr - locall;

                Shape s = l.getLogicalHighlightShape(st, fn, b);
                s = AffineTransform.getTranslateInstance(left, y).createTransformedShape(s);
                if (gp == null) {
                    gp = new GeneralPath(s);
                } else {
                    gp.append(s, false);
                }
            }

            pos += lcc;
        }

        return gp;
    }

    float nextTab(Tabs tabs, float x) {
        return tabs == null ? x : tabs.nextTab(x);
    }
}
