/*
 * @(#)src/classes/sov/javax/swing/text/Bidi.java, as131, as131, 20031014 1.9.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM Java(tm)2 SDK, Standard Edition, v 1.3.1
 * (C) Copyright IBM Corp. 1998, 2001. All Rights Reserved
 * ===========================================================================
 */

/**
 * @(#)Bidi.java	1.7 99/07/22
 *
 * (C) Copyright Taligent, Inc. 1997 - All Rights Reserved
 *
 * Bidi implementation for Unicode 3.0.
 *
 *   The original version of this source code and documentation is copyrighted
 * and owned by Taligent, Inc., a wholly-owned subsidiary of IBM. These
 * materials are provided under terms of a License Agreement between Taligent
 * and Sun. This technology is protected by multiple US and International
 * patents. This notice and attribution to Taligent may not be removed.
 *   Taligent is a registered trademark of Taligent, Inc.
 */

package javax.swing.text;

import java.io.*;

/**
 * Bidi 3.0 implementation.
 */
final class Bidi {
  private boolean ltr;
  private byte[] dirs;
  private byte[] levels;
  private int[] l2vMap;
  private int[] v2lMap;

  /**
   * Return the number of elements the bidi represents.
   */
  int getLength() {
    return levels.length;
  }

  /**
   * Return true if the base run direction defined for this bidi is
   * left to right, false if it is right to left.
   */
  boolean isDirectionLTR() {
    return ltr;
  }

  /**
   * Return a mapping of the bidirectional information from
   * logical to visual position.  If mapping is canonical, return
   * null;
   */
  int[] getLogicalToVisualMap() {
    if (l2vMap == null) {
      l2vMap = getInverseOrder(getVisualToLogicalMap());
    }
    return l2vMap;
  }

  /**
   * Return a mapping of the bidirectional information from
   * visual to logical position.  If mapping is canonical, return
   * null;
   */
  int[] getVisualToLogicalMap() {
    if (v2lMap == null) {
      v2lMap = createVisualToLogicalMap(levels);
    }
    return v2lMap;
  }

  /**
   * Return the resolved level array.
   */
  byte[] getLevels() {
    return levels;
  }

  /**
   * Return a bidi representing the given subrange of this bidi, with
   * line reordering applied to counterdirectional trailing whitespace.
   */
  Bidi createLineBidi(int start, int limit) {
    byte[] newLevels = new byte[limit - start];
    System.arraycopy(levels, start, newLevels, 0, newLevels.length);

    if (dirs != null) {
      byte x = (byte)(ltr ? 0 : 1);
	loop:                                                              //ibm.597
      for (int i = newLevels.length; --i >= 0;) {
                if ((newLevels[i] % 2) != x) {                      //ibm.597
		    switch (dirs[start + i]) {                                    //ibm.597
		    case WS:                                                      //ibm.597
		    case PDF:                                                     //ibm.597
		    case LRE:                                                     //ibm.597
		    case LRO:                                                     //ibm.597
		    case RLE:                                                     //ibm.597
		    case RLO:                                                     //ibm.597
		    case BN:                                                      //ibm.597
	newLevels[i] = x;
			continue loop;                                                   //ibm.597
		    default:                                                      //ibm.597
			break loop;                                                      //ibm.597
      }
    }
		break loop;                                                       //ibm.597
	    }                                                              //ibm.597
	}                                                                  //ibm.597

    return new Bidi(newLevels, ltr);
  }

  /**
   * Return the level of the element at pos.
   */
  int getLevelAt(int pos) {
    return levels[pos];
  }

  /**
   * Get the limit of the level run starting at start.
   */
  int getLevelLimit(int start) {
    byte l = levels[start];
    while (++start < levels.length && levels[start] == l) {}
    return start;
  }

  /**
   * Create bidi information about the provided paragraph of text.
   * Use bidi rules to default line direction.
   */
  Bidi(char[] text) {
        this(getDirectionCodeArray(text), null, defaultIsLTR(text, 0, text.length, true));  //ibm.597
  }

  /**
   * Create bidi information about the provided paragraph of text.
   * Embedding codes in the text will be processed.
   * @param text the characters in the paragraph.
   * @param ltr true if the paragraph run direction is LTR,
   * false if it is RTL.
   */
  Bidi(char[] text, boolean ltr) {
        this(getDirectionCodeArray(text), null, ltr);               //ibm.597
  }

  /**
   * Create bidi information about the provided paragraph of text, using the
   * provided embedding information.  Explicit embedding codes in the text, if
   * any, are not processed.
   * @param text the characters in the paragraph.
   * @param embs the embedding data.  Values from 0-15 are embeddings at the
   * indicated level.  Values from 16-31 are overrides at the corresponding levels.
   * This may modify the data in embs.
   * @param ltr true if the run direction for the paragraph is LTR,
   * false if it is RTL.
   */
  Bidi(char[] text, byte[] embs, boolean ltr) {
        this(getDirectionCodeArray(text), embs, ltr);               //ibm.597
      }


    /*                                                              //ibm.597
     * Create bidi information about the provided paragraph of text, using the  //ibm.597
     * provided embedding information.  Explicit embedding codes, if any, are   //ibm.597
     * not processed.                                               //ibm.597
     * @param dirs the bidi direction codes in the paragraph.       //ibm.597
     * @param embs the embedding data.  The low seven bits indicate the level,  //ibm.597
     * only values from 0-61 are valid.  The high bit, if set, indicates an //ibm.597
     * override instead of an embedding.                            //ibm.597
     * This may modify the data in embs.                            //ibm.597
     * @param ltr true if the run direction for the paragraph is LTR,   //ibm.597
     * false if it is RTL.                                          //ibm.597
     */                                                             //ibm.597
    Bidi(byte[] dirs, byte[] embs, boolean ltr) {                   //ibm.597
        long flags = getFlags(dirs);                                //ibm.597
        if (!ltr) {                                                 //ibm.597
            flags |= FLAG_RTL;                                      //ibm.597
    }
        if (embs != null) {                                         //ibm.597
            flags |= FLAG_EXTERNAL_EMBS;                            //ibm.597
        } else {                                                    //ibm.597
            embs = new byte[dirs.length];                           //ibm.597
        }                                                           //ibm.597

        applyBidiRules(dirs, embs, flags);                          //ibm.597

    this.ltr = ltr;
        this.dirs = dirs; // save original dirs                     //ibm.597
    this.levels = embs;
  }

  /**
     * !!! for testing purposes
     * Protected constructor used by createLineBidi.
   */
    Bidi(byte[] levels, boolean ltr) {                              //ibm.597
    this.ltr = ltr;
    this.dirs = null;
    this.levels = levels;
  }

    /* so clients can construct any direction type array. */        //ibm.597

  static final byte L  = 0;    /* left to right (strong) */
    static final byte LRE = 1;   /* left to right embedding */      //ibm.597
    static final byte LRO = 2;   /* left to right override */       //ibm.597
    static final byte R   = 3;   /* right to left (strong) */       //ibm.597
    static final byte AL  = 4;   /* arabic right to left (strong) */    //ibm.597
    static final byte RLE = 5;   /* right to left embedding */      //ibm.597
    static final byte RLO = 6;   /* right to left override */       //ibm.597
    static final byte PDF = 7;   /* pop display formatting */       //ibm.597
    static final byte EN  = 8;   /* european number (weak) */       //ibm.597
    static final byte ES  = 9;   /* european number separator (weak) */ //ibm.597
    static final byte ET  = 10;  /* european number terminator (weak) */    //ibm.597
    static final byte AN  = 11;  /* arabic number (weak) */         //ibm.597
    static final byte CS  = 12;  /* common number separator (weak) */   //ibm.597
    static final byte NSM = 13;  /* non-spacing mark */             //ibm.597
    static final byte BN  = 14;  /* boundary neutral */             //ibm.597
    static final byte B   = 15;  /* block separator */              //ibm.597
    static final byte S   = 16;  /* segment separator */            //ibm.597
    static final byte WS  = 17;  /* whitespace */                   //ibm.597
    static final byte ON  = 18;  /* other neutral */                //ibm.597

    /* The embedding/override level stack limit */                  //ibm.597

    static final char NUMLEVELS = 62;                               //ibm.597

    static final long FLAG_RTL = 1 << 30;                           //ibm.597
    static final long FLAG_EXTERNAL_EMBS = 1 << 29;                 //ibm.597

    static final long FLAG_L = 1 << L;                              //ibm.597
    static final long FLAG_LRE = 1 << LRE;                          //ibm.597
    static final long FLAG_LRO = 1 << LRO;                          //ibm.597
    static final long FLAG_R = 1 << R;                              //ibm.597
    static final long FLAG_AL = 1 << AL;                            //ibm.597
    static final long FLAG_RLE = 1 << RLE;                          //ibm.597
    static final long FLAG_RLO = 1 << RLO;                          //ibm.597
    static final long FLAG_PDF = 1 << PDF;                          //ibm.597
    static final long FLAG_EN = 1 << EN;                            //ibm.597
    static final long FLAG_ES = 1 << ES;                            //ibm.597
    static final long FLAG_ET = 1 << ET;                            //ibm.597
    static final long FLAG_AN = 1 << AN;                            //ibm.597
    static final long FLAG_CS = 1 << CS;                            //ibm.597
    static final long FLAG_NSM = 1 << NSM;                          //ibm.597
    static final long FLAG_BN = 1 << BN;                            //ibm.597
    static final long FLAG_B = 1 << B;                              //ibm.597
    static final long FLAG_S = 1 << S;                              //ibm.597
    static final long FLAG_WS = 1 << WS;                            //ibm.597
    static final long FLAG_ON = 1 << ON;                            //ibm.597

    static final long MASK_EMBS = FLAG_LRO | FLAG_LRE | FLAG_RLO | FLAG_RLE | FLAG_PDF; //ibm.597
    static final long MASK_R = FLAG_R | FLAG_AL | FLAG_AN | FLAG_RLO | FLAG_RLE | FLAG_RTL; //ibm.597
    static final long MASK_NUM = FLAG_EN | FLAG_AN;                 //ibm.597
    static final long MASK_SEP = FLAG_ES | FLAG_CS;                 //ibm.597

  /**
   * Modify the dir and levels arrays by applying the bidi algorithm to the arrays, changing
   * them in place.  On output, the direction array contains only L, R, or AR codes, and
   * the levels array contains the resolved levels.
   */
    static void applyBidiRules(byte[] dirs, byte[] levels, long flags) {    //ibm.597

        byte[] wdirs = null;                                        //ibm.597

	if ((flags & FLAG_EXTERNAL_EMBS) != 0) {                           //ibm.597
	    wdirs = (byte[])dirs.clone();                                  //ibm.597
	    flags = processEmbeddingArray(wdirs, levels, flags);           //ibm.597
	} else if ((flags & MASK_EMBS) != 0) {                             //ibm.597
	    wdirs = (byte[])dirs.clone();                                  //ibm.597
	    flags = getLevelArray(wdirs, levels, flags);                   //ibm.597
	} else {                                                           //ibm.597
	    byte baselevel = (byte)(((flags & FLAG_RTL) == 0) ? 0 : 1);    //ibm.597
	    for (int i = 0; i < levels.length; ++i) {                      //ibm.597
		levels[i] = baselevel;                                            //ibm.597
  }
	}                                                                  //ibm.597

	if ((flags & MASK_R) != 0) {                                       //ibm.597
	    if (wdirs == null) {                                           //ibm.597
		wdirs = (byte[])dirs.clone();                                     //ibm.597
	    }                                                              //ibm.597
	    resolveWeakTypes(wdirs, levels, flags);                        //ibm.597
	    resolveNeutralTypes(wdirs, levels, flags);                     //ibm.597
	    resolveImplicitLevels(wdirs, levels, flags);                   //ibm.597
	    resolveLineLevels(dirs, levels, flags);                        //ibm.597
	}                                                                  //ibm.597
    }                                                               //ibm.597

    /**
     * Scan dirs and set flags appropriately.
   */
    static long getFlags(byte[] dirs) {                             //ibm.597
        long flags = 0;                                             //ibm.597
        for (int i = 0; i < dirs.length; ++i) {                     //ibm.597
            flags |= 1 << dirs[i];                                  //ibm.597
      }
        return flags;                                               //ibm.597
    }                                                               //ibm.597

    /*                                                              //ibm.597
     * Weak rule processing.                                        //ibm.597
     * Level boundaries are treated as though an L or R started the level for   //ibm.597
     * purposes of NSM and EN processing.                           //ibm.597
     */                                                             //ibm.597
    private static void resolveWeakTypes(byte[] wdirs, byte[] levels, long flags) { //ibm.597
        int start = 0;                                              //ibm.597
	byte paragraphType = ((flags & FLAG_RTL) == 0) ? L : R;            //ibm.597

        while (start < wdirs.length) {                              //ibm.597
            byte level = levels[start];                             //ibm.597

            int limit = start;                                      //ibm.597
            if ((flags & FLAG_EXTERNAL_EMBS) == 0) {                //ibm.597
                // explicit embeddings work ok just off the codes   //ibm.597
                limit = wdirs.length;                               //ibm.597
            } else {                                                //ibm.597
                while (++limit < wdirs.length && levels[limit] == level) {  //ibm.597
        }
            }                                                       //ibm.597

            // NSM processing                                       //ibm.597
            if ((flags & FLAG_NSM) != 0) {                          //ibm.597
		// byte nsmType = (start == 0) ? ON : (((level & 0x1) == 0) ? L : R); //ibm.597
		byte nsmType;                                                     //ibm.597
		if (start > 0) {                                                  //ibm.597
		    byte prevLevel = levels[start-1];                             //ibm.597
		    nsmType = ((Math.max(level, prevLevel) & 0x1) == 0) ? L : R;  //ibm.597
		} else {                                                          //ibm.597
		    nsmType = ((level & 0x1) == 0) ? L : R;                       //ibm.597
		}                                                                 //ibm.597

                // if no external embeddings, start == 0, and the rest just works   //ibm.597
                for (int j = start; j < limit; ++j) {               //ibm.597
                    if (wdirs[j] == NSM) {                          //ibm.597
                        wdirs[j] = nsmType;                         //ibm.597
                    } else if (wdirs[j] != BN) {                    //ibm.597
                        nsmType = wdirs[j];                         //ibm.597
                    }                                               //ibm.597
                }                                                   //ibm.597
            }                                                       //ibm.597

            // AL processing                                        //ibm.597
            if ((flags & FLAG_AL) != 0) {                           //ibm.597
                boolean lastStrongIsArabic = false;                 //ibm.597
                for (int j = start; j < limit; ++j) {               //ibm.597
                    switch (wdirs[j]) {                             //ibm.597
        case L:
        case R:
                        lastStrongIsArabic = false;                 //ibm.597
          break;
                    case AL:                                        //ibm.597
                        lastStrongIsArabic = true;                  //ibm.597
                        wdirs[j] = R;                               //ibm.597
          break;
                    case EN:                                        //ibm.597
                        if (lastStrongIsArabic) {                   //ibm.597
                            wdirs[j] = AN;                          //ibm.597
                        }                                           //ibm.597
          break;
                    }                                               //ibm.597
                }                                                   //ibm.597
            }                                                       //ibm.597

            // Numeric type processing                              //ibm.597
            if ((flags & MASK_NUM) != 0) {                          //ibm.597

                // BN processing                                    //ibm.597
                if ((flags & FLAG_BN) != 0) {                       //ibm.597
                    for (int i = start; i < limit;) {               //ibm.597
                        if (wdirs[i] == BN) {                       //ibm.597
                            int j = i + 1;                          //ibm.597
                            while (j < limit && wdirs[j] == BN) {   //ibm.597
                                ++j;                                //ibm.597
              }

			    byte leading = i > start ? wdirs[i-1] : ON;                  //ibm.597
			    byte trailing = j < limit ? wdirs[j] : ON;                   //ibm.597
                            byte newtype = ON;                      //ibm.597
                            if (leading == EN || trailing == EN) {  //ibm.597
                                newtype = EN;                       //ibm.597
                            } else if (leading == ET || trailing == ET) {   //ibm.597
                                newtype = ET;                       //ibm.597
                            } else if (leading == AN || trailing == AN) {   //ibm.597
                                newtype = AN;                       //ibm.597
                            } else {                                //ibm.597
				flags |= FLAG_ON;                                               //ibm.597
              }
                            while (i < j) {                         //ibm.597
                                wdirs[i++] = newtype;               //ibm.597
            }
          } else {
                            ++i;                                    //ibm.597
          }
                    }                                               //ibm.597
                    // all BN removed by this point                 //ibm.597
                }                                                   //ibm.597

                // ES, CS                                           //ibm.597
                if ((flags & MASK_SEP) != 0) {                      //ibm.597
                    if (start + 2 < limit) {                        //ibm.597
                        byte prev = wdirs[start];                   //ibm.597
                        byte cur = wdirs[start + 1];                //ibm.597
                        for (int i = start + 2; i < limit; ++i) {   //ibm.597
                            byte next = wdirs[i];                   //ibm.597
                            switch (next) {                         //ibm.597
                            case AN:                                //ibm.597
                                if (prev == AN && cur == CS) {      //ibm.597
                                    wdirs[i-1] = AN;                //ibm.597
                                }                                   //ibm.597
          break;

                            case EN:                                //ibm.597
                                if (prev == EN && (cur == CS || cur == ES)) {   //ibm.597
                                    wdirs[i-1] = EN;                //ibm.597
                                }                                   //ibm.597
          break;
        }
                            prev = cur;                             //ibm.597
        cur = next;
      }
    }

                    // remaining ES, CS set to neutral below        //ibm.597
  }

                if ((flags & FLAG_ET) != 0) {                       //ibm.597
                    for (int i = start; i < limit;) {               //ibm.597
                        if (wdirs[i] == ET) {                       //ibm.597
                            int j = i + 1;                          //ibm.597
                            while (j < limit && wdirs[j] == ET) {   //ibm.597
                                ++j;                                //ibm.597
                            }                                       //ibm.597
                            byte newtype = ON;                      //ibm.597
                            if (i > start && wdirs[i-1] == EN) {    //ibm.597
                                newtype = EN;                       //ibm.597
				flags |= EN;                                                    //ibm.597
                            } else if (j < limit && wdirs[j] == EN) {   //ibm.597
                                newtype = EN;                       //ibm.597
				flags |= EN;                                                    //ibm.597
                            } else {                                //ibm.597
				flags |= ON;                                                    //ibm.597
			    }                                                            //ibm.597
                            while (i < j) {                         //ibm.597
                                wdirs[i++] = newtype;               //ibm.597
                            }                                       //ibm.597
                        } else {                                    //ibm.597
                            ++i;                                    //ibm.597
                        }                                           //ibm.597
                    }                                               //ibm.597

                    // all ET removed by this point                 //ibm.597
      }
            }                                                       //ibm.597

            // everything else becomes neutrals                     //ibm.597
            for (int i = start; i < limit; ++i) {                   //ibm.597
                switch (wdirs[i]) {                                 //ibm.597
        case ES:
        case CS:
                case ET:                                            //ibm.597
                case BN:                                            //ibm.597
        case S:
                case B:                                             //ibm.597
                case WS:                                            //ibm.597
		    flags |= ON;                                                  //ibm.597
                    wdirs[i] = ON;                                  //ibm.597
          break;
                }                                                   //ibm.597
            }                                                       //ibm.597

            if ((flags & FLAG_EN) != 0) {                           //ibm.597
                boolean lastStrongIsL = false;                      //ibm.597
		if (start == 0) {                                                 //ibm.597
		    lastStrongIsL = (level & 0x1) == 0;                           //ibm.597
		} else {                                                          //ibm.597
		    lastStrongIsL = (Math.max(level, levels[start-1]) & 0x1) == 0;    //ibm.597
		}                                                                 //ibm.597

                for (int i = start; i < limit; ++i) {               //ibm.597
                    switch (wdirs[i]) {                             //ibm.597
              case L:
                        lastStrongIsL = true;                       //ibm.597
                        break;                                      //ibm.597
              case R:
                        lastStrongIsL = false;                      //ibm.597
                        break;                                      //ibm.597
              case EN:
                        if (lastStrongIsL) {                        //ibm.597
			    flags |= L;                                                  //ibm.597
                            wdirs[i] = L;                           //ibm.597
              }
            }
          }
        }

            start = limit;                                          //ibm.597
      }
    }

  /*
     * This operation should never span a level boundary.  When we have external    //ibm.597
     * levels, we have to be careful.                               //ibm.597
   *
     * The start and end of the level are treated as though there was the appropriate   //ibm.597
     * embedding character there, this means a character of the current levels  //ibm.597
     * direction if the current level is higher, and a character of the outside levels  //ibm.597
     * direction if the current level is lower.  sot and eot are treated like the   //ibm.597
     * base line direction.                                         //ibm.597
   */
    private static void resolveNeutralTypes(byte[] wdirs, byte[] levels, long flags) {  //ibm.597

        int limit = wdirs.length;                                   //ibm.597

        boolean levelaware = (flags & FLAG_EXTERNAL_EMBS) != 0;     //ibm.597

        byte baselevel = (byte)(((flags & FLAG_RTL) == 0) ? 0 : 1); //ibm.597
        byte basedir = ((baselevel & 0x1) == 0) ? L : R;            //ibm.597

        byte prevlevel = baselevel;                                 //ibm.597
        byte level = baselevel;                                     //ibm.597
        byte leveldir = basedir;                                    //ibm.597

        byte prevdir = basedir;                                     //ibm.597

        int i = 0;                                                  //ibm.597
        while (i < limit) {                                         //ibm.597
            level = levels[i];                                      //ibm.597
            if (level != prevlevel) {                               //ibm.597
                byte newleveldir = ((level & 0x1) == 0) ? L : R;    //ibm.597
                if (levelaware) {                                   //ibm.597
                    if (level > prevlevel) {                        //ibm.597
                        prevdir = newleveldir;                      //ibm.597
      } else {
                        prevdir = leveldir;                         //ibm.597
        }
      }
		leveldir = newleveldir;                                           //ibm.597
            }                                                       //ibm.597

            switch (wdirs[i]) {                                     //ibm.597
            case ON:                                                //ibm.597
                if (prevdir == leveldir) {                          //ibm.597
                    wdirs[i++] = leveldir;                          //ibm.597
                } else {                                            //ibm.597
                    byte nextdir = leveldir;                        //ibm.597
                    int j = i;                                      //ibm.597
                    if (levelaware) {                               //ibm.597
                        while (++j < limit && wdirs[j] == ON && levels[j] == level) {   //ibm.597
                        }                                           //ibm.597
                        if (j < limit) {                            //ibm.597
                            byte nextlevel = levels[j];             //ibm.597
                            if (nextlevel != level) {               //ibm.597
                                if (level < nextlevel) {            //ibm.597
                                    nextdir = ((nextlevel & 0x1) == 0) ? L : R; //ibm.597
                                }                                   //ibm.597
                            } else {                                //ibm.597
                                nextdir = (wdirs[j] == L) ? L : R;  //ibm.597
                            }                                       //ibm.597
                        }                                           //ibm.597
                    } else {                                        //ibm.597
                        while (++j < limit && wdirs[j] == ON) {     //ibm.597
                        }                                           //ibm.597
                        if (j < limit) {                            //ibm.597
                            nextdir = (wdirs[j] == L) ? L : R;      //ibm.597
                        }                                           //ibm.597
                    }                                               //ibm.597

                    while (i < j) {                                 //ibm.597
                        wdirs[i++] = nextdir;                       //ibm.597
                    }                                               //ibm.597
                }                                                   //ibm.597
      break;

            case L:                                                 //ibm.597
                prevdir = L;                                        //ibm.597
                ++i;                                                //ibm.597
                break;                                              //ibm.597

            default:                                                //ibm.597
                prevdir = R;                                        //ibm.597
                ++i;                                                //ibm.597
                break;                                              //ibm.597
        }

            prevlevel = level;                                      //ibm.597
        }                                                           //ibm.597
    }                                                               //ibm.597

    /*                                                              //ibm.597
     * Process implicit levels.                                     //ibm.597
     */                                                             //ibm.597
    private static void resolveImplicitLevels(byte[] wdirs, byte[] levels, long flags) {    //ibm.597

	for (int i = 0; i < wdirs.length; ++i) {                           //ibm.597
	    boolean odd = (levels[i] & 0x1) != 0;                          //ibm.597
	    switch (wdirs[i]) {                                            //ibm.597
	    case L:                                                        //ibm.597
		if (odd) {                                                        //ibm.597
		    levels[i] += 1;                                               //ibm.597
		}                                                                 //ibm.597
        break;

	    case R:                                                        //ibm.597
		if (!odd) {                                                       //ibm.597
		    levels[i] += 1;                                               //ibm.597
      }
		break;                                                            //ibm.597

	    case AN:                                                       //ibm.597
	    case EN:                                                       //ibm.597
		levels[i] += odd ? 1 : 2;                                         //ibm.597
		break;                                                            //ibm.597
      }
	}                                                                  //ibm.597
    }                                                               //ibm.597

    /**
     * Take B, S, and leading white space (according to original data) and set to
     * base level, so we won't have to do this each time we get a line bidi.
     */                                                             //ibm.597
    private static void resolveLineLevels(byte[] dirs, byte[] levels, long flags) { //ibm.597
        byte baselevel = (byte)(((flags & FLAG_RTL) == 0) ? 0 : 1); //ibm.597
        boolean resolveWS = true;                                   //ibm.597
        for (int i = dirs.length - 1; i >= 0; --i) {                //ibm.597
            switch (dirs[i]) {                                      //ibm.597
            case WS:                                                //ibm.597
                if (resolveWS) {                                    //ibm.597
                    levels[i] = baselevel;                          //ibm.597
    }
                break;                                              //ibm.597

	    case BN:                                                       //ibm.597
	    case LRE:                                                      //ibm.597
	    case RLE:                                                      //ibm.597
	    case LRO:                                                      //ibm.597
	    case RLO:                                                      //ibm.597
	    case PDF:                                                      //ibm.597
                if (resolveWS) {                                    //ibm.597
                    levels[i] = baselevel;                          //ibm.597
                } else {                                            //ibm.597
		    levels[i] = levels[i+1];                                      //ibm.597
    }
                break;                                              //ibm.597

            case S:                                                 //ibm.597
            case B:                                                 //ibm.597
                levels[i] = baselevel;                              //ibm.597
                resolveWS = true;                                   //ibm.597
                break;                                              //ibm.597

            default:                                                //ibm.597
                resolveWS = false;                                  //ibm.597
                break;                                              //ibm.597
  }
        }                                                           //ibm.597
    }                                                               //ibm.597


  Bidi(Bidi paragraphBidi, int start, int limit) {
    byte[] indirs = paragraphBidi.dirs;
    byte[] newLevels = createLineLevels(indirs, paragraphBidi.levels, paragraphBidi.ltr, start, limit);

    this.ltr = paragraphBidi.ltr;
    this.dirs = null;
    this.levels = newLevels;
  }

  /**
   * Return a level array representing the levels between lineStart and lineLimit using
   * the direction information to identify trailing whitespace that might need to switch
   * levels.
   */
  static byte[] createLineLevels(byte[] dirs, byte[] levels, boolean ltr, int lineStart, int lineLimit) {
    byte[] lineLevels = new byte[lineLimit - lineStart];
    System.arraycopy(levels, lineStart, lineLevels, 0, lineLevels.length);

    byte x = (byte)(ltr ? 0 : 1);
    for (int i = lineLimit - lineStart - 1; i >= 0; --i) {
      if (lineLevels[i] == x || dirs[lineStart + i] != WS) {
        break;
      }
      lineLevels[i] = x;
    }

    return lineLevels;
  }

  /**
   * Return true if the default bidi rules indicate a run direction of LTR for the
   * provided range of the char array.
   */
    static boolean defaultIsLTR(char[] chars, int start, int limit, boolean defaultLTR) {   //ibm.597
    while (start < limit) {
      char c = chars[start++];
      byte dir = getDirectionCode(c);
      switch (dir) {
      case L:
          return true;
            case AL:                                                //ibm.597
      case R:
          return false;

      default:
          break;
      }
    }

        return defaultLTR;                                          //ibm.597
    }                                                               //ibm.597


    /**
     * Return true if the default bidi rules indicate a run direction of LTR for the
     * provided range of the direction code array.
     */                                                             //ibm.597
    static boolean defaultIsLTR(byte[] dirs, int start, int limit, boolean defaultLTR) {    //ibm.597
        while (start < limit) {                                     //ibm.597
            byte dir = dirs[start++];                               //ibm.597
            switch (dir) {                                          //ibm.597
            case L:                                                 //ibm.597
    return true;
            case AL:                                                //ibm.597
            case R:                                                 //ibm.597
                return false;                                       //ibm.597

            default:                                                //ibm.597
                break;                                              //ibm.597
  }
        }                                                           //ibm.597

        return defaultLTR;                                          //ibm.597
    }                                                               //ibm.597


  /**
   * Return true if the character suggests a need for bidi processing.
   * Characters in the arabic extended blocks return false by default.
   * Other rtl characters and rtl explicit codes return true.
   */
  static boolean requiresBidi(char c) {
    if (c < '\u0591') return false;
    if (c > '\u202e') return false; // if contains arabic extended data, presume already ordered
    byte dc = getDirectionCode(c);
        return (MASK_R & (1 << dc)) != 0;                           //ibm.597
  }

  /**
     * Return the bidi direction code of the provided character.
   */
  static byte getDirectionCode(char c) {
    return dirValues[(dirIndices[c >> 7] << 7) + (c & 0x7f)];
  }

  /**
     * Return an array of bidi direction codes for the provided characters.
   */
    static byte[] getDirectionCodeArray(char[] chars) {             //ibm.597
    byte[] dirs = new byte[chars.length];

        for (int i = 0; i < chars.length; ++i) {                    //ibm.597
            dirs[i] = getDirectionCode(chars[i]);                   //ibm.597
      }

    return dirs;
  }

  /**
   * Check embedding array to see if it conforms, return flags.  On return embs is
   * a level array (override bits have been cleared), and wdirs has incorporated
   * effects of override processing.
   */
    private static long processEmbeddingArray(byte[] wdirs, byte[] embs, long flags) {  //ibm.597
        if (embs.length != wdirs.length) {                          //ibm.597
            throw new IllegalArgumentException("embedding length != direction types length");   //ibm.597
        }                                                           //ibm.597
        byte minlevel = (byte)(((flags & FLAG_RTL) == 0) ? 0 : 1);  //ibm.597
        for (int i = 0; i < embs.length; i++) {                     //ibm.597
            if ((embs[i] & 0x80) != 0) {                            //ibm.597
                embs[i] &= 0x7f;                                    //ibm.597
                if ((embs[i] & 0x1) == 0) {                         //ibm.597
                    wdirs[i] = L;                                   //ibm.597
                    flags |= FLAG_LRO;                              //ibm.597
        } else {
                    wdirs[i] = R;                                   //ibm.597
                    flags |= FLAG_RLO;                              //ibm.597
                }                                                   //ibm.597
            } else if (embs[i] > minlevel) {                        //ibm.597
                if ((embs[i] & 0x1) == 0) {                         //ibm.597
                    flags |= FLAG_LRE;                              //ibm.597
          } else {
                    flags |= FLAG_RLE;                              //ibm.597
                }                                                   //ibm.597
            }                                                       //ibm.597
            if (embs[i] < minlevel || embs[i] >= NUMLEVELS) {       //ibm.597
                throw new IllegalArgumentException("bad embedding level (" + embs[i] + ") at index " + i);  //ibm.597
            }                                                       //ibm.597
        }                                                           //ibm.597

        return flags;                                               //ibm.597
            }

    /**
     * Process bidi codes in wdirs, and set levels array accordingly.  If there are
     * override codes, wdirs may change to reflect overrides.  Return new flags.
     */                                                             //ibm.597
    static long getLevelArray(byte[] wdirs, byte[] levels, long flags) {    //ibm.597
        // optimization                                             //ibm.597
        byte paragraphEmbeddingLevel = (byte)(((flags & FLAG_RTL) == 0) ? 0 : 1);   //ibm.597

        if ((flags & MASK_EMBS) == 0) {                             //ibm.597
            for (int i = 0; i < levels.length; ++i) {               //ibm.597
                levels[i] = paragraphEmbeddingLevel;                //ibm.597
          }

            return flags;                                           //ibm.597
        }

        int textLength = wdirs.length;                              //ibm.597

        byte[] embeddingValueStack = new byte[NUMLEVELS];           //ibm.597
        int stackCounter = 0;                                       //ibm.597
        int overflowCounter = 0;                                    //ibm.597

	byte currentEmbeddingValue = paragraphEmbeddingLevel; // low 7 bits plus high bit for override status  //ibm.597
        byte currentEmbeddingLevel = paragraphEmbeddingLevel;       //ibm.597

	byte actualEmbeddingValue = currentEmbeddingValue; // one last applied to a non-format character   //ibm.597
	byte actualEmbeddingLevel = currentEmbeddingLevel;                 //ibm.597

    loop:                                                           //ibm.597
        for (int i = 0; i < textLength; ++i) {                      //ibm.597
            byte t = wdirs[i];                                      //ibm.597

            switch (t) {                                            //ibm.597
      case RLE:
            case LRE:                                               //ibm.597
            case RLO:                                               //ibm.597
            case LRO:                                               //ibm.597
		levels[i] = actualEmbeddingLevel;                                 //ibm.597
		wdirs[i] = BN;                                                    //ibm.597

                if (overflowCounter == 0) {                         //ibm.597
                    byte newLevel;                                  //ibm.597
                    if (t == RLE || t == RLO) {                     //ibm.597
                        newLevel = (byte)((currentEmbeddingLevel + 1) | 1); // least greater odd    //ibm.597
                    } else { // t == LRE || t == LRO                //ibm.597
                        newLevel = (byte)((currentEmbeddingLevel + 2) & ~1); // least greater even  //ibm.597
            }

                    if (newLevel < NUMLEVELS) {                     //ibm.597
                        embeddingValueStack[stackCounter] = currentEmbeddingValue;  //ibm.597
                        stackCounter++;                             //ibm.597

                        currentEmbeddingLevel = newLevel;           //ibm.597

                        switch (t) {                                //ibm.597
                        case LRO:                                   //ibm.597
                        case RLO:                                   //ibm.597
			    currentEmbeddingValue = (byte)(newLevel | 0x80); // override flag    //ibm.597
                            break;                                  //ibm.597
			default:                                                         //ibm.597
			    currentEmbeddingValue = newLevel;                            //ibm.597
                            break;                                  //ibm.597
          }

                        continue loop;                              //ibm.597
        }
                    // Otherwise new level is bad                   //ibm.597
                }                                                   //ibm.597
                // Otherwise old or new level is bad                //ibm.597

                overflowCounter++;                                  //ibm.597
                continue loop;                                      //ibm.597

      case PDF:
		levels[i] = actualEmbeddingLevel;                                 //ibm.597
                wdirs[i] = BN;                                      //ibm.597

                if (overflowCounter == 0) {                         //ibm.597
		    if (stackCounter > 0) {                                       //ibm.597
			--stackCounter;                                                  //ibm.597
			currentEmbeddingValue = embeddingValueStack[stackCounter];       //ibm.597
			currentEmbeddingLevel = (byte)(currentEmbeddingValue & 0x7f);    //ibm.597

                        continue loop;                              //ibm.597
          }
                    // Only get here if no matching code for the PDF.   //ibm.597

                    continue loop;                                  //ibm.597
        }
                // Only get here if stack overflowed.               //ibm.597

                --overflowCounter;                                  //ibm.597
                continue loop;                                      //ibm.597

            case B:                                                 //ibm.597
                stackCounter = 0;                                   //ibm.597
                overflowCounter = 0;                                //ibm.597
                currentEmbeddingValue = paragraphEmbeddingLevel;    //ibm.597
                currentEmbeddingLevel = paragraphEmbeddingLevel;    //ibm.597

		actualEmbeddingValue = currentEmbeddingValue;                     //ibm.597
		actualEmbeddingLevel = currentEmbeddingLevel;                     //ibm.597

                levels[i] = actualEmbeddingLevel;                   //ibm.597
                continue loop;                                      //ibm.597

	    case BN:                                                       //ibm.597
		// doesn't activate an embedding                                  //ibm.597
		levels[i] = actualEmbeddingLevel;                                 //ibm.597
		continue loop;                                                    //ibm.597

      default:
		if (actualEmbeddingValue != currentEmbeddingValue) {              //ibm.597
		    if (actualEmbeddingLevel != currentEmbeddingLevel) {          //ibm.597
			int maxLevel = Math.max(actualEmbeddingLevel, currentEmbeddingLevel);    //ibm.597
			byte maxType = (maxLevel & 0x1) == 0 ? L : R;                    //ibm.597
			wdirs[i-1] = maxType;                                            //ibm.597
      }
		    actualEmbeddingValue = currentEmbeddingValue;                 //ibm.597
		    actualEmbeddingLevel = (byte)(actualEmbeddingValue & 0x7f);   //ibm.597
		}                                                                 //ibm.597

                levels[i] = actualEmbeddingLevel;                   //ibm.597
                if (actualEmbeddingLevel != actualEmbeddingValue) { //ibm.597
		    if ((actualEmbeddingLevel & 0x1) == 0) { // LTR               //ibm.597
			wdirs[i] = L;                                                    //ibm.597
			flags |= FLAG_L;                                                 //ibm.597
		    } else {                                                      //ibm.597
			wdirs[i] = R;                                                    //ibm.597
			flags |= FLAG_R;                                                 //ibm.597
    }
		}                                                                 //ibm.597

                continue loop;                                      //ibm.597
  }

            // Never get here.  Java compiler verifies this.        //ibm.597
        }                                                           //ibm.597

	flags |= FLAG_BN;                                                  //ibm.597

        return flags;                                               //ibm.597
    }                                                               //ibm.597

  /**
   * Given a level array, compute a a visual to logical ordering.
   */
  static int[] createVisualToLogicalMap(byte[] levels) {
    int len = levels.length;
    int[] mapping = new int[len];

    byte lowestOddLevel = (byte)(NUMLEVELS + 1);
    byte highestLevel = 0;

    // initialize mapping and levels

    for (int i = 0; i < len; i++) {
      mapping[i] = i;

      byte level = levels[i];
      if (level > highestLevel) {
        highestLevel = level;
      }

      if ((level & 0x01) != 0 && level < lowestOddLevel) {
        lowestOddLevel = level;
      }
    }

    while (highestLevel >= lowestOddLevel) {
      int i = 0;
      for (;;) {
        while (i < len && levels[i] < highestLevel) {
          i++;
        }
        int begin = i++;

        if (begin == levels.length) {
          break; // no more runs at this level
        }

        while (i < len && levels[i] >= highestLevel) {
          i++;
        }
        int end = i - 1;

        while (begin < end) {
          int temp = mapping[begin];
          mapping[begin] = mapping[end];
          mapping[end] = temp;
          ++begin;
          --end;
        }
      }

      --highestLevel;
    }

    return mapping;
  }

  /**
   * Reorder the objects in the array into visual order based on their levels.
   */
  static void reorderVisually(byte[] levels, Object[] objects) {
    int len = levels.length;

    byte lowestOddLevel = (byte)(NUMLEVELS + 1);
    byte highestLevel = 0;

    // initialize mapping and levels

    for (int i = 0; i < len; i++) {
      byte level = levels[i];
      if (level > highestLevel) {
        highestLevel = level;
      }

      if ((level & 0x01) != 0 && level < lowestOddLevel) {
        lowestOddLevel = level;
      }
    }

    while (highestLevel >= lowestOddLevel) {
      int i = 0;
      for (;;) {
        while (i < len && levels[i] < highestLevel) {
          i++;
        }
        int begin = i++;

        if (begin == levels.length) {
          break; // no more runs at this level
        }

        while (i < len && levels[i] >= highestLevel) {
          i++;
        }
        int end = i - 1;

        while (begin < end) {
          Object temp = objects[begin];
          objects[begin] = objects[end];
          objects[end] = temp;
          ++begin;
          --end;
        }
      }

      --highestLevel;
    }
  }

  /**
   * Return the inverse array, source array must map 1-1
   *
   * i.e. if values[i] = j, then inverse[j] = i.
   */
  static int[] getInverseOrder(int[] values) {
    if (values == null) {
      return null;
    }

    int[] result = new int[values.length];
    for (int i = 0; i < values.length; i++) {
      result[values[i]] = i;
    }

    return result;
  }

  /**
   * Compute a contiguous order for the range start, limit.
   */
  private static int[] computeContiguousOrder(int[] values, int start,
                                              int limit) {

    int[] result = new int[limit-start];
    for (int i=0; i < result.length; i++) {
      result[i] = i + start;
    }

    // now we'll sort result[], with the following comparison:
    // result[i] lessthan result[j] iff values[result[i]] < values[result[j]]

    // selection sort for now;  use more elaborate sorts if desired
    for (int i=0; i < result.length-1; i++) {
      int minIndex = i;
      int currentValue = values[result[minIndex]];
      for (int j=i; j < result.length; j++) {
        if (values[result[j]] < currentValue) {
          minIndex = j;
          currentValue = values[result[minIndex]];
        }
      }
      int temp = result[i];
      result[i] = result[minIndex];
      result[minIndex] = temp;
    }

    // shift result by start:
    if (start != 0) {
      for (int i=0; i < result.length; i++) {
        result[i] -= start;
      }
    }

    // next, check for canonical order:
    int k;
    for (k=0; k < result.length; k++) {
      if (result[k] != k) {
        break;
      }
    }

    if (k == result.length) {
      return null;
    }

    // now return inverse of result:
    return getInverseOrder(result);
  }

  /**
   * Return an array containing contiguous values from 0 to length
   * having the same ordering as the source array. If this would be
   * a canonical ltr ordering, return null.  values[] is NOT
   * required to be a permutation.
   */
  static int[] getContiguousOrder(int[] values) {
    if (values != null) {
      return computeContiguousOrder(values, 0, values.length);
    }

    return null;
  }

  /**
   * Return an array containing the values from start up to limit,
   * normalized to fall within the range from 0 up to limit - start.
   * If this would be a canonical ltr ordering, return null.
   * NOTE: This method assumes that values[] is a permutation
   * generated from levels[].
   */
  static int[] getNormalizedOrder(int[] values, byte[] levels,
                                         int start, int limit) {

    if (values != null) {
      if (start != 0 || limit != values.length) {
        // levels optimization
        boolean copyRange, canonical;
        byte primaryLevel;

        if (levels == null) {
          primaryLevel = (byte) 0x0;
          copyRange = true;
          canonical = true;
        }
        else {
          if (levels[start] == levels[limit-1]) {
            primaryLevel = levels[start];
            canonical = (primaryLevel & (byte)0x1) == 0;

            // scan for levels below primary
            int i;
            for (i=start; i < limit; i++) {
              if (levels[i] < primaryLevel) {
                break;
              }
              if (canonical) {
                canonical = levels[i] == primaryLevel;
              }
            }

            copyRange = (i == limit);
          }
          else {
            copyRange = false;

            // these don't matter;  but the compiler cares:
            primaryLevel = (byte) 0x0;
            canonical = false;
          }
        }

        if (copyRange) {
          if (canonical) {
            return null;
          }

          int[] result = new int[limit-start];
          int baseValue;

          if ((primaryLevel & (byte)0x1) != 0) {
            baseValue = values[limit-1];
          } else {
            baseValue = values[start];
          }

          if (baseValue == 0) {
            System.arraycopy(values, start, result, 0, limit-start);
          }
          else {
            for (int j=0; j < result.length; j++) {
              result[j] = values[j+start] - baseValue;
            }
          }

          return result;
        }
        else {
          return computeContiguousOrder(values, start, limit);
        }
      }
      else {
        return values;
      }
    }

    return null;
  }

  // convenience method for compatibility with old tests
    static Bidi createBidi(char[] text) {
        return new Bidi(text);
    }

    // from Unicode Data 3.0                                        //ibm.3537
    // gets reset in static init
    private static byte[] dirIndices = {
        41, 0, 1, 2, 2, 3, 4, 5, 6, 2, 7, 8, 9, 10, 11, 12,         //ibm.597
        13, 14, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27,     //ibm.597
        28, 29, 30, 31, 32, 33, 34, 35, 36, 37, -125, 2, 6, 38, 39, 14,     //ibm.597
        40, 41, 42, -118, 14, 22, 2, 43, 44, 45, 46, 47, 48, 49, 50, 51,    //ibm.597
        52, 53, 54, 55, 56, 57, 58, 14, 59, 60, 56, 56, -117, 14, 11, 61,   //ibm.597
        56, 62, 63, 64, 65, 66, 67, 68, 69, 70, -77, 2, 1, 71, -2, 2,   //ibm.597
        -91, 2, 1, 72, -119, 2, 1, 73, -114, 14, -41, 2, 1, 74, -60, 2,     //ibm.597
        12, 75, 14, 76, 77, 78, 78, 79, 80, 81, 82, 83, 84,         //ibm.597
    };

    // gets reset in static init
    private static byte[] dirValues = {
        -119, 14, 5, 16, 15, 16, 17, 15, -114, 14, -125, 15, 4, 16, 17, 18,     //ibm.3537
        18, -125, 10, -123, 18, 5, 10, 12, 10, 12, 9, -118, 8, 1, 12, -122,     //ibm.597
        18, -102, 0, -122, 18, -102, 0, -124, 18, -122, 14, 1, 15, -102, 14, 2,     //ibm.3537
        12, 18, -124, 10, -124, 18, 1, 0, -123, 18, 6, 10, 10, 8, 8, 18,    //ibm.3537
        0, -125, 18, 2, 8, 0, -123, 18, -105, 0, 1, 18, -97, 0, 1, 18,  //ibm.3537
        -2, 0, -86, 0, 2, 16, 16, -110, 0, -100, 16, -34, 0, 2, 16, 16,     //ibm.3537
        -119, 0, 2, 18, 18, -121, 0, -114, 18, 2, 0, 0, -114, 18, -123, 0,  //ibm.3537
        -119, 18, 1, 0, -111, 16, -49, 13, -111, 16, -125, 13, -111, 16, 2, 18,     //ibm.3537
        18, -124, 16, 1, 0, -125, 16, 1, 18, -123, 16, 4, 18, 18, 0, 18,    //ibm.3537
        -125, 0, 3, 16, 0, 16, -108, 0, 1, 16, -84, 0, 1, 16, -120, 0,  //ibm.3537
        2, 16, 16, -102, 0, -116, 16, -125, 0, -124, 13, 5, 16, 13, 13, 16,     //ibm.3537
        16, -71, 0, 8, 16, 16, 0, 0, 16, 16, 0, 0, -125, 16, -90, 0,    //ibm.3537
        4, 16, 16, 0, 0, -73, 16, -90, 0, 2, 16, 16, -121, 0, 1, 16,    //ibm.3537
        -89, 0, 3, 16, 0, 18, -122, 16, -111, 13, 1, 16, -105, 13, 1, 16,   //ibm.3537
        -125, 13, 7, 3, 13, 3, 13, 13, 3, 13, -117, 16, -101, 3, -123, 16,  //ibm.3537
        -123, 3, -105, 16, 1, 12, -114, 16, 1, 4, -125, 16, 2, 4, 16, -102,     //ibm.3537
        4, -123, 16, -117, 4, -117, 13, -118, 16, -118, 11, 7, 10, 11, 11, 4,   //ibm.3537
        16, 16, 13, -27, 4, -113, 13, 5, 4, 4, 13, 13, 18, -124, 13, 2,     //ibm.3537
        16, 16, -118, 8, -123, 4, 1, 16, -114, 4, 4, 16, 14, 4, 13, -101,   //ibm.3537
        4, -125, 16, -101, 13, -75, 16, -90, 4, -117, 13, -2, 16, -46, 16, 4,   //ibm.3537
        13, 13, 0, 16, -75, 0, 3, 16, 16, 13, -124, 0, -120, 13, -124, 0,   //ibm.3537
        4, 13, 16, 16, 0, -124, 13, -125, 16, -118, 0, 2, 13, 13, -115, 0,  //ibm.3537
        -112, 16, 4, 13, 0, 0, 16, -120, 0, 6, 16, 16, 0, 0, 16, 16,    //ibm.3537
        -106, 0, 1, 16, -121, 0, 2, 16, 0, -125, 16, -124, 0, 4, 16, 16,    //ibm.3537
        13, 16, -125, 0, -124, 13, 9, 16, 16, 0, 0, 16, 16, 0, 0, 13,   //ibm.3537
        -119, 16, 1, 0, -124, 16, 3, 0, 0, 16, -125, 0, 4, 13, 13, 16,  //ibm.3537
        16, -116, 0, 2, 10, 10, -121, 0, -121, 16, 3, 13, 16, 16, -122, 0,  //ibm.3537
        -124, 16, 4, 0, 0, 16, 16, -106, 0, 1, 16, -121, 0, 13, 16, 0,  //ibm.3537
        0, 16, 0, 0, 16, 0, 0, 16, 16, 13, 16, -125, 0, 2, 13, 13,  //ibm.3537
        -124, 16, 4, 13, 13, 16, 16, -125, 13, -117, 16, -124, 0, 2, 16, 0,     //ibm.3537
        -121, 16, -118, 0, 2, 13, 13, -125, 0, -116, 16, 4, 13, 13, 0, 16,  //ibm.3537
        -121, 0, 3, 16, 0, 16, -125, 0, 1, 16, -106, 0, 1, 16, -121, 0,     //ibm.3537
        4, 16, 0, 0, 16, -123, 0, 3, 16, 16, 13, -124, 0, -123, 13, 11,     //ibm.3537
        16, 13, 13, 0, 16, 0, 0, 13, 16, 16, 0, -113, 16, 1, 0, -123,   //ibm.3537
        16, -118, 0, -111, 16, 4, 13, 0, 0, 16, -120, 0, 6, 16, 16, 0,  //ibm.3537
        0, 16, 16, -106, 0, 1, 16, -121, 0, 5, 16, 0, 0, 16, 16, -124,  //ibm.3537
        0, 7, 16, 16, 13, 0, 0, 13, 0, -125, 13, -125, 16, 7, 0, 0,     //ibm.3537
        16, 16, 0, 0, 13, -120, 16, 2, 13, 0, -124, 16, 3, 0, 0, 16,    //ibm.3537
        -125, 0, -124, 16, -117, 0, -111, 16, 3, 13, 0, 16, -122, 0, -125, 16,  //ibm.3537
        -125, 0, 1, 16, -124, 0, -125, 16, 7, 0, 0, 16, 0, 16, 0, 0,    //ibm.3537
        -125, 16, 2, 0, 0, -125, 16, -125, 0, -125, 16, -120, 0, 1, 16, -125,   //ibm.3537
        0, -124, 16, 5, 0, 0, 13, 0, 0, -125, 16, -125, 0, 1, 16, -125,     //ibm.3537
        0, 1, 13, -119, 16, 1, 0, -113, 16, -116, 0, -114, 16, -125, 0, 1,  //ibm.3537
        16, -120, 0, 1, 16, -125, 0, 1, 16, -105, 0, 1, 16, -118, 0, 1,     //ibm.597
        16, -123, 0, -124, 16, -125, 13, -124, 0, 1, 16, -125, 13, 1, 16, -124,     //ibm.3537
        13, -121, 16, 2, 13, 13, -119, 16, 2, 0, 0, -124, 16, -118, 0, -110,    //ibm.3537
        16, 3, 0, 0, 16, -120, 0, 1, 16, -125, 0, 1, 16, -105, 0, 1,    //ibm.3537
        16, -118, 0, 1, 16, -123, 0, -124, 16, 2, 0, 13, -123, 0, 9, 16,    //ibm.3537
        13, 0, 0, 16, 0, 0, 13, 13, -121, 16, 2, 0, 0, -121, 16, 4,     //ibm.3537
        0, 16, 0, 0, -124, 16, -118, 0, -110, 16, 3, 0, 0, 16, -120, 0,     //ibm.3537
        1, 16, -125, 0, 1, 16, -105, 0, 1, 16, -112, 0, -124, 16, -125, 0,  //ibm.3537
        -125, 13, 2, 16, 16, -125, 0, 1, 16, -125, 0, 1, 13, -119, 16, 1,   //ibm.3537
        0, -120, 16, 2, 0, 0, -124, 16, -118, 0, -110, 16, 3, 0, 0, 16,     //ibm.3537
        -110, 0, -125, 16, -104, 0, 1, 16, -119, 0, 4, 16, 0, 16, 16, -121,     //ibm.3537
        0, -125, 16, 1, 13, -124, 16, -125, 0, -125, 13, 3, 16, 13, 16, -120,   //ibm.3537
        0, -110, 16, -125, 0, -116, 16, -80, 0, 3, 13, 0, 0, -121, 13, -124,    //ibm.3537
        16, 1, 10, -121, 0, -120, 13, -115, 0, -91, 16, 13, 0, 0, 16, 0,    //ibm.3537
        16, 16, 0, 0, 16, 0, 16, 16, 0, -122, 16, -124, 0, 1, 16, -121,     //ibm.3537
        0, 1, 16, -125, 0, 9, 16, 0, 16, 0, 16, 16, 0, 0, 16, -124,     //ibm.3537
        0, 3, 13, 0, 0, -122, 13, 6, 16, 13, 13, 0, 16, 16, -123, 0,    //ibm.3537
        3, 16, 0, 16, -122, 13, 2, 16, 16, -118, 0, 4, 16, 16, 0, 0,    //ibm.3537
        -94, 16, -104, 0, 2, 13, 13, -101, 0, 5, 13, 0, 13, 0, 13, -124,    //ibm.3537
        18, -118, 0, 1, 16, -94, 0, -122, 16, -114, 13, 1, 0, -123, 13, 3,  //ibm.3537
        0, 13, 13, -124, 0, -124, 16, -120, 13, 1, 16, -92, 13, 1, 16, -120,    //ibm.3537
        0, 1, 13, -122, 0, 3, 16, 16, 0, -80, 16, -94, 0, 1, 16, -123,  //ibm.3537
        0, 5, 16, 0, 0, 16, 0, -124, 13, 2, 0, 13, -125, 16, 4, 13,     //ibm.3537
        13, 0, 13, -122, 16, -104, 0, 2, 13, 13, -58, 16, -90, 0, -118, 16,     //ibm.3537
        -89, 0, -124, 16, 1, 0, -124, 16, -38, 0, -123, 16, -60, 0, -123, 16,   //ibm.3537
        -46, 0, -122, 16, -121, 0, 1, 16, -65, 0, 3, 16, 0, 16, -124, 0,    //ibm.3537
        2, 16, 16, -121, 0, 3, 16, 0, 16, -124, 0, 2, 16, 16, -89, 0,   //ibm.3537
        3, 16, 0, 16, -124, 0, 2, 16, 16, -97, 0, 3, 16, 0, 16, -124,   //ibm.3537
        0, 2, 16, 16, -121, 0, 3, 16, 0, 16, -124, 0, 2, 16, 16, -121,  //ibm.3537
        0, 1, 16, -121, 0, 1, 16, -105, 0, 1, 16, -97, 0, 3, 16, 0,     //ibm.3537
        16, -124, 0, 2, 16, 16, -121, 0, 1, 16, -89, 0, 1, 16, -109, 0,     //ibm.3537
        -122, 16, -100, 0, -93, 16, -43, 0, -116, 16, -2, 0, -8, 0, -119, 16,   //ibm.3537
        1, 17, -102, 0, 2, 18, 18, -125, 16, -47, 0, -113, 16, -73, 0, -121,    //ibm.3537
        13, -120, 0, 3, 13, 0, 0, -117, 13, -121, 0, 2, 10, 0, -125, 16,    //ibm.3537
        -118, 0, -106, 16, -117, 18, -124, 14, 1, 16, -118, 0, -122, 16, -40, 0,    //ibm.3537
        -120, 16, -87, 0, 1, 13, -42, 16, -100, 0, -124, 16, -38, 0, -122, 16,  //ibm.3537
        -106, 0, 2, 16, 16, -122, 0, 2, 16, 16, -90, 0, 2, 16, 16, -122,    //ibm.3537
        0, 2, 16, 16, -120, 0, 7, 16, 0, 16, 0, 16, 0, 16, -97, 0,  //ibm.3537
        2, 16, 16, -75, 0, 1, 16, -121, 0, 2, 18, 0, -125, 18, -125, 0,     //ibm.3537
        1, 16, -121, 0, -125, 18, -124, 0, 2, 16, 16, -122, 0, 1, 16, -125,     //ibm.3537
        18, -115, 0, -125, 18, 2, 16, 16, -125, 0, 1, 16, -121, 0, 3, 18,   //ibm.3537
        18, 16, -117, 17, -125, 14, 2, 0, 3, -104, 18, 8, 17, 15, 1, 5,     //ibm.3537
        7, 2, 6, 17, -123, 10, -110, 18, 1, 16, -122, 18, -100, 16, -122, 14,   //ibm.3537
        1, 8, -125, 16, -122, 8, 2, 10, 10, -125, 18, 1, 0, -118, 8, 2,     //ibm.3537
        10, 10, -125, 18, -111, 16, -112, 10, -96, 16, -108, 13, -100, 16, 3, 18,   //ibm.3537
        18, 0, -124, 18, 3, 0, 18, 18, -118, 0, 2, 18, 0, -125, 18, -123,   //ibm.3537
        0, -122, 18, 6, 0, 18, 0, 18, 0, 18, -124, 0, 1, 10, -125, 0,   //ibm.3537
        1, 18, -121, 0, 1, 18, -104, 16, -115, 18, -92, 0, -116, 16, -28, 18,   //ibm.3537
        -116, 16, -110, 18, 2, 10, 10, -2, 18, -32, 18, -114, 16, -74, 18, -59,     //ibm.3537
        0, 2, 18, 16, -104, 18, 1, 0, -123, 18, -27, 16, -89, 18, -103, 16,     //ibm.3537
        -117, 18, -107, 16, -68, 8, -50, 0, 1, 8, -107, 16, -2, 18, -104, 18,   //ibm.3537
        -118, 16, -40, 18, -120, 16, -108, 18, -123, 16, -39, 18, -113, 16, -124, 18,   //ibm.3537
        1, 16, -124, 18, 2, 16, 16, -100, 18, 1, 16, -93, 18, 3, 16, 18,    //ibm.3537
        16, -124, 18, -125, 16, 2, 18, 16, -121, 18, 2, 16, 16, -121, 18, -114,     //ibm.3537
        16, -97, 18, -125, 16, -104, 18, 1, 16, -114, 18, -63, 16, -102, 18, 1,     //ibm.3537
        16, -39, 18, -116, 16, -42, 18, -102, 16, -116, 18, -124, 16, 1, 17, -124,  //ibm.3537
        18, -125, 0, -103, 18, -119, 0, -122, 13, 1, 18, -123, 0, 2, 18, 18,    //ibm.3537
        -125, 0, -125, 16, 3, 18, 18, 16, -44, 0, -124, 16, 8, 13, 13, 18,  //ibm.3537
        18, 0, 0, 16, 16, -38, 0, 1, 18, -125, 0, -122, 16, -88, 0, -124,   //ibm.3537
        16, -34, 0, 1, 16, -88, 0, -56, 16, -99, 0, -125, 16, -92, 0, -100,     //ibm.3537
        16, -100, 0, -125, 16, -78, 0, -113, 16, -116, 0, -124, 16, -81, 0, 1,  //ibm.3537
        16, -9, 0, -124, 16, -29, 0, 2, 16, 16, -97, 0, 1, 16, -74, 0,  //ibm.3537
        -54, 16, -90, 0, -38, 16, -115, 0, -125, 16, -110, 18, 2, 16, 16, -112,     //ibm.3537
        18, 1, 16, -116, 18, 1, 16, -125, 18, 2, 16, 18, -71, 16, -92, 0,   //ibm.3537
        -36, 16, -82, 0, -46, 16, -121, 0, -116, 16, -123, 0, -123, 16, 2, 3,   //ibm.3537
        13, -118, 3, 1, 10, -115, 3, 1, 16, -123, 3, 9, 16, 3, 16, 3,   //ibm.3537
        3, 16, 3, 3, 16, -118, 3, -30, 4, -95, 16, -2, 4, -19, 4, 2,    //ibm.3537
        18, 18, -112, 16, -64, 4, 2, 16, 16, -74, 4, -88, 16, -116, 4, -92,     //ibm.3537
        16, -124, 13, -116, 16, -107, 18, -124, 16, -121, 18, 6, 12, 18, 12, 16,    //ibm.3537
        18, 12, -119, 18, 5, 10, 18, 18, 10, 10, -125, 18, 5, 16, 18, 10,   //ibm.3537
        10, 18, -124, 16, -125, 4, 3, 16, 4, 16, -2, 4, -119, 4, 6, 16,     //ibm.3537
        16, 14, 16, 18, 18, -125, 10, -123, 18, 5, 10, 12, 10, 12, 9, -118,     //ibm.3537
        8, 1, 12, -122, 18, -102, 0, -122, 18, -102, 0, -124, 18, 2, 16, 16,    //ibm.3537
        -123, 18, -39, 0, -125, 16, -122, 0, 2, 16, 16, -122, 0, 2, 16, 16,     //ibm.3537
        -122, 0, 2, 16, 16, -125, 0, -125, 16, 2, 10, 10, -125, 18, 3, 10,  //ibm.3537
        10, 16, -121, 18, -118, 16, -125, 14, 4, 18, 18, 16, 16,    //ibm.3537
    };

    static {
        dirIndices = RLEUtilities.readRLE(dirIndices);
        dirValues = RLEUtilities.readRLE(dirValues);
    }
}
