/*
 * @(#)src/classes/sov/javax/swing/text/html/BlockView.java, swing, as131, 20031014 1.9.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM Java(tm)2 SDK, Standard Edition, v 1.3.1
 * (C) Copyright IBM Corp. 1998, 2001. All Rights Reserved
 * ===========================================================================
 */

/*
 * @(#)BlockView.java	1.24 99/12/04
 *
 * Copyright 1997-2000 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the proprietary information of Sun Microsystems, Inc.  
 * Use is subject to license terms.
 * 
 */
package javax.swing.text.html;

import java.util.Enumeration;
import java.awt.*;
import javax.swing.SizeRequirements;
import javax.swing.border.*;
import javax.swing.event.DocumentEvent;
import javax.swing.text.*;

/**
 * A view implementation to display a block (as a box)
 * with CSS specifications.
 *
 * @author  Timothy Prinzing
 * @version 1.24 12/04/99
 */
public class BlockView extends BoxView  {

    /**
     * Creates a new view that represents an
     * html box.  This can be used for a number
     * of elements.
     *
     * @param elem the element to create a view for
     * @param axis either View.X_AXIS or View.Y_AXIS
     */
    public BlockView(Element elem, int axis) {
	super(elem, axis);
    }

    /**
     * Establishes the parent view for this view.  This is
     * guaranteed to be called before any other methods if the
     * parent view is functioning properly.
     * <p> 
     * This is implemented
     * to forward to the superclass as well as call the
     * <a href="#setPropertiesFromAttributes">setPropertiesFromAttributes</a>
     * method to set the paragraph properties from the css
     * attributes.  The call is made at this time to ensure
     * the ability to resolve upward through the parents 
     * view attributes.
     *
     * @param parent the new parent, or null if the view is
     *  being removed from a parent it was previously added
     *  to
     */
    public void setParent(View parent) {
	super.setParent(parent);
	setPropertiesFromAttributes();
    }

    /**
     * Calculate the requirements of the block along the major
     * axis (i.e. the axis along with it tiles).  This is implemented
     * to provide the superclass behavior and then adjust it if the 
     * CSS width or height attribute is specified and applicable to
     * the axis.
     */
    protected SizeRequirements calculateMajorAxisRequirements(int axis, SizeRequirements r) {
	if (r == null) {
	    r = new SizeRequirements();
	}
	if (! spanSetFromAttributes(axis, r)) {
	    r = super.calculateMajorAxisRequirements(axis, r);
	}
	return r;
    }

    /**
     * Calculate the requirements of the block along the minor
     * axis (i.e. the axis orthoginal to the axis along with it tiles).  
     * This is implemented
     * to provide the superclass behavior and then adjust it if the 
     * CSS width or height attribute is specified and applicable to
     * the axis.
     */
    protected SizeRequirements calculateMinorAxisRequirements(int axis, SizeRequirements r) {
	if (r == null) {
	    r = new SizeRequirements();
	}

	if (! spanSetFromAttributes(axis, r)) {

	    /*
	     * The requirements were not directly specified by attributes, so
	     * compute the aggregate of the requirements of the children.  The
	     * children that have a percentage value specified will be treated 
	     * as completely stretchable since that child is not limited in any
	     * way.
	     */ 
/*
	    int min = 0;
	    long pref = 0;
	    int max = 0;
	    int n = getViewCount();
	    for (int i = 0; i < n; i++) {
		View v = getView(i);
		min = Math.max((int) v.getMinimumSpan(axis), min);
		pref = Math.max((int) v.getPreferredSpan(axis), pref);
		if (
		max = Math.max((int) v.getMaximumSpan(axis), max);

	    }
	    r.preferred = (int) pref;
	    r.minimum = min;
	    r.maximum = max;
	    */
	    r = super.calculateMinorAxisRequirements(axis, r);
	}

	/*
	 * Set the alignment based upon the CSS properties if it is
	 * specified.  For X_AXIS this would be text-align, for 
	 * Y_AXIS this would be vertical-align.
	 */
	if (axis == X_AXIS) {
	    Object o = getAttributes().getAttribute(CSS.Attribute.TEXT_ALIGN);
	    if (o != null) {
		String align = o.toString();
		if (align.equals("center")) {
		    r.alignment = 0.5f;
		} else if (align.equals("right")) {
		    r.alignment = 1.0f;
		} else {
		    r.alignment = 0.0f;
		}
	    }
	}
	// Y_AXIS TBD

	return r;
    }

    boolean isPercentage(int axis, AttributeSet a) {
	if (axis == X_AXIS) {
	    if (cssWidth != null) {
		return cssWidth.isPercentage();
	    }
	} else {
	    if (cssHeight != null) {
		return cssHeight.isPercentage();
	    }
	}
	return false;
    }
	    
    /**
     * Adjust the given requirements to the CSS width or height if
     * it is specified along the applicable axis.  Return true if the
     * size is exactly specified, false if the span is not specified 
     * in an attribute or the size specified is a percentage.
     */
    boolean spanSetFromAttributes(int axis, SizeRequirements r) {
	if (axis == X_AXIS) {
	    if ((cssWidth != null) && (! cssWidth.isPercentage())) {
		r.minimum = r.preferred = r.maximum = (int) cssWidth.getValue();
		return true;
	    }
	} else {
	    if ((cssHeight != null) && (! cssHeight.isPercentage())) {
		r.minimum = r.preferred = r.maximum = (int) cssHeight.getValue();
		return true;
	    }
	}
	return false;
    }

    /**
     * Perform layout for the minor axis of the box (i.e. the
     * axis orthoginal to the axis that it represents).  The results 
     * of the layout should be placed in the given arrays which represent 
     * the allocations to the children along the minor axis.
     *
     * @param targetSpan the total span given to the view, which
     *  whould be used to layout the children.
     * @param axis the axis being layed out.
     * @param offsets the offsets from the origin of the view for
     *  each of the child views.  This is a return value and is
     *  filled in by the implementation of this method.
     * @param spans the span of each child view.  This is a return
     *  value and is filled in by the implementation of this method.
     * @returns the offset and span for each child view in the
     *  offsets and spans parameters.
     */
    protected void layoutMinorAxis(int targetSpan, int axis, int[] offsets, int[] spans) {
	int n = getViewCount();
	Object key = (axis == X_AXIS) ? CSS.Attribute.WIDTH : CSS.Attribute.HEIGHT;
	for (int i = 0; i < n; i++) {
	    View v = getView(i);
	    int min = (int) v.getMinimumSpan(axis);
	    int max = (int) v.getMaximumSpan(axis);

	    // check for percentage span
	    AttributeSet a = v.getAttributes();
	    CSS.LengthValue lv = (CSS.LengthValue) a.getAttribute(key);
	    if ((lv != null) && lv.isPercentage()) {
		// bound the span to the percentage specified
		min = (int) lv.getValue(targetSpan);
		max = min;
	    }

	    // assign the offset and span for the child
	    if (max < targetSpan) {
		// can't make the child this wide, align it
		float align = v.getAlignment(axis);
		offsets[i] = (int) ((targetSpan - max) * align);
		spans[i] = max;
	    } else {
		// make it the target width, or as small as it can get.
		offsets[i] = 0;
		spans[i] = Math.max(min, targetSpan);
	    }
	}
    }


    /**
     * Renders using the given rendering surface and area on that
     * surface.  This is implemented to delegate to the css box
     * painter to paint the border and background prior to the
     * interior.
     *
     * @param g the rendering surface to use
     * @param allocation the allocated region to render into
     * @see View#paint
     */
    public void paint(Graphics g, Shape allocation) {
	Rectangle a = (Rectangle) allocation;
	painter.paint(g, a.x, a.y, a.width, a.height, this);
	super.paint(g, a);
    }

    /**
     * Fetches the attributes to use when rendering.  This is
     * implemented to multiplex the attributes specified in the
     * model with a StyleSheet.
     */
    public AttributeSet getAttributes() {
	if (attr == null) {
	    StyleSheet sheet = getStyleSheet();
	    attr = sheet.getViewAttributes(this);
	}
	return attr;
    }

    /**
     * Gets the resize weight.
     *
     * @param axis may be either X_AXIS or Y_AXIS
     * @return the weight
     * @exception IllegalArgumentException for an invalid axis
     */
    public int getResizeWeight(int axis) {
	switch (axis) {
	case View.X_AXIS:
	    return 1;
	case View.Y_AXIS:
	    return 0;
	default:
	    throw new IllegalArgumentException("Invalid axis: " + axis);
	}
    }

    /**
     * Gets the alignment.
     *
     * @param axis may be either X_AXIS or Y_AXIS
     * @return the alignment
     */
    public float getAlignment(int axis) {
	switch (axis) {
	case View.X_AXIS:
	    return 0;
	case View.Y_AXIS:
	    if (getViewCount() == 0) {
		return 0;
	    }
	    float span = getPreferredSpan(View.Y_AXIS);
	    View v = getView(0);
	    float above = v.getPreferredSpan(View.Y_AXIS);
	    float a = (((int)span) != 0) ? (above * v.getAlignment(View.Y_AXIS)) / span: 0;
	    return a;
	default:
	    throw new IllegalArgumentException("Invalid axis: " + axis);
	}
    }

    public void changedUpdate(DocumentEvent changes, Shape a, ViewFactory f) {
	super.changedUpdate(changes, a, f);
	int pos = changes.getOffset();
	if (pos <= getStartOffset() && (pos + changes.getLength()) >=
	    getEndOffset()) {
	    setPropertiesFromAttributes();
	}
    }

    /**
     * Update any cached values that come from attributes.
     */
    protected void setPropertiesFromAttributes() {

	// update attributes
	StyleSheet sheet = getStyleSheet();
	attr = sheet.getViewAttributes(this);

	// Reset the painter
	painter = sheet.getBoxPainter(attr);
	if (attr != null) {
	    setInsets((short) painter.getInset(TOP, this),
		      (short) painter.getInset(LEFT, this),
		      (short) painter.getInset(BOTTOM, this),
		      (short) painter.getInset(RIGHT, this));
	}

	// Get the width/height
	cssWidth = (CSS.LengthValue) attr.getAttribute(CSS.Attribute.WIDTH);
	cssHeight = (CSS.LengthValue) attr.getAttribute(CSS.Attribute.HEIGHT);
    }

    protected StyleSheet getStyleSheet() {
	HTMLDocument doc = (HTMLDocument) getDocument();
	return doc.getStyleSheet();
    }


    private AttributeSet attr;
    private StyleSheet.BoxPainter painter;

    private CSS.LengthValue cssWidth;
    private CSS.LengthValue cssHeight;

}
