/*
 * IBM Confidential OCO Source Material
 * 5639-D57 (C) COPYRIGHT International Business Machines Corp. 2002.
 * The source code for this program is not published or otherwise divested
 * of its trade secrets, irrespective of what has been deposited with the
 * U.S. Copyright Office.
 */

import java.io.IOException;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;
import org.xml.sax.XMLReader;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

/**
 * This class is a parser wrapper and an extension of DefaultHandler to serve as an SAX parser
 * with error handling.
 */
public class XMLSAXParser extends DefaultHandler {
    // Standard XML Parser features.
    public static final String NAMESPACES_FEATURE_ID = "http://xml.org/sax/features/namespaces";
    public static final String NAMESPACE_PREFIXES_FEATURE_ID = "http://xml.org/sax/features/namespace-prefixes";
    public static final String VALIDATION_FEATURE_ID = "http://xml.org/sax/features/validation";

    // Standard XML Parser property.
    public static final String SCHEMA_LOCATION_PROPERTY_KEY = "http://apache.org/xml/properties/schema/external-schemaLocation";

    // Indicate whether there is any parsing error?
    protected static boolean parsingSuccess = true;

    // Message types for printing parsing error messages.
    protected static final String MSG_WARNING = "Warning";
    protected static final String MSG_ERROR = "Error";
    protected static final String MSG_FATAL_ERROR = "Fatal Error";

    protected XMLReader parser;

    /**
     * Default constructor
     */
    public XMLSAXParser() {
		this(false);
    } // Constructor

	public XMLSAXParser(boolean validating) {
        try {
			SAXParserFactory factory = SAXParserFactory.newInstance();
			factory.setValidating(validating);
			parser = factory.newSAXParser().getXMLReader();
		}
		catch (Exception e) {
			System.err.println("error: Unable to instantiate a parser.");
		}
		parser.setErrorHandler(this);
    } // Constructor

    /**
     * Parse the input XML file.
     *
     * @param The URI of the XML file.
     */
    public void parse(String uri) throws IOException, SAXException {
        parsingSuccess = true;
        parser.parse(uri);
    } // parse

    /**
     * Check whether XML parsing is successful.
     *
     * @return true/false
     */
    public boolean getParsingSuccess() {
		return parsingSuccess;
    } // getParsingSuccess

    /**
     * Turn on/off XML Parser's feature.
     *
     * @param featureId - Parser's feature ID.
     * @param state - true/false
     */
    public void setFeature(String featureId, boolean state) throws SAXException {
		if (parser == null) {
			return;
		}
		parser.setFeature(featureId, state);
    } // setFeature

    /**
     * Sets XML Parser's property.
     *
     * @param name - Property name.
     * @param value - Property value.
     */
    public void setProperty(String name, String value) throws SAXException {
		if (parser == null) {
			return;
		}
		parser.setProperty(name, value);
    } // setProperty

    //
    // ErrorHandler methods
    //

    /**
     * Called by the XML Parser when there is a parsing warning.
     *
     * @param ex - The warning information encoded as an exception.
     */
    public void warning(SAXParseException ex) throws SAXException {
        printError(MSG_WARNING, ex);
    } // warning

    /**
     * Called by the XML Parser when there is a parsing error.
     *
     * @param ex - The error information encoded as an exception.
     */
    public void error(SAXParseException ex) throws SAXException {
        printError(MSG_ERROR, ex);
    } // error

    /**
     * Called by the XML Parser when there is a fatal error.
     *
     * @param ex - The fatal error information encoded as an exception.
     */
    public void fatalError(SAXParseException ex) throws SAXException {
        printError(MSG_FATAL_ERROR, ex);
    } // fatalError

    /**
     * A standard method used to print parsing warnings, errors, and fatal errors.
     *
     * @param type - Error type (Warning/Error/Fatal Error)
     * @param ex - Instance of SAXParseException.
     */
    protected void printError(String type, SAXParseException ex) {
        parsingSuccess = false;

        System.err.print("[");
        System.err.print(type);
        System.err.print("] ");
        if (ex== null) {
            System.out.println("!!!");
        }
        String systemId = ex.getSystemId();
        if (systemId != null) {
            int index = systemId.lastIndexOf('/');
            if (index != -1)
                systemId = systemId.substring(index + 1);
            System.err.print(systemId);
        }
        System.err.print(':');
        System.err.print(ex.getLineNumber());
        System.err.print(':');
        System.err.print(ex.getColumnNumber());
        System.err.print(": ");
        System.err.print(ex.getMessage());
        System.err.println();
        System.err.flush();
    } // printError
} // XMLSAXParser
