
/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2003  All Rights Reserved
 */

package com.ibm.oti.vm;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.HttpURLConnection;
import java.util.Vector;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.jar.Attributes;
import java.security.Policy;
import java.security.CodeSource;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.ProtectionDomain;
import java.security.PermissionCollection;
import java.util.Hashtable;
import java.io.*;

public abstract class AbstractClassLoader extends java.lang.ClassLoader {
	private static ClassLoader systemClassLoader;
	String[] parsedPath;
	int[] types;
	Object[] cache;
	FilePermission permissions[];
	private static final RuntimePermission permissionToExitVM =
		new RuntimePermission("exitVM");

public AbstractClassLoader(){
	super();
}
public AbstractClassLoader(ClassLoader p1){
	super(p1);
}

String parsePath (String classPath) {
	int index = 0, count = 0, end = classPath.length();
	while (index < end) {
		int next = classPath.indexOf(File.pathSeparatorChar, index);
		if (next == -1) next = end;
		if (next - index > 0) count++;
		index = next + 1;
	}
	parsedPath = new String[count];
	types = new int[count];
	cache = new Object[count];
	index = count = 0;
	StringBuffer newPath = new StringBuffer();
	while (index < end) {
		int next = classPath.indexOf(File.pathSeparatorChar, index);
		if (next == -1) next = end;
		if (next - index > 0) {
			String path = classPath.substring(index, next);
			parsedPath[count++] = path;
		}
		index = next + 1;
	}
	return newPath.toString();
}

private void fillCache(int i) {
	types[i] = VM.getClassPathEntryType(this, i);
	switch (types[i]) {
		case VM.CPE_TYPE_UNKNOWN:
			cache[i] = cache;
			return;
		case VM.CPE_TYPE_DIRECTORY:
		case VM.CPE_TYPE_JAR:
			if (parsedPath[i] == null)
				parsedPath[i] = com.ibm.oti.util.Util.toString(VM.getPathFromClassPath(i));
			File f = new File(parsedPath[i]);
			String path;
			try {
				path = f.getCanonicalPath();
			} catch (IOException e) {
				path = f.getAbsolutePath();
			}
			if (types[i] == VM.CPE_TYPE_DIRECTORY) {
				if (path.charAt(path.length() -1) != File.separatorChar)
					path = new StringBuffer(path.length() + 1).
						append(path).append(File.separatorChar).toString();
				parsedPath[i] = path;
				cache[i] = cache;
			} else {
				// Must store absolute path so correct URL is created
				parsedPath[i] = path;
				try {
					ZipFile zf = new JarFile(parsedPath[i]);
					cache[i] = zf;
					return;
				} catch (IOException e) {}
				types[i] = VM.CPE_TYPE_UNUSABLE;
				cache[i] = cache;
			}
			return;
		case VM.CPE_TYPE_TCP:
			if (parsedPath[i] == null)
				parsedPath[i] = com.ibm.oti.util.Util.toString(VM.getPathFromClassPath(i));
			path = "http:" + parsedPath[i].substring(5, parsedPath[i].length()).replace('\\', '/');
			if (!path.endsWith("/")) path += '/';
			parsedPath[i] = path;
			cache[i] = cache;
			return;
		case VM.CPE_TYPE_JXE:
			types[i] = VM.CPE_TYPE_UNUSABLE;
			// fall into next case
		case VM.CPE_TYPE_UNUSABLE:
			cache[i] = cache;
			return;
	}
}

public InputStream getResourceAsStream(String resName) {
	if (resName == null || resName.length() < 1 || resName.charAt(0) == '/')
		return null;	// Do not allow absolute resource references!
	InputStream answer;
	if (this != systemClassLoader) {
		if (getParent() == null)
			answer = systemClassLoader.getResourceAsStream(resName);
		else
			answer = getParent().getResourceAsStream(resName);
		if (answer != null) return answer;
	}

	for (int i = 0; i < cache.length; ++i) {
		try {
			if (cache[i] == null) fillCache(i);
			switch (types[i]) {
				case VM.CPE_TYPE_JAR:
					ZipFile zf = (ZipFile)cache[i];
					ZipEntry entry;
					if ((entry = zf.getEntry(resName)) != null) {
						SecurityManager security = System.getSecurityManager();
						if (security != null) {
							if (permissions == null)
								permissions = new FilePermission[cache.length];
							FilePermission perm = permissions[i];
							if (perm == null) perm = permissions[i] = new FilePermission(parsedPath[i], "read");
							security.checkPermission(perm);
						}
						try {
							return zf.getInputStream(entry);
						} catch (IOException e) {}
					}
					continue;
				case VM.CPE_TYPE_DIRECTORY:
					String resourcePath = new StringBuffer(parsedPath[i].length() + resName.length()).
						append(parsedPath[i]).append(resName).toString();
					File f = new File(resourcePath);
					if (f.exists())
						try {
							return new BufferedInputStream(new FileInputStream(f));
						} catch (FileNotFoundException e) { }
					continue;
			}
		} catch (SecurityException e) {}
	}
	return null;
}

protected URL findResource(final String res) {
	URL result = (URL)AccessController.doPrivileged(new PrivilegedAction() {
		public Object run() {
			for (int i=0; i<cache.length; i++) {
				URL result = findResourceImpl(i, res);
				if (result != null) return result;
			}
			return null;
		}});
	if (result != null) {
		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
			try {
				sm.checkPermission(result.openConnection().getPermission());
			} catch (IOException e) {
				return null;
			} catch (SecurityException e) {
				return null;
			}
		}
	}
	return result;
}

private URL findResourceImpl(int i, String res) {
	if (res == null || res.length() == 0)
		return null;
	if (res.charAt(0) == '/')
		return null;	// Do not allow absolute resource references!

	if (cache[i] == null) fillCache(i);
	try {
		switch (types[i]) {
			case VM.CPE_TYPE_JAR:
				ZipFile zf = (ZipFile)cache[i];
				if (zf.getEntry(res) != null)
					return new URL("jar", null, -1, toURLString(parsedPath[i]) + "!/" + res, null);
				return null;
			case VM.CPE_TYPE_DIRECTORY:
				String resourcePath = new StringBuffer(parsedPath[i].length() + res.length()).
					append(parsedPath[i]).append(res).toString();
				File f = new File(resourcePath);
				if (f.exists())
					return new URL(toURLString(resourcePath));
				return null;
		}
	} catch (MalformedURLException e) {
	}
	return null;
}

protected Enumeration findResources(final String res) throws IOException {
	Vector result = (Vector)AccessController.doPrivileged(new PrivilegedAction() {
		public Object run() {
			Vector resources = new Vector();
			for (int i=0; i < cache.length; i++) {
				URL resource = findResourceImpl(i, res);
				if (resource != null) resources.addElement(resource);
			}
			return resources;
		}});
	SecurityManager sm;
	int length = result.size();
	if (length > 0 && (sm = System.getSecurityManager()) != null) {
		Vector reduced = new Vector(length);
		for (int i=0; i<length; i++) {
			URL url = (URL)result.elementAt(i);
			try {
				sm.checkPermission(url.openConnection().getPermission());
				reduced.addElement(url);
			} catch (IOException e) {
			} catch (SecurityException e) {
			}
		}
		result = reduced;
	}
	return result.elements();
}

/**
 * Answers a string representing the URL which matches the
 * given filename. The argument should be specified using the
 * standard platform rules. If it is not absolute, then it
 * is converted before use. The result will end in a slash
 * if the original path ended in a file separator.
 *
 * @author		OTI
 * @version		initial
 *
 */
static String toURLString(String filename) {
	String name = filename;
	if (!filename.startsWith("http:")) {
		if (File.separatorChar != '/') {
			// Must convert slashes.
			name = name.replace(File.separatorChar, '/');
		}
		StringBuffer buf = new StringBuffer(name.length() + 6).append("file:");
		if (!name.startsWith("/")) {
			// On Windows, absolute paths might not start with sep.
			buf.append('/');
		}
		name = buf.append(name).toString();
	}
	return name;
}

public static void setBootstrapClassLoader(ClassLoader bootstrapClassLoader) {
	// Should only be called once
	if (systemClassLoader != null)
		throw new IllegalArgumentException();
	systemClassLoader = bootstrapClassLoader;
}

/**
 * Answers the ProtectionDomain which should be associated with
 * the code source constructed using the given path and no signers.
 * <p>
 * This method is invoked by the VM to set the ProtectionDomain of
 * classes which are created internally (i.e. without calling one of
 * the defineClass methods).
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		ProtectionDomain
 *					the ProtectionDomain to be used for a class
 *					loaded from the given directory or JAR.
 * @param 		path String
 *					the path to compute the protection domain
 *					for.
 */
Object getFilePD(int index) {
	String path;
	if (index >= 0) {
		if (cache[index] == null) fillCache(index);
		path = parsedPath[index];
	} else path = File.separator;
	// The path will be one of the entries in the class path
	// in absolute form. The policy object should give read
	// access to all subdirs of this directory.
	ProtectionDomain pd = (ProtectionDomain) getProtectionDomainCache().get(path);
	if (pd == null) {
		URL url = null;
		if (path != null) {
			String urlString = toURLString(path);
			try {
				url = new URL(urlString);
			} catch (MalformedURLException ex) {}
		}
		// Need privilege to get the policy
		Policy policy = (Policy)AccessController.doPrivileged(new PrivilegedAction() {
			public Object run() {
				return Policy.getPolicy();
			}});
		if (policy != null) {
			CodeSource cs = new CodeSource(url, null);
			PermissionCollection pc = policy.getPermissions(cs);
			if (path != null && url.getProtocol().equals("file")) {
				// Special case: Add in a read permission for the
				// directory tree where the file lives.
				if (path.endsWith(File.separator)) {
					// It's a directory.
					pc.add(new FilePermission(path + "-", "read"));
				} else {
					// It's a JAR
					pc.add(new FilePermission(path, "read"));
				}
			}
			if (addExitPermission())
				pc.add(permissionToExitVM);
			pd = new ProtectionDomain(cs, pc);
			getProtectionDomainCache().put(path, pd);
		}
	}
	return pd;
}

boolean addExitPermission() {
	return false;
}

synchronized void definePackage(String packageName, final int cacheIndex) {
	if (getPackage(packageName) != null) return;

	if (cacheIndex >= 0 && cache[cacheIndex] == null) {
		AccessController.doPrivileged(new PrivilegedAction() {
		public Object run() {
			fillCache(cacheIndex);
			return null;
		}});
	}

	if (cacheIndex >= 0 && types[cacheIndex] == VM.CPE_TYPE_JAR) {
		Manifest manifest = null;
		try {
			JarFile jf = (JarFile)cache[cacheIndex];
			manifest = jf.getManifest();
		} catch (IOException e) {}
		if (manifest != null) {
			Attributes mainAttributes = manifest.getMainAttributes();
			String value = mainAttributes.getValue(Attributes.Name.SEALED);
			boolean sealed = value != null &&
				value.toLowerCase().equals ("true");
			String dirName = packageName.replace('.', '/') + "/";
			Attributes attributes = manifest.getAttributes(dirName);
			if (attributes != null) {
				value = attributes.getValue(Attributes.Name.SEALED);
				if (value != null)
					sealed = value.toLowerCase().equals("true");
			}
			URL url = null;
			try {
				if (sealed) url = new URL(toURLString(parsedPath[cacheIndex]));
			} catch (MalformedURLException e) {}

			definePackage(packageName,
				mainAttributes.getValue(Attributes.Name.SPECIFICATION_TITLE),
				mainAttributes.getValue(Attributes.Name.SPECIFICATION_VERSION),
				mainAttributes.getValue(Attributes.Name.SPECIFICATION_VENDOR),
				mainAttributes.getValue(Attributes.Name.IMPLEMENTATION_TITLE),
				mainAttributes.getValue(Attributes.Name.IMPLEMENTATION_VERSION),
				mainAttributes.getValue(Attributes.Name.IMPLEMENTATION_VENDOR),
				url);
			return;
		}
	}
	definePackage(packageName, null, null, null, null, null, null, null);
}

Hashtable getProtectionDomainCache() {
	return null;
}

/**
 * Answers the name of the package to which newClass belongs.
 *
 * @author		OTI
 * @version		initial
 */
String getPackageName(Class theClass)
{
	String name;
	int index;

	name = theClass.getName();
	if((index = name.lastIndexOf('.')) == -1) return null;
	return name.substring(0, index);
}

}
