
package java.lang;

import java.util.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2004  All Rights Reserved
 */

/**
 * ThreadGroups are containers of Threads and ThreadGroups, therefore providing
 * a tree-like structure to organize Threads. The root ThreadGroup name is "system"
 * and it has no parent ThreadGroup. All other ThreadGroups have exactly one parent
 * ThreadGroup. All Threads belong to exactly one ThreadGroup.
 *
 *
 * @author		OTI
 * @version		initial
 *
 * @see			VM
 * @see			Thread
 * @see			SecurityManager
 */

 public class ThreadGroup {
	private String name;					// Name of this ThreadGroup
	private int maxPriority = Thread.MAX_PRIORITY;		// Maximum priority for Threads inside this ThreadGroup
	ThreadGroup parent = null;			// The ThreadGroup to which this ThreadGroup belongs
	int numThreads = 0;
	private Thread[] childrenThreads = new Thread[5];		// The Threads this ThreadGroup contains
	int numGroups = 0;
	private ThreadGroup[] childrenGroups = new ThreadGroup[3];		// The ThreadGroups this ThreadGroup contains
	private boolean isDaemon = false;			// Whether this ThreadGroup is a daemon ThreadGroup or not
	private boolean isDestroyed = false;			// Whether this ThreadGroup has already been destroyed or not
	private int memorySpace;				// Memory space to associate all new threads with

/**
 * Used by the vm to create the system ThreadGroup.
 *
 * @author		OTI
 * @version		initial
 */
private ThreadGroup() {
}

/**
 * Constructs a new ThreadGroup with the name provided.
 * The new ThreadGroup will be child of the ThreadGroup to which
 * the <code>Thread.currentThread()</code> belongs.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		name		Name for the ThreadGroup being created
 *
 * @exception	SecurityException
 *					if <code>checkAccess()</code> for the parent group fails with a SecurityException
 *
 * @see			java.lang.Thread#currentThread
 */

public ThreadGroup(String name) {
	this(Thread.currentThread().getThreadGroup(), name);
}

/**
 * Constructs a new ThreadGroup with the name provided, as child of
 * the ThreadGroup <code>parent</code>
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		parent		Parent ThreadGroup
 * @param		name		Name for the ThreadGroup being created
 *
 *
 * @exception	NullPointerException
 *					if <code>parent<code> is <code>null<code>
 * @exception	SecurityException
 *					if <code>checkAccess()</code> for the parent group fails with a SecurityException
 * @exception	IllegalThreadStateException
 *					if <code>parent<code> has been destroyed already
 */
public ThreadGroup(ThreadGroup parent, String name) {
	super();
	if (Thread.currentThread() != null) {
		// If parent is null we must throw NullPointerException, but that will be done "for free"
		// with the message send below
		parent.checkAccess();
	}

	this.name = name;
	this.setParent(parent);
	if (parent != null) {
		this.setMaxPriority(parent.getMaxPriority());
		if (parent.isDaemon()) this.setDaemon(true);
	}
}
/**
 * Returns the number of Threads which are children of
 * the receiver, directly or indirectly.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @return		Number of children Threads
 *
 */

public int activeCount() {
	int count = numThreads;
	synchronized (this.childrenGroups) { // Lock this subpart of the tree as we walk
		for (int i = 0; i < numGroups; i++)
			count += this.childrenGroups[i].activeCount();
	}
	return count;
}
/**
 * Returns the number of ThreadGroups which are children of
 * the receiver, directly or indirectly.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @return		Number of children ThreadGroups
 *
 */

public int activeGroupCount() {
	int count = 0;
	synchronized (this.childrenGroups) { // Lock this subpart of the tree as we walk
		for (int i = 0 ; i < numGroups; i++)
			// One for this group & the subgroups
			count += 1 + this.childrenGroups[i].activeGroupCount();
	}
	return count;
}
/**
 * Adds a Thread to the receiver. This should only be visible to class
 * java.lang.Thread, and should only be called when a new Thread is created
 * and initialized by the constructor.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		thread		Thread to add to the receiver
 *
 * @exception	IllegalThreadStateException
 *					if the receiver has been destroyed already
 *
 * @see			remove
 */

final void add(Thread thread) throws IllegalThreadStateException {
	synchronized (this.childrenThreads) {
		if (!isDestroyed) {
			if (childrenThreads.length == numThreads) {
				Thread[] newThreads = new Thread[childrenThreads.length * 2];
				System.arraycopy(childrenThreads, 0, newThreads, 0, numThreads);
				synchronized (newThreads) {
					newThreads[numThreads++] = thread;
					childrenThreads = newThreads;
				}
			} else childrenThreads[numThreads++] = thread;
		} else throw new IllegalThreadStateException();
	}
}
/**
 * Adds a ThreadGroup to the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		g		ThreadGroup to add to the receiver
 *
 * @exception	IllegalThreadStateException
 *					if the receiver has been destroyed already
 *
 */
private void add(ThreadGroup g) throws IllegalThreadStateException {
	synchronized (this.childrenGroups) {
		if (!isDestroyed) {
			if (childrenGroups.length == numGroups) {
				ThreadGroup[] newGroups = new ThreadGroup[childrenGroups.length * 2];
				System.arraycopy(childrenGroups, 0, newGroups, 0, numGroups);
				synchronized (newGroups) {
					newGroups[numGroups++] = g;
					childrenGroups = newGroups;
				}
			} else childrenGroups[numGroups++] = g;
		}
		else throw new IllegalThreadStateException();
	}
}
/**
 * The definition of this method depends on the deprecated method <code>suspend()</code>.
 * The behavior of this call was never specified.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	b	boolean	Used to control low memory implicit suspension
 *
 * @deprecated Required deprecated method suspend().
 */

public boolean allowThreadSuspension(boolean b) {
	// Does not apply to this VM, no-op
	return true;
}
/**
 * If there is a SecurityManager installed, call <code>checkAccess</code>
 * in it passing the receiver as parameter, otherwise do nothing.
 *
 * @author		OTI
 * @version		initial
 *
 */
public final void checkAccess() {
	// Forwards the message to the SecurityManager (if there's one)
	// passing the receiver as parameter

	SecurityManager currentManager = System.getSecurityManager();
	if (currentManager != null) currentManager.checkAccess(this);
}
/**
 * Destroys the receiver and recursively all its subgroups. It is only legal
 * to destroy a ThreadGroup that has no Threads.
 *  Any daemon ThreadGroup is destroyed automatically when it becomes empty
 * (no Threads and no ThreadGroups in it).
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	IllegalThreadStateException
 *					if the receiver or any of its subgroups has been destroyed already
 * @exception	SecurityException
 *					if <code>this.checkAccess()</code> fails with a SecurityException
 */

public final void destroy(){
	checkAccess();

	synchronized (this.childrenThreads) { // Lock this subpart of the tree as we walk
		synchronized (this.childrenGroups) {
			if (isDestroyed)
				throw new IllegalThreadStateException(com.ibm.oti.util.Msg.getString("K0056"));
			if (numThreads > 0)
				throw new IllegalThreadStateException(com.ibm.oti.util.Msg.getString("K0057"));

			int toDestroy = numGroups;
			// Call recursively for subgroups
			for (int i = 0 ; i < toDestroy; i++) {
				// We always get the first element - remember,
				// when the child dies it removes itself from our collection. See below.
				this.childrenGroups[0].destroy();
			}

			if (parent != null) parent.remove(this);

			// Now that the ThreadGroup is really destroyed it can be tagged as so
			this.isDestroyed = true;
		}
	}
}
/**
 * Auxiliary method that destroys the receiver and recursively all its subgroups
 * if the receiver is a daemon ThreadGroup.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			destroy
 * @see			setDaemon
 * @see			isDaemon
 */

private void destroyIfEmptyDaemon() {
	// Has to be non-destroyed daemon to make sense
	synchronized (childrenThreads) {
		if (isDaemon && !isDestroyed && numThreads == 0) {
			synchronized (childrenGroups) {
				if (numGroups == 0) destroy();
			}
		}
	}
}
/**
 * Copies an array with all Threads which are children of
 * the receiver (directly or indirectly) into the array <code>threads</code>
 * passed as parameters. If the array passed as parameter is too small no
 * exception is thrown - the extra elements are simply not copied.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		Thread	array into which the Threads will be copied
 *
 * @return		How many Threads were copied over
 *
 */

public int enumerate(Thread[] threads) {
	return enumerate(threads, true);
}
/**
 * Copies an array with all Threads which are children of
 * the receiver into the array <code>threads</code>
 * passed as parameter. Children Threads of subgroups are recursively copied
 * as well if parameter <code>recurse</code> is <code>true</code>.
 *
 * If the array passed as parameter is too small no
 * exception is thrown - the extra elements are simply not copied.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		threads		array into which the Threads will be copied
 * @param		recurse		Indicates whether Threads in subgroups should be recursively copied as well or not
 *
 * @return		How many Threads were copied over
 *
 */

public int enumerate(Thread[] threads, boolean recurse) {
	return enumerateGeneric(threads, recurse, 0, true);
}
/**
 * Copies an array with all ThreadGroups which are children of
 * the receiver (directly or indirectly) into the array <code>groups</code>
 * passed as parameters. If the array passed as parameter is too small no
 * exception is thrown - the extra elements are simply not copied.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		ThreadGroup	array into which the ThreadGroups will be copied
 *
 * @return		How many ThreadGroups were copied over
 *
 */

public int enumerate(ThreadGroup[] groups) {
	return enumerate(groups, true);
}
/**
 * Copies an array with all ThreadGroups which are children of
 * the receiver into the array <code>groups</code>
 * passed as parameter. Children ThreadGroups of subgroups are recursively copied
 * as well if parameter <code>recurse</code> is <code>true</code>.
 *
 * If the array passed as parameter is too small no
 * exception is thrown - the extra elements are simply not copied.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		groups		array into which the ThreadGroups will be copied
 * @param		recurse		Indicates whether ThreadGroups in subgroups should be recursively copied as well or not
 *
 * @return		How many ThreadGroups were copied over
 *
 */

public int enumerate(ThreadGroup[] groups, boolean recurse) {
	return enumerateGeneric(groups, recurse, 0, false);
}
/**
 * Copies into <param>enumeration</param> starting at </param>enumerationIndex</param>
 * all Threads or ThreadGroups in the receiver. If </param>recurse</param>
 * is true, recursively enumerate the elements in subgroups.
 *
 * If the array passed as parameter is too small no
 * exception is thrown - the extra elements are simply not copied.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		threads		array into which the elements will be copied
 * @param		recurse		Indicates whether </param>recurseCollection</param> should be enumerated or not
 * @param		enumerationIndex		Indicates in which position of the enumeration array we are
 * @param		enumeratingThreads		Indicates whether we are enumerating Threads or ThreadGroups
 *
 * @return		How many elements were enumerated/copied over
 *
 */

private int enumerateGeneric(Object[] enumeration, boolean recurse, int enumerationIndex, boolean enumeratingThreads) {
	checkAccess();

	Object[] immediateCollection = enumeratingThreads ? (Object[])childrenThreads : (Object[])childrenGroups;
	synchronized (immediateCollection) { // Lock this subpart of the tree as we walk
		for (int i = enumeratingThreads ? numThreads : numGroups; --i >= 0;) {
			if (!enumeratingThreads || ((Thread)immediateCollection[i]).isAlive()) {
				if (enumerationIndex >= enumeration.length) return enumerationIndex;
				enumeration[enumerationIndex++] = immediateCollection[i];
			}
		}
	}

	if (recurse) { // Lock this subpart of the tree as we walk
		synchronized (childrenGroups) {
			for (int i = 0; i < numGroups; i++) {
				if (enumerationIndex >= enumeration.length) return enumerationIndex;
				enumerationIndex = childrenGroups[i].
						enumerateGeneric(enumeration, recurse, enumerationIndex, enumeratingThreads);
			}
		}
	}
	return enumerationIndex;
}
/**
 * Answers the maximum allowed priority for a Thread in the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the maximum priority (an <code>int</code>)
 *
 * @see			setMaxPriority
 */

public final int getMaxPriority() {
	return maxPriority;
}
/**
 * Answers the name of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's name (a java.lang.String)
 */

public final String getName() {
	return name;
}
/**
 * Answers the receiver's parent ThreadGroup. It can be null if the receiver
 * is the the root ThreadGroup.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the parent ThreadGroup
 *
 */

 public final ThreadGroup getParent() {
	if (parent != null)
		parent.checkAccess();
	return parent;
}
/**
 * Interrupts every Thread in the receiver and recursively in all its subgroups.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	SecurityException
 *					if <code>this.checkAccess()</code> fails with a SecurityException
 *
 * @see			Thread#interrupt
 */

public final void interrupt() {
	checkAccess();
	synchronized (this.childrenThreads) { // Lock this subpart of the tree as we walk
		for (int i = 0 ; i < numThreads; i++)
			this.childrenThreads[i].interrupt();
	}
	synchronized (this.childrenGroups) { // Lock this subpart of the tree as we walk
		for (int i = 0 ; i < numGroups; i++)
			this.childrenGroups[i].interrupt();
	}
}
/**
 * Answers true if the receiver is a daemon ThreadGroup, false otherwise.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		if the receiver is a daemon ThreadGroup
 *
 * @see			setDaemon
 * @see			destroy
 */

public final boolean isDaemon() {
	return isDaemon;
}
/**
 * Answers true if the receiver has been destroyed already, false otherwise.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		if the receiver has been destroyed already
 *
 * @see			destroy
 */

public boolean isDestroyed(){
	return isDestroyed;
}
/**
 * Outputs to <code>System.out</code> a text representation of the hierarchy of
 * Threads and ThreadGroups in the receiver (and recursively). Proper identation
 * is done to suggest the nesting of groups inside groups and threads inside groups.
 *
 * @author		OTI
 * @version		initial
 *
 */

public void list() {
	System.out.println();	// We start in a fresh line
	list(0);
}
/**
 * Outputs to <code>System.out</code>a text representation of the hierarchy of
 * Threads and ThreadGroups in the receiver (and recursively). The identation
 * will be four spaces per level of nesting.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	levels	How many levels of nesting, so that proper identetion can be output.
 *
 */

private void list(int levels) {
	for (int i = 0; i < levels; i++)
		System.out.print("    ");	// 4 spaces for each level

	// Print the receiver
	System.out.println(this.toString());

	// Print the children threads, with 1 extra identation
	synchronized (this.childrenThreads) {
		for (int i = 0; i < numThreads; i++) {
			for (int j = 0; j <= levels; j++)
				System.out.print("    "); // children get an extra identation, 4 spaces for each level
			System.out.println(this.childrenThreads[i]);
		}
	}
	synchronized (this.childrenGroups) {
		for (int i = 0; i < numGroups; i++)
			this.childrenGroups[i].list(levels + 1);
	}
}
/**
 * Answers true if the receiver is a direct or indirect parent group of
 * ThreadGroup <code>g</code>, false otherwise.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		g	ThreadGroup to test
 *
 * @return		if the receiver is parent of the ThreadGroup passed as parameter
 *
 */

 public final boolean parentOf(ThreadGroup g) {
	while (g != null) {
		if (this == g) return true;
		g = g.parent;
	}
	return false;
}
/**
 * Removes a Thread from the receiver. This should only be visible to class
 * java.lang.Thread, and should only be called when a Thread dies.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		thread		Thread to remove from the receiver
 *
 * @see			add
 */

final void remove(java.lang.Thread thread) {
	synchronized (childrenThreads) {
		for (int i=0; i<numThreads; i++) {
			if (childrenThreads[i].equals(thread)) {
				numThreads--;
				System.arraycopy (childrenThreads, i + 1, childrenThreads, i, numThreads - i);
				childrenThreads[numThreads] = null;
				break;
			}
		}
	}
	destroyIfEmptyDaemon();
}
/**
 * Removes an immediate subgroup from the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		g		Threadgroup to remove from the receiver
 *
 * @see			add
 */

private void remove(ThreadGroup g) {
	synchronized (childrenGroups) {
		for (int i=0; i<numGroups; i++) {
			if (childrenGroups[i].equals(g)) {
				numGroups--;
				System.arraycopy (childrenGroups, i + 1, childrenGroups, i, numGroups - i);
				childrenGroups[numGroups] = null;
				break;
			}
		}
	}
	destroyIfEmptyDaemon();
}
/**
 * Resumes every Thread in the receiver and recursively in all its subgroups.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	SecurityException
 *					if <code>this.checkAccess()</code> fails with a SecurityException
 *
 * @see			Thread#resume
 * @see			suspend
 *
 * @deprecated Requires deprecated method Thread.resume().
 */
public final void resume() {
	checkAccess();
	synchronized (this.childrenThreads) { // Lock this subpart of the tree as we walk
		for (int i = 0 ; i < numThreads; i++)
			this.childrenThreads[i].resume();
	}
	synchronized (this.childrenGroups) { // Lock this subpart of the tree as we walk
		for (int i = 0 ; i < numGroups; i++)
			this.childrenGroups[i].resume();
	}
}
/**
 * Configures the receiver to be a daemon ThreadGroup or not.
 * Daemon ThreadGroups are automatically destroyed when they become empty.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		isDaemon	new value defining if receiver should be daemon or not
 *
 * @exception	SecurityException
 *					if <code>checkAccess()</code> for the parent group fails with a SecurityException
 *
 * @see			isDaemon
 * @see			destroy
 */

public final void setDaemon(boolean isDaemon) {
	checkAccess();
	this.isDaemon = isDaemon;
}
/**
 * Configures the maximum allowed priority for a Thread in the receiver
 * and recursively in all its subgroups.
 *
 * One can never change the maximum priority of a ThreadGroup to be
 * higher than it was. Such an attempt will not result in an exception, it will
 * simply leave the ThreadGroup with its current maximum priority.

 *
 * @author		OTI
 * @version		initial
 *
 * @param		newMax		the new maximum priority to be set
 *
 * @exception	SecurityException
 *					if <code>checkAccess()</code> fails with a SecurityException
 * @exception	IllegalArgumentException
 *					if the new priority is greater than Thread.MAX_PRIORITY or less than
 *					Thread.MIN_PRIORITY
 *
 * @see			getMaxPriority
 */

public final void setMaxPriority(int newMax) {
	checkAccess();

	if (newMax <= this.maxPriority) {
		if (newMax < Thread.MIN_PRIORITY) newMax = Thread.MIN_PRIORITY;

		int parentPriority = parent == null ? newMax : parent.getMaxPriority();
		this.maxPriority = parentPriority <= newMax ? parentPriority : newMax;
		synchronized (this.childrenGroups) { // Lock this subpart of the tree as we walk
			for (int i = 0 ; i < numGroups; i++)
				this.childrenGroups[i].setMaxPriority(newMax); // ??? why not maxPriority
		}
	}
}
/**
 * Sets the parent ThreadGroup of the receiver, and adds the receiver to the parent's
 * collection of immediate children (if <code>parent</code> is not <code>null</code>).
 *
 * @author		OTI
 * @version		initial
 *
 * @param		parent		The parent ThreadGroup, or null if the receiver is to be the root ThreadGroup
 *
 *
 * @see			getParent
 * @see			parentOf
 */

private void setParent(ThreadGroup parent) {
	if (parent != null) parent.add(this);
	this.parent = parent;
}
/**
 * Stops every Thread in the receiver and recursively in all its subgroups.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	SecurityException
 *					if <code>this.checkAccess()</code> fails with a SecurityException
 *
 * @see			Thread#stop
 * @see			ThreadDeath
 *
 * @deprecated Requires deprecated method Thread.stop().
 */
public final void stop() {
	if (stopHelper())
		Thread.currentThread().stop();
}

private final boolean stopHelper() {
	checkAccess();

	boolean stopCurrent = false;
	synchronized (this.childrenThreads) { // Lock this subpart of the tree as we walk
		Thread current = Thread.currentThread();
		for (int i = 0 ; i < numThreads; i++)
			if (this.childrenThreads[i] == current) {
				stopCurrent = true;
			} else {
				this.childrenThreads[i].stop();
			}
	}
	synchronized (this.childrenGroups) { // Lock this subpart of the tree as we walk
		for (int i = 0 ; i < numGroups; i++)
			stopCurrent |= this.childrenGroups[i].stopHelper();
	}
	return stopCurrent;
}
/**
 * Suspends every Thread in the receiver and recursively in all its subgroups.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	SecurityException
 *					if <code>this.checkAccess()</code> fails with a SecurityException
 *
 * @see			Thread#suspend
 * @see			resume
 *
 * @deprecated Requires deprecated method Thread.suspend().
 */
public final void suspend() {
	if (suspendHelper())
		Thread.currentThread().suspend();
}

private final boolean suspendHelper() {
	checkAccess();

	boolean suspendCurrent = false;
	synchronized (this.childrenThreads) { // Lock this subpart of the tree as we walk
		Thread current = Thread.currentThread();
		for (int i = 0 ; i < numThreads; i++)
			if (this.childrenThreads[i] == current) {
				suspendCurrent = true;
			} else {
				this.childrenThreads[i].suspend();
			}
	}
	synchronized (this.childrenGroups) { // Lock this subpart of the tree as we walk
		for (int i = 0 ; i < numGroups; i++)
			suspendCurrent |= this.childrenGroups[i].suspendHelper();
	}
	return suspendCurrent;
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver.
 */

public String toString() {
	return getClass().getName() + "[name=" + this.getName() + ",maxpri=" + this.getMaxPriority() + "]" ;
}
/**
 * Any uncaught exception in any Thread has to be forwarded (by the VM) to the Thread's ThreadGroup
 * by sending this message (uncaughtException). This allows users to define custom ThreadGroup classes
 * and custom behavior for when a Thread has an uncaughtException or when it does (ThreadDeath).
 *
 * @author		OTI
 * @version		initial
 *
 * @param		t		Thread with an uncaught exception
 * @param		e		The uncaught exception itself
 *
 * @see			Thread#stop
 * @see			ThreadDeath
 */
public void uncaughtException(Thread t, Throwable e) {
	if (parent != null)
		parent.uncaughtException(t,e);
	else if (!(e instanceof ThreadDeath)) {
		// No parent group, has to be 'system' Thread Group
		e.printStackTrace(System.err);
	}
}
}
