package java.lang.ref;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2004  All Rights Reserved
 */

/**
 * Abstract class which describes behavior common to all reference objects.
 *
 * @author		OTI
 * @version		initial
 * @since		JDK1.2
 */
public abstract class Reference extends Object {
	private Object referent;
	private ReferenceQueue queue;
	private boolean enqueued;

/**
 * Make the referent null.  This does not force the reference object to be enqueued.
 *
 * @author		OTI
 * @version		initial
 */
public void clear () {
	referent = null;
}

/**
 * Force the reference object to be enqueued if it has been associated with a queue.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		boolean
 *					true if Reference is enqueued.
 *                  false otherwise.
 */
public boolean enqueue() {
	return enqueueImpl();
}

/**
 * Return the referent of the reference object.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Object
 *					Referent to which reference refers,
 *					or null if object has been cleared.
 */
public Object get() {
	return referent;
}

/**
 * Return whether the reference object has been enqueued.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		boolean
 *					true if Reference has been enqueued.
 *					false otherwise.
 */
public boolean isEnqueued () {
	synchronized(this) {
		return enqueued;
	}
}

/**
 * Enqueue the reference object on the associated queue.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		boolean
 *					true if the Reference was successfully
 *					enqueued.  false otherwise.
 */
boolean enqueueImpl() {
	ReferenceQueue tempQueue;
	boolean result;
	synchronized(this) {
		/* Static order for the following code (DO NOT CHANGE) */
		tempQueue = queue;
		queue = null;
		if(enqueued || tempQueue == null) return false;
		result = tempQueue.enqueue(this);
		if(result) {
			enqueued = true;
		}
		return result;
	}
}

/**
 * Constructs a new instance of this class.
 *
 * @author		OTI
 * @version		initial
 */
Reference() {
}

/**
 * Initialize a newly created reference object.
 *
 * @author		OTI
 * @version		initial
 */
void initReference (Object r) {
	initReferenceImpl(r);
}

/**
 * Initialize a newly created reference object.
 *
 * @author		OTI
 * @version		initial
 */
void initReference (Object r, ReferenceQueue q) {
	if (q == null) throw new NullPointerException();
	queue = q;
	enqueued = false;
	initReferenceImpl(r);
}

/**
 * Associate the reference object with the referent.
 *
 * @author		OTI
 * @version		initial
 */
private final native void initReferenceImpl(Object r);

void dequeue() {
	enqueued = false;
}

}

