module Entities

LINK_LAYER     = 1
IP_LAYER       = 2
IP_PROTO_LAYER = 3
PAYLOAD_LAYER  = 4

Field = Struct.new('Field', :name, :offset, :len)

def self.merge_bytes bytes
  return if bytes.index nil

  sum = 0; shift = (bytes.size-1) * 8
  bytes.each { |b|
    sum += b << shift
    shift -= 8
  }
  sum
end

# Wrapper for a layer.
# Allows to retreive packet field values by name.
# Subclasses define specific fields through self.fields.
class Header
  class << self
    attr_accessor :dataOffset, :fields, :fcache
  end

  # Common code for parsing IP header options and TCP header options.
  def self.parseOptions bytes, optionDescs
    result = []
    return result unless bytes

    curr = 0
    loop {
      return result unless option = bytes[curr]
      result << option

      # Return immediately when `End of options list' is encountered.
      return result if option == 0

      # Have length defined in the option definition?
      if optionDef = optionDescs[option]
        length = optionDef.first
      end

      # Option length is variable, must be found in the pattern.
      unless length
        length = bytes[curr+1]
        return result unless length
      end
      curr += length
    }
  end

  attr_reader :layer

  def self.getFldDef name
    raise "#{name}: undefined field" unless fld = fcache[name]
    fld
  end

  # Delete bytes belonging to field `name' from layer.
  def delFldBytes name
    fld = getFldDef name
    delLayerBytes fld[1], fld[1] + fld[2] - 1
  end

  def delLayerBytes from, to=nil
    from.upto(to || @layer.keys.max) { |offset| @layer.delete offset }
  end

  def dataOffset
    self.class.dataOffset
  end

  def initialize(layer)
    @layer = layer
    unless self.class.fcache
      self.class.fcache = {}
      self.class.fields.each{|f|
	fld = Field.new(*f)
	self.class.fcache[fld.name.kind_of?(Symbol) ? fld.name : fld.name.intern] = fld
      }
    end
  end

  def [](name)
    fld = getFldDef(name.kind_of?(Symbol) ? name : name.intern)
    if fld.len == 1
      data = @layer[fld.offset]
    else
      data = Pattern.get_range(@layer, fld.offset, fld.len);
    end
    data
  end

  def getFldDef name
    self.class.getFldDef name
  end

  def has_fld?(name)
    raise "parameter is not symbol: #{name}" \
      unless name.is_a?(Symbol)
    self.class.fcache[name] != nil
  end

  # If a field consists of more than one byte, the bytes are merged into a
  # single number.
  def get_fld(symbol)
    raise "parameter is not symbol: #{symbol}" \
      unless symbol.is_a?(Symbol)
    data = self[symbol]
    if data.is_a?(Array)
      Entities.merge_bytes(data)
    else
      data
    end
  end
  private :get_fld

  def maxOffset
    @layer.keys.max
  end

  def method_missing(*a)
    raise NoMethodError.new("#{a.first} in class: #{self.class}") \
      unless a.size == 1 && self.class.fcache[a.first]
    get_fld(a.first)
  end
  private :method_missing
end

class EtherHeader < Header
  # Note: multicast MPLS (0x8848) isn't supported in tcpdump (or anywhere else
  # in firewalls), so I don't include it.
  TYPE_MPLS_UNICAST = 0x8847
  TYPE_VLAN = 0x8100

  self.fields = [['dstmac', 0, 6], ['srcmac', 6, 6], ['ethtype', 12, 2]]

  def type_mpls
    self.ethtype == TYPE_MPLS_UNICAST
  end

  def self.mac_to_s mac
    mac.map { |b| "%02X" % b }.join(':')
  end

  def dstmac;	  (ret = self[:dstmac]).index(nil) ? nil : ret		     end
  def srcmac;     (ret = self[:srcmac]).index(nil) ? nil : ret		     end

  def type_vlan
    self.ethtype == TYPE_VLAN
  end
end

class IPHeader < Header
  self.fields = [['ver_len', 0, 1], ['tos', 1, 1], ['total_len', 2, 2],
    ['ip_id', 4, 2], ['flags', 6, 1], ['frag_offset', 6, 2], ['ttl', 8, 1],
    ['proto', 9, 1], ['hchksum', 10, 2], ['srcip', 12, 4], ['dstip', 16, 4]]

  # The Internet Protocol (IP) has provision for optional header fields
  # identified by an option type field.  Options 0 and 1 are exactly one
  # octet which is their type field.  All other options have their one
  # octet type field, followed by a one octet length field, followed by
  # length-2 octets of option data.  The option type field is sub-divided
  # into a one bit copied flag, a two bit class field, and a five bit
  # option number.  These taken together form an eight bit value for the
  # option type field.  IP options are commonly refered to by this value.
  # http://www.iana.org/assignments/ip-parameters
  #
  # The OPTIONS hash contains `option type'=>[length, description] pairs.
  OPTIONS = {
    0=>[1, 'End of options list'],
    1=>[1, 'NOP'],
    130=>[11, 'Security'],
    131=>[nil, 'Loose Source Routing'],
    68=>[nil, 'Internet Timestamp'],
    133=>[nil, 'Extended Security'],
    134=>[nil, 'Commercial Security'],
    7=>[nil, 'Record Route'],
    136=>[4, 'Stream Identifier'],
    137=>[nil, 'Strict Source Routing'],
    10=>[nil, 'Experimental Measurement'],
    11=>[4, 'MTU Probe'],
    12=>[4, 'MTU Reply'],
    205=>[nil, 'Experimental Flow Control'],
    142=>[nil, 'Experimental Access Control'],
    15=>[nil, 'Encode'],
    144=>[nil, 'IMI Traffic Descriptor'],
    145=>[nil, 'Extended Internet Protocol'],
    82=>[12, 'Traceroute'],
    147=>[10, 'Address Extension'],
    148=>[4, 'Router Alert'],
    149=>[nil, 'Selective Directed Broadcast Mode'],
    150=>[nil, 'NSAP Addresses Option'],
    151=>[nil, 'Dynamic Packet State'],
    152=>[nil, 'Upstream Multicast Packet'],
    25=>[nil, 'Quick-Start'] }

  #protocols
  PROTO_EGP = 8
  PROTO_EIGRP = 88
  PROTO_GRE = 47
  PROTO_ICMP = 1
  PROTO_IGMP = 2
  PROTO_IPIP = 94
  PROTO_OSPF = 89
  PROTO_TCP = 6
  PROTO_UDP = 17
  PROTO_IPSEC_ESP = 50
  PROTO_IPSEC_AH = 51
  PROTOCOLS = (PROTO_IPSEC_ESP | PROTO_IPSEC_AH | PROTO_ICMP | PROTO_TCP | PROTO_UDP)
  #flags
  RESERVED_FLAG = 0x80
  DF_FLAG = 0x40
  MORE_FLAG = 0x20
  ANY_FLAG = (RESERVED_FLAG | DF_FLAG | MORE_FLAG)
  ALL_FLAGS = [RESERVED_FLAG, DF_FLAG, MORE_FLAG]
  class << self
    attr_accessor :desc_proto, :proto_desc
  end
  self.proto_desc = {PROTO_EGP => 'egp', PROTO_EIGRP => 'eigrp',
    PROTO_GRE => 'gre', PROTO_ICMP=>'icmp', PROTO_IGMP=>'igmp',
    PROTO_IPIP => 'ipip', PROTO_OSPF => 'ospf', PROTO_TCP=>'tcp',
    PROTO_UDP=>'udp'}
  self.desc_proto = {}
  self.proto_desc.each { |k, v| self.desc_proto.store v, k }

  def dscp
    return unless tos = self[:tos]
    tos >> 2
  end

  def dataOffset
    header_len_bytes
  end

  def dscp
    self[:tos] && self[:tos] & 0b00111111
  end

  def flags
    byte = self[:flags]
    return unless byte
    byte & (RESERVED_FLAG | DF_FLAG | MORE_FLAG)
  end

  def flags_df
    flags ? (flags & DF_FLAG > 0) : nil
  end

  def flags_more
    flags ? (flags & MORE_FLAG > 0) : nil
  end

  def flags_reserved
    flags ? (flags & RESERVED_FLAG > 0) : nil
  end

  def frag_offset
    byte = Entities.merge_bytes(self[:frag_offset])
    return nil unless byte
    byte & 0x1fff
  end

  def frag_offset_bytes
    $_*8 if $_ = frag_offset
  end

  def header_len
    self[:ver_len] ? self[:ver_len] & 0xf : nil
  end

  def header_len_bytes
    $_*4 if $_ = header_len
  end

  # Returns an array of IP options types/ids (e.g. Timestamp option would be
  # identified by 68).
  def ipopts
    self.class.parseOptions Pattern.get_range(@layer, 20), OPTIONS
  end

  def precedence
    return unless tos = self[:tos]
    tos & 0x03
  end

  def proto_ipsec_ah
    self.proto && (self.proto == PROTO_IPSEC_AH)
  end

  def proto_ipsec_esp
    self.proto && (self.proto == PROTO_IPSEC_ESP)
  end

  def proto_tcp
    self.proto && (self.proto == PROTO_TCP)
  end

  def proto_udp
    self.proto && (self.proto == PROTO_UDP)
  end

  def proto_icmp
    self.proto && (self.proto == PROTO_ICMP)
  end

  def proto_igmp
    self.proto && (self.proto == PROTO_IGMP)
  end

  def version
    self[:ver_len] ? self[:ver_len] >> 4 & 0xf : nil
  end
end

class TCPHeader < Header
  self.fields = [['srcport', 0, 2], ['dstport', 2, 2], ['seq', 4, 4],
    ['ack', 8, 4], ['doffs_res', 12, 1], ['flags', 13, 1], ['window', 14, 2],
    ['chksum', 16, 2], ['urg_pnt', 18, 2]]
  FIN_FLAG = 0x01
  SYN_FLAG = 0x02
  RST_FLAG = 0x04
  PSH_FLAG = 0x08
  ACK_FLAG = 0x10
  URG_FLAG = 0x20
  ECE_FLAG = 0x40
  CWR_FLAG = 0x80
  ANY_FLAG = 0xff
  ALL_FLAGS = [FIN_FLAG, SYN_FLAG, RST_FLAG, PSH_FLAG, ACK_FLAG, URG_FLAG]
    #,ECE_FLAG, CWR_FLAG]
  class << self
    attr_accessor :flag_desc
  end
  self.flag_desc = {FIN_FLAG=>"fin", SYN_FLAG=>"syn", RST_FLAG=>"rst",
		    PSH_FLAG=>"psh", ACK_FLAG=>"ack", URG_FLAG=>"urg",
		    ECE_FLAG=>"ece", CWR_FLAG=>"cwr", ANY_FLAG=>"any"}

  OPTIONS = {
    0=>[1, 'End of option list'],
    1=>[1, 'No-Operation'],
    2=>[4, 'Maximum Segment Size'],
    3=>[3, 'TCP Window Scale'],
    4=>[2, 'Selective Acknowledgement'],
    8=>[10, 'TCP Timestamps']
  }

  def doffs_bytes
    doffs*4 if $_ = doffs
  end
  alias :dataOffset :doffs_bytes

  def doffs
    b = self[:doffs_res]
    b >> 4 if b
  end

  # Before using flags_*, check that flags is not nil.
  def flags_ack;	    flags ? (flags & ACK_FLAG > 0) : nil	     end
  def flags_cwr;	    flags ? (flags & CWR_FLAG > 0) : nil	     end
  def flags_ece;	    flags ? (flags & ECE_FLAG > 0) : nil	     end
  def flags_fin;	    flags ? (flags & FIN_FLAG > 0) : nil	     end
  def flags_psh;	    flags ? (flags & PSH_FLAG > 0) : nil	     end
  def flags_rst;	    flags ? (flags & RST_FLAG > 0) : nil	     end
  def flags_syn;	    flags ? (flags & SYN_FLAG > 0) : nil	     end
  def flags_urg;            flags ? (flags & URG_FLAG > 0) : nil	     end

  def reserved
    b = self[:doffs_res]
    b & 0x0f if b 
  end

  def tcpopts
    self.class.parseOptions Pattern.get_range(@layer, 20), OPTIONS
  end
end

class UDPHeader < Header
  self.fields = [['srcport', 0, 2], ['dstport', 2, 2], ['length', 4, 2],
    ['checksum', 6, 2]]
  self.dataOffset = 8
end

class ICMPHeader < Header
  # itype (and not type) is used because Object#type is defined. icode is
  # used for uniformity.
  self.fields = [['itype', 0, 1], ['icode', 1, 1], ['checksum', 2, 2],
    ['pointer', 4, 1], ['gateway', 4, 4], ['iid', 4, 2], ['iseq', 6, 2]]
  self.dataOffset = 8
end

# NOTE: this header describes verion 0 of IGMP. Currently, IGMP v3 is
# also defined, which leaves 4 versions of IGMP messages to support. To do
# it, I would have to create a separate header for each of IGMP versions.
# For now, IGMPHeader objects will no longer be created when IP protocol is
# IGMP.
class IGMPHeader < Header
  self.fields = [['itype', 0, 1], ['icode', 1, 1], ['checksum', 2, 2],
    ['id', 4, 4], ['group_addr', 8, 4], ['access_key', 12, 4]]
  self.dataOffset = 16;
end

class IPsecAH < Header
  self.fields = [['next_header', 0, 1], ['payload_len', 1, 1],
    ['reserved', 2, 2], ['spi', 4, 4], ['sequence_number', 8, 4],
    ['icv', 12, 4]]
end

class IPsecESP < Header
  self.fields = [['spi', 0, 4], ['sequence_number', 4, 4]]
end

class UnknownHeader < Header
  self.fields = []
end

class Pattern
  attr_reader :layers

  IP_PROTO_HEADER_MAP = Hash.new(UnknownHeader)
  IP_PROTO_HEADER_MAP.merge!({IPHeader::PROTO_IPSEC_ESP=>IPsecESP,
    IPHeader::PROTO_IPSEC_AH=>IPsecAH, IPHeader::PROTO_TCP=>TCPHeader,
    IPHeader::PROTO_UDP=>UDPHeader, IPHeader::PROTO_ICMP=>ICMPHeader})

  def initialize(layers)
    @layers = layers
  end

  def self.parse(str, maxLayer=nil)
    layers = {}
    layer_id = nil
    str.split(' ').each { |item|
      item =~ /(\d+):(\d+)=([\da-z]{1,2})/
      layer_id = $1.to_i
      break if layer_id > maxLayer if maxLayer
      (layers[layer_id] ||= {})[$2.to_i] = $3.hex
    }
    Pattern.new(layers)
  end

  def self.get_range(layer, ostart, len=nil)
    res = []
    to = len.nil? ? layer.keys.max : ostart+len-1

    ostart.upto(to) { |i| res.push layer[i] }
    res.empty? ? nil : res
  end

  def get_header(layer_id)
    res = nil
    layer = @layers[layer_id]
    return nil unless layer

    ipHeader = IPHeader.new(@layers[IP_LAYER])
    case layer_id
    when LINK_LAYER
      res = EtherHeader.new layer
    when IP_LAYER
      res = ipHeader
    when IP_PROTO_LAYER
      # Only ICMP, TCP and UDP headers are placed at layer 3. All other protocol
      # headers and payload go into layer 4.
      return unless ipHeader.proto_icmp || ipHeader.proto_tcp || ipHeader.proto_udp
      res = IP_PROTO_HEADER_MAP[ipHeader[:proto]].new(layer)
    when PAYLOAD_LAYER
      res = unless ipHeader.proto_ipsec_ah || ipHeader.proto_ipsec_esp
	UnknownHeader.new layer
      else
	# IPSEC-AH and IPSEC-ESP are placed on layer 4.
	IP_PROTO_HEADER_MAP[ipHeader[:proto]].new(layer)
      end
    end
    res
  end

  def ip
    @ip ||= get_header(IP_LAYER)
  end

  def ipp
    @ipp ||= get_header(IP_PROTO_LAYER)
  end

  def link
    @link ||= get_header(LINK_LAYER)
  end

  def payload
    @payload ||= get_header(PAYLOAD_LAYER)
  end

  # Return payload without padded bytes (Ethernet frames are padded to
  # minimum Ethernet frame size, which equals 64 bytes).
  def packetPayload
    return @packetPayload if @packetPayload

    return unless ip && payload

    # Min Ethernet frame length - min Ethernet header.
    minPktLen = 60 - 14

    pktLen = ip.total_len
    # Packet length >= minPktLen? Not padded! Don't need to cut payload.
    if pktLen && pktLen >= minPktLen
      return (@packetPayload = UnknownHeader.new payload.layer.clone)
    end

    # Maybe minimum packet length for the protocols + payload length is >=
    # than minPktLen? Then the packet is not padded.
    if !pktLen && ip.proto
      ipHeader = (ip.header_len_bytes || 20)
      ippHeader = ipp.dataOffset if ipp
      unless ippHeader
	ippHeader = if ip.proto_tcp 
	  20
	elsif ip.proto_icmp || ip.proto_udp
	  8
	else
	  0
	end
      end

      # Not padded? Don't need to cut payload.
      return (@packetPayload = UnknownHeader.new payload.layer.clone) \
	if (ipHeader + ippHeader + payload.maxOffset + 1) >= minPktLen
    end

    if pktLen && ip.proto
      # Packet length is known, and the packet is padded.
      layer = {}
      # If IP header length and IP protocol header length is known, we can
      # calculate payload length and strip padding after payload ends.
      if ipHeader = ip.header_len_bytes
	ippHeader = if ip.proto_icmp || ip.proto_tcp || ip.proto_udp
	  ipp ? ipp.dataOffset : nil
	else
	  0
	end

	if ippHeader
	  payloadLen = pktLen - ipHeader - ippHeader

	  payload.layer.each { |offset, val|
	    layer[offset] = val if offset < payloadLen
	  }
	  return if layer.empty?
	  return (@packetPayload = UnknownHeader.new layer)
	end
      end
    end

    # It is not known where padding starts, so remove all zero bytes from
    # the end of payload.
    layer = payload.layer.clone
    layer.keys.sort { |a, b| b <=> a }.each { |offset|
      break unless layer[offset] == 0

      layer.delete offset
    }
    return if layer.empty?
    @packetPayload = UnknownHeader.new layer
  end
end

end
