#!/bin/bash
usage() {
  echo "
Usage: $0 [commands...]

General Commands:
  --start-all-accepted                                     Start firewalld service
  --start-on-boot                                          Start firewalld on machine boot
  --stop                                                   Stop firewalld service
  --stop-on-boot                                           Disable start of firewall serivce on machine boot

Port Forwarding Rules:
  --redirect-port [src-port] [dest-port] [protocol]        Redirect incoming packets of the protocol [protocol] from [src-port] to [dest-port].
  --recall-redirect-port [src-port] [dest-port] [protocol] Recall redirection rule of the protocol [protocol] from [src-port] to [dest-port].
"
  exit 1
}

set_forward_rule_ipv4() {
  if ! firewall-cmd --quiet --permanent --direct --query-rule ipv4 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --add-rule ipv4 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2)
    echo "Setting forward rule, external ipv4 packets of $3 protocol from port $1 to port $2..$result"
  fi
  if ! firewall-cmd --quiet --permanent --direct --query-rule ipv4 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --add-rule ipv4 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2)
    echo "Setting forward rule, local(127.0.0.1) ipv4 packets of $3 protocol from port $1 to port $2..$result"
  fi
}

set_forward_rule_ipv6() {
  if ! firewall-cmd --quiet --permanent --direct --query-rule ipv6 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --add-rule ipv6 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2)
    echo "Setting forward rule, external ipv6 packets of $3 protocol from port $1 to port $2..$result"
  fi
  if ! firewall-cmd --quiet --permanent --direct --query-rule ipv6 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --add-rule ipv6 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2)
    echo "Setting forward rule, local(::1) ipv6 packets of $3 protocol from port $1 to port $2..$result"
  fi
}

unset_forward_rule_ipv4(){
  if firewall-cmd --quiet --permanent --direct --query-rule ipv4 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --remove-rule ipv4 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2)
    echo "Removing forward rule, external ipv4 packets of $3 protocol from port $1 to port $2..$result"
  fi
  if firewall-cmd --quiet --permanent --direct --query-rule ipv4 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --remove-rule ipv4 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2)
    echo "Removing forward rule, local(127.0.0.1) ipv4 packets of $3 protocol from port $1 to port $2..$result"
  fi
}

unset_forward_rule_ipv6(){
  if firewall-cmd --quiet --permanent --direct --query-rule ipv6 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --remove-rule ipv6 nat PREROUTING 0 -p $3 --dport $1 -j REDIRECT --to-ports $2)
    echo "Removing forward rule, external ipv6 packets of $3 protocol from port $1 to port $2..$result"
  fi
  if firewall-cmd --quiet --permanent --direct --query-rule ipv6 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2; then
    local result=$(firewall-cmd --permanent --direct --remove-rule ipv6 nat OUTPUT 0 -p $3 -o lo --dport $1 -j REDIRECT --to-ports $2)
    echo "Removing forward rule, local(::1) ipv6 packets of $3 protocol from port $1 to port $2..$result"
  fi
}

validate_forward_arguments() {
  re='^[0-9]+$'
  if ! [[ $1 =~ $re ]] || [[ $1 -gt 65535 ]] ; then
    echo "$1 is not valid port number"
	exit 1
  fi
  if ! [[ $2 =~ $re ]] || [[ $2 -gt 65535 ]] ; then
    echo "$2 is not valid port number"
    exit 1
  fi
}

set_forward_rule() {
  if ! firewall-cmd --state; then
    echo "Cannot create forwarding rule since firewall is down"
	exit 1
  fi
  set_forward_rule_ipv4 $1 $2 $3
  set_forward_rule_ipv6 $1 $2 $3
  echo "Reloading firewall..$(firewall-cmd --reload)"
}

unset_forward_rule() {
  if ! firewall-cmd --state; then
    echo "Cannot remove forwarding rule since firewall is down"
	exit 1
  fi
  unset_forward_rule_ipv4 $1 $2 $3
  unset_forward_rule_ipv6 $1 $2 $3
  echo "Reloading firewall..$(firewall-cmd --reload)"
}

if [[ `id -un` != "root" ]]; then
	echo "Script must run with sudo privilges"
	exit 1
fi

PORT_FORWARD_RULES=()

while (( "$#" )); do
  case $1 in
    --stop)
      firewall-cmd --state >& /dev/null && systemctl stop firewalld.service
      exit $?;;
    --stop-on-boot)
	  systemctl is-enabled firewalld.service >& /dev/null && systemctl disable firewalld.service;;
	--start-all-accepted)
	  ! firewall-cmd --state && systemctl start firewalld
	  [[ "$(firewall-cmd --get-default-zone)" != "trusted" ]] && echo "Setting all connection on ACCEPT filter..$(firewall-cmd --set-default-zone=trusted)";;
	--start-on-boot)
	  ! systemctl is-enabled firewalld >& /dev/null && systemctl enable firewalld.service;;
	--redirect-port)
	  validate_forward_arguments $2 $3
	  set_forward_rule $2 $3 $4
	  shift; shift; shift;;
	--recall-redirect-port)
	  validate_forward_arguments $2 $3
	  unset_forward_rule $2 $3 $4
	  shift; shift; shift;;
	 --help)
	   usage
  esac
  shift
done

exit 0

