#!/bin/sh -e

## New script to run purging stored procedures(functions)
##  Run once per day (on 04:00 AM) / or only for jobs with parameter jobs (cli purge start command)

## Environment variables
# LOG_FORMAT           - Prints logs to stdout stream with given format (available formats: JSON).
# DEFAULT_AGING_PERIOD - If yes use default purging value (i.e. 366 days).
# DB_HOST              - DB host default is 127.0.0.1
# DB_PASSWORD          - By default execution is password-less, in case which DB_PASSWORD exist appending password.
# DB_USER              - DB user

log_file=/opt/allot/log/db_purge.log

log_stdout_json() {
  TIMESTAMP="$(date +"%Y-%m-%dT%H:%M:%S.%3N")"
  echo "{\"time\": \"$TIMESTAMP\", \"format_version\": \"1\", \"message\": \"$2\", \"logger_name\": \"purge.db.job\", \"thread_name\": \"main\", \"level\": \"$1\", \"app\":\"as-dsc-db-purge-job\"}"
}

log_file() {
  echo "$(date +"%Y-%m-%dT%H:%M:%S.%3N") - $1: $2" >> "$log_file"
}

log() {
  if [ -n "$LOG_FORMAT" ] && [ "$LOG_FORMAT" = "JSON" ]; then
    log_stdout_json "$1" "$2"
  else
    log_file "$1" "$2"
  fi
}

rolling_logger_adapter() {
  MSIZE=5000000
  if [ -e ${log_file} ] ; then
      FILESIZE=$(stat -c%s ${log_file})
  else
      FILESIZE=0
  fi
  if [ $FILESIZE -gt $MSIZE ] ; then
    /bin/cp -fp ${log_file} ${log_file}.1
    touch ${log_file}
  fi
}

set_aging_by_disk_size() {
   #Adjust aging period according to disk usage
   #get available disk size in bytes with limit 90%
   disk_size=$(df -kP /opt/shared | grep shared | awk '{print ($4*1024)*0.9}')
   if [ -n "$disk_size" ] ; then
      log "INFO" "Common storage detected, /opt/shared"
      disk_size=$(printf "%0.f" "$disk_size")
      full_disk_size=$(df -kP /opt/shared | grep shared | awk '{print ($2*1024)}')
	else
	  disk_size=$(df -kP /opt | grep opt | awk '{print ($4*1024)*0.9}')
	  if [ -n "$disk_size" ] ; then
	     log "INFO" "Standalone installation detected, /opt"
	     disk_size=$(printf "%0.f" "$disk_size")
       full_disk_size=$(df -kP /opt | grep opt | awk '{print ($2*1024)}')
	   else
	      log "WARN" "Cannot find DB partition, calculation by / path"
        disk_size=$(df -kP / | grep " /" | awk '{print ($4*1024)*0.9}')
        disk_size=$(printf "%0.f" "$disk_size")
        full_disk_size=$(df -kP / | grep " /" | awk '{print ($2*1024)}')
	    fi
    fi
   log "INFO" "Available disk_size = $disk_size"
   log "INFO" "Full disk_size = $full_disk_size"

   #get adjusted aging period
   New_Aging_PeriodDays=$(/opt/allot/bin/psql-query -c "select purge.get_aging_days($disk_size,$full_disk_size)")
   log "INFO" "New_Aging_PeriodDays=$New_Aging_PeriodDays"
   Aging_PeriodDays=$New_Aging_PeriodDays
}

rolling_logger_adapter

#check only jobs parameter
jobs_only=1
if [ x$1 = 'x' ] || [ "$1" != "jobs" ]  ; then
  log "INFO" "Complete purge procedure enabled"
  jobs_only=0
fi

# Default aging period day 45 days
Aging_PeriodDays=45
log "INFO" "Purge process started.."

if [ $jobs_only -eq 0 ] ; then

    Default_Aging_PeriodDays=$(/opt/allot/bin/psql-query -c "select public.get_int_param('Purge','history')")

    # Set aging by disk size, if environment variable DEFAULT_AGING_PERIOD does not exist or differ than yes.
    if [ -z "$DEFAULT_AGING_PERIOD" ] || [ "$DEFAULT_AGING_PERIOD" != "yes" ]; then
      set_aging_by_disk_size
    else
      Aging_PeriodDays=$Default_Aging_PeriodDays
    fi

   log "INFO" "Aging period is $Aging_PeriodDays days"
   Min_available_disk_space=$(/opt/allot/bin/psql-query -c "select public.get_int_param('Purge','available_disk_space')")
   
   if [ $Aging_PeriodDays -ne $Default_Aging_PeriodDays ] ; then
      log "WARN" "!!! Available disk space for data less or equal to $Min_available_disk_space % !!!"
      log "WARN" "Aging period is less than $Default_Aging_PeriodDays due absence of available disk space for DB"
   fi    

   if [ $Aging_PeriodDays -lt 30 ] ; then
      log "WARN" "!!! Available disk space for data less or equal to $Min_available_disk_space % !!!"
      log "WARN" "!!! Aging period is critical low due absence of available disk space for DB !!!"
   fi
   
   
	# Processing detect purge
	log "INFO" "Processing detect_purge started "
	/opt/allot/bin/psql-exec -c "select purge.detect_purge($Aging_PeriodDays)"
	ret=$?
	
	if [ $ret -eq 0 ] ; then
		log "INFO" "Processing detect_purge finished"
	else
		log "ERROR" "Processing detect_purge failed"
	fi	
	
	# Processing quarantine purge
	log "INFO" "Processing quarantine_purge started"
	/opt/allot/bin/psql-exec -c "select purge.quarantine_purge($Aging_PeriodDays)"
	ret=$?

	if [ $ret -eq 0 ] ; then
		log "INFO" "Processing quarantine_purge finished"
	else
		log "ERROR" "Processing quarantine_purge failed"
	fi	
	
	# Processing smp purge
	log "INFO" "Processing smp_purge started"
	/opt/allot/bin/psql-exec -c "select purge.smp_purge($Aging_PeriodDays)"
	ret=$?

	if [ $ret -eq 0 ] ; then
		log "INFO" "Processing smp_purge finished"
	else
		log "ERROR" "Processing smp_purge failed"
	fi	

	log "INFO" "Processing stats_purge started"
	/opt/allot/bin/psql-exec -c "select purge.stats_purge($Aging_PeriodDays)"
	ret=$?

	if [ $ret -eq 0 ] ; then
		log "INFO" "Processing stats_purge finished"
	else
		log "ERROR" "Processing stats_purge failed"
	fi	
	
	# Processing fast purge
	log "INFO" "Processing fast_purge started"
	/opt/allot/bin/psql-exec -c "select purge.fast_purge($Aging_PeriodDays)"
	ret=$?

	if [ $ret -eq 0 ] ; then
		log "INFO" "Processing fast_purge finished"
	else
		log "ERROR" "Processing fast_purge failed  "
	fi	
fi	

# Processing  jobs purge
	log "INFO" "Processing jobs_purge started"
	/opt/allot/bin/psql-exec -c  "select purge.jobs_purge()"
	ret=$?

	if [ $ret -eq 0 ] ; then
		log "INFO" "Processing jobs_purge finished"
	else
		log "ERROR" "Processing jobs_purge failed"
	fi	


log "INFO" "Purge finished "
exit 0

