-- New BGP tables

DO LANGUAGE plpgsql $$
       DECLARE
		subnet_community text;
		host_community text;
		
		BEGIN
		
		SET search_path = public, pg_catalog; 
		
		PERFORM 1
		    FROM
		        pg_catalog.pg_class c
		        JOIN pg_catalog.pg_namespace n ON n.oid = c.relnamespace
		    WHERE
		        n.nspname = 'public'
		        AND c.relname in ('bgp_config', 'bgp_peers_config', 'bgp_communities')
		        AND c.relkind = 'r';
		        
		    IF NOT FOUND THEN
			  -- bgp_config
		      CREATE TABLE bgp_config (
               id SERIAL NOT NULL,
			   name text,
               ips inet[],
               default_trigger_duration integer DEFAULT 3600,
               max_num_hosts smallint DEFAULT 20,
               host_coverage_threshold real DEFAULT 20,
               default_host_community integer,
               default_subnet_community integer,
               username text,
               password text,
               constraint pk_bgp_config primary key(id)
              );

              ALTER TABLE bgp_config OWNER TO nms;
                         
              REVOKE ALL ON TABLE bgp_config FROM PUBLIC;
              REVOKE ALL ON TABLE bgp_config FROM nms;
              GRANT ALL ON TABLE bgp_config TO nms;
              GRANT SELECT ON TABLE bgp_config TO ndb_readers;
              GRANT SELECT ON TABLE bgp_config TO ndb_writers;
              GRANT ALL ON TABLE bgp_config TO PUBLIC;
              GRANT ALL ON TABLE bgp_config TO ndbadm;
			  
			  -- bgp_peers_config
			  CREATE TABLE bgp_peers_config (
               id  SERIAL NOT NULL,
			   name text,
               ip inet,
               asn integer,
               password text,
	           constraint pk_bgp_peers_config primary key(id)
              );

              ALTER TABLE bgp_peers_config OWNER TO nms;
			              		  
              REVOKE ALL ON TABLE bgp_peers_config FROM PUBLIC;
              REVOKE ALL ON TABLE bgp_peers_config FROM nms;
              GRANT ALL ON TABLE bgp_peers_config TO nms;
              GRANT SELECT ON TABLE bgp_peers_config TO ndb_readers;
              GRANT SELECT ON TABLE bgp_peers_config TO ndb_writers;
              GRANT ALL ON TABLE bgp_peers_config TO PUBLIC;
              GRANT ALL ON TABLE bgp_peers_config TO ndbadm;
			  
			  -- bgp_communities 
               CREATE TABLE bgp_communities (
                id SERIAL NOT NULL,
                community text,
	            constraint pk_bgp_communities primary key(id)
               );
			   
			  ALTER TABLE bgp_communities OWNER TO nms;
              		  
              REVOKE ALL ON TABLE bgp_communities FROM PUBLIC;
              REVOKE ALL ON TABLE bgp_communities FROM nms;
              GRANT ALL ON TABLE bgp_communities TO nms;
              GRANT SELECT ON TABLE bgp_communities TO ndb_readers;
              GRANT SELECT ON TABLE bgp_communities TO ndb_writers;
              GRANT ALL ON TABLE bgp_communities TO PUBLIC;
              GRANT ALL ON TABLE bgp_communities TO ndbadm;
             
		    END IF;
			
		SET search_path = flood, pg_catalog;
		
		PERFORM 1
		    FROM
		        pg_catalog.pg_class c
		        JOIN pg_catalog.pg_namespace n ON n.oid = c.relnamespace
		    WHERE
		        n.nspname = 'flood'
		        AND c.relname in ('bgp_events', 'bgp_peers_status')
		        AND c.relkind = 'r';
		        
		    IF NOT FOUND THEN
			
			  -- new domain flood.bgp_events_status
			  CREATE DOMAIN bgp_events_status AS text
	           CONSTRAINT bgp_events_status_check CHECK ((VALUE = ANY (ARRAY['SUCCESS'::text, 'UNREACHABLE'::text, 'NOT_CONFIGURED'::text])));

              ALTER DOMAIN flood.bgp_events_status OWNER TO ndbadm;
			  
			  COMMENT ON DOMAIN bgp_events_status IS 'Allowed values (SUCCESS / UNREACHABLE / NOT_CONFIGURED)';
			  		  
			  -- bgp_events
			  CREATE TABLE bgp_events (
			   id SERIAL NOT NULL,
               subnet INET,
               flood_id integer,
               pattern_id integer,
               community integer,
               since timestamp with time zone DEFAULT now() NOT NULL,
               source action_source_type,
               source_id integer,
			   status bgp_events_status NOT NULL,
               expiry timestamp with time zone,
               type event_types,
			   target_type text,
			   constraint pk_bgp_events primary key(id)
              );

			  ALTER TABLE bgp_events OWNER TO nms;
			  
			  COMMENT ON COLUMN bgp_events.target_type IS 'HOST/SUBNET';
              		  
              REVOKE ALL ON TABLE bgp_events FROM PUBLIC;
              REVOKE ALL ON TABLE bgp_events FROM nms;
              GRANT ALL ON TABLE bgp_events TO nms;
              GRANT SELECT ON TABLE bgp_events TO ndb_readers;
              GRANT SELECT ON TABLE bgp_events TO ndb_writers;
              GRANT ALL ON TABLE bgp_events TO PUBLIC;
              GRANT ALL ON TABLE bgp_events TO ndbadm;
			  
			  -- bgp_peers_status
			  CREATE TABLE bgp_peers_status ( 
               id  SERIAL NOT NULL, 
               status status_type NOT NULL, 
               since timestamp with time zone DEFAULT now() NOT NULL,
               constraint pk_bgp_peers_status primary key(id)
              );
               
			  ALTER TABLE bgp_peers_status OWNER TO nms;
              		  
              REVOKE ALL ON TABLE bgp_peers_status FROM PUBLIC;
              REVOKE ALL ON TABLE bgp_peers_status FROM nms;
              GRANT ALL ON TABLE bgp_peers_status TO nms;
              GRANT SELECT ON TABLE bgp_peers_status TO ndb_readers;
              GRANT SELECT ON TABLE bgp_peers_status TO ndb_writers;
              GRANT ALL ON TABLE bgp_peers_status TO PUBLIC;
              GRANT ALL ON TABLE bgp_peers_status TO ndbadm; 
			  
			END IF; 

        -- upgrade from old flood.BGP tables to new BGP table (only if floog.bgp_config.ip is NOT NULL)
        
		PERFORM 1
		    FROM flood.bgp_config
            WHERE ip is NOT NULL;
         
            IF 	FOUND THEN
			
			 -- prepare data for public.bgp_communities table
			 UPDATE flood.bgp_config SET ip = '{' || ip || '}';
			 INSERT INTO public.bgp_communities (community) SELECT rtbhcom FROM flood.bgp_config WHERE rtbhcom is not NULL ; -- id =1
			 INSERT INTO public.bgp_communities (community) SELECT dvrtcom FROM flood.bgp_config 
			                                                 WHERE dvrtcom is not NULL and (dvrtcom <> rtbhcom or rtbhcom is NULL); -- id =2

			 -- prepare data for public.bgp_config
			 INSERT INTO public.bgp_config (name, ips, default_trigger_duration, max_num_hosts, host_coverage_threshold, username)
			   SELECT 'default', ip::inet[], timeout, top_hosts_limit, top_hosts_bw, 'root' FROM flood.bgp_config ;
			 UPDATE  public.bgp_config SET default_host_community = bgp_com.id FROM public.bgp_communities bgp_com, flood.bgp_config fl_bgp
     			 WHERE bgp_com.community=fl_bgp.rtbhcom; 
			 UPDATE  public.bgp_config SET default_subnet_community = bgp_com.id FROM public.bgp_communities bgp_com, flood.bgp_config fl_bgp
     			 WHERE bgp_com.community=fl_bgp.dvrtcom; 
			 
			 -- prepare data for public.bgp_peers_config
			 INSERT INTO public.bgp_peers_config(name, ip, asn, password)
               SELECT peer_ip, peer_ip::inet, as_number::integer, password 	FROM flood.bgp_peers ;	

             -- prepare data for flood.bgp_peers_status
             INSERT INTO flood.bgp_peers_status (id,status,since) 
               SELECT id, 'DOWN', now() FROM public.bgp_peers_config ;	
			   
			 -- clean/update BGP targets
              	SELECT ' ' || TRIM(community) into subnet_community from public.bgp_communities
                   WHERE id in (SELECT default_subnet_community FROM public.bgp_config); 	
                SELECT ' ' || TRIM(community) into host_community from public.bgp_communities
                   WHERE id in (SELECT default_host_community FROM public.bgp_config); 	

                IF 	subnet_community is NOT NULL AND subnet_community <> ' ' THEN 
                    UPDATE notify.targets SET addr = addr || subnet_community WHERE target_type_id = 12 AND name LIKE '%-subnet-%'
					       AND target_id in (SELECT target_id FROM notify.filter_targets );  
                END IF;	

                IF 	host_community is NOT NULL AND host_community <> ' ' THEN 
                    UPDATE notify.targets SET addr = addr || host_community WHERE target_type_id = 12 AND name LIKE '%-host-%'
					       AND target_id in (SELECT target_id FROM notify.filter_targets );   
                END IF;		
                
                --	targets tables
                DELETE FROM notify.complete_targets 
				  WHERE target_id in (SELECT target_id FROM notify.targets WHERE target_type_id = 12 ) AND
				        target_id not in ( SELECT target_id FROM notify.filter_targets 
						                    WHERE target_id in (SELECT target_id FROM notify.targets WHERE target_type_id = 12));
				DELETE FROM notify.targets WHERE target_type_id = 12 AND 
				  	        target_id not in ( SELECT target_id FROM notify.filter_targets 
						                       WHERE target_id in (SELECT target_id FROM notify.targets WHERE target_type_id = 12));							
	    					
            ELSE
              -- clean BGP targets
               DELETE FROM notify.filter_targets WHERE target_id in (SELECT target_id FROM notify.targets WHERE target_type_id = 12);	 
               DELETE FROM notify.complete_targets WHERE target_id in (SELECT target_id FROM notify.targets WHERE target_type_id = 12);	
               DELETE FROM notify.targets WHERE target_type_id = 12;			   

            END IF;
        

        -- drop old flood.BGP tables

        DROP TABLE IF EXISTS flood.bgp_config;
        DROP TABLE IF EXISTS flood.bgp_peers;		
		DROP TABLE IF EXISTS flood.bgp_events_log;
        DROP TABLE IF EXISTS flood.bgp_requests_log;
        DROP TABLE IF EXISTS flood.bgp_peers_log;
        DROP TABLE IF EXISTS flood.bgp_live;
		DROP TABLE IF EXISTS flood.bgp_gui_current_pattern;
		
		-- alter domain
		ALTER DOMAIN flood.event_types drop constraint event_types_check;

        ALTER DOMAIN flood.event_types add constraint event_types_check 
		                               CHECK ((VALUE = ANY (ARRAY['TRIGGER'::text, 'WITHDRAW'::text, 'WITHDRAW_ALL'::text,
                                        'WITHDRAW_HOSTS'::text, 'WITHDRAW_SUBNETS'::text])));
        COMMENT ON DOMAIN flood.event_types IS 'Allowed values - (TRIGGER / WITHDRAW / WITHDRAW_ALL / WITHDRAW_HOSTS / WITHDRAW_SUBNETS)';
		
		END;
$$;  