#!/bin/bash
################################################################################
# The scrip will install/update/uninstall Allot product
# Appropriate TGZ file must be located in the same directory as the script
################################################################################
declare MD5SUM="e10acddf797295c97d3bfa49a920584b"
declare VERBUILD=17.3.10-68
declare ARCH="x86_64"
declare ALLOTVER
declare THERPM
declare CURVER
declare RCODE=0
declare SCRIPT=`basename $0`
declare TPATH=`dirname $0`
declare INSTD=/tmp/inst.`date +%s`
declare NM=${SCRIPT#Allot_}
declare ALLOTPRODUCT=${NM%.sh}
declare PRODP=/opt
declare TMPP=/dev/shm
declare db_space=2000 # KB
declare LOGD=/opt/allot/log
declare LOGF=${SCRIPT%.sh}_inst.log
declare ACP_MINV=17.13.1
# ----------------------------------------- #
usage() {
echo "Usage:  $SCRIPT [ OPTION ]
    -i, -install : install the product,
    -u, -upgrade : upgrade the product,
    -r, -remove  : uninstall the product,
    -h, -help    : show help message.
"
exit_scr 1
}
# ----------------------------------------- #
# Clean temporary directory and exit script
exit_scr () {
rm_dir
exit $1
}
# ----------------------------------------- #
# Function compares versions given in number-dot notation
# Returns: 0 if $1 = $2
#          1 -  $1 > $2
#          2 -  $1 < $2
#########################
checkver () {
if [[ $1 == $2 ]]
then
    return 0
fi
local IFS=.
local i ver1=($1) ver2=($2)
# fill empty fields in ver1 with zeros
for ((i=${#ver1[@]}; i<${#ver2[@]}; i++))
do
    ver1[i]=0
done
for ((i=0; i<${#ver1[@]}; i++))
do
    if [[ -z ${ver2[i]} ]]
    then
        # fill empty fields in ver2 with zeros
        ver2[i]=0
    fi
    if ((10#${ver1[i]} > 10#${ver2[i]}))
    then
        return 1
    fi
    if ((10#${ver1[i]} < 10#${ver2[i]}))
    then
        return 2
    fi
done
return 0
} # checkver
# ----------------------------------------- #
function check_ACP () {
# check allottype - $ACP_MINV or higher version needed
allottype >& /dev/null 
if [ $? -ne 0 ] ; then
    echo "ACP Installation should be used."
    let RCODE=$RCODE+1
else
    ACPVER=`allottype | grep -E "kickstart|ACP" | awk -F " " '{print $2}'`
#    if [ `echo "$ACPVER >= $ACP_MINV" | bc -q` -ne 1 ] ; then 
    checkver $ACPVER $ACP_MINV
    if [ $? -eq 2 ] ; then
        echo "ACP $ACP_MINV or higher version should be installed."
        let RCODE=$RCODE+1
    fi
fi
} # check_ACP
# ----------------------------------------- #
check_space () {
# Make sure there is free space the size of uncompressed tgz file on tmp
# and additional space for DB on /opt.
local free_space=`df -P -k $PRODP | tail -n 1 | awk '{print $4}'` # KB
local tmp_space=`df -P -k $TMPP | tail -n 1 | awk '{print $4}'` # KB
local need_space
local ret=0
#local db_space=2000000
let need_space=`gzip -l ${TPATH}/Allot_${ALLOTPRODUCT}*.tgz | tail -n 1 | awk '{print $2}'`/1024 # KB
if [ $tmp_space -lt $need_space ] ; then
    echo "In order to install this software version you need $need_space KB on $TMPP."
    echo "You have only $tmp_space KB available which may cause a problem."
    ask_yn
fi
if [ "$1" == "-i" ] || [ "$1" == "-install" ] ; then 
    let need_space=$need_space+$db_space
fi
if [ $free_space -lt $need_space ] ; then
    echo "In order to install this software version you need $need_space KB on $PRODP."
    echo "You have only $free_space KB available which is not enough."
    echo "Please free some disk space and re-run the installation."
    let ret=$ret+1    
fi
let RCODE=$RCODE+$ret
}
# ----------------------------------------- #
# Check user - must be root
check_user () {
you=`id -un`
if [ ! "$you" = "root" ] ; then
    let RCODE=$RCODE+1
    echo "Operation should be performed by root user."
fi
}
# ----------------------------------------- #
pkg_upgrade () {
# Upgrade system  or DB packages if needed
#
# (Placeholder)
OSLOGF=/root/allotlog.txt
if [ ! -f $OSLOGF ] ; then
    return
fi

## stop postgresql(-12) services
if [ -f /var/lib/pgsql/12/data/PG_VERSION ] ; then
    systemctl stop postgresql-12.service
else
    systemctl stop postgresql.service
fi

pushd ${INSTD}/packages >& /dev/null
## install/upgrade postgresql packages
rpm -Uvh postgresql12-libs-12.6-1PGDG.rhel7.x86_64.rpm postgresql12-server-12.6-1PGDG.rhel7.x86_64.rpm postgresql12-12.6-1PGDG.rhel7.x86_64.rpm >& /dev/null
rpm -Uvh postgresql12-contrib-12.6-1PGDG.rhel7.x86_64.rpm >& /dev/null

 popd

} # pkg_upgrade
# ----------------------------------------- #
# Check if the product is installed on the host
check_installed () {
rpm -q --quiet $1
INSTALLED=$?
# 0 - installed
if [ $INSTALLED -eq 0 ] ; then
    CURVER=`rpm -q $1`
else
    CURVER=
fi
}
# ----------------------------------------- #
# Check all
check_all () {
check_user
check_ACP
pushd ${INSTD} >& /dev/null
for name in $(rpm -qp --queryformat '%{Name} %{Obsoletes}'  ${ALLOTPRODUCT}*.rpm); do
    check_installed ${name}
    [[ ! -z ${CURVER} ]] && CURRENT_INSTALLED_PRODUCT=${name} && break
done
popd >& /dev/null
check_space $1
if [ $RCODE -gt 0 ] ; then exit_scr $RCODE ; fi
}
# ----------------------------------------- #
ask_yn () {
while [ 1 ] ; do 
    if [ "$MODE" == "AUTO" ] ; then
        ANS="y"
    else
        read -e -p "Continue (y/n) [y]? " ANS
    fi
    if [ -z $ANS ] ; then ANS="y" ; fi      # Default yes
    if [[ $ANS =~ ^[yn]$ ]] ; then
        break
    else 
        echo "Please asnwer 'y' or 'n'."
    fi
done
if [ "$ANS" == "n" ] ; then 
    exit_scr 1
fi
}
# ----------------------------------------- #
# extract TGZ to temporary directory
# there must be exactly one Allot_*.tgz file in the same directory as the script
# the function should be modified if rpm architecture is not i386
extract_tgz() {
local NTGZ=`ls -1 $TPATH/Allot_${ALLOTPRODUCT}*.tgz 2>/dev/null | wc -l`
if [ $NTGZ -ne 1 ] ; then
    echo "There must be one Allot_${ALLOTPRODUCT} TGZ file in the script directory."
    usage
else
    if [ ! "$MD5SUM" = `md5sum $TPATH/Allot_${ALLOTPRODUCT}*.tgz | awk '{print $1}'` ] ; then
        echo "Error checking md5 sum. Please check TGZ file."
        exit_scr 1
    fi
    rm_dir
    mkdir $INSTD
    tar zxf $TPATH/Allot_*.tgz -C $INSTD || { echo "Corrupted archive detected, exiting." ; exit_scr 1 ; }
    local NRPM=`ls -1 ${INSTD}/*.rpm 2>/dev/null | wc -l`
    if [ $NRPM -ne 1 ] ; then
        echo "There must be one RPM file in the archive."
        exit_scr 1
    else
        pushd ${INSTD} >& /dev/null
        THERPM=`ls -1 *.rpm`
        THEPRODUCT=`echo $THERPM | sed 's/-[0-9].*$//'`
        if [ ! "$ALLOTPRODUCT" = "$THEPRODUCT" ] ; then
            echo "TGZ file contains $THEPRODUCT instead of $ALLOTPRODUCT."
            exit_scr 1
        fi
        ALLOTVER=`echo $THERPM | sed "s/$ALLOTPRODUCT-//" | sed "s/\.$ARCH.rpm//"`
        if [ ! "$ALLOTVER" = "$VERBUILD" ] ; then
            echo "$VERBUILD and $THERPM: different verions."
            exit_scr 1
        fi
        popd >& /dev/null
    fi
fi
} # extract_tgz
# ----------------------------------------- #
rm_dir () {
# Remove temporary directory $INSTD"
rm -rf $INSTD >& /dev/null
}
# ----------------------------------------- #
show_help () {
echo "The script will install/upgrade/uninstall Allot $ALLOTPRODUCT $VERBUILD on the host."
echo "To perform install/upgrade, put an appropriate TGZ file into the script directory."
echo "To perform uninstall, it is recommended to use the script of the same version as the product."
echo
} # show_help
# ----------------------------------------- #
# Perform operations needed before installation
pre_install () {
local ret=0
# If the product is already installed
if [ $INSTALLED -eq 0 ] ; then
    echo "$ALLOTPRODUCT is already installed on the host."
    exit_scr 1
else
    echo "You are going to install $ALLOTPRODUCT $ALLOTVER"
    ask_yn
fi

pkg_upgrade

} # pre_install
# ----------------------------------------- #
# Perform operations needed after installation
post_install () {
# if installation succeeded

if [ ! -f  /opt/allot/spconfig/configuration.properties ] ; then
  touch /opt/allot/spconfig/configuration.properties
  chmod 644 /opt/allot/spconfig/configuration.properties

fi
if [ ! -f  /opt/allot/webui/configuration.properties ] ; then
  touch /opt/allot/webui/configuration.properties
  chmod 644 /opt/allot/webui/configuration.properties
fi
if [ ! -f   /opt/allot/dsc/configuration.properties ] ; then
  touch  /opt/allot/dsc/configuration.properties
  chmod 644 /opt/allot/dsc/configuration.properties
fi
if [ ! -f   /opt/allot/spconfig/cli/configuration.properties ] ; then
   touch  /opt/allot/spconfig/cli/configuration.properties
   chmod 644 /opt/allot/spconfig/cli/configuration.properties
fi

if [ ! -f  /opt/allot/log/simplan.log ] ; then
  touch /opt/allot/log/simplan.log
fi
chown allot:allot ${ALLOT_HOME}/log/simplan.log >& /dev/null

## fix symbolic link for PG(postgresql) 12 version
if [ ! -f /var/lib/pgsql/12/data/PG_VERSION ] ; then
    /opt/allot/bin/fix_symb_link.sh
fi

[ -d /var/lib/pgsql ] ||  mkdir /var/lib/pgsql
export LANG=en_US.UTF-8
postgresql-12-setup initdb
systemctl enable postgresql-12.service

cp /opt/allot/DB/conf/{pg_hba,postgresql}.conf /var/lib/pgsql/12/data/
/opt/allot/bin/postgresConfig.sh # set memory parameters for postgres
systemctl start postgresql-12.service
sleep 15

mkdir -p /opt/shared/PostgreSQL/spc_data/pgnetx 
chown -R postgres:postgres /opt/shared/PostgreSQL 
[ -d /opt/allot/log ] || mkdir /opt/allot/log
chmod a+w /opt/allot/log
touch /opt/allot/log/db_purge.log
chown postgres:postgres /opt/allot/log/db_purge.log
pushd /var/lib/pgsql
sudo -u postgres /usr/bin/psql -f /opt/allot/DB/sql/postgres_netx_db_clean_instal.sql > /opt/allot/log/create_netx_db.log 2> /opt/allot/log/create_netx_db_err.log
popd
if [ ! -h /var/lib/pgsql ] ; then
    #mv /var/lib/pgsql /opt/shared/
    cp -rp /var/lib/pgsql /opt/shared/ && rm -rf /var/lib/pgsql
    chown -R postgres:postgres /opt/shared/pgsql
    ln -s /opt/shared/pgsql /var/lib/pgsql
fi

create_audit_log
replace_rsyslog
# copy the script to /root for further use
if [ $RCODE -eq 0 ] ; then
    if [ ! "${TPATH%/}" == "/root" ] ; then
        cp -f $0 /root/ && { echo "$SCRIPT was copied to /root."; }
        cp -f $0 /opt/allot/bin/ && { echo "$SCRIPT was copied to /opt/allot/bin."; }
    fi
    if [ ! -d /opt/allot/products ] ; then mkdir -p /opt/allot/products ; fi
    echo "$ALLOTPRODUCT $VERBUILD" > /opt/allot/products/${ALLOTPRODUCT}.ver
    chown -R admin:allot /opt/allot/products

    if [ ! -d /opt/allot/webui/reports ] ; then mkdir -p /opt/allot/webui/reports ; fi
    chown -R admin:allot /opt/allot/webui/reports

    grep -q "HA enabled" /opt/allot/products/${ALLOTPRODUCT}.ver && add_vip_ha
    restart_product
    # Set NTP server
    /opt/allot/bin/firewall_utils.sh --start-all-accepted --start-on-boot --redirect-port 80 8080 tcp --redirect-port 443 8443 tcp
    echo "********* NTP ************"
    /opt/allot/bin/set_main_ntp_server.sh
fi
systemctl enable ${ALLOTPRODUCT}
} # post_install
# ----------------------------------------- #
# Perform operations needed before upgrade
pre_upgrade () {
if [ $INSTALLED -ne 0 ] ; then
    echo "$ALLOTPRODUCT is not installed on the host."
    exit_scr 1
else
    ##StrValidate=`echo $CURVER | sed 's/\.i386//' | sed "s/${ALLOTPRODUCT}-//" | sed 's/-/b/'`    
    if [[ "$( echo $CURVER | cut -f4 -d'.' )" == "i386" ]]
      then
       echo "You are going to upgrade ${CURVER%%.i386} to version $ALLOTVER"
      else
       echo "You are going to upgrade ${CURVER%%.x86_64} to version $ALLOTVER"
    fi
    ask_yn
fi
    # Since starting 16.5.10 sp product name changed to dsc
    mv ${ALLOT_HOME}/products/${CURRENT_INSTALLED_PRODUCT}.ver ${ALLOT_HOME}/products/${ALLOTPRODUCT}.ver
   # Backup old tomcat reports directory
  [[ -d /opt/allot/tomcat/webapps/webui/reports/ ]] && echo "Backuping old tomcat reports directory.." && mv /opt/allot/tomcat/webapps/webui/reports/ /opt/allot/tmp/reports.bak/
   # Backup reports directory
  [[ -d /opt/allot/webui/reports/ ]] && echo "Backuping reports directory.." && mv /opt/allot/webui/reports/ /opt/allot/tmp/reports.bak/

  SP_PORT=$(grep -Po '(controllerServerPort" value=")\K.+(?="\s*/>)' ${ALLOT_HOME}/dsc/conf/nms_config.xml)

  pkg_upgrade

}
# ----------------------------------------- #
# Perform operations needed after upgrade
post_upgrade () {

rm -rf /opt/allot/spconfig/cli/.terminalfx/

if [ ! -f  /opt/allot/spconfig/configuration.properties ] ; then
    touch /opt/allot/spconfig/configuration.properties
    chmod 644 /opt/allot/spconfig/configuration.properties
fi
if [ ! -f  /opt/allot/webui/configuration.properties ] ; then
    touch /opt/allot/webui/configuration.properties
    chmod 644 /opt/allot/webui/configuration.properties
fi
if [ ! -f   /opt/allot/dsc/configuration.properties ] ; then
    touch  /opt/allot/dsc/configuration.properties
    chmod 644 /opt/allot/dsc/configuration.properties
fi
if [ ! -f   /opt/allot/spconfig/cli/configuration.properties ] ; then
    touch  /opt/allot/spconfig/cli/configuration.properties
    chmod 644 /opt/allot/spconfig/cli/configuration.properties
fi
# if upgrade succeeded 
if [ $RCODE -eq 0 ] ; then
    # copy the script to /root for further use
    if [ ! "${TPATH%/}" == "/root" ] ; then
        cp -f $0 /root/ && { echo "$SCRIPT was copied to /root."; }
        cp -f $0 /opt/allot/bin/ && { echo "$SCRIPT was copied to /opt/allot/bin."; }
    fi

    # Since old RPMs didn't placed work directory of tomcat as RPM's file.
    chown allot:allot ${ALLOT_HOME}/log/webui.log >& /dev/null
    chown allot:allot ${ALLOT_HOME}/log/simplan.log >& /dev/null
    chown allot:allot ${ALLOT_HOME}/webui/temp >& /dev/null
    # Since old RPMs didn't placed /opt/allot/sp as work directory
    rm -rf /opt/allot/sp
    rm -rf /opt/shared/sp
    # Since old RPMs didn't placed /opt/allot/sp as work directory
    rm -rf /opt/allot/java

    sed -i "s/$CURRENT_INSTALLED_PRODUCT .*/$ALLOTPRODUCT $VERBUILD/" ${ALLOT_HOME}/products/${ALLOTPRODUCT}.ver

    # Restore reports
    [[ -d /opt/allot/tmp/reports.bak ]] && mv /opt/allot/tmp/reports.bak /opt/allot/webui/reports

    chown -R admin:allot /opt/allot/products
    grep -q "HA enabled" /opt/allot/products/${ALLOTPRODUCT}.ver && post_upgrade_ha || { systemctl enable ${ALLOTPRODUCT} &&  restart_product; }
    /opt/allot/bin/firewall_utils.sh --start-all-accepted --start-on-boot --redirect-port 80 8080 tcp --redirect-port 443 8443 tcp

    create_audit_log
    replace_rsyslog
    # Set NTP server
    echo "********* NTP ************"
    /opt/allot/bin/set_main_ntp_server.sh

    # Changing the cli.log owner to allot
    CLI_LOG_FILE=${ALLOT_HOME}/log/cli.log
    if [ -f "${CLI_LOG_FILE}" ]; then
        if [ `stat -c '%U' ${CLI_LOG_FILE}` = "root" ]; then
            chown allot:allot "${CLI_LOG_FILE}"
        fi
    fi

    if [ -n "$SP_PORT" ];
    	then  sed -i "s/server.port=.*/server.port=$SP_PORT/g" "${ALLOT_HOME}/dsc/application.properties";
    fi
fi
}

add_vip_ha() {
    VIRTUALIP=$(/opt/allot/bin/get_vip.sh | grep -o '".*"' | sed 's/"//g')
    if [[ -n ${VIRTUALIP} ]]; then
      echo "adding VIP to allowed hosts ${VIRTUALIP}"
      sed -i "s/^com.allot.sp.configuration.host-headers=.*/com.allot.sp.configuration.host-headers=${VIRTUALIP}/" /opt/allot/webui/application.properties
      sed -i "s/^com.allot.sp.configuration.host-headers=.*/com.allot.sp.configuration.host-headers=${VIRTUALIP}/" /opt/allot/spconfig/application.properties
    else
        echo "Error: Virtual IP is empty, webui will not be accessible"
    fi
}

post_upgrade_ha() {
    local GROUP=`pcs resource --groups | grep sp | head -1 | cut -d':' -f1`

    if pcs resource show sp >& /dev/null; then
      RESOURCE_NAME=sp
    elif pcs resource show sp-service; then
      RESOURCE_NAME=sp-service
    fi

    if [[ -n ${RESOURCE_NAME} ]]; then
        pcs resource delete ${RESOURCE_NAME} >& /dev/null || echo "Failed to delete old SP resource, ${RESOURCE_NAME}, from pacemaker" &&
        pcs resource create dsc lsb:dsc op start timeout="60"  op stop timeout="60"  op monitor interval="15" timeout="40" on-fail="restart" ||
        echo "Failed to create dsc lsb resource at pacemaker" &&
        pcs resource group add ${GROUP} dsc || echo "failed to add dsc lsb resource to cluster group: ${GROUP}"
    fi

    add_vip_ha
}
# ----------------------------------------- #
# Perform operations needed before uninstallation
pre_uninstall () {
firewall-cmd --state --quiet
local firewall_state=$?
if [ $INSTALLED -ne 0 ] ; then
    echo "$ALLOTPRODUCT is not installed on the host."
    exit_scr 1
elif [ $RCODE -ne 0 ] ; then
    echo "Must be root."
    exit_scr 2
else
     if [[ "$( echo $CURVER | cut -f4 -d'.' )" == "i386" ]]
      then
       echo "You are going to uninstall ${CURVER%%.i386}"
      else
       echo "You are going to uninstall ${CURVER%%.x86_64} "
    fi
    ask_yn
fi
systemctl restart postgresql-12.service
sleep 10
pushd /opt/allot 
sudo -u postgres /usr/bin/psql -f /opt/allot/DB/sql/drop_netx_db.sql > /opt/allot/log/drop_netx_db.log 2> /opt/allot/log/drop_netx_db_err.log
systemctl stop postgresql-12.service
systemctl disable postgresql-12.service

[[ firewall_state -ne 0 ]] && systemctl start firewalld
/opt/allot/bin/firewall_utils.sh --recall-redirect-port 80 8080 tcp --recall-redirect-port 443 8443 tcp
[[ firewall_state -ne 0  ]] && systemctl stop firewalld

popd
} # pre_uninstal
# ----------------------------------------- #
# Perform operations needed after removing the product
post_uninstall () {
local ret=0
if [ $RCODE -eq 0 ] ; then
    rm -f /opt/allot/products/${ALLOTPRODUCT}.ver >&/dev/null
    rm -rf /opt/shared
    
    rm -f /var/lib/pgsql
    mkdir /var/lib/pgsql/12
    chown -R postgres:postgres /var/lib/pgsql

    pushd /opt/allot/
    rm -rf `ls | grep -v products`
    popd 
fi
}
# ----------------------------------------- #

verify_service_up(){
    for I in 1 2 3 4 5
    do
        sleep 10
        echo "Verifying $1 service is up. Attempt $I"
        state=$(systemctl is-active $1)
        echo "Service $1 status is $state"
        if [ $state == "active" ]; then
            return 0
        fi
    done
        echo "Max retries reached, service $1 failed to start"
    return 1
}

verify_service_down(){
    for I in 1 2 3 4 5
    do
        sleep 10
        echo "Verifying $1 service is down. Attempt $I"
        state=$(systemctl is-active $1)
        echo "Service $1 status is $state"
        if [ $state != "active" ]; then
            return 0
        fi
    done
        echo "Max retries reached, service $1 failed to stop"
    return 1
}

restart_product () {
    postgresql_12_service="postgresql-12.service"
    echo "Restarting ..."
    echo "Stopping $ALLOTPRODUCT service"
    systemctl stop $ALLOTPRODUCT >&/dev/null
    verify_service_down $ALLOTPRODUCT
    echo "Restarting $postgresql_12_service service"
    systemctl restart $postgresql_12_service >&/dev/nul
    verify_service_up $postgresql_12_service
    echo "Starting $ALLOTPRODUCT service"
    systemctl start $ALLOTPRODUCT >&/dev/null
    verify_service_up $ALLOTPRODUCT
    echo "Restart Done."
}

create_audit_log () {
log_file=/opt/allot/log/audit.log
conf_file=/etc/logrotate.d/dsc.logs
if [ ! -f $log_file ] ; then
    touch $log_file
    chmod 644 $log_file
fi
if [ ! -f $conf_file ] ; then
    /bin/cp /opt/allot/conf/dsc.logs /etc/logrotate.d/
    chmod 644 $conf_file
fi
}

replace_rsyslog () {
/bin/cp /opt/allot/conf/rsyslog.conf /etc/
systemctl restart rsyslog
}

# ----------------------------------------- #
{
############ M A I N ###############
# Check command line parameter
if [ ! $# -eq 1 ] ; then 
    if [ "$2" == "-a" ] ; then
        MODE=AUTO
    else
        if [ "$2" == "-f" ] ; then
            MODE=FORCE
        else
            usage   
        fi  
    fi
fi
date
# Uninstall and help - no need in TGZ
case $1 in
  '-r'|'-remove')
    check_user
    check_installed ${ALLOTPRODUCT}
    echo "Uninstall..."
    pre_uninstall
    rpm -e $ALLOTPRODUCT
    RCODE=$?
    post_uninstall
     ;;
  '-h' | '-help')
    show_help
    usage
     ;;
  '-i'|'-install')
    echo "Install..."
     ;;
  '-u'|'-upgrade')
    echo "Upgrade..."
     ;;
  *)
    usage
     ;;      
esac

# Perform an operation requested
case $1 in
  '-i'|'-install')
     extract_tgz
     check_all $1
     pre_install
     pushd $INSTD >& /dev/null
     rpm -ivh *.rpm
     let RCODE=$RCODE+$?
     popd >& /dev/null
     post_install
     ;;
  '-u'|'-upgrade')
     extract_tgz
     check_all $1
     pre_upgrade
     pushd $INSTD >& /dev/null
     rpm -Uvh *.rpm
     RCODE=$?
     popd >& /dev/null
     post_upgrade
     ;;    
esac

date
} 2>&1 | tee -a $LOGF
if [ -d /opt/allot/log ] ; then
    cat $LOGF >> $LOGD/$LOGF
    chown admin:allot $LOGD/$LOGF
fi
exit_scr 0
